//
//   File : gnutellasearch.cpp
//   Creation date : Sat Sep  8 01:24:47 2001 GMT by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2001 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "gnutellasearch.h"
#include "gnutellawindow.h"
#include "gnutellathread.h"

#include "kvi_iconmanager.h"
#include "kvi_locale.h"
#include "kvi_memmove.h"
#include "kvi_config.h"
#include "kvi_filedialog.h"
#include "kvi_malloc.h"

#include <qlayout.h>
#include <qregexp.h>
#include <qvbox.h>
#include <qtabwidget.h>
#include <qasciidict.h>


KviGnutellaHitItem::KviGnutellaHitItem(QListView * par,KviGnutellaQueryHitInfo * inf)
: QListViewItem(par)
{
	m_pInfo = inf;

	int iPixmap = KVI_SMALLICON_BOMB;                                       // > T1 (T3 or a lie)

	if(inf->uSpeed < 5)iPixmap = KVI_SMALLICON_BLACKSQUARE;               // Unknown ?
	else if(inf->uSpeed < 34)iPixmap = KVI_SMALLICON_BLUESQUARE;          // <= 33 K modem
	else if(inf->uSpeed < 57)iPixmap = KVI_SMALLICON_CYANSQUARE;          // 56 K modem
	else if(inf->uSpeed < 65)iPixmap = KVI_SMALLICON_VIOLETSQUARE;        // ISDN
	else if(inf->uSpeed < 129)iPixmap = KVI_SMALLICON_REDSQUARE;          // Dual ISDN
	else if(inf->uSpeed < 513)iPixmap = KVI_SMALLICON_YELLOWSQUARE;       // Cable modem
	else if(inf->uSpeed < 769)iPixmap = KVI_SMALLICON_DKGREENSQUARE;      // Adsl
	else if(inf->uSpeed < 1025)iPixmap = KVI_SMALLICON_GREENSQUARE;       // T1

	setPixmap(0,*(g_pIconManager->getSmallIcon(iPixmap)));

	setText(0,m_pInfo->szName.ptr());
	KviStr tmp(KviStr::Format,"%u",m_pInfo->uSize);
	setText(1,tmp.ptr());
	tmp.sprintf("%u",m_pInfo->uSpeed);
	setText(2,tmp.ptr());
	setText(3,m_pInfo->szIp.ptr());
	tmp.sprintf("%u",m_pInfo->uPort);
	setText(4,tmp.ptr());
	if(m_pInfo->szExtraInfo.hasData())
	{
		setText(5,m_pInfo->szExtraInfo.ptr());
	}
}

KviGnutellaHitItem::~KviGnutellaHitItem()
{
	delete m_pInfo;
}


QString KviGnutellaHitItem::key(int col,bool) const
{
	if((col == 1) || (col == 2) || (col == 3))
	{
		QString txt = text(col);
		int len = txt.length();
		// senseless check
		if(len > 15)len = 15;
		txt.prepend(QChar(((char)('a' + len))));
		return txt;
	}
	return text(col);
}







KviGnutellaSearchTab::KviGnutellaSearchTab(QWidget * par,KviGnutellaWindow * wnd)
: QWidget(wnd)
{
	m_pGnutellaWindow = wnd;

	m_pPendingHits = new KviPtrList<KviGnutellaQueryHitInfo>;
	m_pPendingHits->setAutoDelete(true);

	QGridLayout * g = new QGridLayout(this,4,4,2,2);

	m_pListView = new QListView(this);
	m_pListView->setAllColumnsShowFocus(true);
	m_pListView->setSelectionMode(QListView::Extended);

	m_pListView->addColumn(__tr("File"),200);
	m_pListView->addColumn(__c2q(__tr("Size (Bytes)")));
	m_pListView->addColumn(__tr("Speed (KBit/sec)"));
	m_pListView->addColumn(__tr("Host"));
	m_pListView->addColumn(__tr("Port"));
	m_pListView->addColumn(__tr("Extra info"));

	// People use horrible filenames...
	m_pListView->setColumnWidthMode(0,QListView::Manual);

	connect(m_pListView,SIGNAL(doubleClicked(QListViewItem *)),this,SLOT(searchResultDoubleClicked(QListViewItem *)));
	connect(m_pListView,SIGNAL(selectionChanged()),this,SLOT(listViewSelectionChanged()));

	g->addMultiCellWidget(m_pListView,0,0,0,2);




	QTabWidget * tab = new QTabWidget(this);
	tab->setTabPosition(QTabWidget::Bottom);
	g->addWidget(tab,0,3);

	QVBox * filtertab = new QVBox(tab);
	filtertab->setSpacing(2);
	filtertab->setMargin(2);

	QLabel * lab = new QLabel(filtertab);
	lab->setText(__tr("File name regexp:"));

	m_pRegExpEdit = new QLineEdit(filtertab);

	m_pApplyFilterButton = new QPushButton(__tr("Apply filter"),filtertab);
	connect(m_pApplyFilterButton,SIGNAL(clicked()),this,SLOT(applyFilter()));

	QFrame * sep = new QFrame(filtertab);
	sep->setFrameStyle(QFrame::Sunken | QFrame::HLine);

	QPushButton * pb = new QPushButton(__c2q(__tr("Clear duplicates")),filtertab);
	connect(pb,SIGNAL(clicked()),this,SLOT(clearDuplicates()));

	lab = new QLabel(filtertab);
	filtertab->setStretchFactor(lab,1);

	tab->addTab(filtertab,__tr("Filter"));

	QVBox * savetab = new QVBox(tab);
	savetab->setSpacing(2);
	savetab->setMargin(2);

	m_pSaveAllButton = new QPushButton(__c2q(__tr("Save hit list")),savetab);
	connect(m_pSaveAllButton,SIGNAL(clicked()),this,SLOT(saveHitList()));
	m_pSaveAllButton->setEnabled(false);

	m_pSaveSelectedButton = new QPushButton(__c2q(__tr("Save selected")),savetab);
	connect(m_pSaveSelectedButton,SIGNAL(clicked()),this,SLOT(saveSelectedHitList()));
	m_pSaveSelectedButton->setEnabled(false);

	m_pLoadButton = new QPushButton(__c2q(__tr("Load hit list")),savetab);
	connect(m_pLoadButton,SIGNAL(clicked()),this,SLOT(loadHitList()));

	lab = new QLabel(savetab);
	savetab->setStretchFactor(lab,1);

	tab->addTab(savetab,__tr("Store"));




	m_pInfoLabel = new QLabel(this);
	m_pInfoLabel->setFrameStyle(QFrame::Sunken | QFrame::Panel);

	g->addMultiCellWidget(m_pInfoLabel,1,1,0,1);

	updateInfoLabel();


	m_pLockCheck = new QCheckBox(__c2q(__tr("Lock")),this);
	connect(m_pLockCheck,SIGNAL(toggled(bool)),this,SLOT(lockToggled(bool)));
	g->addWidget(m_pLockCheck,1,2);


	QLabel * l = new QLabel(this);
	l->setText(__c2q(__tr("Search for")));

	g->addWidget(l,2,0);

/*
	m_pSearchString = new QLineEdit(this);
	connect(m_pSearchString,SIGNAL(returnPressed()),this,SLOT(doSearch()));
	connect(m_pSearchString,SIGNAL(textChanged(const QString &)),this,SLOT(updateSearchButtonState(const QString &)));
*/
	m_pSearchString = new QComboBox(true,this);
	m_pSearchString->setDuplicatesEnabled(false);
	connect(m_pSearchString->lineEdit(),SIGNAL(returnPressed()),this,SLOT(doSearch()));
	connect(m_pSearchString,SIGNAL(textChanged(const QString &)),this,SLOT(updateSearchButtonState(const QString &)));
	connect(m_pSearchString,SIGNAL(activated(const QString &)),this,SLOT(searchStringActivated(const QString &)));

	g->addMultiCellWidget(m_pSearchString,2,2,1,2);


	l = new QLabel(this);
	l->setText(__tr("Minimum speed"));

	g->addWidget(l,3,0);


	m_pSearchMinSpeed = new QLineEdit(this);
	connect(m_pSearchMinSpeed,SIGNAL(returnPressed()),this,SLOT(doSearch()));
	connect(m_pSearchMinSpeed,SIGNAL(textChanged(const QString &)),this,SLOT(updateSearchButtonState(const QString &)));

	g->addWidget(m_pSearchMinSpeed,3,1);


	m_pSearchButton = new QPushButton(__c2q(__tr("Search")),this);
	m_pSearchButton->setEnabled(false);
	connect(m_pSearchButton,SIGNAL(clicked()),this,SLOT(doSearch()));

	g->addWidget(m_pSearchButton,3,2);


	m_pClearAll = new QPushButton(__c2q(__tr("Clear all")),this);
	connect(m_pClearAll,SIGNAL(clicked()),this,SLOT(clearAll()));

	g->addWidget(m_pClearAll,1,3);


	m_pClearSelected = new QPushButton(__c2q(__tr("Clear selected")),this);
	m_pClearSelected->setEnabled(false);
	connect(m_pClearSelected,SIGNAL(clicked()),this,SLOT(clearSelected()));

	g->addWidget(m_pClearSelected,2,3);


	m_pDownloadSelected = new QPushButton(__c2q(__tr("Download selected")),this);
	m_pDownloadSelected->setEnabled(false);
	connect(m_pDownloadSelected,SIGNAL(clicked()),this,SLOT(downloadSelected()));


	g->addWidget(m_pDownloadSelected,3,3);

	g->setColStretch(1,1);
	g->setRowStretch(0,1);
}

KviGnutellaSearchTab::~KviGnutellaSearchTab()
{
	delete m_pPendingHits;
}

void KviGnutellaSearchTab::lockToggled(bool bChecked)
{
	if(!bChecked)
	{
		// add the pending results
		m_pPendingHits->setAutoDelete(false);
//		m_pListView->setUpdatesEnabled(false);
		KviGnutellaHitItem * it;
		while(KviGnutellaQueryHitInfo * inf = m_pPendingHits->first())
		{
			it = new KviGnutellaHitItem(m_pListView,inf);
			m_pPendingHits->removeFirst();
		}
//		m_pListView->setUpdatesEnabled(true);
//		m_pListView->update(); // this is kinda buggy in qt 2.x...the scroll bars do not get updated!
		m_pPendingHits->setAutoDelete(true);
		updateInfoLabel();
	}
}

void KviGnutellaSearchTab::listViewSelectionChanged()
{
	QListViewItem * it = m_pListView->firstChild();
	while(it)
	{
		if(it->isSelected())
		{
			m_pClearSelected->setEnabled(true);
			m_pDownloadSelected->setEnabled(true);
			m_pSaveSelectedButton->setEnabled(true);
			return;
		}
		it = it->nextSibling();
	}
	m_pClearSelected->setEnabled(false);
	m_pDownloadSelected->setEnabled(false);
	m_pSaveSelectedButton->setEnabled(false);
}

void KviGnutellaSearchTab::searchResultDoubleClicked(QListViewItem * it)
{
	if(!it)return;
	m_pGnutellaWindow->startDownload(((KviGnutellaHitItem *)it)->m_pInfo);
}

KviGnutellaQueryHitInfo * KviGnutellaSearchTab::allocQueryHitInfo(KviGnutellaHit * h,KviGnutellaHitThreadEvent *e)
{
	KviGnutellaQueryHitInfo * inf = new KviGnutellaQueryHitInfo;
	inf->szName = h->szName;
	inf->uSize = h->uSize;
	inf->uIndex = h->uIndex;
	inf->uSpeed = e->m_uSpeed;
	inf->uPort = e->m_uPort;
	inf->szIp = e->m_szIp;
	inf->iNodeId = e->m_iNodeId;
	inf->szExtraInfo = h->szExtraInfo;
	kvi_memmove(inf->servId,e->m_servId,16);

	return inf;
}

void KviGnutellaSearchTab::addQueryHit(KviGnutellaHit * h,KviGnutellaHitThreadEvent *e)
{
	KviGnutellaQueryHitInfo * info = allocQueryHitInfo(h,e);
	addQueryHit(info);
}

void KviGnutellaSearchTab::addQueryHit(KviGnutellaQueryHitInfo * info)
{
	if(m_pLockCheck->isChecked())
	{
//#warning "Maybe post an event about removed hits for no space in the list ?"
		if(m_pPendingHits->count() > 20000)m_pPendingHits->removeFirst();
		m_pPendingHits->append(info);
	} else {
		if(m_pListView->childCount() > 10000)
		{
			QListViewItem * it = m_pListView->firstChild();
			delete it;
		}
		KviGnutellaHitItem * it = new KviGnutellaHitItem(m_pListView,info);
	}

	updateInfoLabel();
}

void KviGnutellaSearchTab::updateInfoLabel()
{
	KviStr tmp(KviStr::Format,__tr("%d hits (%d visible, %d pending)"),m_pListView->childCount() + m_pPendingHits->count(),m_pListView->childCount(),m_pPendingHits->count());
	if(m_pListView->childCount() > 0)
	{
		if(!(m_pSaveAllButton->isEnabled()))m_pSaveAllButton->setEnabled(true);
	} else {
		if(m_pSaveAllButton->isEnabled())m_pSaveAllButton->setEnabled(false);
	}
	m_pInfoLabel->setText(tmp.ptr());
}

void KviGnutellaSearchTab::doSearch()
{
	doSearchString(m_pSearchString->currentText());
}

void KviGnutellaSearchTab::doSearchString(const QString &str)
{
	KviStr tmp = str;
	if(tmp.hasData())
	{
		KviStr num = m_pSearchMinSpeed->text();
		unsigned short int uShort;

		if(num.isEmpty())uShort = 0;
		else {
			bool bOk;
			uShort = num.toUShort(&bOk);
			if(!bOk)return;
		}

		KviGnutellaThreadEvent * e = new KviGnutellaThreadEvent(KVI_GNUTELLA_WINDOW_EVENT_DO_SEARCH);
		e->m_szData = tmp;
		e->m_uPort = uShort;
		m_pGnutellaWindow->mainGnutellaThread()->enqueueEvent(e);	
		if(m_pSearchString->lineEdit())m_pSearchString->lineEdit()->setText("");
		m_pSearchMinSpeed->setText("");
		updateSearchButtonState(QString::null);
	}
}

void KviGnutellaSearchTab::searchStringActivated(const QString &)
{
	m_pSearchString->lineEdit()->setFocus();
}

void KviGnutellaSearchTab::updateSearchButtonState(const QString &str)
{
	KviStr tmp = m_pSearchMinSpeed->text();
	tmp.stripWhiteSpace();
	KviStr text = str;//m_pSearchString->text();
	m_pSearchButton->setEnabled((tmp.isUnsignedNum() || tmp.isEmpty()) &&
		text.hasData() && (m_pGnutellaWindow->connectedNodes() > 0));
}

void KviGnutellaSearchTab::clearAll()
{
	m_pListView->clear();
	updateInfoLabel();
	listViewSelectionChanged();
}

void KviGnutellaSearchTab::clearSelected()
{
	KviPtrList<QListViewItem> l;
	l.setAutoDelete(true);
	QListViewItem * it = m_pListView->firstChild();
	while(it)
	{
		if(it->isSelected())l.append(it);
		it = it->nextSibling();
	}
	updateInfoLabel();
	listViewSelectionChanged();
}

void KviGnutellaSearchTab::downloadSelected()
{
	QListViewItem * it = m_pListView->firstChild();
	while(it)
	{
		if(it->isSelected())searchResultDoubleClicked(it);
		it = it->nextSibling();
	}
}

KviGnutellaQueryHitInfo * KviGnutellaSearchTab::findBestQueryHit(const char * fileName,unsigned int uSize,KviPtrList<KviStr> * excludeHosts)
{
	KviPtrList<KviGnutellaHitItem> l;
	l.setAutoDelete(false);

	KviGnutellaHitItem * it = (KviGnutellaHitItem *)m_pListView->firstChild();
	while(it)
	{
		if(it->m_pInfo->uSize == uSize)
		{
			if(kvi_strEqualCS(it->m_pInfo->szName.ptr(),fileName))
			{
				bool bGotIt = false;
				if(excludeHosts)
				{
					for(KviStr * s = excludeHosts->first();s && !bGotIt;s = excludeHosts->next())
					{
						if(kvi_strEqualCS(it->m_pInfo->szIp.ptr(),s->ptr()))bGotIt = true;
					}
				}
				if(!bGotIt)
				{
					// append with greater speeds first
					int idx = 0;
					for(KviGnutellaHitItem * h = l.first();h && !bGotIt;h = l.next())
					{
						if(h->m_pInfo->uSpeed < it->m_pInfo->uSpeed)
						{
							l.insert(idx,it);
							bGotIt = true;
						}
						idx++;
					}
					if(!bGotIt)l.append(it);
				}
			}
		}
		it = (KviGnutellaHitItem *)it->nextSibling();
	}
	KviGnutellaQueryHitInfo * inf = l.first() ? l.first()->m_pInfo : 0;
	return inf;
}

void KviGnutellaSearchTab::putHit(KviConfig * cfg,KviGnutellaHitItem * it,int idx)
{
	KviStr tmp(KviStr::Format,"Hit%d",idx);
	cfg->setGroup(tmp.ptr());

	cfg->writeEntry("szName",it->m_pInfo->szName.ptr());
	cfg->writeEntry("szIp",it->m_pInfo->szIp.ptr());
	cfg->writeEntry("uPort",it->m_pInfo->uPort);
	cfg->writeEntry("uSpeed",it->m_pInfo->uSpeed);
	cfg->writeEntry("uSize",it->m_pInfo->uSize);
	cfg->writeEntry("uIndex",it->m_pInfo->uIndex);

	tmp.bufferToHex((char *)(it->m_pInfo->servId),16);
	cfg->writeEntry("servId",tmp.ptr());

}

KviGnutellaQueryHitInfo * KviGnutellaSearchTab::getHit(KviConfig * cfg,int idx)
{
	KviStr tmp(KviStr::Format,"Hit%d",idx);
	cfg->setGroup(tmp.ptr());

	KviGnutellaQueryHitInfo * inf = new KviGnutellaQueryHitInfo;

	inf->iNodeId = -1;
	inf->szName = cfg->readEntry("szName","unknown");
	inf->szIp = cfg->readEntry("szIp","0.0.0.0");
	inf->uPort = cfg->readUShortEntry("uPort",0);
	inf->uIndex = cfg->readUIntEntry("uIndex",0);
	inf->uSpeed = cfg->readUIntEntry("uSpeed",0);
	inf->uSize = cfg->readUIntEntry("uSize",0);

	tmp = cfg->readEntry("servId","00000000000000000000000000000000");
	char * buf;
	int ret = tmp.hexToBuffer(&buf);
	if(ret == 16)
	{
		kvi_memmove(inf->servId,buf,16);
	}
	if(ret > 0)KviStr::freeBuffer(buf);
	return inf;
}

void KviGnutellaSearchTab::saveHitListToFile(bool bSelectedOnly)
{
	KviStr buffer;
	if(KviFileDialog::askForSaveFileName(buffer,__tr("Select a name for the hit list"),0,0,false,true))
	{
		if(buffer.hasData())
		{
			KviConfig cfg(buffer.ptr());
			cfg.clear();

			int idx = 0;

			for(QListViewItem * it = m_pListView->firstChild();it;it = it->nextSibling())
			{
				if(it->isSelected() || (!bSelectedOnly))putHit(&cfg,(KviGnutellaHitItem *)it,idx);
				idx++;
			}

			cfg.setGroup("KviGnutellaHitsList");
			cfg.writeEntry("HitCount",idx);
		}
	}
}

void KviGnutellaSearchTab::saveHitList()
{
	saveHitListToFile(false);
}

void KviGnutellaSearchTab::saveSelectedHitList()
{
	saveHitListToFile(true);
}

void KviGnutellaSearchTab::loadHitList()
{
	KviStr buffer;
	if(KviFileDialog::askForOpenFileName(buffer,__tr("Select a hit list file")))
	{
		if(buffer.hasData())
		{
			KviConfig cfg(buffer.ptr(),true);

			cfg.setGroup("KviGnutellaHitsList");
			int count = cfg.readIntEntry("HitCount",0);

			for(int i=0;i<count;i++)
			{
				KviGnutellaQueryHitInfo * inf = getHit(&cfg,i);
				addQueryHit(inf);
			}
		}
	}	
}

void KviGnutellaSearchTab::applyFilter()
{
	QString tmp = m_pRegExpEdit->text();
	QRegExp exp(tmp);

	KviPtrList<KviGnutellaHitItem> l;
	l.setAutoDelete(true);

	for(QListViewItem * it = m_pListView->firstChild();it;it = it->nextSibling())
	{
#if QT_VERSION >= 300
		if(exp.search(((KviGnutellaHitItem *)it)->m_pInfo->szName.ptr(),0) < 0)l.append((KviGnutellaHitItem *)it);
#else
		if(exp.find(((KviGnutellaHitItem *)it)->m_pInfo->szName.ptr(),0) < 0)l.append((KviGnutellaHitItem *)it);
#endif
	}
}

void KviGnutellaSearchTab::clearDuplicates()
{
	QAsciiDict<QString> theDict;
	theDict.setAutoDelete(false);

	KviPtrList<KviGnutellaHitItem> l;
	l.setAutoDelete(true);

	QString dummy;

	for(QListViewItem * it = m_pListView->firstChild();it;it = it->nextSibling())
	{
		KviStr unique(KviStr::Format,"%s_%s_%u",
			((KviGnutellaHitItem *)it)->m_pInfo->szName.ptr(),
			((KviGnutellaHitItem *)it)->m_pInfo->szIp.ptr(),
			((KviGnutellaHitItem *)it)->m_pInfo->uSize);
		if(theDict.find(unique.ptr()))l.append((KviGnutellaHitItem *)it);
		else theDict.insert(unique.ptr(),&dummy);
	}
}

#include "m_gnutellasearch.moc"
