//
//   File : gnutellathread.cpp
//   Creation date : Mon Apr 16 2001 17:53:11 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2001 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the gnutellas of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

//#warning "Timeout for nodes ?"

#define _GNUTELLAROUTING_CPP_

#include "gnutellathread.h"
#include "gnutellawindow.h"
#include "gnutellanode.h"
#include "gnutellashares.h"
#include "gnutellaoptions.h"
#include "gnutelladcache.h"
#include "gnutellatransfer.h"

#include "kvi_error.h"
#include "kvi_netutils.h"
#include "kvi_locale.h"
#include "kvi_memmove.h"
#include "kvi_malloc.h"
#include "kvi_socket.h"
#include "kvi_time.h"

#include <stdlib.h>
#include <errno.h>

#define _GNUTELLADEBUG

// libkvignutella.cpp
extern unsigned int                     g_uGnutellaSharedFilesCount;
extern unsigned int                     g_uGnutellaSharedBytes;

extern KviPtrList<KviGnutellaSearchQuery>  * g_pGnutellaSearchQueryList;
extern KviMutex                       * g_pGnutellaSearchQueryMutex;

extern KviPtrList<KviGnutellaSharedFile>   * g_pGnutellaSharedFiles;
extern KviMutex                       * g_pGnutellaSharedFilesMutex;

extern unsigned int                     g_uGnutellaLocalIpAddress;






void KviGnutellaThread::dropCachedDescriptors(KviGnutellaNode * n)
{
	m_pRoutedPingCache->removeAllByNode(n);
	m_pRoutedQueryCache->removeAllByNode(n);
	m_pRoutedPushCache->removeAllByNode(n);
	m_pQueryHitServentsCache->removeAllByNode(n);
}

void KviGnutellaThread::sendPing(KviGnutellaNode *n)
{

	unsigned char * buffer = n->allocOutgoingBuffer(GNUTELLA_DESCRIPTOR_SIZE);
	buildDescriptor(buffer,GNUTELLA_PING,m_uDefaultTtl,0,0);

	m_pOwnPingCache->cache(buffer,0);
}


void KviGnutellaThread::doSearch(KviStr & string,unsigned short int uMinSpeed)
{

	int len = GNUTELLA_DESCRIPTOR_SIZE + 3 + string.len();
	unsigned char * buffer = (unsigned char *)kvi_malloc(len);
	buildDescriptor(buffer,GNUTELLA_QUERY,m_uDefaultTtl,3 + string.len(),0);

	m_pOwnQueryCache->cache(buffer,0);

	*((Q_UINT16 *)(buffer + GNUTELLA_DESCRIPTOR_SIZE)) = hToGnutellaS(uMinSpeed);
	kvi_memmove(buffer + GNUTELLA_DESCRIPTOR_SIZE + 2,string.ptr(),string.len() + 1);

	int nSent = sendToAll(buffer,len);

	kvi_free(buffer);

	if(nSent == 0)
		deferredPostEvent(new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_WARNING,
			new KviStr(__tr_no_lookup("Can't start search: failed to send all the query packets"))));
	else {
		deferredPostEvent(new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_MESSAGE,
			new KviStr(KviStr::Format,
				__tr_no_lookup("Search started for \"%s\" and minimum speed of %u KB/S: sent %d packets to neighbours"),
				string.ptr(),uMinSpeed,nSent)));
	}
}

void KviGnutellaThread::sendPushRequest(KviGnutellaNode * n,KviGnutellaTransferPushRequest *r)
{
	//debug("IN send %d",GNUTELLA_DESCRIPTOR_SIZE + 26);
	unsigned char * buffer = n->allocOutgoingBuffer(GNUTELLA_DESCRIPTOR_SIZE + 26);
	// FIXME: The ttl of the push should be equal to the hops of the query hit that generated it!
	buildDescriptor(buffer,GNUTELLA_PUSH,m_uDefaultTtl,26,0);

	kvi_memmove(buffer + GNUTELLA_DESCRIPTOR_SIZE,r->serventId,16);
	*((Q_UINT32 *)(buffer + GNUTELLA_DESCRIPTOR_SIZE + 16)) = hToGnutellaL(r->uFileIndex);
	*((Q_UINT32 *)(buffer + GNUTELLA_DESCRIPTOR_SIZE + 20)) = hToGnutellaReversedL(g_uGnutellaLocalIpAddress);
	*((Q_UINT16 *)(buffer + GNUTELLA_DESCRIPTOR_SIZE + 24)) = hToGnutellaS(r->uPort);
}

void KviGnutellaThread::processHandshakingIncoming(KviGnutellaNode *n)
{
	// We're expecting a GNUTELLA CONNECT/0.4\n\n
	// or GNUTELLA CONNECT/0.6\r\n{HEADER\r\n}\r\n
	switch(n->protocol())
	{
		case KviGnutellaNode::Unknown:

		if(n->incomingDataLength() >= 22)
		{
			if(kvi_strEqualCSN((char *)n->incomingData(),"GNUTELLA CONNECT/0.4\n\n",22))
			{
				// Ok...connected
				n->eatIncomingData(22);
				n->youAreConnected(KviGnutellaNode::Gnutella04);
				deferredPostEvent(buildThreadEvent(KVI_GNUTELLA_THREAD_EVENT_NODE_CONNECTED_V4,n->id()));
				// append the reply
				n->appendOutgoingData((unsigned char *)"GNUTELLA OK\n\n",13);
				sendPing(n);
			} else if(kvi_strEqualCSN((char *)n->incomingData(),"GNUTELLA CONNECT/0.6\r\n",22))
			{
				int iDataLen = n->incomingDataLength();
				if(iDataLen > 1024)
				{
					// this is a DOS!
					forceKillNode(n,__tr_no_lookup("Handshake failed: response header longer than 1024 bytes"));
					return;
				}
				// might be valid
				KviStr theData((char *)n->incomingData(),iDataLen);
				int idx = theData.findFirstIdx("\r\n\r\n");
				if(idx < 0)
				{
					// headers not finished... wait
					return;
				}

				n->eatIncomingData(idx + 4);
				theData.cutToFirst("\r\n");
				theData.cutFromFirst("\r\n\r\n");
				n->appendPublicHeaders(theData.ptr());
				n->setProtocol(KviGnutellaNode::Gnutella06);
				n->appendOutgoingData((unsigned char *)"GNUTELLA/0.6 200 OK\r\n",21);

				KviStr uagent(KviStr::Format,"User-Agent: %s\r\n\r\n",KVI_GNUTELLA_SERVER_NAME);
				n->appendOutgoingData((unsigned char *)(uagent.ptr()),uagent.len());

			} else {
				// Hanshake failed
				KviStr tmp;
				if(kvi_strEqualCSN((char *)n->incomingData(),"GET ",4))
				{
					// broken client: it shouldn't be connecting here
					tmp.sprintf("HTTP 503 Service Unavaiable\r\n" \
								"Warning: This server listens for http transfer connections on another port: your client is broken, send a bug report to the client author.\r\n" \
								"Server: %s\r\n\r\n",KVI_GNUTELLA_SERVER_NAME);
					n->appendOutgoingData((unsigned char *)tmp.ptr(),tmp.len());
					n->flushQueue(); // we want this user to know it!
				}
				KviStr data((char *)n->incomingData(),22);
				data.append("...");
				tmp.sprintf(__tr_no_lookup("Handshake failed: unexpected handshake action \"%s\""),data.ptr());
				forceKillNode(n,tmp.ptr());
			}
		}
		break;
		case KviGnutellaNode::Gnutella06:
			// This is the second step of the Gnutella06 proto
		if(n->incomingDataLength() > 21)
		{
			if(kvi_strEqualCSN((char *)n->incomingData(),"GNUTELLA/0.6 200 OK\r\n",21))
			{
				int iDataLen = n->incomingDataLength();
				if(iDataLen > 1024)
				{
					// this is a DOS!
					forceKillNode(n,__tr_no_lookup("Handshake failed: response header longer than 1024 bytes"));
					return;
				}
				// might be valid
				KviStr theData((char *)n->incomingData(),iDataLen);
				int idx = theData.findFirstIdx("\r\n\r\n");
				if(idx == -1)
				{
					// headers not finished... wait
					return;
				}
				theData.cutFromFirst("\r\n\r\n");
				//debug("This is 200 oK :)) 0.6");
				n->eatIncomingData(idx + 4);
				theData.cutToFirst("\r\n"); // kill the first part of the header
				n->youAreConnected(KviGnutellaNode::Gnutella06,theData.ptr());
				deferredPostEvent(buildThreadEvent(KVI_GNUTELLA_THREAD_EVENT_NODE_CONNECTED_V6,n->id()));
				sendPing(n);
				return;
			}
			// nope...invalid!
			// Kill it: handshake failed
			KviStr data((char *)n->incomingData(),19);
			data.append("...");
			data.replaceAll("\r\n","\n--- ");
			KviStr tmp(KviStr::Format,__tr_no_lookup("Handshake failed: unexpected response (proto 6 step 2):\n%s"),data.ptr());
			forceKillNode(n,tmp.ptr());
		}
		break;
		default:
		// huh ?
		forceKillNode(n,__tr_no_lookup("Internal error: inconsistent handshaking node state"));
		break;
	}
}

void KviGnutellaThread::processHandshakingOutgoing(KviGnutellaNode *n)
{
	// We're expecting a GNUTELLA OK\n\n
	// or a GNUTELLA OK/VERSION\r\n{HEADERS\r\n}\r\n

	int iDataLen = n->incomingDataLength();

	if(iDataLen >= 13)
	{
		if(kvi_strEqualCSN((char *)n->incomingData(),"GNUTELLA OK\n\n",13))
		{
			// Ok..connected (for any protocol we have attempted , this is a valid response
			// the remote servent supports only Gnutella04
			n->eatIncomingData(13);
			n->youAreConnected(KviGnutellaNode::Gnutella04);
			deferredPostEvent(buildThreadEvent(KVI_GNUTELLA_THREAD_EVENT_NODE_CONNECTED_V4,n->id()));
			sendPing(n);
			return;
		} else {
			if(n->protocol() != KviGnutellaNode::Gnutella04)
			{
				// we have another set of valid responses:
				//debug("GOT RESPONSE %s",(char *)n->incomingData());
				if(kvi_strEqualCSN((char *)n->incomingData(),"GNUTELLA/",9))
				{
					if(iDataLen > 1024)
					{
						// this is a DOS!
						forceKillNode(n,__tr_no_lookup("Handshake failed: response header longer than 1024 bytes"));
						return;
					}
					// might be valid
					KviStr theData((char *)n->incomingData(),iDataLen);
					int idx = theData.findFirstIdx("\r\n\r\n");
					if(idx == -1)
					{
						// headers not finished... wait
						return;
					}
					theData.cutFromFirst("\r\n\r\n");
					KviStr base = theData;
					base.cutFromFirst("\r\n");
					base.stripWhiteSpace();
					// we expect either GNUTELLA/0.4 200 OK
					// or GNUTELLA/0.6 200 OK
					if(kvi_strEqualCS(base.ptr(),"GNUTELLA/0.4 200 OK"))
					{
						n->eatIncomingData(idx + 4);
						n->youAreConnected(KviGnutellaNode::Gnutella04);
						deferredPostEvent(buildThreadEvent(KVI_GNUTELLA_THREAD_EVENT_NODE_CONNECTED_V4,n->id()));
						sendPing(n);
						return;
					}
					if(kvi_strEqualCS(base.ptr(),"GNUTELLA/0.6 200 OK"))
					{
						//debug("This is 200 oK :)) 0.6");
						n->eatIncomingData(idx + 4);
						theData.cutToFirst("\r\n"); // kill the first part of the header
						n->youAreConnected(KviGnutellaNode::Gnutella06,theData.ptr());
						n->appendOutgoingData((const unsigned char *)"GNUTELLA/0.6 200 OK\r\n\r\n",23);
						deferredPostEvent(buildThreadEvent(KVI_GNUTELLA_THREAD_EVENT_NODE_CONNECTED_V6,n->id()));
						sendPing(n);
						return;
					}
					// Kill it: handshake failed
					theData.replaceAll("\r\n","\n-- ");
					KviStr tmp(KviStr::Format,__tr_no_lookup("Handshake failed: invalid response:\n%s"),theData.ptr());
					forceKillNode(n,tmp.ptr());

					if(m_bTryGnutella04After06Failure)
					{
						// Try Gnutella04
						connectToNode(n->ip(),n->port(),KviGnutellaNode::Gnutella04);
					}
					return;
				}
				// nope...invalid!
			}
			// Kill it: handshake failed
			KviStr data((char *)n->incomingData(),13);
			data.append("...");
			data.replaceAll("\r\n","\n--- ");
			KviStr tmp(KviStr::Format,__tr_no_lookup("Handshake failed: unexpected response:\n%s"),data.ptr());
			forceKillNode(n,tmp.ptr());
		}
	}
}

void KviGnutellaThread::processHandshaking(KviPtrList<KviGnutellaNode> * pHandshakingNodes)
{
	// handle handshaking nodes

	for(KviGnutellaNode * n = pHandshakingNodes->first();n;n = pHandshakingNodes->next())
	{
		if(n->connectionType() == KviGnutellaNode::Incoming)processHandshakingIncoming(n);
		else processHandshakingOutgoing(n);
		
	}
}

void KviGnutellaThread::buildDescriptor(unsigned char * buffer,Q_UINT8 uPayloadDescriptor,Q_UINT8 uTtl,
		Q_UINT32 uPayloadLength,unsigned char * descriptorId)
{
	if(descriptorId)
	{
		kvi_memmove(buffer,descriptorId,16);
	} else {
		kvi_memmove(buffer,m_nextDescriptorIdentifier,16);

		// incrementing the first Q_UINT32 helps in distributing the entries in the own-packets caches
		*((Q_UINT32 *)(&m_nextDescriptorIdentifier)) += 1;
	}

	*((Q_UINT8 *)(buffer + 16))  = uPayloadDescriptor;
	*((Q_UINT8 *)(buffer + 17))  = uTtl;
	*((Q_UINT8 *)(buffer + 18))  = 0;
	*((Q_UINT32 *)(buffer + 19)) = hToGnutellaL(uPayloadLength);
}

int KviGnutellaThread::routeToAllButOne(unsigned char * data,int len,KviGnutellaNode * pExcludeNode)
{
	int nRet = 0;
	for(KviGnutellaNode * n = m_pConnectedNodes->first();n;n = m_pConnectedNodes->next())
	{
		if(n != pExcludeNode)
		{
			if(n->state() == KviGnutellaNode::Connected)
			{
				nRet++;
				n->appendOutgoingData(data,len);
			}
		}
	}
	return nRet;
}

int KviGnutellaThread::sendToAll(unsigned char * data,int len)
{
	int nRet = 0;
	for(KviGnutellaNode * n = m_pConnectedNodes->first();n;n = m_pConnectedNodes->next())
	{
		if(n->state() == KviGnutellaNode::Connected)
		{
			n->appendOutgoingData(data,len);
			nRet++;
		}
	}
	return nRet;
}



bool KviGnutellaThread::processPing(KviGnutellaDescriptor * d)
{
	if(d->uPayloadLength != 0)
	{
		KviStr tmp(KviStr::Format,__tr_no_lookup("Invalid payload length %u for PING descriptor (should be 0)"),
						d->uPayloadLength);
		forceKillNode(d->pSourceNode,tmp.ptr());
		return false;
	}

	d->pSourceNode->m_stats.uTotalPings++;

	// we never change the last Q_UINT32 of m_nextDescriptorId
	if(*((Q_UINT32 *)(d->pData + 12)) == *((Q_UINT32 *)(m_nextDescriptorIdentifier + 12)))
	{
		KviGnutellaCachedDescriptor * cd = m_pOwnPingCache->find(d->pData);
		if(cd)
		{
			d->pSourceNode->m_stats.uTotalPingsLoopedBack++;
			// evil! evil!
			if(d->uHops < 2)
			{
				forceKillNode(d->pSourceNode,__tr_no_lookup("Evil: loops back my own PING packets with hops < 2"));
				return false;
			} // still evil...but not critical: there is a near loop in the network
			return true;
		}
	}

	unsigned char * buffer = d->pSourceNode->allocOutgoingBuffer(GNUTELLA_DESCRIPTOR_SIZE + 14);
	buildDescriptor(buffer,GNUTELLA_PONG,d->uHops,14,d->pData);
	*((Q_UINT16 *)(buffer + 23)) = hToGnutellaS(m_uNetworkListenPort);
	*((Q_UINT32 *)(buffer + 25)) = hToGnutellaReversedL(g_uGnutellaLocalIpAddress);
	// Read only access to this...should be ok
	*((Q_UINT32 *)(buffer + 29)) = hToGnutellaL(g_uGnutellaSharedFilesCount);
	*((Q_UINT32 *)(buffer + 33)) = hToGnutellaL(g_uGnutellaSharedBytes / 1024);

	// k...ping replied

	// now eventually route it

	if(d->uTtl > 0)
	{
		if(!m_pRoutedPingCache->find(d->pData))
		{
//			d->pSourceNode->m_stats.uTotalPingsRouted++;

			if(routeToAllButOne(d->pData,GNUTELLA_DESCRIPTOR_SIZE,d->pSourceNode) > 0)
			{
				m_pRoutedPingCache->cache(d->pData,d->pSourceNode);
			}
		} else {
			// already routed
			d->pSourceNode->m_stats.uTotalPingsDuplicated++;
		}
	} else {
		// ttl expired
		d->pSourceNode->m_stats.uTotalPingsTtlExpired++;
	}

	return true;

}

bool gnutella_is_routable_ip(unsigned char * ip)
{

//#warning "This function should also check if we are not in the same network class!"
//#warning "192.168.0.2 might be routable if we are 192.168.0.1...mmmh!"

	if(ip[0] == 0)return false;    // old-style broadcast
	if(ip[0] == 10)return false;   // Class A VPN
	if(ip[0] == 127)return false;   // loopback
	if((ip[0] == 172) && (ip[1] >= 16) && (ip[1] <= 31))return false; // Class B VPN
	if((ip[0] == 192) && (ip[1] == 168))return false; // Class C VPN
	if((ip[0] == 169) && (ip[1] == 254))return false; // APIPA
	if((ip[0] == 192) && (ip[1] == 0) && (ip[2] == 2))return false; // Class B VPN
	if(ip[0] >= 224)return false; // class D multicast and class E reserved

	return true;
}

bool KviGnutellaThread::processPong(KviGnutellaDescriptor * d)
{
	//debug("This is pong");
	if(d->uPayloadLength != 14)
	{
		KviStr tmp(KviStr::Format,__tr_no_lookup("Invalid payload length %u for PONG descriptor (should be 14)"),
						d->uPayloadLength);
		forceKillNode(d->pSourceNode,tmp.ptr());
		return false;
	}

	d->pSourceNode->m_stats.uTotalPongs++;


	struct in_addr ia;
	ia.s_addr = htonl(gnutellaReversedToHL(*((Q_UINT32 *)(d->pData + 25))));

	KviStr szIp;

	if(kvi_binaryIpToStringIp(ia,szIp))
	{
		//debug("This is valid pong with ip %s",szIp.ptr());
		// Ignore unroutable addresses
		if(gnutella_is_routable_ip((unsigned char *)&ia.s_addr))
		{
			unsigned short int uPort = gnutellaToHS(*((Q_UINT16 *)(d->pData + 23)));
			if(!findNonDeadNode(szIp.ptr(),uPort))
			{
//#warning "Avoid duplicated entries"
/*
				if(!m_pNodesCaughtEvent)
				{
					m_pNodesCaughtEvent = new KviGnutellaNodesCaughtThreadEvent();
					deferredPostEvent(m_pNodesCaughtEvent);
				}

				KviGnutellaNodeInfo * inf = new KviGnutellaNodeInfo;
				inf->szIp = szIp;
				inf->szPort.setNum(uPort);
				inf->szHops.setNum(d->uHops);
*/
				cacheHost(szIp,uPort,d->uHops);
/*
				m_pNodesCaughtEvent->m_pNodeList->append(inf);
*/
			} //else {
				//debug("Node %s : %u",szIp.ptr(),uPort);
			//}// else already connected to this node: don't notify
		}
	}

	// we never change the last Q_UINT32 of m_nextDescriptorId
	if(*((Q_UINT32 *)(d->pData + 12)) == *((Q_UINT32 *)(m_nextDescriptorIdentifier + 12)))
	{
		// this might be our ping
		KviGnutellaCachedDescriptor * cd = m_pOwnPingCache->find(d->pData);
		if(cd)
		{
			// Our ping...we were expecting this one
			d->pSourceNode->m_stats.uTotalPongsDirectedToMe++;

			return true; // no need to route it
		}
	}	
	// Need to route it ?

	if(d->uTtl > 0)
	{
		KviGnutellaCachedDescriptor * cd = m_pRoutedPingCache->find(d->pData);
		if(cd)
		{
			// Ok...got the return path

//			d->pSourceNode->m_stats.uTotalPongsRouted++;
			if(cd->pSourceNode->isConnected())
			{
				cd->pSourceNode->appendOutgoingData(d->pData,GNUTELLA_DESCRIPTOR_SIZE + 14);
			}
		}  else {
			// no return path: unroutable
			d->pSourceNode->m_stats.uTotalPongsUnroutable++;
		}
	} else {
		d->pSourceNode->m_stats.uTotalPongsTtlExpired++;
	}

	return true;
}

bool KviGnutellaThread::processQuery(KviGnutellaDescriptor * d)
{
	if(d->uPayloadLength < 2)
	{
		KviStr tmp(KviStr::Format,__tr_no_lookup("Invalid payload length %u for QUERY descriptor (should be >= 2)"),
						d->uPayloadLength);
		forceKillNode(d->pSourceNode,tmp.ptr());
		return false;
	}

	d->pSourceNode->m_stats.uTotalQueries++;

	// we never change the last Q_UINT32 of m_nextDescriptorId
	if(*((Q_UINT32 *)(d->pData + 12)) == *((Q_UINT32 *)(m_nextDescriptorIdentifier + 12)))
	{
		KviGnutellaCachedDescriptor * cd = m_pOwnQueryCache->find(d->pData);
		if(cd)
		{
			d->pSourceNode->m_stats.uTotalQueriesLoopedBack++;
			// evil! evil!
			if(d->uHops < 2)
			{
				forceKillNode(d->pSourceNode,__tr_no_lookup("Evil: loops back my own QUERY packets with hops < 2"));
				return false;
			} // else still evil...but not critical: there is a near loop in the network
			return true;
		}
	}

	if(!m_pRoutedQueryCache->find(d->pData))
	{
		// non duplicated query packet
		if((m_uConnectionSpeed >= *((Q_UINT16 *)(d->pPayload))) && (d->uPayloadLength > 3) && (m_transferListeningSocket != KVI_INVALID_SOCKET))
		{
			g_pGnutellaSearchQueryMutex->lock();
		
			if(g_pGnutellaSearchQueryList->count() < KVI_GNUTELLA_MAX_PENDING_SEARCH_QUERIES)
			{
				KviGnutellaSearchQuery * q = new KviGnutellaSearchQuery;
				q->uSourceNodeId = d->pSourceNode->id();
				q->szQueryString = (char *)(d->pPayload + 2);
				//q->szQueryString.stripWhiteSpace();
				kvi_memmove((void *)q->descriptorId,(void *)d->pData,16);
				g_pGnutellaSearchQueryList->append(q);
			}
	
			g_pGnutellaSearchQueryMutex->unlock();
		}

		if(d->uTtl > 0)
		{
//			d->pSourceNode->m_stats.uTotalQueriesRouted++;
			if(routeToAllButOne(d->pData,GNUTELLA_DESCRIPTOR_SIZE + d->uPayloadLength,d->pSourceNode) > 0)
			{
				// At least one host will receive it
				m_pRoutedQueryCache->cache(d->pData,d->pSourceNode);
			}
		} else {
			d->pSourceNode->m_stats.uTotalQueriesTtlExpired++;
		}
	} else {
		// already routed (non sense to reply or route again)
		d->pSourceNode->m_stats.uTotalQueriesDuplicated++;

	}

	return true;
}

bool KviGnutellaThread::processQueryHit(KviGnutellaDescriptor * d)
{
	if(d->uPayloadLength < 27)
	{
		KviStr tmp(KviStr::Format,__tr_no_lookup("Invalid payload length %u for QUERYHIT descriptor (should be >= 27)"),
						d->uPayloadLength);
		forceKillNode(d->pSourceNode,tmp.ptr());
		return false;
	}

	d->pSourceNode->m_stats.uTotalQueryHits++;

	// First of all , check if it is our query hit

	// we never change the last Q_UINT32 of m_nextDescriptorId
	if(*((Q_UINT32 *)(d->pData + 12)) == *((Q_UINT32 *)(m_nextDescriptorIdentifier + 12)))
	{
		// might be our QUERY
		KviGnutellaCachedDescriptor * cd = m_pOwnQueryCache->find(d->pData);
		if(cd)
		{
			// Ok..query hit
			d->pSourceNode->m_stats.uTotalQueryHitsDirectedToMe++;

			unsigned char nHits      = *(d->pPayload);
	
			if(nHits == 0)return true; // senseless
	
			// Check the packet size now...each hit MUST be at least 10 bytes long
			unsigned int totalPayload = 11 + nHits * 10;
			if(d->uPayloadLength < totalPayload)
			{
				KviStr tmp(KviStr::Format,
						__tr_no_lookup("Invalid payload length %u for QUERYHIT descriptor (should be >= %u)"),
						d->uPayloadLength,totalPayload);
				forceKillNode(d->pSourceNode,tmp.ptr());
				return false;
			}
			// ok..the packet looks to be sane (at least its size)
	
			KviGnutellaHitThreadEvent * e = new KviGnutellaHitThreadEvent();
			e->m_uPort = gnutellaToHS(*((Q_UINT16 *)(d->pPayload + 1)));
	
			struct in_addr ia;
			ia.s_addr = gnutellaToHL(*((Q_UINT32 *)(d->pPayload + 3)));
	
			if(!kvi_binaryIpToStringIp(ia,e->m_szIp))
			{
				// Invalid IP address ? Huh ?
				delete e;
				return true;
			}
	
			e->m_uSpeed = gnutellaToHL(*((Q_UINT32 *)(d->pPayload + 7)));
			e->m_iNodeId = d->pSourceNode->id();
			kvi_memmove(e->m_servId,d->pPayload + d->uPayloadLength - 16,16);
			// now process the result set
	
			// put a 0 in the first byte of the serverId , so we are sure that we will not overrun
			// the packet bound when processing broken (rather malicious) packets
			*(d->pPayload + d->uPayloadLength - 16) = 0;
	
			unsigned char * pNextHit = d->pPayload + 11;
//#warning "We should check the packet bounds!!!"
//			unsigned char * pEndPtr = (d->pPayload + (d->uPayloadLength - (16 + 10)));


			for(unsigned int i=0;i<nHits;i++)
			{
				KviGnutellaHit * h = new KviGnutellaHit;
				h->uIndex = gnutellaToHL(*((Q_UINT32 *)pNextHit));
				pNextHit += 4;
				h->uSize = gnutellaToHL(*((Q_UINT32 *)pNextHit));
				pNextHit += 4;
				h->szName = (char *)pNextHit;
				pNextHit += h->szName.len() + 1;
				if(*pNextHit != '\0')
				{
					// Extra info in the packet
					h->szExtraInfo = (char *)pNextHit;
					pNextHit += h->szExtraInfo.len() + 1;
				} else {
					// Normal packet
					pNextHit++;
				}
				e->m_pHitList->append(h);
			}

			unsigned char * pEndPtr = (d->pPayload + d->uPayloadLength - 16);

			if(pNextHit + 5 <= pEndPtr)
			{
				//Extended QueryHit Descriptor 
				//A valid EQHD must contain at least 5 bytes of data: 
				// 0-3 : Vendor code (ASCII)
				// 4   : Open data size
				// N   : Open data
				// M   : Vendor data

				// remember this
				unsigned char odSize = *(pNextHit + 4);
				*(pNextHit + 4) = 0;
				KviStr szVendor = (char *)pNextHit;
				for(KviGnutellaHit * gh = e->m_pHitList->first();gh;gh = e->m_pHitList->next())
				{
					if(gh->szExtraInfo.hasData())gh->szExtraInfo.append(" ");
					gh->szExtraInfo.append(KviStr::Format,"[Vendor: \"%s\" , OpenData: %u bytes]",szVendor.ptr(),odSize);
				}

				// Vendor codes:
				//   BEAR BearShare 
				//   LIME LimeWire 
				//   CULT Cultiv8r 
				//   GNOT Gnotella 
				//   GNUC Gnucleus 
				//   GNUT gnut 
				//   GTKG Gtk-Gnutella 
				//   HSLG Hagelslag 
				//   MACT Mactella 
				//   NAPS NapShare 
				//   OCFG OpenCola 
				//   TOAD ToadNode 
				// The OpenData usually contains two bytes: 
				//           7 6 5 ...
				//       1st byte: r r r validUploadSpeed  validHaveUploaded  validBusy  r flagPush 
				//       2nd byte: r r r flagUploadSpeed flagHaveUploaded flagBusy r validPush 
				//       r = reserved for future use. 
				//       flagUploadSpeed = 1 if and only if the Speed field of this QueryHit descriptor contains the highest average transfer rate (in kbps) of the last 10 uploads. 
				//       validUploadSpeed = 1 if and only if the flagUploadSpeed bit is meaningful. 
				//       flagHaveUploaded = 1 if and only if the servent has successfully uploaded at least one file. 
				//       validHaveUploaded = 1 if and only if the flagHaveUploaded bit is meaningful. 
				//       flagBusy = 1 if and only if all of the servent's upload slots are full (at the time this QueryHit was generated) 
				//       validBusy = 1 if and only if the flagBusy bit is meaningful. 
				//       flagPush = 1 if and only if the servent is firewalled or has not yet accepted an incoming connection. 
				//       validPush = 1 if and only if the flagPush bit is meaningful. 
			}
			deferredPostEvent(e);
	
			return true; // no need to route it
		}
	}

	// Ok...route it

	if(d->uTtl > 0)
	{
		// ttl ok
		KviGnutellaCachedDescriptor * cd = m_pRoutedQueryCache->find(d->pData);
		if(cd)
		{
			// Ok...got the return path
//			d->pSourceNode->m_stats.uTotalQueryHitsRouted++;
			if(cd->pSourceNode->isConnected())
			{
				// ok..don't reroute again this packet
				// and remember the servent for the eventual push
				m_pQueryHitServentsCache->cache(d->pPayload + d->uPayloadLength - 16,d->pSourceNode);

				cd->pSourceNode->appendOutgoingData(d->pData,GNUTELLA_DESCRIPTOR_SIZE + d->uPayloadLength);
			}
		} else {
			// no return path: unroutable
			d->pSourceNode->m_stats.uTotalQueryHitsUnroutable++;
		}
	} else {
		// ttl expired
		d->pSourceNode->m_stats.uTotalQueryHitsTtlExpired++;
	}

	return true;
}



bool KviGnutellaThread::processPush(KviGnutellaDescriptor * d)
{
	if(d->uPayloadLength < 26)
	{
		KviStr tmp(KviStr::Format,__tr_no_lookup("Invalid payload length %u for PUSH descriptor (should be 26)"),
						d->uPayloadLength);
		forceKillNode(d->pSourceNode,tmp.ptr());
		return false;
	}

	d->pSourceNode->m_stats.uTotalPushes++;

	if(gnutella_compare_descriptor(d->pPayload,m_serventIdentifier))
	{
		// directed to me!
		d->pSourceNode->m_stats.uTotalPushesDirectedToMe++;

		// ok..first the file index
		unsigned int uFileIndex = gnutellaToHL(*((Q_UINT32 *)(d->pPayload + 16)));

		KviStr szFileName;
		unsigned int uFileSize;
		KviStr szPath;
		g_pGnutellaSharedFilesMutex->lock();

		KviGnutellaSharedFile * f;
		for(f = g_pGnutellaSharedFiles->first();f;f = g_pGnutellaSharedFiles->next())
		{
			if(f->uId == uFileIndex)break;
		}
		if(f)
		{
			// got it
			szFileName = f->szFileName;
			uFileSize  = f->uFileSize;
			szPath     = f->szPath;
		}
		g_pGnutellaSharedFilesMutex->unlock();

		struct in_addr ia;
		ia.s_addr = htonl(gnutellaReversedToHL(*((Q_UINT32 *)(d->pPayload + 20))));

		unsigned short int uPort = gnutellaToHS(*((Q_UINT16 *)(d->pPayload + 24)));

		KviStr szIp;

		if(kvi_binaryIpToStringIp(ia,szIp) && f && szFileName.hasData() && uFileSize)
		{
			// ok..theoretically we could serve the file.
			// FIXME: if there are too many running transfers... reject the push request!!!
			//KviStr tmp(KviStr::Format,__tr_no_lookup("Rejecting valid PUSH request from %s for file index %u: too many outgoing transfers"),
			//				szIp.ptr(),uFileIndex);

			KviThreadDataEvent<KviGnutellaPushRequestInfo> * ev = new KviThreadDataEvent<KviGnutellaPushRequestInfo>(KVI_GNUTELLA_THREAD_EVENT_PUSH_REQUEST);
			KviGnutellaPushRequestInfo * inf = new KviGnutellaPushRequestInfo;
			inf->szFileName = szFileName;
			inf->szFilePath = szPath;
			inf->uFileSize = uFileSize;
			inf->uFileIndex = uFileIndex;
			inf->uPort = uPort;
			inf->szIp  = szIp;
//			inf->uAddress = ntohl(ia.s_addr);
			kvi_memmove(inf->localServentId,m_serventIdentifier,16);

			ev->setData(inf);

			deferredPostEvent(ev);
		}

		return true;
	}


	if(d->uTtl > 0)
	{
		// ttl ok
		KviGnutellaCachedDescriptor * cd = m_pQueryHitServentsCache->find(d->pPayload);
		if(cd)
		{
			// Ok...got the return path
			if(!m_pRoutedPushCache->find(d->pData))
			{
//				d->pSourceNode->m_stats.uTotalPushesRouted++;
				if(cd->pSourceNode->isConnected())
				{
					// ok..don't reroute again this packet
					m_pRoutedPushCache->cache(d->pData,d->pSourceNode);
					cd->pSourceNode->appendOutgoingData(d->pData,GNUTELLA_DESCRIPTOR_SIZE + d->uPayloadLength);
				}
			} else {
				// already routed
				d->pSourceNode->m_stats.uTotalPushesDuplicated++;
			}
		} else {
			// no return path: unroutable
			d->pSourceNode->m_stats.uTotalPushesUnroutable++;
		}
	} else {
		// ttl expired
		d->pSourceNode->m_stats.uTotalPushesTtlExpired++;
	}

	return true;
}

void KviGnutellaThread::processConnected(KviPtrList<KviGnutellaNode> * pConnectedNodes)
{
	// connected nodes
	KviGnutellaNode * n = pConnectedNodes->first();
	while(n)
	{
		KviGnutellaDescriptor d;

		while((n->incomingDataLength() >= 23) && (!(n->isDead())))
		{
			//debug("There is data to read: %d (%s)",n->incomingDataLength(),(char *)n->incomingData());
			// Has at least one descriptor header
			d.uPayloadLength = *((Q_UINT32 *)(n->incomingData() + 19));
			d.uPayloadLength = gnutellaToHL(d.uPayloadLength);

			//debug("Payload length is: %u",d.uPayloadLength);

			if(n->incomingDataLength() >= ((int)(23 + d.uPayloadLength)))
			{
				// Has a full descriptor
//#warning "should also check the payload length"

				// Decrement TTL
				// ABSURD gcc's neurons took a day off today (19/04/2001):
				// I can't use
				// *((Q_UINT8 *)((unsigned char *)n->incomingData() + 17))--
				// but I can use the following
				if(*((Q_UINT8 *)(n->incomingData() + 17)) > 0)
				{
					*((Q_UINT8 *)(n->incomingData() + 17)) -= 1;
				}
				// Increment HOPS
				*((Q_UINT8 *)(n->incomingData() + 18)) += 1;

				d.pSourceNode        = n;
				d.pData              = n->incomingData();
				d.uPayloadDescriptor = *(n->incomingData() + 16);
				d.uTtl               = *(n->incomingData() + 17);
				d.uHops              = *(n->incomingData() + 18);
				d.pPayload           = d.uPayloadLength ? n->incomingData() + 23 : 0;

				switch(d.uPayloadDescriptor)
				{
					case GNUTELLA_PING:
						if(!processPing(&d))goto next_node;
					break;
					case GNUTELLA_PONG:
						if(!processPong(&d))goto next_node;
					break;
					case GNUTELLA_PUSH:
						if(!processPush(&d))goto next_node;
					break;
					case GNUTELLA_QUERY:
						if(!processQuery(&d))goto next_node;
					break;
					case GNUTELLA_QUERYHIT:
						if(!processQueryHit(&d))goto next_node;
					break;
					default:
						n->m_stats.uTotalUnknownPackets++;
						//debug("UNKNOWN PACKET %u",d.uPayloadDescriptor);
					break;
				}
				n->eatIncomingData(23 + d.uPayloadLength);
			} else break;
		}
next_node:
		n = pConnectedNodes->next();
	}
}

void KviGnutellaThread::processIncomingData()
{
	// Copy list
	KviPtrList<KviGnutellaNode> cl;
	KviPtrList<KviGnutellaNode> hl;

	cl.setAutoDelete(false);
	hl.setAutoDelete(false);

	for(KviGnutellaNode * n = m_pConnectedNodes->first();n;n = m_pConnectedNodes->next())
	{
		if(n->state() == KviGnutellaNode::Connected)
		{
			if(n->incomingDataLength() > 0)cl.append(n);
		}
		else if(n->state() == KviGnutellaNode::Handshake)
		{
			if(n->msecondsSinceOperationStart(&m_currentTime) > (int)m_uHandshakeTimeoutInMSecs)
			{
				n->die(__tr_no_lookup("Hanshake timeout"));
			} else {
				if(n->incomingDataLength() > 0)hl.append(n);
			}
		}
	}

	processHandshaking(&hl);

	processConnected(&cl);

}
