//
//   File : gnutellanode.cpp
//   Creation date : Tue Apr 17 2001 21:18:43 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2001 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the gnutellas of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "gnutellanode.h"

#include "kvi_malloc.h"
#include "kvi_memmove.h"
#include "kvi_error.h"
#include "kvi_locale.h"
#include "kvi_netutils.h"
#include "kvi_time.h"

#include <stdlib.h>


#define _GNUTELLADEBUG

static int g_iNextGnutellaNodeId = 0;

KviGnutellaNode::KviGnutellaNode(const char * szIp,unsigned short int uPort,ConnectionType t,Protocol p)
{
	m_szIp = szIp;
	m_uPort = uPort;

	m_connectionType = t;

	m_sock = KVI_INVALID_SOCKET;
	m_incomingDataBuffer = 0;
	m_iIncomingDataLen = 0;
	m_outgoingDataBuffer = 0;
	m_iOutgoingDataLen = 0;

	m_protocol = p;

	m_iId = g_iNextGnutellaNodeId;
	g_iNextGnutellaNodeId++;


	kvi_memset(&m_stats,0,sizeof(KviGnutellaNodeStats));

	resetBandwidthStats();

	m_state = JustBorn;
}

KviGnutellaNode::~KviGnutellaNode()
{
	if(m_sock != KVI_INVALID_SOCKET)kvi_socket_close(m_sock);
	if(m_incomingDataBuffer)
	{
//		KviStr tmp((char *)m_incomingDataBuffer,m_iIncomingDataLen);
//
//		debug("Dying with data : (%s)",tmp.ptr());
		kvi_free(m_incomingDataBuffer);
		m_incomingDataBuffer = 0;
		m_iIncomingDataLen = 0;
	}
	if(m_outgoingDataBuffer)
	{
		kvi_free(m_outgoingDataBuffer);
		m_outgoingDataBuffer = 0;
		m_iOutgoingDataLen = 0;
	}
}

void KviGnutellaNode::eatIncomingData(int len)
{

	if(len > m_iIncomingDataLen)len = m_iIncomingDataLen;
	int remaining = m_iIncomingDataLen - len;
	if(remaining > 0)
	{
		kvi_memmove(m_incomingDataBuffer,m_incomingDataBuffer + len,remaining);
		m_incomingDataBuffer = (unsigned char *)kvi_realloc(m_incomingDataBuffer,remaining);
		m_iIncomingDataLen = remaining;
	} else {
		kvi_free(m_incomingDataBuffer);
		m_incomingDataBuffer = 0;
		m_iIncomingDataLen = 0;
	}
	m_stats.uTotalPacketsReceived++;
}

bool KviGnutellaNode::readData()
{
	unsigned char buffer[1024];
	int readed = kvi_socket_read(m_sock,buffer,1024);

	if(readed > 0)
	{
		m_incomingDataBuffer = (unsigned char *)kvi_realloc(m_incomingDataBuffer,m_iIncomingDataLen + readed);
		kvi_fastmove(m_incomingDataBuffer + m_iIncomingDataLen,buffer,readed);

//		KviStr tmp((char *)m_incomingDataBuffer,m_iIncomingDataLen + readed);
//
//		debug("Readed : (%s)",tmp.ptr());

		m_iIncomingDataLen += readed;
		m_stats.uTotalBytesReceived += readed;
		return true;
	}
	// Read error ? EOF ?
	if(readed == 0)
	{
		// EOF
		m_szDeathReason = __tr_no_lookup("EOF");
	} else {
		// Read error
		int err = kvi_socket_error();
#ifdef COMPILE_ON_WINDOWS
		if((err == EAGAIN) || (err == EINTR) || (err == WSAEWOULDBLOCK))return true; //temporeneous
#else
		if((err == EAGAIN) || (err == EINTR))return true; //temporeneous
#endif
		m_szDeathReason.sprintf(__tr_no_lookup("Read error: %s"),kvi_getErrorString(kvi_errorFromSystemError(err)));
	}

	kvi_socket_close(m_sock);
	m_sock = KVI_INVALID_SOCKET;
	m_state = Dead;
	return false;
}

bool KviGnutellaNode::connect()
{
	m_sock = kvi_socket_create(KVI_SOCKET_PF_INET,KVI_SOCKET_TYPE_STREAM,KVI_SOCKET_PROTO_TCP); //tcp
	if(m_sock < 0)
	{
		m_state = Dead;
		m_szDeathReason = __tr_no_lookup("Failed to create the socket");
		return false;
	}

	if(!kvi_socket_setNonBlocking(m_sock))
	{
		kvi_socket_close(m_sock);
		m_sock = KVI_INVALID_SOCKET;
		m_state = Dead;
		m_szDeathReason = __tr_no_lookup("Failed to enter non blocking mode");
		return false;
	}

	sockaddr_in saddr;
//	struct in_addr ia;
	if(!kvi_stringIpToBinaryIp(m_szIp.ptr(),&(saddr.sin_addr)))
	{
		kvi_socket_close(m_sock);
		m_sock = KVI_INVALID_SOCKET;
		m_state = Dead;
		m_szDeathReason = __tr_no_lookup("Invalid target address");
		return false;
	}
	saddr.sin_port = htons(m_uPort);
	saddr.sin_family = AF_INET;

	if(!kvi_socket_connect(m_sock,(struct sockaddr *)&saddr,sizeof(saddr)))
	{
		int err = kvi_socket_error();
		if(!kvi_socket_recoverableConnectError(err))
		{
			m_state = Dead;
			kvi_socket_close(m_sock);
			m_sock = KVI_INVALID_SOCKET;
			m_szDeathReason.sprintf(__tr_no_lookup("Failed to connect : errno = %d (%s)"),err,
				kvi_getErrorString(kvi_errorFromSystemError(err)));
			return false;
		}
	}

	m_state = Connecting;
	return true;

}

bool KviGnutellaNode::flushQueue()
{
	if(m_sock == KVI_INVALID_SOCKET)return false;

	if(m_iOutgoingDataLen > 0)
	{
		//debug("Have to flush");
		int written = kvi_socket_send(m_sock,(const void *)m_outgoingDataBuffer,m_iOutgoingDataLen);

		if(written >= 0)
		{
			// Succesfull write
			if(written > 0)
			{
				//debug("Write success");
				//if(written > m_iOutgoingDataLen)written = m_iOutgoingDataLen; //absurd check
				int remaining = m_iOutgoingDataLen - written;
				if(remaining > 0)
				{
					//debug("Remaining data in buffer");
					kvi_memmove(m_outgoingDataBuffer,m_outgoingDataBuffer + written,remaining);
					m_iOutgoingDataLen = remaining;
					m_outgoingDataBuffer = (unsigned char *)kvi_realloc(m_outgoingDataBuffer,m_iOutgoingDataLen);
				} else {
					//debug("Clean send");
					kvi_free(m_outgoingDataBuffer);
					m_outgoingDataBuffer = 0;
					m_iOutgoingDataLen = 0;
				}
				m_stats.uTotalBytesSent += written;
			}
		} else {
			// Opzz.. error ?
			int err = kvi_socket_error();
#ifdef COMPILE_ON_WINDOWS
			if((err != EINTR) && (err != EAGAIN) && (err != WSAEWOULDBLOCK))
#else
			if((err != EINTR) && (err != EAGAIN))
#endif
			{
			
				m_state = Dead;
				kvi_socket_close(m_sock);
				m_sock = KVI_INVALID_SOCKET;
				m_szDeathReason.sprintf(__tr_no_lookup("Write error : errno = %d (%s)"),err,
					kvi_getErrorString(kvi_errorFromSystemError(err)));
				return false;
			}
		}
	}
	return true;
}

void KviGnutellaNode::die(const char * reason)
{
	if(m_sock != KVI_INVALID_SOCKET)
	{
		kvi_socket_close(m_sock);
		m_sock = KVI_INVALID_SOCKET;
	}
	m_state = Dead;
	m_szDeathReason = reason;


}

void KviGnutellaNode::appendOutgoingData(const unsigned char * buffer,int len)
{
	m_outgoingDataBuffer = (unsigned char *)kvi_realloc(m_outgoingDataBuffer,m_iOutgoingDataLen + len);
	kvi_fastmove(m_outgoingDataBuffer + m_iOutgoingDataLen , buffer,len);
	m_iOutgoingDataLen += len;
	m_stats.uTotalPacketsSent++;
}


unsigned char * KviGnutellaNode::allocOutgoingBuffer(int len)
{
	//debug("Allocating outgoing buffer (orig = %d)(add = %d)[ptr=%d]",m_iOutgoingDataLen,len,m_outgoingDataBuffer);
	m_outgoingDataBuffer = (unsigned char *)kvi_realloc(m_outgoingDataBuffer,m_iOutgoingDataLen + len);
	unsigned char * ret = (m_outgoingDataBuffer + m_iOutgoingDataLen);
	//debug("[ptr now=%d],[ret ptr=%d]",m_outgoingDataBuffer,ret);
	m_iOutgoingDataLen += len;
	m_stats.uTotalPacketsSent++;
	return ret;
}

void KviGnutellaNode::startOperation()
{
	kvi_gettimeofday(&m_operationStartTime,0);
}


void KviGnutellaNode::youAreHandshaking()
{
	m_state = Handshake;
	startOperation();
}

void KviGnutellaNode::youAreConnected(KviGnutellaNode::Protocol p,const char * publicHeaders)
{
	m_protocol = p;
	m_state = Connected;
	if(publicHeaders)
	{
		if(m_szPublicHeaders.hasData())m_szPublicHeaders.append("\r\n");
		m_szPublicHeaders.append(publicHeaders);
	}
	startOperation();
}


void KviGnutellaNode::resetBandwidthStats()
{
	m_uLastBandwidthSentBytes = m_stats.uTotalBytesSent;
	m_uLastBandwidthRecvBytes = m_stats.uTotalBytesReceived;
}

void KviGnutellaNode::setConnectedFd(kvi_socket_t fd)
{
	m_sock = fd;
	m_state = Handshake;
	startOperation();
}

int KviGnutellaNode::msecondsSinceOperationStart(struct timeval * now)
{
	return (((now->tv_sec - m_operationStartTime.tv_sec) * 1000) + ((now->tv_usec - m_operationStartTime.tv_usec) / 1000));
}
