//
//   File : gnutelladcache.cpp
//   Creation date : Mon Apr 23 2001 14:18:34 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2001 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the gnutellas of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//


#include "gnutelladcache.h"

#include "kvi_memmove.h"


//#define _GNUTELLADEBUG

KviGnutellaDescriptorCache::KviGnutellaDescriptorCache()
{
	for(int i = 0;i < KVI_GNUTELLA_DESCRIPTOR_CACHE_HASH_ENTRIES;i++)
	{
		m_pList[i] = new KviPtrList<KviGnutellaCachedDescriptor>;
		m_pList[i]->setAutoDelete(true);
	}
}

KviGnutellaDescriptorCache::~KviGnutellaDescriptorCache()
{
	for(int i = 0;i < KVI_GNUTELLA_DESCRIPTOR_CACHE_HASH_ENTRIES;i++)
	{
		delete m_pList[i];
	}
}

static inline int descriptor_cache_hash(unsigned char * descriptorId)
{
	return ((*((Q_UINT32 *)(descriptorId))) & KVI_GNUTELLA_DESCRIPTOR_CACHE_HASH_BITS);
}


void KviGnutellaDescriptorCache::cache(unsigned char * descriptorId,KviGnutellaNode * n)
{
	int index = descriptor_cache_hash(descriptorId);

#ifdef _GNUTELLADEBUG
	debug("cache index = %d",index);
#endif

	// remove some old entries
	while(m_pList[index]->count() >= KVI_GNUTELLA_DESCRIPTOR_CACHE_MAX_ENTRIES_PER_LIST)
			m_pList[index]->removeLast();

	KviGnutellaCachedDescriptor * d = new KviGnutellaCachedDescriptor;
	d->pSourceNode = n;
	kvi_memmove((void *)d->descriptorId,(void *)descriptorId,16);

	m_pList[index]->prepend(d);
}

void KviGnutellaDescriptorCache::removeAllByNode(KviGnutellaNode * n)
{
	for(int i = 0;i < KVI_GNUTELLA_DESCRIPTOR_CACHE_HASH_ENTRIES;i++)
	{
		if(m_pList[i])
		{
			KviPtrList<KviGnutellaCachedDescriptor> l;
			l.setAutoDelete(false);
			KviGnutellaCachedDescriptor * d;
			for(d = m_pList[i]->first();d;d = m_pList[i]->next())
			{
				if(d->pSourceNode == n)l.append(d);
			}
			for(d = l.first();d;d = l.next())
			{
				m_pList[i]->removeRef(d);
			}
		}
	}
}

KviGnutellaCachedDescriptor * KviGnutellaDescriptorCache::find(unsigned char * descriptorId)
{
	// this function finds a descriptor in at max 16 tries
	// the average number of comparisons is 4-5 , since 
	// the newest descriptors are the ones that are most often looked up
	// and they are at the beginning of the lists.
	// The bad thing (tm) is that if the descriptor is NOT there: we always
	// search the entire list before knowing it
	int index = descriptor_cache_hash(descriptorId);

#ifdef _GNUTELLADEBUG
	int iTries = 0;
#endif
	for(KviGnutellaCachedDescriptor * d = m_pList[index]->first();d;d = m_pList[index]->next())
	{
#ifdef _GNUTELLADEBUG
		iTries++;
#endif
		if(gnutella_compare_descriptor(d->descriptorId,descriptorId))
		{
#ifdef _GNUTELLADEBUG
			debug("DESCRIPTOR %u,%u,%u,%u found after %d tries",((Q_UINT32 *)descriptorId)[0],
				((Q_UINT32 *)descriptorId)[1],((Q_UINT32 *)descriptorId)[2],((Q_UINT32 *)descriptorId)[3],iTries);
#endif
			return d;
		}
	}
#ifdef _GNUTELLADEBUG
			debug("DESCRIPTOR %u,%u,%u,%u not found",((Q_UINT32 *)descriptorId)[0],
				((Q_UINT32 *)descriptorId)[1],((Q_UINT32 *)descriptorId)[2],((Q_UINT32 *)descriptorId)[3]);
#endif

	return 0;
}
