//
//   File : libkvifile.cpp
//   Creation date : Fri Nov  9 03:27:59 2001 GMT by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2001 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//


#include "kvi_module.h"
#include "kvi_uparser.h"
#include "kvi_fileutils.h"
#include "kvi_locale.h"
#include "kvi_command.h"
#include "kvi_malloc.h"
#include "kvi_app.h"


#include <qfileinfo.h>
#include <qfile.h>

/*
	@doc: file.copy
	@type:
		command
	@title:
		file.copy
	@keyterms:
		copying files
	@short:
		Makes a copy of a file
	@syntax:
		file.copy [-o] <source> <destination>
	@description:
		Makes a copy of the <source> file as <destination>.[br]
		If the [-o] switch is used , the <destination> file is overwritten , if already exists.[br]
		With no [-o] switch , this command does not overwrite files.[br]
		The destination path must be already existing: if you want to ensure this, use [fnc]$file.mkdir[/fnc] first.[br]
		The paths (<source> and <destination>) are adjusted according to the system that KVIrc
		is running on so you don't have to bother about portability: it *should* be automatically
		guaranteed. Just use UNIX style paths for them.[br]
	@seealso:
		[cmd]file.rename[/cmd], [fnc]$file.exists[/fnc]()
*/

static bool file_module_cmd_copy(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"file_module_cmd_copy");
	KviStr szSrc,szDst;
	if(!g_pUserParser->parseCmdSingleToken(c,szSrc))return false;
	if(!g_pUserParser->parseCmdFinalPart(c,szDst))return false;

	kvi_adjustFilePath(szSrc);
	kvi_adjustFilePath(szDst);

	if(kvi_fileExists(szDst.ptr()) && !c->hasSwitch('o'))
	{
		c->warning(__tr("Destination file exists: no copy made"));
		return c->leaveContext();
	}

	if(!kvi_copyFile(szSrc.ptr(),szDst.ptr()))
		c->warning(__tr("Failed to copy %s to %s"),szSrc.ptr(),szDst.ptr());

	return c->leaveContext();
}


/*
	@doc: file.write
	@type:
		command
	@title:
		file.write
	@keyterms:
		writing data to files
	@short:
		Writes an ascii data string to a file
	@syntax:
		file.write [-a] <filename> <data>
	@description:
		Writes <data> (which is an ASCII string) to the file <filename>.[br]
		It does NOT append a traling LF character: if you want it you must explicitly specify it in the <data> parameter.[br]
		-a causes the command to append the <data> to the file instead of overwriting the entire file.[br]
		The path is adjusted according to the system that KVIrc
		is running on so you don't have to bother about portability: it *should* be automatically
		guaranteed. Just use UNIX style paths for them.[br]
	@seealso:
		[cmd]file.rename[/cmd], [fnc]$file.exists[/fnc]()
*/

static bool file_module_cmd_write(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"file_module_cmd_write");
	KviStr szFile,szData;
	if(!g_pUserParser->parseCmdSingleToken(c,szFile))return false;
	if(!g_pUserParser->parseCmdFinalPart(c,szData))return false;

	kvi_adjustFilePath(szFile);

	if(!kvi_writeFile(szFile.ptr(),szData,c->hasSwitch('a')))
		c->warning(__tr("Failed to write to file %s"),szFile.ptr());

	return c->leaveContext();
}


/*
	@doc: file.rename
	@type:
		command
	@title:
		file.rename
	@keyterms:
		copying files
	@short:
		Makes a copy of a file
	@syntax:
		file.rename <oldname> <newname>
	@description:
		Renames a file from <oldname> to <newname>.[br]
		This command can also rename directories.[br]
		If the <newname> file already exists , this command fails.[br]
		The paths (<source> and <destination>) are adjusted according to the system that KVIrc
		is running on so you don't have to bother about portability: it *should* be automatically
		guaranteed.Just use UNIX style paths for them.[br]
	@seealso:
		[cmd]file.copy[/cmd], [fnc]$file.exists[/fnc]()
*/

static bool file_module_cmd_rename(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"file_module_cmd_rename");
	KviStr szSrc,szDst;
	if(!g_pUserParser->parseCmdSingleToken(c,szSrc))return false;
	if(!g_pUserParser->parseCmdFinalPart(c,szDst))return false;

	kvi_adjustFilePath(szSrc);
	kvi_adjustFilePath(szDst);

	if(kvi_fileExists(szDst.ptr()))
	{
		c->warning(__tr("Destination file exists: file not renamed"));
		return c->leaveContext();
	}

	if(!kvi_renameFile(szSrc.ptr(),szDst.ptr()))
		c->warning(__tr("Failed to rename %s to %s"),szSrc.ptr(),szDst.ptr());

	return c->leaveContext();
}

/*
	@doc: file.mkdir
	@type:
		command
	@title:
		file.mkdir
	@keyterms:
		creating directories
	@short:
		Creates a directory
	@syntax:
		file.mkdir <directory>
	@description:
		Creates the <directory>.[br]
		The path is adjusted according to the system that KVIrc
		is running on so you don't have to bother about portability: it *should* be automatically
		guaranteed. Just use an UNIX style path.[br]
	@seealso:
		[fnc]$file.exists[/fnc]()
*/

static bool file_module_cmd_mkdir(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"file_module_cmd_mkdir");
	KviStr szDst;
	if(!g_pUserParser->parseCmdFinalPart(c,szDst))return false;

	kvi_adjustFilePath(szDst);

	if(!kvi_makeDir(szDst.ptr()))
		c->warning(__tr("Failed to make the directory %s"),szDst.ptr());

	return c->leaveContext();
}


/*
	@doc: file.remove
	@type:
		command
	@title:
		file.remove
	@keyterms:
		removing files
	@short:
		Removes a file
	@syntax:
		file.remove [-q] <name>
	@description:
		Removes the file <name>.[br]
		-q suppresses any warning message (about non existing file , for example).[br]
		The path is adjusted according to the system that KVIrc
		is running on so you don't have to bother about portability: it *should* be automatically
		guaranteed. Just use an UNIX style path.[br]
	@seealso:
		[fnc]$file.exists[/fnc]()
*/

static bool file_module_cmd_remove(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"file_module_cmd_remove");
	KviStr szDst;
	if(!g_pUserParser->parseCmdFinalPart(c,szDst))return false;

	kvi_adjustFilePath(szDst);

	if(!kvi_removeFile(szDst.ptr()))
	{
		if(!c->hasSwitch('q'))c->warning(__tr("Failed to remove the file %s"),szDst.ptr());
	}

	return c->leaveContext();
}


/*
	@doc: file.rmdir
	@type:
		command
	@title:
		file.rmdir
	@keyterms:
		removing directories
	@short:
		Removes a directory
	@syntax:
		file.rmdir [-q] <name>
	@description:
		Removes the directory <name>.[br]
		The directory must be empty for this command to success.[br]
		-q suppresses any warning message (about non existing directory , for example).[br]
		The path is adjusted according to the system that KVIrc
		is running on so you don't have to bother about portability: it *should* be automatically
		guaranteed. Just use an UNIX style path.[br]
	@seealso:
		[fnc]$file.exists[/fnc]()
		[cmd]file.remove[/cmd]
*/

static bool file_module_cmd_rmdir(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"file_module_cmd_rmdir");
	KviStr szDst;
	if(!g_pUserParser->parseCmdFinalPart(c,szDst))return false;

	kvi_adjustFilePath(szDst);

	if(!kvi_removeDir(szDst.ptr()))
	{
		if(!c->hasSwitch('q'))c->warning(__tr("Failed to remove the directory %s"),szDst.ptr());
	}

	return c->leaveContext();
}

/*
	@doc: file.exists
	@type:
		function
	@title:
		$file.exists
	@short:
		Check if a file exists
	@syntax:
		$file.exists(<filename>)
	@description:
		Returns 1 if the file <filename> exists (this is also valid for directories!).[br]
		The <filename> should be an unix-style file path and is adjusted according to the system that KVIrc is running on.[br]
*/

static bool file_module_fnc_exists(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	//ENTER_CONTEXT(c,"file_module_fnc_exists");
	KviStr szFile = parms->safeFirstParam();
	kvi_adjustFilePath(szFile);
	QFileInfo f(szFile.ptr());
	buffer.append(f.exists() ? "1" : "0");
	return true; //c->leaveContext();
}

/*
	@doc: file.type
	@type:
		function
	@title:
		$file.type
	@short:
		Checks the type of a path
	@syntax:
		$file.type(<filename>)
	@description:
		Returns "file" if the <filename> points to a real file , "dir" if <filename>
		is the name of a directory or "link" if it is a symbolic link.[br]
		The <filename> should be an unix-style file path and is adjusted according to the system that KVIrc is running on.[br]
*/

static bool file_module_fnc_type(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	//ENTER_CONTEXT(c,"file_module_fnc_type");
	KviStr szFile = parms->safeFirstParam();
	kvi_adjustFilePath(szFile);
	QFileInfo f(szFile.ptr());
	if(f.isFile())buffer.append("file");
	else if(f.isDir())buffer.append("dir");
	else if(f.isSymLink())buffer.append("link");
	return true; //c->leaveContext();
}


/*
	@doc: file.size
	@type:
		function
	@title:
		$file.size
	@short:
		Returns the size of a file
	@syntax:
		$file.size(<filename>)
	@description:
		Returns the size of the file pointed by <filename>.[br]
		If the file does not exist , this function returns 0.[br]
		The <filename> should be an unix-style file path and is adjusted according to the system that KVIrc is running on.[br]
*/

static bool file_module_fnc_size(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	//ENTER_CONTEXT(c,"file_module_fnc_size");
	KviStr szFile = parms->safeFirstParam();
	kvi_adjustFilePath(szFile);
	QFileInfo f(szFile.ptr());
	buffer.append(KviStr::Format,"%u",f.size());
	return true; //c->leaveContext();
}


/*
	@doc: file.fixpath
	@type:
		function
	@title:
		$file.fixpath
	@short:
		Converts file paths
	@syntax:
		$file.fixpath(<filename>)
	@description:
		Returns the <filename> adjusted to match the current operating
		system file path conventions.[br] This means that on UNIX , a path like "C:\folder\file.mp3"
		will be returned as "/folder/file.mp3" and vice-versa.[br]
		There is a minor problem with unix paths converted to the windows system: unix
		has no "drive" concept thus the unix paths do not contain a drive. KVIrc will
		always map the paths to the "default" C: drive.[br]
		This is a good reason to avoid using absolute hard-coded paths :).[br]
		Please note that you DON'T NEED to call this function on paths that you
		pass to the other file.* functions: they are adjusted automatically.[br]
	@seealso:
		[fnc]$file.ps[/file]
*/

static bool file_module_fnc_fixpath(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	//ENTER_CONTEXT(c,"file_module_fnc_size");
	KviStr szFile = parms->safeFirstParam();
	kvi_adjustFilePath(szFile);
	buffer.append(szFile);
	return true; //c->leaveContext();
}


/*
	@doc: file.ps
	@type:
		function
	@title:
		$file.ps
	@short:
		Returns the file path separator
	@syntax:
		$file.ps
	@description:
		Returns the file path separator for the current operating system.[br]
		On windows , '\' is returned , on UNIX , '/'.[br]
	@seealso:
		[fnc]$file.fixpath[/file]
*/

static bool file_module_fnc_ps(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	//ENTER_CONTEXT(c,"file_module_fnc_size");
	buffer.append(KVI_PATH_SEPARATOR_CHAR);
	return true; //c->leaveContext();
}



/*
	@doc: file.read
	@type:
		function
	@title:
		$file.read
	@short:
		Reads a text file
	@syntax:
		$file.read(<filename>[,<size>])
	@description:
		Reads at most <size> bytes of the file pointed by <filename>.[br]
		The data read is returned as a string , so if the file contains binary data,
		expect strange results.[br] If <size> is not specified , the whole file is readed.[br]
		WARNING: always check the file size before attemting to read a whole file...
		reading a CDROM iso image may sit down your system :) (and will prolly crash while
		allocating memory , before attempting to read anything)[br]
		An empty string is returned if a serious error occures.[br]
		The <filename> is adjusted according to the system that KVIrc is running on.[br]
	@seealso:
		[fnc]$file.readbinary[/file]
*/

static bool file_module_fnc_read(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_CONTEXT(c,"file_module_fnc_read");
	KviStr szPath = parms->safeFirstParam();
	kvi_adjustFilePath(szPath);
	QFile f(szPath.ptr());
	if(!f.open(IO_ReadOnly))
	{
		c->warning(__tr("Can't open the file \"%s\" for reading"),szPath.ptr());
		return c->leaveContext();
	}

	KviStr szSize = parms->safeNextParam();

	bool bOk;

	unsigned int uSize = szSize.toUInt(&bOk);

	if(!bOk)uSize = f.size();
	else if(uSize > f.size())uSize = f.size();

	char * buf = (char *)kvi_malloc(sizeof(char) * (uSize + 1));

	unsigned int uReaded = 0;

	unsigned int uRetries = 0;


	while(uReaded != uSize)
	{
		int readedNow = f.readBlock(buf + uReaded,uSize - uReaded);
		if(readedNow < 0)
		{
			kvi_free(buf);
			c->warning(__tr("Read error for file %s"),szPath.ptr());
			return c->leaveContext();
		} else readedNow += uReaded;
		uRetries ++;
		if(uRetries > 1000)
		{
			// ops
			kvi_free(buf);
			c->warning(__tr("Read error for file %s (have been unable to read the requested size in 1000 retries)"),szPath.ptr());
			return c->leaveContext();
		}
		uReaded += readedNow;
	}

	buf[uSize] = '\0';

	buffer.append(buf);

	kvi_free(buf);

	return c->leaveContext();
}


/*
	@doc: file.localdir
	@type:
		function
	@title:
		$file.localdir
	@short:
		Get the KVIrc local directory
	@syntax:
		$file.localdir(<relative_path>)
		$file.localdir
	@description:
		Returns the path to the KVIrc local data directory.[br]
		The KVIrc local data directory is always writeable and contains
		the various subdirectories that KVIrc uses internally: audio , avatars ,
		config , help , incoming , log , modules , msgcolors and pics.[br]
		If <relative_path> is passed , then it is appended at the end of the directory
		to form a complete filepath.[br]
		The path is adjusted to contain single separators suitable for the platform
		that KVIrc is atually running on (thus you not need to care about path
		separators in the <relative_path> , KVIrc will adjust them).[br]
	@examples:
		[example]
			echo KVIrc looks for pictures in $file.localdir(pics)
			echo panic.png would be translated to $file.localdir(pics/panic.png)
		[/example]
*/

static bool file_module_fnc_localdir(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_CONTEXT(c,"file_module_fnc_localdir");
	KviStr szFile = parms->safeFirstParam();

	if(szFile.isEmpty())szFile.append(KVI_PATH_SEPARATOR_CHAR);

	KviStr szPath;
	g_pApp->getLocalKvircDirectory(szPath,KviApp::None,szFile.hasData() ? szFile.ptr() : 0);

	kvi_adjustFilePath(szPath);

	buffer.append(szPath);
	return c->leaveContext();
}


/*
	@doc: file.globaldir
	@type:
		function
	@title:
		$file.globaldir
	@short:
		Get the KVIrc global directory
	@syntax:
		$file.globaldir(<relative_path>)
		$file.globaldir
	@description:
		Returns the path to the KVIrc global data directory.[br]
		The KVIrc local data directory is always readable but usually not writeable and contains
		the various subdirectories that KVIrc uses internally: audio , avatars ,
		config , help , incoming , log , modules , msgcolors and pics.[br]
		If <relative_path> is passed , then it is appended at the end of the directory
		to form a complete filepath.[br]
		The path is adjusted to contain single separators suitable for the platform
		that KVIrc is atually running on (thus you not need to care about path
		separators in the <relative_path> , KVIrc will adjust them).[br]
	@examples:
		[example]
			echo KVIrc looks for pictures in $file.globaldir(pics)
			echo panic.png would be translated to $file.globaldir(pics/panic.png)
		[/example]
*/

static bool file_module_fnc_globaldir(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_CONTEXT(c,"file_module_fnc_globaldir");
	KviStr szFile = parms->safeFirstParam();

	if(szFile.isEmpty())szFile.append(KVI_PATH_SEPARATOR_CHAR);

	KviStr szPath;
	g_pApp->getGlobalKvircDirectory(szPath,KviApp::None,szFile.ptr());

	kvi_adjustFilePath(szPath);

	buffer.append(szPath);
	return c->leaveContext();
}

/*
	@doc: file.extractpath
	@type:
		function
	@title:
		$file.extractpath
	@short:
		Extract the path from a filename
	@syntax:
		$file.extractpath(<filepath>)
	@description:
		Returns the path part of the <filepath> translated to match the current
		platform filesystem conventions.[br]
		The path will NOT contain a trailing path separator.[br]
		For example, if <filepath> is /usr/arch/mp3/Carisma_SuonoDelSilenzio.mp3 then
		this function will return /usr/arch/mp3 on UNIX and C:\usr\arch\mp3 on Windows.
	@seealso:
		[fnc]$file.extractfilename[/file]
*/

static bool file_module_fnc_extractpath(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	//ENTER_CONTEXT(c,"file_module_fnc_size");

	KviStr szPath = parms->safeFirstParam();
	kvi_adjustFilePath(szPath);

	szPath.cutFromLast(KVI_PATH_SEPARATOR_CHAR);

	while(szPath.lastCharIs(KVI_PATH_SEPARATOR_CHAR))szPath.cutRight(1);

	//szPath.ensureLastCharIs(KVI_PATH_SEPARATOR_CHAR);

	//kvi_adjustFilePath(szPath);

	buffer.append(szPath);
	return true; //c->leaveContext();
}

/*
	@doc: file.extractfilename
	@type:
		function
	@title:
		$file.extractfilename
	@short:
		Extract the filename from a file path
	@syntax:
		$file.extractpath(<filepath>)
	@description:
		Returns the filename part of the filepath translated to match the current
		platform filesystem conventions.[br]
		For example, if <filepath> is /usr/arch/mp3/Carisma_SuonoDelSilenzio.mp3 then
		this function will return Carisma_SuonoDelSilenzio.mp3
	@seealso:
		[fnc]$file.extractpath[/file]
*/

static bool file_module_fnc_extractfilename(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	//ENTER_CONTEXT(c,"file_module_fnc_size");

	KviStr szPath = parms->safeFirstParam();

	kvi_adjustFilePath(szPath);

	szPath.cutToLast(KVI_PATH_SEPARATOR_CHAR);

	//kvi_adjustFilePath(szPath);

	buffer.append(szPath);
	return true; //c->leaveContext();
}

static bool file_module_init(KviModule * m)
{
	m->registerCommand("copy",file_module_cmd_copy);
	m->registerCommand("rename",file_module_cmd_rename);
	m->registerCommand("mkdir",file_module_cmd_mkdir);
	m->registerCommand("write",file_module_cmd_write);
	m->registerCommand("remove",file_module_cmd_remove);
	m->registerCommand("rmdir",file_module_cmd_rmdir);

	m->registerFunction("exists",file_module_fnc_exists);
	m->registerFunction("type",file_module_fnc_type);
	m->registerFunction("size",file_module_fnc_size);
	m->registerFunction("fixpath",file_module_fnc_fixpath);
	m->registerFunction("ps",file_module_fnc_ps);
	m->registerFunction("read",file_module_fnc_read);
	m->registerFunction("localdir",file_module_fnc_localdir);
	m->registerFunction("globaldir",file_module_fnc_globaldir);
	m->registerFunction("extractpath",file_module_fnc_extractpath);
	m->registerFunction("extractfilename",file_module_fnc_extractfilename);

	return true;
}

static bool file_module_cleanup(KviModule *m)
{
	return true;
}

KVIMODULEEXPORTDATA KviModuleInfo kvirc_module_info=
{
	"File",                                                 // module name
	"1.0.0",                                                // module version
	"Copyright (C) 2001 Szymon Stefanek (stefanek@tin.it)", // author & (C)
	"Interface to the file system",
	file_module_init,
	0,
	0,
	file_module_cleanup
};
