
#include "tableview.h"
#include "qscrollbar.h"
#include "qpainter.h"
#include "qdrawutil.h"



enum ScrollBarDirtyFlags
{
	verGeometry = 0x01, verSteps    = 0x02, verRange    = 0x04, verValue    = 0x08,
	horGeometry = 0x10, horSteps    = 0x20, horRange    = 0x40, horValue    = 0x80,
	verMask     = 0x0F, horMask     = 0xF0
};


#define HSBEXT horizontalScrollBar()->sizeHint().height()
#define VSBEXT verticalScrollBar()->sizeHint().width()

SSEXTableView::SSEXTableView( QWidget *parent, const char *name, WFlags f )
#if QT_VERSION >= 300
	: QFrame( parent, name, f)
#else
    : QFrame( parent, name, f, FALSE )
#endif
{
	m_iRows           = 0;
	m_iTopCell        = 0;                   // zero offset
	m_iXOffset        = 0;
	m_iYOffset        = 0;	// zero total pixel offset

	m_iCellHeight     = 10;
	m_iCellWidth      = 10; // user defined cell size

	m_uTableFlags     = 0;

	m_pVScrollBar = new QScrollBar( QScrollBar::Vertical,this);
	m_pVScrollBar->setCursor( arrowCursor );
	m_pVScrollBar->resize(m_pVScrollBar->sizeHint() ); // height is irrelevant
	m_pVScrollBar->setTracking(false);
	m_pVScrollBar->setFocusPolicy( NoFocus );
	connect(m_pVScrollBar, SIGNAL(valueChanged(int)),SLOT(verSbValue(int)));
	connect(m_pVScrollBar, SIGNAL(sliderMoved(int)),SLOT(verSbSliding(int)));
	m_pVScrollBar->hide();

	m_pHScrollBar = new QScrollBar( QScrollBar::Horizontal,this);
	m_pHScrollBar->setCursor(arrowCursor);
	m_pHScrollBar->resize(m_pHScrollBar->sizeHint()); // width is irrelevant
	m_pHScrollBar->setFocusPolicy(NoFocus);
	m_pHScrollBar->setTracking( FALSE );
	connect(m_pHScrollBar, SIGNAL(valueChanged(int)),SLOT(horSbValue(int)));
	connect(m_pHScrollBar, SIGNAL(sliderMoved(int)),SLOT(horSbSliding(int)));
	m_pHScrollBar->hide();

	m_pCornerSquare = new QWidget(this);
	m_pCornerSquare->setGeometry( maxViewX() + frameWidth() + 1,maxViewY() + frameWidth() + 1,VSBEXT,HSBEXT);

	m_uSbDirtyFlag          = 0;
	m_bCoveringCornerSquare = false;
	m_bInSbUpdate           = false;
#if QT_VERSION < 300
    setFontPropagation(SamePalette);
    setPalettePropagation(SamePalette);
#endif
}



SSEXTableView::~SSEXTableView()
{
    delete m_pVScrollBar;
    delete m_pHScrollBar;
    delete m_pCornerSquare;
}

void SSEXTableView::show()
{
    showOrHideScrollBars();
    QWidget::show();
}

void SSEXTableView::repaint(int x,int y,int w,int h)
{
	if(!isVisible() || testWState(WState_BlockUpdates))return;
	if(w < 0)w = width() - x;
	if(h < 0)h = height() - y;

	QRect r(x,y,w,h);
   	if(r.isEmpty())return;

	QPaintEvent e(r);

	paintEvent(&e);
}


void SSEXTableView::setNumRows( int rows )
{
	if(rows < 0)rows = 0;
	if(m_iRows == rows)return;

	if(isVisible())
	{
		int oldLastVisible = lastRowVisible();
		int oldTopCell = topCell();
		m_iRows = rows;
		if((oldLastVisible != lastRowVisible()) || (oldTopCell != topCell()))repaint();
	} else m_iRows = rows;
	updateScrollBars( verRange );
	updateFrameSize();
}

void SSEXTableView::setTopCell( int row )
{
	int newY = 0;
	if(row < 0)row = 0;
	newY = row * m_iCellHeight;
	if(newY > maxYOffset())newY = maxYOffset();
	setOffset(m_iXOffset,newY);
}

void SSEXTableView::setOffset(int x, int y, bool updateScrBars)
{
	if(x == m_iXOffset && y == m_iYOffset)return;

	if(x < 0)x = 0;
	if(y < 0)y = 0;

	if(x > maxXOffset())x = maxXOffset();
	if(y > maxYOffset())y = maxYOffset();

	m_iTopCell = y / m_iCellHeight;

	int dx = (x - m_iXOffset);
	int dy = (y - m_iYOffset);

	m_iXOffset = x % m_iCellWidth;
	m_iYOffset = m_iTopCell * m_iCellHeight;

	if(isVisible())scroll(-dx,-dy,contentsRect());
	if(updateScrBars)updateScrollBars( verValue | horValue );
}


void SSEXTableView::setCellWidth(int cellWidth)
{
	if(m_iCellWidth== cellWidth)return;
	if(cellWidth < 1)cellWidth = 1;
	if(cellWidth > 32000)cellWidth = 32000;
	m_iCellWidth= (short)cellWidth;
	updateScrollBars(horSteps | horRange);
	if(isVisible())repaint();
}


void SSEXTableView::setCellHeight( int cellHeight )
{
    if( m_iCellHeight== cellHeight )return;
	if(cellHeight < 1)cellHeight = 1;
	if(cellHeight > 32000)cellHeight = 32000;
    m_iCellHeight= (short)cellHeight;
	setMinimumHeight((cellHeight * 2) + VSBEXT);
    updateScrollBars( verSteps | verRange );
    if(isVisible())repaint();
}

void SSEXTableView::updateCell(int row)
{
	int xPos, yPos;
	xPos = minViewX() - m_iXOffset;
	if(!rowYPos(row,&yPos))return;

	QRect uR = QRect(xPos,yPos,m_iCellWidth,m_iCellHeight);
	repaint(uR.intersect(viewRect()),false);
}

QRect SSEXTableView::viewRect() const
{
    return QRect(frameWidth(), frameWidth(), viewWidth(), viewHeight());
}

int SSEXTableView::lastRowVisible() const
{
	int last = (m_iTopCell + maxVisibleCells() - 1);
	if(last >= m_iRows)return m_iRows - 1;
	return last;
}

void SSEXTableView::coverCornerSquare( bool enable )
{
    m_bCoveringCornerSquare = enable;
	if ( enable )m_pCornerSquare->show();
	else m_pCornerSquare->hide();
}

void SSEXTableView::horSbValue( int val )
{
	setOffset(val,m_iYOffset,false);
}

void SSEXTableView::horSbSliding( int val )
{
	setOffset(val,m_iYOffset,false);
}

void SSEXTableView::verSbValue( int val )
{
	setOffset(m_iXOffset,val * m_iCellHeight,false);
}

void SSEXTableView::verSbSliding( int val )
{
	setOffset(m_iXOffset,val * m_iCellHeight,false);
}

void SSEXTableView::paintEvent( QPaintEvent *e )
{
	QRect updateR = e->rect();

	if(m_uSbDirtyFlag)updateScrollBars(0);

	QPainter paint(this);

	if(!contentsRect().contains(updateR,true))
	{
		drawFrame(&paint);
		if(updateR.left() < frameWidth())updateR.setLeft( frameWidth());
		if(updateR.top() < frameWidth())updateR.setTop(frameWidth());
	}

	int maxWX = maxViewX();
	int maxWY = maxViewY();

	if(updateR.right() > maxWX)updateR.setRight(maxWX);
	if(updateR.bottom() > maxWY)updateR.setBottom(maxWY);

	int firstRow = findRow(updateR.y());
	int xStart = minViewX() - m_iXOffset;
	int yStart;

	bool bRowOk = rowYPos(firstRow,&yStart);

	if((updateR.x() > (xStart + m_iCellWidth)) && !bRowOk)
	{
		paint.eraseRect(updateR);
		return;
	}

	int maxX = updateR.right();
	int maxY = updateR.bottom();
	int row = firstRow;
	int yPos = yStart;
	int nextY;
    QRect winR = viewRect();
    QRect cellR;
    QRect cellUR;

	while((yPos <= maxY) && (row < m_iRows))
	{
		nextY = yPos + m_iCellHeight;
		if(xStart <= maxX)
		{
			cellR.setRect(xStart,yPos,m_iCellWidth,m_iCellHeight);
			cellUR = cellR.intersect(updateR);

			if(cellUR.isValid())
			{
				m_rectCellUpdate = cellUR;
				m_rectCellUpdate.moveBy(-xStart,-yPos); // cell coordinates

				paint.translate(xStart, yPos);
				if((frameWidth() > 0) && !winR.contains(cellR))
				{
					paint.setClipRect(cellUR);
					paintCell(&paint, row);
					paint.setClipping(false);
				} else paintCell(&paint,row);
				paint.translate(-xStart,-yPos);
			}
		}
		row++;
		yPos = nextY;
	}

    if((xStart + m_iCellWidth) <= maxX)
	{
		QRect r = viewRect();
		r.setLeft( xStart + m_iCellWidth);
		r.setBottom(yPos < maxY ? yPos : maxY );
		paint.eraseRect( r.intersect( updateR ));
    }
    if(yPos <= maxY)
	{
		QRect r = viewRect();
		r.setTop(yPos);
		paint.eraseRect(r.intersect(updateR));
    }
}

void SSEXTableView::resizeEvent(QResizeEvent *)
{
    updateScrollBars( horValue | verValue | horSteps | horGeometry | horRange | verSteps | verGeometry | verRange );
    showOrHideScrollBars();
    updateFrameSize();
    int maxX = QMIN(m_iXOffset, maxXOffset());
    int maxY = QMIN(m_iYOffset, maxYOffset());
    setOffset(maxX, maxY);
}

void SSEXTableView::setHorScrollBar(bool on)
{
	if ( on )
	{
		m_uTableFlags |= Tbl_hScrollBar;
		m_uSbDirtyFlag = m_uSbDirtyFlag | (horMask | verMask);
		if(m_uTableFlags & Tbl_vScrollBar)coverCornerSquare(true);
	} else {
		m_uTableFlags &= ~Tbl_hScrollBar;
		coverCornerSquare(false);
		bool hideScrollBar = m_pHScrollBar->isVisible();
		if(hideScrollBar)m_pHScrollBar->hide();
		else m_uSbDirtyFlag = m_uSbDirtyFlag | verMask;
		if(hideScrollBar && isVisible() )repaint( m_pHScrollBar->x(), m_pHScrollBar->y(),
												     width() - m_pHScrollBar->x(), m_pHScrollBar->height() );
    }
}


void SSEXTableView::setVerScrollBar(bool on)
{
    if(on)
	{
		m_uTableFlags |= Tbl_vScrollBar;
		m_uSbDirtyFlag = m_uSbDirtyFlag | (horMask | verMask);
		if(m_uTableFlags & Tbl_hScrollBar)coverCornerSquare(true);
    } else {
		m_uTableFlags &= ~Tbl_vScrollBar;
		coverCornerSquare(false);
		bool hideScrollBar = m_pVScrollBar->isVisible();
		if(hideScrollBar)m_pVScrollBar->hide();
		else m_uSbDirtyFlag = m_uSbDirtyFlag | horMask;
		if(hideScrollBar && isVisible())repaint( m_pVScrollBar->x(), m_pVScrollBar->y(),
				     m_pVScrollBar->width(), height() - m_pVScrollBar->y() );
    }
}

int SSEXTableView::findRow(int yPos) const
{
	if(m_iRows == 0)return -1;

	int row;

	if(yPos >= minViewY())row = m_iTopCell + ((yPos - minViewY()) / m_iCellHeight);
	else return m_iTopCell;

	if(row > lastRowVisible())return lastRowVisible();
	return row;
}


bool SSEXTableView::rowYPos( int row, int *yPos ) const
{
	int y;
	if(row >= m_iTopCell)
	{
		int lastVisible = lastRowVisible();
		if ( row > lastVisible || lastVisible == -1 )return FALSE;
		y = (row - m_iTopCell)*m_iCellHeight+ minViewY();
	} else return FALSE;
	if(yPos)*yPos = y;
    return TRUE;
}


int SSEXTableView::maxViewX() const
{
    return width() - 1 - frameWidth() - ((m_uTableFlags & Tbl_vScrollBar) ? VSBEXT : 0);
}


int SSEXTableView::maxViewY() const
{
    return height() - 1 - frameWidth() - ((m_uTableFlags & Tbl_hScrollBar) ? HSBEXT : 0);
}

void SSEXTableView::doAutoScrollBars()
{
    int viewW = width()	 - frameWidth() - minViewX();
    int viewH = height() - frameWidth() - minViewY();

//    bool vScrollOn = m_uTableFlags & Tbl_vScrollBar;
//    bool hScrollOn = m_uTableFlags & Tbl_hScrollBar;

	int w = m_iCellWidth;
	int h = m_iCellHeight * m_iRows;

	bool hScrollOn = w > viewW;
	bool vScrollOn = h > viewH;

	if(vScrollOn && !hScrollOn )
	{
		if( w > viewW - VSBEXT )hScrollOn = true;
	}

	if(hScrollOn && !vScrollOn )
	{
		if( h > viewH - HSBEXT )vScrollOn = true;
	}

	setHorScrollBar(hScrollOn);
	setVerScrollBar(vScrollOn);
 	updateFrameSize();
}


void SSEXTableView::updateScrollBars( uint f )
{
	m_uSbDirtyFlag = m_uSbDirtyFlag | f;

 	if(m_bInSbUpdate)return;

	m_bInSbUpdate = true;

	if((m_uSbDirtyFlag & horRange) || (m_uSbDirtyFlag & verRange))doAutoScrollBars();

	if(yOffset() > 0 && !(m_uTableFlags & Tbl_vScrollBar))setYOffset(0);
	if(xOffset() > 0 && !(m_uTableFlags & Tbl_hScrollBar))setXOffset(0);

    if(!isVisible())
	{
		m_bInSbUpdate = false;
		return;
	}

	if((m_uTableFlags & Tbl_hScrollBar) && (m_uSbDirtyFlag & horMask) != 0 )
	{
		if( m_uSbDirtyFlag & horGeometry )m_pHScrollBar->setGeometry( 0,height() - HSBEXT,viewWidth() + frameWidth()*2,HSBEXT);
		if( m_uSbDirtyFlag & horSteps )m_pHScrollBar->setSteps( QMIN(m_iCellWidth,viewWidth() / 2) , viewWidth() );
		if( m_uSbDirtyFlag & horRange )m_pHScrollBar->setRange( 0, maxXOffset() );
		if( m_uSbDirtyFlag & horValue )m_pHScrollBar->setValue( m_iXOffset );
		if( !m_pHScrollBar->isVisible() )m_pHScrollBar->show();
	}

	if ( (m_uTableFlags & Tbl_vScrollBar) && (m_uSbDirtyFlag & verMask) != 0 )
	{
		if( m_uSbDirtyFlag & verGeometry )m_pVScrollBar->setGeometry( width() - VSBEXT, 0,VSBEXT,viewHeight() + frameWidth()*2 );
		if( m_uSbDirtyFlag & verSteps )m_pVScrollBar->setSteps(1,maxVisibleCells());
		if( m_uSbDirtyFlag & verRange )m_pVScrollBar->setRange(0,maxTopCell());
		if( m_uSbDirtyFlag & verValue )m_pVScrollBar->setValue(m_iTopCell);
		if( !m_pVScrollBar->isVisible() )m_pVScrollBar->show();
	}

	if(m_bCoveringCornerSquare && ( (m_uSbDirtyFlag & verGeometry ) || (m_uSbDirtyFlag & horGeometry)) )
		m_pCornerSquare->move( maxViewX() + frameWidth() + 1,maxViewY() + frameWidth() + 1 );

	m_uSbDirtyFlag = 0;
	m_bInSbUpdate = FALSE;
}


void SSEXTableView::updateFrameSize()
{
	int rw = width()  - ((m_uTableFlags & Tbl_vScrollBar) ? VSBEXT : 0);
	int rh = height() - ((m_uTableFlags & Tbl_hScrollBar) ? HSBEXT : 0);
	if(rw < 0)rw = 0;
	if(rh < 0)rh = 0;

	int fh = frameRect().height();
	int fw = frameRect().width();
	setFrameRect(QRect(0,0,rw,rh));

	if(rw != fw)update(QMIN(fw,rw) - frameWidth() - 2, 0, frameWidth()+4, rh);
	if(rh != fh)update(0, QMIN(fh,rh) - frameWidth() - 2, rw, frameWidth()+4);
}


int SSEXTableView::maxXOffset() const
{
    int maxOffset = m_iCellWidth - viewWidth();
    return maxOffset > 0 ? maxOffset : 0;
}

int SSEXTableView::maxTopCell() const
{
	int m = maxVisibleCells();
	if(m_iRows <= m)return 0;
	return m_iRows - m;
}

void SSEXTableView::showOrHideScrollBars()
{
	if(m_uTableFlags & Tbl_vScrollBar)
	{
		if(!m_pVScrollBar->isVisible())m_uSbDirtyFlag = m_uSbDirtyFlag | verMask;
	} else {
		if(m_pVScrollBar->isVisible())m_pVScrollBar->hide();
	}
	if(m_uTableFlags & Tbl_hScrollBar)
	{
		if(!m_pHScrollBar->isVisible())m_uSbDirtyFlag = m_uSbDirtyFlag | horMask;
	} else {
		if(m_pHScrollBar->isVisible())m_pHScrollBar->hide();
	}
	if((m_uTableFlags & Tbl_hScrollBar) && (m_uTableFlags & Tbl_vScrollBar))
	{
		if(!m_pCornerSquare->isVisible())m_pCornerSquare->show();
	} else {
		if(m_pCornerSquare->isVisible())m_pCornerSquare->hide();
	}
}

#include "m_tableview.moc"
