#ifndef _SEX_EDITOR_
#define _SEX_EDITOR_

//=============================================================================
//
//   This file is part of the SSEX text editor distribution
//   Copyright (C) 2001 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//=============================================================================

//
// This is a self-contained text editor widget.
// It relies ont eh tableview.cpp and tableview.h files , so if you include it
// in an external project , don't forget to add that two files too.
//

#include "tableview.h"

#include <qcolor.h>
#include <qcstring.h>
#include "kvi_list.h"
#include <qpixmap.h>
#include <qtimer.h>
#include <qpopupmenu.h>
#include <qframe.h>
#include <qlineedit.h>
#include <qcheckbox.h>
#include <qpushbutton.h>
#include <qfont.h>
#include <qtabdialog.h>
#include <qhbox.h>

class SSEXEditor;
class SSEXOptions;
class SSEXOptionsDialog;

class SSEXColorEditor : public QHBox
{
	Q_OBJECT
public:
	SSEXColorEditor(SSEXOptionsDialog * dlg,QWidget * par,QColor * pClr,const char * txt);
	~SSEXColorEditor();
protected:
	QColor * m_pClr;
	QColor m_actualClr;
	QPushButton * m_pButton;
protected slots:
	void chooseColor();
public slots:
	void commit();
};

class SSEXFontEditor : public QHBox
{
	Q_OBJECT
public:
	SSEXFontEditor(SSEXOptionsDialog * dlg,QWidget * par,QFont * pFont,const char * txt);
	~SSEXFontEditor();
protected:
	QFont * m_pFont;
	QPushButton * m_pButton;
protected slots:
	void chooseFont();
public slots:
	void commit();
};


class SSEXOptionsDialog : public QTabDialog
{
	friend class SSEXOptions;
	Q_OBJECT
protected:
	SSEXOptionsDialog();
public:
	~SSEXOptionsDialog();
public:
	virtual void showEvent(QShowEvent *e);
	virtual void closeEvent(QCloseEvent *e);
public slots:
	void okClicked();
	void cancelClicked();
signals:
	void doCommit();
};

class SSEXOptions
{
	friend class SSEXEditor;
	friend class SSEXOptionsDialog;
protected:
	SSEXOptions();
	~SSEXOptions();
private:
	KviPtrList<SSEXEditor> * m_pRefList;
	SSEXOptionsDialog * m_pOptionsDialog;
public:
	static void attach(SSEXEditor * ed);
	static void detach(SSEXEditor * ed);
	void edit();
	void apply();
protected:
	QColor clrTxtBackground;
	QColor clrTxtDisabledBackground;
	QColor clrTxtExtBackground;
	QColor clrTxtMarkBackground;
	QColor clrTxtNormalText;
	QColor clrTxtCursor;
	QFont  fntTxt;

	QColor clrCppBackground;
	QColor clrCppDisabledBackground;
	QColor clrCppMarkBackground;
	QColor clrCppExtBackground;
	QColor clrCppNormalText;
	QColor clrCppCursor;
	QColor clrCppTab;
	QColor clrCppBrace;
	QColor clrCppLineComment;
	QColor clrCppMultilineComment;
	QColor clrCppParenthesis;
	QColor clrCppOperator;
	QColor clrCppEscape;
	QColor clrCppChar;
	QColor clrCppString;
	QColor clrCppKeyword;
	QColor clrCppType;
	QColor clrCppNumber;
	QColor clrCppPunctuation;
	QColor clrCppSystemIdentifier;
	QColor clrCppPreprocessor;
	QColor clrCppInclude;
	QColor clrCppMemberVariable;
	QColor clrCppGlobalVariable;
	QColor clrCppQClass;
	QColor clrCppQSignals;
	QColor clrCppKClass;
	QColor clrCppXStuff;
	QColor clrCppGtkStruct;
	QColor clrCppGdkStruct;
	QColor clrCppGType;
	QColor clrCppGtkCall;
	QColor clrCppGdkCall;
	QColor clrCppGtkMacro;
	QColor clrCppGdkMacro;
	QColor clrCppGMacro;
	QColor clrCppSSStuff;
	QColor clrCppSpecial;
	QFont  fntCpp;

	QColor clrHtmlBackground;
	QColor clrHtmlDisabledBackground;
	QColor clrHtmlExtBackground;
	QColor clrHtmlMarkBackground;
	QColor clrHtmlNormalText;
	QColor clrHtmlCursor;
	QColor clrHtmlComment;
	QColor clrHtmlTag;
	QColor clrHtmlString;
	QColor clrHtmlTagInternal;
	QFont  fntHtml;

	QColor clrPerlBackground;
	QColor clrPerlDisabledBackground;
	QColor clrPerlExtBackground;
	QColor clrPerlMarkBackground;
	QColor clrPerlNormalText;
	QColor clrPerlCursor;
	QColor clrPerlComment;
	QColor clrPerlTab;
	QColor clrPerlVariable;
	QColor clrPerlOperator;
	QColor clrPerlBrace;
	QFont  fntPerl;

	QColor clrAutoconfBackground;
	QColor clrAutoconfDisabledBackground;
	QColor clrAutoconfExtBackground;
	QColor clrAutoconfMarkBackground;
	QColor clrAutoconfNormalText;
	QColor clrAutoconfCursor;
	QColor clrAutoconfComment;
	QColor clrAutoconfShellCommand;
	QColor clrAutoconfDnlComment;
	QColor clrAutoconfBuiltinMacro;
	QColor clrAutoconfTab;
	QColor clrAutoconfVariable;
	QColor clrAutoconfOperator;
	QColor clrAutoconfBrace;
	QFont  fntAutoconf;

	unsigned int uUndoDepth;
	unsigned int uTabsNumSpaces;

protected:
	void optionsDialogTerminated();
private:
	void load();
	void save();
	void initDefaults();
	void attachRef(SSEXEditor * ed){ m_pRefList->append(ed); };
	void detachRef(SSEXEditor * ed){ m_pRefList->removeRef(ed); };
	bool hasRef(){ return (m_pRefList->count() > 0); };
};


#define SSEX_EDITOR_BEGIN_IN_COMMENT 1
#define SSEX_EDITOR_END_IN_COMMENT 2
#define SSEX_EDITOR_BEGIN_IN_TAG 4
#define SSEX_EDITOR_END_IN_TAG 8

typedef struct _SSEXEditorTextLine
{
	int      width;
	QCString text;
	int      length;
	int      flags;
} SSEXEditorTextLine;

typedef struct _SSEXEditorKeystroke
{
	int  ascii;
	int  state;
	int  key;
} SSEXEditorKeystroke;


class SSEXUndo
{
	friend class SSEXEditor;
protected:
	enum Type { Insert , Remove };
	QCString m_szName;
	Type     m_type;
	int      m_iRow1;
	int      m_iRow2;
	int      m_iCol1;
	int      m_iCol2;
	QCString m_szText;
protected:
	// delete constructor
	SSEXUndo(const QCString &name,int row1,int col1,int row2,int col2)
	: m_szName(name), m_iRow1(row1), m_iCol1(col1), m_iRow2(row2), m_iCol2(col2) { m_type = Remove; };
	// insert constructor
	SSEXUndo(const QCString &name,int row1,int col1,const QCString & text)
	: m_szName(name), m_iRow1(row1), m_iCol1(col1), m_szText(text) { m_type = Insert; };
public:
	~SSEXUndo(){};
};



class SSEXFindWidget : public QFrame
{
	friend class SSEXEditor;
	Q_OBJECT
public:
	SSEXFindWidget(SSEXEditor * parent);
	~SSEXFindWidget();
protected:
	QPoint        m_pressPoint;
	SSEXEditor  * m_pEditor;
	QLineEdit   * m_pFindStringEdit;
	QLineEdit   * m_pReplaceStringEdit;
	QCheckBox   * m_pRegExpCheckBox;
	QCheckBox   * m_pCaseSensitiveCheckBox;
	QPushButton * m_pReplace;
	QPushButton * m_pReplaceAndFindNext;
	QPushButton * m_pReplaceAllInSelection;
protected:
	virtual void mousePressEvent(QMouseEvent *e);
	virtual void mouseMoveEvent(QMouseEvent *e);
protected slots:
	void findNextClicked();
	void findPrevClicked();
	void replaceAllClicked();
	void replaceAndFindNextClicked();
	void replaceAllInSelectionClicked();
};

/*
class SSEXEditorMode
{
public:
	SSEXEditorMode(const QString &name);
	~SSEXEditorMode();
protected:
	QString m_szName;
public:
	const QString & name(){ return m_szName; };

	void paintCell(SSEXEditor * ed,QPainter * p,SSEXEditorTextLine * l,int row) = 0;
};

SSEXEditorMode::SSEXEditorMode(const QString &name)
{
	m_szName = name;
}

SSEXEditorMode::~SSEXEditorMode()
{
}


void SSEXEditorMode::paintCell(SSEXEditor *ed,QPainter *p,SSEXEditorTextLine *l,int row)
{

}

class SSEXEditorPlainTextMode : public SSEXEditorMode
{
public:
	SSEXEditorPlainTextMode();
	~SSEXEditorPlainTextMode();
public:
	virtual void paintCell(SSEXEditor * ed,QPainter *p,SSEXEditorTextLine * l,int row);
};


SSEXEditorPlainTextMode::SSEXEditorPlainTextMode()
: SSEXEditorMode("Plain text")
{
}

SSEXEditorPlainTextMode::~SSEXEditorPlainTextMode()
{
}

void SSEXEditorPlainTextMode::paintCell(SSEXEditor *ed,QPainter *p,SSEXEditorTextLine *l,int row)
{
	QRect updateR = cellUpdateRect();

	QPainter paint(m_pMemBuffer);

	paint.fillRect(updateR,ed->isEnabled() ? g_pSSEXOptions->clrTxtBackground : g_pSSEXOptions->clrTxtDisabledBackground);

	if(m_bHasBlockMark)paintBlockMark(&paint,l,row,updateR,&(g_pSSEXOptions->clrTxtMarkBackground));

	paint.setPen(g_pSSEXOptions->clrTxtNormalText);
	paint.setFont(p->font());

	const char * c = l->text.data();
	int curXPos = SSEX_EDITOR_BORDER;
	int lastTabPos = SSEX_EDITOR_BORDER;

	while(*c)
	{
		if(*c == '\t')
		{
			// only a tab
			while(lastTabPos <= curXPos)lastTabPos += m_iTabsNumPixels;
			curXPos = lastTabPos;
			c++;
		} else {
			// text block
			const char * begin = c;
			int blockWidth = 0;
			while(*c && *c != '\t'){
				blockWidth += m_iCharWidth[(unsigned char)*c];
				c++;
			}
			paint.drawText(curXPos,m_iFontAscent,begin,c - begin);
			curXPos += blockWidth;
		}
	}

	if(m_iCursorRow == row)paintCursor(&paint,l,&(g_pSSEXOptions->clrTxtCursor));
	if(m_bHasSelection)paintSelection(&paint,l,row,updateR);

	int col_xPos,row_yPos;

	p->worldMatrix().map(0,0,&col_xPos,&row_yPos);

	bitBlt(p->device(),col_xPos + updateR.x(),row_yPos + updateR.y(),m_pMemBuffer,updateR.x(),updateR.y(),updateR.width(),updateR.height());

}

*/

class SSEXEditor : public SSEXTableView
{
	Q_OBJECT
	Q_PROPERTY( int KviProperty_FocusOwner READ cursorCol )
public:
	SSEXEditor(QWidget * parent);
	~SSEXEditor();
	enum ColorMode { Normal , Cpp , Html , Perl , Autoconf };
protected:
//	SSEXEditorColors           * m_pColors;
	KviPtrList<SSEXEditorTextLine>  * m_pLines;
	KviPtrList<SSEXEditorKeystroke> * m_pKeystrokes;
	ColorMode                    m_mode;
//	int                          m_iTabsNumSpaces;
	int                          m_iBlinkTime;
	QPopupMenu                 * m_pContextPopup;
	QPopupMenu                 * m_pFindPopup;
	QPopupMenu                 * m_pEditPopup;
	SSEXFindWidget             * m_pFindWidget;
private:
	// Font dependant variables
	int                          m_iTabsNumPixels;
	int                          m_iCursorRow;      // Line in which the cursor is
	int                          m_iCursorPosition;  // The cursor is just before this character in the line
	int                          m_iCursorPositionInPixels;
	int                          m_iLastCursorRow;
	int                          m_iLastCursorPosition;
	int                          m_iCharWidth[256];
	// Internal
	int                          m_iMaxTextWidth;
	int                          m_iMaxTextWidthLine;
	int                          m_iFontAscent;
	int                          m_iFontLineSpacing;
	QPoint                       m_selection1; // selection anchors (char coords)
	QPoint                       m_selection2; // selection anchors (char coords)
	QPoint                       m_mouseAnchor; // mouse press selection anchor (char coords)
	QPoint                       m_lastMousePressCoords;
	bool                         m_bHasSelection;
	QTimer                     * m_pBlinkTimer;
	QTimer                     * m_pDragTimer;
	QPixmap                    * m_pMemBuffer;
	QCString                     m_szFileName;
	bool                         m_bCursorOn;
	bool                         m_bOverwrite;
	bool                         m_bRecordingKeystrokes;
	bool                         m_bModified;
	QPoint                       m_lastFindWidgetPosition;

	QPoint                       m_blockMark1;
	QPoint                       m_blockMark2;
	bool                         m_bHasBlockMark;

	KviPtrList<SSEXUndo>            * m_pUndoList;
public:
	ColorMode mode(){ return m_mode; };
	const char * modeName();
	QPopupMenu * editPopup(){ return m_pEditPopup; };
	QPopupMenu * findPopup(){ return m_pFindPopup; };
	void fillEditPopup(QPopupMenu * p);
	void fillFindPopup(QPopupMenu * p);
	bool closeFile();
	void updateOptions();
	bool isModified(){ return m_bModified; };
	QCString fileName(){ return m_szFileName; };
	void setModified(bool bModified = true);
	void setBlinkTime(int blinkTime);
	int blinkTime(){ return m_iBlinkTime; };
	virtual void setFont(const QFont &f);
	virtual void setEnabled(bool bEnabled);
	void setText(const QCString &text);
	void getText(QCString &buffer);
	void clear();
	bool loadFile(const char *filename);
	bool saveFile(const char *filename);
	void cursorUp(bool bSelect);
	void cursorDown(bool bSelect);
	void cursorLeft(bool bSelect);
	void cursorRight(bool bSelect);
	void cursorPageUp(bool bSelect);
	void cursorPageDown(bool bSelect);
	void cursorHome(bool bSelect);
	void cursorEnd(bool bSelect);
	int cursorRow(){ return m_iCursorRow; };
	int cursorCol() const;
	bool curTextLine(QCString &buf);
	void clearSelection(bool bUpdate = true);
	bool hasSelection(){ return m_bHasSelection; };
	void clearBlockMark(bool bUpdate = true);
	void insertChar(char c);
	void newLine();
	void backspace();
	void del();
	QCString selectedText();
	void insertText(QCString &text,bool bUpdate = true,bool bRecalcWidths = true,bool bCreateUndo = true);

	void commentOut(bool bAlternative);
	void removeComment();
	void setMode(ColorMode mode);
	virtual void keyPressEvent(QKeyEvent *e);
	void goToLineNum(int lineNum);
	void findRegExp(QString &regexp);
public slots:
	void editOptions();
	void chooseFont();
	void undo();
	void markBlock();
	bool saveFile();
	bool saveFileAs();
	void recordKeystrokes();
	void replayKeystrokes();
	void cut();
	void copy();
	void paste();
	void indent();
	void unindent();
	void indentSelected();
	void unindentSelected();
	void switchMode();
	void toggleFindWidget();
	void findNext();
	void findPrev();
	void findNextRegExp();
	void findPrevRegExp();
	void replace();
	void replaceAll();
	void replaceAllRegExp();
	void replaceAllInSelection();
	void replaceAllInSelectionRegExp();
	void replaceAndFindNext();
	void replaceAndFindNextRegExp();
	void goToLine();
	void goToBeginning();
	void goToEnd();
signals:
	void modifyStateChanged(SSEXEditor *ed,bool bModified);
	void fileNameChanged(SSEXEditor *ed,const QCString &newFileName);
	void saved(const QCString &fileName);
	void saveProgress(const QCString &fileName,int progress);
	void cursorPositionChanged(SSEXEditor *ed,int row,int col);
	void recordingKeystrokes(bool bRecording);
	void gainedFocus(SSEXEditor *ed);
	void rightClickOnTextRow(SSEXEditor *ed,const QCString &text,int charIndex,QPopupMenu *contextPopup);
	void textMessage(SSEXEditor *ed,const QCString &msg);
	void modeChanged(SSEXEditor *ed);
protected:
	void addUndo(SSEXUndo * und);
	void clearUndoList();
	void contextPopup(SSEXEditorTextLine *l,int charIndex);
	void startBlinking();
	void stopBlinking();
	void setHasSelection(bool bHasSelection);
	void contextPopup(const QPoint &pos);
	virtual void focusInEvent(QFocusEvent *e);
	virtual void focusOutEvent(QFocusEvent *e);
	virtual void paintCell(QPainter * p,int row);
	virtual void mousePressEvent(QMouseEvent *e);
	virtual void mouseMoveEvent(QMouseEvent *e);
	virtual void mouseReleaseEvent(QMouseEvent *e);
	virtual void wheelEvent(QWheelEvent *e);
#if QT_VERSION >= 300
	virtual void contextMenuEvent(QContextMenuEvent *e);
#endif
	void paintCellNormal(QPainter * p,SSEXEditorTextLine * l,int row);
	void paintCellCpp(QPainter * p,SSEXEditorTextLine *l,int row);
	void paintCellHtml(QPainter * p,SSEXEditorTextLine *l,int row);
	void paintCellPerl(QPainter * p,SSEXEditorTextLine *l,int row);
	void paintCellAutoconf(QPainter * p,SSEXEditorTextLine *l,int row);
	void paintSelection(QPainter * p,SSEXEditorTextLine *l,int row,QRect &updateR);
	void paintBlockMark(QPainter * p,SSEXEditorTextLine *l,int row,const QRect & updateR,QColor * pColor);
	void paintCursor(QPainter * p,SSEXEditorTextLine *l,QColor * pColor);
	void ensureSelectionCoherent();
	void setSelectionCoords(int x1,int row1,int x2,int row2);
	void selectionCursorMovement(const QPoint &oldCursorPos,const QPoint &newCursorPos);
	void killSelection(bool bUpdate,bool bRecalcWidths = true,bool bCreateUndo = true);
	void recalcCursorPosition(SSEXEditorTextLine *l);
	void ensureCursorVisible();
	void setBottomCell(int row);
	int getTextWidthWithTabsForCursorPosition(const char *text,int cursorPos);
	int getTextWidthWithTabs(const char *text);
	int findCharacterAt(int xPositionInCell,SSEXEditorTextLine * l);
	void updateCellSize();
	void updateMaxTextWidth();
	void updateFontDependantVariables();
	QColor * cppModeGetTokenColor(const char * token,int len);
	void initializeCurrentMode();
	void htmlModeComputeTagState(SSEXEditorTextLine *start);
	void cppModeComputeCommentState(SSEXEditorTextLine * start);
protected slots:
	void blinkTimer();
	void dragTimer();
	void editPopupAboutToShow();
	void findPopupAboutToShow();
};

#endif
