//=============================================================================
//
//   This file is part of the SSEX text editor distribution
//   Copyright (C) 2001 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//=============================================================================

#include "editor.h"

#undef Unsorted


#include <qframe.h>
#include <qpainter.h>
#include <qfontmetrics.h>
#include <qfile.h>
#include <qnamespace.h>
#include <qcursor.h>
#include <qclipboard.h>
#include <qapplication.h>
#include <qfiledialog.h>
#include <qfileinfo.h>
#include <qmessagebox.h>
#include <qlayout.h>
#include <qtoolbutton.h>
#include <qlabel.h>
#include <qinputdialog.h>
#include <qfontdialog.h>
#include <qregexp.h>
#include <qcolordialog.h>
#include <qfontdialog.h>
#include <qvbox.h>
#include <qdatastream.h>
#include <qdir.h>

#include <ctype.h>
#include <string.h>

#include "kvi_settings.h"

#ifdef COMPILE_ON_WINDOWS
	#define strncasecmp _strnicmp
	#define strcasecmp _stricmp
#endif

#define __DEBUG


#ifdef __DEBUG
	#define __assert(_expr) if(! (_expr) )if(!(_expr))debug("RANGE ASSERT : \"%s\" is false in %s (%d)",#_expr,__FILE__,__LINE__);
#else
	#define __assert(_expr)
#endif

static SSEXOptions * g_pSSEXOptions = 0;



SSEXColorEditor::SSEXColorEditor(SSEXOptionsDialog * dlg,QWidget * par,QColor * pClr,const char * txt)
: QHBox(par)
{
	m_pClr = pClr;
	m_actualClr = *pClr;
	QLabel * l = new QLabel(txt,this);
	m_pButton = new QPushButton(this);
	m_pButton->setPalette(QPalette(*pClr));
	m_pButton->setMinimumWidth(120);
	m_pButton->setMaximumWidth(120);
	connect(m_pButton,SIGNAL(clicked()),this,SLOT(chooseColor()));
	connect(dlg,SIGNAL(doCommit()),this,SLOT(commit()));
}


SSEXColorEditor::~SSEXColorEditor()
{
}

void SSEXColorEditor::chooseColor()
{
	QColor clr = QColorDialog::getColor(*m_pClr);
	if(clr.isValid())
	{
		m_actualClr = clr;
		m_pButton->setPalette(QPalette(clr));
	}
}

void SSEXColorEditor::commit()
{
	*m_pClr = m_actualClr;
}

SSEXFontEditor::SSEXFontEditor(SSEXOptionsDialog * dlg,QWidget * par,QFont * pFont,const char * txt)
: QHBox(par)
{
	m_pFont = pFont;
	QLabel * l = new QLabel(txt,this);
	m_pButton = new QPushButton(this);
	m_pButton->setFont(*pFont);
	m_pButton->setText(pFont->family());
	m_pButton->setMinimumWidth(120);
	m_pButton->setMaximumWidth(120);
	connect(m_pButton,SIGNAL(clicked()),this,SLOT(chooseFont()));
	connect(dlg,SIGNAL(doCommit()),this,SLOT(commit()));
}


SSEXFontEditor::~SSEXFontEditor()
{
}

void SSEXFontEditor::chooseFont()
{
	bool bOk = false;
	QFont fnt = QFontDialog::getFont(&bOk,m_pButton->font());
	if(bOk)
	{
		m_pButton->setFont(fnt);
		m_pButton->setText(fnt.family());
	}
}

void SSEXFontEditor::commit()
{
	*m_pFont = m_pButton->font();
}


SSEXOptionsDialog::SSEXOptionsDialog()
: QTabDialog()
{
	setOkButton("Ok");
	setCancelButton("Cancel");
	connect(this,SIGNAL(cancelButtonPressed()),this,SLOT(cancelClicked()));
	connect(this,SIGNAL(applyButtonPressed()),this,SLOT(okClicked()));

	SSEXColorEditor * ed;
	SSEXFontEditor * fe;
	QLabel * l;

	QVBox * page = new QVBox(this);
	page->setMargin(5);


	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrPerlBackground),"Background");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrPerlDisabledBackground),"Disabled background");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrPerlExtBackground),"External background");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrPerlMarkBackground),"Mark background");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrPerlNormalText),"NormalText");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrPerlCursor),"Cursor");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrPerlComment),"Comment");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrPerlTab),"Tab");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrPerlVariable),"Variable");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrPerlOperator),"Operator");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrPerlBrace),"Brace");
	fe = new SSEXFontEditor(this,page,&(g_pSSEXOptions->fntPerl),"Font");

	addTab(page,"Perl/Kvirc script");

	page = new QVBox(this);
	page->setMargin(5);


	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrTxtBackground),"Background");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrTxtDisabledBackground),"Disabled background");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrTxtExtBackground),"External background");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrTxtMarkBackground),"Mark background");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrTxtNormalText),"Foreground");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrTxtCursor),"Cursor");

	fe = new SSEXFontEditor(this,page,&(g_pSSEXOptions->fntTxt),"Font");

	l = new QLabel("",page);

	page->setStretchFactor(l,1);

	addTab(page,"Plain text");

	page = new QVBox(this);
	page->setMargin(5);


	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppBackground),"Background");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppDisabledBackground),"Disabled background");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppMarkBackground),"Mark background");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppExtBackground),"External background");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppNormalText),"Normal text");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppCursor),"Cursor");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppTab),"Tab mark");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppBrace),"Brace");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppLineComment),"Line comment");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppMultilineComment),"Multiline comment");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppParenthesis),"Parenthesis");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppOperator),"Operator");
	addTab(page,"C/C++ (1)");

	page = new QVBox(this);
	page->setMargin(5);

	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppEscape),"Escape");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppChar),"Char");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppString),"String");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppKeyword),"Keyword");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppType),"Type");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppNumber),"Number");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppPunctuation),"Punctuation");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppSystemIdentifier),"System identifier");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppPreprocessor),"Preprocessor");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppInclude),"Include");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppMemberVariable),"Member variable");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppGlobalVariable),"Global variable");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppQClass),"Qt class (Q*)");

	addTab(page,"C/C++ (2)");

	page = new QVBox(this);
	page->setMargin(5);

	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppQSignals),"Qt signal");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppKClass),"KDE class (K*)");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppXStuff),"X stuff (X*)");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppGtkStruct),"Gtk struct");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppGdkStruct),"Gdk struct");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppGType),"Gtk type");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppGtkCall),"Gtk call");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppGdkCall),"Gdk call");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppGtkMacro),"Gtk macro");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppGdkMacro),"Gdk macro");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppGMacro),"G macro");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppSSStuff),"SS stuff (what's that ?)");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrCppSpecial),"Special");
	fe = new SSEXFontEditor(this,page,&(g_pSSEXOptions->fntCpp),"Font");


	addTab(page,"C/C++ (3)");

	page = new QVBox(this);
	page->setMargin(5);


	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrHtmlBackground),"Background");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrHtmlDisabledBackground),"Disabled background");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrHtmlExtBackground),"External background");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrHtmlMarkBackground),"Mark background");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrHtmlNormalText),"Normal text");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrHtmlCursor),"Cursor");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrHtmlComment),"Comment");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrHtmlTag),"Tag");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrHtmlString),"String");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrHtmlTagInternal),"Tag internal");
	fe = new SSEXFontEditor(this,page,&(g_pSSEXOptions->fntHtml),"Font");

	addTab(page,"HTML");

	page = new QVBox(this);
	page->setMargin(5);


	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrAutoconfBackground),"Background");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrAutoconfDisabledBackground),"Disabled background");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrAutoconfExtBackground),"External background");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrAutoconfMarkBackground),"Mark background");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrAutoconfNormalText),"Normal text");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrAutoconfCursor),"Cursor");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrAutoconfComment),"Comment");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrAutoconfDnlComment),"Dnl comment");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrAutoconfShellCommand),"Shell command");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrAutoconfBuiltinMacro),"Builtin bacro");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrAutoconfTab),"Tab");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrAutoconfVariable),"Variable");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrAutoconfOperator),"Operator");
	ed = new SSEXColorEditor(this,page,&(g_pSSEXOptions->clrAutoconfBrace),"Brace");
	fe = new SSEXFontEditor(this,page,&(g_pSSEXOptions->fntAutoconf),"Font");

	addTab(page,"Autoconf");

}

SSEXOptionsDialog::~SSEXOptionsDialog()
{
}

void SSEXOptionsDialog::showEvent(QShowEvent *e)
{
	move((QApplication::desktop()->width() - width()) / 2,(QApplication::desktop()->height() - height()) / 2);
	QDialog::showEvent(e);
}

void SSEXOptionsDialog::closeEvent(QCloseEvent *e)
{
	e->ignore();
	g_pSSEXOptions->optionsDialogTerminated();
}

void SSEXOptionsDialog::okClicked()
{
	emit doCommit();
	g_pSSEXOptions->optionsDialogTerminated();
	g_pSSEXOptions->apply();
}

void SSEXOptionsDialog::cancelClicked()
{
	g_pSSEXOptions->optionsDialogTerminated();
}



////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////
//// SSEXFindWidget
////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////


SSEXFindWidget::SSEXFindWidget(SSEXEditor * parent)
:QFrame(parent)
{
	setFrameStyle(QFrame::WinPanel | QFrame::Raised);
	m_pEditor = parent;
	setCursor(arrowCursor);

	QGridLayout *g = new QGridLayout(this,13,3,4,0);
	QToolButton *tb = new QToolButton(DownArrow,this);
	tb->setFixedSize(12,12);
	tb->setAutoRepeat(false);
	connect(tb,SIGNAL(clicked()),m_pEditor,SLOT(toggleFindWidget()));
	g->addWidget(tb,0,2);

	QFrame *f = new QFrame(this);
	f->setFrameStyle(QFrame::HLine | QFrame::Sunken);
	g->addMultiCellWidget(f,1,1,0,2);

	QLabel *l = new QLabel("String to find :",this);
	g->addMultiCellWidget(l,2,2,0,2);

	m_pFindStringEdit = new QLineEdit(this);
	g->addMultiCellWidget(m_pFindStringEdit,3,3,0,2);
	
	setFocusProxy(m_pFindStringEdit);

	m_pRegExpCheckBox = new QCheckBox("Regular expression",this);
	g->addMultiCellWidget(m_pRegExpCheckBox,4,4,0,2);

	m_pCaseSensitiveCheckBox = new QCheckBox("Case sensitive",this);
	m_pCaseSensitiveCheckBox->setChecked(true);
	g->addMultiCellWidget(m_pCaseSensitiveCheckBox,5,5,0,2);


	QPushButton *b = new QPushButton("Find next",this);
	b->setDefault(true);
	g->addWidget(b,6,0);
	connect(b,SIGNAL(clicked()),this,SLOT(findNextClicked()));

	b = new QPushButton("Find prev",this);
	g->addMultiCellWidget(b,6,6,1,2);
	connect(b,SIGNAL(clicked()),this,SLOT(findPrevClicked()));

	f = new QFrame(this);
	f->setFrameStyle(QFrame::HLine | QFrame::Sunken);
	g->addMultiCellWidget(f,7,7,0,2);

	l = new QLabel("Replacement text :",this);
	g->addMultiCellWidget(l,8,8,0,2);

	m_pReplaceStringEdit = new QLineEdit(this);
	g->addMultiCellWidget(m_pReplaceStringEdit,9,9,0,2);

	m_pReplace = new QPushButton("Replace",this);
	connect(m_pReplace,SIGNAL(clicked()),m_pEditor,SLOT(replace()));
	g->addWidget(m_pReplace,10,0);
	m_pReplace->setEnabled(m_pEditor->hasSelection());

	b = new QPushButton("Replace all",this);
	connect(b,SIGNAL(clicked()),this,SLOT(replaceAllClicked()));
	g->addMultiCellWidget(b,10,10,1,2);

	m_pReplaceAndFindNext = new QPushButton("Replace and find next",this);
	connect(m_pReplaceAndFindNext,SIGNAL(clicked()),this,SLOT(replaceAndFindNextClicked()));
	g->addMultiCellWidget(m_pReplaceAndFindNext,11,11,0,2);
	m_pReplaceAndFindNext->setEnabled(m_pEditor->hasSelection());

	m_pReplaceAllInSelection = new QPushButton("Replace all in selection",this);
	connect(m_pReplaceAllInSelection,SIGNAL(clicked()),this,SLOT(replaceAllInSelectionClicked()));
	g->addMultiCellWidget(m_pReplaceAllInSelection,12,12,0,2);
	m_pReplaceAllInSelection->setEnabled(m_pEditor->hasSelection());


	g->setResizeMode(QGridLayout::Fixed);
}

SSEXFindWidget::~SSEXFindWidget()
{
}

void SSEXFindWidget::findNextClicked()
{
	if(m_pRegExpCheckBox->isChecked())m_pEditor->findNextRegExp();
	else m_pEditor->findNext();
}

void SSEXFindWidget::findPrevClicked()
{
	if(m_pRegExpCheckBox->isChecked())m_pEditor->findPrevRegExp();
	else m_pEditor->findPrev();
}

void SSEXFindWidget::replaceAllClicked()
{
	if(m_pRegExpCheckBox->isChecked())m_pEditor->replaceAllRegExp();
	else m_pEditor->replaceAll();
}

void SSEXFindWidget::replaceAndFindNextClicked()
{
	if(m_pRegExpCheckBox->isChecked())m_pEditor->replaceAndFindNextRegExp();
	else m_pEditor->replaceAndFindNext();
}

void SSEXFindWidget::replaceAllInSelectionClicked()
{
	if(m_pRegExpCheckBox->isChecked())m_pEditor->replaceAllInSelectionRegExp();
	else m_pEditor->replaceAllInSelection();
}

void SSEXFindWidget::mousePressEvent(QMouseEvent *e)
{
	m_pressPoint = e->pos();
}

void SSEXFindWidget::mouseMoveEvent(QMouseEvent *)
{
	QPoint p=m_pEditor->mapFromGlobal(QCursor::pos());
	p-=m_pressPoint;
	if(p.x() < 0)p.setX(0);
	else if((p.x() + width()) > m_pEditor->width())p.setX(m_pEditor->width() - width());
	if(p.y() < 0)p.setY(0);
	else if((p.y() + height()) > m_pEditor->height())p.setY(m_pEditor->height() - height());
	move(p);
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////
//// SSEXEditorOptions
////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////





SSEXOptions::SSEXOptions()
{
	m_pRefList = new KviPtrList<SSEXEditor>;
	m_pRefList->setAutoDelete(false);
	m_pOptionsDialog = 0;
	load();
}

SSEXOptions::~SSEXOptions()
{
	save();
	if(m_pOptionsDialog)delete m_pOptionsDialog;
	delete m_pRefList;
}

void SSEXOptions::edit()
{
	if(!m_pOptionsDialog)
	{
		m_pOptionsDialog = new SSEXOptionsDialog();
	}
	m_pOptionsDialog->show();
	m_pOptionsDialog->raise();
	m_pOptionsDialog->setFocus();
}

void SSEXOptions::optionsDialogTerminated()
{
	delete m_pOptionsDialog;
	m_pOptionsDialog = 0;
}

void SSEXOptions::apply()
{
	for(SSEXEditor * e = m_pRefList->first();e;e = m_pRefList->next())
	{
		e->updateOptions();
	}
}

void SSEXOptions::attach(SSEXEditor * ed)
{
	if(!g_pSSEXOptions)g_pSSEXOptions = new SSEXOptions();
	g_pSSEXOptions->attachRef(ed);
}

void SSEXOptions::detach(SSEXEditor * ed)
{
	g_pSSEXOptions->detachRef(ed);
	if(!(g_pSSEXOptions->hasRef()))
	{
		delete g_pSSEXOptions;
		g_pSSEXOptions = 0;
	}
}

void SSEXOptions::initDefaults()
{
	clrTxtBackground          = QColor(0,0,0);
	clrTxtDisabledBackground  = QColor(40,40,40);
	clrTxtExtBackground       = QColor(40,40,40);
	clrTxtMarkBackground      = QColor(25,25,45);
	clrTxtNormalText          = QColor(50,255,0);
	clrTxtCursor              = QColor(255,0,0);
	fntTxt                    = QFont("fixed",11);

	clrCppBackground          = QColor(0,0,0);
	clrCppDisabledBackground  = QColor(40,40,40);
	clrCppMarkBackground      = QColor(25,25,45);
	clrCppExtBackground       = QColor(40,40,40);
	clrCppNormalText          = QColor(80,255,0);
	clrCppCursor              = QColor(255,0,0);
	clrCppTab                 = QColor(70,70,70);
	clrCppBrace               = QColor(255,0,0);
	clrCppLineComment         = QColor(40,150,0);
	clrCppMultilineComment    = QColor(20,180,0);
	clrCppParenthesis         = QColor(170,130,130);
	clrCppOperator            = QColor(150,150,40);
	clrCppEscape              = QColor(50,130,240);
	clrCppChar                = QColor(100,140,250);
	clrCppString              = QColor(80,170,250);
	clrCppKeyword             = QColor(130,130,130);
	clrCppType                = QColor(160,160,160);
	clrCppNumber              = QColor(190,200,255);
	clrCppPunctuation         = QColor(180,180,50);
	clrCppSystemIdentifier    = QColor(255,0,255);
	clrCppPreprocessor        = QColor(255,255,255);
	clrCppInclude             = QColor(200,200,200);
	clrCppMemberVariable      = QColor(190,170,80);
	clrCppGlobalVariable      = QColor(230,200,110);
	clrCppQClass              = QColor(255,255,50);
	clrCppQSignals            = QColor(255,150,0);
	clrCppKClass              = QColor(255,255,0);
	clrCppXStuff              = QColor(255,255,90);
	clrCppGtkStruct           = QColor(255,255,50);
	clrCppGdkStruct           = QColor(255,205,90);
	clrCppGType               = QColor(190,190,190);
	clrCppGtkCall             = QColor(150,150,180);
	clrCppGdkCall             = QColor(150,120,180);
	clrCppGtkMacro            = QColor(220,170,180);
	clrCppGdkMacro            = QColor(230,170,160);
	clrCppGMacro              = QColor(240,170,140);
	clrCppSSStuff             = QColor(255,200,0);
	clrCppSpecial             = QColor(150,150,190);
	fntCpp                    = QFont("fixed",11);

	clrHtmlBackground         = QColor(0,0,0);
	clrHtmlDisabledBackground = QColor(40,40,40);
	clrHtmlExtBackground      = QColor(40,40,40);
	clrHtmlMarkBackground     = QColor(25,25,45);
	clrHtmlNormalText         = QColor(20,255,20);
	clrHtmlCursor             = QColor(255,0,0);
	clrHtmlComment            = QColor(35,180,0);
	clrHtmlTag                = QColor(180,100,30);
	clrHtmlString             = QColor(40,180,255);
	clrHtmlTagInternal        = QColor(180,150,20);
	fntHtml                   = QFont("fixed",11);

	clrPerlBackground         = QColor(0,0,0);
	clrPerlDisabledBackground = QColor(40,40,40);
	clrPerlExtBackground      = QColor(40,40,40);
	clrPerlMarkBackground     = QColor(25,25,45);
	clrPerlNormalText         = QColor(20,255,20);
	clrPerlCursor             = QColor(255,0,0);
	clrPerlComment            = QColor(35,180,0);
	clrPerlTab                = QColor(70,70,70);
	clrPerlVariable           = QColor(255,255,50);
	clrPerlOperator           = QColor(150,150,40);
	clrPerlBrace              = QColor(255,0,0);
	fntPerl                   = QFont("fixed",11);

	clrAutoconfBackground         = QColor(0,0,0);
	clrAutoconfDisabledBackground = QColor(40,40,40);
	clrAutoconfExtBackground      = QColor(40,40,40);
	clrAutoconfMarkBackground     = QColor(25,25,45);
	clrAutoconfNormalText         = QColor(20,255,20);
	clrAutoconfCursor             = QColor(255,0,0);
	clrAutoconfComment            = QColor(40,150,0);
	clrAutoconfDnlComment         = QColor(20,180,0);
	clrAutoconfShellCommand       = QColor(130,130,130);
	clrAutoconfBuiltinMacro       = QColor(130,100,200);
	clrAutoconfTab                = QColor(70,70,70);
	clrAutoconfVariable           = QColor(255,255,50);
	clrAutoconfOperator           = QColor(150,150,40);
	clrAutoconfBrace              = QColor(255,0,0);
	fntAutoconf                   = QFont("fixed",11);

	uUndoDepth                = 256;
	uTabsNumSpaces            = 4;
}

void SSEXOptions::load()
{
	QString thedir = QDir::homeDirPath();
	thedir += "/.ssexeditorrc";
	QFile f(thedir.ascii());
	if(!f.open(IO_ReadOnly))
	{
		initDefaults();
		return;
	}

	QDataStream s(&f);


	s >> clrTxtBackground;
	s >> clrTxtDisabledBackground;
	s >> clrTxtExtBackground;
	s >> clrTxtMarkBackground;
	s >> clrTxtNormalText;
	s >> clrTxtCursor;
	s >> fntTxt;

	s >> clrCppBackground;
	s >> clrCppDisabledBackground;
	s >> clrCppMarkBackground;
	s >> clrCppExtBackground;
	s >> clrCppNormalText;
	s >> clrCppCursor;
	s >> clrCppTab;
	s >> clrCppBrace;
	s >> clrCppLineComment;
	s >> clrCppMultilineComment;
	s >> clrCppParenthesis;
	s >> clrCppOperator;
	s >> clrCppEscape;
	s >> clrCppChar;
	s >> clrCppString;
	s >> clrCppKeyword;
	s >> clrCppType;
	s >> clrCppNumber;
	s >> clrCppPunctuation;
	s >> clrCppSystemIdentifier;
	s >> clrCppPreprocessor;
	s >> clrCppInclude;
	s >> clrCppMemberVariable;
	s >> clrCppGlobalVariable;
	s >> clrCppQClass;
	s >> clrCppQSignals;
	s >> clrCppKClass;
	s >> clrCppXStuff;
	s >> clrCppGtkStruct;
	s >> clrCppGdkStruct;
	s >> clrCppGType;
	s >> clrCppGtkCall;
	s >> clrCppGdkCall;
	s >> clrCppGtkMacro;
	s >> clrCppGdkMacro;
	s >> clrCppGMacro;
	s >> clrCppSSStuff;
	s >> clrCppSpecial;
	s >> fntCpp;

	s >> clrHtmlBackground;
	s >> clrHtmlDisabledBackground;
	s >> clrHtmlExtBackground;
	s >> clrHtmlMarkBackground;
	s >> clrHtmlNormalText;
	s >> clrHtmlCursor;
	s >> clrHtmlComment;
	s >> clrHtmlTag;
	s >> clrHtmlString;
	s >> clrHtmlTagInternal;
	s >> fntHtml;

	s >> clrPerlBackground;
	s >> clrPerlDisabledBackground;
	s >> clrPerlExtBackground;
	s >> clrPerlMarkBackground;
	s >> clrPerlNormalText;
	s >> clrPerlCursor;
	s >> clrPerlComment;
	s >> clrPerlTab;
	s >> clrPerlVariable;
	s >> clrPerlOperator;
	s >> clrPerlBrace;
	s >> fntPerl;

	s >> clrAutoconfBackground;
	s >> clrAutoconfDisabledBackground;
	s >> clrAutoconfExtBackground;
	s >> clrAutoconfMarkBackground;
	s >> clrAutoconfNormalText;
	s >> clrAutoconfCursor;
	s >> clrAutoconfComment;
	s >> clrAutoconfDnlComment;
	s >> clrAutoconfShellCommand;
	s >> clrAutoconfBuiltinMacro;
	s >> clrAutoconfTab;
	s >> clrAutoconfVariable;
	s >> clrAutoconfOperator;
	s >> clrAutoconfBrace;
	s >> fntAutoconf;

	s >> uUndoDepth;
	s >> uTabsNumSpaces;


	f.close();
}

void SSEXOptions::save()
{
	QString thedir = QDir::homeDirPath();
	thedir += "/.ssexeditorrc";
	QFile f(thedir.ascii());
	if(!f.open(IO_WriteOnly))return;

	QDataStream s(&f);

	s << clrTxtBackground;
	s << clrTxtDisabledBackground;
	s << clrTxtExtBackground;
	s << clrTxtMarkBackground;
	s << clrTxtNormalText;
	s << clrTxtCursor;
	s << fntTxt;

	s << clrCppBackground;
	s << clrCppDisabledBackground;
	s << clrCppMarkBackground;
	s << clrCppExtBackground;
	s << clrCppNormalText;
	s << clrCppCursor;
	s << clrCppTab;
	s << clrCppBrace;
	s << clrCppLineComment;
	s << clrCppMultilineComment;
	s << clrCppParenthesis;
	s << clrCppOperator;
	s << clrCppEscape;
	s << clrCppChar;
	s << clrCppString;
	s << clrCppKeyword;
	s << clrCppType;
	s << clrCppNumber;
	s << clrCppPunctuation;
	s << clrCppSystemIdentifier;
	s << clrCppPreprocessor;
	s << clrCppInclude;
	s << clrCppMemberVariable;
	s << clrCppGlobalVariable;
	s << clrCppQClass;
	s << clrCppQSignals;
	s << clrCppKClass;
	s << clrCppXStuff;
	s << clrCppGtkStruct;
	s << clrCppGdkStruct;
	s << clrCppGType;
	s << clrCppGtkCall;
	s << clrCppGdkCall;
	s << clrCppGtkMacro;
	s << clrCppGdkMacro;
	s << clrCppGMacro;
	s << clrCppSSStuff;
	s << clrCppSpecial;
	s << fntCpp;

	s << clrHtmlBackground;
	s << clrHtmlDisabledBackground;
	s << clrHtmlExtBackground;
	s << clrHtmlMarkBackground;
	s << clrHtmlNormalText;
	s << clrHtmlCursor;
	s << clrHtmlComment;
	s << clrHtmlTag;
	s << clrHtmlString;
	s << clrHtmlTagInternal;
	s << fntHtml;

	s << clrPerlBackground;
	s << clrPerlDisabledBackground;
	s << clrPerlExtBackground;
	s << clrPerlMarkBackground;
	s << clrPerlNormalText;
	s << clrPerlCursor;
	s << clrPerlComment;
	s << clrPerlTab;
	s << clrPerlVariable;
	s << clrPerlOperator;
	s << clrPerlBrace;
	s << fntPerl;

	s << clrAutoconfBackground;
	s << clrAutoconfDisabledBackground;
	s << clrAutoconfExtBackground;
	s << clrAutoconfMarkBackground;
	s << clrAutoconfNormalText;
	s << clrAutoconfCursor;
	s << clrAutoconfComment;
	s << clrAutoconfDnlComment;
	s << clrAutoconfShellCommand;
	s << clrAutoconfBuiltinMacro;
	s << clrAutoconfTab;
	s << clrAutoconfVariable;
	s << clrAutoconfOperator;
	s << clrAutoconfBrace;
	s << fntAutoconf;

	s << uUndoDepth;
	s << uTabsNumSpaces;

	f.close();

}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////
//// SSEXEditor
////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////



#define SSEX_EDITOR_BORDER 5
#define SSEX_EDITOR_BLINK_TIME 500
#define SSEX_EDITOR_DRAG_TIMEOUT 150

SSEXEditor::SSEXEditor(QWidget * parent)
:SSEXTableView(parent)
{
	SSEXOptions::attach(this);

	setFrameStyle(QFrame::Panel|QFrame::Sunken);
	setNumRows(1);

	setCursor(ibeamCursor);

	m_pLines = new KviPtrList<SSEXEditorTextLine>;
	m_pLines->setAutoDelete(true);

	m_mode = Normal;

	m_iCursorRow              = 0;
	m_iCursorPosition         = 0;
	m_iCursorPositionInPixels = 0;
	m_iLastCursorRow          = 0;
	m_iLastCursorPosition     = 0;

	m_iMaxTextWidth = 0;
	m_pMemBuffer = new QPixmap(cellWidth() + 1,cellHeight() + 1);
	m_pBlinkTimer = new QTimer();
	m_bCursorOn = false;
	m_bOverwrite = false;

	m_bRecordingKeystrokes = false;
	m_pKeystrokes = new KviPtrList<SSEXEditorKeystroke>;
	m_pKeystrokes->setAutoDelete(true);

	m_bHasSelection = false;

	m_bHasBlockMark = false;

	m_pContextPopup = new QPopupMenu(this);
	m_pFindPopup = new QPopupMenu(this);
	connect(m_pFindPopup,SIGNAL(aboutToShow()),this,SLOT(findPopupAboutToShow()));
	m_pEditPopup = new QPopupMenu(this);
	connect(m_pEditPopup,SIGNAL(aboutToShow()),this,SLOT(editPopupAboutToShow()));

	m_pUndoList = new KviPtrList<SSEXUndo>;
	m_pUndoList->setAutoDelete(true);

	connect(m_pBlinkTimer,SIGNAL(timeout()),this,SLOT(blinkTimer()));

	m_bModified = false;
	m_pDragTimer = new QTimer();
	connect(m_pDragTimer,SIGNAL(timeout()),this,SLOT(dragTimer()));

	updateFontDependantVariables();
	updateCellSize();

	setBackgroundMode(NoBackground);

	setFocusPolicy(QWidget::StrongFocus);

	m_iBlinkTime = SSEX_EDITOR_BLINK_TIME;
	m_szFileName = "";

	m_pFindWidget = new SSEXFindWidget(this);
	m_pFindWidget->hide();

	m_lastFindWidgetPosition = QPoint(20,20);

	initializeCurrentMode();
	setText("");

}

SSEXEditor::~SSEXEditor()
{
	closeFile(); // cancel does nothing here

	delete m_pFindWidget;
	delete m_pLines;
	delete m_pMemBuffer;
	delete m_pBlinkTimer;
	delete m_pDragTimer;
	delete m_pKeystrokes;
	delete m_pUndoList;

	SSEXOptions::detach(this);
}

static const char * m_modeTable[]=
{
	"C/C++/Java",
	"Plain text",
	"Html",
	"Perl",
	"Autoconf"
};

const char * SSEXEditor::modeName()
{
	switch(m_mode)
	{
		case Cpp: return m_modeTable[0]; break;
		case Normal: return m_modeTable[1]; break;
		case Html: return m_modeTable[2]; break;
		case Perl: return m_modeTable[3]; break;
		case Autoconf: return m_modeTable[4]; break;
		default: return m_modeTable[1]; break;
	}
}


bool SSEXEditor::closeFile()
{
	if(m_bModified)
	{
		QString msg;
		msg.sprintf("The file %s has been modified.\nDo you want to save your changes ?",m_szFileName.isEmpty() ? "unnamed" : m_szFileName.data());
		int ret = QMessageBox::warning(this,"Warning",msg,"Save","Don't save","Cancel");
		if(ret == 0){
			if(!saveFile())return closeFile(); //cancelled after this...repeat plz...
			else return true; //saved
		} else if(ret == 1){
			m_bModified = false; // don't call closeFile in destructor
			return true;
		} else return false; //cancelled;
	}
	return true; // safe to close
}


void SSEXEditor::getText(QCString &buffer)
{
	buffer = "";
	for(SSEXEditorTextLine *l=m_pLines->first();l;l=m_pLines->next())
	{
		buffer.append(l->text);
		buffer.append("\n");
	}
}

void SSEXEditor::markBlock()
{
	if(m_bHasBlockMark)clearBlockMark(false);

	int iPos = m_iCursorPosition;
	int iRow = m_iCursorRow;


	char chFound = 0;
	char chMatch = 0;

	int leftNormalPar = 0;
	int leftSquarePar = 0;
	int leftBrackets  = 0;
	int leftSquares   = 0;
	

	SSEXEditorTextLine * l = m_pLines->at(iRow);

	if(iPos > l->length)iPos = l->length;

	while((chFound == 0) && iRow >= 0)
	{
		iPos--;
		if(iPos < 0)
		{
			iRow--;
			if(iRow >= 0)
			{
				l = m_pLines->at(iRow);
				iPos = l->length - 1;
			}
		}
		if(iPos > -1)
		{
			switch(l->text.at(iPos))
			{
				case ')': leftNormalPar++; break;
				case '(': if(leftNormalPar == 0){ chFound = '('; chMatch = ')'; } else leftNormalPar--; break;
				case ']': leftSquarePar++; break;
				case '[': if(leftSquarePar == 0){ chFound = '['; chMatch = ']'; }else leftSquarePar--; break;
				case '}': leftBrackets++; break;
				case '{': if(leftBrackets == 0){ chFound = '{'; chMatch = '}'; } else leftBrackets--; break;
				case '>': if(m_mode == Html)leftSquares++; break;
				case '<': if(m_mode == Html){ if(leftSquares == 0){ chFound = '<'; chMatch = '>'; } else leftSquares--; } break;
			}
		}
	}

	if(chFound)
	{
//		debug("Found left end");

		// Ok..we have the left end
		m_blockMark1 = QPoint(iPos,iRow);
		// Now do the same in the right direction

		iPos = m_iCursorPosition;
		iRow = m_iCursorRow;

		
		l = m_pLines->at(iRow);
	

		while((chFound != 0) && (iRow < numRows()))
		{
			if(iPos < l->length)
			{
				char c = l->text.at(iPos);
				if(c == chFound)leftNormalPar++;
				else if(c == chMatch)
				{
					if(leftNormalPar == 0)chFound = 0;
					else leftNormalPar--;
				}
			}
			iPos++;
			if((iPos >= l->length) && (chFound != 0))
			{
				iPos = 0;
				iRow++;
				if(iRow < numRows())l = m_pLines->at(iRow);
			}
		}

		if(chFound == 0)
		{
//			debug("Found right end");
			// Ok..have the whole mark
			m_blockMark2 = QPoint(iPos,iRow);
			m_bHasBlockMark = true;
		}

	}

	update();
}

void SSEXEditor::clearBlockMark(bool bUpdate)
{
	if(!m_bHasBlockMark)return;
	m_bHasBlockMark = false;
	if(bUpdate)update();
}
/*
void SSEXEditor::setColors(SSEXEditorColors *clrs)
{
	if(clrs)
	{
		if(clrs != m_pColors)
		{
			delete m_pColors;
			m_pColors = clrs;
			initializeCurrentMode();
			update();
		}
	}
}
*/
void SSEXEditor::toggleFindWidget()
{
	if(m_pFindWidget->isVisible())
	{
		m_lastFindWidgetPosition = m_pFindWidget->pos();
		m_pFindWidget->hide();
		setFocus();
	} else {
		if((m_lastFindWidgetPosition.x() + m_pFindWidget->width()) > width())m_lastFindWidgetPosition.setX(width() - m_pFindWidget->width());
		if((m_lastFindWidgetPosition.y() + m_pFindWidget->height()) > height())m_lastFindWidgetPosition.setY(height() - m_pFindWidget->height());
		m_pFindWidget->move(m_lastFindWidgetPosition);
		m_pFindWidget->show();
		m_pFindWidget->m_pFindStringEdit->setFocus();
	}
}

void SSEXEditor::setModified(bool bModified)
{
	if(bModified != m_bModified)
	{
		m_bModified = bModified;
		emit modifyStateChanged(this,m_bModified);
	}
}

void SSEXEditor::setHasSelection(bool bHasSelection)
{
	m_bHasSelection = bHasSelection;
	if(m_pFindWidget)
	{
		m_pFindWidget->m_pReplace->setEnabled(bHasSelection);
		m_pFindWidget->m_pReplaceAndFindNext->setEnabled(bHasSelection);
		m_pFindWidget->m_pReplaceAllInSelection->setEnabled(bHasSelection);
	}
}

void SSEXEditor::blinkTimer()
{
	m_bCursorOn = !m_bCursorOn;
	updateCell(m_iCursorRow);
}

void SSEXEditor::setBlinkTime(int blinkTime)
{
	if(blinkTime < 100)blinkTime = 100;
	if(blinkTime > 10000)blinkTime = 10000;
	m_iBlinkTime = 100;
	if(hasFocus() && isEnabled())startBlinking();
}

void SSEXEditor::startBlinking()
{
	if(m_pBlinkTimer->isActive())m_pBlinkTimer->stop();
	m_bCursorOn = true;
	m_pBlinkTimer->start(m_iBlinkTime);	
}


void SSEXEditor::stopBlinking()
{
	if(m_pBlinkTimer->isActive())m_pBlinkTimer->stop();
	if(m_bCursorOn){
		m_bCursorOn = false;
		updateCell(m_iCursorRow);
	}
}

void SSEXEditor::focusInEvent(QFocusEvent *e)
{
	if(isEnabled())startBlinking();
	SSEXTableView::focusInEvent(e);
	emit gainedFocus(this);
}

void SSEXEditor::focusOutEvent(QFocusEvent *e)
{
	stopBlinking();
	SSEXTableView::focusOutEvent(e);
}

void SSEXEditor::setEnabled(bool bEnabled)
{
	SSEXTableView::setEnabled(bEnabled);
	if(bEnabled && hasFocus())startBlinking();
}

/*
void SSEXEditor::setTabsNumSpaces(int spaces)
{
	if(spaces < 1)spaces = 1;
	if(spaces > 64)spaces = 64;
	g_pSSEXOptions->uTabsNumSpaces = spaces;
	updateFontDependantVariables();
	updateMaxTextWidth();
	updateCellSize();
	recalcCursorPosition(m_pLines->at(m_iCursorRow));
	update();
}
*/
void SSEXEditor::updateMaxTextWidth()
{
	m_iMaxTextWidth = 0;
	int line = 0;
	for(SSEXEditorTextLine * l=m_pLines->first();l;l=m_pLines->next())
	{
		if(l->width > m_iMaxTextWidth)
		{
			m_iMaxTextWidth = l->width;
			m_iMaxTextWidthLine = line;
		}
		line++;
	}
}

//////////////////////////////////////////////////////////////////////////////////////////////////
//
// PAINT
//
//////////////////////////////////////////////////////////////////////////////////////////////////


void SSEXEditor::paintCell(QPainter * p,int row)
{
	SSEXEditorTextLine * l = m_pLines->at(row);
	if(!l)return;

	__assert(l->text.length() == l->length);
	
	switch(m_mode)
	{
		case Normal:
			paintCellNormal(p,l,row);
			break;
		case Cpp:
			paintCellCpp(p,l,row);
			break;
		case Html:
			paintCellHtml(p,l,row);
			break;
		case Perl:
			paintCellPerl(p,l,row);
			break;
		case Autoconf:
			paintCellAutoconf(p,l,row);
			break;
	}
}

void SSEXEditor::paintBlockMark(QPainter * p,SSEXEditorTextLine * l,int row,const QRect & updateR,QColor * pColor)
{
	if((m_blockMark1.y() <= row) && (m_blockMark2.y() >= row))
	{
		int selectionBeginX = SSEX_EDITOR_BORDER;
		int selectionEndX = SSEX_EDITOR_BORDER + cellWidth();
		if(m_blockMark1.y() == row)
		{
			// the selection begins in this row
			selectionBeginX += getTextWidthWithTabsForCursorPosition(l->text.data(),m_blockMark1.x());
		}
		if(m_blockMark2.y() == row)
		{
			// the selection ends in this row
			selectionEndX = SSEX_EDITOR_BORDER + getTextWidthWithTabsForCursorPosition(l->text.data(),m_blockMark2.x());
		}
		p->fillRect(selectionBeginX,updateR.y(),selectionEndX - selectionBeginX,updateR.height(),*pColor);
	}
}

void SSEXEditor::paintCursor(QPainter * p,SSEXEditorTextLine *l,QColor * pColor)
{
	if(!isEnabled())return;

	if((m_iCursorRow != m_iLastCursorRow) || (m_iCursorPosition != m_iLastCursorPosition))
	{
		m_iLastCursorRow = m_iCursorRow;
		m_iLastCursorPosition = m_iCursorPosition;
//		debug("Cursor pos = %d,%d",m_iCursorRow,m_iCursorPosition);
		emit cursorPositionChanged(this,m_iCursorRow,(m_iCursorPosition < l->length) ? m_iCursorPosition : l->length);
	}
	if(!m_bCursorOn)return;

	int curXPos = m_iCursorPositionInPixels + SSEX_EDITOR_BORDER - 1;

	p->setPen(*pColor);

	int rowLowPos = cellHeight() - 1;

	p->drawLine(curXPos,0,curXPos,rowLowPos);
	p->drawLine(curXPos + 1,0,curXPos + 1,rowLowPos);

	p->drawLine(curXPos - 3,0,curXPos + 4,0);
	p->drawLine(curXPos - 3,rowLowPos,curXPos + 4,rowLowPos);
}


void SSEXEditor::paintSelection(QPainter * p,SSEXEditorTextLine *l,int row,QRect &updateR)
{
	if((m_selection1.y() <= row) && (m_selection2.y() >= row))
	{
		int selectionBeginX = SSEX_EDITOR_BORDER;
		int selectionEndX = SSEX_EDITOR_BORDER;
		if(m_selection1.y() == row)
		{
			// the selection begins in this row
			selectionBeginX += getTextWidthWithTabsForCursorPosition(l->text.data(),m_selection1.x());
		}
		if(m_selection2.y() == row)
		{
			// the selection ends in this row
			selectionEndX += getTextWidthWithTabsForCursorPosition(l->text.data(),m_selection2.x());
		} else {
			// selection ends below
			selectionEndX += getTextWidthWithTabsForCursorPosition(l->text.data(),l->length);
		}
		p->setRasterOp(NotROP);
		p->fillRect(selectionBeginX,updateR.y(),selectionEndX - selectionBeginX,updateR.height(),Qt::black);
		p->setRasterOp(CopyROP);
	}
}


void SSEXEditor::paintCellNormal(QPainter * p,SSEXEditorTextLine * l,int row)
{
	QRect updateR = cellUpdateRect();

	QPainter paint(m_pMemBuffer);

	paint.fillRect(updateR,isEnabled() ? g_pSSEXOptions->clrTxtBackground : g_pSSEXOptions->clrTxtDisabledBackground);
	if(m_bHasBlockMark)paintBlockMark(&paint,l,row,updateR,&(g_pSSEXOptions->clrTxtMarkBackground));

	paint.setPen(g_pSSEXOptions->clrTxtNormalText);
	paint.setFont(p->font());

	const char * c = l->text.data();
	int curXPos = SSEX_EDITOR_BORDER;
	int lastTabPos = SSEX_EDITOR_BORDER;

	while(*c)
	{
		if(*c == '\t')
		{
			// only a tab
			while(lastTabPos <= curXPos)lastTabPos += m_iTabsNumPixels;
			curXPos = lastTabPos;
			c++;
		} else {
			// text block
			const char * begin = c;
			int blockWidth = 0;
			while(*c && *c != '\t'){
				blockWidth += m_iCharWidth[(unsigned char)*c];
				c++;
			}
			paint.drawText(curXPos,m_iFontAscent,begin,c - begin);
			curXPos += blockWidth;
		}
	}

	if(m_iCursorRow == row)paintCursor(&paint,l,&(g_pSSEXOptions->clrTxtCursor));
	if(m_bHasSelection)paintSelection(&paint,l,row,updateR);

	int col_xPos,row_yPos;

	p->worldMatrix().map(0,0,&col_xPos,&row_yPos);

	bitBlt(p->device(),col_xPos + updateR.x(),row_yPos + updateR.y(),m_pMemBuffer,updateR.x(),updateR.y(),updateR.width(),updateR.height());

}


bool cmp(const char *token1,const char *token2,int len1,int len2)
{
	if(len1 != len2)return false;
	return (strncmp(token1,token2,len2) == 0);
}



bool cmpnocase(const char *token1,const char *token2,int len1,int len2)
{
	if(len1 != len2)return false;
	return (strncasecmp(token1,token2,len2) == 0);
}

bool cmpnolen(const char *token1,const char *token2,int len)
{
	return (strncmp(token1,token2,len) == 0);
}

void SSEXEditor::paintCellCpp(QPainter * p,SSEXEditorTextLine * l,int row)
{
	QPainter paint(m_pMemBuffer);
	paint.setFont(font());

	QRect updateR = cellUpdateRect();

	paint.fillRect(updateR,isEnabled() ? g_pSSEXOptions->clrCppBackground : g_pSSEXOptions->clrCppDisabledBackground);
	if(m_bHasBlockMark)paintBlockMark(&paint,l,row,updateR,&(g_pSSEXOptions->clrCppMarkBackground));

	const char * c = l->text.data();
	int curXPos = SSEX_EDITOR_BORDER;
	int lastTabPos = SSEX_EDITOR_BORDER;

	const char * begin;
	QColor * pClr;
	int blockWidth;

	bool bInLineComment = false;
	bool bInString = false;
	bool bInChar = false;
	bool bInMultilineComment = l->flags & SSEX_EDITOR_BEGIN_IN_COMMENT;
	bool bPreprocessor = false;
	bool bGotInclude = false;

	while(*c)
	{
		switch(*c)
		{
			case '\t':
				// a tab character
				while(lastTabPos <= curXPos)lastTabPos += m_iTabsNumPixels;
				curXPos = lastTabPos;
				paint.setPen(QPen(g_pSSEXOptions->clrCppTab,0,DashLine));
				paint.drawLine(curXPos - m_iTabsNumPixels,0,curXPos - m_iTabsNumPixels,m_iFontAscent);
				c++;
				break;
			case ' ':
				// a set of spaces
				begin = c;
				while(*c == ' ')c++;
				curXPos += (m_iCharWidth[(unsigned char)' '] * (c - begin));
//				debug("LIne %d, a set of %d spaces brings us to %d",row,c - begin,curXPos);
				break;
			default:
				begin      = c;
				blockWidth = 0;
				{
					//
					// EXTRACT TOKEN ####################################################################################
					//
					if(bInLineComment)
					{
						// We are in a line comment...up to the end of the line!
						while((*c) && (*c != ' ') && (*c != '\t'))
						{
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
						}
						pClr = &(g_pSSEXOptions->clrCppLineComment);

					} else if(bInMultilineComment)
					{
						// Multiline comment
						while((*c) && (*c != ' ') && (*c != '\t') && bInMultilineComment)
						{
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							if(*c == '*')
							{
								c++;
								if(*c == '/'){
									// End of comment!!!
									blockWidth += m_iCharWidth[(unsigned char)'/'];
									c++;
									bInMultilineComment = false;
								}
							} else c++;
						}
						pClr = &(g_pSSEXOptions->clrCppMultilineComment);

					} else if(bInString)
					{
						// Inside a string
						if(*c == '\\'){
							// escape sequence inside a string
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
							if((*c) && (*c != ' ') && (*c != '\t'))
							{
								blockWidth += m_iCharWidth[(unsigned char)(*c)];
								c++;
								pClr = &(g_pSSEXOptions->clrCppEscape);
							} else pClr = &(g_pSSEXOptions->clrCppString);
						} else {
							while((*c) && (*c != ' ') && (*c != '\t') && (*c != '\\') && bInString)
							{
								blockWidth += m_iCharWidth[(unsigned char)(*c)];
								if(*c == '"')
								{
									c++;
									bInString = false;
								} else c++;
							}
							pClr = &(g_pSSEXOptions->clrCppString);
						}
					} else if(bInChar)
					{
						// Inside a char constant
						if(*c == '\\'){
							// escape sequence inside a string
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
							if((*c) && (*c != ' ') && (*c != '\t'))
							{
								blockWidth += m_iCharWidth[(unsigned char)(*c)];
								c++;
								pClr = &(g_pSSEXOptions->clrCppEscape);
							} else pClr = &(g_pSSEXOptions->clrCppChar);
						} else {
							while((*c) && (*c != ' ') && (*c != '\t') && (*c != '\\') && bInChar)
							{
								blockWidth += m_iCharWidth[(unsigned char)(*c)];
								if(*c == '\'')
								{
									c++;
									bInChar = false;
								} else c++;
							}
							pClr = &(g_pSSEXOptions->clrCppChar);
						}
					} else if(bGotInclude){
						bGotInclude = false;
						while((*c) && (*c != ' ') && (*c != '\t'))
						{
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
						}
						pClr = &(g_pSSEXOptions->clrCppInclude);
					} else if(	((*c >= 'a') && (*c <= 'z')) ||
								((*c >= 'A') && (*c <= 'Z')) ||
								(*c == '_'))
					{
						// A text token
						blockWidth += m_iCharWidth[(unsigned char)(*c)];
						c++;
						while(	((*c >= 'a') && (*c <= 'z')) ||
								((*c >= 'A') && (*c <= 'Z')) ||
								((*c >= '0') && (*c <= '9')) ||
								(*c == '_'))
						{
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
						}
						if(bPreprocessor){
							pClr = &(g_pSSEXOptions->clrCppPreprocessor);
							bPreprocessor = false;
							if(cmp(begin,"include",7,c - begin))bGotInclude = true;
						} else pClr = cppModeGetTokenColor(begin,c - begin);
					} else if(	(*c == '(') || (*c == ')') ||
								(*c == '[') || (*c == ']'))
					{
						// A parenthesis
						blockWidth += m_iCharWidth[(unsigned char)(*c)];
						c++;
						while(	(*c == '(') || (*c == ')') ||
								(*c == '[') || (*c == ']'))
						{
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
						}
						pClr = &(g_pSSEXOptions->clrCppParenthesis);
					} else if(  (*c == ':'))
					{
						blockWidth += m_iCharWidth[(unsigned char)(*c)];
						c++;
						if(*c == ':'){
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
						}
						pClr = &(g_pSSEXOptions->clrCppPunctuation);
					} else if(	(*c == ';') || (*c == ',') || (*c == '.'))
					{
						blockWidth += m_iCharWidth[(unsigned char)(*c)];
						c++;
						pClr = &(g_pSSEXOptions->clrCppPunctuation);
					} else if(	(*c >= '0') && (*c <= '9'))
					{
						// A parenthesis
						blockWidth += m_iCharWidth[(unsigned char)(*c)];
						c++;
						while(	((*c >= '0') && (*c <= '9')) ||
								((*c >= 'a') && (*c <= 'z')) ||
								((*c >= 'A') && (*c <= 'Z')))
						{
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
						}
						pClr = &(g_pSSEXOptions->clrCppNumber);
					} else if(	*c == '#')
					{
						// A preprocessor code
						blockWidth += m_iCharWidth[(unsigned char)(*c)];
						c++;
						bPreprocessor = true;
						pClr = &(g_pSSEXOptions->clrCppPreprocessor);
					} else if(	(*c == '+') || (*c == '-') ||
								(*c == '*') || (*c == '?') ||
								(*c == '!') || (*c == '<') ||
								(*c == '>') || (*c == '^') ||
								(*c == '%') || (*c == '&') ||
								(*c == '|') || (*c == '=') ||
								(*c == '~') || (*c == '\\'))
					{
						// A parenthesis
						blockWidth += m_iCharWidth[(unsigned char)(*c)];
						c++;
						while(	(*c == '+') || (*c == '-') ||
								(*c == '*') || (*c == '?') ||
								(*c == '!') || (*c == '<') ||
								(*c == '>') || (*c == '^') ||
								(*c == '%') || (*c == '&') ||
								(*c == '|') || (*c == '=') ||
								(*c == '~') || (*c == '\\'))
						{
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
						}
						pClr = &(g_pSSEXOptions->clrCppOperator);
					} else if( (*c == '{') || (*c == '}') )
					{
						// brace
						blockWidth += m_iCharWidth[(unsigned char)(*c)];
						c++;
						pClr = &(g_pSSEXOptions->clrCppBrace);
					} else if( (*c == '"'))
					{
						// string
						blockWidth += m_iCharWidth[(unsigned char)(*c)];
						c++;
						bInString = true;
						pClr = &(g_pSSEXOptions->clrCppString);
					} else if( (*c == '\''))
					{
						// string
						blockWidth += m_iCharWidth[(unsigned char)(*c)];
						c++;
						bInChar = true;
						pClr = &(g_pSSEXOptions->clrCppChar);
					} else if( (*c == '/'))
					{
						// Comment begin ?
						blockWidth += m_iCharWidth[(unsigned char)(*c)];
						c++;
						if(*c == '/'){
							bInLineComment = true;
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
							pClr = &(g_pSSEXOptions->clrCppLineComment);
						} else if(*c == '*'){
							bInMultilineComment = true;
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
							pClr = &(g_pSSEXOptions->clrCppMultilineComment);
						} else pClr = &(g_pSSEXOptions->clrCppOperator);	
					} else {
						while(	(*c) && (*c != ' ') && (*c != '\t') &&
								(*c != '{') && (*c != '}') && (*c != '/') &&
								(*c != '(') && (*c != ')') && (*c != '[') &&
								(*c != ']') && (*c != '"') && (*c != '\'') &&
								(*c != '+') && (*c != '-') && (*c != '*') &&
								(*c != '!') && (*c != '<') && (*c != '.') &&
								(*c != '>') && (*c != '^') && (*c != '%') &&
								(*c != '&') && (*c != '|') && (*c != '=') &&
								(*c != ',') && (*c != ';') && (*c != ':') &&
								(*c != '~') && (*c != '?') && (*c != '#') && (*c != '\\'))
						{
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
						}
						pClr = &(g_pSSEXOptions->clrCppNormalText);
					}
					//
					// EXTRACT TOKEN ####################################################################################
					//
				}
//				debug("LIne %d, block width: %d, starting at %d, of %d chars of (%s)",row,blockWidth,curXPos,c - begin,begin);
//				XSetForeground(dpy,gc,pClr->pixel());
				paint.setPen(*pClr);
				paint.drawText(curXPos,m_iFontAscent,begin,c - begin);
//				XDrawString(dpy,drw,gc,curXPos,m_iFontAscent,begin,c - begin);
				curXPos += blockWidth;
				break;
		}
	}

	if(m_iCursorRow == row)paintCursor(&paint,l,&(g_pSSEXOptions->clrCppCursor));
	if(m_bHasSelection)paintSelection(&paint,l,row,updateR);

	int col_xPos,row_yPos;

	p->worldMatrix().map(0,0,&col_xPos,&row_yPos);

	bitBlt(p->device(),col_xPos + updateR.x(),row_yPos + updateR.y(),m_pMemBuffer,updateR.x(),updateR.y(),updateR.width(),updateR.height());
}

QColor * SSEXEditor::cppModeGetTokenColor(const char * token,int len)
{
	if(*(token + len - 1) == 't')
	{
		if(len > 2)
		{
			if((*token != '_') && (*(token + len - 2) == '_'))return &(g_pSSEXOptions->clrCppType);
		}
	}

	switch(*token)
	{
		case '_':
			return &(g_pSSEXOptions->clrCppSystemIdentifier);
		break;
		case 'a':
			switch(len)
			{
				case 3:
					if(cmpnolen(token,"asm",3))return &(g_pSSEXOptions->clrCppKeyword);
				break;
				case 4:
					if(cmpnolen(token,"auto",4))return &(g_pSSEXOptions->clrCppKeyword);
				break;
			}
		break;
		case 'b':
			switch(len)
			{
				case 4:
					if(cmpnolen(token,"bool",4))return &(g_pSSEXOptions->clrCppType);
				break;
				case 5:
					if(cmpnolen(token,"break",5))return &(g_pSSEXOptions->clrCppKeyword);
				break;
				case 7:
					if(cmpnolen(token,"boolean",7))return &(g_pSSEXOptions->clrCppKeyword);
				break;
			}
		break;
		case 'c':
			switch(len)
			{
				case 4:
					if(cmpnolen(token,"case",4))return &(g_pSSEXOptions->clrCppKeyword);
					if(cmpnolen(token,"char",4))return &(g_pSSEXOptions->clrCppType);
				break;
				case 5:
					if(cmpnolen(token,"const",5))return &(g_pSSEXOptions->clrCppKeyword);
					if(cmpnolen(token,"class",5))return &(g_pSSEXOptions->clrCppKeyword);
					if(cmpnolen(token,"catch",5))return &(g_pSSEXOptions->clrCppKeyword);
					if(cmpnolen(token,"catch",5))return &(g_pSSEXOptions->clrCppKeyword);
				case 8:
					if(cmpnolen(token,"continue",8))return &(g_pSSEXOptions->clrCppKeyword);
				break;
				case 10:
					if(cmpnolen(token,"const_cast",10))return &(g_pSSEXOptions->clrCppKeyword);
				break;

			}
		break;
		case 'd':
			if(	cmp(token,"default",7,len) ||
				cmp(token,"do",2,len) ||
				cmp(token,"delete",6,len) ||
				cmp(token,"dynamic_cast",11,len) ||
				cmp(token,"dec",3,len)
			) return &(g_pSSEXOptions->clrCppKeyword);
			if(	cmp(token,"double",6,len)
			) return &(g_pSSEXOptions->clrCppType);
			if( cmp(token,"defined",7,len)
			) return &(g_pSSEXOptions->clrCppPreprocessor);
//			if(	cmp(token,"dup",3,len) ||
//				cmp(token,"dup2",4,len) ||
//				cmp(token,"domainname",10,len) ||
//				cmp(token,"daemon",6,len) ||
//				cmp(token,"dlopen",6,len) ||
//				cmp(token,"dlsym",5,len) ||
//				cmp(token,"dlclose",7,len) ||
//				cmp(token,"dlerror",7,len)
//			) return &(g_pSSEXOptions->clrCppSysCall);
		break;
		case 'e':
			if(	cmp(token,"extern",6,len) ||
				cmp(token,"else",4,len) ||
				cmp(token,"enum",4,len) ||
				cmp(token,"extends",7,len)
			) return &(g_pSSEXOptions->clrCppKeyword);
			if( cmp(token,"emit",4,len))return &(g_pSSEXOptions->clrCppQSignals);
//			if(	cmp(token,"execl",5,len) ||
//				cmp(token,"execle",6,len) ||
//				cmp(token,"execlp",6,len) ||
//				cmp(token,"execv",5,len) ||
//				cmp(token,"execve",6,len) ||
//				cmp(token,"execvp",6,len) ||
//				cmp(token,"encrypt",7,len)
//			) return &(g_pSSEXOptions->clrCppSysCall);
		break;
		case 'f':
			if(	cmp(token,"for",3,len) ||
				cmp(token,"fixed",5,len) ||
				cmp(token,"false",5,len) ||
				cmp(token,"friend",6,len) ||
				cmp(token,"final",5,len) || 
				cmp(token,"finally",7,len)     // this is really java
			) return &(g_pSSEXOptions->clrCppKeyword);
			if(	cmp(token,"float",5,len)
			) return &(g_pSSEXOptions->clrCppType);
//			if(	cmp(token,"fork",4,len) ||
//				cmp(token,"fcntl",5,len) ||
//				cmp(token,"fclose",6,len) ||
//				cmp(token,"fexecve",7,len) ||
//				cmp(token,"fopen",5,len) ||
//				cmp(token,"freopen",7,len) ||
//				cmp(token,"fdopen",6,len) ||
//				cmp(token,"feof",4,len) ||
//				cmp(token,"ferror",6,len) ||
//				cmp(token,"fflush",6,len) ||
//				cmp(token,"free",4,len) ||
//				cmp(token,"fgetc",5,len) ||
//				cmp(token,"fputc",5,len) ||
//				cmp(token,"fgets",5,len) ||
//				cmp(token,"fgetc",5,len) ||
//				cmp(token,"fread",5,len) ||
//				cmp(token,"fprintf",7,len) ||
//				cmp(token,"fileno",6,len) ||
//				cmp(token,"fscanf",6,len) ||
//				cmp(token,"fseek",5,len) ||
//				cmp(token,"fwrite",6,len) ||
//				cmp(token,"ftime",5,len)
//			) return &(g_pSSEXOptions->clrCppSysCall);
		break;
		case 'g':
			if(len > 1)
			{
				if(*(token + 1) == '_')return &(g_pSSEXOptions->clrCppGlobalVariable);
				if(strncmp(token,"gtk_",4) == 0)return &(g_pSSEXOptions->clrCppGtkCall);
				if(strncmp(token,"gdk_",4) == 0)return &(g_pSSEXOptions->clrCppGdkCall);
				switch(len)
				{
					case 4:
						if(cmpnolen(token,"goto",4))return &(g_pSSEXOptions->clrCppKeyword);
						if(cmpnolen(token,"gint",4))return &(g_pSSEXOptions->clrCppGType);
					break;
					case 5:
						if(cmpnolen(token,"guint",5))return &(g_pSSEXOptions->clrCppGType);
						if(cmpnolen(token,"glong",5))return &(g_pSSEXOptions->clrCppGType);
						if(cmpnolen(token,"gchar",5))return &(g_pSSEXOptions->clrCppGType);
						if(cmpnolen(token,"gsize",5))return &(g_pSSEXOptions->clrCppGType);
						if(cmpnolen(token,"gint8",5))return &(g_pSSEXOptions->clrCppGType);
					break;
					case 6:
						if(cmpnolen(token,"gulong",6))return &(g_pSSEXOptions->clrCppGType);
						if(cmpnolen(token,"guchar",6))return &(g_pSSEXOptions->clrCppGType);
						if(cmpnolen(token,"gshort",6))return &(g_pSSEXOptions->clrCppGType);
						if(cmpnolen(token,"gfloat",6))return &(g_pSSEXOptions->clrCppGType);
						if(cmpnolen(token,"gssize",6))return &(g_pSSEXOptions->clrCppGType);
						if(cmpnolen(token,"guint8",6))return &(g_pSSEXOptions->clrCppGType);
						if(cmpnolen(token,"gint16",6))return &(g_pSSEXOptions->clrCppGType);
						if(cmpnolen(token,"gint32",6))return &(g_pSSEXOptions->clrCppGType);
						if(cmpnolen(token,"gint64",6))return &(g_pSSEXOptions->clrCppGType);
					break;
					case 7:
						if(cmpnolen(token,"gushort",7))return &(g_pSSEXOptions->clrCppGType);
						if(cmpnolen(token,"gdouble",7))return &(g_pSSEXOptions->clrCppGType);
						if(cmpnolen(token,"guint16",7))return &(g_pSSEXOptions->clrCppGType);
						if(cmpnolen(token,"guint32",7))return &(g_pSSEXOptions->clrCppGType);
						if(cmpnolen(token,"guint64",7))return &(g_pSSEXOptions->clrCppGType);
					break;
					case 8:
						if(cmpnolen(token,"gboolean",8))return &(g_pSSEXOptions->clrCppGType);
						if(cmpnolen(token,"gldouble",8))return &(g_pSSEXOptions->clrCppGType);
						if(cmpnolen(token,"gpointer",8))return &(g_pSSEXOptions->clrCppGType);
					break;
					case 13:
						if(cmpnolen(token,"gconstpointer",13))return &(g_pSSEXOptions->clrCppGType);
					break;
				}
			}
		break;
		case 'i':
			if(	cmp(token,"if",2,len) ||
				cmp(token,"inline",6,len) ||
				cmp(token,"internal",8,len) ||
				cmp(token,"import",6,len) // this is java
			) return &(g_pSSEXOptions->clrCppKeyword);
			if(	cmp(token,"int",3,len)
			) return &(g_pSSEXOptions->clrCppType);
		break;
		case 'k':
			if( strncmp(token,"kvi_",4) == 0)return &(g_pSSEXOptions->clrCppSpecial);
		break;
		case 'l':
			if(	cmp(token,"long",4,len)
			) return &(g_pSSEXOptions->clrCppType);
		break;
		case 'm':
			if( (strncmp(token,"m_",2) == 0)
			) return &(g_pSSEXOptions->clrCppMemberVariable);
			if(	cmp(token,"mutable",7,len)
			) return &(g_pSSEXOptions->clrCppKeyword);
		break;
		case 'n':
			if(	cmp(token,"namespace",9,len) ||
				cmp(token,"new",3,len) ||
				cmp(token,"null",4,len)
			) return &(g_pSSEXOptions->clrCppKeyword);
		break;
		case 'o':
			if(	cmp(token,"overload",8,len) ||
				cmp(token,"operator",8,len)
			) return &(g_pSSEXOptions->clrCppKeyword);
		break;
		case 'p':
			if(	cmp(token,"public",6,len) ||
				cmp(token,"protected",9,len) ||
				cmp(token,"private",7,len) ||
				cmp(token,"package",7,len) // this is java
			) return &(g_pSSEXOptions->clrCppKeyword);
		break;
//		case 'q':
//			if(	cmp(token,"public",6,len) ||
//				cmp(token,"protected",9,len) ||
//				cmp(token,"private",7,len)
//			) return &(g_pSSEXOptions->clrCppKeyword);
//		break;
		case 'r':
			if(	cmp(token,"return",6,len) ||
				cmp(token,"register",8,len) ||
				cmp(token,"reinterpret_cast",16,len)
			) return &(g_pSSEXOptions->clrCppKeyword);
		break;
		case 's':
			if(	cmp(token,"switch",6,len) ||
				cmp(token,"struct",6,len) ||
				cmp(token,"static",6,len) ||
				cmp(token,"sizeof",6,len) ||
				cmp(token,"static_cast",11,len) ||
				cmp(token,"synchronized",12,len) // this is java
			) return &(g_pSSEXOptions->clrCppKeyword);
			if(	cmp(token,"signed",6,len) ||
				cmp(token,"short",5,len)
			) return &(g_pSSEXOptions->clrCppType);
			if(	cmp(token,"signals",7,len) ||
				cmp(token,"slots",5,len)
			) return &(g_pSSEXOptions->clrCppQSignals);
			if( strncmp(token,"ss",2) == 0)return &(g_pSSEXOptions->clrCppSpecial);
		break;
		case 't':
			if(	cmp(token,"typedef",7,len) ||
				cmp(token,"template",8,len) ||
				cmp(token,"this",4,len) ||
				cmp(token,"throw",5,len) ||
				cmp(token,"try",3,len) ||
				cmp(token,"true",4,len) ||
				cmp(token,"typeid",6,len)
			) return &(g_pSSEXOptions->clrCppKeyword);
			
		break;
		case 'u':
			if(	cmp(token,"union",5,len) ||
				cmp(token,"using",5,len)
			) return &(g_pSSEXOptions->clrCppKeyword);
			if(	cmp(token,"uint",4,len) ||
				cmp(token,"uchar",5,len) ||
				cmp(token,"ulong",5,len) ||
				cmp(token,"ushort",6,len) ||
				cmp(token,"unsigned",8,len) ||
				cmp(token,"u_int",5,len) ||
				cmp(token,"u_char",6,len) ||
				cmp(token,"u_long",6,len) ||
				cmp(token,"u_short",7,len)
			) return &(g_pSSEXOptions->clrCppType);
		break;
		case 'v':
			if(	cmp(token,"volatile",8,len) ||
				cmp(token,"virtual",7,len)
			) return &(g_pSSEXOptions->clrCppKeyword);
			if(	cmp(token,"void",4,len)
			) return &(g_pSSEXOptions->clrCppType);
		break;
		case 'w':
			if(	cmp(token,"while",5,len)
			) return &(g_pSSEXOptions->clrCppKeyword);
		break;
		case 'A':
			if(	cmp(token,"Atom",4,len)
			) return &(g_pSSEXOptions->clrCppXStuff);
		break;
		case 'B':
			if(	cmp(token,"Bool",4,len)
			) return &(g_pSSEXOptions->clrCppXStuff);
		break;
		case 'C':
			if(	cmp(token,"Colormap",8,len) ||
				cmp(token,"Cursor",6,len) ||
				cmp(token,"Cardinal",8,len)
			) return &(g_pSSEXOptions->clrCppXStuff);
		break;
		case 'D':
			if(	cmp(token,"Depth",5,len) ||
				cmp(token,"Display",7,len) ||
				cmp(token,"Drawable",8,len)
			) return &(g_pSSEXOptions->clrCppXStuff);
			if(	cmp(token,"DIR",3,len))return &(g_pSSEXOptions->clrCppType);
		break;
		case 'F':
			if(	cmpnocase(token,"FALSE",5,len)
			) return &(g_pSSEXOptions->clrCppKeyword);
			if(	cmp(token,"FILE",4,len)
			) return &(g_pSSEXOptions->clrCppType);
			if(	cmp(token,"Font",4,len)
			) return &(g_pSSEXOptions->clrCppXStuff);
		break;
		case 'G':
			if(len > 2)
			{
				if(*(token + 1)=='_')return &(g_pSSEXOptions->clrCppGMacro);
				if(len > 4)
				{
					if(strncmp(token,"GTK_",4) == 0)return &(g_pSSEXOptions->clrCppGtkMacro);
					if(strncmp(token,"GDK_",4) == 0)return &(g_pSSEXOptions->clrCppGdkMacro);
				}
				if(strncmp(token,"Gtk",3) == 0)return &(g_pSSEXOptions->clrCppGtkStruct);
				if(strncmp(token,"Gdk",3) == 0)return &(g_pSSEXOptions->clrCppGdkStruct);
			} else if(cmp(token,"GC",2,len))return &(g_pSSEXOptions->clrCppXStuff);
//			return &(g_pSSEXOptions->clrCppGStruct);
		break;
		case 'K':
			return &(g_pSSEXOptions->clrCppKClass);
		break;
		case 'M':
			if(	cmp(token,"Mask",4,len)
			) return &(g_pSSEXOptions->clrCppXStuff);
		break;
		case 'N':
			if(	cmp(token,"NULL",4,len)
			) return &(g_pSSEXOptions->clrCppKeyword);
		break;
		case 'P':
			if(	cmp(token,"Pixmap",6,len)
			) return &(g_pSSEXOptions->clrCppXStuff);
		break;
		case 'Q':
			return &(g_pSSEXOptions->clrCppQClass);
		break;
		case 'S':
			if(	cmp(token,"SLOT",4,len) ||
				cmp(token,"SIGNAL",6,len)
			) return &(g_pSSEXOptions->clrCppQSignals);
			if(	cmp(token,"Status",6,len) ||
				cmp(token,"Screen",6,len) ||
				cmp(token,"ScreenFormat",12,len)) return &(g_pSSEXOptions->clrCppXStuff);
			if( strncmp(token,"SS",2) == 0)return &(g_pSSEXOptions->clrCppSSStuff);
		break;
		case 'T':
			if(	cmpnocase(token,"TRUE",4,len)
			) return &(g_pSSEXOptions->clrCppKeyword);
			if(	cmp(token,"Time",4,len)
			) return &(g_pSSEXOptions->clrCppXStuff);
		break;
		case 'V':
			if(	cmp(token,"Visual",6,len) ||
				cmp(token,"VisualID",8,len)
			) return &(g_pSSEXOptions->clrCppXStuff);
		break;
		case 'W':
			if(	cmp(token,"Window",6,len) ||
				cmp(token,"Widget",6,len)
			) return &(g_pSSEXOptions->clrCppXStuff);
		break;
		case 'X':
			return &(g_pSSEXOptions->clrCppXStuff);
		break;
	}
	return &(g_pSSEXOptions->clrCppNormalText);
}


void SSEXEditor::paintCellPerl(QPainter * p,SSEXEditorTextLine * l,int row)
{
	QPainter paint(m_pMemBuffer);

	paint.setFont(font());


	QRect updateR = cellUpdateRect();
	paint.fillRect(updateR,isEnabled() ? g_pSSEXOptions->clrPerlBackground : g_pSSEXOptions->clrPerlDisabledBackground);
	if(m_bHasBlockMark)paintBlockMark(&paint,l,row,updateR,&(g_pSSEXOptions->clrPerlMarkBackground));

	const char * c = l->text.data();
	int curXPos = SSEX_EDITOR_BORDER;
	int lastTabPos = SSEX_EDITOR_BORDER;

	const char * begin;
	QColor * pClr;
	int blockWidth;

	bool bInLineComment = false;

	unsigned int bGotEscape = 0;

	while(*c)
	{

		if(bGotEscape > 0)bGotEscape--;
		switch(*c)
		{
			case '\t':
				// a tab character
				while(lastTabPos <= curXPos)lastTabPos += m_iTabsNumPixels;
				curXPos = lastTabPos;
				paint.setPen(QPen(g_pSSEXOptions->clrPerlTab,0,DashLine));
				paint.drawLine(curXPos - m_iTabsNumPixels,0,curXPos - m_iTabsNumPixels,m_iFontAscent);
				c++;
				break;
			case ' ':
				// a set of spaces
				begin = c;
				while(*c == ' ')c++;
				curXPos += (m_iCharWidth[(unsigned char)' '] * (c - begin));
				break;
			default:
				begin      = c;
				blockWidth = 0;
				{
					//
					// EXTRACT TOKEN ####################################################################################
					//
					if(bInLineComment)
					{
						// We are in a line comment...up to the end of the line!
						while((*c) && (*c != ' ') && (*c != '\t'))
						{
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
						}
						pClr = &(g_pSSEXOptions->clrPerlComment);

					} else if(	((*c >= 'a') && (*c <= 'z')) ||
								((*c >= 'A') && (*c <= 'Z')) ||
								(*c == '_'))
					{
						// A text token
						blockWidth += m_iCharWidth[(unsigned char)(*c)];
						c++;
						while(	((*c >= 'a') && (*c <= 'z')) ||
								((*c >= 'A') && (*c <= 'Z')) ||
								((*c >= '0') && (*c <= '9')) ||
								(*c == '_'))
						{
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
						}
						//pClr = cppModeGetTokenColor(begin,c - begin);
						pClr = &(g_pSSEXOptions->clrPerlNormalText);
					} else if(*c == '#')
					{
						blockWidth += m_iCharWidth[(unsigned char)(*c)];
						bInLineComment = true;
						c++;
						pClr = &(g_pSSEXOptions->clrPerlComment);
					} else if((*c == '$') || (*c == '%') || (*c == '@'))
					{
						// A variable
						blockWidth += m_iCharWidth[(unsigned char)(*c)];
						c++;
						if(bGotEscape)
						{
							pClr = &(g_pSSEXOptions->clrPerlNormalText);
						} else {
							while(	((*c >= 'a') && (*c <= 'z')) ||
									((*c >= 'A') && (*c <= 'Z')) ||
									((*c >= '0') && (*c <= '9')) ||
									(*c == '_'))
							{
								blockWidth += m_iCharWidth[(unsigned char)(*c)];
								c++;
							}
							pClr = &(g_pSSEXOptions->clrPerlVariable);
						}
					} else if(*c == '\\')
					{
						// escape
						bGotEscape = 2;
						blockWidth += m_iCharWidth[(unsigned char)(*c)];
						c++;
						pClr = &(g_pSSEXOptions->clrPerlOperator);
					} else if( (*c == '{') || (*c == '}') )
					{
						blockWidth += m_iCharWidth[(unsigned char)(*c)];
						c++;
						pClr = &(g_pSSEXOptions->clrPerlBrace);
					} else if((*c == '/') || (*c == '(') || (*c == ')') || (*c == '[') || (*c == ']') ||
								(*c == '"') || (*c == '\'') || (*c == '+') || (*c == '-') || (*c == '*') ||
								(*c == '>') || (*c == '<') || (*c == '~') || (*c == '?') || (*c == '=') ||
								(*c == '!') || (*c == '&') || (*c == '|') || (*c == '.') || (*c == ':') ||
								(*c == ';') || (*c == '^') || (*c == ',') )
					{
						blockWidth += m_iCharWidth[(unsigned char)(*c)];
						c++;
						pClr = bGotEscape ? &(g_pSSEXOptions->clrPerlNormalText) : &(g_pSSEXOptions->clrPerlOperator);
					} else {
						while(	(*c) && (*c != ' ') && (*c != '\t') &&
								(*c != '{') && (*c != '}') && (*c != '/') &&
								(*c != '(') && (*c != ')') && (*c != '[') &&
								(*c != ']') && (*c != '"') && (*c != '\'') &&
								(*c != '+') && (*c != '-') && (*c != '*') &&
								(*c != '!') && (*c != '<') && (*c != '.') &&
								(*c != '>') && (*c != '^') && (*c != '%') &&
								(*c != '&') && (*c != '|') && (*c != '=') &&
								(*c != ',') && (*c != ';') && (*c != ':') &&
								(*c != '~') && (*c != '?') && (*c != '#') &&
								(*c != '\\') && (*c != '$') && (*c != '@'))
						{
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
						}
						pClr = &(g_pSSEXOptions->clrPerlNormalText);
					}
					//
					// EXTRACT TOKEN ####################################################################################
					//
				}
				paint.setPen(*pClr);
				paint.drawText(curXPos,m_iFontAscent,begin,c-begin);
				curXPos += blockWidth;
				break;
		}
	}

	if(m_iCursorRow == row)paintCursor(&paint,l,&(g_pSSEXOptions->clrPerlCursor));
	if(m_bHasSelection)paintSelection(&paint,l,row,updateR);

	int col_xPos,row_yPos;

	p->worldMatrix().map(0,0,&col_xPos,&row_yPos);

	bitBlt(p->device(),col_xPos + updateR.x(),row_yPos + updateR.y(),m_pMemBuffer,updateR.x(),updateR.y(),updateR.width(),updateR.height());
}


void SSEXEditor::paintCellHtml(QPainter * p,SSEXEditorTextLine * l,int row)
{
	QPainter paint(m_pMemBuffer);

	paint.setFont(font());


	QRect updateR = cellUpdateRect();


	paint.fillRect(updateR,isEnabled() ? g_pSSEXOptions->clrHtmlBackground : g_pSSEXOptions->clrHtmlDisabledBackground);
	if(m_bHasBlockMark)paintBlockMark(&paint,l,row,updateR,&(g_pSSEXOptions->clrHtmlMarkBackground));

	const char * c = l->text.data();
	int curXPos = SSEX_EDITOR_BORDER;
	int lastTabPos = SSEX_EDITOR_BORDER;

	const char * begin;
	QColor * pClr;
	int blockWidth;

	bool bInString = false;
	bool bInComment = l->flags & SSEX_EDITOR_BEGIN_IN_COMMENT;
	bool bInTag = l->flags & SSEX_EDITOR_BEGIN_IN_TAG;

	while(*c)
	{
		switch(*c)
		{
			case '\t':
				// a tab character
				while(lastTabPos <= curXPos)lastTabPos += m_iTabsNumPixels;
				curXPos = lastTabPos;
				c++;
				break;
			case ' ':
				// a set of spaces
				begin = c;
				while(*c == ' ')c++;
				curXPos += (m_iCharWidth[(unsigned char)' '] * (c - begin));
				break;
			default:
				begin      = c;
				blockWidth = 0;
				{
					//
					// EXTRACT TOKEN ####################################################################################
					//
					if(bInComment)
					{
						if(*c == '-')
						{
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
							if(*c == '-')
							{
								blockWidth += m_iCharWidth[(unsigned char)(*c)];
								c++;
								if(*c == '>')
								{
									blockWidth += m_iCharWidth[(unsigned char)(*c)];
									c++;
									bInComment = false;
									bInTag = false;
									bInString = false;
								} else pClr = &(g_pSSEXOptions->clrHtmlComment);
							} else pClr = &(g_pSSEXOptions->clrHtmlComment);
						} else {
							while(*c && (*c != '-') && (*c != ' ') && (*c != '\t'))
							{
								blockWidth += m_iCharWidth[(unsigned char)(*c)];
								c++;
							}
							pClr = &(g_pSSEXOptions->clrHtmlComment);
						}
					} else if(bInTag)
					{
						if(*c == '"')
						{
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
							bInString = !bInString;
							pClr = &(g_pSSEXOptions->clrHtmlString);
						} else if(*c == '>')
						{
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
							bInString = false;
							bInTag = false;
							pClr = &(g_pSSEXOptions->clrHtmlTag);
						} else {
							while(*c && (*c != '>') && (*c != '"'))
							{
								blockWidth += m_iCharWidth[(unsigned char)(*c)];
								c++;
							}
							pClr = bInString ? &(g_pSSEXOptions->clrHtmlString) : &(g_pSSEXOptions->clrHtmlTagInternal);
						}
					} else {
						// normal text
						if(*c == '<')
						{
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
							bInString = false;
							bInTag = true;
							if(*c == '!')
							{
								blockWidth += m_iCharWidth[(unsigned char)(*c)];
								c++;
								if(*c == '-')
								{
									blockWidth += m_iCharWidth[(unsigned char)(*c)];
									c++;
									if(*c == '-')
									{
										blockWidth += m_iCharWidth[(unsigned char)(*c)];
										c++;
										bInString = false;
										bInTag =false;
										bInComment = true;
										pClr = &(g_pSSEXOptions->clrHtmlComment);
									} else pClr = &(g_pSSEXOptions->clrHtmlTag);
								} else pClr = &(g_pSSEXOptions->clrHtmlTag);
							} else pClr = &(g_pSSEXOptions->clrHtmlTag);
						} else {
							while(*c && (*c != '<'))
							{
								blockWidth += m_iCharWidth[(unsigned char)(*c)];
								c++;
							}
							pClr = &(g_pSSEXOptions->clrHtmlNormalText);
						}
					}
					//
					// EXTRACT TOKEN ####################################################################################
					//
				}
				paint.setPen(*pClr);
				paint.drawText(curXPos,m_iFontAscent,begin,c - begin);
				curXPos += blockWidth;
				break;
		}
	}
	if(m_iCursorRow == row)paintCursor(&paint,l,&(g_pSSEXOptions->clrHtmlCursor));
	if(m_bHasSelection)paintSelection(&paint,l,row,updateR);

	int col_xPos,row_yPos;

	p->worldMatrix().map(0,0,&col_xPos,&row_yPos);

	bitBlt(p->device(),col_xPos + updateR.x(),row_yPos + updateR.y(),m_pMemBuffer,updateR.x(),updateR.y(),updateR.width(),updateR.height());

}




void SSEXEditor::paintCellAutoconf(QPainter * p,SSEXEditorTextLine * l,int row)
{
	QPainter paint(m_pMemBuffer);

	paint.setFont(font());

	QRect updateR = cellUpdateRect();
	paint.fillRect(updateR,isEnabled() ? g_pSSEXOptions->clrAutoconfBackground : g_pSSEXOptions->clrAutoconfDisabledBackground);
	if(m_bHasBlockMark)paintBlockMark(&paint,l,row,updateR,&(g_pSSEXOptions->clrAutoconfMarkBackground));

	const char * c = l->text.data();
	int curXPos = SSEX_EDITOR_BORDER;
	int lastTabPos = SSEX_EDITOR_BORDER;

	const char * begin;
	QColor * pClr;
	int blockWidth;

	bool bInLineComment = false;
	bool bInDnlComment = false;

	unsigned int bGotEscape = 0;

	while(*c)
	{

		if(bGotEscape > 0)bGotEscape--;
		switch(*c)
		{
			case '\t':
				// a tab character
				while(lastTabPos <= curXPos)lastTabPos += m_iTabsNumPixels;
				curXPos = lastTabPos;
				paint.setPen(QPen(g_pSSEXOptions->clrAutoconfTab,0,DashLine));
				paint.drawLine(curXPos - m_iTabsNumPixels,0,curXPos - m_iTabsNumPixels,m_iFontAscent);
				c++;
				break;
			case ' ':
				// a set of spaces
				begin = c;
				while(*c == ' ')c++;
				curXPos += (m_iCharWidth[(unsigned char)' '] * (c - begin));
				break;
			default:
				begin      = c;
				blockWidth = 0;
				{
					//
					// EXTRACT TOKEN ####################################################################################
					//
					if(bInLineComment || bInDnlComment)
					{
						// We are in a line comment...up to the end of the line!
						while(*c && (*c != ' ') && (*c != '\t'))
						{
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
						}
						pClr = bInLineComment ? &(g_pSSEXOptions->clrAutoconfComment) : &(g_pSSEXOptions->clrAutoconfDnlComment);

					} else if(	((*c >= 'a') && (*c <= 'z')) ||
								((*c >= 'A') && (*c <= 'Z')) ||
								(*c == '_'))
					{
						// A text token
						blockWidth += m_iCharWidth[(unsigned char)(*c)];
						c++;
						while(	((*c >= 'a') && (*c <= 'z')) ||
								((*c >= 'A') && (*c <= 'Z')) ||
								((*c >= '0') && (*c <= '9')) ||
								(*c == '_'))
						{
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
						}
						pClr = 0;
						int len = c - begin;
						// TOKEN COLORISATION
						switch(*begin)
						{
							case 'a':
								if(cmp(begin,"alias",5,len))
										pClr = &(g_pSSEXOptions->clrAutoconfShellCommand);
							break;
							case 'b':
								if(cmp(begin,"bg",2,len) ||
									cmp(begin,"bind",4,len) ||
									cmp(begin,"break",5,len) ||
									cmp(begin,"builtin",7,len))
										pClr = &(g_pSSEXOptions->clrAutoconfShellCommand);
							break;
							case 'c':
								if(cmp(begin,"case",4,len) ||
									cmp(begin,"cd",2,len) ||
									cmp(begin,"command",7,len) ||
									cmp(begin,"continue",8,len))
										pClr = &(g_pSSEXOptions->clrAutoconfShellCommand);
							break;
							case 'd':
								if(len == 3)
								{
									if(cmpnolen(begin,"dnl",3))
									{
										pClr = &(g_pSSEXOptions->clrAutoconfDnlComment);
										bInDnlComment = true;
									}
								} else {
									if(cmp(begin,"done",4,len) ||
										cmp(begin,"do",2,len) ||
										cmp(begin,"dirs",4,len) ||
										cmp(begin,"declare",7,len) ||
										cmp(begin,"disown",6,len))
										pClr = &(g_pSSEXOptions->clrAutoconfShellCommand);
								}
							break;
							case 'e':
								if(len == 4)
								{
									if(cmpnolen(begin,"else",4) || 
										cmpnolen(begin,"elif",4) ||
										cmpnolen(begin,"esac",4) ||
										cmpnolen(begin,"exec",4) ||
										cmpnolen(begin,"echo",4) ||
										cmpnolen(begin,"eval",4) ||
										cmpnolen(begin,"exit",4))
											pClr = &(g_pSSEXOptions->clrAutoconfShellCommand);
								} else {
									if(cmp(begin,"enable",6,len) ||
										cmp(begin,"export",6,len))
											pClr = &(g_pSSEXOptions->clrAutoconfShellCommand);
								}
							break;
							case 'f':
								if(len == 2)
								{
									if(cmpnolen(begin,"fi",2) ||
										cmpnolen(begin,"fc",2) ||
										cmpnolen(begin,"fg",2))
										pClr = &(g_pSSEXOptions->clrAutoconfShellCommand);
								} else if(len == 3)
								{
									if(cmpnolen(begin,"for",3))
										pClr = &(g_pSSEXOptions->clrAutoconfShellCommand);
								} else if(len == 8)
								{
									if(cmpnolen(begin,"function",8))
										pClr = &(g_pSSEXOptions->clrAutoconfShellCommand);
								}
							break;
							case 'g':
								if(cmp(begin,"getopts",7,len))
										pClr = &(g_pSSEXOptions->clrAutoconfShellCommand);
							break;
							case 'h':
								if(cmp(begin,"hash",4,len) ||
									cmp(begin,"help",4,len) ||
									cmp(begin,"history",7,len))
										pClr = &(g_pSSEXOptions->clrAutoconfShellCommand);
							break;
							case 'i':
								if(len == 2)
								{
									if(cmpnolen(begin,"if",2) || 
										cmpnolen(begin,"in",2))
										pClr = &(g_pSSEXOptions->clrAutoconfShellCommand);
								}
							break;
							case 'j':
								if(cmp(begin,"jobs",4,len))
										pClr = &(g_pSSEXOptions->clrAutoconfShellCommand);
							break;
							case 'k':
								if(cmp(begin,"kill",4,len))
										pClr = &(g_pSSEXOptions->clrAutoconfShellCommand);
							break;
							case 'l':
								if(cmp(begin,"let",3,len) ||
									cmp(begin,"local",5,len))
										pClr = &(g_pSSEXOptions->clrAutoconfShellCommand);
							break;
							case 'p':
								if(cmp(begin,"popd",4,len) ||
									cmp(begin,"pushd",5,len) ||
									cmp(begin,"pwd",3,len))
										pClr = &(g_pSSEXOptions->clrAutoconfShellCommand);
							break;
							case 'r':
								if(cmp(begin,"return",6,len) ||
									cmp(begin,"readonly",8,len))
										pClr = &(g_pSSEXOptions->clrAutoconfShellCommand);
							break;
							case 's':
								if(cmp(begin,"select",6,len) ||
									cmp(begin,"set",3,len) ||
									cmp(begin,"shift",5,len) ||
									cmp(begin,"shopt",5,len) ||
									cmp(begin,"source",6,len) ||
									cmp(begin,"suspend",7,len))
										pClr = &(g_pSSEXOptions->clrAutoconfShellCommand);
							break;
							case 't':
								if(cmp(begin,"then",4,len) ||
									cmp(begin,"time",4,len) ||
									cmp(begin,"times",5,len) ||
									cmp(begin,"test",4,len) ||
									cmp(begin,"trap",4,len) ||
									cmp(begin,"type",4,len) ||
									cmp(begin,"typeset",7,len))
										pClr = &(g_pSSEXOptions->clrAutoconfShellCommand);
							break;

							case 'u':
								if(cmp(begin,"until",5,len) ||
									cmp(begin,"ulimit",6,len) ||
									cmp(begin,"umask",5,len) ||
									cmp(begin,"unalias",7,len) ||
									cmp(begin,"unset",5,len))
										pClr = &(g_pSSEXOptions->clrAutoconfShellCommand);
							break;
							case 'w':
								if(cmp(begin,"while",5,len) ||
									cmp(begin,"wait",4,len))
										pClr = &(g_pSSEXOptions->clrAutoconfShellCommand);
							break;
							case 'A':
								if(len > 3)
								{
									if(cmpnolen(begin,"AC_",3))
									{
										const char * ptr = begin + 3;
										int newLen = len - 3;
										switch(*ptr)
										{
											case 'A':
												if(cmp(ptr,"AIX",3,newLen) ||
													cmp(ptr,"ALLOCA",6,newLen) ||
													cmp(ptr,"ARG_WITH",8,newLen) ||
													cmp(ptr,"ARG_ENABLE",10,newLen) ||
													cmp(ptr,"ARG_PROGRAM",11,newLen))
													pClr = &(g_pSSEXOptions->clrAutoconfBuiltinMacro);
											break;
											case 'B':
												if(cmp(ptr,"BEFORE",6,newLen))
													pClr = &(g_pSSEXOptions->clrAutoconfBuiltinMacro);
											break;
											case 'C':
												if(len > 7)
												{
													if(cmpnolen(ptr,"CACHE_",6))
													{
														if( cmp(ptr,"CACHE_CHECK",11,newLen) ||
															cmp(ptr,"CACHE_LOAD",10,newLen) ||
															cmp(ptr,"CACHE_SAVE",10,newLen) ||
															cmp(ptr,"CACHE_VAL",9,newLen))
																pClr = &(g_pSSEXOptions->clrAutoconfBuiltinMacro);
													} else if(cmpnolen(ptr,"CHECK_",6))
													{
														if( cmp(ptr,"CHECK_FILE",10,newLen) ||
															cmp(ptr,"CHECK_FUNC",10,newLen) ||
															cmp(ptr,"CHECK_FUNCS",11,newLen) ||
															cmp(ptr,"CHECK_FILES",11,newLen) ||
															cmp(ptr,"CHECK_HEADER",12,newLen) ||
															cmp(ptr,"CHECK_HEADERS",13,newLen) ||
															cmp(ptr,"CHECK_LIB",9,newLen) ||
															cmp(ptr,"CHECK_PROG",10,newLen) ||
															cmp(ptr,"CHECK_PROGS",11,newLen) ||
															cmp(ptr,"CHECK_SIZEOF",12,newLen) ||
															cmp(ptr,"CHECK_TOOL",10,newLen) ||
															cmp(ptr,"CHECK_TYPE",10,newLen))
																pClr = &(g_pSSEXOptions->clrAutoconfBuiltinMacro);
													} else {
														if( cmp(ptr,"COMPILE_CHECK",13,newLen) ||
															cmp(ptr,"CONFIG_AUX_DIR",14,newLen) ||
															cmp(ptr,"CONFIG_HEADER",13,newLen) ||
															cmp(ptr,"CONFIG_SUBDIRS",14,newLen) ||
															cmp(ptr,"CANONICAL_HOST",14,newLen) ||
															cmp(ptr,"CANONICAL_BUILD",15,newLen) ||
															cmp(ptr,"CANONICAL_SYSTEM",16,newLen) ||
															cmp(ptr,"CHECKING",8,newLen) ||
															cmp(ptr,"C_BIGENDIAN",11,newLen) ||
															cmp(ptr,"C_CHAR_UNSIGNED",15,newLen) ||
															cmp(ptr,"C_CONST",7,newLen) ||
															cmp(ptr,"C_CROSS",7,newLen) ||
															cmp(ptr,"C_INLINE",8,newLen) ||
															cmp(ptr,"C_LONG_DOUBLE",13,newLen) ||
															cmp(ptr,"C_STRINGIZE",11,newLen))
																pClr = &(g_pSSEXOptions->clrAutoconfBuiltinMacro);
													}
												}
											break;
											case 'D':
												if(cmp(ptr,"DEFUN",5,newLen) ||
													cmp(ptr,"DEFINE",6,newLen) ||
													cmp(ptr,"DEFINE_UNQUOTED",15,newLen) ||
													cmp(ptr,"DIR_HEADER",10,newLen) ||
													cmp(ptr,"DECL_YYTEXT",11,newLen) ||
													cmp(ptr,"DYNIX_SEQ",9,newLen))
													pClr = &(g_pSSEXOptions->clrAutoconfBuiltinMacro);
											break;
											case 'E':
												if(cmp(ptr,"EGREP_CPP",9,newLen) ||
													cmp(ptr,"ENABLE",6,newLen) ||
													cmp(ptr,"EGREP_HEADER",12,newLen) ||
													cmp(ptr,"EXEEXT",6,newLen))
													pClr = &(g_pSSEXOptions->clrAutoconfBuiltinMacro);
											break;
											case 'F':
												if(cmpnolen(ptr,"FUNC_",5))
												{
													if(cmp(ptr,"FUNC_ALLOCA",11,newLen) ||
														cmp(ptr,"FUNC_CLOSEDIR_VOID",18,newLen) ||
														cmp(ptr,"FUNC_FNMATH",11,newLen) ||
														cmp(ptr,"FUNC_GETLOADAVG",15,newLen) ||
														cmp(ptr,"FUNC_GETMNTENT",14,newLen) ||
														cmp(ptr,"FUNC_GETPGRP",12,newLen) ||
														cmp(ptr,"FUNC_MEMCMP",11,newLen) ||
														cmp(ptr,"FUNC_MMAP",9,newLen) ||
														cmp(ptr,"FUNC_SELECT_ARGTYPES",20,newLen) ||
														cmp(ptr,"FUNC_SETPGRP",12,newLen) ||
														cmp(ptr,"FUNC_SETVBUF_REVERSED",21,newLen) ||
														cmp(ptr,"FUNC_STRCOLL",12,newLen) ||
														cmp(ptr,"FUNC_STRFTIME",13,newLen) ||
														cmp(ptr,"FUNC_UTIME_NULL",15,newLen) ||
														cmp(ptr,"FUNC_VFORK",10,newLen) ||
														cmp(ptr,"FUNC_VPRINTF",12,newLen) ||
														cmp(ptr,"FUNC_WAIT3",11,newLen))
															pClr = &(g_pSSEXOptions->clrAutoconfBuiltinMacro);
												} else {
													if(cmp(ptr,"F77_LIBRARY_LDFLAGS",19,newLen))
															pClr = &(g_pSSEXOptions->clrAutoconfBuiltinMacro);
												}
											break;
											case 'H':
												if(cmp(ptr,"HAVE_LIBRARY",12,newLen) ||
													cmp(ptr,"HEADER_DIRENT",13,newLen) ||
													cmp(ptr,"HEADER_MAJOR",12,newLen) ||
													cmp(ptr,"HEADER_STAT",11,newLen) ||
													cmp(ptr,"HEADER_STDC",11,newLen) ||
													cmp(ptr,"HEADER_SYS_WAIT",15,newLen) ||
													cmp(ptr,"HEADER_TIME",11,newLen))
										 				pClr = &(g_pSSEXOptions->clrAutoconfBuiltinMacro);
											break;
											case 'I':
												if(cmp(ptr,"INIT",4,newLen) ||
													cmp(ptr,"INT_16_BITS",11,newLen) ||
													cmp(ptr,"IRIX_SUN",8,newLen) ||
													cmp(ptr,"ISC_POSIX",9,newLen))
														pClr = &(g_pSSEXOptions->clrAutoconfBuiltinMacro);
											break;
											case 'L':
												if(cmp(ptr,"LANG_C",6,newLen) ||
													cmp(ptr,"LANG_CPLUSPLUS",14,newLen) ||
													cmp(ptr,"LANG_FORTRAN77",14,newLen) ||
													cmp(ptr,"LANG_SAVE",9,newLen) ||
													cmp(ptr,"LANG_RESTORE",12,newLen) ||
													cmp(ptr,"LINK_FILES",10,newLen) ||
													cmp(ptr,"LONG_64_BITS",12,newLen))
														pClr = &(g_pSSEXOptions->clrAutoconfBuiltinMacro);
											break;
											case 'M':
												if(cmp(ptr,"MSG_ERROR",9,newLen) ||
													cmp(ptr,"MSG_RESULT",10,newLen) ||
													cmp(ptr,"MSG_CHECKING",12,newLen) ||
													cmp(ptr,"MSG_WARN",8,newLen) ||
													cmp(ptr,"MEMORY_H",8,newLen) ||
													cmp(ptr,"MINGW32",7,newLen) ||
													cmp(ptr,"MINIX",5,newLen))
														pClr = &(g_pSSEXOptions->clrAutoconfBuiltinMacro);
											break;
											case 'O':
												if(cmp(ptr,"OUTPUT",6,newLen) ||
													cmp(ptr,"OBJEXT",6,newLen))
														pClr = &(g_pSSEXOptions->clrAutoconfBuiltinMacro);
											break;
											case 'P':
												if(len > 7)
												{
													if(cmpnolen(ptr,"PATH_",5))
													{
														if( cmp(ptr,"PATH_PROGS",10,newLen) ||
															cmp(ptr,"PATH_PROG",9,newLen) ||
															cmp(ptr,"PATH_X",6,newLen) ||
															cmp(ptr,"PATH_XTRA",9,newLen))
																pClr = &(g_pSSEXOptions->clrAutoconfBuiltinMacro);
													} else if(cmpnolen(ptr,"PROG_",5))
													{
														if( cmp(ptr,"PROG_AWK",8,newLen) ||
															cmp(ptr,"PROG_CC",7,newLen) ||
															cmp(ptr,"PROG_CC_C_O",11,newLen) ||
															cmp(ptr,"PROG_CPP",8,newLen) ||
															cmp(ptr,"PROG_CXX",8,newLen) ||
															cmp(ptr,"PROG_CXXCPP",11,newLen) ||
															cmp(ptr,"PROG_F77_C_O",12,newLen) ||
															cmp(ptr,"PROG_FORTRAN",12,newLen) ||
															cmp(ptr,"PROG_GCC_TRADITIONAL",20,newLen) ||
															cmp(ptr,"PROG_INSTALL",12,newLen) ||
															cmp(ptr,"PROG_LEX",8,newLen) ||
															cmp(ptr,"PROG_LN_S",9,newLen) ||
															cmp(ptr,"PROG_MAKE_SET",13,newLen) ||
															cmp(ptr,"PROG_RANLIB",11,newLen) ||
															cmp(ptr,"PROG_YACC",9,newLen))
																pClr = &(g_pSSEXOptions->clrAutoconfBuiltinMacro);
													} else {
														if( cmp(ptr,"PREFIX_PROGRAM",14,newLen) ||
															cmp(ptr,"PREREQ",6,newLen) ||
															cmp(ptr,"PROVIDE",7,newLen))
																pClr = &(g_pSSEXOptions->clrAutoconfBuiltinMacro);
													}
												}
											break;
											case 'R':
												if(cmp(ptr,"REQUIRE",7,newLen) ||
													cmp(ptr,"REPLACE_FUNCS",13,newLen) ||
													cmp(ptr,"REQUIRE_CPP",11,newLen) ||
													cmp(ptr,"REVISION",8,newLen))
														pClr = &(g_pSSEXOptions->clrAutoconfBuiltinMacro);
											break;
											case 'S':
												if(cmp(ptr,"SCO_INTL",8,newLen) ||
													cmp(ptr,"SEARCH_LIBS",11,newLen) ||
													cmp(ptr,"STAT_MACROS_BROKEN",18,newLen) ||
													cmp(ptr,"STRUCT_ST_BLKSIZE",17,newLen) ||
													cmp(ptr,"STRUCT_ST_BLOCKS",16,newLen) ||
													cmp(ptr,"STRUCT_ST_RDEV",14,newLen) ||
													cmp(ptr,"STRUCT_TIMEZONE",15,newLen) ||
													cmp(ptr,"STRUCT_TM",9,newLen) ||
													cmp(ptr,"SUBST",5,newLen) ||
													cmp(ptr,"SUBST_FILE",10,newLen) ||
													cmp(ptr,"SYS_INTERPRETER",15,newLen) ||
													cmp(ptr,"SYS_LONG_FILE_NAMES",19,newLen) ||
													cmp(ptr,"SYS_RESTARTABLE_SYSCALLS",24,newLen))
														pClr = &(g_pSSEXOptions->clrAutoconfBuiltinMacro);
											break;
											case 'T':
												if(cmp(ptr,"TRY_COMPILE",11,newLen) ||
													cmp(ptr,"TRY_CPP",7,newLen) ||
													cmp(ptr,"TRY_LINK",8,newLen) ||
													cmp(ptr,"TRY_RUN",7,newLen) ||
													cmp(ptr,"TRY_LINK_FUNC",13,newLen) ||
													cmp(ptr,"TYPE_GETGROUPS",14,newLen) ||
													cmp(ptr,"TYPE_MODE_T",11,newLen) ||
													cmp(ptr,"TYPE_OFF_T",10,newLen) ||
													cmp(ptr,"TYPE_PID_T",10,newLen) ||
													cmp(ptr,"TYPE_SIGNAL",11,newLen) ||
													cmp(ptr,"TYPE_SIZE_T",11,newLen) ||
													cmp(ptr,"TYPE_UID_T",10,newLen))
														pClr = &(g_pSSEXOptions->clrAutoconfBuiltinMacro);
											break;
											case 'U':
												if(cmp(ptr,"UNISTD_H",8,newLen) ||
													cmp(ptr,"USG",3,newLen))
														pClr = &(g_pSSEXOptions->clrAutoconfBuiltinMacro);
											break;
											case 'V':
												if(cmp(ptr,"VALIDATE_CACHED_SYSTEM_TUPLE",28,newLen) ||
													cmp(ptr,"VERBOSE",7,newLen))
														pClr = &(g_pSSEXOptions->clrAutoconfBuiltinMacro);
											break;
											case 'W':
												if(cmp(ptr,"WITH",4,newLen))
														pClr = &(g_pSSEXOptions->clrAutoconfBuiltinMacro);
											break;
											case 'X':
												if(cmp(ptr,"XENIX_DIR",9,newLen))
														pClr = &(g_pSSEXOptions->clrAutoconfBuiltinMacro);
											break;
										}
									}
								}
							break;
						}
							
						// TOKEN COLORISATION
						//pClr = cppModeGetTokenColor(begin,c - begin);
						if(!pClr)pClr = &(g_pSSEXOptions->clrAutoconfNormalText);
					} else if(*c == '#')
					{
						blockWidth += m_iCharWidth[(unsigned char)(*c)];
						bInLineComment = true;
						c++;
						pClr = &(g_pSSEXOptions->clrAutoconfComment);
					} else if(*c == '$')
					{
						// A variable
						blockWidth += m_iCharWidth[(unsigned char)(*c)];
						c++;
						if(bGotEscape)
						{
							pClr = &(g_pSSEXOptions->clrAutoconfNormalText);
						} else {
							while(	((*c >= 'a') && (*c <= 'z')) ||
									((*c >= 'A') && (*c <= 'Z')) ||
									((*c >= '0') && (*c <= '9')) ||
									(*c == '_'))
							{
								blockWidth += m_iCharWidth[(unsigned char)(*c)];
								c++;
							}
							pClr = &(g_pSSEXOptions->clrAutoconfVariable);
						}
					} else if(*c == '\\')
					{
						// escape
						bGotEscape = 2;
						blockWidth += m_iCharWidth[(unsigned char)(*c)];
						c++;
						pClr = &(g_pSSEXOptions->clrAutoconfOperator);
					} else if( (*c == '(') || (*c == ')') || (*c == '[') || (*c == ']') )
					{
						blockWidth += m_iCharWidth[(unsigned char)(*c)];
						c++;
						pClr = &(g_pSSEXOptions->clrAutoconfBrace);
					} else if((*c == '/') || (*c == '{') || (*c == '}') ||
								(*c == '"') || (*c == '\'') || (*c == '+') || (*c == '-') || (*c == '*') ||
								(*c == '>') || (*c == '<') || (*c == '~') || (*c == '?') || (*c == '=') ||
								(*c == '!') || (*c == '&') || (*c == '|') || (*c == '.') || (*c == ':') || (*c == '`') ||
								(*c == ';') || (*c == '^') || (*c == ',') || (*c == '@') || (*c == '%') || (*c == '\''))
					{
						blockWidth += m_iCharWidth[(unsigned char)(*c)];
						c++;
						pClr = bGotEscape ? &(g_pSSEXOptions->clrAutoconfNormalText) : &(g_pSSEXOptions->clrAutoconfOperator);
					} else {
						while(	(*c) && (*c != ' ') && (*c != '\t') && (*c != '\'') && (*c != '`') &&
								(*c != '{') && (*c != '}') && (*c != '/') &&
								(*c != '(') && (*c != ')') && (*c != '[') &&
								(*c != ']') && (*c != '"') && (*c != '\'') &&
								(*c != '+') && (*c != '-') && (*c != '*') &&
								(*c != '!') && (*c != '<') && (*c != '.') &&
								(*c != '>') && (*c != '^') && (*c != '%') &&
								(*c != '&') && (*c != '|') && (*c != '=') &&
								(*c != ',') && (*c != ';') && (*c != ':') &&
								(*c != '~') && (*c != '?') && (*c != '#') &&
								(*c != '\\') && (*c != '$') && (*c != '@'))
						{
							blockWidth += m_iCharWidth[(unsigned char)(*c)];
							c++;
						}
						pClr = &(g_pSSEXOptions->clrAutoconfNormalText);
					}
					//
					// EXTRACT TOKEN ####################################################################################
					//
				}
				paint.setPen(*pClr);
				paint.drawText(curXPos,m_iFontAscent,begin,c-begin);
				curXPos += blockWidth;
				break;
		}
	}

	if(m_iCursorRow == row)paintCursor(&paint,l,&(g_pSSEXOptions->clrAutoconfCursor));
	if(m_bHasSelection)paintSelection(&paint,l,row,updateR);

	int col_xPos,row_yPos;

	p->worldMatrix().map(0,0,&col_xPos,&row_yPos);

	bitBlt(p->device(),col_xPos + updateR.x(),row_yPos + updateR.y(),m_pMemBuffer,updateR.x(),updateR.y(),updateR.width(),updateR.height());
}






inline void SSEXEditor::recalcCursorPosition(SSEXEditorTextLine *l)
{
	m_iCursorPositionInPixels = getTextWidthWithTabsForCursorPosition(l->text.data(),m_iCursorPosition);
}

int SSEXEditor::cursorCol() const
{
	SSEXEditorTextLine *l = m_pLines->at(m_iCursorRow);
	if(!l)return 0;
	return (m_iCursorPosition < l->length) ? m_iCursorPosition : l->length;
}

/////////////////////////////////////////////////////////////////////////////////////////////////
//
// SELECTION HANDLING HELPERS
//
/////////////////////////////////////////////////////////////////////////////////////////////////

void SSEXEditor::ensureSelectionCoherent()
{
	if((m_selection2.y() < m_selection1.y()) || ((m_selection1.y() == m_selection2.y()) && (m_selection2.x() < m_selection1.x())))
	{
		QPoint aux = m_selection1;
		m_selection1 = m_selection2;
		m_selection2 = aux;
	}
}

void SSEXEditor::setSelectionCoords(int x1,int row1,int x2,int row2)
{
	setHasSelection(true);
	m_selection1 = QPoint(x1,row1);
	m_selection2 = QPoint(x2,row2);
}

void SSEXEditor::clearSelection(bool bUpdate)
{
	setHasSelection(false);
	if(bUpdate)update();	
}

void SSEXEditor::selectionCursorMovement(const QPoint &oldCursorPos,const QPoint &newCursorPos)
{
	if(m_bHasSelection)
	{
		if(m_selection1 == oldCursorPos)m_selection1 = newCursorPos;
		else if(m_selection2 == oldCursorPos)m_selection2 = newCursorPos;
		else setSelectionCoords(oldCursorPos.x(),oldCursorPos.y(),newCursorPos.x(),newCursorPos.y());
	} else setSelectionCoords(oldCursorPos.x(),oldCursorPos.y(),newCursorPos.x(),newCursorPos.y());

	ensureSelectionCoherent();
}
/////////////////////////////////////////////////////////////////////////////////////////////////
//
// CURSOR MOVEMENT
//
/////////////////////////////////////////////////////////////////////////////////////////////////

void SSEXEditor::cursorUp(bool bSelect)
{

	if(m_iCursorRow > 0)
	{
		if(bSelect)selectionCursorMovement(QPoint(m_iCursorPosition,m_iCursorRow),QPoint(m_iCursorPosition,m_iCursorRow - 1));
		else if(m_bHasSelection)clearSelection(true);

		m_iCursorRow--;
		recalcCursorPosition(m_pLines->at(m_iCursorRow));
		ensureCursorVisible();
		if(bSelect)update();
		else {
			updateCell(m_iCursorRow + 1);
			updateCell(m_iCursorRow);
		}
	}
}

void SSEXEditor::cursorDown(bool bSelect)
{
	if(m_iCursorRow < (m_pLines->count() - 1))
	{
		if(bSelect)selectionCursorMovement(QPoint(m_iCursorPosition,m_iCursorRow),QPoint(m_iCursorPosition,m_iCursorRow + 1));
		else if(m_bHasSelection)clearSelection(true);

		m_iCursorRow++;

		recalcCursorPosition(m_pLines->at(m_iCursorRow));
		ensureCursorVisible();
		if(bSelect)update();
		else {
			updateCell(m_iCursorRow - 1);
			updateCell(m_iCursorRow);
		}
	}
}

void SSEXEditor::cursorPageUp(bool bSelect)
{
	if(m_iCursorRow > 0)
	{
		int oldCursorRow = m_iCursorRow;
		m_iCursorRow -=(viewHeight() / cellHeight());
		if(m_iCursorRow < 0)m_iCursorRow = 0;

		if(bSelect)selectionCursorMovement(QPoint(m_iCursorPosition,oldCursorRow),QPoint(m_iCursorPosition,m_iCursorRow));
		else if(m_bHasSelection)clearSelection(true);

		recalcCursorPosition(m_pLines->at(m_iCursorRow));
		ensureCursorVisible();
		update();
	}
}

void SSEXEditor::cursorPageDown(bool bSelect)
{
	if(m_iCursorRow < (m_pLines->count() - 1))
	{
		int oldCursorRow = m_iCursorRow;
		m_iCursorRow += (viewHeight() / cellHeight());
		if(m_iCursorRow >= m_pLines->count())m_iCursorRow = m_pLines->count() - 1;

		if(bSelect)selectionCursorMovement(QPoint(m_iCursorPosition,oldCursorRow),QPoint(m_iCursorPosition,m_iCursorRow));
		else if(m_bHasSelection)clearSelection(true);


		recalcCursorPosition(m_pLines->at(m_iCursorRow));
		ensureCursorVisible();
		update();
	}
}

void SSEXEditor::cursorLeft(bool bSelect)
{	
	SSEXEditorTextLine * l = m_pLines->at(m_iCursorRow);
	if(m_iCursorPosition > l->length)m_iCursorPosition = l->length;
	if(m_iCursorPosition > 0)
	{
		if(bSelect)selectionCursorMovement(QPoint(m_iCursorPosition,m_iCursorRow),QPoint(m_iCursorPosition - 1,m_iCursorRow));
		else if(m_bHasSelection)clearSelection(true);

		m_iCursorPosition--;
		recalcCursorPosition(l);
		ensureCursorVisible();
		if(bSelect)update();
		else {
			updateCell(m_iCursorRow);
		}
	} else {
		if(m_iCursorRow > 0)
		{
			QPoint old(m_iCursorPosition,m_iCursorRow);

			m_iCursorRow--;
			l = m_pLines->at(m_iCursorRow);
			m_iCursorPosition = l->length;

			if(bSelect)selectionCursorMovement(old,QPoint(m_iCursorPosition,m_iCursorRow));
			else if(m_bHasSelection)clearSelection(true);

			recalcCursorPosition(l);
			ensureCursorVisible();
			if(bSelect)update();
			else {
				updateCell(m_iCursorRow + 1);
				updateCell(m_iCursorRow);
			}
		}
	}
}

void SSEXEditor::cursorRight(bool bSelect)
{
	SSEXEditorTextLine * l = m_pLines->at(m_iCursorRow);
	if(l->length > m_iCursorPosition)
	{
		if(bSelect)selectionCursorMovement(QPoint(m_iCursorPosition,m_iCursorRow),QPoint(m_iCursorPosition + 1,m_iCursorRow));
		else if(m_bHasSelection)clearSelection(true);

		m_iCursorPosition++;
		recalcCursorPosition(l);
		ensureCursorVisible();
		if(bSelect)update();
		else updateCell(m_iCursorRow);
	} else {
		if(m_iCursorRow < (m_pLines->count() - 1))
		{
			if(bSelect)selectionCursorMovement(QPoint(m_iCursorPosition,m_iCursorRow),QPoint(0,m_iCursorRow));
			else if(m_bHasSelection)clearSelection(true);

			m_iCursorRow++;
			m_iCursorPosition = 0;
			m_iCursorPositionInPixels = 0;
			ensureCursorVisible();
			if(bSelect)update();
			else {
				updateCell(m_iCursorRow - 1);
				updateCell(m_iCursorRow);
			}
		}
	}
}

void SSEXEditor::cursorHome(bool bSelect)
{
	if(bSelect)selectionCursorMovement(QPoint(m_iCursorPosition,m_iCursorRow),QPoint(0,m_iCursorRow));
	else if(m_bHasSelection)clearSelection(true);

	SSEXEditorTextLine * l = m_pLines->at(m_iCursorRow);
	if(m_iCursorPosition > l->length)m_iCursorPosition = l->length;
	m_iCursorPosition = 0;
	recalcCursorPosition(l);
	ensureCursorVisible();
	if(bSelect)update();
	else updateCell(m_iCursorRow);

}

void SSEXEditor::cursorEnd(bool bSelect)
{	
	SSEXEditorTextLine * l = m_pLines->at(m_iCursorRow);
	if(bSelect)selectionCursorMovement(QPoint(m_iCursorPosition,m_iCursorRow),QPoint(l->length,m_iCursorRow));
	else if(m_bHasSelection)clearSelection(true);
	if(m_iCursorPosition > l->length)m_iCursorPosition = l->length;
	m_iCursorPosition = l->length;
	recalcCursorPosition(l);
	ensureCursorVisible();
	if(bSelect)update();
	else updateCell(m_iCursorRow);
}

/////////////////////////////////////////////////////////////////////////////////////////////////
//
// VIEW ADJUSTEMENTS
//
/////////////////////////////////////////////////////////////////////////////////////////////////

void SSEXEditor::ensureCursorVisible()
{
	if(topCell() > m_iCursorRow)
	{
		setTopCell(m_iCursorRow);
	} else if(topCell() == m_iCursorRow){
		// We want it to be totally visible
//		debug("TOP CELL IS HERE");
		setYOffset(topCell() * cellHeight());
	} else {
		if(lastRowVisible() <= m_iCursorRow)setBottomCell(m_iCursorRow);
	}

	if((m_iCursorPositionInPixels) < xOffset())setXOffset(m_iCursorPositionInPixels);
	else  if((m_iCursorPositionInPixels + SSEX_EDITOR_BORDER + 20) > (viewWidth() + xOffset()))
			setXOffset((m_iCursorPositionInPixels + SSEX_EDITOR_BORDER + 20) - viewWidth());
}

void SSEXEditor::setBottomCell(int row)
{
//	int row_yPos = row * cellHeight();

	int nCellsVisible = viewHeight() / cellHeight();
	int topRow = (row - nCellsVisible) + 1;
	setTopCell(topRow);

//	setYOffset(row_yPos - (viewHeight() - cellHeight()));
}

/////////////////////////////////////////////////////////////////////////////////////////////////
//
// TEXT OPERATIONS
//
/////////////////////////////////////////////////////////////////////////////////////////////////

void SSEXEditor::indentSelected()
{
	if(!m_bHasSelection)return;
	if(m_bHasBlockMark)clearBlockMark(false);


	if(m_selection1.x() > 0)m_selection1.setX(0);

	QCString tmp = selectedText();

	int lastIdx = 0;

	int idx;

	while((idx = tmp.find('\n',lastIdx)) != -1)
	{
		tmp.insert(lastIdx,"\t");
		lastIdx = idx + 2;
	}

	tmp.insert(lastIdx,"\t");

	QPoint sel1 = m_selection1;
	QPoint sel2 = m_selection2;

	killSelection(false);
	insertText(tmp);

	setSelectionCoords(0,sel1.y(),sel2.x() + 1,sel2.y());

	update();
}

void SSEXEditor::unindentSelected()
{
//#warning "This should have an UNIQUE undo...not in two steps! (same for indentSelected())"
	if(!m_bHasSelection)return;
	if(m_bHasBlockMark)clearBlockMark(false);

	if(m_selection1.x() > 0)m_selection1.setX(0);

	QCString tmp = selectedText();

	if(tmp.isNull())tmp = "";

	int lastIdx = 0;
	int idx;

	while((idx = tmp.find('\n',lastIdx)) != -1)
	{
		if(*(tmp.data() + lastIdx) == '\t')
		{
			tmp.remove(lastIdx,1);
			lastIdx = idx;
		} else {
			lastIdx = idx + 1;
		}
	}

	bool bLast = false;

	if(*(tmp.data() + lastIdx) == '\t')
	{
		bLast = true;
		tmp.remove(lastIdx,1);
	}

	QPoint sel1 = m_selection1;
	QPoint sel2 = m_selection2;

	killSelection(false);
	insertText(tmp);

	setSelectionCoords(0,sel1.y(),(sel2.x() > 0 && bLast) ? sel2.x() - 1: sel2.x(),sel2.y());

	update();
}

void SSEXEditor::indent()
{
	if(m_bHasSelection)
	{
		indentSelected();
		return;
	}

	if(m_bHasBlockMark)clearBlockMark(false);

	SSEXEditorTextLine *l = m_pLines->at(m_iCursorRow);
	addUndo(new SSEXUndo("indent line",m_iCursorRow,0,m_iCursorRow,1));
	l->text.insert(0,'\t');
	l->length++;
	l->width  = getTextWidthWithTabs(l->text.data());
	if(m_iMaxTextWidthLine == m_iCursorRow)
	{
		m_iMaxTextWidth = l->width;
		updateCellSize();
	} else {
		if(m_iMaxTextWidth < l->width)
		{
			m_iMaxTextWidth = l->width;
			m_iMaxTextWidthLine = m_iCursorRow;
			updateCellSize();
		}
	}

	if(m_iCursorRow < m_pLines->count() - 1)m_iCursorRow++;
	recalcCursorPosition(m_pLines->at(m_iCursorRow));

	ensureCursorVisible();
	updateCell(m_iCursorRow);
	updateCell(m_iCursorRow - 1);

	setModified(true);
}

void SSEXEditor::unindent()
{
	if(m_bHasSelection)
	{
		unindentSelected();
		return;
	}

	if(m_bHasBlockMark)clearBlockMark(true);

	SSEXEditorTextLine *l = m_pLines->at(m_iCursorRow);

	if(*(l->text.data()) == '\t')
	{
		addUndo(new SSEXUndo("unindent line",m_iCursorRow,0,"\t"));
		l->text.remove(0,1);
		l->length--;
		l->width = getTextWidthWithTabs(l->text.data());
		if(m_iMaxTextWidthLine == m_iCursorRow)
		{
			updateMaxTextWidth();
			updateCellSize();
		}
	} else l=0;
	if(m_iCursorRow < m_pLines->count() - 1)m_iCursorRow++;
	recalcCursorPosition(m_pLines->at(m_iCursorRow));
	ensureCursorVisible();
	updateCell(m_iCursorRow);
	updateCell(m_iCursorRow - 1);
	if(l)setModified(true);
}

bool SSEXEditor::curTextLine(QCString &buf)
{
	SSEXEditorTextLine * l = m_pLines->at(m_iCursorRow);
	if(l)buf = l->text;
	return (l != 0);
}

void SSEXEditor::commentOut(bool bAlternative)
{
	if(m_bHasBlockMark)clearBlockMark(true);
	if(m_bHasSelection)clearSelection(true);

	if((m_mode == Cpp) || (m_mode == Perl)){
		SSEXEditorTextLine *l = m_pLines->at(m_iCursorRow);
		if(m_mode == Cpp)
		{
			if(bAlternative)
			{
				addUndo(new SSEXUndo("add c comment (1)",m_iCursorRow,0,m_iCursorRow,2));
				l->text.insert(0,"/*");
				addUndo(new SSEXUndo("add c comment (2)",m_iCursorRow,l->length + 2,m_iCursorRow,l->length + 4));
				l->text.append("*/");
				l->length += 4;
			} else {
				addUndo(new SSEXUndo("add cpp comment",m_iCursorRow,0,m_iCursorRow,2));
				l->text.insert(0,"//");
				l->length += 2;
			}
		} else {
			addUndo(new SSEXUndo("add shell comment",m_iCursorRow,l->length,m_iCursorRow,l->length + 1));
			l->text.insert(0,"#");
			l->length += 1;
		}
		l->width  = getTextWidthWithTabs(l->text.data());
		if(m_iMaxTextWidthLine == m_iCursorRow){
			m_iMaxTextWidth = l->width;
			updateCellSize();
		} else {
			if(m_iMaxTextWidth < l->width)
			{
				m_iMaxTextWidth = l->width;
				m_iMaxTextWidthLine = m_iCursorRow;
				updateCellSize();
			}
		}

		if(m_iCursorRow < m_pLines->count() - 1)m_iCursorRow++;
		recalcCursorPosition(m_pLines->at(m_iCursorRow));

		ensureCursorVisible();
		updateCell(m_iCursorRow);
		updateCell(m_iCursorRow - 1);
		setModified(true);
	}
}

void SSEXEditor::removeComment()
{
	if(m_bHasBlockMark)clearBlockMark(true);
	if(m_bHasSelection)clearSelection(true);

	if((m_mode == Cpp) || (m_mode == Perl))
	{

		SSEXEditorTextLine *l = m_pLines->at(m_iCursorRow);

		if(m_mode == Cpp)
		{

			if(strncmp(l->text.data(),"//",2) == 0)
			{
				addUndo(new SSEXUndo("remove cpp comment",m_iCursorRow,0,"//"));
				l->text.remove(0,2);
				l->length -= 2;
				l->width = getTextWidthWithTabs(l->text.data());
				if(m_iMaxTextWidthLine == m_iCursorRow){
					updateMaxTextWidth();
					updateCellSize();
				}
			} else if(strncmp(l->text.data(),"/*",2) == 0)
			{
				if((strncmp(l->text.right(2).data(),"*/",2) == 0) && (l->length > 3))
				{
					addUndo(new SSEXUndo("remove c comment (1)",m_iCursorRow,0,"/*"));
					addUndo(new SSEXUndo("remove c comment (2)",m_iCursorRow,l->length - 2,"*/"));

					l->text.remove(0,2);
					l->text.remove(l->text.length() - 2,2);
					l->length -= 4;
					l->width = getTextWidthWithTabs(l->text.data());
	
					if(m_iMaxTextWidthLine == m_iCursorRow)
					{
						updateMaxTextWidth();
						updateCellSize();
					}
				} else l=0;
			} else l=0;
		} else {
			if(strncmp(l->text.data(),"#",1) == 0)
			{
				addUndo(new SSEXUndo("remove shell comment",m_iCursorRow,0,"#"));
				l->text.remove(0,1);
				l->length -= 1;
				l->width = getTextWidthWithTabs(l->text.data());
				if(m_iMaxTextWidthLine == m_iCursorRow){
					updateMaxTextWidth();
					updateCellSize();
				}
			} else l = 0;
		}

		if(m_iCursorRow < m_pLines->count() - 1)m_iCursorRow++;
		recalcCursorPosition(m_pLines->at(m_iCursorRow));
		ensureCursorVisible();
		updateCell(m_iCursorRow);
		updateCell(m_iCursorRow - 1);
		if(l)setModified(true);
	}
}

void SSEXEditor::insertChar(char c)
{
#ifdef __DEBUG
	if((((unsigned char)c) < 32) && (((unsigned char)c) != ((unsigned char)'\t')))debug("INSERTING A STRANGE CHAR! (%c:%d)",c,c);
#endif

	if(m_bHasBlockMark)clearBlockMark(true);
	if(m_bHasSelection)killSelection(true); // trigger the total update too

	SSEXEditorTextLine *l = m_pLines->at(m_iCursorRow);
	char deletedChar = 0;
	if(m_iCursorPosition <= l->length)
	{
		if(m_bOverwrite && (m_iCursorPosition < l->length))
		{
			deletedChar = *(l->text.data() + m_iCursorPosition);
			addUndo(new SSEXUndo("kill char",m_iCursorRow,m_iCursorPosition,l->text.mid(m_iCursorPosition,1)));
			l->text.remove(m_iCursorPosition,1);
		} else l->length++;
		addUndo(new SSEXUndo("insert char",m_iCursorRow,m_iCursorPosition,m_iCursorRow,m_iCursorPosition + 1));
		l->text.insert(m_iCursorPosition,c);
		m_iCursorPosition++;
	} else {
		addUndo(new SSEXUndo("insert char",m_iCursorRow,l->length,m_iCursorRow,l->length + 1));
		l->text.insert(l->length,c);
		l->length++;
		m_iCursorPosition = l->length;
	}
	l->width  = getTextWidthWithTabs(l->text.data());
	recalcCursorPosition(l);
	if(m_iMaxTextWidthLine == m_iCursorRow){
		if(m_bOverwrite)updateMaxTextWidth();
		else m_iMaxTextWidth = l->width;
		updateCellSize();
	} else {
		if(m_iMaxTextWidth < l->width)
		{
			m_iMaxTextWidth = l->width;
			m_iMaxTextWidthLine = m_iCursorRow;
			updateCellSize();
		}
	}

	__assert(l->text.length() == l->length);

	if((!m_bOverwrite) || (c != deletedChar))
	{
		if(m_mode == Cpp)
		{
			// In cpp mode update the comments state if needed
			if((c == '/')||(deletedChar == '/')){
				if((*(l->text.data() + m_iCursorPosition) == '*') || (*(l->text.data() + m_iCursorPosition) == '/') || (l->flags & SSEX_EDITOR_END_IN_COMMENT)){
					cppModeComputeCommentState(l);
					update();
				} else if(m_iCursorPosition > 1){
					if(*(l->text.data() + m_iCursorPosition - 2) == '*'){
						cppModeComputeCommentState(l);
						update();
					}
				}
			} else {
				if(*(l->text.data() + m_iCursorPosition) == '/'){
					cppModeComputeCommentState(l);
					update();
				} else if(m_iCursorPosition > 1){
					if(*(l->text.data() + m_iCursorPosition - 2) == '/'){
						cppModeComputeCommentState(l);
						update();
					}
				}
			}
		} else if(m_mode == Html)
		{
			if((c == '<') ||
			(deletedChar == '<') ||
			(c == '>') ||
			(deletedChar == '>') ||
			(c == '!') ||
			(deletedChar == '!') ||
			(c == '-') ||
			(deletedChar == '-')
			){
				htmlModeComputeTagState(l);
				update();
			}
		}
	}
	ensureCursorVisible();
	updateCell(m_iCursorRow);

	setModified(true);
}

void SSEXEditor::newLine()
{
	if(m_bHasSelection)clearSelection(false);
	if(m_bHasBlockMark)clearBlockMark(false);


	SSEXEditorTextLine *l = m_pLines->at(m_iCursorRow);
	int pos = ((m_iCursorPosition <= l->length) ? m_iCursorPosition : l->length);

	int oldCursorRow = m_iCursorRow;

	SSEXEditorTextLine *n = new SSEXEditorTextLine;
	n->text = (l->text.data() + pos);
	n->length = n->text.length();
	l->length -= n->length;
	l->text.truncate(l->length);
	l->width  = getTextWidthWithTabs(l->text.data());
	n->width  = getTextWidthWithTabs(n->text.data());
	m_pLines->insert(m_iCursorRow + 1,n);
	setNumRows(m_pLines->count());
	if(m_iMaxTextWidthLine == m_iCursorRow){
		updateMaxTextWidth();
		updateCellSize();
	} else {
		// MaxTextWidthLine is scrolled down!
		if(m_iMaxTextWidthLine > m_iCursorRow)m_iMaxTextWidthLine++;
	}
	m_iCursorRow++;
	m_iCursorPosition = 0;

	addUndo(new SSEXUndo("new line",oldCursorRow,pos,m_iCursorRow,m_iCursorPosition));

	recalcCursorPosition(l);

	if(m_mode == Cpp)
	{
		// In cpp mode comments state if needed
		if((l->flags & SSEX_EDITOR_END_IN_COMMENT) && (*(n->text.data()) != '*'))
		{
			n->flags = SSEX_EDITOR_BEGIN_IN_COMMENT | SSEX_EDITOR_END_IN_COMMENT;
		} else cppModeComputeCommentState(l);
	} else if(m_mode == Html)
	{
		if((l->flags & SSEX_EDITOR_END_IN_COMMENT) && (*(n->text.data()) != '-') &&(*(n->text.data()) != '>'))
		{
			n->flags = SSEX_EDITOR_BEGIN_IN_COMMENT | SSEX_EDITOR_END_IN_COMMENT;
		} else	if(l->flags & SSEX_EDITOR_END_IN_TAG)
		{
			n->flags = SSEX_EDITOR_BEGIN_IN_TAG | SSEX_EDITOR_END_IN_TAG;
		} else htmlModeComputeTagState(l);
	}
	ensureCursorVisible();
	update();
	setModified(true);
}

void SSEXEditor::backspace()
{
	if(m_bHasBlockMark)clearBlockMark(true);

	if(m_bHasSelection)
	{
		killSelection(true);
		return;
	}

	SSEXEditorTextLine *l = m_pLines->at(m_iCursorRow);
	int pos = (m_iCursorPosition <= l->length) ? m_iCursorPosition : l->length;

	if(pos == 0){
		// join lines
		if(m_iCursorRow > 0){
			m_iCursorRow--;
			SSEXEditorTextLine *prev = m_pLines->at(m_iCursorRow);
			prev->text.append(l->text);
			m_iCursorPosition = prev->length;
			recalcCursorPosition(prev);
			prev->length += l->length;
			prev->width = getTextWidthWithTabs(prev->text.data());
			if(m_mode == Cpp){
				prev->flags = ((prev->flags & SSEX_EDITOR_BEGIN_IN_COMMENT) | (l->flags & SSEX_EDITOR_END_IN_COMMENT));
				m_pLines->setAutoDelete(false);
				m_pLines->removeRef(l);
				m_pLines->setAutoDelete(true);
				if((*(l->text.data()) == '*') || (*(l->text.data()) == '/'))cppModeComputeCommentState(prev);
				delete l;
			} else if(m_mode == Html)
			{
				prev->flags = ((prev->flags & SSEX_EDITOR_BEGIN_IN_COMMENT) | (l->flags & SSEX_EDITOR_END_IN_COMMENT)) | ((prev->flags & SSEX_EDITOR_BEGIN_IN_TAG) | (l->flags & SSEX_EDITOR_END_IN_TAG));
				m_pLines->setAutoDelete(false);
				m_pLines->removeRef(l);
				m_pLines->setAutoDelete(true);
				if((*(l->text.data()) == '>') || (*(l->text.data()) == '<') || (*(l->text.data()) == '-') || (*(l->text.data()) == '!'))htmlModeComputeTagState(l);
				delete l;
			} else m_pLines->removeRef(l);
			setNumRows(m_pLines->count());

			if((m_iMaxTextWidthLine == m_iCursorRow)||(m_iMaxTextWidthLine == (m_iCursorRow + 1))){
				m_iMaxTextWidth = prev->width;
				m_iMaxTextWidthLine = m_iCursorRow;
				updateCellSize();
			} else {
				if(m_iMaxTextWidth < prev->width)
				{
					m_iMaxTextWidth = prev->width;
					m_iMaxTextWidthLine = m_iCursorRow;
					updateCellSize();
				} else {
					// The maxtextwidthline is scrolled up!
					if(m_iMaxTextWidthLine > (m_iCursorRow + 1))m_iMaxTextWidthLine--;
				}
			}

			addUndo(new SSEXUndo("join lines",m_iCursorRow,m_iCursorPosition,"\n"));

			update(); // we could calc the right rect here
		}
	} else {
		char c = *(l->text.data() + pos - 1);

		QCString undoText = l->text.mid(pos - 1,1);

		l->text.remove(pos - 1,1);
		l->length--;
		l->width = getTextWidthWithTabs(l->text.data());
		m_iCursorPosition = pos - 1;
		recalcCursorPosition(l);
		if(m_iMaxTextWidthLine == m_iCursorRow)
		{
			updateMaxTextWidth();
			updateCellSize();
		}
		pos--;
		if(m_mode == Cpp)
		{
			// In cpp mode update the comments state if needed
			if(c == '/'){
				if((*(l->text.data() + pos) == '*') || (*(l->text.data() + pos) == '/') || (l->flags & SSEX_EDITOR_END_IN_COMMENT)){
					cppModeComputeCommentState(l);
					update();
				} else if(pos > 0){
					if(*(l->text.data() + pos - 1) == '*'){
						cppModeComputeCommentState(l);
						update();
					}
				}
			} else {
				if(*(l->text.data() + pos) == '/'){
					cppModeComputeCommentState(l);
					update();
				} else if(pos > 0){
					if(*(l->text.data() + pos - 1) == '/'){
						cppModeComputeCommentState(l);
						update();
					}
				}
			}
		} else if(m_mode == Html)
		{
			// In html mode update the tag state if needed
			if(	(c == '<') ||
				(c == '>') ||
				(c == '!')||
				(c == '-')
			){
				htmlModeComputeTagState(l);
				update();
			}
		}

		addUndo(new SSEXUndo("delete char",m_iCursorRow,m_iCursorPosition,undoText));


		ensureCursorVisible();
		updateCell(m_iCursorRow);
	}
	setModified(true);
}

void SSEXEditor::goToLineNum(int lineNum)
{
	if(lineNum < 0)return;
	SSEXEditorTextLine * l = m_pLines->at(lineNum);
	if(l)
	{
		clearSelection();
		m_iCursorRow = lineNum;
		m_iCursorPosition = 0;
		recalcCursorPosition(l);
		ensureCursorVisible();
	}
}

void SSEXEditor::goToLine()
{
	int lineNum = QInputDialog::getInteger("Go to line","Insert the line number");
	lineNum--;
	goToLineNum(lineNum);
}

void SSEXEditor::goToBeginning()
{
	goToLineNum(0);
}

void SSEXEditor::goToEnd()
{
	if(m_pLines->count() > 0)goToLineNum(m_pLines->count() - 1);
}

void SSEXEditor::del()
{
	if(m_bHasBlockMark)clearBlockMark(true);

	if(m_bHasSelection)
	{
		killSelection(true);
		return;
	}

	SSEXEditorTextLine *l = m_pLines->at(m_iCursorRow);
	if(m_iCursorPosition > l->length)m_iCursorPosition = l->length;

	if(m_iCursorPosition == l->length){
		// join lines

		if(m_iCursorRow < (m_pLines->count() - 1))
		{
			SSEXEditorTextLine *next = m_pLines->at(m_iCursorRow + 1);
			l->text.append(next->text);
			l->length += next->length;
			l->width = getTextWidthWithTabs(l->text.data());
			if(m_mode == Cpp){
				l->flags = ((l->flags & SSEX_EDITOR_BEGIN_IN_COMMENT) | (next->flags & SSEX_EDITOR_END_IN_COMMENT));
				m_pLines->setAutoDelete(false);
				m_pLines->removeRef(next);
				m_pLines->setAutoDelete(true);
				if((*(next->text.data()) == '*') || (*(next->text.data()) == '/'))cppModeComputeCommentState(l);
				delete next;
			} else if(m_mode == Html)
			{
				l->flags = ((l->flags & SSEX_EDITOR_BEGIN_IN_COMMENT) | (next->flags & SSEX_EDITOR_END_IN_COMMENT)) | ((l->flags & SSEX_EDITOR_BEGIN_IN_TAG) | (next->flags & SSEX_EDITOR_END_IN_TAG));
				m_pLines->setAutoDelete(false);
				m_pLines->removeRef(next);
				m_pLines->setAutoDelete(true);
				if((*(next->text.data()) == '>') || (*(next->text.data()) == '<')|| (*(next->text.data()) == '-') || (*(next->text.data()) == '!'))htmlModeComputeTagState(l);
				delete next;
			} else m_pLines->removeRef(next);

			setNumRows(m_pLines->count());
			if((m_iMaxTextWidthLine == m_iCursorRow)||(m_iMaxTextWidthLine == (m_iCursorRow + 1))){
				m_iMaxTextWidth = l->width;
				m_iMaxTextWidthLine = m_iCursorRow;
				updateCellSize();
			} else {
				if(m_iMaxTextWidth < l->width)
				{
					m_iMaxTextWidth = l->width;
					m_iMaxTextWidthLine = m_iCursorRow;
					updateCellSize();
				}
			}

			addUndo(new SSEXUndo("join lines",m_iCursorRow,m_iCursorPosition,"\n"));

			update(); // we could calc the right rect here
		}
	} else {

		char c = *(l->text.data() + m_iCursorPosition);

		addUndo(new SSEXUndo("kill char",m_iCursorRow,m_iCursorPosition,l->text.mid(m_iCursorPosition,1)));

		l->text.remove(m_iCursorPosition,1);
		l->length--;
		l->width = getTextWidthWithTabs(l->text.data());
		if(m_iMaxTextWidthLine == m_iCursorRow)
		{
			updateMaxTextWidth();
			updateCellSize();
		}
		if(m_mode == Cpp)
		{
			// In cpp mode update the comments state if needed
			if(c == '/'){
				if((*(l->text.data() + m_iCursorPosition) == '*') || (*(l->text.data() + m_iCursorPosition) == '/') || (l->flags & SSEX_EDITOR_END_IN_COMMENT)){
					cppModeComputeCommentState(l);
					update();
				} else if(m_iCursorPosition > 0){
					if(*(l->text.data() + m_iCursorPosition - 1) == '*'){
						cppModeComputeCommentState(l);
						update();
					}
				}
			} else {
				if(*(l->text.data() + m_iCursorPosition) == '/'){
					cppModeComputeCommentState(l);
					update();
				} else if(m_iCursorPosition > 0){
					if(*(l->text.data() + m_iCursorPosition - 1) == '/'){
						cppModeComputeCommentState(l);
						update();
					}
				}
			}
		} else if(m_mode == Html)
		{
			// In html mode update the tag state if needed
			if(	(c == '<') ||
				(c == '>') ||
				(c == '!')||
				(c == '-')
			){
				htmlModeComputeTagState(l);
				update();
			}
		}
		ensureCursorVisible();
		updateCell(m_iCursorRow);
	}
	setModified(true);
}

void SSEXEditor::copy()
{
	if(m_bHasSelection)QApplication::clipboard()->setText(selectedText());
}

void SSEXEditor::paste()
{
	QCString cText = QApplication::clipboard()->text().ascii();
	if(!(cText.isNull() || cText.isEmpty()))insertText(cText);
}

void SSEXEditor::cut()
{
	if(!m_bHasSelection)return;
	copy();
	if(m_bHasBlockMark)clearBlockMark(false);
	killSelection(true);
}


void SSEXEditor::addUndo(SSEXUndo * und)
{
	m_pUndoList->append(und);
	while(m_pUndoList->count() > g_pSSEXOptions->uUndoDepth)m_pUndoList->removeFirst();
}

void SSEXEditor::clearUndoList()
{
	delete m_pUndoList;
	m_pUndoList = new KviPtrList<SSEXUndo>;
	m_pUndoList->setAutoDelete(true);
}

void SSEXEditor::undo()
{
	SSEXUndo * u = m_pUndoList->last();
	if(!u)return;
	if(u->m_type == SSEXUndo::Insert)
	{
		// Insert
		if(m_bHasSelection)clearSelection(true);
		m_iCursorRow = u->m_iRow1;
		m_iCursorPosition = u->m_iCol1;
		insertText(u->m_szText,true,true,false);
	} else {
		// Remove
		if(m_bHasSelection)clearSelection(true);
		setSelectionCoords(u->m_iCol1,u->m_iRow1,u->m_iCol2,u->m_iRow2);
		killSelection(true,true,false);
	}
	m_pUndoList->removeLast();
}

void SSEXEditor::killSelection(bool bUpdate,bool bRecalcWidths,bool bCreateUndo)
{
	if(!m_bHasSelection)return;
	SSEXEditorTextLine * first = m_pLines->at(m_selection1.y());
	if(!first)return;
	// Single line selection
	QCString undoText;
	if(m_selection1.y() == m_selection2.y())
	{
		if(bCreateUndo)undoText = first->text.mid(m_selection1.x(),m_selection2.x() - m_selection1.x());
		first->text.remove(m_selection1.x(),m_selection2.x() - m_selection1.x());
		first->length = first->text.length();
	} else {
		// Multiple lines selection
		if(bCreateUndo)undoText = first->text.mid(m_selection1.x(),first->length - m_selection1.x());
		first->text.remove(m_selection1.x(),first->length - m_selection1.x());
		int idx = m_selection1.y() + 1;
		SSEXEditorTextLine * l = m_pLines->next();
		KviPtrList<SSEXEditorTextLine> list;
		list.setAutoDelete(false);
		while(l && (idx <= m_selection2.y()))
		{
			if(bCreateUndo)undoText.append("\n");
			if(idx == m_selection2.y())
			{
				if(bCreateUndo)undoText.append(l->text.left(m_selection2.x()));

				l->text.remove(0,m_selection2.x());
				first->text.append(l->text);
				list.append(l);
			} else {
				if(bCreateUndo)undoText.append(l->text);

				list.append(l);
			}
			l = m_pLines->next();
			idx++;
		}
		first->length = first->text.length();
		// Kill the middle lines
		for(SSEXEditorTextLine *line=list.first();line;line=list.next())m_pLines->removeRef(line);
		setNumRows(m_pLines->count());
	}

	first->width = getTextWidthWithTabs(first->text.data());

	setHasSelection(false);
	m_iCursorRow = m_selection1.y();
	m_iCursorPosition = m_selection1.x() < first->length ? m_selection1.x() : first->length;
	
	if(bCreateUndo)addUndo(new SSEXUndo("kill selection",m_iCursorRow,m_iCursorPosition,undoText));


	recalcCursorPosition(first);
	if(bRecalcWidths)
	{
		updateMaxTextWidth();
		updateCellSize();
		if(m_mode == Cpp)cppModeComputeCommentState(first);
		else if(m_mode == Html)htmlModeComputeTagState(first);
	}
	if(bUpdate){
		ensureCursorVisible();
		update();
	}
	setModified(true);
}

void SSEXEditor::insertText(QCString &text,bool bUpdate,bool bRecalcWidths,bool bCreateUndo)
{
	if(m_bHasSelection)killSelection(false,false);
	if(m_bHasBlockMark)clearBlockMark(false);

	SSEXEditorTextLine * first = m_pLines->at(m_iCursorRow);
	if(!first)return;



	QCString rightPart = (m_iCursorPosition < first->length) ? (first->text.data() + m_iCursorPosition) : "";
	__assert(first->text.length() == first->length);
	first->text.remove(m_iCursorPosition,first->length - m_iCursorPosition);

	SSEXEditorTextLine * l = first;
	int row = m_iCursorRow;

	int oldCursorRow = m_iCursorRow;
	int oldCursorPosition = m_iCursorPosition;

	QCString theText = text;
	int idx = theText.find('\n');
	while(idx != -1)
	{
		l->text.append(theText.left(idx));
		theText.remove(0,idx + 1);
		l->length = l->text.length();
		l->width = getTextWidthWithTabs(l->text.data());
		row++;
		l = new SSEXEditorTextLine;
		l->text = "";
		l->length = 0;
		l->width = 0;
		m_pLines->insert(row,l);
		idx = theText.find('\n');
	}
	setNumRows(m_pLines->count());

	m_iCursorRow = row;
	m_iCursorPosition = l->text.length() + theText.length(); 

	if(bCreateUndo)addUndo(new SSEXUndo("insert text",oldCursorRow,oldCursorPosition,m_iCursorRow,m_iCursorPosition));

	l->text.append(theText);
	l->text.append(rightPart);
	l->length = l->text.length();
	l->width = getTextWidthWithTabs(l->text.data());


	recalcCursorPosition(l);
	if(bRecalcWidths)
	{
		updateMaxTextWidth();
		updateCellSize();
		if(m_mode == Cpp)cppModeComputeCommentState(first);
		else if(m_mode == Html)htmlModeComputeTagState(first);
	}
	if(bUpdate)
	{
		ensureCursorVisible();
		update();
	}
	setModified(true);
}

void SSEXEditor::wheelEvent(QWheelEvent *e)
{
 	if(e->delta () < 0)
	{
 		cursorDown(0);
 	} else {
 		cursorUp(0);
 	}
}

QCString SSEXEditor::selectedText()
{
	QCString ret = "";
	if(!m_bHasSelection)return ret;
	SSEXEditorTextLine * l = m_pLines->at(m_selection1.y());
	if(!l)return ret;
	if(m_selection1.y() == m_selection2.y()){
		return l->text.mid(m_selection1.x(),m_selection2.x() - m_selection1.x());
	} else {
		if(m_selection1.x() < l->length)ret = (l->text.data() + m_selection1.x());
		int idx = m_selection1.y();
		do{
			l = m_pLines->next();
			idx++;
			ret.append("\n");
			if(m_selection2.y() == idx)
			{
				ret.append(l->text.left(m_selection2.x()));
			} else ret.append(l->text);
		} while(l && (idx < m_selection2.y()));
	}
	return ret;	
}

void SSEXEditor::findNext()
{
	QCString toFind = m_pFindWidget->m_pFindStringEdit->text().ascii();
	if(toFind.isEmpty() || toFind.isNull()){
		QMessageBox::warning(this,"Find next","No text to find",QMessageBox::Ok | QMessageBox::Default,0);
		return;
	}

	SSEXEditorTextLine *l = m_pLines->at(m_iCursorRow);

	int row = m_iCursorRow;
	int col = m_iCursorPosition;

	while(l){

		if(col < l->length){
			int idx = l->text.find(toFind.data(),col,m_pFindWidget->m_pCaseSensitiveCheckBox->isChecked());
			if(idx != -1)
			{
				// Found an occurence
				m_iCursorRow = row;
				m_iCursorPosition = idx + toFind.length();
				recalcCursorPosition(l);
				setSelectionCoords(idx,row,m_iCursorPosition,row);
				ensureCursorVisible();
				m_bCursorOn = true;
				update();
				setFocus();
				return;
			}
		}
		if(row >= (m_pLines->count() - 1))
		{
			// End reached
			if(QMessageBox::information(this,"Find next","No occurences found.\nContinue from the beginning ?",
				QMessageBox::Yes | QMessageBox::Default,QMessageBox::No | QMessageBox::Escape) != QMessageBox::Yes)return;
			row = 0;

		} else row++;

		col = 0;
		l = m_pLines->at(row);
	}
	// newer here
}

void SSEXEditor::findPrev()
{
	QCString toFind = m_pFindWidget->m_pFindStringEdit->text().ascii();
	if(toFind.isEmpty() || toFind.isNull())
	{
		QMessageBox::warning(this,"Find prev","No text to find",QMessageBox::Ok | QMessageBox::Default,0);
		return;
	}

	SSEXEditorTextLine *l = m_pLines->at(m_iCursorRow);

	int row = m_iCursorRow;
	int col = m_iCursorPosition - 1;

	while(l){

		if(col >= (toFind.length() - 1)){
			int idx = l->text.findRev(toFind.data(),col,m_pFindWidget->m_pCaseSensitiveCheckBox->isChecked());
			if(idx != -1)
			{
				// Found an occurence
				m_iCursorRow = row;
				m_iCursorPosition = idx;
				recalcCursorPosition(l);
				setSelectionCoords(idx,row,idx + toFind.length(),row);
				ensureCursorVisible();
				m_bCursorOn = true;
				update();
				setFocus();
				return;
			}
		}
		if(row <= 0)
		{
			// End reached
			if(QMessageBox::information(this,"Find prev","No occurences found.\nContinue from the end of the file ?",
				QMessageBox::Yes | QMessageBox::Default,QMessageBox::No | QMessageBox::Escape) != QMessageBox::Yes)return;
			row = m_pLines->count() - 1;

		} else row--;

		l = m_pLines->at(row);
		col = l->length;
	}
	// newer here
}

void SSEXEditor::findRegExp(QString &regexp)
{
	int row = 0;
	int col = 0;

	QRegExp rx(regexp,true);

	int len = 1;

	SSEXEditorTextLine *l = m_pLines->at(0);


	while(l){

		if(col < l->length)
		{
#if QT_VERSION >= 300
			int idx = rx.search(l->text,col);
			len = rx.matchedLength();
#else
			int idx = rx.match(l->text,col,&len);
#endif
			if(idx != -1)
			{
				// Found an occurence
				m_iCursorRow = row;
				m_iCursorPosition = idx + len;
				recalcCursorPosition(l);
//				setSelectionCoords(idx,row,m_iCursorPosition,row);
				ensureCursorVisible();
				m_bCursorOn = true;
				update();
				setFocus();
				return;
			}
		}
		if(row >= (m_pLines->count() - 1))
		{
			// End reached
			return;

		} else row++;

		col = 0;
		l = m_pLines->at(row);
	}
}

void SSEXEditor::findNextRegExp()
{
	QCString toFind = m_pFindWidget->m_pFindStringEdit->text().ascii();
	if(toFind.isEmpty() || toFind.isNull()){
		QMessageBox::warning(this,"Find next reg exp","No regular expression to find",QMessageBox::Ok | QMessageBox::Default,0);
		return;
	}

	SSEXEditorTextLine *l = m_pLines->at(m_iCursorRow);

	int row = m_iCursorRow;
	int col = m_iCursorPosition;

	QRegExp rx(toFind,m_pFindWidget->m_pCaseSensitiveCheckBox->isChecked());

	int len = 1;

	while(l){

		if(col < l->length){
#if QT_VERSION >= 300
			int idx = rx.search(l->text,col);
			len = rx.matchedLength();
#else
			int idx = rx.match(l->text,col,&len);
#endif
			if(idx != -1)
			{
				// Found an occurence
				m_iCursorRow = row;
				m_iCursorPosition = idx + len;
				recalcCursorPosition(l);
				setSelectionCoords(idx,row,m_iCursorPosition,row);
				ensureCursorVisible();
				m_bCursorOn = true;
				update();
				setFocus();
				return;
			}
		}
		if(row >= (m_pLines->count() - 1))
		{
			// End reached
			if(QMessageBox::information(this,"Find next reg exp","No occurences found.\nContinue from the beginning ?",
				QMessageBox::Yes | QMessageBox::Default,QMessageBox::No | QMessageBox::Escape) != QMessageBox::Yes)return;
			row = 0;

		} else row++;

		col = 0;
		l = m_pLines->at(row);
	}
	// newer here
}

void SSEXEditor::findPrevRegExp()
{
	QCString toFind = m_pFindWidget->m_pFindStringEdit->text().ascii();
	if(toFind.isEmpty() || toFind.isNull()){
		QMessageBox::warning(this,"Find prev reg exp","No regular expression to find",QMessageBox::Ok | QMessageBox::Default,0);
		return;
	}

	SSEXEditorTextLine *l = m_pLines->at(m_iCursorRow);

	int row = m_iCursorRow;
	int col = m_iCursorPosition - 1;

	QRegExp rx(toFind,m_pFindWidget->m_pCaseSensitiveCheckBox->isChecked());
	int len = 1;

	while(l){

		if(col >= (toFind.length() - 1)){
			int idx = col;
			bool bFound = false;

			while((idx >= 0) && (!bFound)){
#if QT_VERSION >= 300
				if(rx.search(l->text,idx) != idx)
				{
					len = rx.matchedLength();
					idx--;
				}
#else
				if(rx.match(l->text,idx,&len) != idx)idx--;
#endif
				else bFound = true;
			}

			if(idx != -1)
			{
				// Found an occurence
				m_iCursorRow = row;
				m_iCursorPosition = idx;
				recalcCursorPosition(l);
				setSelectionCoords(idx,row,idx + len,row);
				ensureCursorVisible();
				m_bCursorOn = true;
				update();
				setFocus();
				return;
			}
		}
		if(row <= 0)
		{
			// End reached
			if(QMessageBox::information(this,"Find prev reg exp","No occurences found.\nContinue from the end of the file ?",
				QMessageBox::Yes | QMessageBox::Default,QMessageBox::No | QMessageBox::Escape) != QMessageBox::Yes)return;
			row = m_pLines->count() - 1;

		} else row--;

		l = m_pLines->at(row);
		col = l->length;
	}
	// newer here
}

void SSEXEditor::replace()
{
	if(!m_bHasSelection)
	{
		QMessageBox::warning(this,"Replace","No text selected",QMessageBox::Ok,0);
		return;
	}
	QCString toReplace = m_pFindWidget->m_pReplaceStringEdit->text().ascii();
	if(toReplace.isNull())toReplace = "";
	insertText(toReplace);
	setFocus();
}

void SSEXEditor::replaceAndFindNext()
{
	replace();
	findNext();
}

void SSEXEditor::replaceAndFindNextRegExp()
{
	replace();
	findNextRegExp();
}

void SSEXEditor::replaceAllInSelection()
{
	if(!m_bHasSelection)
	{
		QMessageBox::warning(this,"Replace all in selection","No selection to search in",QMessageBox::Ok | QMessageBox::Default,0);
		return;
	}

	QPoint topLeft = m_selection1;
	QPoint bottomRight = m_selection2;

	clearSelection(false);

	QCString toReplace = m_pFindWidget->m_pReplaceStringEdit->text().ascii();
	if(toReplace.isNull())toReplace = "";
	QCString toFind = m_pFindWidget->m_pFindStringEdit->text().ascii();
	if(toFind.isEmpty() || toFind.isNull())
	{
		QMessageBox::warning(this,"Replace all in selection","No text to find",QMessageBox::Ok | QMessageBox::Default,0);
		return;
	}

	SSEXEditorTextLine *l = m_pLines->at(topLeft.y());

	int row = topLeft.y();
	int col = topLeft.x();

	bool bFound;
	int count = 0;

	while(l && (row <= bottomRight.y())){

		bFound = false;

		if(col < l->length){
			int idx = l->text.find(toFind.data(),col,m_pFindWidget->m_pCaseSensitiveCheckBox->isChecked());
			if(idx != -1)
			{
				if(row == bottomRight.y())
				{
					if((idx + toFind.length()) > bottomRight.x())
					{
						// finished
						break;
					}
				}
				// Found an occurence
				m_iCursorRow = row;
				m_iCursorPosition = idx + toFind.length();
				recalcCursorPosition(l);
				setSelectionCoords(idx,row,m_iCursorPosition,row);
				insertText(toReplace,false,false);
				col = m_iCursorPosition;
				bFound = true;
				count ++;
			} else bFound = false;
		}


		if(!bFound)
		{
			row++;
			col = 0;
			l = m_pLines->at(row);
		}
	}

	updateMaxTextWidth();
	updateCellSize();
	if(m_mode == Cpp)cppModeComputeCommentState(m_pLines->first());
	else if(m_mode == Html)htmlModeComputeTagState(m_pLines->first());
	m_bCursorOn = true;
	ensureCursorVisible();
	update();

	QCString msg;
	msg.sprintf("Replaced %d occurences",count);
	emit textMessage(this,msg);
	setFocus();
}

void SSEXEditor::replaceAllInSelectionRegExp()
{
	if(!m_bHasSelection)
	{
		QMessageBox::warning(this,"Replace all in selection reg exp","No selection to search in",QMessageBox::Ok | QMessageBox::Default,0);
		return;
	}

	QPoint topLeft = m_selection1;
	QPoint bottomRight = m_selection2;

	clearSelection(false);

	QCString toReplace = m_pFindWidget->m_pReplaceStringEdit->text().ascii();
	if(toReplace.isNull())toReplace = "";
	QCString toFind = m_pFindWidget->m_pFindStringEdit->text().ascii();
	if(toFind.isEmpty() || toFind.isNull()){
		QMessageBox::warning(this,"Replace all in selection reg exp","No regular expression to find",QMessageBox::Ok | QMessageBox::Default,0);
		return;
	}

	QRegExp rx(toFind,m_pFindWidget->m_pCaseSensitiveCheckBox->isChecked());

	SSEXEditorTextLine *l = m_pLines->at(topLeft.y());

	int row = topLeft.y();
	int col = topLeft.x();

	bool bFound;
	int count = 0;
	int len = 1;

	while(l && (row <= bottomRight.y())){

		bFound = false;

		if(col < l->length){
#if QT_VERSION >= 300
			int idx = rx.search(l->text,col);
			len = rx.matchedLength();
#else
			int idx = rx.match(l->text,col,&len);
#endif
			if(idx != -1)
			{
				if(row == bottomRight.y())
				{
					if((idx + len) > bottomRight.x())
					{
						// finished
						break;
					}
				}
				// Found an occurence
				m_iCursorRow = row;
				m_iCursorPosition = idx + len;
				recalcCursorPosition(l);
				setSelectionCoords(idx,row,m_iCursorPosition,row);
				insertText(toReplace,false,false);
				col = m_iCursorPosition;
				bFound = true;
				count ++;
			} else bFound = false;
		}


		if(!bFound)
		{
			row++;
			col = 0;
			l = m_pLines->at(row);
		}
	}

	updateMaxTextWidth();
	updateCellSize();
	if(m_mode == Cpp)cppModeComputeCommentState(m_pLines->first());
	else if(m_mode == Html)htmlModeComputeTagState(m_pLines->first());
	m_bCursorOn = true;
	ensureCursorVisible();
	update();

	QCString msg;
	msg.sprintf("Replaced %d occurences",count);
	emit textMessage(this,msg);
	setFocus();
}



void SSEXEditor::replaceAll()
{
	if(QMessageBox::warning(this,"Replace all",
		"This may be a really destructive action\n" \
		"Really replace all the occurences from\n" \
		"the cursor position to end of the file ?",
		QMessageBox::Yes | QMessageBox::Default,QMessageBox::No | QMessageBox::Escape) == QMessageBox::No)return;

	QCString toReplace = m_pFindWidget->m_pReplaceStringEdit->text().ascii();
	if(toReplace.isNull())toReplace = "";
	QCString toFind = m_pFindWidget->m_pFindStringEdit->text().ascii();
	if(toFind.isEmpty() || toFind.isNull()){
		QMessageBox::warning(this,"Replace all","No text to find",QMessageBox::Ok,0);
		return;
	}

	SSEXEditorTextLine *l = m_pLines->at(m_iCursorRow);

	int row = m_iCursorRow;
	int col = m_iCursorPosition;

	bool bFound;
	int count = 0;

	while(l){

		bFound = false;

		if(col < l->length){
			int idx = l->text.find(toFind.data(),col,m_pFindWidget->m_pCaseSensitiveCheckBox->isChecked());
			if(idx != -1)
			{
				// Found an occurence
				m_iCursorRow = row;
				m_iCursorPosition = idx + toFind.length();
				recalcCursorPosition(l);
				setSelectionCoords(idx,row,m_iCursorPosition,row);
				insertText(toReplace,false,false);
				col = m_iCursorPosition;
				bFound = true;
				count ++;
			} else bFound = false;
		}

		if(!bFound)
		{
			if(row >= (m_pLines->count() - 1))
			{
				// End reached
				updateMaxTextWidth();
				updateCellSize();
				if(m_mode == Cpp)cppModeComputeCommentState(m_pLines->first());
				else if(m_mode == Html)htmlModeComputeTagState(m_pLines->first());
				m_bCursorOn = true;
				ensureCursorVisible();
				update();
				QCString msg;
				msg.sprintf("Replaced %d occurences",count);
				emit textMessage(this,msg);
				if(QMessageBox::information(this,"Replace all","No more occurences found.\nContinue from the beginning ?",
					QMessageBox::Yes | QMessageBox::Default,QMessageBox::No | QMessageBox::Escape) != QMessageBox::Yes)
				{
					setFocus();
					return;
				}
				row = 0;
				count = 0;
			} else row++;

			col = 0;
			l = m_pLines->at(row);
		}
	}
}

void SSEXEditor::replaceAllRegExp()
{
	if(QMessageBox::warning(this,"Replace all (regular expression)",
		"This may be a really destructive action\n" \
		"Really replace all the occurences from\n" \
		"the cursor position to end of the file ?",
		QMessageBox::Yes | QMessageBox::Default,QMessageBox::No | QMessageBox::Escape) == QMessageBox::No)return;


	QCString toReplace = m_pFindWidget->m_pReplaceStringEdit->text().ascii();
	if(toReplace.isNull())toReplace = "";
	QCString toFind = m_pFindWidget->m_pFindStringEdit->text().ascii();
	if(toFind.isEmpty() || toFind.isNull()){
		QMessageBox::warning(this,"Replace all","No regular expression to find",QMessageBox::Ok | QMessageBox::Default,0);
		return;
	}

	QRegExp rx(toFind,m_pFindWidget->m_pCaseSensitiveCheckBox->isChecked());

	SSEXEditorTextLine *l = m_pLines->at(m_iCursorRow);

	int row = m_iCursorRow;
	int col = m_iCursorPosition;

	bool bFound;
	int len = 1;

	while(l){

		bFound = false;

		if(col < l->length){
#if QT_VERSION >= 300
			int idx = rx.search(l->text,col);
			len =rx.matchedLength();
#else
			int idx = rx.match(l->text,col,&len);
#endif
			if(idx != -1)
			{
				// Found an occurence
				m_iCursorRow = row;
				m_iCursorPosition = idx + len;
				recalcCursorPosition(l);
				setSelectionCoords(idx,row,m_iCursorPosition,row);
				insertText(toReplace,false,false);
				col = m_iCursorPosition;
				bFound = true;
			} else bFound = false;
		}

		if(!bFound)
		{
			if(row >= (m_pLines->count() - 1))
			{
				// End reached
				updateMaxTextWidth();
				updateCellSize();
				if(m_mode == Cpp)cppModeComputeCommentState(m_pLines->first());
				else if(m_mode == Html)htmlModeComputeTagState(m_pLines->first());
				m_bCursorOn = true;
				ensureCursorVisible();
				update();
				if(QMessageBox::information(this,"Replace all reg exp","No more occurences found.\nContinue from the beginning ?",
					QMessageBox::Yes | QMessageBox::Default,QMessageBox::No | QMessageBox::Escape) != QMessageBox::Yes)
				{
					setFocus();
					return;
				}
				row = 0;

			} else row++;

			col = 0;
			l = m_pLines->at(row);
		}
	}

}

void SSEXEditor::switchMode()
{
	switch(m_mode)
	{
		case Normal: setMode(Cpp);      break;
		case Cpp:    setMode(Html);     break;
		case Html:   setMode(Perl);     break;
		case Perl:   setMode(Autoconf); break;
		default:     setMode(Normal);   break;
	}
}

void SSEXEditor::setMode(ColorMode mode)
{
	if(mode != m_mode)
	{
		m_mode = mode;
		emit modeChanged(this);
		initializeCurrentMode();
		update();
	}
}

void SSEXEditor::recordKeystrokes()
{
	if(m_bRecordingKeystrokes)
	{
		m_bRecordingKeystrokes = false;
		emit recordingKeystrokes(false);
		return;
	}
	m_pKeystrokes->clear();
	m_bRecordingKeystrokes = true;
	emit recordingKeystrokes(true);
}

#undef KeyPress

void SSEXEditor::replayKeystrokes()
{
	if(m_bRecordingKeystrokes)
	{
		m_bRecordingKeystrokes = false;
		emit recordingKeystrokes(false);
		return;
	}

	for(SSEXEditorKeystroke *k=m_pKeystrokes->first();k;k=m_pKeystrokes->next())
	{
		QKeyEvent ev(QEvent::KeyPress,k->key,k->ascii,k->state);
		keyPressEvent(&ev);
	}
}

void SSEXEditor::keyPressEvent(QKeyEvent *e)
{
	if(	(e->key() == Qt::Key_Alt) ||
		(e->key() == Qt::Key_Meta) ||
		(e->key() == Qt::Key_Control) ||
		(e->key() == Qt::Key_Shift))
	{
		e->ignore();
		return;
	}

	if(m_bRecordingKeystrokes)
	{
		if(!(((e->key() == Qt::Key_T) || (e->key() == Qt::Key_R)) && (e->state() & ControlButton)))
		{
			SSEXEditorKeystroke * k = new SSEXEditorKeystroke;
			k->ascii = e->ascii();
			k->key = e->key();
			k->state = e->state();
			m_pKeystrokes->append(k);
		}
	}

	m_bCursorOn = true;
	bool bShiftOn = (e->state() & ShiftButton);

	if(e->state() & ControlButton)
	{
		switch(e->key())
		{
			case Qt::Key_Q:     replaceAndFindNext();          e->accept(); return; break;
			case Qt::Key_W:     replaceAndFindNextRegExp();    e->accept(); return; break;
			case Qt::Key_E:     replaceAllInSelection();       e->accept(); return; break;
			case Qt::Key_R:     recordKeystrokes();            e->accept(); return; break;
			case Qt::Key_T:     replayKeystrokes();            e->accept(); return; break;
			case Qt::Key_Y:     goToLine();                    e->accept(); return; break;
			case Qt::Key_U:     unindent();                    e->accept(); return; break;
			case Qt::Key_I:     indent();                      e->accept(); return; break;
			case Qt::Key_O:     commentOut(bShiftOn);          e->accept(); return; break;
			case Qt::Key_P:     removeComment();               e->accept(); return; break;

			case Qt::Key_A:     saveFileAs();                  e->accept(); return; break;
			case Qt::Key_S:     saveFile();                    e->accept(); return; break;
			case Qt::Key_D:     replaceAllInSelectionRegExp(); e->accept(); return; break;
			case Qt::Key_F:     toggleFindWidget();            e->accept(); return; break;
			case Qt::Key_G:     findPrevRegExp();              e->accept(); return; break;
			case Qt::Key_H:     findPrev();                    e->accept(); return; break;
			case Qt::Key_J:     replace();                     e->accept(); return; break;
			case Qt::Key_K:     replaceAll();                  e->accept(); return; break;
			case Qt::Key_L:     replaceAllRegExp();            e->accept(); return; break;

			case Qt::Key_Z:     undo();                        e->accept(); return; break;
			case Qt::Key_X:     cut();                         e->accept(); return; break;
			case Qt::Key_C:     copy();                        e->accept(); return; break;
			case Qt::Key_V:     paste();                       e->accept(); return; break;
			case Qt::Key_B:     findNextRegExp();              e->accept(); return; break;
			case Qt::Key_N:     findNext();                    e->accept(); return; break;
			case Qt::Key_M:     switchMode();                  e->accept(); return; break;
			case Qt::Key_Comma: markBlock();                   e->accept(); return; break;
		}
		e->ignore();
		return;
	}


	switch(e->key())
	{
		case Qt::Key_Up:        cursorUp(bShiftOn);         e->accept(); return; break;
		case Qt::Key_Down:      cursorDown(bShiftOn);       e->accept(); return; break;
		case Qt::Key_Right:     cursorRight(bShiftOn);      e->accept(); return; break;
		case Qt::Key_Left:      cursorLeft(bShiftOn);       e->accept(); return; break;
		case Qt::Key_PageUp:    cursorPageUp(bShiftOn);     e->accept(); return; break;
		case Qt::Key_PageDown:  cursorPageDown(bShiftOn);   e->accept(); return; break;
		case Qt::Key_Home:      cursorHome(bShiftOn);       e->accept(); return; break;
		case Qt::Key_End:       cursorEnd(bShiftOn);        e->accept(); return; break;

		case Qt::Key_Backspace: backspace();                e->accept(); return; break;
		case Qt::Key_Delete:    del();                      e->accept(); return; break;
		case Qt::Key_Insert:    m_bOverwrite = !m_bOverwrite; e->accept(); return; break;
		case Qt::Key_Return:
		case Qt::Key_Enter:
			if(m_pFindWidget->isVisible())findNext();
			else newLine();
			e->accept();
			return;
		break;
		default:
			if(    ((((unsigned char)e->ascii()) >= 32) && (((unsigned char)e->ascii()) <= 255)) || (e->ascii() == '\t')){
				insertChar(e->ascii());
				e->accept();
				return;
			}
			break;
	}

	e->ignore();
}

#if QT_VERSION >= 300
void SSEXEditor::contextMenuEvent(QContextMenuEvent *e)
{
//	debug("CONTEXT EVENT");
//	contextPopup(e->pos());
	e->ignore();
}
#endif

void SSEXEditor::contextPopup(const QPoint &pos)
{
	int theRow = findRow(pos.y());
	if(theRow < 0)theRow = m_pLines->count() - 1;
	SSEXEditorTextLine * l = m_pLines->at(theRow);
	int charIndex = findCharacterAt((pos.x() - frameWidth()) + xOffset(),l);
	if(charIndex >= l->length)charIndex = l->length - 1;
	contextPopup(l,charIndex);
}

void SSEXEditor::mousePressEvent(QMouseEvent *e)
{
//	debug("MOUSE PRESS");
	if(e->button() & LeftButton){
		m_iCursorRow = findRow(e->pos().y());
		if(m_iCursorRow < 0)m_iCursorRow = m_pLines->count() - 1;
		SSEXEditorTextLine * l = m_pLines->at(m_iCursorRow);
		m_iCursorPosition = findCharacterAt((e->pos().x() - frameWidth()) + xOffset(),l);
		recalcCursorPosition(l);
		ensureCursorVisible();
		if(m_bHasSelection)clearSelection(false);
		m_selection1 = QPoint(m_iCursorPosition,m_iCursorRow);
		m_selection2 = m_selection1;
		m_mouseAnchor = m_selection1;
		m_lastMousePressCoords = QPoint(e->pos().x() + xOffset(),e->pos().y() + yOffset());
		update();
	} else if(e->button() & MidButton)paste();
	else if(e->button() & RightButton)
	{
//		debug("RIGHT BUTTON");
		contextPopup(e->pos());
	}
//	SSEXTableView::mousePressEvent(e);
}

void SSEXEditor::fillEditPopup(QPopupMenu * p)
{
	QCString tmp;
	SSEXUndo * u = m_pUndoList->last();
	if(u)
	{
		tmp = "&Undo \"";
		tmp.append(u->m_szName);
		tmp.append("\" (CTRL+Z)");
	} else {
		tmp = "Can't undo";
	}


	int id = p->insertItem(QString(tmp.data()),this,SLOT(undo()));
	if(!u)p->setItemEnabled(id,false);
	p->insertSeparator();
	id = p->insertItem("&Copy (CTRL+C)",this,SLOT(copy()));
	if(!m_bHasSelection)p->setItemEnabled(id,false);
	id = p->insertItem("&Paste (CTRL+V)",this,SLOT(paste()));
	QString t = QApplication::clipboard()->text();
	if(t.isNull() || t.isEmpty())p->setItemEnabled(id,false);
	id = p->insertItem("&Cut (CTRL+X)",this,SLOT(cut()));
	if(!m_bHasSelection)p->setItemEnabled(id,false);

	p->insertSeparator();
	p->insertItem("&Go to line (CTRL+Y)",this,SLOT(goToLine()));
	p->insertItem("Go to &beginning",this,SLOT(goToBeginning()));
	p->insertItem("Go to &end",this,SLOT(goToEnd()));
	p->insertSeparator();
	if(m_bHasSelection)
	{
		id = p->insertItem("&Indent selected (CTRL+I)",this,SLOT(indentSelected()));
		id = p->insertItem("&Unindent selected (CTRL+U)",this,SLOT(unindentSelected()));
	} else {
		id = p->insertItem("&Indent current line (CTRL+I)",this,SLOT(indent()));
		id = p->insertItem("&Unindent current line (CTRL+U)",this,SLOT(unindent()));
	}
	p->insertSeparator();
	p->insertItem("&Mark block (CTRL+.)",this,SLOT(markBlock()));
	p->insertSeparator();

	p->insertItem(m_bRecordingKeystrokes ? "Stop &recording keystrokes (CTRL+R)" : "&Record keystrokes (CTRL+R)",this,SLOT(recordKeystrokes()));
	id = p->insertItem("Replay keys&trokes (CTRL+T)",this,SLOT(replayKeystrokes()));
	if(m_bRecordingKeystrokes || (m_pKeystrokes->count() == 0))p->setItemEnabled(id,false);
}

void SSEXEditor::fillFindPopup(QPopupMenu * p)
{
	p->insertItem(m_pFindWidget->isVisible() ? "Hide &find widget (CTRL+F)" : "&Show &find widget (CTRL+F)",this,SLOT(toggleFindWidget()));
	p->insertSeparator();

	QCString toFind = m_pFindWidget->m_pFindStringEdit->text().ascii();
	bool bHasToFind = !(toFind.isEmpty() || toFind.isNull());
	int id = p->insertItem("Find &next (CTRL+N)",this,SLOT(findNext()));
	if(!bHasToFind)p->setItemEnabled(id,false);
	id = p->insertItem("Find &prev (CTRL+H)",this,SLOT(findPrev()));
	if(!bHasToFind)p->setItemEnabled(id,false);
	id = p->insertItem("Find next &reg exp (CTRL+B)",this,SLOT(findNextRegExp()));
	if(!bHasToFind)p->setItemEnabled(id,false);
	id = p->insertItem("Find prev reg &exp (CTRL+G)",this,SLOT(findPrevRegExp()));
	if(!bHasToFind)p->setItemEnabled(id,false);
	id = p->insertItem("Rep&lace (CTRL+J)",this,SLOT(replace()));
	if(!m_bHasSelection)p->setItemEnabled(id,false);
	id = p->insertItem("Replace &All (CTRL+K)",this,SLOT(replaceAll()));
	if(!bHasToFind)p->setItemEnabled(id,false);
	id = p->insertItem("Replace All (Re&gexp) (CTRL+L)",this,SLOT(replaceAllRegExp()));
	if(!bHasToFind)p->setItemEnabled(id,false);
	id = p->insertItem("Replace all in &selection (CTRL+E)",this,SLOT(replaceAllInSelection()));
	if(!(bHasToFind && m_bHasSelection))p->setItemEnabled(id,false);
	id = p->insertItem("Replace all &in selection (regexp) (CTRL+D)",this,SLOT(replaceAllInSelectionRegExp()));
	if(!(bHasToFind && m_bHasSelection))p->setItemEnabled(id,false);
}

void SSEXEditor::editPopupAboutToShow()
{
	m_pEditPopup->clear();

	fillEditPopup(m_pEditPopup);
}

void SSEXEditor::findPopupAboutToShow()
{
	m_pFindPopup->clear();

	fillFindPopup(m_pFindPopup);
}

void SSEXEditor::contextPopup(SSEXEditorTextLine *l,int charIndex)
{
	if(m_pContextPopup->isVisible())return;
	m_pContextPopup->clear();

	m_pContextPopup->insertItem("&Edit",m_pEditPopup);
	m_pContextPopup->insertItem("&Find",m_pFindPopup);

	m_pContextPopup->insertSeparator();
	int id = m_pContextPopup->insertItem("&Save (CTRL+S)",this,SLOT(saveFile()));
	if(!m_bModified)m_pContextPopup->setItemEnabled(id,false);
	m_pContextPopup->insertItem("Save &as (CTRL+A)",this,SLOT(saveFileAs()));

	m_pContextPopup->insertSeparator();

	m_pContextPopup->insertItem("Choose font (temporary)",this,SLOT(chooseFont()));

	m_pContextPopup->insertSeparator();

	m_pContextPopup->insertItem("Configure...",this,SLOT(editOptions()));

//#warning "The openinclude stuff should be inside the editor!"
	emit rightClickOnTextRow(this,l->text,charIndex,m_pContextPopup);

	m_pContextPopup->popup(QCursor::pos());
}



void SSEXEditor::dragTimer()
{
	// Send fake mouse events....
	QMouseEvent e(QEvent::MouseMove,mapFromGlobal(QCursor::pos()),LeftButton,LeftButton);
	mouseMoveEvent(&e);
}

void SSEXEditor::mouseMoveEvent(QMouseEvent *e)
{
	if(e->state() & LeftButton)
	{
		QPoint mousePos = e->pos();

		int lastCursorYPosition = m_iCursorRow;

		int rowUnderMouse = findRow(e->pos().y());
		if(mousePos.y() < minViewY())
		{
			if(!m_pDragTimer->isActive())m_pDragTimer->start(SSEX_EDITOR_DRAG_TIMEOUT);
			if(m_iCursorRow > 0)rowUnderMouse = m_iCursorRow - 1;
			else return;
			mousePos.setY(frameWidth() + 1);
		} else if(mousePos.y() > maxViewY())
		{
			if(!m_pDragTimer->isActive())m_pDragTimer->start(SSEX_EDITOR_DRAG_TIMEOUT);
			if(m_iCursorRow < (m_pLines->count() - 1))rowUnderMouse = m_iCursorRow + 1;
			else return;
			mousePos.setY(frameWidth() + viewHeight() - 1);
		} else if(m_pDragTimer->isActive())m_pDragTimer->stop();

		m_iCursorRow = rowUnderMouse;

		SSEXEditorTextLine * l = m_pLines->at(m_iCursorRow);
		m_iCursorPosition = findCharacterAt((mousePos.x() - frameWidth()) + xOffset(),l);
		recalcCursorPosition(l);
		ensureCursorVisible();

		if(!m_bHasSelection)setHasSelection(true);

		lastCursorYPosition = ((lastCursorYPosition * cellHeight()) - yOffset()) + frameWidth();
		
		if(m_mouseAnchor.y() == m_iCursorRow)
		{
			if(m_lastMousePressCoords.x() <= mousePos.x())
			{
				m_selection1 = m_mouseAnchor;
				m_selection2 = QPoint(m_iCursorPosition,m_iCursorRow);
			} else {
				m_selection2 = m_mouseAnchor;
				m_selection1 = QPoint(m_iCursorPosition,m_iCursorRow);
			}
		} else {
			if(m_lastMousePressCoords.y() <= (mousePos.y() + yOffset()))
			{
				m_selection1 = m_mouseAnchor;
				m_selection2 = QPoint(m_iCursorPosition,m_iCursorRow);
			} else {
				m_selection2 = m_mouseAnchor;
				m_selection1 = QPoint(m_iCursorPosition,m_iCursorRow);
			}
		}

		int cH = cellHeight() + 1;
		if(lastCursorYPosition > mousePos.y())update(QRect(0,mousePos.y() - cH,width(),(lastCursorYPosition - mousePos.y()) + (cH << 1)));
		else update(QRect(0,lastCursorYPosition - cH,width(),(mousePos.y() - lastCursorYPosition) + (cH << 1)));
	}
}

void SSEXEditor::mouseReleaseEvent(QMouseEvent *)
{
	if(m_pDragTimer->isActive())m_pDragTimer->stop();
}

void SSEXEditor::updateCellSize()
{
	setCellHeight(m_iFontLineSpacing);
	setCellWidth(m_iMaxTextWidth + (SSEX_EDITOR_BORDER * 2));
	m_pMemBuffer->resize(cellWidth() + 1,cellHeight() + 1);
}


void SSEXEditor::updateFontDependantVariables()
{
	QFontMetrics fm(font());
	int spaceSize = fm.width(' ');
	m_iTabsNumPixels = spaceSize * g_pSSEXOptions->uTabsNumSpaces;
	for(int i=0;i<256;i++)m_iCharWidth[i] = fm.width((char)i);
	m_iFontAscent = fm.ascent();
	m_iFontLineSpacing = fm.lineSpacing();
	updateCellSize();
}

void SSEXEditor::chooseFont()
{
	bool bOk;
	QFont f = QFontDialog::getFont(&bOk,font(),this);
	if(bOk)setFont(f);
}

void SSEXEditor::setFont(const QFont &f)
{
	SSEXTableView::setFont(f);
	updateFontDependantVariables();
	if(m_pLines->count() > m_iCursorRow)
	{
		SSEXEditorTextLine *l = m_pLines->at(m_iCursorRow);
		if(l)recalcCursorPosition(l);
	}
	update();
}

void SSEXEditor::updateOptions()
{
	initializeCurrentMode();
	update();
}

void SSEXEditor::clear()
{
	setText("");
	setModified(true);
}

int SSEXEditor::getTextWidthWithTabsForCursorPosition(const char *text,int cursorPos)
{
	int totWidth = 0;
	int lastTabStop = 0;
	while(*text && cursorPos)
	{
		if(*text == '\t')
		{
			while(lastTabStop <= totWidth)lastTabStop += m_iTabsNumPixels;
			totWidth = lastTabStop;
		} else totWidth += m_iCharWidth[(unsigned char)(*text)];
		text++;
		if(!(--cursorPos))return totWidth;
	}
	return totWidth;
}

int SSEXEditor::findCharacterAt(int xPositionInCell,SSEXEditorTextLine * l)
{
	const char *aux = l->text.data();
	int curXPos = SSEX_EDITOR_BORDER;
	int lastTabStop = SSEX_EDITOR_BORDER;
	while(*aux)
	{
		if(*aux == '\t')
		{
			while(lastTabStop <= curXPos)lastTabStop += m_iTabsNumPixels;
			if(xPositionInCell < (curXPos + ((lastTabStop - curXPos) >> 1)))return (aux - l->text.data());
			else {
				curXPos = lastTabStop;
				aux++;
			}
		} else {
			if(xPositionInCell < (curXPos + (m_iCharWidth[(unsigned char)*aux] >> 1)))return (aux - l->text.data());
			else {
				curXPos += m_iCharWidth[(unsigned char)*aux];
				aux++;
			}
		}
	}
	return l->length;
}

int SSEXEditor::getTextWidthWithTabs(const char *text)
{
	int totWidth = 0;
	int lastTabStop = 0;
	while(*text)
	{
		if(*text == '\t')
		{
			while(lastTabStop <= totWidth)lastTabStop += m_iTabsNumPixels;
			totWidth = lastTabStop;
		} else totWidth += m_iCharWidth[(unsigned char)*text];
		text++;
	}
	return totWidth;
}

void SSEXEditor::editOptions()
{
	g_pSSEXOptions->edit();
}

void SSEXEditor::setText(const QCString &text)
{
	m_iCursorRow = 0;
	m_iCursorPosition = 0;

	m_bHasSelection = false;
	m_bHasBlockMark = false;

	m_pLines->clear();

	const char *aux = text.data();

	if(!aux)aux = "";

	do {
		const char * begin = aux;
		while(*aux && (*aux != '\n'))aux++;
		SSEXEditorTextLine * l = new SSEXEditorTextLine;
		l->length = aux - begin;
		l->text = QCString(begin,l->length + 1);
		if(*aux)aux++;
		l->width = getTextWidthWithTabs(l->text.data());
		l->flags = 0;
		m_pLines->append(l);
	} while(*aux);

	setNumRows(m_pLines->count());

	initializeCurrentMode();

	updateMaxTextWidth();
	updateCellSize();
	recalcCursorPosition(m_pLines->first());

	setOffset(0,0);

	clearUndoList();

	update();
}

bool SSEXEditor::loadFile(const char *filename)
{
	QFile f(filename);
	if(!f.open(IO_ReadOnly))return false;
	QCString buffer;
	buffer.resize(f.size() + 1);
	int readed = f.readBlock(buffer.data(),f.size());
	*(buffer.data() + readed) = '\0';
	f.close();
	QCString fExtension = filename;
	int idx = fExtension.findRev('.');

	if(readed != buffer.length())
	{
		// Binary file!
	}

	// Lookup #!/bin/something magic files!

	ColorMode mode = Normal;
	if(idx != -1)
	{
		fExtension.remove(0,idx+1);
		if(strcmp(fExtension.data(),"cpp") == 0)mode = Cpp;
		else if(strcasecmp(fExtension.data(),"c") == 0)mode = Cpp;
		else if(strcasecmp(fExtension.data(),"cc") == 0)mode = Cpp;
		else if(strcasecmp(fExtension.data(),"pl") == 0)mode = Perl;
		else if(strcasecmp(fExtension.data(),"sh") == 0)mode = Perl;
		else if(strcasecmp(fExtension.data(),"am") == 0)mode = Perl;
		else if(strcasecmp(fExtension.data(),"in") == 0)mode = Autoconf;
		else if(strcasecmp(fExtension.data(),"m4") == 0)mode = Autoconf;
		else if(strcasecmp(fExtension.data(),"kvs") == 0)mode = Perl;
		else if(strncasecmp(filename,"Makefile",8) == 0)mode = Perl;
		else if(strcasecmp(fExtension.data(),"java") == 0)mode = Cpp;
		else if(strcasecmp(fExtension.data(),"cxx") == 0)mode = Cpp;
		else if(strcasecmp(fExtension.data(),"CPP") == 0)mode = Cpp;
		else if(strcasecmp(fExtension.data(),"h") == 0)mode = Cpp;
		else if(strcasecmp(fExtension.data(),"hxx") == 0)mode = Cpp;
		else if(strcasecmp(fExtension.data(),"hpp") == 0)mode = Cpp;
		else if(strcasecmp(fExtension.data(),"moc") == 0)mode = Cpp;
		else if(strcasecmp(fExtension.data(),"s") == 0)mode = Cpp;
		else if(strcasecmp(fExtension.data(),"xpm") == 0)mode = Cpp;
		else if(strcasecmp(fExtension.data(),"html") == 0)mode = Html;
		else if(strcasecmp(fExtension.data(),"htm") == 0)mode = Html;
		else if(strcasecmp(fExtension.data(),"SGML") == 0)mode = Html;
		else if(strcasecmp(fExtension.data(),"xml") == 0)mode = Html;
	}
	setMode(mode);
	setText(buffer);
	m_szFileName = filename;
	setModified(false);
	emit fileNameChanged(this,m_szFileName);
	return true;
}

bool SSEXEditor::saveFile()
{
	if(m_szFileName.isEmpty() || m_szFileName.isNull())return saveFileAs();
	else return saveFile(m_szFileName.data());
}

bool SSEXEditor::saveFile(const char *filename)
{
	QFile f(filename);
	if(!f.open(IO_WriteOnly)){
		QMessageBox::warning(this,"Warning","Can not open the file for writing.\nSave failed",QMessageBox::Ok | QMessageBox::Default,0);
		return false;
	}
	int i = 0;
	int lastProg = -1;
	int progress;
	if(m_pLines->count() == 0){
		emit saveProgress(m_szFileName,100);
	} else {
		emit saveProgress(m_szFileName,0);
		for(SSEXEditorTextLine *l=m_pLines->first();l;l=m_pLines->next())
		{
			__assert(l->text.length() == l->length);
//#ifdef __DEBUG
//			if(l->text.length() != l->length)debug("WARNING : Wrong line size (%d) , should be %d for (%s)",l->length,l->text.length(),l->text.data());
//#endif
			if((f.writeBlock(l->text.data(),l->length) != l->length) || (f.writeBlock("\n",1) != 1))
			i++;
			progress = (i * 100) / m_pLines->count();
			if(progress != lastProg){
				emit saveProgress(m_szFileName,progress);
				lastProg = progress;
			}
		}
	}
	f.close();
	if(strcmp(m_szFileName.data(),filename) != 0){
		m_szFileName = filename;
		emit fileNameChanged(this,m_szFileName);
	}
	setModified(false);
	emit saved(m_szFileName);
	return true;
}

bool SSEXEditor::saveFileAs()
{
	QString newName = QFileDialog::getSaveFileName(m_szFileName,QString::null,0);
	if(newName.isNull())return false; //Aborted
	QFileInfo fi(newName);
	if(fi.exists())
	{
		QString msg;
		msg.sprintf("The file %s already exists.\nDo you want to overwrite it ?",newName.ascii());
		int result = QMessageBox::information(this,"Warning",msg,QMessageBox::Yes,QMessageBox::No | QMessageBox::Default);
		if(result != QMessageBox::Yes)return false; // Aborted
	}
	return saveFile(newName.ascii());
}

void SSEXEditor::initializeCurrentMode()
{
	QPalette pal = palette();
	switch(m_mode)
	{
		case Cpp:
			cppModeComputeCommentState(m_pLines->first());
			setBackgroundColor(g_pSSEXOptions->clrCppExtBackground);
			setBackgroundMode(NoBackground);
			setFont(g_pSSEXOptions->fntCpp);
		break;
		case Html:
			htmlModeComputeTagState(m_pLines->first());
			setBackgroundColor(g_pSSEXOptions->clrHtmlExtBackground);
			setBackgroundMode(NoBackground);
			setFont(g_pSSEXOptions->fntHtml);
		break;
		case Perl:
			setBackgroundColor(g_pSSEXOptions->clrPerlExtBackground);
			setBackgroundMode(NoBackground);
			setFont(g_pSSEXOptions->fntPerl);
		break;
		case Autoconf:
			setBackgroundColor(g_pSSEXOptions->clrAutoconfExtBackground);
			setBackgroundMode(NoBackground);
			setFont(g_pSSEXOptions->fntAutoconf);
		break;
		default:
			setBackgroundColor(g_pSSEXOptions->clrTxtExtBackground);
			setBackgroundMode(NoBackground);
			setFont(g_pSSEXOptions->fntTxt);
		break;
	}
}

void SSEXEditor::cppModeComputeCommentState(SSEXEditorTextLine * start)
{
	if(!start)return;
	if(m_pLines->findRef(start) < 0)return;
	bool bInComment = (start->flags & SSEX_EDITOR_BEGIN_IN_COMMENT);
	for(SSEXEditorTextLine * l=start;l;l=m_pLines->next())
	{
		l->flags = (bInComment ? SSEX_EDITOR_BEGIN_IN_COMMENT : 0);
		bool bInString = false;
		bool bInChar   = false;
		const char *c = l->text.data();
		while(*c)
		{
			if(bInComment){
				while(*c){
					if(*c == '*'){
						c++;
						if(*c == '/'){
							c++;
							bInComment = false;
							break;
						}
					} else c++;
				}
			} else {	
				while(*c){
					if(*c == '/'){
						c++;
						if(!(bInString || bInChar)){
							if(*c == '/'){
								// line comment
								while(*c)c++;
							} else if(*c =='*'){
								c++;
								bInComment = true;
								break;
							}
						} // else in string..no comment begins
					} else if(*c == '"'){
						// no comment begins in strings
						if(bInString){
							if(c != l->text.data()){
								const char * aux = c - 1;
								if(*aux != '\\')bInString = !bInString;
							}
						} else bInString = !bInString;
						c++;
					} else if(*c == '\''){
						// no comment begins in character constants
						if(bInChar){
							if(c != l->text.data()){
								const char * aux = c - 1;
								if(*aux != '\\')bInChar = !bInChar;
							}
						} else bInChar = !bInChar;
						c++;
					} else c++;
				}
			}
		}
		if(bInComment)l->flags |= SSEX_EDITOR_END_IN_COMMENT;
	}
}


void SSEXEditor::htmlModeComputeTagState(SSEXEditorTextLine * start)
{
	if(!start)return;
	if(m_pLines->findRef(start) < 0)return;

	bool bInComment = (start->flags & SSEX_EDITOR_BEGIN_IN_COMMENT);
	bool bInTag     = (start->flags & SSEX_EDITOR_BEGIN_IN_TAG);

	for(SSEXEditorTextLine * l=start;l;l=m_pLines->next())
	{
		l->flags = (bInComment ? SSEX_EDITOR_BEGIN_IN_COMMENT : (bInTag ? SSEX_EDITOR_BEGIN_IN_TAG : 0));

		const char *c = l->text.data();

		while(*c)
		{
			if(bInComment){
				while(*c){
					if(*c == '-'){
						c++;
						if(!(*c))break;
						if(*c == '-'){
							c++;
							if(!(*c))break;
							if(*c == '>'){
								c++;
								bInComment = false;
								break;
							}
						} else c++;
					} else c++;
				}
			} else if(bInTag){
				while(*c){
					if(*c == '>')
					{
						c++;
						bInTag = false;
						break;
					} else if(*c == '<')
					{
						c++;
						if(!(*c))break;
						if(*c == '!')
						{
							c++;
							if(!(*c))break;
							if(*c == '-')
							{
								c++;
								if(!(*c))break;
								if(*c == '-')
								{
									c++;
									bInTag = false;
									bInComment = true;
									break;
								} else c++;
							} else c++;
						} else c++;
					} else c++;
				}
			} else {
				// not in tag nor comment
				while(*c){
					if(*c == '<'){
						c++;
						bInTag = true;
						if(!(*c))break;
						if(*c == '!')
						{
							c++;
							if(!(*c))break;
							if(*c == '-')
							{
								c++;
								if(!(*c))break;
								if(*c == '-')
								{
									c++;
									bInTag = false;
									bInComment = true;
								}
							}
						}
						break;
					} else c++;
				}
			}
		}

		if(bInComment)l->flags |= SSEX_EDITOR_END_IN_COMMENT;
		else if(bInTag)l->flags |= SSEX_EDITOR_END_IN_TAG;
	}
}

#include "m_editor.moc"
