//
//   File : dirbrowserwidget.cpp
//   Creation date : Thu Aug 10 2000 17:42:12 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//


#include "dirbrowserwidget.h"

#include "kvi_frame.h"
#include "kvi_iconmanager.h"
#include "kvi_locale.h"
#include "kvi_process.h"
#include "kvi_app.h"
#include "kvi_msgbox.h"
#include "kvi_module.h"
#include "kvi_thread.h"
#include "kvi_mediatype.h"
#include "kvi_uparser.h"
#include "kvi_console.h"

#include "dirbrowserwindow.h"

#include <qmessagebox.h>
#include <qtoolbutton.h>
#include <qlineedit.h>
#include <qtooltip.h>
#include <qtimer.h>
#include <qiconview.h>
#include <qdir.h>
#include <qcursor.h>

#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>

extern KviPtrList<KviDirBrowserWindow> * g_pDirBrowserWindowList;
extern KviPtrList<KviDirBrowserWidget> * g_pDirBrowserWidgetList;
extern KviMediaManager * g_pMediaManager;
//extern KviDirBrowserIcons          * g_pDirBrowserIcons;




KviDirBrowserThread::KviDirBrowserThread(KviDirBrowserWidget * w,const char * szDir)
: KviSensitiveThread()
{
	m_pDirBrowserWidget = w;
	m_szDir = szDir;
}

KviDirBrowserThread::~KviDirBrowserThread()
{
}

bool KviDirBrowserThread::processEvents()
{
	bool bContinue = true;
	while(KviThreadEvent * e = dequeueEvent())
	{
		if(e->id() == KVI_THREAD_EVENT_TERMINATE)
		{
			bContinue = false;
		}
		delete e;
	}
	return bContinue;
}

static KviMediaType * analyze_directory_entry(const char * szFullPath,KviMediaType * data = 0)
{
	g_pMediaManager->lock();
	KviMediaType * t = g_pMediaManager->findMediaType(szFullPath);
	if(t)
	{
		if(!data)data = new KviMediaType;
		if(t->szIcon.hasData())
		{
			if(g_pApp->findImage(data->szIcon,t->szIcon.ptr()))
			{
				t->szIcon = data->szIcon;
			} else {
				t->szIcon = ""; // not found
			}
		}
		KviMediaManager::copyMediaType(data,t);
		g_pMediaManager->unlock();
		return data;
	}
	g_pMediaManager->unlock();

	return 0;
}

void KviDirBrowserThread::processEntry(const char * fName)
{
	KviStr tmp(KviStr::Format,"%s%s",m_szDir.ptr(),fName);

	KviMediaType * t = analyze_directory_entry(tmp.ptr());
	if(t)
	{
		KviDirBrowserData * d = new KviDirBrowserData;
		d->szFileName = fName;
		d->szKey.sprintf("%c%s%s",kvi_strEqualCI(t->szIanaType.ptr(),"inode/directory") ? '0' : '1',t->szIanaType.ptr(),d->szFileName.ptr());
		d->pMediaType = t;

		KviThreadDataEvent<KviDirBrowserData> * tde = new KviThreadDataEvent<KviDirBrowserData>(KVI_THREAD_EVENT_DATA);
		tde->setData(d);
		postEvent(m_pDirBrowserWidget,tde);
	}
}

void KviDirBrowserThread::run()
{
	m_szDir.ensureLastCharIs('/');

	DIR * d = opendir(m_szDir.ptr());
	if(!d)
	{
		KviThreadDataEvent<KviStr> * e = new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_ERROR);
		KviStr * tmp = new KviStr(KviStr::Format,__tr_no_lookup("Can't open directory %s"),m_szDir.ptr());
		e->setData(tmp);
		postEvent(m_pDirBrowserWidget,e);
		return;
	}

	struct stat st;

	while(struct dirent * de = readdir(d))
	{
		if(!kvi_strEqualCS(de->d_name,"."))
		{
			processEntry(de->d_name);
		}

		if(!processEvents())
		{
			closedir(d);
			return;
		}
	}

	closedir(d);

	postEvent(m_pDirBrowserWidget,new KviThreadEvent(KVI_THREAD_EVENT_TERMINATE));
}





KviStr KviFileIconDrag::m_szMime1 = "application/x-qiconlist";
KviStr KviFileIconDrag::m_szMime2 = "text/uri-list";

KviFileIconDrag::KviFileIconDrag( QWidget * dragSource, const char* name )
: QIconDrag( dragSource, name )
{

}

const char * KviFileIconDrag::format( int i ) const
{
	if(i == 0)return m_szMime1.ptr();
	else {
		if ( i == 1 )return m_szMime2.ptr();
	}
	return 0;
}

QByteArray KviFileIconDrag::encodedData(const char * mime) const
{
	QByteArray a;
	if(kvi_strEqualCI(mime,m_szMime1.ptr()))
	{
		a = QIconDrag::encodedData( mime );
    } else if(kvi_strEqualCI(mime,m_szMime2.ptr()))
	{
		QString s = m_lUrls.join("\r\n");
		a.resize(s.length());
		memcpy(a.data(),s.latin1(),s.length());
    }
    return a;
}

bool KviFileIconDrag::canDecode( QMimeSource* e )
{
	return (e->provides(m_szMime1.ptr()) || e->provides(m_szMime2.ptr()));
}

void KviFileIconDrag::append(const QIconDragItem &item,const QRect &pr,const QRect &tr,const QString &url)
{
	QIconDrag::append( item, pr, tr );
	m_lUrls << url;
}





KviDirBrowserIconView::KviDirBrowserIconView(KviDirBrowserWidget * par,const char * nam)
: QIconView(par,nam)
{
	m_pDirBrowserWidget = par;
}

KviDirBrowserIconView::~KviDirBrowserIconView()
{
}

QDragObject * KviDirBrowserIconView::dragObject()
{
	if(!currentItem())return 0;

 	QPoint orig = viewportToContents(viewport()->mapFromGlobal(QCursor::pos()));

	KviFileIconDrag *drag = new KviFileIconDrag(viewport());
	drag->setPixmap(*currentItem()->pixmap(),
		QPoint(currentItem()->pixmapRect().width() / 2, currentItem()->pixmapRect().height() / 2));

	for(QIconViewItem *item = firstItem(); item; item = item->nextItem())
	{
		if (item->isSelected())
		{
			QIconDragItem id;

			KviStr szFName = item->text();
			KviStr tmp(KviStr::Format,"file:%s%s",m_pDirBrowserWidget->currentDirectory(),szFName.ptr());
			//debug("data is %s",tmp.ptr());
			id.setData(QCString(tmp.ptr()));
			drag->append(id,
				QRect(item->pixmapRect(false).x() - orig.x(),item->pixmapRect(false).y() - orig.y(),item->pixmapRect().width(),item->pixmapRect().height()),
				QRect(item->textRect(false).x() - orig.x(),item->textRect(false).y() - orig.y(),item->textRect().width(),item->textRect().height()),
				tmp.ptr());
		}
	}

    return drag;
}



KviDirBrowserWidget::KviDirBrowserWidget(QWidget * par,KviFrame * lpFrm,bool bIsStandalone)
: QWidget(par,"dirbrowser_widget")
{

	if(bIsStandalone)g_pDirBrowserWidgetList->append(this);
	m_bIsStandalone = bIsStandalone;

	m_pSlaveThread = 0;

	m_pMenuBar = new QMenuBar(this);

	QPopupMenu * filePopup = new QPopupMenu(m_pMenuBar);
//	filePopup->insertItem(__tr("&New window"),this,SLOT(newWindow()));
	filePopup->insertSeparator();
	filePopup->insertItem(__tr("&Close"),this,SLOT(doClose()));

	m_pMenuBar->insertItem(__tr("&File"),filePopup);

	QPopupMenu * goPopup = new QPopupMenu(m_pMenuBar);

	goPopup->insertItem(__c2q(__tr("&Home")),this,SLOT(home()));
	goPopup->insertItem(__tr("&Up"),this,SLOT(cdUp()));

	m_pMenuBar->insertItem(__tr("&Go"),goPopup);



	m_pStatusLabel = new QLabel(__tr("."),this);

//	m_pSearchJob = 0;
	m_pIconView = new KviDirBrowserIconView(this,"icon_view");
	m_pIconView->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	m_pIconView->setFocusPolicy(QWidget::NoFocus);

	m_pIconView->setGridX(70);
	m_pIconView->setGridY(50);
	m_pIconView->setAutoArrange(true);
	m_pIconView->setResizeMode(QIconView::Adjust);
	m_pIconView->setSorting(true,true);
	m_pIconView->setSelectionMode(QIconView::Extended);
	connect(m_pIconView,SIGNAL(doubleClicked(QIconViewItem *)),this,SLOT(itemDoubleClicked(QIconViewItem *)));

	m_pToolBar = new QToolBar("the_toolbar",lpFrm,this,"dirbrowser_tool_bar");
//	m_pBtnIndex = new QToolButton(*g_pIconManager->getBigIcon(KVI_BIGICON_DIRBROWSERINDEX),
//		__tr("Main index"),__tr("Main index"),this,SLOT(showIndex()),m_pToolBar);
//	m_pBtnIndex->setUsesBigPixmap(true);
//	m_pBtnBackward = new QToolButton(*g_pIconManager->getBigIcon(KVI_BIGICON_DIRBROWSERBACKWARD),
//		__tr("Previous document"),__tr("Previous document"),m_pTextBrowser,SLOT(backward()),m_pToolBar);
//	m_pBtnBackward->setEnabled(false);
//	m_pBtnBackward->setUsesBigPixmap(true);
//	m_pBtnForward = new QToolButton(*g_pIconManager->getBigIcon(KVI_BIGICON_DIRBROWSERFORWARD),
//		__tr("Next document"),__tr("Next document"),m_pTextBrowser,SLOT(forward()),m_pToolBar);
//	m_pBtnForward->setEnabled(false);
//	m_pBtnForward->setUsesBigPixmap(true);
//	if(bIsStandalone)
//	{
//		QToolButton * b = new QToolButton(*g_pIconManager->getBigIcon(KVI_BIGICON_DIRBROWSERCLOSE),
//		__tr("Close this window"),__tr("Close this window"),this,SLOT(doClose()),m_pToolBar);
//		b->setUsesBigPixmap(true);
//	}
//	m_pBtnSearch = new QToolButton(*g_pIconManager->getBigIcon(KVI_BIGICON_DIRBROWSERSEARCH),
//		__tr("Search"),__tr("Search"),this,SLOT(doSearch()),m_pToolBar);
//	m_pBtnSearch->setUsesBigPixmap(true);
//
//	m_pSearchInput = new QLineEdit(m_pToolBar);
//	QToolTip::add(m_pSearchInput,__tr("Search keywords"));
	m_pDirectoryEdit = new QLineEdit(m_pToolBar);
	connect(m_pDirectoryEdit,SIGNAL(returnPressed()),this,SLOT(returnPressed()));
	QToolTip::add(m_pDirectoryEdit,__tr("Current directory"));
	m_pToolBar->setStretchableWidget(m_pDirectoryEdit);
}

KviDirBrowserWidget::~KviDirBrowserWidget()
{
	if(m_bIsStandalone)g_pDirBrowserWidgetList->removeRef(this);
	if(m_pSlaveThread)killSlave();
	KviThreadManager::killPendingEvents(this);
}

void KviDirBrowserWidget::killSlave()
{
	if(!m_pSlaveThread)return;

	m_pSlaveThread->terminate();
	delete m_pSlaveThread;
	m_pSlaveThread = 0;
}

void KviDirBrowserWidget::returnPressed()
{
	KviStr szDir = m_pDirectoryEdit->text();
	if(szDir.hasData())browse(szDir.ptr());
}

void KviDirBrowserWidget::browse(const char * szDir)
{
	if(m_pSlaveThread)killSlave();
	m_pIconView->clear();
	m_szCurrentDir = szDir;
	m_szCurrentDir.ensureLastCharIs('/');
	m_pDirectoryEdit->setText(m_szCurrentDir.ptr());
	m_pSlaveThread = new KviDirBrowserThread(this,m_szCurrentDir.ptr());
	m_pSlaveThread->start();
	emit directoryChanged();
	
}

void KviDirBrowserWidget::home()
{
	KviStr tmp = QDir::homeDirPath();
	browse(tmp.ptr());
}

void KviDirBrowserWidget::cdUp()
{
	KviStr szFullPath = m_szCurrentDir;
	if(szFullPath.lastCharIs('/'))szFullPath.cutFromLast('/');
	szFullPath.cutFromLast('/');
	if(szFullPath.isEmpty())szFullPath = "/";
	browse(szFullPath.ptr());
}

void KviDirBrowserWidget::itemDoubleClicked(QIconViewItem * it)
{
	if(!it)return;
	KviStr tmp = it->text();
	KviStr szFullPath = tmp;
	szFullPath.prepend(m_szCurrentDir);
	KviMediaType m;
	KviMediaType * pM = analyze_directory_entry(szFullPath.ptr(),&m);
	if(!pM)return;

	if(kvi_strEqualCI(pM->szIanaType,"inode/directory"))
	{
		// cd
		if(kvi_strEqualCI(tmp.ptr(),".."))
		{
			cdUp();
		} else {
			browse(szFullPath.ptr());
		}
	} else {
		if(pM->szCommandline.hasData())
		{
			KviParameterList * parms = new KviParameterList(new KviStr(szFullPath));
			g_pUserParser->parseCommandBuffer(pM->szCommandline.ptr(),g_pApp->activeConsole(),parms);
		}
	}
}

void KviDirBrowserWidget::addItem(KviDirBrowserData * data)
{
	QPixmap * pix = 0;
	QIconViewItem * it = new QIconViewItem(m_pIconView,data->szFileName.ptr());
	it->setKey(data->szKey.ptr());

	if(data->pMediaType->szIcon.hasData())pix = g_pIconManager->getImage(data->pMediaType->szIcon.ptr());

#warning "We need a fallback image here!!!"
	if(pix)it->setPixmap(*pix);
	else it->setPixmap(QPixmap(32,32));
}

bool KviDirBrowserWidget::event(QEvent *e)
{
	if(e->type() == (QEvent::Type)KVI_THREAD_EVENT)
	{
		switch(((KviThreadEvent *)e)->id())
		{
			case KVI_THREAD_EVENT_ERROR:
			{
				KviStr * str = ((KviThreadDataEvent<KviStr> *)e)->getData();
				QMessageBox::critical(this,__tr("Directory browser error"),str->ptr(),QMessageBox::Ok,QMessageBox::NoButton);
				killSlave();
				delete str;
			}
			break;
			case KVI_THREAD_EVENT_DATA:
			{
				KviDirBrowserData * data = ((KviThreadDataEvent<KviDirBrowserData> *)e)->getData();
				addItem(data);
				delete data->pMediaType;
				delete data;
			}
			break;
			case KVI_THREAD_EVENT_TERMINATE:
				killSlave();
			break;
		}
		return true;
	}
	return QWidget::event(e);
}


void KviDirBrowserWidget::resizeEvent(QResizeEvent *e)
{
	int hght = m_pMenuBar->sizeHint().height();
	int hght2 = m_pToolBar->sizeHint().height();
	int hght3 = m_pStatusLabel->sizeHint().height();
	m_pMenuBar->setGeometry(0,0,width(),hght);
	m_pToolBar->setGeometry(0,hght,width(),hght2);
	m_pIconView->setGeometry(0,hght + hght2,width(),height() - (hght + hght2 + hght3 + 2));
	m_pStatusLabel->setGeometry(0,height() - (hght3 + 2),width(),hght3);
}

void KviDirBrowserWidget::doClose()
{
	// hack needed to workaround "QToolBar::emulateButtonClicked()"
	// that refers to the "this" pointer after this slot has been
	// called (from the "too-small-toolbar-for-all-items-popup")
	QTimer::singleShot(0,this,SLOT(suicide()));
}

void KviDirBrowserWidget::suicide()
{
	// goodbye cruel wolrd
	if(m_bIsStandalone)delete this;
	else ((QWidget *)parent())->close();
}

QSize KviDirBrowserWidget::sizeHint() const
{
	int wdth = m_pIconView->sizeHint().width();
	if(m_pToolBar->sizeHint().width() > wdth)wdth = m_pToolBar->sizeHint().width();
	QSize s(wdth,m_pIconView->sizeHint().height() + m_pToolBar->sizeHint().height() + m_pMenuBar->sizeHint().height() + m_pStatusLabel->sizeHint().height());
	return s;
}

#include "m_dirbrowserwidget.moc"
