//
//   File : libkvidialog.cpp
//   Creation date : Sat Sep 15 2001 01:13:25 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2001 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//


#include "libkvidialog.h"

#include <qmessagebox.h>
#include <qlayout.h>
#include <qhbox.h>
#include <qlineedit.h>
#include <qmultilineedit.h>
#include <qlabel.h>
#include <qpushbutton.h>

#include "kvi_locale.h"
#include "kvi_uparser.h"
#include "kvi_module.h"
#include "kvi_modulemanager.h"
#include "kvi_command.h"
#include "kvi_error.h"
#include "kvi_app.h"
#include "kvi_iconmanager.h"
#include "kvi_console.h"
#include "kvi_iconmanager.h"

static KviPtrList<KviScriptCallbackDialog> * g_pDialogModuleDialogList = 0;

//#warning "Other dialogs!"

KviScriptCallbackDialog::KviScriptCallbackDialog(KviWindow * pWnd,KviStr &code,KviParameterList * parms)
{
	g_pDialogModuleDialogList->append(this);
	m_pWindow = pWnd;
	m_szCode = code;
	m_pParams = parms;
	if(!m_pParams)
	{
		m_pParams = new KviParameterList();
		m_pParams->setAutoDelete(true);
	}
}

KviScriptCallbackDialog::~KviScriptCallbackDialog()
{
	g_pDialogModuleDialogList->removeRef(this);
	if(m_pParams)delete m_pParams;
}

void KviScriptCallbackDialog::executeCallback()
{
	if(!g_pApp->windowExists(m_pWindow))m_pWindow = g_pApp->activeConsole();

	KviCommand cmd(m_szCode.ptr(),m_pWindow);
	if(m_pParams)cmd.setParams(m_pParams);

	m_pParams = 0;

	ENTER_CONTEXT((&cmd),"dialog_callback");

	if(!g_pUserParser->parseCommand(&cmd))
	{
		if(cmd.hasError())g_pUserParser->printError(&cmd);
	}
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

KviScriptCallbackMessageBox::KviScriptCallbackMessageBox(KviStr &caption,KviStr &text,KviStr &icon,KviStr &b1,KviStr &b2,KviStr &b3,KviParameterList * parms,KviWindow * pWnd,KviStr &code)
: QMessageBox(caption.ptr(),text.ptr(),QMessageBox::NoIcon,
		b1.hasData() ? QMessageBox::Ok | QMessageBox::Default : QMessageBox::NoButton,
		b2.hasData() ? (b3.hasData() ? QMessageBox::No : QMessageBox::No | QMessageBox::Escape) : QMessageBox::NoButton,
		b3.hasData() ? QMessageBox::Cancel | QMessageBox::Escape : QMessageBox::NoButton,
		0,0,false) ,
	KviScriptCallbackDialog(pWnd,code,parms)
{
	setIcon(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_KVIRC)));

	QPixmap * pix = g_pIconManager->getImage(icon.ptr());
	if(pix)setIconPixmap(*pix);
	else {
		if(kvi_strEqualCI(icon.ptr(),"information"))setIcon(QMessageBox::Information);
		else if(kvi_strEqualCI(icon.ptr(),"warning"))setIcon(QMessageBox::Warning);
		else if(kvi_strEqualCI(icon.ptr(),"critical"))setIcon(QMessageBox::Critical);
	}
	if(b1.hasData())setButtonText(QMessageBox::Ok,b1.ptr());
	if(b2.hasData())setButtonText(QMessageBox::No,b2.ptr());
	if(b3.hasData())setButtonText(QMessageBox::Cancel,b3.ptr());
}

KviScriptCallbackMessageBox::~KviScriptCallbackMessageBox()
{
}

void KviScriptCallbackMessageBox::done(int code)
{
	QMessageBox::done(code);

	int iVal = 0;
	
	switch(code)
	{
		case QMessageBox::No: iVal = 1; break;
		case QMessageBox::Cancel: iVal = 2; break;
	}

	m_pParams->prepend(new KviStr(KviStr::Format,"%d",iVal));
	executeCallback();

	delete this;
}



/*
	@doc: dialog.message
	@type:
		command
	@title:
		dialog.message
	@short:
		Shows a message box
	@syntax:
		dialog.message(<caption>,<message_text>,<icon>,<button0>[,<button1>[,<button2>[,<magic1>[,<magic2>[...]]]]])
		{
			<callback_command>
		}
	@description:
		Shows a message dialog box with the specified <caption> , <message_text> , <icon> and
		buttons.[br]
		<caption> is a text string that will appear in the caption of the dialog box.[br]
		<message_text> is a text string that will appear in the dialog box and can contain HTML formatting.[br]
		<icon> is an [doc:image_id]image identifier[/doc] that defines an icon to be placed in the dialog box.
		<icon> can be a relative or absolute path to an image file , a signed number (in that case it defines
		an internal KVIrc image) or one of the special strings "critical", "information" and "warning".[br]
		<button0> is the text of the first button (on the left).[br]
		<button1> is the text of the second button (if empty or not given at all, only one button will appear in the dialog).[br]
		<button2> is the text of the third button (if empty or not given, only two buttons will appear in the dialog).[br]
		<magic1>,<magic2>... are the magic parameters: evaluated at dialog.message call time and passed
		to the <callback_command> as positional parameters.[br]
		Once the dialog has been shown , the user will click one of the buttons. At this point the dialog
		is hidden and the <callback_command> is executed passing the number of the button clicked
		as $0 and the magic parameters as positional parameters $1 , $2 , $3....[br]
		Please note that if the user closes the window with the window manager close button , 
		the action is interpreted as a button2 click (that is usually sth as "Cancel").[br]
	@examples:
		[example]
		[comment]# Just a warning dialog[/comment]
		dialog.message("Warning","You're being <b>warned</b>",warning,Ok){ echo The user clicked OK; }
		[comment]# A question[/comment]
		dialog.message("And now ?","What do you want to do ?",information,"Go home","Watch TV","Scream")
		{
			if($0 == 0)echo "The user want's to go home"
			else if($0 == 1)echo "The user want's to watch TV"
			else echo "The user wants to scream!"
		}
		[/example]
*/

static bool dialog_module_cmd_message(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"dialog_module_cmd_message");

	KviParameterList paramList;
	paramList.setAutoDelete(true);

	KviStr cmd;

	if(!g_pUserParser->parseCallbackCommand(c,&paramList,&cmd))return false;

	KviStr caption = paramList.safeFirstParam();
	KviStr message = paramList.safeNextParam();
	KviStr icon    = paramList.safeNextParam();
	KviStr button1 = paramList.safeNextParam();
	KviStr button2 = paramList.safeNextParam();
	KviStr button3 = paramList.safeNextParam();

	KviParameterList * parms = new KviParameterList();
	parms->setAutoDelete(true);
	while(KviStr * n = paramList.next())parms->append(new KviStr(*n));

	KviScriptCallbackMessageBox * box = new KviScriptCallbackMessageBox(caption,message,icon,button1,button2,button3,parms,c->window(),cmd);
	box->show();

	return c->leaveContext();
}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

KviScriptCallbackTextInput::KviScriptCallbackTextInput(KviStr &caption,KviStr &label,KviStr &deftext,KviStr &icon,bool bMultiLine,KviStr &b1,KviStr &b2,KviStr &b3,KviParameterList * parms,KviWindow * pWnd,KviStr &code)
: QDialog() , KviScriptCallbackDialog(pWnd,code,parms)
{
	setIcon(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_KVIRC)));

	setCaption(caption.ptr());

	QGridLayout * g = new QGridLayout(this,2,3,5,5);

	QPixmap * pix = g_pIconManager->getImage(icon.ptr());

	if(pix)
	{
		QLabel * il = new QLabel(this);
		il->setPixmap(*pix);
		il->setAlignment(AlignCenter);
		g->addWidget(il,0,0);
		QLabel * tl = new QLabel(label.ptr(),this);
		g->addWidget(tl,0,1);
	} else {
		QLabel * tl = new QLabel(label.ptr(),this);
		g->addMultiCellWidget(tl,0,0,0,1);
	}

	g->setColStretch(1,1);

	m_bMultiLine = bMultiLine;

	if(m_bMultiLine)
	{
		m_pEdit = new QMultiLineEdit(this);
		((QMultiLineEdit *)m_pEdit)->setText(deftext.ptr());
	} else {
		m_pEdit = new QLineEdit(this);
		((QLineEdit *)m_pEdit)->setText(deftext.ptr());
	}

	g->addMultiCellWidget(m_pEdit,1,1,0,1);

	QHBox * box = new QHBox(this);
	g->addMultiCellWidget(box,2,2,0,1);

	if(b1.hasData())
	{
		QPushButton * pb1 = new QPushButton(b1.ptr(),box);
		connect(pb1,SIGNAL(clicked()),this,SLOT(b1Clicked()));
	}

	if(b2.hasData())
	{
		QPushButton * pb2 = new QPushButton(b2.ptr(),box);
		connect(pb2,SIGNAL(clicked()),this,SLOT(b2Clicked()));
		
	}

	if(b3.hasData())
	{
		QPushButton * pb3 = new QPushButton(b3.ptr(),box);
		connect(pb3,SIGNAL(clicked()),this,SLOT(b3Clicked()));
	}
}

KviScriptCallbackTextInput::~KviScriptCallbackTextInput()
{
}

void KviScriptCallbackTextInput::b1Clicked()
{
	done(0);
}

void KviScriptCallbackTextInput::b2Clicked()
{
	done(1);
}

void KviScriptCallbackTextInput::b3Clicked()
{
	done(2);
}

void KviScriptCallbackTextInput::closeEvent(QCloseEvent *e)
{
	e->ignore();
	done(2);
}

void KviScriptCallbackTextInput::done(int code)
{
	QDialog::done(code);

	KviStr * txt = new KviStr();

	if(m_bMultiLine)
	{
		*txt = ((QMultiLineEdit *)m_pEdit)->text();
	} else {
		*txt = ((QLineEdit *)m_pEdit)->text();
	}
	m_pParams->prepend(txt);

	m_pParams->prepend(new KviStr(KviStr::Format,"%d",code));
	executeCallback();

	delete this;
}

void KviScriptCallbackTextInput::showEvent(QShowEvent *e)
{
	move((g_pApp->desktop()->width() - width())/2,(g_pApp->desktop()->height() - height()) / 2);
	QDialog::showEvent(e);
}

/*
	@doc: dialog.textinput
	@type:
		command
	@title:
		dialog.textinput
	@short:
		Shows a dialog that accepts user input as text
	@syntax:
		dialog.textinput [-d=<default text>] [-i=<icon>] [-m] (<caption>,<info_text>,<button0>[,<button1>[,<button2>[,<magic1>[,<magic2>[...]]]]])
		{
			<callback_command>
		}
	@description:
		Shows a text input dialog box with the specified <caption> , <info_text> , <icon> and
		buttons.[br]
		<caption> is a text string that will appear in the caption of the dialog box.[br]
		<info_text> is a fixed text string that will appear in the dialog box and can contain HTML formatting.[br]
		<button0> is the text of the first button (on the left).[br]
		<button1> is the text of the second button (if empty or not given at all, only one button will appear in the dialog).[br]
		<button2> is the text of the third button (if empty or not given, only two buttons will appear in the dialog).[br]
		If the -m switch is used , the dialog will be a multi-line text input, otherwise the user will be able to
		input only a single line of text.[br]
		If the -d switch is used , the initial text input value is set to <default text>.[br]
		If the -i switch is used , the dialog displays also the icon <icon> , just on the left ot the <info_text>
		In that case <icon> is an [doc:image_id]image identifier[/doc] (can be a relative or absolute
		path to an image file or a signed number (in that case it defines an internal KVIrc image).[br]
		<magic1>,<magic2>... are the magic parameters: evaluated at dialog.textinput call time and passed
		to the <callback_command> as positional parameters.[br]
		Once the dialog has been shown , the user will click one of the buttons. At this point the dialog
		is hidden and the <callback_command> is executed passing the text input value in $1, the number of the button clicked
		as $0, and the magic parameters as positional parameters $2 , $3 , $4....[br]
		Please note that if the user closes the window with the window manager close button , 
		the action is interpreted as a button2 click (that is usually sth as "Cancel").[br]
	@examples:
		[example]
		[comment]# We need a single line "reason"[/comment]
		dialog.textinput -d="Working !" (Away,<center>Please enter the <h1>away message</h1></center>,"Ok","Cancel")
		{
			switch($0)
			{
				case 0:
					away $1-
				break;
				default:
					# Cancelled
				break;
			}
		}
		[/example]
*/

static bool dialog_module_cmd_textinput(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"dialog_module_cmd_input");

	KviParameterList paramList;
	paramList.setAutoDelete(true);

	KviStr cmd;

	if(!g_pUserParser->parseCallbackCommand(c,&paramList,&cmd))return false;

	KviStr caption  = paramList.safeFirstParam();
	KviStr infotext = paramList.safeNextParam();
	KviStr defaultv;
	KviStr icon;
	KviStr button1  = paramList.safeNextParam();
	KviStr button2  = paramList.safeNextParam();
	KviStr button3  = paramList.safeNextParam();

	if(c->hasSwitch('i'))c->getSwitchValue('i',icon);
	if(c->hasSwitch('d'))c->getSwitchValue('d',defaultv);

	KviParameterList * parms = new KviParameterList();
	parms->setAutoDelete(true);
	while(KviStr * n = paramList.next())parms->append(new KviStr(*n));

	KviScriptCallbackTextInput * box = new KviScriptCallbackTextInput(caption,infotext,defaultv,icon,c->hasSwitch('m'),button1,button2,button3,parms,c->window(),cmd);
	box->show();

	return c->leaveContext();
}



//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////


KviScriptCallbackFileDialog::KviScriptCallbackFileDialog(KviStr &caption,KviStr &initial,KviStr &filter,KviParameterList * parms,KviWindow * pWnd,KviStr &code)
 : KviFileDialog(initial.ptr(),filter.ptr()) , KviScriptCallbackDialog(pWnd,code,parms)
{
	setCaption(caption.ptr());
}


KviScriptCallbackFileDialog::~KviScriptCallbackFileDialog()
{
}

void KviScriptCallbackFileDialog::done(int code)
{
	KviFileDialog::done(code);

	if(code == QDialog::Accepted)
	{
		if(mode() == QFileDialog::ExistingFiles)
		{
			KviStr joined = selectedFiles().join(",");
			m_pParams->prepend(new KviStr(joined));
		} else {
			m_pParams->prepend(new KviStr(selectedFile()));
		}
	} else {
		m_pParams->prepend(new KviStr());
	}

	hide(); // ensure we're hidden

	// ugly workaround for the Qt filedialog "destructive accept() before this reference" bug
	// we can't delete ourselves in this moment.... :(((
	// ...so skip out of this call stack and ask KviApp to destroy us just
	// when the control returns to the main loop.
	// If the module is unloaded then , KviApp will notice it and will NOT delete the dialog
	g_pApp->collectGarbage(this);

	// calling dialog.unload here WILL lead to a sigsegv (this is SURE
	// with a lot of qt versions that have the ugly file dialog "accept before this reference" bug)
	// to avoid it, we can execute the callback triggered by a timer...
	// ... umpf ...
	executeCallback();
}


/*
	@doc: dialog.file
	@type:
		command
	@title:
		dialog.file
	@short:
		Shows a file dialog
	@syntax:
		dialog.file(<mode>,<caption>,<initial_selection>,<file_filter>[,<magic1>[,<magic2>[...]]]]])
		{
			<callback_command>
		}
	@description:
		Shows an openfile dialog box with the specified <caption> , <initial_selection> , and <file_filter>.[br]
		<mode> can be "open" , "openm" , "save" or "dir":[br]
		"open" causes the dialog to return an existing file[br]
		"openm" is similar to open but allows returning multiple files as a comma separated list[br]
		"save" causes the dialog to return any file name (no overwrite confirmation is built in the dialog!)[br]
		"dir" causes the dialog to return an existing directory name[br]
		<mode> defaults to "open".[br]
		<caption> is a text string that will appear in the caption of the dialog box.[br]
		<initial_selection> can be a directory or filename that will be initially selected in the dialog.[br]
		Only files matching <file_filter> are selectable. If filter is an empty string, all files are selectable.[br]
		In the filter string multiple filters can be specified separated by either two semicolons next to each
		other or separated by newlines. To add two filters, one to show all C++ files and one to show all 
		header files, the filter string could look like "C++ Files (*.cpp *.cc *.C *.cxx *.c++);;Header Files (*.h *.hxx *.h++)" 
		<magic1>,<magic2>... are the magic parameters: evaluated at dialog.message call time and passed
		to the <callback_command> as positional parameters.[br]
		Once the dialog has been shown , the user will select an EXISTING file and click either
		Ok or Cancel. At this point the dialog is hidden and the <callback_command> is executed passing the selected file(s) as $0
		and the magic parameters as positional parameters $1 , $2 , $3....[br]
		If the user clicks "Cancel" or does not select any file the positional parameter $0 will be empty.[br]
	@examples:
		[example]
			dialog.file(open,Choose an audio file,/home/pragma/TheAudio.au,"Audio files (*.au *.wav *.snd)")
			{
				if("$0" != "")run play $0
			}
		[/example]
*/

//#warning "Examples for these dialogs!"

static bool dialog_module_cmd_file(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"dialog_module_cmd_file");

	KviParameterList paramList;
	paramList.setAutoDelete(true);

	KviStr cmd;

	if(!g_pUserParser->parseCallbackCommand(c,&paramList,&cmd))return false;

	KviStr mode    = paramList.safeFirstParam();
	KviStr caption = paramList.safeNextParam();
	KviStr initial = paramList.safeNextParam();
	KviStr filter  = paramList.safeNextParam();

	KviParameterList * parms = new KviParameterList();
	parms->setAutoDelete(true);
	while(KviStr * n = paramList.next())parms->append(new KviStr(*n));

	KviScriptCallbackFileDialog * box = new KviScriptCallbackFileDialog(caption,initial,filter,parms,c->window(),cmd);

	QFileDialog::Mode md = QFileDialog::ExistingFile;

	if(kvi_strEqualCI(mode.ptr(),"openm"))md = QFileDialog::ExistingFiles;
	else if(kvi_strEqualCI(mode.ptr(),"save"))md = QFileDialog::AnyFile;
	else if(kvi_strEqualCI(mode.ptr(),"dir"))md = QFileDialog::DirectoryOnly;

	box->setMode(md);

	box->show();

	return c->leaveContext();
}


/*

static int g_iLocalEventLoops = 0;

static bool dialog_module_cmd_unload(KviModule *m,KviCommand *c)
{
	// We use local loops in this module: we must FORBID explicit unloading of the
	// module while local even loops are running
	ENTER_CONTEXT(c,"dialog_module_cmd_unload");
	c->warning(__tr("The dialog module can't be explicitly unloaded: a modal dialog is currently open"));
	return c->leaveContext();
}

static void dialog_module_entering_local_loop(KviModule * m)
{
	// Replace unload
	g_iLocalEventLoops++;
	if(g_iLocalEventLoops == 1)m->registerCommand("unload",dialog_module_cmd_unload);
}

static void dialog_module_exiting_local_loop(KviModule * m)
{
	g_pModuleManager->registerDefaultCommands(m);
	g_iLocalEventLoops--;
}

static bool dialog_module_fnc_textline(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_CONTEXT(c,"dialog_module_fnc_textline");

	KviStr caption = parms->safeFirstParam();
	KviStr info    = parms->safeNextParam();
	KviStr initial = parms->safeNextParam();

	dialog_module_entering_local_loop(m);
	QMessageBox::information(0,caption.ptr(),info.ptr(),QMessageBox::Ok);
	dialog_module_exiting_local_loop(m);

	// It might be that the current window is no longer available!!!

	return c->leaveContext();
}

*/

/*
	@doc: noblockingdialogs
	@type:
		generic
	@title:
		Why there are no blocking dialogs in KVIrc ?
	@short:
		Tecnical answer
	@description:
		Why there are no blocking dialogs in KVIrc ?[br]
		The answer is simple: because they're more confusing and tricky than it seems.[br]
		Blocking the entire program control flow while showing a dialog is
		rather a bad idea since we have to deal with external entities (servers and other users)
		that are NOT blocked. This means that the blocking dialogs must block only the
		script control-flow but let the rest of the application running.
		Such blocking dialogs actually seem to simplify scripting because
		the programmer "feels" that the control is always left in the script snippet that he is writing.
		This is actually confusing: the control IS in the script snippet but while the dialog
		is open the whole world can change: you can return from the dialog call and discover
		that the server connection no longer exists and the application is about to quit.[br]
		This may happen even with non-blocking dialogs ,but in non-blocking mode you have
		a way to handle this event. Consider the following snippet of code:[br]
		[example]
			echo My name is $?
		[/example]
		Where $? stands for a blocking input dialog that asks the user for some text.[br]
		When the input dialog returns the window that the echo was directed to no longer
		exists and you have no way to stop the echo! (Well...I could add extra code
		in the executable to handle all these situations but that would be really too expensive).[br]
		With object scripting this is actually dangerous: you might use a blocking dialog
		in an object signal handler and when returning discover that this object has been deleted!
		(The example refers to a simple object , but think about a complex hierarchy of objects
		where one random gets deleted...).[br]
		This is why the dialogs in KVIrc are non-blocking :)[br]
		That's REAL programming.
*/







static bool dialog_module_init(KviModule * m)
{
	g_pDialogModuleDialogList = new KviPtrList<KviScriptCallbackDialog>;
	g_pDialogModuleDialogList->setAutoDelete(false);

//	m->registerFunction("textline",dialog_module_fnc_textline);

	m->registerCommand("message",dialog_module_cmd_message);
	m->registerCommand("file",dialog_module_cmd_file);
	m->registerCommand("textinput",dialog_module_cmd_textinput);

    return true;
}

static bool dialog_module_cleanup(KviModule *m)
{
	// Here we get a tragedy if g_iLocalEventLoops > 0!
	while(g_pDialogModuleDialogList->first())delete g_pDialogModuleDialogList->first();
	delete g_pDialogModuleDialogList;

	m->unregisterMetaObject("KviScriptCallbackMessageBox");
	m->unregisterMetaObject("KviScriptCallbackFileDialog");
	m->unregisterMetaObject("KviScriptCallbackTextInput");
	return true;
}

static bool dialog_module_can_unload(KviModule *m)
{
	return g_pDialogModuleDialogList->isEmpty();
}


KVIMODULEEXPORTDATA KviModuleInfo kvirc_module_info =
{
    "KVIrc script dialogs",
	"1.0.0",
	"Szymon Stefanek <stefanek@tin.it>" ,
	"Adds the /dialog.* commands functionality\n",
    dialog_module_init ,
    dialog_module_can_unload,
    0,
	dialog_module_cleanup
};

#include "libkvidialog.moc"
