//
//   File : chat.cpp
//   Creation date : Tue Sep 20 09 2000 15:13:13 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "send.h"
#include "broker.h"
#include "marshal.h"

#ifdef COMPILE_ON_WINDOWS
	// Ugly Windoze compiler...
	#include "dialogs.h"
#endif

#define _KVI_DEBUG_CHECK_RANGE_
#include "kvi_debug.h"
#include "kvi_app.h"
#include "kvi_options.h"
#include "kvi_ircview.h"
#include "kvi_iconmanager.h"
#include "kvi_locale.h"
#include "kvi_error.h"
#include "kvi_out.h"
#include "kvi_netutils.h"
#include "kvi_console.h"
#include "kvi_frame.h"
#include "kvi_malloc.h"
#include "kvi_memmove.h"
#include "kvi_thread.h"
#include "kvi_ircsocket.h"
#include "kvi_uparser.h"
#include "kvi_mediatype.h"
#include "kvi_socket.h"
#include "kvi_event.h"
#include "kvi_parameterlist.h"

#include <qsplitter.h>
#include <qevent.h>
#include <qtooltip.h>
#include <qfile.h>
#include <qtoolbutton.h>

//#include <unistd.h> //close()

// FIXME: The events OnDCCConnect etc are in wrong places here...!

extern KviDccBroker * g_pDccBroker;

extern KVIRC_API KviMediaManager * g_pMediaManager; // kvi_app.cpp

//#warning "The events that have a KviStr data pointer should become real classes, that take care of deleting the data pointer!"
//#warning "Otherwise, when left undispatched we will be leaking memory (event class destroyed but not the data ptr)"

KviDccRecvThread::KviDccRecvThread(KviWindow *wnd,kvi_socket_t fd,KviDccRecvThreadOptions * opt)
: KviDccThread(wnd,fd)
{
	m_pOpt                  = opt;
	m_iProgress             = 0;
	m_iAverageSpeed         = 0;
	m_iInstantSpeed         = 0;
	m_iFilePosition         = 0;
	m_iStartTime            = time(0);

	m_iTotalReceivedBytes   = 0;
	m_iInstantReceivedBytes = 0;
	m_pFile                 = 0;
}

KviDccRecvThread::~KviDccRecvThread()
{
	if(m_pOpt)delete m_pOpt;
	if(m_pFile)delete m_pFile;
}

bool KviDccRecvThread::sendAck(int filePos)
{
	int size = htonl(filePos);
	if(kvi_socket_send(m_fd,(void *)(&size),4) != 4)
	{
		postErrorEvent(KviError_acknowledgeError);
		return false;
	}
	return true;
}

void KviDccRecvThread::updateStats()
{
	int iCurTime = (int)time(0);
	int iDiffTime = iCurTime - m_iLastInstantCheckTime;

	m_pMutex->lock();
	int iElapsedTime = (int)(iCurTime - m_iStartTime);
	if(iElapsedTime < 1)iElapsedTime = 1;
	m_iFilePosition = m_pFile->at();
	if(m_pOpt->iTotalFileSize > 0)
		m_iProgress = ((m_iFilePosition * 100) / m_pOpt->iTotalFileSize);
	m_iAverageSpeed = m_iTotalReceivedBytes / iElapsedTime;
	if(iDiffTime > 1)
	{
		m_iInstantSpeed = m_iInstantReceivedBytes / iDiffTime;
		m_iLastInstantCheckTime = iCurTime;
		m_iInstantReceivedBytes = 0;
	}
	m_pMutex->unlock();
}

void KviDccRecvThread::postMessageEvent(const char * m)
{
	KviThreadDataEvent<KviStr> * e = new KviThreadDataEvent<KviStr>(KVI_DCC_THREAD_EVENT_MESSAGE);
	e->setData(new KviStr(m));
	postEvent(m_pWindow,e);
}


void KviDccRecvThread::run()
{
	m_pMutex->lock();
	m_iStartTime              = time(0);
	m_pMutex->unlock();

	m_iLastInstantCheckTime = time(0);

	int iProbableTerminationTime = 0;

	m_pFile = new QFile(m_pOpt->szFileName.ptr());

	if(m_pOpt->bResume)
	{
		if(!m_pFile->open(IO_WriteOnly | IO_Append))
		{
			postErrorEvent(KviError_cantOpenFileForAppending);
			goto exit_dcc;
		} // else pFile is already at end
	} else {
		if(!m_pFile->open(IO_WriteOnly))
		{
			postErrorEvent(KviError_cantOpenFileForWriting);
			goto exit_dcc;
		}
	}

	if(m_pOpt->bSendZeroAck && (!m_pOpt->bNoAcks))
	{
		if(!sendAck(m_pFile->at()))goto exit_dcc;
	}


	for(;;)
	{
		// Dequeue events
		while(KviThreadEvent * e = dequeueEvent())
		{
			if(e->id() == KVI_THREAD_EVENT_TERMINATE)
			{
				delete e;
				goto exit_dcc;
			} else {
				// Other events are senseless to us
				delete e;
			}
		}

		bool bCanRead;
		bool bDummy;

		if(kvi_select(m_fd,&bCanRead,&bDummy))
		{
			if(bCanRead)
			{
				// Read a data block
				char buffer[1024];
				int readLen = kvi_socket_recv(m_fd,buffer,1024);

				if(readLen > 0)
				{
					// Readed something useful...write back
					if((m_pOpt->iTotalFileSize > -1) && ((readLen + m_pFile->at()) > m_pOpt->iTotalFileSize))
					{
						postMessageEvent(__tr_no_lookup("WARNING: The peer is sending garbage data past the end of the file"));
						postMessageEvent(__tr_no_lookup("WARNING: Ignoring data after the declared length and forcibly closing the connection"));

						readLen = m_pOpt->iTotalFileSize - m_pFile->at();
						if(readLen > 0)
						{
							if(m_pFile->writeBlock(buffer,readLen) != readLen)
								postErrorEvent(KviError_fileIOError);
						}
						break;

					} else {
						if(m_pFile->writeBlock(buffer,readLen) != readLen)
						{
							postErrorEvent(KviError_fileIOError);
							break;
						}
					}

					// Update stats
					m_iTotalReceivedBytes += readLen;
					m_iInstantReceivedBytes += readLen;

					updateStats();
					// Now send the ack
					if(m_pOpt->bNoAcks)
					{
						// No acks...
						// Interrupt if the whole file has been received
						if(m_pOpt->iTotalFileSize > 0)
						{
							if(m_pFile->at() == m_pOpt->iTotalFileSize)
							{
								// Received the whole file...die
								KviThreadEvent * e = new KviThreadEvent(KVI_DCC_THREAD_EVENT_SUCCESS);
								postEvent(m_pWindow,e);
								break;
							}
						}
					} else {
						// Must send the ack... the peer must close the connection
						if(!sendAck(m_pFile->at()))break;
					}
				} else {
					updateStats();
					// Read problem...

					if(readLen == 0)
					{
						// readed EOF..
						if((m_pFile->at() == m_pOpt->iTotalFileSize) || (m_pOpt->iTotalFileSize < 0))
						{
							// success if we got the whole file or if we don't know the file size (we trust the peer)
							KviThreadEvent * e = new KviThreadEvent(KVI_DCC_THREAD_EVENT_SUCCESS);
							postEvent(m_pWindow,e);
							break;
						}
					}
					if(!handleInvalidSocketRead(readLen))break;
				}
			} else {
				// Can't read stuff
				updateStats();

				if(m_pFile->at() == m_pOpt->iTotalFileSize)
				{
					// Wait for the peer to close the connection
					if(iProbableTerminationTime == 0)
					{
						iProbableTerminationTime = (int)time(0);
						m_pFile->flush();
						postMessageEvent(__tr_no_lookup("Data transfer terminated: waiting 30 secs for the peer to close the connection"));
						// FIXME: Close the file ?
					} else {
						int iDiff = (((int)time(0)) - iProbableTerminationTime);
						if(iDiff > 30)
						{
							// success if we got the whole file or if we don't know the file size (we trust the peer)
							postMessageEvent(__tr_no_lookup("Data transfer was terminated 30 secs ago: forcibly closing the connection"));
							KviThreadEvent * e = new KviThreadEvent(KVI_DCC_THREAD_EVENT_SUCCESS);
							postEvent(m_pWindow,e);
							break;
						}
					}
				}
			}
			msleep(10);
		} else {
			updateStats();
			msleep(50);
		}
	}

exit_dcc:
	if(m_pFile)
	{
		m_pFile->close();
		delete m_pFile;
		m_pFile = 0;
	}
	kvi_socket_close(m_fd);
	m_fd = KVI_INVALID_SOCKET;
}

int KviDccRecvThread::elapsedTime()
{
	return (int)(time(0) - m_iStartTime);
}

void KviDccRecvThread::initGetInfo()
{
	m_pMutex->lock();
}

void KviDccRecvThread::doneGetInfo()
{
	m_pMutex->unlock();
}

KviDccSendThread::KviDccSendThread(KviWindow *wnd,kvi_socket_t fd,KviDccSendThreadOptions * opt)
: KviDccThread(wnd,fd)
{
	m_pOpt = opt;
	// stats
	m_iStartTime     = time(0);
	m_iProgress      = 0;
	m_iAverageSpeed  = 0;
	m_iInstantSpeed  = 0;
	m_iFilePosition  = 0;
	m_iAckedProgress = 0;
}

KviDccSendThread::~KviDccSendThread()
{
	if(m_pOpt)delete m_pOpt;
}

void KviDccSendThread::run()
{

	m_pMutex->lock();
	m_iStartTime              = time(0);
	m_pMutex->unlock();

	int iLastInstantCheckTime = time(0);
	int iTotalSentBytes       = 0;
	int iInstantSentBytes     = 0;
	int iLastAck;

	if(m_pOpt->iPacketSize < 32)m_pOpt->iPacketSize = 32;
	char * buffer = (char *)kvi_malloc(m_pOpt->iPacketSize * sizeof(char));

	QFile * pFile = new QFile(m_pOpt->szFileName.ptr());

	if(!pFile->open(IO_ReadOnly))
	{
		postErrorEvent(KviError_cantOpenFileForReading);
		goto exit_dcc;
	}

	if(pFile->size() < 1)
	{
		postErrorEvent(KviError_cantSendAZeroSizeFile);
		goto exit_dcc;
	}

	if(m_pOpt->iStartPosition > 0)
	{
		// seek
		if(!(pFile->at(m_pOpt->iStartPosition)))
		{
			postErrorEvent(KviError_fileIOError);
			goto exit_dcc;
		}
	}

	iLastAck = m_pOpt->iStartPosition;
	for(;;)
	{
		// Dequeue events
		while(KviThreadEvent * e = dequeueEvent())
		{
			if(e->id() == KVI_THREAD_EVENT_TERMINATE)
			{
				delete e;
				goto exit_dcc;
			} else {
				// Other events are senseless to us
				delete e;
			}
		}

		bool bCanRead;
		bool bCanWrite;

		if(kvi_select(m_fd,&bCanRead,&bCanWrite))
		{
			if(bCanRead)
			{
				if(!m_pOpt->bNoAcks)
				{
					int iAck;
					int readLen = kvi_socket_recv(m_fd,(void *)&iAck,4);
					if(readLen == 4)iLastAck = ntohl(iAck);
					else {
						if(readLen > 0)
						{
							postErrorEvent(KviError_acknowledgeError);
							break;
						} else {
							if(!handleInvalidSocketRead(readLen))break;
						}
					}
					// update stats
					m_pMutex->lock(); // is this really necessary ?
					m_iAckedProgress = (iLastAck * 100) / pFile->size();
					m_pMutex->unlock(); 
	
					if(iLastAck == (int)(pFile->size()))
					{
						KviThreadEvent * e = new KviThreadEvent(KVI_DCC_THREAD_EVENT_SUCCESS);
						postEvent(m_pWindow,e);
						break;
					}
				} else {
					// No acknowledges
					if(m_pOpt->bIsTdcc)
					{
						// We expect the remote end to close the connection when the whole file has been sent
						if(pFile->atEnd())
						{
							int iAck;
							int readLen = kvi_socket_recv(m_fd,(void *)&iAck,4);
							if(readLen == 0)
							{
								// done...success
								KviThreadEvent * e = new KviThreadEvent(KVI_DCC_THREAD_EVENT_SUCCESS);
								postEvent(m_pWindow,e);
								break;
							} else {
								if(readLen < 0)
								{
									if(!handleInvalidSocketRead(readLen))break;
								} else {
									KviThreadDataEvent<KviStr> * e = new KviThreadDataEvent<KviStr>(KVI_DCC_THREAD_EVENT_MESSAGE);
									e->setData(new KviStr(__tr_no_lookup("WARNING: Received data in a DCC TSEND: there should be no acknowledges")));
									postEvent(m_pWindow,e);
								}
							}
						}
					}
				}
			}
			if(bCanWrite)
			{
				if(!pFile->atEnd())
				{
					if(m_pOpt->bFastSend || m_pOpt->bNoAcks || (iLastAck == pFile->at()))
					{
						int toRead = pFile->size() - pFile->at();
						if(toRead > m_pOpt->iPacketSize)toRead = m_pOpt->iPacketSize;
						int readed = pFile->readBlock(buffer,toRead);
						if(readed < toRead)
						{
							postErrorEvent(KviError_fileIOError);
							break;
						}
						int written = kvi_socket_send(m_fd,buffer,toRead);
						if(written < toRead)
						{
							if(written < 0)
							{
								// error ?
								if(!handleInvalidSocketRead(written))break;
							} else {
								pFile->at(pFile->at() - (toRead - written));
							}
						}
						iTotalSentBytes += written;
						iInstantSentBytes += written;
						int iCurTime = (int)time(0);
						int iDiffTime = iCurTime - iLastInstantCheckTime;
						m_pMutex->lock();
						int iElapsedTime = (int)(iCurTime - m_iStartTime);
						if(iElapsedTime < 1)iElapsedTime = 1;
						m_iFilePosition = pFile->at();
						m_iProgress = ((m_iFilePosition * 100) / pFile->size());
						m_iAverageSpeed = iTotalSentBytes / iElapsedTime;
						if(iDiffTime > 1)
						{
							m_iInstantSpeed = iInstantSentBytes / iDiffTime;
							iLastInstantCheckTime = iCurTime;
							iInstantSentBytes = 0;
						}
						m_pMutex->unlock();
					}
				} else {
					if(m_pOpt->bNoAcks && !m_pOpt->bIsTdcc)
					{
						// at end of the file in a blind dcc send...
						// not in a tdcc: we can close the file...
						KviThreadEvent * e = new KviThreadEvent(KVI_DCC_THREAD_EVENT_SUCCESS);
						postEvent(m_pWindow,e);
						break;
					}
				}
			}
		}
//#warning "iIdleStepLengthInMSec could increase if stalled or decrease of received data"
//#warning "obviously into a predefined range... (adaptive dcc)"
		msleep(m_pOpt->iIdleStepLengthInMSec);
	}

exit_dcc:
	kvi_free(buffer);
	pFile->close();
	delete pFile;
	pFile = 0;
	kvi_socket_close(m_fd);
	m_fd = KVI_INVALID_SOCKET;
}

int KviDccSendThread::elapsedTime()
{
	return (int)(time(0) - m_iStartTime);
}

void KviDccSendThread::initGetInfo()
{
	m_pMutex->lock();
}

void KviDccSendThread::doneGetInfo()
{
	m_pMutex->unlock();
}




KviDccSend::KviDccSend(KviFrame *pFrm,KviDccBrokerDescriptor * dcc,const char * name)
: KviWindow(KVI_WINDOW_TYPE_DCCSEND,pFrm,name)
{
	m_pDescriptor = dcc;
	m_pSplitter = new QSplitter(QSplitter::Horizontal,this,"splitter");
	m_pIrcView = new KviIrcView(m_pSplitter,pFrm,this);

	m_pVBox = new QVBox(this,"v_box");
	KviStr tmp(KviStr::Format,__tr("File: %s (%s bytes)"),
			dcc->szFileName.ptr(),
			dcc->bRecvFile ? dcc->szFileSize.ptr() :  dcc->szLocalFileSize.ptr());
	m_pFileInfoLabel = new QLabel(tmp.ptr(),m_pVBox);
	m_pFileInfoLabel->setFrameStyle(QFrame::Sunken | QFrame::Panel);
	tmp.sprintf(__tr("Local file name: %s"),dcc->szLocalFileName.ptr());
	QToolTip::add(m_pFileInfoLabel,tmp.ptr());

	m_pDataProgressBar = new QProgressBar(100,m_pVBox,"data_progress");
	QToolTip::add(m_pDataProgressBar,__tr("Data progress"));
	if((!dcc->bRecvFile) && (!dcc->bNoAcks))
	{
		m_pAckProgressBar = new QProgressBar(100,m_pVBox,"ack_progress");
		QToolTip::add(m_pAckProgressBar,__tr("Acknowledge progress"));
	} else m_pAckProgressBar = 0;

	QHBox * box = new QHBox(m_pVBox,"h_box");
	m_pElapsedTimeLabel = new QLabel(__tr("0 h 0 m 0 s"),box);
	m_pElapsedTimeLabel->setFrameStyle(QFrame::Sunken | QFrame::Panel);
	QToolTip::add(m_pElapsedTimeLabel,__tr("Elapsed time"));
	m_pBytesLabel = new QLabel(__tr("0 bytes"),box);
	m_pBytesLabel->setFrameStyle(QFrame::Sunken | QFrame::Panel);
	QToolTip::add(m_pBytesLabel,__tr("Bytes transferred"));
	m_pAvgSpeedLabel = new QLabel(__tr("0 bytes/sec"),box);
	m_pAvgSpeedLabel->setFrameStyle(QFrame::Sunken | QFrame::Panel);
	QToolTip::add(m_pAvgSpeedLabel,__tr("Average transfer speed"));
	m_pInstantSpeedLabel = new QLabel(__tr("0 bytes/sec"),box);
	m_pInstantSpeedLabel->setFrameStyle(QFrame::Sunken | QFrame::Panel);
	QToolTip::add(m_pInstantSpeedLabel,__tr("Instant transfer speed"));

	box->setStretchFactor(m_pElapsedTimeLabel,1);
	box->setStretchFactor(m_pBytesLabel,1);
	box->setStretchFactor(m_pAvgSpeedLabel,1);
	box->setStretchFactor(m_pInstantSpeedLabel,1);

	m_pFilePopup = new QPopupMenu(this);
	connect(m_pFilePopup,SIGNAL(aboutToShow()),this,SLOT(setupFilePopup()));
	QToolButton * btn = new QToolButton(DownArrow,box);
	btn->setPopup(m_pFilePopup);
	btn->setPopupDelay(1);


	setFocusHandler(m_pIrcView,this);

	m_pMarshal = new KviDccMarshal(this);
	connect(m_pMarshal,SIGNAL(error(int)),this,SLOT(handleMarshalError(int)));
	connect(m_pMarshal,SIGNAL(connected()),this,SLOT(connected()));
	connect(m_pMarshal,SIGNAL(inProgress()),this,SLOT(connectionInProgress()));
#ifdef COMPILE_SSL_SUPPORT
	connect(m_pMarshal,SIGNAL(startingSSLHandshake()),this,SLOT(startingSSLHandshake()));
	connect(m_pMarshal,SIGNAL(sslError(const char *)),this,SLOT(sslError(const char *)));
#endif
	m_szDccType = dcc->bIsTdcc ? (dcc->bRecvFile ? "TRECV" : "TSEND") : (dcc->bRecvFile ? "RECV" : "SEND");


	m_pSlaveRecvThread = 0;
	m_pSlaveSendThread = 0;

	m_pUpdateTimer = new QTimer();
	m_bRunning = true;

	startConnection();
}

KviDccSend::~KviDccSend()
{
	g_pDccBroker->unregisterDccWindow(this);
	if(m_pSlaveRecvThread)
	{
		m_pSlaveRecvThread->terminate();
		delete m_pSlaveRecvThread;
		m_pSlaveRecvThread = 0;
	} else {
		if(m_pSlaveSendThread)
		{
			m_pSlaveSendThread->terminate();
			delete m_pSlaveSendThread;
			m_pSlaveSendThread = 0;
		}
	}
	KviThreadManager::killPendingEvents(this);
	delete m_pUpdateTimer;
	delete m_pDescriptor;
	delete m_pMarshal;
}

void KviDccSend::connectionInProgress()
{
	if(!(m_pDescriptor->bActive))
	{
		// PASSIVE CONNECTION
		output(KVI_OUT_DCCMSG,__tr("Listening on interface %s port %s"),
			m_pMarshal->localIp(),m_pMarshal->localPort());

		if(m_pDescriptor->bSendRequest)
		{
			KviStr ip     = m_pDescriptor->szFakeIp.hasData() ? m_pDescriptor->szFakeIp : m_pDescriptor->szListenIp;
			KviStr port   = m_pDescriptor->szFakePort.hasData() ? m_pDescriptor->szFakePort.ptr() : m_pMarshal->localPort();
//#warning "OPTION FOR SENDING 127.0.0.1 and so on (not an unsigned nuumber)"
			struct in_addr a;
			if(kvi_stringIpToBinaryIp(ip.ptr(),&a))ip.setNum(htonl(a.s_addr));

			KviStr fName  = m_pDescriptor->szFileName.ptr();
			fName.cutToLast('/'); // just to be sure...
			fName.replaceAll(' ',m_pDescriptor->bRecvFile ? "\\ " : "_"); // be cool :)
			m_pDescriptor->pConsole->socket()->sendFmtData("PRIVMSG %s :%cDCC %s %s %s %s %s%c",
					m_pDescriptor->szNick.ptr(),0x01,m_szDccType.ptr(),
					fName.ptr(),ip.ptr(),port.ptr(),
					m_pDescriptor->szLocalFileSize.ptr(),0x01);
			output(KVI_OUT_DCCMSG,__tr("Sent DCC %s request to %s...waiting for the remote client to connect"),
				m_szDccType.ptr(),m_pDescriptor->szNick.ptr());
		} else output(KVI_OUT_DCCMSG,__tr("DCC %s request not sent: awaiting manual connections"),m_szDccType.ptr());
	} else {
		// ACTIVE CONNECTION
		if((kvi_strEqualCS(m_szDccType.ptr(), "RECV")) || (kvi_strEqualCS(m_szDccType.ptr(),"TRECV"))) 
		{
			// FIXME: that's not true!... we're NOT connected here
			if(TRIGGER_EVENT_5PARAM_RETVALUE(KviEvent_OnDCCGetConnected,this,m_pDescriptor->szPort.ptr(),m_pDescriptor->szFileName.ptr(),m_pDescriptor->szNick.ptr(),m_pDescriptor->szUser.ptr(),m_pDescriptor->szHost.ptr()));
		} else {
			if(TRIGGER_EVENT_5PARAM_RETVALUE(KviEvent_OnDCCSendConnected,this,m_pDescriptor->szPort.ptr(),m_pDescriptor->szFileName.ptr(),m_pDescriptor->szNick.ptr(),m_pDescriptor->szUser.ptr(),m_pDescriptor->szHost.ptr()));
		}

		output(KVI_OUT_DCCMSG,__tr("Contacting host %s on port %s"),m_pDescriptor->szIp.ptr(),m_pDescriptor->szPort.ptr());
	}
}

void KviDccSend::startingSSLHandshake()
{
#ifdef COMPILE_SSL_SUPPORT
	outputNoFmt(KVI_OUT_SSL,__tr("Low lewel transport connection estabilished"));
	outputNoFmt(KVI_OUT_SSL,__tr("Starting Secure Socket Layer handshake"));
#endif
}

void KviDccSend::sslError(const char * msg)
{
#ifdef COMPILE_SSL_SUPPORT
	output(KVI_OUT_DCCERROR,__tr("[SSL ERROR]: %s"),msg);
#endif
}

void KviDccSend::startConnection()
{
	if(!(m_pDescriptor->bActive))
	{
		// PASSIVE CONNECTION
		output(KVI_OUT_DCCMSG,__tr("Attempting a passive DCC %s connection"),m_szDccType.ptr());
		int ret = m_pMarshal->dccListen(m_pDescriptor->szListenIp.ptr(),m_pDescriptor->szListenPort.ptr(),m_pDescriptor->bDoTimeout);
		if(ret != KviError_success)handleMarshalError(ret);
	} else {
		// ACTIVE CONNECTION
		output(KVI_OUT_DCCMSG,__tr("Attempting an active DCC %s connection"),m_szDccType.ptr());
		if(m_pDescriptor->bResume && m_pDescriptor->bRecvFile)
		{
			KviStr fName  = m_pDescriptor->szFileName.ptr();
			if(fName.contains(' '))
			{
				fName.prepend("\"");
				fName.append("\"");
			}
			m_pDescriptor->pConsole->socket()->sendFmtData("PRIVMSG %s :%cDCC RESUME %s %s %s%c",
				m_pDescriptor->szNick.ptr(),0x01,fName.ptr(),m_pDescriptor->szPort.ptr(),
				m_pDescriptor->szLocalFileSize.ptr(),0x01);
			output(KVI_OUT_DCCMSG,__tr("Sent DCC RESUME request...waiting for ACCEPT"));
		} else {
			int ret = m_pMarshal->dccConnect(m_pDescriptor->szIp.ptr(),m_pDescriptor->szPort.ptr(),m_pDescriptor->bDoTimeout);
			if(ret != KviError_success)handleMarshalError(ret);
		}
	}
}


void KviDccSend::setupFilePopup()
{
	m_pFilePopup->clear();
	m_pFilePopup->insertItem(__tr("Guess media type"),this,SLOT(slotGuessMediaType()));
	m_pFilePopup->insertItem(__tr("Open folder"),this,SLOT(slotOpenFolder()));
	m_pFilePopup->insertItem(__tr("Open file (/play)"),this,SLOT(slotOpenFile()));
}

const char * KviDccSend::target()
{
	// This may change on the fly...
	m_szTarget.sprintf("%s@%s:%s",
		m_pDescriptor->szNick.ptr(),m_pDescriptor->szIp.ptr(),m_pDescriptor->szPort.ptr());
	return m_szTarget.ptr();
}


void KviDccSend::slotGuessMediaType()
{
	g_pMediaManager->lock();
	KviMediaType * t = g_pMediaManager->findMediaType(m_pDescriptor->szLocalFileName.ptr());
	if(t)
	{
		output(KVI_OUT_SYSTEMMESSAGE,__tr("File: %s"),m_pDescriptor->szLocalFileName.ptr());
		output(KVI_OUT_SYSTEMMESSAGE,__tr("Iana type: %s"),t->szIanaType.ptr());
		output(KVI_OUT_SYSTEMMESSAGE,__tr("Description: %s"),t->szDescription.ptr());
		output(KVI_OUT_SYSTEMMESSAGE,__tr("File mask: %s"),t->szFileMask.ptr());
		output(KVI_OUT_SYSTEMMESSAGE,__tr("Magic bytes: %s"),t->szMagicBytes.ptr());
		output(KVI_OUT_SYSTEMMESSAGE,__tr("Commandline: %s"),t->szCommandline.ptr());
	} else {
		output(KVI_OUT_SYSTEMMESSAGE,__tr("No media type match for file \"%s\""),m_pDescriptor->szLocalFileName.ptr());
	}
	g_pMediaManager->unlock();
}

void KviDccSend::slotOpenFolder()
{
	KviStr tmp = m_pDescriptor->szLocalFileName.ptr();
	tmp.cutFromLast('/');
	KviStr szCommand(KviStr::Format,"dirbrowser.browse \"%s\"",tmp.ptr());
	g_pUserParser->parseCommandBuffer(szCommand.ptr(),this,0);
}

void KviDccSend::slotOpenFile()
{
	KviStr szCommand(KviStr::Format,"play \"%s\"",m_pDescriptor->szLocalFileName.ptr());
	g_pUserParser->parseCommandBuffer(szCommand.ptr(),this,0);
}

void KviDccSend::getBaseLogFileName(KviStr &buffer)
{
	buffer.sprintf("%s_%s_%s",m_pDescriptor->szNick.ptr(),m_pDescriptor->szLocalFileName.ptr(),m_pDescriptor->szPort.ptr());
}

void KviDccSend::fillCaptionBuffers()
{
	KviStr tmp(KviStr::Format,"dcc %s %s@%s:%s %s",
		m_pDescriptor->bIsTdcc ? (m_pDescriptor->bRecvFile ? "trecv" : "tsend") : (m_pDescriptor->bRecvFile ? "recv" : "send"),
		m_pDescriptor->szNick.ptr(),m_pDescriptor->szIp.ptr(),m_pDescriptor->szPort.ptr(),
		m_pDescriptor->szLocalFileName.ptr());

	m_szPlainTextCaption = tmp;

	m_szHtmlActiveCaption.sprintf("<nobr><font color=\"%s\"><b>%s</b></font></nobr>",
		KVI_OPTION_COLOR(KviOption_colorCaptionTextActive).name().ascii(),tmp.ptr());
	m_szHtmlInactiveCaption.sprintf("<nobr><font color=\"%s\"><b>%s</b></font></nobr>",
		KVI_OPTION_COLOR(KviOption_colorCaptionTextInactive).name().ascii(),tmp.ptr());
}

QPixmap * KviDccSend::myIconPtr()
{
	return g_pIconManager->getSmallIcon(KVI_SMALLICON_DCCREQUEST);
}

void KviDccSend::fillContextPopup(QPopupMenu * p)
{
	p->insertSeparator();
	int id = p->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_DCCERROR)),__tr("Close all terminated transfers"),g_pDccBroker,SLOT(closeAllTerminatedDccSendTransfers()));
	if(!g_pDccBroker->terminatedDccSendTransfersCount())p->setItemEnabled(id,false);
}

bool KviDccSend::event(QEvent *e)
{
	if(e->type() == KVI_THREAD_EVENT)
	{
		switch(((KviThreadEvent *)e)->id())
		{
			case KVI_DCC_THREAD_EVENT_ERROR:
			{
				m_bRunning = false;
				int * err = ((KviThreadDataEvent<int> *)e)->getData();
				output(KVI_OUT_DCCERROR,__tr("ERROR: %s"),kvi_getErrorString(*err));
				delete err;
				m_pUpdateTimer->stop();
				if(m_pDescriptor->bRecvFile)updateDccRecv();
				else updateDccSend();
				return true;
			}
			break;
			case KVI_DCC_THREAD_EVENT_SUCCESS:
				m_bRunning = false;
				outputNoFmt(KVI_OUT_DCCMSG,__tr("Transfer succesfull"));
				if(m_pDescriptor->bRecvFile)updateDccRecv();
				else updateDccSend();
				m_pUpdateTimer->stop();
				if ((kvi_strEqualCS(m_szDccType.ptr(), "RECV")) || (kvi_strEqualCS(m_szDccType.ptr(),"TRECV"))) 
				{
					if(TRIGGER_EVENT_5PARAM_RETVALUE(KviEvent_OnDCCGetTransferComplete,this,m_pDescriptor->szPort.ptr(),m_pDescriptor->szFileName.ptr(),m_pDescriptor->szNick.ptr(),m_pDescriptor->szUser.ptr(),m_pDescriptor->szHost.ptr()));
				} else {
					if(TRIGGER_EVENT_5PARAM_RETVALUE(KviEvent_OnDCCSendTransferComplete,this,m_pDescriptor->szPort.ptr(),m_pDescriptor->szFileName.ptr(),m_pDescriptor->szNick.ptr(),m_pDescriptor->szUser.ptr(),m_pDescriptor->szHost.ptr()));
				}

				if(KVI_OPTION_BOOL(KviOption_boolNotifyDccSendSuccessInConsole))
				{
					g_pApp->activeConsole()->output(KVI_OUT_DCCMSG,__tr("DCC %s transfer with %s@%s:%s succesfull: file \r![!dbl]play $0\r%s\r"),
						m_pDescriptor->bIsTdcc ? (m_pDescriptor->bRecvFile ? "trecv" : "tsend") : (m_pDescriptor->bRecvFile ? "recv" : "send"),
						m_pDescriptor->szNick.ptr(),m_pDescriptor->szIp.ptr(),m_pDescriptor->szPort.ptr(),
						m_pDescriptor->szLocalFileName.ptr());
				}
				if(m_pDescriptor->bRecvFile)g_pApp->dccFileReceived(m_pDescriptor->szFileName.ptr(),m_pDescriptor->szLocalFileName.ptr(),m_pDescriptor->szNick.ptr());				
				if(KVI_OPTION_BOOL(KviOption_boolAutoCloseDccSendOnSuccess))close();
				return true;
			break;
			case KVI_DCC_THREAD_EVENT_MESSAGE:
			{
				KviStr * str = ((KviThreadDataEvent<KviStr> *)e)->getData();
				outputNoFmt(KVI_OUT_DCCMSG,__tr_no_xgettext(str->ptr()));
				delete str;
				return true;
			}
			break;
			default:
				debug("Invalid event type %d received",((KviThreadEvent *)e)->id());
			break;
		}
	}
//#warning "Remove this!"
//	if(e->type() == QEvent::Close)debug("Close event received");
	return KviWindow::event(e);
}

void KviDccSend::resizeEvent(QResizeEvent *e)
{
//	int hght = m_pInput->heightHint();
	int hght2 = m_pVBox->sizeHint().height();
	m_pVBox->setGeometry(0,0,width(),hght2);
	m_pSplitter->setGeometry(0,hght2,width(),height() - hght2);
//	m_pInput->setGeometry(0,height() - hght,width(),hght);
}

QSize KviDccSend::sizeHint() const
{
	int w = m_pIrcView->sizeHint().width();
	int w2 = m_pVBox->sizeHint().width();
	QSize ret(w > w2 ? w : w2,
		m_pIrcView->sizeHint().height() + m_pVBox->sizeHint().height());
	return ret;
}

void KviDccSend::handleMarshalError(int err)
{
	m_bRunning = false;
	output(KVI_OUT_DCCERROR,__tr("DCC Failed: %s"),kvi_getErrorString(err));	
}

void KviDccSend::connected()
{
	output(KVI_OUT_DCCMSG,__tr("Connected to %s:%s"),
		m_pMarshal->remoteIp(),m_pMarshal->remotePort());
	output(KVI_OUT_DCCMSG,__tr("Local end is %s:%s"),
		m_pMarshal->localIp(),m_pMarshal->localPort());
	if(!(m_pDescriptor->bActive))
	{
		m_pDescriptor->szIp   = m_pMarshal->remoteIp();
		m_pDescriptor->szPort = m_pMarshal->remotePort();
		m_pDescriptor->szHost = m_pMarshal->remoteIp();
	}
	updateCaption();
	if(m_pDescriptor->bRecvFile)
	{
		KviDccRecvThreadOptions * o = new KviDccRecvThreadOptions;
		o->szFileName     = m_pDescriptor->szLocalFileName.ptr();
		bool bOk;
		o->iTotalFileSize = m_pDescriptor->szFileSize.toInt(&bOk);
		if(!bOk)o->iTotalFileSize = -1;
		o->bResume        = m_pDescriptor->bResume;
		o->bIsTdcc        = m_pDescriptor->bIsTdcc;
		o->bSendZeroAck   = KVI_OPTION_BOOL(KviOption_boolSendZeroAckInDccRecv);
		o->bNoAcks         = m_pDescriptor->bNoAcks;
		m_pSlaveRecvThread = new KviDccRecvThread(this,m_pMarshal->releaseSocket(),o);
		connect(m_pUpdateTimer,SIGNAL(timeout()),this,SLOT(updateDccRecv()));
		m_pUpdateTimer->start(1000);
		m_pSlaveRecvThread->start();
	} else {
		KviDccSendThreadOptions * o = new KviDccSendThreadOptions;
		o->szFileName     = m_pDescriptor->szLocalFileName.ptr();
		o->bFastSend      = KVI_OPTION_BOOL(KviOption_boolUseFastDccSend);
		o->iIdleStepLengthInMSec = KVI_OPTION_UINT(KviOption_uintDccSendIdleStepInMSec);
		bool bOk;
		o->bIsTdcc        = m_pDescriptor->bIsTdcc;
		o->iStartPosition  = m_pDescriptor->szFileSize.toInt(&bOk);
		if(!bOk || (o->iStartPosition < 0))o->iStartPosition = 0;
		o->iPacketSize     = KVI_OPTION_UINT(KviOption_uintDccSendPacketSize);
		if(o->iPacketSize < 32)o->iPacketSize = 32;
		o->bNoAcks         = m_pDescriptor->bNoAcks;
		m_pSlaveSendThread = new KviDccSendThread(this,m_pMarshal->releaseSocket(),o);
		connect(m_pUpdateTimer,SIGNAL(timeout()),this,SLOT(updateDccSend()));
		m_pUpdateTimer->start(1000);
		m_pSlaveSendThread->start();
	}
}

void KviDccSend::updateDccRecv()
{
	if(!m_pSlaveRecvThread)
	{
		m_pUpdateTimer->stop();
		return;
	}
	m_pSlaveRecvThread->initGetInfo();
	int dataProgress = m_pSlaveRecvThread->progress();
	m_pDataProgressBar->setProgress(dataProgress);
	setProgress(dataProgress);
	KviStr tmp(KviStr::Format,__tr("Received %d bytes"),m_pSlaveRecvThread->filePosition());
	m_pBytesLabel->setText(tmp.ptr());
	int el = m_pSlaveRecvThread->elapsedTime();
	int elh = el / 3600;
	el = el % 3600;
	int elm = el / 60;
	el = el % 60;
	tmp.sprintf(__tr("%d h %d m %d s"),elh,elm,el);
	m_pElapsedTimeLabel->setText(tmp.ptr());
	tmp.sprintf(__tr("Avg: %d (bytes/sec)"),m_pSlaveRecvThread->averageSpeed());
	m_pAvgSpeedLabel->setText(tmp.ptr());
	tmp.sprintf(__tr("Spd: %d (bytes/sec)"),m_pSlaveRecvThread->instantSpeed());
	m_pInstantSpeedLabel->setText(tmp.ptr());
	m_pSlaveRecvThread->doneGetInfo();
}

void KviDccSend::updateDccSend()
{
	if(!m_pSlaveSendThread)
	{
		m_pUpdateTimer->stop();
		return;
	}
	m_pSlaveSendThread->initGetInfo();
	int dataProgress = m_pSlaveSendThread->progress();
	m_pDataProgressBar->setProgress(dataProgress);
	if(!m_pDescriptor->bNoAcks)
	{
		dataProgress = m_pSlaveSendThread->ackedProgress();
		m_pAckProgressBar->setProgress(dataProgress);
		setProgress(dataProgress);
	} else {
		setProgress(dataProgress);
	}
	KviStr tmp(KviStr::Format,__tr("Sent %d bytes"),m_pSlaveSendThread->filePosition());
	m_pBytesLabel->setText(tmp.ptr());
	int el = m_pSlaveSendThread->elapsedTime();
	int elh = el / 3600;
	el = el % 3600;
	int elm = el / 60;
	el = el % 60;
	tmp.sprintf(__tr("%d h %d m %d s"),elh,elm,el);
	m_pElapsedTimeLabel->setText(tmp.ptr());
	tmp.sprintf(__tr("Avg: %d (bytes/sec)"),m_pSlaveSendThread->averageSpeed());
	m_pAvgSpeedLabel->setText(tmp.ptr());
	tmp.sprintf(__tr("Spd: %d (bytes/sec)"),m_pSlaveSendThread->instantSpeed());
	m_pInstantSpeedLabel->setText(tmp.ptr());
	m_pSlaveSendThread->doneGetInfo();
}

bool KviDccSend::resumeAccepted(const char *filename,const char* port)
{
	if(kvi_strEqualCI(filename,m_pDescriptor->szFileName.ptr()) &&
		kvi_strEqualCI(port,m_pDescriptor->szPort.ptr()) &&
		(!m_pSlaveRecvThread) && m_pDescriptor->bResume && m_pDescriptor->bRecvFile)
	{
		output(KVI_OUT_DCCMSG,__tr("RESUME accepted: the transfer will begin at position %s"),
			m_pDescriptor->szLocalFileSize.ptr());
		int ret = m_pMarshal->dccConnect(m_pDescriptor->szIp.ptr(),
						m_pDescriptor->szPort.ptr(),m_pDescriptor->bDoTimeout);
		if(ret != KviError_success)handleMarshalError(ret);
		else output(KVI_OUT_DCCMSG,__tr("Contacting host %s on port %s"),
				m_pDescriptor->szIp.ptr(),m_pDescriptor->szPort.ptr());

		return true;
	}
	return false;
}

bool KviDccSend::doResume(const char * filename,const char * port,unsigned int filePos)
{
	if(kvi_strEqualCI(port,m_pMarshal->dccPort()) &&
		(!m_pSlaveRecvThread) && (!m_pDescriptor->bRecvFile))
	{
		if(kvi_strEqualCI(filename,m_pDescriptor->szFileName.ptr()) || KVI_OPTION_BOOL(KviOption_boolAcceptBrokenFileNameDccResumeRequests))
		{
			bool bOk;
			unsigned int iLocalFileSize = m_pDescriptor->szLocalFileSize.toUInt(&bOk);
			if(!bOk)
			{
				// ops...internal error
				outputNoFmt(KVI_OUT_DCCERROR,__tr("Internal error in RESUME request"));
				return false;
			}
			if(iLocalFileSize <= filePos)
			{
				output(KVI_OUT_DCCERROR,__tr("Invalid RESUME request: position %u is 'out of the file'"),filePos);
				return false;
			}
			output(KVI_OUT_DCCMSG,__tr("Accepting RESUME request: transfer will initiate at position %u"),filePos);
			m_pDescriptor->szFileSize.setNum(filePos);
			m_pDescriptor->pConsole->socket()->sendFmtData("PRIVMSG %s :%cDCC ACCEPT %s %s %u%c",
				m_pDescriptor->szNick.ptr(),0x01,filename,port,filePos,0x01);
			return true;
		}
	}
	return false;
}

#include "m_send.moc"
