//
//   File : libkviobjects.cpp
//   Creation date : Wed Sep 09 2000 20:59:01 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2000-2001 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#define _KVI_DEBUG_CHECK_RANGE_
#include "kvi_debug.h"
#include "kvi_settings.h"
#include "kvi_string.h"
#include "kvi_module.h"
#include "kvi_sparser.h"
#include "kvi_locale.h"
#include "kvi_out.h"
#include "kvi_console.h"
#include "kvi_netutils.h"
#include "kvi_frame.h"
#include "kvi_console.h"
#include "kvi_uparser.h"
#include "kvi_error.h"
#include "kvi_options.h"
#include "kvi_defaults.h"
#include "kvi_filetrader.h"
#include "kvi_mirccntrl.h"
#include "kvi_app.h"

#include "gsmcodec.h"
#include "broker.h"
#include "voice.h"

#ifdef COMPILE_ON_WINDOWS
	// Ugly Windoze compiler...
	#include "dialogs.h"
#endif

//#warning "KviOption_boolIgnoreDccChat and other types too"

extern KVIRC_API KviFileTrader * g_pFileTrader;

KviDccBroker * g_pDccBroker = 0;

static void dcc_module_reply_errmsg(KviDccRequest * dcc,const char * text)
{
	dcc->ctcpMsg->msg->console()->socket()->sendFmtData(
		"NOTICE %s :%cERRMSG %s%c",
		dcc->ctcpMsg->pSource->nick(),0x01,text,0x01);
}

static void dcc_module_request_error(KviDccRequest * dcc,const char * errText)
{
	dcc->ctcpMsg->msg->console()->output(KVI_OUT_DCCERROR,
			__tr("Unable to process the above request: %s: %s"),
			errText,
			KVI_OPTION_BOOL(KviOption_boolNotifyFailedDccHandshakes) ? __tr("ignoring and notifying failure") : __tr("ignoring"));

	if(KVI_OPTION_BOOL(KviOption_boolNotifyFailedDccHandshakes))
	{
		KviStr szError(KviStr::Format,"Sorry, your dcc %s request can't be satisfied: %s",
			dcc->szType.ptr(),errText);
		dcc_module_reply_errmsg(dcc,szError.ptr());
	}
}

static bool normalizeTargetData(KviDccRequest * dcc,KviStr &ipaddr,KviStr &port)
{
	if(!port.isUnsignedNum())
	{
		if(!dcc->ctcpMsg->msg->haltOutput())
		{
			KviStr szError(KviStr::Format,__tr("invalid port number %s"),port.ptr());
			dcc_module_request_error(dcc,szError.ptr());
		}
		return false;
	}

	struct in_addr addr;

	if(ipaddr.isUnsignedNum())
	{
		addr.s_addr = htonl((unsigned long)ipaddr.toULong());
		KviStr tmp;
		if(!kvi_binaryIpToStringIp(addr,tmp))
		{
			if(!dcc->ctcpMsg->msg->haltOutput())
			{
				KviStr szError(KviStr::Format,__tr("invalid ip address in old format %s"),ipaddr.ptr());
				dcc_module_request_error(dcc,szError.ptr());
			}
			return false;
		}
		ipaddr = tmp;
	} else {
		if(!kvi_stringIpToBinaryIp(ipaddr,&addr))
		{
#ifdef COMPILE_IPV6_SUPPORT
			struct in6_addr addr6;
			if(kvi_stringIpToBinaryIp_V6(ipaddr,&addr6))
			{
				dcc->bIpV6 = true;
				return true; // IPV6 address.
			}
#endif
			if(!dcc->ctcpMsg->msg->haltOutput())
			{
				KviStr szError(KviStr::Format,__tr("invalid ip address %s"),ipaddr.ptr());
				dcc_module_request_error(dcc,szError.ptr());
			}
			return false;
		}
	}
	return true;
}

static bool dcc_module_get_listen_ip_address(KviCommand *c,KviConsole * pConsole,KviStr &szListenIp)
{
	//
	// Find an interface suitable for listening....
	// Either from user options or from the current connection...
	//

	if(KVI_OPTION_BOOL(KviOption_boolDccListenOnSpecifiedInterfaceByDefault))
	{
		KVI_OPTION_STRING(KviOption_stringDccListenDefaultInterface).stripWhiteSpace();
		if(kvi_isValidStringIp(KVI_OPTION_STRING(KviOption_stringDccListenDefaultInterface).ptr()))
		{
			if(kvi_strEqualCI(KVI_OPTION_STRING(KviOption_stringDccListenDefaultInterface).ptr(),"0.0.0.0"))
			{
				// Try to find the first available IpV4 interface
				if(!kvi_getLocalHostAddress(szListenIp))
				{
					if(c)c->warning(__tr("Can't retrieve a suitable local IPV4 address"),
							KVI_OPTION_STRING(KviOption_stringDccListenDefaultInterface).ptr());
					return false;
				}
			} else {
				szListenIp = KVI_OPTION_STRING(KviOption_stringDccListenDefaultInterface);
			}
		} else {
#ifdef COMPILE_IPV6_SUPPORT
			if(kvi_isValidStringIp_V6(KVI_OPTION_STRING(KviOption_stringDccListenDefaultInterface).ptr()))
			{
				szListenIp = KVI_OPTION_STRING(KviOption_stringDccListenDefaultInterface);
			} else {
#endif
				if(!kvi_getInterfaceAddress(KVI_OPTION_STRING(KviOption_stringDccListenDefaultInterface).ptr(),szListenIp))
				{
					KVI_OPTION_BOOL(KviOption_boolDccListenOnSpecifiedInterfaceByDefault) = false;
					if(c)c->warning(__tr("Can't listen on default interface '%s': fix it in the options dialog, disabling the option (so the next dcc will work)"),
							KVI_OPTION_STRING(KviOption_stringDccListenDefaultInterface).ptr());
					return false;
				}
#ifdef COMPILE_IPV6_SUPPORT
			}
#endif
		}
	} else {
		if(pConsole)
		{
			if(pConsole->isConnected())
			{
				//#warning "The IPV6 choice is not OK here.... and maybe allow to bind to specified ports"
				pConsole->socket()->getLocalHostIp(szListenIp,pConsole->isIpV6Connection());
			} else {
				szListenIp = "0.0.0.0"; // huh ? :)
			}
		} else {
			szListenIp = "0.0.0.0";
		}
	}
	return true;
}

static bool dcc_module_parse_default_parameters(KviDccBrokerDescriptor * d,KviCommand *c)
{
	d->bIsTdcc = c->hasSwitch('t');

	d->bOverrideMinimize = c->hasSwitch('m');

	if(d->bOverrideMinimize)
	{
		KviStr tmpVal;
		if(!(c->getSwitchValue('m',tmpVal)))d->bShowMinimized = false;
		else d->bShowMinimized = kvi_strEqualCI(tmpVal.ptr(),"1");
	}

	if(!d->pConsole)
	{
		// We don't need a console with -c and -n , otherwise we need it
		if(!(c->hasSwitch('c') || c->hasSwitch('n')))
		{
			delete d;
			c->warning(__tr("This window has no associated irc context (you need one unless you pass -c or -n)"));
			return false;
		} else d->pConsole = c->window()->frame()->firstConsole();
	}

	__range_valid(d->pConsole);

	if(!d->pConsole->isConnected())
	{
		// We don't need a connection with -c and -n , otherwise we need it
		if(!(c->hasSwitch('c') || c->hasSwitch('n')))
		{
			delete d;
			c->warning(__tr("You are not connected to a server (you need a connection unless you pass -c or -n)"));
			return false;
		} else {
			// -c or -n , grab a local nick from somewhere
			d->szLocalNick  = KVI_OPTION_STRING(KviOption_stringNickname1);
			d->szLocalNick.stripWhiteSpace();
			if(d->szLocalNick.isEmpty())d->szLocalNick = KVI_DEFAULT_NICKNAME1;
			d->szLocalUser  = __tr("unknown"); // we can live without it
			d->szLocalHost  = d->szLocalUser;  // we can live without it
		}
	} else {
		// We know everything
		d->szLocalNick  = d->pConsole->currentNickName();
		d->szLocalUser  = d->pConsole->currentUserName();
		d->szLocalHost  = d->pConsole->currentLocalHostName();
	}

	if(c->hasSwitch('i'))
	{
		c->getSwitchValue('i',d->szListenIp);
		if(!(d->szListenIp.contains('.') || d->szListenIp.contains(':')))
		{
			// This will magically work with the same buffer as source and dst!
			if(!kvi_getInterfaceAddress(d->szListenIp.ptr(),d->szListenIp))
			{
				delete d;
				return c->error(KviError_invalidParameter,__tr("Can't get address of interface %s"),d->szListenIp.ptr());
			}
		}
	} else {
		if(!dcc_module_get_listen_ip_address(c,d->pConsole,d->szListenIp))
		{
			delete d;
			return c->error(KviError_invalidParameter,__tr("No suitable interface to listen on : use -i"));
		}
	}

	if(c->hasSwitch('p'))c->getSwitchValue('p',d->szListenPort);
	else d->szListenPort = "0"; // any port is ok

	if(c->hasSwitch('a'))c->getSwitchValue('a',d->szFakeIp);
	else {
		if(KVI_OPTION_BOOL(KviOption_boolDccSendFakeAddressByDefault))
		{
			d->szFakeIp = KVI_OPTION_STRING(KviOption_stringDefaultDccFakeAddress);
			if(d->szFakeIp.isEmpty())KVI_OPTION_BOOL(KviOption_boolDccSendFakeAddressByDefault) = false;
		}
	}

	if(c->hasSwitch('f'))c->getSwitchValue('f',d->szFakePort);

	d->bDoTimeout = (!c->hasSwitch('u'));
#ifdef COMPILE_SSL_SUPPORT
	d->bIsSSL = c->hasSwitch('s');
#else
	if(c->hasSwitch('s'))c->warning(__tr("This executable was build without SSL support: -s switch ignored"));
#endif

	return true;
}

static bool dcc_module_check_concurrent_transfers_limit(KviDccRequest * dcc)
{
	if(KVI_OPTION_UINT(KviOption_uintMaxDccSendTransfers) > 0)
	{
		unsigned int uTransfers = g_pDccBroker->runningDccSendTransfersCount();
		if(uTransfers >= KVI_OPTION_UINT(KviOption_uintMaxDccSendTransfers))
		{
			KviStr szError(KviStr::Format,__tr("concurrent transfers limit reached (%u of %u transfers running)"),
				uTransfers,KVI_OPTION_UINT(KviOption_uintMaxDccSendTransfers));
			dcc_module_request_error(dcc,szError.ptr());
			return false;
		}
	}
	return true;
}

static bool dcc_module_check_limits(KviDccRequest * dcc)
{
	if(KVI_OPTION_UINT(KviOption_uintMaxDccSlots) > 0)
	{
		unsigned int uWindows = g_pDccBroker->dccWindowsCount();
		if(uWindows >= KVI_OPTION_UINT(KviOption_uintMaxDccSlots))
		{
			KviStr szError(KviStr::Format,__tr("slots limit reached (%u slots of %u)"),
				uWindows,KVI_OPTION_UINT(KviOption_uintMaxDccSlots));
			dcc_module_request_error(dcc,szError.ptr());
			return false;
		}
	}
	if(g_pDccBroker->dccBoxCount() >= 32)
	{
		// there are too many pending dcc requests: the user isn't watching....
		dcc_module_request_error(dcc,__tr("too many pending connections"));
		return false;
	}
	return true;
}

static void dcc_module_set_dcc_type(KviDccBrokerDescriptor * d,const char * szBaseType)
{
	d->szType = szBaseType;
#ifdef COMPILE_SSL_SUPPORT
	if(d->bIsSSL)d->szType.prepend('S');
#endif
	if(d->bIsTdcc)d->szType.prepend('T');
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// CHAT
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

static void dccModuleParseDccChat(KviDccRequest *dcc)
{
	//
	// We have received a DCC CHAT request in the following form:
	//
	//     DCC CHAT chat <ipaddress> <port>
	//
	// This means that we're requested to setup an ACTIVE chat connection
	// ... Easy task :)
	//
	// Anybody understands the meaning of the secondo "chat" in there ?
	// It was meant to simplify the parsing ? :DDD
	//

	if(!dcc_module_check_limits(dcc))return;

	if(!normalizeTargetData(dcc,dcc->szParam2,dcc->szParam3))return;

	if(!kvi_strEqualCI(dcc->szParam1.ptr(),"chat"))
	{
		if(!dcc->ctcpMsg->msg->haltOutput())
		{
			dcc->ctcpMsg->msg->console()->output(KVI_OUT_DCCMSG,
				__tr("The above request is broken: the second parameter is '%s' and shoud be 'chat'; trying to continue"),dcc->szParam1.ptr());
		}
	}

	KviStr szExtensions = dcc->szType;
	szExtensions.cutRight(4); // cut off CHAT

#ifdef COMPILE_SSL_SUPPORT
	bool bSSLExtension   = szExtensions.contains('S',false);
#else //!COMPILE_SSL_SUPPORT
	if(szExtensions.contains('S',false))
	{
		dcc_module_request_error(dcc,__tr("This executable has been compiled without SSL support: the SSL extension to DCC CHAT is not available"));
		return;
	}
#endif //!COMPILE_SSL_SUPPORT

	KviDccBrokerDescriptor * d = new KviDccBrokerDescriptor;
	d->szNick            = dcc->ctcpMsg->pSource->nick();
	d->szUser            = dcc->ctcpMsg->pSource->username();
	d->szHost            = dcc->ctcpMsg->pSource->host();
	d->szLocalNick       = dcc->pConsole->currentNickName();
	d->szLocalUser       = dcc->pConsole->currentUserName();
	d->szLocalHost       = dcc->pConsole->currentLocalHostName();
	d->szIp              = dcc->szParam2.ptr();
	d->szPort            = dcc->szParam3.ptr();
	d->pConsole          = dcc->pConsole;
	d->bActive           = true; // we have to connect
	d->bIsTdcc           = false;
	d->bNoAcks           = false;
	d->bOverrideMinimize = false;
#ifdef COMPILE_SSL_SUPPORT
	d->bIsSSL            = bSSLExtension;
#endif
	d->bAutoAccept       = KVI_OPTION_BOOL(KviOption_boolAutoAcceptDccChat);
	dcc_module_set_dcc_type(d,"CHAT");

	g_pDccBroker->activeChatManage(d);
}

/*
	@doc: dcc.chat
	@type:
		command
	@title:
		dcc.chat
	@short:
		Starts a DCC CHAT connection
	@syntax:
		dcc.chat [-s] [-n] [-c] [-u] [-m[=<boolean>]] [-i=<interface>] [-p=<port>] [-a=<fake address>] [-f=<fake port>] <nickname>
	@description:
		Attempts a dcc connection to <nickname>.[br]
		The simplest case "dcc.chat <nickname>" will work just as in all
		the other IRC clients, but this command is really more powerful...[br]
		Before attempting to understand the possibilities of this command,
		be sure to know how a [doc:dcc_connection]DCC negotiation and connection[/doc] works.
		If the 'i' switch is specified , the local listening socket
		will be bound to the specified <interface> (which is an IP address: IPV4 or IPV6),
		otherwise it will be bound to the interface of the
		current irc connection.[br]
		You can also specify a local interface name to get the address from (this works only for IPV4 interfaces
		since IPV6 ones seems to be unsupported by the system ioctl() calls at the moment (for linux at least)).[br]
		Here go some examples:[br]
		-i=215.243.12.12: this will bind to the IPV4 interface with the specified address.[br]
		-i=3ffe:1001::1: this will bind to the IPV6 interface with the specified address.[br]
		-i=ppp0: this will bind to the IPV4 address of the interface ppp0 (if supported by the underlying system).[br]
		The -i switch parameter may serve also as a target address when the -c switch is used.[br]
		If the 'p' switch is specified , the local listening socket
		will be bound to the <port>, otherwise it will be bound to
		a random port choosen by the kernel.[br]
		If the 'a' switch is specified , the requesting CTCP message
		will contain <fake address> as target for the remote user,
		otherwise the CTCP message will contain the real IP address
		of the listening interface.
		If the 'f' switch is specified , the requesting CTCP message
		will contain <fake port> as target for the remote user,
		otherwise the CTCP message will contain the real port of the
		listening socket.
		All these switches are meant to allow maximum felxibility of the
		dcc negotiation: older KVIrc releases had serious problems
		with firewalled and/or masqueraded machines. With the -a and -f switches
		you can workaround it.
		If the 'n' switch is specified , KVIrc will NOT send the CTCP request
		to the target user: you will have to do it manually , or the remote user
		will have to connect manually (for example by using dcc.chat -c).
		If the 'c' switch is specified , KVIrc will attempt to connect
		to the remote host specified as <interface> and <port>, instead
		of listening (active connection instead of a passive one).
		In this case the -i and -p switches are mandatory.[br]
		The 'c' switch takes precedence over 'n' (In fact both should
		be mutually exclusive).[br]
		If the 'c' and 'n' switches are missing, this commands
		needs to be executed in a window that is bound to a connected
		irc context (you need a third entity to accomplish the negotiation).[br]
		If the 'u' switch is given , the connection attempt will
		never time out; this might be useful if you want to leave
		a listening socket for a friend of yours while you are sleeping
		and have the CTCP processing disabled. The 'u' switch works either
		in active and passive mode.[br]
		If the -m switch is passed, the default boolCreateMinimizedDccChat option
		is overridden with the <boolean> parameter passed. So actually
		by passing -m=1 will create a minimized dcc chat even
		if the [fnc]$option[/fnc](boolCreateMinimizedDccChat) returns false.[br]
		In the same way, by passing -m=0 you will create a non minimized dcc chat.
		If no <boolean> value is specified , it defaults to 1.[br]
		-s will cause the DCC chat to be attempted in Secure Sockets Layer mode:
		the connection will be encrypted with a private key algorithm after a
		public key handshake. -s will work only if the kvirc executable was compiled
		with SSL support enabled and if the remote end supports it.
		The eventual DCC request will contain the string SCHAT instead of CHAT.[br]
		When requesting a SSL based dcc chat to someone you probably will need a
		certificate. If you don't have one , create it (for example with CA.pl -newcert)
		and set it in the options dialog.
	@examples:
		Simple examples:
		[example]
			# Simple dcc chat to Pragma
			dcc.chat Pragma
			# Dcc chat to pragma , listen on address 127.0.0.1
			dcc.chat -i=127.0.0.1 Pragma
			# Dcc chat to pragma , listen on address 168.0.0.1 and port 1025
			dcc.chat -i=168.0.0.1 -p=1025 Pragma
		[/example]
		More tricky ones:
		[example]
			# Dcc chat to pragma , listen on address 127.0.0.1 and port 1080
			# but tell him to connect to address 212.134.22.11 port 1080
			dcc.chat -i=127.0.0.1 -p=1080 -a=212.134.22.11 Pragma
			# Dcc chat to pragma , listen on address 127.0.0.1 and port 1080
			# but tell him to connect to address 212.134.22.11 port 1090
			dcc.chat -i=127.0.0.1 -p=1080 -a=212.134.22.11 -f=1090 Pragma
		[/example]
		Now run completly out of bounds. Use dcc.chat connections
		to do various things:
		[example]
			# Tricky : simulate a HTTP server
			dcc.chat -n -i=127.0.0.1 -p=80 Netscape
			# Now open http://127.0.0.1 with netscape
			# and type "<html><body>Hello!</body></html>" :)
			#
			# Tricky 2: simulate the ident daemon (if you have none)
			dcc.chat -n -i=127.0.0.1 -p=113 Client
			#
			# Now a self-DCC connection without the IRC negotiation
			# Src: Setup a listening socket awaiting the "Destination" user connection
			dcc.chat -n -i=127.0.0.1 -p=1080 Dst
			# Dst: Connect to the listening socket at addr 127.0.0.1 and port 1080
			dcc.chat -c -i=127.0.0.1 -p=1080 Src
			# The above example will mess you a bit...
			# Try to guess why in both windows YOU have the same nickname
			# that is probably not Dst nor Src... :)
		[/example]
		Using the shell ftp proggie is too easy:
		we're [b]real hackers[/b] and want to do complicated things...
		[example]
			# Connect to the local ftp server and get the list of contents
			/dcc.chat -c -i=127.0.0.1 -p=21 FtpServer
			# Now login: type in the new window (the following lines are NOT commands):
			USER youruser
			PASS yourpass
			# Now enter passive mode
			PASV
			# And watch the line that you have received...sth like
			# 227 Entering passive mode (127,0,0,1,210,195)
			/dcc.chat -c -i=127.0.0.1 -p=$((210 * 256) + 195) FtpList
			# (Change the port numbers accordingly)
			# And then type in the FtpServer window (this is NOT a kvirc command):
			LIST
			# Then watch the ls output in the FtpList window. :)
			# In this way you can also read ascii files by ftp:
			# Assume that in the previous ls output you have seen
			# a README file.
			# In the FtpServer window type:
			PASV
			# Watch the message
			# 227 Entering passive mode (127,0,0,1,22,227)
			/dcc.chat -c -i=127.0.0.1 -p=$((22 * 256) + 227) README
			# In the FtpServer window type:
			RETR README
			# And read the file in the README window :)
		[/example]
*/

static bool dcc_module_cmd_chat(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"dcc_module_cmd_chat");

	KviStr target;
	if(!g_pUserParser->parseCmdFinalPart(c,target))return false;

	if(target.isEmpty())return c->error(KviError_notEnoughParameters,"%s",__tr("Missing target nickname"));

	KviDccBrokerDescriptor * d = new KviDccBrokerDescriptor;

	d->szNick       = target.ptr();    // we always specify the nickname
	d->szUser       = __tr("unknown"); // username is always unknown
	d->szHost       = d->szUser;       // host is always unknown
	d->pConsole     = c->window()->console();

	if(!dcc_module_parse_default_parameters(d,c))return false;
	dcc_module_set_dcc_type(d,"CHAT");

	if(c->hasSwitch('c'))
	{
		if(!(c->hasSwitch('i') && c->hasSwitch('p')))
		{
			delete d;
			return c->error(KviError_notEnoughParameters,__tr("-c requires -i and -p"));
		}
		d->szIp         = d->szListenIp;
		d->szPort       = d->szListenPort;
		d->szListenIp   = ""; // useless
		d->szListenPort = ""; // useless
		d->bActive      = true;

		g_pDccBroker->activeChatExecute(0,d);
	} else {
		d->szIp         = __tr("unknown");
		d->szPort       = d->szIp;
		d->bActive      = false;
		d->bSendRequest = !c->hasSwitch('n');

		g_pDccBroker->passiveChatExecute(d);
	}

	return c->leaveContext();
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// SEND
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

static void dccModuleParseDccSend(KviDccRequest *dcc)
{
//#warning "Ignore files depending on file type ? (MediaType ?)"
	//
	// We have received a DCC SEND request in the following form
	//
	//      DCC [ST]SEND <filename> <ipaddress> <port> <filesize>
	//
	// Now the things are a bit tricky... we eventually can
	// reply with a DCC RESUME and receive a DCC ACCEPT then
	// The format of these requests is:
	//
	//      DCC RESUME <filename> <port> <resumepos>
	//      ACCEPT <filename> <port> <resumepos>
	//

	if(!dcc_module_check_limits(dcc))return;
	if(!dcc_module_check_concurrent_transfers_limit(dcc))return;

	if(!normalizeTargetData(dcc,dcc->szParam2,dcc->szParam3))return;

	if(!(dcc->szParam4.isUnsignedNum()))
	{
		if(!dcc->ctcpMsg->msg->haltOutput())
		{
			dcc->ctcpMsg->msg->console()->output(KVI_OUT_DCCMSG,
				__tr("The above request is broken: the fourth parameter should be the file size but does not look as an unsigned number; trying to continue"),dcc->szParam4.ptr());
		}		
		dcc->szParam4 = __tr("<unknown size>");
	}

	if(dcc->szParam1.contains('/'))
	{
		if(!dcc->ctcpMsg->msg->haltOutput())
		{
			dcc->ctcpMsg->msg->console()->output(KVI_OUT_DCCMSG,
				__tr("The above request is broken: the filename contains path components; stripping the leading path and trying to continue"),dcc->szParam1.ptr());
		}		
		dcc->szParam1.cutToLast('/');
	}

	KviStr szExtensions = dcc->szType;
	szExtensions.cutRight(4); // cut off SEND

	bool bTurboExtension = szExtensions.contains('T',false);
#ifdef COMPILE_SSL_SUPPORT
	bool bSSLExtension   = szExtensions.contains('S',false);
#else //!COMPILE_SSL_SUPPORT
	if(szExtensions.contains('S',false))
	{
		dcc_module_request_error(dcc,__tr("This executable has been compiled without SSL support: the SSL extension to DCC SEND is not available"));
		return;
	}
#endif //!COMPILE_SSL_SUPPORT

	KviDccBrokerDescriptor * d = new KviDccBrokerDescriptor;
	d->szNick            = dcc->ctcpMsg->pSource->nick();
	d->szUser            = dcc->ctcpMsg->pSource->username();
	d->szHost            = dcc->ctcpMsg->pSource->host();
	d->szLocalNick       = dcc->pConsole->currentNickName();
	d->szLocalUser       = dcc->pConsole->currentUserName();
	d->szLocalHost       = dcc->pConsole->currentLocalHostName();
	d->szIp              = dcc->szParam2.ptr();
	d->szPort            = dcc->szParam3.ptr();
	d->szFileName        = dcc->szParam1.ptr();
	d->szFileSize        = dcc->szParam4.ptr();
	d->pConsole          = dcc->pConsole;
	d->bActive           = true; // we have to connect
	d->bResume           = false;
	d->bRecvFile         = true;
	d->bIsTdcc           = bTurboExtension;
	d->bNoAcks           = d->bIsTdcc;
#ifdef COMPILE_SSL_SUPPORT
	d->bIsSSL            = bSSLExtension;
#endif
	d->bOverrideMinimize = false;
	d->bAutoAccept       = KVI_OPTION_BOOL(KviOption_boolAutoAcceptDccSend);
	d->bIsIncomingAvatar = g_pApp->findPendingAvatarChange(dcc->pConsole,d->szNick.ptr(),d->szFileName.ptr());
	dcc_module_set_dcc_type(d,"RECV");
	if(KVI_OPTION_BOOL(KviOption_boolAutoAcceptIncomingAvatars))d->bAutoAccept = d->bAutoAccept || d->bIsIncomingAvatar;
	g_pDccBroker->recvFileManage(d);
}

static void dccModuleParseDccAccept(KviDccRequest *dcc)
{
	if(!g_pDccBroker->handleResumeAccepted(dcc->szParam1.ptr(),dcc->szParam2.ptr()))
	{
//#warning "IF KviOption_boolReplyCtcpErrmsgOnInvalidAccept..."
		if(!dcc->ctcpMsg->msg->haltOutput())
		{
//			dcc->ctcpMsg->msg->console()->output(KVI_OUT_DCCERROR,
//				__tr("Can't proceed with dcc recv: transfer not initiated for file %s on port %s"),
//					dcc->szParam1.ptr(),dcc->szParam2.ptr());

			KviStr szError(KviStr::Format,__tr("can't proceed with dcc recv: transfer not initiated for file %s on port %s"),dcc->szParam1.ptr(),dcc->szParam2.ptr());
			dcc_module_request_error(dcc,szError.ptr());
		}
	}
}

static void dccModuleParseDccResume(KviDccRequest *dcc)
{
	bool bOk;
	unsigned int filePos = dcc->szParam3.toUInt(&bOk);
	if(!bOk)
	{
		if(!dcc->ctcpMsg->msg->haltOutput())
		{
			KviStr szError(KviStr::Format,__tr("invalid resume position argument '%s'"),dcc->szParam3.ptr());
			dcc_module_request_error(dcc,szError.ptr());

//			dcc->ctcpMsg->msg->console()->output(KVI_OUT_DCCERROR,
//				__tr("Unable to process the above request: invalid resume position argument '%s'"),dcc->szParam3.ptr());
//			if(KVI_OPTION_BOOL(KviOption_boolNotifyFailedDccHandshakes))
//			{
//				dcc_module_reply_errmsg(dcc,"Sorry, your resume request seems to be invalid");
//			}
		}
		return;
	}

	if(!g_pDccBroker->handleResumeRequest(dcc->szParam1.ptr(),dcc->szParam2.ptr(),filePos))
	{
//#warning "IF KviOption_boolReplyCtcpErrmsgOnInvalidResume..."
		if(!dcc->ctcpMsg->msg->haltOutput())
		{
			KviStr szError(KviStr::Format,
					__tr("can't proceed with dcc send: transfer not initiated for file %s on port %s, or invalid resume size"),
					dcc->szParam1.ptr(),dcc->szParam2.ptr());
			dcc_module_request_error(dcc,szError.ptr());
		}
	}
}

/*
	@doc: dcc.send
	@type:
		command
	@title:
		dcc.send
	@short:
		Sends a file
	@syntax:
		dcc.send [-s] [-n] [-c] [-u] [-b] [-g[=<file size>]] [-t] [-m[=<boolean>]] [-i=<interface>] [-p=<port>] [-a=<fake address>] [-f=<fake port>] <nickname> [filename]
	@description:
		Attempts to send the file <filename> to <nickname>.[br]
		If [filename] is specified it must be an absolute file path,
		otherwise a file selection dialog is opened.[br]
		The simplest case "dcc.send <nickname> <filename>" will work just as in all
		the other IRC clients, but this command is really more powerful...[br]
		Before attempting to understand the possibilities of this command,
		be sure to know how a [doc:dcc_connection]DCC negotiation and connection[/doc] works.[br]
		The file will be sent as a sequence of packets which must
		be acknowledged one by one by the active client.[br]
		There is a special option (see $option()) called "fast send" (also known
		as "send ahead") that makes KVirc avoid to wait for the acknowledge
		of the last packet before sending the next one.[br]
		Anyway, the connection is declared as succesfull only
		when the whole file (all the packets) has been acknowledged.[br]
		The command supports the following switches:[br]
		[b]-b[/b][br]
		Assume that no acknowledges are sent.
		Assume that the transfer was succesfull when the whole file has been sent,
		then close the socket.[br]
		This is called "blind dcc send".[br]
		[b]-t[/b][br]
		Emulate the TDCC protocol: use the TDCC CTCP message (DCC TSEND) for requesting the connection
		and assume that no acknowledges are sent. Wait for the remote end to close the connection.[br]
		[b]-s[/b][br]
		Use a Secure Sockets Layer for the transfer: the whole communication will be encrypted
		with a private key algorithm after a public key handshake.[br]
		This option will work only if the KVIrc executable has been compiled with SSL support
		and the remote end supports the SSL protocol too.[br]
		Please note that this will may down the transfer somewhat.[br]
		-s can be combined with -t.[br]
		The CTCP negotiation will use SSEND as parameter (or eventually TSSEND).[br]
		When requesting a SSL based dcc send to someone you probably will need a
		certificate. If you don't have one , create it (for example with CA.pl -newcert)
		and set it in the options dialog.
		[b]-i=<interface>[/b][br]
		Bind the local listening socket to the specified <interface> (which is an IP address: IPV4 or IPV6).
		If this switch is NOT specified , the socket is bound to the interface of
		the current irc connection (if any) or to "127.0.0.1".[br]
		You can also specify a local interface name to get the address from (this works only for IPV4 interfaces
		since IPV6 ones seems to be unsupported by the system ioctl() calls at the moment (for linux at least)).[br]
		Here go some examples:[br]
		-i=215.243.12.12: this will bind to the IPV4 interface with the specified address.[br]
		-i=3ffe:1001::1: this will bind to the IPV6 interface with the specified address.[br]
		-i=ppp0: this will bind to the IPV4 address of the interface ppp0 (if supported by the underlying system).[br]
		The -i switch parameter may serve also as a target address when the -c switch is used.[br]
		[b]-p=<port>[/b][br]
		Bind the local listening socket to the specified <port>.
		If this switch is NOT specified, the port will be a "random" one choosen by the kernel.[br]
		[b]-a=<fake address>[/b][br]
		Send the <fake address> as target for the remote client in the requesting CTCP message.
		If this switch is not given , the CTCP will contain the real IP address of the listening
		interface.[br]
		[b]-f=<fake port>[/b][br]
		Send the <fake port> as target port for the remote client in the requesting CTCP message.
		If this switch is not given , the CTCP will contain the real port of the listening socket.
		[br][br]
		All these switches are meant to allow maximum felxibility of the
		dcc negotiation: older KVIrc releases had serious problems
		with firewalled and/or masqueraded machines. With the -a and -f switches
		you can workaround it.[br]
		[br]
		[b]-n[/b][br]
		Do NOT send the CTCP request to the target user: you will have to do it manually,
		or the remote user will have to connect manually (for example by using dcc.recv -c).[br]
		[b]-c[/b][br]
		Attempt to CONNECT to the remote host specified as <interface> and <port>,
		instead of listening (active connection instead of a passive one).
		In this case the -i and -p switches are mandatory.[br]
		The 'c' switch takes precedence over 'n' (In fact both should
		be mutually exclusive).[br]
		If the 'c' and 'n' switches are missing, this commands
		needs to be executed in a window that is bound to a connected
		irc context (you need a third entity to accomplish the negotiation).[br]
		[b]-u[/b][br]
		If the 'u' switch is given , the connection attempt will
		never time out; this might be useful if you want to leave
		a listening socket for a friend of yours while you are sleeping
		and have the CTCP processing disabled. The 'u' switch works either
		in active and passive mode.[br]
		[b]-m[/b][br]
		If the -m switch is passed, the default boolCreateMinimizedDccSend option
		is overridden with the <boolean> parameter passed. So actually
		by passing -m=1 will create a minimized dcc send even
		if the [fnc]$option[/fnc](boolCreateMinimizedDccSend) returns false.[br]
		In the same way, by passing -m=0 you will create a non minimized dcc send.
		If no <boolean> value is specified , it defaults to 1.[br]
		[b]-g[/b][br]
		This switch is a dirty trick: you can use it to receive files from people
		behind a firewall with masquerading enabled.[br]
		It causes the transfer direction to be inverted: this client will receive
		the file from the remote host instead of sending it.[br]
		<file size> is the expected file size in bytes: this parameter can be omitted,
		and in this case the dcc will "blindly" trush the remote end and assume
		that the file has been transferred correctly when the remote end will close the connection.[br]
		If you don't pass the -n option , the remote end will receive an informational DCC RECV request,
		specifying the IP address and the port to connect to.[br]
		-t can be used to prevent sending acknowledges to the remote end , and -u can be used
		to avoid the listening socket to timeout.[br]
		-a and -f can be used as well, but I see no real reason for that...[br]
		here goes an example of usage of this option:[br]
		BAfH can't accept connections (is behind a firewall with masquerading or some other reason...),
		to his machine.[br]
		BAfH wants to send the file important.jpg to Pragma.[br]
		BAfH tells to Pragma that he wants to send him the file and Pragma sets up a connection in the following way:[br]
		[b]dcc.send -g BAfH important.png[/b][br]
		BAfH will see the informational DCC RECV request with the IP address and port that Pragma is listening on.
		Assume that the address was 212.212.231.220 and the port 32344.[br]
		BAfH will then use the following command:[br]
		[b]dcc.send -c -i=212.212.231.220 -p=32344 Pragma /home/bafh/important.jpg[/b][br]
		Et voila!..the file is transferring.[br]
		Pragma will see no file progress indication , since the file size is unknown on Pragma's side.[br]
		If BAfH had specified the file size, Pragma could use -g=<file size> while setting up the connection,
		to be able to see the progress indications.[br]
		If Pragma used the the -n option , the DCC RECV indication wouldn't have been sent: in this case
		Pragma would need to communicate the Ip address and the port "manually" to BAfH.[br]
	@examples:

*/

//#warning "Example for -r"
//#warning "OPTION FOR NO GUI ? (this is hard...)"
//#warning "When in IPV6 mode, should be able to use the IPV4 binding!"


static bool dcc_module_cmd_send(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"dcc_module_cmd_send");

	KviStr target , filename;
	if(!g_pUserParser->parseCmdSingleToken(c,target))return false;
	if(!g_pUserParser->parseCmdFinalPart(c,filename))return false;

	if(target.isEmpty())return c->error(KviError_notEnoughParameters,__tr("Missing target nickname"));

	KviDccBrokerDescriptor * d = new KviDccBrokerDescriptor;

	d->szNick            = target.ptr();    // we always specify the nickname

	if(c->hasSwitch('g'))
	{
		d->szFileName        = filename.ptr();
		d->szFileName.cutToLast('/');
		c->getSwitchValue('g',d->szFileSize);
		if(!d->szFileSize.isUnsignedNum())d->szFileSize = __tr("<unknown size>");
	} else {
		d->szFileName      = filename.ptr();
		d->szLocalFileName = filename.ptr();
	}

	d->szUser            = __tr("unknown"); // username is always unknown
	d->szHost            = d->szUser;       // host is always unknown
	d->pConsole          = c->window()->console();
	d->bRecvFile         = c->hasSwitch('g');
	d->bNoAcks           = c->hasSwitch('b') || c->hasSwitch('t');
	d->bResume           = false;
	d->bAutoAccept       = c->hasSwitch('g');
	d->bIsIncomingAvatar = false;

	if(!dcc_module_parse_default_parameters(d,c))return false;

	if(c->hasSwitch('c'))
	{
		if(!(c->hasSwitch('i') && c->hasSwitch('p')))
		{
			delete d;
			return c->error(KviError_notEnoughParameters,__tr("-c requires -i and -p"));
		}
		d->szIp         = d->szListenIp;
		d->szPort       = d->szListenPort;
		d->szListenIp   = ""; // useless
		d->szListenPort = ""; // useless
		d->bActive      = true;
	} else {
		d->szIp         = __tr("unknown");
		d->szPort       = d->szIp;
		d->bActive      = false;
		d->bSendRequest = !c->hasSwitch('n');
	}

	if(c->hasSwitch('g'))
	{
		dcc_module_set_dcc_type(d,"RECV");
		g_pDccBroker->recvFileManage(d);
	} else {
		dcc_module_set_dcc_type(d,"SEND");
		if(d->szLocalFileName.hasData())
		{
			g_pDccBroker->sendFileExecute(0,d);
		} else {
			g_pDccBroker->sendFileManage(d);
		}
	}

	return c->leaveContext();
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// RECV
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

static void dccModuleParseDccRecv(KviDccRequest * dcc)
{
	// DCC [TS]RECV <filename> <ipaddr> <port> <resume-filesize>
	if(!dcc_module_check_limits(dcc))return;
	if(!dcc_module_check_concurrent_transfers_limit(dcc))return;

	if(!normalizeTargetData(dcc,dcc->szParam2,dcc->szParam3))return;

	if(!(dcc->szParam4.isUnsignedNum()))
	{
		if(!dcc->ctcpMsg->msg->haltOutput())
		{
			dcc->ctcpMsg->msg->console()->outputNoFmt(KVI_OUT_DCCMSG,
				__tr("The above request has resume file size missing: we assume a resume file size of 0"));
		}		
		dcc->szParam4 = __tr("0");
	}

	if(dcc->szParam1.contains('/'))
	{
		if(!dcc->ctcpMsg->msg->haltOutput())
		{
			dcc->ctcpMsg->msg->console()->output(KVI_OUT_DCCMSG,
				__tr("The above request is broken: the filename contains path components; stripping the leading path and trying to continue"),dcc->szParam1.ptr());
		}		
		dcc->szParam1.cutToLast('/');
	}

	KviStr szExtensions = dcc->szType;
	szExtensions.cutRight(4); // cut off RECV

	bool bTurboExtension = szExtensions.contains('T',false);
#ifdef COMPILE_SSL_SUPPORT
	bool bSSLExtension   = szExtensions.contains('S',false);
#else //!COMPILE_SSL_SUPPORT
	if(szExtensions.contains('S',false))
	{
		dcc_module_request_error(dcc,__tr("This executable has been compiled without SSL support: the SSL extension to DCC RECV is not available"));
		return;
	}
#endif //!COMPILE_SSL_SUPPORT

	// If we have a file offer for this...do it automatically
	KviFileOffer * o = g_pFileTrader->lookupOffer(dcc->szParam1.ptr(),dcc->ctcpMsg->pSource,0);
	if(o)
	{

		unsigned int uResumeSize = dcc->szParam4.toUInt(); // this will NEVER fail
		if(uResumeSize >= o->fileSize())
		{
			// senseless request
			KviStr szError(KviStr::Format,
					__tr("invalid recv request: position %u is 'out of the file'"),uResumeSize);
			dcc_module_request_error(dcc,szError.ptr());
			return;
		}

		// ok...we have requested this send
//		#warning "Maybe remove this file offer now ?"
		KviDccBrokerDescriptor * d = new KviDccBrokerDescriptor;

		d->szNick            = dcc->ctcpMsg->pSource->nick();
		d->szUser            = dcc->ctcpMsg->pSource->user();
		d->szHost            = dcc->ctcpMsg->pSource->host();

		d->szFileName        = dcc->szParam1.ptr();
		d->szFileSize        = dcc->szParam4.ptr();
  
		//d->bResume           = false; // This is actually useless

		d->szLocalFileName   = o->absFilePath();
		d->szLocalFileSize.setNum(o->fileSize()); // Should we look it up again ?


		d->pConsole          = dcc->pConsole;

		d->bRecvFile         = false;
		d->bNoAcks           = bTurboExtension;

		d->bAutoAccept       = true;
		d->bIsIncomingAvatar = false;

		d->bIsTdcc           = bTurboExtension;
#ifdef COMPILE_SSL_SUPPORT
		d->bIsSSL            = bSSLExtension;
#endif

		d->bOverrideMinimize = false;

		// We know everything
		d->szLocalNick       = d->pConsole->currentNickName();
		d->szLocalUser       = d->pConsole->currentUserName();
		d->szLocalHost       = d->pConsole->currentLocalHostName();
		d->bDoTimeout        = true;

		d->szIp              = dcc->szParam2.ptr();
		d->szPort            = dcc->szParam3.ptr();

		d->bActive           = true;
		dcc_module_set_dcc_type(d,"SEND");
		g_pDccBroker->sendFileExecute(0,d);

		return;

	} else {

		dcc->ctcpMsg->msg->console()->output(KVI_OUT_DCCMSG,
			__tr("%s [%s@%s] is ready to receive the file \"%s\""),
			dcc->ctcpMsg->pSource->nick(),dcc->ctcpMsg->pSource->username(),dcc->ctcpMsg->pSource->host(),
			dcc->szParam1.ptr());
		dcc->ctcpMsg->msg->console()->output(KVI_OUT_DCCMSG,
			__tr("The remote client is listening on interface %s and port %s"),dcc->szParam2.ptr(),dcc->szParam3.ptr());
		KviStr szSwitches = "-c";
		if(bTurboExtension)szSwitches.prepend("-t ");
#ifdef COMPILE_SSL_SUPPORT
		if(bSSLExtension)szSwitches.prepend("-s ");
#endif
		dcc->ctcpMsg->msg->console()->output(KVI_OUT_DCCMSG,
			__tr("Use %c\r![!dbl]dcc.send %s -i=%s -p=%s %s\r/dcc.send %s -i=%s -p=%s %s\r%c to send the file (or simply double click on the link)"),
			KVI_TEXT_BOLD,
			szSwitches.ptr(),
			dcc->szParam2.ptr(),dcc->szParam3.ptr(),dcc->ctcpMsg->pSource->nick(),
			szSwitches.ptr(),
			dcc->szParam2.ptr(),dcc->szParam3.ptr(),dcc->ctcpMsg->pSource->nick(),
			KVI_TEXT_BOLD);
	}
}

/*
	@doc: dcc.recv
	@type:
		command
	@title:
		dcc.recv
	@short:
		Sets up a file receiving connection
	@syntax:
		dcc.recv [-s] [-t] [-u] [-b] [-n] [-c] [-i=<interface>] [-p=<port>] [-m[=<boolean>]] <nickname> <filename> <remote file size>
	@description:
		Sets up a connection ready to receive a file.[br]
		In most 'common' cases you will not need this command: you might want to use [cmd]dcc.get[/cmd] instead.[br]
		This works like dcc.send but has two main differences: the file is INCOMING, and the CTCP sent to the other side
		is a CTCP RECV.[br]
		This command is the counterpart of [cmd]dcc.send[/cmd] and its parameters are exactly the same, so please refer to that
		help page for the full discussion. This help page contains only a brief resume of these parameters.[br]
		The [doc:dcc_connection]dcc documentation[/doc] explains the DCC Recv subprotocol in detail.[br]
		Parameters:[br]
		-t: Assume the transfer will be a TDcc (send a TRECV instead of a RECV and use no acks).[br]
		-s: Assume the transfer will be based on the Secure Sockets Layer (send a SRECV or a TSRECV instead of a RECV or a TRECV)
		-m=<boolean>: Force window minimize if boolean evaluates to 1[br]
		-b: Blind transfer (no acks)[br]
		-n: send no CTCP request[br]
		-u: never timeout[br]
		-i=<interface>: listen on this interface (interface may be an IP address or a local interface name)[br]
		-p=<port>: listen on this port[br]
		-a=<fake ip>: send this fake IP address in the request[br]
		-f=<fake port>: send this fake port in the request[br]
		-c: accepted for compatibility (don't use it :)
	@examples:

*/

//#warning "ENCODE THE CTCP'S!!!!!!!"
//#warning "DOCS FOR dcc.recv (examples!)"

static bool dcc_module_cmd_recv(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"dcc_module_cmd_recv");

	KviStr target , filename , filesize;
	if(!g_pUserParser->parseCmdSingleToken(c,target))return false;
	if(!g_pUserParser->parseCmdSingleToken(c,filename))return false;
	if(!g_pUserParser->parseCmdFinalPart(c,filesize))return false;

	if(target.isEmpty())return c->error(KviError_notEnoughParameters,__tr("Missing target nickname"));
	if(filename.isEmpty())return c->error(KviError_notEnoughParameters,__tr("Missing filename"));
	if(!filesize.isUnsignedNum())return c->error(KviError_invalidParameter,__tr("The file size is not an unsigned number"));

	KviDccBrokerDescriptor * d = new KviDccBrokerDescriptor;
	d->szNick            = target.ptr();
	d->szUser            = __tr("unknown");
	d->szHost            = d->szUser;
	d->szIp              = __tr("unknown");
	d->szPort            = d->szIp;
	d->pConsole          = c->window()->console();

	// -c is senseless here...but we accept it for compatibility

	if(!dcc_module_parse_default_parameters(d,c))return false;

	d->szFileName        = filename;
	d->szFileSize        = filesize;

	d->bActive           = false; // we have to listen!
	d->bResume           = false; 
	d->bRecvFile         = true;  // we have to receive the file!
	d->bSendRequest      = !c->hasSwitch('n');
	d->bNoAcks           = d->bIsTdcc || c->hasSwitch('b');
	d->bAutoAccept       = KVI_OPTION_BOOL(KviOption_boolAutoAcceptDccSend);
	d->bIsIncomingAvatar = g_pApp->findPendingAvatarChange(d->pConsole,d->szNick.ptr(),d->szFileName.ptr());

	if(KVI_OPTION_BOOL(KviOption_boolAutoAcceptIncomingAvatars))d->bAutoAccept = d->bAutoAccept || d->bIsIncomingAvatar;
	dcc_module_set_dcc_type(d,"RECV");
	g_pDccBroker->recvFileManage(d);

	return c->leaveContext();
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// RSEND
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

static void dccModuleParseDccRSend(KviDccRequest *dcc)
{
	// DCC RSEND <filename> <filesize>
//#warning "Ignore files depending on file type ? (MediaType ?)"
	//
	// We have received a DCC RSEND request in the following form
	//
	//      DCC [ST]RSEND <filename> <filesize>
	//

	if(!dcc_module_check_limits(dcc))return;
	if(!dcc_module_check_concurrent_transfers_limit(dcc))return;

	if(!(dcc->szParam2.isUnsignedNum()))
	{
		if(!dcc->ctcpMsg->msg->haltOutput())
		{
			dcc->ctcpMsg->msg->console()->output(KVI_OUT_DCCMSG,
				__tr("The above request is broken: the fourth parameter should be the file size but does not look as an unsigned number; trying to continue"),dcc->szParam2.ptr());
		}		
		dcc->szParam2 = __tr("<unknown size>");
	}

	if(dcc->szParam1.contains('/'))
	{
		if(!dcc->ctcpMsg->msg->haltOutput())
		{
			dcc->ctcpMsg->msg->console()->output(KVI_OUT_DCCMSG,
				__tr("The above request is broken: the filename contains path components; stripping the leading path and trying to continue"),dcc->szParam1.ptr());
		}		
		dcc->szParam1.cutToLast('/');
	}

	KviStr szExtensions = dcc->szType;
	szExtensions.cutRight(4); // cut off SEND

	bool bTurboExtension = szExtensions.contains('T',false);
#ifdef COMPILE_SSL_SUPPORT
	bool bSSLExtension   = szExtensions.contains('S',false);
#else //!COMPILE_SSL_SUPPORT
	if(szExtensions.contains('S',false))
	{
		dcc_module_request_error(dcc,__tr("This executable has been compiled without SSL support: the SSL extension to DCC RSEND is not available"));
		return;
	}
#endif //!COMPILE_SSL_SUPPORT

//#warning "When behind a firewall, we should reply an error message and avoid setting up the listening connection"

	KviDccBrokerDescriptor * d = new KviDccBrokerDescriptor;
	d->szNick            = dcc->ctcpMsg->pSource->nick();
	d->szUser            = dcc->ctcpMsg->pSource->username();
	d->szHost            = dcc->ctcpMsg->pSource->host();
	d->szIp              = __tr("unknown");
	d->szPort            = d->szIp;
	d->pConsole          = dcc->pConsole;

	if(!dcc_module_get_listen_ip_address(0,d->pConsole,d->szListenIp))
	{
		d->pConsole->output(KVI_OUT_DCCMSG,
			__tr("No suitable interface to listen on: Trying to continue anyway... but don't expect nice results"));
		d->szListenIp = "0.0.0.0";
	}

	d->szListenPort      = "0";
	d->szLocalNick       = dcc->pConsole->currentNickName();
	d->szLocalUser       = dcc->pConsole->currentUserName();
	d->szLocalHost       = dcc->pConsole->currentLocalHostName();
	d->szFileName        = dcc->szParam1.ptr();
	d->szFileSize        = dcc->szParam2.ptr();
	d->bActive           = false; // we have to listen!
	d->bResume           = false; 
	d->bRecvFile         = true;  // we have to receive the file!

#ifdef COMPILE_SSL_SUPPORT
	d->bIsSSL            = bSSLExtension;
#endif
	d->bIsTdcc           = bTurboExtension;
	d->bSendRequest      = true; // we have to send the [ST]RECV request back
	d->bNoAcks           = d->bIsTdcc;
	d->bOverrideMinimize = false;
	d->bAutoAccept       = KVI_OPTION_BOOL(KviOption_boolAutoAcceptDccSend);
	d->bIsIncomingAvatar = g_pApp->findPendingAvatarChange(dcc->pConsole,d->szNick.ptr(),d->szFileName.ptr());

	if(KVI_OPTION_BOOL(KviOption_boolDccSendFakeAddressByDefault))
	{
		d->szFakeIp = KVI_OPTION_STRING(KviOption_stringDefaultDccFakeAddress);
		if(d->szFakeIp.isEmpty())KVI_OPTION_BOOL(KviOption_boolDccSendFakeAddressByDefault) = false;
	}

	if(KVI_OPTION_BOOL(KviOption_boolAutoAcceptIncomingAvatars))d->bAutoAccept = d->bAutoAccept || d->bIsIncomingAvatar;

	dcc_module_set_dcc_type(d,"RECV");
	g_pDccBroker->recvFileManage(d);
}

/*
	@doc: dcc.rsend
	@type:
		command
	@title:
		dcc.rsend
	@short:
		Sends a file by using the reverse DCC send protocol
	@syntax:
		dcc.rsend [-s] [-t] <nickname> [filename]
	@description:
		Sends a DCC RSEND request to <nickname> notifying him that you want to
		send him the file [filename].[br]
		The remote end may acknowledge the request by sending a DCC RECV request.
		This command effects are similar to [cmd]dcc.send[/cmd] , but will work also on machines
		that can't accept incoming connections (firewalling or masquerading problems).[br]
		A 120 seconds file offer is added for the specified file and mask "<nickname>!*@*".
	@examples:

*/

//#warning "ENCODE THE CTCP'S!!!!!!!"
//#warning "DOCS FOR dcc.rsend"

static bool dcc_module_cmd_rsend(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"dcc_module_cmd_rsend");

	KviStr target , filename;
	if(!g_pUserParser->parseCmdSingleToken(c,target))return false;
	if(!g_pUserParser->parseCmdFinalPart(c,filename))return false;

	if(target.isEmpty())return c->error(KviError_notEnoughParameters,"%s",__tr("Missing target nickname"));

	if(!c->window()->console())return c->noIrcContext();
	if(!c->window()->console()->isConnected())return c->notConnectedToServer();

	KviDccBrokerDescriptor * d = new KviDccBrokerDescriptor;
	d->szNick            = target;
	d->szLocalFileName   = filename;
	d->pConsole          = c->window()->console();
	d->bIsTdcc           = c->hasSwitch('t');
#ifdef COMPILE_SSL_SUPPORT
	d->bIsSSL            = c->hasSwitch('s');
#else
	if(c->hasSwitch('s'))c->warning(__tr("This executable has been compiled without SSL support: -s switch ignored"));
#endif //!COMPILE_SSL_SUPPORT

	dcc_module_set_dcc_type(d,"RSEND");
	g_pDccBroker->rsendManage(d);

	return c->leaveContext();
}




////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// GET
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

static void dccModuleParseDccGet(KviDccRequest *dcc)
{
	// DCC [TS]GET <filename> [filesize]
	// -> DCC [TS]SEND <filename> <ipaddr> <port> <filesize>
	// ...

	bool bOk;
	unsigned int uSize = dcc->szParam2.toUInt(&bOk);
	if(!bOk)uSize = 0;

	if(!dcc_module_check_limits(dcc))return;
	if(!dcc_module_check_concurrent_transfers_limit(dcc))return;

	KviStr szExtensions = dcc->szType;
	szExtensions.cutRight(3); // cut off GET

	bool bTurboExtension = szExtensions.contains('T',false);
#ifdef COMPILE_SSL_SUPPORT
	bool bSSLExtension   = szExtensions.contains('S',false);
#else //!COMPILE_SSL_SUPPORT
	if(szExtensions.contains('S',false))
	{
		dcc_module_request_error(dcc,__tr("This executable has been compiled without SSL support: the SSL extension to DCC GET is not available"));
		return;
	}
#endif //!COMPILE_SSL_SUPPORT

	KviFileOffer * o = g_pFileTrader->lookupOffer(dcc->szParam1.ptr(),dcc->ctcpMsg->pSource,uSize);
	if(!o)
	{
		if(!dcc->ctcpMsg->msg->haltOutput())
		{
			KviStr szError(KviStr::Format,
					__tr("no file offer named '%s' (with size %s) available for %s [%s@%s]"),
					dcc->szParam1.ptr(),uSize > 0 ? dcc->szParam2.ptr() : __tr("\"any\""),dcc->ctcpMsg->pSource->nick(),
					dcc->ctcpMsg->pSource->username(),
					dcc->ctcpMsg->pSource->host());
			dcc_module_request_error(dcc,szError.ptr());
		}
		return;
	}

//#warning "IF NOT IGNORE DCC GET!"

//#warning "CREATE IT MINIMIZED ETC..."
//#warning "MAYBE USE A DIALOG TO ACCEPT THE REQUEST ?"
//#warning "DO NOT ACCEPT /etc/* requests..."

	if(KVI_OPTION_BOOL(KviOption_boolCantAcceptIncomingDccConnections))
	{
		// we have to use DCC RSEND , otherwise it will not work
		KviStr szSubproto("RSEND");
		szSubproto.prepend(szExtensions);

		if(!dcc->ctcpMsg->msg->haltOutput())
		{
			dcc->ctcpMsg->msg->console()->output(KVI_OUT_DCCMSG,
				__tr("Accepting file request from %s [%s@%s] for '%s' (real file: %s): offering DCC %s since we can't accept incoming connections (user option)"),
				dcc->ctcpMsg->pSource->nick(),dcc->ctcpMsg->pSource->username(),
				dcc->ctcpMsg->pSource->host(),dcc->szParam1.ptr(),
				o->absFilePath(),szSubproto.ptr());
		}

		dcc->pConsole->socket()->sendFmtData("PRIVMSG %s :%cDCC %s %s %u%c",dcc->ctcpMsg->pSource->nick(),
			0x01,szSubproto.ptr(),dcc->szParam1.ptr(),o->fileSize(),0x01);
		return;
	}


	KviDccBrokerDescriptor * d = new KviDccBrokerDescriptor;
	d->szNick            = dcc->ctcpMsg->pSource->nick();
	d->szLocalFileName   = o->absFilePath();
	d->szUser            = dcc->ctcpMsg->pSource->username();
	d->szHost            = dcc->ctcpMsg->pSource->host();
	d->pConsole          = dcc->pConsole;
	d->bRecvFile         = false;
	d->szLocalNick       = dcc->pConsole->currentNickName();
	d->szLocalUser       = dcc->pConsole->currentUserName();
	d->szLocalHost       = dcc->pConsole->currentLocalHostName();

	if(!dcc_module_get_listen_ip_address(0,d->pConsole,d->szListenIp))
	{
		d->pConsole->output(KVI_OUT_DCCMSG,
			__tr("No suitable interface to listen on: Trying to continue anyway... but don't expect nice results"));
		d->szListenIp = "0.0.0.0";
	}

//#warning "DO STH WITH THIS PORT (HOW TO SPECIFY IT ?)"
	d->szListenPort      = "0"; // any port is ok

	if(KVI_OPTION_BOOL(KviOption_boolDccSendFakeAddressByDefault))
	{
		d->szFakeIp = KVI_OPTION_STRING(KviOption_stringDefaultDccFakeAddress);
		if(d->szFakeIp.isEmpty())KVI_OPTION_BOOL(KviOption_boolDccSendFakeAddressByDefault) = false;
	}

	d->bDoTimeout        = true;
	d->szIp              = __tr("unknown");
	d->szPort            = d->szIp;
	d->bActive           = false;
	d->bSendRequest      = true;
	d->bIsTdcc           = bTurboExtension;
#ifdef COMPILE_SSL_SUPPORT
	d->bIsSSL            = bSSLExtension;
#endif
	d->bNoAcks           = d->bIsTdcc;
	d->bOverrideMinimize = false;

	dcc_module_set_dcc_type(d,"SEND");

	if(!dcc->ctcpMsg->msg->haltOutput())
	{
		dcc->ctcpMsg->msg->console()->output(KVI_OUT_DCCMSG,
			__tr("Accepting file request from %s [%s@%s] for '%s' (real file: %s): offering DCC %s"),
			dcc->ctcpMsg->pSource->nick(),dcc->ctcpMsg->pSource->username(),
			dcc->ctcpMsg->pSource->host(),dcc->szParam1.ptr(),
			o->absFilePath(),d->szType.ptr());
	}
	g_pDccBroker->sendFileExecute(0,d);
}

/*
	@doc: dcc.get
	@type:
		command
	@title:
		dcc.get
	@short:
		Requests a file
	@syntax:
		dcc.get [-s] [-t] <nickname> <filename> [filesize]
	@description:
		Sends a CTCP DCC GET to <nickname> requesting the file <filename>.
		The remote end should reply with a DCC SEND request CTCP.
		<filename> must not contain any leading path.
		If the -t switch is given , the message is a DCC TGET, expecting
		a TSEND reply.[br]
		If the -s switch is given , the message will be a DCC SGET, expecting
		a SSEND reply.[br]
		-t and -s can be combined together to obtain a "turbo"+"SSL" extension transfer.[br]
		-s will work only if the KVIrc executable has been compiled with SSL support and
		the remote client supports it.[br]
	@examples:

*/

//#warning "ENCODE THE CTCP'S!!!!!!!"
//#warning "DOCS FOR dcc.get"


static bool dcc_module_cmd_get(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"dcc_module_cmd_get");

	KviStr target , filename ,size;
	if(!g_pUserParser->parseCmdSingleToken(c,target))return false;
	if(!g_pUserParser->parseCmdSingleToken(c,filename))return false;
	if(!g_pUserParser->parseCmdFinalPart(c,size))return false;

	if(size.hasData() && !size.isUnsignedNum())
	{
		c->warning(__tr("Invalid file size specified: no message sent"));
		return c->leaveContext();
	}

	filename.cutToLast('/');

	if(target.isEmpty())return c->error(KviError_notEnoughParameters,"%s",__tr("Missing target nickname"));
	if(filename.isEmpty())return c->error(KviError_notEnoughParameters,"%s",__tr("Missing filename"));

	if(!c->window()->console())return c->noIrcContext();
	if(!c->window()->console()->isConnected())return c->notConnectedToServer();

	if(filename.contains(' '))
	{
		filename.prepend('"');
		filename.append('"');
	}

	KviStr szDCC("GET");
#ifdef COMPILE_SSL_SUPPORT
	if(c->hasSwitch('s'))szDCC.prepend('S');
#else
	if(c->hasSwitch('s'))c->warning(__tr("this executable has no SSL support: -s switch ignored"));
#endif
	if(c->hasSwitch('t'))szDCC.prepend('T');

	if(size.hasData())
	{
		c->window()->console()->socket()->sendFmtData("PRIVMSG %s :%cDCC %s %s%c",
			target.ptr(),0x01,szDCC.ptr(),filename.ptr(),0x01);
	} else {
		c->window()->console()->socket()->sendFmtData("PRIVMSG %s :%cDCC %s %s %s%c",
			target.ptr(),0x01,szDCC.ptr(),filename.ptr(),size.ptr(),0x01);
	}

	return c->leaveContext();
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// VOICE
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

static void dccModuleParseDccVoice(KviDccRequest *dcc)
{
	//
	// We have received a DCC VOICE request in the following form:
	//
	//     DCC VOICE codec <ipaddress> <port> <sample-rate>
	//
	// This means that we're requested to setup an ACTIVE voice connection
	// ... Easy task :)
	//

	if(!dcc_module_check_limits(dcc))return;

	if(!normalizeTargetData(dcc,dcc->szParam2,dcc->szParam3))return;

	//  Actually unused parameter
	if(!kvi_dcc_voice_is_valid_codec(dcc->szParam1.ptr()))
	{
		if(!dcc->ctcpMsg->msg->haltOutput())
		{
			dcc->ctcpMsg->msg->console()->output(KVI_OUT_DCCERROR,
				__tr("The above request can't be accepted: unsupported codec '%s'"),dcc->szParam1.ptr());
			return;
		}
	}

	bool bOk;

	int iSampleRate = dcc->szParam4.toInt(&bOk);
	if(!bOk)
	{
		if(!dcc->ctcpMsg->msg->haltOutput())
		{
			dcc->ctcpMsg->msg->console()->output(KVI_OUT_DCCMSG,
				__tr("The above request looks to be broken: invalid sample-rate '%s': defaulting to 8000"),dcc->szParam4.ptr());
		}
		iSampleRate = 8000;
	}


	KviDccBrokerDescriptor * d = new KviDccBrokerDescriptor;
	d->szNick            = dcc->ctcpMsg->pSource->nick();
	d->szUser            = dcc->ctcpMsg->pSource->username();
	d->szHost            = dcc->ctcpMsg->pSource->host();
	d->szLocalNick       = dcc->pConsole->currentNickName();
	d->szLocalUser       = dcc->pConsole->currentUserName();
	d->szLocalHost       = dcc->pConsole->currentLocalHostName();
	d->szIp              = dcc->szParam2.ptr();
	d->szPort            = dcc->szParam3.ptr();
	d->pConsole          = dcc->pConsole;
	d->bActive           = true; // we have to connect
	d->bIsTdcc           = false;
	d->bNoAcks           = false; // this has no meaning in voice
	d->szCodec           = dcc->szParam1;
	d->iSampleRate       = iSampleRate;
	d->bOverrideMinimize = false;
	d->bAutoAccept       = KVI_OPTION_BOOL(KviOption_boolAutoAcceptDccVoice);
	dcc_module_set_dcc_type(d,"VOICE");
	g_pDccBroker->activeVoiceManage(d);
}

// FIXME: SSL support for DCC VOICE ? (has a sense ?)

/*
	@doc: dcc.voice
	@type:
		command
	@title:
		dcc.voice
	@short:
		Starts a DCC VOICE connection
	@syntax:
		dcc.voice [-g=<codec>] [-n] [-c] [-u] [-h=<sample_rate_in_hz>] [-m[=<boolean>]] [-i=<interface>] [-p=<port>] [-a=<fake address>] [-f=<fake port>] <nickname>
	@description:
		Attempts a dcc VOICE connection to <nickname>.[br]
		The -g option is used to select the GSM codec: available codecs are "gsm" , "adpcm" and "null".[br]
		The adpcm codec is the one that was used in previous KVIrc versions: it provides a 1:4 compression rate
		and is designed for 8 KHz audio sampling rate (but will work also with other sampling rates).[br]
		The gsm codec offers 1:10 compression at the cost of some quality and cpu time. If you have a good
		cpu and plan to transmit voice only , use this codec.<br> The null codec
		offers no compression and may be used to transfer plain audio data over a fast connection (usually loopback
		connection or local networks). The null codec with 44100 sampling rate would provide CD quality audio
		streaming , but it is practically not usable (at the time I'm writing) since requires a
		monster bandwidth. If the -g switch is not present, the adpcm codec is used by default (for compatibility reasons).[br]
 		The -h switch is used to select the sampling rate , if not given the sampling rate defaults to 8000 Hz.
		This switch accepts any value , but in fact the soundcards have limits on the values. Typically
		the lowest limit is 5 KHz and the upper limit is 44.1 KHz (but some soundcards support 96 KHz).
		It is also possible that the soundcard can't support a continous range of frequencies and
		will select a discrete closest match instead.[br]
		The "commonly used" sample rates are 8000 , 11025 , 22050 and 44100 Hz.[br]
		The remaining parameters are equivalent to the ones used in [cmd]dcc.send[/cmd], so please refer to that
		help page for the full discussion. This help page contains only a brief resume of these parameters.[br]
		Parameters:[br]
		-t: ignored.[br]
		-m=<boolean>: Force window minimize if boolean evaluates to 1[br]
		-n: send no CTCP request, just setup a listening socket[br]
		-u: never timeout[br]
		-i=<interface>: listen on this interface (interface may be an IP address or a local interface name)[br]
		-p=<port>: listen on this port[br]
		-a=<fake ip>: send this fake IP address in the request[br]
		-f=<fake port>: send this fake port in the request[br]
		-c: connect to the interface and port specified by the -i and -p switches (instead of sending a request and listening)
	@examples:
		[example]
			[comment]# Setup a DCC VOICE connection with Pragma (irc user)[/comment]
			dcc.voice Pragma
			[comment]# Setup a DCC VOICE connection with Pragma and use the gsm codec[/comment]
			dcc.voice -g=gsm Pragma
			[comment]# Setup a DCC VOICE connection with Pragma , use the gsm codec and 22050 Hz sampling rate[/comment]
			dcc.voice -g=gsm -h=22050 Pragma
			[comment]# Setup a listening socket for a DCC VOICE connection on interface 127.0.0.1 and port 8088[/comment]
			dcc.voice -n -i=127.0.0.1 -p=8088 Pippo
			[comment]# Connect to the socket above[/comment]
			dcc.voice -c -i=127.0.0.1 -p=8088 Pluto
			[comment]# Same as above but use the NULL codec with 11025 Hz sampling rate[/comment]
			dcc.voice -g=null -h=11025 -n -i=127.0.0.1 -p=8088 Pippo
			[comment]# Connect to the socket above[/comment]
			dcc.voice -g=null -h=11025 -c -i=127.0.0.1 -p=8088 Pluto
		[/example]
*/

static bool dcc_module_cmd_voice(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"dcc_module_cmd_voice");

	KviStr target;
	if(!g_pUserParser->parseCmdFinalPart(c,target))return false;

	if(target.isEmpty())return c->error(KviError_notEnoughParameters,__tr("Missing target nickname"));

	KviDccBrokerDescriptor * d = new KviDccBrokerDescriptor;

	d->szNick       = target.ptr();    // we always specify the nickname
	d->szUser       = __tr("unknown"); // username is always unknown
	d->szHost       = d->szUser;       // host is always unknown
	d->pConsole     = c->window()->console();
	d->iSampleRate  = 8000;

	if(!dcc_module_parse_default_parameters(d,c))return false;

	if(c->hasSwitch('h'))
	{
		KviStr szSampleRate;
		c->getSwitchValue('h',szSampleRate);
		bool bOk;
		d->iSampleRate = szSampleRate.toInt(&bOk);
		if(!bOk)
		{
			c->warning(__tr("Invalid sample rate '%s': defaulting to 8000"),szSampleRate.ptr());
			d->iSampleRate = 8000;
		}
	}

	d->szCodec = "adpcm";

	if(c->hasSwitch('g'))
	{
		c->getSwitchValue('g',d->szCodec);

		if(!kvi_dcc_voice_is_valid_codec(d->szCodec.ptr()))
		{
			c->warning(__tr("Invalid codec '%s': defaulting to 'adpcm'"),d->szCodec.ptr());
			d->szCodec = "adpcm";
		}
	}

	dcc_module_set_dcc_type(d,"VOICE");
	if(c->hasSwitch('c'))
	{
		if(!(c->hasSwitch('i') && c->hasSwitch('p')))
		{
			delete d;
			return c->error(KviError_notEnoughParameters,__tr("-c requires -i and -p"));
		}
		d->szIp         = d->szListenIp;
		d->szPort       = d->szListenPort;
		d->szListenIp   = ""; // useless
		d->szListenPort = ""; // useless
		d->bActive      = true;

		g_pDccBroker->activeVoiceExecute(0,d);
	} else {
		d->szIp         = __tr("unknown");
		d->szPort       = d->szIp;
		d->bActive      = false;
		d->bSendRequest = !c->hasSwitch('n');

		g_pDccBroker->passiveVoiceExecute(d);
	}

	return c->leaveContext();
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// CANVAS
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

static void dccModuleParseDccCanvas(KviDccRequest *dcc)
{
	//
	// We have received a DCC CANVAS request in the following form:
	//
	//     DCC CANVAS unused <ipaddress> <port>
	//
	// This means that we're requested to setup an ACTIVE canvas connection
	// ... Easy task :)
	//
	if(!dcc_module_check_limits(dcc))return;

	if(!normalizeTargetData(dcc,dcc->szParam2,dcc->szParam3))return;

//  Actually unused parameter
//	if(!(kvi_strEqualCI("canvas",dcc->szParam1.ptr())))
//	{
//		if(!dcc->ctcpMsg->msg->haltOutput())
//		{
//			dcc->ctcpMsg->msg->console()->output(KVI_OUT_DCCMSG,
//				__tr("The above request is broken: the second parameter is '%s' and shoud be 'chat'; trying to continue"),dcc->szParam1.ptr());
//		}
//	}

	KviDccBrokerDescriptor * d = new KviDccBrokerDescriptor;
	d->szNick            = dcc->ctcpMsg->pSource->nick();
	d->szUser            = dcc->ctcpMsg->pSource->username();
	d->szHost            = dcc->ctcpMsg->pSource->host();
	d->szLocalNick       = dcc->pConsole->currentNickName();
	d->szLocalUser       = dcc->pConsole->currentUserName();
	d->szLocalHost       = dcc->pConsole->currentLocalHostName();
	d->szIp              = dcc->szParam2.ptr();
	d->szPort            = dcc->szParam3.ptr();
	d->pConsole          = dcc->pConsole;
	d->bActive           = true; // we have to connect
	d->bIsTdcc           = false;
	d->bNoAcks           = false; // this has no meaning in canvas
	d->bOverrideMinimize = false;
	d->bAutoAccept       = KVI_OPTION_BOOL(KviOption_boolAutoAcceptDccCanvas);
	dcc_module_set_dcc_type(d,"CANVAS");
	g_pDccBroker->activeCanvasManage(d);
}

/*
	@doc: dcc.canvas
	@type:
		command
	@title:
		dcc.canvas
	@short:
		Starts a DCC CHAT connection
	@syntax:
		dcc.canvas [-n] [-c] [-u] [-m[=<boolean>]] [-i=<interface>] [-p=<port>] [-a=<fake address>] [-f=<fake port>] <nickname>
	@description:
		Attempts a dcc canvas connection to <nickname>.[br]
	@examples:
*/

static bool dcc_module_cmd_canvas(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"dcc_module_cmd_canvas");

	KviStr target;
	if(!g_pUserParser->parseCmdFinalPart(c,target))return false;

	if(target.isEmpty())return c->error(KviError_notEnoughParameters,"%s",__tr("Missing target nickname"));

	KviDccBrokerDescriptor * d = new KviDccBrokerDescriptor;

	d->szNick       = target.ptr();    // we always specify the nickname
	d->szUser       = __tr("unknown"); // username is always unknown
	d->szHost       = d->szUser;       // host is always unknown
	d->pConsole     = c->window()->console();

/*
	d->bIsTdcc = c->hasSwitch('t');

	d->bOverrideMinimize = c->hasSwitch('m');

	if(d->bOverrideMinimize)
	{
		KviStr tmpVal;
		if(!(c->getSwitchValue('m',tmpVal)))d->bShowMinimized = false;
		else d->bShowMinimized = kvi_strEqualCI(tmpVal.ptr(),"1");
	}


	if(!d->pConsole)
	{
		// We don't need a console with -c and -n , otherwise we need it
		if(!(c->hasSwitch('c') || c->hasSwitch('n')))return c->noIrcContext();
		else d->pConsole = c->window()->frame()->firstConsole();
	}

	__range_valid(d->pConsole);

	if(!d->pConsole->isConnected())
	{
		// We don't need a connection with -c and -n , otherwise we need it
		if(!(c->hasSwitch('c') || c->hasSwitch('n')))return c->notConnectedToServer();
		else {
			// -c or -n , grab a local nick from somewhere
			d->szLocalNick  = KVI_OPTION_STRING(KviOption_stringNickname1);
			d->szLocalNick.stripWhiteSpace();
			if(d->szLocalNick.isEmpty())d->szLocalNick = KVI_DEFAULT_NICKNAME1;
			d->szLocalUser  = __tr("unknown"); // we can live without it
			d->szLocalHost  = d->szLocalUser;  // we can live without it
		}
	} else {
		// We know everything
		d->szLocalNick  = d->pConsole->currentNickName();
		d->szLocalUser  = d->pConsole->currentUserName();
		d->szLocalHost  = d->pConsole->currentLocalHostName();
	}


	if(c->hasSwitch('i'))
	{
		c->getSwitchValue('i',d->szListenIp);
		if(!(d->szListenIp.contains('.') || d->szListenIp.contains(':')))
		{
			// This will magically work with the same buffer as source and dst!
			if(!kvi_getInterfaceAddress(d->szListenIp.ptr(),d->szListenIp))
			{
				return c->error(KviError_invalidParameter,__tr("Can't get address of interface %s"),d->szListenIp.ptr());
			}
		}
	} else {
		if(d->pConsole->isConnected())
		{
			d->pConsole->socket()->getLocalHostIp(d->szListenIp,d->pConsole->isIpV6Connection());
		} else d->szListenIp = "127.0.0.1"; // huh ? :)
	}

	if(c->hasSwitch('p'))c->getSwitchValue('p',d->szListenPort);
	else d->szListenPort = "0"; // any port is ok

	if(c->hasSwitch('a'))c->getSwitchValue('a',d->szFakeIp);

	if(c->hasSwitch('f'))c->getSwitchValue('f',d->szFakePort);

	d->bDoTimeout = (!c->hasSwitch('u'));

*/
	if(!dcc_module_parse_default_parameters(d,c))return false;
	dcc_module_set_dcc_type(d,"VOICE");

	if(c->hasSwitch('c'))
	{
		if(!(c->hasSwitch('i') && c->hasSwitch('p')))
		{
			delete d;
			return c->error(KviError_notEnoughParameters,__tr("-c requires -i and -p"));
		}
		d->szIp         = d->szListenIp;
		d->szPort       = d->szListenPort;
		d->szListenIp   = ""; // useless
		d->szListenPort = ""; // useless
		d->bActive      = true;

		g_pDccBroker->activeCanvasExecute(0,d);
	} else {
		d->szIp         = __tr("unknown");
		d->szPort       = d->szIp;
		d->bActive      = false;
		d->bSendRequest = !c->hasSwitch('n');

		g_pDccBroker->passiveCanvasExecute(d);
	}

	return c->leaveContext();
}


static void dccModuleParseDccList(KviDccRequest *dcc)
{
	// DCC LIST <mask> <ipaddr> <port>
	// FIXME!
}


typedef void (*dccParseProc)(KviDccRequest *);
typedef struct _dccParseProcEntry
{
	const char * type;
	dccParseProc proc;
} dccParseProcEntry;

#define KVI_NUM_KNOWN_DCC_TYPES 27

static dccParseProcEntry dccParseProcTable[KVI_NUM_KNOWN_DCC_TYPES]=
{
	{ "CHAT"   , dccModuleParseDccChat   },
	{ "SCHAT"  , dccModuleParseDccChat   },
	{ "SEND"   , dccModuleParseDccSend   },
	{ "TSEND"  , dccModuleParseDccSend   },
	{ "SSEND"  , dccModuleParseDccSend   },
	{ "TSSEND" , dccModuleParseDccSend   },
	{ "STSEND" , dccModuleParseDccSend   },
	{ "GET"    , dccModuleParseDccGet    },
	{ "SGET"   , dccModuleParseDccGet    },
	{ "TGET"   , dccModuleParseDccGet    },
	{ "STGET"  , dccModuleParseDccGet    },
	{ "TSGET"  , dccModuleParseDccGet    },
	{ "LIST"   , dccModuleParseDccList   },
	{ "ACCEPT" , dccModuleParseDccAccept },
	{ "RESUME" , dccModuleParseDccResume },
	{ "RECV"   , dccModuleParseDccRecv   },
	{ "SRECV"  , dccModuleParseDccRecv   },
	{ "TRECV"  , dccModuleParseDccRecv   },
	{ "TSRECV" , dccModuleParseDccRecv   },
	{ "STRECV" , dccModuleParseDccRecv   },
	{ "RSEND"  , dccModuleParseDccRSend  },
	{ "SRSEND" , dccModuleParseDccRSend  },
	{ "TRSEND" , dccModuleParseDccRSend  },
	{ "STRSEND", dccModuleParseDccRSend  },
	{ "TSRSEND", dccModuleParseDccRSend  },
	{ "CANVAS" , dccModuleParseDccCanvas },
	{ "VOICE"  , dccModuleParseDccVoice  }
};


// We want C linkage on this one: we want to be able to dlsym() it with a simple name
// FIXME: Is this portable enough ? Or is better to have a table entry ?

KVIMODULEEXPORTFUNC void dccModuleCtcpDccParseRoutine(KviDccRequest *dcc)
{
	dcc->szType.toUpper();
	
	for(int i=0;i<KVI_NUM_KNOWN_DCC_TYPES;i++)
	{
		if(kvi_strEqualCS(dccParseProcTable[i].type,dcc->szType.ptr()))
		{
			(dccParseProcTable[i].proc)(dcc);
			return;
		}
	}
	// ops...we don't know this dcc type
	if(!dcc->ctcpMsg->msg->haltOutput())
	{
		KviStr szError(KviStr::Format,
				__tr("unknown DCC type '%s'"),dcc->szType.ptr());
		dcc_module_request_error(dcc,szError.ptr());
	}
}

/*
	@doc: dcc_connection
	@type:
		generic
	@title:
		Dcc negotiation and connection
	@short:
		Overview of the dcc internals
	@keyterms:
		dcc without irc
	@body:
		[big]What is DCC ?[/big][br]
		'DCC' stands for Direct Client Connection: it is used to exchange data
		directly between two IRC clients (with no IRC server data routing).[br]
		DCC itself is not a well-defined protocol , but rather a set of
		subprotocols with (more or less) standardized rules.[br]
		Sub-protocols are also (historically) called [b]"DCC types"[/b]; this term often
		leads to confusion and it will become clear later.[br]
		Each subprotocol has two main parts: the [b]DCC negotiation[/b] and the [b]DCC transfer[/b].[br]
		The [b]DCC negotiation[/b] part is used to request the [b]DCC transfer[/b] and define its necessary parameters,[br]
		while the [b]DCC transfer[/b] part is the real data transfer between clients.[br]
		The [b]DCC negotiation[/b] requires a third entity that routes the negotiation data between clients;
		this is usually an IRC server.[br]
		[br]
		[big]DCC Negotiation[/big][br]
		This part of the protocol is the most tricky and difficult one, and is different for almost every DCC subprotocol.[br]
		The "constant" scenario of the negotiation is more or less the following:[br]
		There are two IRC clients connected to the same IRC network and they want to exchange some data in
		a direct client connection.[br]
		Both clients know the other client by "nickname" only (and eventually by the host displayed by the IRC server,
		but this can not be trusted for several reasons), and can send text messages to each other by using the
		IRC network as data channel.[br]
		To initiate a direct client connection , one of the clients must start listening on some port (this is called [b]passive client[/b])
		and the other must connect to that port on the first client's machine (this is the [b]active client[/b]).[br]
		Both clients must agree on who is the passive and who is the active client. The active client must also
		know the passive client's IP address and port (in order to be able to contact it).[br]
		Finally , both clients must agree on the transfer type that has to be initiated.[br]
		The negotiation exchanges these informations between clients by using IRC as channel and CTCP messages
		as encoding method.[br]
		An example will make things clearer:[br]
		DCC CHAT is the simplest (and most widely implemented) DCC subprotocol:
		it is used to exchange <cr><lf> separated text data between clients.[br]
		Assume that you want to estabilish a DCC CHAT
		connection to 'Sarah' that is actually connected to your IRC network (so
		she/he is an IRC user just like you).
		All you have to do is type sth as "/dcc chat Sarah" in your IRC client.
		The client will setup a listening socket on a random port choosen
		usually by the kernel of your OS. In this case YOU are the [b]passive client[/b], and Sarah is the active one.[br]
		Once the socket is ready to accept connections,
		your client will send a [doc:ctcp_handling]CTCP message[/doc] to Sarah using the IRC connection (and protocol) as channel:[br]
		PRIVMSG Sarah :<0x01>DCC CHAT chat <ip_address> <port><0x01>[br]
		where <ip_address> is the address of the listening socket and <port>
		is the port that it has been bound to (these informations are obtained
		after the socket has been setup). Once Sarah has received the CTCP message,
		and agreed to connect, her (active) client will attempt to connect to the
		specified <ip_address> and <port> (eg. to your listening socket).[br]
		Once the connection has been estabilished, it continues using the
		specific CHAT transfer protocol.[br]
		Some IRC clients allow modifications of this procedure:[br]
		First of all, the port to listen on can be specified by the user
		and not by the kernel; this is useful when the passive client
		is behind a firewall that "shades" some sets of ports.
		The ip address for the listening socket
		can be specified by the user as well (especially when the machine has more than one network interface).[br]
		A more challenging trick is to listen on a specified ip address and port
		and notify different ones to the remote user (eg, <ip_address> and <port>
		parameters of the CTCP message are not the ones that the client is listening on).
		This is especially useful with "transparent proxy" firewalls that
		often are not transparent enough to allow the DCC connections.
		(If you have one of these firewalls you know what I'm talking about,
		otherwise just read on). KVIrc allows to avoid the usage of a third entity
		for the protocol negotiation too.
		You can setup a listening socket on a specified port and ip address
		without notyfying anyone of this.
		You can also manually connect to a specified port and ip address without
		having been notified of a DCC request.[br][br][br]
		Is everything clear ?...I don't think so... my English is really too bad...
		[br]
		[big]DCC transfer[/big][br]
		The DCC transfer part is different for every DCC subprotocol: the only "constant" thing is
		that it happens over a direct client to client TCP connection.[br]
		[br]
		[big]DCC subprotocols[/big][br]
		There are two main standardized DCC subprotocols that are widely implemented in IRC clients:
		[b]DCC Chat[/b] and [b]DCC Send[/b].[br]
		DCC Chat is quite simple and the protocol is more or less completly defined.[br]
		The DCC Send is a *real mess* (tm): the original definition was not very flexible
		so many IRC clients tried to enchance both the negotiation and the transfer, leading
		often to incompatible implementations. (I can remember of the Turbo File Transfer implemented
		by VIrc, the Send-Ahead enchancement implemented in many clients, the RESUME facility...)[br]
		Many clients introduced new DCC subprotocols with non-standard implementations,
		leading again to client incompatibility.[br]
		Some of the notable subprotocols are DCC Voice, DCC Draw, DCC Whiteboard...[br]
		[br]
		[big]DCC Chat[/big][br]
		This is the simpliest and most standardized DCC subprotocol. Almost any IRC client implements it.[br]
		It is used to exchange lines of text between the two clients.[br]
		The negotiation is quite simple: assume that client A wants to estabilish a DCC CHAT connection to client B.
		Client A sets up a listening socket and retrieves its adress (ip address and port).[br]
		Once the socket is ready client A sends a CTCP request to B, in the following form:[br]
		[b]DCC CHAT chat <ipaddress> <port>[/b][br]
		Where <ipaddress> is a string rappresenting an unisnged integer that is the A socket's IP address
		in network byte order, and where <port> is a string rappresenting an unsigned integer that is the
		A socket's port.[br]
		The meaning of the second "chat" string in the CTCP request is quite obscure: it was probably
		introduced to have the <ipaddress> as second parameter , as in the DCC Send subprotocol.[br]
		Client B receives the CTCP , parses it , eventually asks the user for permission and connects
		to the specified ip address and port.
		The transfer protocol is quite simple: both clients can send text lines separated by <cr><lf> pairs.[br]
		Some clients use only <lf> as line terminator so the general idea is that one of <cr> <cr><lf> or <lf>
		can be used as line terminator.[br]
		As extension to the protocol , KVIrc allows <ipaddress> to be an IPV6 address in the standard hexadecimal
		notation: the connection will be made over the IPV6 protocol in this case (obviously if both clients
		support this feature).[br]
		(I can't understand why the original DCC specification used the unsigned int format instead of the
		standard string rappresentation of the IP address...).[br]
		KVIrc adds the Secure Sockets Layer to the DCC CHAT protocol. In this case the negotiation string becomes:[br]
		[b]DCC SCHAT chat <ipaddress> <port>[/b][br]
		where "SCHAT" stands for Secure CHAT.[br] The external protocol is exactly the same but is built on a Secure Sockets Layer
		implementation (actually OpenSSL). The connection will be encrypted with a private key algorithm after
		a public key handshake.[br]
		[br]
		[big]DCC Send[/big][br]
		DCC Send is another standard subprotocol. Most clients implement this as well, many have tried
		to enchance it.[br]
		The basic DCC Send protocol allows transferring a file from the requesting client to the receiving client.[br]
		The requesting client (the one that sends the file) is always passive and the receiving client is always active.[br]
		This is a huge protocol limitation since firewalled clients are often unable to accept incoming connections.[br]
		The negotiation protocol is a bit complexier than DCC Chat: assume that client A wants to send the file F to client B.[br]
		Client A sets up a listening socket and retrieves its ip address and port.[br]
		Client A sends a CTCP request to client B in the following form:[br]
		[b]DCC SEND <filename> <ipaddress> <port> <filesize>[/b][br]
		<ipaddress> and <port> have the same semantics as in the DCC Chat subprotocol.[br]
		<filename> is the name (without path!) of the file to be sent , and <filesize> is (yeah), the file size.[br]
		Client B receives the CTCP , parses it , eventually asks the user for confirmation and connects to the
		specified ip address and port; the transfer then begins.[br]
		The client A sends blocks of data (usually 1-2 KB) and at every block awaits confirmation from the client B,
		that when receiving a block should reply 4 bytes containing an unsigned number specifying the total size
		of the file received up to that moment.[br]
		The transmission closes when the last acknowledge is received by client A.[br]
		The acknowledges were meant to include some sort of coherency check in the transmission, but in fact
		no client can "recover" from an acknowledge error/desync: all of them just close the connection declaring the
		transfer as failed (the situation is even worse in fact: often acknowledge errors aren't even detected!).[br]
		Since the packet-acknowledge round trip eats a lot of time , many clients included
		the "send-ahead" feature: the client A does NOT wait for the acknowledge of the first packet before sending the second one.[br]
		The acknowledges are still sent , but just a reverse independent stream.[br] This makes the DCC Send considerably faster.[br]
		Since the acknowledge stream has non-zero bandwidth usage, no client can recover from an acknowledge error and
		having them as an independant stream is more or less like having no acknowledges, the "Turbo" ( :) ) extension has been added:
		client B will send no acknowledges and will just close the connection when he has received all the expected data.[br]
		This makes the DCC Send as fast as FTP transfers.[br]
		The "Turbo" extension is specified during the negotiation phase, bu using TSEND as DCC message type (instead of SEND).[br]
		The "Turbo" extension is not widely implemented.[br]
		Later implementations have added the support for resuming interrupted DCC Send transfers:[br]
		Client A sets up the socket and sends the CTCP request as before.[br]
		If client B discovers that the file has been partially received in a previous DCC Send session it sends
		a resume request in the following form:[br]
		[b]DCC RESUME <filename> <port> <resume position>[/b][br]
		Where <port> is the <port> sent in the DCC SEND request and <resume position> is the position in the file
		from where the transfer should start.[br]
		Cilent A receives the request, parses it and eventually replies with:[br]
		[b]DCC ACCEPT <filename> <port> <resume position>[/b][br]
		Client B receives the ACCEPT message, connects to client A and the transfer initiates as before.[br]
		The "Send-ahead" and "Turbo" extensions can obviously be used also in this case (But 'T' is NOT prepended to the RESUME and ACCEPT messages).[br]
		The IPV6 extension can be used also in this subprotocol, so <ipaddress> can be also an IPV6 address in hexadecimal notation.[br]
		KVIrc introduces the SSL extension also to DCC SEND. The protocol remains the same again but it is build over
		a Secure Sockets Layer implementation just like DCC CHAT.[br]
		With SSL the negotiation string becomes:[br]
		[b]DCC SSEND <filename> <ipaddress> <port> <filesize>[/b][br]
		where "SSEND" stands for Secure SEND.[br]
		The "turbo" extension can be combined with the SSL extension too. In this case the second parameter
		of the negotiation string must be "TSSEND" or "STSEND".[br]
		[br]
		[big]DCC Recv[/big][br]
		DCC Recv is the counterpart of DCC Send. This is a KVIrc extension and is not a standard yet.[br]
		The purpose of this subprotocol will not be immediately clear, but keep reading on.[br]
		It is used to request a file from another client: assume that Client A knows that client B has
		a specific file and is able/wants to send it.[br]
		Client A sets up a listening socket, retrieves its address and port and then
		sends a CTCP request to client B in the following form:[br]
		[b]DCC RECV <filename> <ipaddress> <port> <resume position>[/b][br]
		where <filename> is the name of the requested file without path, <ipaddress> and <port> have the usual meaning and <resume position>
		is the position from that the transfer should start from.[br]
		<ipaddress> can be an IPV6 address as well.[br]
		Client B receives the CTCP message, parses it, looks for the file to send (in some unspecified way)
		and connects to the specified ip address and port. The transfer then begins just as in the DCC send, but in the inverse way:
		client B sends blocks of data to client A and client B sends back acknowledges.[br]
		This subprotocol is useful in transferring data from clients that are behind a firewall and are not able to accept
		incoming connections (this is not possible with a normal DCC Send). In this case the client that receives
		the file is passive and the client that sends it is active (as opposite to DCC Send).[br]
		The "Send ahead" extension can be used also in this case and the "Turbo" extension is activated by prepending a 'T' to the
		DCC message: "TRECV" instead of "RECV". The SSL extension is activated by prepending an 'S' to the
		DCC message: "SRECV" , "STRECV" or "TSRECV".[br]
		This subprotocol has an implicit resume capability and thus has no need for RESUME and ACCEPT messages.[br]
		DCC Recv requires the initiating (passive) client to know that the file to be transferred is avaiable on the B's side
		and probably also know the file size. This subprotocol does not specify how this information is obtained , but it
		will become clear soon that it can be obtained either manually (User B can simply tell the info to User A),
		or automatically (as in the DCC Rsend subprotocol (keep reading)).[br]
		[br]
		[big]DCC RSend[/big][br]
		DCC RSend stands for Reverse Send. This is a KVIrc extension to the SEND protocol to allow firewalled clients
		to send files.[br] In fact, this is a "half" subprotocol , since it defines only a part of the DCC negotiation;
		the transfer is defined by another subprotocol (and specifically bu DCC Recv).[br]
		The requesting client (the one that sends the file) is active and the receiving client is passive.[br]
		Assume that client A wants to send a file to client B and that client A can not accept incoming connections.[br]
		Client A sends a CTCP request to client B in the following form:[br]
		[b]DCC RSEND <filename> <filesize>[/b][br]
		Client B receives the request , parses it , eventually asks the user for confirmation , sets up a listening socket , retrieves
		its ip address and port and switches to the DCC Recv subprotocol by effectively sending the following CTCP message:[br]
		[b]DCC RECV <filename> <ipaddress> <port> <resume position>[/b][br]
		The rest of the transfer is defined by the DCC Recv subprotocol.[br]
		The "Turbo" extension is again activated by prepending a 'T' to the RSEND string, so the initial CTCP will become:[br]
		[b]DCC TRSEND <filename> <filesize>[/b][br]
		The "SSL" extension is also activated by prepending an 'S' to the RSEND string. It can be again combined
		with the "turbo" extension. The negotiation parameter becomes then "SRSEND","TSRSEND" or "STRSEND".[br]
		Easy , no ? :)[br]
		[br]
		[big]DCC Get[/big][br]
		This is again a "half" subprotocol in fact since it defines only a part of the negotiation for file transfers.[br]
		It is also NON standard, since actually no client except KVIrc implements it (AFAIK).[br]
		DCC Get is used to request a file from a remote client. Assume that client A wants to request a file from client B
		(and assume that client A knows that B has that file and wants to send it).[br]
		Client A sends a CTCP message to client B in the following form:[br]
		[b]DCC GET <filename>[/b][br]
		Where <filename> is a name of a file without path.[br]
		Client B receives the message , parses it , looks for an association of the <filename> to a real filesystem file
		and starts one of the two DCC File transfer subprotocols: DCC Send or DCC RSend.[br]
		The client B should prefer the DCC Send method and choose DCC RSend only if it is not able to accept incoming connections.[br]
		This subprotocol can be used by firewalled clients that can't accept connections but still want to request a file
		from another client: this one can fail only if both clients are firewalled (in this case no DCC transfer is possible at all).[br]
		This subprotocol also does not need to "magically" know the file size: the size definition
		is found in the subprotocol that the remote client will choose.[br]
		The association of <filename> with a real file on the B's machine is not explicitly defined by the subprotocol;
		KVIrc uses an internal "file-offer" table with a list of files that are available for download.[br]
		The "turbo" and "SSL" extensions are activated as usual: "TGET" , "SGET" , "TSGET" and "STGET" are supported.[br]
		[br]
		[big]DCC File transfers resume[/big][br]
		DCC SEND : send a file, sender is passive, receiver is active (not good for firewalled senders)[br]
		DCC RECV : receive a file, sender is active, receiver is passive (not good for firewalled receivers)[br]
		DCC RSEND : send a file, sender is active , receiver is passive (not good for firewalled receivers)[br]
		DCC GET : receive a file, sender is passive if not firewalled, receiver active if sender not firewalled (will fail only if both are firewalled)[br]
		The "turbo" extension disables the stream of acknowledges and is activated by prepending the 'T' character to the DCC subprotocol name[br]
		The "SSL" extension causes a Secure Sockets Layer to be used and is activated by prepending the 'S' character to the DCC subprotocol name[br]
		[br]
		[big]DCC VOICE[/big][br]
		DCC VOICE is a KVIrc extension. (Not really: there is a Windows client called VIrc that implements such
		a protocol but it is incompatible with KVIrc actually).[br]
		DCC VOICE allows audio level communication between two clients: the audio stream is compressed
		with a particular codec.[br]
		KVIrc currently supports the ADPCM (core support) and the GSM codec (if the libgsm is available on the target system).[br]
		[b]TODO: Finish the DCC VOICE doc :)[/b]
		[big]More tricks[/big][br]
		KVIrc supports another "hack" to the DCC negotiation: it recognizes "XDCC" as
		negotiation CTCP parameter.[br]
		This can be used to circumvent some irc client limitations (read mIrc) that will not allow
		you to send a /DCC GET since it is an unrecognized DCC type.[br]
		"XDCC" has exactly the same meaning as "DCC" (at least in KVIrc).[br]
*/




static bool dcc_module_init(KviModule * m)
{
	g_pDccBroker = new KviDccBroker();

	m->registerCommand("chat",dcc_module_cmd_chat);
	m->registerCommand("send",dcc_module_cmd_send);
	m->registerCommand("voice",dcc_module_cmd_voice);
	m->registerCommand("rsend",dcc_module_cmd_rsend);
	m->registerCommand("recv",dcc_module_cmd_recv);
	m->registerCommand("get",dcc_module_cmd_get);
	m->registerCommand("canvas",dcc_module_cmd_canvas);
	return true;
}

static bool dcc_module_cleanup(KviModule *m)
{
	delete g_pDccBroker;
	g_pDccBroker = 0;
	m->unregisterMetaObject("KviDccBroker");
	m->unregisterMetaObject("KviDccMarshal");
	m->unregisterMetaObject("KviDccSend");
	m->unregisterMetaObject("KviDccChat");
	m->unregisterMetaObject("KviDccVoice");
	m->unregisterMetaObject("KviDccAcceptBox");
	m->unregisterMetaObject("KviDccSaveFileBox");
	m->unregisterMetaObject("KviDccLoadFileBox");
	m->unregisterMetaObject("KviDccRenameBox");
	m->unregisterMetaObject("KviDccCanvas");
	m->unregisterMetaObject("KviCanvasWidget");
	m->unregisterMetaObject("KviCanvasView");
	m->unregisterMetaObject("KviCanvasItemPropertiesWidget");

#ifdef COMPILE_USE_GSM
	kvi_gsm_codec_done();
#endif

	return true;
}

static bool dcc_module_can_unload(KviModule *m)
{
	return g_pDccBroker ? g_pDccBroker->canUnload() : true;
}

KVIMODULEEXPORTDATA KviModuleInfo kvirc_module_info=
{
	"Dcc",
	"1.0.0",
	"Copyright (C) 2000:\n" \
	"  Szymon Stefanek (stefanek@tin.it)\n",
	"Dcc extension for KVIrc\n",
	dcc_module_init,
	dcc_module_can_unload,
	0,
	dcc_module_cleanup
};
