//
//   File : chat.cpp
//   Creation date : Tue Sep 20 09 2000 15:13:13 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "chat.h"
#include "marshal.h"

#ifdef COMPILE_ON_WINDOWS
	// Ugly Windoze compiler...
	#include "dialogs.h"
#endif

#define _KVI_DEBUG_CHECK_RANGE_
#include "kvi_debug.h"
#include "kvi_options.h"
#include "kvi_input.h"
#include "kvi_ircview.h"
#include "kvi_iconmanager.h"
#include "kvi_locale.h"
#include "kvi_error.h"
#include "kvi_out.h"
#include "kvi_netutils.h"
#include "kvi_console.h"
#include "kvi_frame.h"
#include "kvi_malloc.h"
#include "kvi_memmove.h"
#include "kvi_thread.h"
#include "kvi_ircsocket.h"
#include "kvi_settings.h"
#include "kvi_themedlabel.h"
#include "kvi_socket.h"
#include "kvi_app.h"
#include "kvi_event.h"
#include "kvi_parameterlist.h"

#ifdef COMPILE_CRYPT_SUPPORT
	#include "kvi_crypt.h"
	#include "kvi_cryptcontroller.h"
	#include "kvi_mirccntrl.h"
#endif

#include <qsplitter.h>
#include <qevent.h>
#include <qvbox.h>


#ifdef COMPILE_SSL_SUPPORT
	#include "kvi_sslmaster.h"
#endif

extern KviDccBroker * g_pDccBroker;

KviDccChatThread::KviDccChatThread(KviWindow *wnd,kvi_socket_t fd)
: KviDccThread(wnd,fd)
{
//	debug("KviDccChatThread constructor");
	m_pOutBuffers = new KviPtrList<KviDataBuffer>;
	m_pOutBuffers->setAutoDelete(true);
}


KviDccChatThread::~KviDccChatThread()
{
	if(m_pOutBuffers)delete m_pOutBuffers;
}


void KviDccChatThread::run()
{
	KviDccThreadIncomingData data;
	data.iLen   = 0;
	data.buffer = 0;

//	debug("KviDccChatThread::THREAD STARTING WITH SSL=%d",m_pSSL);

	for(;;)
	{
		// Dequeue events
		while(KviThreadEvent * e = dequeueEvent())
		{
			if(e->id() == KVI_THREAD_EVENT_TERMINATE)
			{
				delete e;
				goto out_of_the_loop;
			} else {
				// Other events are senseless to us
				delete e;
			}
		}

		bool bCanRead;
		bool bCanWrite;
		if(kvi_select(m_fd,&bCanRead,&bCanWrite))
		{
			if(bCanWrite)
			{
				if(!tryFlushOutBuffers())goto out_of_the_loop;
			}
			if(bCanRead)
			{
				data.buffer = (char *) kvi_realloc(data.buffer,(data.iLen + 512) * sizeof(char));
				int readLen;
#ifdef COMPILE_SSL_SUPPORT
				if(m_pSSL)
				{
					readLen = m_pSSL->read(data.buffer + data.iLen,512);
				} else {
#endif
					readLen = kvi_socket_recv(m_fd,data.buffer + data.iLen,512);
#ifdef COMPILE_SSL_SUPPORT
				}
#endif
				if(readLen > 0)
				{
					data.iLen += readLen;
					data.buffer = (char *)kvi_realloc(data.buffer,data.iLen * sizeof(char));
					if(!handleIncomingData(&data,false))break; // non critical...
				} else {

#ifdef COMPILE_SSL_SUPPORT
					if(m_pSSL)
					{
						// ssl error....?
						switch(m_pSSL->getProtocolError(readLen))
						{
							case KviSSL::ZeroReturn:
								readLen = 0;
							break;
							case KviSSL::WantRead:
							case KviSSL::WantWrite:
								// hmmm...
							break;
							case KviSSL::SyscallError:
							{
								int iE = m_pSSL->getLastError(true);
								if(iE != 0)
								{
									raiseSSLError();
									postErrorEvent(KviError_SSLError);
									goto out_of_the_loop;
								}
							}
							break;
							case KviSSL::SSLError:
							{
								raiseSSLError();
								postErrorEvent(KviError_SSLError);
								goto out_of_the_loop;
							}
							break;
							default:
								// Raise unknown SSL ERROR
								postErrorEvent(KviError_SSLError);
								goto out_of_the_loop;
							break;
						}
					}
#endif
					if(data.iLen > 0)
					{
						data.buffer = (char *)kvi_realloc(data.buffer,data.iLen * sizeof(char));
					} else {
						kvi_free(data.buffer);
						data.buffer = 0;
					}

					if(!handleInvalidSocketRead(readLen))
					{
						if(data.iLen)handleIncomingData(&data,true); // critical
						__range_invalid(data.iLen);
						break; // error
					}
				}
			}
			msleep(100);
		}
	}

out_of_the_loop:


	if(data.iLen)kvi_free(data.buffer);

#ifdef COMPILE_SSL_SUPPORT
	if(m_pSSL)
	{
		KviSSLMaster::freeSSL(m_pSSL);
		m_pSSL = 0;
	}
#endif

	if(m_fd != KVI_INVALID_SOCKET)::kvi_socket_close(m_fd);
	m_fd = KVI_INVALID_SOCKET;
}

bool KviDccChatThread::handleIncomingData(KviDccThreadIncomingData * data,bool bCritical)
{
	__range_valid(data->iLen);
	__range_valid(data->buffer);
	char * aux = data->buffer;
	char * end = data->buffer + data->iLen;
	while(aux != end)
	{
		if((*aux == '\n') || (*aux == '\0'))
		{
			KviThreadDataEvent<KviStr> * e = new KviThreadDataEvent<KviStr>(KVI_DCC_THREAD_EVENT_DATA);
			// The left part is len chars long
			int len = aux - data->buffer;
//			debug("LEN = %d, iLen = %d",len,data->iLen);
//#warning "DO IT BETTER (the \r cutting)"
			KviStr * s = new KviStr(data->buffer,len);
			if(s->lastCharIs('\r'))s->cutRight(1);
			e->setData(s);
			// but we cut also \n (or \0)
			++aux;
			// so len += 1; --> new data->iLen -= len;
			data->iLen -= (len + 1);
//			debug("iLen now = %d",data->iLen);
			__range_valid(data->iLen >= 0);
			if(data->iLen > 0)
			{
				// memmove the remaining part to the beginning
				// aux points after \n or \0
				kvi_memmove(data->buffer,aux,data->iLen);
				data->buffer = (char *)kvi_realloc(data->buffer,data->iLen);
				end = data->buffer + data->iLen;
				aux = data->buffer;
			} else {
				// no more data in the buffer
				__range_valid(data->iLen == 0);
				kvi_free(data->buffer);
				data->buffer = end = aux = 0;
			}
			postEvent(m_pWindow,e);
		} else aux++;
//		debug("PASSING CHAR %c",*aux);
	}
	// now aux == end
	if(bCritical)
	{
		// need to flush everything...
		if(data->iLen > 0)
		{
			// in the last part there are no NULL and \n chars
			KviThreadDataEvent<KviStr> * e = new KviThreadDataEvent<KviStr>(KVI_DCC_THREAD_EVENT_DATA);
			KviStr * s = new KviStr(data->buffer,data->iLen);
			if(s->lastCharIs('\r'))s->cutRight(1);
			e->setData(s);
			data->iLen = 0;
			kvi_free(data->buffer);
			data->buffer = 0;
			postEvent(m_pWindow,e);
		}
	}
	return true;
}

void KviDccChatThread::sendRawData(const void * buffer,int len)
{
	m_pMutex->lock();
	m_pOutBuffers->append(new KviDataBuffer((unsigned int)len,(const unsigned char *)buffer));
	m_pMutex->unlock();
}

bool KviDccChatThread::tryFlushOutBuffers()
{
	bool bRet = true;
	m_pMutex->lock();
	while(KviDataBuffer * b = m_pOutBuffers->first())
	{
		int sentLen;
#ifdef COMPILE_SSL_SUPPORT
		if(m_pSSL)
		{
			sentLen = m_pSSL->write((const char *)b->data(),b->size());
		} else {
#endif
			sentLen = kvi_socket_send(m_fd,b->data(),b->size());
#ifdef COMPILE_SSL_SUPPORT
		}
#endif
		if(sentLen > 0)
		{
			if(sentLen == b->size())m_pOutBuffers->removeFirst();
			else {
				// just a part
				b->remove((unsigned int)sentLen);
				break;
			}
		} else {
#ifdef COMPILE_SSL_SUPPORT
			if(m_pSSL)
			{
				// ops...might be an SSL error
				switch(m_pSSL->getProtocolError(sentLen))
				{
					case KviSSL::WantWrite:
					case KviSSL::WantRead:
						// Async continue...
						goto handle_system_error;
					break;
					case KviSSL::SyscallError:
						if(sentLen == 0)
						{
							raiseSSLError();
							postErrorEvent(KviError_remoteEndClosedConnection);
							bRet = false;
							goto out_of_the_loop;
						} else {
							int iSSLErr = m_pSSL->getLastError(true);
							if(iSSLErr != 0)
							{
								raiseSSLError();
								postErrorEvent(KviError_SSLError);
								bRet = false;
								goto out_of_the_loop;
							} else {
								goto handle_system_error;
							}
						}
					break;
					case KviSSL::SSLError:
						raiseSSLError();
						postErrorEvent(KviError_SSLError);
						bRet = false;
						goto out_of_the_loop;
					break;
					default:
						postErrorEvent(KviError_SSLError);
						bRet = false;
						goto out_of_the_loop;
					break;
				}
			}
#endif
handle_system_error:
			if(sentLen < 0)
			{
				int err = kvi_socket_error();
#ifdef COMPILE_ON_WINDOWS
				if((err != EAGAIN) || (err != EINTR) || (err != WSAEWOULDBLOCK))
#else
				if((err != EAGAIN)||(err != EINTR))
#endif
				{
					postErrorEvent(kvi_errorFromSystemError(err));
					bRet = false;
					goto out_of_the_loop;
				}
			}
			break; // send error
		}
	}
out_of_the_loop:
	m_pMutex->unlock();
	return bRet;
}


KviDccChat::KviDccChat(KviFrame *pFrm,KviDccBrokerDescriptor * dcc,const char * name)
: KviWindow(KVI_WINDOW_TYPE_DCCCHAT,pFrm,name)
{
	m_pDescriptor = dcc;
	m_pTopSplitter = new QSplitter(QSplitter::Horizontal,this,"top_splitter");
	KviThemedLabel * dummy = new KviThemedLabel(m_pTopSplitter,"dummy_label");
	QVBox * box = new QVBox(m_pTopSplitter);

#ifdef COMPILE_CRYPT_SUPPORT
	createCryptControllerButton(box);
#endif
	m_pSplitter = new QSplitter(QSplitter::Horizontal,this,"splitter");
	m_pIrcView = new KviIrcView(m_pSplitter,pFrm,this);
	m_pInput   = new KviInput(this);

	setFocusHandler(m_pInput,this);

	m_pSlaveThread = 0;

	if(KVI_OPTION_BOOL(KviOption_boolAutoLogDccChat))m_pIrcView->startLogging(0);

	m_pMarshal = new KviDccMarshal(this);
	connect(m_pMarshal,SIGNAL(error(int)),this,SLOT(handleMarshalError(int)));
	connect(m_pMarshal,SIGNAL(connected()),this,SLOT(connected()));
	connect(m_pMarshal,SIGNAL(inProgress()),this,SLOT(connectionInProgress()));
#ifdef COMPILE_SSL_SUPPORT
	connect(m_pMarshal,SIGNAL(startingSSLHandshake()),this,SLOT(startingSSLHandshake()));
	connect(m_pMarshal,SIGNAL(sslError(const char *)),this,SLOT(sslError(const char *)));
#endif

	m_pSlaveThread = 0;

	startConnection();
}

KviDccChat::~KviDccChat()
{
	g_pDccBroker->unregisterDccWindow(this);
	if(m_pSlaveThread)
	{
		m_pSlaveThread->terminate();
		delete m_pSlaveThread;
		m_pSlaveThread = 0;
	}
	KviThreadManager::killPendingEvents(this);
	delete m_pDescriptor;
	delete m_pMarshal;
}


void KviDccChat::startConnection()
{
	if(!(m_pDescriptor->bActive))
	{
		// PASSIVE CONNECTION
		output(KVI_OUT_DCCMSG,__tr("Attempting a passive DCC %s connection"),m_pDescriptor->szType.ptr());
#ifdef COMPILE_SSL_SUPPORT
		int ret = m_pMarshal->dccListen(m_pDescriptor->szListenIp.ptr(),m_pDescriptor->szListenPort.ptr(),m_pDescriptor->bDoTimeout,m_pDescriptor->bIsSSL);
#else
		int ret = m_pMarshal->dccListen(m_pDescriptor->szListenIp.ptr(),m_pDescriptor->szListenPort.ptr(),m_pDescriptor->bDoTimeout);
#endif
		if(ret != KviError_success)handleMarshalError(ret);

	} else {
		// ACTIVE CONNECTION
		output(KVI_OUT_DCCMSG,__tr("Attempting an active DCC %s connection"),m_pDescriptor->szType.ptr());
#ifdef COMPILE_SSL_SUPPORT
		int ret = m_pMarshal->dccConnect(m_pDescriptor->szIp.ptr(),m_pDescriptor->szPort.ptr(),m_pDescriptor->bDoTimeout,m_pDescriptor->bIsSSL);
#else
		int ret = m_pMarshal->dccConnect(m_pDescriptor->szIp.ptr(),m_pDescriptor->szPort.ptr(),m_pDescriptor->bDoTimeout);
#endif
		if(ret != KviError_success)handleMarshalError(ret);
	}
}

void KviDccChat::connectionInProgress()
{
	if(m_pDescriptor->bActive)
	{
		output(KVI_OUT_DCCMSG,__tr("Contacting host %s on port %s"),m_pDescriptor->szIp.ptr(),m_pDescriptor->szPort.ptr());
	} else {
		output(KVI_OUT_DCCMSG,__tr("Listening on interface %s port %s"),m_pMarshal->localIp(),m_pMarshal->localPort());

		if(m_pDescriptor->bSendRequest)
		{
			KviStr ip     = m_pDescriptor->szFakeIp.hasData() ? m_pDescriptor->szFakeIp : m_pDescriptor->szListenIp;
			KviStr port   = m_pDescriptor->szFakePort.hasData() ? m_pDescriptor->szFakePort.ptr() : m_pMarshal->localPort();
			//FIXME: #warning "OPTION FOR SENDING 127.0.0.1 and so on (not an unsigned number)"
			struct in_addr a;
			if(kvi_stringIpToBinaryIp(ip.ptr(),&a))ip.setNum(htonl(a.s_addr));
			m_pDescriptor->pConsole->socket()->sendFmtData("PRIVMSG %s :%cDCC %s chat %s %s%c",
				m_pDescriptor->szNick.ptr(),0x01,m_pDescriptor->szType.ptr(),ip.ptr(),port.ptr(),0x01);
			output(KVI_OUT_DCCMSG,__tr("Sent DCC %s request to %s...waiting for the remote client to connect"),
				m_pDescriptor->szType.ptr(),m_pDescriptor->szNick.ptr());
		} else output(KVI_OUT_DCCMSG,__tr("DCC %s request not sent: awaiting manual connections"),m_pDescriptor->szType.ptr());
	}
}

void KviDccChat::startingSSLHandshake()
{
#ifdef COMPILE_SSL_SUPPORT
	outputNoFmt(KVI_OUT_SSL,__tr("Low lewel transport connection estabilished"));
	outputNoFmt(KVI_OUT_SSL,__tr("Starting Secure Socket Layer handshake"));
#endif
}

void KviDccChat::sslError(const char * msg)
{
#ifdef COMPILE_SSL_SUPPORT
	output(KVI_OUT_DCCERROR,__tr("[SSL ERROR]: %s"),msg);
#endif
}

const char * KviDccChat::target()
{
	// This may change on the fly...
	m_szTarget.sprintf("%s@%s:%s",
		m_pDescriptor->szNick.ptr(),m_pDescriptor->szIp.ptr(),m_pDescriptor->szPort.ptr());
	return m_szTarget.ptr();
}

void KviDccChat::fillCaptionBuffers()
{
	KviStr tmp(KviStr::Format,"dcc %s %s@%s:%s",
#ifdef COMPILE_SSL_SUPPORT
		m_pDescriptor->bIsSSL ? "schat" : "chat",
#else
		"chat",
#endif
		m_pDescriptor->szNick.ptr(),m_pDescriptor->szIp.ptr(),m_pDescriptor->szPort.ptr());

	m_szPlainTextCaption = tmp;

	m_szHtmlActiveCaption.sprintf("<nobr><font color=\"%s\"><b>%s</b></font></nobr>",
		KVI_OPTION_COLOR(KviOption_colorCaptionTextActive).name().ascii(),tmp.ptr());
	m_szHtmlInactiveCaption.sprintf("<nobr><font color=\"%s\"><b>%s</b></font></nobr>",
		KVI_OPTION_COLOR(KviOption_colorCaptionTextInactive).name().ascii(),tmp.ptr());
}

QPixmap * KviDccChat::myIconPtr()
{
	return g_pIconManager->getSmallIcon(KVI_SMALLICON_DCCMSG);
}


void KviDccChat::getBaseLogFileName(KviStr &buffer)
{
	buffer.sprintf("%s_%s_%s",m_pDescriptor->szNick.ptr(),m_pDescriptor->szIp.ptr(),m_pDescriptor->szPort.ptr());
}

void KviDccChat::ownMessage(const char * text)
{
	if(!m_pSlaveThread)
	{
		output(KVI_OUT_SYSTEMWARNING,__tr("Can't send data: no active connection"));
		return;
	}
#ifdef COMPILE_CRYPT_SUPPORT
	if(cryptSessionInfo())
	{
		if(cryptSessionInfo()->bDoEncrypt)
		{
			if(*text != KVI_TEXT_CRYPT)
			{
				KviStr encrypted;
				cryptSessionInfo()->pEngine->setMaxEncryptLen(-1);
				if(cryptSessionInfo()->pEngine->encrypt(text,encrypted))
				{
					if(cryptSessionInfo()->pEngine->isCryptographicEngine())
					{
						KviStr buf(KviStr::Format,"%c%s\r\n",KVI_TEXT_CRYPT,encrypted.ptr());
						m_pSlaveThread->sendRawData(buf.ptr(),buf.len());
						m_pFrm->firstConsole()->outputPrivmsg(this,KVI_OUT_OWNPRIVMSGCRYPTED,
							m_pDescriptor->szLocalNick.ptr(),m_pDescriptor->szLocalUser.ptr(),
							m_pDescriptor->szLocalHost.ptr(),text);
					} else {
						KviStr buf(KviStr::Format,"%s\r\n",encrypted.ptr());
						m_pSlaveThread->sendRawData(buf.ptr(),buf.len());
						m_pFrm->firstConsole()->outputPrivmsg(this,KVI_OUT_OWNPRIVMSG,
							m_pDescriptor->szLocalNick.ptr(),m_pDescriptor->szLocalUser.ptr(),
							m_pDescriptor->szLocalHost.ptr(),encrypted.ptr());
					}
					return;
				} else {
					output(KVI_OUT_SYSTEMERROR,
						__tr("The crypt engine was not able to encrypt the current message (%s): %s: no data sent to the remote end"),
						text,cryptSessionInfo()->pEngine->lastError());
					return;
				}
			} else {
				text++; //eat the escape code
			}
		}
	}
#endif
	KviStr buf(KviStr::Format,"%s\r\n",text);

	m_pSlaveThread->sendRawData(buf.ptr(),buf.len());
	m_pFrm->firstConsole()->outputPrivmsg(this,KVI_OUT_OWNPRIVMSG,
		m_pDescriptor->szLocalNick.ptr(),m_pDescriptor->szLocalUser.ptr(),
		m_pDescriptor->szLocalHost.ptr(),text);
}

void KviDccChat::ownAction(const char * text)
{
	if(m_pSlaveThread)
	{
		KviStr buf(KviStr::Format,"%cACTION %s%c\r\n",text);
		m_pSlaveThread->sendRawData(buf.ptr(),buf.len());
		output(KVI_OUT_ACTION,"%s %s",m_pDescriptor->szLocalNick.ptr(),text);
	} else {
		output(KVI_OUT_SYSTEMWARNING,__tr("Can't send data: no active connection"));
	}
}

bool KviDccChat::event(QEvent *e)
{
	if(e->type() == KVI_THREAD_EVENT)
	{
		switch(((KviThreadEvent *)e)->id())
		{
			case KVI_DCC_THREAD_EVENT_ERROR:
			{
				int * err = ((KviThreadDataEvent<int> *)e)->getData();
				if(!TRIGGER_EVENT_5PARAM_RETVALUE(KviEvent_OnDCCChatTerminated,g_pApp->activeWindow(),m_pDescriptor->szPort.ptr(),m_pDescriptor->szNick.ptr(),m_pDescriptor->szUser.ptr(),m_pDescriptor->szHost.ptr(),kvi_getErrorString(*err)))
					output(KVI_OUT_DCCERROR,__tr("ERROR: %s"),kvi_getErrorString(*err));
				delete err;
				return true;
			}
			break;
			case KVI_DCC_THREAD_EVENT_DATA:
			{
				KviStr * d = ((KviThreadDataEvent<KviStr> *)e)->getData();
				if(d->firstCharIs(0x01))
				{
					d->cutLeft(1);
					if(d->lastCharIs(0x01))d->cutRight(1);
					if(kvi_strEqualCIN("ACTION",d->ptr(),6))d->cutLeft(6);
					d->stripLeftWhiteSpace();
					output(KVI_OUT_ACTION,"%s %s",m_pDescriptor->szNick.ptr(),d->ptr());
				} else {

#ifdef COMPILE_CRYPT_SUPPORT
					if(KviCryptSessionInfo * cinf = cryptSessionInfo())
					{
						if(cinf->bDoDecrypt)
						{
							if(cinf->pEngine->isCryptographicEngine() && (*(d->ptr()) == KVI_TEXT_CRYPT))
							{
								KviStr decryptedStuff;
								if(cinf->pEngine->decrypt(d->ptr() + 1,decryptedStuff))
								{
									if(!TRIGGER_EVENT_5PARAM_RETVALUE(KviEvent_OnDCCChatMessage,this,m_pDescriptor->szPort.ptr(),m_pDescriptor->szNick.ptr(),m_pDescriptor->szUser.ptr(),m_pDescriptor->szHost.ptr(),decryptedStuff.ptr()))
									{
										m_pFrm->firstConsole()->outputPrivmsg(this,KVI_OUT_DCCCHATMSGCRYPTED,
											m_pDescriptor->szNick.ptr(),m_pDescriptor->szUser.ptr(),
											m_pDescriptor->szHost.ptr(),decryptedStuff.ptr());
									}
								} else {
									output(KVI_OUT_SYSTEMERROR,
										__tr("The following message looks like an encrypted one, but the crypting engine failed to decode it: %s"),
										cinf->pEngine->lastError());
									if(!TRIGGER_EVENT_5PARAM_RETVALUE(KviEvent_OnDCCChatMessage,this,m_pDescriptor->szPort.ptr(),m_pDescriptor->szNick.ptr(),m_pDescriptor->szUser.ptr(),m_pDescriptor->szHost.ptr(),decryptedStuff.ptr()))
									{
										m_pFrm->firstConsole()->outputPrivmsg(this,KVI_OUT_DCCCHATMSG,
											m_pDescriptor->szNick.ptr(),m_pDescriptor->szUser.ptr(),
											m_pDescriptor->szHost.ptr(),d->ptr() + 1);
									}
								}
								delete d;
								return true;
							} else {
								if(!(cinf->pEngine->isCryptographicEngine()))
								{
									KviStr decryptedStuff;
									if(cinf->pEngine->decrypt(d->ptr(),decryptedStuff))
									{
										if(!TRIGGER_EVENT_5PARAM_RETVALUE(KviEvent_OnDCCChatMessage,this,m_pDescriptor->szPort.ptr(),m_pDescriptor->szNick.ptr(),m_pDescriptor->szUser.ptr(),m_pDescriptor->szHost.ptr(),decryptedStuff.ptr()))
										{
											m_pFrm->firstConsole()->outputPrivmsg(this,KVI_OUT_DCCCHATMSG,
												m_pDescriptor->szNick.ptr(),m_pDescriptor->szUser.ptr(),
												m_pDescriptor->szHost.ptr(),decryptedStuff.ptr());
										}
										delete d;
										return true;
									} else {
										output(KVI_OUT_SYSTEMERROR,
											__tr("The following message looks like an encrypted one, but the crypting engine failed to decode it: %s"),
											cinf->pEngine->lastError());
									}
								}
							}
						}
					}
#endif
					if(!TRIGGER_EVENT_5PARAM_RETVALUE(KviEvent_OnDCCChatMessage,this,m_pDescriptor->szPort.ptr(),m_pDescriptor->szNick.ptr(),m_pDescriptor->szUser.ptr(),m_pDescriptor->szHost.ptr(),d->ptr()))
						m_pFrm->firstConsole()->outputPrivmsg(this,KVI_OUT_DCCCHATMSG,
							m_pDescriptor->szNick.ptr(),m_pDescriptor->szUser.ptr(),
							m_pDescriptor->szHost.ptr(),d->ptr());
				}
				delete d;
				return true;
			}
			break;
		}
	}
	return KviWindow::event(e);
}

void KviDccChat::resizeEvent(QResizeEvent *e)
{
	int hght = m_pInput->heightHint();
	int hght2 = m_pTopSplitter->sizeHint().height();
	m_pTopSplitter->setGeometry(0,0,width(),hght2);
	m_pSplitter->setGeometry(0,hght2,width(),height() - (hght + hght2));
	m_pInput->setGeometry(0,height() - hght,width(),hght);
}

QSize KviDccChat::sizeHint() const
{
	QSize ret(m_pIrcView->sizeHint().width(),
		m_pIrcView->sizeHint().height() + m_pInput->heightHint());
	return ret;
}

void KviDccChat::handleMarshalError(int err)
{
	output(KVI_OUT_DCCERROR,__tr("DCC %s Failed: %s"),m_pDescriptor->szType.ptr(),kvi_getErrorString(err));	
}

void KviDccChat::connected()
{
	if(!(m_pDescriptor->bActive))
	{
		// PASSIVE CONNECTION...Find out the remote end
		m_pDescriptor->szIp   = m_pMarshal->remoteIp();
		m_pDescriptor->szPort = m_pMarshal->remotePort();
		m_pDescriptor->szHost = m_pMarshal->remoteIp();
	}

	updateCaption();
	
	m_pSlaveThread = new KviDccChatThread(this,m_pMarshal->releaseSocket());
#ifdef COMPILE_SSL_SUPPORT
	KviSSL * s = m_pMarshal->releaseSSL();
	if(s)
	{
		KviSSLMaster::printSSLConnectionInfo(this,s);
		m_pSlaveThread->setSSL(s);
	}
#endif
	m_pSlaveThread->start();

	if(!TRIGGER_EVENT_6PARAM_RETVALUE(KviEvent_OnDCCChatConnected,this, \
				m_pMarshal->remoteIp(), \
				m_pMarshal->remotePort(), \
				m_pMarshal->localPort(), \
				m_pDescriptor->szNick.ptr(), \
				m_pDescriptor->szUser.ptr(), \
				m_pDescriptor->szHost.ptr()))
	{
		output(KVI_OUT_DCCMSG,__tr("Connected to %s:%s"),
			m_pMarshal->remoteIp(),m_pMarshal->remotePort());
		output(KVI_OUT_DCCMSG,__tr("Local end is %s:%s"),
			m_pMarshal->localIp(),m_pMarshal->localPort());
	}
}


#include "m_chat.moc"
