//
//   File : broker.cpp
//   Creation date : Tue Sep 19 09 2000 10:21:54 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "broker.h"
#include "dialogs.h"
#include "chat.h"
#include "send.h"
#include "canvas.h"
#include "voice.h"

#include "kvi_app.h"
#include "kvi_frame.h"
#include "kvi_locale.h"
#include "kvi_options.h"
#include "kvi_console.h"
#include "kvi_fileutils.h"
#include "kvi_out.h"
#include "kvi_mediatype.h"
#include "kvi_ircsocket.h"
#include "kvi_filetrader.h"

// kvi_app.cpp
extern KVIRC_API KviMediaManager * g_pMediaManager;
extern KVIRC_API KviFileTrader * g_pFileTrader;

#include <qfileinfo.h>


//#warning "The broker might lookup the remote host name"

KviDccBroker::KviDccBroker()
: QObject(0,"dcc_broker")
{
	m_pBoxList = new KviPtrList<KviDccBox>;
	m_pBoxList->setAutoDelete(false);

	m_pDccWindowList = new KviPtrList<KviWindow>;
	m_pDccWindowList->setAutoDelete(false);
}

KviDccBroker::~KviDccBroker()
{
//	m_pBoxList->setAutoDelete(true);
	while(m_pBoxList->first())delete m_pBoxList->first();
	delete m_pBoxList;
	m_pBoxList = 0;
	while(m_pDccWindowList->first())delete m_pDccWindowList->first();
	delete m_pDccWindowList;
}

unsigned int KviDccBroker::runningDccSendTransfersCount()
{
	unsigned int cnt = 0;
	for(KviWindow * wnd = m_pDccWindowList->first();wnd;wnd = m_pDccWindowList->next())
	{
		if(wnd->type() == KVI_WINDOW_TYPE_DCCSEND)
		{
			if(((KviDccSend *)wnd)->isRunning())cnt++;
		}
	}
	return cnt;
}

unsigned int KviDccBroker::terminatedDccSendTransfersCount()
{
	unsigned int cnt = 0;
	for(KviWindow * wnd = m_pDccWindowList->first();wnd;wnd = m_pDccWindowList->next())
	{
		if(wnd->type() == KVI_WINDOW_TYPE_DCCSEND)
		{
			if(!((KviDccSend *)wnd)->isRunning())cnt++;
		}
	}
	return cnt;
}

void KviDccBroker::closeAllTerminatedDccSendTransfers()
{
	KviPtrList<KviWindow> l;
	l.setAutoDelete(false);
	KviWindow * wnd;
	for(wnd = m_pDccWindowList->first();wnd;wnd = m_pDccWindowList->next())
	{
		if(wnd->type() == KVI_WINDOW_TYPE_DCCSEND)
		{
			if(!((KviDccSend *)wnd)->isRunning())l.append(wnd);
		}
	}
	for(wnd = l.first();wnd;wnd = l.next())
	{
		wnd->close();
	}
}

void KviDccBroker::copyDescriptor(KviDccBrokerDescriptor * dst,KviDccBrokerDescriptor * src)
{
	/* Erm... wouldn't a simple memcpy(dst, src, sizeof(KviDccBrokerDescriptor)); work as well (faster ?) ?  | Kristoff*/
	/* No , because we need to copy strings...and not string pointers... | Pragma */

	dst->szType                = src->szType;
	dst->szNick                = src->szNick;
	dst->szUser                = src->szUser;
	dst->szHost                = src->szHost;
	dst->szLocalNick           = src->szLocalNick;
	dst->szLocalUser           = src->szLocalUser;
	dst->szLocalHost           = src->szLocalHost;
	dst->szIp                  = src->szIp;
	dst->szPort                = src->szPort;
	dst->pConsole              = src->pConsole;
	dst->bActive               = src->bActive;
	dst->szListenIp            = src->szListenIp;
	dst->szListenPort          = src->szListenPort;
	dst->szFakeIp              = src->szFakeIp;
	dst->szFakePort            = src->szFakePort;
	dst->bSendRequest          = src->bSendRequest;
	dst->bDoTimeout            = src->bDoTimeout;
	dst->szFileName            = src->szFileName;
	dst->szFileSize            = src->szFileSize;
	dst->bResume               = src->bResume;
	dst->bRecvFile             = src->bRecvFile;
	dst->bNoAcks               = src->bNoAcks;
	dst->bIsTdcc               = src->bIsTdcc;
	dst->bOverrideMinimize     = src->bOverrideMinimize;
	dst->bShowMinimized        = src->bShowMinimized;
	dst->bAutoAccept           = src->bAutoAccept;
	dst->bIsIncomingAvatar     = src->bIsIncomingAvatar;
#ifdef COMPILE_SSL_SUPPORT
	dst->bIsSSL                = src->bIsSSL;
#endif
}

void KviDccBroker::unregisterDccWindow(KviWindow *wnd)
{
	m_pDccWindowList->removeRef(wnd);
}

void KviDccBroker::unregisterDccBox(KviDccBox * box)
{
	//debug("Forgetting box %d",box);
	m_pBoxList->removeRef(box);
}

//void KviDccBroker::killBox(KviDccBox * box)
//{
//	if(box)
//	{
//		// Ok...this is somewhat ugly...
//		if(box)box->forgetDescriptor();
//
//		// the boxes will get garbage collected
//		// so we forget them
//		m_pBoxList->setAutoDelete(false);
//		m_pBoxList->removeRef(box);
//		m_pBoxList->setAutoDelete(true);
//	}
//}

void KviDccBroker::cancelDcc(KviDccBox *box,KviDccBrokerDescriptor * dcc)
{
	if(box)box->forgetDescriptor();
	delete dcc;
}

///////////////////////////////////////////////////////////////////////////////
// RSEND
///////////////////////////////////////////////////////////////////////////////

void KviDccBroker::rsendManage(KviDccBrokerDescriptor * dcc)
{
	// We need the filename...
	QFileInfo fi(dcc->szLocalFileName.ptr());
	if(fi.exists())rsendExecute(0,dcc);
	else rsendAskForFileName(dcc);
}

void KviDccBroker::rsendAskForFileName(KviDccBrokerDescriptor * dcc)
{
	KviDccLoadFileBox * box = new KviDccLoadFileBox(this,dcc);
	m_pBoxList->append(box);
	connect(box,SIGNAL(accepted(KviDccBox *,KviDccBrokerDescriptor *)),
			this,SLOT(rsendExecute(KviDccBox *,KviDccBrokerDescriptor *)));
	connect(box,SIGNAL(rejected(KviDccBox *,KviDccBrokerDescriptor *)),
			this,SLOT(cancelDcc(KviDccBox *,KviDccBrokerDescriptor *)));
	box->show();
}

void KviDccBroker::rsendExecute(KviDccBox * box,KviDccBrokerDescriptor * dcc)
{
	if(box)box->forgetDescriptor();

	if(!g_pApp->windowExists(dcc->pConsole))
	{
		// No way...we NEED the right IRC context...
		g_pApp->activeConsole()->output(KVI_OUT_DCCERROR,
			__tr("Can't send the DCC %s request to %s: the connection has been terminated"),
			dcc->szType.ptr(),dcc->szNick.ptr());
		delete dcc;
		return;
	}

	// Ok...we need the file to exist
	QFileInfo fi(dcc->szLocalFileName.ptr());
	if(!(fi.exists() && fi.isReadable() && (fi.isFile()) && (fi.size() > 0)))
	{
		dcc->pConsole->output(KVI_OUT_DCCERROR,__tr("Can't open file %s for reading"),
			dcc->szLocalFileName.ptr());
		delete dcc;
		return;
	}

	dcc->szFileName = dcc->szLocalFileName;
	dcc->szFileName.cutToLast('/');

	dcc->pConsole->socket()->sendFmtData("PRIVMSG %s :%cDCC %s %s %u%c",
		dcc->szNick.ptr(),0x01,dcc->szType.ptr(),dcc->szFileName.ptr(),fi.size(),0x01);

	// now add a file offer , so he we will accept it automatically
	// 120 secs is a reasonable timeout
	KviStr mask(KviStr::Format,"%s!*@*",dcc->szNick.ptr());

	g_pFileTrader->addOffer(dcc->szFileName.ptr(),dcc->szLocalFileName.ptr(),mask.ptr(),120);

	delete dcc;
}


///////////////////////////////////////////////////////////////////////////////
// ACTIVE CHAT
///////////////////////////////////////////////////////////////////////////////

void KviDccBroker::activeChatManage(KviDccBrokerDescriptor * dcc)
{

	if(!dcc->bAutoAccept)
	{
		// FIXME: better message ? Secure Direct Client Connection...eventually
		// need confirmation
		KviStr tmp(KviStr::Format,__tr( \
				"<center>" \
					"<b>%s [%s@%s]</b> requests a<br>" \
					"<b>Direct Client Connection</b> in <b>%s</b> mode.<br>" \
				),dcc->szNick.ptr(),dcc->szUser.ptr(),dcc->szHost.ptr(),
				dcc->szType.ptr());

#ifdef COMPILE_SSL_SUPPORT
		if(dcc->bIsSSL)tmp.append(__tr("The connection will be secured using SSL.<br>"));
#endif

		tmp.append(KviStr::Format,__tr( \
					"The connection target will be host <b>%s</b> on port <b>%s</b><br>" \
					"<br><hr><br>" \
					"Do you accept ?<br>" \
				"</center>"
			),dcc->szIp.ptr(),dcc->szPort.ptr());

		KviStr caption(KviStr::Format,__tr("DCC %s request"),dcc->szType.ptr());
		KviDccAcceptBox * box = new KviDccAcceptBox(this,dcc,tmp.ptr(),caption.ptr());
		m_pBoxList->append(box);
		connect(box,SIGNAL(accepted(KviDccBox *,KviDccBrokerDescriptor *)),
				this,SLOT(activeChatExecute(KviDccBox *,KviDccBrokerDescriptor *)));
		connect(box,SIGNAL(rejected(KviDccBox *,KviDccBrokerDescriptor *)),
				this,SLOT(cancelDcc(KviDccBox *,KviDccBrokerDescriptor *)));
		box->show();
	} else {
		// auto accept
		activeChatExecute(0,dcc);
	}
}

void KviDccBroker::activeChatExecute(KviDccBox *box,KviDccBrokerDescriptor * dcc)
{
	if(box)box->forgetDescriptor();

	if(!g_pApp->windowExists(dcc->pConsole))
	{
		// rebind to the first available console....
		dcc->pConsole = g_pApp->activeConsole();
	}

	KviStr szSubProto = dcc->szType;
	szSubProto.toLower();

	KviStr tmp(KviStr::Format,"dcc: %s %s@%s:%s",szSubProto.ptr(),dcc->szNick.ptr(),dcc->szIp.ptr(),dcc->szPort.ptr());
	KviDccChat * chat = new KviDccChat(dcc->pConsole->frame(),dcc,tmp.ptr());

	bool bMinimized = dcc->bOverrideMinimize ? dcc->bShowMinimized : \
			(KVI_OPTION_BOOL(KviOption_boolCreateMinimizedDccChat) || \
				(dcc->bAutoAccept && KVI_OPTION_BOOL(KviOption_boolCreateMinimizedDccChatWhenAutoAccepted)));

	dcc->pConsole->frame()->addWindow(chat,!bMinimized);
	if(bMinimized)chat->minimize();

	m_pDccWindowList->append(chat);
}

///////////////////////////////////////////////////////////////////////////////
// PASSIVE CHAT
///////////////////////////////////////////////////////////////////////////////

void KviDccBroker::passiveChatExecute(KviDccBrokerDescriptor * dcc)
{
	KviStr szSubProto = dcc->szType;
	szSubProto.toLower();

	KviStr tmp(KviStr::Format,"dcc: %s %s@%s:%s",szSubProto.ptr(),dcc->szNick.ptr(),dcc->szIp.ptr(),dcc->szPort.ptr());
	KviDccChat * chat = new KviDccChat(dcc->pConsole->frame(),dcc,tmp.ptr());
	bool bMinimized = dcc->bOverrideMinimize ? dcc->bShowMinimized : KVI_OPTION_BOOL(KviOption_boolCreateMinimizedDccChat);
	dcc->pConsole->frame()->addWindow(chat,!bMinimized);
	if(bMinimized)chat->minimize();
	m_pDccWindowList->append(chat);
}


///////////////////////////////////////////////////////////////////////////////
// ACTIVE VOICE
///////////////////////////////////////////////////////////////////////////////

void KviDccBroker::activeVoiceManage(KviDccBrokerDescriptor * dcc)
{
	if(!dcc->bAutoAccept)
	{
		// need confirmation
		KviStr tmp(KviStr::Format,__tr(
				"<center>" \
					"<b>%s [%s@%s]</b> requests a<br>" \
					"<b>Direct Client Connection</b> in <b>VOICE</b> mode.<br>" \
					"The connection target will be host <b>%s</b> on port <b>%s</b><br>" \
					"<br><hr><br>" \
					"Do you accept ?<br>" \
				"</center>"
			),dcc->szNick.ptr(),dcc->szUser.ptr(),dcc->szHost.ptr(),
				dcc->szIp.ptr(),dcc->szPort.ptr());

		KviDccAcceptBox * box = new KviDccAcceptBox(this,dcc,tmp.ptr(),__tr("DCC VOICE request"));
		m_pBoxList->append(box);
		connect(box,SIGNAL(accepted(KviDccBox *,KviDccBrokerDescriptor *)),
				this,SLOT(activeVoiceExecute(KviDccBox *,KviDccBrokerDescriptor *)));
		connect(box,SIGNAL(rejected(KviDccBox *,KviDccBrokerDescriptor *)),
				this,SLOT(cancelDcc(KviDccBox *,KviDccBrokerDescriptor *)));
		box->show();
	} else {
		// auto accept
		activeVoiceExecute(0,dcc);
	}
}

void KviDccBroker::activeVoiceExecute(KviDccBox *box,KviDccBrokerDescriptor * dcc)
{
	if(box)box->forgetDescriptor();

	if(!g_pApp->windowExists(dcc->pConsole))
	{
		// rebind to the first available console....
		dcc->pConsole = g_pApp->activeConsole();
	}

	KviStr tmp(KviStr::Format,"dcc: voice %s@%s:%s",dcc->szNick.ptr(),dcc->szIp.ptr(),dcc->szPort.ptr());
	KviDccVoice * v = new KviDccVoice(dcc->pConsole->frame(),dcc,tmp.ptr());

	bool bMinimized = dcc->bOverrideMinimize ? dcc->bShowMinimized : \
			(KVI_OPTION_BOOL(KviOption_boolCreateMinimizedDccVoice) || \
				(dcc->bAutoAccept && KVI_OPTION_BOOL(KviOption_boolCreateMinimizedDccVoiceWhenAutoAccepted)));

	dcc->pConsole->frame()->addWindow(v,!bMinimized);
	if(bMinimized)v->minimize();

	m_pDccWindowList->append(v);
}


///////////////////////////////////////////////////////////////////////////////
// PASSIVE VOICE
///////////////////////////////////////////////////////////////////////////////

void KviDccBroker::passiveVoiceExecute(KviDccBrokerDescriptor * dcc)
{
	KviStr tmp(KviStr::Format,"dcc: voice %s@%s:%s",dcc->szNick.ptr(),dcc->szIp.ptr(),dcc->szPort.ptr());
	KviDccVoice * v = new KviDccVoice(dcc->pConsole->frame(),dcc,tmp.ptr());
//#warning "Create minimized dcc voice ?... or maybe it's too much ? :)"
	bool bMinimized = dcc->bOverrideMinimize ? dcc->bShowMinimized : KVI_OPTION_BOOL(KviOption_boolCreateMinimizedDccChat);
	dcc->pConsole->frame()->addWindow(v,!bMinimized);
	if(bMinimized)v->minimize();
	m_pDccWindowList->append(v);
}


///////////////////////////////////////////////////////////////////////////////
// ACTIVE CANVAS
///////////////////////////////////////////////////////////////////////////////

void KviDccBroker::activeCanvasManage(KviDccBrokerDescriptor * dcc)
{
	if(!dcc->bAutoAccept)
	{
		// need confirmation
		KviStr tmp(KviStr::Format,__tr(
				"<center>" \
					"<b>%s [%s@%s]</b> requests a<br>" \
					"<b>Direct Client Connection</b> in <b>CANVAS</b> mode.<br>" \
					"The connection target will be host <b>%s</b> on port <b>%s</b><br>" \
					"<br><hr><br>" \
					"Do you accept ?<br>" \
				"</center>"
			),dcc->szNick.ptr(),dcc->szUser.ptr(),dcc->szHost.ptr(),
				dcc->szIp.ptr(),dcc->szPort.ptr());

		KviDccAcceptBox * box = new KviDccAcceptBox(this,dcc,tmp.ptr(),__tr("DCC CANVAS request"));
		m_pBoxList->append(box);
		connect(box,SIGNAL(accepted(KviDccBox *,KviDccBrokerDescriptor *)),
				this,SLOT(activeCanvasExecute(KviDccBox *,KviDccBrokerDescriptor *)));
		connect(box,SIGNAL(rejected(KviDccBox *,KviDccBrokerDescriptor *)),
				this,SLOT(cancelDcc(KviDccBox *,KviDccBrokerDescriptor *)));
		box->show();
	} else {
		// auto accept
		activeCanvasExecute(0,dcc);
	}
}

void KviDccBroker::activeCanvasExecute(KviDccBox *box,KviDccBrokerDescriptor * dcc)
{
	if(box)box->forgetDescriptor();

	if(!g_pApp->windowExists(dcc->pConsole))
	{
		// rebind to the first available console....
		dcc->pConsole = g_pApp->activeConsole();
	}

	KviStr tmp(KviStr::Format,"dcc: canvas %s@%s:%s",dcc->szNick.ptr(),dcc->szIp.ptr(),dcc->szPort.ptr());
	KviDccCanvas * cnv = new KviDccCanvas(dcc->pConsole->frame(),dcc,tmp.ptr());

//#warning "This option should be dedicated to Dcc Canvas!....for now we are using the DccChat options"
	bool bMinimized = dcc->bOverrideMinimize ? dcc->bShowMinimized : \
			(KVI_OPTION_BOOL(KviOption_boolCreateMinimizedDccChat) || \
				(dcc->bAutoAccept && KVI_OPTION_BOOL(KviOption_boolCreateMinimizedDccChatWhenAutoAccepted)));

	dcc->pConsole->frame()->addWindow(cnv,!bMinimized);
	if(bMinimized)cnv->minimize();

	m_pDccWindowList->append(cnv);
}

///////////////////////////////////////////////////////////////////////////////
// PASSIVE CANVAS
///////////////////////////////////////////////////////////////////////////////

void KviDccBroker::passiveCanvasExecute(KviDccBrokerDescriptor * dcc)
{
	KviStr tmp(KviStr::Format,"dcc: canvas %s@%s:%s",dcc->szNick.ptr(),dcc->szIp.ptr(),dcc->szPort.ptr());
	KviDccCanvas * cnv = new KviDccCanvas(dcc->pConsole->frame(),dcc,tmp.ptr());
//#warning "This option should be dedicated to Dcc Canvas!....for now we are using the DccChat options"
	bool bMinimized = dcc->bOverrideMinimize ? dcc->bShowMinimized : KVI_OPTION_BOOL(KviOption_boolCreateMinimizedDccChat);
	dcc->pConsole->frame()->addWindow(cnv,!bMinimized);
	if(bMinimized)cnv->minimize();
	m_pDccWindowList->append(cnv);
}


///////////////////////////////////////////////////////////////////////////////
// SEND
///////////////////////////////////////////////////////////////////////////////

void KviDccBroker::recvFileManage(KviDccBrokerDescriptor * dcc)
{
	if(!dcc->bAutoAccept)
	{
		// need confirmation
		KviStr tmp;

		if(dcc->bActive)
		{
			// Normal active send: we will be connecting
			tmp.sprintf(__tr(
					"<center>" \
						"<b>%s [%s@%s]</b> wants to send you the file<br>" \
						"'<b>%s</b>' <b>%s</b> bytes long.<br>" \
						"The connection target will be host <b>%s</b> on port <b>%s</b><br>" \
						"<br><hr><br>" \
						"Do you accept ?<br>" \
					"</center>"
				),dcc->szNick.ptr(),dcc->szUser.ptr(),dcc->szHost.ptr(),
					dcc->szFileName.ptr(),dcc->szFileSize.ptr(),
					dcc->szIp.ptr(),dcc->szPort.ptr());

		} else {
			// passive: we will be listening!
			tmp.sprintf(__tr(
					"<center>" \
						"<b>%s [%s@%s]</b> wants to send you the file<br>" \
						"'<b>%s</b>' <b>%s</b> bytes long.<br>" \
						"You will be the passive side of the connection.<br>" \
						"<br><hr><br>" \
						"Do you accept ?<br>" \
					"</center>"
				),dcc->szNick.ptr(),dcc->szUser.ptr(),dcc->szHost.ptr(),
					dcc->szFileName.ptr(),dcc->szFileSize.ptr());
		}

		if(dcc->bIsIncomingAvatar)
		{
			tmp.append(KviStr::Format,__tr(
				"<br><hr><center>" \
					"<b>Notes:</b><br>" \
					"The file looks to be an avatar that you have requested.<br>" \
					"You should not change its file name and<br>" \
					"save it in a place where KVIrc can find it:<br>" \
					"this can be the KVIrc 'avatars' directory, your home directory,<br>" \
					"the KVIrc 'incoming' directory, the KVIrc 'pics' directory,<br>" \
					"or the save directory for the incoming file type...<br>" \
					"Usually accepting the default save path suggested in the<br>" \
					"dialog will do the job.<br>" \
					"You can also instruct KVirc to auto-accept the incoming avatars<br>" \
					"by setting the option 'boolAutoAcceptIncomingAvatars' to true.<br>" \
				"</center>"
				)
			);
		}

//#warning "Maybe remove the pending avatar if rejected ?"

		KviStr title(KviStr::Format,__tr("DCC %s request"),dcc->szType.ptr());

		KviDccAcceptBox * box = new KviDccAcceptBox(this,dcc,tmp.ptr(),title.ptr());
		m_pBoxList->append(box);
		connect(box,SIGNAL(accepted(KviDccBox *,KviDccBrokerDescriptor *)),
				this,SLOT(chooseSaveFileName(KviDccBox *,KviDccBrokerDescriptor *)));
		connect(box,SIGNAL(rejected(KviDccBox *,KviDccBrokerDescriptor *)),
				this,SLOT(cancelDcc(KviDccBox *,KviDccBrokerDescriptor *)));
		box->show();
	} else {
		// auto accept

		if(KVI_OPTION_BOOL(KviOption_boolBeVerbose))
		{
			dcc->pConsole->output(KVI_OUT_DCCMSG,__tr("Auto-accepting DCC %s request from %s!%s@%s for file %s"),
				dcc->szType.ptr(),dcc->szNick.ptr(),dcc->szUser.ptr(),
				dcc->szHost.ptr(),dcc->szFileName.ptr());
		}
		chooseSaveFileName(0,dcc);
	}
}

void KviDccBroker::chooseSaveFileName(KviDccBox *box,KviDccBrokerDescriptor *dcc)
{
	if(box)box->forgetDescriptor();

	// Lookup the suggested save directory

	dcc->szLocalFileName = "";

	if(dcc->bIsIncomingAvatar)g_pApp->getLocalKvircDirectory(dcc->szLocalFileName,KviApp::Avatars);
	else {

		if(KVI_OPTION_BOOL(KviOption_boolUseIncomingDccMediaTypeSavePath))
		{
			g_pMediaManager->lock();
			if(KviMediaType * mt = g_pMediaManager->findMediaType(dcc->szFileName.ptr(),false))
			{
				if(mt->szSavePath.hasData())
				{
					if(kvi_directoryExists(mt->szSavePath.ptr()))dcc->szLocalFileName = mt->szSavePath;
				}
			}
			g_pMediaManager->unlock();
		}
	
		if(dcc->szLocalFileName.isEmpty())g_pApp->getLocalKvircDirectory(dcc->szLocalFileName,KviApp::Incoming);
	}

	dcc->szLocalFileName.ensureLastCharIs('/');

	if(!(dcc->bAutoAccept))
	{
		KviDccSaveFileBox * box = new KviDccSaveFileBox(this,dcc);
		m_pBoxList->append(box);
		connect(box,SIGNAL(accepted(KviDccBox *,KviDccBrokerDescriptor *)),
				this,SLOT(renameOverwriteResume(KviDccBox *,KviDccBrokerDescriptor *)));
		connect(box,SIGNAL(rejected(KviDccBox *,KviDccBrokerDescriptor *)),
				this,SLOT(cancelDcc(KviDccBox *,KviDccBrokerDescriptor *)));
		box->show();
	} else {
		// auto accept
		// WE choose the filename
		dcc->szLocalFileName.append(dcc->szFileName);

		if(KVI_OPTION_BOOL(KviOption_boolBeVerbose))
		{
			dcc->pConsole->output(KVI_OUT_DCCMSG,__tr("Auto-saving DCC %s file %s as \r![!dbl]play $0\r%s\r"),
				dcc->szType.ptr(),dcc->szFileName.ptr(),dcc->szLocalFileName.ptr());
		}

		renameOverwriteResume(0,dcc);
	}
}

void KviDccBroker::renameOverwriteResume(KviDccBox *box,KviDccBrokerDescriptor * dcc)
{
	if(box)box->forgetDescriptor();

	// Check if file exists
	QFileInfo fi(dcc->szLocalFileName.ptr());
	if(fi.exists())
	{
		dcc->szLocalFileSize.setNum(fi.size());
		if(!dcc->bAutoAccept)
		{
//#warning "SHOULDN'T BE ABLE TO RESUME IF dcc->szFileSize < fi.size()"
			KviStr tmp(KviStr::Format,__tr(
					"<center>" \
						"<b>The file %s already exists</b><br>" \
						"and is <b>%d</b> bytes long.<br>" \
						"Do you want to <b>overwrite</b> it, <b>auto-rename</b> the incoming one, or" \
						"threat it as an incomplete download and <b>resume</b> it ?<br>" \
					"</center>"
				),dcc->szLocalFileName.ptr(),fi.size());

			KviDccRenameBox * box = new KviDccRenameBox(this,dcc,tmp.ptr());
			m_pBoxList->append(box);
			connect(box,SIGNAL(renameSelected(KviDccBox *,KviDccBrokerDescriptor *)),
					this,SLOT(renameDccSendFile(KviDccBox *,KviDccBrokerDescriptor *)));
			connect(box,SIGNAL(overwriteSelected(KviDccBox *,KviDccBrokerDescriptor *)),
					this,SLOT(recvFileExecute(KviDccBox *,KviDccBrokerDescriptor *)));
			connect(box,SIGNAL(cancelSelected(KviDccBox *,KviDccBrokerDescriptor *)),
					this,SLOT(cancelDcc(KviDccBox *,KviDccBrokerDescriptor *)));
			box->show();
			return;
		} else {
			// auto rename
			renameDccSendFile(0,dcc);
			return;
		}
	} else dcc->szLocalFileSize = "0";

	// everything OK
	recvFileExecute(0,dcc);
}

void KviDccBroker::renameDccSendFile(KviDccBox *box,KviDccBrokerDescriptor * dcc)
{
	if(box)box->forgetDescriptor();

	while(kvi_fileExists(dcc->szLocalFileName.ptr()))
	{
		if(KVI_OPTION_BOOL(KviOption_boolBeVerbose))
		{
			dcc->pConsole->output(KVI_OUT_DCCMSG,__tr("File %s exists: auto-renaming to %s.rnm"),
				dcc->szLocalFileName.ptr(),dcc->szLocalFileName.ptr());
		}
		dcc->szLocalFileName.append(".rnm");
	}

	dcc->szLocalFileSize = "0"; // 0 for sure

	recvFileExecute(0,dcc);
}

void KviDccBroker::recvFileExecute(KviDccBox *box,KviDccBrokerDescriptor * dcc)
{
	if(box)box->forgetDescriptor();

	if(!g_pApp->windowExists(dcc->pConsole))
	{
		// rebind to the first available console....
		dcc->pConsole = g_pApp->activeConsole();
	}

	KviStr szSubProto = dcc->szType;
	szSubProto.toLower();

	KviStr tmp(KviStr::Format,"dcc: %s %s@%s:%s %s",szSubProto.ptr(),dcc->szNick.ptr(),dcc->szIp.ptr(),dcc->szPort.ptr(),dcc->szLocalFileName.ptr());
	KviDccSend * send = new KviDccSend(dcc->pConsole->frame(),dcc,tmp.ptr());

	bool bMinimized = dcc->bOverrideMinimize ? dcc->bShowMinimized : \
			(KVI_OPTION_BOOL(KviOption_boolCreateMinimizedDccSend) || \
				(dcc->bAutoAccept && KVI_OPTION_BOOL(KviOption_boolCreateMinimizedDccSendWhenAutoAccepted)));

	dcc->pConsole->frame()->addWindow(send,!bMinimized);
	if(bMinimized)send->minimize();
	m_pDccWindowList->append(send);
}


void KviDccBroker::sendFileManage(KviDccBrokerDescriptor * dcc)
{
	KviDccLoadFileBox * box = new KviDccLoadFileBox(this,dcc);
	m_pBoxList->append(box);
	connect(box,SIGNAL(accepted(KviDccBox *,KviDccBrokerDescriptor *)),
			this,SLOT(sendFileExecute(KviDccBox *,KviDccBrokerDescriptor *)));
	connect(box,SIGNAL(rejected(KviDccBox *,KviDccBrokerDescriptor *)),
			this,SLOT(cancelDcc(KviDccBox *,KviDccBrokerDescriptor *)));
	box->show();
}

void KviDccBroker::sendFileExecute(KviDccBox * box,KviDccBrokerDescriptor *dcc)
{
	if(box)box->forgetDescriptor();

	if(!g_pApp->windowExists(dcc->pConsole))
	{
		// rebind to the first available console....
		dcc->pConsole = g_pApp->activeConsole();
	}

	QFileInfo fi(dcc->szLocalFileName.ptr());
	if(!(fi.exists() && fi.isReadable() && (fi.isFile()) && (fi.size() > 0)))
	{
		dcc->pConsole->output(KVI_OUT_DCCERROR,__tr("Can't open file %s for reading"),
			dcc->szLocalFileName.ptr());
		delete dcc;
		return;
	}

	dcc->szFileName = dcc->szLocalFileName;
	dcc->szFileName.cutToLast(KVI_PATH_SEPARATOR_CHAR);

	dcc->szLocalFileSize.setNum(fi.size());

	KviStr szSubProto = dcc->szType;
	szSubProto.toLower();

	KviStr tmp(KviStr::Format,"dcc: %s %s@%s:%s %s",szSubProto.ptr(),dcc->szNick.ptr(),dcc->szIp.ptr(),dcc->szPort.ptr(),dcc->szLocalFileName.ptr());
	KviDccSend * send = new KviDccSend(dcc->pConsole->frame(),dcc,tmp.ptr());
	bool bMinimized = dcc->bOverrideMinimize ? dcc->bShowMinimized : KVI_OPTION_BOOL(KviOption_boolCreateMinimizedDccSend);

	dcc->pConsole->frame()->addWindow(send,!bMinimized);
	if(bMinimized)send->minimize();
	m_pDccWindowList->append(send);
}

bool KviDccBroker::canUnload()
{
	if(m_pBoxList)
	{
		if((m_pBoxList->count() != 0) || (m_pDccWindowList->count() != 0))return false;
	} // else in the destructor anyway (going to die)
	return true;
}

bool KviDccBroker::handleResumeAccepted(const char * filename,const char * port)
{
	for(KviWindow * w = m_pDccWindowList->first();w;w = m_pDccWindowList->next())
	{
		if(w->type() == KVI_WINDOW_TYPE_DCCSEND)
		{
			if(((KviDccSend *)w)->resumeAccepted(filename,port))return true;
		}
	}
	return false;
}

bool KviDccBroker::handleResumeRequest(const char * filename,const char * port,unsigned int filePos)
{
	for(KviWindow * w = m_pDccWindowList->first();w;w = m_pDccWindowList->next())
	{
		if(w->type() == KVI_WINDOW_TYPE_DCCSEND)
		{
			if(((KviDccSend *)w)->doResume(filename,port,filePos))return true;
		}
	}
	return false;
}


#include "m_broker.moc"
