//
//   File : libkvichan.cpp
//   Creation date : Sun Feb 02 2002 05:27:11 GMT by Szymon Stefanek
//
//   This chan is part of the KVirc irc client distribution
//   Copyright (C) 2002 Szymon Stefanek (pragma@kvirc.net)
//   Copyright (C) 2002 Juanjo Alvarez  (juanjux@yahoo.es)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "kvi_module.h"
#include "kvi_string.h"
#include "kvi_app.h"
#include "kvi_channel.h"
#include "kvi_uparser.h"
#include "kvi_command.h"
#include "kvi_locale.h"
#include "kvi_topicw.h"
#include "kvi_datacontainer.h"
#include "kvi_ircmask.h"
#include "kvi_maskeditor.h"

static KviChannel * chan_module_find_channel(KviCommand *c,KviParameterList * parms,bool bNext = false)
{
	KviStr szChan = bNext ? parms->safeNextParam() : parms->safeFirstParam();
	if(szChan.isEmpty())
	{
		if(c->window()->type() == KVI_WINDOW_TYPE_CHANNEL)return (KviChannel *)(c->window());
		c->warning(__tr("The current window is not a channel"));
	} else {
		KviWindow * w = g_pApp->findWindow(szChan.ptr());
		if(!w)
		{
			c->warning(__tr("Can't find the window with id '%s'"),szChan.ptr());
			return 0;
		}
		if(w->type() == KVI_WINDOW_TYPE_CHANNEL)return (KviChannel *)w;
		c->warning(__tr("The specified window (%s) is not a channel"),szChan.ptr());
	}
	return 0;
}

/*
	@doc: chan.name
	@type:
		function
	@title:
		$chan.name
	@short:
		Returns the name of a channel
	@syntax:
		$chan.name
		$chan.name(<window_id>)
	@description:
		The first form returns the name of the current channel (assuming that the current window
		is a channel at all). If the current window is not a channel, a warning is printed
		and an empty string is returned.[br]
		The second form returns the name of the channel specified by <window_id>.[br]
*/


static bool chan_module_fnc_name(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_CONTEXT(c,"chan_module_fnc_name");

	KviChannel * ch = chan_module_find_channel(c,parms);
	if(ch)buffer.append(ch->name());
	return c->leaveContext();
}

/*
	@doc: chan.topic
	@type:
		function
	@title:
		$chan.topic
	@short:
		Returns the topic of a channel
	@syntax:
		$chan.topic
		$chan.topic(<window_id>)
	@description:
		The first form returns the topic of the current channel (assuming that the current window
		is a channel at all). If the current window is not a channel, a warning is printed
		and an empty string is returned.[br]
		The second form returns the topic of the channel specified by <window_id>.[br]
		The topic is returned as it is known form at the call time: this means that 
		if the channel is not synchronized with the server (as just after the join, for example)
		you might get an empty string (the topic is not yet known).[br]
*/


static bool chan_module_fnc_topic(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_CONTEXT(c,"chan_module_fnc_topic");

	KviChannel * ch = chan_module_find_channel(c,parms);
	if(ch)buffer.append(ch->topicWidget()->topic());
	return c->leaveContext();
}


/*
	@doc: chan.topicsetby
	@type:
		function
	@title:
		$chan.topicsetby
	@short:
		Returns the author of the topic of a channel
	@syntax:
		$chan.topicsetby
		$chan.topicsetby(<window_id>)
	@description:
		The first form returns the author of the topic of the current channel (assuming that the current window
		is a channel at all). If the current window is not a channel, a warning is printed
		and an empty string is returned.[br]
		The second form returns the author of the topic of the channel specified by <window_id>.[br]
		The topic author nickname is returned if it is known form at the call time: this means that 
		if the channel is not synchronized with the server (as just after the join, for example)
		you might get an empty string (the topic is not yet known).[br]
*/


static bool chan_module_fnc_topicsetby(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_CONTEXT(c,"chan_module_fnc_topicsetby");

	KviChannel * ch = chan_module_find_channel(c,parms);
	if(ch)buffer.append(ch->topicWidget()->topicSetBy());
	return c->leaveContext();
}


/*
	@doc: chan.topicsetat
	@type:
		function
	@title:
		$chan.topicsetat
	@short:
		Returns the set time of the topic of a channel
	@syntax:
		$chan.topicsetat
		$chan.topicsetat(<window_id>)
	@description:
		The first form returns the set time of the topic of the current channel (assuming that the current window
		is a channel at all). If the current window is not a channel, a warning is printed
		and an empty string is returned.[br]
		The second form returns the set time of the topic of the channel specified by <window_id>.[br]
		The topic set time is returned if it is known form at the call time: this means that 
		if the channel is not synchronized with the server (as just after the join, for example)
		you might get an empty string (the topic is not yet known).[br]
*/


static bool chan_module_fnc_topicsetat(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_CONTEXT(c,"chan_module_fnc_topicsetat");

	KviChannel * ch = chan_module_find_channel(c,parms);
	if(ch)buffer.append(ch->topicWidget()->topicSetAt());
	return c->leaveContext();
}



/*
	@doc: chan.usercount
	@type:
		function
	@title:
		$chan.usercount
	@short:
		Returns the number of users on a channel
	@syntax:
		$chan.usercount
		$chan.usercount(<window_id>)
	@description:
		The first form returns the number of users on the current channel (assuming that the current window
		is a channel at all). If the current window is not a channel, a warning is printed
		and an empty string is returned.[br]
		The second form returns the number of users on the channel specified by <window_id>.[br]
		The number of users is returned if it is known form at the call time: this means that 
		if the channel is not synchronized with the server (as just after the join, for example)
		you might get a number that is actually smaller.[br]
*/


static bool chan_module_fnc_usercount(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_CONTEXT(c,"chan_module_fnc_usercount");

	KviChannel * ch = chan_module_find_channel(c,parms);
	if(ch)buffer.append(KviStr::Format,"%u",ch->count());
	return c->leaveContext();
}

/*
	@doc: chan.opcount
	@type:
		function
	@title:
		$chan.opcount
	@short:
		Returns the number of op users on a channel
	@syntax:
		$chan.opcount
		$chan.opcount(<window_id>)
	@description:
		The first form returns the number of op users on the current channel (assuming that the current window
		is a channel at all). If the current window is not a channel, a warning is printed
		and an empty string is returned.[br]
		The second form returns the number of op users on the channel specified by <window_id>.[br]
		The number of ops is returned if it is known form at the call time: this means that 
		if the channel is not synchronized with the server (as just after the join, for example)
		you might get a number that is actually smaller.[br]
*/


static bool chan_module_fnc_opcount(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_CONTEXT(c,"chan_module_fnc_opcount");

	KviChannel * ch = chan_module_find_channel(c,parms);
	if(ch)buffer.append(KviStr::Format,"%d",ch->opCount());
	return c->leaveContext();
}


/*
	@doc: chan.voicecount
	@type:
		function
	@title:
		$chan.voicecount
	@short:
		Returns the number of voiced users on a channel
	@syntax:
		$chan.voicecount
		$chan.voicecount(<window_id>)
	@description:
		The first form returns the number of voiced users on the current channel (assuming that the current window
		is a channel at all). If the current window is not a channel, a warning is printed
		and an empty string is returned.[br]
		The second form returns the number of voiced users on the channel specified by <window_id>.[br]
		The number of voiced users is returned if it is known form at the call time: this means that 
		if the channel is not synchronized with the server (as just after the join, for example)
		you might get a number that is actually smaller.[br]
*/


static bool chan_module_fnc_voicecount(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_CONTEXT(c,"chan_module_fnc_voicecount");

	KviChannel * ch = chan_module_find_channel(c,parms);
	if(ch)buffer.append(KviStr::Format,"%d",ch->voiceCount());
	return c->leaveContext();
}

/*
	@doc: chan.halfopcount
	@type:
		function
	@title:
		$chan.halfopcount
	@short:
		Returns the number of voiced users on a channel
	@syntax:
		$chan.halfOpCount
		$chan.halfOpCount(<window_id>)
	@description:
		The first form returns the number of half-op users on the current channel (assuming that the current window
		is a channel at all). If the current window is not a channel, a warning is printed
		and an empty string is returned.[br]
		The second form returns the number of half-op users on the channel specified by <window_id>.[br]
		The number of half-op users is returned if it is known form at the call time: this means that 
		if the channel is not synchronized with the server (as just after the join, for example)
		you might get a number that is actually smaller.[br]
*/


static bool chan_module_fnc_halfopcount(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_CONTEXT(c,"chan_module_fnc_halfopcount");

	KviChannel * ch = chan_module_find_channel(c,parms);
	if(ch)buffer.append(KviStr::Format,"%d",ch->halfOpCount());
	return c->leaveContext();
}

/*
	@doc: chan.ison
	@type:
		function
	@title:
		$chan.ison
	@short:
		Checks if an user is on a channel
	@syntax:
		$chan.ison(<nickname>[,<window_id>])
	@description:
		Returns 1 if <nickname> is on the channel identified by <window_id>, 0 otherwise.[br]
		If <window_id> is not specified the current window is used (assuming that it is a channel at all).[br]
		If the window is not a channel , a warning is printed and an empty string is returned.[br]
*/


static bool chan_module_fnc_ison(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_CONTEXT(c,"chan_module_fnc_ison");
	KviStr szNick = parms->safeFirstParam();
	KviChannel * ch = chan_module_find_channel(c,parms,true);
	if(ch)buffer.append(ch->isOn(szNick.ptr()) ? '1' : '0');
	return c->leaveContext();
}

/*
	@doc: chan.getflag
	@type:
		function
	@title:
		$chan.getflag
	@short:
		Returns the channel-user mode flag of an user
	@syntax:
		$chan.getflag(<nickname>[,<window_id>])
	@description:
		Returns the channe-user mode flag of an user on the channel specified by <window_id>.[br]
		If <window_id> is not passed, the current window is used.[br]
		If the specified window is not a channel, a warning is printed and an empty string is returned.[br]
		If the specified user is not on the channel identified by <window_id>, and empty string is returned.[br]
		The user flag is '@' for ops , '%' for halfops , '+' for voiced users.[br]
*/


static bool chan_module_fnc_getflag(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_CONTEXT(c,"chan_module_fnc_getflag");
	KviStr szNick = parms->safeFirstParam();
	KviChannel * ch = chan_module_find_channel(c,parms,true);
	if(ch)
	{
		char flag = ch->userListView()->getUserFlag(szNick.ptr());
		if(flag)buffer.append(flag);
	}
	return c->leaveContext();
}

/*
	@doc: chan.isop
	@type:
		function
	@title:
		$chan.isop
	@short:
		Checks if an user is op on a channel
	@syntax:
		$chan.isop(<nickname>[,<window_id>])
	@description:
		Returns 1 if <nickname> is an operator on the channel identified by <window_id>, 0 otherwise.[br]
		If <window_id> is not specified the current window is used (assuming that it is a channel at all).[br]
		If the window is not a channel , a warning is printed and an empty string is returned.[br]
		Note that if the user is not on the channel at all, you will get 0 as return value.[br]
*/


static bool chan_module_fnc_isop(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_CONTEXT(c,"chan_module_fnc_isop");
	KviStr szNick = parms->safeFirstParam();
	KviChannel * ch = chan_module_find_channel(c,parms,true);
	if(ch)buffer.append(ch->isOp(szNick.ptr()) ? '1' : '0');
	return c->leaveContext();
}

/*
	@doc: chan.isvoice
	@type:
		function
	@title:
		$chan.isvoice
	@short:
		Checks if an user is voiced on a channel
	@syntax:
		$chan.isop(<nickname>[,<window_id>])
	@description:
		Returns 1 if <nickname> is voiced on the channel identified by <window_id>, 0 otherwise.[br]
		If <window_id> is not specified the current window is used (assuming that it is a channel at all).[br]
		If the window is not a channel , a warning is printed and an empty string is returned.[br]
		Note that if the user is not on the channel at all, you will get 0 as return value.[br]
*/


static bool chan_module_fnc_isvoice(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_CONTEXT(c,"chan_module_fnc_isvoice");
	KviStr szNick = parms->safeFirstParam();
	KviChannel * ch = chan_module_find_channel(c,parms,true);
	if(ch)buffer.append(ch->isVoice(szNick.ptr()) ? '1' : '0');
	return c->leaveContext();
}

/*
	@doc: chan.ishalfop
	@type:
		function
	@title:
		$chan.ishalfop
	@short:
		Checks if an user is half-op on a channel
	@syntax:
		$chan.ishalfop(<nickname>[,<window_id>])
	@description:
		Returns 1 if <nickname> is a half-operator on the channel identified by <window_id>, 0 otherwise.[br]
		If <window_id> is not specified the current window is used (assuming that it is a channel at all).[br]
		If the window is not a channel , a warning is printed and an empty string is returned.[br]
		Note that if the user is not on the channel at all, you will get 0 as return value.[br]
*/


static bool chan_module_fnc_ishalfop(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_CONTEXT(c,"chan_module_fnc_ishalfop");
	KviStr szNick = parms->safeFirstParam();
	KviChannel * ch = chan_module_find_channel(c,parms,true);
	if(ch)buffer.append(ch->isHalfOp(szNick.ptr()) ? '1' : '0');
	return c->leaveContext();
}

/*
	@doc: chan.ismeop
	@type:
		function
	@title:
		$chan.isMeOp
	@short:
		Checks if the current user is op on a channel
	@syntax:
		$chan.isMeOp
		$chan.isMeOp(<window_id>)
	@description:
		Returns 1 if the current user is op on the channel specified by <window_id> , 0 otherwise.[br]
		If <window_id> is not passed , the current window is used (assuming it is a channel at all).[br]
		If the window is not a channel , a warning is printed and an empty string is returned.[br]
		This function is a "shortcut" for [fnc]$chan.isop[/fnc]([fnc]$me[/fnc]).[br]
*/


static bool chan_module_fnc_ismeop(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_CONTEXT(c,"chan_module_fnc_ismeop");

	KviChannel * ch = chan_module_find_channel(c,parms);
	if(ch)buffer.append(ch->isMeOp() ? '1' : '0');
	return c->leaveContext();
}

/*
	@doc: chan.mode
	@type:
		function
	@title:
		$chan.mode
	@short:
		Returns the mode string of a channel
	@syntax:
		$chan.mode
		$chan.mode(<window_id>)
	@description:
		Returns the mode string of the channel identified by <window_id>.[br]
		If no <window_id> is passed , the current channel mode string is returned (assuming that
		the current window is a chnannel at all).[br]
		If the window is not a channel , a warning is printed and an empty string is returned.[br]
*/


static bool chan_module_fnc_mode(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_CONTEXT(c,"chan_module_fnc_mode");

	KviChannel * ch = chan_module_find_channel(c,parms);
	if(ch)
	{
		KviStr chMode;
		ch->getChannelModeString(chMode);
		buffer.append(chMode);
	}
	return c->leaveContext();
}

/*
	@doc: chan.key
	@type:
		function
	@title:
		$chan.key
	@short:
		Returns the key of a channel
	@syntax:
		$chan.key
		$chan.key(<window_id>)
	@description:
		Returns the key of the channel identified by <window_id>.[br]
		If no <window_id> is passed , the current channel key is returned (assuming that
		the current window is a chnannel at all).[br]
		If the window is not a channel , a warning is printed and an empty string is returned.[br]
		If the channel has no key set, an empty string is returned.[br]
*/


static bool chan_module_fnc_key(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_CONTEXT(c,"chan_module_fnc_key");

	KviChannel * ch = chan_module_find_channel(c,parms);
	if(ch)buffer.append(ch->channelKey());
	return c->leaveContext();
}

/*
	@doc: chan.limit
	@type:
		function
	@title:
		$chan.limit
	@short:
		Returns the limit of a channel
	@syntax:
		$chan.limit
		$chan.limit(<window_id>)
	@description:
		Returns the limit of the channel identified by <window_id>.[br]
		If no <window_id> is passed , the current channel limit is returned (assuming that
		the current window is a chnannel at all).[br]
		If the window is not a channel , a warning is printed and an empty string is returned.[br]
		If the channel has no limit set, "0" is returned.[br]
*/


static bool chan_module_fnc_limit(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_CONTEXT(c,"chan_module_fnc_limit");

	KviChannel * ch = chan_module_find_channel(c,parms);
	if(ch)
	{
		if(ch->hasChannelLimit())buffer.append(ch->channelLimit());
		else buffer.append('0');
	}
	return c->leaveContext();
}


/*
	@doc: chan.users
	@type:
		function
	@title:
		$chan.users
	@short:
		Returns an array of channel user nicknames
	@syntax:
		$chan.users([window_id],[mask],[flags])
	@description:
		Returns an array of nicknames on the channel specified by [window_id].[br]
		If [window_id] is empty , the current window is used.[br]
		If the window designated by [window_id] is not a channel a warning is printed and an empty array is returned.[br]
		If [mask] is given , each user is added to the array only
		if matches the [mask].[br]
		[flags] may contain a subset of the letters "ovhnmi":[br]
		"ovhn" are mode flags: the users are added to the array only if they are operators ('o'),
		voiced users ('v'), half-operators ('h') or unflagged ('n') users. (Unflagged means not operators , not
		voiced and not halt-operators). If none of the "ovhn" flags are used, KVIrc behaves like all four were passed.[br]
		The flag 'm' causes the entire user masks to be added to the 
		array entries , as known by KVIrc at the moment of this function call.[br]
		The flag 'i' causes KVIrc to invert the match and add only the users that do NOT match [mask].[br]
	@example:
		[example]
			[comment]# Get the nickname list[/comment]
			%test[] = $chan.users
			[comment]# And loop thru the items[/comment]
			%i = 0
			[comment]# %test[]# returns the number of elements in the array[/comment]
			%count = %test[]#
			while(%i < %count)
			{
				echo "User %i : %test[%i]"
				%i++
			}
			[comment]# Another way of looping[/comment]
			foreach(%tmp,%test[])echo "User %tmp"
			[comment]# Find all the users that come from the *.org domain[/comment]
			%test[]=$chan.users(,*!*@*.org)
			echo %test[]
			[comment]# This will work too![/comment]
			echo $chan.users(,*!*@*.org)
			[comment]# Find all the channel operators[/comment]
			%test[] = $chan.users(,,o)
			echo %test[]
			[comment]# Find all the voiced users that do NOT come from *.edu[/comment]
			[comment]# See also their whole mask[/comment]
			%test[] = $chan.users(,*!*@*.edu,vim)
			echo %test[]
		[/example]
		
*/


static bool chan_module_fnc_users(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_CONTEXT(c,"chan_module_fnc_users");

	c->beginListArrayOrDictionaryReturnIdentifier();

	KviChannel * ch = chan_module_find_channel(c,parms);
	if(!ch)return c->leaveContext();

	KviStr * szMask = parms->safeNext();
	KviStr * szFlags = parms->safeNext();

//	QAsciiDictIterator<KviUserListEntry> it(*(((KviChannel *)(c->window()))->entryDict()));
	KviUserListEntry * e = ((KviChannel *)(c->window()))->userListView()->firstItem();

	bool bCheckMask = szMask->hasData();
	bool bOp = szFlags->contains('o');
	bool bVoice = szFlags->contains('v');
	bool bHalfOp = szFlags->contains('h');
	bool bNone = szFlags->contains('n');
	bool bCheckFlags = bOp || bVoice || bHalfOp || bNone;
	bool bAddMask = szFlags->contains('m');

	int idx = 0;

	if(bAddMask || bCheckFlags || bCheckMask)
	{
		bool bMaskMustMatch = !(szFlags->contains('i'));
		KviIrcMask mask(szMask->ptr());

		while(e)
		{
			if(bCheckFlags)
			{
				if(bOp)
				{
					if(e->flags() & KVI_USERFLAG_OP)goto check_mask;
				}
				if(bVoice)
				{
					if(e->flags() & KVI_USERFLAG_VOICE)goto check_mask;
				}
				if(bHalfOp)
				{
					if(e->flags() & KVI_USERFLAG_HALFOP)goto check_mask;
				}
				if(bNone)
				{
					if(!(e->flags() & KVI_USERFLAG_MASK))goto check_mask;
				}
				goto next_item;
			}
check_mask:
			if(bCheckMask)
			{
				if(mask.matchesFixed(e->nick().ptr(),e->globalData()->user(),e->globalData()->host()) == bMaskMustMatch)goto add_item;
				goto next_item;
			}
add_item:
			if(bAddMask)
			{
				KviStr x(e->nick());
				x.append('!');
				x.append(e->globalData()->user());
				x.append('@');
				x.append(e->globalData()->host());
				c->addListArrayOrDictionaryReturnValue(idx,x,buffer);
			} else {
				c->addListArrayOrDictionaryReturnValue(idx,e->nick(),buffer);
			}
			idx++;
next_item:
			e = e->next();
		}
	} else {
		while(e)
		{
			c->addListArrayOrDictionaryReturnValue(idx,e->nick(),buffer);
			idx++;
			e = e->next();
		}
	}
	return c->leaveContext();

}

/*
	@doc: chan.modearray
	@type:
		command
	@title:
		chan.modearray
	@short:
		Sets an array of channel ban/ban exception/invite exception masks
	@syntax:
		chan.modearray [-s=<start_idx>] [-r=<window_id>] <mode> <arrayname>
	@description:
		Sets the array <arrayname> to the list of <mode> masks on the current channel.[br]
		If the current window is not a channel , a warning is printed and no array is set.[br]
		If you use the -r switch and pass <window_id> , the command is rebound
		to the specified window: this allows you to set the mode array for channels
		that are not "current".[br]
		<mode> must be one of 'b' , 'I' or 'e': 'b' returns the channel ban list, 'I' returns
		the channel invite exception list (if the server supports it), 'e' returns the
		ban exception list.[br]
		The <arrayname> is in fact a name of a local or global dictionary, but must
		NOT contain the leading '%' character.[br]
		The dictionary is treated as array since the entries are set by numberic keys
		starting from '<start_idx>' and ending at "<start_idx> + number-of-users - 1".[br]
		If <start_idx> is not given (no -s switch), it is assumed to be 0.[br]
		The dictionary is not cleared before filling it so if you had previous entries
		(with non numberic keys , for example) they will be preserved.[br]
	@example:
		[example]
			[comment]# Get the invite exception list[/comment]
			chan.modearray I test
			echo %test[]
			[comment]# Something more complex... clearbans alias[/comment]
			alias(clearbans)
			{
				[comment]# %thebans[] is local... do not need to clear it[/comment]
				chan.modearray b thebans
				%c = %thebans[]#
				%n = $(%c / 3)
				%i = 0
				while(%i < %n)
				{
					%idx = $(%i * 3)
					mode $chan.name -bbb %thebans[%idx] %thebans[$(%idx + 1)] %thebans[$(%idx + 2)]
					%i++
				}
				%n = $(%c % 3)
				switch(%n)
				{
					case(1):
						mode $chan.name -b %thebans[$(%c - 1)]
					break
					case(2):
						mode $chan.name -bb %thebans[$(%c - 2)] %thebans[$(%c - 1)]
					break
				}
			}
		[/example]
		
*/
/*
static bool chan_module_cmd_modearray(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"chan_module_cmd_banarray");
	KviStr szArray,szMode;
	if(!g_pUserParser->parseCmdSingleToken(c,szMode))return false;
	if(!g_pUserParser->parseCmdFinalPart(c,szArray))return false;

	if(szArray.isEmpty())
	{
		c->warning(__tr("Missing array name"));
		return c->leaveContext();
	}

	if(c->window()->type() != KVI_WINDOW_TYPE_CHANNEL)
	{
		c->warning(__tr("The current window is not a channel"));
		return c->leaveContext();
	}

	KviDictionary * d = 0;

	if(isupper(*(szArray.ptr())))d = g_pUserParser->globalDataContainer()->lookupDictionary(szArray.ptr(),false);
	else d = c->dataContainer()->lookupDictionary(szArray.ptr(),false);

	int idx = 0;
	if(c->hasSwitch('s'))
	{
		KviStr val;
		c->getSwitchValue('s',val);
		bool bOk;
		idx = val.toInt(&bOk);
		if(!bOk)
		{
			c->warning(__tr("Invalid start index (%s): assuming 0"),val.ptr());
			idx = 0;
		}
	}

	KviStr tmp;

	KviPtrList<KviMaskEntry> * l;

	switch(*(szMode.ptr()))
	{
		case 'b':
			l = ((KviChannel *)(c->window()))->banList();
		break;
		case 'I':
			l = ((KviChannel *)(c->window()))->inviteExceptionList();
		break;
		case 'e':
			l = ((KviChannel *)(c->window()))->banExceptionList();
		break;
		default:
			c->warning(__tr("Invalid mode specification (%s)"),szMode.ptr());
			return c->leaveContext();
		break;
	}

	for(KviMaskEntry * e = l->first();e;e = l->next())
	{
		tmp.setNum(idx);
		d->replace(tmp.ptr(),new KviStr(e->szMask));
		idx++;
	}

	return c->leaveContext();
}
*/

static bool chan_module_init(KviModule * m)
{
	m->registerFunction("name",chan_module_fnc_name);

	m->registerFunction("topic",chan_module_fnc_topic);
	m->registerFunction("topicsetby",chan_module_fnc_topicsetby);
	m->registerFunction("topicsetat",chan_module_fnc_topicsetat);

	m->registerFunction("usercount",chan_module_fnc_usercount);

	m->registerFunction("opcount",chan_module_fnc_opcount);
	m->registerFunction("voicecount",chan_module_fnc_voicecount);
	m->registerFunction("halfopcount",chan_module_fnc_halfopcount);

	m->registerFunction("ison",chan_module_fnc_ison);

	m->registerFunction("isop",chan_module_fnc_isop);
	m->registerFunction("isvoice",chan_module_fnc_isvoice);
	m->registerFunction("ishalfop",chan_module_fnc_ishalfop);

	m->registerFunction("getflag",chan_module_fnc_getflag);
	
	m->registerFunction("ismeop",chan_module_fnc_ismeop);

	m->registerFunction("mode",chan_module_fnc_mode);

	m->registerFunction("key",chan_module_fnc_key);
	m->registerFunction("limit",chan_module_fnc_limit);

	m->registerFunction("users",chan_module_fnc_users);

//	m->registerCommand("userarray",chan_module_cmd_userarray);
//
//	m->registerCommand("modearray",chan_module_cmd_modearray);

	return true;
}

static bool chan_module_cleanup(KviModule *m)
{
	return true;
}


KVIMODULEEXPORTDATA KviModuleInfo kvirc_module_info=
{
	"Chan",                                                  // module name
	"1.0.0",                                                // module version
	"Copyright (C) 2002 Szymon Stefanek (pragma@kvirc.net)"\
	"          (C) 2002 Juanjo Alvarez (juanjux@yahoo.es)",
	"Scripting interface for the channel management",
	chan_module_init,
	0,
	0,
	chan_module_cleanup
};
