//
//   File : kvi_uparser.cpp
//   Creation date : Sun Jul 02 2000 14:48:54 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
#define __KVIRC__

#define _KVI_UPARSER_CPP_

#define _KVI_DEBUG_CHECK_RANGE_
#include "kvi_debug.h"

#include "kvi_uparser.h"
#include "kvi_window.h"
#include "kvi_out.h"
#include "kvi_command.h"
#include "kvi_error.h"
#include "kvi_locale.h"
#include "kvi_datacontainer.h"
#include "kvi_mirccntrl.h"
#include "kvi_ircsocket.h"
#include "kvi_console.h"
#include "kvi_channel.h"
#include "kvi_query.h"
#include "kvi_app.h"
#include "kvi_modulemanager.h"
#include "kvi_event.h"
#include "kvi_options.h"
#include "kvi_scriptobject.h"
#include "kvi_aliasmanager.h"
#include "kvi_eventhandler.h"

#include <ctype.h>

KviUserParser * g_pUserParser = 0;

/*
	@doc: language_overview
	@type:
		language
	@keyterms:
		command syntax,escape character,backslash escape,functions,
		quotes,parameters,positional parameters,positional parameter,expression evaluation,
		parameters evaluation,command switch,switch,comments,indentation
	@title:
		KVIrc scripting language overview
	@short:
		README.FIRST
	@body:
		[big]Introduction[/big][br]
		KVIrc has an internal language 'inspired' by C++,
		sh, perl and mIrc scripting language implementation.
		It is a compromise between flexibility and speed,
		a 'workaround' for many intrinsic problems of an IRC oriented language.
		It is an [b]interpreted[/b] language : there is no compilation stage and
		there are no syntactic trees (except for the algebraic and boolean expressions);
		commands are executed sequentially, while being parsed.
		I can say that the language is 'fast in its intrinsic slowness'.
		I'll try to explain the language by using examples
		instead of strict syntactic rules. (Actually I have even
		tried to write the rules...take a look [doc:syntactic_rules]here[/doc][br][br]
		And please...forgive me for my "fantastic" english :)[br]

		[big]Basic command syntax[/big][br]
		[b]A "command" is a keyword followed by a "list" of space separated parameters.[/b]
		The simplest command (and the one that I prefer at all) is '[cmd]echo[/cmd]';
		it prints the parameter text to a KVIrc window.[br]
		[example]
			[cmd]echo[/cmd] This is the text to print
		[/example]
		[b]Commands are case insensitive[/b]; so 'echo' is equivalent to 'Echo' ,
		to 'ECHO' and to 'eChO'...
		A single command is usually terminated by a newline, but if you want to put more
		commands on a single line, you can separate it by semicolons:[br]
		[example]
			[cmd]echo[/cmd] This is the first line; [cmd]echo[/cmd] this is the second line
		[/example]
		Note:[br]
		Yes...this is a problem for those that want to use
		a ';)' at the end of the IRC commands like /msg...
		My solution: You have to escape the ';' character:[br]
		[example]
			[cmd]echo[/cmd] I can do it now! \;)
		[/example]
		If you're able to find a better solution for this , please drop me a mail.
		Note that using '|' or any other character as the command terminator
		will NOT solve the problem...; there always will be people that
		will not be able to use a ':|' or a ':]' at the end of the command...
		On the other side , a command separator is really needed:
		in this way you can enter a sequence of commands in the window
		commandline , that can not handle multiline text.[br]

		[big]Switches[/big][br]
		Many commands accept switch parameters.[br]
		[b]A switch modifies the behaviour of a command.[/b][br]
		Any switch can optionally accept a parameter, that must
		be specified after an equal ('=') sign following the switch.
		[example]
			[cmd]echo[/cmd] [b]-i = 2[/b] This text uses a specific color scheme
		[/example]
		The -i switch (just for example) changes the attributes
		and the icon of the printed text.[br]
		[b]The switch must be specified immediately after the command keyword.[/b][br]
		[example]
			[cmd]echo[/cmd] This -i = 2 will not work...
		[/example]
		If you want to start the first parameter of a command (that is not a switch)
		with a literal '-' you must 'escape' it:[br]
		[example]
			[cmd]echo[/cmd] \--- This text has three minus signs on the left
		[/example]
		Or use the quotes (see next paragraph):[br]
		[example]
			[cmd]echo[/cmd] "--- This text has three minus signs on the left"
		[/example]

		[big]Parameter processing[/big]
		The literal parameters are processed in a (more or less) "bash-like" manner.
		All the spaces and tabs are reduced to exactly one space unless they are enclosed in
		quotation marks or are escaped.
		[example]
			[cmd]echo[/cmd] This &nbsp; &nbsp; text &nbsp; &nbsp; will &nbsp; &nbsp; have &nbsp; &nbsp; spaces &nbsp; &nbsp; simplified
			[cmd]echo[/cmd] But &nbsp; &nbsp; "this &nbsp; &nbsp; &nbsp; one &nbsp; &nbsp; not"
		[/example]
		The quotes are substantially a trick used to insert spaces inside a
		single parameter (that otherwise wuold be splitted in two or more).[br]
		[example]
			[cmd]echo[/cmd] Parameter1 Parameter2 "Parameter  3 ( with spaces )" Parameter4
		[/example]

		[big]Escape character[/big][br]
		As you might have guessed , the '\' character is really important in the
		KVIrc scripting language.
		For example , it can be used to 'escape' a newline:[br]
		[example]
			[cmd]echo[/cmd] This text will be \
			&nbsp; &nbsp; printed on a single line!
		[/example]
		After an 'escaped' newline all the leading space and tab characters are skipped,
		so you must include the needed spaces [b]before[/b] the escape character.
		The previous example will be printed as:[br][br]
		[i]This text will be printed on a single line[/i][br][br]
		Another example:[br]
		[example]
			[cmd]echo[/cmd] "The new kvirc &nbsp; &nbsp; &nbsp \
            &nbsp; &nbsp; IS OUT!"
			[cmd]echo[/cmd] Check it out at http://www.kvi\
            &nbsp; &nbsp; rc.org!
		[/example]
		This will be printed as:[br][br]
		[i]
		The new kvirc &nbsp; &nbsp; &nbsp; IS OUT![br]
		Check it out at http://www.kvirc.org!
		[/i][br][br]
		The backslash can be used to remove any "semantic" associated
		to a character. For example, you can use it to insert a literal
		quote ('"') character in your parameters.[br]
		[example]
			[cmd]echo[/cmd] And she said \"hello!\"
		[/example]
		Will be printed as[br][br]
		[i]And she said "hello!"[/i][br][br]
		Later we will discover other usages of the backslash escape, such
		as preventing KVIrc from interpreting a literal percent character as a variable
		name begin or separating variable names from subsequent text.

		[big]Command blocks[/big][br]
		Commands can be 'grouped' in blocks by using the classic C++ braces.
		Here is a single line example:[br]
		[example]
			{ [cmd]echo[/cmd] First command; [cmd]echo[/cmd] Second command; } [cmd]echo[/cmd] Third command
		[/example]
		Multi line example:[br]
		[example]
			{
			    [cmd]echo[/cmd] First command
				[cmd]echo[/cmd] Second command
			}
			[cmd]echo[/cmd] Third command
		[/example]
		[i]
		Tip : copy the example above to a text file
		and then use /[cmd]parse[/cmd] &lt;filename&gt;
		[/i][br]
		In this case the command block has no special meaning
		other than making the code more readable , but command blocks
		will be useful later (see [cmd]if[/cmd],[cmd]while[/cmd]...).[br]
		Forward reference:[br]
		Unlike in C or C++, the braces do NOT automatically define a variable scope
		(with few exceptions to this rule ... just to complicate the things a bit more).
		You will recall this last assertion later, when reading about [doc:data_structures]data structures[/doc].[br]

		[big]Comments[/big]
		KVirc supports comments in command sequences.[br]
		A comment startw with the character '#' and terminates with a newline.
		You can start a comment anywhere a command can start.[br]
		[example]
			# This is a comment , it occupies the whole line
			[cmd]echo[/cmd] After the comment!; # This is an end-line comment
		[/example]
		You can not 'escape' newline characters in this case.
		(or better: 'escape' characters have no meaning in comments...
		maybe one day I'll implement it).[br]

		[big]Indentation[/big]
		You can use spaces or tabs to indent your code.
		but please note that the command parameters should be separated by
		space characters (ascii 32). Tabs are not granted to work as parameter separators.[br]
		[example]
		{
		&lt;tab&gt;[cmd]echo[/cmd] Indented command
		&lt;tab&gt;{
		&lt;tab&gt;&lt;tab&gt;# Comment
		&lt;tab&gt;&lt;tab&gt;[cmd]echo[/cmd] Really Really long indented \
		&lt;tab&gt;&lt;tab&gt;&lt;tab&gt;command
		&lt;tab&gt;}
		}
		[/example]
		Tabs behave better than spaces as indentation characters.[br]

		[big]Functions[/big][br]
		KVirc has many internal [doc]functions[/doc]
		that can be used as command parameters.[br]
		[b]All the function names start with a literal '$' character.[/b][br]
		[example]
			[cmd]echo[/cmd] This window caption is [fnc]$caption[/fnc]
		[/example]
		The [fnc]$caption[/fnc] [doc:functions]function[/doc]
		is evaluated before the command executes,
		and it is 'changed' into the 'current' window caption text.[br]
		The [doc]functions[/doc] can be used also as switch parameters.[br]
		[example]
			[cmd]echo[/cmd] -w = [fnc]$window[/fnc] This text will be surely \
			&nbsp; &nbsp; printed in the current window
		[/example]
		The -w switch allows to redirect the echo text to a specified window --- in this
		case the one that you are typing in.[br]
		[i](Surprise: in this case the -w switch is useless ,
		since echo prints text to the current window by default...
		but it will work correctly. :)[/i]
		Normal function names can be made of "anycase" letters, digits and underscores,
		with the restriction that the first character is not a digit.[br]
		Some kind of functions can contain a dot '.' character inside the name
		and these are assumed to be module references (see [doc:modules]the modules documentation[/doc]).[br]
		By now we have seen only simple functions, but there's more...[br]
		The functions can accept parameters; the general syntax for a function call is:[br]
		[b]$<function name>['('<parameter_list>')'][/b][br]
		where <parameter_list> is a list of comma separated parameters,
		eventually empty.
		[example]
			[cmd]echo[/cmd] The word 'neural' is [fnc]$str.len[/fnc](neural) characters long
		[/example]
		The function [fnc]$str.len[/fnc] accepts a single parameter and returns the
		length in characters of the parameter string. The returned value is always
		a string: in this case it can be also interpreted as a number.[br]
		When passing an empty list you can avoid the parenthesis.
		(And you have found the "simple" functions shown above).
		So the followind two calls are equal:[br]
		[example]
			[cmd]echo[/cmd] [fnc]$caption[/fnc]
			[cmd]echo[/cmd] [fnc]$caption()[/fnc]
		[/example]
		If you want to pass an "empty" string as the first parameter you have to use
		the following syntax:[br]
		[example]
			[cmd]echo[/cmd] [fnc]$str.len[/fnc]("")
		[/example]
		Obviously a function is valid as a function parameter.[br]
		[example]
			[cmd]echo[/cmd] [fnc]$str.len[/fnc]([fnc]$caption[/fnc])
		[/example]
		If you want to place a literal '(' or ')' in the function parameters
		you must escape it.
		A special case for when you want to use 'matching' parentheses:
		an opened '(' corresponds to a closed ')'.
		In this case you can omit the 'escape' character.[br]
		[example]
			[cmd]echo[/cmd] The length of '(a+b)' is : [fnc]$str.len[/fnc]( (a+b) )
		[/example]
		This is useful for algebraic and boolean expressions , like the ones
		accepted by the special function $() (see next paragraph).[br]

		[big]Special functions[/big][br]
		The functions of type [b]$N[-[M]][/b] (where N and M are positive
		numbers starting from 0 and N < M) evaluate to the sequence of
		[b]positional parameters[/b] from Nth to Mth.[br]
		If M is omitted , the function evaluate to the sequence of [b]positional
		parameters[/b] from Nth to the last one. If the whole -M block is omitted
		the function evaluate to the Nth positional parameter.
		We will discover more on the [b]positional parameters[/b] when talking
		of aliases and events.[br]
		[example]
			$0 evaluates to the 1st positional parameter
			$0-4 evaluates to the parameters from first to 5th
			$41- evaluates to the parameters from 41st to the last avaiable
		[/example]
		The function [b]$#[/b] evaluates to the number of positional parameters available.
		The [b]positional parameter[/b] functions do not accept parameters.[br]
		The special function [b]$(<expression>)[/b] returns the result
		of the evaluation of the <expression>. In previous versions of KVIrc this
		function was called [fnc]$calc[/fnc].[br]
		[example]
			[cmd]echo[/cmd] $(2 + (3 ^ 7) <= 1 * (3 && 2))
		[/example]
		The special function [b]${<command sequence>}[/b] evaluates to the
		return value of the <command sequence>.[br]
		Forward reference:[br]
		This special function designates a new command scope. The local
		variables declared inside the <command sequence> are local to this one,
		and no external local variables are visible in the <command sequence>.[br]
		The special function [b]$$[/b] evaluates to the current object id,
		but it is too early to explain it here...[br]
		
		[big]Next suggested lectures[/big]
		The KVirc scripting language is more than this...[br]
		It is quite difficult to choose a "next" argument for beginners.
		It might be a good idea to take a look at the KVIrc
		[doc:data_types]data structures documentation[/doc].
		You may also take a first look at the available [doc:command_index_all]commands[/doc]
		and [doc:function_index_all]functions[/doc]. Once you got familiar with the
		basic syntax, you can start reading the [doc:advanced_scripting]advanced scripting documentation[/doc].
*/

/*
	@doc: advanced_scripting
	@type:
		language
	@keyterms:
		aliases , objects
	@title:
		Advanced scripting techniques
	@short:
		README.NEXT
	@keyterms:
		aliases , events
	@body:
		Once you got familiar with the [doc:language_overview]basic KVIrc command syntax[/doc],
		and with the [doc:data_types]local and global variables[/doc],
		you can start exploring the advanced scripting features.[br]
		KVirc allows you to define your own commands called aliases,
		and to define custom re-actions to certain [doc:events]events[/doc].
		The [doc:objects]objects[/doc] expand the scripting possibilities by
		introducing complex data structures and C++ like classes.
		The predefined [doc:classes]object classes[/doc] allow you
		to create your own dialogs, change the KVIrc GUI behaviour and use
		sockets. Once you have learned to use the KVIrc objects,
		you can safely grab a C++ book and start programming KVIrc [doc:modules]modules[/doc] :).
*/


KviUserParser::KviUserParser()
: QObject(0,"user_command_parser")
{
	m_pGlobalDataContainer = new KviDataContainer(true);

	initFunctionDict();
	initCommandDict();
}

KviUserParser::~KviUserParser()
{
	delete m_pFunctionDict;
	delete m_pCommandExecDict;
	delete m_pCommandSkipDict;
	delete m_pGlobalDataContainer;
}

/*
	@doc: coding_tips
	@type:
		generic
	@title:
		Coding tips
	@keyterms:
		indentation,indent,readability
	@short:
		Generic coding tips for scripters (and not only)
	@body:

		Here comes a small list of "coding tips".[br]
		These apply to programming in general , not only to KVIrc scripting.[br]
		[br]
		1. [b]Comment your code[/b][br]
		A well commented code is easy to mantain, and easy to read by others.[br]
		[br]
		2. [b]Indent your code[/b][br]
		Indentation increases the code readability; this is again for you and
		other developers that will be going to read your code.[br]
		[br]
		3. [b]Use TABS to indent your code[/b][br]
		...and use ONLY TABS to indent.[br]
		Tabs are better than space since most code editors allow you
		to set the tab sice and thus to have the indentation steps smaller or bigger.[br]
		This is really important since the indentation size is really a matter of personal taste.[br]
		Mixing spaces and tabs is Evil (tm), since it makes the code look really
		ugly in editors that have the tab size different than yours; in some cases the
		code gets really unreadable.[br]
		[br]
		4. [b]Use descriptive variable names[/b][br]
		Using 'foo' as variable name implies tracing its semantic the next
		time that you're going to read the code that uses it.[br]
		This is really annoying and time-consuming, especially if the project
		is getting large.[br]
		Obviously using "thisIsACounterVariable" as name for a simple counter
		is also a suicide.[br]
		A good convention on variable names can speed up writing , debugging and mantaining code.[br]
		Encoding the type of the variable in the variable name might be also a good idea,
		but this is a matter of taste; personally I feel really well with that.[br]
		Just as example, here go my fundamental convention rules for C++:[br]
		[br]
		- The type of the variable is encoded at the beginning of the variable name:[br]
		[br]
			- b prefix for the boolean varables[br]
			- i prefix for signed integers[br]
			- u prefix for unsigned integers[br]
			- f and d prefixes for floating point stuff[br]
			- sz prefix for strings (this is rather for string classes)[br]
			- ...[br]
		[br]
		- Pointers have a "p" prefix prepended[br]
		- Global variables start with a "g_" prefix[br]
		- Member variables start with a "m_" prefix[br]
		- Exception comes for local variables with obvious semantics[br]
		[br]
			- i,j,k,l for local loop counters[br]
			- "aux" and "tmp" for local obvious short-term temporary variables[br]
		[br]
		So actually by ONLY reading "g_pszQuitMessage" I know that this is a global pointer to a string variable
		containing a quit message. :)[br]
		[/p]
*/

// FIXME: #warning "FINISH THE SYNTACTIC RULES DOC"

/*
	@doc: syntactic_rules
	@type:
		language
	@keyterms:
		productions
	@title:
		Syntactic rules
	@short:
		Syntactic rules of the KVIrc scripting language
	@body:

		In the following table you can find a good part of the
		KVIrc scripting language syntactic rules.[br]
		[br]
		<entity> indicates a ENTITY THAT CAN APPEAR EXACTLY ONE TIME.[br]
		[<entity>] indicates an OPTIONAL ENTITY.[br]
		{<entity>} indicates an ENTITY THAT CAN APPEAR ONE OR MORE TIMES.[br] 
		'entity' indicates a LITERAL ENTITY: written exactly as it is.[br]
		<entity1>|<entity2> indicates mutually exclusive choices.[br]
		The mutually exclusive choices are often separated in two or more
		rules (productions), to improve readability.[br]
		[table]
		[tr]
		[td]<command buffer>[/td]
		[td][<whitespace>][<command block>]{<command buffer>}[/td]
		[/tr]
		[tr]
		[td]<command buffer>[/td]
		[td][<whitespace>][<single command>]{<command buffer>}[/td]
		[/tr]
		[tr]
		[td]<whitespace>[/td]
	    [td]{<space>|<tab>|<newline>}['\'<newline>][<whitespace>][/td]
		[/tr]
		[tr]
		[td]<space>[/td]
		[td]' '['\'<newline>][<space>] (Ascii space character)[/td]
		[/tr]
		[tr]
		[td]<tab>[/td]
		[td]'\t' (Ascii horizontal tabulation character)[/td]
		[/tr]
		[tr]
		[td]<newline>[/td]
		[td]'\n' (Ascii line feed (LF) character)[/td]
		[/tr]
		[tr]
		[td]<command block>[/td]
		[td]'{' <command buffer>[<whitespace>] '}'[/td]
		[/tr]
		[tr]
		[td]<single command>[/td]
		[td]<comment>[/td]
		[/tr]
		[tr]
		[td]<single command>[/td]
		[td]<lvalue command> <command terminator>[/td]
		[/tr]
		[tr]
		[td]<single command>[/td]
		[td]<rvalue command> <command terminator>[/td]
		[/tr]
		[tr]
		[td]<comment>[/td]
		[td]'#' {<non comment terminator>} <comment terminator>[/td]
		[/tr]
		[tr]
		[td]<comment terminator>[/td]
		[td]<newline> | <end of string>[/td]
		[/tr]
		[tr]
		[td]<end of string>[/td]
		[td]No character (internally Ascii character 0)[/td]
		[/tr]
		[tr]
		[td]<command terminator>[/td]
		[td]<newline> | <end of string> | ';'[/td]
		[/tr]
		[tr]
		[td]<non comment-terminator>[/td]
		[td]Any Ascii character except <newline> and <end of string>[/td]
		[/tr]
		[tr]
		[td]<simple command>[/td]
		[td][<module name>'.']<command name>[<switch list>]{<space>}<command dependant part>[/td]
		[/tr]	
		[tr]
		[td]<lvalue command>[/td]
		[td]<variable>[<space>]<operation>[/td]
		[/tr]
		[tr]
		[td]<lvalue command>[/td]
		[td]<variable>'->'<object command>[/td]
		[/tr]
		[tr]
		[td]<lvalue command>[/td]
		[td]<identifier>'->'<object command>[/td]
		[/tr]
		[tr]
		[td]<operation>[/td]
		[td]<one op operator>[/td]
		[/tr]
	    [tr]
		[td]<operation>[/td]
		[td]<two op operator>[<space>]<param string>[/td]
		[/tr]
		[tr]
		[td]<switch list>[/td]
		[td]{<space>}'-'<alpha char>[{<space>}'='<single parameter>][<switch list>][/td]
		[/tr]
		[tr]
		[td]<command name>[/td]
		[td]<alphanumeric char>{<alphanumeric char>}[/td]
		[/tr]
		[tr]
		[td]<module name>[/td]
		[td]<alphanumeric char>{<alphanumeric char>}[/td]
		[/tr]
		[tr]
		[td]<alphanumeric char>[/td]
		[td]Ascii characters 'A' to 'Z' , 'a' to 'z' , '0' to '9' and '_'[/td]
		[/tr]
		[tr]
		[td]<variable>[/td]
		[td]<global variable> | <local variable>[/td]
		[/tr]
		[tr]
		[td]<global variable>[/td]
		[td]'%' <uppercase letter> [<alphanumeric char>]['['<param string>']'][/td]
		[/tr]
		[tr]
		[td]<local variable>[/td]
		[td]'%' <lowercase letter> [<alphanumeric char>]['['<param string>']'][/td]
		[/tr]
		[tr]
		[td]<param string>[/td]
		[td][<single parameter>][<space>[<param string>]][/td]
		[/tr]
		[tr]
		[td]<single parameter>[/td]
		[td]<variable> | <identifier> | <nonterminator token> | <string>[/td]
		[/tr]
		[tr]
		[td]<nonterminator token>[/td]
		[td]<nonterminator char>['\'<newline><nonterminator char>][/td]
		[/tr]
		[tr]
		[td]<nonterminator char>[/td]
		[td]Any ascii character except <space> and <command terminator>[/td]
		[/tr]
		[tr]
		[td]<command dependant part>[/td]
		[td][b]Production in each command help page[/b][/td]
		[/tr]
		[/table]
		To be continued...
*/

void KviUserParser::printError(KviCommand *c,bool bCallStackOnly,int * stackFrames)
{

	// FIXME: #warning "PRINT PARAMETERS ?"
	if(!bCallStackOnly && (c->m_iError != KviError_success))
	{
		c->m_pWnd->output(KVI_OUT_PARSERERROR,__tr("Error in command: %c%c%s"),
			KVI_TEXT_BOLD,KVI_TEXT_UNDERLINE,kvi_getErrorString(c->m_iError));
		if(c->m_szErrDetail.hasData())
		{
			c->m_pWnd->output(KVI_OUT_PARSERERROR,__tr("Error token: %c%s"),
				KVI_TEXT_BOLD,c->m_szErrDetail.ptr());
		}
		int nLine = 1;
		const char * aux = c->m_szCmdBuffer.ptr();
		const char * linePtr = aux;
		// find the line in that the parsing stopped
		while(aux != c->m_ptr)
		{
			if(*aux == '\n'){
				nLine++;
				linePtr = ++aux;
			} else ++aux;
		}
		int chIdx = c->m_ptr - linePtr;
		KviStr tmp = linePtr;

		tmp.insert(chIdx,KVI_TEXT_REVERSE);

		chIdx++;

		if(chIdx < (tmp.len() - 1))
		{
			tmp.insert(chIdx + 1,KVI_TEXT_RESET);
		} else {
			tmp.append(" "); // need a trailing space to show the char (we were pointing to the null terminator!)
			tmp.append(KVI_TEXT_RESET);
		}

		if(tmp.len() > 50)
		{
			if(chIdx > 30)
			{
				tmp.cutLeft(chIdx - 25);
				tmp.prepend("...");
			}
		}
		bool bAddPoints = false;
		int theLen = tmp.len();
		tmp.cutFromFirst('\n');
		if(theLen < tmp.len())bAddPoints = true;
		if(tmp.len() > 50)
		{
			tmp.setLen(50);
			bAddPoints = true;
		}
		if(bAddPoints)tmp.append("...");

		c->m_pWnd->output(KVI_OUT_PARSERERROR,__tr("Parsing stopped at %cline %d , character %d"),
			KVI_TEXT_BOLD,nLine,chIdx);
		c->m_pWnd->output(KVI_OUT_PARSERERROR,__tr("   %s"),tmp.ptr());
		if(c->scopeObject())
		{
			if(g_pScriptObjectController->objectExists(c->scopeObject()))
			{
				c->m_pWnd->output(KVI_OUT_PARSERERROR,__tr("Object scope: name (%s), class (%s)"),
					c->scopeObject()->name(),c->scopeObject()->getClass()->name());
			} else {
				c->m_pWnd->outputNoFmt(KVI_OUT_PARSERERROR,__tr("Object scope: deleted object"));
			}
		}
		if(c->thisPointer())
		{
			if(g_pScriptObjectController->objectExists(c->thisPointer()))
			{
				c->m_pWnd->output(KVI_OUT_PARSERERROR,__tr("This pointer: name (%s), class (%s)"),
					c->thisPointer()->name(),c->thisPointer()->getClass()->name());
			} else {
				c->m_pWnd->outputNoFmt(KVI_OUT_PARSERERROR,__tr("This pointer: deleted object"));
			}
		}
		c->m_pWnd->output(KVI_OUT_PARSERERROR,__tr("Internal call stack:"));
	}

	int parentCount = 0;
	KviCommand * par = c->parent();
	while(par)
	{
		parentCount++;
		par = par->parent();
	}

	int i = stackFrames ? *stackFrames : 0;
	for(const char * ch=c->m_pContextStack->last();ch;ch=c->m_pContextStack->prev())
	{
		i++;
		if(i == 1)c->m_pWnd->output(KVI_OUT_PARSERERROR,"   L%c%d::%d: %s",
			KVI_TEXT_BOLD,parentCount,i,ch);
		else c->m_pWnd->output(KVI_OUT_PARSERERROR,"   L%d::%d: %s",
			parentCount,i,ch);
	}

	if(c->parent()){
		if(stackFrames){
			*stackFrames += i;
			printError(c->parent(),true,stackFrames);
		} else {
			int iStackCount = i;
			printError(c->parent(),true,&iStackCount);
		}
	}

}

bool KviUserParser::parseUserCommand(KviStr &buffer,KviWindow *wnd)
{
	char * c = buffer.ptr();
	while((*c == ' ') || (*c == '\n'))c++;
	if(!*c)return false;

	if(*c == '\\')c++;
	else {
		if(*c == '/')return parseCommandBuffer(++c,wnd);
	}

	switch(wnd->type())
	{
		case KVI_WINDOW_TYPE_CONSOLE:
			if(!((KviConsole *)wnd)->socket()->sendData(c))
			{
				wnd->output(KVI_OUT_SYSTEMERROR,__tr("Not connected to server"));
				return false;
			} else {
				wnd->output(KVI_OUT_RAW,__tr("[RAW]: %s"),c);
				return true;
			}
		break;
		case KVI_WINDOW_TYPE_CHANNEL:
		case KVI_WINDOW_TYPE_QUERY:
			if (wnd->console()->userIsAway()) 
				if(KVI_OPTION_BOOL(KviOption_boolExitAwayOnInput)) 
						g_pUserParser->parseCommandBuffer("back", wnd->console());
			wnd->ownMessage(c);
		break;
		case KVI_WINDOW_TYPE_DCCCHAT:
			wnd->ownMessage(c);
		break;
		default: return parseCommandBuffer(c,wnd); break;
	}
	return true;
}


bool KviUserParser::parseCommandBuffer(char * buffer,KviWindow *wnd,KviParameterList * params)
{
	KviCommand cmd(buffer,wnd);
	if(params)cmd.setParams(params);
	if(!parseCommand(&cmd))
	{
		if(cmd.hasError())
		{
			printError(&cmd);
			return false;
		}
	}
	return true;
}

bool KviUserParser::parseCommand(KviCommand *c)
{
	ENTER_CONTEXT(c,"parseCommand");

	// This one parses a whole command buffer executing it
	// Returns false in case of execution stopped
	// If cmd->err is non-zero in that moment , it
	// means that an error occured , otherwise the execution was simply halted
	c->skipWhiteSpace();
	while(*(c->m_ptr))
	{
		if(*(c->m_ptr) == '{')
		{
			if(!parseCommandBlock(c))return false;
			// Here c->m_ptr points to the char immediately after the closing brace
		} else {
			if(!parseSingleCommand(c))return false;
			// Here c->m_ptr points to the char immediately after the separator...
		}
		c->skipWhiteSpace();
	}
	// All ok.
	return c->leaveContext();
}

bool KviUserParser::parseCommandBlock(KviCommand *c)
{
	ENTER_CONTEXT(c,"parseCommandBlock");

	__range_valid(*(c->m_ptr) == '{');
	++(c->m_ptr);
	c->skipWhiteSpace();
	while(*(c->m_ptr)){
		switch(*(c->m_ptr)){
			case '}': ++(c->m_ptr);             return c->leaveContext(); break;
			case '{': if(!parseCommandBlock(c)) return false;             break;
			default : if(!parseSingleCommand(c))return false;             break;
		}
		c->skipWhiteSpace();
	}
	// We should newer reach this point
	return c->error(KviError_missingClosingBrace);
}

/*
	@doc: command_rebinding
	@type:
		language
	@keyterms:
		standard -r switch
	@title:
		Command rebinding
	@short:
		Rebinding a command to a specified window
	@body:
		Every command that KVIrc executes is bound to a specific window.
		For user-typed commands it is the window that the command was typed
		in, for events it is the window that the event relates to, etc...[br]
		It is possible to change "on-the-fly" the binding of a single command
		to a specific window by using the standard -r switch.[br]
		The syntax is:[br]
		[b]command -r=[doc:window_naming_conventions]<window_id>[/doc] <parameters>[/b][br]
		The <window_id> is the window identifier as described in the
		[doc:window_naming_conventions]standard window naming conventions document[/doc].
		Please note that if the -r switch is specified, all the command parameters
		are "resolved" by using the new window.[br]
		If the window can not be found, no rebinding is made: so if you
		rely on the rebinding first make sure that the provided window is existing
		and is the one that you're looking for.[br]
		For example, [fnc]$window[/fnc] will return the new window id.[br]
*/

// FIXME: #warning "ADD EXAMPLES OF WINDOW REBINDING"

bool KviUserParser::parseSingleCommand(KviCommand *c)
{
	ENTER_CONTEXT(c,"parseSingleCommand");

	// Ok , we first check for comments
	if(*(c->m_ptr) == '#')
	{
		skipComment(c);
		return c->leaveContext();
	}

	// not a comment...look for lValue stuff , variables and function calls
	if((*(c->m_ptr) == '%') || (*(c->m_ptr) == '$'))
	{
		return (parseLValueCommand(c) ? c->leaveContext() : false);
	}

	// empty command ?
	if(*(c->m_ptr) == ';'){
		++(c->m_ptr);
		return c->leaveContext();
	}

	// ok...it must be a full command at this point
	char * end = c->m_ptr;
	end++;
	while(isalnum(*end) || (*end == '_'))end++;

	KviStr command;

	// check if it is a module name
	if(*end == '.')
	{
		// a module command
		KviStr module(c->m_ptr,end);
		c->m_ptr = ++end;

		KviModule * pModule = g_pModuleManager->getModule(module.ptr());
		if(!pModule)return c->error(KviError_errorInLoadingModule,__tr("%s (module %s)"),g_pModuleManager->lastError().ptr(),module.ptr());

		while(isalnum(*end) || (*end == '_') || (*end == '.'))end++;
		command.extractFromString(c->m_ptr,end); // FIXME: make the module dict case sensitive and add KviStr::extractStringToUpper() ?
		//command.toUpper();

		c->m_ptr = end;

		KviModuleCommandParseProc * mProc = pModule->findCommandParseProc(command.ptr());
		if(!mProc)
		{
			mProc = pModule->genericCommandParseProc();
			if(!mProc)return c->error(KviError_noSuchModuleCommand,__tr("Module '%s', Command '%s'"),module.ptr(),command.ptr());
		}

		if(!extractSwitches(c))return false;

		KviWindow *pOld = 0;

		if(c->hasSwitch('r'))
		{
			// temp rebinding requested
			KviStr winId;
			if(c->getSwitchValue('r',winId))
			{
				KviWindow * pAux = g_pApp->findWindow(winId.ptr());
				if(pAux){
					pOld = c->window();
					c->rebindToWindow(pAux);
				} else c->warning(__tr("Can't rebind the command: window with id %s not found"),winId.ptr());
			} else c->warning(__tr("Can't rebind the command: missing window identifier after the -r switch"));
		}

		c->setCurrentEntity(command.ptr());
		if(! (*mProc)(pModule,c) )
		{
			if(pOld)c->rebindToWindow(pOld);
			return false;
		} else {
			if(pOld)c->rebindToWindow(pOld);
			return c->leaveContext();
		}
	
	}

	// not a module name... it is a complete command

	command.extractFromString(c->m_ptr,end);
	command.toUpper();

	// lookup the builtin ones
	KviCommandParseProc * proc = m_pCommandExecDict->find(command.ptr());

	if(proc)
	{
		// builtin command
		c->m_ptr = end;
		if(!extractSwitches(c))return false;

		KviWindow *pOld = 0;

		if(c->hasSwitch('r'))
		{
			// temp rebinding requested
			KviStr winId;
			if(c->getSwitchValue('r',winId))
			{
				KviWindow * pAux = g_pApp->findWindow(winId.ptr());
				if(pAux){
					pOld = c->window();
					c->rebindToWindow(pAux);
				} else c->warning(__tr("Can't rebind the command: window with id %s not found"),winId.ptr());
			} else c->warning(__tr("Can't rebind the command: missing window identifier after the -r switch"));
		}

		c->setCurrentEntity(command.ptr());
		if(! ((this->*(proc->proc))(c)) )
		{
			if(pOld)c->rebindToWindow(pOld);
			return false;
		} else {
			if(pOld)c->rebindToWindow(pOld);
			return c->leaveContext();
		}
	}

	// not a builtin command: alias ?

	KviAlias * a = g_pAliasManager->lookupAlias(command.ptr());
	if(a)
	{
		// Yes...an alias
		// We can easily check for infinite recursion here
		if(KVI_OPTION_BOOL(KviOption_boolLimitAliasRecursion))
		{
			if(a->lock() > 1024)
			{
				a->totalUnlock();
				return c->error(KviError_recursionTooDeep);
			}
		} else {
			// In fact the locking mechanism will stop to work if we exceed 2^32 iterations
			// assuming that unsigned long is 32 bits wide. So we still try to
			// avoid such monster recursions. Anyway, we will prolly never get it
			// to be executed: we will get a stack overflow first (2^32 quite large stack frames
			// can't live in our addressing space: the stack pointer would overflow!)
			// So this check is not a real limit (at least on 32 bit systems)
			if(a->lock() > 0xFFFFFFFE)
			{
				a->totalUnlock();
				return c->error(KviError_recursionTooDeep);
			}
		}

		c->m_ptr = end;

		// extract the parameters...
		KviStr parambuffer;
		if(!parseCmdFinalPart(c,parambuffer))return false;
		KviParameterList * l = new KviParameterList(parambuffer.ptr());

		// We want nice output in case of errors...
		ENTER_CONTEXT(c,a->name());

		// child command buffer
		KviCommand cmd(a->dataBuffer(),c->window(),c);
		cmd.setParams(l,false); // do not transfer param ownership

		// parse it
		if(!parseCommand(&cmd))
		{
			if(cmd.hasError())
			{
				printError(&cmd);
				delete l;
				a->unlock();
				return false; // this is a "halt" for us
			}
		}

		// propagate the return value
		c->m_szRetBuffer= cmd.m_szRetBuffer;

		delete l;

		c->leaveContext();
		a->unlock();

		return c->leaveContext();
	}


	// not an alias...
	// check if the user wants to send it as "raw" to server
	if(KVI_OPTION_BOOL(KviOption_boolSendUnknownCommandsAsRaw))
	{
		if(c->window()->console())
		{
			if(c->window()->console()->isConnected())
			{
				KviStr parbuf;
				c->m_ptr = end;
				if(!parseCmdFinalPart(c,parbuf))return false;
				c->window()->console()->socket()->sendFmtData("%s %s",command.ptr(),parbuf.ptr());
				return c->leaveContext();
			}
		}
	}

	// nope...trigger an error: this is a syntactic one...better to stop
	KviStr tmp(c->m_ptr,end);
	return c->error(KviError_unknownCommand,tmp.ptr());
}


void KviUserParser::completeModuleCommand(const KviStr &mod,const KviStr &cmd,QList<KviStr> * matches)
{
	KviModule * m = g_pModuleManager->getModule(mod.ptr());
	if(!m)return;
	m->completeCommand(cmd,matches);
}

void KviUserParser::completeModuleFunction(const KviStr &mod,const KviStr &cmd,QList<KviStr> * matches)
{
	KviModule * m = g_pModuleManager->getModule(mod.ptr());
	if(!m)return;
	m->completeFunction(cmd,matches);
}

void KviUserParser::completeCommand(const KviStr &cmd,QList<KviStr> * matches)
{
	// get a list of commands that match
	if(cmd.contains('.'))
	{
		// this is a module name!
		// load it!
		KviStr szMod = cmd;
		szMod.cutFromFirst('.');
		KviStr szCmd = cmd;
		szCmd.cutToFirst('.');
		completeModuleCommand(szMod,szCmd,matches);
		return;
	}

	QAsciiDictIterator<KviCommandParseProc> it(*m_pCommandExecDict);

	while(it.current())
	{
		if(kvi_strEqualCIN(cmd.ptr(),it.currentKey(),cmd.len()))
		{
			KviStr * x = new KviStr(it.currentKey());
			x->toLower();
			matches->append(x);
		}
		++it;
	}

	// now match the modules
	QList<KviStr> tmp;
	tmp.setAutoDelete(true);
	g_pModuleManager->completeModuleNames(cmd,&tmp);
	for(KviStr * s = tmp.first();s;s = tmp.next())completeModuleCommand(*s,KviStr::emptyString(),matches);

	g_pAliasManager->completeCommand(cmd,matches);

	// FIXME: complete aliases too!
}

void KviUserParser::completeFunction(const KviStr &cmd,KviPtrList<KviStr> * matches)
{
	if(cmd.contains('.'))
	{
		// this is a module name!
		// load it!
		KviStr szMod = cmd;
		szMod.cutFromFirst('.');
		KviStr szCmd = cmd;
		szCmd.cutToFirst('.');
		completeModuleFunction(szMod,szCmd,matches);
		return;
	}

	QAsciiDictIterator<KviFunctionParseProc> it(*m_pFunctionDict);

	while(it.current())
	{
		if(kvi_strEqualCIN(cmd.ptr(),it.currentKey(),cmd.len()))
		{
			KviStr * x = new KviStr(it.currentKey());
			x->toLower();
			matches->append(x);
		}
		++it;
	}

	// now match the modules
	QList<KviStr> tmp;
	tmp.setAutoDelete(true);
	g_pModuleManager->completeModuleNames(cmd,&tmp);
	for(KviStr * s = tmp.first();s;s = tmp.next())completeModuleFunction(*s,KviStr::emptyString(),matches);

	g_pAliasManager->completeCommand(cmd,matches);
}

bool KviUserParser::parseCallbackCommand(KviCommand * c,KviParameterList * parms,KviStr * cmd)
{
	ENTER_CONTEXT(c,"parseCallbackCommand");

	c->skipWhiteSpace();
	if(*(c->m_ptr) != '(')return c->error(KviError_openParenthesisExpected);

	if(!extractFunctionParameters(c,parms))return false;

	c->skipWhiteSpace();

	if(*(c->m_ptr) != '{')return c->error(KviError_openBraceExpected);
	const char * aux = c->m_ptr;
	if(!skipCommand(c))return false;

	cmd->extractFromString(aux,c->m_ptr);
	// Don't need real formatting here: it's just an internal buffer
	cmd->cutLeft(1);
	cmd->cutRight(1);
	cmd->stripWhiteSpace();

	return c->leaveContext();
}

bool KviUserParser::triggerEvent(int index,KviWindow *wnd,KviParameterList * params,bool bTransferOwnership,bool bIsRaw)
{
	//__range_valid((index >= 0) && (index < KVI_NUM_SCRIPT_EVENTS));
	bool bHaltCalled = false;

	KviPtrList<KviEventHandler> * l = bIsRaw ? g_pEventManager->m_rawNumericEventTable[index] : g_pEventManager->m_eventTable[index].handlers;

	if(l)
	{
		if(params)params->init();
		for(KviEventHandler *s = l->first();s; s = l->next())
		{
			if(s->type() == KviEventHandler::Script)
			{
				if(((KviScriptEventHandler *)s)->enabled())
				{
					KviCommand cmd(((KviScriptEventHandler *)s)->codeString(),wnd);
					if(params)cmd.setParams(params,false); // do not transfer param ownership
					if(!parseCommand(&cmd))
					{
						if(cmd.hasError())
						{
							printError(&cmd);
							if(bIsRaw)
							{
								KviStr tmp(KviStr::Format,"RAW%d",index);
								wnd->output(KVI_OUT_PARSERERROR,
									__tr("Error triggered from raw event handler %c%s::%s"),KVI_TEXT_BOLD,
									tmp.ptr(),((KviScriptEventHandler *)s)->name());
							} else {
								wnd->output(KVI_OUT_PARSERERROR,
									__tr("Error triggered from event handler %c%s::%s"),KVI_TEXT_BOLD,
									g_pEventManager->m_eventTable[index].szName,((KviScriptEventHandler *)s)->name());
							}
							if(KVI_OPTION_BOOL(KviOption_boolDisableBrokenEventHandlers))
							{
								((KviScriptEventHandler *)s)->setEnabled(false);
								if(bIsRaw)
								{
									KviStr tmp(KviStr::Format,"RAW%d",index);
									wnd->output(KVI_OUT_PARSERERROR,
										__tr("Raw event handler %s::%s is broken: disabling"),
										tmp.ptr(),((KviScriptEventHandler *)s)->name());
								} else {
									wnd->output(KVI_OUT_PARSERERROR,
										__tr("Event handler %s::%s is broken: disabling"),
										g_pEventManager->m_eventTable[index].szName,((KviScriptEventHandler *)s)->name());
								}
							}
						} else {
							if(cmd.haltEncountered())bHaltCalled = true;
						}
					}
				}
			} else {
				// a module handler
				if(!((KviModuleEventHandler *)s)->proc()(((KviModuleEventHandler *)s)->module(),wnd,params))bHaltCalled = true; // halt-like
			}
		}
	}

	if(params && bTransferOwnership)delete params;
	return bHaltCalled;
}

bool KviUserParser::extractSwitches(KviCommand *c)
{
	ENTER_CONTEXT(c,"extractSwitches");
	// This one 
	c->clearSwitchList();
	c->skipSpace();
	while(*(c->m_ptr) == '-'){
		if(!extractSingleSwitch(c))return false;
		c->skipSpace();
	}
	return c->leaveContext();
}


#ifdef JMP
#undef JMP
#endif
#define JMP 1
static char command_parameter_parsing_char_table[256]=
{
	//	000 001 002 003 004 005 006 007   008 009 010 011 012 013 014 015 
	//	NUL SOH STX ETX EOT ENQ ACK BEL   BS  HT  LF  VT  FF  CR  SO  SI
		JMP,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,JMP,JMP,0  ,0  ,0  ,0  ,0  ,
	//	016 017 018 019 020 021 022 023   024 025 026 027 028 029 030 031 
	//	DLE DC1 DC2 DC3 DC4 NAK SYN ETB   CAN EM  SUB ESC FS  GS  RS  US
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	032 033 034 035 036 037 038 039   040 041 042 043 044 045 046 047 
	//	    !   "   #   $   %   &   '     (   )   *   +   ,   -   .   /   
		JMP,0  ,JMP,0  ,JMP,JMP,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	048 049 050 051 052 053 054 055   056 057 058 059 060 061 062 063 
	//	0   1   2   3   4   5   6   7     8   9   :   ;   <   =   >   ?   
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,JMP,0  ,0  ,0  ,0  ,
	//	064 065 066 067 068 069 070 071   072 073 074 075 076 077 078 079 
	//	@   A   B   C   D   E   F   G     H   I   J   K   L   M   N   O   
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	080 081 082 083 084 085 086 087   088 089 090 091 092 093 094 095 
	//	P   Q   R   S   T   U   V   W     X   Y   Z   [   \   ]   ^   _   
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,JMP,0  ,0  ,0  ,
	//	096 097 098 099 100 101 102 103   104 105 106 107 108 109 110 111 
	//	`   a   b   c   d   e   f   g     h   i   j   k   l   m   n   o   
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	112 113 114 115 116 117 118 119   120 121 122 123 124 125 126 127 
	//	p   q   r   s   t   u   v   w     x   y   z   {   |   }   ~      
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	128 129 130 131 132 133 134 135   136 137 138 139 140 141 142 143 
	//	                                                  
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	144 145 146 147 148 149 150 151   152 153 154 155 156 157 158 159 
	//	                                                  
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	160 161 162 163 164 165 166 167   168 169 170 171 172 173 174 175 
	//	                                                  
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	176 177 178 179 180 181 182 183   184 185 186 187 188 189 190 191 
	//	                                                  
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	192 193 194 195 196 197 198 199   200 201 202 203 204 205 206 207 
	//	                                                  
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	208 209 210 211 212 213 214 215   216 217 218 219 220 221 222 223 
	//	                                                  
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	224 225 226 227 228 229 230 231   232 233 234 235 236 237 238 239 
	//	                                                  
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	240 241 242 243 244 245 246 247   248 249 250 251 252 253 254 255 
	//	                                                  
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0
};
#undef JMP



bool KviUserParser::extractSingleSwitch(KviCommand *c)
{
	ENTER_CONTEXT(c,"extractSingleSwitch");

	__range_valid(*(c->m_ptr) == '-');

	++(c->m_ptr); // skip the - char

	// now a LETTER is required
	if( (!(*(c->m_ptr))) || (!isalpha(*(c->m_ptr))) )
	{
		return c->error(KviError_switchDashWithoutSwitchLetter);
	}

	KviStr * sw = new KviStr();
	sw->append(*(c->m_ptr));

	++(c->m_ptr); // skip the switch letter

	c->skipSpace();

	// now if we have a '=' , a token follows the switch

	if(*(c->m_ptr) == '=')
	{
		// a token follows the switch
		++(c->m_ptr);
		c->skipSpace();
		sw->append('=');

		char *aux_ptr = c->m_ptr;

		bool bInString = false;


		for(;;){
			// Now skip all the non interesting chars
			/*
			while(*aux_ptr && (*aux_ptr!='\\') && (*aux_ptr!='\n') &&
					(*aux_ptr!=' ') && (*aux_ptr!='\t') && (*aux_ptr!=';') &&
					(*aux_ptr!='$') && (*aux_ptr!='\"') &&
					(*aux_ptr!='%'))aux_ptr++;
			*/
			while(!command_parameter_parsing_char_table[*aux_ptr])aux_ptr++;
			// Interesting char
			switch(*aux_ptr){
				case ' ' :
				case '\t':
				case ';' :
					if(!bInString){ // End of the token...append the last block to the buffer and return
						sw->append(c->m_ptr,aux_ptr - c->m_ptr);
						c->m_ptr = aux_ptr;
						c->m_pSwitchList->append(sw);
						return c->leaveContext();
					} else while((*aux_ptr == ' ')||(*aux_ptr == '\t')||
								(*aux_ptr == ';'))aux_ptr++; //In string...must skip it
				break;
				case '\n': // end of command
				case '\0': // end of command buffer...append the last block to the buffer and return
					if(bInString){
						// unsecaped newline or end of data
						delete sw;
						return c->error(KviError_unexpectedEndInString);
					} // else it is the end of the command
					sw->append(c->m_ptr,aux_ptr - c->m_ptr);
					c->m_ptr = aux_ptr;
					c->m_pSwitchList->append(sw);
					return c->leaveContext();
				break;
				case '\"': // beginning or end of a string
					sw->append(c->m_ptr,aux_ptr - c->m_ptr);
					bInString = !bInString;
					c->m_ptr = ++aux_ptr;
				break;
				case '%': //variable: append the last block to the buffer and process the var
					sw->append(c->m_ptr,aux_ptr - c->m_ptr);
					c->m_ptr = aux_ptr;
					if(!parseVariable(c,*sw)){
						delete sw;
						return false;
					}
					aux_ptr = c->m_ptr;
				break;
				case '$': //system identifier: append the last block to the buffer and process the ident
					sw->append(c->m_ptr,aux_ptr - c->m_ptr);
					c->m_ptr = aux_ptr;
					if(!parseIdentifier(c,*sw)){
						delete sw;
						return false;
					}
					aux_ptr = c->m_ptr;
				break;
				case '\\': //escape character: append the last block to the processed buffer...
					sw->append(c->m_ptr,aux_ptr - c->m_ptr);
					c->m_ptr = ++aux_ptr;
					switch(*aux_ptr){
						case '\0':
							if(bInString){
								delete sw;
								return c->error(KviError_unexpectedEndInString);
							}
							c->m_pSwitchList->append(sw);
							return c->leaveContext();
						break;
						case '\n':  //escaped newline
							c->m_ptr = ++aux_ptr; //skip it
							c->skipSpace();       //skip leading spaces
							aux_ptr = c->m_ptr;   //continue
						break;
					case 'r':
						if(bInString)sw->append('\r');
						else sw->append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
					case 'n':
						if(bInString)sw->append('\n');
						else sw->append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
						default: // Must be copied to the buffer without modification
							sw->append(*aux_ptr);
							c->m_ptr = ++aux_ptr;
						break;
					}
				break;
			}
		}
	}

	// not '='
	c->m_pSwitchList->append(sw);

	return c->leaveContext();
}

bool KviUserParser::parseCmdSingleToken(KviCommand *c,KviStr &buffer)
{
	ENTER_CONTEXT(c,"parseCmdSingleToken");

	// This one must be called when c->m_ptr points
	// somewhere in a command buffer.
	// It extracts the first token encountered and
	// preprocesses it (subst variables , etc...)
	// then it appends everything to the c->m_buffer.
	// The parsing is stopped by a '\n',' ',';' or '\0'
	// Note that this function does nothing if c->m_ptr
	// points to the end of a command...

	// Skip leading spaces...even the escaped ones
	c->skipSpace();

	char *aux_ptr = c->m_ptr;

	bool bInString = false;

	for(;;){
		// Now skip all the non interesting chars
		/*
		while(*aux_ptr && (*aux_ptr!='\\') && (*aux_ptr!='\n') &&
				(*aux_ptr!=' ') && (*aux_ptr!='\t') && (*aux_ptr!=';') &&
				(*aux_ptr!='$') && (*aux_ptr!='\"') &&
				(*aux_ptr!='%'))aux_ptr++;
		*/
		while(!command_parameter_parsing_char_table[*aux_ptr])aux_ptr++;

		// Interesting char
		switch(*aux_ptr){
			case ' ' :
			case '\t':
			case ';' :
				if(!bInString){ // End of the token...append the last block to the buffer and return
					buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
					c->m_ptr = aux_ptr;
					return c->leaveContext();
				} else while((*aux_ptr == ' ')||(*aux_ptr == '\t')||
							(*aux_ptr == ';'))aux_ptr++; //In string...must skip it
			break;
			case '\n': // end of command
			case '\0': // end of command buffer...append the last block to the buffer and return
				if(bInString){
					// unsecaped newline or end of data
					return c->error(KviError_unexpectedEndInString);
				} // else it is the end of the command
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				return c->leaveContext();
			break;
			case '\"': // beginning or end of a string
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				bInString = !bInString;
				c->m_ptr = ++aux_ptr;
			break;
			case '%': //variable: append the last block to the buffer and process the var
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				if(!parseVariable(c,buffer))return false;
				aux_ptr = c->m_ptr;
			break;
			case '$': //system identifier: append the last block to the buffer and process the ident
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				if(!parseIdentifier(c,buffer))return false;
				aux_ptr = c->m_ptr;
			break;
			case '\\': //escape character: append the last block to the processed buffer...
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = ++aux_ptr;
				switch(*aux_ptr){
					case '\0':
						if(bInString)return c->error(KviError_unexpectedEndInString);
						return c->leaveContext();
					break;
					case '\n':  //escaped newline
						c->m_ptr = ++aux_ptr; //skip it
						c->skipSpace();       //skip leading spaces
						aux_ptr = c->m_ptr;   //continue
					break;
					case 'r':
						if(bInString)buffer.append('\r');
						else buffer.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
					case 'n':
						if(bInString)buffer.append('\n');
						else buffer.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
					default: // Must be copied to the buffer without modification
						buffer.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
				}
			break;
		}
	}
	// Newer arrive here...
	return c->error(KviError_internalError);
}


bool KviUserParser::parseCmdFinalPart(KviCommand *c,KviStr &buffer)
{
	ENTER_CONTEXT(c,"parseCmdFinalPart");

	// Preprocesses the final part of a standard command...
	// It is stopped by a newline , a ';' or the end of the buffer.
	// This one extracts the command parameters , evaluates it , 
	// and appends everything to cmd->buffer.
	// skip spaces...but keep the terminators!!!

	c->skipSpace();
	char *aux_ptr = c->m_ptr;

	bool bInString = false;

	for(;;){
		// Now skip all the non interesting chars
		/*
		while(*aux_ptr && (*aux_ptr!='\\') && (*aux_ptr!='\n') &&
				(*aux_ptr!=' ') && (*aux_ptr!='\t') && (*aux_ptr!=';') &&
				(*aux_ptr!='$') && (*aux_ptr!='\"') &&
				(*aux_ptr!='%'))aux_ptr++;
		*/

		while(!command_parameter_parsing_char_table[*aux_ptr])aux_ptr++;


		// Interesting char
		switch(*aux_ptr)
		{
			case '\0': // end of command buffer...
			case '\n':
				if(bInString){
					// ops....end of buffer while looking for matching '"'
					return c->error(KviError_unexpectedEndInString);
				}
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				if(*aux_ptr == '\n')aux_ptr++; // point after the semicolon
				c->m_ptr = aux_ptr;
				return c->leaveContext();
			break;
			case '\"': // beginning or end of a string
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				bInString = !bInString;
				c->m_ptr = ++aux_ptr; //don't forget to point to next char
			break;
			case ' ': // Spaces...need to reduce it to one...
			case '\t':
				if(!bInString)
				{
					// append the last part
					buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
					// skip the other spaces including escaped ones
					c->m_ptr = aux_ptr;
					c->skipSpace();
					aux_ptr = c->m_ptr;
					if((*aux_ptr != '\0') && (*aux_ptr != '\n') &&
						(*aux_ptr != ';'))buffer.append(' ');

				} else while((*aux_ptr == ' ')||(*aux_ptr == '\t'))aux_ptr++;
			break;
			case ';' :
				if(!bInString){ // end of command...
					buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
					c->m_ptr = ++aux_ptr; //don't forget to point to next char
					return c->leaveContext();
				} else ++aux_ptr; // just skip it
			break;
			case '%': //variable
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				if(!parseVariable(c,buffer))return false;
				aux_ptr = c->m_ptr;
			break;
			case '$': //system identifier
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				if(!parseIdentifier(c,buffer))return false;
				aux_ptr = c->m_ptr;
			break;
			case '\\': //escape character: append the last block to the processed buffer
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = ++aux_ptr;
				switch(*aux_ptr){
					case '\0':
						if(bInString)return c->error(KviError_unexpectedEndInString);
						else return c->leaveContext();
					break; //escaped nothing...
					case '\n':  //escaped newline
						c->m_ptr = ++aux_ptr; //skip it
						c->skipSpace();       //skip leading spaces
						aux_ptr = c->m_ptr;   //continue
					break;
					case 'r':
						if(bInString)buffer.append('\r');
						else buffer.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
					case 'n':
						if(bInString)buffer.append('\n');
						else buffer.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
					default: // Must be copied to the buffer without modification
						buffer.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
				}
			break;
		}
	}
	return c->leaveContext();
}


bool KviUserParser::parseCmdUpTo(KviCommand *c,KviStr &buffer,char terminator)
{
	ENTER_CONTEXT(c,"parseCmdUpTo");

	// Preprocesses the final part of a standard command...
	// It is stopped by a newline , a ';' the end of the buffer or the terminator char.
	// This one extracts the command parameters , evaluates it , 
	// and appends everything to cmd->buffer.
	// skip spaces...but keep the terminators!!!

	c->skipSpace();
	char *aux_ptr = c->m_ptr;
	bool bInString = false;

	for(;;){
		// Now skip all the non interesting chars
		while(!command_parameter_parsing_char_table[*aux_ptr] && (*aux_ptr != terminator))aux_ptr++;

		if(*aux_ptr == terminator)
		{
			if(!bInString)
			{ // end of command...
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = ++aux_ptr; //don't forget to point to next char
				return c->leaveContext();
			} else ++aux_ptr; // just skip it

		} else {
			// Interesting char
			switch(*aux_ptr)
			{
				case '\0': // end of command buffer...
				case '\n':
					if(bInString){
						// ops....end of buffer while looking for matching '"'
						return c->error(KviError_unexpectedEndInString);
					}
					buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
					if(*aux_ptr == '\n')aux_ptr++; // point after the semicolon
					c->m_ptr = aux_ptr;
					return c->leaveContext();
				break;
				case '\"': // beginning or end of a string
					buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
					bInString = !bInString;
					c->m_ptr = ++aux_ptr; //don't forget to point to next char
				break;
				case ' ': // Spaces...need to reduce it to one...
				case '\t':
					if(!bInString)
					{
						// append the last part
						buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
						// skip the other spaces including escaped ones
						c->m_ptr = aux_ptr;
						c->skipSpace();
						aux_ptr = c->m_ptr;
						if((*aux_ptr != '\0') && (*aux_ptr != '\n') &&
							(*aux_ptr != ';'))buffer.append(' ');
	
					} else while((*aux_ptr == ' ')||(*aux_ptr == '\t'))aux_ptr++;
				break;
				case ';' :
					if(!bInString){ // end of command...
						buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
						c->m_ptr = ++aux_ptr; //don't forget to point to next char
						return c->leaveContext();
					} else ++aux_ptr; // just skip it
				break;
				case '%': //variable
					buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
					c->m_ptr = aux_ptr;
					if(!parseVariable(c,buffer))return false;
					aux_ptr = c->m_ptr;
				break;
				case '$': //system identifier
					buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
					c->m_ptr = aux_ptr;
					if(!parseIdentifier(c,buffer))return false;
					aux_ptr = c->m_ptr;
				break;
				case '\\': //escape character: append the last block to the processed buffer
					buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
					c->m_ptr = ++aux_ptr;
					switch(*aux_ptr){
						case '\0':
							if(bInString)return c->error(KviError_unexpectedEndInString);
							else return c->leaveContext();
						break; //escaped nothing...
						case '\n':  //escaped newline
							c->m_ptr = ++aux_ptr; //skip it
							c->skipSpace();       //skip leading spaces
							aux_ptr = c->m_ptr;   //continue
						break;
						case 'r':
							if(bInString)buffer.append('\r');
							else buffer.append(*aux_ptr);
							c->m_ptr = ++aux_ptr;
						break;
						case 'n':
							if(bInString)buffer.append('\n');
							else buffer.append(*aux_ptr);
							c->m_ptr = ++aux_ptr;
						break;
						default: // Must be copied to the buffer without modification
							buffer.append(*aux_ptr);
							c->m_ptr = ++aux_ptr;
						break;
					}
				break;
			}
		}
	}
	return c->leaveContext();
}


/*
	@doc: data_types
	@type:
		language
	@keyterms:
		variable,variables,array,arrays,dictionary,dictionaries,percent sign,global variables,local variables,
		global variable,local variable,variable creation and destruction,variable evaluation,dictionary keys,
		associative arrays,hash,hashes,scalar,scalars,command scope,data types,extended scope,array reference,
		dictionary reference
	@title:
		Data types
	@short:
		KVIrc built-in data types
	@body:
		KVirc has three basic built-in data types: simple variables, arrays of
		simple variables and associative arrays of variables.[br]
		There is a fourth builtin data type: object
		that allows creation of complex structures and object-oriented programming;
		the object data type is described in another document.[br]

		[big]Variables[/big]
		The variables can be either global to the application or local
		to the command scope. The content of a variable is basically a string:
		any sequence of ASCII characters. In some contests the string may be interpreted
		as a number. All the variable names start with a percent '%' sign.
		
		[big]Global variables[/big][br]
		A global variable name is formed by a "percent" sign (%),
		followed by an uppercase letter from A to Z, followed
		by a sequence of characters in range ('a' to 'z','0' to '9','.','_').[br]
		"%INDEX","%My_nickname","%Foo","%Bar1" and "%Foo.BAR" are examples of valid
		global variable names.[br]
		A global variable is [b]global to the entire application[/b], not
		to the current frame or irc context; be sure to remember this.[br]
		You can type[br]
		[example]
			/%Hello = "Hello world!"
		[/example]
		in the commandline of any KVIrc window and then execute[br]
		[example]
			echo %Hello
		[/example]
		in the commandline of any other KVIrc window. You will see "Hello world!"
		printed in the second window view.

		[big]Local variables[/big][br]
		A local variable name is formed by a "percent" sign (%),
		followed by an lowercase letter from a to z, followed
		by a sequence of characters in range ('a' to 'z','0' to '9','.','_').[br]
		"%index","%my_nickname","%foo","%bAR1" and "%foo.BAR" are examples of valid
		local variable names.[br]
		A local variable exists only in the current command scope.
		The exact command scope definition is rather tricky and depends
		on the internal KVIrc implementation. Just be aware that:[br]
		- An alias body is a command scope.[br]
		- An event body is a command scope.[br]
		- Any sequence of commands executed [b]at once[/b] in a window commandline
		is a command scope.[br]
		You will notice that finding out the current command scope is rather intuitive.[br]
		When you type[br]
		[example]
			%text = "very important text";
		[/example]
		in the commandline of any KVIrc window and then try to execute[br]
		[example]
			echo %text
		[/example]
		you will see nothing printed. The variable %text was local to the "command scope"
		and disappeared immediately after the execution of the assignment.[br]
		But if you execute[br]
		[example]
			%text = "hello"; echo %text wold!
		[/example]
		you will see  "hello world!" printed in the current window.[br]

		[big]Extended scope variables[/big]
		Variables that start with a ':' character are "extended scope" variables.
		"%:index" , "%:Hello" , "%:something.else" are all valid special scope variable names.[br]
		They're actually used in popups and in timers (but later I might find other usages as well :).[br]
		"Extended scope" means that these variables are somewhere in the middle between
		global and local variables. They normally act as local , but in some cases their [b]lifetime[/b] and [b]visibility[/b]
		may be extended.[br]
		For example , in the popups , all the special scope variables
		are visible during all the "lifetime" of a popup (so from the prologue code call to
		the moment when the user selects an item and the corresponding code is executed).[br]
		This allows you to pre-calculate some data or conditions in the popup prologue
		and use this data in the popup item conditions and item handlers.[br]

		[big]Variable creation and destruction[/big]
		You don't need to declare variables.
		A variable starts to exist at the time that you assing
		something to it.[br]
		[example]
			%MyVar = some funky text
			%X = 2445833
			%SevenSpaces = "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;"
		[/example]
		It terminates its existence at the time that you assingn it an empty string:
		[example]
			%MyVar = ""
			%X = %MyVar
			%SevenSpaces =
		[/example]
		The example is based on global variables, but it is valid for local ones as well.[br]
		A non existing variable is equivalent to an empty one: you will
		never get a warning about non existing variables. This convention
		allows KVIrc to automatically manage the variable creation and destruction.[br]

		[big]Variable evaluation[/big]
		A variable can appear in every place where a parameter
		is expected: so after the command name, after a switch or inside
		an identifier parameters.
		KVirc will try to extract the longest possible variable name after a literal percent '%'
		sign everywhere in the parameter string. So the command sequence[br]
		[example]
			%number = 1st; echo this is my %number variable test
		[/example]
		will first assign "1st" to the variable "%number" and then execute
		"echo this is my 1st variable test".
		The following example will NOT work as expected.[br]
		[example]
			%number = 1; echo this is my %numberst variable test
		[/example]
		KVirc will assign "1" to %number in this case but the next variable
		name extracted will be "%numberst" that is actually empty; so finally
		"echo this is my variable test" will be executed.
		To avoid this problem you can use the backslash escape character:[br]
		[example]
			%number = 1; echo this is my %number\st variable test
		[/example]

		[big]Arrays[/big]
		Arrays are collections of items indexed by numbers.[br]
		The general syntax for an array is:[br]
		[b]%<name>[<index>][/b][br]
		<name> is the name of the array and follows the rules
		valid for the simple variables: the names starting with an uppercase letter
		designate a global array, the others designate local ones.[br]
		Extended scope arrays can be created as well: normally they act as local,
		but may have extended lifetime in some scopes.[br]
		The arrays have its own namespace thus %Array[] has nothing to do with %Array
		(that is a simple variable).[br]
		<index> must be a subscript that evaluates to a positive integer and it selects an item in the array.
		The first index of the array is 0 and the last is equal to size-1.[br]
		You can obtain the size of the array by evaluating %<name>[]#.[br]
		You don't need to declare the size of the array: it is automatically
		handled. When you assign a non-empty string to an item, the array
		is automatically enlarged to contain the index that you are assigning to.
		If the array was not existing before the assignment, it is created.[br]
		If the first assignment index is greater than 0 the items below that index
		will be empty: will just behave as empty/unset variables.[br]
		[example]
			[comment]# Create an array with 21 elements[/comment]
			%Names[20]=Pragma
		[/example]
		To remove an item from the array you assign it an empty string.[br]
		When you remove the highest indexed item the array is automatically shrunk to
		the next highest non-empty item. If there are no other non-empty items the array is destroyed.[br]
		Please note that the memory occupation of the array depends on the <index>.[br]
		By assigning a value to the 10000'th item (index 9999) you allocate 10000 entries! (in fact
		at least ((10000 * 4) + value size) bytes).[br]
		An array is destroyed when it contains no more items or when it goes out of scope (a local array obviously).[br]
		[example]
			[comment]# Creating an array with 800 entries[/comment]
			%Array[799]=test
			echo %Array[]#
			%Array[299]=test
			echo %Array[]#
			%Array[799]=
			echo %Array[]#
			[comment]# Now it contains 300 elements[/comment]
		[/example]
		[br]
		[big]Array references[/big][br]
		When <index> is an empty string then the subscript %<name>[] is called "array reference" (maybe a bit unproperly :).
		You may think that notation as referencing the "whole array".[br]
		The following example shows how the array assignment.[br]
		[example]
			%A[100]=x
			%A[200]=y
			%B[]=%A[]
			echo %B[200]
		[/example]
		When you pass an array reference to a command or function, it is evaluated
		as a comma separated list of entries.[br]
		[example]
			%Array[0]=Pippo
			%Array[1]=Pluto
			%Array[2]=Paperino
			echo %Array[]
		[/example]
		By assigning a string to an array you assign it to all the array entries:[br]
		[example]
			%Array[0]=Pippo
			%Array[1]=Pluto
			%Array[2]=Paperino
			echo %Array[]
			%Array[]=undefined
			echo %Array[]
		[/example]
		This is useful when you want to unset an array: just assign an empty string to all its entries:[br]
		[example]
			%Array[200]=Test
			echo %Array[]#
			%Array[]=
			echo %Array[]#
		[/example]
		You can loop through all the array items by using the [cmd]foreach[/cmd] command:[br]
		[example]
			%Array[0]=Pippo
			%Array[1]=Pluto
			%Array[2]=Paperino
			[cmd]foreach[/cmd](%item,%Array[])[cmd]echo[/cmd] %item
		[/example]
		Obviously also the traditional [cmd]for[/cmd] and [cmd]while[/cmd] indexed-looping methods
		are available:[br]
		[example]
			%Array[0]=Pippo
			%Array[1]=Never show this
			%Array[2]=Pluto
			%Array[5]=Hidden again
			%Array[8]=Paperino
			for(%i=0;%i < %Array[]#;%i+=2)echo Entry %i: \"%Array[%i]\";
		[/example]
		[br]
		[big]Dictionaries[/big]
		Dictionaries are associative arrays of strings. They look close
		to the perl hashes. The general syntax for a dictionary name is:[br]
		[b]%<name>{<key>}[/b][br]
		<name> is the name of the dictionary and follows the same rule
		as for the variables: the names starting with an uppercase letter
		designate a global dictionary, the others designate local ones.[br]
		Again , the dictionaries have its own namespace: you can safely use
		%Names , %Names[] and %Names{} as different entities in your script.[br]
		Extended scope dictionaries can be created as well: normally they act as local,
		but may have extended lifetime in some scopes.[br]
		A dictionary associates a "data string" to each "key string" used.
		The key can be [b]any string[/b] not containing an "unescaped" '}' character
		and is [b]case insensitive[/b]: "key" is equivalent to "KEY" or "KeY".[br]
		[example]
			%Ages{Pragma} = 24
		[/example]
		This assignment associates the key "Pragma" to the number "24" in
		the global dictionary "%Ages". If the array was not existing yet,
		it is created first. If the key "Pragma" was already existing,
		the value associated is replaced with the new value.[br]
		To remove an association you simply assign the empty string to it:[br]
		[example]
			%Ages{pragma} =
		[/example]
		The dictionary is automatically destroyed when there are no more associations
		in it (eg. when you assign the empty string to the last key).[br]
		Dictionaries can be used easily to simulate arrays:[br]
		[example]
			%Links{21} = "http://www.kvirc.net"
		[/example]
		Even multidimensional ones:[br]
		[example]
			%Pixels{324.312} = 1
		[/example]
		Remember that in the example above the key "324.312" is a single string.[br]
		Obviously the key can contain variables and functions just as any other parameter.[br]
		An empty key performs operations on the whole dictionary (just like for the arrays):[br]
		You can assign dictionaries:[br]
		[example]
			%Test2{}=%Test1{}
		[/example]
		Assign a scalar to all the items
		[example]
			%Data{} = "unspecified"
		[/example]
		The example above associates the string "unspecified" to all the existing
		keys in the dictionary %Data. By using this method you can easily destroy
		a dictionary: simply assign the empty string to it.[br]
		[example]
			%Data{} =
		[/example]
		Other [doc]operators[/doc] have similar semantic when working on "empty keys".[br]

		[big]Dictionary evaluation[/big]
		A dictionary can appear in every place where a variable can appear.[br]
		[example]
			[cmd]echo[/cmd] My age is %Ages{[fnc]$mynick[/fnc]}
		[/example]
		If you pass an empty key, the dictionary evaluates to the comma separated
		list of values stored in it. The values have no defined order.
		The list is interpreted as single "string" in contexts where a "string" is required,
		and as a list of "strings" in context where lists of strings are expected. (<-- hehe :)[br]
		The special syntax %<name>{}# returns the number of keys in the
		dictionary.[br]
		The special syntax %<name>{}@ returns a comma
		separated list of keys in the dictionary; the keys have no
		defined order (well, you may be only sure that the order
		of the keys is exactly equal to the values order (%name{})).[br]
		[example]
			%Songs{Jimi Hendrix} = Voodo child
			%Songs{Shawn Lane} = Gray piano's flying
			%Songs{Mina} = Brava
			%Songs{Greg Howe} = "Full Throttle"
			# This is a "single string" evaluation context
			[cmd]echo[/cmd] %Songs{}
			# This is a "list of strings" evaluation context
			[cmd]foreach[/cmd](%var,%Songs{})[cmd]echo[/cmd] %var
		[/example]
*/


bool KviUserParser::lookupDataType(KviCommand *c,KviDataType *d)
{
	ENTER_CONTEXT(c,"lookupDataType");
	// %<name> : variable
	// %<name>[<key>] : dictionary
	// %<name>[] : dictionary reference
	// ->%[a-z_.]+ : object scope variable
	// %[A-Z][a-z_.]+ : global variable
	// %[a-z][a-z_.]+ : local variable
	// %:[A-Za-z_.]+ : special scope variable

	__range_valid(*c->m_ptr == '%');
	char * aux = ++(c->m_ptr);

	if(c->scopeObject())d->pDataContainer = c->scopeObject()->dataContainer();
	else {
		if(*aux == ':')
		{
			d->pDataContainer = c->extendedScopeDataContainer();
			if(!d->pDataContainer)d->pDataContainer = c->m_pDataContainer;
			aux = ++(c->m_ptr);
		} else {
			if(isupper(*aux))d->pDataContainer = m_pGlobalDataContainer;
			else d->pDataContainer = c->m_pDataContainer;
		}
	}

	while(isalnum(*aux) || (*aux == '.') || (*aux == '_'))aux++;
	d->szName.extractFromString(c->m_ptr,aux);

	c->m_ptr = aux;

	if(d->szName.isEmpty())
	{
		d->iDataType = KVI_DATA_TYPE_NONE;
		return c->leaveContext();
	}

	if(*(c->m_ptr) == '{')
	{
		++(c->m_ptr);
		// looks like a dictionary
		// no scope object for the key
		KviScriptObject * back = c->scopeObject();
		c->setScopeObject(0);
		if(!parseVariableKey(c,d->szKey))
		{
			c->setScopeObject(back);
			return false;
		}
		c->setScopeObject(back);

		d->pDictionary = d->pDataContainer->lookupDictionary(d->szName.ptr(),true);
		if(d->szKey.isEmpty())
		{
			if(*(c->m_ptr) == '@')
			{
				++(c->m_ptr);	
				d->iDataType = KVI_DATA_TYPE_DICTKEYSREFERENCE;
			} else if(*(c->m_ptr) == '#')
			{
				++(c->m_ptr);
				d->iDataType = KVI_DATA_TYPE_DICTITEMCOUNT;
			} else d->iDataType = KVI_DATA_TYPE_DICTREFERENCE;
		} else {
			d->iDataType = KVI_DATA_TYPE_DICTVARIABLE;
			d->pVariable = d->pDataContainer->lookupDictionaryVariable(d->szName.ptr(),d->szKey.ptr(),true);
		}
	} else if(*(c->m_ptr) == '[')
	{
		++(c->m_ptr);
		// looks like an array
		// no scope object for the index
		KviScriptObject * back = c->scopeObject();
		c->setScopeObject(0);
		if(!parseVariableIndex(c,d->szKey))
		{
			c->setScopeObject(back);
			return false;
		}
		c->setScopeObject(back);

		d->pArray = d->pDataContainer->lookupArray(d->szName.ptr(),true);
		if(d->szKey.isEmpty())
		{
			if(*(c->m_ptr) == '#')
			{
				++(c->m_ptr);
				d->iDataType = KVI_DATA_TYPE_ARRAYITEMCOUNT;
			} else d->iDataType = KVI_DATA_TYPE_ARRAYREFERENCE;
		} else {
			bool bOk;
			d->uIndex = d->szKey.toUInt(&bOk);
			if(!bOk)return c->error(KviError_invalidArrayIndex);
			d->iDataType = KVI_DATA_TYPE_ARRAYVARIABLE;
			d->pVariable = d->pDataContainer->lookupArrayVariable(d->szName.ptr(),d->uIndex,true);
		}
	} else {
		// it is a variable
		d->iDataType = KVI_DATA_TYPE_VARIABLE;
		d->pVariable = d->pDataContainer->lookupVariable(d->szName.ptr(),true);
	}

	return c->leaveContext();
}

bool KviUserParser::parseLValueCommand(KviCommand *c)
{
	ENTER_CONTEXT(c,"parseLValueCommand");
	__range_valid((*(c->m_ptr)=='%') || (*(c->m_ptr) == '$'));
	__range_invalid(c->scopeObject());

	bool bLastWasAVariable = false;

	KviDataType d;
	KviStr dummy;

	do {
		if(*(c->m_ptr) == '%')
		{
			bLastWasAVariable = true;
			if(!parseVariable(c,dummy,&d,false))return false;
		} else if(*(c->m_ptr) == '$')
		{
			bLastWasAVariable = false;
			if(!parseIdentifier(c,dummy))return false;
		} else return c->error(KviError_variableOrIdentifierExpected);
	} while(c->scopeObject());
	// check object scope

	if(bLastWasAVariable)
	{
		switch(d.iDataType)
		{
			case KVI_DATA_TYPE_VARIABLE:
				if(!d.pVariable)d.pVariable = d.pDataContainer->lookupVariable(d.szName.ptr(),false); // RW access
			break;
			case KVI_DATA_TYPE_DICTVARIABLE:
				if(!d.pDictionary)d.pDictionary = d.pDataContainer->lookupDictionary(d.szName.ptr(),false);
				if(!d.pVariable)d.pVariable = d.pDataContainer->lookupDictionaryVariable(d.szName.ptr(),d.szKey.ptr(),false);
			break;
			case KVI_DATA_TYPE_DICTREFERENCE:
			case KVI_DATA_TYPE_DICTKEYSREFERENCE:
			case KVI_DATA_TYPE_DICTITEMCOUNT:
				if(!d.pDictionary)d.pDictionary = d.pDataContainer->lookupDictionary(d.szName.ptr(),false);
			break;
			case KVI_DATA_TYPE_ARRAYVARIABLE:
				if(!d.pArray)d.pArray = d.pDataContainer->lookupArray(d.szName.ptr(),false);
				if(!d.pVariable)d.pVariable = d.pDataContainer->lookupArrayVariable(d.szName.ptr(),d.uIndex,false);
			break;
			case KVI_DATA_TYPE_ARRAYREFERENCE:
			case KVI_DATA_TYPE_ARRAYITEMCOUNT:
				if(!d.pArray)d.pArray = d.pDataContainer->lookupArray(d.szName.ptr(),false);
			break;
			case KVI_DATA_TYPE_NONE:
				// this will be error in parseOperator
				return c->error(KviError_missingVariableName);
			break;
		}
		c->skipSpace();
		if(!parseOperator(c,&d))
		{
			// may we have created unnecessary empty variables ?
			// .... yes...this is ugly
			switch(d.iDataType)
			{
				case KVI_DATA_TYPE_VARIABLE:
					if(d.pVariable->isEmpty())d.pDataContainer->removeVariable(d.szName.ptr());
				break;
				case KVI_DATA_TYPE_DICTVARIABLE:
					if(d.pVariable->isEmpty())d.pDataContainer->removeDictionaryVariable(d.szName.ptr(),d.szKey.ptr());
					if(d.pDictionary->count() == 0)d.pDataContainer->removeDictionary(d.szName.ptr());
				break;
				case KVI_DATA_TYPE_DICTREFERENCE:
				case KVI_DATA_TYPE_DICTKEYSREFERENCE:
				case KVI_DATA_TYPE_DICTITEMCOUNT:
					if(d.pDictionary->count() == 0)d.pDataContainer->removeDictionary(d.szName.ptr());
				break;
				case KVI_DATA_TYPE_ARRAYVARIABLE:
					if(d.pVariable->isEmpty())d.pDataContainer->removeArrayVariable(d.szName.ptr(),d.uIndex);
					if(d.pArray->size() == 0)d.pDataContainer->removeArray(d.szName.ptr());
				break;
				case KVI_DATA_TYPE_ARRAYREFERENCE:
				case KVI_DATA_TYPE_ARRAYITEMCOUNT:
					if(d.pArray->size() == 0)d.pDataContainer->removeArray(d.szName.ptr());
				break;
			}
			return false;
		}
		return c->leaveContext();
	} else {
		// function call (probably object function)
		return parseCmdFinalPart(c,dummy) ? c->leaveContext() : false;
	}

	// never here

	return c->leaveContext();
}

bool KviUserParser::parseVariable(KviCommand *c,KviStr &buffer,KviDataType * d,bool bIncludeLists)
{
	ENTER_CONTEXT(c,"parseVariable");
	// This one parses the variable pointed by ptr
	// adds its value to c->m_buffer
	// (or an empty string if the var is not defined)
	// then moves c->m_ptr to the characted immediately 
	// after the variable and returns

	KviDataType dataTypeBuffer;
	if(!d)d = &dataTypeBuffer; // local one

	if(!lookupDataType(c,d))return false;

	KviStr tmpBuffer;

	switch(d->iDataType)
	{
		case KVI_DATA_TYPE_NONE:
			tmpBuffer = '%';
		break;
		case KVI_DATA_TYPE_VARIABLE:
		case KVI_DATA_TYPE_DICTVARIABLE:
		case KVI_DATA_TYPE_ARRAYVARIABLE:
			if(d->pVariable)tmpBuffer = *(d->pVariable);
		break;
		case KVI_DATA_TYPE_DICTITEMCOUNT:
			if(d->pDictionary)tmpBuffer.sprintf("%d",d->pDictionary->count());
			else tmpBuffer = '0';
		break;
		case KVI_DATA_TYPE_DICTKEYSREFERENCE:
			if(d->pDictionary && bIncludeLists)
			{
				KviDictionaryIterator it(*(d->pDictionary));
				bool bFirst = true;
				while(it.current())
				{
					if(bFirst)bFirst = false;
					else tmpBuffer.append(',');

					tmpBuffer.append(it.currentKey());
					++it;
				}
			}
		break;
		case KVI_DATA_TYPE_DICTREFERENCE:
			if(d->pDictionary && bIncludeLists)
			{
				KviDictionaryIterator it(*(d->pDictionary));
				bool bFirst = true;
				while(it.current())
				{
					if(bFirst)bFirst = false;
					else tmpBuffer.append(',');

					tmpBuffer.append(*(it.current()));
					++it;
				}
			}
		break;
		case KVI_DATA_TYPE_ARRAYITEMCOUNT:
			if(d->pArray)tmpBuffer.sprintf("%d",d->pArray->size());
			else tmpBuffer = '0';
		break;
		case KVI_DATA_TYPE_ARRAYREFERENCE:
			if(d->pArray && bIncludeLists)
			{
				for(unsigned int u = 0;u < d->pArray->size();u++)
				{
					if(u > 0)tmpBuffer.append(',');
					KviStr * t = d->pArray->uncheckedAt(u);
					if(t)tmpBuffer.append(*t);
				}
			}
		break;
		default:
			__range_valid(false); //ooops...
		break;
	}

	if(*(c->m_ptr) == '-')
	{
		// scope object ?
		char * aux = c->m_ptr;
		++aux;
		if(*aux == '>')
		{
			// yup...it looks as!
			++aux;
			// ok..we expect a variable or function call now
			if((*aux == '%') || (*aux == '$'))
			{
				c->m_ptr = aux;
				// yup...object scope operator
				KviScriptObject * o = g_pScriptObjectController->lookupObject(tmpBuffer.ptr());
				if(!o)return c->error(KviError_noSuchObject,__tr("Variable evaluated to \"%s\""),tmpBuffer.ptr());
				c->setScopeObject(o);
				return c->leaveContext(); //yup...new object scope
			} // else just a "->"....? :)
		}
	}

	c->setScopeObject(0); // evaluated the variable in the object scope...back to global scope

	buffer.append(tmpBuffer);

	return c->leaveContext();
}


bool KviUserParser::parseVariableKey(KviCommand *c,KviStr &buffer)
{
	ENTER_CONTEXT(c,"parseVariableKey");
	// This one must be called when c->m_ptr points
	// somewhere in a variable parameters block.
	// The parsing is stopped by a '\n', '}' or '\0'
	// on '\n' and '\0' an error is returned
	// The parsing is NOT stopped if '}'
	// is in a string (delimitated by two '"' chars)
	// Skips the ending '}' character
	// This function EATS leading and trailing spaces and tabs
	// and reduces the middle ones to one per group.
	// A key like this:
	//          string    "full  of  words"  and  spaces      ,
	// Becomes like this:
	// string full  of  words and spaces ,

	// Skip leading spaces and tabs (the escaped ones too)
	c->skipSpace();
	register char *aux_ptr = c->m_ptr;

	bool inString      = false;

	for(;;){
		// First skip all the non interesting chars
		while(	*aux_ptr         && (*aux_ptr!='%') &&
				(*aux_ptr!='\n') && (*aux_ptr!='}')  && (*aux_ptr!='$')  && 
				(*aux_ptr!=' ')  && (*aux_ptr!='"')  && (*aux_ptr!='\\') &&
				(*aux_ptr!='\t'))aux_ptr++;
		// Interesting char
		switch(*aux_ptr){
			case '\0':
			case '\n': // End of the string ('\n' is not escaped)...fail here
				if(inString)return c->error(KviError_unexpectedEndInString);
				else return c->error(KviError_unexpectedEndInDictionaryKey);
			break;
			case ' ': // Spaces...need to reduce it to one...
			case '\t':
				if(!inString){
					// append the last part
					buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
					// skip the other spaces including escaped ones
					c->m_ptr = aux_ptr;
					c->skipSpace();
					aux_ptr = c->m_ptr;
					if((*aux_ptr != '}'))buffer.append(' ');
				} else while((*aux_ptr == ' ')||(*aux_ptr == '\t'))aux_ptr++;
			break;
			case '"': // Not escaped '"'...enter or exit the string (skip the '"')
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = ++aux_ptr;
				inString = !inString;
			break;
			case '}': // If not in string , exit
				if(!inString){
					buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
					c->m_ptr = ++aux_ptr;
					return c->leaveContext();
				}
				++aux_ptr;
			break;
			case '%': //variable: append the last block to the buffer and process the var
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				if(!parseVariable(c,buffer))return false;
				aux_ptr = c->m_ptr;
			break;
			case '$': //system identifier: append the last block to the buffer and process the ident
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				if(!parseIdentifier(c,buffer))return false;
				aux_ptr = c->m_ptr;
			break;
			case '\\': //escape character: append the last block to the processed buffer...
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = ++aux_ptr;
				switch(*aux_ptr){
					case '\0':
						if(inString)return c->error(KviError_unexpectedEndInString);
						else return c->error(KviError_unexpectedEndInDictionaryKey);
					break;
					case '\n':  //escaped newline
						c->m_ptr = ++aux_ptr; //skip it
						c->skipSpace();       //skip leading spaces
						aux_ptr = c->m_ptr;   //continue
					break;
					case 'r':
						if(inString)buffer.append('\r');
						else buffer.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
					case 'n':
						if(inString)buffer.append('\n');
						else buffer.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
					default: // Must be copied to the buffer without modification
						buffer.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
				}
			break;
		}
	}
	// Newer arrive here...
	return c->error(KviError_internalError);
}


bool KviUserParser::parseVariableIndex(KviCommand *c,KviStr &buffer)
{
	ENTER_CONTEXT(c,"parseVariableIndex");
	// This one must be called when c->m_ptr points
	// somewhere in a variable parameters block.
	// The parsing is stopped by a '\n', ']' or '\0'
	// on '\n' and '\0' an error is returned
	// The parsing is NOT stopped if ']'
	// is in a string (delimitated by two '"' chars)
	// Skips the ending ']' character
	// This function EATS leading and trailing spaces and tabs
	// and reduces the middle ones to one per group.
	// A key like this:
	//          string    "full  of  words"  and  spaces      ,
	// Becomes like this:
	// string full  of  words and spaces ,

	// Skip leading spaces and tabs (the escaped ones too)
	c->skipSpace();

	register char *aux_ptr = c->m_ptr;

	for(;;){
		// First skip all the non interesting chars
		while(	*aux_ptr         && (*aux_ptr!='%') &&
				(*aux_ptr!='\n') && (*aux_ptr!=']')  && (*aux_ptr!='$')  && 
				(*aux_ptr!=' ')  && (*aux_ptr!='\\') &&
				(*aux_ptr!='\t'))aux_ptr++;
		// Interesting char
		switch(*aux_ptr)
		{
			case '\0':
			case '\n': // End of the string ('\n' is not escaped)...fail here
				return c->error(KviError_unexpectedEndInArrayIndex);
			break;
			case ' ': // Spaces...need to reduce it to one...
			case '\t':
				// append the last part
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				// skip the other spaces including escaped ones
				c->m_ptr = aux_ptr;
				c->skipSpace();
				aux_ptr = c->m_ptr;
				if((*aux_ptr != ']'))return c->error(KviError_unexpectedCharactersInArrayIndex);
			break;
			case ']': // If not in string , exit
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = ++aux_ptr;
				return c->leaveContext();
			break;
			case '%': //variable: append the last block to the buffer and process the var
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				if(!parseVariable(c,buffer))return false;
				aux_ptr = c->m_ptr;
			break;
			case '$': //system identifier: append the last block to the buffer and process the ident
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				if(!parseIdentifier(c,buffer))return false;
				aux_ptr = c->m_ptr;
			break;
			case '\\': //escape character: append the last block to the processed buffer...
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = ++aux_ptr;
				switch(*aux_ptr)
				{
					case '\0':
						return c->error(KviError_unexpectedEndInArrayIndex);
					break;
					default: // Must be copied to the buffer without modification
						buffer.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
				}
			break;
		}
	}
	// Newer arrive here...
	return c->error(KviError_internalError);
}

bool KviUserParser::parseParameterIdentifier(KviCommand *c,KviStr &buffer)
{
	ENTER_CONTEXT(c,"parseParameterIdentifier");
	// This one expects c->m_ptr to point to a digit or '#'
	__range_valid(isdigit(*(c->m_ptr)) || ((*(c->m_ptr))=='#') || ((*(c->m_ptr)) == '$'));
	if(*(c->m_ptr) == '#')
	{
		c->getParamCount(buffer);
		++(c->m_ptr);
		return c->leaveContext();
	}
	if(*(c->m_ptr) == '$')
	{
		if(c->thisPointer())buffer.append(c->thisPointer()->id());
		else buffer.append('0');
		++(c->m_ptr);
		return c->leaveContext();
	}
	int param = (*(c->m_ptr) - '0');
	++(c->m_ptr);
	while(isdigit(*(c->m_ptr)))
	{
		param = param * 10;
		param += (*(c->m_ptr) - '0');
		++(c->m_ptr);
	}
	if(*(c->m_ptr) == '-'){
		++(c->m_ptr);
		if(isdigit(*(c->m_ptr)))
		{
			int parEnd = (*(c->m_ptr) - '0');
			++(c->m_ptr);
			while(isdigit(*(c->m_ptr)))
			{
				parEnd = parEnd * 10;
				parEnd += (*(c->m_ptr) - '0');
				++(c->m_ptr);
			}
			c->getParam(param,parEnd,buffer);
		} else {
			if(*(c->m_ptr) == '>')
			{
				// ops!...was an object scope operator!
				--(c->m_ptr);
				c->getSingleParam(param,buffer);
			} else c->getParam(param,-1,buffer); //up to the end
		}
	} else c->getSingleParam(param,buffer);
	return c->leaveContext();
}

bool KviUserParser::parseObjectFunction(KviCommand *c,KviStr &buffer)
{
	ENTER_CONTEXT(c,"parseObjectFunction");

	__range_valid(c->scopeObject());

	KviStr classOverride;

	char * aux = c->m_ptr;

	while(isalnum(*aux) || (*aux == '_'))aux++;

	if(*aux == ':')
	{
		classOverride.extractFromString(c->m_ptr,aux);
		c->m_ptr = ++aux;
		if(*(c->m_ptr) == ':') // allow the C++ syntax class::function
		{
			++(c->m_ptr);
			++aux;
		}
		while(isalnum(*aux) || (*aux == '_'))aux++;
	}

	KviStr funcName(c->m_ptr,aux);
	c->m_ptr = aux;

	KviParameterList * paramList = new KviParameterList;
	paramList->setAutoDelete(true);
	
	if(!extractFunctionParameters(c,paramList))
	{
		delete paramList;
		return false;
	}

	KviScriptObjectFunctionHandler * h = c->scopeObject()->lookupFunctionHandler(funcName.ptr(),
			classOverride.hasData() ? classOverride.ptr() : 0);

	if(!h)return c->error(KviError_noSuchObjectFunction,"%s::%s",classOverride.ptr(),funcName.ptr());

	if(!callObjectFunction(h,paramList,buffer,c))return false;

	delete paramList;

	return c->leaveContext();
}

bool KviUserParser::callObjectFunction(KviScriptObjectFunctionHandler * h,KviParameterList * paramList,
	KviStr &buffer,KviCommand *c)
{
	ENTER_CONTEXT(c,"callObjectFunction");
	paramList->init();

	if(h->szScriptHandler.hasData())
	{
		KviCommand cmd(h->szScriptHandler.ptr(),c->window(),c);
		cmd.setThisPointer(c->scopeObject());
		cmd.setParams(paramList,false);
		if(!parseCommand(&cmd))
		{
			if(cmd.hasError())
			{
				printError(&cmd);
				return false; // We could do better ? (just a halt)
			}
		}
		buffer = cmd.m_szRetBuffer;
	} else {
		if(h->proc)
		{
			if(!(((c->scopeObject())->*(h->proc))(c,paramList,buffer)))return false;
		}
	}

	//debug("CALLED OBJECT FUNCTION for object (%s)",c->scopeObject()->id());

	return c->leaveContext();
}

	/*
		@doc: calc
		@type:
			function
		@title:
			$calc
		@short:
			$calc() has been replaced by $() !
		@syntax:
			$(<expression>)
		@description:
			In KVIrc 3.0.0 , $calc() has been replaced by the $().[br]
			This is just a shorter and faster form.[br]
			You can use $() to evaluate any integer expression.[br]
		@examples:
			[example]
				echo $(1 + 2)
				%var = 2
				echo $(10 * %val << 1)
			[/example]
		@seealso:
	*/

bool KviUserParser::parseIdentifier(KviCommand *c,KviStr &buffer)
{
	//
	// This is a rather huge function that parses an identifier
	// This is something as $identifier , $module.function , 
	// $1 , $2 , $$ , $# , $alias...
	//

	ENTER_CONTEXT(c,"parseIdentifier");

	__range_valid(*c->m_ptr == '$');

	++(c->m_ptr);              // skip the $

	// This is our return buffer
	KviStr tmpBuffer;

	if(c->scopeObject())
	{
		// We have a scope object... it must be an object function
		if(!parseObjectFunction(c,tmpBuffer))return false;
		goto check_object_scope;
	}

	if(isdigit(*(c->m_ptr))||(*(c->m_ptr) == '#')||(*(c->m_ptr) == '$'))
	{
		// this must be a parameter identifier
		if(!parseParameterIdentifier(c,tmpBuffer))return false;
		goto check_object_scope;
	}

	if(*(c->m_ptr) == '{')
	{
		// A command call
		char * aux = c->m_ptr;
		if(!skipCommand(c))return false;
		KviCommand cmd(aux,c->m_ptr,c->window(),c);
		if(!parseCommand(&cmd))
		{
			if(cmd.hasError())
			{
				printError(&cmd);
				return false; // We could do better ? (just a halt)
			}
		}
		tmpBuffer = cmd.m_szRetBuffer;
		goto check_object_scope;
	}

	if(*(c->m_ptr) == '(')
	{
		// an expression to evaluate
		long res;
		++(c->m_ptr);
		if(!evaluateExpression(c,&res))return false;
		tmpBuffer.setNum(res);
		goto check_object_scope;
	}


	{
		// Normal identifier
		KviModule * pModule;
		KviModuleFunctionParseProc * mProc = 0;
		KviFunctionParseProc * proc = 0;
	
		char * aux = c->m_ptr;
	
		while(*aux && (isalnum(*aux)||(*aux == '_')))aux++;
	
		KviStr ident(c->m_ptr,aux);       // and get the ident name
		c->m_ptr = aux;

		if(ident.isEmpty())
		{
			tmpBuffer.append('$');  // single $...nope
			goto processing_done;
		}
	

		if(*(c->m_ptr) == '.')
		{
			// module function
			++(c->m_ptr);

			pModule = g_pModuleManager->getModule(ident.ptr());
			if(!pModule)return c->error(KviError_errorInLoadingModule,__tr("%s (module %s)"),g_pModuleManager->lastError().ptr(),ident.ptr());

			aux = c->m_ptr;

			while(isalnum(*aux) || (*aux == '_') || (*aux == '.'))aux++;
			ident.extractFromString(c->m_ptr,aux); // FIXME: extractFromStringToUpper ???
			//ident.toUpper();
	
			c->m_ptr = aux;
	
			mProc = pModule->findFunctionParseProc(ident.ptr());
			if(!mProc)
			{
				mProc = pModule->genericFunctionParseProc();
				if(!mProc)return c->error(KviError_noSuchModuleFunction,__tr("Module '%s', Function '%s'"),pModule->name(),ident.ptr());
			}

			goto extract_parameters;
		}

	    // have an identifier
		ident.toUpper();

		proc = m_pFunctionDict->find(ident.ptr());
	
extract_parameters:
	
		KviParameterList * paramList = new KviParameterList();
		paramList->setAutoDelete(true);
	
		if(!extractFunctionParameters(c,paramList))
		{
			delete paramList;
			return false;
		}
	
	
		paramList->init(); // point to the first item

		c->setCurrentEntity(ident.ptr());

		if(proc)
		{
			// internal function
			if(!(this->*(proc->proc))(c,paramList,tmpBuffer))
			{
				delete paramList;
				return false;
			}
			delete paramList;
			goto check_object_scope;
		}
	
		// was a module function ?
		if(mProc)
		{
			if(!(*mProc)(pModule,c,paramList,tmpBuffer))
			{
				delete paramList;
				return false;
			}
			delete paramList;
			goto check_object_scope;
		}
	
		// check for aliases
	
		KviAlias * a = g_pAliasManager->lookupAlias(ident.ptr());
		if(a)
		{
			// Yes...an alias
			// We can easily check for infinite recursion here
			if(KVI_OPTION_BOOL(KviOption_boolLimitAliasRecursion))
			{
				if(a->lock() > 1024)
				{
					a->totalUnlock();
					return c->error(KviError_recursionTooDeep);
				}
			} else {
				// In fact the locking mechanism will stop to work if we exceed 2^32 iterations
				// assuming that unsigned long is 32 bits wide. So we still try to
				// avoid such monster recursions. Anyway, we will prolly never get it
				// to be executed: we will get a stack overflow first (2^32 quite large stack frames
				// can't live in our addressing space: the stack pointer would overflow!)
				// So this check is not a real limit (at least on 32 bit systems)
				if(a->lock() > 0xFFFFFFFE)
				{
					a->totalUnlock();
					return c->error(KviError_recursionTooDeep);
				}
			}
		
			// We want nice output in case of errors...
			ENTER_CONTEXT(c,a->name());
		
			// child command buffer
			KviCommand cmd(a->dataBuffer(),c->window(),c);
			cmd.setParams(paramList,false); // do not transfer param ownership
		
			// parse it
			if(!parseCommand(&cmd))
			{
				if(cmd.hasError())
				{
					printError(&cmd);
					delete paramList;
					a->unlock();
					return false; // this is a "halt" for us
				}
			}
		
			// propagate the return value
			tmpBuffer = cmd.m_szRetBuffer;
		
			delete paramList;
		
			c->leaveContext();
			a->unlock();
	
			goto check_object_scope;
		}
	
		delete paramList;
		return c->error(KviError_unknownFunction,"%s",ident.ptr());
	}
	
check_object_scope:	

	// check object scope
	if(*(c->m_ptr) == '-')
	{
		// scope object ?
		char * aux = c->m_ptr;
		++aux;
		if(*aux == '>')
		{
			// yup...it looks as!
			++aux;
			// ok..we expect a variable or function call now
			if((*aux == '%') || (*aux == '$'))
			{
				c->m_ptr = aux;
				// yup...object scope operator
				KviScriptObject * o = g_pScriptObjectController->lookupObject(tmpBuffer.ptr());
				if(!o)return c->error(KviError_noSuchObject,__tr("Identifier evaluated to \"%s\""),tmpBuffer.ptr());
				c->setScopeObject(o);
				return c->leaveContext(); //yup...new object scope
			} // else just a "->"....? :)
		}
	}

processing_done:

	c->setScopeObject(0); //evaluated in the object scope...back to global scope now

	buffer.append(tmpBuffer);
	return c->leaveContext();
}

bool KviUserParser::extractParameters(KviCommand *c,KviParameterList *paramList)
{
	// This one extracts a list of comma separated parameters and puts them in the paramList
	// The parameter list is terminated by a ')' character.

	ENTER_CONTEXT(c,"extractParameters");
	c->skipSpace();
	if(*(c->m_ptr) != ')')
	{

		KviScriptObject * o = c->scopeObject();
		c->setScopeObject(0);
		// the ident is followed by parameters...
		for(;;)
		{
			if(!parseFncSingleParam(c,paramList))
			{
				c->setScopeObject(o);
				return false;
			} else {
				__range_valid((*(c->m_ptr)==',')||(*(c->m_ptr)==')'));
				if(*(c->m_ptr) == ')')
				{
					++(c->m_ptr);
					break;
				} else ++(c->m_ptr);
			}
		}
		c->setScopeObject(o);
	} else ++(c->m_ptr);
	return c->leaveContext();
}


bool KviUserParser::extractFunctionParameters(KviCommand *c,KviParameterList * paramList)
{
	// This one extracts the parameters of a function and puts them in the paramList
	// If the first character is not '(' , the function is assumed to have no parameters.
//	ENTER_CONTEXT(c,"extractFunctionParameters"); <-- let's avoid a stack entry

	if(*(c->m_ptr) == '(')
	{
		++(c->m_ptr);
		if(!extractParameters(c,paramList))return false;
	}
//	return c->leaveContext(); <-- let's avoid such a simple stack entry
	return true;
}

#ifdef JMP
#undef JMP
#endif
#define JMP 1
static char function_parameter_parsing_char_table[256]=
{
	//	000 001 002 003 004 005 006 007   008 009 010 011 012 013 014 015 
	//	NUL SOH STX ETX EOT ENQ ACK BEL   BS  HT  LF  VT  FF  CR  SO  SI
		JMP,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,JMP,JMP,0  ,0  ,0  ,0  ,0  ,
	//	016 017 018 019 020 021 022 023   024 025 026 027 028 029 030 031 
	//	DLE DC1 DC2 DC3 DC4 NAK SYN ETB   CAN EM  SUB ESC FS  GS  RS  US
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	032 033 034 035 036 037 038 039   040 041 042 043 044 045 046 047 
	//	    !   "   #   $   %   &   '     (   )   *   +   ,   -   .   /   
		JMP,0  ,JMP,0  ,JMP,JMP,0  ,0    ,JMP,JMP,0  ,0  ,JMP,0  ,0  ,0  ,
	//	048 049 050 051 052 053 054 055   056 057 058 059 060 061 062 063 
	//	0   1   2   3   4   5   6   7     8   9   :   ;   <   =   >   ?   
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	064 065 066 067 068 069 070 071   072 073 074 075 076 077 078 079 
	//	@   A   B   C   D   E   F   G     H   I   J   K   L   M   N   O   
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	080 081 082 083 084 085 086 087   088 089 090 091 092 093 094 095 
	//	P   Q   R   S   T   U   V   W     X   Y   Z   [   \   ]   ^   _   
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,JMP,0  ,0  ,0  ,
	//	096 097 098 099 100 101 102 103   104 105 106 107 108 109 110 111 
	//	`   a   b   c   d   e   f   g     h   i   j   k   l   m   n   o   
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	112 113 114 115 116 117 118 119   120 121 122 123 124 125 126 127 
	//	p   q   r   s   t   u   v   w     x   y   z   {   |   }   ~      
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	128 129 130 131 132 133 134 135   136 137 138 139 140 141 142 143 
	//	                                                  
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	144 145 146 147 148 149 150 151   152 153 154 155 156 157 158 159 
	//	                                                  
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	160 161 162 163 164 165 166 167   168 169 170 171 172 173 174 175 
	//	                                                  
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	176 177 178 179 180 181 182 183   184 185 186 187 188 189 190 191 
	//	                                                  
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	192 193 194 195 196 197 198 199   200 201 202 203 204 205 206 207 
	//	                                                  
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	208 209 210 211 212 213 214 215   216 217 218 219 220 221 222 223 
	//	                                                  
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	224 225 226 227 228 229 230 231   232 233 234 235 236 237 238 239 
	//	                                                  
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	240 241 242 243 244 245 246 247   248 249 250 251 252 253 254 255 
	//	                                                  
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0
};
#undef JMP

bool KviUserParser::parseFncSingleParam(KviCommand *c,KviParameterList * paramList,bool bSkipIdentAndVars)
{
	ENTER_CONTEXT(c,"parseFncSingleParam");

	// This one must be called when c->m_ptr points
	// somewhere in a function parameters block.
	// It extracts the first parameter encountered and
	// preprocesses it (subst variables , etc...)
	// then it appends everything to the buffer
	// The parsing is stopped by a '\n', ',' , ')' or '\0'
	// on '\n' and '\0' an error is returned
	// The parsing is NOT stopped if one of ',' or ')'
	// is in a string (delimitated by two '"' chars)
	// Note that this function does nothing if c->m_ptr
	// points to the end of the param buffer...
	// (eg. c->m_ptr comes from previous calls to processFncSingleParam())
	// It will also skip matching parenthesis :
	// so () is a valid parameter
	// It will return false in case of an error ;
	// If it returns true , c->m_ptr surely points to a ',' or a ')'
	// This function EATS leading and trailing spaces and tabs
	// and reduces the middle ones to one per group.
	// A param like this:
	//          string    "full  of  words"  and  spaces      ,
	// Becomes like this:
	// string full  of  words and spaces,

	// Skip leading spaces and tabs (the escaped ones too)


	c->skipSpace();

	register char *aux_ptr = c->m_ptr;

	int parLevel       = 0;
	bool inString      = false;

	KviStr buffer;



	for(;;){
		// First skip all the non interesting chars
		/*
		while(	*aux_ptr         && (*aux_ptr!='%') &&
				(*aux_ptr!='\n') && (*aux_ptr!=')')  && 
				(*aux_ptr!=',')  && (*aux_ptr!='$')  && 
				(*aux_ptr!=' ')  && (*aux_ptr!='(')  &&
				(*aux_ptr!='"')  && (*aux_ptr!='\\') &&
				(*aux_ptr!='\t'))aux_ptr++;
		*/
		while(!function_parameter_parsing_char_table[*aux_ptr])aux_ptr++;

		// Interesting char
		switch(*aux_ptr)
		{
			case '\0':
			case '\n': // End of the string ('\n' is not escaped)...fail here
				if(inString)return c->error(KviError_unexpectedEndInString);
				else return c->error(KviError_unexpectedEndInFunctionParams);
			break;
			case ' ': // Spaces...need to reduce it to one...
			case '\t':
				if(!inString)
				{
					// append the last part
					buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
					// skip the other spaces including escaped ones
					c->m_ptr = aux_ptr;
					c->skipSpace();
					aux_ptr = c->m_ptr;
					if((*aux_ptr != ')') && (*aux_ptr != ','))buffer.append(' ');
				} else while((*aux_ptr == ' ')||(*aux_ptr == '\t'))aux_ptr++;
			break;
			case '"': // Not escaped '"'...enter or exit the string (skip the '"')
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = ++aux_ptr;
				inString = !inString;
			break;
			case '(': // Keep track of matching parenthesis (if not in string)
				++aux_ptr;
				if(!inString)++parLevel;
			break;
			case ')': // If not in string , exit if parenthesis level is 0 , otherwise decrease it
				if(!inString)
				{
					if(parLevel == 0)
					{
						buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
						c->m_ptr = aux_ptr;
						paramList->append(new KviStr(buffer));
						return c->leaveContext();
					} else --parLevel;
				}
				++aux_ptr;
			break;
			case ',': // end of command buffer...append the last block to the buffer and return
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				if(!inString)
				{
					c->m_ptr = aux_ptr;
					paramList->append(new KviStr(buffer));
					return c->leaveContext();
				}
				c->m_ptr = aux_ptr++; //in string...',' must be included
			break;
			case '%': //variable: append the last block to the buffer and process the var
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;

				if(bSkipIdentAndVars)
				{
					if(!skipVariable(c))return false;
					buffer.append(aux_ptr,c->m_ptr - aux_ptr);
				} else {
					KviDataType d;
					if(!parseVariable(c,buffer,&d,inString))return false;
					if(!inString)
					{
						switch(d.iDataType)
						{
							case KVI_DATA_TYPE_DICTKEYSREFERENCE:
								if(d.pDictionary)
								{
									unsigned int cnt = d.pDictionary->count();
									if(cnt > 0)
									{
										KviDictionaryIterator it(*(d.pDictionary));
										
										buffer.append(it.currentKey());
										if(cnt > 1)
										{
											paramList->append(new KviStr(buffer));
											++it;
											cnt--;
	
											while(cnt > 1)
											{
												paramList->append(new KviStr(it.currentKey()));
												++it;
												cnt--;
											}
											buffer = it.currentKey();
										}
									}
								}
							break;
							case KVI_DATA_TYPE_DICTREFERENCE:
								if(d.pDictionary)
								{
									unsigned int cnt = d.pDictionary->count();
									if(cnt > 0)
									{
										KviDictionaryIterator it(*(d.pDictionary));
										buffer.append(*(it.current()));
										if(cnt > 1)
										{
											paramList->append(new KviStr(buffer));
											++it;
											cnt--;
											while(cnt > 1)
											{
												paramList->append(new KviStr(*(it.current())));
												++it;
												cnt--;
											}
											buffer = *(it.current());
										}
									}
								}
							break;
							case KVI_DATA_TYPE_ARRAYREFERENCE:
								if(d.pArray)
								{
									if(d.pArray->size() > 0)
									{
										unsigned int cnt = d.pArray->size() - 1;
										KviStr * t = d.pArray->uncheckedAt(0);
										if(*t)buffer.append(*t);
										if(cnt > 0)
										{
											paramList->append(new KviStr(buffer));
											unsigned int u = 1;
											for(;u < cnt;u++)
											{
												t = d.pArray->uncheckedAt(u);
												if(t)paramList->append(new KviStr(*t));
												else paramList->append(new KviStr());
											}
											t = d.pArray->uncheckedAt(u);
											if(t)buffer = *t;
										}
									}
								}
							break;
						}
					}
				}
				aux_ptr = c->m_ptr;
			break;
			case '$': //system identifier: append the last block to the buffer and process the ident
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				if(bSkipIdentAndVars)
				{
					if(!skipIdentifier(c))return false;
					buffer.append(aux_ptr,c->m_ptr - aux_ptr);
				} else {
					KviParameterList * old = c->returnParameterList();
					c->setReturnParameterList(paramList);
					if(!parseIdentifier(c,buffer))return false;
					c->setReturnParameterList(old);
				}
				aux_ptr = c->m_ptr;
			break;
			case '\\': //escape character: append the last block to the processed buffer...
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = ++aux_ptr;
				switch(*aux_ptr)
				{
					case '\0':
						if(inString)return c->error(KviError_unexpectedEndInString);
						else return c->error(KviError_unexpectedEndInFunctionParams);
					break;
					case '\n':  //escaped newline
						c->m_ptr = ++aux_ptr; //skip it
						c->skipSpace();       //skip leading spaces
						aux_ptr = c->m_ptr;   //continue
					break;
					case 'r':
						if(inString)buffer.append('\r');
						else buffer.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
					case 'n':
						if(inString)buffer.append('\n');
						else buffer.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
					default: // Must be copied to the buffer without modification
						buffer.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
				}
			break;
		}
	}
	// Newer arrive here...
	return c->error(KviError_internalError);
}

/*
bool KviUserParser::parseFncSingleParamNoIdent(KviCommand *c,KviParameterList * paramList)
{
	ENTER_CONTEXT(c,"parseFncSingleParamNoIdent");
	// Skip leading spaces and tabs (the escaped ones too)

	c->skipSpace();

	register char *aux_ptr = c->m_ptr;

	int parLevel       = 0;
	bool inString      = false;

	KviStr buffer;

	for(;;){
		// First skip all the non interesting chars
//		while(	*aux_ptr         && (*aux_ptr!='%') &&
//				(*aux_ptr!='\n') && (*aux_ptr!=')')  && 
//				(*aux_ptr!=',')  && (*aux_ptr!='$')  && 
//				(*aux_ptr!=' ')  && (*aux_ptr!='(')  &&
//				(*aux_ptr!='"')  && (*aux_ptr!='\\') &&
//				(*aux_ptr!='\t'))aux_ptr++;
		while(!parameter_parsing_char_table[*aux_ptr])aux_ptr++;


		// Interesting char
		switch(*aux_ptr)
		{
			case '\0':
			case '\n': // End of the string ('\n' is not escaped)...fail here
				if(inString)return c->error(KviError_unexpectedEndInString);
				else return c->error(KviError_unexpectedEndInFunctionParams);
			break;
			case ' ':
			case '\t': // Spaces...need to reduce it to one...
				if(!inString)
				{
					// append the last part
					buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
					// skip the other spaces including escaped ones
					c->m_ptr = aux_ptr;
					c->skipSpace();
					aux_ptr = c->m_ptr;
					if((*aux_ptr != ')') && (*aux_ptr != ','))buffer.append(' ');
				} else {
					while((*aux_ptr == ' ')||(*aux_ptr == '\t'))aux_ptr++;
				}
			break;
			case '"': // Not escaped '"'...enter or exit the string (skip the '"')
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = ++aux_ptr;
				inString = !inString;
			break;
			case '(': // Keep track of matching parenthesis (if not in string)
				++aux_ptr;
				if(!inString)++parLevel;
			break;
			case ')': // If not in string , exit if parenthesis level is 0 , otherwise decrease it
				if(!inString)
				{
					if(parLevel == 0)
					{
						buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
						c->m_ptr = aux_ptr;
						paramList->append(new KviStr(buffer));
						return c->leaveContext();
					} else --parLevel;
				}
				++aux_ptr;
			break;
			case ',': // end of command buffer...append the last block to the buffer and return
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				if(!inString)
				{
					c->m_ptr = aux_ptr;
					paramList->append(new KviStr(buffer));
					return c->leaveContext();
				}
				c->m_ptr = aux_ptr++; //in string...',' must be included
			break;
			case '%': //variable: append the last block to the buffer and process the var
			{	
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				if(!skipVariable(c))return false;
				buffer.append(aux_ptr,c->m_ptr - aux_ptr);
				aux_ptr = c->m_ptr;
			}
			break;
			case '$': //system identifier: append the last block to the buffer and process the ident
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				if(!skipIdentifier(c))return false;
				buffer.append(aux_ptr,c->m_ptr - aux_ptr);
				aux_ptr = c->m_ptr;
			break;
			case '\\': //escape character: append the last block to the processed buffer...
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = ++aux_ptr;
				switch(*aux_ptr){
					case '\0':
						if(inString)return c->error(KviError_unexpectedEndInString);
						else return c->error(KviError_unexpectedEndInFunctionParams);
					break;
					case '\n':  //escaped newline
						c->m_ptr = ++aux_ptr; //skip it
						c->skipSpace();       //skip leading spaces
						aux_ptr = c->m_ptr;   //continue
					break;
					case 'r':
						if(inString)buffer.append('\r');
						else buffer.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
					case 'n':
						if(inString)buffer.append('\n');
						else buffer.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
					default: // Must be copied to the buffer without modification
						buffer.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
				}
			break;
		}
	}
	// Newer arrive here...
	return c->error(KviError_internalError);
}
*/

#include "kvi_uparser.moc"
