//
//   File : kvi_up_fnc.cpp
//   Creation date : Mon Jul 03 2000 20:41:21 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2002 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#define __KVIRC__

#include "kvi_debug.h"
#include "kvi_uparser.h"
#include "kvi_command.h"
#include "kvi_app.h"
#include "kvi_error.h"
#include "kvi_window.h"
#include "kvi_locale.h"
#include "kvi_console.h"
#include "kvi_query.h"
#include "kvi_channel.h"
#include "kvi_scriptobject.h"
#include "kvi_modulemanager.h"
#include "kvi_parameterlist.h"
#include "kvi_ircuserdb.h"
#include "kvi_avatar.h"
#include "kvi_timermanager.h"
#include "kvi_ircuserdb.h"
#include "kvi_mirccntrl.h"


#include <stdlib.h> // for rand()
#include <time.h>   // for time()

#include <qregexp.h>


// kvi_app.cpp
extern KviTimerManager * g_pTimerManager;

void KviUserParser::initFunctionDict()
{
	m_pFunctionDict = new QAsciiDict<KviFunctionParseProc>(109);
	m_pFunctionDict->setAutoDelete(true);
/*
#ifdef COMPILE_ON_WINDOWS
	#define ALLOC_PARSE_PROC(__proc) new functionParseProc(&KviUserParser::__proc)
#else
	#define ALLOC_PARSE_PROC(__proc) new functionParseProc(&(KviUserParser::__proc))
#endif
*/

	KviFunctionParseProc * p;

#define ALLOC_PARSE_PROC(__proc) \
	p = new KviFunctionParseProc; \
	p->proc = KVI_PTR2MEMBER(KviUserParser::__proc)

#define FNC_REG(__name,__proc) \
		ALLOC_PARSE_PROC(__proc); \
		m_pFunctionDict->insert(__name,p);

	FNC_REG("WINDOW",parseFnc_WINDOW);
	FNC_REG("CHANNEL",parseFnc_CHANNEL);
	FNC_REG("QUERY",parseFnc_QUERY);
	FNC_REG("CONSOLE",parseFnc_CONSOLE);
	FNC_REG("CONTEXT",parseFnc_CONTEXT);
	FNC_REG("IC",parseFnc_CONTEXT);
	FNC_REG("CAPTION",parseFnc_CAPTION);
	FNC_REG("NEW",parseFnc_NEW);
	FNC_REG("THIS",parseFnc_THIS);
	FNC_REG("RAND",parseFnc_RAND);
	FNC_REG("UNIXTIME",parseFnc_UNIXTIME);
	FNC_REG("SELECTED",parseFnc_SELECTED);
	FNC_REG("ISMEOP",parseFnc_ISMEOP);
	FNC_REG("TARGET",parseFnc_TARGET);
	FNC_REG("OPTION",parseFnc_OPTION);
	FNC_REG("MASK",parseFnc_MASK);
	FNC_REG("AVATAR",parseFnc_AVATAR);
	FNC_REG("ISTIMER",parseFnc_ISTIMER);
	FNC_REG("AWAY",parseFnc_AWAY);
	FNC_REG("ISWELLKNOWN",parseFnc_ISWELLKNOWN);
	FNC_REG("USERNAME",parseFnc_USERNAME);
	FNC_REG("HOSTNAME",parseFnc_HOSTNAME);
	FNC_REG("SERVER",parseFnc_SERVER);
	FNC_REG("FMTLINK",parseFnc_FMTLINK);
	FNC_REG("LF",parseFnc_LF);
	FNC_REG("CR",parseFnc_CR);
	FNC_REG("ASCII",parseFnc_ASCII);
	FNC_REG("CHAR",parseFnc_CHAR);
	FNC_REG("K",parseFnc_K);
	FNC_REG("B",parseFnc_B);
	FNC_REG("U",parseFnc_U);
	FNC_REG("O",parseFnc_O);
	FNC_REG("R",parseFnc_R);
	FNC_REG("ISANYCONSOLECONNECTED",parseFnc_ISANYCONSOLECONNECTED);
	FNC_REG("ME",parseFnc_ME);
	FNC_REG("TIME",parseFnc_TIME);
	FNC_REG("BASE64TOTEXT",parseFnc_BASE64TOASCII);
	FNC_REG("HEXTOASCII",parseFnc_HEXTOASCII);
	FNC_REG("ACTIVE",parseFnc_ACTIVE);
	FNC_REG("ISMEVOICE", parseFnc_ISMEVOICE);
	FNC_REG("ISMEHALFOP", parseFnc_ISMEHALFOP);
	FNC_REG("CLASSDEFINED",parseFnc_CLASSDEFINED);
	FNC_REG("FEATURES",parseFnc_FEATURES);
	FNC_REG("SPLIT",parseFnc_SPLIT);
//	FNC_REG("JOIN",parseFnc_JOIN);
}

bool KviUserParser::parseFnc_BASE64TOASCII(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: base64toAscii
		@type:
			function
		@title:
			$base64ToAscii
		@short:
			Returns a decoded base64 string
		@syntax:
			$bast64ToAscii(<base_64_encoded_string>)
		@description:
		@examples:
	*/

	ENTER_CONTEXT(c,"parseFnc_BASE64TOASCII");
	char * buf;

	KviStr * pStr = params->first();
	if(pStr)
	{
		int len = pStr->base64ToBuffer(&buf,true);
		if(len > 0)
		{
			buffer.append(buf,len);
			KviStr::freeBuffer(buf);
		}
	}
	return c->leaveContext();
}

bool KviUserParser::parseFnc_HEXTOASCII(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: hextoAscii
		@type:
			function
		@title:
			$hexToAscii
		@short:
			Returns a decoded hex string
		@syntax:
			$hexToAscii(<hex_encoded_string>)
		@description:
		@examples:
	*/

	ENTER_CONTEXT(c,"parseFnc_HEXTOASCII");
	char * buf;

	KviStr * pStr = params->first();
	if(pStr)
	{
		int len = pStr->hexToBuffer(&buf,true);
		if(len > 0)
		{
			buffer.append(buf,len);
			KviStr::freeBuffer(buf);
		}
	}
	return c->leaveContext();
}

bool KviUserParser::parseFnc_CAPTION(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: caption
		@type:
			function
		@title:
			$caption
		@short:
			Returns the caption text of the specified window.
		@syntax:
			$caption
			$caption(<window id>)
		@description:
			The form with the <window id> parameter returns the caption text
			of the window that has the specified id.
			The form without parameters returns the caption of the current window,
			thus it is equivalent to calling $caption([fnc]$window[/fnc])
		@examples:
			[example]
			[cmd]echo[/cmd] $caption
			[cmd]echo[/cmd] $caption([fnc]$console[/fnc])
			[/example]
		@seealso:
			[fnc]$window[/fnc],
			[fnc]$console[/fnc],
			[doc:window_naming_conventions]Window naming conventions[/doc]
	*/

	ENTER_CONTEXT(c,"parseFnc_CAPTION");

	KviStr * pStr = params->first();
	if(pStr)
	{
		KviWindow * wnd = g_pApp->findWindow(pStr->ptr());
		if(wnd)buffer.append(wnd->plainTextCaption());
		else c->warning(__tr("Window with id '%s' not found, returning empty string"),pStr->ptr());
	} else buffer.append(c->window()->plainTextCaption());

	return c->leaveContext();
}



bool KviUserParser::parseFnc_TARGET(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: target
		@type:
			function
		@title:
			$target
		@short:
			Returns the target of the current window
		@syntax:
			$target
			$target(<window id>)
		@description:
			The form with the <window id> parameter returns the target
			of the channel,query or dcc that has the specified id.
			The form without parameters returns the target of the current window,
			thus it is equivalent to calling $target([fnc]$window[/fnc]).
			For channel windows the target is the channel name,
			for query windows it is the list of the "queried" users, for the
			dcc windows it is the remote end of the connection.
			The other windows have an empty target.
		@examples:
			[example]
			[cmd]echo[/cmd] $target
			[/example]
		@seealso:
			[fnc]$window[/fnc],
			[fnc]$console[/fnc],
			[fnc]$channel[/fnc],
			[fnc]$query[/fnc],
			[doc:window_naming_conventions]Window naming conventions[/doc]
	*/

	ENTER_CONTEXT(c,"parseFnc_TARGET");

	KviStr * pStr = params->first();
	if(pStr)
	{
		KviWindow * wnd = g_pApp->findWindow(pStr->ptr());
		if(wnd)buffer.append(wnd->target());
		else c->warning(__tr("Window with id '%s' not found, returning empty string"),pStr->ptr());
	} else buffer.append(c->window()->target());

	return c->leaveContext();
}



bool KviUserParser::parseFnc_CHANNEL(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: channel
		@type:
			function
		@title:
			$channel
		@short:
			Retrieve the window id of a specified channel
		@syntax:
			$channel[(<channel name>[,<irc context id>])]
		@description:
			Returns the [b]window id[/b] of channel matching the
			<channel name> and bound to the connection specified by
			<irc context id>[br]
			If no window matches the specified name or connection, and invalid
			window id is returned (0).[br]
			If no <irc context id> is specified , this function looks for
			the channel in the current connection context (if any).[br]
			If no <channel name> is specified, this function returns the current
			channel window id, if executed in a channel , 0 otherwise.[br]
		@examples:
			[example]
			[/example]
		@seealso:
			[fnc]$window[/fnc],
			[fnc]$query[/fnc],
			[fnc]$console[/fnc]
			[doc:window_naming_conventions]Window naming conventions[/doc]
	*/

	ENTER_CONTEXT(c,"parseFnc_CHANNEL");

	KviWindow *wnd = 0;

	KviStr * pName = params->first();
	KviStr * pCntx = params->next();

	if(pName)
	{
		if(pCntx)
		{
			bool bOk;
			unsigned int ircContextId = pCntx->toUInt(&bOk);
			if(!bOk)return c->error(KviError_invalidIrcContextId,pCntx->ptr());
			KviConsole * cons = g_pApp->findConsole(ircContextId);
			if(!cons)c->warning(__tr("No such irc context (%u)"),ircContextId);
			else wnd = cons->findChannel(pName->ptr());
		} else {
			if(c->window()->console())wnd = c->window()->console()->findChannel(pName->ptr());
			else c->warning(__tr("This window has no associated irc context"));
		}
	} else if(c->window()->type() == KVI_WINDOW_TYPE_CHANNEL)wnd = c->window();

	buffer.append(wnd ? wnd->id() : "0");

	return c->leaveContext();
}

bool KviUserParser::parseFnc_CONSOLE(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: console
		@type:
			function
		@title:
			$console
		@short:
			Retrieve the window id of a specified console
		@syntax:
			$console[(<irc context id>)]
		@description:
			Returns the [b]window id[/b] of the console bound
			to the irc context specified by <irc context id>.
			If no window matches the specified irc context, and invalid
			window id is returned (0).[br]
			If no <irc context id> is specified , this function looks for
			the console in the current irc context (if any).[br]
		@examples:
			[example]
			[/example]
		@seealso:
			[fnc]$window[/fnc],
			[fnc]$channel[/fnc],
			[fnc]$query[/fnc]
			[doc:window_naming_conventions]Window naming conventions[/doc]
	*/

	ENTER_CONTEXT(c,"parseFnc_CONSOLE");

	KviWindow *wnd = 0;

	KviStr * pCntx = params->first();

	if(pCntx)
	{
		bool bOk;
		unsigned int ircContextId = pCntx->toUInt(&bOk);
		if(!bOk)return c->error(KviError_invalidIrcContextId,pCntx->ptr());
		wnd = g_pApp->findConsole(ircContextId);
	} else {
		if(c->window()->console())wnd = c->window()->console();
		else c->warning(__tr("This window has no associated irc context"));
	}

	buffer.append(wnd ? wnd->id() : "0");

	return c->leaveContext();
}


bool KviUserParser::parseFnc_ACTIVE(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: active
		@type:
			function
		@title:
			$active
		@short:
			Retrieve the window id of the active window
		@syntax:
			$active[(<irc context id>)]
		@description:
			Returns the [b]window id[/b] of the active window 
			bound to the irc context specified by <irc context id>.
			If no window matches the specified irc context, and invalid
			window id is returned (0).[br]
			If no <irc context id> is specified , this function returns
			the active window of the current irc context (if any).[br]
			If the current window is not bound to any irc context,
			its own id is returned.
		@examples:
			[example]
			[/example]
		@seealso:
			[fnc]$window[/fnc],
	*/

	ENTER_CONTEXT(c,"parseFnc_ACTIVE");

	KviWindow *wnd = 0;

	KviStr * pCntx = params->first();

	if(pCntx)
	{
		bool bOk;
		unsigned int ircContextId = pCntx->toUInt(&bOk);
		if(!bOk)return c->error(KviError_invalidIrcContextId,pCntx->ptr());
		wnd = g_pApp->findConsole(ircContextId);
		if(wnd)wnd = ((KviConsole *)wnd)->activeWindow();
	} else {
		if(c->window()->console())
		{
			wnd = c->window()->console()->activeWindow();
		} else {
			wnd = c->window();
		}
	}

	buffer.append(wnd ? wnd->id() : "0");

	return c->leaveContext();
}



bool KviUserParser::parseFnc_QUERY(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: query
		@type:
			function
		@title:
			$query
		@short:
			Retrieve the window id of a specified query
		@syntax:
			$query[(<target>[,<irc context id>])]
		@description:
			Returns the [b]window id[/b] of the query that has <target>
			in the list of targets and is bound to the connection specified by
			<irc context id>[br]
			If no window matches the specified target or context, and invalid
			window id is returned (0).[br]
			If no <irc context id> is specified , this function looks for
			the query in the current connection context (if any).[br]
			If no <target> is specified, this function returns the current
			query window id, if executed in a query , 0 otherwise.[br]
		@examples:
			[example]
			[/example]
		@seealso:
			[fnc]$window[/fnc],
			[fnc]$channel[/fnc],
			[fnc]$console[/fnc]
			[doc:window_naming_conventions]Window naming conventions[/doc]
	*/

	ENTER_CONTEXT(c,"parseFnc_QUERY");

	KviWindow *wnd = 0;

	KviStr * pName = params->first();
	KviStr * pCntx = params->next();

	if(pName)
	{
		if(pCntx)
		{
			bool bOk;
			unsigned int ircContextId = pCntx->toUInt(&bOk);
			if(!bOk)return c->error(KviError_invalidIrcContextId,pCntx->ptr());
			KviConsole * cons = g_pApp->findConsole(ircContextId);
			if(!cons)c->warning(__tr("No such irc context (%u)"),ircContextId);
			else wnd = cons->findQuery(pName->ptr());
		} else {
			if(c->window()->console())wnd = c->window()->console()->findQuery(pName->ptr());
			else c->warning(__tr("This window has no associated irc context"));
		}
	} else if(c->window()->type() == KVI_WINDOW_TYPE_QUERY)wnd = c->window();

	buffer.append(wnd ? wnd->id() : "0");

	return c->leaveContext();
}

bool KviUserParser::parseFnc_CONTEXT(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: context
		@type:
			function
		@title:
			$context
		@short:
			Retrieve the id of the specified irc context
		@syntax:
			$context[(<server>,<nickname>)]
		@description:
			Returns the [b]irc context id[/b] of the IRC context that uses
			the specified <server> and <nickname>.[br] This function can
			find only connected irc contexts.
			If no context matches the server and nickname, and invalid
			[b]irc context id[/b] is returned (0).[br]
			If <server> is an empty string, the first context that matches
			the specified nickname is returned. If <nickname> is an empty string
			the first context that uses the specified server is returned.
			If both parameters are missing this function returns the
			id of the current irc context, or '0' if the
			window in that this call is executed is not bound to any irc context.
			Please note that in this last case you may find an [b]irc context[/b]
			that is 'not connected'.
			This can only happen if the current window is a console that is
			currently in "idle" state: no connection has been estabilished yet.[br]
			It is a good idea to take a look at the
			[doc:window_naming_conventions]window naming conventions document[/doc].
			This identifier is equivalent to [fnc]$ic[/fnc].[br]
		@examples:
			[example]
			[/example]
		@seealso:
	*/

	/*
		@doc: ic
		@type:
			function
		@title:
			$ic
		@short:
			Retrieve the id of the specified irc context
		@syntax:
			$ic[(<server>,<nickname>)]
		@description:
			This identifier is equivalent to [fnc]$context[/fnc].
			See the related documentation.[br]
		@examples:
			[example]
			[/example]
		@seealso:
	*/


	ENTER_CONTEXT(c,"parseFnc_CONTEXT");

	KviStr * pServ = params->first();
	KviStr * pNick = params->next();

	if(!pServ)pServ = &m_szEmptyString;
	if(!pNick)pNick = &m_szEmptyString;

	KviConsole * cons = 0;

	if(pServ->hasData() || pNick->hasData())
	{
		// some parameters passed
		cons = g_pApp->findConsole(*pServ,*pNick);
	} else {
		// both are empty
		cons = c->window()->console();
		if(!cons)c->warning(__tr("This window has no associated irc context"));
	}

	if(cons)buffer.append(KviStr::Format,"%u",cons->ircContextId());
	else buffer.append('0');

	return c->leaveContext();
}

// FIXME: #warning "$window should accept the type as optional parameter"
// FIXME: #warning "$type"

bool KviUserParser::parseFnc_WINDOW(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: window
		@type:
			function
		@title:
			$window
		@short:
			Retrieve the id of a specified window.
		@syntax:
			$window[(<caption text>)]
		@description:
			Returns the [b]window id[/b] of the first window that
			has the specified <caption text>.[br]
			If no window matches the specified <caption text>, and invalid
			window id is returned (0).[br]
			If no <caption text> is specified , this function returns the id
			of the current window.[br]
		@examples:
			[example]
			[/example]
		@seealso:
			[fnc]$channel[/fnc],
			[fnc]$query[/fnc],
			[fnc]$console[/fnc],
			[doc:window_naming_conventions]Window naming conventions[/doc]
	*/

	ENTER_CONTEXT(c,"parseFnc_WINDOW");

	KviWindow *wnd = c->window();

	KviStr * pStr = params->first();

	if(pStr)
	{
		wnd = g_pApp->findWindowByCaption(pStr->ptr());
		if(!wnd)
		{
			c->warning(__tr("Window with caption '%s' not found, returning 0"),pStr->ptr());
			buffer.append('0');
			return c->leaveContext();
		}
	}

	buffer.append(wnd->id());

	return c->leaveContext();
}

bool KviUserParser::parseFnc_NEW(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: new
		@type:
			function
		@title:
			$new
		@short:
			Create a new object
		@syntax:
			$new(<class>,[<parent_id> [,<name>]])
		@description:
			Creates a new instance of the object <class> with
			the parent object <parent_id> and the specified <name>.[br]
			<name> and <parent_id> are optional: if not specified , <name>
			is assumed to be an empty string and <parent_id> default to 0 (parentless object).[br]
			Please see the [doc:objects]objects documentation[/doc] for more informations.[br]
		@examples:
			[example]
				%myobj = $new(widget,0,pippo)
			[/example]
		@seealso:
			[doc:objects]Objects documentation[/doc], [cmd]delete[/cmd]

	*/

	ENTER_CONTEXT(c,"parseFnc_NEW");

	KviStr * pszClass  = params->first();
	if(!pszClass)return c->error(KviError_missingObjectClassName);

	KviStr * pszParent = params->next();
	KviScriptObject * pParent = 0;
	if(pszParent)
	{
		if(!kvi_strEqualCS(pszParent->ptr(),"0"))
		{
			pParent = g_pScriptObjectController->lookupObject(pszParent->ptr());
			if(!pParent)return c->error(KviError_noSuchObject,"%s",pszParent->ptr());
		}
	}

	KviStr * pszName   = pszParent ? params->next() : 0;

	KviScriptObjectClass * pClass = g_pScriptObjectController->lookupClass(pszClass->ptr());

	if(!pClass)return c->error(KviError_noSuchObjectClass,"%s",pszClass->ptr());

	KviScriptObject * ob = g_pScriptObjectController->allocateObjectInstance(pClass,
			pParent,pszName ? pszName->ptr() : "");

	if(ob)
	{
		KviStr ret;
		if(!ob->callEventFunction("constructor",&ret))
		{
			// ops...constructor failed (script error!)
			delete ob;
			ob = 0;
		} else {
			if(kvi_strEqualCI(ret.ptr(),"0"))
			{
				// implementation failure...
				delete ob;
				ob = 0;
			}
		}
	}

	buffer.append(ob ? ob->id() : "0");
	return c->leaveContext();
}

bool KviUserParser::parseFnc_THIS(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: this
		@type:
			function
		@title:
			$this
		@short:
			Retrieve the id of the current object
		@syntax:
			$this
		@description:
			Returns the id of the current object or ('0') if there is
			none. This function has a "quick" version with syntax:
			[b]$$[/b][br]
		@examples:
			[example]
			[/example]
		@seealso:
			
	*/

	/*
		@doc: $
		@type:
			function
		@title:
			$$
		@short:
			Retrieve the id of the current object
		@syntax:
			$$
		@description:
			Returns the id of the current object or ('0') if there is
			none. This function has equivalent to [fnc]$this[/fnc]
		@examples:
			[example]
			[/example]
		@seealso:
			
	*/

// FIXME: #warning "BETTER DOCS FOR THIS"

	ENTER_CONTEXT(c,"parseFnc_THIS");

	if(c->thisPointer())buffer.append(c->thisPointer()->id());
	else buffer.append('0');

	return c->leaveContext();
}


bool KviUserParser::parseFnc_CLASSDEFINED(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: classdefined
		@type:
			function
		@title:
			$classdefined
		@short:
			Check if a class is defined
		@syntax:
			$classdefined(<class_name>)
		@description:
			Returns 1 if the class <class_name> is defined, 0 otherwise.[br]
		@examples:
			[example]
			[/example]
		@seealso:
			
	*/

	ENTER_CONTEXT(c,"parseFnc_CLASSDEFINED");

	KviStr szClass = params->safeFirstParam();

	buffer.append(g_pScriptObjectController->lookupClass(szClass.ptr()) ? '1' : '0');

	return c->leaveContext();
}

bool KviUserParser::parseFnc_RAND(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: rand
		@type:
			function
		@title:
			$rand
		@short:
			Generate a random number
		@syntax:
			$rand(<max>)
		@description:
			Returns a random integer number from 0 to max inclusive.
			You can repeat sequences of random numbers by calling [cmd]srand[/cmd]
			with the same seed value. If [cmd]srand[/cmd] has not been called
			$rand is automatically seeded with value of 1.
			If no <max> is specified , this function returns an integer between
			0 and RAND_MAX that is system dependant.
		@examples:
			[example]
			[/example]
		@seealso:
			[cmd]srand[/cmd]
	*/

	ENTER_CONTEXT(c,"parseFnc_RAND");

	bool bOk;
	int max = params->safeFirst()->toInt(&bOk);
	if(!bOk)max = RAND_MAX;
	buffer.append(KviStr::Format,"%d",rand() % max);
	return c->leaveContext();
}

bool KviUserParser::parseFnc_UNIXTIME(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: unixtime
		@type:
			function
		@title:
			$unixTime
		@short:
			Returns the current unix time
		@syntax:
			$unixtime
		@description:
			Returns the time since the Epoch (00:00:00 UTC, January 1, 1970),
			measured in seconds.  
		@seealso:
			[fnc]$time[/fnc]
		@examples:
			[example]
			[/example]
	*/

	buffer.append(KviStr::Format,"%u",time(0));
	return true;
}

bool KviUserParser::parseFnc_TIME(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: time
		@type:
			function
		@title:
			$time
		@short:
			Returns a formatted time string
		@syntax:
			$time(<unixtime>)
			$time
		@description:
			Returns the string rappresentation of the <unixtime> or
			of the current time if <unixtime> is not given.[br]  
		@examples:
			[example]
			[/example]
	*/

	ENTER_CONTEXT(c,"parseFnc_TIME");

	KviStr * pStr = params->safeFirst();

	time_t t;
	if(pStr->hasData())
	{
		bool bOk;
		t = (time_t)pStr->toUInt(&bOk);
		if(!bOk)
		{
			c->warning(__tr("The specified unixtime is not valid (%s)"),pStr->ptr());
			return c->leaveContext();
		}
	} else {
		t = time(0);
	}

	buffer.append(ctime(&t));
	if(buffer.lastCharIs('\n'))buffer.cutRight(1);
	return c->leaveContext();
}


bool KviUserParser::parseFnc_SELECTED(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
// FIXME: #warning "THIS HAS TO WORK FOR QUERIES TOO!"
	/*
		@doc: selected
		@type:
			function
		@title:
			$selected
		@short:
			Returns the list of selected nicknames in the channel
		@syntax:
			$selected
			$selected(<window id>)
		@description:
			The form with the <window id> parameter returns an array of the selected
			nicknames in the channel designated by <window id>.
			The form without parameters returns an array of the selected nicknames
			in the current window (assuming that it is a channel),
			thus it is equivalent to calling $selected([fnc]$window[/fnc])
			The returned value may be assigned to a dictionary too: it will be used to simulate an array.[br]
			In a non-array/dictionary context it returns the selected nicknames as a comma separated list.
		@examples:
			[example]
				[cmd]echo[/cmd] $selected
				[cmd]foreach[/cmd](%i,$selected)[cmd]echo[/cmd] %i
			[/example]
		@seealso:
			[fnc]$window[/fnc],
			[fnc]$channel[/fnc],
			[doc:window_naming_conventions]Window naming conventions[/doc]
	*/

	ENTER_CONTEXT(c,"parseFnc_SELECTED");

	KviStr * pStr = params->first();
	KviWindow * wnd = c->window();
	if(pStr)
	{
		wnd = g_pApp->findWindow(pStr->ptr());
		if(!wnd)
		{
			c->warning(__tr("Window with id '%s' not found, returning empty string"),pStr->ptr());
			return c->leaveContext();
		}
	}

	if(wnd->type() != KVI_WINDOW_TYPE_CHANNEL)
	{
		c->warning(__tr("The specified window is not a channel"));
		return c->leaveContext();
	}

	c->beginListArrayOrDictionaryReturnIdentifier();
	int id = 0;

	for(KviStr * s = ((KviChannel *)wnd)->firstSelectedNickname();s;s = ((KviChannel *)wnd)->nextSelectedNickname())
	{
		c->addListArrayOrDictionaryReturnValue(id++,*s,buffer);
	}

	return c->leaveContext();
}

bool KviUserParser::parseFnc_ISMEOP(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: ismeop
		@type:
			function
		@title:
			$isMeOp
		@short:
			Returns 1 if the current user has operator status
		@syntax:
			$ismeop
			$ismeop(<window id>)
		@description:
			The form with the <window id> parameter returns '1' if the current
			user has operator status on channel designated by <window id>.
			The form without parameters is equivalent to $ismeop([fnc]$window[/fnc])
		@seealso:
			[fnc]$window[/fnc],
			[fnc]$channel[/fnc],
			[doc:window_naming_conventions]Window naming conventions[/doc]
	*/
	ENTER_CONTEXT(c,"parseFnc_ISMEOP");

	KviStr * pStr = params->first();
	KviWindow * wnd = c->window();
	if(pStr)
	{
		wnd = g_pApp->findWindow(pStr->ptr());
		if(!wnd)
		{
			c->warning(__tr("Window with id '%s' not found, returning empty string"),pStr->ptr());
			return c->leaveContext();
		}
	}

	if(wnd->type() != KVI_WINDOW_TYPE_CHANNEL)
	{
		c->warning(__tr("The specified window is not a channel"));
		return c->leaveContext();
	}
		
	buffer.append(((KviChannel *)wnd)->isMeOp() ? '1' : '0');

	return c->leaveContext();
}

bool KviUserParser::parseFnc_ISMEVOICE(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
    /*
        @doc: ismevoice
        @type:
            function
        @title:
            $isMeVoice
        @short:
            Returns 1 if the current user has voice status
        @syntax:
            $ismevoice
            $ismevoice(<window id>)
        @description:
            The form with the <window id> parameter returns '1' if the current
            user has voice status on channel designated by <window id>.
            The form without parameters is equivalent to $ismevoice([fnc]$window[/fnc])
        @seealso:
			[fnc]$isvoice[/fnc],
			[fnc]$isop[/fnc],
			[fnc]$ismeop[/fnc],
			[fnc]$ishalfop[/fnc],
			[fnc]$ismehalfop[/fnc],
            [fnc]$window[/fnc],
            [fnc]$channel[/fnc],
            [doc:window_naming_conventions]Window naming conventions[/doc]

    */
	ENTER_CONTEXT(c,"parseFnc_ISMEVOICE");

	KviStr * pStr = params->first();
	KviWindow * wnd = c->window();
	if(pStr)
	{
		wnd = g_pApp->findWindow(pStr->ptr());
		if(!wnd)
		{
			c->warning(__tr("Window with id '%s' not found, returning empty string"),pStr->ptr());
			return c->leaveContext();
		}
	}

	if(wnd->type() != KVI_WINDOW_TYPE_CHANNEL)
	{
		c->warning(__tr("The specified window is not a channel"));
		return c->leaveContext();
	}
	buffer.append(((KviChannel *)wnd)->isMeVoice() ? '1' : '0');

	return c->leaveContext();
}
	
bool KviUserParser::parseFnc_ISMEHALFOP(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
    /*
        @doc: ismehalfop
        @type:
            function
        @title:
            $isMeHalfOp
        @short:
            Returns 1 if the current user has half-op status
        @syntax:
            $ismehalfop
            $ismehalfop(<window id>)
        @description:
            The form with the <window id> parameter returns '1' if the current
            user has half-op status on channel designated by <window id>.
            The form without parameters is equivalent to $ismehalfop([fnc]$window[/fnc])
        @seealso:
			[fnc]$ismevoice[/fnc],
            [fnc]$isvoice[/fnc],
            [fnc]$isop[/fnc],
            [fnc]$ismeop[/fnc],
            [fnc]$ishalfop[/fnc],
            [fnc]$window[/fnc],
            [fnc]$channel[/fnc],
            [doc:window_naming_conventions]Window naming conventions[/doc]

    */
	ENTER_CONTEXT(c,"parseFnc_ISMEHALFOP");

	KviStr * pStr = params->first();
	KviWindow * wnd = c->window();
	if(pStr)
	{
		wnd = g_pApp->findWindow(pStr->ptr());
		if(!wnd)
		{
			c->warning(__tr("Window with id '%s' not found, returning empty string"),pStr->ptr());
            return c->leaveContext();
		}
	}

	if(wnd->type() != KVI_WINDOW_TYPE_CHANNEL)
	{
		c->warning(__tr("The specified window is not a channel"));
		return c->leaveContext();
	}

	buffer.append(((KviChannel *)wnd)->isMeHalfOp() ? '1' : '0');

	return c->leaveContext();
}

bool KviUserParser::parseFnc_ISTIMER(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: istimer
		@type:
			function
		@title:
			$isTimer
		@short:
			Checks for a timer existence
		@syntax:
			$istimer(<name>)
		@description:
			Returns 1 if the timer named <name> is actually running, 0 otherwise
		@seealso:
			[cmd]timer[/cmd],[cmd]killtimer[/cmd]
	*/

	ENTER_CONTEXT(c,"parseFnc_ISTIMER");

	KviStr * pStr = params->first();
	if(pStr)buffer.append(g_pTimerManager->isTimer(pStr->ptr()) ? '1' : '0');
	else return c->error(KviError_notEnoughParameters,__tr("Missing timer name"));

	return c->leaveContext();
}

bool KviUserParser::parseFnc_OPTION(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: option
		@type:
			function
		@title:
			$option
		@short:
			Returns the value of an option
		@syntax:
			$option(<optionName>)
		@description:
			Returns the current value of the internal option named <optionName>.
			See the [cmd]option[/cmd] command documentation for more info about options.
		@examples:
			[example]
				[cmd]echo[/cmd] $option(fontIrcView)
			[/example]
		@seealso:
			[cmd]option[/cmd],
	*/

	ENTER_CONTEXT(c,"parseFnc_OPTION");

	KviStr * pStr = params->first();
	if(pStr)
	{
		KviStr tmp;
		if(g_pApp->getOptionString(pStr->ptr(),tmp))buffer.append(tmp);
		else c->warning(__tr("No option named '%s'"),pStr->ptr());
	} else c->warning(__tr("No option name specified"));

	return c->leaveContext();
}


bool KviUserParser::parseFnc_MASK(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: mask
		@type:
			function
		@title:
			$mask
		@short:
			Retrieve the mask of an user
		@syntax:
			$mask[(<nickname>[,<mask_type>])]
		@description:
			Returns the specified type of mask for the user with <nickname>.[br]
			If the host or username are not known , the mask may contain less information
			than requested.[br]
			If the <nickname> is not given it is assumed to be the current nickname.[br]
			If <mask_type> is not given or is invalid, it is assumed to be 0.[br]
			Available mask types:[br]
			0 : nick!user@machine.host.top  (nick!user@XXX.XXX.XXX.XXX) (default)[br]
			1 : nick!user@*.host.top        (nick!user@XXX.XXX.XXX.*)[br]
			2 : nick!user@*[br]
			3 : nick!*@machine.host.top     (nick!user@XXX.XXX.XXX.XXX)[br]
			4 : nick!*@*.host.top           (nick!user@XXX.XXX.XXX.*)[br]
			5 : nick!*@*[br]
			6 : *!user@machine.host.top     (*!user@XXX.XXX.XXX.XX)[br]
			7 : *!user@*.host.top           (*!user@XXX.XXX.XXX.*)[br]
			8 : *!user@*[br]
			9 : *!*@machine.host.top        (*!*@XXX.XXX.XXX.XXX)[br]
			10: *!*@*.host.top              (*!*@XXX.XXX.XXX.*)[br]
			11: nick!*user@machine.host.top (nick!*user@XXX.XXX.XXX.XXX)[br]
			12: nick!*user@*.host.top       (nick!*user@XXX.XXX.XXX.*)[br]
			13: nick!*user@*[br]
			14: *!*user@machine.host.top    (*!*user@XXX.XXX.XXX.XXX)[br]
			15: *!*user@*.host.top          (*!*user@XXX.XXX.XXX.*)[br]
			16: *!*user@*[br]
			If some data is missing , these types may change:[br]
			For example , if hostname is missing , the mask type 3 or 4 may be reduced to type 5.[br]
			If the user with <nickname> is not found in the current irc context user database,
			an empty string is returned.[br]
		@examples:
		@seealso:
	*/

	ENTER_CONTEXT(c,"parseFnc_MASK");

	KviStr * nk = params->safeFirst();
	KviStr * mt = params->safeNext();
	bool bOk;
	int maskType = mt->toInt(&bOk);
	if((!bOk) || (maskType < 0) || (maskType > 16))maskType = 0;

	if(!(c->window()->console()))return c->noIrcContext();
	KviIrcUserDataBase * db = c->window()->console()->userDataBase();
	if(!db)return c->notConnectedToServer();

	const char * nick = nk->hasData() ? nk->ptr() : c->window()->console()->currentNickName();

	KviIrcUserEntry * e = db->find(nick);
	if(e)
	{
		KviIrcMask u;
		u.setNick(nick);
		u.setUsername(e->user());
		u.setHost(e->host());

		KviStr tmp;
		u.mask(tmp,maskType);

		buffer.append(tmp);
	}

	return c->leaveContext();
}

bool KviUserParser::parseFnc_AVATAR(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: avatar
		@type:
			function
		@title:
			$avatar
		@short:
			Retrieve the current user's avatar
		@syntax:
			$avatar[([<nickname>])]
		@description:
			Returns the full path name of the [doc:ctcp_avatar]avatar[/doc] currently
			associated to the <nickname>. If the <nickname> is not in the
			used database of the current irc context , or it has no associated
			avatar, an empty string is returned.[br]
			Please note that if you're going to send a CTCP AVATAR manually (ie. /[cmd]raw[/cmd] or /[cmd]ctcp[/cmd]),
			you have to cut the leading path.
			If <nickname> is missing , it is assumed to be the current user nickname ($me).[br]
		@examples:
		@seealso:
			[doc:ctcp_avatar]Ctcp avatar protocol documentation[/doc],
			[cmd]avatar[/cmd]
	*/

	ENTER_CONTEXT(c,"parseFnc_AVATAR");

	KviStr * nk = params->safeFirst();

	if(!(c->window()->console()))return c->noIrcContext();

	KviIrcUserDataBase * db = c->window()->console()->userDataBase();
	if(!db)return c->notConnectedToServer();

	const char * nick = nk->hasData() ? nk->ptr() : c->window()->console()->currentNickName();

	KviIrcUserEntry * e = db->find(nick);
	if(e)
	{
		KviAvatar * a = e->avatar();
		if(a)buffer.append(a->path());
	}

	return c->leaveContext();
}

bool KviUserParser::parseFnc_AWAY(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: away
		@type:
			function
		@title:
			$away
		@short:
			Returns 1 if the current user is away
		@syntax:
			$away
		@description:
			Returns 1 if the current user is away , 0 otherwise.
			If the current IRC context is not connected at all , this function prints
			a warning and returns 0 anyway.
	*/

	ENTER_CONTEXT(c,"parseFnc_AWAY");

	bool bAway = false;

	if(!c->window()->console())c->warnNoIrcContext();
	else {
		if(!c->window()->console()->isConnected())c->warnNotConnectedToServer();
		else bAway = c->window()->console()->userIsAway();
	}

	buffer.append(bAway ? '1' : '0');

	return c->leaveContext();
}

bool KviUserParser::parseFnc_ISANYCONSOLECONNECTED(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
	   @doc: isanyconsoleconnected
	   @type:
	   	   function
	   @title:
	       $isAnyConsoleConnected
	   @short:
	       Returns 1 if there is any console connected
	   @syntax:
	       $isAnyConsoleConnected()
	   @description:
	       Returns 1 if there is at least one console connected to a server. [br]If that's not the case,
		   returns 0.[br]
	*/
	
	ENTER_CONTEXT(c,"parseFnc_ISANYCONSOLECONNECTED");

	if (g_pApp->isAnyConsoleConnected()) buffer.append('1');
	else buffer.append('0');

	return c->leaveContext();
}
	

bool KviUserParser::parseFnc_ISWELLKNOWN(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: iswellknown
		@type:
			function
		@title:
			$isWellKnown
		@short:
			Returns 1 if the specified user is well known
		@syntax:
			$isWellKnown(<nickname>)
		@description:
			Returns 1 if KVIrc has the basic user informations about the specified <nickname>.[br]
			The basic informations include the username and hostname.[br]
			This is almost always true if the user is on a channel with you or
			you have an open query with him.[br]
			If $isWellKnown returns 0 , [fnc]$username[/fnc] and [fnc]$hostname[/fnc]
			will return empty strings.[br]
			In this case you must use [cmd]awhois[/cmd] to obtain the user basic informations.[br]
	*/

	ENTER_CONTEXT(c,"parseFnc_ISWELLKNOWN");

	if(!c->window()->console())return c->noIrcContext();
	else {
		if(!c->window()->console()->isConnected())return c->notConnectedToServer();
	}

	KviIrcUserEntry * e = c->window()->console()->userDataBase()->find(params->safeFirstParam());
	bool bGotIt = false;
	if(e)
	{
		bGotIt = (e->hasHost() && e->hasUser());
	}

	buffer.append(bGotIt ? '1' : '0');

	return c->leaveContext();
}

bool KviUserParser::parseFnc_USERNAME(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: username
		@type:
			function
		@title:
			$username
		@short:
			Returns the username of the specified user
		@syntax:
			$username(<nickname>)
		@description:
			Returns the username of the specified irc user IF it is known.[br]
			The username is known if [fnc]$isWellKnown[/fnc] returns 1.[br]
			The username is generally known if the user is on a channel with you
			or has an open query with you.[br]
			Detailed explaination:[br]
			KVirc has an internal database of users that are currently
			visible by *this client*: this includes users on open channels
			and queries.[br] The other IRC users are NOT in the database:
			this means that KVIrc knows NOTHING about them and can't return
			any information immediately. In this case this function will return
			an EMPTY string.[br]
			If an user is in the database then at least his nickname is known.[br]
			The username and hostname is known only if the server provides this information
			spontaneously or after a KVIrc request.[br]
			KVIrc requests user informations for all the users in open queries
			and channels. These informations take some time to be received:
			in this interval of time KVIrc knows only the user's nickname.
			This function will return the string "*" in this case.[br]
		@seealso:
			[fnc]$iswellknown[/fnc], [$fnc]$hostname[/fnc], [cmd]awhois[/cmd]
	*/

	ENTER_CONTEXT(c,"parseFnc_USERNAME");

	if(!c->window()->console())return c->noIrcContext();
	else {
		if(!c->window()->console()->isConnected())return c->notConnectedToServer();
	}

	KviIrcUserEntry * e = c->window()->console()->userDataBase()->find(params->safeFirstParam());

	if(e)buffer.append(e->user());

	return c->leaveContext();
}


bool KviUserParser::parseFnc_HOSTNAME(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: hostname
		@type:
			function
		@title:
			$hostname
		@short:
			Returns the hostname of the specified user
		@syntax:
			$hostname(<nickname>)
		@description:
			Returns the hostname of the specified irc user IF it is known.[br]
			The hostname is known if [fnc]$isWellKnown[/fnc] returns 1.[br]
			The hostname is generally known if the user is on a channel with you
			or has an open query with you.[br]
			Detailed explaination:[br]
			KVirc has an internal database of users that are currently
			visible by *this client*: this includes users on open channels
			and queries.[br] The other IRC users are NOT in the database:
			this means that KVIrc knows NOTHING about them and can't return
			any information immediately. In this case this function will return
			an EMPTY string.[br]
			If an user is in the database then at least his nickname is known.[br]
			The username and hostname is known only if the server provides this information
			spontaneously or after a KVIrc request.[br]
			KVIrc requests user informations for all the users in open queries
			and channels. These informations take some time to be received:
			in this interval of time KVIrc knows only the user's nickname.
			This function will return the string "*" in this case.[br]
		@seealso:
			[fnc]$iswellknown[/fnc], [$fnc]$username[/fnc], [cmd]awhois[/cmd]
	*/

	ENTER_CONTEXT(c,"parseFnc_HOSTNAME");

	if(!c->window()->console())return c->noIrcContext();
	else {
		if(!c->window()->console()->isConnected())return c->notConnectedToServer();
	}

	KviIrcUserEntry * e = c->window()->console()->userDataBase()->find(params->safeFirstParam());

	if(e)buffer.append(e->host());

	return c->leaveContext();
}


bool KviUserParser::parseFnc_SERVER(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: server
		@type:
			function
		@title:
			$server
		@short:
			Returns the current server name (if any)
		@syntax:
			$server[(irc_context_id)]
		@description:
			Returns the current server name of the specified irc context.[br]
			If no irc_context_id is specified , the current irc context is assumed.[br]
			If you are not connected to a server, this function will return an empty string.[br]
			If the current window does not belong to any IRC context and no irc_context_id
			is specified this function prints a warning and still returns an empty string.[br]
	*/

	ENTER_CONTEXT(c,"parseFnc_CONSOLE");

	KviConsole *wnd = 0;

	KviStr * pCntx = params->first();

	if(pCntx)
	{
		bool bOk;
		unsigned int ircContextId = pCntx->toUInt(&bOk);
		if(!bOk)return c->error(KviError_invalidIrcContextId,pCntx->ptr());
		wnd = g_pApp->findConsole(ircContextId);
	} else {
		if(c->window()->console())wnd = c->window()->console();
		else c->warning(__tr("This window has no associated irc context"));
	}

	if(wnd)
	{
		if(wnd->isConnected())buffer.append(wnd->currentServerName());
	}

	return c->leaveContext();

}


bool KviUserParser::parseFnc_ME(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: me
		@type:
			function
		@title:
			$me
		@short:
			Returns the current nick name
		@syntax:
			$me[(irc_context_id)]
		@description:
			Returns the current nick name used in the specified irc context.[br]
			If no irc_context_id is specified , the current irc context is assumed.[br]
			If you are not connected to a server, this function will return an empty string.[br]
			If the current window does not belong to any IRC context and no irc_context_id
			is specified this function prints a warning and still returns an empty string.[br]
	*/

	ENTER_CONTEXT(c,"parseFnc_CONSOLE");

	KviConsole *wnd = 0;

	KviStr * pCntx = params->first();

	if(pCntx)
	{
		bool bOk;
		unsigned int ircContextId = pCntx->toUInt(&bOk);
		if(!bOk)return c->error(KviError_invalidIrcContextId,pCntx->ptr());
		wnd = g_pApp->findConsole(ircContextId);
	} else {
		if(c->window()->console())wnd = c->window()->console();
		else c->warning(__tr("This window has no associated irc context"));
	}

	if(wnd)
	{
		if(wnd->isConnected())buffer.append(wnd->currentNickName());
	}

	return c->leaveContext();

}

bool KviUserParser::parseFnc_FMTLINK(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: fmtlink
		@type:
			function
		@title:
			$fmtlink
		@short:
			Returns a formatted link buffer
		@syntax:
			$fmtlink(<link_text>,<double_click_command>[,<tooltip_text>])
		@description:
			Returns a link formatted for the [cmd]echo[/cmd] command.[br]
			If you pass the returned string to the echo command , the string will be displayed
			as a link and will be highlighted when the user moves the mouse over it.[br]
			If the user will leave the mouse for a few seconds over the link , the <tooltip_text>
			will be displayed in a small tooltip window. If <tooltip_text> is not given,
			then no tooltip will be shown.[br]
			The <double_click_command> will be executed when the user will double click on the link.[br]
			Please remember that if <double_click_command> contains identifiers
			that must be evaluated at double-click time, you MUST escape them in the $fmtlink() call
			to prevent the evaluation.[br]
			You might also take a look at [doc:escape_sequences]the escape sequences documentation[/doc]
			to learn more about how the links are implemented and how to create more powerful links (add
			right and middle button actions , use predefined kvirc links etc...)
		@seealso:
			[doc:escape_sequences]the escape sequences documentation[/doc]
	*/

	ENTER_CONTEXT(c,"parseFnc_FMTLINK");

	KviStr * pLinkText = params->safeFirst();
	KviStr * pCommand  = params->safeNext();
	KviStr * pTooltip  = params->safeNext();

	if(pLinkText->isEmpty())
	{
		c->warning(__tr("No link text specified"));
		return c->leaveContext();
	}

	if(pCommand->isEmpty())
	{
		c->warning(__tr("No command specified"));
		return c->leaveContext();
	}

	KviStr szCmd(KviStr::Format,"[!dbl]%s",pCommand->ptr());
	if(pTooltip->hasData())szCmd.append(KviStr::Format,"[!txt]%s",pTooltip->ptr());

	buffer.append(KviStr::Format,"\r!%s\r%s\r",szCmd.ptr(),pLinkText->ptr());

	return c->leaveContext();

}

bool KviUserParser::parseFnc_CR(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: cr
		@type:
			function
		@title:
			$cr
		@short:
			Returns a carriage return character
		@syntax:
			$cr
		@description:
			Returns a carriage return character
		@seealso:
			[fnc]$lf[/fnc],[fnc]$ascii[/fnc],[fnc]$char[/fnc]
	*/

	buffer.append("\r");
	return true;
}


bool KviUserParser::parseFnc_LF(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: lf
		@type:
			function
		@title:
			$lf
		@short:
			Returns a line feed character
		@syntax:
			$lf
		@description:
			Returns a line feed character
		@seealso:
			[fnc]$cr[/fnc],[fnc]$ascii[/fnc],[fnc]$char[/fnc]
	*/

	buffer.append("\n");
	return true;
}

bool KviUserParser::parseFnc_CHAR(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: char
		@type:
			function
		@title:
			$char
		@short:
			Returns a character specified by ascii code
		@syntax:
			$char(<ascii_code>)
		@description:
			Returns a character corresponding to the ASCII code <ascii_code>.[br]
			This function can not return NUL character (ASCII 0). Basically
			you should never need it: if you do , drop me a mail.[br]
			If the <ascii_code> is not a valid ASCII code (or is 0), this function returns
			an empty string.[br]
		@seealso:
			[fnc]$cr[/fnc],[fnc]$lf[/fnc],[fnc]$char[/fnc]
	*/

	KviStr * pCode = params->safeFirst();
	bool bOk;
	int iCode = pCode->toInt(&bOk) % 256;
	if(bOk && (iCode != 0))buffer.append(KviStr::Format,"%c",iCode);
	return true;
}

bool KviUserParser::parseFnc_ASCII(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: ascii
		@type:
			function
		@title:
			$ascii
		@short:
			Returns the ASCII code of a character
		@syntax:
			$char(<char>)
		@description:
			Returns the ASCII code corresponding to the given character.[br]
		@seealso:
			[fnc]$cr[/fnc],[fnc]$lf[/fnc],[fnc]$char[/fnc]
	*/

	KviStr * pChar = params->safeFirst();
	if(pChar->hasData())
	{
		unsigned char c = (unsigned char) *(pChar->ptr());
		buffer.append(KviStr::Format,"%u",c);
	}
	return true;
}

bool KviUserParser::parseFnc_K(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: k
		@type:
			function
		@title:
			$k
		@short:
			Returns the COLOR mirc control character
		@syntax:
			$k(<foreground>[,<background>])
			$k
		@description:
			Returns the COLOR mirc control character (CTRL+K).[br]
			If <foreground> and <background> are passed , a standard mirc
			color escape is returned.[br]
		@seealso:
			[fnc]$b[/fnc]
	*/

	KviStr * pCode1 = params->safeFirst();
	KviStr * pCode2 = params->safeNext();
	buffer.append(KVI_TEXT_COLOR);
	if(pCode1->hasData())
	{
		buffer.append(pCode1->ptr());
		if(pCode2->hasData())
		{
			buffer.append(',');
			buffer.append(pCode2->ptr());
		}
	}
	return true;
}

bool KviUserParser::parseFnc_B(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: b
		@type:
			function
		@title:
			$b
		@short:
			Returns the BOLD mirc control character
		@syntax:
			$b
		@description:
			Returns the BOLD mirc control character (CTRL+B).[br]
		@seealso:
			[fnc]$k[/fnc],[fnc]$u[/fnc]
	*/

	buffer.append(KVI_TEXT_BOLD);
	return true;
}

bool KviUserParser::parseFnc_U(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: u
		@type:
			function
		@title:
			$u
		@short:
			Returns the UNDERLINE mirc control character
		@syntax:
			$u
		@description:
			Returns the UNDERLINE mirc control character (CTRL+U).[br]
		@seealso:
			[fnc]$k[/fnc],[fnc]$b[/fnc],[fnc]$r[/fnc],[fnc]$o[/fnc]
	*/

	buffer.append(KVI_TEXT_UNDERLINE);
	return true;
}

bool KviUserParser::parseFnc_R(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: r
		@type:
			function
		@title:
			$r
		@short:
			Returns the REVERSE mirc control character
		@syntax:
			$u
		@description:
			Returns the REVERSE mirc control character (CTRL+R).[br]
		@seealso:
			[fnc]$k[/fnc],[fnc]$b[/fnc],[fnc]$u[/fnc],[fnc]$o[/fnc]
	*/

	buffer.append(KVI_TEXT_REVERSE);
	return true;
}

bool KviUserParser::parseFnc_O(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: o
		@type:
			function
		@title:
			$o
		@short:
			Returns the RESET mirc control character
		@syntax:
			$o
		@description:
			Returns the RESET mirc control character (CTRL+O).[br]
		@seealso:
			[fnc]$k[/fnc],[fnc]$b[/fnc],[fnc]$u[/fnc],[fnc]$r[/fnc]
	*/

	buffer.append(KVI_TEXT_RESET);
	return true;
}

bool KviUserParser::parseFnc_SPLIT(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: split
		@type:
			function
		@title:
			$split
		@short:
			Splits a string to an array
		@syntax:
			$split(<separator>,<string>[,<flags>])
		@description:
			Splits the <string> by <separator> and returns an array of substrings.[br]
			<flags> may be a combination of the characters s w and r.[br]
			If s is specified , <separator> matching is case sensitive, otherwise is case insensitive.[br]
			If w is specified , <separator> is treated as a wildcard-type regular expression
			(with * and ? wildcars).[br]
			If r is specified , <separator> is treated as a extended-type regular expression
			(with character classes, special escapes etc..).[br]
			If both w and r are specified w takes precedence.[br]
			If none of w and r are specified <separator> is treated as a simple string to be matched.[br]
			The returned value may be assigned to a dictionary too: it will be used to simulate an array.[br]
		@examples:
			[example]
				[comment]# Split the fields[/comment]
				%test[] = $split(!,"Field0!Field1!Field2!Field3!!Field5")
				echo %test[]
				%i = 0
				[cmd]while[/cmd](%i < %test[]#)
				{
					[cmd]echo[/cmd] "Field %i: %test[%i]"
					%i++;
				}
			[/example]
			Regexp splitting:
			[example]
				%Test[] = $split("[ ]*[0-9][0-9]*-","AllOfThem: 1-Balboy 2-Pragma 3-Iakkolo 4-Crocodile",r)
				echo %Test[]
				%Test[] = $split("Y*H","hihiYeaHhohohoyeahYepYEAHhi",cw)
				echo %Test[]
			[/example]
			If used in "non-array" context it returns just a comma separated list of substrings:[br]
			[example]
				[cmd]echo[/cmd] $split("[ ]*","Condense spaces and change &nbsp; &nbsp; all &nbsp; &nbsp; &nbsp; it in commas",r)
			[/example]
		@seealso:
			[fnc]$join[/fnc]()
	*/

	KviStr * pSep = params->safeFirst();
	KviStr * pDat = params->safeNext();
	KviStr * pFla = params->safeNext();

	c->beginListArrayOrDictionaryReturnIdentifier();

	if(pSep->isEmpty())
	{
		c->addListArrayOrDictionaryReturnValue(0,pDat->ptr(),buffer);
		return true;
	}

	bool bWild = pFla->contains('w');
	bool bContainsR = pFla->contains('r');
	bool bCaseSensitive = pFla->contains('s');

	int id = 0;
	char * begin = pDat->ptr();

	if(bContainsR || bWild)
	{
		QRegExp re(pSep->ptr(),bCaseSensitive,bWild);
		QString szSearch = pDat->ptr();
		
		__range_valid(szSearch.length() == pDat->len());
		int iMatch = 0;
		int iOldMatch;
#if QT_VERSION >= 300
		while((iMatch != -1) && *begin)
		{
			iOldMatch = iMatch;
			iMatch = re.search(szSearch,iMatch);
			if(iMatch != -1)
			{
				int len = re.matchedLength();
				if((len == 0) && (iOldMatch == iMatch))iMatch++; // safety measure for empty string matching

				char * ptr = pDat->ptr() + iMatch;
				char tmp = *ptr;
				*ptr = '\0';
				c->addListArrayOrDictionaryReturnValue(id,begin,buffer);
				*ptr = tmp;

				id++;
				iMatch += len;
				begin = pDat->ptr() + iMatch;
			}
		}
#else
		int len;
		while((iMatch != -1) && *begin)
		{
			iOldMatch = iMatch;
			iMatch = re.match(szSearch,iMatch,&len,false);
			if(iMatch != -1)
			{
				if((len == 0) && (iOldMatch == iMatch))iMatch++; // safety measure for empty string matching
				char * ptr = pDat->ptr() + iMatch;
				char tmp = *ptr;
				*ptr = '\0';
				c->addListArrayOrDictionaryReturnValue(id,begin,buffer);
				*ptr = tmp;

				id++;
				iMatch += len;
				begin = pDat->ptr() + iMatch;
			}
		}
#endif
		if(*begin)c->addListArrayOrDictionaryReturnValue(id,begin,buffer);
		return true;
	}

	char chr = *(pSep->ptr());
	char * p = pDat->ptr();

	if(pSep->len() == 1)
	{
		if(bCaseSensitive)
		{
			while(*p)
			{
				if(*p == chr)
				{
					*p = '\0';
					c->addListArrayOrDictionaryReturnValue(id,begin,buffer);
					*p = chr;
					id++;
					p++;
					begin = p;
				} else p++;
			}
		} else {
			char lowchr = tolower(chr);
			while(*p)
			{
				if(tolower(*p) == lowchr)
				{
					*p = '\0';
					c->addListArrayOrDictionaryReturnValue(id,begin,buffer);
					*p = chr;
					id++;
					p++;
					begin = p;
				} else p++;
			}
		}
	} else {
		int len = pSep->len();
		while(*p)
		{
			if(bCaseSensitive)
			{
				if(*p == chr)
				{
					if(kvi_strEqualCSN(p,pSep->ptr(),len))
					{
						*p = '\0';
						c->addListArrayOrDictionaryReturnValue(id,begin,buffer);
						*p = chr;
						id++;
						p += len;
						begin = p;
					} else p++;
				} else p++;
			} else {
				char lowchr = tolower(chr);
				if(tolower(*p) == lowchr)
				{
					if(kvi_strEqualCIN(p,pSep->ptr(),len))
					{
						*p = '\0';
						c->addListArrayOrDictionaryReturnValue(id,begin,buffer);
						*p = chr;
						id++;
						p += len;
						begin = p;
					} else p++;
				} else p++;
			}
		}
	}
	if(begin != p)c->addListArrayOrDictionaryReturnValue(id,begin,buffer);
	return true;
}

bool KviUserParser::parseFnc_FEATURES(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: features
		@type:
			function
		@title:
			$features
		@short:
			Returns the features that KVIrc supports
		@syntax:
			$features
			$features(<test_feature>)
		@description:
			The parameterless form returns an array of feature descripton strings that this KVIrc executable supports.[br]
			This function is useful when some part of your script depends on
			an optional KVIrc feature (like SSL support or IPV6 support).[br]
			The returned value may be assigned to a dictionary too: it will be used to simulate an array.[br]
			The form with the [test_feature] parameter returns 1 if and only if [test_feature] is available.[br]
		@examples:
			[example]
			%myfeats[] = $features
			echo %myfeats[]
			%i = %myfeats[]#
			[cmd]while[/cmd](%i > 0)
			{
				[cmd]echo[/cmd] "Supporting feature %myfeats[%i]"
				%i--;
			}
			[/example]
			You can test for a specific feature in the following way:
			[example]
			[cmd]if[/cmd]($features("SSL"))[cmd]echo[/cmd] "Yes! SSL is available";
			[/example]
			If used in "non-array" context it returns just a comma separated list of entries:[br]
			[example]
			[cmd]echo[/cmd] $features
			[/example]
		@seealso:
	*/
	KviStr * p = params->safeFirst();
	bool bHasParam = p->hasData();

	static const char * feature_array[]=
	{
		"IRC",
#ifdef COMPILE_IPV6_SUPPORT
		"IPv6",
#endif
#ifdef COMPILE_CRYPT_SUPPORT
		"Crypt",
#endif
#ifdef COMPILE_SSL_SUPPORT
		"SSL",
#endif
#ifdef COMPILE_LOCALE_STUFF
		"Locale",
#endif
#ifdef COMPILE_GET_INTERFACE_ADDRESS
		"IfAddr",
#endif
#ifndef COMPILE_NO_IPC
		"IPC",
#endif
#ifdef COMPILE_KDE_SUPPORT
		"KDE",
#endif
#ifdef COMPILE_OSS_SUPPORT
		"OSS",
#endif
#ifdef COMPILE_ARTS_SUPPORT
		"ARTS",
#endif
#ifdef COMPILE_ESD_SUPPORT
		"ESD",
#endif
#ifdef COMPILE_AUDIOFILE_SUPPORT
		"Audiofile",
#endif
#ifdef COMPILE_PSEUDO_TRANSPARENCY
		"Transparency",
#endif
#ifdef COMPILE_SPLASH_SCREEN
		"Splash",
#endif
#ifdef COMPILE_ix86_ASM
		"ix86-ASM",
#endif
#if QT_VERSION >= 300
		"Qt3",
#endif
		0
	};

	if(bHasParam)
	{
		for(int i=0;feature_array[i];i++)
		{
			if(kvi_strEqualCI(feature_array[i],p->ptr()))
			{
				buffer.append('1');
				return true;
			}
		}
		buffer.append('0');
	} else {
		c->beginListArrayOrDictionaryReturnIdentifier();
		int id = 0;
		for(int i=0;feature_array[i];i++)
		{
			c->addListArrayOrDictionaryReturnValue(id++,feature_array[i],buffer);
		}
	}

	return true;
}

