//
//   File : kvi_up_expr.cpp
//   Creation date : Sat Feb 27 1999 19:31:10 by Szymon Stefanek
//   Original name: kvi_expression.cpp
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
#define __KVIRC__
#define _KVI_DEBUG_CHECK_RANGE_
#include "kvi_debug.h"

#include "kvi_app.h"
#include "kvi_uparser.h"
#include "kvi_command.h"
#include "kvi_error.h"
#include "kvi_locale.h"
#include "kvi_exprtree.h"
#include "kvi_scriptobject.h"


bool KviUserParser::evaluateExpression(KviCommand *c,long * result,bool bExpectClosedParenthesis)
{
	ENTER_CONTEXT(c,"evaluateExpression");
	// This one evaluates an expression
	KviExprTree * root = parseExpression(c);
	if(!root)return false; // error inside!

	if(bExpectClosedParenthesis)
	{
		if(*(c->m_ptr) != ')')return c->error(KviError_unterminatedExpression,c->m_ptr);
	} else {
		if(*(c->m_ptr) != '\0')c->warning(__tr("Unmatched parenthesis in expression body: ignoring trailing garbage"));
	}

	*result = root->calculate(c);
	delete root;
	++(c->m_ptr);

	return c->hasError() ? false : c->leaveContext();
}

KviExprTree * KviUserParser::parseExpression(KviCommand *c)
{
	ENTER_CONTEXT(c,"parseExpression");
	// This one evaluates the expression up to a (not matching) ')' or a \0
	KviExprTree * op = parseExprOperand(c);
	if(!op)return 0; // error inside!

	c->skipSpace();
	if((*(c->m_ptr) == ')')||(*(c->m_ptr) == '\0'))
	{
		c->leaveContext();
		return op;//single operand
	}
	KviExprTree * curOper = parseExprOperator(c);
	if(!curOper){
		delete op;
		return 0; // error inside!
	}
	curOper->setLeft(op);
	KviExprTree * rOp = parseExpression_RightOperand(curOper,c);

	if(rOp)c->leaveContext();
	return rOp; // if 0 , error inside!
}

KviExprTree * KviUserParser::parseExprOperand(KviCommand *c)
{
	ENTER_CONTEXT(c,"parseExprOperand");
	// This one tries to extract a numeric operand from the data buffer
	// Moves m_ptr to the character after the last char of the operand
	// Returns a valid KviExprTreeNode of succesful 
	// otherwise returns 0 and sets c->m_err appropriately
	c->skipSpace();

	KviExprTree * pTop = 0;
	KviExprTree * pCur = 0;
	KviExprTree * pLast = 0;

	// Check for unary ops...
	while(*(c->m_ptr) == '-' || *(c->m_ptr) == '~' || *(c->m_ptr) == '!'){
		// Unary op
		switch(*(c->m_ptr)){
			case '-':
				pCur = new KviExprTree(KviExprTree::UnaryOperator,OP_NEGATE);
			break;
			case '~':
				pCur = new KviExprTree(KviExprTree::UnaryOperator,OP_BITNOT);
			break;
			case '!':
				pCur = new KviExprTree(KviExprTree::UnaryOperator,OP_NOT);
			break;
		}
		// And we work with the child
		if(pTop == 0)pTop = pCur;
		else pLast->setLeft(pCur);
		pLast = pCur;
		++(c->m_ptr);
		c->skipSpace();
	}

	if(*(c->m_ptr) == '('){
		// subexpression
		++(c->m_ptr);
		pCur = parseExpression(c);
		if(!pCur){                              // failed somewhere in the call
			if(pTop)delete pTop;
			return 0; // error inside
		}
		if(*(c->m_ptr) != ')'){                         // failed here
			if(pTop)delete pTop;
			delete pCur;
			c->error(KviError_unterminatedSubexpression);
			return 0;
		}
		++(c->m_ptr);
	} else {
		// here *m_ptr may be something that is not a digit too!
		pCur = parseExprSimpleOperand(c);
		if(!pCur){                            // failed somewere
			if(pTop)delete pTop;
			return 0; // error inside
		}
	}
	if(pTop){
		pLast->setLeft(pCur);
		c->leaveContext();
		return pTop;
	} else {
		if(pCur)c->leaveContext();
		return pCur;
	}
}
/*
KviExprTree * KviUserParser::parseExprStringOperand(KviCommand *c)
{
	ENTER_CONTEXT(c,"parseExprStringOperand");

	__range_valid(*(c->m_ptr) == '"');
	KviExprTree * node = new KviExprTree(KviExprTree::StringOperand,0);
	char *aux_ptr = ++(c->m_ptr);

	for(;;){
		// Now skip all the non interesting chars
		while(*aux_ptr && (*aux_ptr != '\\') && (*aux_ptr != '\n') &&
				(*aux_ptr != '"') && (*aux_ptr != '$') && (*aux_ptr != '%'))++aux_ptr;
		// Interesting char
		switch(*aux_ptr){
			case '\n':
			case '\0': // end of command buffer...append the last block to the buffer and return
				c->error(KviError_unexpectedEndInString);
				delete node;
				return 0;
			break;
			case '"' : // append the last block to the buffer and return
				node->m_string.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = ++aux_ptr; //don't forget to point to next char
				c->leaveContext();
				return node;
			break;
			case '%': //variable: append the last block to the buffer and process the var
				node->m_string.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				if(!parseVariable(c,node->m_string)){
					delete node;
					return 0;
				}
				aux_ptr = c->m_ptr;
			break;
			case '$': //system identifier: append the last block to the buffer and process the ident
				node->m_string.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				if(!parseIdentifier(c,node->m_string)){
					delete node;
					return 0;
				}
				aux_ptr = c->m_ptr;
			break;
			case '\\': //escape character: append the last block to the processed buffer...
				node->m_string.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = ++aux_ptr;
				switch(*aux_ptr){
					case '\0':
						c->error(KviError_unexpectedEndInString);
						delete node;
						return 0;
					break;
					case '\n':  //escaped newline or terminator
						c->m_ptr = ++aux_ptr; //skip it
						c->skipSpace();       //skip leading spaces
						aux_ptr = c->m_ptr;   //continue
					break;
					default: // Must be copied to the buffer without modification
						node->m_string.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
				}
			break;
		}
	}
	c->error(KviError_internalError);
	return 0; //Newer here
}
*/

KviExprTree * KviUserParser::parseExprSimpleOperand(KviCommand *c)
{
	ENTER_CONTEXT(c,"parseExprSimpleOperand");

	// Skip leading spaces...even the escaped ones
	c->skipSpace();
	char *aux_ptr = c->m_ptr;

	KviStr buffer;

	bool bInString = false;

	for(;;)
	{
		switch(*aux_ptr)
		{
			case '=' :
			case '+' :
			case '/' :
			case '*' :
			case '^' :
			case '@' :
			case '-' :
			case '!' :
			case '>' :
			case '<' :
			case '&' : 
			case '|' :
			case ' ' :
			case '\t':
			case ')' :
				if(!bInString)
				{ // End of the token...append the last block to the buffer and return
					buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
					c->m_ptr = aux_ptr;
					goto got_operand;
				} else ++aux_ptr;
			break;
			case '\n': 
			case '\0':
				if(!bInString)
				{ // End of the token...append the last block to the buffer and return
					buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
					c->m_ptr = aux_ptr;
					goto got_operand;
				} else {
					c->error(KviError_unexpectedEndInString);
					return 0;
				}
			break;
			case '\"': // beginning or end of a string
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				bInString = !bInString;
				c->m_ptr = ++aux_ptr;
			break;
			case '%': //variable: append the last block to the buffer and process the var
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				if(!parseVariable(c,buffer))return 0;
				aux_ptr = c->m_ptr;
			break;
			case '$': //system identifier: append the last block to the buffer and process the ident
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				if(!parseIdentifier(c,buffer))return 0;
				aux_ptr = c->m_ptr;
			break;
			case '\\': //escape character: append the last block to the processed buffer...
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = ++aux_ptr;
				switch(*aux_ptr)
				{
					case '\0':
						c->error(bInString ? KviError_unexpectedEndInString : KviError_unexpectedEndInExpression);
						return 0;
					break;
					case '\n':  //escaped newline
						c->m_ptr = ++aux_ptr; //skip it
						c->skipSpace();       //skip leading spaces
						aux_ptr = c->m_ptr;   //continue
					break;
					default: // Must be copied to the buffer without modification
						buffer.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
				}
			break;
			default:
				++aux_ptr;
			break;
		}
	}

got_operand:

	c->leaveContext();

	if((!isdigit(*(buffer.ptr()))) && (*(buffer.ptr()) != '-') && (*(buffer.ptr()) != '+'))
	{
		//debug("Buffer with not a digit : %s",buffer.ptr());
		return new KviExprTree(buffer);
	}

	char * aux = buffer.ptr();
	int base = 0;
	char ccc;

	if(*aux == '0')
	{
		++aux;
		ccc = *aux;
		switch(tolower(ccc))
		{
			case 'x': buffer.cutLeft(2); base = 16; break;
			case 'h': buffer.cutLeft(2); base = 16; break;

			case 'i': buffer.cutLeft(2); base = 2;  break; // can not use b because it is a hex digit
			case 'j': buffer.cutLeft(2); base = 3;  break;
			case 'k': buffer.cutLeft(2); base = 4;  break;
			case 'l': buffer.cutLeft(2); base = 5;  break;
			case 'm': buffer.cutLeft(2); base = 6;  break;
			case 'n': buffer.cutLeft(2); base = 7;  break;
			case 'o': buffer.cutLeft(2); base = 8;  break;
			case 'p': buffer.cutLeft(2); base = 9;  break;
			case 'q': buffer.cutLeft(2); base = 10; break;
			case 'r': buffer.cutLeft(2); base = 11; break;
			case 's': buffer.cutLeft(2); base = 12; break;
			case 't': buffer.cutLeft(2); base = 13; break;
			case 'u': buffer.cutLeft(2); base = 14; break;
			case 'v': buffer.cutLeft(2); base = 15; break;
			default: break; // unknown ...maybe a single 0
		}
	}

	bool bOk;
	long res = buffer.toLongExt(&bOk,base > 1 ? base : 10);

	//debug("Computing:(%s),%d,%d,%d",buffer.ptr(),res,base,bOk);

	if(!bOk)
	{
		if(base > 0)
		{
			buffer.prepend(ccc);
			buffer.prepend('0');
		}
		//debug("A string");
		return new KviExprTree(buffer);
	}
	//debug("A numeric");
	return new KviExprTree(KviExprTree::NumericOperand,res);
}

/*
KviExprTree * KviUserParser::parseExprSimpleOperand(KviCommand *c)
{
	ENTER_CONTEXT(c,"parseExprSimpleOperand");
	// This one extracts a number from the data buffer
	KviStr tempNum;
	const char *aux =  c->m_ptr;
	switch(*aux)
	{
		case '%':
		case '$':
			do
			{
				if(*(c->m_ptr) == '%')
				{
					if(!parseVariable(c,tempNum))return 0;
				} else if(*(c->m_ptr) == '$'){
					if(!parseIdentifier(c,tempNum))return 0;
				} else {
					c->error(KviError_variableOrIdentifierExpected);
					return 0; //variable or identifier expected
				}
			} while(c->scopeObject());

			if(tempNum.isEmpty())
			{
				c->error(KviError_numericOperandEvaluatedToEmpty);
				return 0;
			}
		break;
		case '"':
			{
				KviExprTree * tree = parseExprStringOperand(c);;
				if(tree)c->leaveContext();
				return tree;
			}
		break;
		case '0':
			++(c->m_ptr);
			if(! (  isalpha(*(c->m_ptr))  ||   isxdigit(*(c->m_ptr))  )  )tempNum = "0";
			else {
				++(c->m_ptr);
				if(isalpha(*(c->m_ptr))||isxdigit(*(c->m_ptr)))++(c->m_ptr);
				while(isxdigit(*(c->m_ptr)))++(c->m_ptr);
				tempNum.setStr(aux,(c->m_ptr) - aux);
			}
		break;
		default:
			if(!isdigit(*(c->m_ptr))){
				c->error(KviError_unexpectedCharacter,__tr("Decimal numeric operand was expected (forgot double-quotes somewhere ?)"));
				return 0;
			}
			while(isdigit(*(c->m_ptr)))++(c->m_ptr);
			tempNum.setStr(aux,(c->m_ptr) - aux);
		break;

	}

	aux = tempNum.ptr();
	int base = 10;
	if(*aux == '0')
	{
		++aux;
		switch(tolower(*aux))
		{
			case 'x': tempNum.cutLeft(2); base = 16; break;
			case 'h': tempNum.cutLeft(2); base = 16; break;

			case 'i': tempNum.cutLeft(2); base = 2;  break; // can not use b because it is a hex digit
			case 'j': tempNum.cutLeft(2); base = 3;  break;
			case 'k': tempNum.cutLeft(2); base = 4;  break;
			case 'l': tempNum.cutLeft(2); base = 5;  break;
			case 'm': tempNum.cutLeft(2); base = 6;  break;
			case 'n': tempNum.cutLeft(2); base = 7;  break;
			case 'o': tempNum.cutLeft(2); base = 8;  break;
			case 'p': tempNum.cutLeft(2); base = 9;  break;
			case 'q': tempNum.cutLeft(2); base = 10; break;
			case 'r': tempNum.cutLeft(2); base = 11; break;
			case 's': tempNum.cutLeft(2); base = 12; break;
			case 't': tempNum.cutLeft(2); base = 13; break;
			case 'u': tempNum.cutLeft(2); base = 14; break;
			case 'v': tempNum.cutLeft(2); base = 15; break;
			default: break; // unknown ...maybe a single 0
		}
	}
	bool bOk;
	long res = tempNum.toLongExt(&bOk,base);
	if(!bOk)
	{
		c->error(KviError_invalidNumericOperand,tempNum.ptr());
		return 0;
	}
	c->leaveContext();
	return new KviExprTree(KviExprTree::NumericOperand,res);
}
*/

KviExprTree * KviUserParser::parseExpression_RightOperand(KviExprTree * curTopOperator,KviCommand *c)
{
	ENTER_CONTEXT(c,"parseExpression_RightOperand");
	__range_valid(curTopOperator);
	// Now curTopOperator has the left subtree (one node) set
	// and it points to the TOP (=ROOT) node
	// Evaluate the rest
	KviExprTree * operand;
	KviExprTree * incompleteOperator = curTopOperator;
	KviExprTree * auxOperator;

	for(;;){
		operand = parseExprOperand(c);
		if(!operand){ delete curTopOperator; return 0; }
		c->skipSpace();
		if(*(c->m_ptr) == ')' || *(c->m_ptr) == '\0')
		{
			incompleteOperator->setRight(operand);
			c->leaveContext();
			return curTopOperator;
		}

		auxOperator = parseExprOperator(c);
		if(!auxOperator){ delete curTopOperator; delete operand; return 0; }

		//now compare operators...
		if(incompleteOperator->firstBinaryOperator() >= auxOperator->firstBinaryOperator()){
			incompleteOperator->setRight(auxOperator);
			auxOperator->setLeft(operand);
		} else {
			// precedence of operators...
			incompleteOperator->setRight(operand); //right tree complete
			// go up until we find an operator with lower precedence than auxOperator (>=)
			KviExprTree * tempOperator = incompleteOperator->parentWithPrecedenceLowerThan(auxOperator->m_value);
			if(tempOperator == 0){ //reached the top
				auxOperator->setLeft(curTopOperator);
				curTopOperator = auxOperator;
			} else {
				__range_valid(tempOperator->right());
				auxOperator->setLeft(tempOperator->right());
				tempOperator->setRight(auxOperator);
			}
		}
		incompleteOperator = auxOperator;
		__range_valid(incompleteOperator->right() == 0);
	}
	c->error(KviError_internalError);
	return 0; //newer here
}

KviExprTree * KviUserParser::parseExprOperator(KviCommand *c)
{
	ENTER_CONTEXT(c,"parseExprOperator");
	// This one extracts the first operator it finds
	c->skipSpace();
	long op;
	switch(*(c->m_ptr))
	{
		// single ops
		case '^': op = OP_POWER; ++(c->m_ptr); break;
		case '*': op = OP_MUL;   ++(c->m_ptr); break;
		case '/': op = OP_DIV;   ++(c->m_ptr); break;
		case '%': op = OP_MOD;   ++(c->m_ptr); break;
		case '+': op = OP_ADD;   ++(c->m_ptr); break;
		case '-': op = OP_SUB;   ++(c->m_ptr); break;
		case '@':
			++(c->m_ptr);
			if(*(c->m_ptr) == '=')op = OP_CSEQ , ++(c->m_ptr);
			else {
				c->error(KviError_unknownOperator,c->m_ptr);
				return 0;
			}
		break;
		case '=':
			++(c->m_ptr);
			if(*(c->m_ptr) == '=')op = OP_EQ , ++(c->m_ptr);
			else {
				c->error(KviError_unknownOperator,c->m_ptr);
				return 0;
			}
		break;
		case '!':
			++(c->m_ptr);
			if(*(c->m_ptr) == '=')op = OP_NE , ++(c->m_ptr);
			else if(*(c->m_ptr) == '@')op = OP_CSNE , ++(c->m_ptr);
			else {
				c->error(KviError_unknownOperator,c->m_ptr);
				return 0;
			}
		break;
		case '>':
			++(c->m_ptr);
			if(*(c->m_ptr) == '=')op = OP_GE , ++(c->m_ptr);
			else if(*(c->m_ptr) == '>')op = OP_SHR , ++(c->m_ptr);
			else op = OP_GT;
		break;
		case '<':
			++(c->m_ptr);
			if(*(c->m_ptr) == '=')op = OP_LE , ++(c->m_ptr);
			else if(*(c->m_ptr) == '<')op = OP_SHL , ++(c->m_ptr);
			else op = OP_LT;
		break;
		case '&':
			++(c->m_ptr);
			if(*(c->m_ptr) == '&')op = OP_AND , ++(c->m_ptr);
			else op = OP_BITAND;
		break;
		case '|':
			++(c->m_ptr);
			if(*(c->m_ptr) == '|')op = OP_OR  , ++(c->m_ptr);
			else op = OP_BITOR;
		break;
		default:
			c->error(KviError_unknownOperator,c->m_ptr);
			return 0;
		break;
	}
	c->leaveContext();
	return new KviExprTree(KviExprTree::BinaryOperator,op);
}
