//
//   File : kvi_popupmenu.cpp
//   Creation date : Sun Sep 1 2000 15:02:21 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
#define __KVIRC__
#define _KVI_DEBUG_CHECK_RANGE_


#include "kvi_debug.h"
#include "kvi_popupmenu.h"
#include "kvi_iconmanager.h"
#include "kvi_window.h"
#include "kvi_console.h"
#include "kvi_uparser.h"
#include "kvi_command.h"
#include "kvi_config.h"
#include "kvi_app.h"
#include "kvi_out.h"
#include "kvi_locale.h"
#include "kvi_datacontainer.h"
#include "kvi_cmdformatter.h"
#include "kvi_popupmanager.h"
#include "kvi_menubar.h"
#ifdef COMPILE_ON_WINDOWS
	#include "kvi_malloc.h"
#endif

#include <qlabel.h>

KviPopupMenuItem::KviPopupMenuItem(const KviPopupMenuItem * src)
{
	m_type = src->m_type;
	m_szText = src->m_szText;
	m_szCode = src->m_szCode;
	m_szIcon = src->m_szIcon;
	m_szExpr = src->m_szExpr;

	if(src->m_pMenu && (m_type != ExtMenu))
	{
		m_pMenu = new KviPopupMenu(src->m_pMenu->name());
		m_pMenu->copyFrom(src->m_pMenu);
	} else m_pMenu = 0;

	m_pLabel = 0;
}

void KviPopupMenuItem::setMenu(KviPopupMenu * m)
{
	if(m_pMenu)delete m_pMenu;
	m_pMenu = m;
}


#ifdef COMPILE_ON_WINDOWS

	// On windows we need to override new and delete operators
	// to ensure that always the right new/delete pair is called for an object instance
	// This bug is present in all the classes exported by a module that
	// can be instantiated/destroyed from external modules.
	// (this is a well known bug described in Q122675 of MSDN)

	void * KviPopupMenuItem::operator new(size_t tSize)
	{
		return kvi_malloc(tSize);
	}

	void KviPopupMenuItem::operator delete(void * p)
	{
		kvi_free(p);
	}

#endif






KviPopupMenuTopLevelData::KviPopupMenuTopLevelData(KviDataContainer * d,KviParameterList * l,KviWindow * wnd)
{
	pDataContainer = d;
	pParamList = l;
	pWnd = wnd;
	bLocked = false;
}

KviPopupMenuTopLevelData::~KviPopupMenuTopLevelData()
{
	if(pParamList)delete pParamList;
	if(pDataContainer)delete pDataContainer;
}

#ifdef COMPILE_ON_WINDOWS

	// On windows we need to override new and delete operators
	// to ensure that always the right new/delete pair is called for an object instance
	// This bug is present in all the classes exported by a module that
	// can be instantiated/destroyed from external modules.
	// (this is a well known bug described in Q122675 of MSDN)

	void * KviPopupMenuTopLevelData::operator new(size_t tSize)
	{
		return kvi_malloc(tSize);
	}

	void KviPopupMenuTopLevelData::operator delete(void * p)
	{
		kvi_free(p);
	}

#endif







KviPopupMenu::KviPopupMenu(const char * name)
:QPopupMenu(0,name)
{
	m_pItemList = new KviPtrList<KviPopupMenuItem>;
	m_pItemList->setAutoDelete(true);
	m_pParentPopup = 0;
	m_pTopLevelData = 0;
	m_pTempTopLevelData = 0;
	m_bSetupDone = false;
	connect(this,SIGNAL(activated(int)),this,SLOT(itemClicked(int)));
	connect(this,SIGNAL(aboutToShow()),this,SLOT(setupMenuContents()));
}


KviPopupMenu::~KviPopupMenu()
{
	delete m_pItemList;
	if(m_pTopLevelData)delete m_pTopLevelData;
	if(m_pTempTopLevelData)delete m_pTempTopLevelData;
}

#ifdef COMPILE_ON_WINDOWS

	// On windows we need to override new and delete operators
	// to ensure that always the right new/delete pair is called for an object instance
	// This bug is present in all the classes exported by a module that
	// can be instantiated/destroyed from external modules.
	// (this is a well known bug described in Q122675 of MSDN)

	void * KviPopupMenu::operator new(size_t tSize)
	{
		return kvi_malloc(tSize);
	}

	void KviPopupMenu::operator delete(void * p)
	{
		kvi_free(p);
	}

#endif

void KviPopupMenu::copyFrom(const KviPopupMenu * src)
{
	doClear();
	m_szPrologueCode = src->m_szPrologueCode;
	m_szEpilogueCode = src->m_szEpilogueCode;
	for(const KviPopupMenuItem * it = src->m_pItemList->first();it;it = src->m_pItemList->next())
	{
		addItem(new KviPopupMenuItem(it));
	}
}

KviPopupMenuTopLevelData * KviPopupMenu::topLevelData()
{
	if(parentPopup())return parentPopup()->topLevelData();
	return m_pTopLevelData;
}

bool KviPopupMenu::isLocked()
{
	if(topLevelPopup()->isVisible())return true;
	KviPopupMenuTopLevelData * d = topLevelData();
	return d ? d->bLocked : false;
}

KviPopupMenu * KviPopupMenu::topLevelPopup()
{
	if(parentPopup())return parentPopup();
	return this;
}

void KviPopupMenu::addItem(KviPopupMenuItem * it)
{
	__range_invalid(isLocked());
	m_pItemList->append(it);
	if(it->menu())
	{
		it->menu()->setParentPopup(this);
		KviStr tmp(KviStr::Format,"%s_sub%d",name(),m_pItemList->count());
		it->menu()->setName(tmp.ptr());
	}
}

void KviPopupMenu::doPopup(const QPoint & pnt,KviWindow * wnd,KviParameterList * params)
{
	__range_invalid(isLocked());
	__range_invalid(parentPopup());
	// This might be a compat problem later :(((((
	// it is also an ugly trick
	clearMenuContents();
	m_pTempTopLevelData = new KviPopupMenuTopLevelData(new KviDataContainer(false),params,wnd);
	QPopupMenu::popup(pnt);
}

void KviPopupMenu::clearMenuContents()
{
	m_bSetupDone = false;

	clear();

	for(KviPopupMenuItem * it = m_pItemList->first();it;it = m_pItemList->next())
	{
		if(it->isExtMenu())it->setMenu(0);
		else {
			if(it->menu())it->menu()->clearMenuContents();
		}
	}

	if(m_pTopLevelData)
	{
		delete m_pTopLevelData;
		m_pTopLevelData = 0;
	}
	if(m_pTempTopLevelData)
	{
		delete m_pTempTopLevelData;
		m_pTempTopLevelData = 0;
	}
}

void KviPopupMenu::doClear()
{
	clear();
	if(m_pTopLevelData)
	{
		delete m_pTopLevelData;
		m_pTopLevelData = 0;
	}
	if(m_pTempTopLevelData)
	{
		delete m_pTempTopLevelData;
		m_pTempTopLevelData = 0;
	}
	m_bSetupDone = false;
	m_pItemList->clear();
	m_szPrologueCode = "";
	m_szEpilogueCode = "";
}

void KviPopupMenu::lock(bool bLock)
{
	KviPopupMenuTopLevelData * d = topLevelData();
	if(!d)return;
	d->bLocked = bLock;
}

void KviPopupMenu::setupMenuContents()
{
	// This might be a compat problem later :((((

	if(parentPopup() == 0)
	{
		if(m_pTempTopLevelData == 0)
		{
			// We have been called by a KviMenuBar!
			// m_bSetupDone is not valid here
			clearMenuContents();
			m_pTopLevelData = new KviPopupMenuTopLevelData(new KviDataContainer(false),0,g_pApp->activeWindow());
		} else {
			if(m_bSetupDone)return;
			// we have been called by doPopup
			// the menu contents have been already cleared
			if(m_pTopLevelData)debug("Ops.. something got messed in KviPopupMenu activation system");
			// Swap the top level data from temporary to the permanent
			m_pTopLevelData = m_pTempTopLevelData;
			m_pTempTopLevelData = 0;
		}
	} else {
		if(m_bSetupDone)return;
	}

	m_bSetupDone = true;


	// HACK...this is to remove the separator inserted by Qt when popup() is called and the popup is empty
	clear();


	KviPopupMenuTopLevelData * d = topLevelData();

	if(!d)
	{
		debug("Ops...menu contents changed behind my back!");
		return;
	}


	lock(true);

// FIXME: #warning "Maybe say that the window has changed"
	if(!g_pApp->windowExists(d->pWnd))d->pWnd = g_pApp->activeConsole();

	// Execute the prologue code
	if(m_szPrologueCode.hasData())
	{
		KviCommand cmdp(m_szPrologueCode.ptr(),d->pWnd,0,d->pDataContainer);
		if(d->pParamList)cmdp.setParams(d->pParamList,false);
		if(!g_pUserParser->parseCommand(&cmdp))
		{
			if(cmdp.hasError())
			{
				d->pWnd->output(KVI_OUT_PARSERWARNING,__tr("Broken prologue code for menu '%s': error detail follows"),name());
				g_pUserParser->printError(&cmdp);
			}
		}
		cmdp.forgetExtendedScopeDataContainer();
	}

	// Fill this menu contents
	int idx = 0;
	for(KviPopupMenuItem * it = m_pItemList->first();it;it = m_pItemList->next())
	{
		if(it->isSeparator())
		{
			insertSeparator();
		} else {
			// Find the pixmap (if any)
			QPixmap * pix = g_pIconManager->getImage(it->icon());
			// Parse the text

// FIXME: #warning "Maybe first parse the condition, no ?"
			KviCommand cmd(it->text(),d->pWnd,0,d->pDataContainer);
			if(d->pParamList)cmd.setParams(d->pParamList,false);
			KviStr parsed;
			if(!g_pUserParser->parseCmdFinalPart(&cmd,parsed))parsed = it->text();
			// Parse the condition (if present)
			bool bInsertIt = true;
			if(it->hasExpression())
			{
				cmd.setCmdBuffer(it->expr());
				long result;
				if(!g_pUserParser->evaluateExpression(&cmd,&result,false))
				{
					// broken expression
					g_pUserParser->printError(&cmd);
					d->pWnd->output(KVI_OUT_PARSERWARNING,__tr("Broken expression for menu item '%s': ignoring"),it->text());
				} else {
					if(result == 0)bInsertIt = false;
				}
			}
			// Eventually insert the item
			if(bInsertIt)
			{
				int id;
				if(it->isLabel())
				{
					it->setLabel(new QLabel(parsed.ptr(),this));
					id = insertItem(it->label());
					it->label()->setFrameStyle(QFrame::Panel | QFrame::Raised);
					if(pix)it->label()->setPixmap(*pix);
				} else if(it->isExtMenu())
				{
					it->setMenu(0);
					KviPopupMenu * source = g_pPopupManager->lookupPopup(it->code());
					if(source)
					{
						KviStr tmp(KviStr::Format,"ontheflycopyof_%s",it->code());
						KviPopupMenu * onTheFly = new KviPopupMenu(tmp.ptr());
						onTheFly->copyFrom(source);
						onTheFly->setParentPopup(this);
						it->setMenu(onTheFly);
						if(pix)id = insertItem(*pix,parsed.ptr(),it->menu());
						else id = insertItem(parsed.ptr(),it->menu());
					} else {
						d->pWnd->output(KVI_OUT_PARSERWARNING,__tr("Can't find the external popup '%s': ignoring"),it->code());
						bInsertIt = false;
					}
				} else if(it->menu())
				{
					if(pix)id = insertItem(*pix,parsed.ptr(),it->menu());
					else id = insertItem(parsed.ptr(),it->menu());
				} else {
					if(pix)id = insertItem(*pix,parsed.ptr());
					else id = insertItem(parsed.ptr());
				}
				if(bInsertIt)
				{
					//debug("Inserted and set the param");
					if(!setItemParameter(id,idx))debug("Ops...failed to set it ?");
				}
			}

			cmd.forgetExtendedScopeDataContainer();
		}
		++idx;
	}

	// Now the epilogue
	if(m_szEpilogueCode.hasData())
	{
		KviCommand cmde(m_szEpilogueCode.ptr(),d->pWnd,0,d->pDataContainer);
		if(d->pParamList)cmde.setParams(d->pParamList,false);
		if(!g_pUserParser->parseCommand(&cmde))
		{
			if(cmde.hasError())
			{
				d->pWnd->output(KVI_OUT_PARSERWARNING,__tr("Broken epilogue code for menu '%s': error details follow"),name());
				g_pUserParser->printError(&cmde);
			}
		}
		cmde.forgetExtendedScopeDataContainer();
	}

	lock(false);

}


void KviPopupMenu::itemClicked(int itemId)
{
	lock(true);
	int param = itemParameter(itemId);
	KviPopupMenuItem * it = m_pItemList->at(param);
	KviPopupMenuTopLevelData * d = topLevelData();
	if(it && d)
	{
// FIXME: #warning "Maybe tell that the window has changed"
		if(!g_pApp->windowExists(d->pWnd))d->pWnd = g_pApp->activeConsole();
		KviCommand cmd(it->code(),d->pWnd,0,d->pDataContainer);
		cmd.setParams(d->pParamList,false);
		if(!g_pUserParser->parseCommand(&cmd))
		{
			if(cmd.hasError())g_pUserParser->printError(&cmd);
		}
		cmd.forgetExtendedScopeDataContainer();
	} else debug("oops....no menu item at position %d",param);
	lock(false);
	// UGLY Qt 3.0.0.... we can't clear menu contents here :(
//#if QT_VERSION < 300
//	topLevelPopup()->clearMenuContents();
//#endif
	//debug("Cleared menu contents");
}


void KviPopupMenu::load(const char * prefix,KviConfig * cfg)
{
	doClear();
	KviStr tmp(KviStr::Format,"%s_Count",prefix);
	int cnt = cfg->readIntEntry(tmp.ptr(),0);
	tmp.sprintf("%s_Prologue",prefix);
	m_szPrologueCode = cfg->readEntry(tmp.ptr(),"");
	tmp.sprintf("%s_Epilogue",prefix);
	m_szEpilogueCode = cfg->readEntry(tmp.ptr(),"");
	for(int idx = 0;idx < cnt;idx++)
	{
		KviStr pre(KviStr::Format,"%s_%d",prefix,idx);
		tmp.sprintf("%s_Type",pre.ptr());
		int type = cfg->readIntEntry(tmp.ptr(),3);
		switch(type)
		{
			case 0: // separator
				addItem(new KviPopupMenuItem(KviPopupMenuItem::Separator));
			break;
			case 1: // item
			{
				KviStr text,icon,code,expr;
				tmp.sprintf("%s_Text",pre.ptr());
				text = cfg->readEntry(tmp.ptr(),"Unnamed");
				tmp.sprintf("%s_Icon",pre.ptr());
				icon = cfg->readEntry(tmp.ptr(),"");
				tmp.sprintf("%s_Code",pre.ptr());
				code = cfg->readEntry(tmp.ptr(),"");
				tmp.sprintf("%s_Expr",pre.ptr());
				expr = cfg->readEntry(tmp.ptr(),"");
				addItem(new KviPopupMenuItem(KviPopupMenuItem::Item,text.ptr(),icon.ptr(),expr.ptr(),code.ptr()));
			}
			break;
			case 2: // menu
			{
				KviStr text,icon,nome,expr;
				tmp.sprintf("%s_Text",pre.ptr());
				text = cfg->readEntry(tmp.ptr(),"Unnamed");
				tmp.sprintf("%s_Icon",pre.ptr());
				icon = cfg->readEntry(tmp.ptr(),"");
				tmp.sprintf("%s_Name",pre.ptr());
				nome = cfg->readEntry(tmp.ptr(),"Unnamed");
				tmp.sprintf("%s_Expr",pre.ptr());
				expr = cfg->readEntry(tmp.ptr(),"");
				KviPopupMenu * pop = new KviPopupMenu(nome.ptr());
				pop->load(pre.ptr(),cfg);
				addItem(new KviPopupMenuItem(KviPopupMenuItem::Menu,text.ptr(),icon.ptr(),expr.ptr(),0,pop));
			}
			break;
			case 3: // label
			{
				KviStr text,icon,expr;
				tmp.sprintf("%s_Text",pre.ptr());
				text = cfg->readEntry(tmp.ptr(),"Unnamed");
				tmp.sprintf("%s_Icon",pre.ptr());
				icon = cfg->readEntry(tmp.ptr(),"");
				tmp.sprintf("%s_Expr",pre.ptr());
				expr = cfg->readEntry(tmp.ptr(),"");
				addItem(new KviPopupMenuItem(KviPopupMenuItem::Label,text.ptr(),icon.ptr(),expr.ptr()));
			}
			break;
			case 4: // extmenu
			{
				KviStr text,icon,code,expr;
				tmp.sprintf("%s_Text",pre.ptr());
				text = cfg->readEntry(tmp.ptr(),"Unnamed");
				tmp.sprintf("%s_Icon",pre.ptr());
				icon = cfg->readEntry(tmp.ptr(),"");
				tmp.sprintf("%s_ExtName",pre.ptr());
				code = cfg->readEntry(tmp.ptr(),"");
				tmp.sprintf("%s_Expr",pre.ptr());
				expr = cfg->readEntry(tmp.ptr(),"");
				addItem(new KviPopupMenuItem(KviPopupMenuItem::ExtMenu,text.ptr(),icon.ptr(),expr.ptr(),code.ptr()));
			}
			break;
			default: // ignore
			break;
		}
	}
}

// FIXME: #warning "NOBODY EDITS THE POPUPS IN THE CONFIG!...A binary config would be faster and work better for sure here"

void KviPopupMenu::save(const char * prefix,KviConfig * cfg)
{
	KviStr tmp(KviStr::Format,"%s_Count",prefix);
	cfg->writeEntry(tmp.ptr(),m_pItemList->count());
	int idx = 0;
	if(m_szPrologueCode.hasData())
	{
		tmp.sprintf("%s_Prologue",prefix);
		cfg->writeEntry(tmp.ptr(),m_szPrologueCode.ptr());
	}
	if(m_szEpilogueCode.hasData())
	{
		tmp.sprintf("%s_Epilogue",prefix);
		cfg->writeEntry(tmp.ptr(),m_szEpilogueCode.ptr());
	}
	for(KviPopupMenuItem * it = m_pItemList->first();it;it = m_pItemList->next())
	{
		KviStr pre(KviStr::Format,"%s_%d",prefix,idx);
		tmp.sprintf("%s_Type",pre.ptr());
		int typeCode = 0;
		switch(it->type())
		{
			case KviPopupMenuItem::Label:        typeCode = 3; break;
			case KviPopupMenuItem::Separator:    typeCode = 0; break;
			case KviPopupMenuItem::Menu:         typeCode = 2; break;
			case KviPopupMenuItem::Item:         typeCode = 1; break;
			case KviPopupMenuItem::ExtMenu:      typeCode = 4; break;
		}

		cfg->writeEntry(tmp.ptr(),typeCode);

		if(!it->isSeparator())
		{
			tmp.sprintf("%s_Text",pre.ptr());
			cfg->writeEntry(tmp.ptr(),it->text());
			if(it->hasIcon())
			{
				tmp.sprintf("%s_Icon",pre.ptr());
				cfg->writeEntry(tmp.ptr(),it->icon());
			}
			if(it->hasExpression())
			{
				tmp.sprintf("%s_Expr",pre.ptr());
				cfg->writeEntry(tmp.ptr(),it->expr());
			}
			if(it->isMenu())
			{
				tmp.sprintf("%s_Name",pre.ptr());
				cfg->writeEntry(tmp.ptr(),it->menu()->name());
				it->menu()->save(pre.ptr(),cfg);
			} else if(it->isExtMenu())
			{
				tmp.sprintf("%s_ExtName",pre.ptr());
				cfg->writeEntry(tmp.ptr(),it->code());
			} else {
				tmp.sprintf("%s_Code",pre.ptr());
				cfg->writeEntry(tmp.ptr(),it->code());
			}
		}
		++idx;
	}
}

void KviPopupMenu::generateDefPopupCore(KviStr &buffer)
{
	KviStr tmp;

	buffer = "";

	m_szPrologueCode.stripWhiteSpace();

	if(m_szPrologueCode.hasData())
	{
		buffer.append("prologue\n");
		tmp = m_szPrologueCode;
		KviCommandFormatter::blockFromBuffer(tmp);
		buffer.append(tmp);
		buffer.append('\n');
	}


	m_szEpilogueCode.stripWhiteSpace();

	if(m_szEpilogueCode.hasData())
	{
		buffer.append("epilogue\n");
		tmp = m_szEpilogueCode;
		KviCommandFormatter::blockFromBuffer(tmp);
		buffer.append(tmp);
		buffer.append('\n');
	}

	for(KviPopupMenuItem * it = m_pItemList->first();it;it = m_pItemList->next())
	{
		switch(it->type())
		{
			case KviPopupMenuItem::Item:
				if(it->hasIcon())buffer.append(KviStr::Format,"item(%s,%s)",it->text(),it->icon());
				else buffer.append(KviStr::Format,"item(%s)",it->text());
				if(it->hasExpression())buffer.append(KviStr::Format," (%s)",it->expr());
				buffer.append("\n");
				tmp = it->code();
				KviCommandFormatter::blockFromBuffer(tmp);
				buffer.append(tmp);
				buffer.append("\n");
			break;
			case KviPopupMenuItem::Menu:
				if(it->hasIcon())buffer.append(KviStr::Format,"popup(%s,%s)",it->text(),it->icon());
				else buffer.append(KviStr::Format,"popup(%s)",it->text());
				if(it->hasExpression())buffer.append(KviStr::Format," (%s)",it->expr());
				buffer.append("\n");
				it->menu()->generateDefPopupCore(tmp);
				KviCommandFormatter::blockFromBuffer(tmp);
				buffer.append(tmp);
				buffer.append("\n");
			break;
			case KviPopupMenuItem::Separator:
				buffer.append("separator\n\n");
			break;
			case KviPopupMenuItem::Label:
				buffer.append(KviStr::Format,"label(%s)",it->text());
				if(it->hasExpression())buffer.append(KviStr::Format," (%s)",it->expr());
				buffer.append("\n\n");
			break;
			case KviPopupMenuItem::ExtMenu:
				if(it->hasIcon())buffer.append(KviStr::Format,"extpopup(%s,%s,%s)",it->text(),it->code(),it->icon());
				else buffer.append(KviStr::Format,"popup(%s,%s)",it->text(),it->code());
				if(it->hasExpression())buffer.append(KviStr::Format," (%s)",it->expr());
				buffer.append("\n\n");
			break;
		}
	}
}

void KviPopupMenu::generateDefPopup(KviStr &buffer)
{
	buffer.sprintf("defpopup(%s)\n",name());

	KviStr core;

	generateDefPopupCore(core);

	KviCommandFormatter::blockFromBuffer(core);

	buffer.append(core);

}

#include "kvi_popupmenu.moc"
