//
//   File : kvi_event.cpp
//   Creation date : Thu Aug 17 2000 16:42:12 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#define __KVIRC__

#define _KVI_EVENT_CPP_
#include "kvi_event.h"
#include "kvi_eventhandler.h"
#include "kvi_config.h"

// FIXME: #warning "DOCS FOR THE EVENTS!"

// FIXME: OnSpam ??
//


/*
	@doc: events
	@type:
		language
	@keyterms:
		events,event handlers,event
	@title:
		Events
	@short:
		Events: user reactions
	@body:
		KVIrc triggers an event when a particular situation occurs (hehe :D).[br]
		You can define a set of event handlers for each event type.[br]
		An event handler is a snippet of user-defined code that gets executed when the event is triggered.[br]
		Event handlers can be created or destroyed by using the scriptcenter (graphic interface)
		or even from the commandline (or script) by using the [cmd]event[/cmd] command.[br]
		For example, the [event:onirc]OnIrc[/event] is triggered when the login operations have
		been terminated and you can consider yourself "completly" on IRC. For example , you might
		want to "auto-join" some channels. Nothing easier! The following snippet of code
		adds a handler to the OnIrc event that joins three channels:[br]
		[example]
		[cmd]event[/cmd](OnIrc,autojoin)
		{
			[cmd]echo[/cmd] Auto-joining my preferred channels...
			[cmd]join[/cmd] #kvirc,#siena,#linux
		}
		[/example]
		Now try to connect to a server and you'll see that it joins automatically the three channels!.[br]
		You might also want to do some other actions just after the connection has been estabilished,
		for example you might want to look immediately for a friend of yours by issuing a [cmd]whois[/cmd]
		to the server (well.. you could use the notify list for that, but well, this is an example).[br]
		You can add the [cmd]whois[/cmd] request to the handler above or just create a new one:[br]
		[example]
		[cmd]event[/cmd](OnIrc,lookforfred)
		{
			[cmd]echo[/cmd] Looking for fred...
			[cmd]whois[/cmd] fred
		}
		[/example]
		(An even nicer idea would be to use the [cmd]awhois[/cmd] command...but that's left to the reader as exercise.[br]
		To remove an event handler you still use the [cmd]event[/cmd] command , but with an empty code block:[br]
		[example]
		[cmd]event[/cmd](OnIrc,lookforfred){}[br]
		[/example]
		[br]
		Certain events will pass you some data in the positional parameters.[br]
		For example, when you are being banned from a channel, KVIrc triggers the [event:onmeban]OnMeBan[/event]
		event: you might be interested in WHO has banned you. KVIrc will pass the "ban source" informations
		in the positional parameters $0,$1 and $2.[br]
		(Please note that the parameters started from $1 in KVIrc versions older than 3.0.0!).[br]
		You may take a look at the list of available [doc:event_list_all]events[/doc].[br]
*/


KviEventManager * g_pEventManager = 0;

KviEventManager::KviEventManager()
{
	for(int i=0;i<1000;i++)m_rawNumericEventTable[i] = 0;
}

KviEventManager::~KviEventManager()
{
	clear();
}

#define EVENT(_name,_parm) { _name , _parm , 0 }

KviEventStruct KviEventManager::m_eventTable[KVI_NUM_SCRIPT_EVENTS]=
{
	/*
		@doc: onkvircstartup
		@type:
			event
		@title:
			OnKVIrcStartup
		@short:
			Triggered on KVIrc startup
		@parameters:
			None
		@window:
			Console of the first irc context
		@description:
			This event is triggered on KVIrc startup.
			It is bound to the console of the first created irc context.
			This is a little tricky, since this event is triggered
			before the related [event:onirccontextcreated]OnIrcContextCreated[/event] event.
			This is to make OnApplicationStartup be the absolutely first
			event triggered at all.
			This event is a good place to load your script configuration
			and eventually initialize your variables.
		@seealso:
			[event:onkvircshutdown]OnKVIrcShutdown[/event]
	*/
	EVENT("OnKVIrcStartup",""),
	/*
		@doc: onkvircshutdown
		@type:
			event
		@title:
			OnKVIrcShutdown
		@short:
			Triggered on KVIrc shutdown
		@parameters:
			None
		@window:
			Console of the last irc context
		@description:
			This event is triggered on KVIrc shutdown.
			It is bound to the console of the last existing irc context.
			This is a little tricky, since this event is triggered
		    immediately after the on [event:onirccontextdestroyed]OnIrcContextDestroyed[/event] event,
			so in theory the console should be already destroyed.
			This is to make OnApplicationShutdown be the absolutely last event
			triggered at all.
			This event is a good place to save your script configuration.
		@seealso:
			[event:onkvircstartup]OnKVIrcStartup[/event]
	*/
	EVENT("OnKVIrcShutdown",""),
	/*
		@doc: onirccontextcreated
		@type:
			event
		@title:
			OnIrcContextCreated
		@short:
			Triggered when a new irc context is created
		@parameters:
			None
		@window:
			Console of the created irc context
		@description:
			This event is triggered when a new irc context is created.
			It is bound to the console of the related irc context,
			that is the first window that is created.
		@seealso:
			[event:onirccontextdestroyed]OnIrcContextDestroyed[/event]
	*/
	EVENT("OnIrcContextCreated",""),
	/*
		@doc: onirccontextdestroyed
		@type:
			event
		@title:
			OnIrcContextDestroyed
		@short:
			Triggered when an irc context is destroyed
		@parameters:
			None
		@window:
			Console of the destroyed irc context
		@description:
			This event is triggered when an irc context is being destroyed.
			It is bound to the console of the related irc context,
			that is the last window of the context that is destroyed.
		@seealso:
			[event:onirccontextcreated]OnIrcContextCreated[/event]
	*/
	EVENT("OnIrcContextDestroyed",""),
	/*
		@doc: onircconnectionestabilished
		@type:
			event
		@title:
			OnIrcConnectionEstabilished
		@short:
			Triggered when a new irc connection has been estabilished
		@parameters:
			None
		@window:
			Console of the related irc context
		@description:
			This event is triggered immediately after an irc connection
			has been estabilished. No data has been sent to the server yet
			and you have not logged in: it is not a good idea
			attempting to send data to the server (READ: [b]This is NOT
			a good place for joining channels[/b]).
			Your nickname is still "undefined" too.
			If you're looking for the appropriate moment for sending
			data to the server look at the [event:onirc]OnIrc[/event] event.
			This event is a good place for initializing connection related
			variables. By calling [cmd]halt[/cmd] in this event you will stop the
			"Connection estabilished" output.
		@seealso:
			[event:onircconnectionterminated]OnIrcConnectionTerminated[/event]
	*/
	EVENT("OnIrcConnectionEstabilished",""),
	/*
		@doc: onircconnectionterminated
		@type:
			event
		@title:
			OnIrcConnectionTerminated
		@short:
			Triggered when an irc connection has been terminated
		@parameters:
			None
		@window:
			Console of the related irc context
		@description:
			This event is triggered immediately after an irc connection
			has been terminated. The socket is already closed, so you
			can't send data to the server. You can still read
			the connection parameters like the server name or
			network name. By calling [cmd]halt[/cmd] in this
			event you will stop the "Connection terminated..." output.
		@seealso:
			[event:onircconnectionestabilished]OnIrcConnectionEstabilished[/event]
	*/
	EVENT("OnIrcConnectionTerminated",""),
	/*
		@doc: onirc
		@type:
			event
		@title:
			OnIrc
		@short:
			Triggered when the login operations are terminated
		@parameters:
			None
		@window:
			Console of the related irc context
		@description:
			This event is triggered immediately after the login
			operations have been terminated. From now on you are
			"completly on IRC": your nickname has been defined
			and the server is ready to accept commands from you.
			This is a good place to join the "preferred channels".
			By calling [cmd]halt[/cmd] in this event you will
			stop the "Login operations complete..." output.
		@seealso:
			[event:onircconnectionestabilished]OnIrcConnectionEstabilished[/event]
			[event:onircconnectionterminated]OnIrcConnectionTerminated[/event]
	*/
	EVENT("OnIrc",""),
	/*
		@doc: onchannelnickpopuprequest
		@type:
			event
		@title:
			OnChannelNickPopupRequest
		@short:
			Allows customization of the nick-on-channel context popup
		@parameters:
			$0 = nick list
		@window:
			Channel window
		@description:
			This event is triggered when the user right-clicks in the channel user list
			or on a nickname link in the channel text view.[br]
			The first parameter ($0) contains a comma separated list of selected nicknames (if it is an user list click)
			or a single nickname (if it is a link).[br]
			In both cases all the nicknames are currently on the channel.[br]
			You should popup a popup-menu with a set of possible channel-related actions,
			like op/deop , voice/devoice , whois etc...
		@seealso:
			[event:onchannelnickdefaultactionrequest]OnChannelNickDefaultActionRequest[/event]
	*/
	EVENT("OnChannelNickPopupRequest","$0 = nick list"),
	/*
		@doc: onchannelnickdefaultactionrequest
		@type:
			event
		@title:
			OnChannelNickDefaultActionRequest
		@short:
			Allows customization of the double click on nicknames
		@parameters:
			$0 = nick list
		@window:
			Channel window
		@description:
		@seealso:
			[event:onchannelnickpopuprequest]OnChannelNickPopupRequest[/event]
	*/
	EVENT("OnChannelNickDefaultActionRequest","$0 = nick list"),
	/*
		@doc: onframewindowcreated
		@type:
			event
		@title:
			OnFrameWindowCreated
		@short:
			Triggered when a new frame window is opened
		@parameters:
			None
		@window:
			First console in the frame window
		@description:
			This event is triggered in the first console of a new frame window.
			A frame window is the toplevel window that contains the consoles
			and the other MDI children windows.
		@seealso:
			[event:onframewindowdestroyed]OnFrameWindowDestroyed[/event]
	*/
	EVENT("OnFrameWindowCreated",""),
	/*
		@doc: onframewindowdestroyed
		@type:
			event
		@title:
			OnFrameWindowDestroyed
		@short:
			Triggered when a frame window is being destroyed
		@parameters:
			None
		@window:
			Last console in the frame window
		@description:
			This event is triggered in the last console of a frame window that is
			being destroyed. In fact the frame window is not destroyed yet, but it
			is going to: there would be no way to trigger this event after the
			last frame window destruction.
		@seealso:
			[event:onframewindowcreated]OnFrameWindowCreated[/event]
	*/
	EVENT("OnFrameWindowDestroyed",""),
	/*
		@doc: onchannelpopuprequest
		@type:
			event
		@title:
			OnChannelPopupRequest
		@short:
			Triggered when the channel text view is right-clicked
		@parameters:
			None
		@window:
			Channel window
		@description:
			This event is triggered when the user clicks the channel
			text view with the right mouse button. It is a good place
			to activate a [cmd]popup[/cmd] menu :)
		@seealso:
			[event:onconsolepopuprequest]OnConsolePopupRequest[/event],
			[event:onquerypopuprequest]OnQueryPopupRequest[/event]
	*/
	EVENT("OnChannelPopupRequest",""),
	/*
		@doc: onconsolepopuprequest
		@type:
			event
		@title:
			OnConsolePopupRequest
		@short:
			Triggered when the console text view is right-clicked
		@parameters:
			None
		@window:
			Console window
		@description:
			This event is triggered when the user clicks the console
			text view with the right mouse button. It is a good place
			to activate a [cmd]popup[/cmd] menu :)
		@seealso:
			[event:onchannelpopuprequest]OnChannelPopupRequest[/event],
			[event:onquerypopuprequest]OnQueryPopupRequest[/event]
	*/
	EVENT("OnConsolePopupRequest",""),
	/*
		@doc: onquerypopuprequest
		@type:
			event
		@title:
			OnQueryPopupRequest
		@short:
			Triggered when the query text view is right-clicked
		@parameters:
			None
		@window:
			Query
		@description:
			This event is triggered when the user clicks the query
			text view with the right mouse button. It is a good place
			to activate a [cmd]popup[/cmd] menu :)
		@seealso:
			[event:onconsolepopuprequest]OnConsolePopupRequest[/event],
			[event:onchannelpopuprequest]OnChannelPopupRequest[/event]
	*/
	EVENT("OnQueryPopupRequest",""),
	/*
		@doc: onchannelsync
		@type:
			event
		@title:
			OnChannelSync
		@short:
			Triggered when all the channel informations have been received
		@parameters:
			$0 = channel sync time in milliseconds
		@window:
			Channel
		@description:
			This event is triggered after you have joined a channel
			and all the informations have been received.[br]
			This includes the channel users list, the ban list,
			the channel mode, the topic and eventually
			the invite and ban exception lists.[br]
			Calling halt in this even stops the "channel synchronized" output.[br]
			(the output will be halted anyway if [fnc]$option[/fnc](boolShowChannelSyncTime) evaluates to false)
	*/
	EVENT("OnChannelSync","$0 = Channel sync time in milliseconds"),
	/*
		@doc: onurl
		@type:
			event
		@title:
			OnUrl
		@short:
			Triggered when an url appears in one of the output windows
		@parameters:
			$0 = Url
		@window:
			The window that the URL appeared in
		@description:
			This event is triggered every time that an url is going to appear
			in a KVIrc window output widget. You can not stop it nor modify it.
			This event is good for implementing an url catcher.[br]
			WARNING: If you attempt to "echo $0" inside this event kvirc will hang:
			I can do nothing about it...this is infinite recursion....
	*/
	EVENT("OnUrl","$0 = Url"),
	/*
		@doc: onwindowpopuprequest
		@type:
			event
		@title:
			OnWindowPopupRequest
		@short:
			Used for window popup customization
		@parameters:
			none
		@window:
			The window that the user has clicked in
		@description:
			This event is here to allow easy customization of the window popup (it was builtin in KVIrc version prior to 3.0.0).[br]
			Actually it is triggered when the user middle-clicks in a text view (or right-clicks with the CTRL key pressed).[br]
			You should use it to popup your own version of the window popup that should interface the
			logging and window-behaviour functions.[br]
	*/
	EVENT("OnWindowPopupRequest",""),
	/*
		@doc: onhostlinkpopuprequest
		@type:
			event
		@title:
			OnHostLinkPopupRequest
		@short:
			Host link popup customization
		@parameters:
			$0 = host name
		@window:
			The window that the user has clicked in
		@description:
			This event is triggered when an user has right clicked on a hostname link in
			the text view. The double click link action is builtin and it causes KVIrc
			to perform a DNS query on the hostname.
			This is a good place to show a [cmd]popup[/cmd] menu with some operations
			that can be done on the hostname like "telnet" , "traceroute" , some special
			kind of DNS lookup (maybe an xterm with nslookup or sth).[br]
	*/
	EVENT("OnHostLinkPopupRequest","$0 = hostname"),
	/*
		@doc: onurllinkpopuprequest
		@type:
			event
		@title:
			OnUrlLinkPopupRequest
		@short:
			Url link popup customization
		@parameters:
			$0 = url
		@window:
			The window that the user has clicked in
		@description:
			This event is triggered when an user has right clicked on an URL link in
			the text view. The double click link action is builtin and it causes KVIrc
			to open the URL with a browser (konqueror in KDE compilation mode or netscape).[br]
			This is a good place to show a [cmd]popup[/cmd] menu with some operations
			that can be performed with the URL , like bookmarking in some way or opening
			with a specific browser.[br]
	*/
	EVENT("OnUrlLinkPopupRequest","$0 = url"),
	/*
		@doc: onserverlinkpopuprequest
		@type:
			event
		@title:
			OnServerLinkPopupRequest
		@short:
			Server link popup customization
		@parameters:
			$0 = server name
		@window:
			The window that the user has clicked in
		@description:
			This event is triggered when an user has right clicked on an server name link in
			the text view. The double click link action is builtin and it causes KVIrc
			to query the MOTD of the server.[br]
			This is a good place to show a [cmd]popup[/cmd] menu with some operations
			that can be performed with the server name , like querying stats or links.[br]
			Please note that the server name is generally NOT a hostname: some parts of
			it are often hidden. The name of the server is generally valid only within
			the IRC network that you are connected to.[br]
	*/
	EVENT("OnServerLinkPopupRequest","$0 = server name"),
	/*
		@doc: onchannellinkpopuprequest
		@type:
			event
		@title:
			OnChannelLinkPopupRequest
		@short:
			Channel link popup customization
		@parameters:
			$0 = channel name
		@window:
			The window that the user has clicked in
		@description:
			This event is triggered when an user has right clicked on an channel link in
			the text view. The double click link action is builtin and it causes KVIrc
			to join the channel (unless the user is already on).[br]
			This is a good place to show a [cmd]popup[/cmd] menu with some operations
			that can be performed with the channel name like bookmarking, joining with a password,
			performing a [cmd]who[/cmd] etc...
	*/
	EVENT("OnChannelLinkPopupRequest","$0 = channel name"),
	/*
		@doc: onnicklinkpopuprequest
		@type:
			event
		@title:
			OnNickLinkPopupRequest
		@short:
			Nick link popup customization
		@parameters:
			$0 = nickname
		@window:
			The window that the user has clicked in
		@description:
			This event is triggered when an user has right clicked on an "unknown" nickname link in
			the text view.[br]
			This is a good place to show a [cmd]popup[/cmd] menu with some operations
			that can be performed with the nickname like querying , sending a file by dcc or opening a dcc.chat.
			Please note that NOT ALL the nickname links will receive this event.[br]
			The nickname links that appear in the text view of the channel that they are actually in
			are handled by the [event:onchannelnickpopuprequest]OnChannelNickPopupRequest[/event] event.[br]
			The nickname links that appear in the text view of the query in that they are a remote end
			are handled by the [event:onquerynickpopuprequest]OnQueryNickPopupRequest[/event] event.[br]
	*/
	EVENT("OnNickLinkPopupRequest","$0 = nick name"),
	/*
		@doc: onquerynickpopuprequest
		@type:
			event
		@title:
			OnQueryNickPopupRequest
		@short:
			Query nicklist popup customization
		@parameters:
			$0 = nickname list
		@window:
			The window that the user has clicked in
		@description:
			This event is triggered when an user has right clicked on a set of nicknames in
			the query target list.[br]
			This is a good place to show a [cmd]popup[/cmd] menu with some operations
			that can be performed with the nickname like sending a file by dcc, opening a dcc.chat or
			performing a whois.[br]
	*/
	EVENT("OnQueryNickPopupRequest","$0 = nickname list"),
	/*
		@doc: onquerynickdefaultactionrequest
		@type:
			event
		@title:
			OnQueryNickDefaultActionRequest
		@short:
			Query double click customization
		@parameters:
			$0 = nickname
		@window:
			The window that the user has clicked in
		@description:
			This event is triggered when an user has double clicked on a nickname in
			the query target list.[br]
			You should use this event to do some significant action associated with the double click.[br]
			A good example might be a [cmd]whois[/cmd] query or a [cmd]dcc.chat[/cmd]
	*/
	EVENT("OnQueryNickDefaultActionRequest","$0 = nickname"),
	/*
		@doc: onnicklinkdefaultactionrequest
		@type:
			event
		@title:
			OnNickLinkDefaultActionRequest
		@short:
			Nick link double click customization
		@parameters:
			$0 = nickname
		@window:
			The window that the user has clicked in
		@description:
			This event is triggered when an user has double clicked on a nickname link.[br]
			You should use this event to perform some significant action associated with the double click.[br]
			A good example might be a [cmd]whois[/cmd] or a [cmq]query[/cmd] call.[br]
			Please note that NOT ALL nickname links will cause this event to be triggered: some nickname
			links default actions are actually handled by the
			[event:onquerynickdefaultactionrequest]OnQueryNickDefaultActionRequest[/event] and
			[event:onchannelnickdefatulactionrequest]OnChannelNickDefaultActionRequest[/event] events.[br]
	*/
	EVENT("OnNickLinkDefaultActionRequest","$0 = nickname"),
	/*
		@doc: ontextviewdoubleclicked
		@type:
			event
		@title:
			OnTextViewDoubleClicked
		@short:
			Text view link double click customization
		@parameters:
			None
		@window:
			The window that the user has clicked in
		@description:
			This event is triggered when an user has double clicked on a text view and has not
			hit any link.[br]
			I have no idea what you can do in this event....maybe some sort of channel-central dialog ?[br]
			:)[br]
	*/
	EVENT("OnTextViewDoubleClicked",""),
	/*
		@doc: onnotifylistdefaultactionrequest
		@type:
			event
		@title:
			OnNotifyListDefaultActionRequest
		@short:
			Notify list double click customization
		@parameters:
			$0 = nickname
		@window:
			The window that the user has clicked in
		@description:
			This event is triggered when an user has double clicked on a nickname in
			the notify list (console window).[br]
			You should use this event to do some significant action associated with the double click.[br]
			A good example might be a [cmd]whois[/cmd] query or a [cmd]dcc.chat[/cmd]
	*/
	EVENT("OnNotifyListDefaultActionRequest","$0 = nickname"),
	/*
		@doc: onnotifylistpopuprequest
		@type:
			event
		@title:
			OnNotifyListPopupRequest
		@short:
			Notify list context popup customization
		@parameters:
			$0 = nickname list
		@window:
			The window that the user has clicked in
		@description:
			This event is triggered when an user has right clicked on a list of nicknames in
			the notify list (console window).[br]
			This is a good place to show a [cmd]popup[/cmd] with some actions that can be performed
			on the nicknames like querying , performing a whois lookup or sth similar.[br]
	*/
	EVENT("OnNotifyListPopupRequest","$0 = nickname list"),
	/*
		@doc: onping
		@type:
			event
		@title:
			OnPing
		@short:
			Server ping handler
		@parameters:
			$0 = server
			$1 = ping message parameters
		@window:
			Console
		@description:
			Triggered when a PING message was received from a server.[br]
			Calling 'halt' in this event will stop the informational message about 'ping->pong'.[br]
	*/
	EVENT("OnPing","$0 = server\n$1 = message parameters"),
	/*
		@doc: onmejoin
		@type:
			event
		@title:
			OnMeJoin
		@short:
			Local user joining a channel
		@parameters:
			None
		@window:
			Channel
		@description:
			Triggered just after the local user has joined a channel.[br]
			Sine this event is triggered in the chanel window,
			you can get the channel name by using the [fnc]$target[/fnc] function.[br]
			Calling 'halt' in this event will stop the join message output.[br]
		@seealso:
			[event:onjoin]OnJoin[/event]
	*/
	EVENT("OnMeJoin",""),
	/*
		@doc: onjoin
		@type:
			event
		@title:
			OnJoin
		@short:
			Non-local user joining a channel
		@parameters:
			$0 = nickname
			$1 = username
			$2 = hostname
		@window:
			Channel
		@description:
			Triggered just after a non-local user has joined a channel.[br]
			Sine this event is triggered in the chanel window,
			you can get the channel name by using the [fnc]$target[/fnc] function.[br]
			Calling 'halt' in this event will stop the join message output.[br]
		@seealso:
			[event:onmejoin]OnMeJoin[/event]
	*/
	EVENT("OnJoin","$0 = nickname\n$1 = username\n$2 = hostname"),
	/*
		@doc: ontopic
		@type:
			event
		@title:
			OnTopic
		@short:
			Topic change on a channel
		@parameters:
			$0 = nickname
			$1 = username
			$2 = hostname
			$3 = new topic string
		@window:
			Channel
		@description:
			Triggered when a TOPIC message is received for a given channel.[br]
			The topic change has not been executed effectively in the current window
			so you can still access the old topic by the channel functions.[br]
			The new topic is passed as fourth parameter and will effectively
			become the current topic just after this event handler has been executed.[br]
			Sine this event is triggered in the chanel window,
			you can get the channel name by using the [fnc]$target[/fnc] function.[br]
			Calling 'halt' in this event will stop the topic message output.[br]
		@seealso:
			[event:onmejoin]OnMeJoin[/event]
	*/
	EVENT("OnTopic","$0 = nickname\n$1 = username\n$2 = hostname\n$3 = topic"),
	/*
		@doc: onmepart
		@type:
			event
		@title:
			OnMePart
		@short:
			Local user is leaving a channel
		@parameters:
			$0 = part message
		@window:
			Channel
		@description:
			Triggered when the local user is leaving a channel.[br]
			The PART message has already been processed by the server , so
			the user is effectively no longer on the channel.
			On the other side, it might be useful to access the channel data from
			this event, thus the handler is called effectively just BEFORE the
			user effectively leaves the channel (and thus BEFORE the channel window gets
			destroyed).[br]
			Sine this event is triggered in the chanel window,
			you can get the channel name by using the [fnc]$target[/fnc] function.[br]
			Calling 'halt' in this event will stop the part message output.[br]
		@seealso:
			[event:onmejoin]OnMeJoin[/event], [event:OnJoin]OnJoin[/event],
			[event:onpart]OnPart[/event]
	*/
	EVENT("OnMePart","$0 = part message"),
	/*
		@doc: onpart
		@type:
			event
		@title:
			OnPart
		@short:
			Non-local user is leaving a channel
		@parameters:
			$0 = nickname
			$1 = username
			$2 = hostname
			$3 = part message
		@window:
			Channel
		@description:
			Triggered when a NON-local user is leaving a channel.[br]
			The PART message has already been processed by the server , so
			the user is effectively no longer on the channel.
			On the other side, it might be useful to access the user-related channel data from
			this event, thus the handler is called effectively just BEFORE the
			user effectively leaves the channel (and thus BEFORE the user-related channel data gets
			destroyed).[br]
			Since this event is triggered in the chanel window,
			you can get the channel name by using the [fnc]$target[/fnc] function.[br]
			Calling 'halt' in this event will stop the part message output.[br]
		@seealso:
			[event:onmejoin]OnMeJoin[/event], [event:OnJoin]OnJoin[/event],
			[event:onmepart]OnMePart[/event]
	*/
	EVENT("OnPart","$0 = nickname\n$1 = username\n$2 = host\n$3 = part message"),
	/*
		@doc: onquit
		@type:
			event
		@title:
			OnQuit
		@short:
			An user is leaving a channel
		@parameters:
			$0 = nickname
			$1 = username
			$2 = hostname
			$3 = quit message
		@window:
			Console
		@description:
			Triggered when an user user is quitting IRC.[br]
			The QUIT message has already been processed by the server , so
			the user is effectively no longer on IRC.
			On the other side, it might be useful to access the user-related data from
			this event, thus the handler is called effectively just BEFORE the
			user is effectively removed from the IRC user list (and thus from the channels and query windows).[br]
			Calling 'halt' in this event will stop the quit message output (in ALL the windows!).[br]
		@seealso:
			[event:onpart]OnPart[/event],
			[event:onmepart]OnMePart[/event]
	*/
	EVENT("OnQuit","$0 = nickname\n$1 = username\n$2 = host\n$3 = part message"),
	/*
		@doc: onusermode
		@type:
			event
		@title:
			OnUserMode
		@short:
			The local user has changed mode
		@parameters:
			$0 = mode flags
		@window:
			Console
		@description:
			Triggered when the local user has changed his user mode.[br]
			The first parameter is the changed mode flag with a leading '+' or '-' sign:
			if the mode was removed it will be something like "-x" , otherwise something like "+x".[br]
			Calling 'halt' in this event will stop the "mode" message output.[br]
	*/
	EVENT("OnUserModeChanged","$0 = mode flags"),
	/*
		@doc: onmekick
		@type:
			event
		@title:
			OnMeKick
		@short:
			Local user is being kicked from a channel
		@parameters:
			$0 = source nickname
			$1 = source username
			$2 = source hostname
			$3 = kick reason
		@window:
			Channel
		@description:
			Triggered when the local user has been kicked from a channel.[br]
			The KICK message has already been processed by the server , so
			the user is effectively no longer on the channel.
			On the other side, it might be useful to access the user-related channel data from
			this event, thus the handler is called effectively just BEFORE the
			user is effectively removed from the channel internal structures
			(and thus BEFORE the channel data itself gets destroyed).[br]
			Sine this event is triggered in the chanel window,
			you can get the channel name by using the [fnc]$target[/fnc] function.[br]
			Calling 'halt' in this event will stop the kick message output.[br]
		@seealso:
			[event:onkick]OnKick[/event]
	*/
	EVENT("OnMeKick","$0 = source nick\n$1 = source username\n$2 = source hostname\n$3 = reason"),
	/*
		@doc: onkick
		@type:
			event
		@title:
			OnKick
		@short:
			Non-local user is being kicked from a channel
		@parameters:
			$0 = source nickname
			$1 = source username
			$2 = source hostname
			$3 = kicked nickname
			$4 = kick reason
		@window:
			Channel
		@description:
			Triggered when a non-local user is being kicked from a channel.[br]
			The KICK message has already been processed by the server , so
			the user is effectively no longer on the channel.
			On the other side, it might be useful to access the user-related channel data from
			this event, thus the handler is called effectively just BEFORE the
			user is effectively removed from the channel internal structures.[br]
			Sine this event is triggered in the chanel window,
			you can get the channel name by using the [fnc]$target[/fnc] function.[br]
			Calling 'halt' in this event will stop the kick message output.[br]
		@seealso:
			[event:onmekick]OnMeKick[/event]
	*/
	EVENT("OnKick","$0 = source nick\n$1 = source username\n$2 = source hostname\n$3 = kicked nick\n$4 = reason"),
	/*
		@doc: onmeaway
		@type:
			event
		@title:
			OnMeAway
		@short:
			Local user sets the AWAY status
		@parameters:
			$0 = server message trailing
		@window:
			Console
		@description:
			Triggered when the local user enters the AWAY state.
			Calling 'halt' in this event disables the server "You have been marked as being away" message.[br]
		@seealso:
			[event:onmeback]OnMeBack[/event]
	*/
	EVENT("OnMeAway","$0- = server message trailing"),
	/*
		@doc: onmeback
		@type:
			event
		@title:
			OnMeBack
		@short:
			Local user cancels the AWAY status
		@parameters:
			$0 = away start time
			$1- = server message trailing
		@window:
			Console
		@description:
			Triggered when the local user leaves the AWAY state.[br]
			The $0 parameter contains the [b]unix time[/b] of the away status start.[br]
			You can obtain the current [b]unix time[/b] from the function [fnc]$unixtime[/fnc].[br]
			Calling 'halt' in this event disables the server "You are no longer marked as being away" message.[br]
		@seealso:
			[event:onmeaway]OnMeAway[/event]
	*/
	EVENT("OnMeBack","$0 = away start time\n$1- = server message trailing"),
	/*
		@doc: onchannelwindowcreated
		@type:
			event
		@title:
			OnChannelWindowCreated
		@short:
			A channel window has just been opened
		@parameters:
			none
		@window:
			Channel window
		@description:
			Triggered when a channel window has just been created
		@seealso:
			[event:onquerywindowcreated]OnQueryWindowCreated[/event]
	*/
	EVENT("OnChannelWindowCreated",""),
	/*
		@doc: onquerywindowcreated
		@type:
			event
		@title:
			OnQueryWindowCreated
		@short:
			A query window has just been opened
		@parameters:
			none
		@window:
			Query window
		@description:
			Triggered when a query window has just been created
		@seealso:
			[event:onchannelwindowcreated]OnChannelWindowCreated[/event]
	*/
	EVENT("OnQueryWindowCreated",""),
	/* 
	 	@doc: onban
		@type:
			event
		@title:
			OnBan
		@short:
			Someone has set a +b flag on the channel
		@parameters:
			$0 = source nick
			$1 = source username
			$2 = source hostname
			$3 = ban mask			
		@window:
			Channel window
		@description:
			Triggered when someone sets a +b flag on the channel
		@seealso:
			[event:onunban]OnUnban[/event]
	*/
	EVENT("OnBan","$0 = source nick\n$1 = source username\n$2 = source hostname\n$3 = ban mask"),
	/*
	 	@doc: onunban
		@type:
			event
		@title:
			OnUnban
		@short:
			Someone has set a -b flag on the channel
		@parameters:
			$0 = source nick
			$1 = source username
			$2 = source hostname
			$3 = unban mask
		@window:
			Channel window
		@description:
			Triggered when someone sets a -b flag on the channel
		@seealso:
			[event:onban]OnBan[/event]
	*/		
	EVENT("OnUnBan","$0 = source nick\n$1 = source username\n$2 = source hostname\n$3 = ban mask"),
	/*
		@doc: onmeban
		@type:
			event
		@title:
			OnMeBan
		@short:
			Local user is being banned in a channel
		@parameters:
			$0 = source nickname
			$1 = source username
			$2 = source hostname
		@window:
			Channel
		@description:
			Triggered when someone has set a +b flag that matches your mask (bans you) in the channel.[br]
		@seealso:
			[event:onmeunban]OnMeUnban[/event]
	*/
	EVENT("OnMeBan","$0 = source nick\n$1 = source username\n$2 = source hostname"),
	/*
		@doc: onmeunban
		@type:
			event
		@title:
			OnMeUnban
		@short:
			Local user is being de-banned in a channel
		@parameters:
			$0 = source nickname
			$1 = source username
			$2 = source hostname
		@window:
			Channel
		@description:
			Triggered when someone has set a -b flag that matches your mask (debans you) in the channel.[br]
		@seealso:
			[event:onmeban]OnMeBan[/event]
	*/
EVENT("OnMeUnban","$0 = source nick\n$1 = source username\n$2 = source hostname"),
 	/*
        @doc: onbanexception
        @type:
            event
        @title:
            OnBanException
        @short:
            Someone has set a +e flag on the channel
        @parameters:
            $0 = source nick
            $1 = source username
            $2 = source hostname
            $3 = banexception mask
        @window:
            Channel window
        @description:
            Triggered when someone sets a +e (Ban Exception) flag on the channel
        @seealso:
            [event:onbanexceptionremove]OnBanExceptionRemove[/event]
    */

	EVENT("OnBanException","$0 = source nick\n$1 = source username\n$2 = source hostname\n$3 = ban exception mask"),
    /*
        @doc: onbanexceptionremove
        @type:
            event
        @title:
            OnBanExceptionRemove
        @short:
            Someone has set a -e flag on the channel
        @parameters:
            $0 = source nick
            $1 = source username
            $2 = source hostname
            $3 = unban mask
        @window:
            Channel window
        @description:
            Triggered when someone sets a -e (Removed a Ban Exception) flag on the channel
        @seealso:
            [event:onbanexception]OnBanException[/event]
    */
	EVENT("OnBanExceptionRemove","$0 = source nick\n$1 = source username\n$2 = source hostname\n$3 = ban exception mask"),
    /*
        @doc: onmebanexception
        @type:
            event
        @title:
            OnMeBanException
        @short:
            Someone has set a +e flag on the local user
        @parameters:
            $0 = source nick
            $1 = source username
            $2 = source hostname
        @window:
            Channel window
        @description:
            Triggered when someone sets a +e (puts a Ban Exception) flag on the local user in the active
channel
        @seealso:
            [event:onmebanexceptionremove]OnBanMEExceptionRemove[/event]
    */
	EVENT("OnMeBanException","$0 = source nick\n$1 = source username\n$2 = source hostname"),
    /*
        @doc: onmebanexceptionremove
        @type:
            event
        @title:
            OnMeBanExceptionRemove
        @short:
            Someone has set a -e flag on the local user
        @parameters:
            $0 = source nick
            $1 = source username
            $2 = source hostname
        @window:
            Channel window
        @description:
            Triggered when someone sets a -e (removes a Ban Exception) flag on the local user in the active channel
        @seealso:
            [event:onmebanexception]OnBanMEException[/event]
    */
	EVENT("OnMeBanExceptionRemove","$0 = source nick\n$1 = source username\n$2 = source hostname"),
	/*
	 	@doc: onop
		@type:
			event
		@title:
			OnOp
		@short:
			Someone has give some user a +o flag
		@parameters:
			$0 = source nick
			$1 = source username
			$2 = source host
			$3 = opped nick
		@window:
			Channels window
		@description:
			Triggered when someone sets a +o (has made Op) flag on a user of the active channel.
		@seealso:
			[event:ondeop]OnDeOp[/event]
	*/
	EVENT("OnOp","$0 = source nick\n$1 = source username\n$2 = source hostname\n$3 = opped nick"),
	/*
	 	@doc: ondeop
		@type:
			event
		@title:
			OnDeOp
		@short:
			Someone has give some user a -o flag
		@parameters:
			$0 = source nick
			$1 = source username
			$2 = source host
			$3 = deopped nick
		@window:
			Channels window
		@description:
			Triggered when someone sets a -o (DeOp) flag on a user in the active channel.
		@seealso:
			[event:onop]OnOp[/event]
	*/
	EVENT("OnDeOp","$0 = source nick\n$1 = source username\n$2 = source hostname\n$3 = deopped nick"),
	/*
	 	@doc: onmeop
		@type:
			event
		@title:
			OnMeOp
		@short:
			Someone has give some user a +o flag to the local user
		@parameters:
			$0 = source nick
			$1 = source username
			$2 = source host
		@window:
			Channels window
		@description:
			Triggered when someone sets a +o (has made Op) flag on the local user in the active channel.
		@seealso:
			[event:onmedeop]OnMeDeOp[/event]
	*/
	EVENT("OnMeOp","$0 = source nick\n$1 = source username\n$2 = source hostname"),
	/*
	 	@doc: onmedeop
		@type:
			event
		@title:
			OnMeDeOp
		@short:
			Someone has give some user a -o flag to the local user
		@parameters:
			$0 = source nick
			$1 = source username
			$2 = source host
		@window:
			Channels window
		@description:
			Triggered when someone sets a -o (has deoped) flag on the local user in the active channel.
		@seealso:
			[event:onmeop]OnMeOp[/event]
	*/
	EVENT("OnMeDeOp","$0 = source nick\n$1 = source username\n$2 = source hostname"),
	/*
	 	@doc: onlimitset
		@type:
			event
		@title:
			OnLimitSet
		@short:
			The channel users limit has been set
		@parameters:
			$0 = source nick
			$1 = source username
			$2 = source host
			$3 = limit
		@window:
			Channels window
		@description:
			Triggered when someone change or sets the active channel user limit (+l)
		@seealso:
			[event:onlimitunset]OnLimitUnset[/event]
	*/
	EVENT("OnLimitSet","$0 = source nick\n$1 = source username\n$2 = source hostname\n$3 = limit"),
    /*
        @doc: onlimitunset
        @type:
            event
        @title:
            OnLimitUnSet
        @short:
            The channel users limit has been unset
        @parameters:
            $0 = source nick
            $1 = source username
            $2 = source host
        @window:
            Channels window
        @description:
            Triggered when someone unsets the active channel user limit (-l)
        @seealso:
            [event:onlimitunset]OnLimitUnset[/event]
    */
	EVENT("OnLimitUnset","$0 = source nick\n$1 = source username\n$2 = source hostname"),
    /*
        @doc: onkeyset
        @type:
            event
        @title:
            OnKeySet
        @short:
            The channel users key has been set
        @parameters:
            $0 = source nick
            $1 = source username
            $2 = source host
            $3 = key
        @window:
            Channels window
        @description:
            Triggered when someone change or sets the active channel access key (+k)
        @seealso:
            [event:onkeyunset]OnKeyUnset[/event]
    */
	EVENT("OnKeySet","$0 = source nick\n$1 = source username\n$2 = source hostname\n$3 = key"),
    /*
        @doc: onkeyunset
        @type:
            event
        @title:
            OnKeyUnset
        @short:
            The channel users key has been unset
        @parameters:
            $0 = source nick
            $1 = source username
            $2 = source host
        @window:
            Channels window
        @description:
            Triggered when someone unsets the active channel access key (-k)
        @seealso:
            [event:onkeyunset]OnKeyUnset[/event]
    */
	EVENT("OnKeyUnset","$0 = source nick\n$1 = source username\n$2 = source hostname"),
    /*
        @doc: oninvite
        @type:
            event
        @title:
            OnInvite
        @short:
            The local user has received an invitation
        @parameters:
            $0 = source nick
            $1 = source username
            $2 = source host
            $3 = channel
        @window:
            Active window/console
        @description:
            Triggered when someone invites the local user to join a channel
    */
	EVENT("OnInvite","$0 = source nick\n$1 = source username\n$2 = source hostname\n$3 = channel"),
    /*
        @doc: onchannelmessage
        @type:
            event
        @title:
            OnChannelMessage
        @short:
            A message has been received from the channel
        @parameters:
            $0 = source nick
            $1 = source username
            $2 = source host
            $3 = message
        @window:
            Channels window
        @description:
            Triggered when a channel message is received
        @seealso:
            [event:onquerymessage]OnQueryMessage[/event]
			[event:ondccchatmessage]OnDCCChatMessage[/event]
    */
	EVENT("OnChannelMessage","$0 = source nick\n$1 = source username\n$3 = source hostname\n$3 = message"),
    /*
        @doc: onquerymessage
        @type:
            event
        @title:
            OnQueryMessage
        @short:
            A message has been received from a query
        @parameters:
            $0 = source nick
            $1 = source username
            $2 = source host
            $3 = message
        @window:
            Query window
        @description:
            Triggered when a query message is received.[br]
			If you call [cmd]halt[/cmd] in this event, the message output will be stopped
			and if the message was going to cause a query window creation, the window will NOT be created.[br]
        @seealso:
            [event:onchannelmessage]OnChannelMessage[/event]
			[event:ondccchatmessage]OnDCCChatMessage[/event]
    */
	EVENT("OnQueryMessage","$0 = source nick\n$1 = source username\n$3 = source hostname\n$3 = message"),
    /*
        @doc: onvoice
        @type:
            event
        @title:
            OnVoice
        @short:
            Someone has give some user a +v flag
        @parameters:
            $0 = source nick
            $1 = source username
            $2 = source host
		    $3 = voiced nick
        @window:
            Channels window
        @description:
            Triggered when someone sets a +v (voice) flag to someone in the active channel.
        @seealso:
            [event:ondevoice]OnDeVoice[/event]
	*/
	EVENT("OnVoice","$0 = source nick\n$1 = source username\n$2 = source hostname\n$3 = voiced nick"),
    /*
        @doc: ondevoice
        @type:
            event
        @title:
            OnDeVoice
        @short:
            Someone has give some user a -v flag
        @parameters:
            $0 = source nick
            $1 = source username
            $2 = source host
			$3 = devoiced nick
        @window:
            Channels window
        @description:
            Triggered when someone sets a -v (devoice) flag to someone in the active channel.
        @seealso:
            [event:ondevoice]OnDeVoice[/event]
    */
	EVENT("OnDeVoice","$0 = source nick\n$1 = source username\n$2 = source hostname\n$3 = voiced nick"),
    /*
        @doc: onmevoice
        @type:
            event
        @title:
            OnMeVoice
        @short:
            Someone has give the local user a +v flag
        @parameters:
            $0 = source nick
            $1 = source username
            $2 = source host
        @window:
            Channels window
        @description:
            Triggered when someone sets a +v (devoice) flag to the local user in the active channel.
        @seealso:
            [event:onmedevoice]OnMeDeVoice[/event]
    */
	EVENT("OnMeVoice","$0 = source nick\n$1 = source username\n$2 = source hostname"),
    /*
        @doc: onmedevoice
        @type:
            event
        @title:
            OnMeDeVoice
        @short:
            Someone has give the local user a -v flag
        @parameters:
            $0 = source nick
            $1 = source username
            $2 = source host
        @window:
            Channels window
        @description:
            Triggered when someone sets a -v (devoice) flag to the local user in the active channel.
        @seealso:
            [event:onmevoice]OnMeVoice[/event]
    */
	EVENT("OnMeDeVoice","$0 = source nick\n$1 = source username\n$2 = source hostname"),
    /*
        @doc: onnickchange
        @type:
            event
        @title:
            OnNickChange
        @short:
            Someone has changed his nickname
        @parameters:
			$0 = source nick
			$1 = source username
			$2 = source host
			$3 = new nickname
        @window:
            Console window
        @description:
            Triggered when someone has changed his nickname.
			The change has already been processed by the server, but not by the
			local KVirc database.
        @seealso:
            [event:onmenickchange]OnMeNickChange[/event]
    */
	EVENT("OnNickChange","$0 = source nick\n$1 = source username\n$2 = source hostname\n$3 = new nickname"),
    /*
        @doc: onmenickchange
        @type:
            event
        @title:
            OnMeNickChange
        @short:
            The local user has changed his nickname
        @parameters:
			$0 = old nick
			$1 = new nickname
        @window:
            Console window
        @description:
            Triggered when the local user is going to change his nickname.
			The change has already been processed by the server, but not by the
			local KVirc database.
        @seealso:
            [event:onnickchange]OnNickChange[/event]
    */
	EVENT("OnMeNickChange","$0 = old nick\n$1 = new nick"),
    /*
        @doc: onctcprequest
        @type:
            event
        @title:
            OnCtcpRequest
        @short:
            A ctcp request has been received
        @parameters:
			$0 = source nick
			$1 = source user
			$2 = source host
			$3 = target
			$4 = ctcp type
			$5- = ctcp parameters
        @window:
            Console window
        @description:
            A ctcp request has been received.[br]
			If you call [cmd]halt[/cmd] in this event , you will stop the further
			processing of the CTCP (thus you can disable some of the kvirc features).
			Be sure to know what you're doing.
        @seealso:
            [event:onctcpreply]OnCtcpReply[/event]
    */
	EVENT("OnCtcpRequest","$0 = source nick\n$1 = source user\n$2 = source host\n$3 = target\n$4 = ctcp type\n$5- = ctcp parameters"),
    /*
        @doc: onctcpreply
        @type:
            event
        @title:
            OnCtcpReply
        @short:
            A ctcp reply has been received
        @parameters:
			$0 = source nick
			$1 = source user
			$2 = source host
			$3 = target
			$4 = ctcp type
			$5- = ctcp parameters
        @window:
            Console window
        @description:
            A ctcp reply has been received.[br]
			If you call [cmd]halt[/cmd] in this event , you will stop the further
			processing of the CTCP (thus you can disable some of the kvirc features).
			Be sure to know what you're doing.
        @seealso:
            [event:onctcpreply]OnCtcpReply[/event]
    */
	EVENT("OnCtcpReply","$0 = source nick\n$1 = source user\n$2 = source host\n$3 = target\n$4 = ctcp type\n$5- = ctcp parameters"),
    /*
        @doc: onunhandledliteral
        @type:
            event
        @title:
            OnUnhandledLiteral
        @short:
            An unhandled literal server message has been received
        @parameters:
			$0 = source mask
			$1 = message (literal)
			$2- = parameters
        @window:
            Console window
        @description:
			An unhandled/unrecognized literal server message has been received.[br]
			The KVIrc core code hasn't been able to recognize it nor handle it
			in any way.[br]
			If you call [cmd]halt[/cmd] you will stop the standard or warning message output.[br]
			For server numerics there is no "unhandled" event, you should use the
			raw numeric events to handle them.[br]
        @seealso:
    */
	EVENT("OnUnhandledLiteral","$0 = source mask\n$1 = message\n$2- parameters"),
    /*
        @doc: onmehalfop
        @type:
            event
        @title:
            OnMeHalfOp
        @short:
            Someone has given the local user the +h flag
        @parameters:
            $0 = source nick
            $1 = source username
            $2 = source host
        @window:
            Channels window
        @description:
            Triggered when someone sets a +h (halfop) flag to the local user in the active channel.
        @seealso:
            [event:onmedehalfop]OnMeDeHalfOp[/event]
    */
	EVENT("OnMeHalfOp","$0 = source nick\n$1 = source username\n$2 = source hostname"),
    /*
        @doc: onmedehalfop
        @type:
            event
        @title:
            OnMeDeHalfOp
        @short:
            Someone has given the local user the -h flag
        @parameters:
            $0 = source nick
            $1 = source username
            $2 = source host
        @window:
            Channels window
        @description:
            Triggered when someone sets a -h (dehalfop) flag to the local user in the active channel.
        @seealso:
            [event:onmehalfop]OnMeHalfOp[/event]
    */
	EVENT("OnMeDeHalfOp","$0 = source nick\n$1 = source username\n$2 = source hostname"),
	/*
	 	@doc: onhalfop
		@type:
			event
		@title:
			OnHalfOp
		@short:
			Someone has give some user a +h flag
		@parameters:
			$0 = source nick
			$1 = source username
			$2 = source host
			$3 = halfopped nick
		@window:
			Channels window
		@description:
			Triggered when someone sets a +h flag on a user of the active channel.
		@seealso:
			[event:ondehalfop]OnDeHalfOp[/event]
	*/
	EVENT("OnHalfOp","$0 = source nick\n$1 = source username\n$2 = source hostname\n$3 = halfopped nick"),
	/*
	 	@doc: ondehalfop
		@type:
			event
		@title:
			OnDeHalfOp
		@short:
			Someone has give some user a -h flag
		@parameters:
			$0 = source nick
			$1 = source username
			$2 = source host
			$3 = dehalfopped nick
		@window:
			Channels window
		@description:
			Triggered when someone sets a -h (DeHalfOp) flag on a user in the active channel.
		@seealso:
			[event:onhalfop]OnHalfOp[/event]
	*/
	EVENT("OnDeHalfOp","$0 = source nick\n$1 = source username\n$2 = source hostname\n$3 = dehalfopped nick"),
	/*
	 	@doc: oninviteexception
		@type:
			event
		@title:
			OnInviteException
		@short:
			A +I mode has been set
		@parameters:
			$0 = source nick
			$1 = source username
			$2 = source host
			$3 = target mask
		@window:
			Channels window
		@description:
			Someone has set mode +I on a channel
		@seealso:
			[event:oninviteexceptionremove]OnInviteExceptionRemove[/event]
	*/
	EVENT("OnInviteException","$0 = source nick\n$1 = source username\n$2 = source hostname\n$3 = target mask"),
	/*
	 	@doc: oninviteexceptionremove
		@type:
			event
		@title:
			OnInviteExceptionRemove
		@short:
			Move -I has been set
		@parameters:
			$0 = source nick
			$1 = source username
			$2 = source host
			$3 = target mask
		@window:
			Channels window
		@description:
			Someone has set mode -I on a channel
		@seealso:
			[event:oninviteexceptionremove]OnInviteExceptionRemove[/event]
	*/
	EVENT("OnInviteExceptionRemove","$0 = source nick\n$1 = source username\n$2 = source hostname\n$3 = target mask"),
	/*
	 	@doc: onmeinviteexception
		@type:
			event
		@title:
			OnMeInviteException
		@short:
			A +I mode has been set
		@parameters:
			$0 = source nick
			$1 = source username
			$2 = source host
		@window:
			Channels window
		@description:
			Someone has set mode +I on a channel with a target mask that matches
			the local user mask
		@seealso:
			[event:onmeinviteexceptionremove]OnMeInviteExceptionRemove[/event]
	*/
	EVENT("OnMeInviteException","$0 = source nick\n$1 = source username\n$2 = source hostname"),
	/*
	 	@doc: onmeinviteexceptionremove
		@type:
			event
		@title:
			OnMeInviteExceptionRemove
		@short:
			Move -I has been set
		@parameters:
			$0 = source nick
			$1 = source username
			$2 = source host
		@window:
			Channels window
		@description:
			Someone has set mode -I on a channel with a target mask that matches
			the local user mask
		@seealso:
			[event:onmeinviteexceptionremove]OnMeInviteExceptionRemove[/event]
	*/
	EVENT("OnMeInviteExceptionRemove","$0 = source nick\n$1 = source username\n$2 = source hostname"),
	/*
	 	@doc: onaction
		@type:
			event
		@title:
			OnAction
		@short:
			Someone issued a CTTP ACTION
		@parameters:
			$0 = source nick
			$1 = source username
			$2 = source host
		@window:
			Channels and query windows
		@description:
			Someone issued a CTTP ACTION (usually using a /me command with his client) 
			on this channel/query.
	*/
			
	EVENT("OnAction", "$0 = source nick\n$1 = source username\n$2 = source hostname"),	
	/*
		@doc: onctcpflood
		@type:
			event
		@title:
			OnCTCPFlood
		@short:
			Someone tried to flood you with CTCP requests
		@parameters:
			$0 = source nick
			$1 = source username
			$2 = source hostname
			$3 = target
			$4 = ctcp type
			$5- = ctcp parameters
		@window:
			Console window
		@description:
			On IRC there is a kind of attack called 'CTCP Flood'. It consist of a lot of CTCP request in a short period of time, trying to made your client reply to them and made the server kick you. This event is triggered when one of such attacks is detected.
		@seealso:
			[event:onctcprequest]OnCtcpRequest[/event]
			[event:onctcpreply]OnCtcpReply[/event]
	*/	
	EVENT("OnCTCPFlood", "$0 = source nick\n$1 = source username\n$2 = source hostname\n$3 = target\n$4 = ctcp type\n$5- = ctcp parameters"),
	/*
	 	@doc: onchannelnotice
		@type:
			event
		@title:
			OnChannelNotice
		@short:
			A NOTICE have been issued to a channel
		@parameters:
			$0 = source nick
			$1 = message
		@window:
			Channel window
		@description:
			Someone has issued a NOTICE with a channel as his target
		@seealso:
			[event:onservernotice]OnServerNotice[/event]
	*/
	EVENT("OnChannelNotice", "$0 = source nick\n$1 = message"),
    /*
        @doc: onservernotice
        @type:
            event
        @title:
            OnServerNotice
        @short:
            A server NOTICE have been issued.
        @parameters:
            $0 = source nick
            $1 = message
        @window:
            Console window
        @description:
            Someone has issued a server NOTICE.
        @seealso:
            [event:onchannelnotice]OnChannelNotice[/event]
    */
	EVENT("OnServerNotice", "$0 = source nick\n$1 = message"),
    /*
	 	@doc: ondccchatconnected
		@type:
			event
		@title:
			OnDCCChatConnected
		@short:
	 		A DCC chat connection have been initiated
		@parameters:
			$0 = remote ip
			$1 = remote port
			$2 = local port
			$3 = target nick
			$4 = target username
			$5 = target hostname
		@window:
			DCC chat window
		@description:
			Triggered when a DCC chat connection has been succesfully initiated
		@seealso:
			[event:ondccsendconnected]OnDCCSendConnected[/event]
			[event:ondccgetconnected]OnDCCGetConnected[/event]
			[event:ondccvoiceconnected]OnDCCVoiceConnected[/event]	 
	*/
	EVENT("OnDCCChatConnected", "$0 = remote port\n$1 = local port\n$2 = target nick\n$3 = target username\n$4 = target host"),
	/* 
	 	@doc: ondccchatmessage
		@type:
			event
		@title:
			OnDCCChatMessage
		@short:
			A message have been received from a DCC Chat
		@parameters:
			$0 = remote port
			$1 = target nick
			$2 = target username
			$3 = target hostname
			$4 = message
		@window:
			DCC Chat window
		@description:
			Triggered when a text message have been received on a DCC chat.
		@seealso:
			[event:onquerymessage]OnQueryMessage[/event]
			[event:onchannelmessage]OnChannelMessage[/event]
		*/
	EVENT("OnDCCChatMessage", "$0 = remote port\n$1 = target nick\n$2 = target username\n$3 = target hostname\n$4 = message"),
	/*
	 	@doc: ondccsendtransfercomplete
		@type:
			event
		@title:
			OnDCCSendTransferComplete
		@short:
			A DCC SEND transfer has ended succesfully
		@parameters:
			$0 = remote ip
			$1 = remote port
			$2 = filename
			$3 = source nick
			$4 = source username
			$5 = source hostname
		@window:
			DCC Send window
		@description:
			A DCC Send transfer has ended succesfully.
		@seealso:
			[event:ondccgettransfercomplete]OnDccGetTransferComplete[/event]	 
	*/
	EVENT("OnDCCSendTransferComplete", "$0 = remote port\n$1 = filename\n$2 = target nick\n$3 = target username\n$4 = target hostname"),
	/*
	 	@doc: ondccgettransfercomplete
		@type:
			event
		@title:
			OnDCCGetTransferComplete
		@short:
			A DCC GET transfer has ended succesfully
		@parameters:
			$0 = remote port
			$1 = filename
			$2 = source nick
			$3 = source username
			$4 = source hostname
		@window:
			DCC Get window
		@description:
			A DCC GET transfer has ended succesfully and the file is on the disk
		@seealso:
			[event:ondccsendtransfercomplete]OnDCCSendTransferComplete[/event]	

	*/
	EVENT("OnDCCGetTransferComplete", "$0 = remote port\n$1 = filename\n$2 = target nick\n$3 = target username\n$4 = target hostname"),
    /*
        @doc: ondccsendconnected
        @type:
            event
        @title:
            OnDCCSendConnected
        @short:
            A DCC Send transfer has just been initiated
        @parameters:
            $0 = remote port
            $1 = filename
            $2 = target nick
            $3 = target username
            $4 = target hostname
        @window:
            DCC Send window
        @description:
            A DCC Send transfer has just been initiated the connection 
        @seealso:
            [event:ondccgetconnected]OnDCCGetConnected[/event]
 
    */
	EVENT("OnDCCSendConnected", "$0 = remote port\n$1 = filename\n$2 = target nick\n$3 = target username\n$4 = target hostname"),
    /*
        @doc: ondccgetconnected
        @type:
            event
        @title:
            OnDCCGetConnected
        @short:
            A DCC Get transfer has just been initiated
        @parameters:
            $0 = remote port
            $1 = filename
            $2 = source nick
            $3 = source username
            $4 = source hostname
        @window:
            DCC Get window
        @description:
            A DCC Get transfer has just been initiated the connection
        @seealso:
            [event:ondccsendconnected]OnDCCSendConnected[/event]
 
    */
	EVENT("OnDCCGetConnected", "$0 = remote port\n$1 = filename\n$2 = source nick\n$3 = source username\n$4 = source hostname"),
    /*
        @doc: ondccchatterminated
        @type:
            event
        @title:
            OnDCCchatTerminated
        @short:
            A DCC Chat connection has ended
        @parameters:
			$0 = remote port
            $1 = target nick
            $2 = target username
            $3 = target hostname
			$4 = termination message
        @window:
            DCC Chat window
        @description:
            A DCC Chat has ended. It could have ended by an error or be a normal termination (one of the two hosts closed the connection)
    */
	EVENT("OnDCCChatTerminated", "$0 = remote port\n$1 source nick\n$2 = source username\n$3 = source hostname\n$4 = termination message"),
	/*
		@doc: onaccelkeypressed
		@type:
			event
		@title:
			OnAccelKeyPressed
		@short:
			An accelerator key was pressed
		@parameters:
			$0 = keys
		@window:
			Active window
		@description:
			An accelerator key has been pressed by the user.[br]
			The key sequence that the user has pressed is passed in $0 as a 
			human readable string.[br]
			The keys that actually work with KVIrc are:[br]
			F1 to F12, F1 to F12 with SHIFT key pressed, 0 to 9 with CTRL Key pressed.[br]
			This means that you can set up 34 "quick" aliases...that's really more than
			an user can remember.[br]
			If you need more keys , mail me , I'll see what I can do :)[br]
    */
	EVENT("OnAccelKeyPressed", "$0 = keys"),
	/*
		@doc: onhighlight
		@type:
			event
		@title:
			OnHighlight
		@short:
			A channel or query message is going to be highlighted
		@parameters:
			$0 = talker nick
			$1 = target username
			$2 = target hostname
			$3 = message
			$4 = highlight word
		@window:
			Channel or query window
		@description:
			A channel or query message is going to be highlighted.[br]
			The highlight may be either caused by a "highlight word" or by your own nickname
			(if that option was specified).[br]
			If you call [cmd]halt[/cmd] in this event the message output will be stopped at all
			(thus KVIrc assumes that YOU have provided the necessary output and feedback to the user).[br]
    */
	EVENT("OnHighlight", "$0 = source nick\n$1 source username\n$2 = source hostname\n$3 = message\n$4 = highlight word"),
	/*
		@doc: onwindowactivated
		@type:
			event
		@title:
			OnWindowActivated
		@short:
			A window has just been activated
		@parameters:
			none
		@window:
			Any window
		@description:
			This event is triggered when a KVIrc MDI window gets activated.[br]
			This might be a good place to update your toolbar button status (for example).[br]
	*/
	EVENT("OnWindowActivated",""),
	/*
		@doc: onnotifyonline
		@type:
			event
		@title:
			OnNotifyOnLine
		@short:
			An user in the notify list is on-line
		@parameters:
			$0 = nickname
		@window:
			Console or active window of the irc context
		@description:
			This event is triggered when one of the users in your [doc:notify_list]notify[/doc] lists
			is detected to be ON-Line.[br]
			The event is triggered in the console window or the active window of the irc context depending
			on the user options.[br]
			(Note: If you call [cmd]halt[/cmd] in this event , you will stop the "Nick is on irc" output,
			but this is rather a bad idea since KVIrc gives a lot of additional informations to the
			user with some notify-methods).[br]
			This is a good place to play a sound or attract the user attention in some other way.[br]
			
	*/
	EVENT("OnNotifyOnLine","$0 = nickname"),
	/*
		@doc: onnotifyoffline
		@type:
			event
		@title:
			OnNotifyOffLine
		@short:
			An user in the notify list went off-line
		@parameters:
			$0 = nickname
		@window:
			Console or active window of the irc context
		@description:
			This event is triggered when one of the users in your [doc:notify_list]notify[/doc] lists
			has just left IRC.[br]
			The event is triggered in the console window or the active window of the irc context depending
			on the user options.[br]
			(Note: If you call [cmd]halt[/cmd] in this event , you will stop the "Nick has left irc" output,
			but this is rather a bad idea since KVIrc gives a lot of additional informations to the
			user with some notify-methods).[br]
			This is a good place to play a sound or attract the user attention in some other way.[br]
	*/
	EVENT("OnNotifyOffLine","$0 = nickname")

};

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Clear functions
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void KviEventManager::clear()
{
	int i;
	for(i = 0;i < KVI_NUM_SCRIPT_EVENTS;i++)
	{
		if(m_eventTable[i].handlers)
		{
			delete m_eventTable[i].handlers;
			m_eventTable[i].handlers = 0;
		}
	}
	for(i = 0;i < 1000;i++)
	{
		delete m_rawNumericEventTable[i];
		m_rawNumericEventTable[i] = 0;
	}
}

void KviEventManager::clearScriptHandlers(KviEventHandlerList * l)
{
	KviEventHandlerList dl;
	dl.setAutoDelete(false);
	KviEventHandler * e;
	for(e = l->first();e;e = l->next())
	{
		if(e->type() == KviEventHandler::Script)dl.append(e);
	}
	for(e = dl.first();e;e = dl.next())
	{
		l->removeRef(e);
	}
}

void KviEventManager::clearScriptHandlers()
{
	for(int i=0;i< KVI_NUM_SCRIPT_EVENTS;i++)
	{
		if(m_eventTable[i].handlers)
		{
			clearScriptHandlers(m_eventTable[i].handlers);
			if(m_eventTable[i].handlers->isEmpty())
			{
				delete m_eventTable[i].handlers;
				m_eventTable[i].handlers = 0;
			}
		}
	}
}

void KviEventManager::clearRawScriptHandlers()
{
	for(int i=0;i< 1000;i++)
	{
		if(m_rawNumericEventTable[i])
		{
			clearScriptHandlers(m_rawNumericEventTable[i]);
			if(m_rawNumericEventTable[i]->isEmpty())
			{
				delete m_rawNumericEventTable[i];
				m_rawNumericEventTable[i] = 0;
			}
		}
	}
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Lookup functions
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int KviEventManager::lookupEventByName(const char * n)
{
	for(int i = 0;i < KVI_NUM_SCRIPT_EVENTS;i++)
	{
		if(kvi_strEqualCI(m_eventTable[i].szName,n))return i;
	}
	return -1;
}

KviScriptEventHandler * KviEventManager::lookupScriptEventHandler(int evIdx,const char * hndName)
{
	if(!m_eventTable[evIdx].handlers)return 0;
	for(KviEventHandler * s = m_eventTable[evIdx].handlers->first();s;s = m_eventTable[evIdx].handlers->next())
	{
		if(s->type() == KviEventHandler::Script)
		{
			if(kvi_strEqualCI(((KviScriptEventHandler *)s)->name(),hndName))return (KviScriptEventHandler *)s;
		}
	}
	return 0;
}

KviScriptEventHandler * KviEventManager::lookupRawNumericScriptEventHandler(int evIdx,const char * hndName)
{
	if(!m_rawNumericEventTable[evIdx])return 0;
	for(KviEventHandler * s = m_rawNumericEventTable[evIdx]->first();s;s = m_rawNumericEventTable[evIdx]->next())
	{
		if(s->type() == KviEventHandler::Script)
		{
			if(kvi_strEqualCI(((KviScriptEventHandler *)s)->name(),hndName))return (KviScriptEventHandler *)s;
		}
	}
	return 0;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Code lookup functions
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

const char * KviEventManager::lookupScriptHandlerCode(const char * evName,const char * hndName)
{
	int idx = lookupEventByName(evName);
	if(idx < 0)return 0;
	KviScriptEventHandler * s = lookupScriptEventHandler(idx,hndName);
	if(!s)return 0;
	return s->code();
}

const char * KviEventManager::lookupRawNumericScriptHandlerCode(int idx,const char * hndName)
{
	KviScriptEventHandler * s = lookupRawNumericScriptEventHandler(idx,hndName);
	if(!s)return 0;
	return s->code();
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Registration functions for plain script handlers
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool KviEventManager::registerScriptHandler(const char * evName,const char * hndName,const char * code,bool bEnabled)
{
	int idx = lookupEventByName(evName);
	if(idx < 0)return false;
	KviScriptEventHandler * s = lookupScriptEventHandler(idx,hndName);
	if(!s)
	{
		s = new KviScriptEventHandler();
		if(!m_eventTable[idx].handlers)
		{
			m_eventTable[idx].handlers = new KviEventHandlerList;
			m_eventTable[idx].handlers->setAutoDelete(true);
		}
		m_eventTable[idx].handlers->append(s);
		s->setName(hndName);
	}
	s->setEnabled(bEnabled);
	s->setCode(code);
	return true;
}

bool KviEventManager::unregisterScriptHandler(const char * evName,const char * hndName)
{
	int idx = lookupEventByName(evName);
	if(idx < 0)return false;
	if(!m_eventTable[idx].handlers)return false;
	for(KviEventHandler * s = m_eventTable[idx].handlers->first();s;s = m_eventTable[idx].handlers->next())
	{
		if(s->type() == KviEventHandler::Script)
		{
			if(kvi_strEqualCI(((KviScriptEventHandler *)s)->name(),hndName))
			{
				m_eventTable[idx].handlers->removeRef(s);
				if(m_eventTable[idx].handlers->isEmpty())
				{
					delete m_eventTable[idx].handlers;
					m_eventTable[idx].handlers = 0;
					return true;
				}
			}
		}
	}
	return false;
}

bool KviEventManager::enableScriptHandler(const char * evName,const char * hndName,bool bEnable)
{
	int idx = lookupEventByName(evName);
	if(idx < 0)return false;
	KviScriptEventHandler * s = lookupScriptEventHandler(idx,hndName);
	if(s)
	{
		s->setEnabled(bEnable);
		return true;
	}
	return false;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Registration functions for raw numeric script handlers
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool KviEventManager::registerRawNumericScriptHandler(int idx,const char * hndName,const char * code,bool bEnabled)
{
	KviScriptEventHandler * s = lookupRawNumericScriptEventHandler(idx,hndName);
	if(!s)
	{
		s = new KviScriptEventHandler();
		if(!m_rawNumericEventTable[idx])
		{
			m_rawNumericEventTable[idx] = new KviEventHandlerList;
			m_rawNumericEventTable[idx]->setAutoDelete(true);
		}
		m_rawNumericEventTable[idx]->append(s);
		s->setName(hndName);
	}
	s->setEnabled(bEnabled);
	s->setCode(code);
	return true;
}

bool KviEventManager::unregisterRawNumericScriptHandler(int idx,const char * hndName)
{
	if(!m_rawNumericEventTable[idx])return false;
	for(KviEventHandler * s = m_rawNumericEventTable[idx]->first();s;s = m_rawNumericEventTable[idx]->next())
	{
		if(s->type() == KviEventHandler::Script)
		{
			if(kvi_strEqualCI(((KviScriptEventHandler *)s)->name(),hndName))
			{
				m_rawNumericEventTable[idx]->removeRef(s);
				if(m_rawNumericEventTable[idx]->isEmpty())
				{
					delete m_rawNumericEventTable[idx];
					m_rawNumericEventTable[idx] = 0;
					return true;
				}
			}
		}
	}
	return false;
}

bool KviEventManager::enableRawNumericScriptHandler(int idx,const char * hndName,bool bEnable)
{
	KviScriptEventHandler * s = lookupRawNumericScriptEventHandler(idx,hndName);
	if(s)
	{
		s->setEnabled(bEnable);
		return true;
	}
	return false;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Registration functions for plain module handlers
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void KviEventManager::registerModuleHandler(int evIdx,KviModuleEventHandler * h)
{
	unregisterModuleHandler(evIdx,h->module());
	if(!m_eventTable[evIdx].handlers)
	{
		m_eventTable[evIdx].handlers = new KviEventHandlerList;
		m_eventTable[evIdx].handlers->setAutoDelete(true);
	}
	m_eventTable[evIdx].handlers->append(h);
}

void KviEventManager::unregisterModuleHandler(int evIdx,KviModule * m)
{
	if(!m_eventTable[evIdx].handlers)return;
	for(KviEventHandler * e = m_eventTable[evIdx].handlers->first();e;e = m_eventTable[evIdx].handlers->next())
	{
		if(e->type() == KviEventHandler::Module)
		{
			if(((KviModuleEventHandler *)e)->module() == m)
			{
				m_eventTable[evIdx].handlers->removeRef(e);
				if(m_eventTable[evIdx].handlers->isEmpty())
				{
					delete m_eventTable[evIdx].handlers;
					m_eventTable[evIdx].handlers = 0;
				}
				return;
			}
		}
	}
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Registration functions for raw numeric module handlers
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void KviEventManager::registerRawNumericModuleHandler(int evIdx,KviModuleEventHandler * h)
{
	unregisterRawNumericModuleHandler(evIdx,h->module());
	if(!m_rawNumericEventTable[evIdx])
	{
		m_rawNumericEventTable[evIdx] = new KviEventHandlerList;
		m_rawNumericEventTable[evIdx]->setAutoDelete(true);
	}
	m_rawNumericEventTable[evIdx]->append(h);
}

void KviEventManager::unregisterRawNumericModuleHandler(int evIdx,KviModule * m)
{
	if(!m_rawNumericEventTable[evIdx])return;
	for(KviEventHandler * e = m_rawNumericEventTable[evIdx]->first();e;e = m_rawNumericEventTable[evIdx]->next())
	{
		if(e->type() == KviEventHandler::Module)
		{
			if(((KviModuleEventHandler *)e)->module() == m)
			{
				m_rawNumericEventTable[evIdx]->removeRef(e);
				if(m_rawNumericEventTable[evIdx]->isEmpty())
				{
					delete m_rawNumericEventTable[evIdx];
					m_rawNumericEventTable[evIdx] = 0;
				}
				return;
			}
		}
	}
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Registration functions for all module handlers
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void KviEventManager::unregisterAllModuleHandlers(KviModule * m)
{
	int i;
	for(i=0;i<KVI_NUM_SCRIPT_EVENTS;i++)unregisterModuleHandler(i,m);
	for(i=0;i<1000;i++)unregisterRawNumericModuleHandler(i,m);
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Load & save
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void KviEventManager::saveRaw(const char * filename)
{
	KviConfig cfg(filename);
	cfg.clear();
	int i;

	for(i=0;i<1000;i++)
	{
		if(m_rawNumericEventTable[i])
		{
			KviStr tmp(KviStr::Format,"RAW%d",i);
			cfg.setGroup(tmp.ptr());
			int index = 0;
			for(KviEventHandler* s = m_rawNumericEventTable[i]->first();s; s = m_rawNumericEventTable[i]->next())
			{
				if(s->type() == KviEventHandler::Script)
				{
					KviStr tmp(KviStr::Format,"Name%d",index);
					cfg.writeEntry(tmp.ptr(),((KviScriptEventHandler *)s)->name());
					tmp.sprintf("Buffer%d",index);
					cfg.writeEntry(tmp.ptr(),((KviScriptEventHandler *)s)->code());
					tmp.sprintf("Enabled%d",index);
					cfg.writeEntry(tmp.ptr(),((KviScriptEventHandler *)s)->enabled());
					index++;
				}
			}
			cfg.writeEntry("NHandlers",index);
		}
	}
}

void KviEventManager::loadRaw(const char * filename)
{
	KviConfig cfg(filename);
	clearRawScriptHandlers();
	int i;


	for(i=0;i<1000;i++)
	{
		KviStr tmp(KviStr::Format,"RAW%d",i);
		if(cfg.hasGroup(tmp.ptr()))
		{
			cfg.setGroup(tmp.ptr());
			unsigned int nHandlers = cfg.readUIntEntry("NHandlers",0);
			if(nHandlers)
			{
				m_rawNumericEventTable[i] = new KviEventHandlerList;
				m_rawNumericEventTable[i]->setAutoDelete(true);
				for(unsigned int index = 0;index < nHandlers;index++)
				{
					KviStr tmp(KviStr::Format,"Name%u",index);
					KviScriptEventHandler *s = new KviScriptEventHandler();
					s->setName(cfg.readEntry(tmp.ptr(),"unnamed"));
					tmp.sprintf("Buffer%u",index);
					s->setCode(cfg.readEntry(tmp.ptr(),""));
					tmp.sprintf("Enabled%u",index);
					s->setEnabled(cfg.readBoolEntry(tmp.ptr(),false));
					m_rawNumericEventTable[i]->append(s);
				}
			}
		}
	}
}

void KviEventManager::save(const char * filename)
{
	KviConfig cfg(filename);
	cfg.clear();
	int i;

	for(i=0;i<KVI_NUM_SCRIPT_EVENTS;i++)
	{
		if(m_eventTable[i].handlers)
		{
			cfg.setGroup(m_eventTable[i].szName);
			int index = 0;
			for(KviEventHandler* s = m_eventTable[i].handlers->first();s; s = m_eventTable[i].handlers->next())
			{
				if(s->type() == KviEventHandler::Script)
				{
					KviStr tmp(KviStr::Format,"Name%d",index);
					cfg.writeEntry(tmp.ptr(),((KviScriptEventHandler *)s)->name());
					tmp.sprintf("Buffer%d",index);
					cfg.writeEntry(tmp.ptr(),((KviScriptEventHandler *)s)->code());
					tmp.sprintf("Enabled%d",index);
					cfg.writeEntry(tmp.ptr(),((KviScriptEventHandler *)s)->enabled());
					index++;
				}
			}
			cfg.writeEntry("NHandlers",index);
		}
	}
}


void KviEventManager::load(const char * filename)
{
	KviConfig cfg(filename);
	clearScriptHandlers();
	int i;
	for(i=0;i<KVI_NUM_SCRIPT_EVENTS;i++)
	{
		//debug("Loading event %d",i);
		if(cfg.hasGroup(m_eventTable[i].szName))
		{
			//debug("Is there");
			cfg.setGroup(m_eventTable[i].szName);
			unsigned int nHandlers = cfg.readUIntEntry("NHandlers",0);
			if(nHandlers)
			{
				//debug("nHandlers is %d",nHandlers);
				m_eventTable[i].handlers = new KviEventHandlerList;
				m_eventTable[i].handlers->setAutoDelete(true);
				for(unsigned int index = 0;index < nHandlers;index++)
				{
					//debug("Loading it");
					KviStr tmp(KviStr::Format,"Name%u",index);
					KviScriptEventHandler *s = new KviScriptEventHandler();
					s->setName(cfg.readEntry(tmp.ptr(),"unnamed"));
					tmp.sprintf("Buffer%u",index);
					s->setCode(cfg.readEntry(tmp.ptr(),""));
					tmp.sprintf("Enabled%u",index);
					s->setEnabled(cfg.readBoolEntry(tmp.ptr(),false));
					m_eventTable[i].handlers->append(s);
				}
			}
		}
	}
}
