//
//   File : kvi_alias.cpp
//   Creation date : Sun Sep 1 2000 15:02:21 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#define __KVIRC__

#define _KVI_DEBUG_CHECK_RANGE_

#include "kvi_debug.h"
#include "kvi_alias.h"
#include "kvi_config.h"


/*
	@doc: aliases
	@type:
		language
	@keyterms:
		aliases,alias functions,alias
	@title:
		Aliases
	@short:
		Aliases: user definable command sequences
	@body:
		An alias is an user definable command sequence that can be
		triggered by using the alias name as it was a single command or function.[br]
		It can be used to rename the builtin kvirc commands or functions , to
		automatize complex tasks or as structured programming mean.[br]
		Aliases can be created or destroyed by using the scriptcenter (graphic interface)
		or even from the commandline (or script) by using the [cmd]alias[/cmd] command.[br]
		A couple of examples will make the things clear:[br]
		[br]
		[cmd]join[/cmd] is a really commonly used command. It might be a good idea
		to rename it to simply "j" .. just to type it faster.[br]
		Nothing easier in KVirc: just try this commandline:[br]
		[example]
			[cmd]alias[/cmd](j){ [cmd]join[/cmd] $0-; };
		[/example]
		This will create the alias "j". From this moment you can use /j as it was a normal command.[br]
		[example]
			j #kvirc
		[/example]
		will in fact execute "join $0-" where $0- is the list of parameters passed to /j (in this case
		it contains a single parameter: #kvirc).[br]
		[br]
		A common task in channel management is the kick & ban action.[br]
		You first ban an user from the channel and then eventually kick him (obviously assuming that
		he is actually on the channel).[br]
		This involves using two commands: [cmd]ban[/cmd] and then [cmd]kick[/cmd].[br]
		It could be a nice idea to have a single "kb" command to perform this action.[br]
		Well...easy:[br]
		[example]
			[cmd]alias[/cmd](kb){ [cmd]ban[/cmd] $0; [cmd]kick[/cmd] $0-; };
		[/example]
		This adds the "kb" alias: it can be called as a normal command:[br]
		[example]
			kb spammer You're not welcome here!
		[/example]
		This will first execute "ban spammer" and then "kick spammer You're not welcome here".[br]
		Our kb is a really simple example... it doesn't check for the validity of the parameters:
		the server will warn us if the parameters passed to kb were empty.[br]
		The alias can be modified at any time by re-using the [cmd]alias[/cmd] command.[br]
		Let's make our "kb" a bit more intelligent and add a check for the parameters.[br]
		TIP: It is a good idea to write the following examples in a text file and then use /[cmd]parse[/cmd] <filename>
		to execute it.[br]
		[example]
			[cmd]alias[/cmd](kb)
			{
				[cmd]if[/cmd]("$0" == "")
				{
					[cmd]echo[/cmd] "Usage: /kb <nickname> <kick reason>"
					[cmd]halt[/cmd]
				}
				[cmd]ban[/cmd] $0

				%reason = $1-

				[cmd]if[/cmd]("%reason" == "")%reason = "You're not welcome here!"
				[cmd]kick[/cmd] $0 %reason
			}
		[/example]
		The example above will first check the validity of the <nickname> passed to kb: if no nickname
		was passed , it will warn the user and stop.[br]
		The next step will be the "ban <nickname>" call. Another enchancement is the "default reason":
		we first assign the remaining parameters ($1- means "from $1 to the end") to a temporary variable,
		if the variable is empty , a default kick reason is assigned.[br]
		Finally the "kick <nickname> <reason>" will be executed.[br]
		[br]
		Aliases can be used as a mean for structured programming.[br]
		In large scripts you will SURELY have "common tasks" to perform (like having specially colored output
		or calculating a value from a set of other values)...[br]
		Aliases are the way of writing the common tasks: they are equivalent to the "procedures" or "functions"
		in many high-level programming languages.[br]
		The alias as a procedure (subroutine or sub-task) has been shown in the "kb" example above: it might be commonly called
		from complexier scripts or other aliases in case that a kick & ban action is needed.[br]
		The aliases can be used also as functions.[br]
		Assume that you need really often to calculate the sum of three numbers: a function-alias is the way.[br]
		[example]
			[cmd]alias[/cmd](sum3){ [cmd]setreturn[/cmd] $($0 + $1 + $2); };
		[/example]
		This will add the alias "sum3" and make it available both as a command and a function.[br]
		The "setreturn" command sets the return value of a sequence of commands (an alias is a sequence of commands...remember ?).[br]
		So setreturn $($0 + $1 + $2); will set the return value of the alias to the value computed by $($0 + $1 + $2)
		that actually is the sum of the first three parameters passed.[br]
		You will then use it in the following way:[br]
		[example]
			...
			%myfirstsum = $sum3(%somevalue,%someothervalue,4)
			%anothersum = $sum3(12,%somevalue,%anothervalue)
			...
		[/example]
		This example is again really simple , but you might have complexier function-aliases.[br]
		The function-aliases are also normal aliases.... you can use it as a command:[br]
		[example]
			/sum3 1 2 3
		[/example]
		Is a perfectly valid call.... it's just that it will have no visible results (just because
		a command call implies ignoring the return value.[br]
		In fact there is no difference al all between function-aliases and normal-aliases:
		the caller makes the difference: by calling an alias as a command the return value just disappears in hyperspace,
		by calling an alias as a function , the return value is propagated (and in fact "used").[br]
		(There are some "nice" exceptions to this rule...but you don't need to care about it, for now).[br]
		If [cmd]setreturn[/cmd] is not called inside an alias body , the return value will be
		just an empty string.[br] (Note that setreturn can be called multiple times inside an alias:
		it sets the "return buffer" that is initially empty).[br]
		You might also write aliases that can work both as commands and functions.. they could
		perform some tasks and return a secondary result (that might be often ignored).[br]
		To remove an alias use again the [cmd]alias[/cmd] command with an empty body:[br]
		[example]
			[cmd]alias[/cmd](somealias){}
		[/example]
		This will remove the alias "somealias".[br]
		Please note that you CAN'T remove an alias from inside itself; the following form
		is forbidden and will throw an execution-time error:[br]
		[example]
			[cmd]alias[/cmd](test){ [cmd]alias[/cmd](test){}; }
		[/example]
		In fact , any self-modification of an alias is forbidden.[br]
*/

KviAlias::KviAlias(const char * name)
{
	m_szName = name;
	m_uLock = 0;
}

KviAlias::~KviAlias()
{
}

void KviAlias::load(const char * prefix,KviConfig * cfg)
{
	KviStr tmp(KviStr::Format,"%s_Buffer",prefix);
	m_szDataBuffer = cfg->readEntry(tmp.ptr(),"");
}

void KviAlias::save(const char * prefix,KviConfig * cfg)
{
	KviStr tmp(KviStr::Format,"%s_Buffer",prefix);
	cfg->writeEntry(tmp.ptr(),m_szDataBuffer.ptr());
}
