//
//   File : kvi_window.cpp
//   Creation date : Tue Jul 6 1999 14:52:11 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
#define __KVIRC__
#define KVI_WINDOW_MIN_WIDTH 100
#define KVI_WINDOW_MIN_HEIGHT 100

#define _KVI_WINDOW_CPP_

#define _KVI_DEBUG_CHECK_RANGE_

#include "kvi_debug.h"
#include "kvi_app.h"
#include "kvi_window.h"
#include "kvi_frame.h"
#include "kvi_taskbar.h"
#include "kvi_iconmanager.h"
#include "kvi_mdichild.h"
#include "kvi_locale.h"
#include "kvi_ircview.h"
#include "kvi_out.h"
#include "kvi_malloc.h"
#include "kvi_input.h"
#include "kvi_fileutils.h"
#include "kvi_options.h"


#include <qpixmap.h>
#include <qpopupmenu.h>
#include <qcursor.h>
#include <qtimer.h>
#include <qsplitter.h>
#include <qmetaobject.h>
#include <qdatetime.h>
#include <qobjcoll.h>
#include <qvariant.h>
#include <qaccel.h>

#ifdef COMPILE_CRYPT_SUPPORT
	#include <qtooltip.h>
	#include <qtoolbutton.h>
	#include "kvi_crypt.h"
	#include "kvi_cryptcontroller.h"
#endif

unsigned int g_uUniqueWindowId = 1;

// FIXME: #warning "Mouse wheel suppport"

KviWindow::KviWindow(int type,KviFrame * lpFrm,const char * name,KviConsole * lpConsole)
: QWidget(0,name)
{
	m_szId.setNum(g_uUniqueWindowId);
	g_uUniqueWindowId++;

	g_pApp->registerWindow(this);

	m_pFrm                  = lpFrm;
	m_type                  = type;
	m_pFocusHandler         = 0;
	m_pContextPopup         = new QPopupMenu();
	m_pIrcView              = 0;
	m_pInput                = 0;
	m_pSplitter             = 0;
	m_pButtonBox            = 0;
	m_pConsole              = lpConsole;
	m_pLastPressedChild     = 0;
	m_pWheelEventHandler    = 0;
	m_bNewLastPressedChild  = true;

#ifdef COMPILE_CRYPT_SUPPORT
	m_pCryptControllerButton = 0;
	m_pCryptController = 0;
	m_pCryptSessionInfo = 0;
#endif

	m_pAccel = 0;

	m_pTaskBarItem = 0;
	createTaskBarItem();
	setMinimumSize(KVI_WINDOW_MIN_WIDTH,KVI_WINDOW_MIN_HEIGHT);
	setBackgroundMode(NoBackground);
	setFocusPolicy(StrongFocus);

//	applyOptions();
//	updateCaption();
}

KviWindow::~KviWindow()
{
	destroyTaskBarItem();
	g_pApp->unregisterWindow(this);
	delete m_pContextPopup;
#ifdef COMPILE_CRYPT_SUPPORT
	if(m_pCryptSessionInfo)KviCryptController::destroyCryptSessionInfo(&m_pCryptSessionInfo);
#endif
}

void KviWindow::createTaskBarItem()
{
	if(m_pTaskBarItem)return;
//	if(!m_pFrm->m_pTaskBar)return;
	m_pTaskBarItem = m_pFrm->m_pTaskBar->addItem(this);
}

void KviWindow::destroyTaskBarItem()
{
	if(!m_pTaskBarItem)return;
//	if(!m_pFrm->m_pTaskBar)return;
	m_pFrm->m_pTaskBar->removeItem(m_pTaskBarItem);
//	m_pTaskBarItem = 0; // actually the taskBarItem destructor sets it
}

QToolButton * KviWindow::createToolButton(QWidget * par,const char * nam,int pixon,int pixoff,const char * tooltip,bool bOn)
{
	QToolButton * b = new QToolButton(par,nam);
	b->setToggleButton(true);
	b->setUsesBigPixmap(false);
#if QT_VERSION >= 300
	QIconSet is1;
	is1.setPixmap(*(g_pIconManager->getSmallIcon(pixon)),QIconSet::Small,QIconSet::Normal,QIconSet::On);
	is1.setPixmap(*(g_pIconManager->getSmallIcon(pixoff)),QIconSet::Small,QIconSet::Normal,QIconSet::Off);
	b->setIconSet(is1);
#else
	b->setOnIconSet(*(g_pIconManager->getSmallIcon(pixon)));
	b->setOffIconSet(*(g_pIconManager->getSmallIcon(pixoff)));
#endif
	QToolTip::add(b,__c2q(tooltip));
	b->setOn(bOn);
	return b;
}

// This is always defined...
void KviWindow::createCryptControllerButton(QWidget * par)
{
#ifdef COMPILE_CRYPT_SUPPORT
	m_pCryptControllerButton = createToolButton(par,"crypt_controller_button",KVI_SMALLICON_UNLOCKEDOFF,KVI_SMALLICON_UNLOCKED,__tr("Toggle crypt controller"),false);
	connect(m_pCryptControllerButton,SIGNAL(clicked()),this,SLOT(toggleCryptController()));
#endif // COMPILE_CRYPT_SUPPORT
}

const char * KviWindow::lastLineOfText()
{
	if(m_pIrcView)return m_pIrcView->lastLineOfText();
	return 0;
}

// The following three have to be here even if the crypt support is disabled...moc does not support conditional compilations
void KviWindow::toggleCryptController()
{
#ifdef COMPILE_CRYPT_SUPPORT
	if(m_pCryptController)
	{
		delete m_pCryptController;
		m_pCryptController = 0;
		if(!m_pCryptControllerButton)return;
		if(m_pCryptControllerButton->isOn())m_pCryptControllerButton->setOn(false);
	} else {
		if(m_pSplitter && m_pInput)
		{
			m_pCryptController = new KviCryptController(m_pSplitter,"crypt_controller",this,m_pCryptSessionInfo);
			connect(m_pCryptController,SIGNAL(done()),this,SLOT(cryptControllerFinished()));
			setFocusHandlerNoClass(m_pInput,m_pCryptController,"QLineEdit"); //link it!
			m_pCryptController->show();
			if(!m_pCryptControllerButton)return;
			if(!(m_pCryptControllerButton->isOn()))m_pCryptControllerButton->setOn(true);
		}
	}
#endif // COMPILE_CRYPT_SUPPORT
}

void KviWindow::cryptControllerFinished()
{
#ifdef COMPILE_CRYPT_SUPPORT
	if(m_pCryptSessionInfo)KviCryptController::destroyCryptSessionInfo(&m_pCryptSessionInfo);
	m_pCryptSessionInfo = m_pCryptController->getNewSessionInfo();
	if(m_pCryptSessionInfo)
	{
		connect(m_pCryptSessionInfo->pEngine,SIGNAL(destroyed()),this,SLOT(cryptSessionInfoDestroyed()));
	}
	if(m_pCryptControllerButton)
	{
#if QT_VERSION >= 300
		QIconSet is;
		is.setPixmap(*(g_pIconManager->getSmallIcon(m_pCryptSessionInfo ? KVI_SMALLICON_LOCKEDOFF : KVI_SMALLICON_UNLOCKEDOFF)),QIconSet::Small,QIconSet::Normal,QIconSet::Off);
		is.setPixmap(*(g_pIconManager->getSmallIcon(m_pCryptSessionInfo ? KVI_SMALLICON_LOCKED : KVI_SMALLICON_UNLOCKED)),QIconSet::Small,QIconSet::Normal,QIconSet::On);
		m_pCryptControllerButton->setIconSet(is);
#else
		m_pCryptControllerButton->setOnIconSet(
			*(g_pIconManager->getSmallIcon(m_pCryptSessionInfo ? KVI_SMALLICON_LOCKEDOFF : KVI_SMALLICON_UNLOCKEDOFF)));
		m_pCryptControllerButton->setOffIconSet(
			*(g_pIconManager->getSmallIcon(m_pCryptSessionInfo ? KVI_SMALLICON_LOCKED : KVI_SMALLICON_UNLOCKED)));
#endif
		if(m_pCryptControllerButton->isOn())m_pCryptControllerButton->setOn(false);
	}
	delete m_pCryptController;
	m_pCryptController = 0;
#endif
}

void KviWindow::cryptSessionInfoDestroyed()
{
#ifdef COMPILE_CRYPT_SUPPORT
	output(KVI_OUT_SYSTEMERROR,__tr("Ops...I've accidentally lost the crypting engine..."));
	m_pCryptSessionInfo->pEngine = 0;
	delete m_pCryptSessionInfo;
	m_pCryptSessionInfo = 0;
#endif
}

const char * KviWindow::m_typeTable[KVI_WINDOW_NUM_TYPES]=
{
	"console",
	"channel",
	"query",
	"help",
	"terminal",
	"editor",
	"dccchat",
	"dccsend",
	"socketspy",
	"links",
	"icq",
	"gnutella",
	"dirbrowser",
	"dcccanvas",
	"dccvoice",
	"list",
	"offer",
    "logview"
};

void KviWindow::setProgress(int progress)
{
	m_pTaskBarItem->setProgress(progress);
}

void KviWindow::listWindowTypes()
{
	outputNoFmt(KVI_OUT_SYSTEMMESSAGE,__tr("List of window types available in this release of KVIrc:"));
	for(int i=0;i< KVI_WINDOW_NUM_TYPES;i++)outputNoFmt(KVI_OUT_SYSTEMMESSAGE,m_typeTable[i]);
}

void KviWindow::getConfigGroupName(KviStr &buf)
{
	buf = typeString();
}

void KviWindow::getDefaultLogFileName(KviStr &buffer)
{
// FIXME: #warning "Make it configurable ?"
	QDate dt(QDate::currentDate());
	KviStr date(KviStr::Format,"%d.",dt.year());
	if(dt.month() < 10)date.append('0');
	date.append(KviStr::Format,"%d.",dt.month());
	if(dt.day() < 10)date.append('0');
	date.append(KviStr::Format,"%d",dt.day());
	KviStr base;
	kvi_encodeFileName(base);
	getBaseLogFileName(base);
	base.replaceAll(' ',"_");
	base.replaceAll(KVI_PATH_SEPARATOR_CHAR,"-");
	KviStr tmp(KviStr::Format,"%s_%s_%s.log",m_typeTable[type()],base.ptr(),date.ptr());
	g_pApp->getLocalKvircDirectory(buffer,KviApp::Log,tmp.ptr());
}

void KviWindow::getBaseLogFileName(KviStr &buffer)
{
	buffer = name();
}

void KviWindow::saveProperties(KviConfig *)
{
}

void KviWindow::loadProperties(KviConfig *)
{
}

QPixmap * KviWindow::myIconPtr()
{
	return g_pIconManager->getSmallIcon(KVI_SMALLICON_DEFAULTICON);
}

void KviWindow::fillCaptionBuffers()
{
	m_szPlainTextCaption    = name();
	m_szHtmlActiveCaption   = name();
	m_szHtmlInactiveCaption = name();
}

void KviWindow::updateCaption()
{
	fillCaptionBuffers();
	if(mdiParent())mdiParent()->setCaption(plainTextCaption(),htmlActiveCaption(),htmlInactiveCaption());
	else setCaption(plainTextCaption());
	m_pTaskBarItem->captionChanged();
	if(mdiParent() && isMaximized() && (m_pFrm->activeWindow() == this))m_pFrm->updateCaption();
}

void KviWindow::systemPopupRequest(const QPoint &pnt)
{
	m_pContextPopup->clear();

	if(mdiParent())
		m_pContextPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_UNDOCK)),
			__c2q(__tr("&Undock")),this,SLOT(undock()));
	else
		m_pContextPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_DOCK)),
			__c2q(__tr("&Dock")),this,SLOT(dock()));

	m_pContextPopup->insertSeparator();

	if(!isMinimized())
		m_pContextPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_MINIMIZE)),
				__c2q(__tr("M&inimize")),this,SLOT(minimize()));
	if(!isMaximized())
		m_pContextPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_MAXIMIZE)),
				__c2q(__tr("&Maximize")),this,SLOT(maximize()));
	if(isMinimized() | isMaximized())
		m_pContextPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_RESTORE)),
				__c2q(__tr("&Restore")),this,SLOT(restore()));

	m_pContextPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_CLOSE)),
			__c2q(__tr("&Close")),this,SLOT(close()));

	m_pContextPopup->insertSeparator();

	m_pContextPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_XY)),
			__c2q(__tr("&Save properties as default for this window type")),this,SLOT(savePropertiesAsDefault()));

	fillContextPopup(m_pContextPopup);

	m_pContextPopup->popup(pnt);
}

void KviWindow::savePropertiesAsDefault()
{
	m_pFrm->saveWindowProperties(this,typeString());
}

void KviWindow::contextPopup()
{
	systemPopupRequest(QCursor::pos());
}

void KviWindow::fillContextPopup(QPopupMenu *)
{
	// nothing here
}

void KviWindow::undock()
{
	m_pFrm->undockWindow(this);
}

void KviWindow::dock()
{
	m_pFrm->dockWindow(this);
}

void KviWindow::delayedAutoRaise()
{
	QTimer::singleShot(0,this,SLOT(autoRaise()));
}

void KviWindow::autoRaise()
{
	if(!mdiParent()){
		raise();
		setActiveWindow();
	}
	if(m_pFocusHandler)m_pFocusHandler->setFocus();
	else setFocus();
}

void KviWindow::closeEvent(QCloseEvent *e)
{
	e->ignore();
	m_pFrm->childWindowCloseRequest(this);
}

void KviWindow::youAreDocked()
{
	if(m_pAccel)
	{
		delete m_pAccel;
		m_pAccel = 0;
	}
	((KviMdiChild *)parent())->setIcon(*myIconPtr());
	updateCaption();
	connect(((KviMdiChild *)parent()),SIGNAL(systemPopupRequest(const QPoint &)),this,SLOT(systemPopupRequest(const QPoint &)));
}

void KviWindow::youAreUndocked()
{
	m_pAccel = m_pFrm->installAccelerators(this);
	setIcon(*myIconPtr());
	updateCaption();
}

#ifdef FocusIn
	// Hack for X.h
	#undef FocusIn
#endif

bool KviWindow::eventFilter(QObject *o,QEvent *e)
{
	if(e->type() == QEvent::FocusIn)
	{
		if(o != m_pContextPopup)
		{
			m_pFrm->childWindowGainFocus(this);
			m_pFrm->m_pTaskBar->setActiveItem(m_pTaskBarItem);
		}
	} else if(e->type() == QEvent::MouseButtonPress)
	{
		// Hack for wheel events handling
		if(m_pLastPressedChild != o)
		{
			//debug("New last pressed child!");
			m_pLastPressedChild    = o;
			m_bNewLastPressedChild = true;
		}
	}
	return false;
}

QObject * KviWindow::findChildScrollBar(QObject * par)
{
	if(par->inherits("QScrollBar"))
	{
		if(((QWidget *)par)->isVisible())
		{
			if(((QScrollBar *)par)->orientation() == Vertical)return par;
		}
	}
	QObjectList * l = par->queryList("QScrollBar",0,false,true);
	if(l)
	{
		QObjectListIt it(*l);
		while(QObject *o = it.current())
		{
			if(((QWidget *)o)->isVisible())
			{
				if(((QScrollBar *)o)->orientation() == Vertical)
				{
					delete l;
					return o;
				}
			}
			++it;
		}
		delete l;
	}
	return 0;
}

void KviWindow::setWheelEventHandler(QObject * o)
{
	//debug("Setting new wheel event handler");
	if(m_pWheelEventHandler != o)
	{
//		if(m_pWheelEventHandler)disconnect(m_pWheelEventHandler,SIGNAL(destroyed()),this,SLOT(wheelEventHandlerDestroyed()));
		m_pWheelEventHandler = o;
//		if(o)connect(o,SIGNAL(destroyed()),this,SLOT(wheelEventHandlerDestroyed()));
	}
}

void KviWindow::childEvent(QChildEvent *e)
{
	if(e->removed())
	{
		if(e->child() == m_pWheelEventHandler)
		{
			m_pWheelEventHandler = 0;
		} else if(e->child() == m_pLastPressedChild)
		{
			m_pLastPressedChild = 0;
			m_bNewLastPressedChild = true; // next wheelEvent will try to locate a new event handler
		}
	}
	QWidget::childEvent(e);
}

//void KviWindow::wheelEventHandlerDestroyed()
//{
//	m_pWheelEventHandler = 0;
//	setWheelEventHandler(findChildScrollBar(this));
//}

void KviWindow::wheelEvent(QWheelEvent *e)
{
	//debug("Wheel event");
	if(m_bNewLastPressedChild)
	{
		m_bNewLastPressedChild = false;
		if(m_pLastPressedChild)
		{
			QObject * o = findChildScrollBar(m_pLastPressedChild);
			if(o)
			{
				// Ok...valid new wheel event handler
				setWheelEventHandler(o);
			}
		} else {
			// No child has been pressed yet...
			if(m_pWheelEventHandler == 0)
			{
				// Let's search the first available scroll bar...
				setWheelEventHandler(findChildScrollBar(this));
			}
		}
	}
// FIXME: #warning "Double check it for crashes....this could be dangerous...this widget could be dead (?)"
	if(m_pWheelEventHandler)
	{
		//debug("Have a handler (%s)",m_pWheelEventHandler->className());
		g_pApp->sendEvent(m_pWheelEventHandler,e);
		if(!e->isAccepted())e->accept();
	} else e->ignore();
}

void KviWindow::setFocusHandler(QWidget *focusedOne,QWidget *w)
{
	__range_valid(focusedOne);
	if(m_pFocusHandler != 0)m_pFocusHandler = focusedOne;
	if(w==0)w=this;
	if(w != focusedOne)w->setFocusProxy(focusedOne);
	w->installEventFilter(this);
	if(w->focusPolicy() == QWidget::NoFocus)w->setFocusPolicy(QWidget::ClickFocus);
	KviPtrList<QObject> *list = (KviPtrList<QObject> *)(w->children());
	if(list){
		for (unsigned int i=0; i< list->count(); i++){
			QObject *child = list->at(i);
			if(child->inherits("QWidget"))setFocusHandler(focusedOne,((QWidget *)child));
		}
	}
}

void KviWindow::setFocusHandlerNoClass(QWidget *focusedOne,QWidget *w,const char * clName)
{
	__range_valid(focusedOne);
	if(m_pFocusHandler != 0)m_pFocusHandler = focusedOne;
	if(w==0)w=this;
	if((w != focusedOne) && (!(w->inherits(clName))))w->setFocusProxy(focusedOne);
	w->installEventFilter(this);
	if(w->focusPolicy() == QWidget::NoFocus)w->setFocusPolicy(QWidget::ClickFocus);
	KviPtrList<QObject> *list = (KviPtrList<QObject> *)(w->children());
	if(list){
		for (unsigned int i=0; i< list->count(); i++){
			QObject *child = list->at(i);
			if(child->inherits("QWidget"))setFocusHandlerNoClass(focusedOne,((QWidget *)child),clName);
		}
	}
}

void KviWindow::setFocusHandlerNoChildren(QWidget *focusedOne,QWidget *w,bool bInChildTree)
{
	__range_valid(focusedOne);
	if(m_pFocusHandler != 0)m_pFocusHandler = focusedOne;
	if(w==0)w=this;
	if(w != focusedOne)
	{
		if(!bInChildTree)w->setFocusProxy(focusedOne);
	} else bInChildTree = true;

	w->installEventFilter(this);
	if(w->focusPolicy() == QWidget::NoFocus)w->setFocusPolicy(QWidget::ClickFocus);
	KviPtrList<QObject> *list = (KviPtrList<QObject> *)(w->children());
	if(list){
		for (unsigned int i=0; i< list->count(); i++){
			QObject *child = list->at(i);
			if(child->inherits("QWidget"))setFocusHandlerNoChildren(focusedOne,((QWidget *)child),bInChildTree);
		}
	}
}

void KviWindow::updateBackgrounds(QObject * obj)
{
	if(!obj)obj = this;
	QObjectList * list = (QObjectList *)(obj->children());
	if(list)
	{
		for(unsigned int i=0;i<list->count();i++)
		{
			QObject * child = list->at(i);
			QVariant v = list->at(i)->property("TransparencyCapable");
			if(v.isValid())((QWidget *)child)->update();

//			if(list->at(i)->metaObject()->classInfo("TransparencyCapable"))
//				((QWidget *)child)->update();
			updateBackgrounds(child);
		}
	}
}

void KviWindow::moveEvent(QMoveEvent *e)
{
#ifdef COMPILE_PSEUDO_TRANSPARENCY
	updateBackgrounds();
#endif
	QWidget::moveEvent(e);
}

void KviWindow::minimize()
{
	if(mdiParent())
	{
		if(!isMinimized())mdiParent()->minimize();
	} else showMinimized();
}

void KviWindow::maximize()
{
	if(mdiParent()){
		if(!isMaximized())mdiParent()->maximize();
	} else showMaximized();
	autoRaise();
}

bool KviWindow::isMinimized()
{
	if(mdiParent())return (mdiParent()->state() == KviMdiChild::Minimized);
	else return QWidget::isMinimized();
}

bool KviWindow::isMaximized()
{
	if(mdiParent())return (mdiParent()->state() == KviMdiChild::Maximized);
	// Heh...how to check it ?
	// Empirical check
	int wdth = (g_pApp->desktop()->width() * 75) / 100;
	int hght = (g_pApp->desktop()->height() * 75) / 100;
	
	return ((x() <= 1)&&(y() <= 1)&&(width() >= wdth)&&(height() >= hght));
}

void KviWindow::restore()
{
	if(mdiParent()){
		if(isMinimized()||isMaximized())mdiParent()->restore();
	} else showNormal();
	autoRaise();
}

QRect KviWindow::externalGeometry()
{
	return mdiParent() ? mdiParent()->geometry() : frameGeometry();
}

void KviWindow::applyOptions()
{
	updateCaption();
}

KviWindow * KviWindow::outputProxy()
{
	return 0;
}

void KviWindow::output(int msg_type,const char *format,...)
{
	char txt_ptr[512]; //It should be enough for all outputs...
	va_list list;
	va_start(list,format);
	if(kvi_vsnprintf(txt_ptr,512,format,list) < 0){
		//Just in case...
		va_end(list);
		int len = 512;
		char *long_txt_ptr = 0;
		int result;
		do{
			len += 512;
			//first time long_txt_ptr == 0 so it is equivalent to malloc
			//At least the man page says that...
			long_txt_ptr = (char *)kvi_realloc((void *)long_txt_ptr,len);
			va_start(list,format);
			result = kvi_vsnprintf(long_txt_ptr,len,format,list);
			va_end(list);
		} while(result < 0);
		if(m_pIrcView)m_pIrcView->appendText(msg_type,long_txt_ptr);
		else {
			// Redirect to the output proxy
			KviWindow *wnd = outputProxy();
			if(wnd)wnd->outputNoFmt(msg_type,long_txt_ptr);
		}
		kvi_free((void *)long_txt_ptr);
	} else {
		//Succesful vsnprintf
		va_end(list);
		if(m_pIrcView)m_pIrcView->appendText(msg_type,txt_ptr);
		else {
			// Redirect to the output proxy
			KviWindow *wnd = outputProxy();
			if(wnd)wnd->outputNoFmt(msg_type,txt_ptr);
		}
	}
	m_pTaskBarItem->highlight(KVI_OPTION_MSGTYPE(msg_type).level());
}

//================ outputNoFmt ================//

void KviWindow::outputNoFmt(int msg_type,const char * text)
{
	__range_valid(text);
	if(m_pIrcView)m_pIrcView->appendText(msg_type,text);
	else {
		// Redirect to the output proxy
		KviWindow *wnd = outputProxy();
		if(wnd)wnd->outputNoFmt(msg_type,text);
	}
	m_pTaskBarItem->highlight(KVI_OPTION_MSGTYPE(msg_type).level());
}

#include "kvi_window.moc"
