//
//   File : kvi_userlistview.cpp
//   Creation date : Tue Aug 1 2000 21:05:22 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
#define __KVIRC__
#define _KVI_USERLISTVIEW_CPP_

#define _KVI_DEBUG_CHECK_RANGE_
#include "kvi_debug.h"

#include "kvi_userlistview.h"
#include "kvi_settings.h"
#include "kvi_locale.h"
#include "kvi_options.h"
#include "kvi_defaults.h"
#include "kvi_iconmanager.h"
#include "kvi_regusersdb.h"
#include "kvi_event.h"
#include "kvi_parameterlist.h"
#include "kvi_window.h"
#include "kvi_console.h"


#include <qlabel.h>
#include <qscrollbar.h>
#include <qpainter.h>
#include <qpixmap.h>
#include <qfontmetrics.h>
#include <qdatetime.h>
#include <qmime.h>

#ifdef COMPILE_PSEUDO_TRANSPARENCY
	extern QPixmap * g_pShadedChildGlobalDesktopBackground;
#endif

// kvi_app.cpp
extern KviRegisteredUserDataBase * g_pRegisteredUserDataBase;

// kvi_app.cpp (loaded and destroyed by KviIconManager)
extern QPixmap * g_pUserChanStatePixmap;

// Yet another really complex widget :)

#define KVI_USERLIST_BORDER_WIDTH 2

// FIXME: #warning "Button to show/hide avatars"
// FIXME: #warning "We want to be able to navigate the list with the keyboard!"

KviUserListToolTip::KviUserListToolTip(KviUserListView * v,KviUserListViewArea * a)
: QToolTip(a,0)
{
	m_pListView = v;
}

KviUserListToolTip::~KviUserListToolTip()
{
}

void KviUserListToolTip::maybeTip(const QPoint &pnt)
{
	m_pListView->maybeTip(this,pnt);
}




KviUserListEntry::KviUserListEntry(KviUserListView * parent,const char * nick,
	KviIrcUserEntry * e,short int iFlags,bool bJoinTimeUnknown)
{
	m_pListView      = parent;
	m_szNick         = nick;
	m_pGlobalData    = e;
	m_iFlags         = iFlags;
	m_lastActionTime = (time_t)0;
	m_joinTime       = bJoinTimeUnknown ? (time_t)0 : time(0);
	m_iTemperature   = bJoinTimeUnknown ? 0 : KVI_USERACTION_JOIN;

	m_bSelected      = false;
	recalcSize();
}

KviUserListEntry::~KviUserListEntry()
{
}

void KviUserListEntry::recalcSize()
{
	KviAvatar * av = m_pGlobalData->avatar();
	m_iHeight = m_pListView->m_iFontHeight + 4;
	if(av)
	{
		m_iHeight += KVI_OPTION_BOOL(KviOption_boolScaleAvatars) ? (int)KVI_OPTION_UINT(KviOption_uintAvatarScaleHeight) : av->pixmap()->height();
	}
    if(m_iHeight < 22)m_iHeight = 22;
}

///////////////////////////////////////////////



KviUserListView::KviUserListView(QWidget * par,KviIrcUserDataBase * db,KviWindow * pWnd,int dictSize,const char * label_text,const char * name)
: QWidget(par,name)
{
	m_pKviWindow  = pWnd;
	m_pEntryDict  = new QAsciiDict<KviUserListEntry>(dictSize,false,true);
	m_pEntryDict->setAutoDelete(true);
	m_pUsersLabel = new QLabel(this);
	QToolTip::add(m_pUsersLabel,__c2q(label_text));
	m_pViewArea   = new KviUserListViewArea(this);
	m_pToolTip    = new KviUserListToolTip(this,m_pViewArea); // valgrind signals this as leaked memory ????
	m_pTopItem    = 0;
	m_pHeadItem   = 0;
	m_pTailItem   = 0;
	m_iOpCount    = 0;
	m_iHalfOpCount = 0;
	m_iVoiceCount = 0;
	m_pIrcUserDataBase = db;
	m_iTotalHeight = 0;
	m_ibEntries   = 0;
	m_ieEntries   = 0;
	m_iIEntries   = 0;
//	setBackgroundMode(QWidget::NoBackground);

	applyOptions();

}

KviUserListView::~KviUserListView()
{
	removeAllEntries();
	delete m_pEntryDict;
	delete m_pToolTip;
}

void KviUserListView::emitRightClick()
{
	int ev = -1;
	switch(m_pKviWindow->type())
	{
		case KVI_WINDOW_TYPE_CHANNEL: ev = KviEvent_OnChannelNickPopupRequest;  break;
		case KVI_WINDOW_TYPE_QUERY:   ev = KviEvent_OnQueryNickPopupRequest;    break;
		case KVI_WINDOW_TYPE_CONSOLE: ev = KviEvent_OnNotifyListPopupRequest;   break;
		default:                      ev = KviEvent_OnNickLinkPopupRequest;     break; // this should actually never happen
	}
	if(ev > -1)
	{
		if(g_pEventManager->hasEventHandlers(ev))
		{
			KviStr * nicks = new KviStr();
			appendSelectedNicknames(*nicks);
			g_pUserParser->triggerEvent(ev,m_pKviWindow,new KviParameterList(nicks));
		}
	}
}

void KviUserListView::emitDoubleClick()
{
	int ev = -1;
	switch(m_pKviWindow->type())
	{
		case KVI_WINDOW_TYPE_CHANNEL: ev = KviEvent_OnChannelNickDefaultActionRequest;  break;
		case KVI_WINDOW_TYPE_QUERY:   ev = KviEvent_OnQueryNickDefaultActionRequest;    break;
		case KVI_WINDOW_TYPE_CONSOLE: ev = KviEvent_OnNotifyListDefaultActionRequest;   break;
		default:                      ev = KviEvent_OnNickLinkDefaultActionRequest;     break; // this should actually never happen
	}
	if(ev > -1)
	{
		if(g_pEventManager->hasEventHandlers(ev))
		{
			KviStr * nick = new KviStr();
			appendSelectedNicknames(*nick);
			g_pUserParser->triggerEvent(ev,m_pKviWindow,new KviParameterList(nick));
		}
	}
}

void KviUserListView::updateScrollBarRange()
{
	int max = m_iTotalHeight - (m_pViewArea->height() - (KVI_USERLIST_BORDER_WIDTH * 2));
	m_pViewArea->m_pScrollBar->setRange(0,max > 0 ? max : 0);
}

void KviUserListView::applyOptions()
{
	setFont(KVI_OPTION_FONT(KviOption_fontUserListView));
	QFontMetrics fm(KVI_OPTION_FONT(KviOption_fontUserListView));
	m_iFontHeight = fm.lineSpacing();
	KviUserListEntry * e = m_pHeadItem;
	m_iTotalHeight = 0;
	while(e)
	{
		e->recalcSize();
		m_iTotalHeight += e->m_iHeight;
		e = e->m_pNext;
	}
	updateScrollBarRange();
	m_pUsersLabel->setFont(KVI_OPTION_FONT(KviOption_fontUserListView));
	resizeEvent(0); // this will call update() too
}

void KviUserListView::enableUpdates(bool bEnable)
{
	m_pViewArea->setUpdatesEnabled(bEnable);
	if(bEnable)triggerUpdate();
}

void KviUserListView::completeNickBashLike(const KviStr &begin,KviPtrList<KviStr> *l,bool bAppendMask)
{
	KviUserListEntry * entry = m_pHeadItem;

	while(entry)
	{
		if(kvi_strEqualCIN(begin.ptr(),entry->m_szNick.ptr(),begin.len()))
		{
			if(bAppendMask)
				l->append(new KviStr(KviStr::Format,"%s!%s@%s",entry->m_szNick.ptr(),
					entry->m_pGlobalData->user(),entry->m_pGlobalData->host()));
			else
				l->append(new KviStr(entry->m_szNick));
		}
		entry = entry->m_pNext;
	}
}

bool KviUserListView::completeNickStandard(const KviStr &begin,const KviStr &skipAfter,KviStr &buffer,bool bAppendMask)
{
	KviUserListEntry * entry = m_pHeadItem;

	if(skipAfter.hasData())
	{
		while(entry)
		{
			if(kvi_strEqualCI(skipAfter.ptr(),entry->m_szNick.ptr()))
			{
				entry = entry->m_pNext;
				break;
			}
			entry = entry->m_pNext;
		}
	}

	// FIXME: completion should skip my own nick or place it as last entry in the chain (?)

//	if(KviConsole * c = m_pKviWindow->console())
//	{
//		if(kvi_strEqualCI(entry->m_szNick.ptr(),c->currentNickName())
//	}

	// Ok...now the real completion
	while(entry)
	{
		if(entry->m_szNick.len() >= begin.len())
		{
			char save = *(entry->m_szNick.ptr() + begin.len());
			*(entry->m_szNick.ptr() + begin.len()) = '\0';
			int result = kvi_strcmpCI(begin.ptr(),entry->m_szNick.ptr());
			*(entry->m_szNick.ptr() + begin.len()) = save;
			if(result == 0)
			{
				// This is ok.
				buffer = entry->m_szNick;
				if(bAppendMask)buffer.append(KviStr::Format,"!%s@%s",
									entry->m_pGlobalData->user(),entry->m_pGlobalData->host());
				return true;
			} else if(result < 0)
			{
				// No match...begin is lower than the current entry
				if(entry->m_iFlags == 0)return false;
				else {
					int flags = entry->m_iFlags;
					// skip the current flag
					while(entry)
					{
						if(entry->m_iFlags != flags)break;
						entry = entry->m_pNext;
					}
					continue;
				}
			}
		}
		entry = entry->m_pNext;
	}

	return false;
}


void KviUserListView::insertUserEntry(const char * nick,KviUserListEntry * e)
{
	// Complex insertion task :)

	m_pEntryDict->insert(nick,e);
	m_iTotalHeight += e->m_iHeight;

	bool bGotTopItem = false;

	int flag = 0;
	if(e->m_iFlags != 0)
	{
		if(e->m_iFlags & KVI_USERFLAG_VOICE)
		{
			flag = KVI_USERFLAG_VOICE;
			m_iVoiceCount++;
		}
		if(e->m_iFlags & KVI_USERFLAG_HALFOP)
		{
			flag = KVI_USERFLAG_HALFOP;
			m_iHalfOpCount++;
		}
		if(e->m_iFlags & KVI_USERFLAG_OP)
		{
			flag = KVI_USERFLAG_OP; // op takes precedence
			m_iOpCount++;
		}
	}


	if(m_pHeadItem)
	{
		KviUserListEntry * entry = m_pHeadItem;

		if(!(e->m_iFlags & KVI_USERFLAG_OP))
		{
			// the new user is not an op...
			// skip ops
			while(entry && (entry->m_iFlags & KVI_USERFLAG_OP))
			{
				if(entry == m_pTopItem)bGotTopItem = true;
				entry = entry->m_pNext;
			}

			// is half oped ?
			if(!(e->m_iFlags & KVI_USERFLAG_HALFOP))
			{
				// nope , skip halfops
				while(entry && (entry->m_iFlags & KVI_USERFLAG_HALFOP))
				{
					if(entry == m_pTopItem)bGotTopItem = true;
					entry = entry->m_pNext;
				}

				// is voiced ?
				if(!(e->m_iFlags & KVI_USERFLAG_VOICE))
				{
					// nope , not voiced so skip voiced users
					while(entry && (entry->m_iFlags & KVI_USERFLAG_VOICE))
					{
						if(entry == m_pTopItem)bGotTopItem = true;
						entry = entry->m_pNext;
					}
				} // else it is voiced , ops and halfops are skipped
			} // else it is halfop ,  ops are skipped
		} // else it is op! , so nothing to skip: the ops are first in the list

		// now strcmp within the current user-flag group...
		while(entry && (kvi_strcmpCI(entry->m_szNick.ptr(),e->m_szNick.ptr()) < 0) &&
				((entry->m_iFlags & flag) || (flag == 0)))
		{
			if(entry == m_pTopItem)bGotTopItem = true;
			entry = entry->m_pNext;
		}
		if(entry)
		{
//			if(entry == m_pTopItem)bGotTopItem = true;
			// inserting
			e->m_pNext = entry;
			e->m_pPrev = entry->m_pPrev;
			if(e->m_pPrev == 0)m_pHeadItem = e;
			else e->m_pPrev->m_pNext = e;
			entry->m_pPrev = e;
			// need to adjust the item offsets now...
			// ok... if we're inserting something after
			// the top item, we move everything down
			// otherwise we only update the scrollbar values
			if(!bGotTopItem)
			{
				// Inserting BEFORE the top item
				if((e == m_pHeadItem) && (m_pTopItem == e->m_pNext) && (m_pViewArea->m_iTopItemOffset == 0))
				{
					// special case...the top item is the head one
					// and it has zero offset...change the top item too
					m_pTopItem = e;
					triggerUpdate();
				} else {
					// invisible insertion
					m_pViewArea->m_bIgnoreScrollBar = true;
					m_pViewArea->m_iLastScrollBarVal += e->m_iHeight;
					updateScrollBarRange();
					m_pViewArea->m_pScrollBar->setValue(m_pViewArea->m_iLastScrollBarVal);
					m_pViewArea->m_bIgnoreScrollBar = false;
					updateUsersLabel();
				}
			} else {
				triggerUpdate();
			}
		} else {
			// appending to the end (may be visible)
			m_pTailItem->m_pNext = e;
			e->m_pNext = 0;
			e->m_pPrev = m_pTailItem;
			m_pTailItem = e;
			triggerUpdate();
		}
	} else {
		// There were no items (is rather visible)
		m_pHeadItem = e;
		m_pTailItem = e;
		m_pTopItem = e;
		e->m_pNext = 0;
		e->m_pPrev = 0;
		triggerUpdate();
	}
}

KviUserListEntry * KviUserListView::join(const char *nick,const char * user,
		const char * host,int iFlags)
{
	// Ok..an user joins the channel
	KviUserListEntry * it = m_pEntryDict->find(nick);
	if(it == 0)
	{
		// add an entry to the global dict
		KviIrcUserEntry * pGlobalData = m_pIrcUserDataBase->insertUser(nick,user,host);
		// calculate the flags and update the counters
		it = new KviUserListEntry(this,nick,pGlobalData,iFlags,(user == 0));
		insertUserEntry(nick,it);
	} else {
		if(host)
		{
			// Bad luck...we must be out of sync , or the server got drunk
// FIXME: #warning "DO SOMETHING REASONABLE HERE!"
		} else {
			// Ok...the user was already on...
			// Probably this is a NAMES reply , and the user IS_ME
			// (already joined after the JOIN message)
//			if((bOp && (!(it->m_iFlags & KVI_USERFLAG_OP))) || ((!bOp) && (it->m_iFlags & KVI_USERFLAG_OP)) )
//			{
//				// WE WANT A LOGICAL XOR! :))))
//				op(nick,bOp);
//			}
//			if((bVoice && (!(it->m_iFlags & KVI_USERFLAG_VOICE))) || ((!bVoice) && (it->m_iFlags & KVI_USERFLAG_VOICE)) )
//			{
//				voice(nick,bVoice);
//			}
			if(iFlags != it->m_iFlags)
			{
//// FIXME: #warning "Maybe say to the channel that we're oped : and the op is guessed from the names reply"
				if((iFlags & KVI_USERFLAG_OP) != (it->m_iFlags & KVI_USERFLAG_OP))op(nick,iFlags & KVI_USERFLAG_OP);
				if((iFlags & KVI_USERFLAG_HALFOP) != (it->m_iFlags & KVI_USERFLAG_HALFOP))halfop(nick,iFlags & KVI_USERFLAG_HALFOP);
				if((iFlags & KVI_USERFLAG_VOICE) != (it->m_iFlags & KVI_USERFLAG_VOICE))voice(nick,iFlags & KVI_USERFLAG_VOICE);
			}
		}
	}
	return it;
}

void KviUserListView::triggerUpdate()
{
	// This stuff is useful on joins only
	if(m_pViewArea->isUpdatesEnabled())
	{
		//m_pViewArea->m_pScrollBar->setRange(0,m_iTotalHeight);
		updateScrollBarRange();
		m_pViewArea->update();
		updateUsersLabel();
	}
}

bool KviUserListView::avatarChanged(const char * nick)
{
	KviUserListEntry * it = m_pEntryDict->find(nick);
	if(it)
	{
		int oldH = it->m_iHeight;
		m_iTotalHeight -= it->m_iHeight;
		it->recalcSize();
		m_iTotalHeight += it->m_iHeight;
		// if this was "over" the top item , we must adjust the scrollbar value
		// otherwise scroll everything down
		KviUserListEntry * e = m_pHeadItem;
		bool bGotTopItem = false;
		while(e != it)
		{
			if(e == m_pTopItem)
			{
				bGotTopItem = true;
				e = it;
			} else e = e->m_pNext;
		}
		if(!bGotTopItem && (m_pTopItem != it))
		{
			// we're "over" the top item , so over the
			// upper side of the view...adjust the scroll bar value
			int hDiff = it->m_iHeight - oldH;
			m_pViewArea->m_iLastScrollBarVal += hDiff;
			m_pViewArea->m_bIgnoreScrollBar = true;
//			m_pViewArea->m_pScrollBar->setRange(0,m_iTotalHeight);
			updateScrollBarRange();
			m_pViewArea->m_pScrollBar->setValue(m_pViewArea->m_iLastScrollBarVal);
			m_pViewArea->m_bIgnoreScrollBar = false;
		} else {
			// the item may be visible!
			// the scroll bar should take care of the case
			// in that the current value runs out of the allowed
			// range.... it should set the value to a good one
			// and emit the signal
			updateScrollBarRange();
//			m_pViewArea->m_pScrollBar->setRange(0,m_iTotalHeight);
			m_pViewArea->update();
		}
		return true;
	}
	return false;
}

void KviUserListView::userAction(KviIrcMask *user,int actionTemperature)
{
	// This is called when an user "acts" in some visible way
	// on the channel, so we can keep track of his channeel
	// idle time. This will also update the username and hostname
	// if needed.
	KviUserListEntry * it = m_pEntryDict->find(user->nick());
	if(it)
	{
		it->m_lastActionTime = time(0);
		if(user->hasUsername())it->m_pGlobalData->setUser(user->username());
		if(user->hasHost())it->m_pGlobalData->setHost(user->host());
		it->m_iTemperature += actionTemperature;
		if(itemVisible(it))triggerUpdate();
	}
}

void KviUserListView::userAction(const char * nick,int actionTemperature)
{
	// This is called when an user "acts" in some visible way
	// on the channel, so we can keep track of his channeel
	// idle time. This will also update the username and hostname
	// if needed.
	KviUserListEntry * it = m_pEntryDict->find(nick);
	if(it)
	{
		it->m_lastActionTime = time(0);
		it->m_iTemperature += actionTemperature;
		if(itemVisible(it))triggerUpdate();
	}
}


bool KviUserListView::op(const char *nick,bool bOp)
{
	KviUserListEntry * it = m_pEntryDict->find(nick);
	if(!it)return false;
	m_pEntryDict->setAutoDelete(false);
	partInternal(nick,false);
	m_pEntryDict->setAutoDelete(true);
	if(bOp)
	{
		if(!(it->m_iFlags & KVI_USERFLAG_OP))
		{
			it->m_iFlags |= KVI_USERFLAG_OP;
		}
	} else {
		if(it->m_iFlags & KVI_USERFLAG_OP)
		{
			it->m_iFlags &= ~KVI_USERFLAG_OP;
		}
	}
	insertUserEntry(nick,it);
//	updateUsersLabel();
	m_pViewArea->update();
	return true;
}

bool KviUserListView::halfop(const char *nick,bool bHalfOp)
{
	KviUserListEntry * it = m_pEntryDict->find(nick);
	if(!it)return false;
	m_pEntryDict->setAutoDelete(false);
	partInternal(nick,false);
	m_pEntryDict->setAutoDelete(true);
	if(bHalfOp)
	{
		if(!(it->m_iFlags & KVI_USERFLAG_HALFOP))
		{
			it->m_iFlags |= KVI_USERFLAG_HALFOP;
		}
	} else {
		if(it->m_iFlags & KVI_USERFLAG_HALFOP)
		{
			it->m_iFlags &= ~KVI_USERFLAG_HALFOP;
		}
	}
	insertUserEntry(nick,it);
//	updateUsersLabel();
	m_pViewArea->update();
	return true;
}

char KviUserListView::getUserFlag(const char * nick)
{
	KviUserListEntry * it = m_pEntryDict->find(nick);
	if(!it)return 0;
	if(it->m_iFlags & (KVI_USERFLAG_OP | KVI_USERFLAG_VOICE | KVI_USERFLAG_HALFOP))
	{
		if(it->m_iFlags & KVI_USERFLAG_OP)return '@';
		if(it->m_iFlags & KVI_USERFLAG_HALFOP)return '%';
		if(it->m_iFlags & KVI_USERFLAG_VOICE)return '+';
	}
	return 0;
}

void KviUserListView::prependUserFlag(const char * nick,KviStr &buffer)
{
	char uFlag = getUserFlag(nick);
	if(uFlag)buffer.prepend(uFlag);
}

bool KviUserListView::isOp(const char * nick)
{
	KviUserListEntry * it = m_pEntryDict->find(nick);
	return it ? (it->m_iFlags & KVI_USERFLAG_OP) : false;
}

int KviUserListView::flags(const char * nick)
{
	KviUserListEntry * it = m_pEntryDict->find(nick);
	return it ? it->m_iFlags : 0;
}

bool KviUserListView::isVoice(const char * nick)
{
	KviUserListEntry * it = m_pEntryDict->find(nick);
	return it ? (it->m_iFlags & KVI_USERFLAG_VOICE) : false;
}

bool KviUserListView::isHalfOp(const char * nick)
{
	KviUserListEntry * it = m_pEntryDict->find(nick);
	return it ? (it->m_iFlags & KVI_USERFLAG_HALFOP) : false;
}

bool KviUserListView::voice(const char *nick,bool bVoice)
{
	KviUserListEntry * it = m_pEntryDict->find(nick);
	if(!it)return false;
	m_pEntryDict->setAutoDelete(false);
	partInternal(nick,false);
	m_pEntryDict->setAutoDelete(true);
	if(bVoice)
	{
		if(!(it->m_iFlags & KVI_USERFLAG_VOICE))
		{
			it->m_iFlags |= KVI_USERFLAG_VOICE;
		}
	} else {
		if(it->m_iFlags & KVI_USERFLAG_VOICE)
		{
			it->m_iFlags &= ~KVI_USERFLAG_VOICE;
		}
	}
	insertUserEntry(nick,it);
//	updateUsersLabel();
	m_pViewArea->update();
	return true;
}

KviStr * KviUserListView::firstSelectedNickname()
{
	m_pIterator = m_pHeadItem;
	while(m_pIterator)
	{
		if(m_pIterator->m_bSelected)
		{
			KviStr * s = &(m_pIterator->m_szNick);
			m_pIterator = m_pIterator->m_pNext;
			return s;
		}
		m_pIterator = m_pIterator->m_pNext;
	}
	return 0;
}

KviStr * KviUserListView::nextSelectedNickname()
{
	while(m_pIterator)
	{
		if(m_pIterator->m_bSelected)
		{
			KviStr * s = &(m_pIterator->m_szNick);
			m_pIterator = m_pIterator->m_pNext;
			return s;
		}
		m_pIterator = m_pIterator->m_pNext;
	}
	return 0;
}

void KviUserListView::appendSelectedNicknames(KviStr &buffer)
{
	KviUserListEntry * aux = m_pHeadItem;
	bool bFirst = true;
	while(aux)
	{
		if(aux->m_bSelected)
		{
			if(!bFirst)buffer.append(',');
			else bFirst = false;
			buffer.append(aux->m_szNick);
		}
		aux = aux->m_pNext;
	}
}

bool KviUserListView::partInternal(const char * nick,bool bRemove)
{
	KviUserListEntry * it = m_pEntryDict->find(nick);
	if(it)
	{
		// so, first of all..check if this item is over, or below the top item
		KviUserListEntry * e = m_pHeadItem;
		bool bGotTopItem = false;
		while(e != it)
		{
			if(e == m_pTopItem)
			{
				bGotTopItem = true;
				e = it;
			} else e = e->m_pNext;
		}
		if(bRemove)m_pIrcUserDataBase->removeUser(nick,it->m_pGlobalData);

		// now just remove it
		if(it->m_iFlags & KVI_USERFLAG_OP)m_iOpCount--;
		if(it->m_iFlags & KVI_USERFLAG_VOICE)m_iVoiceCount--;
		if(it->m_iFlags & KVI_USERFLAG_HALFOP)m_iHalfOpCount--;
		if(it->m_pPrev)it->m_pPrev->m_pNext = it->m_pNext;
		if(it->m_pNext)it->m_pNext->m_pPrev = it->m_pPrev;
		if(m_pTopItem == it)
		{
			bGotTopItem = true; //!!! the previous while() does not handle it!
			m_pTopItem = it->m_pNext;
			if(m_pTopItem == 0)m_pTopItem = it->m_pPrev;
		}
		if(it == m_pHeadItem)m_pHeadItem = it->m_pNext;
		if(it == m_pTailItem)m_pTailItem = it->m_pPrev;
		m_iTotalHeight -= it->m_iHeight;

		int iHeight = it->m_iHeight;

		m_pEntryDict->remove(nick);

		if(bGotTopItem)
		{
			// removing after (or exactly) the top item, may be visible
			if(bRemove)triggerUpdate();
		} else {
			// removing over (before) the top item...not visible
			m_pViewArea->m_bIgnoreScrollBar = true;
			m_pViewArea->m_iLastScrollBarVal -= iHeight;
			m_pViewArea->m_pScrollBar->setValue(m_pViewArea->m_iLastScrollBarVal);
//			m_pViewArea->m_pScrollBar->setRange(0,m_iTotalHeight);
			updateScrollBarRange();
			m_pViewArea->m_bIgnoreScrollBar = false;
			if(bRemove)updateUsersLabel();
		}

		return true;
	}
	return false;
}

bool KviUserListView::nickChange(const char *oldNick,const char * newNick)
{
	KviUserListEntry * it = m_pEntryDict->find(oldNick);
	if(it)
	{
		KviStr user   = it->m_pGlobalData->user();
		KviStr host   = it->m_pGlobalData->host();
		int iFlags    = it->m_iFlags;
		time_t joint  = it->m_joinTime;
		bool bSelect  = it->m_bSelected;
		KviAvatar * av  = it->m_pGlobalData->forgetAvatar();
		part(oldNick);
		__range_invalid(m_pEntryDict->find(oldNick));
		it = join(newNick,user.ptr(),host.ptr(),iFlags);
		it->m_joinTime = joint;
		it->m_lastActionTime = time(0);
		it->m_bSelected = bSelect;
		it->m_iTemperature += KVI_USERACTION_NICK;
		if(av)
		{
//			if(!it->m_pGlobalData->avatar())
//			{
				it->m_pGlobalData->setAvatar(av);
				avatarChanged(newNick);
//			} else delete av; //it should never happen anyway
		}
		return true;
	}
	return false;
}


void KviUserListView::updateUsersLabel()
{
	KviStr tmp;
	tmp.sprintf(" u:%u",m_pEntryDict->count());
	if(m_iOpCount)tmp.append(KviStr::Format," o:%d",m_iOpCount);
	if(m_iHalfOpCount)tmp.append(KviStr::Format," h:%d",m_iHalfOpCount);
	if(m_iVoiceCount)tmp.append(KviStr::Format," v:%d",m_iVoiceCount);
	if(m_ibEntries)tmp.append(KviStr::Format," b:%d",m_ibEntries);
	if(m_ieEntries)tmp.append(KviStr::Format," e:%d",m_ieEntries);
	if(m_iIEntries)tmp.append(KviStr::Format," I:%d",m_iIEntries);
	m_pUsersLabel->setText(tmp.ptr());
}

void KviUserListView::removeAllEntries()
{
	QAsciiDictIterator<KviUserListEntry> it(*m_pEntryDict);
	while(it.current())
	{
		m_pIrcUserDataBase->removeUser(it.currentKey(),
			((KviUserListEntry *)it.current())->m_pGlobalData);
		++it;
	}
	m_pEntryDict->clear();
	m_pHeadItem = 0;
	m_pTopItem = 0;
	m_iVoiceCount = 0;
	m_iHalfOpCount = 0;
	m_iOpCount = 0;
	m_pViewArea->m_iTopItemOffset = 0;
	m_pViewArea->m_iLastScrollBarVal = 0;
	m_pViewArea->m_bIgnoreScrollBar = true;
	m_pViewArea->m_pScrollBar->setValue(0);
//	m_pViewArea->m_pScrollBar->setRange(0,0);
	m_iTotalHeight = 0;

	updateScrollBarRange();
	m_pViewArea->m_bIgnoreScrollBar = true;
}

void KviUserListView::partAll()
{
	removeAllEntries();
	triggerUpdate();
}

void KviUserListView::resizeEvent(QResizeEvent *)
{
	int hght = m_pUsersLabel->sizeHint().height();
	m_pUsersLabel->setGeometry(0,0,width(),hght);
	m_pViewArea->setGeometry(0,hght,width(),height() - hght);
	updateScrollBarRange();
}

bool KviUserListView::itemVisible(KviUserListEntry * e)
{
	KviUserListEntry * le = m_pTopItem;
	int curTop = KVI_USERLIST_BORDER_WIDTH - m_pViewArea->m_iTopItemOffset;
	int hght   = height();
	while(le && (curTop < hght))
	{
		if(le == e)return true;
		curTop += le->m_iHeight;
		le = le->m_pNext;
	}
	return false;
}

KviUserListEntry * KviUserListView::itemAt(const QPoint &pnt,QRect * rct)
{
	if(!m_pTopItem)return 0;
	if(pnt.y() < 0)return 0;
	int curTop = KVI_USERLIST_BORDER_WIDTH - m_pViewArea->m_iTopItemOffset;
	KviUserListEntry * e = m_pTopItem;
	int curBottom = 0;
	while(e && (curTop <= m_pViewArea->height()))
	{
		curBottom = curTop + e->m_iHeight;
		if((pnt.y() >= curTop) && (pnt.y() < curBottom))
		{
			if(rct)
			{
				rct->setX(0);
				rct->setY(curTop);
				rct->setWidth(m_pViewArea->width());
				rct->setHeight(e->m_iHeight);
			}
			return e;
		}
		curTop = curBottom;
		e = e->m_pNext;
	}
	return 0;
}


void KviUserListView::maybeTip(KviUserListToolTip * tip,const QPoint &pnt)
{
	QRect itRect;
	KviUserListEntry * it = (KviUserListEntry *)itemAt(pnt,&itRect);
	if(it)
	{
		if(m_pKviWindow->console())
		{
			KviStr tmp;
			m_pKviWindow->console()->getUserTipText(it->m_szNick.ptr(),it->m_pGlobalData,tmp);

			bool bFirst = true;

			if(it->m_joinTime != 0)
			{
				QDateTime dt;
				dt.setTime_t(it->m_joinTime);
				tmp.append(KviStr::Format,
					__tr("<hr><nobr>Joined on <b>%s</b></nobr>"),
					dt.toString().ascii());
				bFirst = false;
			}
	
			if(it->m_lastActionTime != 0)
			{
				int secs = time(0) - it->m_lastActionTime;
				int mins = secs / 60;
				secs = secs % 60;
				int hors = mins / 60;
				mins = mins % 60;
				tmp.append(KviStr::Format,
					__tr("<%s><nobr>Quiet since <b>%d h %d m %d s</b></nobr>"),
					bFirst ? "hr" : "br",hors,mins,secs);
				bFirst = false;
			}
	
			tip->doTip(itRect,tmp.ptr());
		}
	}
}

///////////////////////////////////////////////////////////////

KviUserListViewArea::KviUserListViewArea(KviUserListView * par)
: QWidget(par)
{
	m_pListView = par;
	setBackgroundMode(QWidget::NoBackground);
	m_pScrollBar = new QScrollBar(QScrollBar::Vertical,this);
	m_pScrollBar->setRange(0,0);
	m_pScrollBar->setValue(0);
	connect(m_pScrollBar,SIGNAL(valueChanged(int)),this,SLOT(scrollBarMoved(int)));
	m_pScrollBar->setPageStep(height());
	m_pScrollBar->setLineStep(m_pListView->m_iFontHeight);
	m_pMemBuffer = new QPixmap();
	m_iLastScrollBarVal = 0;
	m_iTopItemOffset = 0;
	m_bIgnoreScrollBar = false;
	m_pLastEntryUnderMouse = 0;
}

KviUserListViewArea::~KviUserListViewArea()
{
	delete m_pMemBuffer;
}

//void KviUserListViewArea::updateScrollBar()
//{
//	m_pScrollBar->setRange(0,m_pListView->m_iTotalHeight - 1);
////	showOrHideScrollbar();
//}

void KviUserListViewArea::scrollBarMoved(int newVal)
{
	if(m_bIgnoreScrollBar)return;
	int diff = newVal - m_iLastScrollBarVal;
	if(m_pListView->m_pTopItem)
	{
		while(diff > 0)
		{
			int nextH = (m_pListView->m_pTopItem->m_iHeight - m_iTopItemOffset);
			if(diff >= nextH)
			{
				// the diff is greater than the top item visible part
				diff -= nextH;
				if(m_pListView->m_pTopItem->m_pNext)
				{
					// There is a next item (offset to 0)
					m_pListView->m_pTopItem = m_pListView->m_pTopItem->m_pNext;
					m_iTopItemOffset = 0;
				} else {
					// No next item (rather a bug) (offset to the top item size)
					m_iTopItemOffset = m_pListView->m_pTopItem->m_iHeight;
					diff = 0;
				}
			} else {
				// just offset the top item
				m_iTopItemOffset += diff;
				diff = 0;
			}
		}
		while(diff < 0)
		{
			if((-diff) <= m_iTopItemOffset)
			{
				// just move the top item
				m_iTopItemOffset += diff;
				diff = 0;
			} else {
				diff += m_iTopItemOffset;
				if(m_pListView->m_pTopItem->m_pPrev)
				{
					// There is a prev item (offset to 0)
					m_pListView->m_pTopItem = m_pListView->m_pTopItem->m_pPrev;
					m_iTopItemOffset = m_pListView->m_pTopItem->m_iHeight;
				} else {
					// No prev item (rather a bug) (offset to the top item size)
					m_iTopItemOffset = 0;
					diff = 0;
				}
			}
		}
	}
	m_iLastScrollBarVal = newVal;
	update();

}

//void KviUserListViewArea::showOrHideScrollbar()
//{
//	if((m_pListView->m_iTotalHeight + (KVI_USERLIST_BORDER_WIDTH << 1)) > height())
//	{
//		if(!m_pScrollBar->isVisible())
//		{
//			m_pScrollBar->show();
//			m_pScrollBar->setGeometry(width() - KVI_USERLIST_SCROLLBAR_WIDTH,0,KVI_USERLIST_SCROLLBAR_WIDTH,height());
//		}
//	} else {
//		if(m_pScrollBar->isVisible())
//		{
//			m_pScrollBar->hide();
//			m_iLastScrollBarVal = 0;
//		}
//	}
//
//}

void KviUserListViewArea::paintEvent(QPaintEvent *ev)
{
	// update the scroll bar

	if(!isVisible())return;
	int wdth = width() - m_pScrollBar->width();

	QRect r = ev->rect();
	if(r.right() > wdth)r.setRight(wdth);

	QPainter p(m_pMemBuffer);
	p.setFont(KVI_OPTION_FONT(KviOption_fontUserListView));

#ifdef COMPILE_PSEUDO_TRANSPARENCY
	if(g_pShadedChildGlobalDesktopBackground)
	{
		QPoint pnt = mapToGlobal(QPoint(r.left(),r.top()));
		p.drawTiledPixmap(r.left(),r.top(),r.width(),r.height(),*g_pShadedChildGlobalDesktopBackground,pnt.x(),pnt.y());
	} else {
#endif

		QPixmap * pix = KVI_OPTION_PIXMAP(KviOption_pixmapUserListViewBackground).pixmap();
		if(pix)
		{
			p.drawTiledPixmap(r.left(),r.top(),r.width(),r.height(),*pix,r.left(),r.top());
		} else {
			p.fillRect(r.left(),r.top(),r.width(),r.height(),KVI_OPTION_COLOR(KviOption_colorUserListViewBackground));
		}

#ifdef COMPILE_PSEUDO_TRANSPARENCY
	}
#endif

	KviUserListEntry * e = m_pListView->m_pTopItem;

	int theY = KVI_USERLIST_BORDER_WIDTH - m_iTopItemOffset;

	time_t curTime = time(0);


	bool bShowIcons = KVI_OPTION_BOOL(KviOption_boolShowUserChannelIcons);
	bool bShowState = KVI_OPTION_BOOL(KviOption_boolShowUserChannelState);

	int theX;
	int bottom;

	while(e && theY < r.bottom())
	{
		bottom = theY + e->m_iHeight;
		if(bottom >= r.top())
		{

			theX = KVI_USERLIST_BORDER_WIDTH;
	
			if(e->m_bSelected)
			{
			    p.fillRect(0,theY,wdth,e->m_iHeight,KVI_OPTION_COLOR(KviOption_colorUserListViewSelectionBackground));
				p.setPen(KVI_OPTION_COLOR(KviOption_colorUserListViewSelectionForeground));
			} else {
				if(e->m_iFlags == 0)p.setPen(KVI_OPTION_COLOR(KviOption_colorUserListViewNormalForeground));
				else {
					p.setPen(KVI_OPTION_COLOR((e->m_iFlags & KVI_USERFLAG_OP) ? KviOption_colorUserListViewOpForeground : \
							((e->m_iFlags & KVI_USERFLAG_HALFOP) ? KviOption_colorUserListViewHalfOpForeground : KviOption_colorUserListViewVoiceForeground)));
				}
		    }
	
			if(bShowState)
			{
				if(e->m_lastActionTime)
				{
					// the g_pUserChanStatePixmap is 36 x 80 pixels
					// divided into 6 rows of 5 pixmaps
					// row 0 is hot , row 5 is cold
					// left is most active , right is least active
					// e->m_iTemperature is a signed short , negative values are cold
					// e->m_lastActionTime is the time of the last action (eventually 0 , if not known)
					// 6 bit right shift is an aprox division for 64 : really aprox minutes
					unsigned int uTimeDiff = (((unsigned int)(curTime - e->m_lastActionTime)) >> 6);
					if(uTimeDiff < 16)
					{
						p.drawRect(theX,theY + 2,10,e->m_iHeight - 4);
						static int xOffTable[16] =
						{
							0  , 8  , 16 , 16 ,
							24 , 24 , 24 , 24 ,
							32 , 32 , 32 , 32 ,
							32 , 32 , 32 , 32
						};
						// the temperature now
						// temp > 100 is hot (offset y = 0)
						// temp < -100 is cold (offset y = 80)
						// temp > 30 is half-hot (offset y = 16)
						// temp < -30 is half-cold (offset y = 64)
						// temp > 0 is a-bit-hot (offset y = 32)
						// temp < 0 is a-bit-cold (offset y = 48)

						if(e->m_iTemperature > KVI_MID_TEMPERATURE)
						{
							if(e->m_iTemperature > KVI_HALF_HOT_TEMPERATURE)
							{
								if(e->m_iTemperature > KVI_HOT_TEMPERATURE)
								{
									// hot
									p.drawPixmap(theX + 1,theY + 3,*g_pUserChanStatePixmap,xOffTable[uTimeDiff],0,8,16);
								} else {
									// half-hot
									p.drawPixmap(theX + 1,theY + 3,*g_pUserChanStatePixmap,xOffTable[uTimeDiff],16,8,16);
								}
							} else {
								// bit-hot
								p.drawPixmap(theX + 1,theY + 3,*g_pUserChanStatePixmap,xOffTable[uTimeDiff],32,8,16);
							}
						} else {
							if(e->m_iTemperature < KVI_HALF_COLD_TEMPERATURE)
							{
								if(e->m_iTemperature < KVI_COLD_TEMPERATURE)
								{
									// cold
									p.drawPixmap(theX + 1,theY + 3,*g_pUserChanStatePixmap,xOffTable[uTimeDiff],80,8,16);
								} else {
									// half-cold
									p.drawPixmap(theX + 1,theY + 3,*g_pUserChanStatePixmap,xOffTable[uTimeDiff],64,8,16);
								}
							} else {
								// bit-cold
								p.drawPixmap(theX + 1,theY + 3,*g_pUserChanStatePixmap,xOffTable[uTimeDiff],48,8,16);
							}
						} 
					}
				}
				theX += 11;
			}
	
			if(bShowIcons)
			{
				p.drawRect(theX,theY + 2,18,e->m_iHeight - 4);
				if(e->m_iFlags != 0)
				{
					QPixmap * ico = g_pIconManager->getSmallIcon((e->m_iFlags & KVI_USERFLAG_OP) ? KVI_SMALLICON_OP : \
								((e->m_iFlags & KVI_USERFLAG_HALFOP) ? KVI_SMALLICON_HALFOP :  KVI_SMALLICON_VOICE));
					p.drawPixmap(theX + 1,theY + 3,*ico);
				}
				theX += 19;
			}
	
			theX++;
	
			int partialY = 2;
			KviAvatar * av = e->m_pGlobalData->avatar();
			if(av)
			{
				QPixmap * pix = KVI_OPTION_BOOL(KviOption_boolScaleAvatars) ? \
						av->scaledPixmap(KVI_OPTION_UINT(KviOption_uintAvatarScaleWidth),KVI_OPTION_UINT(KviOption_uintAvatarScaleHeight)) : \
						av->pixmap();
				p.drawPixmap(theX,theY + 2,*pix);
	
				partialY += (pix->height());
				//e->m_iHeight = pix->height() + 4 + m_pListView->m_iFontHeight;
			} //else e->m_iHeight = 4 + m_pListView->m_iFontHeight;
	
			p.drawText(theX,theY + partialY,wdth - theX,m_pListView->m_iFontHeight,AlignLeft|AlignVCenter,e->m_szNick.ptr());
		}
	
		theY = bottom;
		e = e->m_pNext;
	}

	p.setPen(colorGroup().dark());
	p.drawLine(0,0,wdth,0);
	p.drawLine(0,0,0,height());
	p.setPen(colorGroup().light());
	p.drawLine(1,height()-1,wdth,height()-1);
	p.drawLine(wdth - 1,1,wdth - 1,height());

	bitBlt(this,r.left(),r.top(),m_pMemBuffer,r.left(),r.top(),r.width(),r.height(),Qt::CopyROP,false);
}

void KviUserListViewArea::resizeEvent(QResizeEvent *)
{
	m_pMemBuffer->resize(width(),height());
	int iScr = m_pScrollBar->sizeHint().width();
	m_pScrollBar->setGeometry(width() - iScr,0,iScr,height());
	m_pScrollBar->setPageStep(height());
	m_pScrollBar->setLineStep(m_pListView->m_iFontHeight - 1);
}

void KviUserListViewArea::mousePressEvent(QMouseEvent *e)
{
	if(e->button() & LeftButton)
	{
		KviUserListEntry * entry = m_pListView->itemAt(e->pos());
		if(entry)
		{
			if(e->state() & ShiftButton)
			{
				// Multiselect mode
				entry->m_bSelected = true;
				update();
			} else if(e->state() & ControlButton)
			{
				// Invert mode
				entry->m_bSelected = ! entry->m_bSelected;
				update();
			} else {
				// Single select mode
				KviUserListEntry * aux = m_pListView->m_pHeadItem;
				while(aux)
				{
					aux->m_bSelected = false;
					aux = aux->m_pNext;
				}
				entry->m_bSelected = true;
				update();
			}
		}
		m_pLastEntryUnderMouse = entry;
	} else if(e->button() & RightButton)
	{
		KviUserListEntry * entry = m_pListView->itemAt(e->pos());
		if(entry)
		{
			entry->m_bSelected = true;
			update();
		}
		m_pListView->emitRightClick();
	}
}

void KviUserListViewArea::mouseDoubleClickEvent(QMouseEvent *e)
{
	m_pListView->emitDoubleClick();
}

void KviUserListViewArea::mouseMoveEvent(QMouseEvent *e)
{
	if(e->state() & LeftButton)
	{
		KviUserListEntry * entry = m_pListView->itemAt(e->pos());
		if(entry && (entry != m_pLastEntryUnderMouse))
		{
			if(e->state() & ControlButton)entry->m_bSelected = ! entry->m_bSelected;
			else entry->m_bSelected = true;
			update();
			m_pLastEntryUnderMouse = entry;
		} else {
			// out of the widget ?
			if(entry == m_pLastEntryUnderMouse)return;
			if(e->pos().y() < KVI_USERLIST_BORDER_WIDTH)
			{
				KviUserListEntry * top = m_pListView->m_pTopItem;
				if(top)
				{
					m_pScrollBar->setValue(m_pScrollBar->value() - top->m_iHeight);
					if(m_pListView->m_pTopItem != top)
					{
						if(e->state() & ControlButton)m_pListView->m_pTopItem->m_bSelected = ! m_pListView->m_pTopItem->m_bSelected;
						else m_pListView->m_pTopItem->m_bSelected = true;
						update();
					}
				}
				m_pLastEntryUnderMouse = top;
			} else if(e->pos().y() > (height() - KVI_USERLIST_BORDER_WIDTH))
			{
				KviUserListEntry * bottom = m_pListView->m_pTopItem;
				if(bottom)
				{
					int theY = KVI_USERLIST_BORDER_WIDTH - m_iTopItemOffset;
					while(bottom && (theY < height()))
					{
						theY+= bottom->m_iHeight;
						bottom = bottom->m_pNext;
					}
					if(!bottom)bottom = m_pListView->m_pTailItem;
					if(bottom)
					{
						m_pScrollBar->setValue(m_pScrollBar->value() + bottom->m_iHeight);
						if(bottom != m_pLastEntryUnderMouse)
						{
							if(e->state() & ControlButton)bottom->m_bSelected = ! bottom->m_bSelected;
							else bottom->m_bSelected = true;
							update();
						}
					}
				}
				m_pLastEntryUnderMouse = bottom;
			} else m_pLastEntryUnderMouse = 0;
		}
	}
}

void KviUserListViewArea::mouseReleaseEvent(QMouseEvent *)
{
	m_pLastEntryUnderMouse = 0;
}




#include "kvi_userlistview.moc"
