//
//   File : kvi_taskbar.cpp
//   Last major modification : Thu Jan 7 1999 03:59:43 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
#define __KVIRC__
#define KVI_TASKBARBUTTON_MIN_WIDTH 100
#define KVI_TASKBARBUTTON_CONTEXTINDICATORHEIGHT 6
#define KVI_TASKBAR_MIN_WIDTH 120

#define _KVI_DEBUG_CHECK_RANGE_
#include "kvi_debug.h"
#include "kvi_taskbar.h"
#include "kvi_frame.h"
#include "kvi_window.h"
#include "kvi_string.h"
#include "kvi_options.h"
#include "kvi_console.h"
#include "kvi_locale.h"
#include "kvi_dynamictooltip.h"
#include "kvi_settings.h"
#include "kvi_channel.h"

// FIXME: #warning "The tree taskbar min width should be configurable"

#include <qfontmetrics.h>
#include <qpainter.h>
#include <qpixmap.h>
#include <qheader.h>

#ifdef COMPILE_PSEUDO_TRANSPARENCY
	extern QPixmap * g_pShadedChildGlobalDesktopBackground;
#endif


//////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// KviTaskBarBase
//

//: QToolBar(__tr("Taskbar"),lpFrm,QMainWindow::Bottom,false,"taskbar")
//: KToolBar(lpFrm,QMainWindow::Bottom,false,0,false,false)

KviTaskBarBase::KviTaskBarBase(KviFrame * lpFrm)
#if QT_VERSION >= 300
: KviTalToolBar(__tr("Taskbar"),lpFrm,Qt::DockBottom,false,"taskbar")
#else
: KviTalToolBar(__tr("Taskbar"),lpFrm,QMainWindow::Bottom,false,"taskbar")
#endif
{
	m_pFrm = lpFrm;
}

KviTaskBarBase::~KviTaskBarBase()
{
}

void KviTaskBarBase::switchWindow(bool bNext,bool bInContextOnly)
{
	if(!m_pFrm->activeWindow())return;

	KviConsole * cons = m_pFrm->activeWindow()->console();

	KviTaskBarItem * cur = m_pFrm->activeWindow()->taskBarItem();
	KviTaskBarItem * it = cur;

	if(!setIterationPointer(cur))return;

	while(it)
	{
		it = bNext ? nextItem() : prevItem();
		if(!it)it = bNext ? firstItem() : lastItem();
		if(it)
		{
			if(bInContextOnly)
			{
				if(it->window()->console() == cons)
				{
					m_pFrm->setActiveWindow(it->window());
					return;
				}
			} else {
				m_pFrm->setActiveWindow(it->window());
				return;
			}
		}
		if(it == cur)return; // did a complete loop.... no window to switch to
	}
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// KviTaskBarItem
//

KviTaskBarItem::KviTaskBarItem(KviWindow * wnd)
{
	m_pWindow = wnd;
}

KviTaskBarItem::~KviTaskBarItem()
{
	m_pWindow->m_pTaskBarItem = 0;
}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// KviTaskBarButton
//


KviTaskBarButton::KviTaskBarButton(QWidget * par,KviWindow * wnd,const char * name)
: QWidget(par,name) , KviTaskBarItem(wnd)
{
	m_iHighlightLevel = 0;
	m_bActive         = false;
	m_iProgress       = -1;
	m_pTip            = new KviDynamicToolTip(this);
	connect(m_pTip,SIGNAL(tipRequest(KviDynamicToolTip *,const QPoint &)),this,SLOT(tipRequest(KviDynamicToolTip *,const QPoint &)));
	setBackgroundMode(QWidget::NoBackground);
}

KviTaskBarButton::~KviTaskBarButton()
{
	delete m_pTip; //not necessary ?
}

void KviTaskBarButton::tipRequest(KviDynamicToolTip *,const QPoint &pnt)
{
	m_pTip->tip(rect(),m_pWindow->plainTextCaption());
}

void KviTaskBarButton::mousePressEvent(QMouseEvent *e)
{
	if(e->button() & LeftButton)
	{
		if((m_pWindow->frame()->activeWindow() != m_pWindow)  || (m_pWindow->isMinimized()))m_pWindow->frame()->setActiveWindow(m_pWindow);
		else m_pWindow->minimize();
	} else m_pWindow->contextPopup();
}

#if QT_VERSION >= 300
void KviTaskBarButton::contextMenuEvent(QContextMenuEvent *e)
{
	m_pWindow->contextPopup();
	e->accept();
}
#endif

void KviTaskBarButton::setActive(bool bActive)
{
	if(bActive)
	{
		m_bActive         = true;
		m_iHighlightLevel = 0;
	} else {
		if(m_bActive)
		{
			// was active... unset the highlighting if it was silently turned on while being active...
			m_iHighlightLevel = 0;
		}
		m_bActive      = false;
	}
	update();
}

static void draw_frame_helper(QPainter * p,int x1,int y1,int x2,int y2,QColor light,QColor dark)
{
	p->setPen(light);
	p->drawLine(x1,y1,x2,y1);
	p->drawLine(x1,y1,x1,y2);

	p->setPen(dark);
	p->drawLine(x1,y2,x2,y2);
	p->drawLine(x2,y1,x2,y2);
}

//   ##################  #####################
//   #@@@@@@@@@@@@@@@@@  #@@@@@@@@@@@@@@@@@@@@
//   #@                  #@
//   #@ ***********      #@ ************* <-|
//   #@ ***********      #@ *************   |
//   #@ ***********      #@ *************    16
//   #@ ***********      #@ *************   |
//   #@ ***********      #@ *************   |
//   #@ ***********      #@ ************* <-|
//   #@                  #@
//   #@@@@@@@@@@@@@@@@@  #@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
//   ##################  #++++++++++++++++++++   <-|
//                       #+                        KVI_TASKBARBUTTON_CONTEXTINDICATORHEIGHT
//                       #++++++++++++++++++++   <-|
//                       ####################################
//
//
//

/*
#include <qbitmap.h>

static unsigned char filler_bits[]=
{
	0x08 , 0x55 ,
	0x01 , 0x2F
};
*/

void KviTaskBarButton::paintEvent(QPaintEvent *e)
{
	QPainter p(this);

	draw_frame_helper(&p,0,0,width() - 1,height() - 1,
		m_pWindow->mdiParent() ? (m_bActive ? colorGroup().shadow() : colorGroup().midlight()) : colorGroup().background(),
		m_pWindow->mdiParent() ? (m_bActive ? colorGroup().midlight() : colorGroup().shadow()) : colorGroup().background());

	int h = height();

	if(KVI_OPTION_BOOL(KviOption_boolUseTaskBarIrcContextIndicator))
	{
		h -= KVI_TASKBARBUTTON_CONTEXTINDICATORHEIGHT;
		QColor base = colorGroup().background();
		if(m_pWindow->console())
		{
			QColor cntx = KVI_OPTION_ICCOLOR(m_pWindow->console()->ircContextId() % KVI_NUM_ICCOLOR_OPTIONS);
			base.setRgb((base.red() + cntx.red()) >> 1,(base.green() + cntx.green()) >> 1,
				(base.blue() + cntx.blue()) >> 1);
			p.fillRect(2,h,width() - 4,KVI_TASKBARBUTTON_CONTEXTINDICATORHEIGHT - 2,base);
		} else {
			p.fillRect(2,h,width() - 4,KVI_TASKBARBUTTON_CONTEXTINDICATORHEIGHT - 2,colorGroup().brush(QColorGroup::Background));
		}

		draw_frame_helper(&p,1,h - 1,width() - 2,height() - 2,
			m_bActive ? base.dark() : base.light(180),
			m_bActive ? base.light(180) : base.dark());
	}

	draw_frame_helper(&p,1,1,width() - 2,h - 2,
		m_bActive ? colorGroup().dark() : colorGroup().light(),
		m_bActive ? colorGroup().light() : colorGroup().dark());

	p.fillRect(2,2,width() - 4,h - 4,colorGroup().brush(QColorGroup::Background));

	p.drawPixmap(3,3,*(m_pWindow->myIconPtr()));

	QRect cRect(20,3,width() - 40,h - 6);

	if(m_iProgress >= 0)
	{
		// paint the progress bar
		int wdth = (m_iProgress * cRect.width()) / 100;
		p.setPen(KVI_OPTION_COLOR(KviOption_colorTaskBarProgressBar));
		p.drawRect(cRect);
		p.fillRect(20,3,wdth,cRect.height(),KVI_OPTION_COLOR(KviOption_colorTaskBarProgressBar));
	}

	QRect bRect;
//	cRect.setWidth(cRect.width());
//	cRect.setWidth(cRect.width() - 13);

	bool bMinimized = m_pWindow->isMinimized();

	if(m_bActive)
	{
		p.setPen(KVI_OPTION_COLOR(KviOption_colorTaskBarNormalText));
	} else {
		int iLevel;
	
		switch(m_iHighlightLevel)
		{
			case 0: iLevel = bMinimized ? KviOption_colorTaskBarMinimizedText : KviOption_colorTaskBarNormalText; break;
			case 1: iLevel = KviOption_colorTaskBarHighlight1Text; break;
			case 2: iLevel = KviOption_colorTaskBarHighlight2Text; break;
			case 3: iLevel = KviOption_colorTaskBarHighlight3Text; break;
			case 4: iLevel = KviOption_colorTaskBarHighlight4Text; break;
			default: iLevel = KviOption_colorTaskBarHighlight5Text; break;
		}
		p.setPen(KVI_OPTION_COLOR(iLevel));
	}


	if(bMinimized)
	{
		KviStr tmp(KviStr::Format,"(%s)",m_pWindow->plainTextCaption());
		p.drawText(cRect,AlignLeft | AlignVCenter,tmp.ptr(),-1,&bRect);
	} else {
		p.drawText(cRect,AlignLeft | AlignVCenter,m_pWindow->plainTextCaption(),-1,&bRect);
	}
/*
	if(bRect.width() > cRect.width())
	{
		QBitmap bm(16,2,filler_bits,false);
		QBrush br(colorGroup().color(QColorGroup::Background),bm);
		p.fillRect(cRect.right() - 16,cRect.y(),16,cRect.height(),br);
	}
*/
	if(bRect.width() > cRect.width())
	{
		//QRect old = p.clipRect();
		p.setClipRect(cRect.right(),cRect.y(),10,cRect.height());
		QColor base = p.pen().color();
		QColor bg   = colorGroup().color(QColorGroup::Background);
		base.setRgb((base.red() + bg.red()) / 2,(base.green() + bg.green()) / 2,(base.blue() + bg.blue()) / 2);
		p.setPen(base);
		cRect.setWidth(cRect.width() + 10);
		p.drawText(cRect,AlignLeft | AlignVCenter,m_pWindow->plainTextCaption(),-1);
		p.setClipRect(cRect.right(),cRect.y(),5,cRect.height());
		base.setRgb((base.red() + bg.red()) / 2,(base.green() + bg.green()) / 2,(base.blue() + bg.blue()) / 2);
		p.setPen(base);
		cRect.setWidth(cRect.width() + 10);
		p.drawText(cRect,AlignLeft | AlignVCenter,m_pWindow->plainTextCaption(),-1);
		//p.setClipRect(old);
//		p.drawText(cRect.right(),cRect.y(),15,cRect.height(),AlignLeft | AlignVCenter,"...");
	}
}


//bool KviTaskBarButton::active()
//{
//	return m_bActive;
//}

//int KviTaskBarButton::highlightLevel()
//{
//	return m_iHighlightLevel;
//}

//bool KviTaskBarButton::altColor()
//{
//	return m_bAltColor;
//}

void KviTaskBarButton::captionChanged()
{
	update();
}

void KviTaskBarButton::setProgress(int progress)
{
	if(progress == m_iProgress)return;
	m_iProgress = progress;
	update(); // repaint(false) ?
}

void KviTaskBarButton::highlight(int iLevel)
{
	if(iLevel <= m_iHighlightLevel)return;
//	if(m_bHighlighted && (m_bAltColor || (bAltColor == m_bAltColor)))return;
	if(m_bActive && m_pWindow->frame()->isActiveWindow())return;
	m_iHighlightLevel = iLevel;
//	m_bAltColor = bAltColor;
	if(m_pWindow->frame()->dockExtension())m_pWindow->frame()->dockExtension()->refresh();
	if(m_bActive)return;
	update(); // repaint(false) ?
}



//////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// KviTaskBarToolTip
//
/*
KviTaskBarToolTip::KviTaskBarToolTip(KviTaskBarButton * b)
: QToolTip(b,0)
{
	m_pTaskBarButton = b;
}

KviTaskBarToolTip::~KviTaskBarToolTip()
{
}

void KviTaskBarToolTip::maybeTip(const QPoint &pnt)
{
	tip(m_pTaskBarButton->rect(),m_pTaskBarButton->m_pWindow->plainTextCaption());
}
*/

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// KviClasicTaskBar
//


KviClassicTaskBar::KviClassicTaskBar(KviFrame * lpFrm)
: KviTaskBarBase(lpFrm)
{
	m_pButtonList = new KviPtrList<KviTaskBarButton>;
	m_pButtonList->setAutoDelete(true);

	calcButtonHeight();

	m_pBase = new QWidget(this);
//	m_pBase->setBackgroundMode(NoBackground);

	setStretchableWidget(m_pBase);

	setVerticalStretchable(true);
	setHorizontalStretchable(true);

	m_pBase->setMinimumWidth(KVI_TASKBAR_MIN_WIDTH);
}

KviClassicTaskBar::~KviClassicTaskBar()
{
	delete m_pButtonList;
	m_pButtonList = 0;
}

void KviClassicTaskBar::fontChange(const QFont &old)
{
	calcButtonHeight();
	doLayout();
	QToolBar::fontChange(old);
}

void KviClassicTaskBar::calcButtonHeight()
{
	QFontMetrics fm(font());
	m_iButtonHeight = fm.lineSpacing() + 6;
	if(m_iButtonHeight < 22)m_iButtonHeight = 22;
	if(KVI_OPTION_BOOL(KviOption_boolUseTaskBarIrcContextIndicator))
		m_iButtonHeight += KVI_TASKBARBUTTON_CONTEXTINDICATORHEIGHT;
}

void KviClassicTaskBar::insertButton(KviTaskBarButton * b)
{
	int idx = 0;
//	if(KVI_OPTION_BOOL(KviOption_boolSortTaskbarButtons))
//	{
		// first sort by irc context
		for(KviTaskBarButton * btn = m_pButtonList->first();btn;btn = m_pButtonList->next())
		{
			if(btn->window()->console() == b->window()->console())
			{
				// same irc context (or none)
				// sort by type now
				for(;btn;btn = m_pButtonList->next())
				{
					if((btn->window()->type() > b->window()->type()) ||
						(btn->window()->console() != b->window()->console()))
					{
						// greater type or another irc context
						m_pButtonList->insert(idx,b);
						return;
					} else if(btn->window()->type() == b->window()->type())
					{
						// same type!
						// sort by name
						if(kvi_strcmpCI(btn->window()->name(),b->window()->name()) > 0)
						{
							// got a "higher one"
							m_pButtonList->insert(idx,b);
							return;
						}
					}
					idx++;
				}
				// ran out of buttons
				m_pButtonList->append(b);
				return;
			} else {
				if(!(btn->window()->console()) && b->window()->console())
				{
					// this must be a new console...insert before the contextless windows
					__range_valid(b->window()->console() == b->window());
					m_pButtonList->insert(idx,b);
					return;
				} else idx++; // wrong irc contet...go on searching
			}
		}
//	}
	// no sorting selected , or no match for this irc context
	m_pButtonList->append(b);
}


KviTaskBarItem * KviClassicTaskBar::addItem(KviWindow * wnd)
{
	KviStr tmp(KviStr::Format,"taskbar_button_%s",wnd->name());
	KviTaskBarButton * b = new KviTaskBarButton(m_pBase,wnd,tmp.ptr());
	insertButton(b);
	b->show();
	doLayout();
	if(m_pFrm->dockExtension())m_pFrm->dockExtension()->refresh();
	return b;
}

bool KviClassicTaskBar::removeItem(KviTaskBarItem * it)
{
	m_pButtonList->removeRef((KviTaskBarButton *)it);
	doLayout();
	if(m_pFrm->dockExtension())m_pFrm->dockExtension()->refresh();
	return true;
}

void KviClassicTaskBar::setActiveItem(KviTaskBarItem * it)
{
	for(KviTaskBarButton * b = m_pButtonList->first();b;b = m_pButtonList->next())
	{
		b->setActive(((KviTaskBarButton *)it) == b);
	}
	if(m_pFrm->dockExtension())m_pFrm->dockExtension()->refresh();

}


void KviClassicTaskBar::doLayout()
{
	if(!m_pButtonList->count())return;

	int btnsInRow = m_pBase->width() / KVI_TASKBARBUTTON_MIN_WIDTH;
	if(btnsInRow < 1)btnsInRow = 1;
	int totCount = m_pButtonList->count();
	int rows = 0;
	while(totCount > 0)
	{
		rows++;
		totCount -= btnsInRow;
	}

	m_pBase->setMinimumHeight(rows * m_iButtonHeight);

	int baseWidth     = m_pBase->width();
	int theWidth      = 0;
	int theX          = 0;
	int theY          = -m_iButtonHeight;
	totCount          = m_pButtonList->count();
	int btnIdx        = 0;
	int btnInRow      = 1;
	for(KviTaskBarButton * b = m_pButtonList->first();b;b = m_pButtonList->next())
	{
		if((btnIdx % btnsInRow) == 0)
		{
			int inRow = ((totCount < btnsInRow ) ? totCount : btnsInRow);
			theWidth = baseWidth / (inRow ? inRow : 1);
			theX     = 0;
			theY    += m_iButtonHeight;
			btnInRow = 1;
		} else {
			if((btnInRow == btnsInRow) || (totCount == 1))theWidth = baseWidth - theX;
		}

		b->setGeometry(theX,theY,theWidth,m_iButtonHeight);

		if(btnInRow != btnsInRow)
		{
			theX += theWidth;
			btnInRow++;
		}
		btnIdx++;
		totCount--;
	}
}

void KviClassicTaskBar::resizeEvent(QResizeEvent *e)
{
	QToolBar::resizeEvent(e);
	doLayout();
}

inline KviTaskBarItem * KviClassicTaskBar::firstItem()
{
	return m_pButtonList->first();
}

inline KviTaskBarItem * KviClassicTaskBar::lastItem(void)
{
	return m_pButtonList->last();
}

inline KviTaskBarItem * KviClassicTaskBar::nextItem()
{
	return m_pButtonList->next();
}

inline KviTaskBarItem * KviClassicTaskBar::prevItem(void)
{
	return m_pButtonList->prev();
}

inline bool KviClassicTaskBar::setIterationPointer(KviTaskBarItem * it)
{
	return (m_pButtonList->findRef((const KviTaskBarButton *)it) != -1);
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// KviTreeTaskBarItem
//


KviTreeTaskBarItem::KviTreeTaskBarItem(QListView * par,KviWindow * wnd)
: QListViewItem(par) , KviTaskBarItem(wnd)
{
//	m_bHighlighted = false;
//	m_bAltColor = false;
	m_iHighlightLevel = 0;
	m_iProgress = -1;
}

KviTreeTaskBarItem::KviTreeTaskBarItem(KviTreeTaskBarItem * par,KviWindow * wnd)
: QListViewItem(par) , KviTaskBarItem(wnd)
{
//	m_bHighlighted = false;
//	m_bAltColor = false;
//	m_bActive = false;
	m_iHighlightLevel = 0;
	m_iProgress = -1;
}

KviTreeTaskBarItem::~KviTreeTaskBarItem()
{
}

void KviTreeTaskBarItem::captionChanged()
{
	// FIXME: can we do better ?
	repaint();
}

void KviTreeTaskBarItem::highlight(int iLevel)
{
//	if(m_bHighlighted && (m_bAltColor || (bAltColor == m_bAltColor)))return;
//	debug("Highlight %d , (%d)",iLevel,m_iHighlightLevel);
	if(iLevel <= m_iHighlightLevel)return;
	if(isSelected() && m_pWindow->frame()->isActiveWindow())return;
//	m_bHighlighted = true;
//	m_bAltColor = bAltColor;
	m_iHighlightLevel = iLevel;
	if(m_pWindow->frame()->dockExtension())m_pWindow->frame()->dockExtension()->refresh();
	if(isSelected())return;
//	debug("And going to repaint");
	repaint(); // repaint(false) ?
}

void KviTreeTaskBarItem::setProgress(int progress)
{
	if(progress == m_iProgress)return;
	m_iProgress = progress;
	repaint(); // repaint(false) ?
}

void KviTreeTaskBarItem::setActive(bool bActive)
{
	if(bActive)
	{
//		m_bHighlighted = false;
//		m_bAltColor    = false;
		m_iHighlightLevel = 0;
		// was not selected: the listView will repaint it
	} else {
		if(isSelected())
		{
			// was active... unset the highlighting if it was silently turned on while being active...
//			m_bHighlighted = false;
//			m_bAltColor = false;
			m_iHighlightLevel = 0;
			// was selected: the list view will repaint it
		}
	}
}

void KviTreeTaskBarItem::paintBranches(QPainter *p,const QColorGroup &,int w,int y,int h,GUIStyle s)
{
//	QRect myRect = listView()->itemRect(this);
//    ((KviTreeTaskBarListView *)listView())->paintEmptyAreaInternal(p,
//			QRect(myRect.topLeft().x(),myRect.bottomLeft().y() + 1,w,totalHeight() - myRect.height()),
//			QRect(0,y,w,totalHeight() - myRect.height()));
	((KviTreeTaskBarListView *)listView())->paintEmptyArea(p,QRect(0,y,w,totalHeight() - height()));
}

void KviTreeTaskBarItem::paintCell(QPainter *p,const QColorGroup &cg,int column,int width,int)
{
//#warning "GOT A SIGSEGV HERE"

	if(isSelected())
	{
		p->fillRect(0,0,width,height(),KVI_OPTION_COLOR(KviOption_colorTreeTaskBarActiveBackground));
	} else {

#ifdef COMPILE_PSEUDO_TRANSPARENCY
		if(g_pShadedChildGlobalDesktopBackground)
		{
			QPoint pnt = listView()->viewport()->mapToGlobal(QPoint(int(p->worldMatrix().dx()),int(p->worldMatrix().dy())));
			p->drawTiledPixmap(0,0,width,height(),*g_pShadedChildGlobalDesktopBackground,pnt.x(),pnt.y());
		} else {
#endif
			QPixmap * pix = KVI_OPTION_PIXMAP(KviOption_pixmapTreeTaskBarBackground).pixmap();
			if(pix)
			{
				QPoint pnt = listView()->viewportToContents(QPoint(int(p->worldMatrix().dx()),int(p->worldMatrix().dy())));
				p->drawTiledPixmap(0,0,width,height(),*pix,pnt.x(),pnt.y());
			} else {
				p->fillRect(0,0,width,height(),KVI_OPTION_COLOR(KviOption_colorTreeTaskBarBackground));
			}
#ifdef COMPILE_PSEUDO_TRANSPARENCY
		}
#endif
	}

	int h = height();

	int im = listView()->itemMargin();

	int yPixmap = (h - 16) >> 1;

	const char * pText;

	QRect cRect(im + 20,0,width - ((im << 1) + 20),height());

	switch(m_pWindow->type())
	{
		case KVI_WINDOW_TYPE_CONSOLE:
			//debug("HI");
			if(KVI_OPTION_BOOL(KviOption_boolUseTaskBarIrcContextIndicator))
			{
				QColor base = cg.background();
				QColor cntx = KVI_OPTION_ICCOLOR(m_pWindow->console()->ircContextId() % KVI_NUM_ICCOLOR_OPTIONS);
				base.setRgb((base.red() + cntx.red()) >> 1,(base.green() + cntx.green()) >> 1,
					(base.blue() + cntx.blue()) >> 1);
				p->fillRect(im + 2,yPixmap + 1,14,15,base);
				draw_frame_helper(p,im + 1,yPixmap,im + 15,yPixmap + 15,base.light(180),base.dark());
				cRect.setLeft(cRect.left() + 20);
				p->drawPixmap(im + 20,yPixmap,*(m_pWindow->myIconPtr()));
			} else {
				p->drawPixmap(im,yPixmap,*(m_pWindow->myIconPtr()));
			}
			pText = ((KviConsole *)m_pWindow)->statusString();
		break;
		case KVI_WINDOW_TYPE_CHANNEL:
			pText = ((KviChannel *)m_pWindow)->nameWithUserFlag();
			p->drawPixmap(im,yPixmap,*(m_pWindow->myIconPtr()));
		break;
		case KVI_WINDOW_TYPE_QUERY:
			pText = m_pWindow->name();
			p->drawPixmap(im,yPixmap,*(m_pWindow->myIconPtr()));
		break;
		default:
			pText = m_pWindow->plainTextCaption();
			p->drawPixmap(im,yPixmap,*(m_pWindow->myIconPtr()));
		break;
	}

	if(m_iProgress >= 0)
	{
		// paint the progress bar
		int wdth = (m_iProgress * cRect.width()) / 100;
		p->fillRect(cRect.x(),cRect.y(),wdth,cRect.height(),KVI_OPTION_COLOR(KviOption_colorTreeTaskBarProgress));
	}

	if(isSelected())
	{
//		debug("SELECTED paint");
		p->setPen(KVI_OPTION_COLOR(KviOption_colorTreeTaskBarActiveForeground));
	} else {
		int iLevel;
//		debug("Non active paint %d",m_iHighlightLevel);
		switch(m_iHighlightLevel)
		{
			case 0: iLevel = KviOption_colorTreeTaskBarForeground; break;
			case 1: iLevel = KviOption_colorTreeTaskBarHighlight1Foreground; break;
			case 2: iLevel = KviOption_colorTreeTaskBarHighlight2Foreground; break;
			case 3: iLevel = KviOption_colorTreeTaskBarHighlight3Foreground; break;
			case 4: iLevel = KviOption_colorTreeTaskBarHighlight4Foreground; break;
			default: iLevel = KviOption_colorTreeTaskBarHighlight5Foreground; break;
		}
		p->setPen(KVI_OPTION_COLOR(iLevel));
	}


	if(m_pWindow->isMinimized())
	{
		KviStr tmp(KviStr::Format,"(%s)",pText);
		p->drawText(cRect,AlignLeft | AlignVCenter,tmp.ptr(),-1,0);
	} else {
		p->drawText(cRect,AlignLeft | AlignVCenter,pText,-1,0);
	}
}

//void KviTreeTaskBarItem::setup()
//{
//	QListViewItem::setup();
//	int h = height();
//	if(h < 18)setHeight(18);
//}

QString KviTreeTaskBarItem::key(int,bool) const
{
	QString ret = m_pWindow->typeString();
	ret.append(m_pWindow->name());
	//debug("key: %s",ret.latin1());
	return ret;
}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// KviTreeTaskBarListView
//


KviTreeTaskBarListView::KviTreeTaskBarListView(QWidget * par)
: QListView(par)
{
	//setSorting(0);
	setItemMargin(2);
}

KviTreeTaskBarListView::~KviTreeTaskBarListView()
{
}

void KviTreeTaskBarListView::contentsMousePressEvent(QMouseEvent *e)
{
	QListViewItem * it = itemAt(contentsToViewport(e->pos()));
	if(it)
	{
		if(e->button() & LeftButton)emit leftMousePress(it);
		else if(e->button() & RightButton)emit rightMousePress(it);
	}
}

void KviTreeTaskBarListView::resizeEvent(QResizeEvent *e)
{
	QListView::resizeEvent(e);
	setColumnWidth(0,viewport()->width());
	resizeContents(viewport()->width(),contentsHeight());
}

//void KviTreeTaskBarListView::paintEmptyAreaInternal(QPainter * p,const QRect &viewportRect,const QRect &painterRect)
//{
//
//}

void KviTreeTaskBarListView::paintEmptyArea(QPainter * p,const QRect &rct)
{
//	paintEmptyAreaInternal(p,rct,rct);
#ifdef COMPILE_PSEUDO_TRANSPARENCY
	if(g_pShadedChildGlobalDesktopBackground)
	{


		QPoint pnt = viewport()->mapToGlobal(QPoint(rct.x() + int(p->worldMatrix().dx()),rct.y() + int(p->worldMatrix().dy())));
		p->drawTiledPixmap(rct.x(),rct.y(),rct.width(),rct.height(),*g_pShadedChildGlobalDesktopBackground,pnt.x(),pnt.y());
	} else {
#endif
		QPixmap * pix = KVI_OPTION_PIXMAP(KviOption_pixmapTreeTaskBarBackground).pixmap();
		if(pix)
		{
			QPoint pnt = viewportToContents(QPoint(rct.x() + int(p->worldMatrix().dx()),rct.y() + int(p->worldMatrix().dy())));
			p->drawTiledPixmap(rct.x(),rct.y(),rct.width(),rct.height(),*pix,pnt.x(),pnt.y());
		} else {
			p->fillRect(rct.x(),rct.y(),rct.width(),rct.height(),KVI_OPTION_COLOR(KviOption_colorTreeTaskBarBackground));
		}
#ifdef COMPILE_PSEUDO_TRANSPARENCY
	}
#endif
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// KviTreeTaskBar
//

KviTreeTaskBar::KviTreeTaskBar(KviFrame * lpFrm)
: KviTaskBarBase(lpFrm)
{
	m_pListView = new KviTreeTaskBarListView(this);
	m_pListView->addColumn(__tr("Window list"),135);
	m_pListView->setAllColumnsShowFocus(true);
	m_pListView->setMultiSelection(false);
	setStretchableWidget(m_pListView);

	setVerticalStretchable(true);
	setHorizontalStretchable(true);
#if QT_VERSION >= 300
    setResizeEnabled(true);
#endif
    setMinimumWidth(KVI_OPTION_UINT(KviOption_uintTreeTaskBarMinimumWidth));
    setMaximumWidth(KVI_OPTION_UINT(KviOption_uintTreeTaskBarMaximumWidth));

	if(KVI_OPTION_UINT(KviOption_uintTreeTaskBarMinimumWidth) < 24)KVI_OPTION_UINT(KviOption_uintTreeTaskBarMinimumWidth) = 24;

	//m_pListView->setMinimumWidth(KVI_OPTION_UINT(KviOption_uintTreeTaskBarMinimumWidth));
    //m_pListView->setMaximumWidth(KVI_OPTION_UINT(KviOption_uintTreeTaskBarMaximumWidth));
	m_pListView->setFocusPolicy(QWidget::NoFocus);
	m_pListView->setStaticBackground(true);
	m_pListView->viewport()->setBackgroundMode(QWidget::NoBackground);
	m_pListView->header()->setResizeEnabled(false); 

	connect(m_pListView,SIGNAL(leftMousePress(QListViewItem *)),this,SLOT(leftMousePress(QListViewItem *)));
	connect(m_pListView,SIGNAL(rightMousePress(QListViewItem *)),this,SLOT(rightMousePress(QListViewItem *)));
#if QT_VERSION >= 300
	connect(m_pListView,SIGNAL(contextMenuRequested(QListViewItem *,const QPoint &,int)),this,SLOT(contextPopupRequested(QListViewItem *,const QPoint &,int)));
#endif
	m_pToolTip = new KviDynamicToolTip(m_pListView->viewport(),"tree_taskbar_tooltip");
	connect(m_pToolTip,SIGNAL(tipRequest(KviDynamicToolTip *,const QPoint &)),this,SLOT(tipRequest(KviDynamicToolTip *,const QPoint &)));
}

KviTreeTaskBar::~KviTreeTaskBar()
{
//	delete m_pToolTip;
}

void KviTreeTaskBar::updatePseudoTransparency()
{
//	debug("Update called");
#ifdef COMPILE_PSEUDO_TRANSPARENCY
	m_pListView->viewport()->update();
//	debug("Calling update");
#endif
}

void KviTreeTaskBar::moveEvent(QMoveEvent *)
{
#ifdef COMPILE_PSEUDO_TRANSPARENCY
//	debug("moveEvent");
	updatePseudoTransparency();
#endif

//	KviTreeTaskBarBase::moveEvent(e);
}

void KviTreeTaskBar::tipRequest(KviDynamicToolTip *,const QPoint &pnt)
{
	QListViewItem * it = m_pListView->itemAt(pnt);
	if(it)
	{
		m_pToolTip->tip(m_pListView->itemRect(it),((KviTreeTaskBarItem *)it)->m_pWindow->plainTextCaption());
	}
}

void KviTreeTaskBar::leftMousePress(QListViewItem * it)
{
	if(it)
	{
		KviWindow * wnd = ((KviTreeTaskBarItem *)it)->window();
		if(!wnd)return;
		if((wnd->frame()->activeWindow() != wnd) || (wnd->isMinimized()))wnd->frame()->setActiveWindow(wnd);
		else wnd->minimize();
	}
}

void KviTreeTaskBar::rightMousePress(QListViewItem * it)
{
	// never called with qt3.0
	if(it)((KviTreeTaskBarItem *)it)->window()->contextPopup();
}

void KviTreeTaskBar::contextPopupRequested(QListViewItem *it,const QPoint &,int)
{
	// this is the qt3.0 only slot
	if(it)((KviTreeTaskBarItem *)it)->window()->contextPopup();
}

KviTaskBarItem * KviTreeTaskBar::addItem(KviWindow * wnd)
{
	// complex insertion task
	if(wnd->console())
	{
		if(wnd->type() != KVI_WINDOW_TYPE_CONSOLE)
		{
			((KviTreeTaskBarItem *)(wnd->console()->m_pTaskBarItem))->setOpen(true);
			return new KviTreeTaskBarItem(((KviTreeTaskBarItem *)(wnd->console()->m_pTaskBarItem)),wnd);
		}
	}

	// console , or a window that has no irc context
	return new KviTreeTaskBarItem(m_pListView,wnd);
}

bool KviTreeTaskBar::removeItem(KviTaskBarItem * it)
{
	delete (KviTreeTaskBarItem *)it;
	return true;
}

void KviTreeTaskBar::setActiveItem(KviTaskBarItem * it)
{
	KviTreeTaskBarItem * cur = (KviTreeTaskBarItem *)m_pListView->currentItem();
	if(cur && (cur != (KviTreeTaskBarItem *)it))
	{
		cur->setActive(false);
	}
	if(((KviTreeTaskBarItem *)it)->parent())
	{
		if(!((KviTreeTaskBarItem *)it)->parent()->isOpen())((KviTreeTaskBarItem *)it)->parent()->setOpen(true);
	}
	((KviTreeTaskBarItem *)it)->setActive(true);
	m_pListView->setSelected(((KviTreeTaskBarItem *)it),true); // this MUST go after it->setActive()
	if(m_pFrm->dockExtension())m_pFrm->dockExtension()->refresh();
}


KviTaskBarItem * KviTreeTaskBar::firstItem()
{
	m_pCurrentItem = (KviTreeTaskBarItem *)m_pListView->firstChild();
	return m_pCurrentItem;
}

KviTaskBarItem * KviTreeTaskBar::nextItem()
{
	if(!m_pCurrentItem)return 0;

	if(m_pCurrentItem->firstChild())
	{
		m_pCurrentItem = (KviTreeTaskBarItem *)m_pCurrentItem->firstChild();
	} else {
		// this item has no children: try the next sibling
		if(m_pCurrentItem->nextSibling())
		{
			m_pCurrentItem = (KviTreeTaskBarItem *)m_pCurrentItem->nextSibling();
		} else {
			if(m_pCurrentItem->parent())
			{
				// child with not siblings : try the sibling of the parent
				m_pCurrentItem = (KviTreeTaskBarItem *)m_pCurrentItem->parent()->nextSibling();
			} else {
				m_pCurrentItem = 0; // toplevel with no siblings
			}
		}
	}
	return m_pCurrentItem;
}

KviTaskBarItem * KviTreeTaskBar::prevItem()
{	
	KviTreeTaskBarItem * it;

	if(!m_pCurrentItem)return 0;

	if(m_pCurrentItem->parent())
	{
		// a child item
		it = (KviTreeTaskBarItem *)m_pCurrentItem->parent()->firstChild();

		while(it)
		{
			if(((KviTreeTaskBarItem *)it->nextSibling()) == m_pCurrentItem)break;
			else it = ((KviTreeTaskBarItem *)(it->nextSibling()));
		}
		if(!it)
		{
			it = (KviTreeTaskBarItem *)m_pCurrentItem->parent();
		}

	} else {
		// a toplevel one
		it = (KviTreeTaskBarItem *)m_pListView->firstChild();
		while(it)
		{
			if(((KviTreeTaskBarItem *)it->nextSibling()) == m_pCurrentItem)break;
			else it = ((KviTreeTaskBarItem *)(it->nextSibling()));
		}
		if(it)
		{
			if(it->firstChild())
			{
				it = ((KviTreeTaskBarItem *)(it->firstChild()));
				while(it->nextSibling())it = ((KviTreeTaskBarItem *)(it->nextSibling()));
			}
		}
	}

	m_pCurrentItem = it;
	return it;
}

KviTaskBarItem * KviTreeTaskBar::lastItem()
{
	// first find last toplevel item
	m_pCurrentItem = (KviTreeTaskBarItem *)m_pListView->firstChild();
	for(;;)
	{
		if(m_pCurrentItem->nextSibling())
		{
			m_pCurrentItem = (KviTreeTaskBarItem *)m_pCurrentItem->nextSibling();
		} else if(m_pCurrentItem->firstChild())
		{
			m_pCurrentItem = (KviTreeTaskBarItem *)m_pCurrentItem->firstChild();
		} else return m_pCurrentItem;
	}
	return 0;
}

bool KviTreeTaskBar::setIterationPointer(KviTaskBarItem * it)
{
	m_pCurrentItem = (KviTreeTaskBarItem *)it;
	if(!it)return true;
	if(((QListView *)m_pListView) == ((KviTreeTaskBarItem *)it)->listView())return true;
	m_pCurrentItem = 0;
	return false;
}



#include "kvi_taskbar.moc"
