//
//   File : kvi_selectors.cpp
//   Creation date : Mon Nov 13 2000 15:22:12 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
#define __KVIRC__

#define _KVI_SELECTORS_CPP_

#include "kvi_selectors.h"
#include "kvi_locale.h"

#include "kvi_filedialog.h"

#include <qlabel.h>
#include <qpainter.h>
#include <qlayout.h>
#include <qobjcoll.h>
#include <qcolordialog.h>
#include <qpalette.h>
#include <qfontdialog.h>


KviBoolSelector::KviBoolSelector(QWidget * par,const char * txt,bool *pOption,bool bEnabled)
: QCheckBox(__c2q(txt),par,txt), KviSelectorInterface()
{
	setEnabled(bEnabled);
	setChecked(*pOption);
	m_pOption = pOption;
}

void KviBoolSelector::setNotEnabled(bool bNotEnabled)
{
	setEnabled(!bNotEnabled);
}

void KviBoolSelector::commit()
{
	*m_pOption = isChecked();
}

KviUIntSelector::KviUIntSelector(QWidget * par,const char * txt,unsigned int *pOption,
	unsigned int uLowBound,unsigned int uHighBound,unsigned int uDefault,bool bEnabled,bool bShortInt)
: QHBox(par,txt) , KviSelectorInterface()
{
	m_pLabel = new QLabel(__c2q(txt),this);
	m_pLineEdit = new QLineEdit(this);
	m_pLineEdit->setMaximumWidth(150);

	m_bIsShortInt = bShortInt;

	setEnabled(bEnabled);

	m_pOption = pOption;

	m_uLowBound = uLowBound;
	m_uHighBound = uHighBound;
	m_uDefault = uDefault;


	KviStr tmp(KviStr::Format,"%u",bShortInt ? (unsigned int) *((unsigned short int *)pOption) : *pOption);
	m_pLineEdit->setText(tmp.ptr());

	setStretchFactor(m_pLabel,10);
}

void KviUIntSelector::commit()
{
	KviStr tmp = m_pLineEdit->text();
	bool bOk;
	unsigned int val = tmp.toUInt(&bOk);
	if(!bOk)val = m_uDefault;
	if((val < m_uLowBound) && (m_uHighBound > m_uLowBound))val = m_uLowBound;
	if((val > m_uHighBound) && (m_uHighBound > m_uLowBound))val = m_uHighBound;

	if(m_bIsShortInt)*((unsigned short int *)m_pOption) = (unsigned short int)val;
	else *m_pOption = val;
}


void KviUIntSelector::setEnabled(bool bEnabled)
{
	m_pLabel->setEnabled(bEnabled);
	m_pLineEdit->setEnabled(bEnabled);
}



KviStringSelector::KviStringSelector(QWidget * par,const char * txt,KviStr * pOption,bool bEnabled)
: QHBox(par,txt) , KviSelectorInterface()
{
	m_pLabel = new QLabel(__c2q(txt),this);
	m_pLineEdit = new QLineEdit(this);
	m_pLineEdit->setMinimumWidth(200);
	m_pLineEdit->setText(pOption->ptr());

	m_pOption = pOption;

	setEnabled(bEnabled);
	setStretchFactor(m_pLabel,10);
}

KviStringSelector::~KviStringSelector()
{
}

void KviStringSelector::commit()
{
	*m_pOption = m_pLineEdit->text();
}

void KviStringSelector::setEnabled(bool bEnabled)
{
	m_pLineEdit->setEnabled(bEnabled);
	m_pLabel->setEnabled(bEnabled);
}

KviPasswordSelector::KviPasswordSelector(QWidget * par,const char * txt,KviStr *pOption,bool bEnabled)
: KviStringSelector(par,txt,pOption,bEnabled)
{
	m_pLineEdit->setEchoMode(QLineEdit::Password);
}




KviPixmapPreview::KviPixmapPreview(QWidget * par)
: QScrollView(par)
{
	m_pPixmap = 0;
	resizeContents(0,0);
}


KviPixmapPreview::~KviPixmapPreview()
{
}

void KviPixmapPreview::setPixmap(KviPixmap * pix)
{
	m_pPixmap = pix;
	if(m_pPixmap)
	{
		if(m_pPixmap->pixmap())
		{
			resizeContents(m_pPixmap->pixmap()->width(),m_pPixmap->pixmap()->height());
			return;
		}
	}
	resizeContents(0,0);
}

void KviPixmapPreview::drawContents(QPainter * p, int clipx, int clipy, int clipw, int cliph)
{
	if(m_pPixmap)
	{
		if(m_pPixmap->pixmap())
		{
			p->drawPixmap(clipx,clipy,*(m_pPixmap->pixmap()),clipx,clipy,clipw,cliph);
		}
	}
}


KviPixmapSelector::KviPixmapSelector(QWidget * par,const char * txt,KviPixmap * pOption,bool bEnabled)
: QWidget(par,txt), KviSelectorInterface()
{
	QGridLayout * g = new QGridLayout(this,3,2,0,2);
	m_pOption = pOption;

	m_localPixmap = *pOption;

	m_pCheckBox = new QCheckBox(__c2q(txt),this);
	m_pCheckBox->setChecked(m_localPixmap.pixmap());
	connect(m_pCheckBox,SIGNAL(toggled(bool)),this,SLOT(checkBoxToggled(bool)));
	g->addMultiCellWidget(m_pCheckBox,0,0,0,1);

	m_pPreview = new KviPixmapPreview(this);
	m_pPreview->setPixmap(&m_localPixmap);
	g->addMultiCellWidget(m_pPreview,1,1,0,1);

	m_pFileNameLabel = new QLabel(this);
	m_pFileNameLabel->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	if(m_localPixmap.pixmap())m_pFileNameLabel->setText(m_localPixmap.path());
	g->addWidget(m_pFileNameLabel,2,0);

	m_pChooseButton = new QPushButton(__c2q(__tr("Choose...")),this);
	g->addWidget(m_pChooseButton,2,1);
	connect(m_pChooseButton,SIGNAL(clicked()),this,SLOT(choosePixmap()));

	g->setRowStretch(1,1);
	g->setColStretch(0,1);
	setEnabled(bEnabled);
}

KviPixmapSelector::~KviPixmapSelector()
{
}

void KviPixmapSelector::checkBoxToggled(bool bEnabled)
{
	setEnabled(isEnabled());
}

void KviPixmapSelector::commit()
{
	if(m_pCheckBox->isChecked())
	{
		*m_pOption = m_localPixmap;
	} else {
		*m_pOption = KviPixmap(); // null pixmap
	}
}

void KviPixmapSelector::choosePixmap()
{
	KviStr tmp;
 	if(KviFileDialog::askForOpenFileName(tmp,__tr("Choose an image file...")))
	{
		m_localPixmap.load(tmp.ptr());
		m_pPreview->setPixmap(&m_localPixmap);
	}
}

void KviPixmapSelector::setEnabled(bool bEnabled)
{
	m_pCheckBox->setEnabled(bEnabled);
	m_pPreview->setEnabled(bEnabled && m_pCheckBox->isChecked());
	m_pFileNameLabel->setEnabled(bEnabled && m_pCheckBox->isChecked());
	m_pChooseButton->setEnabled(bEnabled && m_pCheckBox->isChecked());
	QWidget::setEnabled(bEnabled);
}



// FIXME: #warning "Option for DIR_MUST_EXISTS...(this widget could be turned into a file selector too)"
KviFileSelector::KviFileSelector(QWidget * par,const char * txt,KviStr * pOption,bool bEnabled)
: QHBox(par,txt), KviSelectorInterface()
{
	m_pLabel = new QLabel(__c2q(txt),this);
	m_pLineEdit = new QLineEdit(this);
	m_pLineEdit->setMinimumWidth(200);
	m_pLineEdit->setText(pOption->ptr());
	m_pButton = new QPushButton(__c2q(__tr("Browse...")),this);
	connect(m_pButton,SIGNAL(clicked()),this,SLOT(browseClicked()));

	setStretchFactor(m_pLabel,10);

	m_pOption = pOption;

	setEnabled(bEnabled);
}


void KviFileSelector::commit()
{
	*m_pOption = m_pLineEdit->text();
}

void KviFileSelector::setEnabled(bool bEnabled)
{
	m_pLineEdit->setEnabled(bEnabled);
	m_pLabel->setEnabled(bEnabled);
	m_pButton->setEnabled(bEnabled);
}

void KviFileSelector::browseClicked()
{
	select();
}

void KviFileSelector::select()
{
	KviStr tmp;
 	if(KviFileDialog::askForOpenFileName(tmp,__tr("Choose a file...")))
	{
		m_pLineEdit->setText(tmp.ptr());
	}

}


KviDirectorySelector::KviDirectorySelector(QWidget * par,const char * txt,KviStr * pOption,bool bEnabled)
: KviFileSelector(par,txt,pOption,bEnabled)
{
}

void KviDirectorySelector::select()
{
	KviStr tmp;
 	if(KviFileDialog::askForDirectoryName(tmp,__tr("Choose a directory...")))
	{
		m_pLineEdit->setText(tmp.ptr());
	}

}








KviStringListSelector::KviStringListSelector(QWidget * par,const char * txt,QStringList * pOption,bool bEnabled)
: QVBox(par,txt), KviSelectorInterface()
{
	m_pLabel = new QLabel(__c2q(txt),this);
	m_pListBox = new QListBox(this);
	m_pLineEdit = new QLineEdit(this);
	connect(m_pLineEdit,SIGNAL(textChanged(const QString &)),this,SLOT(textChanged(const QString &)));
	connect(m_pLineEdit,SIGNAL(returnPressed()),this,SLOT(addClicked()));
	QHBox * hBox = new QHBox(this);
	m_pAddButton = new QPushButton(__tr("Add"),hBox);
	connect(m_pAddButton,SIGNAL(clicked()),this,SLOT(addClicked()));
	m_pRemoveButton = new QPushButton(__tr("Remove"),hBox);
	connect(m_pRemoveButton,SIGNAL(clicked()),this,SLOT(removeClicked()));
	m_pOption = pOption;
	m_pListBox->insertStringList(*pOption);
	m_pListBox->setSelectionMode(QListBox::Extended);
	connect(m_pListBox,SIGNAL(selectionChanged()),this,SLOT(selectionChanged()));
	setStretchFactor(m_pListBox,1);
	setEnabled(bEnabled);
}

KviStringListSelector::~KviStringListSelector()
{
}

void KviStringListSelector::selectionChanged()
{
	unsigned int uCount = m_pListBox->count();
	bool bSomeSelected = false;
	for(unsigned int u=0;u<uCount;u++)
	{
		if(m_pListBox->isSelected(u))
		{
			bSomeSelected = true;
			break;
		}
	}
	m_pRemoveButton->setEnabled(isEnabled() && bSomeSelected);
}

void KviStringListSelector::textChanged(const QString &str)
{
	str.stripWhiteSpace();
	m_pAddButton->setEnabled((str.length() > 0) && isEnabled());
}

void KviStringListSelector::setEnabled(bool bEnabled)
{
	QVBox::setEnabled(bEnabled);
	m_pLineEdit->setEnabled(bEnabled);
	m_pLabel->setEnabled(bEnabled);
	QString txt = m_pLineEdit->text();
	txt.stripWhiteSpace();
	m_pAddButton->setEnabled(bEnabled && (txt.length() > 0));
	unsigned int uCount = m_pListBox->count();
	bool bSomeSelected = false;
	for(unsigned int u=0;u<uCount;u++)
	{
		if(m_pListBox->isSelected(u))
		{
			bSomeSelected = true;
			break;
		}
	}
	m_pRemoveButton->setEnabled(bEnabled && bSomeSelected);
	m_pListBox->setEnabled(bEnabled);
}

void KviStringListSelector::commit()
{
	unsigned int uCount = m_pListBox->count();
	m_pOption->clear();
	for(unsigned int u=0;u<uCount;u++)
	{
		QString str = m_pListBox->text(u);
		str.stripWhiteSpace();
		if(str.length() > 0)m_pOption->append(str);
	}
}

void KviStringListSelector::addClicked()
{
	QString str = m_pLineEdit->text();
	str.stripWhiteSpace();
	if(str.length() > 0)m_pListBox->insertItem(str);
	m_pLineEdit->setText("");
}

void KviStringListSelector::removeClicked()
{
	unsigned int uCount = m_pListBox->count();
	for(unsigned int u=0;u<uCount;u++)
	{
		while(m_pListBox->isSelected(u))m_pListBox->removeItem(u);
	}
}


KviColorSelector::KviColorSelector(QWidget * par,const char * txt,QColor * pOption,bool bEnabled)
: QHBox(par,txt), KviSelectorInterface()
{
	m_pLabel = new QLabel(__c2q(txt),this);

	m_pButton = new QPushButton("",this);
	// m_pButton->setMinimumWidth(150);
	connect(m_pButton,SIGNAL(clicked()),this,SLOT(changeClicked()));

	setStretchFactor(m_pLabel,10);

	setButtonPalette(pOption);

	m_pOption = pOption;

	setEnabled(bEnabled);
}

void KviColorSelector::setButtonPalette(QColor * pOption)
{
	QPalette pal(*pOption,colorGroup().background());
	m_memColor = *pOption;
	m_pButton->setPalette(pal);
}

void KviColorSelector::forceColor(QColor clr)
{
	setButtonPalette(&clr);
}

void KviColorSelector::changeClicked()
{
	QColor tmp = QColorDialog::getColor(m_memColor);
	if(tmp.isValid())setButtonPalette(&tmp);
}

void KviColorSelector::commit()
{
	*m_pOption = m_memColor;
}

void KviColorSelector::setEnabled(bool bEnabled)
{
	m_pLabel->setEnabled(bEnabled);
	m_pButton->setEnabled(bEnabled);
}


KviFontSelector::KviFontSelector(QWidget * par,const char * txt,QFont * pOption,bool bEnabled)
: QHBox(par,txt), KviSelectorInterface()
{
	m_pLabel = new QLabel(__c2q(txt),this);

	m_pButton = new QPushButton("",this);
	// m_pButton->setMinimumWidth(150);
	connect(m_pButton,SIGNAL(clicked()),this,SLOT(changeClicked()));

	setStretchFactor(m_pLabel,10);

	setButtonFont(pOption);

	m_pOption = pOption;

	setEnabled(bEnabled);
}

void KviFontSelector::setButtonFont(QFont * pOption)
{
	m_pButton->setText(pOption->family());
	m_pButton->setFont(*pOption);
}

void KviFontSelector::changeClicked()
{
	bool bOk;
	QFont tmp = QFontDialog::getFont(&bOk,m_pButton->font());
	if(bOk)setButtonFont(&tmp);
}

void KviFontSelector::commit()
{
	*m_pOption = m_pButton->font();
}

void KviFontSelector::setEnabled(bool bEnabled)
{
	m_pLabel->setEnabled(bEnabled);
	m_pButton->setEnabled(bEnabled);
}


#include "kvi_selectors.moc"
