//
//   File : kvi_query.cpp
//   Creation date : Tue Aug 7 2000 14:23:22 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
#define __KVIRC__
#include "kvi_query.h"
#include "kvi_console.h"
#include "kvi_iconmanager.h"
#include "kvi_ircview.h"
#include "kvi_input.h"
#include "kvi_options.h"
#include "kvi_locale.h"
#include "kvi_settings.h"
#include "kvi_ircsocket.h"
#include "kvi_out.h"
#include "kvi_taskbar.h"
#include "kvi_config.h"
#include "kvi_event.h"
#include "kvi_uparser.h"
#include "kvi_mirccntrl.h"
#include "kvi_settings.h"
#include "kvi_themedlabel.h"

#ifdef COMPILE_CRYPT_SUPPORT
	#include "kvi_crypt.h"
	#include "kvi_cryptcontroller.h"
#endif

#include <qpixmap.h>
#include <qsplitter.h>
#include <qhbox.h>
#include <qtoolbutton.h>


KviQuery::KviQuery(KviFrame * lpFrm,KviConsole * lpConsole,const char * name)
: KviWindow(KVI_WINDOW_TYPE_QUERY,lpFrm,name,lpConsole)
{
	m_pConsole->registerQuery(this);

	//m_pTopSplitter = new QSplitter(QSplitter::Horizontal,this,"top_splitter");
	m_pButtonBox = new QHBox(this,"button_box");
	KviThemedLabel * l = new KviThemedLabel(m_pButtonBox,"dummy_label");
	m_pButtonBox->setStretchFactor(l,1);
	QToolTip::add(l,__tr("What is this ?"));
	// The button box on the right
	//QHBox * box = new QHBox(m_pTopSplitter,"button_box");

	m_pListViewButton = createToolButton(m_pButtonBox,"list_view_button",KVI_SMALLICON_HIDELISTVIEW,KVI_SMALLICON_SHOWLISTVIEW,__tr("Toggle user list"),true);
	connect(m_pListViewButton,SIGNAL(clicked()),this,SLOT(toggleListView()));

#ifdef COMPILE_CRYPT_SUPPORT
	createCryptControllerButton(m_pButtonBox);
#endif

	m_pSplitter = new QSplitter(QSplitter::Horizontal,this,"main_splitter");
	m_pIrcView = new KviIrcView(m_pSplitter,lpFrm,this);
	connect(m_pIrcView,SIGNAL(rightClicked()),this,SLOT(textViewRightClicked()));
	m_pUserListView = new KviUserListView(m_pSplitter,m_pConsole->userDataBase(),this,7,__tr("Query targets"),"user_list_view");

	m_pInput   = new KviInput(this,m_pUserListView);

	setFocusHandler(m_pInput,this);

	if(KVI_OPTION_BOOL(KviOption_boolAutoLogQueries))m_pIrcView->startLogging(0);
// FIXME: #warning "Maybe tell the user all that we know about the remote end(s)....channels..."

	updateTargets();
}

KviQuery::~KviQuery()
{
	removeAllTargets();
	m_pConsole->unregisterQuery(this);
}

void KviQuery::triggerCreationEvents()
{
	TRIGGER_EVENT(KviEvent_OnQueryWindowCreated,this);
}

void KviQuery::getBaseLogFileName(KviStr &buffer)
{
	buffer.sprintf("%s_%u",name(),console()->ircContextId());
}

void KviQuery::textViewRightClicked()
{
	TRIGGER_EVENT(KviEvent_OnQueryPopupRequest,this);
}

void KviQuery::saveProperties(KviConfig *cfg)
{
	cfg->writeEntry("Splitter",m_pSplitter->sizes());
	cfg->writeEntry("UserListViewVisible",m_pUserListView->isVisible());
}

void KviQuery::loadProperties(KviConfig *cfg)
{
	QValueList<int> def;
	def.append(75);
	def.append(25);
	m_pSplitter->setSizes(cfg->readIntListEntry("Splitter",def));
	showListView(cfg->readBoolEntry("UserListViewVisible",false));
}

void KviQuery::userAction(KviIrcMask *user,int actionTemperature)
{
	m_pUserListView->userAction(user,actionTemperature);
}


KviUserListEntry * KviQuery::addTarget(const char *nick,const char * user,const char * host)
{
	KviUserListEntry * e = m_pUserListView->join(nick,user,host);
	if((!e->globalData()->avatar()) && user)
		m_pConsole->checkDefaultAvatar(e->globalData(),nick,user,host);
	updateTargets();

	if(KVI_OPTION_BOOL(KviOption_boolEnableQueryTracing))
	{
		KviStr szChans;
		int iChans = m_pConsole->getCommonChannels(nick,szChans);
		notifyCommonChannels(nick,user,host,iChans,szChans);
	}

	return e;
}

static const char * star = "*";

void KviQuery::notifyCommonChannels(const char * nick,const char * user,const char * host,int iChans,const char * chans)
{

	if(iChans > 0)
	{
		output(KVI_OUT_QUERYTRACE,__tr("Common channels for \r!n\r%s\r [%s@\r!h\r%s\r]: %s"),nick,user ? user : star,host ? host : star,chans);
	} else {
		output(KVI_OUT_QUERYTRACE,__tr("No common channels for \r!n\r%s\r [%s@\r!h\r%s\r]"),nick,user ? user : star,host ? host : star,chans);
	}
}

bool KviQuery::removeTarget(const char * nick)
{
	bool bRet = m_pUserListView->part(nick);
	updateTargets();
	return bRet;
}

void KviQuery::removeAllTargets()
{
	m_pUserListView->partAll();
	updateTargets();
}

void KviQuery::updateTargets()
{
	m_szTargets = "";

	if(targetCount() > 0)
	{
		QAsciiDictIterator<KviUserListEntry> it(*(m_pUserListView->entryDict()));
		while(it.current())
		{
			if(m_szTargets.hasData())m_szTargets.append(',');
			m_szTargets.append(it.currentKey());
			++it;
		}
	}

	if(m_szTargets.hasData())setName(m_szTargets.ptr());
	else setName(__tr("No targets"));

	updateCaption();
}

void KviQuery::fillCaptionBuffers()
{
	m_szPlainTextCaption.sprintf(__tr("%s [%s on %s]"),name(),
		m_pConsole->currentNickName(),
		m_pConsole->currentServerName());

	m_szHtmlActiveCaption.sprintf(__tr("<nobr><font color=\"%s\"><b>%s</b></font> <font color=\"%s\">[%s on %s]</font></nobr>"),
		KVI_OPTION_COLOR(KviOption_colorCaptionTextActive).name().ascii(),
		name(),
		KVI_OPTION_COLOR(KviOption_colorCaptionTextActive2).name().ascii(),
		m_pConsole->currentNickName(),
		m_pConsole->currentServerName());
	m_szHtmlInactiveCaption.sprintf(__tr("<nobr><font color=\"%s\"><b>%s</b></font> <font color=\"%s\">[%s on %s]</font></nobr>"),
		KVI_OPTION_COLOR(KviOption_colorCaptionTextInactive).name().ascii(),
		name(),
		KVI_OPTION_COLOR(KviOption_colorCaptionTextInactive2).name().ascii(),
		m_pConsole->currentNickName(),
		m_pConsole->currentServerName());
}

bool KviQuery::nickChange(const char * oldNick,const char * newNick)
{
	bool bRet = m_pUserListView->nickChange(oldNick,newNick);
	updateTargets();
	return bRet;
}

void KviQuery::showListView(bool bShow)
{
	if(!bShow)
	{
		m_pUserListView->hide();
		if(m_pListViewButton->isOn())m_pListViewButton->setOn(false);
	} else {
		m_pUserListView->show();
		if(!(m_pListViewButton->isOn()))m_pListViewButton->setOn(true);
	}
}

void KviQuery::toggleListView()
{
	showListView(!m_pUserListView->isVisible());
}

QPixmap * KviQuery::myIconPtr()
{
	return g_pIconManager->getSmallIcon(KVI_SMALLICON_QUERY);
}

void KviQuery::resizeEvent(QResizeEvent *e)
{
	int hght = m_pInput->heightHint();
	int hght2 = m_pButtonBox->sizeHint().height();
	m_pButtonBox->setGeometry(0,0,width(),hght2);
	m_pSplitter->setGeometry(0,hght2,width(),height() - (hght + hght2));
	m_pInput->setGeometry(0,height() - hght,width(),hght);
}

QSize KviQuery::sizeHint() const
{
	QSize ret(m_pSplitter->sizeHint().width(),m_pIrcView->sizeHint().height() + m_pInput->heightHint());
	return ret;
}

void KviQuery::ownMessage(const char * buffer)
{
	if(targetCount() == 0)
	{
		outputNoFmt(KVI_OUT_SYSTEMWARNING,__tr("This query has no active targets: no message sent"));
	} else {

#ifdef COMPILE_CRYPT_SUPPORT
		if(cryptSessionInfo())
		{
			if(cryptSessionInfo()->bDoEncrypt)
			{
				if(*buffer != KVI_TEXT_CRYPT)
				{
					KviStr encrypted;
					cryptSessionInfo()->pEngine->setMaxEncryptLen(500 - kvi_strLen(name()));
					if(cryptSessionInfo()->pEngine->encrypt(buffer,encrypted))
					{
						if(cryptSessionInfo()->pEngine->isCryptographicEngine())
						{
							m_pConsole->socket()->sendFmtData("PRIVMSG %s :%c%s",name(),KVI_TEXT_CRYPT,encrypted.ptr());
							m_pConsole->outputPrivmsg(this,KVI_OUT_OWNPRIVMSGCRYPTED,0,0,0,buffer);
					} else {
							m_pConsole->socket()->sendFmtData("PRIVMSG %s :%s",name(),encrypted.ptr());
							m_pConsole->outputPrivmsg(this,KVI_OUT_OWNPRIVMSG,0,0,0,encrypted.ptr());
						}
						//m_pConsole->outputPrivmsg(this,KVI_OUT_OWNPRIVMSG,0,0,0,encrypted.ptr());
						m_pUserListView->userAction(m_pConsole->currentNickName(),KVI_USERACTION_PRIVMSG);
						return;
					} else {
						output(KVI_OUT_SYSTEMERROR,
							__tr("The crypt engine was not able to encrypt the current message (%s): %s: no data sent to the server"),
							buffer,cryptSessionInfo()->pEngine->lastError());
						return;
					}
				} else {
					buffer++; //eat the escape code
				}
			}
		}
#endif

		m_pConsole->socket()->sendFmtData("PRIVMSG %s :%s",m_szTargets.ptr(),buffer);
		m_pConsole->outputPrivmsg(this,KVI_OUT_OWNPRIVMSG,0,0,0,buffer);
		m_pUserListView->userAction(m_pConsole->currentNickName(),KVI_USERACTION_PRIVMSG);
	}
}

void KviQuery::ownAction(const char * buffer)
{
	if(targetCount() == 0)
	{
		outputNoFmt(KVI_OUT_SYSTEMWARNING,__tr("This query has no active targets: no message sent"));
	} else {
		m_pConsole->socket()->sendFmtData("PRIVMSG %s :%cACTION %s%c",
			m_szTargets.ptr(),0x01,buffer,0x01);
		output(KVI_OUT_ACTION,"%s %s",m_pConsole->currentNickName(),buffer);
		m_pUserListView->userAction(m_pConsole->currentNickName(),KVI_USERACTION_ACTION);
	}
}

#include "kvi_query.moc"
