//
//   File : kvi_mdimanager.cpp
//   Creation date : Wed Jun 21 2000 17:28:04 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
#define __KVIRC__
#define _KVI_DEBUG_CHECK_RANGE_
#include "kvi_debug.h"
#include "kvi_settings.h"
#include "kvi_mdimanager.h"
#include "kvi_mdichild.h"
#include "kvi_locale.h"
#include "kvi_options.h"
#include "kvi_iconmanager.h"
#include "kvi_frame.h"
#include "kvi_menubar.h"

#include <qpopupmenu.h>
#include <qmenubar.h>
#include <qobjcoll.h>
#include <qlayout.h>
#include <qpainter.h>
#include <math.h>
#include <qcursor.h>


#ifdef COMPILE_PSEUDO_TRANSPARENCY
	#include <qpixmap.h>
	extern QPixmap * g_pShadedParentGlobalDesktopBackground;
#endif


KviMdiManager::KviMdiManager(QWidget * parent,KviFrame * pFrm,const char * name)
: QFrame(parent,name)
{
	setFrameStyle(QFrame::Sunken | QFrame::WinPanel);

	m_pZ = new KviPtrList<KviMdiChild>;
	m_pZ->setAutoDelete(true);

	m_pFrm = pFrm;

	m_pSdiFrame          = 0;
	m_pSdiRestoreButton  = 0;
	m_pSdiMinimizeButton = 0;
	m_pSdiCloseButton    = 0;
	m_iSdiIconItemId     = 0;

	m_pWindowPopup = new QPopupMenu(this);
	connect(m_pWindowPopup,SIGNAL(activated(int)),this,SLOT(menuActivated(int)));
	connect(m_pWindowPopup,SIGNAL(aboutToShow()),this,SLOT(fillWindowPopup()));
	m_pTileMethodPopup = new QPopupMenu(this);
	connect(m_pTileMethodPopup,SIGNAL(activated(int)),this,SLOT(tileMethodMenuActivated(int)));

	setBackgroundMode(QWidget::NoBackground);
}

KviMdiManager::~KviMdiManager()
{
	delete m_pZ;
}

void KviMdiManager::drawContents(QPainter *p)
{
#ifdef COMPILE_PSEUDO_TRANSPARENCY
	if(g_pShadedParentGlobalDesktopBackground)
	{
		QPoint pnt = mapToGlobal(contentsRect().topLeft());
		p->drawTiledPixmap(contentsRect(),*(g_pShadedParentGlobalDesktopBackground),pnt);
		return;
	}
#endif
	if(KVI_OPTION_PIXMAP(KviOption_pixmapMdiBackground).pixmap())
	{
		p->drawTiledPixmap(contentsRect(),*(KVI_OPTION_PIXMAP(KviOption_pixmapMdiBackground).pixmap()));
	} else {
		p->fillRect(contentsRect(),KVI_OPTION_COLOR(KviOption_colorMdiBackground));
	}
}

void KviMdiManager::manageChild(KviMdiChild * lpC,bool bCascade,QRect *setGeom)
{
	__range_valid(lpC);

	m_pZ->insert(0,lpC); //hidden -> last in the Z order

	if(bCascade)lpC->move(getCascadePoint(m_pZ->count()-1));
	else {
		if(setGeom)lpC->setGeometry(*setGeom);
	}

//	if(bShow)
//	{
//		lpC->show();
//		setTopChild(lpC,true);
//	}

//	if(KVI_OPTION_BOOL(KviOption_boolAutoTileWindows))tile();
}

void KviMdiManager::showAndActivate(KviMdiChild * lpC)
{
	lpC->show();
	setTopChild(lpC,true);
	if(KVI_OPTION_BOOL(KviOption_boolAutoTileWindows))tile();
}

void KviMdiManager::setTopChild(KviMdiChild *lpC,bool bSetFocus)
{
	__range_valid(lpC);
	// The following check fails safely at startup....
	//	__range_valid(lpC->isVisible() || lpC->testWState(WState_ForceHide));

	if(m_pZ->last() != lpC)
	{
		m_pZ->setAutoDelete(false);

		if(!m_pZ->removeRef(lpC))return; // no such child ?

		//disable the labels of all the other children
		for(KviMdiChild *pC=m_pZ->first();pC;pC=m_pZ->next())
		{
			pC->captionLabel()->setActive(false);
		}

		KviMdiChild *pMaximizedChild=m_pZ->last();
		if(pMaximizedChild)
		{
			if(pMaximizedChild->m_state != KviMdiChild::Maximized)pMaximizedChild=0;
		}

		m_pZ->setAutoDelete(true);
		m_pZ->append(lpC);

		if(pMaximizedChild)
		{
			lpC->maximize(); //do not animate the change
			// In SDI MODE!
//			updateSDIMode();
	
		}
		lpC->raise();
		if(pMaximizedChild)pMaximizedChild->restore();
		
	}

	if(bSetFocus)
	{
		if(!lpC->hasFocus())lpC->setFocus();
	}
}

void KviMdiManager::focusInEvent(QFocusEvent *)
{
	focusTopChild();
}

void KviMdiManager::destroyChild(KviMdiChild *lpC,bool bFocusTopChild)
{
	bool bWasMaximized = lpC->state() == KviMdiChild::Maximized;
	disconnect(lpC);
	lpC->blockSignals(true);
#ifdef _KVI_DEBUG_CHECK_RANGE_
	//Report invalid results in a debug session
	__range_valid(m_pZ->removeRef(lpC));
#else
	m_pZ->removeRef(lpC);
#endif
	if(bWasMaximized)
	{
		KviMdiChild * c=topChild();
		if(c)
		{
			if(c->state() != KviMdiChild::Minimized)c->maximize();
			else {
				// minimized top child...the last one
				leaveSDIMode();
			}
		} else {
			// SDI state change
			leaveSDIMode();
		}
	}
	if(bFocusTopChild)focusTopChild();

	if(KVI_OPTION_BOOL(KviOption_boolAutoTileWindows))tile();
}

void KviMdiManager::resizeEvent(QResizeEvent *e)
{
	//If we have a maximized children at the top , adjust its size
	KviMdiChild *lpC=m_pZ->last();
	if(lpC)
	{
		if(lpC->state()==KviMdiChild::Maximized)
		{
			lpC->resize(width()+ KVI_MDICHILD_DOUBLE_HIDDEN_EDGE,
				height() + lpC->m_pCaptionLabel->heightHint() + KVI_MDICHILD_DOUBLE_HIDDEN_EDGE + KVI_MDICHILD_SPACING);
		} else {
			if(KVI_OPTION_BOOL(KviOption_boolAutoTileWindows))tile();
		}
	}
}

QPoint KviMdiManager::getCascadePoint(int indexOfWindow)
{
	QPoint pnt(0,0);
	if(indexOfWindow==0)return pnt;
	KviMdiChild *lpC=m_pZ->first();
	int step=(lpC ? (lpC->captionLabel()->heightHint()+KVI_MDICHILD_BORDER) : 20);
	int availableHeight=height()-(lpC ? lpC->minimumSize().height() : KVI_MDICHILD_MIN_HEIGHT);
	int availableWidth=width()-(lpC ? lpC->minimumSize().width() : KVI_MDICHILD_MIN_WIDTH);
	int ax=0;
	int ay=0;
	for(int i=0;i<indexOfWindow;i++)
	{
		ax+=step;
		ay+=step;
		if(ax>availableWidth)ax=0;
		if(ay>availableHeight)ay=0;
	}
	pnt.setX(ax);
	pnt.setY(ay);
	return pnt;
}

void KviMdiManager::mousePressEvent(QMouseEvent *e)
{
	//Popup the window menu
	if(e->button() & RightButton)m_pWindowPopup->popup(mapToGlobal(e->pos()));
}

void KviMdiManager::childMaximized(KviMdiChild * lpC)
{
	if(lpC == m_pZ->last())enterSDIMode(lpC);
}

void KviMdiManager::childMinimized(KviMdiChild * lpC,bool bWasMaximized)
{
	__range_valid(lpC);
	if(m_pZ->findRef(lpC) == -1)return;
	if(m_pZ->count() > 1)
	{
		m_pZ->setAutoDelete(false);
#ifdef _KVI_DEBUG_CHECK_RANGE_
		//Report invalid results in a debug session
		__range_valid(m_pZ->removeRef(lpC));
#else
		m_pZ->removeRef(lpC);
#endif
		m_pZ->setAutoDelete(true);
		m_pZ->insert(0,lpC);
		if(bWasMaximized)
		{
			// Need to maximize the top child
			lpC = m_pZ->last();
			if(!lpC)return; //??
			if(lpC->state()==KviMdiChild::Minimized)
			{
				if(bWasMaximized)leaveSDIMode();
				return;
			}
			lpC->maximize(); //do nrot animate the change
		}
		focusTopChild();
	} else {
		// Unique window minimized...it won't loose the focus...!!
		setFocus(); //Remove focus from the child
		if(bWasMaximized)leaveSDIMode();
	}
}

void KviMdiManager::childRestored(KviMdiChild * lpC,bool bWasMaximized)
{
	if((lpC == m_pZ->last()) && bWasMaximized)
	{
		leaveSDIMode();
		if(KVI_OPTION_BOOL(KviOption_boolAutoTileWindows))tile();
	}
}

void KviMdiManager::focusTopChild()
{
	KviMdiChild *lpC=m_pZ->last();
	if(!lpC)return;
	if(!lpC->isVisible())return;
//	if(lpC->state()==KviMdiChild::Minimized)return;
//	debug("Focusing top child %s",lpC->name());
	//disable the labels of all the other children
	for(KviMdiChild *pC=m_pZ->first();pC;pC=m_pZ->next()){
		if(pC != lpC)pC->captionLabel()->setActive(false);
	}
	lpC->raise();
	if(!lpC->hasFocus())lpC->setFocus();
}

void KviMdiManager::minimizeActiveChild()
{
	KviMdiChild * lpC = m_pZ->last();
	if(!lpC)return;
	if(lpC->state() != KviMdiChild::Minimized)lpC->minimize();
}

void KviMdiManager::restoreActiveChild()
{
	KviMdiChild * lpC = m_pZ->last();
	if(!lpC)return;
	if(lpC->state() == KviMdiChild::Maximized)lpC->restore();
}

void KviMdiManager::closeActiveChild()
{
	KviMdiChild * lpC = m_pZ->last();
	if(!lpC)return;
	lpC->closeRequest();
}


void KviMdiManager::updateSDIMode()
{
	__range_valid(m_pSdiFrame);
//	__range_valid(m_pSdiFrame->isVisible()); // Invalid at startup (the parent is not visible)

	KviMenuBar * b = m_pFrm->mainMenuBar();

	KviMdiChild * lpC = m_pZ->last();

	m_pSdiCloseButton->setEnabled(lpC ? lpC->closeEnabled() : false);
	if(m_iSdiIconItemId != 0)b->removeItem(m_iSdiIconItemId);

	const QPixmap * pix = lpC ? lpC->m_pIconButton->pixmap() : 0;
// FIXME: #warning "The accel for this ?"
	if(pix && !(pix->isNull()))m_iSdiIconItemId = b->insertItem(*pix,this,SLOT(activeChildSystemPopup()),0,-1,0);
	else m_iSdiIconItemId = 0;
}

void KviMdiManager::activeChildSystemPopup()
{
	KviMdiChild * lpC = m_pZ->last();
	if(!lpC)return;
//	lpC->systemPopupSlot();
	lpC->emitSystemPopupRequest(QCursor::pos());
}

void KviMdiManager::enterSDIMode(KviMdiChild *lpC)
{
	KviMenuBar * b = m_pFrm->mainMenuBar();

	if(!m_pSdiFrame)
	{
		m_pSdiFrame = new QFrame(b,"sdi_frame");
		m_pSdiFrame->setBackgroundMode(PaletteButton);
		QHBoxLayout * box = new QHBoxLayout(m_pSdiFrame,0,2,"sdi_hbox_layout");
		box->setAutoAdd(true);
		m_pSdiMinimizeButton = new KviMdiCaptionButton(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_MINIMIZE)),m_pSdiFrame,"sdi_minimize_button");
		m_pSdiMinimizeButton->setBackgroundMode(PaletteButton);
		connect(m_pSdiMinimizeButton,SIGNAL(clicked()),this,SLOT(minimizeActiveChild()));
		m_pSdiRestoreButton = new KviMdiCaptionButton(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_RESTORE)),m_pSdiFrame,"sdi_restore_button");
		m_pSdiRestoreButton->setBackgroundMode(PaletteButton);
		connect(m_pSdiRestoreButton,SIGNAL(clicked()),this,SLOT(restoreActiveChild()));
		m_pSdiCloseButton = new KviMdiCaptionButton(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_CLOSE)),m_pSdiFrame,"sdi_close_button");
		m_pSdiCloseButton->setBackgroundMode(PaletteButton);
		connect(m_pSdiCloseButton,SIGNAL(clicked()),this,SLOT(closeActiveChild()));
		b->insertItem(m_pSdiFrame,-1,b->count());
	}

	m_pSdiFrame->show();
	updateSDIMode();

	emit enteredSdiMode();
}

void KviMdiManager::leaveSDIMode()
{
	__range_valid(m_pSdiFrame);
	__range_valid(m_pSdiFrame->isVisible());
	m_pSdiFrame->hide();
	if(m_iSdiIconItemId != 0)m_pFrm->mainMenuBar()->removeItem(m_iSdiIconItemId);
	m_iSdiIconItemId = 0;
	emit leftSdiMode();
}

#define KVI_TILE_METHOD_ANODINE 0
#define KVI_TILE_METHOD_PRAGMA4HOR 1
#define KVI_TILE_METHOD_PRAGMA4VER 2
#define KVI_TILE_METHOD_PRAGMA6HOR 3
#define KVI_TILE_METHOD_PRAGMA6VER 4
#define KVI_TILE_METHOD_PRAGMA9HOR 5
#define KVI_TILE_METHOD_PRAGMA9VER 6

void KviMdiManager::fillWindowPopup()
{
	m_pWindowPopup->clear();
	int i=100;
	for(KviMdiChild *lpC=m_pZ->first();lpC;lpC=m_pZ->next())
	{
		KviStr szItem;
		szItem.setNum(((uint)i)-99);
		szItem+=". ";
		if(lpC->state()==KviMdiChild::Minimized)
		{
			szItem+="( ";
			szItem+=lpC->plainCaption();
			szItem+=" )";
		} else if(((uint)i)==(m_pZ->count()+99))
		{ //Active item :)
			szItem+="* ";
			szItem+=lpC->plainCaption();
			szItem+=" *";
		} else szItem+=lpC->plainCaption();
		const QPixmap * pix = lpC->m_pIconButton->pixmap();
		if(pix && !(pix->isNull()))m_pWindowPopup->insertItem(*pix,szItem.ptr(),i);
		else m_pWindowPopup->insertItem(szItem.ptr(),i);
		i++;
	}

	m_pWindowPopup->insertSeparator();
	m_pWindowPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_CASCADEWINDOWS)),(__c2q(__tr("&Cascade windows"))),this,SLOT(cascadeWindows()));
	m_pWindowPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_CASCADEWINDOWS)),(__c2q(__tr("Cascade &maximized"))),this,SLOT(cascadeMaximized()));
	m_pWindowPopup->insertSeparator();
	m_pWindowPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_MAXVERTICAL)),(__c2q(__tr("Expand &vertical"))),this,SLOT(expandVertical()));
	m_pWindowPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_MAXHORIZONTAL)),(__c2q(__tr("Expand &horizontal"))),this,SLOT(expandHorizontal()));
	m_pWindowPopup->insertSeparator();
	m_pWindowPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_TILEWINDOWS)),(__c2q(__tr("&Tile"))),this,SLOT(tile()));

	m_pTileMethodPopup->clear();
	int ids[7];
	ids[0] = m_pTileMethodPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_TILEWINDOWS)),(__c2q(__tr("Anodine's full-grid"))));
	ids[1] = m_pTileMethodPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_TILEWINDOWS)),(__c2q(__tr("Pragma's horizontal 4-grid"))));
	ids[2] = m_pTileMethodPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_TILEWINDOWS)),(__c2q(__tr("Pragma's vertical 4-grid"))));
	ids[3] = m_pTileMethodPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_TILEWINDOWS)),(__c2q(__tr("Pragma's horizontal 6-grid"))));
	ids[4] = m_pTileMethodPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_TILEWINDOWS)),(__c2q(__tr("Pragma's vertical 6-grid"))));
	ids[5] = m_pTileMethodPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_TILEWINDOWS)),(__c2q(__tr("Pragma's horizontal 9-grid"))));
	ids[6] = m_pTileMethodPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_TILEWINDOWS)),(__c2q(__tr("Pragma's vertical 9-grid"))));

	if(KVI_OPTION_UINT(KviOption_uintTileMethod) > 6)KVI_OPTION_UINT(KviOption_uintTileMethod) = 5;
	m_pTileMethodPopup->setItemChecked(ids[KVI_OPTION_UINT(KviOption_uintTileMethod)],true);

	m_pWindowPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_TILEWINDOWS)),(__c2q(__tr("Tile met&hod"))),m_pTileMethodPopup);
	int id = m_pWindowPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_TILE_HOR_WINDOWS)),(__c2q(__tr("&Auto tile"))),this,SLOT(toggleAutoTile()));
	m_pWindowPopup->setItemChecked(id,KVI_OPTION_BOOL(KviOption_boolAutoTileWindows));

/*
	m_pWindowPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_TILEWINDOWS)),__tr("&Anodine's tile"),this,SLOT(tileAnodine()));
	m_pWindowPopup->insertSeparator();
	m_pWindowPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_TILEWINDOWS)),__tr("&Pragma's tile-9 (vertical)"),this,SLOT(tilePragma9Vertical()));
	m_pWindowPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_TILE_HOR_WINDOWS)),__tr("P&ragma's tile-9 (horizontal)"),this,SLOT(tilePragma9Horizontal()));
	m_pWindowPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_TILEWINDOWS)),__tr("&Pragma's tile-6 (vertical)"),this,SLOT(tilePragma6Vertical()));
	m_pWindowPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_TILE_HOR_WINDOWS)),__tr("P&ragma's tile-6 (horizontal)"),this,SLOT(tilePragma6Horizontal()));
	m_pWindowPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_TILEWINDOWS)),__tr("&Pragma's tile-4 (vertical)"),this,SLOT(tilePragma4Vertical()));
	m_pWindowPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_TILE_HOR_WINDOWS)),__tr("P&ragma's tile-4 (horizontal)"),this,SLOT(tilePragma4Horizontal()));
*/
}

void KviMdiManager::menuActivated(int id)
{
	if(id<100)return;
	id-=100;
	__range_valid(((uint)id) < m_pZ->count());
	KviMdiChild *lpC=m_pZ->at(id);
	if(!lpC)return;
	if(lpC->state()==KviMdiChild::Minimized)lpC->restore();
	setTopChild(lpC,true);
}

void KviMdiManager::tileMethodMenuActivated(int id)
{
	int idx = m_pTileMethodPopup->indexOf(id);
	if(idx < 0)idx = 0;
	if(idx > 6)idx = 5;
	KVI_OPTION_UINT(KviOption_uintTileMethod) = idx;
	if(KVI_OPTION_BOOL(KviOption_boolAutoTileWindows))tile();
}

void KviMdiManager::cascadeWindows()
{
	int idx=0;
	KviPtrList<KviMdiChild> list(*m_pZ);
	list.setAutoDelete(false);
	while(!list.isEmpty()){
		KviMdiChild *lpC=list.first();
		if(lpC->state() != KviMdiChild::Minimized){
			if(lpC->state()==KviMdiChild::Maximized)lpC->restore();
			lpC->move(getCascadePoint(idx));
			lpC->resize(lpC->sizeHint());
			idx++;
		}
		list.removeFirst();
	}
	focusTopChild();
}

void KviMdiManager::cascadeMaximized()
{
	int idx=0;
	KviPtrList<KviMdiChild> list(*m_pZ);

	list.setAutoDelete(false);
	while(!list.isEmpty()){
		KviMdiChild *lpC=list.first();
		if(lpC->state() != KviMdiChild::Minimized){
			if(lpC->state()==KviMdiChild::Maximized)lpC->restore();
			QPoint pnt(getCascadePoint(idx));
			lpC->move(pnt);
			QSize curSize(width()-pnt.x(),height()-pnt.y());
			if((lpC->minimumSize().width() > curSize.width()) ||
				(lpC->minimumSize().height() > curSize.height()))lpC->resize(lpC->minimumSize());
			else lpC->resize(curSize);
			idx++;
		}
		list.removeFirst();
	}
	focusTopChild();
}

void KviMdiManager::expandVertical()
{
	int idx=0;
	KviPtrList<KviMdiChild> list(*m_pZ);
	list.setAutoDelete(false);
	while(!list.isEmpty()){
		KviMdiChild *lpC=list.first();
		if(lpC->state() != KviMdiChild::Minimized){
			if(lpC->state()==KviMdiChild::Maximized)lpC->restore();
			lpC->setGeometry(lpC->x(),0,lpC->width(),height());
			idx++;
		}
		list.removeFirst();
	}
	focusTopChild();
}

void KviMdiManager::expandHorizontal()
{
	int idx=0;
	KviPtrList<KviMdiChild> list(*m_pZ);
	list.setAutoDelete(false);
	while(!list.isEmpty()){
		KviMdiChild *lpC=list.first();
		if(lpC->state() != KviMdiChild::Minimized){
			if(lpC->state()==KviMdiChild::Maximized)lpC->restore();
			lpC->setGeometry(0,lpC->y(),width(),lpC->height());
			idx++;
		}
		list.removeFirst();
	}
	focusTopChild();
}

//void KviMdiManager::expand()
//{
//	int idx=0;
//	KviPtrList<KviMdiChild> list(*m_pZ);
//	list.setAutoDelete(false);
//	while(!list.isEmpty()){
//		KviMdiChild *lpC=list.first();
//		if(lpC->m_state != KviMdiChild::Minimized){
//			if(lpC->m_state==KviMdiChild::Maximized)lpC->setState(KviMdiChild::Normal,false);
//			lpC->setGeometry(0,0,width(),height());
//			idx++;
//		}
//		list.removeFirst();
//	}
//	focusTopChild();
//}

int KviMdiManager::getVisibleChildCount()
{
	int cnt=0;
	for(KviMdiChild *lpC=m_pZ->first();lpC;lpC=m_pZ->next()){
		if(lpC->state() != KviMdiChild::Minimized)cnt++;
	}
	return cnt;
}

void KviMdiManager::tile()
{
	switch(KVI_OPTION_UINT(KviOption_uintTileMethod))
	{
		case KVI_TILE_METHOD_ANODINE:      tileAnodine(); break;
		case KVI_TILE_METHOD_PRAGMA4HOR:   tileAllInternal(4,true); break;
		case KVI_TILE_METHOD_PRAGMA4VER:   tileAllInternal(4,false); break;
		case KVI_TILE_METHOD_PRAGMA6HOR:   tileAllInternal(6,true); break;
		case KVI_TILE_METHOD_PRAGMA6VER:   tileAllInternal(6,false); break;
		case KVI_TILE_METHOD_PRAGMA9HOR:   tileAllInternal(9,true); break;
		case KVI_TILE_METHOD_PRAGMA9VER:   tileAllInternal(9,false); break;
		default:
			KVI_OPTION_UINT(KviOption_uintTileMethod) = KVI_TILE_METHOD_PRAGMA9HOR;
			tileAllInternal(9,true);
		break;
	}
}

void KviMdiManager::toggleAutoTile()
{
	if(KVI_OPTION_BOOL(KviOption_boolAutoTileWindows))
	{
		KVI_OPTION_BOOL(KviOption_boolAutoTileWindows) = false;
	} else {
		KVI_OPTION_BOOL(KviOption_boolAutoTileWindows) = true;
		tile();
	}
}

void KviMdiManager::tileAllInternal(int maxWnds,bool bHorizontal)
{
	//NUM WINDOWS =           1,2,3,4,5,6,7,8,9
	static int colstable[9]={ 1,1,1,2,2,2,3,3,3 }; //num columns
	static int rowstable[9]={ 1,2,3,2,3,3,3,3,3 }; //num rows
	static int lastwindw[9]={ 1,1,1,1,2,1,3,2,1 }; //last window multiplier
	static int colrecall[9]={ 0,0,0,3,3,3,6,6,6 }; //adjust self
	static int rowrecall[9]={ 0,0,0,0,4,4,4,4,4 }; //adjust self

	int * pColstable = bHorizontal ? colstable : rowstable;
	int * pRowstable = bHorizontal ? rowstable : colstable;
	int * pColrecall = bHorizontal ? colrecall : rowrecall;
	int * pRowrecall = bHorizontal ? rowrecall : colrecall;

	KviMdiChild *lpTop=topChild();
	int numVisible=getVisibleChildCount();
	if(numVisible<1)return;
	int numToHandle=((numVisible > maxWnds) ? maxWnds : numVisible);
	int xQuantum=width()/pColstable[numToHandle-1];
	if(xQuantum < ((lpTop->minimumSize().width() > KVI_MDICHILD_MIN_WIDTH) ? lpTop->minimumSize().width() : KVI_MDICHILD_MIN_WIDTH)){
		if(pColrecall[numToHandle-1]==0)debug(__tr("Tile : Not enouh space"));
		else tileAllInternal(pColrecall[numToHandle-1],bHorizontal);
		return;
	}
	int yQuantum=height()/pRowstable[numToHandle-1];
	if(yQuantum < ((lpTop->minimumSize().height() > KVI_MDICHILD_MIN_HEIGHT) ? lpTop->minimumSize().height() : KVI_MDICHILD_MIN_HEIGHT)){
		if(pRowrecall[numToHandle-1]==0)debug(__tr("Tile : Not enough space"));
		else tileAllInternal(pRowrecall[numToHandle-1],bHorizontal);
		return;
	}
	int curX=0;
	int curY=0;
	int curRow=1;
	int curCol=1;
	int curWin=1;
	for(KviMdiChild *lpC=m_pZ->first();lpC;lpC=m_pZ->next()){
		if(lpC->state()!=KviMdiChild::Minimized){
			//restore the window
			if(lpC->state()==KviMdiChild::Maximized)lpC->restore();
			if((curWin%numToHandle)==0)lpC->setGeometry(curX,curY,xQuantum * lastwindw[numToHandle-1],yQuantum);
			else lpC->setGeometry(curX,curY,xQuantum,yQuantum);
			//example : 12 windows : 3 cols 3 rows
			if(curCol<pColstable[numToHandle-1]){ //curCol<3
				curX+=xQuantum; //add a column in the same row
				curCol++;       //increase current column
			} else {
				curX=0;         //new row
				curCol=1;       //column 1
				if(curRow<pRowstable[numToHandle-1]){ //curRow<3
					curY+=yQuantum; //add a row
					curRow++;       //
				} else {
					curY=0;         //restart from beginning
					curRow=1;       //
				}
			}
			curWin++;
		}
	}
	if(lpTop)lpTop->setFocus();
}

void KviMdiManager::tileAnodine()
{
	KviMdiChild *lpTop=topChild();
	int numVisible=getVisibleChildCount(); // count visible windows
	if(numVisible<1)return;
	int numCols=int(sqrt(numVisible)); // set columns to square root of visible count
	// create an array to form grid layout
	int *numRows=new int[numCols];
	int numCurCol=0;
	while(numCurCol<numCols)
	{
		numRows[numCurCol]=numCols; // create primary grid values
		numCurCol++;
	}
	int numDiff=numVisible-(numCols*numCols); // count extra rows
	int numCurDiffCol=numCols; // set column limiting for grid updates
	while(numDiff>0)
	{
		numCurDiffCol--;
		numRows[numCurDiffCol]++; // add extra rows to column grid
		if(numCurDiffCol<1)numCurDiffCol=numCols; // rotate through the grid
		numDiff--;
	}
	numCurCol=0;
	int numCurRow=0;
	int curX=0;
	int curY=0;
	// the following code will size everything based on my grid above
	// there is no limit to the number of windows it will handle
	// it's great when a kick-ass theory works!!!                      // Pragma :)
	int xQuantum=width()/numCols;
	int yQuantum=height()/numRows[numCurCol];
	for(KviMdiChild *lpC=m_pZ->first();lpC;lpC=m_pZ->next()){
		if(lpC->state() != KviMdiChild::Minimized){
			if(lpC->state()==KviMdiChild::Maximized)lpC->restore();
			lpC->setGeometry(curX,curY,xQuantum,yQuantum);
			numCurRow++;
			curY+=yQuantum;
			if(numCurRow==numRows[numCurCol]){
				numCurRow=0;
				numCurCol++;
				curY=0;
				curX+=xQuantum;
				if(numCurCol!=numCols)yQuantum=height()/numRows[numCurCol];
			}
		}
	}
	delete[] numRows;
	if(lpTop)lpTop->setFocus();
}

#include "kvi_mdimanager.moc"
