//
//   File : kvi_mdichild.cpp
//   Creation date : Wed Jun 21 2000 17:35:45 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
#define __KVIRC__

#define _KVI_DEBUG_CHECK_RANGE_
#include "kvi_debug.h"
#include "kvi_mdichild.h"
#include "kvi_mdimanager.h"
#include "kvi_string.h"
#include "kvi_locale.h"
#include "kvi_options.h"
#include "kvi_settings.h"
#include "kvi_iconmanager.h"
#include "kvi_window.h"

#include <qcursor.h>
#include <qnamespace.h>
#include <qapplication.h>
#include <qfontmetrics.h>
#include "kvi_list.h"
#include <qpixmap.h>
#include <qstyle.h>
#include <qpainter.h>
#include <qpopupmenu.h>
#include <qobjectlist.h>

#ifdef COMPILE_PSEUDO_TRANSPARENCY
	extern QPixmap * g_pShadedChildGlobalDesktopBackground;
#endif


KviMdiCaptionButton::KviMdiCaptionButton(const QPixmap &pix,QWidget * parent,const char * name)
: QToolButton(parent,name)
{
	setPixmap(pix);
	setAutoRaise(true);
	setMinimumSize(20,20);
}

KviMdiCaptionButton::~KviMdiCaptionButton()
{
}

void KviMdiCaptionButton::drawButton(QPainter *p)
{
	if (uses3D() || isDown() || isOn())
	{
#if QT_VERSION >= 300
		int flags = 0;
		if(isOn())flags |= QStyle::Style_On;
		if(isDown())flags |= QStyle::Style_Down;
		if(isEnabled())flags |= QStyle::Style_Enabled;
		style().drawPrimitive(QStyle::PE_ButtonTool,p,rect(),colorGroup(),flags);
#else
		style().drawToolButton( this, p );
#endif
	} else if ( parentWidget() && parentWidget()->backgroundPixmap() ){
		// pseudo tranparency
		p->drawTiledPixmap(0,0,width(),height(),*parentWidget()->backgroundPixmap(),x(),y());
	}
	drawButtonLabel(p);
}

KviMdiCaption::KviMdiCaption(KviMdiChild * parent,const char * name)
: QLabel(parent,name)
{
	setFrameStyle(QFrame::Sunken | QFrame::Panel);
	setMargin(0);

	m_lastMousePos = QPoint(-1,-1);
	m_bMouseGrabbed = true;
	m_bActive = false;

	QFontMetrics fm(font());
	m_iLineSpacing = fm.lineSpacing() + 4;
	if(m_iLineSpacing < 20)m_iLineSpacing = 20;
}

KviMdiCaption::~KviMdiCaption()
{
}

int KviMdiCaption::heightHint()
{
	return m_iLineSpacing;
}

void KviMdiCaption::setActive(bool bActive)
{
	if(bActive)
	{
		setText(((KviMdiChild *)parent())->xmlActiveCaption());
		setFrameStyle(QFrame::Raised | QFrame::Panel);
		setBackgroundColor(KVI_OPTION_COLOR(KviOption_colorMdiCaptionActive));
	} else {
		setText(((KviMdiChild *)parent())->xmlInactiveCaption());
		setFrameStyle(QFrame::Sunken | QFrame::Panel);
		setBackgroundColor(KVI_OPTION_COLOR(KviOption_colorMdiCaptionInactive));
	}
	m_bActive = bActive;
}

void KviMdiCaption::fontChange(const QFont &old)
{
	QFontMetrics fm(font());
	m_iLineSpacing = fm.lineSpacing() + 4;
	if(m_iLineSpacing < 18)m_iLineSpacing = 18;
	QLabel::fontChange(old);
	((KviMdiChild *)parent())->resizeEvent(0);
}

void KviMdiCaption::mousePressEvent(QMouseEvent *)
{
	m_bMouseGrabbed = true;
	m_lastMousePos = QCursor::pos();
	grabMouse(Qt::sizeAllCursor);
}

void KviMdiCaption::mouseMoveEvent(QMouseEvent *)
{
	if(m_bMouseGrabbed)
	{
		QPoint p = QCursor::pos();
		int dx = m_lastMousePos.x() - p.x();
		int dy = m_lastMousePos.y() - p.y();
		KviMdiChild * c = (KviMdiChild *)parent();
		c->move(c->pos().x() - dx,c->pos().y() - dy);
		m_lastMousePos = p;
	}
}

void KviMdiCaption::mouseReleaseEvent(QMouseEvent *)
{
	m_bMouseGrabbed = false;
	releaseMouse();
}



#define KVI_MDI_NORESIZE 0
#define KVI_MDI_RESIZE_TOP 1
#define KVI_MDI_RESIZE_LEFT 2
#define KVI_MDI_RESIZE_RIGHT 4
#define KVI_MDI_RESIZE_BOTTOM 8
#define KVI_MDI_RESIZE_TOPLEFT (1|2)
#define KVI_MDI_RESIZE_TOPRIGHT (1|4)
#define KVI_MDI_RESIZE_BOTTOMLEFT (8|2)
#define KVI_MDI_RESIZE_BOTTOMRIGHT (8|4)

KviMdiChild::KviMdiChild(KviMdiManager * par,const char * name)
: QFrame(par,name ? name : "mdi_child")
{
	setFrameStyle(QFrame::WinPanel | QFrame::Raised);

	m_pMaximizeButton    = new KviMdiCaptionButton(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_MAXIMIZE)),this,"maximize_button");
	connect(m_pMaximizeButton,SIGNAL(clicked()),this,SLOT(maximize()));
	m_pMinimizeButton    = new KviMdiCaptionButton(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_MINIMIZE)),this,"minimize_button");
	connect(m_pMinimizeButton,SIGNAL(clicked()),this,SLOT(minimize()));
	m_pCloseButton       = new KviMdiCaptionButton(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_CLOSE)),this,"close_button");
	connect(m_pCloseButton,SIGNAL(clicked()),this,SLOT(closeRequest()));
	m_pIconButton        = new KviMdiCaptionButton(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_DEFAULTICON)),this,"icon_button");
	connect(m_pIconButton,SIGNAL(clicked()),this,SLOT(systemPopupSlot()));
	m_pCaptionLabel      = new KviMdiCaption(this,"mdi_caption");

	m_bResizeMode        = false;
	m_iResizeCorner      = KVI_MDI_NORESIZE;
	m_iLastCursorCorner  = KVI_MDI_NORESIZE;

	m_pClient            = 0;
	m_state              = Normal;
	m_restoredGeometry   = QRect(10,10,100,100);

//	m_pSystemPopup       = new QPopupMenu(this);
//	connect(m_pSystemPopup,SIGNAL(aboutToShow()),this,SLOT(systemPopupAboutToShow()));

	setMouseTracking(true);
	setMinimumSize(KVI_MDICHILD_MIN_WIDTH,KVI_MDICHILD_MIN_HEIGHT);
}

KviMdiChild::~KviMdiChild()
{
	if(m_pClient)delete m_pClient;
}

void KviMdiChild::setBackgroundMode(QWidget::BackgroundMode)
{
	QFrame::setBackgroundMode(QWidget::PaletteBackground);
}

//void KviMdiChild::die()
//{
//	((KviMdiManager *)parent())->destroyChild(this,true);
//}

void KviMdiChild::setIcon(const QPixmap &pix)
{
	m_pIconButton->setPixmap(pix);
}

const QPixmap * KviMdiChild::icon()
{
	return m_pIconButton->pixmap();
}

void KviMdiChild::enableClose(bool bEnable)
{
	m_pCloseButton->setEnabled(bEnable);
}

bool KviMdiChild::closeEnabled()
{
	return m_pCloseButton->isEnabled();
}

void KviMdiChild::setCaption(const char * plain,const char * xmlActive,const char * xmlInactive)
{
	if(plain)m_szPlainCaption = plain;
	if(xmlActive)m_szXmlActiveCaption = xmlActive;
	if(xmlInactive)m_szXmlInactiveCaption = xmlInactive;
	m_pCaptionLabel->setActive(m_pCaptionLabel->active());
}

void KviMdiChild::maximize()
{
	if(m_state == Minimized)restore(); // restore first
	if(m_state == Normal)m_restoredGeometry = geometry();
	setGeometry(- KVI_MDICHILD_HIDDEN_EDGE,- (KVI_MDICHILD_HIDDEN_EDGE + KVI_MDICHILD_SPACING + m_pCaptionLabel->heightHint()),
		((KviMdiManager *)parent())->width() + KVI_MDICHILD_DOUBLE_HIDDEN_EDGE,
		((KviMdiManager *)parent())->height() + KVI_MDICHILD_DOUBLE_HIDDEN_EDGE + m_pCaptionLabel->heightHint() + KVI_MDICHILD_SPACING);
	m_state = Maximized;
	((KviMdiManager *)parent())->childMaximized(this);
}

void KviMdiChild::restore()
{
	if(m_restoredGeometry.x() < 0)m_restoredGeometry.setX(0);
	if(m_restoredGeometry.y() < 0)m_restoredGeometry.setY(0);
	if((m_restoredGeometry.x() + m_restoredGeometry.width()) > ((KviMdiManager *)parent())->width())
		m_restoredGeometry.setWidth(((KviMdiManager *)parent())->width() - m_restoredGeometry.x());
	if((m_restoredGeometry.y() + m_restoredGeometry.height()) > ((KviMdiManager *)parent())->height())
		m_restoredGeometry.setHeight(((KviMdiManager *)parent())->height() - m_restoredGeometry.y());

	switch(m_state)
	{
		case Maximized:
			setGeometry(m_restoredGeometry);
			m_state = Normal;
			((KviMdiManager *)parent())->childRestored(this,true);
		break;
		case Minimized:
			setGeometry(m_restoredGeometry);
			show();
			m_state = Normal;
			((KviMdiManager *)parent())->childRestored(this,false);
		break;
		case Normal:
			m_state = Normal;
			if(!isVisible())show();
			return;
		break;
	}
}

void KviMdiChild::minimize()
{
	switch(m_state)
	{
		case Maximized:
			hide();
			m_state = Minimized;
			((KviMdiManager *)parent())->childMinimized(this,true);
		break;
		case Normal:
			m_restoredGeometry = geometry();
			hide();
			m_state = Minimized;
			((KviMdiManager *)parent())->childMinimized(this,false);
		break;
		case Minimized:
			m_state = Minimized;
			if(isVisible())hide();
			return;
		break;
	}
}

void KviMdiChild::closeRequest()
{
	if(closeEnabled())if(m_pClient)m_pClient->close();
}
/*
void KviMdiChild::systemPopupAboutToShow()
{
	m_pSystemPopup->clear();
	if(m_state != Maximized)m_pSystemPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_MAXIMIZE)),__tr("&Maximize"),this,SLOT(maximize()));
	if(m_state != Minimized)m_pSystemPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_MINIMIZE)),__tr("M&inimize"),this,SLOT(minimize()));
	if(m_state != Normal)m_pSystemPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_RESTORE)),__tr("&Restore"),this,SLOT(restore()));
	if(closeEnabled())
	{
		m_pSystemPopup->insertSeparator();
		m_pSystemPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_CLOSE)),__tr("&Close"),this,SLOT(closeRequest()));
	}
}
*/
void KviMdiChild::moveEvent(QMoveEvent *e)
{
#ifdef COMPILE_PSEUDO_TRANSPARENCY
	if(m_pClient && g_pShadedChildGlobalDesktopBackground)
	{
		if(m_pClient->inherits("KviWindow")) // actually this is always the case
		{
			((KviWindow *)m_pClient)->updateBackgrounds();
		}
	}
#endif
	QFrame::moveEvent(e);
}

void KviMdiChild::systemPopupSlot()
{
	emit systemPopupRequest(m_pIconButton->mapToGlobal(m_pIconButton->rect().bottomLeft()));
}

void KviMdiChild::resizeEvent(QResizeEvent *e)
{
	int s = m_pCaptionLabel->heightHint();
	m_pIconButton->setGeometry(KVI_MDICHILD_BORDER,KVI_MDICHILD_BORDER,s,s);
	m_pCloseButton->setGeometry(width() - (s + KVI_MDICHILD_BORDER),
		KVI_MDICHILD_BORDER,s,s);
	m_pMaximizeButton->setGeometry(width() - ((s << 1) + KVI_MDICHILD_BORDER + KVI_MDICHILD_SPACING),
		KVI_MDICHILD_BORDER,s,s);
	m_pMinimizeButton->setGeometry(width() - ((s * 3) + KVI_MDICHILD_BORDER + (KVI_MDICHILD_SPACING << 1)),
		KVI_MDICHILD_BORDER,s,s);
	m_pCaptionLabel->setGeometry(KVI_MDICHILD_BORDER + s + KVI_MDICHILD_SPACING,KVI_MDICHILD_BORDER,
		width() - ((KVI_MDICHILD_BORDER << 1) + ((s + KVI_MDICHILD_SPACING) << 2)),s);
	if(m_pClient)
	{
		int yPos = KVI_MDICHILD_BORDER + s + KVI_MDICHILD_SPACING;
		m_pClient->setGeometry(KVI_MDICHILD_BORDER,yPos,
			width() - (KVI_MDICHILD_BORDER << 1),height() - (yPos + KVI_MDICHILD_BORDER));
	}
	QFrame::resizeEvent(e);
}

void KviMdiChild::mousePressEvent(QMouseEvent *e)
{
	m_iResizeCorner=getResizeCorner(e->pos().x(),e->pos().y());
	if(m_iResizeCorner != KVI_MDI_NORESIZE)
	{
		grabMouse(getResizeCursor(m_iResizeCorner));
		m_bResizeMode = true;
	}
}

void KviMdiChild::mouseReleaseEvent(QMouseEvent *)
{
	m_iResizeCorner=KVI_MDI_NORESIZE;
	m_iLastCursorCorner=KVI_MDI_NORESIZE;
	if(m_bResizeMode)
	{
		m_bResizeMode = false;
		releaseMouse();
	}
	if(QApplication::overrideCursor())QApplication::restoreOverrideCursor();
}

QCursor KviMdiChild::getResizeCursor(int resizeCorner)
{
	switch (resizeCorner) { 
		case KVI_MDI_RESIZE_LEFT: 
		case KVI_MDI_RESIZE_RIGHT: 
			return Qt::sizeHorCursor;
			break; 
		case KVI_MDI_RESIZE_TOP: 
		case KVI_MDI_RESIZE_BOTTOM: 
			return Qt::sizeVerCursor;
			break; 
		case KVI_MDI_RESIZE_TOPLEFT: 
		case KVI_MDI_RESIZE_BOTTOMRIGHT: 
			return Qt::sizeFDiagCursor;
			break; 
		case KVI_MDI_RESIZE_BOTTOMLEFT: 
		case KVI_MDI_RESIZE_TOPRIGHT: 
			return Qt::sizeBDiagCursor;
			break; 
		default:
			return Qt::arrowCursor;
			break;
	}
}

void KviMdiChild::mouseMoveEvent(QMouseEvent *e)
{
	if(e->state() & LeftButton){
		if(m_iResizeCorner)resizeWindowOpaque(m_iResizeCorner);
	} else setResizeCursor(getResizeCorner(e->pos().x(), e->pos().y()));
}

void KviMdiChild::setResizeCursor(int resizeCorner)
{
	if(resizeCorner == m_iLastCursorCorner)return; //Don't do it twice
	m_iLastCursorCorner = resizeCorner;
	if(resizeCorner == KVI_MDI_NORESIZE){
		if(QApplication::overrideCursor())QApplication::restoreOverrideCursor(); 
	} else QApplication::setOverrideCursor(getResizeCursor(resizeCorner),true);
}

void KviMdiChild::leaveEvent(QEvent *)
{
	if(!m_bResizeMode){
		m_iResizeCorner=KVI_MDI_NORESIZE;
		m_iLastCursorCorner=KVI_MDI_NORESIZE;
		if(QApplication::overrideCursor())QApplication::restoreOverrideCursor();
	} else {
		if(m_iResizeCorner != KVI_MDI_NORESIZE)resizeWindowOpaque(m_iResizeCorner);
	}
}

void KviMdiChild::calculateResizeRect(int resizeCorner,QPoint mousePos,QRect &resizeRect,int minWidth,int minHeight)
{
	switch (resizeCorner){
		case KVI_MDI_RESIZE_LEFT:
			resizeRect.setLeft(mousePos.x() - 1);
			if(resizeRect.width() < minWidth)resizeRect.setLeft(resizeRect.right() - minWidth);
			break; 
		case KVI_MDI_RESIZE_RIGHT: 
			resizeRect.setRight(mousePos.x() + 1);
			if(resizeRect.width() < minWidth)resizeRect.setRight(resizeRect.left() + minWidth);
			break; 
		case KVI_MDI_RESIZE_TOP:
			resizeRect.setTop(mousePos.y() - 1);
			if(resizeRect.height() < minHeight)resizeRect.setTop(resizeRect.bottom() - minHeight);
			break; 
		case KVI_MDI_RESIZE_BOTTOM: 
			resizeRect.setBottom(mousePos.y() + 1);
			if(resizeRect.height() < minHeight)resizeRect.setBottom(resizeRect.top() + minHeight);
			break; 
		case KVI_MDI_RESIZE_BOTTOMRIGHT:
			resizeRect.setBottom(mousePos.y() + 1);
			if(resizeRect.height() < minHeight)resizeRect.setBottom(resizeRect.top() + minHeight);
			resizeRect.setRight(mousePos.x() + 1);
			if(resizeRect.width() < minWidth)resizeRect.setRight(resizeRect.left() + minWidth);
			break;
		case KVI_MDI_RESIZE_TOPRIGHT:
			resizeRect.setTop(mousePos.y() - 1);
			if(resizeRect.height() < minHeight)resizeRect.setTop(resizeRect.bottom() - minHeight);
			resizeRect.setRight(mousePos.x() + 1);
			if(resizeRect.width() < minWidth)resizeRect.setRight(resizeRect.left() + minWidth);
			break;
		case KVI_MDI_RESIZE_BOTTOMLEFT:
			resizeRect.setBottom(mousePos.y() + 1);
			if(resizeRect.height() < minHeight)resizeRect.setBottom(resizeRect.top() + minHeight);
			resizeRect.setLeft(mousePos.x() - 1);			
			if(resizeRect.width() < minWidth)resizeRect.setLeft(resizeRect.right() - minWidth);
			break;
		case KVI_MDI_RESIZE_TOPLEFT:
			resizeRect.setTop(mousePos.y() - 1);
			if(resizeRect.height() < minHeight)resizeRect.setTop(resizeRect.bottom() - minHeight);
			resizeRect.setLeft(mousePos.x() - 1);
			if(resizeRect.width() < minWidth)resizeRect.setLeft(resizeRect.right() - minWidth);
			break;
	}
}

void KviMdiChild::calculateMinimumSize(int &minWidth,int &minHeight)
{
	if(m_pClient){
		minWidth  = m_pClient->minimumSize().width() + (KVI_MDICHILD_BORDER << 1);
		minHeight = m_pClient->minimumSize().height()+ (KVI_MDICHILD_BORDER << 1)+
					m_pCaptionLabel->heightHint() + KVI_MDICHILD_SPACING;
	}
	if(minWidth<KVI_MDICHILD_MIN_WIDTH)minWidth=KVI_MDICHILD_MIN_WIDTH;
	if(minHeight<KVI_MDICHILD_MIN_HEIGHT)minHeight=KVI_MDICHILD_MIN_HEIGHT;
}

void KviMdiChild::resizeWindowOpaque(int resizeCorner)
{
	int minWidth=0;
	int minHeight=0;
	QRect resizeRect(x(),y(),width(),height());
	calculateMinimumSize(minWidth,minHeight);
	QPoint mousePos = ((KviMdiManager *)parent())->mapFromGlobal(QCursor::pos());
	calculateResizeRect(resizeCorner,mousePos,resizeRect,minWidth,minHeight);
	setGeometry(resizeRect.x(),resizeRect.y(),resizeRect.width(),resizeRect.height());
	if(m_state == Maximized){
		m_state=Normal;
		((KviMdiManager *)parent())->childRestored(this,true);
	}

}

int KviMdiChild::getResizeCorner(int ax,int ay)
{
	int ret = KVI_MDI_NORESIZE; 
	if((ax>0)&&(ax<(KVI_MDICHILD_BORDER+2))) ret |= KVI_MDI_RESIZE_LEFT; 
	if((ax<width())&&(ax>(width()-(KVI_MDICHILD_BORDER+2)))) ret |= KVI_MDI_RESIZE_RIGHT; 
	if((ay>0)&&(ay<(KVI_MDICHILD_BORDER+2))) ret |= KVI_MDI_RESIZE_TOP;
	if((ay<(height()))&&(ay>(height()-(KVI_MDICHILD_BORDER+2)))) ret |= KVI_MDI_RESIZE_BOTTOM; 
	return ret; 
}

void KviMdiChild::setClient(QWidget *w)
{
	__range_valid(m_pClient==0);
	__range_valid(w!=0);

	m_pClient = w;
	//resize to match the client
	int clientYPos=m_pCaptionLabel->heightHint()+KVI_MDICHILD_SPACING+KVI_MDICHILD_BORDER;
	resize(w->width()+(KVI_MDICHILD_BORDER << 1),w->height()+KVI_MDICHILD_BORDER+clientYPos);

	//Reparent if needed
	if(w->parent()!=this){
		//reparent to this widget , no flags , point , show it
		QPoint pnt2(KVI_MDICHILD_BORDER,clientYPos);
		w->reparent(this,pnt2,true);
	} else w->move(KVI_MDICHILD_BORDER,clientYPos);

	setFocusProxy(w);
	m_pCaptionLabel->setFocusProxy(w);
	m_pMinimizeButton->setFocusProxy(w);
	m_pMaximizeButton->setFocusProxy(w);
	m_pCloseButton->setFocusProxy(w);
	m_pIconButton->setFocusProxy(w);

	linkChildren(w);

	if(m_pClient->minimumSize().width() > KVI_MDICHILD_MIN_WIDTH && 
		m_pClient->minimumSize().height() > KVI_MDICHILD_MIN_HEIGHT){
		setMinimumWidth(m_pClient->minimumSize().width() + (KVI_MDICHILD_BORDER << 1));
		setMinimumHeight(m_pClient->minimumSize().height()+ (KVI_MDICHILD_BORDER << 1) +
					m_pCaptionLabel->heightHint() + KVI_MDICHILD_SPACING);
	}

	KviStr tmp(KviStr::Format,"mdi_child_%s",w->name());
	setName(tmp.ptr());
}

void KviMdiChild::unsetClient()
{
	__range_valid(m_pClient!=0);
	if(!m_pClient)return;
	//reparent to desktop widget , no flags , point , show it
	unlinkChildren(m_pClient);
	setFocusProxy(0); //remove the focus proxy...
	//Kewl...the reparent function has a small prob now..
	//the new toplelvel widgets gets not reenabled for dnd
	m_pClient->reparent(0,QPoint(0,0),true);
	m_pClient=0;
	setName("mdi_child");
}

void KviMdiChild::linkChildren(QWidget *w)
{
	//recursive : installs an event filter to all childrens of the widget w
	QObjectList *list = (QObjectList *)(w->children());
	if(list){
		for (unsigned int i=0; i< list->count(); i++){
			QObject *o = list->at(i);
			if(o->inherits("QWidget"))linkChildren(((QWidget *)o));
		}
	}
	w->installEventFilter(this);
}

void KviMdiChild::unlinkChildren(QWidget *w)
{
	//recursive : installs an event filter to all childrens of the widget w
	QObjectList *list = (QObjectList *)(w->children());
	if(list){
		for (unsigned int i=0; i< list->count(); i++){
			QObject *o = list->at(i);
			if(o->inherits("QWidget"))unlinkChildren(((QWidget *)o));
		}
	}
	w->removeEventFilter(this);
}

bool KviMdiChild::eventFilter(QObject *o,QEvent *e)
{
	switch(e->type()){
		case QEvent::FocusIn:
			// FIXME : We get FocusIn for invisible widgets (Create minimized query option)
			__range_valid(isVisible()); //Still not sure...but seems a Qt bug. Even_FocusIn from a HIDDEN NoBackground widget!
			m_pCaptionLabel->setActive(true);
			((KviMdiManager *)parent())->setTopChild(this,false);
			return false;
		break;
		case QEvent::Enter:
			if(QApplication::overrideCursor())QApplication::restoreOverrideCursor();
		break;
		default:
			return false;
		break;
	}
	return false; //newer here
}

void KviMdiChild::focusInEvent(QFocusEvent *)
{
	// We gained focus by click , tab or from the caption label
	// Bring this child to top
	m_pCaptionLabel->setActive(true);
	((KviMdiManager *)parent())->setTopChild(this,false); //Do not focus by now...
	/*The client is our focusProxy ! it should be focused by Qt !*/
#ifdef _KVI_DEBUG_CLASS_NAME_
	__range_valid(focusProxy() == m_pClient);
#endif
}

QSize KviMdiChild::sizeHint()
{
	if(m_pClient)
	{
		QSize s = m_pClient->sizeHint();
		QSize ret(s.width() + (KVI_MDICHILD_BORDER << 1),
				s.height() + (KVI_MDICHILD_BORDER << 1) + KVI_MDICHILD_SPACING + m_pCaptionLabel->heightHint());
		return ret;
	}
	return QFrame::sizeHint();
}

#include "kvi_mdichild.moc"
