//
//   File : kvi_input.cpp
//   Creation date : Sun Jan 3 1999 23:11:50 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2002 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#define __KVIRC__
#define _KVI_DEBUG_CHECK_RANGE_
#include "kvi_debug.h"

#define _KVI_INPUT_CPP_

#include "kvi_input.h"

#include "kvi_options.h"
#include "kvi_app.h"
#include "kvi_settings.h"
#include "kvi_defaults.h"
#include "kvi_colorwin.h"
#include "kvi_texticonwin.h"
#include "kvi_window.h"
#include "kvi_uparser.h"
#include "kvi_locale.h"
#include "kvi_mirccntrl.h"
#include "kvi_userlistview.h"
#include "kvi_ircview.h"
#include "kvi_console.h"
#include "kvi_out.h"

#include <ctype.h>
#include <stdlib.h>

#include <qnamespace.h>
#include <qpopupmenu.h>
#include <qclipboard.h>
#include <qdragobject.h>
#include <qstringlist.h>
#include "kvi_list.h"
#include <qpainter.h>

// FIXME: #warning "This hack is temporary...later remove it"
#if QT_VERSION >= 300
	#ifndef QT_CLEAN_NAMESPACE
		#define QT_CLEAN_NAMESPACE
		#include <qcursor.h>
		#undef QT_CLEAN_NAMESPACE
	#else
		#include <qcursor.h>
	#endif
#else
	#include <qcursor.h>
#endif


/*
	@quickhelp: KviInput
	@widget: Commandline input

	This is the commandline input widget.<br>
	<docsubtitle>Default key bindings:</docsubtitle><br>
	CTRL+B : Inserts the 'bold' mIrc control character<br>
	CTRL+K : Inserts the 'color' mIrc control character<br>
	CTRL+R : Inserts the 'reverse' mIrc control character<br>
	CTRL+U : Inserts the 'underline' mIrc control character<br>
	CTRL+O : Inserts the 'reset' mIrc control character<br>
	CTRL+P : Inserts the 'no crypt' (plain text) KVirc control character used to disable crypting of the current text line<br>
	CTRL+C : Copy selected text to clipboard<br>
	CTRL+X : Cuts the selected text<br>
	CTRL+V : Paste the clipboard contents (same as middle mouse click)<br>
	CTRL+I : Inserts the 'icon' control code and pops up the icon list box
	CursorUp : Moves backward in the commandline history<br>
	CursorDown : Moves forward in the commandline history<br>
	CursorRight : Moves cursor right<br>
	CursorLeft : Moves cursor left :)<br>
	Shift+CursorLeft : Moves selection left<br>
	Shift+RightCursor : Moves selection right<br>
	Tab : Nick completion or function/command completion (see below)<br>
	Shift+Tab : Mask completion or function/command completion (see below)<br>
	ALT+&lt;numeric_sequence&gt; : Inserts the character by ASCII code<br>
	<example>
	ALT+32 : Inserts the ASCII character 32 = space
	ALT+00032 : Same as above :)
	ALT+13 : Inserts the Carriage Return (CR) control character
	ALT+77 : Inserts the ASCII character 77 : M
	</example>
	Take a look also at the <a href="shortcuts.kvihelp">global shortcuts</a> description.<br>
	If you drop a file on this widget , a <a href="parse.kvihelp">/PARSE &lt;filename&gt;</a> will be executed.<br>
	You can enable word substitution in the misc options dialog.<br>
	For example, if you choose to substitute "afaik" with "As far as I know",<br>
	when you will type "afaik" somewhere in the commandline , and then
	press space or return , that word will be replaced with "As far as I know".<br>
	Experiment with it :)<br>
	The Tab key activates the completion of the current word.<br>
	If the word start with a '/' sign , it is treated as a command to be completed,
	if it starts with a '$' sign it is treated as a function or identifier to be completed,
	otherwise it is treated as a nickname to be completed.<br>
	<example>
		/ec&lt;tab&gt; will produce /echo&lt;space&gt
		/echo $loca&lt;tab&gt; will produce /echo $localhost
	</example>
	Multiple matches are listed in the view window and the word is completed
	to the common part of all the matches.<br>
	<example>
		$sel&lt;tab;&gt; will find multiple matches and produce $selected
	</example>
	Experiment with that too :)
*/

//This comes from kvi_app.cpp
extern KviColorWindow    * g_pColorWindow;
extern KviTextIconWindow * g_pTextIconWindow;
extern QPopupMenu        * g_pInputPopup;
extern QPixmap           * g_pInputMemBuffer;

// the list ircview widgets
extern KviPtrList<KviInput>   * g_pInputWidgetList;

// This comes from the KviIrcView class
extern const char * getColorBytes(const char *data_ptr,unsigned char *byte_1,unsigned char *byte_2);

#ifdef COMPILE_PSEUDO_TRANSPARENCY
	extern QPixmap * g_pShadedChildGlobalDesktopBackground;
#endif

//=============== Construktor ==============//

KviInput::KviInput(KviWindow *par,KviUserListView * view)
:QWidget(par,"input")
{
	// register this Input widget
	g_pInputWidgetList->append(this);

	m_iCursorPosition      = 0;                             //Index of the char AFTER the cursor
	m_iFirstVisibleChar    = 0;                             //Index of the first visible character
	m_iSelectionBegin      = -1;                            //Index of the first char in the selection
	m_iSelectionEnd        = -1;                            //Index of the last char in the selection
	m_bCursorOn            = false;                         //Cursor state
	m_iCursorTimer         = 0;                             //Timer that iverts the cursor state
	m_iDragTimer           = 0;                             //Timer for drag selection updates
	m_iLastCursorXPosition = KVI_INPUT_BORDER;              //Calculated in paintEvent
	m_iSelectionAnchorChar = -1;                            //Character clicked at the beginning of the selection process
	m_pHistory             = new KviPtrList<KviStr>;        //History buffer
	m_pHistory->setAutoDelete(true);
	m_iCurHistoryIdx       = -1;                            //No data in the history
	m_bUpdatesEnabled      = true;
	m_pKviWindow           = par;
	m_bLastEventWasASubstitution = false;
	m_pUserListView        = view;
	setBackgroundMode(NoBackground);
	setFocusPolicy(StrongFocus);
	setAcceptDrops(true);
	resizeMemBuffer();
	applyOptions();
}

//================= Destruktor ================//

KviInput::~KviInput()
{
	g_pInputWidgetList->removeRef(this);
	resizeMemBuffer();
	delete m_pHistory;
	if(m_iCursorTimer)killTimer(m_iCursorTimer);
	killDragTimer();
}

void KviInput::applyOptions()
{
	recalcFontVariables(KVI_OPTION_FONT(KviOption_fontInput));
	update();
}
 
void KviInput::recalcFontVariables(const QFont &f)
{
// FIXME: #warning "OPTIMIZE THIS: GLOBAL VARIABLES"
	QFontMetrics fm(f);
	for(unsigned int i=0;i<256;i++){
		m_iInputFontCharacterWidth[i]=fm.width((char)i);
	}
	m_iInputFontLineSpacing         =fm.lineSpacing();
	m_iInputFontBaseLineOffset      =fm.descent();
	m_iInputFontLineWidth           =fm.lineWidth();
	m_iInputFontHeight              =fm.height();
	m_iInputFontCharacterWidth[KVI_TEXT_REVERSE]=fm.width("R")+4;
	m_iInputFontCharacterWidth[KVI_TEXT_RESET]=fm.width("O")+4;
	m_iInputFontCharacterWidth[KVI_TEXT_UNDERLINE]=fm.width("U")+4;
	m_iInputFontCharacterWidth[KVI_TEXT_BOLD]=fm.width("B")+4;
	m_iInputFontCharacterWidth[KVI_TEXT_COLOR]=fm.width("K")+4;
	m_iInputFontCharacterWidth[KVI_TEXT_CRYPT]=fm.width("P")+4;
	m_iInputFontCharacterWidth[KVI_TEXT_ICON]=fm.width("I")+4;
}

void KviInput::resizeMemBuffer()
{
	// check if we can make the mem buffer a bit smaller (save some memory)
	int maxw = 16;
	int maxh = 16;
	for(KviInput *i=g_pInputWidgetList->first();i;i=g_pInputWidgetList->next())
	{
		if(i->width() > maxw)maxw = i->width();
		if(i->height() > maxh)maxh = i->height();
	}
	if((maxw != g_pInputMemBuffer->width())||(maxh != g_pInputMemBuffer->height()))
	{
		g_pInputMemBuffer->resize(maxw,maxh);
	}

	// Ok....mem buffer is big enough for all the irc views alive
}


void KviInput::dragEnterEvent(QDragEnterEvent *e)
{
	if(QUriDrag::canDecode(e))
	{
		e->accept(true);
// FIXME: #warning "FIX THIS COMMENTED STUFF"
/*
		m_pKviWindow->m_pFrm->m_pStatusBar->tempText(__tr("Drop the file to /PARSE it"),5000);
*/
	} else e->accept(false);
}

void KviInput::dropEvent(QDropEvent *e)
{
	QStringList list;
	if(QUriDrag::decodeLocalFiles(e,list))
	{
		//debug("Local files decoded");
		if(!list.isEmpty())
		{
			//debug("List not empty");
			QStringList::ConstIterator it = list.begin(); //kewl ! :)
    		for( ; it != list.end(); ++it )
			{
				KviStr tmp = *it; //wow :)
#ifndef COMPILE_ON_WINDOWS
				if(*(tmp.ptr()) != '/')tmp.prepend("/"); //HACK HACK HACK for Qt bug (?!?)
#endif
				tmp.prepend("/PARSE \"");
				tmp.append("\"");
				//debug("Parsing %s",tmp.ptr());
				g_pUserParser->parseUserCommand(tmp,m_pKviWindow);
			}
		}
	}
}

int KviInput::heightHint() const
{
	return m_iInputFontLineSpacing+(KVI_INPUT_BORDER*2);
}

#define KVI_INPUT_DEF_BACK 100
#define KVI_INPUT_DEF_FORE 101

void KviInput::paintEvent(QPaintEvent *)
{
	if(!isVisible())return;

	int widgetWidth       = width();
	int widgetHeight      = height();

	QPainter pa(g_pInputMemBuffer);

#ifdef COMPILE_PSEUDO_TRANSPARENCY
	if(g_pShadedChildGlobalDesktopBackground)
	{
		QPoint pnt = mapToGlobal(QPoint(0,0));
		pa.drawTiledPixmap(0,0,widgetWidth,widgetHeight,*g_pShadedChildGlobalDesktopBackground,pnt.x(),pnt.y());
	} else {
#endif
		if(KVI_OPTION_PIXMAP(KviOption_pixmapInputBackground).pixmap())
		{
			QPoint pnt = mapToGlobal(QPoint(0,0));
			pa.drawTiledPixmap(0,0,widgetWidth,widgetHeight,*(KVI_OPTION_PIXMAP(KviOption_pixmapInputBackground).pixmap()),pnt.x(),pnt.y());
		} else {
			pa.fillRect(0,0,widgetWidth,widgetHeight,KVI_OPTION_COLOR(KviOption_colorInputBackground));
		}
#ifdef COMPILE_PSEUDO_TRANSPARENCY
	}
#endif

	pa.setFont(KVI_OPTION_FONT(KviOption_fontInput));

	int curXPos      = KVI_INPUT_BORDER;
	int maxXPos      = widgetWidth - KVI_INPUT_BORDER;
	m_iCurBack       = KVI_INPUT_DEF_BACK; //transparent
	m_iCurFore       = KVI_INPUT_DEF_FORE; //normal fore color
	m_bCurBold       = false;
	m_bCurUnderline  = false;

	int bottom       = widgetHeight - KVI_INPUT_BORDER;
	int textBaseline = bottom - m_iInputFontBaseLineOffset;
	int top          = bottom - m_iInputFontHeight;

	register char *p = runUpToTheFirstVisibleChar();
	int charIdx = m_iFirstVisibleChar;

	pa.setClipRect(KVI_INPUT_BORDER,KVI_INPUT_BORDER,widgetWidth - (KVI_INPUT_BORDER << 1),widgetHeight - (KVI_INPUT_BORDER << 1) + 1);

	//Control the selection state
	if((m_iSelectionEnd < m_iSelectionBegin)||(m_iSelectionEnd == -1)||(m_iSelectionBegin == -1))
	{
		m_iSelectionEnd = -1;
		m_iSelectionBegin = -1;
	}


	while(extractNextBlock(p,curXPos,maxXPos) && (curXPos < maxXPos))
	{
		if(m_bControlBlock)
		{
			//Only a control char

			if((charIdx >= m_iSelectionBegin) && (charIdx <= m_iSelectionEnd))
			{
				//Paint the selected control char
				pa.fillRect(curXPos,top,m_iBlockWidth,m_iInputFontHeight,KVI_OPTION_COLOR(KviOption_colorInputSelectionBackground));
			}

			pa.setPen(KVI_OPTION_COLOR(KviOption_colorInputControl));

			QString s = QChar(getSubstituteChar(*p));

			pa.drawText(curXPos + 2,textBaseline,s,1);

			pa.drawLine(curXPos,top,curXPos+m_iBlockWidth-1,top);
			pa.drawLine(curXPos,top,curXPos,bottom);
			pa.drawLine(curXPos,bottom,curXPos+m_iBlockWidth,bottom);
			pa.drawLine(curXPos+m_iBlockWidth-1,top,curXPos+m_iBlockWidth-1,bottom);

			p       += m_iBlockLen;
			curXPos += m_iBlockWidth;
			charIdx += m_iBlockLen;
		} else {
			//Paint a normal string
			//First check if it is in the selection
			if((m_iSelectionBegin != -1)&&(m_iSelectionEnd != -1)&&
				(charIdx <= m_iSelectionEnd)&&(charIdx+m_iBlockLen > m_iSelectionBegin))
			{
				//Selection in this block!
				int auxWidth = 0;
				int auxLen   = 0;
				char *aux_ptr=p;
				if(charIdx < m_iSelectionBegin)
				{
					//But have non selected stuff before
					while((charIdx < m_iSelectionBegin)&&(auxLen < m_iBlockLen))
					{
						auxWidth +=m_iInputFontCharacterWidth[(unsigned char)*aux_ptr];
						aux_ptr++;
						charIdx++;
						auxLen++;
					}
					drawTextBlock(&pa,curXPos,textBaseline,p,auxLen,auxWidth);
					curXPos+=auxWidth;
					p=aux_ptr;
					m_iBlockLen -= auxLen;
					auxWidth = 0;
					auxLen   = 0;
				}
				//Selection
				while((charIdx <= m_iSelectionEnd)&&(auxLen < m_iBlockLen)){
					auxWidth +=m_iInputFontCharacterWidth[(unsigned char)*aux_ptr];
					aux_ptr++;
					charIdx++;
					auxLen++;
				}

				pa.fillRect(curXPos,top,auxWidth,m_iInputFontHeight,KVI_OPTION_COLOR(KviOption_colorInputSelectionBackground));
				pa.setPen(KVI_OPTION_COLOR(KviOption_colorInputSelectionForeground));
				pa.drawText(curXPos,textBaseline,__c2q(p),auxLen);

				curXPos+=auxWidth;
				p=aux_ptr;

				m_iBlockLen -= auxLen;
				//Last part
				if(m_iBlockLen > 0)
				{
					auxWidth = 0;
					auxLen   = 0;
					while(auxLen < m_iBlockLen)
					{
						auxWidth +=m_iInputFontCharacterWidth[(unsigned char)*aux_ptr];
						aux_ptr++;
						charIdx++;
						auxLen++;
					}
					drawTextBlock(&pa,curXPos,textBaseline,p,auxLen,auxWidth);
					curXPos+=auxWidth;
					p=aux_ptr;
				}
			} else {
				//No selection
				drawTextBlock(&pa,curXPos,textBaseline,p,m_iBlockLen,m_iBlockWidth);
				p       += m_iBlockLen;
				curXPos += m_iBlockWidth;
				charIdx += m_iBlockLen;
			}
		}
	}

	pa.setClipping(false);

	//Now the cursor
	m_iLastCursorXPosition = KVI_INPUT_BORDER;
	m_iBlockLen = m_iFirstVisibleChar;
	p = m_szTextBuffer.ptr()+m_iFirstVisibleChar;
	while(m_iBlockLen < m_iCursorPosition && *p)
	{
		m_iLastCursorXPosition+=m_iInputFontCharacterWidth[(unsigned char) *p];
		m_iBlockLen++;
		p++;
	}
	if(m_bCursorOn)
	{
		pa.setPen(KVI_OPTION_COLOR(KviOption_colorInputCursor));
		pa.drawLine(m_iLastCursorXPosition,KVI_INPUT_BORDER-1,m_iLastCursorXPosition,(widgetHeight-KVI_INPUT_BORDER)+2);
	} else {
		pa.setPen(KVI_OPTION_COLOR(KviOption_colorInputForeground));
	}

	pa.drawLine(m_iLastCursorXPosition-2,KVI_INPUT_BORDER-2,m_iLastCursorXPosition+3,KVI_INPUT_BORDER-2);
	pa.drawLine(m_iLastCursorXPosition-3,KVI_INPUT_BORDER-3,m_iLastCursorXPosition+4,KVI_INPUT_BORDER-3);
	pa.drawLine(m_iLastCursorXPosition-1,KVI_INPUT_BORDER-1,m_iLastCursorXPosition+2,KVI_INPUT_BORDER-1);
	pa.drawLine(m_iLastCursorXPosition,KVI_INPUT_BORDER,m_iLastCursorXPosition+1,KVI_INPUT_BORDER);

	//Need to draw the sunken rect around the view now...
	pa.setPen(colorGroup().dark());
	pa.drawLine(0,0,widgetWidth,0);
	pa.drawLine(0,0,0,widgetHeight);
	pa.setPen(colorGroup().light());
	pa.drawLine(1,widgetHeight-1,widgetWidth-1,widgetHeight-1);
	pa.drawLine(widgetWidth-1,1,widgetWidth-1,widgetHeight);
	//COPY TO THE DISPLAY

	bitBlt(this,0,0,g_pInputMemBuffer,0,0,widgetWidth,widgetHeight,Qt::CopyROP);

}

void KviInput::drawTextBlock(QPainter * pa,int curXPos,int textBaseline,char *p,int len,int wdth)
{
	if(m_iCurFore == KVI_INPUT_DEF_FORE)
	{
		pa->setPen(KVI_OPTION_COLOR(KviOption_colorInputForeground));
	} else {
		if(((unsigned char)m_iCurFore) > 16)
		{
			pa->setPen(KVI_OPTION_COLOR(KviOption_colorInputBackground));
		} else {
			pa->setPen(KVI_OPTION_MIRCCOLOR((unsigned char)m_iCurFore));
		}
	}

	if(m_iCurBack != KVI_INPUT_DEF_BACK)
	{
		if(((unsigned char)m_iCurBack) > 16)
		{
			pa->fillRect(curXPos,height() - (KVI_INPUT_BORDER + m_iInputFontHeight),wdth,m_iInputFontHeight,KVI_OPTION_COLOR(KviOption_colorInputForeground));
		} else {
			pa->fillRect(curXPos,height() - (KVI_INPUT_BORDER + m_iInputFontHeight),wdth,m_iInputFontHeight,KVI_OPTION_MIRCCOLOR((unsigned char)m_iCurBack));
		}
	}
	pa->drawText(curXPos,textBaseline,__c2q(p),len);

	if(m_bCurBold)pa->drawText(curXPos+1,textBaseline,__c2q(p),len);
	if(m_bCurUnderline)
	{
		pa->drawLine(curXPos,textBaseline+m_iInputFontBaseLineOffset,curXPos+wdth,textBaseline+m_iInputFontBaseLineOffset);
	}
}

char KviInput::getSubstituteChar(char control_code)
{
	switch(control_code)
	{
		case KVI_TEXT_COLOR:
			return 'K';
			break;
		case KVI_TEXT_BOLD:
			return 'B';
			break;
		case KVI_TEXT_RESET:
			return 'O';
			break;
		case KVI_TEXT_REVERSE:
			return 'R';
			break;
		case KVI_TEXT_UNDERLINE:
			return 'U';
			break;
		case KVI_TEXT_CRYPT:
			return 'P';
			break;
		case KVI_TEXT_ICON:
			return 'I';
			break;
		default:
			return control_code;
			break;
	}
}

bool KviInput::extractNextBlock(char *p,int curXPos,int maxXPos)
{
	m_iBlockLen = 0;
	m_iBlockWidth = 0;
	if((*p) && (*p != KVI_TEXT_COLOR) && (*p != KVI_TEXT_BOLD) && (*p != KVI_TEXT_UNDERLINE) &&
		(*p != KVI_TEXT_RESET) && (*p != KVI_TEXT_REVERSE) && (*p != KVI_TEXT_CRYPT) && (*p != KVI_TEXT_ICON))
	{
		m_bControlBlock = false;
		//Not a control code...run..
		while((*p) && (curXPos < maxXPos) && (*p != KVI_TEXT_COLOR) &&
				(*p != KVI_TEXT_BOLD) && (*p != KVI_TEXT_UNDERLINE) &&
				(*p != KVI_TEXT_RESET) && (*p != KVI_TEXT_REVERSE) &&
				(*p != KVI_TEXT_CRYPT) && (*p != KVI_TEXT_ICON))
		{
			m_iBlockLen++;
			m_iBlockWidth+=m_iInputFontCharacterWidth[(unsigned char)*p];
			curXPos      +=m_iInputFontCharacterWidth[(unsigned char)*p];
			p++;
		}
		return true;
	} else {
		if(*p == '\0')return false;
		else {
			m_bControlBlock = true;
			m_iBlockLen = 1;
			m_iBlockWidth = m_iInputFontCharacterWidth[(unsigned char)*p];
			//Control code
			switch(*p)
			{
				case KVI_TEXT_BOLD:
					m_bCurBold = ! m_bCurBold;
					return true;
					break;
				case KVI_TEXT_UNDERLINE:
					m_bCurUnderline = ! m_bCurUnderline;
					return true;
					break;
				case KVI_TEXT_RESET:
					m_iCurFore = KVI_INPUT_DEF_FORE;
					m_iCurBack = KVI_INPUT_DEF_BACK;
					m_bCurBold = false;
					m_bCurUnderline = false;
					return true;
					break;
				case KVI_TEXT_REVERSE:{
					char auxClr = m_iCurFore;
					m_iCurFore  = m_iCurBack;
					m_iCurBack  = auxClr;}
					return true;
					break;
				case KVI_TEXT_CRYPT:
				case KVI_TEXT_ICON:
					return true; // makes a single block
					break;
				case KVI_TEXT_COLOR:
				{
					p++;
					unsigned char fore;
					unsigned char back;
					getColorBytes(p,&fore,&back);
					if(fore != KVI_NOCHANGE)m_iCurFore = fore;
					else m_iCurFore = KVI_INPUT_DEF_FORE;
					if(back != KVI_NOCHANGE)m_iCurBack = back;
					else m_iCurBack = KVI_INPUT_DEF_BACK;
					// Begin Edit by GMC-jimmy: Added condition for handling a CTRL+K code when missing numbers.
					// When KVIrc encounters a CTRL+K code without any trailing numbers, we then use KVIrc's default color value defined by the user in the Options dialog.
					// This is to allow KVIrc to handle mIRC color codes in a similar fashion to most other modern irc clients.
					// See also kvi_ircview.cpp
					// Pragma: merged the code in the lines above (optimized)
					return true;
				}
				break;
				default:
					debug("Ops..");
					exit(0);
					break;
			}
		}
	}
	return true;
}

char * KviInput::runUpToTheFirstVisibleChar()
{
	register int idx = 0;
	register char *p = m_szTextBuffer.ptr();
	while(idx < m_iFirstVisibleChar)
	{
		if(*p < 32)
		{
			switch(*p)
			{
				case KVI_TEXT_BOLD:
					m_bCurBold = ! m_bCurBold;
					p++;
					break;
				case KVI_TEXT_UNDERLINE:
					m_bCurUnderline = ! m_bCurUnderline;
					p++;
					break;
				case KVI_TEXT_RESET:
					m_iCurFore = KVI_INPUT_DEF_FORE;
					m_iCurBack = KVI_INPUT_DEF_BACK;
					m_bCurBold = false;
					m_bCurUnderline = false;
					p++;
					break;
				case KVI_TEXT_REVERSE:{
					char auxClr = m_iCurFore;
					m_iCurFore  = m_iCurBack;
					m_iCurBack  = auxClr;
					p++;}
					break;
				case KVI_TEXT_COLOR:
				{
					p++;
					unsigned char fore;
					unsigned char back;
					getColorBytes(p,&fore,&back);
					if(fore != KVI_NOCHANGE)m_iCurFore = fore;
					else m_iCurFore = KVI_INPUT_DEF_FORE;
					if(back != KVI_NOCHANGE)m_iCurBack = back;
					else m_iCurBack = KVI_INPUT_DEF_BACK;
					// Begin Edit by GMC-jimmy: Added condition for handling a CTRL+K code when missing numbers.
					// When KVIrc encounters a CTRL+K code without any trailing numbers, we then use KVIrc's default color value defined by the user in the Options dialog.
					// This is to allow KVIrc to handle mIRC color codes in a similar fashion to most other modern irc clients.
					// See also kvi_ircview.cpp
					// Pragma: merged the code in the lines above (optimized)
				}
				break;
				case '0':
					debug("KviInput::Encountered invisible end of the string!");
					exit(0);
					break;
				default:
					p++;
					break;
			}
		} else p++;
		idx++;
	}
	return p;
}

void KviInput::resizeEvent(QResizeEvent *)
{
	resizeMemBuffer();
}

void KviInput::mousePressEvent(QMouseEvent *e)
{
	if(e->button() & LeftButton)
	{
		m_iCursorPosition = charIndexFromXPosition(e->pos().x());
		//move the cursor to
		int anchorX =  xPositionFromCharIndex(m_iCursorPosition);
		if(anchorX > (width()-KVI_INPUT_BORDER))m_iFirstVisibleChar++;
		m_iSelectionAnchorChar = m_iCursorPosition;
		selectOneChar(-1);
		grabMouse(QCursor(crossCursor));
		repaintWithCursorOn();
		killDragTimer();
		m_iDragTimer = startTimer(KVI_INPUT_DRAG_TIMEOUT);
	} else if(e->button() & RightButton)
	{
        int type = g_pApp->activeWindow()->type();
		//Popup menu
		g_pInputPopup->clear();
		g_pInputPopup->insertItem(__tr("Copy"),this,SLOT(copy()),0,1);
		g_pInputPopup->setItemEnabled(1,hasSelection());
		g_pInputPopup->insertItem(__tr("Paste"),this,SLOT(paste()),0,2);
		g_pInputPopup->setItemEnabled(2,(g_pApp->clipboard()->text() != 0));
        g_pInputPopup->insertItem(__tr("Paste (slowly)"),this,SLOT(pasteSlow()),0,3);
        if ((type == KVI_WINDOW_TYPE_CHANNEL) || (type == KVI_WINDOW_TYPE_QUERY) || (type == KVI_WINDOW_TYPE_DCCCHAT))
            g_pInputPopup->setItemEnabled(3,(g_pApp->clipboard()->text() != 0));
        else g_pInputPopup->setItemEnabled(3,0);
		g_pInputPopup->insertItem(__tr("Cut"),this,SLOT(cut()),0,4);
		g_pInputPopup->setItemEnabled(4,hasSelection());
		g_pInputPopup->insertSeparator();
		g_pInputPopup->insertItem(__tr("Select All"),this,SLOT(selectAll()),0,5);
		g_pInputPopup->setItemEnabled(5,(m_szTextBuffer.hasData()));
		g_pInputPopup->insertItem(__tr("Clear"),this,SLOT(clear()),0,6);
		g_pInputPopup->setItemEnabled(6,(m_szTextBuffer.hasData()));
		g_pInputPopup->popup(mapToGlobal(e->pos()));

	} else paste();
}

bool KviInput::hasSelection()
{
	return ((m_iSelectionBegin != -1)&&(m_iSelectionEnd != -1));
}

void KviInput::copy()
{
	if(hasSelection()){
		g_pApp->clipboard()->setText(
			m_szTextBuffer.middle(m_iSelectionBegin,(m_iSelectionEnd-m_iSelectionBegin)+1).ptr()
		);
	}
	repaintWithCursorOn();
}

void KviInput::moveCursorTo(int idx,bool bRepaint)
{
	if(idx < 0)idx = 0;
	if(idx > m_szTextBuffer.len())idx = m_szTextBuffer.len();
	if(idx > m_iCursorPosition)
	{
		while(m_iCursorPosition < idx)
		{
			moveRightFirstVisibleCharToShowCursor();
			m_iCursorPosition++;
		}
	} else {
		m_iCursorPosition = idx;
		if(m_iFirstVisibleChar > m_iCursorPosition)m_iFirstVisibleChar = m_iCursorPosition;
	}
	if(bRepaint)repaintWithCursorOn();
}

void KviInput::cut()
{
	if(hasSelection())
	{
		m_szTextBuffer.cut(m_iSelectionBegin,(m_iSelectionEnd-m_iSelectionBegin)+1);
		moveCursorTo(m_iSelectionBegin,false);
		selectOneChar(-1);
		repaintWithCursorOn();
	}
}

void KviInput::insertText(const KviStr & text)
{
	KviStr szText = text; // crop away constness
	if(szText.hasData())
	{
		szText.replaceAll('\t'," "); //Do not paste tabs
		m_bUpdatesEnabled = false;
		cut();
		m_bUpdatesEnabled = true;
		if(szText.findFirstIdx('\n')==-1)
		{
			m_szTextBuffer.insert(m_iCursorPosition,szText.ptr());
			if(m_szTextBuffer.len() > KVI_INPUT_MAX_BUFFER_SIZE){
				m_szTextBuffer.setLen(KVI_INPUT_MAX_BUFFER_SIZE);
			}
			moveCursorTo(m_iCursorPosition + szText.len());
		} else {
			//Multiline paste...do not execute commands here
			KviStr szBlock;
			while(szText.hasData())
			{
				int idx = szText.findFirstIdx('\n');
				if(idx != -1)
				{
					szBlock = szText.left(idx);
					szText.cutLeft(idx+1);
				} else {
					szBlock = szText.ptr();
					szText  = "";
				}
				m_szTextBuffer.insert(m_iCursorPosition,szBlock.ptr());
				if(m_szTextBuffer.len() > KVI_INPUT_MAX_BUFFER_SIZE){
					m_szTextBuffer.setLen(KVI_INPUT_MAX_BUFFER_SIZE);
				}
//				moveCursorTo(m_iCursorPosition + szBlock.len());
				const char * aux = m_szTextBuffer.ptr();
				if(*aux < 33)while(*aux && ((unsigned char)*aux) < 33)++aux;
				if(*aux == '/')m_szTextBuffer.insert(aux - m_szTextBuffer.ptr(),"\\");
				returnPressed(idx != -1);
			}
		}
	}
}

void KviInput::paste()
{
	KviStr szText=g_pApp->clipboard()->text();
	insertText(szText);
}

void KviInput::pasteSlow()
{
    g_pUserParser->parseCommandBuffer("spaste.clipboard",g_pApp->activeWindow());
}

void KviInput::selectAll()
{
	if(m_szTextBuffer.len() > 0)
	{
		m_iSelectionBegin = 0;
		m_iSelectionEnd = m_szTextBuffer.len()-1;
	}
	end();
}

void KviInput::clear()
{
	m_szTextBuffer = "";
	selectOneChar(-1);
	home();
}

void KviInput::setText(const char *text)
{
	m_szTextBuffer = text;
	if(m_szTextBuffer.len() > KVI_INPUT_MAX_BUFFER_SIZE)
	{
		m_szTextBuffer.setLen(KVI_INPUT_MAX_BUFFER_SIZE);
	}
	selectOneChar(-1);
	end();
}

void KviInput::mouseReleaseEvent(QMouseEvent *)
{
	if(m_iDragTimer)
	{
		m_iSelectionAnchorChar =-1;
		releaseMouse();
		killDragTimer();
	}
}

void KviInput::killDragTimer()
{
	if(m_iDragTimer)
	{
		killTimer(m_iDragTimer);
		m_iDragTimer = 0;
	}
}

void KviInput::timerEvent(QTimerEvent *e)
{
	if(e->timerId() == m_iCursorTimer)
	{
		if(!hasFocus() || !isVisibleToTLW())
		{
			killTimer(m_iCursorTimer);
			m_iCursorTimer = 0;
			m_bCursorOn = false;
		} else m_bCursorOn = ! m_bCursorOn;
		paintEvent(0);
	} else {
		//Drag timer
		handleDragSelection();
	}
}

void KviInput::handleDragSelection()
{
	if(m_iSelectionAnchorChar == -1)return;

	QPoint pnt = mapFromGlobal(QCursor::pos());
	

	if(pnt.x() <= 0)
	{
		//Left side dragging
		if(m_iFirstVisibleChar > 0)m_iFirstVisibleChar--;
		m_iCursorPosition = m_iFirstVisibleChar;
	} else if(pnt.x() >= width())
	{
		//Right side dragging...add a single character to the selection on the right
		if(m_iCursorPosition < m_szTextBuffer.len())
		{
			moveRightFirstVisibleCharToShowCursor();
			m_iCursorPosition++;
		} //else at the end of the selection...don't move anything
	} else {
		//Inside the window...
		m_iCursorPosition = charIndexFromXPosition(pnt.x());
	}
	if(m_iCursorPosition == m_iSelectionAnchorChar)selectOneChar(-1);
	else {
		if(m_iCursorPosition > m_iSelectionAnchorChar)
		{
			m_iSelectionBegin = m_iSelectionAnchorChar;
				m_iSelectionEnd   = m_iCursorPosition-1;
		} else {
			m_iSelectionBegin = m_iCursorPosition;
			m_iSelectionEnd   = m_iSelectionAnchorChar-1;
		}
	}
	repaintWithCursorOn();
}

void KviInput::returnPressed(bool bRepaint)
{
	KviStr *pHist = new KviStr(m_szTextBuffer.ptr());
	m_pHistory->insert(0,pHist);
	__range_valid(KVI_INPUT_HISTORY_ENTRIES > 1); //ABSOLUTELY NEEDED, if not, pHist will be destroyed...
	if(m_pHistory->count() > KVI_INPUT_HISTORY_ENTRIES)m_pHistory->removeLast();

	m_iCurHistoryIdx = -1;

	m_szTextBuffer="";

	selectOneChar(-1);

	m_iCursorPosition = 0;
	m_iFirstVisibleChar = 0;

	if(bRepaint)repaintWithCursorOn();

	g_pUserParser->parseUserCommand(*pHist,m_pKviWindow); //the parsed string must be NOT THE SAME AS m_szTextBuffer
}

void KviInput::focusInEvent(QFocusEvent *)
{
	if(m_iCursorTimer==0)
	{
		m_iCursorTimer = startTimer(KVI_INPUT_BLINK_TIME);
		m_bCursorOn = true;
		paintEvent(0);
	}
}

void KviInput::focusOutEvent(QFocusEvent *)
{
	if(m_iCursorTimer)killTimer(m_iCursorTimer);
	m_iCursorTimer = 0;
	m_bCursorOn = false;
	paintEvent(0);
}


void KviInput::keyPressEvent(QKeyEvent *e)
{
    //FIXME: Destroy _all_ ircviews tooltips not just the one associated with this input widget
    if(m_pKviWindow->view())
        m_pKviWindow->view()->inputWidgetKeyPressed();

	if((e->key() == Qt::Key_Tab) || (e->key() == Qt::Key_BackTab))
	{
		completion(e->state() & ShiftButton);
		return;
	} else if(e->key() != Qt::Key_Shift)m_szLastCompletedNick = "";

	if (e->key()!=Qt::Key_Backspace)m_bLastEventWasASubstitution=false;


	if(e->state() & ControlButton)
	{
		//debug("CTRL");
		switch(e->key())
		{
			case Qt::Key_K:
			{
				insertChar(KVI_TEXT_COLOR);
				int xPos = xPositionFromCharIndex(m_iCursorPosition);
				if(xPos > 24)xPos-=24;
				if(!g_pColorWindow)g_pColorWindow = new KviColorWindow();
				if(xPos+g_pColorWindow->width() > width())xPos = width()-(g_pColorWindow->width()+2);
				g_pColorWindow->move(mapToGlobal(QPoint(xPos,-35)));
				g_pColorWindow->popup(this);
				return;
			}
			break;
			case Qt::Key_B:
				insertChar(KVI_TEXT_BOLD);
				return;
			break;
			case Qt::Key_O:
				insertChar(KVI_TEXT_RESET);
				return;
			break;
			case Qt::Key_U:
				insertChar(KVI_TEXT_UNDERLINE);
				return;
			break;
			case Qt::Key_R:
				insertChar(KVI_TEXT_REVERSE);
				return;
			break;
			case Qt::Key_P:
				insertChar(KVI_TEXT_CRYPT); // DO NOT CRYPT THIS STUFF
				return;
			break;
			case Qt::Key_I:
			{
				insertChar(KVI_TEXT_ICON); // THE NEXT WORD IS AN ICON NAME
				int xPos = xPositionFromCharIndex(m_iCursorPosition);
				if(xPos > 24)xPos-=24;
				if(!g_pTextIconWindow)g_pTextIconWindow = new KviTextIconWindow();
				if(xPos+g_pTextIconWindow->width() > width())xPos = width()-(g_pTextIconWindow->width()+2);
				g_pTextIconWindow->move(mapToGlobal(QPoint(xPos,-KVI_TEXTICON_WIN_HEIGHT)));
				g_pTextIconWindow->popup(this);
				return;
			}
			break;
			case Qt::Key_C:
				copy();
				return;
			break;
			case Qt::Key_X:
				cut();
				return;
			break;
			case Qt::Key_V:
				paste();
				return;
			break;
		}
		//if(e->ascii() > 0)insertChar(e->ascii());
		//else e->ignore();
	}

	if(e->state() & AltButton)
	{
		//debug("ALT");
		// Qt::Key_Meta seems to substitute Key_Alt on some keyboards
		if((e->key() == Qt::Key_Alt) || (e->key() == Qt::Key_Meta))
		{
			m_szAltKeyCode = "";
			return;
		} else if((e->ascii() >= '0') && (e->ascii() <= '9'))
		{
			m_szAltKeyCode += e->ascii();
			return;
		}
		//debug("%c",e->ascii());
		if(e->ascii() > 0)insertChar(e->ascii());
		else e->ignore();
		return;
	}

	if(e->state() & ShiftButton)
	{
		switch(e->key())
		{
			case Qt::Key_Insert:
				paste();
				return;
			break;
			case Qt::Key_PageUp:
				if(m_pKviWindow->view())m_pKviWindow->view()->prevLine();
				return;
			break;
			case Qt::Key_PageDown:
				if(m_pKviWindow->view())m_pKviWindow->view()->nextLine();
				return;
			break;
		}
	}

//	debug("KE:%d",e->key());

	switch(e->key())
	{
		case Qt::Key_Right:
			if(m_iCursorPosition < m_szTextBuffer.len())
			{
				moveRightFirstVisibleCharToShowCursor();
				if(e->state() & ShiftButton)
				{
					//Grow the selection if needed
					if((m_iSelectionBegin > -1)&&(m_iSelectionEnd > -1))
					{
						if(m_iSelectionEnd == m_iCursorPosition-1)m_iSelectionEnd++;
						else if(m_iSelectionBegin == m_iCursorPosition)m_iSelectionBegin++;
						else selectOneChar(m_iCursorPosition);
					} else selectOneChar(m_iCursorPosition);
				}
				m_iCursorPosition++;
				repaintWithCursorOn();
			}
			break;
		case Qt::Key_Left:
			if(m_iCursorPosition > 0)
			{
				if(e->state() & ShiftButton)
				{
					if((m_iSelectionBegin > -1)&&(m_iSelectionEnd > -1))
					{
						if(m_iSelectionBegin == m_iCursorPosition)m_iSelectionBegin--;
						else if(m_iSelectionEnd == m_iCursorPosition-1)m_iSelectionEnd--;
						else selectOneChar(m_iCursorPosition - 1);
					} else selectOneChar(m_iCursorPosition - 1);
				}
				m_iCursorPosition--;
				if(m_iFirstVisibleChar > m_iCursorPosition)m_iFirstVisibleChar--;
				repaintWithCursorOn();
			}
			break;
		case Qt::Key_Backspace:
			if(m_iCursorPosition > 0)
			{
				if(m_bLastEventWasASubstitution)
				{
					////
					// restore the old string
					m_szTextBuffer=m_szStringBeforeSubstitution;
					m_iCursorPosition=m_iCursorPositionBeforeSubstitution;
					m_iLastCursorXPosition=m_iCursorXPositionBeforeSubstitution;
					m_iFirstVisibleChar=m_iFirstVisibleCharBeforeSubstitution;
					///
					insertChar(m_cDetonationChar);
					repaintWithCursorOn();
					m_bLastEventWasASubstitution=false;
					break;
				}
				if((m_iSelectionEnd >= m_iCursorPosition) && (m_iSelectionBegin <= m_iCursorPosition))
				{
					//remove the selection
					m_szTextBuffer.cut(m_iSelectionBegin,(m_iSelectionEnd-m_iSelectionBegin)+1);
					m_iCursorPosition = m_iSelectionBegin;
					if(m_iFirstVisibleChar > m_iCursorPosition)m_iFirstVisibleChar = m_iCursorPosition;
				} else {
					m_iCursorPosition--;
					m_szTextBuffer.cut(m_iCursorPosition,1);
					if(m_iFirstVisibleChar > m_iCursorPosition)m_iFirstVisibleChar--;
				}
				selectOneChar(-1);
				repaintWithCursorOn();
			}
			break;
		case Qt::Key_Delete:
			if(m_iCursorPosition < (int)m_szTextBuffer.len())
			{
				if((m_iSelectionEnd >= m_iCursorPosition) && (m_iSelectionBegin <= m_iCursorPosition))
				{
					m_szTextBuffer.cut(m_iSelectionBegin,(m_iSelectionEnd-m_iSelectionBegin)+1);
				} else m_szTextBuffer.cut(m_iCursorPosition,1);
				selectOneChar(-1);
				repaintWithCursorOn();
			}
			break;
		case Qt::Key_Home:
			if(m_iCursorPosition > 0)
			{
				if(e->state() & ShiftButton)
				{
					if((m_iSelectionBegin == -1)&&(m_iSelectionEnd == -1))m_iSelectionEnd = m_iCursorPosition - 1;
					m_iSelectionBegin = 0;
				}
				home();
			}
			break;
		case Qt::Key_End:
			if(m_iCursorPosition < m_szTextBuffer.len())
			{
				if(e->state() & ShiftButton)
				{
					if((m_iSelectionBegin == -1)&&(m_iSelectionEnd == -1))m_iSelectionBegin = m_iCursorPosition;
					m_iSelectionEnd = m_szTextBuffer.len();
				}
				end();
			}
			break;
		case Qt::Key_Up:
			if(m_pHistory->count() > 0)
			{
				if(m_iCurHistoryIdx < 0)
				{
					m_szSaveTextBuffer = m_szTextBuffer;
					m_szTextBuffer = m_pHistory->at(0)->ptr();
					m_iCurHistoryIdx = 0;
				} else if(m_iCurHistoryIdx >= (int)(m_pHistory->count()-1))
				{
					m_szTextBuffer=m_szSaveTextBuffer;
					m_iCurHistoryIdx = -1;
				} else {
					m_iCurHistoryIdx++;
					m_szTextBuffer = m_pHistory->at(m_iCurHistoryIdx)->ptr();
				}
				selectOneChar(-1);
				if(KVI_OPTION_BOOL(KviOption_boolInputHistoryCursorAtEnd))end();
				else home();
			}
			break;
		case Qt::Key_Down:
			if(m_pHistory->count() > 0)
			{
				if(m_iCurHistoryIdx < 0)
				{
					m_szSaveTextBuffer = m_szTextBuffer;
					m_szTextBuffer = m_pHistory->at(m_pHistory->count()-1)->ptr();
					m_iCurHistoryIdx = m_pHistory->count()-1;
				} else if(m_iCurHistoryIdx == 0)
				{
					m_szTextBuffer=m_szSaveTextBuffer;
					m_iCurHistoryIdx = -1;
				} else {
					m_iCurHistoryIdx--;
					m_szTextBuffer = m_pHistory->at(m_iCurHistoryIdx)->ptr();
				}
				selectOneChar(-1);
				if(KVI_OPTION_BOOL(KviOption_boolInputHistoryCursorAtEnd))end();
				else home();
			}
			break;
		case Qt::Key_PageUp:
			if(m_pKviWindow->view())m_pKviWindow->view()->prevPage();
		break;
		case Qt::Key_PageDown:
			if(m_pKviWindow->view())m_pKviWindow->view()->nextPage();
		break;
		case Qt::Key_Return:
		case Qt::Key_Enter:
			checkForSubstitution();
			m_bLastEventWasASubstitution=false;
			returnPressed();
			break;
		case Qt::Key_Alt:
		case Qt::Key_Meta:
			m_szAltKeyCode = "";
			break;
		case Qt::Key_Space:
			m_cDetonationChar=' ';
			checkForSubstitution();
			insertChar(e->ascii());
			break;
		case Qt::Key_Period:
			m_cDetonationChar='.';
			checkForSubstitution();
			insertChar(e->ascii());
			break;
		case Qt::Key_Comma:
			m_cDetonationChar=',';
			checkForSubstitution();
			insertChar(e->ascii());
			break;
		default:
			if(e->ascii() > 0)insertChar(e->ascii());
			else e->ignore();
			break;
	}
}

void KviInput::keyReleaseEvent(QKeyEvent *e)
{
	if((e->key() == Qt::Key_Alt) || (e->key() == Qt::Key_Meta))
	{
		if(m_szAltKeyCode.hasData())
		{
			bool bOk;
			char ch = m_szAltKeyCode.toChar(&bOk);
			if(bOk && ch != 0)
			{
				insertChar(ch);
				e->accept();
			}
		}
		m_szAltKeyCode = "";
	}
	e->ignore();
}

void KviInput::getWordBeforeCursor(KviStr &buffer,bool * bIsFirstWordInLine)
{
	if(m_szTextBuffer.isEmpty() || m_iCursorPosition <= 0)
	{
		buffer = "";
		return;
	}

	buffer = m_szTextBuffer.left(m_iCursorPosition);

	int idx = buffer.findLastIdx(' ');
	int idx2 = buffer.findLastIdx(','); // This is for comma separated lists...
	int idx3 = buffer.findLastIdx('(');
	int idx4 = buffer.findLastIdx('"');
	if(idx2 > idx)idx = idx2;
	if(idx3 > idx)idx = idx3;
	if(idx4 > idx)idx = idx4;
	*bIsFirstWordInLine = false;
	if(idx > -1)buffer.cutLeft(idx+1);
	else *bIsFirstWordInLine = true;
}

void KviInput::completion(bool bShift)
{
	// FIXME: Spaces in directory completion can mess everything completly
	//        On windows the KVI_PATH_SEPARATOR_CHARacters are breaking everything...
	//        Well.... :D

	KviStr word;
	KviStr match;

	bool bFirstWordInLine;
	getWordBeforeCursor(word,&bFirstWordInLine);

	KviPtrList<KviStr> tmp;
	tmp.setAutoDelete(true);

	bool bIsCommand = false;
	bool bIsDir = false;
	bool bIsNick = false;

	if(*(word.ptr()) == '/')
	{
		if(bFirstWordInLine)
		{
			// command completion
			word.cutLeft(1);
			if(word.isEmpty())return;
			g_pUserParser->completeCommand(word,&tmp);
			bIsCommand = true;
		} else {
			// directory completion attempt
			g_pApp->completeDirectory(word,&tmp);
			bIsDir = true;
		}
	} else if(*(word.ptr()) == '$')
	{
		// function/identifer completion
		word.cutLeft(1);
		if(word.isEmpty())return;
		g_pUserParser->completeFunction(word,&tmp);
	} else if(*(word.ptr()) == '#' || *(word.ptr()) == '&')
	{
		if(m_pKviWindow->console())
			m_pKviWindow->console()->completeChannel(word,&tmp);
	} else {
		// empty word will end up here
		if(KVI_OPTION_BOOL(KviOption_boolBashLikeNickCompletion))
		{
			m_pUserListView->completeNickBashLike(word,&tmp,bShift);
			bIsNick = true;
		} else {
			standardNickCompletion(bShift,word,bFirstWordInLine);
			repaintWithCursorOn();
			return;
		}
	}

	// Lookup the longest exact match

	if(tmp.count() > 0)
	{
		if(tmp.count() == 1)
		{
			match = *(tmp.first());
			if(bIsCommand)match.append(' ');
			else if(bIsNick)
			{
				if(KVI_OPTION_STRING(KviOption_stringNickCompletionPostfix).hasData())
				{
					if(bFirstWordInLine || (!KVI_OPTION_BOOL(KviOption_boolUseNickCompletionPostfixForFirstWordOnly)))
						match.append(KVI_OPTION_STRING(KviOption_stringNickCompletionPostfix));
				}
			}
		} else {
			KviStr all;
			match = *(tmp.first());
			int wLen = word.len();
			for(KviStr * s = tmp.first();s;s = tmp.next())
			{
				if(s->len() < match.len())match.cutRight(match.len() - s->len());
				// All the matches here have length >= word.len()!!!
				char * c1 = s->ptr() + wLen;
				char * c2 = match.ptr() + wLen;
				if(bIsDir)while(*c1 && (*c1 == *c2))c1++,c2++;
				else while(*c1 && (tolower(*c1) == tolower(*c2)))c1++,c2++;
				int len = c1 - s->ptr();
				if(len < match.len())match.cutRight(match.len() - len);
				if(all.hasData())all.append(", ");
				all.append(*s);
			}
			m_pKviWindow->output(KVI_OUT_SYSTEMMESSAGE,__tr("%d matches: %s"),tmp.count(),all.ptr());
		}
	} else m_pKviWindow->outputNoFmt(KVI_OUT_SYSTEMMESSAGE,__tr("No matches"));

	if(match.hasData())
	{
		//debug("Word (%s) match (%s)",word.ptr(),match.ptr());
		if(!bIsDir && !bIsNick)match.toLower();
		m_iCursorPosition -= word.len();
		m_szTextBuffer.cut(m_iCursorPosition,word.len());
		m_szTextBuffer.insert(m_iCursorPosition,match.ptr());
		if(m_szTextBuffer.len() > KVI_INPUT_MAX_BUFFER_SIZE){
			m_szTextBuffer.setLen(KVI_INPUT_MAX_BUFFER_SIZE);
		}
		moveCursorTo(m_iCursorPosition + match.len());
	}

	repaintWithCursorOn();
}

void KviInput::replaceWordBeforeCursor(const KviStr &word,const KviStr &replacement,bool bRepaint)
{
	m_iCursorPosition -= word.len();
	m_szTextBuffer.cut(m_iCursorPosition,word.len());
	m_szTextBuffer.insert(m_iCursorPosition,replacement.ptr());
	if(m_szTextBuffer.len() > KVI_INPUT_MAX_BUFFER_SIZE)
	{
		m_szTextBuffer.setLen(KVI_INPUT_MAX_BUFFER_SIZE);
	}
	moveCursorTo(m_iCursorPosition + replacement.len());
	if(bRepaint)repaintWithCursorOn();
}

void KviInput::standardNickCompletion(bool bAddMask,KviStr &word,bool bFirstWordInLine)
{
	// FIXME: this could be really simplified...

	if(!m_pUserListView)return;
	selectOneChar(-1);

	KviStr buffer;
	if(m_szLastCompletedNick.isEmpty())
	{
		// New completion session: we NEED sth to complete
		if(word.isEmpty())return;
		if(m_pUserListView->completeNickStandard(word,m_szLastCompletedNick,buffer,bAddMask))
		{
			// completed: save the buffer
			m_szLastCompletionBuffer          = m_szTextBuffer;
			m_iLastCompletionCursorPosition   = m_iCursorPosition;
			m_iLastCompletionCursorXPosition  = m_iLastCursorXPosition;
			m_iLastCompletionFirstVisibleChar = m_iFirstVisibleChar;
			m_szLastCompletedNick             = buffer;
			if(KVI_OPTION_STRING(KviOption_stringNickCompletionPostfix).hasData())
			{
				if(bFirstWordInLine || (!KVI_OPTION_BOOL(KviOption_boolUseNickCompletionPostfixForFirstWordOnly)))
					buffer.append(KVI_OPTION_STRING(KviOption_stringNickCompletionPostfix));
			}
			replaceWordBeforeCursor(word,buffer,false);
			// REPAINT CALLED FROM OUTSIDE!
		} // else no match at all
	} else {
		// Old session
		// swap the buffers
		m_szTextBuffer                        = m_szLastCompletionBuffer;
		m_iCursorPosition                     = m_iLastCompletionCursorPosition;
		m_iLastCursorXPosition                = m_iLastCompletionCursorXPosition;
		m_iFirstVisibleChar                   = m_iLastCompletionFirstVisibleChar;
		// re-extract 
		word = m_szTextBuffer.left(m_iCursorPosition);

		getWordBeforeCursor(word,&bFirstWordInLine);
		if(word.isEmpty())return;

		if(m_pUserListView->completeNickStandard(word,m_szLastCompletedNick,buffer,bAddMask))
		{
			// completed
			m_szLastCompletedNick             = buffer;
			if(KVI_OPTION_STRING(KviOption_stringNickCompletionPostfix).hasData())
			{
				if(bFirstWordInLine || (!KVI_OPTION_BOOL(KviOption_boolUseNickCompletionPostfixForFirstWordOnly)))
					buffer.append(KVI_OPTION_STRING(KviOption_stringNickCompletionPostfix));
			}
			replaceWordBeforeCursor(word,buffer,false);
			// REPAINT CALLED FROM OUTSIDE!
		} else {
			m_szLastCompletedNick = "";
		}
	}
}

void KviInput::checkForSubstitution()
{
/*
	if(!g_pOptions->m_bUseStringSubstitution || m_szTextBuffer.isEmpty() || !m_iCursorPosition)return;
	KviStr szAuxTextBuffer = m_szTextBuffer;
	szAuxTextBuffer.prepend(' ');
	for(KviStrSubItem * m=g_pOptions->m_pStrSub->m_pList->first();m;m=g_pOptions->m_pStrSub->m_pList->next()) {
		KviStr szAuxOriginal=m->szOriginal;
		szAuxOriginal.prepend(' ');
		if (!kvi_strMatchRevCS(szAuxTextBuffer.ptr(),szAuxOriginal.ptr(),m_iCursorPosition)) {
			//////
			// save buffer in case of backspace
			m_szStringBeforeSubstitution = m_szTextBuffer;
			m_iCursorPositionBeforeSubstitution=m_iCursorPosition;
			m_iCursorXPositionBeforeSubstitution=m_iLastCursorXPosition;
			m_iFirstVisibleCharBeforeSubstitution=m_iFirstVisibleChar;
			///  
			m_iCursorPosition -= m->szOriginal.len();
			m_szTextBuffer.cut(m_iCursorPosition,m->szOriginal.len());
			m_szTextBuffer.insert(m_iCursorPosition,m->szSubstitute.ptr());
			if(m_szTextBuffer.len() > KVI_INPUT_MAX_BUFFER_SIZE){
				m_szTextBuffer.setLen(KVI_INPUT_MAX_BUFFER_SIZE);
			}
			moveCursorTo(m_iCursorPosition + m->szSubstitute.len());
			m_bLastEventWasASubstitution=true;
			repaintWithCursorOn();
			return;
		}
	}
*/
// FIXME: #warning "FIX STRING SUBSTITUTION"
}


//Funky helpers

void KviInput::end()
{
	m_iLastCursorXPosition = KVI_INPUT_BORDER;
	m_iCursorPosition = 0;
	m_iFirstVisibleChar = 0;
	while(m_iCursorPosition < m_szTextBuffer.len())
	{
		moveRightFirstVisibleCharToShowCursor();
		m_iCursorPosition++;
	}
	repaintWithCursorOn();
}

void KviInput::home()
{
	m_iFirstVisibleChar = 0;
	m_iCursorPosition   = 0;
	repaintWithCursorOn();
}

void KviInput::insertChar(char c)
{
	if(m_szTextBuffer.len() >= KVI_INPUT_MAX_BUFFER_SIZE)return;

	// Kill the selection
	if((m_iSelectionBegin > -1) || (m_iSelectionEnd > -1))
	{
		if((m_iCursorPosition >= m_iSelectionBegin) && (m_iCursorPosition <= m_iSelectionEnd))
		{
			m_bUpdatesEnabled = false;
			cut();
			m_bUpdatesEnabled = true;
		}
	}
	selectOneChar(-1);
	m_szTextBuffer.insert(m_iCursorPosition,c);
	moveRightFirstVisibleCharToShowCursor();
	m_iCursorPosition++;
	repaintWithCursorOn();
}

void KviInput::moveRightFirstVisibleCharToShowCursor()
{
	// :)
	m_iLastCursorXPosition += m_iInputFontCharacterWidth[(unsigned char)m_szTextBuffer.at(m_iCursorPosition)];
	while(m_iLastCursorXPosition >= width()-KVI_INPUT_BORDER)
	{
		m_iLastCursorXPosition -= m_iInputFontCharacterWidth[(unsigned char)m_szTextBuffer.at(m_iFirstVisibleChar)];
		m_iFirstVisibleChar++;
	}
}

void KviInput::repaintWithCursorOn()
{
	// :)
	if(m_bUpdatesEnabled)
	{
		m_bCursorOn = true;
		paintEvent(0);
	}
}

void KviInput::selectOneChar(int pos)
{	
	m_iSelectionBegin = pos;
	m_iSelectionEnd   = pos;
}

int KviInput::charIndexFromXPosition(int xPos)
{
	int curXPos = KVI_INPUT_BORDER;
	int curChar = m_iFirstVisibleChar;
	int bufLen  = m_szTextBuffer.len();
	while(curChar < bufLen)
	{
		int widthCh = m_iInputFontCharacterWidth[(unsigned char)m_szTextBuffer.at(curChar)];
		if(xPos < (curXPos+(widthCh/2)))return curChar;
		else if(xPos < (curXPos+widthCh))return (curChar+1);
		{
			curXPos+=widthCh;
			curChar++;
		}
	}
	return curChar;
}

int KviInput::xPositionFromCharIndex(int chIdx)
{
	int curXPos = KVI_INPUT_BORDER;
	int curChar = m_iFirstVisibleChar;
	while(curChar < chIdx)
	{
		curXPos += m_iInputFontCharacterWidth[(unsigned char)m_szTextBuffer.at(curChar)];
		curChar++;
	}
	return curXPos;
}

// FIXME: #warning "Later remove this stuff and use a wrapper for #include <X11/Xlib.h>"


#include "kvi_input.moc"
