//
//   File : kvi_filedialog.cpp
//   Creation date : Mon Nov 20 2000 12:20:42 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2001 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
#define __KVIRC__
#include "kvi_filedialog.h"
#include "kvi_mediatype.h"
#include "kvi_iconmanager.h"
#include "kvi_locale.h"
#include "kvi_app.h"
#include "kvi_fileutils.h"
#include "kvi_frame.h"

#include <qpainter.h>
#include <qtoolbutton.h>
#include <qdir.h>
#include <qstringlist.h>
#include <qtooltip.h>
#include <qmessagebox.h>

extern KviMediaManager * g_pMediaManager;
extern QStringList * g_pBookmarkList;

KviFileDialog::KviFileDialog(const QString &dirName, const QString &filter,
					QWidget *parent, const char *name, bool modal)
: QFileDialog(dirName,filter,parent,name,modal)
{
	setContentsPreviewEnabled(true);
	KviFilePreview * p = new KviFilePreview(this);
	setContentsPreview(p,p);

	setIcon(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_FOLDER)));

	QToolButton * b = new QToolButton(this);
	b->setPixmap(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_HOME)));

	QToolTip::add(b,__tr("Home directory"));

	addToolButton(b,true);
	connect(b,SIGNAL(clicked()),this,SLOT(goHome()));

	b = new QToolButton(this);
	b->setPixmap(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_BOOKMARKS)));
	QToolTip::add(b,__tr("Bookmarks"));
	addToolButton(b,false);
	
	m_pBookmarkPopup = new QPopupMenu(this);
	connect(m_pBookmarkPopup,SIGNAL(activated(int)),this,SLOT(bookmarkSelected(int)));
	b->setPopup(m_pBookmarkPopup);
	connect(b,SIGNAL(clicked()),this,SLOT(addBookmark()));


	b = new QToolButton(this);
	b->setPixmap(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_KVIRC)));
	addToolButton(b,false);
	connect(b,SIGNAL(clicked()),this,SLOT(goKvirc()));
	QToolTip::add(b,__tr("KVIrc local directory"));

	b = new QToolButton(this);
	b->setToggleButton(true);
// FIXME: #warning "Must have an decent icon"
	b->setPixmap(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_FOLDER)));
	connect(b, SIGNAL(toggled(bool)), SLOT(showHiddenFilesToggled(bool)));
	addToolButton(b, false);
	QToolTip::add(b, __tr("Show hidden files"));
	m_pShowHiddenButton = b;

	fillBookmarkPopup();
}

KviFileDialog::~KviFileDialog()
{
}

void KviFileDialog::bookmarkSelected(int id)
{
	if(id != 0)
	{
		setDir(QDir(m_pBookmarkPopup->text(id)));
	}
}

void KviFileDialog::fillBookmarkPopup()
{
	m_pBookmarkPopup->clear();
	m_pBookmarkPopup->insertItem(__tr("Add bookmark"),this,SLOT(addBookmark()),0);
	m_pBookmarkPopup->insertSeparator();
	int id = 1;
	for(QStringList::Iterator it = g_pBookmarkList->begin(); it != g_pBookmarkList->end(); ++it)
	{
		m_pBookmarkPopup->insertItem(*it,id);
		++id;
	}
}

void KviFileDialog::showHiddenFilesToggled(bool b)
{
// FIXME: #warning "Get REAL icons here"
	setShowHiddenFiles(b);
	m_pShowHiddenButton->setPixmap((*g_pIconManager->getSmallIcon( \
		b ? KVI_SMALLICON_WORLD : KVI_SMALLICON_FOLDER )));
	QToolTip::add(m_pShowHiddenButton, b ? __tr("Don't show hidden files") :
		__tr("Show hidden files"));
}

void KviFileDialog::addBookmark()
{
// FIXME: #warning "Make the number of bookmarks configurable ?"
	g_pBookmarkList->append(dirPath());
	if(g_pBookmarkList->count() > 20)g_pBookmarkList->remove(g_pBookmarkList->begin());
	fillBookmarkPopup();
}

void KviFileDialog::goKvirc()
{
	KviStr tmp;
	g_pApp->getLocalKvircDirectory(tmp,KviApp::None);
	setDir(QDir(tmp.ptr()));
}

void KviFileDialog::goHome()
{
	setDir(QDir::home());
}

bool KviFileDialog::askForOpenFileName(KviStr &buffer,const char * caption,const char * initial,const char * filter,bool showHidden)
{
	KviFileDialog * d = new KviFileDialog(initial ? QString(initial) : QString::null,
		filter ? QString(filter) : QString::null,0,"open_file_name_dialog",true);
	d->setCaption(caption);
	d->setMode(QFileDialog::ExistingFile);
	d->setShowHiddenFiles(showHidden);
	if(d->exec() == QDialog::Accepted)
	{
		buffer = d->selectedFile();
		kvi_adjustFilePath(buffer);
		delete d;
		return buffer.hasData();
	}
	delete d;
	return false;
}

bool KviFileDialog::askForSaveFileName(KviStr &buffer,const char * caption,const char * initial,const char * filter,bool showHidden,bool bConfirmOverwrite)
{
	KviFileDialog * d = new KviFileDialog(initial ? QString(initial) : QString::null,
		filter ? QString(filter) : QString::null,0,"save_file_name_dialog",true);
	d->setCaption(caption);
	d->setMode(QFileDialog::AnyFile);
	d->setShowHiddenFiles(showHidden);	

	while(d->exec() == QDialog::Accepted)
	{
		buffer = d->selectedFile();
		kvi_adjustFilePath(buffer);

		if(buffer.hasData())
		{
			if(!bConfirmOverwrite)
			{
				delete d;
				return true;
			}
			// Check for the file existence
			if(kvi_fileExists(buffer.ptr()))
			{
				KviStr tmp(KviStr::Format,__tr("The file %s already exists.\nOverwrite it ?"),buffer.ptr());
				switch(QMessageBox::information(d,__tr("File exists"),tmp.ptr(),QMessageBox::Yes,QMessageBox::No | QMessageBox::Default,QMessageBox::Cancel | QMessageBox::Escape))
				{
					case QMessageBox::Cancel: delete d; return false; break;
					case QMessageBox::Yes:    delete d; return true;  break;
				}
				// QMessageBox::No will continue looping

			} else {
				delete d;
				return true; // ok...file not exists
			}
		} else {
			delete d;
			return false; // empty buffer
		}
	}

	delete d;
	return false;
}

bool KviFileDialog::askForDirectoryName(KviStr &buffer,const char * caption,const char * initial,const char * filter,bool showHidden)
{
	KviFileDialog * d = new KviFileDialog(initial ? QString(initial) : QString::null,
		filter ? QString(filter) : QString::null,0,"directory_name_dialog",true);
	d->setCaption(caption);
	d->setMode(QFileDialog::DirectoryOnly);
	d->setShowHiddenFiles(showHidden);	
	if(d->exec() == QDialog::Accepted)
	{
		buffer = d->selectedFile();
		kvi_adjustFilePath(buffer);
		delete d;
		return buffer.hasData();
	}
	delete d;
	return false;
}


bool KviFileDialog::askForOpenFileNames(QStringList &buffer,const char * caption,const char * initial,const char * filter,bool showHidden)
{
	KviFileDialog * d = new KviFileDialog(initial ? QString(initial) : QString::null,
		filter ? QString(filter) : QString::null,0,"open_file_names_dialog",true);
	d->setCaption(caption);
	d->setMode(QFileDialog::ExistingFiles);
	d->setShowHiddenFiles(showHidden);	
	if(d->exec() == QDialog::Accepted)
	{
		buffer = d->selectedFiles();
		delete d;
		return (buffer.count() > 0);
	}
	delete d;
	return false;
}

KviFilePreview::KviFilePreview(QWidget * par)
: QScrollView(par,"file_preview"), QFilePreview()
{
// FIXME: #warning "Display the matched IANA type"
	m_pMemPixmap = 0;
}

KviFilePreview::~KviFilePreview()
{
	if(m_pMemPixmap)delete m_pMemPixmap;
}

void KviFilePreview::previewUrl(const QUrl &url)
{
	// Guess the mime type
	if(m_pMemPixmap)
	{
		delete m_pMemPixmap;
		m_pMemPixmap = 0;
	}

	if(url.isLocalFile() && url.isValid())
	{
		KviStr fPath = url.path();
		g_pMediaManager->lock();
		KviMediaType * t = g_pMediaManager->findMediaType(fPath.ptr());
		if(t)
		{
// FIXME: #warning "Preview more file types: and maybe be more conservative with g_pMediaManager->locking() (unlock it earlier...before resizing the pixmap and calling paint())"
			if(kvi_strEqualCIN(t->szIanaType.ptr(),"image/",6))
			{
				m_pMemPixmap = new QPixmap(fPath.ptr());
				if(!(m_pMemPixmap->isNull()))
				{
					g_pMediaManager->unlock();
					resizeContents(m_pMemPixmap->width(),m_pMemPixmap->height());
					viewport()->update();
					return;
				}
				delete m_pMemPixmap;
				m_pMemPixmap = 0;
			}
		}
		g_pMediaManager->unlock();
	}

	resizeContents(0,0);
	viewport()->update();
}

void KviFilePreview::drawContents(QPainter *p,int cx,int cy,int cw,int ch)
{
	if(m_pMemPixmap)
	{
		if(!m_pMemPixmap->isNull())
		{
			p->drawPixmap(cx,cy,*m_pMemPixmap,cx,cy,cw,ch);
		}
	}
}


#include "kvi_filedialog.moc"
