//
//   File : kvi_console.cpp
//   Creation date : Sun Jun 25 2000 15:01:34 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#define __KVIRC__

#include "kvi_debug.h"
#include "kvi_app.h"
#include "kvi_console.h"
#include "kvi_frame.h"
#include "kvi_iconmanager.h"
#include "kvi_options.h"
#include "kvi_locale.h"
#include "kvi_ircview.h"
#include "kvi_mirccntrl.h"
#include "kvi_out.h"
#include "kvi_input.h"
#include "kvi_error.h"
#include "kvi_ircserverdb.h"
#include "kvi_proxydb.h"
#include "kvi_netutils.h"
#include "kvi_ircserver.h"
#include "kvi_dns.h"
#include "kvi_debug.h"
#include "kvi_defaults.h"
#include "kvi_ircuserdb.h"
#include "kvi_channel.h"
#include "kvi_query.h"
#include "kvi_event.h"
#include "kvi_parameterlist.h"
#include "kvi_uparser.h"
#include "kvi_regusersdb.h"
#include "kvi_userlistview.h"
#include "kvi_notifylist.h"
#include "kvi_config.h"
#include "kvi_irctoolbar.h"
#include "kvi_internalcmd.h"
#include "kvi_sparser.h"
#include "kvi_themedlabel.h"
#include "kvi_garbage.h"

#ifdef COMPILE_SSL_SUPPORT
	#include "kvi_sslmaster.h"
#endif

#include <qsplitter.h>
#include <qtoolbar.h>
#include <qtoolbutton.h>
#include <qtimer.h>
#include <qhbox.h>
#include <qpopupmenu.h>


extern KVIRC_API KviIrcServerDataBase           * g_pIrcServerDataBase;
extern KVIRC_API KviProxyDataBase               * g_pProxyDataBase;
extern KVIRC_API KviRegisteredUserDataBase      * g_pRegisteredUserDataBase;
extern KVIRC_API KviGarbageCollector            * g_pGarbageCollector;

KviConsole::KviConsole(KviFrame * lpFrm,unsigned int uId,const char * name,int iFlags)
#ifdef COMPILE_ON_WINDOWS
: KviWindow(KVI_WINDOW_TYPE_CONSOLE,lpFrm,name,0)
#else
: KviWindow(KVI_WINDOW_TYPE_CONSOLE,lpFrm,name,this)
#endif
{
	m_pConsole = this;

	m_uIrcContextId = uId;

	m_iFlags = iFlags;

	m_pIrcSocketMonitorList = 0;

	m_pSocket = new KviIrcSocket(this,"irc_socket");
	m_pConnectionInfo = 0;

	m_pLinksWindow = 0;
	m_pListWindow = 0;

	m_pButtonBox = new QHBox(this,"button_box");
	KviThemedLabel * l = new KviThemedLabel(m_pButtonBox,"dummy_label");
	m_pButtonBox->setStretchFactor(l,1);
	QToolTip::add(l,__c2q(__tr("What is this ?")));

	m_pNotifyViewButton = createToolButton(m_pButtonBox,"list_view_button",KVI_SMALLICON_HIDELISTVIEW,KVI_SMALLICON_SHOWLISTVIEW,__tr("Toggle notify list"),true);
	connect(m_pNotifyViewButton,SIGNAL(clicked()),this,SLOT(toggleNotifyView()));

	m_pSplitter = new QSplitter(QSplitter::Horizontal,this,"splitter");
	m_pIrcView = new KviIrcView(m_pSplitter,lpFrm,this);
	connect(m_pIrcView,SIGNAL(rightClicked()),this,SLOT(textViewRightClicked()));

// FIXME: #warning "If notify list is disabled avoid to show this"
// FIXME: #warning "Button to show/hide the notifyListView (NOT DELETE RE_CREATE!)"
	// The userlist on the right
	m_pNotifyListView = new KviUserListView(m_pSplitter,0,this,19,__tr("Notify list"),"notify_list_view");


//	connect(m_pNotifyListView,SIGNAL(rightClicked()),this,SLOT(notifyListViewRightClicked()));
//	connect(m_pNotifyListView,SIGNAL(doubleClicked()),this,SLOT(notifyListViewDoubleClicked()));

	m_pAsyncServerCommandData = 0;
	m_pInput   = new KviInput(this,m_pNotifyListView);

	m_state = NotConnected;

	KviIrcToolBar * tb = m_pFrm->getIrcToolBar();
	m_pIcController = new KviIrcContextController(this,tb->iccStack(),tb);
	tb->addController(m_pIcController);
	

	setFocusHandler(m_pInput,this);

	if(KVI_OPTION_BOOL(KviOption_boolAutoLogConsole))m_pIrcView->startLogging(0);
}




KviConsole::~KviConsole()
{
// FIXME: #warning "WARNING : THIS SHOULD BECOME A COMMAND /QUIT $option() so the idents are parsed!"
	while(m_pIrcSocketMonitorList)
	{
		KviIrcSocketMonitor * m = m_pIrcSocketMonitorList->first();
		if(m)
		{
			m->die();
		} else {
			delete m_pIrcSocketMonitorList;
			m_pIrcSocketMonitorList = 0;
		}
	}

	if(m_pLinksWindow)m_pLinksWindow->die();
	if(m_pListWindow)m_pListWindow->die();

	// Force connection close
	if(m_state == Connected)terminateConnectionRequest(true);

	__range_valid(m_state == NotConnected);
	__range_valid(m_pConnectionInfo == 0);

	TRIGGER_EVENT(KviEvent_OnIrcContextDestroyed,this);

	if(m_pFrm->consoleCount() <= 1)
	{
		TRIGGER_EVENT(KviEvent_OnFrameWindowDestroyed,this);
		if(g_pApp->frameCount() < 1) TRIGGER_EVENT(KviEvent_OnKVIrcShutdown,this);
	}

	delete m_pSocket;

	if(m_pIcController->parentToolBar() == m_pFrm->ircToolBar())
	{
		m_pIcController->parentToolBar()->removeController(m_pIcController);
		m_pIcController = 0;
	} else {
		delete m_pIcController->parentToolBar(); // again a deleted path
	}

	destroyAsyncServerCommandData();
}


void KviConsole::triggerCreationEvents()
{
	if(m_iFlags & KVI_CONSOLE_FLAG_FIRSTINAPP)
	{
		TRIGGER_EVENT(KviEvent_OnKVIrcStartup,this);

		if(KVI_OPTION_BOOL(KviOption_boolShowTipAtStartup)) 
			m_pFrm->executeInternalCommand(KVI_INTERNALCOMMAND_TIP_OPEN);

		if(KVI_OPTION_BOOL(KviOption_boolShowServersConnectDialogOnStart))
			m_pFrm->executeInternalCommand(KVI_INTERNALCOMMAND_SERVERSJOIN_OPEN);
	}

	if(m_iFlags & KVI_CONSOLE_FLAG_FIRSTINFRAME)
	{
		TRIGGER_EVENT(KviEvent_OnFrameWindowCreated,this);
	}

	TRIGGER_EVENT(KviEvent_OnIrcContextCreated,this);
}


void KviConsole::fillContextPopup(QPopupMenu * p)
{
	p->insertSeparator();
	int id = p->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_CHANNEL)),__tr("Part all channels"),this,SLOT(partAllChannels()));
	if(!channelCount())p->setItemEnabled(id,false);
	id = p->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_QUERY)),__tr("Close all queries"),this,SLOT(closeAllQueries()));
	if(!queryCount())p->setItemEnabled(id,false);
}

void KviConsole::completeChannel(const KviStr &word,KviPtrList<KviStr> * matches)
{
	if(!m_pConnectionInfo)return;
	for(KviChannel * c = m_pConnectionInfo->pChannelList->first();c;c = m_pConnectionInfo->pChannelList->next())
	{
		if(kvi_strEqualCIN(c->name(),word,word.len()))matches->append(new KviStr(c->name()));
	}
}


int KviConsole::getCommonChannels(const char * nick,KviStr &chansBuffer,bool bEscapeSeq)
{
	if(!m_pConnectionInfo)return 0;
	int count = 0;
	for(KviChannel * c = m_pConnectionInfo->pChannelList->first();c;c = m_pConnectionInfo->pChannelList->next())
	{
		if(c->isOn(nick))
		{
			if(chansBuffer.hasData())chansBuffer.append(", ");
			char uFlag = c->getUserFlag(nick);
			if(uFlag)
			{
				chansBuffer.append(KviStr::Format,bEscapeSeq ? "\r!c\r%c%s\r" : "%c%s",uFlag,c->name());
			} else {
				if(bEscapeSeq)chansBuffer.append(KviStr::Format,"\r!c\r%s\r",c->name());
				else chansBuffer.append(c->name());
			}
			count ++;
		}
	}
	return count;
}


void KviConsole::getUserTipText(const char * nick,KviIrcUserEntry *e,KviStr &buffer)
{
	buffer.sprintf("<center><b>%s!%s@%s</b>",
		nick,
		*(e->user()) ? e->user() : "*",
		*(e->host()) ? e->host() : "*");

	if(e->avatar())
	{
		QMimeSourceFactory::defaultFactory()->setPixmap("ulv_avatar",*(e->avatar()->pixmap()));
		buffer.append("<br><img src=\"ulv_avatar\">");
	}

	if(e->hasRealName())
	{
		buffer.append(KviStr::Format,"<br>%s",e->realName());
	}
	buffer.append("</center>");


	if(*(e->user()))
	{
		KviRegisteredMask *u = g_pRegisteredUserDataBase->findMatchingMask(nick,
			e->user(),e->host());
		if(u)
		{
			KviStr mask;
			u->mask()->mask(mask);
			buffer.append(KviStr::Format,__tr("<hr>\nRegistered as %s<br>\n(Matched by %s)"),u->user()->name(),mask.ptr());
		}
	}

	KviStr chans;

	bool bHr = false;

	if(getCommonChannels(nick,chans,false))
	{
		bHr = true;
		buffer.append(__tr("<hr>On <b>"));
		buffer.append(chans);
		buffer.append("</b>");
	}

	if(e->hasServer())
	{
		buffer.append(bHr ? "<br>" : "<hr>");
		buffer.append(KviStr::Format,
			__tr("<nobr>Using server <b>%s</b>"),e->server());

		if(e->hasHops())
		{
			buffer.append(KviStr::Format,
				__tr(" (%d hops)</nobr>"),
				e->hops());
		} else {
			buffer.append("</nobr>");
		}
	}
}


void KviConsole::toggleNotifyView()
{
	if(m_pNotifyListView->isVisible())
	{
		m_pNotifyListView->hide();
		if(m_pNotifyViewButton->isOn())m_pNotifyViewButton->setOn(false);
	} else {
		m_pNotifyListView->show();
		if(!(m_pNotifyViewButton->isOn()))m_pNotifyViewButton->setOn(true);
	}
}

void KviConsole::executeInternalCommand(int index)
{
	g_pUserParser->parseCommandBuffer(kvi_getInternalCommandBuffer(index),this);
}

void KviConsole::saveProperties(KviConfig *cfg)
{
	cfg->writeEntry("Splitter",m_pSplitter->sizes());
}

void KviConsole::getBaseLogFileName(KviStr &buffer)
{
	buffer.sprintf("CONSOLE%u",ircContextId());
}

void KviConsole::loadProperties(KviConfig *cfg)
{
	QValueList<int> def;
	def.append(85);
	def.append(15);
	m_pSplitter->setSizes(cfg->readIntListEntry("Splitter",def));
}

void KviConsole::registerSocketMonitor(KviIrcSocketMonitor * m)
{
	if(!m_pIrcSocketMonitorList)
	{
		m_pIrcSocketMonitorList = new KviPtrList<KviIrcSocketMonitor>;
		m_pIrcSocketMonitorList->setAutoDelete(false);
	}
	m_pIrcSocketMonitorList->append(m);
}

void KviConsole::unregisterSocketMonitor(KviIrcSocketMonitor *m)
{
	if(!m_pIrcSocketMonitorList)return;
	m_pIrcSocketMonitorList->removeRef(m);
	if(m_pIrcSocketMonitorList->isEmpty())
	{
		delete m_pIrcSocketMonitorList;
		m_pIrcSocketMonitorList = 0;
	}
}



void KviConsole::textViewRightClicked()
{
	TRIGGER_EVENT(KviEvent_OnConsolePopupRequest,this);
}

//void KviConsole::notifyListViewRightClicked()
//{
//#warning "Trigger the event here"
//}
//
//void KviConsole::notifyListViewDoubleClicked()
//{
//#warning "Trigger the event here"
//}

KviWhoisInfo * KviConsole::lookupAsyncWhois(const char * nick)
{
	if(m_pConnectionInfo->pWhoisInfoList->isEmpty())return 0;
	for(KviWhoisInfo * i = m_pConnectionInfo->pWhoisInfoList->first();i;i = m_pConnectionInfo->pWhoisInfoList->next())
	{
		if(kvi_strEqualCI(nick,i->szNick.ptr()))return i;
	}
	return 0;
}

void KviConsole::partAllChannels()
{
	if(!m_pConnectionInfo)return;
	for(KviChannel * c = m_pConnectionInfo->pChannelList->first();c;c = m_pConnectionInfo->pChannelList->next())
	{
		c->close();
	}
}

void KviConsole::closeAllChannels()
{
	if(!m_pConnectionInfo)return;
	while(m_pConnectionInfo->pChannelList->first())
	{
		m_pFrm->closeWindow(m_pConnectionInfo->pChannelList->first());
	}
}

void KviConsole::closeAllQueries()
{
	if(!m_pConnectionInfo)return;
	while(m_pConnectionInfo->pQueryList->first())
	{
		m_pFrm->closeWindow(m_pConnectionInfo->pQueryList->first());
	}
}

KviWindow * KviConsole::activeWindow()
{
	if(g_pApp->activeWindow())
	{
		if(g_pApp->activeWindow()->console() == this)return g_pApp->activeWindow();
	}
	return this;
}

void KviConsole::createLinksWindow()
{
	if(m_pLinksWindow)return;
	g_pUserParser->parseCommandBuffer("links.open",this);
}

void KviConsole::createListWindow()
{
	if(m_pListWindow)return;
	g_pUserParser->parseCommandBuffer("list.open",this);
}

void KviConsole::createConnectionInfo(KviIrcNetwork * net,KviIrcServer * srv,KviProxy * prx)
{
	__range_invalid(m_pConnectionInfo);
	m_pConnectionInfo = new KviConnectionInfo;
	m_pConnectionInfo->szNetworkName            = net->name();
	m_pConnectionInfo->pIrcServer               = new KviIrcServer(*srv);
	m_pConnectionInfo->pProxy                   = prx ? new KviProxy(*prx) : 0;
	m_pConnectionInfo->pNotifyListManager       = 0;
	m_pConnectionInfo->bSentQuit                = false;
	m_pConnectionInfo->pDns                     = 0;
	m_pConnectionInfo->pUserDataBase            = new KviIrcUserDataBase();
	m_pNotifyListView->setUserDataBase(m_pConnectionInfo->pUserDataBase);
	m_pConnectionInfo->pChannelList             = new KviPtrList<KviChannel>;
	m_pConnectionInfo->pChannelList->setAutoDelete(false);
	m_pConnectionInfo->pQueryList               = new KviPtrList<KviQuery>;
	m_pConnectionInfo->pQueryList->setAutoDelete(false);
	m_pConnectionInfo->pWhoisInfoList           = new KviPtrList<KviWhoisInfo>;
	m_pConnectionInfo->pWhoisInfoList->setAutoDelete(true);
	m_pConnectionInfo->lastNetsplitTime         = 0;
	m_pConnectionInfo->lastCtcpTime             = 0;
	m_pConnectionInfo->uCtcpCount               = 0;
	m_pConnectionInfo->bServerSupportsModeIe    = true;
	m_pConnectionInfo->bServerSupportsWatchList = false;
	m_pConnectionInfo->pNotifyListTimer         = 0;
	m_pConnectionInfo->bAway                    = false;
}


void KviConsole::destroyDnsObject(KviDns * d)
{
	if(!d)return;
	disconnect(d,SIGNAL(lookupDone(KviDns *)),this,0);
	if(d->isRunning())
	{
		// the DNS thread is running in the background... ugh
		// we need to wait it to finish before deleting it if we don't want to be blocked
		g_pGarbageCollector->collect(d);
	} else {
		// this is never blocking.. just delete the object
		delete d;
	}
}

void KviConsole::destroyConnectionInfo()
{
	if(m_pConnectionInfo)
	{
		if(m_pConnectionInfo->pNotifyListTimer)
		{
			delete m_pConnectionInfo->pNotifyListTimer;
			m_pConnectionInfo->pNotifyListTimer = 0;
		}
		closeAllChannels();
		closeAllQueries();
		if(m_pConnectionInfo->pIrcServer)delete m_pConnectionInfo->pIrcServer;
		if(m_pConnectionInfo->pProxy)delete m_pConnectionInfo->pProxy;
		if(m_pConnectionInfo->pDns)destroyDnsObject(m_pConnectionInfo->pDns);
		if(m_pConnectionInfo->pNotifyListManager)delete m_pConnectionInfo->pNotifyListManager; // destroy this before the userDb
		m_pNotifyListView->partAll();
		m_pNotifyListView->setUserDataBase(0); // this is rather for crash tests
		delete m_pConnectionInfo->pUserDataBase;
		delete m_pConnectionInfo->pChannelList;
		delete m_pConnectionInfo->pQueryList;
		delete m_pConnectionInfo->pWhoisInfoList;
		delete m_pConnectionInfo;
		m_pConnectionInfo = 0;
		if(m_pLinksWindow)m_pLinksWindow->control(EXTERNAL_SERVER_DATA_PARSER_CONTROL_RESET);
		if(m_pListWindow)m_pListWindow->control(EXTERNAL_SERVER_DATA_PARSER_CONTROL_RESET);
	}
}

void KviConsole::startNotifyList()
{
	// start the notify list in 3 seconds
	// We have this delay to wait an eventual RPL_PROTOCTL from the server
	// telling us that the WATCH notify list method is supported
	__range_invalid(m_pConnectionInfo->pNotifyListTimer);

	m_pConnectionInfo->pNotifyListTimer = new QTimer();
	connect(m_pConnectionInfo->pNotifyListTimer,SIGNAL(timeout()),this,SLOT(restartNotifyList()));
	m_pConnectionInfo->pNotifyListTimer->start(15000,true);

	// This delay is large enough to fire after the MOTD has been sent,
	// even on the weirdest network.
	// If there is no MOTD, this timer will fire after 15 secs,
	// If there is a MOTD , restartNotifyList() will be triggered by RPL_ENDOFMOTD and
	// will kill the timer before it has fired.
}


void KviConsole::restartNotifyList()
{
	if(!m_pConnectionInfo)return;

	if(m_pConnectionInfo->pNotifyListTimer)
	{
		delete m_pConnectionInfo->pNotifyListTimer;
		m_pConnectionInfo->pNotifyListTimer = 0;
	}

	// clear it
	if(m_pConnectionInfo->pNotifyListManager)
	{
		m_pConnectionInfo->pNotifyListManager->stop(); // may need to remove watch entries
		delete m_pConnectionInfo->pNotifyListManager;
		m_pConnectionInfo->pNotifyListManager = 0;
	}

	if(!KVI_OPTION_BOOL(KviOption_boolUseNotifyList))return;

	if(m_pConnectionInfo->bServerSupportsWatchList && KVI_OPTION_BOOL(KviOption_boolUseWatchListIfAvailable))
	{
		if(KVI_OPTION_BOOL(KviOption_boolBeVerbose))
		{
			output(KVI_OUT_SYSTEMMESSAGE,__tr("The server seems to support the WATCH notify list method: going to use it"));
		}
		m_pConnectionInfo->pNotifyListManager = new KviWatchNotifyListManager(this);
	} else {
		if(KVI_OPTION_BOOL(KviOption_boolUseIntelligentNotifyListManager))
		{
			m_pConnectionInfo->pNotifyListManager = new KviIsOnNotifyListManager(this);
		} else {
			m_pConnectionInfo->pNotifyListManager = new KviStupidNotifyListManager(this);
		}
	}
	m_pConnectionInfo->pNotifyListManager->start();
}

//KviWindow * KviConsole::activeWindow()
//{
//#warning "FIX THIS....ACTIVE WINDOW ?"
//	return this;
//}

KviChannel * KviConsole::createChannel(const char * name)
{
	__range_valid(m_pConnectionInfo);
	KviChannel * c = new KviChannel(m_pFrm,this,name);
	m_pFrm->addWindow(c);
	//TRIGGER_EVENT(KviEvent_OnChannelWindowCreated,c);
	return c;
}

KviQuery * KviConsole::createQuery(const char * name)
{
	__range_valid(m_pConnectionInfo);
	KviQuery * q = new KviQuery(m_pFrm,this,name);
	m_pFrm->addWindow(q,!KVI_OPTION_BOOL(KviOption_boolCreateMinimizedQuery));
	if(KVI_OPTION_BOOL(KviOption_boolCreateMinimizedQuery))q->minimize();
	//TRIGGER_EVENT(KviEvent_OnQueryWindowCreated,q);
	return q;
}

KviChannel * KviConsole::findChannel(const char * name)
{
	if(m_pConnectionInfo)
	{
		for(KviChannel * c = m_pConnectionInfo->pChannelList->first();c;c = m_pConnectionInfo->pChannelList->next())
		{
			if(!(c->isDeadChan()))
			{
				if(kvi_strEqualCI(name,c->name()))return c;
			}
		}
	}
	return 0;
}

KviChannel * KviConsole::findDeadChannel(const char * name)
{
	if(m_pConnectionInfo)
	{
		for(KviChannel * c = m_pConnectionInfo->pChannelList->first();c;c = m_pConnectionInfo->pChannelList->next())
		{
			if(c->isDeadChan())
			{
				if(kvi_strEqualCI(name,c->name()))return c;
			}
		}
	}
	return 0;
}

KviQuery * KviConsole::findQuery(const char * nick)
{
	if(m_pConnectionInfo)
	{
		for(KviQuery * c = m_pConnectionInfo->pQueryList->first();c;c = m_pConnectionInfo->pQueryList->next())
		{
			if(c->haveTarget(nick))return c;
		}
	}
	return 0;
}

KviQuery * KviConsole::findQueryByName(const char * name)
{
	if(m_pConnectionInfo)
	{
		for(KviQuery * c = m_pConnectionInfo->pQueryList->first();c;c = m_pConnectionInfo->pQueryList->next())
		{
			if(kvi_strEqualCI(name,c->name()))return c;
		}
	}
	return 0;
}

void KviConsole::registerChannel(KviChannel * c)
{
	__range_valid(m_pConnectionInfo);
	m_pConnectionInfo->pChannelList->append(c);
	g_pApp->addRecentChannel(c->name());
}

void KviConsole::unregisterChannel(KviChannel * c)
{
	__range_valid(m_pConnectionInfo);
	m_pConnectionInfo->pChannelList->removeRef(c);
}

void KviConsole::registerQuery(KviQuery * c)
{
	__range_valid(m_pConnectionInfo);
	m_pConnectionInfo->pQueryList->append(c);
}

void KviConsole::unregisterQuery(KviQuery * c)
{
	__range_valid(m_pConnectionInfo);
	m_pConnectionInfo->pQueryList->removeRef(c);
}

QPixmap * KviConsole::myIconPtr()
{
	return g_pIconManager->getSmallIcon(KVI_SMALLICON_CONSOLE);
}

void KviConsole::fillCaptionBuffers()
{
	switch(m_state)
	{
		case NotConnected: m_szStatusString = __tr("no connection"); break;
		case Connecting:   m_szStatusString = __tr("connection in progress..."); break;
		case LoggingIn:    m_szStatusString = __tr("login in progress..."); break;
		case Connected:
			m_szStatusString = currentNickName();
			if(m_pConnectionInfo->szUserMode.hasData())
			{
				m_szStatusString.append(KviStr::Format," (+%s",m_pConnectionInfo->szUserMode.ptr());
				if(m_pConnectionInfo->bAway)m_szStatusString.append(__tr(" away) on "));
				else m_szStatusString.append(__tr(") on "));
			} else {
				if(m_pConnectionInfo->bAway)m_szStatusString.append(__tr(" (away) on "));
				else m_szStatusString.append(__tr(" on "));
			}
			m_szStatusString.append(currentServerName());
		break;
	}

	m_szPlainTextCaption.sprintf(__tr("%s [%s]"),name(),m_szStatusString.ptr());
	m_szHtmlActiveCaption.sprintf(__tr("<nobr><font color=\"%s\"><b>%s</b></font> <font color=\"%s\">[%s]</font></nobr>"),
		KVI_OPTION_COLOR(KviOption_colorCaptionTextActive).name().ascii(),name(),
		KVI_OPTION_COLOR(KviOption_colorCaptionTextActive2).name().ascii(),m_szStatusString.ptr());
	m_szHtmlInactiveCaption.sprintf(__tr("<nobr><font color=\"%s\"><b>%s</b></font> <font color=\"%s\">[%s]</font></nobr>"),
		KVI_OPTION_COLOR(KviOption_colorCaptionTextInactive).name().ascii(),name(),
		KVI_OPTION_COLOR(KviOption_colorCaptionTextInactive2).name().ascii(),m_szStatusString.ptr());
}

bool KviConsole::setUserMode(char mode,bool bSet)
{
	__range_valid(m_pConnectionInfo);
	if(bSet)
	{
		if(m_pConnectionInfo->szUserMode.contains(mode))return false;
		m_pConnectionInfo->szUserMode.append(mode);
	} else {
		if(m_pConnectionInfo->szUserMode.occurences(mode) < 1)return false;
		KviStr m = mode;
		m_pConnectionInfo->szUserMode.replaceAll(m.ptr(),"");
	}
	updateCaption();
	m_pIcController->m_pStatusLabel->update();
	return true;
}

void KviConsole::changeAwayState(bool bAway)
{
	__range_valid(m_pConnectionInfo);
	m_pConnectionInfo->bAway = bAway;
	if(bAway)m_pConnectionInfo->tAwayTime = time(0);
	updateCaption();
	m_pIcController->m_pStatusLabel->update();
}
/*
bool KviConsole::userModeEmpty()
{
	return m_pConnectionInfo ? m_pConnectionInfo->szUserMode.isEmpty() : true;
}

const char * KviConsole::currentUserMode()
{
	return m_pConnectionInfo ? m_pConnectionInfo->szUserMode.ptr() : 0;
}

bool KviConsole::hasUserMode(char c)
{
	if(!m_pConnectionInfo)return false;
	return m_pConnectionInfo->szUserMode.contains(c);
}
*/
void KviConsole::resizeEvent(QResizeEvent *e)
{
	int hght = m_pInput->heightHint();
	int hght2 = m_pButtonBox->sizeHint().height();
	m_pButtonBox->setGeometry(0,0,width(),hght2);
	m_pSplitter->setGeometry(0,hght2,width(),height() - (hght + hght2));
	m_pInput->setGeometry(0,height() - hght,width(),hght);
}

QSize KviConsole::sizeHint() const
{
	QSize ret(m_pIrcView->sizeHint().height(),m_pIrcView->sizeHint().height() + m_pInput->heightHint());
	return ret;
}

void KviConsole::asyncServerCommand()
{
	if(m_pConnectionInfo)
	{
		// need a brutal disconnect here
		if(m_state == Connected)m_pSocket->sendFmtData("QUIT :changing server...");
		m_pConnectionInfo->bSentQuit = true;
		connectButtonClicked();
	}
	connectToCurrentServer();
}

void KviConsole::terminateConnectionRequest(bool bForce,const char * quitMsg)
{
	if(!m_pConnectionInfo)return;

	if(m_pConnectionInfo->pDns)
	{
		// was waiting for dns...
		outputNoFmt(KVI_OUT_SYSTEMMESSAGE,__tr("Connection aborted"));
		setState(NotConnected);
	} else {
		if(m_state == Connected)
		{
			// was connected
			KviStr szQuitMsg = quitMsg ? quitMsg : "";
			if(szQuitMsg.isEmpty())szQuitMsg = KVI_OPTION_STRING(KviOption_stringQuitMessage);
			m_pSocket->sendFmtData("QUIT :%s",szQuitMsg.ptr());

			if(KVI_OPTION_BOOL(KviOption_boolForceBrutalQuit) || m_pConnectionInfo->bSentQuit || bForce)
			{
				m_pConnectionInfo->bSentQuit = true;
				m_pSocket->abortConnection();
			} else {
				m_pConnectionInfo->bSentQuit = true;
				outputNoFmt(KVI_OUT_SYSTEMMESSAGE,__tr("Sent QUIT... waiting for the server to close the connection"));
			}
		} else {
			// was waiting for connection or for login
			m_pSocket->abortConnection();
		}
	}
}

void KviConsole::connectButtonClicked()
{
	if(m_pConnectionInfo == 0)
	{
		// No connections in progress
		connectToCurrentServer();
	} else {
		// Sth is going on
		terminateConnectionRequest(false);
	}	
}

void KviConsole::setState(ConnectionState st)
{
	if(m_state != st)
	{
		m_state = st;
		switch(st)
		{
			case NotConnected:
				m_pIcController->m_pConnectButton->setPixmap(*(g_pIconManager->getBigIcon(KVI_BIGICON_CONNECT)));
				m_pIcController->m_pConnectButton->setTextLabel(__tr("Connect"));
				destroyConnectionInfo();
				m_pIcController->m_pStatusLabel->update();
			break;
			case Connecting:
				m_pIcController->m_pConnectButton->setPixmap(*(g_pIconManager->getBigIcon(KVI_BIGICON_CONNECTING)));
				m_pIcController->m_pConnectButton->setTextLabel(__tr("Abort"));
				m_pIcController->m_pStatusLabel->update();
			break;
			case LoggingIn:
			break;
			case Connected:
				m_pIcController->m_pConnectButton->setPixmap(*(g_pIconManager->getBigIcon(KVI_BIGICON_CONNECTED)));
				m_pIcController->m_pConnectButton->setTextLabel(__tr("Disconnect"));
				m_pIcController->m_pStatusLabel->update();
			break;
		}
		updateCaption();
		emit stateChange(m_state);
	}
}
/*
const char * KviConsole::currentNickName()
{
	return (m_pConnectionInfo ? m_pConnectionInfo->szNickName.ptr() : KviStr::emptyString().ptr());
}

//int KviConsole::currentNickNameLen()
//{
//	__range_valid(m_pConnectioInfo);
//	return m_pConnectionInfo->szNickName.len();
//}

const char * KviConsole::currentUserName()
{
	return m_pConnectionInfo->szUserNameFromServer.ptr();
}

const char * KviConsole::currentLocalHostName()
{
	__range_valid(m_pConnectionInfo);
	return m_pConnectionInfo->szLocalHostFromServer.ptr();
}


const char * KviConsole::currentServerName()
{
	__range_valid(m_pConnectionInfo);
	return m_pConnectionInfo->szServerNameFromServer.ptr();
}

const char * KviConsole::currentNetworkName()
{
	__range_valid(m_pConnectionInfo);
	return m_pConnectionInfo->szNetworkName.ptr();
}
*/

void KviConsole::setUserInfoFromServer(const char * username,const char *localhost)
{
	__range_valid(m_pConnectionInfo);
	m_pConnectionInfo->szUserNameFromServer = username;
	m_pConnectionInfo->szLocalHostFromServer = localhost;
}



void KviConsole::nickChange(const char * newNick)
{
	__range_valid(m_pConnectionInfo);
	m_pConnectionInfo->szNickName = newNick;
	output(KVI_OUT_NICK,__tr("You have changed your nickname to %s"),newNick);
	updateCaption();
	m_pIcController->m_pStatusLabel->update();
	g_pApp->addRecentNickname(newNick);
}

void KviConsole::setServerInfoFromServer(const char * servname,const char * umodes,const char * chanmodes)
{
	__range_valid(m_pConnectionInfo);
	if(servname)m_pConnectionInfo->szServerNameFromServer = servname;
	if(umodes)m_pConnectionInfo->szServerSupportedUserModes = umodes;
	if(chanmodes)
	{
		m_pConnectionInfo->szServerSupportedChanModes = chanmodes;
		m_pConnectionInfo->bServerSupportsModeIe = (m_pConnectionInfo->szServerSupportedChanModes.contains('e') &&
			m_pConnectionInfo->szServerSupportedChanModes.contains('I'));
	}
	updateCaption(); // for server name
	m_pIcController->m_pStatusLabel->update();
}

void KviConsole::setAsyncServerCommandData(KviAsyncServerCommandData * d)
{
	destroyAsyncServerCommandData();
	m_pAsyncServerCommandData = d;
}

void KviConsole::destroyAsyncServerCommandData()
{
	if(m_pAsyncServerCommandData)
	{
		delete m_pAsyncServerCommandData;
		m_pAsyncServerCommandData = 0;
	}
}

void KviConsole::connectToCurrentServer()
{
	__range_invalid(m_pConnectionInfo);

	if(m_pAsyncServerCommandData)
	{
		if(m_pAsyncServerCommandData->szServer.hasData())
		{
			g_pIrcServerDataBase->iWantThisServerToBeTheCurrent(
				m_pAsyncServerCommandData->szServer.ptr(),
				m_pAsyncServerCommandData->uPort,
				m_pAsyncServerCommandData->bPortIsOk,
				m_pAsyncServerCommandData->bUseIpV6,
				m_pAsyncServerCommandData->bUseSSL);
		}
	}

	KviIrcNetwork * net = g_pIrcServerDataBase->currentNetwork();
	KviIrcServer  * srv = 0;
	KviProxy      * prx = 0;

	if(net)srv = net->currentServer();

	if(!srv)
	{
		outputNoFmt(KVI_OUT_SYSTEMERROR,__tr("No servers available: check the options dialog or use the /SERVER command"));
		destroyAsyncServerCommandData();
		return;
	}

	if(KVI_OPTION_BOOL(KviOption_boolUseProxyHost))
	{
		prx = g_pProxyDataBase->currentProxy();
		if(!prx)
		{
			outputNoFmt(KVI_OUT_SYSTEMWARNING,__tr("No proxy hosts available: resuming direct connection"));
		}
	}

	createConnectionInfo(net,srv,prx);

	if(m_pAsyncServerCommandData)
	{
		m_pConnectionInfo->szCommandToExecAfterConnect = m_pAsyncServerCommandData->szCommandToExecAfterConnect;
		m_pConnectionInfo->szBindAddress               = m_pAsyncServerCommandData->szBindAddress;
		destroyAsyncServerCommandData();
	}

	setState(Connecting);

	output(KVI_OUT_SYSTEMMESSAGE,__tr("Attempting %s to %s (%s) on port %u"),
		m_pConnectionInfo->pIrcServer->useSSL() ? __tr("secure connection") : __tr("connection"),
		m_pConnectionInfo->pIrcServer->m_szHostname.ptr(),
		m_pConnectionInfo->szNetworkName.ptr(),
		m_pConnectionInfo->pIrcServer->m_uPort);

	if(prx)
	{
		output(KVI_OUT_SYSTEMMESSAGE,__tr("Attempting to 'bounce' on proxy %s on port %u (protocol %s)"),
			m_pConnectionInfo->pProxy->m_szHostname.ptr(),
			m_pConnectionInfo->pProxy->m_uPort,
			m_pConnectionInfo->pProxy->protocolName());
		lookupProxyHostname();
	} else {
		lookupServerHostname();
	}
}

void KviConsole::lookupProxyHostname()
{
	bool bValidIp;

#ifdef COMPILE_IPV6_SUPPORT
	if(m_pConnectionInfo->pProxy->isIpV6())
	{
		bValidIp = kvi_isValidStringIp_V6(m_pConnectionInfo->pProxy->m_szIp.ptr());
	} else {
#endif
		bValidIp = kvi_isValidStringIp(m_pConnectionInfo->pProxy->m_szIp.ptr());
#ifdef COMPILE_IPV6_SUPPORT
	}
#endif

// FIXME: #warning "option for caching proxy ip addresses too ?"
// FIXME: #warning "test IPV6 proxies ?"

	if(bValidIp)
	{
		output(KVI_OUT_SYSTEMMESSAGE,__tr("Using cached proxy IP address (%s)"),
			m_pConnectionInfo->pProxy->m_szIp.ptr());
		lookupServerHostname();
	} else {
		m_pConnectionInfo->pDns = new KviDns();
		connect(m_pConnectionInfo->pDns,SIGNAL(lookupDone(KviDns *)),this,
			SLOT(proxyLookupTerminated(KviDns *)));

		if(!m_pConnectionInfo->pDns->lookup(m_pConnectionInfo->pProxy->m_szHostname.ptr(),
			m_pConnectionInfo->pProxy->isIpV6() ? KviDns::IpV6 : KviDns::IpV4))
		{
			outputNoFmt(KVI_OUT_SYSTEMWARNING,__tr("Unable to lookup the irc proxy hostname: can't start the DNS slave"));
			outputNoFmt(KVI_OUT_SYSTEMWARNING,__tr("Resuming direct server connection"));
// FIXME: #warning "Option for resuming direct connection or not ?"
			delete m_pConnectionInfo->pDns;
			m_pConnectionInfo->pDns = 0;
			delete m_pConnectionInfo->pProxy;
			m_pConnectionInfo->pProxy = 0;
			lookupServerHostname();
		} else {
			output(KVI_OUT_SYSTEMMESSAGE,__tr("Looking up the proxy hostname (%s)"),
				m_pConnectionInfo->pProxy->m_szHostname.ptr());
		}
	}
}


void KviConsole::lookupServerHostname()
{
	bool bValidIp;

#ifdef COMPILE_IPV6_SUPPORT
	if(m_pConnectionInfo->pIrcServer->isIpV6())
	{
		bValidIp = kvi_isValidStringIp_V6(m_pConnectionInfo->pIrcServer->m_szIp.ptr());
	} else {
#endif
		bValidIp = kvi_isValidStringIp(m_pConnectionInfo->pIrcServer->m_szIp.ptr());
#ifdef COMPILE_IPV6_SUPPORT
	}
#endif

	if(bValidIp && m_pConnectionInfo->pIrcServer->cacheIp())
	{
		output(KVI_OUT_SYSTEMMESSAGE,__tr("Using cached server IP address (%s)"),
			m_pConnectionInfo->pIrcServer->m_szIp.ptr());
		haveServerIp();
	} else {
		m_pConnectionInfo->pDns = new KviDns();
		connect(m_pConnectionInfo->pDns,SIGNAL(lookupDone(KviDns *)),this,
			SLOT(serverLookupTerminated(KviDns *)));
		if(!m_pConnectionInfo->pDns->lookup(m_pConnectionInfo->pIrcServer->m_szHostname.ptr(),
			m_pConnectionInfo->pIrcServer->isIpV6() ? KviDns::IpV6 : KviDns::IpV4))
		{
			outputNoFmt(KVI_OUT_SYSTEMERROR,__tr("Unable to lookup the irc server hostname: can't start the DNS slave"));
			setState(NotConnected);
		} else {
			output(KVI_OUT_SYSTEMMESSAGE,__tr("Looking up the irc server hostname (%s)"),
				m_pConnectionInfo->pIrcServer->m_szHostname.ptr());
		}
	}
}

void KviConsole::proxyLookupTerminated(KviDns *)
{
	if(m_pConnectionInfo->pDns->state() != KviDns::Success)
	{
		output(KVI_OUT_SYSTEMERROR,__tr("Can't find the proxy IP address: %s"),
			kvi_getErrorString(m_pConnectionInfo->pDns->error()));

// FIXME: #warning "Option to resume the direct connection if proxy failed ?"

		output(KVI_OUT_SYSTEMERROR,__tr("Resuming direct server connection"));
		delete m_pConnectionInfo->pProxy;
		m_pConnectionInfo->pProxy = 0;
	} else {

		output(KVI_OUT_SYSTEMMESSAGE,__tr("Proxy hostname resolved to %s"),m_pConnectionInfo->pDns->firstIpAddress());
		m_pConnectionInfo->pProxy->m_szIp = m_pConnectionInfo->pDns->firstIpAddress();
		g_pProxyDataBase->updateProxyIp(m_pConnectionInfo->pProxy->m_szIp.ptr(),m_pConnectionInfo->pDns->firstIpAddress());

		if(m_pConnectionInfo->pDns->hostnameCount() > 1)
		{
			KviStr szFirstHostname = m_pConnectionInfo->pDns->firstHostname();

			for(KviStr * addr = m_pConnectionInfo->pDns->hostnameList()->next();addr;addr = m_pConnectionInfo->pDns->hostnameList()->next())
			{
				output(KVI_OUT_SYSTEMMESSAGE,__tr("Proxy %s has a nickname: %s"),szFirstHostname.ptr(),addr->ptr());
			}
		}
	}
	delete m_pConnectionInfo->pDns;
	m_pConnectionInfo->pDns = 0;
// FIXME: #warning "With http proxies we might even avoid lookups for server names!"
	lookupServerHostname();
}


void KviConsole::serverLookupTerminated(KviDns *)
{
	if(m_pConnectionInfo->pDns->state() != KviDns::Success)
	{
		output(KVI_OUT_SYSTEMERROR,__tr("Can't find the server IP address: %s"),
			kvi_getErrorString(m_pConnectionInfo->pDns->error()));
		output(KVI_OUT_SYSTEMERROR,__tr("Connection attempt failed [%s]"),
			m_pConnectionInfo->pIrcServer->m_szHostname.ptr());
#ifdef COMPILE_IPV6_SUPPORT
		if(!(m_pConnectionInfo->pIrcServer->isIpV6()))
		{
			output(KVI_OUT_SYSTEMERROR,__tr("If this server is an IPV6 one, try /server -i %s"),m_pConnectionInfo->pIrcServer->m_szHostname.ptr());
		}
#endif
		setState(NotConnected);
	} else {
		output(KVI_OUT_SYSTEMMESSAGE,__tr("Server hostname resolved to %s"),m_pConnectionInfo->pDns->firstIpAddress());
		g_pIrcServerDataBase->updateServerIp(m_pConnectionInfo->pIrcServer->m_szHostname.ptr(),m_pConnectionInfo->pDns->firstIpAddress());

		if(!kvi_strEqualCI(m_pConnectionInfo->pIrcServer->m_szHostname.ptr(),m_pConnectionInfo->pDns->firstHostname()))
		{
			output(KVI_OUT_SYSTEMMESSAGE,__tr("Real hostname for %s is %s"),
				m_pConnectionInfo->pIrcServer->m_szHostname.ptr(),
				m_pConnectionInfo->pDns->firstHostname());
			m_pConnectionInfo->pIrcServer->m_szHostname = m_pConnectionInfo->pDns->firstHostname();
		}

		m_pConnectionInfo->pIrcServer->m_szIp = m_pConnectionInfo->pDns->firstIpAddress();


		if(m_pConnectionInfo->pDns->hostnameCount() > 1)
		{
			KviStr szFirstHostname = m_pConnectionInfo->pDns->firstHostname();

			for(KviStr * addr = m_pConnectionInfo->pDns->hostnameList()->next();addr;addr = m_pConnectionInfo->pDns->hostnameList()->next())
			{
				output(KVI_OUT_SYSTEMMESSAGE,__tr("Server %s has a nickname: %s"),szFirstHostname.ptr(),addr->ptr());
			}
		}

		delete m_pConnectionInfo->pDns;
		m_pConnectionInfo->pDns = 0;
		haveServerIp();
	}
}

bool KviConsole::validateLocalAddress(const char * szAddress,KviStr &szBuffer)
{
	// szAddress may be an ip address or an interface name
	__range_valid(m_pConnectionInfo);
	__range_valid(m_pConnectionInfo->pIrcServer);

#ifdef COMPILE_IPV6_SUPPORT
	if(m_pConnectionInfo->pIrcServer->isIpV6())
	{
		if(kvi_isValidStringIp_V6(szAddress))
		{
			szBuffer = szAddress;
			return true;
		}
	} else {
#endif
		if(kvi_isValidStringIp(szAddress))
		{
			szBuffer = szAddress;
			return true;
		}
#ifdef COMPILE_IPV6_SUPPORT
	}
#endif

	// is it an interface name ?

	return kvi_getInterfaceAddress(szAddress,szBuffer);
}

void KviConsole::haveServerIp()
{

	if(KVI_OPTION_BOOL(KviOption_boolUseIdentService))m_pFrm->executeInternalCommand(KVI_INTERNALCOMMAND_IDENT_START);

	KviStr bindAddress;

	if(m_pConnectionInfo->szBindAddress.hasData())
	{
		if(!validateLocalAddress(m_pConnectionInfo->szBindAddress,bindAddress))
		{
			if((m_pConnectionInfo->szBindAddress.findFirstIdx('.') != -1) ||
				(m_pConnectionInfo->szBindAddress.findFirstIdx(':') != -1))
			{
				output(KVI_OUT_SYSTEMWARNING,__tr("The specified bind address (%s) is not valid"),
						m_pConnectionInfo->szBindAddress.ptr());
			} else {
				output(KVI_OUT_SYSTEMWARNING,__tr("The specified bind address (%s) is not valid (the interface it refers to might be down)"),
						m_pConnectionInfo->szBindAddress.ptr());
			}
			m_pConnectionInfo->szBindAddress = "";
		}
	} else {
		// the options specify a bind address ?
#ifdef COMPILE_IPV6_SUPPORT
		if(m_pConnectionInfo->pIrcServer->isIpV6())
		{
			if(KVI_OPTION_BOOL(KviOption_boolBindIrcIpV6ConnectionsToSpecifiedAddress))
			{
				if(KVI_OPTION_STRING(KviOption_stringIpV6ConnectionBindAddress).hasData())
				{
					if(!validateLocalAddress(KVI_OPTION_STRING(KviOption_stringIpV6ConnectionBindAddress).ptr(),bindAddress))
					{
						// if it is not an interface name , kill it for now and let the user correct the address
						if(KVI_OPTION_STRING(KviOption_stringIpV6ConnectionBindAddress).findFirstIdx(':') != -1)
						{
							output(KVI_OUT_SYSTEMWARNING,__tr("The system-wide IPV6 bind address (%s) is not valid"),
								KVI_OPTION_STRING(KviOption_stringIpV6ConnectionBindAddress).ptr());
							KVI_OPTION_BOOL(KviOption_boolBindIrcIpV6ConnectionsToSpecifiedAddress) = false;
						} else {
							// this is an interface address: might be down
							output(KVI_OUT_SYSTEMWARNING,__tr("The system-wide IPV6 bind address (%s) is not valid: the interface it refers to might be down"),
								KVI_OPTION_STRING(KviOption_stringIpV6ConnectionBindAddress).ptr());
						}
					}
				} else {
					// empty address....kill it
					KVI_OPTION_BOOL(KviOption_boolBindIrcIpV6ConnectionsToSpecifiedAddress) = false;
				}
			}
		} else {
#endif
			if(KVI_OPTION_BOOL(KviOption_boolBindIrcIpV4ConnectionsToSpecifiedAddress))
			{
				if(KVI_OPTION_STRING(KviOption_stringIpV4ConnectionBindAddress).hasData())
				{
					if(!validateLocalAddress(KVI_OPTION_STRING(KviOption_stringIpV4ConnectionBindAddress).ptr(),bindAddress))
					{
						// if it is not an interface name , kill it for now and let the user correct the address
						if(KVI_OPTION_STRING(KviOption_stringIpV4ConnectionBindAddress).findFirstIdx(':') != -1)
						{
							output(KVI_OUT_SYSTEMWARNING,__tr("The system-wide IPV4 bind address (%s) is not valid"),
								KVI_OPTION_STRING(KviOption_stringIpV4ConnectionBindAddress).ptr());
							KVI_OPTION_BOOL(KviOption_boolBindIrcIpV4ConnectionsToSpecifiedAddress) = false;
						} else {
							// this is an interface address: might be down
							output(KVI_OUT_SYSTEMWARNING,__tr("The system-wide IPV4 bind address (%s) is not valid: the interface it refers to might be down"),
								KVI_OPTION_STRING(KviOption_stringIpV4ConnectionBindAddress).ptr());
						}
					}
				} else {
					// empty address....kill it
					KVI_OPTION_BOOL(KviOption_boolBindIrcIpV4ConnectionsToSpecifiedAddress) = false;
				}
			}
#ifdef COMPILE_IPV6_SUPPORT
		}
#endif
	}

	int iErr = m_pSocket->connectToIrcHost(m_pConnectionInfo->pIrcServer,m_pConnectionInfo->pProxy,
					bindAddress.hasData() ? bindAddress.ptr() : 0);
	if(iErr != KviError_success)
	{
		output(KVI_OUT_SYSTEMERROR,__tr("Connection attempt failed (%s) : %s"),
			m_pConnectionInfo->pIrcServer->m_szHostname.ptr(),
			kvi_getErrorString(iErr));
		setState(NotConnected);
	}
}


void KviConsole::socketEvent(KviIrcSocket::SockEvent ev,void * arg)
{
	// FIXME: icons for SSL message , SSL error , Proxy message and Proxy error ?
	switch(ev)
	{
		case KviIrcSocket::SSLCertificate:
#ifdef COMPILE_SSL_SUPPORT
			KviSSLMaster::printSSLCertificate(this,__tr("Server X509 certificate"),(KviSSLCertificate *)arg);
#endif
		break;
		case KviIrcSocket::SSLCipherInfo:
#ifdef COMPILE_SSL_SUPPORT
			KviSSLMaster::printSSLCipherInfo(this,__tr("Current transmission cipher"),(KviSSLCipherInfo *)arg);
#endif
		break;
		case KviIrcSocket::SSLMessage:
			output(KVI_OUT_SSL,__tr("[SSL]: %s"),(char *)arg);
		break;
		case KviIrcSocket::SSLError:
			output(KVI_OUT_SOCKETERROR,__tr("[SSL ERROR]: %s"),(char *)arg);
		break;
		case KviIrcSocket::ProxyMessage:
			output(KVI_OUT_SOCKETMESSAGE,__tr("[PROXY]: %s"),(char *)arg);
		break;
		case KviIrcSocket::ProxyError:
			output(KVI_OUT_SOCKETERROR,__tr("[PROXY ERROR]: %s"),(char *)arg);
		break;
		case KviIrcSocket::Message:
			output(KVI_OUT_SOCKETMESSAGE,__tr("[SOCKET]: %s"),(char *)arg);
		break;
		case KviIrcSocket::Warning:
			output(KVI_OUT_SOCKETWARNING,__tr("[SOCKET WARNING]: %s"),(char *)arg);
		break;
		case KviIrcSocket::Error:
			output(KVI_OUT_SOCKETERROR,__tr("[SOCKET ERROR]: %s"),kvi_getErrorString((int)arg));
			if(m_state == Connecting)
			{
				if(m_pConnectionInfo->pIrcServer->cacheIp())
				{
					if((((int)arg) == KviError_connectionTimedOut) ||
						(((int)arg) == KviError_connectionRefused) || 
						(((int)arg) == KviError_networkUnreachable) ||
						(((int)arg) == KviError_hostUnreachable))
					{
						output(KVI_OUT_SYSTEMWARNING,__tr("The connection attempt failed while IP address caching for the current server was turned on"));
						output(KVI_OUT_SYSTEMWARNING,__tr("The problem *might* be caused by a changed DNS entry"));
						output(KVI_OUT_SYSTEMWARNING,__tr("You might try to disable caching and reconnect again"));
					}
				}
			}
		break;
		case KviIrcSocket::StateChange:
			switch((KviIrcSocket::SockState)((int)arg))
			{
				case KviIrcSocket::Connected:
				{
					if(m_pIrcSocketMonitorList)
					{
						for(KviIrcSocketMonitor *m =m_pIrcSocketMonitorList->first();m;m =m_pIrcSocketMonitorList->next())
						{
							m->connectionInitiated();
						}
					}

					bool bStopOutput = false;
					if(g_pEventManager->hasEventHandlers(KviEvent_OnIrcConnectionEstabilished))
					{
						bStopOutput = g_pUserParser->triggerEvent(KviEvent_OnIrcConnectionEstabilished,this,0);
					}
					if(!bStopOutput)
					{
						output(KVI_OUT_CONNECTION,__tr("%s estabilished [%s (%s:%u)]"),
							m_pSocket->usingSSL() ? __tr("Secure connection") : __tr("Connection"),
							m_pConnectionInfo->pIrcServer->m_szHostname.ptr(),
							m_pConnectionInfo->pIrcServer->m_szIp.ptr(),
							m_pConnectionInfo->pIrcServer->m_uPort);
					}

					// Add to recent server list (build the url of type irc[6]://<server>:<port>
					KviStr tmp(KviStr::Format,"irc%s//%s:%u",
							m_pConnectionInfo->pIrcServer->isIpV6() ? "6:" : ":",
							m_pConnectionInfo->pIrcServer->m_szHostname.ptr(),m_pConnectionInfo->pIrcServer->m_uPort);

					g_pApp->addRecentServer(tmp.ptr());
					// Ok...we're loggin in now
					setState(LoggingIn);
					resolveLocalHost();
					loginToIrcServer();
				}
				break;
				case KviIrcSocket::Ready:
				{
					if(m_pIrcSocketMonitorList)
					{
						for(KviIrcSocketMonitor *m =m_pIrcSocketMonitorList->first();m;m =m_pIrcSocketMonitorList->next())
						{
							m->connectionTerminated();
						}
					}

					// Prepare data for an eventual reconnect

					KviIrcServer oldServer(*(m_pConnectionInfo->pIrcServer));
					bool bUnexpectedDisconnect = (!(m_pConnectionInfo->bSentQuit)) && (m_state == Connected);
//					bool bConnectFailureBecauseOfCachedIp = (m_pIrcSocket->state() == KviIrcSocket::Connecting) && m_pConnectionInfo->pIrcServer->cacheIp();

					KviStr szReconnectCmd;

					if(bUnexpectedDisconnect && KVI_OPTION_BOOL(KviOption_boolAutoReconnectOnUnexpectedDisconnect))
					{
						if(KVI_OPTION_BOOL(KviOption_boolRejoinChannelsAfterReconnect))
						{
							for(KviChannel * c = m_pConnectionInfo->pChannelList->first();c;c = m_pConnectionInfo->pChannelList->next())
							{
								// FIXME: password protected channels should have the pass appended
								// FIXME: should join groups of 5 channels per message
								if(szReconnectCmd.isEmpty())szReconnectCmd.append("join ");
								else szReconnectCmd.append(",");
								szReconnectCmd.append(c->name());
							}
						}

						KviStr szQuery;

						if(KVI_OPTION_BOOL(KviOption_boolReopenQueriesAfterReconnect))
						{
							for(KviQuery * q = m_pConnectionInfo->pQueryList->first();q;q = m_pConnectionInfo->pQueryList->next())
							{
								// FIXME: should reopen multiple queries as multiple queries!!!
								// FIXME: should join groups of 5 channels per message
								if(szQuery.isEmpty())szQuery.append("query ");
								else szQuery.append(",");
								szQuery.append(q->target());
							}
						}

						if(szQuery.hasData())
						{
							if(szReconnectCmd.hasData())szReconnectCmd.append(';');
							szReconnectCmd.append(szQuery);
						}
					}
					
					setState(NotConnected);

					bool bStopOutput = false;
					if(g_pEventManager->hasEventHandlers(KviEvent_OnIrcConnectionTerminated))
					{
						bStopOutput = g_pUserParser->triggerEvent(KviEvent_OnIrcConnectionTerminated,this,0);
					}
					if(!bStopOutput)
					{
						output(KVI_OUT_CONNECTION,__tr("Connection terminated [%s (%s:%u)]"),
								oldServer.hostname(),
								oldServer.ip(),
								oldServer.port());
					}

					// do reconnect

					if(bUnexpectedDisconnect && KVI_OPTION_BOOL(KviOption_boolAutoReconnectOnUnexpectedDisconnect))
					{
						output(KVI_OUT_CONNECTION,__tr("The connnection terminated unexpectedly: trying to reconnect..."));
						KviAsyncServerCommandData * d = new KviAsyncServerCommandData;
						d->szServer = oldServer.hostname();
						d->uPort = oldServer.port();
						d->bPortIsOk = true;
						d->bUseIpV6 = oldServer.isIpV6();
						d->bUseSSL = oldServer.useSSL();
						d->szCommandToExecAfterConnect = szReconnectCmd;
						setAsyncServerCommandData(d);
						QTimer::singleShot(0,this,SLOT(asyncServerCommand()));
//					} else if(bConnectFailureBecauseOfCachedIp)
//					{
//						g_pIrcServerDataBase->disableCaching(m_pConnectionInfo->pIrcServer->m_szHostname.ptr());
//						output(KVI_OUT_SYSTEMERROR,__tr("A cached server IP address was used and the connection attempt has failed"));
//						output(KVI_OUT_SYSTEMERROR,__tr("Caching has been turned off for this server to ensure that the problem isn't caused by an invalid IP address"));
//						output(KVI_OUT_SYSTEMERROR,__tr("You might try reconnecting again"));
					}

				}
				break;
				case KviIrcSocket::Connecting:
					output(KVI_OUT_CONNECTION,__tr("Contacting %s %s (%s) on port %u"),
						m_pConnectionInfo->pProxy ? __tr("proxy host") : __tr("irc server"),
						m_pConnectionInfo->pProxy ? m_pConnectionInfo->pProxy->m_szHostname.ptr() : m_pConnectionInfo->pIrcServer->m_szHostname.ptr(),
						m_pConnectionInfo->pProxy ? m_pConnectionInfo->pProxy->m_szIp.ptr() : m_pConnectionInfo->pIrcServer->m_szIp.ptr(),
						m_pConnectionInfo->pProxy ? m_pConnectionInfo->pProxy->m_uPort : m_pConnectionInfo->pIrcServer->m_uPort);
				break;
				case KviIrcSocket::SSLHandshake:
					output(KVI_OUT_CONNECTION,__tr("Low level transport connection estabilished [%s (%s:%u)]"),
						m_pConnectionInfo->pProxy ? m_pConnectionInfo->pProxy->m_szHostname.ptr() : m_pConnectionInfo->pIrcServer->m_szHostname.ptr(),
						m_pConnectionInfo->pProxy ? m_pConnectionInfo->pProxy->m_szIp.ptr() : m_pConnectionInfo->pIrcServer->m_szIp.ptr(),
						m_pConnectionInfo->pProxy ? m_pConnectionInfo->pProxy->m_uPort : m_pConnectionInfo->pIrcServer->m_uPort);
					outputNoFmt(KVI_OUT_CONNECTION,__tr("Starting Secure Socket Layer handshake"));
				break;
				case KviIrcSocket::ProxyLogin:
					output(KVI_OUT_CONNECTION,__tr("%s estabilished [%s (%s:%u)]"),
						m_pSocket->usingSSL() ? __tr("Secure proxy connection") : __tr("Proxy connection"),
						m_pConnectionInfo->pProxy->m_szHostname.ptr(),
						m_pConnectionInfo->pProxy->m_szIp.ptr(),
						m_pConnectionInfo->pProxy->m_uPort);
					outputNoFmt(KVI_OUT_CONNECTION,__tr("Negotiating relay informations"));
				break;
				case KviIrcSocket::ProxyFinalV4:
					outputNoFmt(KVI_OUT_CONNECTION,__tr("Sent connection request, waiting for acknowledge"));
				break;
				case KviIrcSocket::ProxyFinalV5:
					outputNoFmt(KVI_OUT_CONNECTION,__tr("Sent target host data, waiting for acknowledge"));
				break;
				case KviIrcSocket::ProxySelectAuthMethodV5:
					outputNoFmt(KVI_OUT_CONNECTION,__tr("Sent auth method request, waiting for acknowledge"));
				break;
				case KviIrcSocket::ProxyUserPassV5:
					outputNoFmt(KVI_OUT_CONNECTION,__tr("Sent username and password, waiting for acknowledge"));
				break;
				case KviIrcSocket::ProxyFinalHttp:
					outputNoFmt(KVI_OUT_CONNECTION,__tr("Sent connection request, waiting for \"HTTP 200\" acknowledge"));
				break;
			}
		break;
	}
}

bool KviConsole::isIpV6Connection()
{
	__range_valid(m_pConnectionInfo);
	return m_pConnectionInfo->pIrcServer->isIpV6();
}

void KviConsole::resolveLocalHost()
{
	if(!m_pSocket->getLocalHostIp(m_pConnectionInfo->szLocalHostIp,
		m_pConnectionInfo->pIrcServer->isIpV6()))
	{
		bool bGotIp = false;
		if(KVI_OPTION_STRING(KviOption_stringLocalHostIp).hasData())
		{
#ifdef COMPILE_IPV6_SUPPORT
			if(m_pConnectionInfo->pIrcServer->isIpV6())
			{
				if(kvi_isValidStringIp_V6(KVI_OPTION_STRING(KviOption_stringLocalHostIp).ptr()))bGotIp = true;
			} else {
#endif
				if(kvi_isValidStringIp(KVI_OPTION_STRING(KviOption_stringLocalHostIp).ptr()))bGotIp = true;
#ifdef COMPILE_IPV6_SUPPORT
			}
#endif
		}
		if(bGotIp)
		{
			m_pConnectionInfo->szLocalHostIp = KVI_OPTION_STRING(KviOption_stringLocalHostIp);
			output(KVI_OUT_SYSTEMWARNING,__tr("Can't resolve local host address : using user supplied one (%s)"),
				m_pConnectionInfo->szLocalHostIp.ptr());

		} else {
			// FIXME : Maybe check for IPV6 here too ?
			m_pConnectionInfo->szLocalHostIp = "127.0.0.1";
			output(KVI_OUT_SYSTEMWARNING,__tr("Can't resolve local host address : using default 127.0.0.1"),
				m_pConnectionInfo->szLocalHostIp.ptr());
		}
	} else output(KVI_OUT_SYSTEMMESSAGE,__tr("Local host address is %s"),m_pConnectionInfo->szLocalHostIp.ptr());
	// For now this is the only we know
	m_pConnectionInfo->szLocalHostFromServer = m_pConnectionInfo->szLocalHostIp;
}

void KviConsole::loginToIrcServer()
{
	// Username
	m_pConnectionInfo->pIrcServer->m_szUser.stripWhiteSpace();
	if(m_pConnectionInfo->pIrcServer->m_szUser.isEmpty())
	{
		m_pConnectionInfo->pIrcServer->m_szUser = KVI_OPTION_STRING(KviOption_stringUsername);
		m_pConnectionInfo->pIrcServer->m_szUser.stripWhiteSpace();
		if(m_pConnectionInfo->pIrcServer->m_szUser.isEmpty())
		{
			m_pConnectionInfo->pIrcServer->m_szUser = KVI_DEFAULT_USERNAME;
		}
	} else {
		output(KVI_OUT_SYSTEMMESSAGE,__tr("Using server specific username (%s)"),
			m_pConnectionInfo->pIrcServer->m_szUser.ptr());
	}

	// For now this is the only we know
	m_pConnectionInfo->szUserNameFromServer = m_pConnectionInfo->pIrcServer->m_szUser.ptr();
	m_pConnectionInfo->szServerNameFromServer = m_pConnectionInfo->pIrcServer->m_szHostname.ptr();

	// Nick stuff
	m_pConnectionInfo->pIrcServer->m_szNick.stripWhiteSpace();
	if(m_pConnectionInfo->pIrcServer->m_szNick.hasData())
	{
		output(KVI_OUT_SYSTEMMESSAGE,__tr("Using server specific nickname (%s)"),
			m_pConnectionInfo->pIrcServer->m_szNick.ptr());
		m_pConnectionInfo->szNickName = m_pConnectionInfo->pIrcServer->m_szNick;
		m_pConnectionInfo->iNicknameUsed = 0;
	} else {
		m_pConnectionInfo->szNickName = KVI_OPTION_STRING(KviOption_stringNickname1);
		m_pConnectionInfo->szNickName.stripWhiteSpace();
		if(m_pConnectionInfo->szNickName.isEmpty())m_pConnectionInfo->szNickName = KVI_DEFAULT_NICKNAME1;
		m_pConnectionInfo->iNicknameUsed = 1;
	}

	// Real name
	m_pConnectionInfo->pIrcServer->m_szRealName.stripWhiteSpace();
	if(m_pConnectionInfo->pIrcServer->m_szRealName.hasData())
	{
		output(KVI_OUT_SYSTEMMESSAGE,__tr("Using server specific real name (%s)"),
			m_pConnectionInfo->pIrcServer->m_szRealName.ptr());
		m_pConnectionInfo->szRealName = m_pConnectionInfo->pIrcServer->m_szRealName;
	} else {
		m_pConnectionInfo->szRealName = KVI_OPTION_STRING(KviOption_stringRealname);
	}

	output(KVI_OUT_SYSTEMMESSAGE,__tr("Logging in as %s!%s :%s"),
		m_pConnectionInfo->szNickName.ptr(),
		m_pConnectionInfo->pIrcServer->m_szUser.ptr(),
		m_pConnectionInfo->szRealName.ptr());

	m_pSocket->sendFmtData("USER %s %s %s :%s",
		m_pConnectionInfo->pIrcServer->m_szUser.ptr(),
		m_pConnectionInfo->szLocalHostIp.ptr(),
		m_pConnectionInfo->pIrcServer->m_szHostname.ptr(),
		m_pConnectionInfo->szRealName.ptr());

	// The pass ?
	m_pConnectionInfo->pIrcServer->m_szPass.stripWhiteSpace();
	if(m_pConnectionInfo->pIrcServer->m_szPass.hasData())
	{
		KviStr szHidden;
		int pLen = m_pConnectionInfo->pIrcServer->m_szPass.len();
		for(int i=0;i<pLen;i++)szHidden.append('*');
	
		output(KVI_OUT_SYSTEMMESSAGE,__tr("Sending %s as password"),szHidden.ptr());
		m_pSocket->sendFmtData("PASS %s",m_pConnectionInfo->pIrcServer->m_szPass.ptr());
	}
	m_pSocket->sendFmtData("NICK %s",m_pConnectionInfo->szNickName.ptr());

	// on connect stuff ?
	KviStr tmp = m_pConnectionInfo->pIrcServer->onConnectCommand();
	tmp.stripWhiteSpace();
	if(tmp.hasData())
	{
		output(KVI_OUT_SYSTEMMESSAGE,__tr("Executing scheduled server specific \"on connect\" commands"));
		g_pUserParser->parseCommandBuffer(tmp.ptr(),this,0);
	}

	// and wait for the server to agree...
}

void KviConsole::loginComplete(const char * servNick)
{
	if(m_state == Connected)return;
	if(!kvi_strEqualCS(servNick,m_pConnectionInfo->szNickName.ptr()))
	{
		output(KVI_OUT_SYSTEMMESSAGE,__tr("The server refused the suggested nickname (%s) and named you %s instead"),
			m_pConnectionInfo->szNickName.ptr(),servNick);
		m_pConnectionInfo->szNickName = servNick;
	}
	g_pApp->addRecentNickname(servNick);
	setState(Connected);
	bool bHaltOutput = false;
	if(g_pEventManager->hasEventHandlers(KviEvent_OnIrc))
	{
		bHaltOutput = g_pUserParser->triggerEvent(KviEvent_OnIrc,this,0);	
	}
	if(!bHaltOutput)outputNoFmt(KVI_OUT_IRC,__tr("Login operations complete: happy ircing!"));
	startNotifyList();

	if(KVI_OPTION_BOOL(KviOption_boolShowChannelsJoinOnIrc))
		m_pFrm->executeInternalCommand(KVI_INTERNALCOMMAND_CHANNELSJOIN_OPEN);

	if(m_pConnectionInfo->szCommandToExecAfterConnect.hasData())
	{
		g_pUserParser->parseCommandBuffer(m_pConnectionInfo->szCommandToExecAfterConnect.ptr(),this,0);
	}
}

void KviConsole::closeEvent(QCloseEvent *e)
{
	if(m_state == NotConnected)
	{
		if(m_pFrm->consoleCount() > 1)
		{
			KviWindow::closeEvent(e);
			return;
		}
	}
	if(m_pConnectionInfo)
	{
		if(m_pConnectionInfo->bSentQuit)
		{
			if(m_pFrm->consoleCount() > 1)
			{
				KviWindow::closeEvent(e);
				return;
			}
// FIXME: #warning "ELSE SAY THAT WE CAN'T CLOSE IT , OR TRY TO QUIT THE APP!"
		}
	}
	e->ignore();
	connectButtonClicked();
}


///////////////////////////////////////////////////////////////////////////////////////////
//
// The highlight check have been moved inside the main outputPrivmsg function
// (just to avoid the call and thus make this (already ugly) process faster)
// A new KviStr::ext_contains() function has been added (see kvi_str.cpp..)
//

//bool KviConsole::checkNickInMsg(const char *msg)
//{
/*
	KviStr Msg(msg);
	if(Msg.contains(currentNickName(),false))return true;
	else return false;
*/
/*
	Is this really faster?
	Yes it is!.. KviStr Msg(msg) copies the message buffer...
	... But it SUX... you're right :D
	...mmmh.. do we need a "container only KviStr variant ?"
	...for now let's use the following:

	KviStr::ext_contains(msg,currentNickName())

	... thus I have moved it to the function below
*/


/*
	const char *aux = msg;
	const char *nick = currentNickName();
	if(nick)
	{
		int len = kvi_strLen(nick);
		char c = tolower(*nick);
		while(*aux)
		{
			while(*aux && (tolower(*aux) != c))aux++;
			if(*aux)
			{
				if(kvi_strEqualCIN(nick,aux,len))return true;
				else aux++;
			}
		}
	}
*/
//	return false;
//}

//bool KviConsole::checkHighlight(const char *msg)
//{    
	// Test for the nick 
	/*
	if (KVI_OPTION_BOOL(KviOption_boolAlwaysHighlightNick) )
	{
		const char *aux = msg;
		const char *nick = currentNickName();
		int len = kvi_strLen(nick);
		if(nick)
		{
			char c = tolower(*nick);
			while(*aux)
			{
				while(*aux && (tolower(*aux) != c))aux++;
				if(*aux)
				{
					if(kvi_strEqualCIN(nick,aux,len))return true;
					else aux++;
				}
			}
		}
	}
	*/
	/*
	for(QStringList::Iterator it = KVI_OPTION_STRINGLIST(KviOption_stringlistHighlightWords).begin();
			it != KVI_OPTION_STRINGLIST(KviOption_stringlistHighlightWords).end() ; ++it)
	{
		// FIXME : This is SLOOOOOOOOW (QString -> ascii translation!!)

		const char *aux = msg;
		const char *str = (*it).ascii();
		if(str)
		{
			int len = kvi_strLen(str);
			char c = tolower(*str); //newer \0 !
			// Run thru the string
			while(*aux)
			{
				while(*aux  && (tolower(*aux) != c))aux++; //run until we find a probable match
				if(*aux)
				{
					// probable match...check
					if(kvi_strEqualCIN(str,aux,len))return true;
					else aux++;
				} // else finished the string
			}
		}
	}
	return false;
	*/
//}

#define KVI_NUM_NICK_COLORS 95

static const char * g_nickColors[KVI_NUM_NICK_COLORS]=
{
	"0,1"  ,"0,2"  ,"0,3"  ,"0,4"  ,"0,5"  ,"0,6"  ,"0,10" ,"0,12" ,"0,14" , //9
	"1,0"  ,"1,4"  ,"1,7"  ,"1,8"  ,"1,9"  ,"1,11" ,"1,15" ,  //7
	"2,0"  ,"2,4"  ,"2,7"  ,"2,8"  ,"2,9"  ,"2,11" ,"2,15" ,  //7
	"3,8"  ,"3,9"  ,"3,0"  ,"3,15" , //4
	"4,0"  ,"4,1"  ,"4,8"  ,"4,9"  ,"4,11" ,"4,15" , //6
	"5,0"  ,"5,7"  ,"5,8"  ,"5,15" , //4
	"6,0"  ,"6,7"  ,"6,8"  ,"6,9"  ,"6,10" ,"6,11" ,"6,15" , //7
	"7,1"  ,"7,2"  ,"7,5"  ,"7,6"  ,"7,14" , //5
	"8,1"  ,"8,2"  ,"8,3"  ,"8,4"  ,"8,5"  ,"8,6"  ,"8,7"  ,"8,10" ,"8,12" ,"8,14" , //10
	"9,1"  ,"9,2"  ,"9,3"  ,"9,5"  ,"9,6"  ,"9,14" , //6
	"10,1" ,"10,2" , //2
	"11,1" ,"11,2" ,"11,3" ,"11,5" ,"11,6" ,"11,14", //6
	"12,0" ,"12,7" ,"12,8" ,"12,9" ,"12,10","12,11","12,15", //7
	"13,0" ,"13,1" ,"13,6" ,"13,8" ,"13,11","13,15", //6
	"14,0" ,"14,8" ,"14,11","14,15", //4
	"15,1" ,"15,2" ,"15,3" ,"15,6" ,"15,14" //5
};

void KviConsole::outputPrivmsg(KviWindow *wnd,int type,const char *nick,
		const char *user,const char *host,const char *msg)
{
// FIXME: #warning "THIS IS USED BY WINDOWS THAT ARE NOT BOUND TO THIS IRC CONTEXT"
// FIXME: #warning "REMEMBER IT IN ESCAPE COMMANDS"
//	__range_valid(wnd);

	if(!nick)nick = currentNickName();

	if(KVI_OPTION_BOOL(KviOption_boolAlwaysHighlightNick))
	{
		if(KviStr::ext_contains(msg,currentNickName(),false))
		{
			type = KVI_OUT_HIGHLIGHT;
			if(TRIGGER_EVENT_5PARAM_RETVALUE(KviEvent_OnHighlight,this,nick,user ? user : currentUserName(),host ? host : currentLocalHostName(),msg,currentNickName()))
				return;

			goto highlight_done;
		}
	}
	
	if(KVI_OPTION_BOOL(KviOption_boolUseWordHighlighting))
	{
		for(QStringList::Iterator it = KVI_OPTION_STRINGLIST(KviOption_stringlistHighlightWords).begin();
				it != KVI_OPTION_STRINGLIST(KviOption_stringlistHighlightWords).end() ; ++it)
		{
			// FIXME : This is SLOOOOOOOOW (QString -> ascii translation!!)
			if(KviStr::ext_contains(msg,(*it).ascii(),false))
			{
				type = KVI_OUT_HIGHLIGHT;
				if(TRIGGER_EVENT_5PARAM_RETVALUE(KviEvent_OnHighlight,this,nick,user ? user : currentUserName(),host ? host : currentLocalHostName(),msg,(*it).ascii()))
					return;

				goto highlight_done;
			}
		}
	}

highlight_done:

	// <PREFIX>nick[!user@host]<POSTFIX>This is a test message

	KviStr szNick;
	szNick.sprintf("\r!n\r%s\r",nick);

	if(KVI_OPTION_BOOL(KviOption_boolShowUserAndHostInPrivmsgView))
	{
		szNick.append(KviStr::Format,"!%s@\r!h\r%s\r",user ? user : currentUserName(),host ? host : currentLocalHostName());
	}

	bool bIsChan = (wnd->type() == KVI_WINDOW_TYPE_CHANNEL);

	if(bIsChan && KVI_OPTION_BOOL(KviOption_boolShowChannelUserFlagInPrivmsgView))
	{
		((KviChannel *)wnd)->prependUserFlag(nick,szNick);
	}

	if(KVI_OPTION_BOOL(KviOption_boolColorNicks))
	{
		int sum = 0;
		int i = 0;
		const char * aux = nick;
		while(aux[i])sum += aux[i++];
		szNick.prepend(g_nickColors[sum % KVI_NUM_NICK_COLORS]);
		szNick.prepend(KVI_TEXT_COLOR);
		szNick.append(KVI_TEXT_COLOR);
	}

	if(KVI_OPTION_BOOL(KviOption_boolBoldedNicks))
	{
		szNick.prepend(KVI_TEXT_BOLD);
		szNick.append(KVI_TEXT_BOLD);
	}


	if(KVI_OPTION_BOOL(KviOption_boolUseExtendedPrivmsgView))
	{
		if(wnd->type() == KVI_WINDOW_TYPE_CHANNEL)
		{
			((KviChannel *)wnd)->outputMessage(type,"%s%s%s%s",
				KVI_OPTION_STRING(KviOption_stringExtendedPrivmsgPrefix).ptr(),
				szNick.ptr(),
				KVI_OPTION_STRING(KviOption_stringExtendedPrivmsgPostfix).ptr(),msg);

		} else {
			wnd->output(type,"%s%s%s%s",
				KVI_OPTION_STRING(KviOption_stringExtendedPrivmsgPrefix).ptr(),
				szNick.ptr(),
				KVI_OPTION_STRING(KviOption_stringExtendedPrivmsgPostfix).ptr(),msg);
		}
	} else {
		// mini optimisation for speed
		if(bIsChan)
		{
			((KviChannel *)wnd)->outputMessage(type,"<%s> %s",szNick.ptr(),msg);
		} else {
			wnd->output(type,"<%s> %s",szNick.ptr(),msg);
		}
	}
}



void KviConsole::avatarChanged(KviAvatar * avatar,const char * nick,const char * user,const char * host,const char * textLine)
{
	if(!m_pConnectionInfo)return; //ops...

	if(KVI_OPTION_BOOL(KviOption_boolSetLastAvatarAsDefaultForRegisteredUsers))
	{
		// Don't even try to do it for myself
		if(!kvi_strEqualCI(nick,currentNickName()))
		{
			KviRegisteredUser * u = g_pRegisteredUserDataBase->findMatchingUser(nick,user,host);
			if(u)
			{
				if(avatar)u->setProperty("avatar",avatar->name());
				else u->setProperty("avatar",0);
			}
		}
	}

	for(KviChannel * c = channelList()->first();c;c = channelList()->next())
	{
		if(c->avatarChanged(nick))
		{
			if(textLine)c->output(KVI_OUT_AVATAR,textLine);
		}
	}
	for(KviQuery * q = queryList()->first();q;q = queryList()->next())
	{
		if(q->avatarChanged(nick))
		{
			if(textLine)q->output(KVI_OUT_AVATAR,textLine);
		}
	}
	m_pNotifyListView->avatarChanged(nick); // recalc the item height here too!
}

void KviConsole::checkDefaultAvatar(KviIrcUserEntry *e,const char * nick,const char * user,const char * host)
{
//	KviStr mask(KviStr::Format,"%s!%s@%s",nick,user,host);
	KviRegisteredUser * u = g_pRegisteredUserDataBase->findMatchingUser(nick,user,host);
	if(u)
	{
		// the user is registered...
		KviStr szAvatar;
		if(u->getProperty("avatar",szAvatar))
		{
			// the user has a default avatar...
			KviAvatar * avatar = g_pIconManager->getAvatar(szAvatar.ptr());
			if(avatar)
			{
				e->setAvatar(avatar);
				avatarChanged(avatar,nick,user,host,0);
			}

		}
	}
}



void KviConsole::setAvatar(const char * nick,const char * user,const char * host,const char * fName)
{
	if(!m_pConnectionInfo)return;
	KviIrcUserEntry * e = m_pConnectionInfo->pUserDataBase->find(nick);
	if(e)
	{
		// User and host must match
		if(user && e->hasUser())
		{
			if(!kvi_strEqualCI(user,e->user()))return;
		}

		if(host && e->hasHost())
		{
			if(!kvi_strEqualCI(host,e->host()))return;
		}

		// Ok...got it

		KviAvatar * avatar = g_pIconManager->getAvatar(fName);
		if(avatar)
		{
			e->setAvatar(avatar);
			avatarChanged(avatar,nick,user,host,0);
		}
	}
}

KviAvatar * KviConsole::currentAvatar()
{
	if(!m_pConnectionInfo)return 0;
	KviIrcUserEntry * e = m_pConnectionInfo->pUserDataBase->find(currentNickName());
	if(e)
	{
		KviAvatar * a = e->avatar();
		if(!a)
		{
			QPixmap * avatar = KVI_OPTION_PIXMAP(KviOption_pixmapMyAvatar).pixmap();
			if(avatar)
			{
				a = new KviAvatar(new QPixmap(*avatar),KVI_OPTION_PIXMAP(KviOption_pixmapMyAvatar).path(),true);
				e->setAvatar(a);
				avatarChanged(a,currentNickName(),0,0,0);
			}
		}
		return a;
	}
	return 0;
}

void KviConsole::applyOptions()
{
	m_pNotifyListView->applyOptions();
	m_pInput->applyOptions();
	m_pIrcView->applyOptions();

	KviWindow::applyOptions();

	// trick
	resize(width() - 1,height() - 1);
	resize(width() + 1,height() + 1);
}

//void KviConsole::setCurrentNick(const char * newNick)
//{
//
//}
//
//void KviConsole::nickChange(KviIrcUser * source,const char * newNick)
//{
//	
//}

#include "kvi_console.moc"
