//
//   File : kvi_channel.cpp
//   Creation date : Tue Aug 1 2000 02:20:22 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

//
// Channel widget : abstraction of an IRC channel
//

#define __KVIRC__

#include "kvi_channel.h"
#include "kvi_console.h"
#include "kvi_iconmanager.h"
#include "kvi_ircview.h"
#include "kvi_input.h"
#include "kvi_options.h"
#include "kvi_locale.h"
#include "kvi_topicw.h"
#include "kvi_ircsocket.h"
#include "kvi_out.h"
#include "kvi_malloc.h"
#include "kvi_taskbar.h"
#include "kvi_frame.h"
#include "kvi_config.h"
#include "kvi_themedlabel.h"
#include "kvi_maskeditor.h"
#include "kvi_event.h"
#include "kvi_uparser.h"
#include "kvi_mirccntrl.h"
#include "kvi_settings.h"
#include "kvi_parameterlist.h"
#include "kvi_modeeditor.h"

#ifdef COMPILE_CRYPT_SUPPORT
	#include "kvi_crypt.h"
	#include "kvi_cryptcontroller.h"
#endif

#include <time.h>

#include <qsplitter.h>
#include <qhbox.h>
#include <qtoolbutton.h>
#include <qlabel.h>
#include <qvaluelist.h>
#include <qpalette.h>
#include <qpopupmenu.h>


#ifndef AVERAGE_CHANNEL_USERS
	#define AVERAGE_CHANNEL_USERS 101
#endif

// kvi_sp_numeric.cpp
extern const char * getChannelModeDescription(char c);

// FIXME: #warning "I'm not op on new channels on irc.openprojects.net ---> Sync with the flag send with the WHO message!!!!"
// FIXME: #warning "+a Anonymous channel mode!"
// FIXME: #warning "+r channel mode (reop)"
// FIXME: #warning "OnChannelFlood event...."



KviChannel::KviChannel(KviFrame * lpFrm,KviConsole * lpConsole,const char * name)
: KviWindow(KVI_WINDOW_TYPE_CHANNEL,lpFrm,name,lpConsole)
{
	// Init some member variables
	m_iStateFlags          = 0;
	m_pBanList             = new KviPtrList<KviMaskEntry>;
	m_pBanList->setAutoDelete(true);
	m_pBanExceptionList    = new KviPtrList<KviMaskEntry>;
	m_pBanExceptionList->setAutoDelete(true);
	m_pInviteExceptionList = new KviPtrList<KviMaskEntry>;
	m_pInviteExceptionList->setAutoDelete(true);
	// Register ourselves
	m_pConsole->registerChannel(this);
	// And create the widgets layout
	// Button box
	m_pButtonBox = new QHBox(this,"button_box");

	m_pTopSplitter = new QSplitter(QSplitter::Horizontal,m_pButtonBox,"top_splitter");

	m_pButtonBox->setStretchFactor(m_pTopSplitter,1);

	// Topic widget on the left
	m_pTopicWidget = new KviTopicWidget(m_pTopSplitter,"topic_widget");

	connect(m_pTopicWidget,SIGNAL(topicSelected(const char *)),
		this,SLOT(topicSelected(const char *)));
	// mode label follows the topic widget
	m_pModeLabel = new KviThemedLabel(m_pTopSplitter,"mode_label");
	QToolTip::add(m_pModeLabel,__tr("Channel mode"));

	// and the related buttons
	m_pDoubleViewButton = createToolButton(m_pButtonBox,"double_view_button",KVI_SMALLICON_HIDEDOUBLEVIEW,KVI_SMALLICON_SHOWDOUBLEVIEW,__tr("Toggle double view"),false);
	connect(m_pDoubleViewButton,SIGNAL(clicked()),this,SLOT(toggleDoubleView()));
	m_pListViewButton = createToolButton(m_pButtonBox,"list_view_button",KVI_SMALLICON_HIDELISTVIEW,KVI_SMALLICON_SHOWLISTVIEW,__tr("Toggle user list"),true);
	connect(m_pListViewButton,SIGNAL(clicked()),this,SLOT(toggleListView()));
	m_pBanEditorButton = createToolButton(m_pButtonBox,"ban_editor_button",KVI_SMALLICON_UNBAN,KVI_SMALLICON_BAN,__tr("Toggle ban editor"),false);
	connect(m_pBanEditorButton,SIGNAL(clicked()),this,SLOT(toggleBanEditor()));

	if(m_pConsole->connectionInfo()->bServerSupportsModeIe)
	{
		m_pBanExceptionEditorButton = createToolButton(m_pButtonBox,"ban_exception_editor_button",KVI_SMALLICON_BANUNEXCEPT,KVI_SMALLICON_BANEXCEPT,__tr("Toggle ban exception editor"),false);
		connect(m_pBanExceptionEditorButton,SIGNAL(clicked()),this,SLOT(toggleBanExceptionEditor()));
		m_pInviteExceptionEditorButton = createToolButton(m_pButtonBox,"invite_exception_editor_button",KVI_SMALLICON_INVITEUNEXCEPT,KVI_SMALLICON_INVITEEXCEPT,__tr("Toggle invite exception editor"),false);
		connect(m_pInviteExceptionEditorButton,SIGNAL(clicked()),this,SLOT(toggleInviteExceptionEditor()));
	} else {
		m_pBanExceptionEditorButton = 0;
		m_pInviteExceptionEditorButton = 0;
	}

	m_pModeEditorButton = createToolButton(m_pButtonBox,"mode_editor_button",KVI_SMALLICON_CHANMODEHIDE,KVI_SMALLICON_CHANMODE,__tr("Toggle mode editor"),false);
	connect(m_pModeEditorButton,SIGNAL(clicked()),this,SLOT(toggleModeEditor()));
	m_pModeEditor = 0;

#ifdef COMPILE_CRYPT_SUPPORT
	createCryptControllerButton(m_pButtonBox);
#endif

	// Central splitter
	m_pSplitter = new QSplitter(QSplitter::Horizontal,this,"main_splitter");
	// Spitted vertially on the left
	m_pVertSplitter = new QSplitter(QSplitter::Vertical,m_pSplitter,
		"vertical_splitter");
	// With the IRC view over
	m_pIrcView = new KviIrcView(m_pVertSplitter,lpFrm,this);
	connect(m_pIrcView,SIGNAL(rightClicked()),this,SLOT(textViewRightClicked()));
	// And the double view (that may be unused)
	m_pMessageView = 0;
	// The userlist on the right
	m_pUserListView = new KviUserListView(m_pSplitter,m_pConsole->userDataBase(),this,
								AVERAGE_CHANNEL_USERS,__tr("User list"),"user_list_view");
	// And finally the input line on the bottom
	m_pInput   = new KviInput(this,m_pUserListView);
	// no mask editors yet
	m_pBanEditor = 0;
	m_pBanExceptionEditor = 0;
	m_pInviteExceptionEditor = 0;
	// Ensure proper focusing
	setFocusHandler(m_pInput,this);
	// And turn on the secondary IRC view if needed
//	if(KVI_OPTION_BOOL(KviOption_boolUseChannelMessageView))toggleDoubleView();

	if(KVI_OPTION_BOOL(KviOption_boolAutoLogChannels))m_pIrcView->startLogging(0);

	applyOptions();

	m_joinTime = QTime::currentTime();
}

KviChannel::~KviChannel()
{
	// Unregister ourself
	m_pConsole->unregisterChannel(this);
	// Then remove all the users and free mem
	m_pUserListView->enableUpdates(false);
	m_pUserListView->partAll();
	delete m_pBanList;
	delete m_pBanExceptionList;
	delete m_pInviteExceptionList;
}

void KviChannel::triggerCreationEvents()
{
	TRIGGER_EVENT(KviEvent_OnChannelWindowCreated,this);
}

void KviChannel::resetState()
{
	m_iStateFlags = 0;
	m_pUserListView->enableUpdates(false);
	m_pUserListView->partAll();
	m_pUserListView->enableUpdates(true);
	m_pBanList->clear();
	m_pBanExceptionList->clear();
	m_pInviteExceptionList->clear();
	m_joinTime = QTime::currentTime();
}

void KviChannel::textViewRightClicked()
{
	TRIGGER_EVENT(KviEvent_OnChannelPopupRequest,this);
}

void KviChannel::getBaseLogFileName(KviStr &buffer)
{
	buffer.sprintf("%s_%u",name(),console()->ircContextId());
}

//void KviChannel::appendSelectedNicknames(KviStr &buffer)
//{
//	m_pUserListView->appendSelectedNicknames(buffer);
//}

void KviChannel::applyOptions()
{
	m_pUserListView->applyOptions();
	m_pTopicWidget->applyOptions();

	m_pInput->applyOptions();
	m_pIrcView->applyOptions();
	if(m_pMessageView)m_pMessageView->applyOptions();

	m_pModeLabel->applyOptions();

	KviWindow::applyOptions();

	// trick
	resize(width() - 1,height() - 1);
	resize(width() + 1,height() + 1);
}

//void KviChannel::fillContextPopup(QPopupMenu * p)
//{
//	p->insertSeparator();
//	p->insertItem(__tr("&Save properties as default for channels"),this,SLOT(savePropertiesAsDefault()));
//}

void KviChannel::getConfigGroupName(KviStr &buf)
{
	buf = name();
}

//void KviChannel::savePropertiesAsDefault()
//{
//	m_pFrm->saveWindowProperties(this,typeString());
//}

void KviChannel::saveProperties(KviConfig *cfg)
{
	cfg->writeEntry("TopSplitter",m_pTopSplitter->sizes());
	cfg->writeEntry("Splitter",m_pSplitter->sizes());
	cfg->writeEntry("VertSplitter",m_pVertSplitter->sizes());
	cfg->writeEntry("PrivateBackground",m_privateBackground);
	cfg->writeEntry("DoubleView",m_pMessageView ? true : false);
}

void KviChannel::loadProperties(KviConfig *cfg)
{
	QValueList<int> def;
	def.append(65);
	def.append(25);
	def.append(10);
	m_pTopSplitter->setSizes(cfg->readIntListEntry("TopSplitter",def));
	def.clear();
	def.append(75);
	def.append(25);
	m_pSplitter->setSizes(cfg->readIntListEntry("Splitter",def));
	m_pVertSplitter->setSizes(cfg->readIntListEntry("VertSplitter",def));
	showDoubleView(cfg->readBoolEntry("DoubleView",false));
	m_privateBackground = cfg->readPixmapEntry("PrivateBackground",KviPixmap());
	if(m_privateBackground.pixmap())
	{
		m_pIrcView->setPrivateBackgroundPixmap(*(m_privateBackground.pixmap()));
		if(m_pMessageView)m_pMessageView->setPrivateBackgroundPixmap(*(m_privateBackground.pixmap()));
	}
}

void KviChannel::showDoubleView(bool bShow)
{
	if(m_pMessageView)
	{
		if(bShow)return;
		m_pIrcView->joinMessagesFrom(m_pMessageView);
		delete m_pMessageView;
		m_pMessageView = 0;
		if(m_pDoubleViewButton->isOn())m_pDoubleViewButton->setOn(false);
	} else {
		if(!bShow)return;
		m_pMessageView = new KviIrcView(m_pVertSplitter,m_pFrm,this);
		setFocusHandler(m_pInput,m_pMessageView); //link it!
		if(!(m_pDoubleViewButton->isOn()))m_pDoubleViewButton->setOn(true);
		if(m_privateBackground.pixmap())
		{
			m_pMessageView->setPrivateBackgroundPixmap(*(m_privateBackground.pixmap()));
		}
		connect(m_pMessageView,SIGNAL(rightClicked()),this,SLOT(textViewRightClicked()));
		m_pMessageView->setMasterView(m_pIrcView);
		m_pIrcView->splitMessagesTo(m_pMessageView);
		m_pMessageView->show();
	}	
}

void KviChannel::toggleDoubleView()
{
	showDoubleView(!m_pMessageView);
}

void KviChannel::toggleListView()
{
	if(m_pUserListView->isVisible())
	{
		m_pUserListView->hide();
		if(m_pListViewButton->isOn())m_pListViewButton->setOn(false);
	} else {
		m_pUserListView->show();
		if(!(m_pListViewButton->isOn()))m_pListViewButton->setOn(true);
	}
}


void KviChannel::toggleModeEditor()
{
	if(m_pModeEditor)
	{
		delete m_pModeEditor;
		m_pModeEditor = 0;
		if(m_pModeEditorButton->isOn())m_pModeEditorButton->setOn(false);
	} else {
		m_pModeEditor = new KviModeEditor(m_pSplitter,"mode_editor",m_szChannelMode.ptr(),m_szChannelKey.ptr(),m_szChannelLimit.ptr());
		connect(m_pModeEditor,SIGNAL(setMode(const char *)),this,SLOT(setMode(const char *)));
		connect(m_pModeEditor,SIGNAL(done()),this,SLOT(modeSelectorDone()));
		m_pModeEditor->show();
		setFocusHandlerNoClass(m_pInput,m_pModeEditor,"QLineEdit");
		if(!m_pModeEditorButton->isOn())m_pModeEditorButton->setOn(true);
	}
}

void KviChannel::modeSelectorDone()
{
	if(m_pModeEditor)toggleModeEditor();
}

void KviChannel::setMode(const char * mode)
{
	m_pConsole->m_pSocket->sendFmtData("MODE %s %s",name(),mode);
}

void KviChannel::toggleBanEditor()
{
	toggleEditor(&m_pBanEditor,&m_pBanEditorButton,
			m_pBanList,'b',"ban_editor");
}

void KviChannel::toggleBanExceptionEditor()
{
	toggleEditor(&m_pBanExceptionEditor,&m_pBanExceptionEditorButton,
			m_pBanExceptionList,'e',"ban_exception_editor");
}

void KviChannel::toggleInviteExceptionEditor()
{
	toggleEditor(&m_pInviteExceptionEditor,&m_pInviteExceptionEditorButton,
			m_pInviteExceptionList,'I',"invite_exception_editor");
}

void KviChannel::toggleEditor(KviMaskEditor ** ppEd,QToolButton ** ppBtn,KviPtrList<KviMaskEntry> *l,char flag,const char *edName)
{
	if(*ppEd)
	{
		delete *ppEd;
		*ppEd = 0;
		if(!(*ppBtn))return;
		if((*ppBtn)->isOn())(*ppBtn)->setOn(false);
	} else {
		*ppEd = new KviMaskEditor(m_pSplitter,l,flag,edName);
		connect(*ppEd,SIGNAL(removeMasks(KviMaskEditor *,KviPtrList<KviMaskEntry> *)),
			this,SLOT(removeMasks(KviMaskEditor *,KviPtrList<KviMaskEntry> *)));
		setFocusHandler(m_pInput,*ppEd); //link it!
		(*ppEd)->show();
		if(!(*ppBtn))return;
		if(!((*ppBtn)->isOn()))(*ppBtn)->setOn(true);
	}
}

void KviChannel::removeMasks(KviMaskEditor *ed,KviPtrList<KviMaskEntry> *l)
{
	KviStr masks;
	KviStr flags;
	unsigned int count = 0;
	for(KviMaskEntry * e = l->first();e;e = l->next())
	{
		if(masks.hasData())masks.append(' ');
		masks.append(e->szMask);
		flags.append(ed->flag());
		count++;
		if(count == KVI_OPTION_UINT(KviOption_uintModeChangesPerMessage))
		{
			console()->socket()->sendFmtData("MODE %s -%s %s",name(),flags.ptr(),masks.ptr());
			flags = "";
			masks = "";
			count = 0;
		}
	}
	if(masks.hasData())
	{
		console()->socket()->sendFmtData("MODE %s -%s %s",name(),flags.ptr(),masks.ptr());
	}
}

QPixmap * KviChannel::myIconPtr()
{
	return g_pIconManager->getSmallIcon(KVI_SMALLICON_CHANNEL);
}

void KviChannel::resizeEvent(QResizeEvent *e)
{
	int hght = m_pInput->heightHint();
	int hght2 = m_pButtonBox->sizeHint().height();
	m_pButtonBox->setGeometry(0,0,width(),hght2);
	m_pSplitter->setGeometry(0,hght2,width(),height() - (hght + hght2));
	m_pInput->setGeometry(0,height() - hght,width(),hght);
}

QSize KviChannel::sizeHint() const
{
	QSize ret(m_pSplitter->sizeHint().width(),
		m_pIrcView->sizeHint().height() + m_pInput->heightHint() + m_pButtonBox->sizeHint().height());
	return ret;
}

void KviChannel::setChannelMode(char  mode,bool bAdd)
{
	if(bAdd)
	{
		if(!(m_szChannelMode.contains(mode)))m_szChannelMode.append(mode);
	} else {
		if(m_szChannelMode.contains(mode))
		{
			KviStr tmp = mode;
			m_szChannelMode.replaceAll(tmp.ptr(),"");
		}
	}
	updateModeLabel();
}

void KviChannel::setChannelKey(const char * key)
{
	m_szChannelKey = key;
	updateModeLabel();
}

void KviChannel::setChannelLimit(const char * limit)
{
	m_szChannelLimit = limit;
	updateModeLabel();
}

void KviChannel::getChannelModeString(KviStr &buffer)
{
	buffer = m_szChannelMode;
	if(m_szChannelKey.hasData())buffer.append('k');
	if(m_szChannelLimit.hasData())buffer.append('l');
}

void KviChannel::setDeadChan()
{
	// We have been kicked!
	m_iStateFlags |= KVI_CHANNEL_STATE_DEADCHAN;
// FIXME: #warning "Do something reasonable here...disable stuff...change the chan name.."
}

void KviChannel::fillCaptionBuffers()
{
	char uFlag = getUserFlag(m_pConsole->currentNickName());

	if(uFlag)
	{
		m_szNameWithUserFlag.sprintf("%c%s",uFlag,name());
	
		m_szPlainTextCaption.sprintf(__tr("%c%s [%s on %s]"),
			uFlag,name(),m_pConsole->currentNickName(),
			m_pConsole->currentServerName());
	
		m_szHtmlActiveCaption.sprintf(
			__tr("<nobr><font color=\"%s\"><b>%c%s</b></font> " \
				"<font color=\"%s\">[%s on %s]</font></nobr>"),
			KVI_OPTION_COLOR(KviOption_colorCaptionTextActive).name().ascii(),
			uFlag,name(),
			KVI_OPTION_COLOR(KviOption_colorCaptionTextActive2).name().ascii(),
			m_pConsole->currentNickName(),
			m_pConsole->currentServerName());

		m_szHtmlInactiveCaption.sprintf(
			__tr("<nobr><font color=\"%s\"><b>%c%s</b></font> " \
				"<font color=\"%s\">[%s on %s]</font></nobr>"),
			KVI_OPTION_COLOR(KviOption_colorCaptionTextInactive).name().ascii(),
			uFlag,name(),
			KVI_OPTION_COLOR(KviOption_colorCaptionTextInactive2).name().ascii(),
			m_pConsole->currentNickName(),
			m_pConsole->currentServerName());
	} else {
		m_szNameWithUserFlag = name();
	
		m_szPlainTextCaption.sprintf(__tr("%s [%s on %s]"),
			name(),m_pConsole->currentNickName(),
			m_pConsole->currentServerName());
	
		m_szHtmlActiveCaption.sprintf(
			__tr("<nobr><font color=\"%s\"><b>%s</b></font> " \
				"<font color=\"%s\">[%s on %s]</font></nobr>"),
			KVI_OPTION_COLOR(KviOption_colorCaptionTextActive).name().ascii(),
			name(),
			KVI_OPTION_COLOR(KviOption_colorCaptionTextActive2).name().ascii(),
			m_pConsole->currentNickName(),
			m_pConsole->currentServerName());

		m_szHtmlInactiveCaption.sprintf(
			__tr("<nobr><font color=\"%s\"><b>%s</b></font> " \
				"<font color=\"%s\">[%s on %s]</font></nobr>"),
			KVI_OPTION_COLOR(KviOption_colorCaptionTextInactive).name().ascii(),
			name(),
			KVI_OPTION_COLOR(KviOption_colorCaptionTextInactive2).name().ascii(),
			m_pConsole->currentNickName(),
			m_pConsole->currentServerName());
	}
}

void KviChannel::ownMessage(const char * buffer)
{
#ifdef COMPILE_CRYPT_SUPPORT
	if(cryptSessionInfo())
	{
		if(cryptSessionInfo()->bDoEncrypt)
		{
			if(*buffer != KVI_TEXT_CRYPT)
			{
				KviStr encrypted;
				cryptSessionInfo()->pEngine->setMaxEncryptLen(500 - kvi_strLen(name()));
				if(cryptSessionInfo()->pEngine->encrypt(buffer,encrypted))
				{
					if(cryptSessionInfo()->pEngine->isCryptographicEngine())
					{
						m_pConsole->socket()->sendFmtData("PRIVMSG %s :%c%s",name(),KVI_TEXT_CRYPT,encrypted.ptr());
						m_pConsole->outputPrivmsg(this,KVI_OUT_OWNPRIVMSGCRYPTED,0,0,0,buffer);
					} else {
						m_pConsole->socket()->sendFmtData("PRIVMSG %s :%s",name(),encrypted.ptr());
						m_pConsole->outputPrivmsg(this,KVI_OUT_OWNPRIVMSG,0,0,0,encrypted.ptr());
					}
					//m_pConsole->outputPrivmsg(this,KVI_OUT_OWNPRIVMSG,0,0,0,encrypted.ptr());
					m_pUserListView->userAction(m_pConsole->currentNickName(),KVI_USERACTION_PRIVMSG);
					return;
				} else {
					output(KVI_OUT_SYSTEMERROR,
						__tr("The crypt engine was not able to encrypt the current message (%s): %s: no data sent to the server"),
						buffer,cryptSessionInfo()->pEngine->lastError());
					return;
				}
			} else {
				buffer++; //eat the escape code
			}
		}
	}
#endif
	m_pConsole->socket()->sendFmtData("PRIVMSG %s :%s",name(),buffer);
	m_pConsole->outputPrivmsg(this,KVI_OUT_OWNPRIVMSG,0,0,0,buffer);
	m_pUserListView->userAction(m_pConsole->currentNickName(),KVI_USERACTION_PRIVMSG);
}

void KviChannel::ownAction(const char * buffer)
{
	m_pConsole->socket()->sendFmtData("PRIVMSG %s :%cACTION %s%c",
		name(),0x01,buffer,0x01);
	outputMessage(KVI_OUT_ACTION,"%s %s",m_pConsole->currentNickName(),buffer);
	m_pUserListView->userAction(m_pConsole->currentNickName(),KVI_USERACTION_ACTION);
}

void KviChannel::topicSelected(const char * topic)
{
	m_pConsole->socket()->sendFmtData("TOPIC %s :%s",name(),topic);
}

void KviChannel::closeEvent(QCloseEvent *e)
{
	if((m_iStateFlags & KVI_CHANNEL_STATE_SENTPART) || (m_iStateFlags & KVI_CHANNEL_STATE_DEADCHAN))KviWindow::closeEvent(e);
	else {
		e->ignore();
		m_iStateFlags |= KVI_CHANNEL_STATE_SENTPART;
// FIXME: #warning "THIS PART SHOULD BECOME A COMMAND /PART $option()..so the identifiers are parsed"
		m_pConsole->socket()->sendFmtData("PART %s :%s",name(),
			KVI_OPTION_STRING(KviOption_stringPartMessage).ptr());
		outputNoFmt(KVI_OUT_SYSTEMMESSAGE,__tr("Sent PART request , waiting for reply..."));
	}
}

bool KviChannel::isMeOp()
{
	return m_pUserListView->isOp(m_pConsole->currentNickName());
}

int KviChannel::myFlags()
{
	return m_pUserListView->flags(m_pConsole->currentNickName());
}

//bool KviChannel::isOp(const char * nick) INLINED : Pragma
//{
//	return m_pUserListView->isOp(nick);
//}

bool KviChannel::isMeVoice()
{
	return m_pUserListView->isVoice(m_pConsole->currentNickName());
}

//bool KviChannel::isVoice(const char * nick) INLINED : Pragma
//{
//	return m_pUserListView->isVoice(nick);
//}

bool KviChannel::isMeHalfOp()
{
	return m_pUserListView->isHalfOp(m_pConsole->currentNickName());
}

//bool KviChannel::isHalfOp(const char *nick) : INLINED : Pragma
//{
//	return m_pUserListView->isHalfOp(nick);
//}

void KviChannel::banMask(const char * mask,bool bAdd,const char * setBy,const char * setAt)
{
	internalMask(mask,bAdd,setBy,setAt,m_pBanList,&m_pBanEditor);
	m_pUserListView->setbEntries((int)m_pBanList->count());
}


void KviChannel::banExceptionMask(const char * mask,bool bAdd,const char * setBy,const char * setAt)
{
	internalMask(mask,bAdd,setBy,setAt,m_pBanExceptionList,&m_pBanExceptionEditor);
	m_pUserListView->seteEntries((int)m_pBanExceptionList->count());
}

void KviChannel::inviteExceptionMask(const char * mask,bool bAdd,const char * setBy,const char * setAt)
{
	internalMask(mask,bAdd,setBy,setAt,m_pInviteExceptionList,&m_pInviteExceptionEditor);
	m_pUserListView->setIEntries((int)m_pInviteExceptionList->count());
}

void KviChannel::internalMask(const char * mask,bool bAdd,const char * setBy,const char * setAt,KviPtrList<KviMaskEntry> *l,KviMaskEditor **ppEd)
{
	KviMaskEntry * e = 0;
	if(bAdd)
	{
		for(e = l->first();e;e = l->next())
		{
			if(kvi_strEqualCI(e->szMask.ptr(),mask))return; //already there
		}
		e = new KviMaskEntry;
		e->szMask = mask;
		e->szSetBy = setBy ? setBy : __tr("unknown");
		e->szSetAt = setAt ? setAt : __tr("unknown");
		l->append(e);
		if(*ppEd)(*ppEd)->addMask(e);
	} else {
		for(e = l->first();e;e = l->next())
		{
			if(kvi_strEqualCI(e->szMask.ptr(),mask))break;
		}
		if(e)
		{
			if(*ppEd)(*ppEd)->removeMask(e);
			l->removeRef(e);
		}
	}
}

void KviChannel::updateModeLabel()
{
	KviStr tmp = m_szChannelMode;
	KviStr tip = __tr("<b>Channel mode</b><br>");
	const char * aux = tmp.ptr();
	while(*aux)
	{
		tip.append(KviStr::Format,"<br>%c: %s",*aux,getChannelModeDescription(*aux));
		++aux;
	}

	if(m_szChannelKey.hasData())
	{
		if(tmp.hasData())tmp.append(' ');
		tmp.append(KviStr::Format,"k:%s",m_szChannelKey.ptr());
		tip.append(KviStr::Format,__tr("<br>key: %s"),m_szChannelKey.ptr());
	}

	if(m_szChannelLimit.hasData())
	{
		if(tmp.hasData())tmp.append(' ');
		tmp.append(KviStr::Format,"l:%s",m_szChannelLimit.ptr());
		tip.append(KviStr::Format,__tr("<br>limit: %s"),m_szChannelLimit.ptr());
	}

	m_pModeLabel->setText(tmp.ptr());
	QToolTip::remove(m_pModeLabel);
	QToolTip::add(m_pModeLabel,tip.ptr());
}

void KviChannel::outputMessage(int msg_type,const char *format,...)
{
	char txt_ptr[512]; //It should be enough for all outputs...
	va_list list;
	va_start(list,format);
	if(kvi_vsnprintf(txt_ptr,512,format,list) < 0){
		//Just in case...
		va_end(list);
		int len = 512;
		char *long_txt_ptr = 0;
		int result;
		do{
			len += 512;
			//first time long_txt_ptr == 0 so it is equivalent to malloc
			//At least the man page says that...
			long_txt_ptr = (char *)kvi_realloc((void *)long_txt_ptr,len);
			va_start(list,format);
			result = kvi_vsnprintf(long_txt_ptr,len,format,list);
			va_end(list);
		} while(result < 0);
		if(m_pMessageView)m_pMessageView->appendText(msg_type,long_txt_ptr);
		else m_pIrcView->appendText(msg_type,long_txt_ptr);
		kvi_free((void *)long_txt_ptr);
	} else {
		//Succesful vsnprintf
		va_end(list);
		if(m_pMessageView)m_pMessageView->appendText(msg_type,txt_ptr);
		else m_pIrcView->appendText(msg_type,txt_ptr);
	}
	m_pTaskBarItem->highlight(KVI_OPTION_MSGTYPE(msg_type).level());
}

void KviChannel::setHasBanList()
{
	m_iStateFlags |= KVI_CHANNEL_STATE_HAVEBANLIST;
	// we already have all the spontaneous server replies
	// (so probably mode, topic (or no topic is set),names)
	// we have already received the I and e lists (if requested)
	bool bStop = false;
	int iSyncTime = m_joinTime.msecsTo(QTime::currentTime());
	if(iSyncTime < 0)iSyncTime += 86400000;

	if(g_pEventManager->hasEventHandlers(KviEvent_OnChannelSync))
	{
		bStop = g_pUserParser->triggerEvent(KviEvent_OnChannelSync,this,
					new KviParameterList(new KviStr(KviStr::Format,"%d",iSyncTime)));
	}
	if(!bStop && KVI_OPTION_BOOL(KviOption_boolShowChannelSyncTime))
	{
		output(KVI_OUT_SYSTEMMESSAGE,__tr("Channel synchronized in %d.%d seconds"),iSyncTime / 1000,iSyncTime % 1000);
	}
}

bool KviChannel::eventFilter(QObject * o, QEvent * e)
{
	if(e->type() == QEvent::FocusOut && o == m_pTopicWidget && \
		m_pTopicWidget->isVisible())
		m_pTopicWidget->deactivate();
	
	return KviWindow::eventFilter(o, e);
}

#include "kvi_channel.moc"
