#ifndef _KVI_SPARSER_H_
#define _KVI_SPARSER_H_

//
//   File : kvi_sparser.h
//   Creation date : Sat Jun 29 2000 14:23:11 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2001 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "kvi_settings.h"

#include <time.h>

#include <qobject.h>
#include "kvi_list.h"

#include "kvi_string.h"
#include "kvi_console.h"

class KviWindow;
class KviChannel;
class KviServerParser;
class KviFrame;

#define KVI_IRCMSG_FLAG_HALTOUTPUT 1
#define KVI_IRCMSG_FLAG_UNRECOGNIZED 2

class KVIRC_API KviIrcMessage
{
public:
	KviIrcMessage(const char * message,KviConsole * console);
	~KviIrcMessage();
private:
	const char    * m_ptr;
	KviStr          m_szPrefix;
	KviStr          m_szCommand;
	KviPtrList<KviStr> * m_pParams;
	KviConsole    * m_pConsole;
	int             m_iNumericCommand;
	KviStr          m_szNullParam;
	int             m_iFlags;
public:
	KviConsole   * console(){ return m_pConsole; };
//	KviWindow    * activeWindow(){ return m_pConsole->activeWindow(); };

	bool           isNumeric(){ return (m_iNumericCommand >= 0); };
	int            numeric(){ return m_iNumericCommand; };
	const char   * prefix(){ return m_szPrefix.ptr(); };
	const char   * safePrefix(){ return m_szPrefix.hasData() ? m_szPrefix.ptr() : m_pConsole->currentServerName(); };
	bool           hasPrefix(){ return m_szPrefix.hasData(); };
	const char   * command(){ return m_szCommand.ptr(); };
	const char   * param(unsigned int idx){ return (idx < m_pParams->count()) ? m_pParams->at(idx)->ptr() : 0; };
	const char   * safeParam(unsigned int idx){ return (idx < m_pParams->count()) ? m_pParams->at(idx)->ptr() : m_szNullParam.ptr(); };
	KviStr       * paramString(unsigned int idx){ return m_pParams->at(idx); };
	const char   * trailing(){ KviStr * tr = m_pParams->last(); return tr ? tr->ptr() : 0; };
	int            paramCount(){ return m_pParams->count(); };
	KviStr       * trailingString(){ return m_pParams->last(); };
	const char   * safeTrailing(){ KviStr * tr = m_pParams->last(); return tr ? tr->ptr() : m_szNullParam.ptr(); };
	const char   * allParams(){ return m_ptr; };
	KviStr       * prefixPtr(){ return &m_szPrefix; };
	KviStr       * commandPtr(){ return &m_szCommand; };
	KviStr       * firstParam(){ return m_pParams->first(); };
	KviStr       * nextParam(){ return m_pParams->next(); };

	void           setHaltOutput(){ m_iFlags |= KVI_IRCMSG_FLAG_HALTOUTPUT; };
	bool           haltOutput(){ return (m_iFlags & KVI_IRCMSG_FLAG_HALTOUTPUT); };
	void           setUnrecognized(){ m_iFlags |= KVI_IRCMSG_FLAG_UNRECOGNIZED; };
	bool           unrecognized(){ return (m_iFlags & KVI_IRCMSG_FLAG_UNRECOGNIZED); };
};

#define UNRECOGNIZED_MESSAGE(_msg,_err) \
		_msg->setUnrecognized(); \
		m_szLastParserError = _err;

#define IS_ME(_msg,_nick) kvi_strEqualCI(_msg->console()->currentNickName(),_nick)

typedef void (KviServerParser::*messageParseProc)(KviIrcMessage *);

typedef struct _KviLiteralMessageParseStruct
{
	const char       * msgName;
	messageParseProc   proc;
} KviLiteralMessageParseStruct;

class KviIrcMask;

typedef struct _KviCtcpMessage
{
	KviIrcMessage * msg;
	const char    * pData;
	const char    * pTarget;
	KviIrcMask    * pSource;
	bool            bIgnored;
	bool            bIsFlood;
	bool            bUnknown;

	KviStr          szTag;
} KviCtcpMessage;

typedef struct _KviDccRequest
{
	KviStr           szType;
	KviStr           szParam1;
	KviStr           szParam2;
	KviStr           szParam3;
	KviStr           szParam4;
	bool             bIpV6;
	KviCtcpMessage * ctcpMsg;
	KviConsole     * pConsole;
} KviDccRequest;

typedef void (KviServerParser::*ctcpParseProc)(KviCtcpMessage *);

typedef struct _KviCtcpMessageParseStruct
{
	const char       * msgName;
	ctcpParseProc      proc;
} KviCtcpMessageParseStruct;


#define EXTERNAL_SERVER_DATA_PARSER_CONTROL_RESET 0
#define EXTERNAL_SERVER_DATA_PARSER_CONTROL_STARTOFDATA 1
#define EXTERNAL_SERVER_DATA_PARSER_CONTROL_ENDOFDATA 2

class KVIRC_API KviExternalServerDataParser
{
public:
	KviExternalServerDataParser(){};
	virtual ~KviExternalServerDataParser(){};
public:
	virtual void processData(KviIrcMessage *){};
	virtual void control(int){};
	virtual void die(){ delete this; };
};


class KVIRC_API KviServerParser : public QObject
{
	Q_OBJECT
public:
	KviServerParser();
	~KviServerParser();
private:
	static messageParseProc             m_numericParseProcTable[1000];
	static KviLiteralMessageParseStruct m_literalParseProcTable[];
	static KviCtcpMessageParseStruct    m_ctcpRequestParseProcTable[];
	static KviCtcpMessageParseStruct    m_ctcpReplyParseProcTable[];
	KviStr                              m_szLastParserError;

//	KviStr                              m_szNoAwayNick; //<-- moved to kvi_console.h in KviConnectionInfo
public:
	void parseMessage(const char * message,KviConsole * console);
private:
	void parseNumeric001(KviIrcMessage *msg);
	void parseNumeric002(KviIrcMessage *msg);
	void parseNumeric003(KviIrcMessage *msg);
	void parseNumeric004(KviIrcMessage *msg);
	void parseNumeric005(KviIrcMessage *msg);

	void parseNumericMotd(KviIrcMessage *msg);
	void parseNumericNames(KviIrcMessage *msg);
	void parseNumericEndOfNames(KviIrcMessage *msg);
	void parseNumericTopic(KviIrcMessage *msg);
	void parseNumericNoTopic(KviIrcMessage *msg);
	void parseNumericTopicWhoTime(KviIrcMessage *msg);
	void parseNumericChannelModeIs(KviIrcMessage *msg);
	void parseNumericBanList(KviIrcMessage *msg);
	void parseNumericEndOfBanList(KviIrcMessage *msg);
	void parseNumericInviteList(KviIrcMessage *msg);
	void parseNumericEndOfInviteList(KviIrcMessage *msg);
	void parseNumericExceptList(KviIrcMessage *msg);
	void parseNumericEndOfExceptList(KviIrcMessage *msg);
	void parseNumericWhoReply(KviIrcMessage *msg);
	void parseNumericEndOfWho(KviIrcMessage *msg);
	void parseNumericNicknameInUse(KviIrcMessage *msg);
	void parseNumericWhoisAway(KviIrcMessage *);
	void parseNumericWhoisUser(KviIrcMessage *msg);
	void parseNumericWhoisChannels(KviIrcMessage *msg);
	void parseNumericWhoisIdle(KviIrcMessage *msg);
	void parseNumericWhoisServer(KviIrcMessage *msg);
	void parseNumericWhoisOther(KviIrcMessage *msg);
	void parseNumericWhowasUser(KviIrcMessage *msg);
	void parseNumericEndOfWhois(KviIrcMessage *msg);
	void parseNumericEndOfWhowas(KviIrcMessage *msg);
	void parseNumericNoSuchNick(KviIrcMessage *msg);
	void parseNumericCreationTime(KviIrcMessage *msg);
	void parseNumericIsOn(KviIrcMessage *msg);
	void parseNumericUnavailResource(KviIrcMessage *msg);
	void parseNumericLinks(KviIrcMessage *msg);
	void parseNumericEndOfLinks(KviIrcMessage *msg);
	void parseNumericUserhost(KviIrcMessage *msg);
	void parseLoginNicknameProblem(KviIrcMessage *msg);
	void parseNumericBackFromAway(KviIrcMessage *);
	void parseNumericAway(KviIrcMessage *);
	void parseNumericWatch(KviIrcMessage *msg);
	void parseNumericList(KviIrcMessage *msg);
	void parseNumericListStart(KviIrcMessage *msg);
	void parseNumericListEnd(KviIrcMessage *msg);
	void parseNumericCantJoinChannel(KviIrcMessage *msg);
	void parseNumericStats(KviIrcMessage *msg);
	void parseNumericUserMode(KviIrcMessage * msg);

	void parseLiteralPing(KviIrcMessage *msg);
	void parseLiteralJoin(KviIrcMessage *msg);
	void parseLiteralTopic(KviIrcMessage *msg);
	void parseLiteralPart(KviIrcMessage *msg);
	void parseLiteralPrivmsg(KviIrcMessage *msg);
	void parseLiteralNotice(KviIrcMessage *msg);
	void parseLiteralQuit(KviIrcMessage *msg);
	void parseLiteralNick(KviIrcMessage *msg);
	void parseLiteralMode(KviIrcMessage *msg);
	void parseLiteralKick(KviIrcMessage *msg);
	void parseLiteralInvite(KviIrcMessage *msg);
	void parseLiteralWallops(KviIrcMessage *msg);
	void parseChannelMode(KviIrcMask &source,KviChannel * chan,KviStr &modefl,KviIrcMessage *msg,int curParam);
	void parseUserMode(KviIrcMessage *msg,const char * modeflptr);


	const char * decodeCtcpEscape(const char * msg_ptr,KviStr &buffer);
	void parseCtcpRequest(KviCtcpMessage *msg);
	void parseCtcpReply(KviCtcpMessage *msg);
	void echoCtcpRequest(KviCtcpMessage *msg);
	void echoCtcpReply(KviCtcpMessage *msg);
	void replyCtcp(KviCtcpMessage *msg,const char * data);
	bool checkCtcpFlood(KviCtcpMessage *msg);

	void parseCtcpRequestPing(KviCtcpMessage *msg);
	void parseCtcpRequestVersion(KviCtcpMessage *msg);
	void parseCtcpRequestUserinfo(KviCtcpMessage *msg);
	void parseCtcpRequestClientinfo(KviCtcpMessage *msg);
	void parseCtcpRequestFinger(KviCtcpMessage *msg);
	void parseCtcpRequestSource(KviCtcpMessage *msg);
	void parseCtcpRequestAction(KviCtcpMessage *msg);
	void parseCtcpRequestTime(KviCtcpMessage *msg);
	void parseCtcpRequestAvatar(KviCtcpMessage *msg);
	void parseCtcpRequestDcc(KviCtcpMessage *msg);
	void parseCtcpRequestPage(KviCtcpMessage *msg);

	void parseCtcpReplyAvatar(KviCtcpMessage *msg);
	void parseCtcpReplyGeneric(KviCtcpMessage *msg);
	void parseCtcpReplyPing(KviCtcpMessage *msg);
	


	const char * formatGenericLink(const char * escape,const char * link,KviStr &buffer);
	const char * formatServerLink(const char * link,KviStr &buffer);
	const char * formatHostLink(const char * link,KviStr &buffer);
	const char * formatChannelLink(const char * link,KviStr &buffer);
	const char * formatNickLink(const char * link,KviStr &buffer);
//	const char * formatMaskLink(const char * link,KviStr &buffer);

//	void parseCtcpReply(const char * msg_ptr,KviIrcUser &source,const char * target);
public:
	const char * extractCtcpParameter(const char * msg_ptr,KviStr &buffer,bool bSpaceBreaks = true);
};


extern KVIRC_API KviServerParser * g_pServerParser;


#endif //_KVI_SPARSER_H_
