//
//   File : kvi_sp_numeric.cpp
//   Creation date : Thu Aug 3 2000 01:30:45 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
#define __KVIRC__

#include "kvi_sparser.h"
#include "kvi_window.h"
#include "kvi_out.h"
#include "kvi_locale.h"
#include "kvi_ircsocket.h"
#include "kvi_options.h"
#include "kvi_channel.h"
#include "kvi_topicw.h"
#include "kvi_ircuserdb.h"
#include "kvi_defaults.h"
#include "kvi_mirccntrl.h"
#include "kvi_frame.h"
#include "kvi_uparser.h"
#include "kvi_parameterlist.h"
#include "kvi_app.h"
#include "kvi_notifylist.h"
#include "kvi_numeric.h"
#include "kvi_event.h"

#include <qpixmap.h>
#include <qdatetime.h>

// Numeric message handlers

// FIXME: #warning "IN ALL OUTPUT ADD ESCAPE SEQUENCES!!!!"
// FIXME: #warning "parseErrorUnknownModeChar() for modes e and I , parseErrorUnknownCommand for WATCH"

void KviServerParser::parseNumeric001(KviIrcMessage *msg)
{
	// 001: RPL_WELCOME
	// :prefix 001 target :Welcome to the Internet Relay Network <usermask>
	// FIXME: #warning "SET THE USERMASK FROM SERVER"
	
	if(!(msg->console()->isConnected()))msg->console()->loginComplete(msg->param(0));
	if(!msg->haltOutput())msg->console()->output(KVI_OUT_SERVERINFO,msg->trailing());
}

void KviServerParser::parseNumeric002(KviIrcMessage *msg)
{
	// 002: RPL_YOURHOST [I,E,U,D]
	// :prefix 002 target :Your host is <server name>, running version <server version>
	if(!(msg->console()->isConnected()))msg->console()->loginComplete(msg->param(0));
	if(!msg->haltOutput())msg->console()->output(KVI_OUT_SERVERINFO,msg->trailing());
}

void KviServerParser::parseNumeric003(KviIrcMessage *msg)
{
	// 003: RPL_CREATED [I,E,U,D]
	// :prefix 003 target :This server was created <date>
	if(!(msg->console()->isConnected()))msg->console()->loginComplete(msg->param(0));
	if(!msg->haltOutput())msg->console()->output(KVI_OUT_SERVERINFO,msg->trailing());
}


const char * getChannelModeDescription(char c)
{
	switch(c)
	{
		case 'o': return __tr("channel operators"); break;
		case 'v': return __tr("voiced users"); break;
		case 'b': return __tr("ban masks"); break;
		case 'c': return __tr("color free (no ANSI colors)"); break;
		case 'e': return __tr("ban exception masks"); break;
		case 'I': return __tr("invite exception masks or forbid /INVITE"); break;
		case 's': return __tr("secret"); break;
		case 'p': return __tr("private"); break;
		case 't': return __tr("topic change restricted"); break;
		case 'i': return __tr("invite only"); break;
		case 'n': return __tr("no external messages"); break;
		case 'a': return __tr("anonymous or protected user"); break;
		case 'q': return __tr("quiet or channel owner"); break;
		case 'l': return __tr("limited number of users"); break;
		case 'k': return __tr("key"); break;
		case 'm': return __tr("moderated"); break;
		case 'r': return __tr("registered"); break;
		case 'G': return __tr("censor swearwords"); break;
		case 'R': return __tr("only registered nicks can join"); break;
		case 'Q': return __tr("no kicks able (unless U-Line)"); break;
		case 'O': return __tr("IRC-Op only channel"); break;
		case 'A': return __tr("Server Admin | Network Admin | Tech Admin only channel"); break;
		case 'K': return __tr("forbid /KNOCK"); break;
		case 'S': return __tr("strip colors"); break;
		case 'L': return __tr("redirect on channel full"); break;
		case 'C': return __tr("forbid channel CTCPs"); break;
		case 'u': return __tr("auditorium : /NAMES and /WHO shows only ops"); break;
		default: return __tr("unknown channel mode"); break;
	}
}

void KviServerParser::parseNumeric004(KviIrcMessage *msg)
{
	// 004: RPL_MYINFO [I,E,U,D]
	// :prefix 004 target <server_name> <srv_version> <u_modes> <ch_modes>
	if(!(msg->console()->isConnected()))msg->console()->loginComplete(msg->param(0));

	KviStr umodes    = msg->safeParam(3);
	KviStr chanmodes = msg->safeParam(4);

	if((umodes.occurences('o') != 1) || (chanmodes.occurences('o') != 1) ||
		(chanmodes.occurences('b') != 1) || (chanmodes.occurences('v') != 1) ||
		(chanmodes.occurences('t') != 1) || (chanmodes.occurences('n') != 1))
	{
		msg->console()->output(KVI_OUT_SYSTEMWARNING,__tr(
			"One or more standard mode flags are missing in the server available modes.\n" \
			"This is caused either by a non RFC1459-compliant irc daemon or a broken server reply.\n" \
			"Server umodes seem to be '%s' and chanmodes seem to be '%s'.\n" \
			"Ignoring this reply and assuming that the basic set of modes is available.\n" \
			"If you have strange problems, try changing the server."));

		umodes = "oiws";    // standard support
		chanmodes = "obtkmlvsn"; // standard support
	}

// FIXME: #warning "CHECK THE e AND I MODE SUPPORT ?"
//	msg->console()->seteModeSupported(chanmodes.occurences('e') == 1);
//	msg->console()->setIModeSupported(chanmodes.occurences('I') == 1);
//
//	if(KVI_OPTION_BOOL(KviOption_boolBeVerbose)){
//		if(m_pFrm->m_global.bServerSupportsEMode)m_pConsole->output(KVI_OUT_INTERNAL,__tr("The server seems to support ban and invite exception lists (channel modes e and I are listed as supported)"));
//		else m_pConsole->output(KVI_OUT_INTERNAL,__tr("The server has no support for invite and ban exception lists (channel modes e and I are not supported both)"));
//	}

	if(KVI_OPTION_BOOL(KviOption_boolShowExtendedServerInfo) && (!msg->haltOutput()))
	{
		if(umodes.hasData())msg->console()->outputNoFmt(KVI_OUT_SERVERINFO,__tr("Available user modes:"));

		const char * aux = umodes.ptr();
		KviStr tmp;

		while(*aux)
		{
			switch(*aux)
			{
				case 'o': tmp = __tr("o: irc operator (OPER)"); break;
				case 'O': tmp = __tr("O: local irc operator (LOCOP)"); break;
				case 'i': tmp = __tr("i: invisible"); break;
				case 'w': tmp = __tr("w: recipient for WALLOPS messages"); break;
				case 'r': tmp = __tr("r: user with restricted connection (or recipient for messages about rejected bots)"); break;
 				case 's': tmp = __tr("s: recipient for server notices"); break;
				case 'z': tmp = __tr("z: recipient for oper wallop messages"); break;
				case 'c': tmp = __tr("c: recipient for cconn messages"); break;
				case 'k': tmp = __tr("k: recipient for server kill messages"); break;
				case 'f': tmp = __tr("f: recipient for full server notices"); break;
				case 'y': tmp = __tr("y: spy :)"); break;
				case 'd': tmp = __tr("d: obscure 'DEBUG' flag"); break;
				case 'n': tmp = __tr("n: recipient for nick changes"); break;
				default: tmp = __tr(": unknown user mode"); tmp.prepend(*aux); break;
			}

			msg->console()->outputNoFmt(KVI_OUT_SERVERINFO,tmp.ptr());
			aux++;
		}

		if(chanmodes.hasData())msg->console()->outputNoFmt(KVI_OUT_SERVERINFO,__tr("Available channel modes:"));

		aux = chanmodes.ptr();

		while(*aux)
		{
			tmp.sprintf("%c: %s",*aux,getChannelModeDescription(*aux));
			msg->console()->outputNoFmt(KVI_OUT_SERVERINFO,tmp.ptr());
			aux++;
		}
	}

	msg->console()->setServerInfoFromServer(msg->param(1),umodes.ptr(),chanmodes.ptr());

// FIXME: #warning "TO ACTIVE ? OR TO CONSOLE ?"
	if(!msg->haltOutput())msg->console()->output(KVI_OUT_SERVERINFO,
		__tr("Server %s version %s supporting modes '%s' and '%s'"),
		msg->safeParam(1),msg->safeParam(2),
		msg->safeParam(3),msg->safeParam(4));
}

void KviServerParser::parseNumeric005(KviIrcMessage *msg)
{
	// 005: RPL_PROTOCTL [D]
	// :prefix 005 target <proto> <proto> .... :are available on this server
	// 005: RPL_BOUNCE [?]
	// :prefix 005 target :Try server <server>, port <port>
	if(!(msg->console()->isConnected()))msg->console()->loginComplete(msg->param(0));

	const char * aux = msg->allParams();


	while(*aux == ' ')aux++;
	while((*aux != ' ') && *aux)aux++;
	if(*aux)aux++;

	if(!msg->haltOutput())msg->console()->outputNoFmt(KVI_OUT_SERVERINFO,aux);

	// Check if it is a RPL_PROTOCTL and if we can use the WATCH list
	KviStr ap = aux;
	KviStr trailing = msg->trailing();

	if(trailing.findFirstIdx("available") != -1)
	{
		if(ap.findFirstIdx("WATCH") != -1)
		{
			msg->console()->connectionInfo()->bServerSupportsWatchList = true;
			if(!msg->haltOutput() && KVI_OPTION_BOOL(KviOption_boolBeVerbose))
				msg->console()->outputNoFmt(KVI_OUT_SYSTEMMESSAGE,
						__tr("This server seems to support the WATCH notify list method: will try to use it."));
		}
	}
}

void KviServerParser::parseNumericMotd(KviIrcMessage *msg)
{
	// 372: RPL_MOTD [I,E,U,D]
	// :prefix 372 target : - <motd>
	// 377: RPL_MOTD2 [?]
	// :prefix 377 target : - <motd>
	// 378: RPL_MOTD3 [Austnet]
	// :prefix 377 target : - <motd>
	// 375: RPL_MOTDSTART [I,E,U,D]
	// :prefix 375 target : - <server> Message of the Day -
	// 372: RPL_ENDOFMOTD [I,E,U,D]
	// :prefix 376 target :End of /MOTD command.
// FIXME: #warning "SKIP MOTD , MOTD IN A SEPARATE WINDOW , SILENT ENDOFMOTD , MOTD IN ACTIVE WINDOW"
	if(!msg->haltOutput())msg->console()->output(KVI_OUT_MOTD,msg->trailing());

	if(msg->numeric() == RPL_ENDOFMOTD)
	{
		if(msg->console()->connectionInfo()->pNotifyListTimer)
		{
			// The timer is there running , not fired yet
			msg->console()->restartNotifyList();
		}
	}
}

void KviServerParser::parseNumericEndOfNames(KviIrcMessage *msg)
{
	// 366: RPL_ENDOFNAMES [I,E,U,D]
	// :prefix 366 target <channel> :End of /NAMES list.
	KviChannel * chan = msg->console()->findChannel(msg->safeParam(1));
	if(chan)
	{
		if(!chan->hasAllNames())
		{
			chan->setHasAllNames();
			return;
		}
	}

	if(!msg->haltOutput())
	{
// FIXME: #warning "KVI_OUT_NAMES missing"
// FIXME: #warning "TO ACTIVE ? OR TO CONSOLE ?"
		KviWindow * pOut = chan ? chan : msg->console()->activeWindow();
		pOut->output(KVI_OUT_UNHANDLED,__tr("End of names for \r!c\r%s\r"),msg->safeParam(1));
	}
}

void KviServerParser::parseNumericNames(KviIrcMessage *msg)
{
	// 353: RPL_NAMREPLY [I,E,U,D]
	// :prefix 353 target [=|*|@] <channel> :<space_separated_list_of_nicks>

	// [=|*|@] is the type of the channel:
	// = --> public  * --> private   @ --> secret
	// ...but we ignore it

	KviChannel * chan = msg->console()->findChannel(msg->safeParam(2));

	// and run to the first nickname
	const char * aux = msg->safeTrailing();
	while((*aux) && (*aux == ' '))aux++;
	// now check if we have that channel

	const char * trailing = aux;


	bool bHalt = msg->haltOutput();

	if(chan)
	{
		bHalt = bHalt || (!chan->hasAllNames());

		// K...time to parse a lot of data
		KviStr aNick;
		char aMode;

		chan->enableUserListUpdates(false);

		int iPrevFlags = chan->myFlags();

		while(*aux)
		{
			// @ = op (+o) , + = voiced (+v) , % = halfop (+h) , ^ = protected (+a) , * = chan owner (+q)
			// ^  +a is a weird mode: it also breaks nicknames on some networks!
			if((*aux == '@')||(*aux == '+')||(*aux == '%')||(*aux == '*'))
			{
				// leading umode flag
				aMode = *aux;
				aux++;
			} else aMode = 0;
			// extract the nickname
			aux = kvi_extractToken(aNick,aux,' ');
			// and make it join
			if(aNick.hasData())
			{
				int iFlags = 0;
				switch(aMode)
				{
					case '@': iFlags = KVI_USERFLAG_OP;      break;
					case '+': iFlags = KVI_USERFLAG_VOICE;   break;
					case '%': iFlags = KVI_USERFLAG_HALFOP;  break;
				// For now we ignore protected and chan owner
				}
				chan->join(aNick.ptr(),0,0,iFlags);
			}
			// run to the next nick (or the end)
			while((*aux) && (*aux == ' '))aux++;
		}

		if(iPrevFlags != chan->myFlags())chan->updateCaption();

		chan->enableUserListUpdates(true);
		// finished a block
	}

	// So it is a result of a /NAMES command or a local desync
	// We handle it in a cool way.

	if(!bHalt)
	{
// FIXME: #warning "KVI_OUT_NAMES missing"
		KviWindow * pOut = chan ? chan : msg->console()->activeWindow();
		pOut->output(KVI_OUT_UNHANDLED,__tr("Names for \r!c\r%s\r: %s"),msg->safeParam(2),trailing);
	}
}

void KviServerParser::parseNumericTopic(KviIrcMessage *msg)
{
	// 332: RPL_TOPIC [I,E,U,D]
	// :prefix 332 target <channel> :<topic>
	KviChannel * chan = msg->console()->findChannel(msg->safeParam(1));
	if(chan)
	{
		chan->topicWidget()->setTopic(msg->safeTrailing());
		chan->topicWidget()->setTopicSetBy(__tr("Unknown"));
		if(KVI_OPTION_BOOL(KviOption_boolEchoNumericTopic))
		{
			if(!msg->haltOutput())
				chan->output(KVI_OUT_TOPIC,__tr("Channel topic is: %s"),msg->safeTrailing());
		}
	} else {
		msg->console()->activeWindow()->output(KVI_OUT_TOPIC,__tr("Topic for \r!c\r%s\r is: %s"),
			msg->safeParam(1),msg->safeTrailing());
	}

}

void KviServerParser::parseNumericNoTopic(KviIrcMessage *msg)
{
	// 331: RPL_NOTOPIC [I,E,U,D]
	// :prefix 331 target <channel> :No topic is set
	KviChannel * chan = msg->console()->findChannel(msg->safeParam(1));
	if(chan)
	{
		chan->topicWidget()->setTopic("");
		if(KVI_OPTION_BOOL(KviOption_boolEchoNumericTopic))
		{
			if(!msg->haltOutput())
				chan->outputNoFmt(KVI_OUT_TOPIC,__tr("No channel topic is set"));
		}
	} else {
		msg->console()->activeWindow()->output(KVI_OUT_TOPIC,__tr("No topic is set for channel \r!c\r%s\r"),
			msg->safeParam(1));
	}
}

void KviServerParser::parseNumericTopicWhoTime(KviIrcMessage *msg)
{
	// 333: RPL_TOPICWHOTIME [e,U,D]
	// :prefix 333 target <channel> <whoset> <time>

	KviChannel * chan = msg->console()->findChannel(msg->safeParam(1));
	KviStr tmp = msg->safeParam(3);
	bool bOk = false;
	unsigned long t = 0;
	if(tmp.hasData())t = tmp.toUInt(&bOk);

	QDateTime dt;
	dt.setTime_t(t);

	tmp = dt.toString();

	if(chan)
	{
		chan->topicWidget()->setTopicSetBy(msg->safeParam(2));
		if(bOk)chan->topicWidget()->setTopicSetAt(tmp.ptr());
		if(KVI_OPTION_BOOL(KviOption_boolEchoNumericTopic))
		{
			if(!msg->haltOutput())
			{
				if(bOk)chan->output(KVI_OUT_TOPIC,__tr("Topic was set by \r!n\r%s\r on %s"),msg->safeParam(2),tmp.ptr());
				else chan->output(KVI_OUT_TOPIC,__tr("Topic was set by \r!n\r%s\r"),msg->safeParam(2));
			}
		}
	} else {
		KviStr buffer;
		if(bOk)
		{
			msg->console()->activeWindow()->output(KVI_OUT_TOPIC,__tr("Topic for \r!c\r%s\r was set by \r!n\r%s\r on %s"),
					msg->safeParam(1),msg->safeParam(2),tmp.ptr());
		} else {
			msg->console()->activeWindow()->output(KVI_OUT_TOPIC,__tr("Topic for \r!c\r%s\r was set by \r!n\r%s\r"),
					msg->safeParam(1),msg->safeParam(2));
		}
	}
}

void KviServerParser::parseNumericChannelModeIs(KviIrcMessage *msg)
{
	// 324: RPL_CHANNELMODEIS [I,E,U,D]
	// :prefix 324 target <channel> +<chanmode>

	KviIrcMask source(msg->safePrefix());
	KviChannel * chan = msg->console()->findChannel(msg->safeParam(1));
	KviStr modefl = msg->safeParam(2);
	if(chan)parseChannelMode(source,chan,modefl,msg,3);
	else {
		msg->console()->activeWindow()->output(KVI_OUT_CHANMODE,__tr("Channel mode for \r!c\r%s\r is %s"),
			msg->safeParam(1),msg->safeParam(2));
	}
}

void KviServerParser::parseNumericEndOfBanList(KviIrcMessage *msg)
{
	// 368: RPL_ENDOFBANLIST [I,E,U,D]
	// :prefix 368 target <channel> :End of channel Ban List.
	KviChannel * chan = msg->console()->findChannel(msg->safeParam(1));
	if(chan)
	{
		if(!chan->hasBanList())
		{
// FIXME: #warning "IF VERBOSE && SHOW INTERNAL BAN LIST REPLIES...."
			chan->setHasBanList();
			return;
		}
	}
// FIXME: #warning "TO ACTIVE ? OR TO CONSOLE ?"
	if(!msg->haltOutput())
	{
		KviWindow * pOut = chan ? chan : msg->console()->activeWindow();
		pOut->output(KVI_OUT_BAN,__tr("End of channel ban list for \r!c\r%s\r"),msg->safeParam(1));
	}
}

void getDateTimeStringFromCharTimeT(KviStr &buffer,const char *time_t_string)
{
	KviStr tmp=time_t_string;
	bool bOk=false;
	unsigned int uTime = tmp.toUInt(&bOk);
	if(bOk){
		QDateTime dt;
		dt.setTime_t(uTime);
		buffer = dt.toString();
	} else buffer = __tr("[Unknown]");
}

void KviServerParser::parseNumericBanList(KviIrcMessage *msg)
{
	// 367: RPL_BANLIST [I,E,U,D]
	// :prefix 367 target <channel> <banmask> [bansetby] [bansetat]

	KviStr banmask  = msg->safeParam(2);
	KviStr bansetby = msg->safeParam(3);
	KviStr bansetat;
	getDateTimeStringFromCharTimeT(bansetat,msg->safeParam(4));
	if(bansetby.isEmpty())bansetby = __tr("[Unknown]");

	KviChannel * chan = msg->console()->findChannel(msg->safeParam(1));
	if(chan)
	{
		if(!chan->hasBanList())
		{
// FIXME: #warning "IF VERBOSE && SHOW INTERNAL BAN LIST REPLIES...."
			chan->banMask(banmask.ptr(),true,bansetby.ptr(),bansetat.ptr());
			return;
		}
	}
// FIXME: #warning "TO ACTIVE ? OR TO CONSOLE ?"
	if(!msg->haltOutput())
	{
		KviWindow * pOut = chan ? chan : msg->console()->activeWindow();
		pOut->output(KVI_OUT_BAN,__tr("Ban entry for \r!c\r%s\r: \r!m-b\r%s\r (set by %s on %s)"),
			msg->safeParam(1),banmask.ptr(),bansetby.ptr(),bansetat.ptr());
	}
}

void KviServerParser::parseNumericEndOfInviteList(KviIrcMessage *msg)
{
	// 347: RPL_ENDOFINVITELIST [I,E,U,D]
	// :prefix 347 target <channel> :End of Channel Invite List.
	KviChannel * chan = msg->console()->findChannel(msg->safeParam(1));
	if(chan)
	{
		if(!chan->hasInviteExceptionList())
		{
// FIXME: #warning "IF VERBOSE && SHOW INTERNAL INVITE LIST REPLIES...."
			chan->setHasInviteExceptionList();
			return;
		}
	}
// FIXME: #warning "TO ACTIVE ? OR TO CONSOLE ?"
	if(!msg->haltOutput())
	{
		KviWindow * pOut = chan ? chan : msg->console()->activeWindow();
		pOut->output(KVI_OUT_INVITEEXCEPT,__tr("End of channel invite list for \r!c\r%s\r"),
			msg->safeParam(1));
	}
}

void KviServerParser::parseNumericInviteList(KviIrcMessage *msg)
{
	// 346: RPL_INVITELIST [I,E,U,D]
	// :prefix 346 target <channel> <invitemask> [invitesetby] [invitesetat]

	KviStr invitemask  = msg->safeParam(2);
	KviStr invitesetby = msg->safeParam(3);
	KviStr invitesetat;
	getDateTimeStringFromCharTimeT(invitesetat,msg->safeParam(4));
	if(invitesetby.isEmpty())invitesetby = __tr("[Unknown]");

	KviChannel * chan = msg->console()->findChannel(msg->safeParam(1));
	if(chan)
	{
		if(!chan->hasInviteExceptionList())
		{
// FIXME: #warning "IF VERBOSE && SHOW INTERNAL INVITE LIST REPLIES...."
			chan->inviteExceptionMask(invitemask.ptr(),true,invitesetby.ptr(),invitesetat.ptr());
			return;
		}
	}
// FIXME: #warning "TO ACTIVE ? OR TO CONSOLE ?"
	if(!msg->haltOutput())
	{
		KviWindow * pOut = chan ? chan : msg->console()->activeWindow();
		pOut->output(KVI_OUT_INVITEEXCEPT,__tr("Invite entry for \r!c\r%s\r: \r!m-I\r%s\r (set by %s on %s)"),
			msg->safeParam(1),invitemask.ptr(),invitesetby.ptr(),invitesetat.ptr());
	}
}

void KviServerParser::parseNumericEndOfExceptList(KviIrcMessage *msg)
{
	// 347: RPL_ENDOFEXCEPTLIST [I,E,U,D]
	// :prefix 347 target <channel> :End of Channel Ban Exception List.
	KviChannel * chan = msg->console()->findChannel(msg->safeParam(1));
	if(chan)
	{
		if(!chan->hasBanExceptionList())
		{
// FIXME: #warning "IF VERBOSE && SHOW INTERNAL EXCEPT LIST REPLIES...."
			chan->setHasBanExceptionList();
			return;
		}
	}
// FIXME: #warning "TO ACTIVE ? OR TO CONSOLE ?"
	if(!msg->haltOutput())
	{
		KviWindow * pOut = chan ? chan : msg->console()->activeWindow();
		pOut->output(KVI_OUT_BANEXCEPT,__tr("End of channel ban exception list for \r!c\r%s\r"),
			msg->safeParam(1));
	}
}

void KviServerParser::parseNumericExceptList(KviIrcMessage *msg)
{
	// 346: RPL_EXCEPTLIST [I,E,U,D]
	// :prefix 346 target <channel> <banmask> [bansetby] [bansetat]

	KviStr banmask  = msg->safeParam(2);
	KviStr bansetby = msg->safeParam(3);
	KviStr bansetat;
	getDateTimeStringFromCharTimeT(bansetat,msg->safeParam(4));
	if(bansetby.isEmpty())bansetby = __tr("[Unknown]");

	KviChannel * chan = msg->console()->findChannel(msg->safeParam(1));
	if(chan)
	{
		if(!chan->hasBanExceptionList())
		{
// FIXME: #warning "IF VERBOSE && SHOW INTERNAL EXCEPT LIST REPLIES...."
			chan->banExceptionMask(banmask.ptr(),true,bansetby.ptr(),bansetat.ptr());
			return;
		}
	}
// FIXME: #warning "TO ACTIVE ? OR TO CONSOLE ?"
	if(!msg->haltOutput())
	{
		KviWindow * pOut = chan ? chan : msg->console()->activeWindow();
		pOut->output(KVI_OUT_BANEXCEPT,__tr("Ban exception entry for \r!c\r%s\r: \r!m-e\r%s\r (set by %s on %s)"),
			msg->safeParam(1),banmask.ptr(),bansetby.ptr(),bansetat.ptr());
	}
}

void KviServerParser::parseNumericWhoReply(KviIrcMessage *msg)
{
	// 352: RPL_WHOREPLY [I,E,U,D]
	// :prefix 352 target <chan> <usr> <hst> <srv> <nck> <stat> :<hops> <real>

	// Update the user entry
	KviIrcUserDataBase * db = msg->console()->userDataBase();
	KviIrcUserEntry * e = db->find(msg->safeParam(5));
	if(e)
	{
		e->setUser(msg->safeParam(2));
		e->setHost(msg->safeParam(3));
		e->setServer(msg->safeParam(4));
		KviStr trailing = msg->safeTrailing();
		KviStr hops = trailing.getToken(' ');
		bool bOk;
		int iHops = hops.toInt(&bOk);
		if(bOk)e->setHops(iHops);
		e->setRealName(trailing.ptr());
		if(!e->avatar())
		{
// FIXME: #warning "THE AVATAR SHOULD BE RESIZED TO MATCH THE MAX WIDTH/HEIGHT"
			// maybe now we can match this user ?
			msg->console()->checkDefaultAvatar(e,msg->safeParam(5),e->user(),e->host());
		}
	}

	KviChannel * chan = msg->console()->findChannel(msg->safeParam(1));
	if(chan)
	{
		if(!chan->hasWhoList())
		{
// FIXME: #warning "IF VERBOSE && SHOW INTERNAL WHO REPLIES...."
			return;
		}
	}

// FIXME: #warning "TO ACTIVE ? OR TO CONSOLE ?"

// FIXME: #warning "SYNC OP/VOICE on channel!!!"

	if(!msg->haltOutput())
	{
		KviWindow * pOut = chan ? chan : msg->console()->activeWindow();
		KviStr hops = msg->safeTrailing();
		hops.cutFromFirst(' ');
		KviStr real = msg->safeTrailing();
		real.cutToFirst(' ');
//		KviStr buffer,buffer2,buffer3, buffer4;
		pOut->output(KVI_OUT_WHO,__tr("Who entry for %c\r!n\r%s\r%c[%s@\r!h\r%s\r]: %cChannel:%c \r!c\r%s\r, %cServer%c: \r!s\r%s\r, %cHops:%c %s, %cAway:%c %s, %cReal name:%c %s"),
			KVI_TEXT_BOLD,msg->safeParam(5), KVI_TEXT_BOLD, 
			msg->safeParam(2),msg->safeParam(3),KVI_TEXT_UNDERLINE,
			KVI_TEXT_UNDERLINE, msg->safeParam(1), KVI_TEXT_UNDERLINE,
			KVI_TEXT_UNDERLINE, msg->safeParam(4), KVI_TEXT_UNDERLINE,
			KVI_TEXT_UNDERLINE, hops.ptr(), KVI_TEXT_UNDERLINE, KVI_TEXT_UNDERLINE,
			kvi_strEqualCI(msg->safeParam(6), "G") ? __tr("Yes") : __tr("No"), KVI_TEXT_UNDERLINE,
			KVI_TEXT_UNDERLINE,real.ptr());
	}
}

void KviServerParser::parseNumericEndOfWho(KviIrcMessage *msg)
{
	// 315: RPL_ENDOFWHO [I,E,U,D]
	// :prefix 315 target <channel/nick> :End of /WHO List.
	KviChannel * chan = msg->console()->findChannel(msg->safeParam(1));
	if(chan)
	{
		if(!chan->hasWhoList())
		{
// FIXME: #warning "IF VERBOSE && SHOW INTERNAL WHO REPLIES...."
			chan->setHasWhoList();
			return;
		}
	}
// FIXME: #warning "TO ACTIVE ? OR TO CONSOLE ?"
	if(!msg->haltOutput())
	{
		KviWindow * pOut = chan ? chan : msg->console()->activeWindow();
		KviStr buffer;
		KviStr whoTarget = msg->safeParam(1);
		bool bIsChan = ((*(whoTarget.ptr())) == '#') || ((*(whoTarget.ptr())) == '&') || ((*(whoTarget.ptr())) == '!');
		pOut->output(KVI_OUT_WHO,__tr("End of /WHO List for %s"),
			bIsChan ? formatChannelLink(whoTarget.ptr(),buffer) : formatNickLink(whoTarget.ptr(),buffer));
	}
}

void KviServerParser::parseLoginNicknameProblem(KviIrcMessage *msg)
{
	// ops...not logged in yet...
	KviStr nextNick;
	KviStr defNick;
	int iNickCnt;
	switch(msg->console()->connectionInfo()->iNicknameUsed)
	{
		case 0:
			nextNick = KVI_OPTION_STRING(KviOption_stringNickname1);
			defNick = KVI_DEFAULT_NICKNAME1;
			iNickCnt = 1;
		case 1:
			nextNick = KVI_OPTION_STRING(KviOption_stringNickname2);
			defNick = KVI_DEFAULT_NICKNAME2;
			iNickCnt = 2;
		break;
		case 2:
			nextNick = KVI_OPTION_STRING(KviOption_stringNickname3);
			defNick = KVI_DEFAULT_NICKNAME3;
			iNickCnt = 3;
		break;
		default:
			// random string...
			nextNick = msg->safeParam(1);
			nextNick.stripWhiteSpace();
			if(nextNick.isEmpty())nextNick = KVI_DEFAULT_NICKNAME1;
			*(nextNick.ptr()) = ((*(nextNick.ptr())) + 1);
			if(*(nextNick.ptr()) > 'z')*(nextNick.ptr()) = 'a';
		break;
	}
	msg->console()->connectionInfo()->szNickName = nextNick;
	msg->console()->connectionInfo()->szNickName.stripWhiteSpace();
	if(msg->console()->connectionInfo()->szNickName.isEmpty())
		msg->console()->connectionInfo()->szNickName = defNick;

	msg->console()->connectionInfo()->iNicknameUsed = iNickCnt;

	if(!msg->haltOutput())
	{
		msg->console()->output(KVI_OUT_NICKNAMEPROBLEM,
			__tr("No way to login as '\r!n\r%s\r' (%d:%s): trying '%s'..."),
				msg->safeParam(1),msg->numeric(),msg->safeTrailing(),nextNick.ptr());
	}

	msg->console()->socket()->sendFmtData("NICK %s",nextNick.ptr());
}

void KviServerParser::parseNumericUnavailResource(KviIrcMessage *msg)
{
	// 437: ERR_UNAVAILRESOURCE [I]
	// :prefix 437 <target> <nick/channel> :Nick/Channel is temporairly unavailable
	if(!(msg->console()->isConnected()))
	{
		parseLoginNicknameProblem(msg);
	} else {
		// already connected... just say that we have problems
		if(!msg->haltOutput())
		{
			msg->console()->activeWindow()->output(KVI_OUT_NICKNAMEPROBLEM,
				"\r!n\r%s\r: %s",msg->safeParam(1),msg->safeTrailing());
		}
	}
}

void KviServerParser::parseNumericCantJoinChannel(KviIrcMessage *msg)
{
	// 471: ERR_CHANNELISFULL [I,E,U,D]
	// 473: ERR_INVITEONLYCHAN [I,E,U,D]
	// 474: ERR_BANNEDFROMCHAN [I,E,U,D]
	// 475: ERR_BADCHANNELKEY [I,E,U,D]
	// :prefix 47* <target> <channel> :Can't join channel (+l/i/b/k)
	if(!msg->haltOutput())
	{
		msg->console()->activeWindow()->output(KVI_OUT_JOINERROR,
			"\r!c\r%s\r: %s",msg->safeParam(1),msg->safeTrailing());
	}
}

void KviServerParser::parseNumericNicknameInUse(KviIrcMessage *msg)
{
	// 433: ERR_NICKNAMEINUSE [I,E,U,D]
	// :prefix 433 <target> <nick> :Nickname is already in use.

	if(!(msg->console()->isConnected()))
	{
		parseLoginNicknameProblem(msg);
	} else {
		// already connected... just say that we have problems
		if(!msg->haltOutput())
		{
			msg->console()->activeWindow()->output(KVI_OUT_NICKNAMEPROBLEM,
				"\r!n\r%s\r: %s",msg->safeParam(1),msg->safeTrailing());
		}
	}
}

void KviServerParser::parseNumericWhoisAway(KviIrcMessage * msg)
{
// FIXME: #warning "Need an icon here too: sth like KVI_OUT_WHOISSERVER, but with 'A' letter"
	if(!msg->haltOutput())
	{
		msg->console()->activeWindow()->output(KVI_OUT_WHOISUSER,__tr("%c\r!n\r%s\r%c is away: %s"),
			KVI_TEXT_BOLD, msg->safeParam(1), KVI_TEXT_BOLD, msg->safeTrailing());
	}
}

void KviServerParser::parseNumericWhoisUser(KviIrcMessage *msg)
{
	// 311: RPL_WHOISUSER [I,E,U,D]
	// :prefix 311 <target> <nick> <user> <host> * :<real_name>
	KviIrcUserDataBase * db = msg->console()->userDataBase();
	KviIrcUserEntry * e = db->find(msg->safeParam(1));
	if(e)
	{
		e->setUser(msg->safeParam(2));
		e->setHost(msg->safeParam(3));
		e->setRealName(msg->safeTrailing());
		if(!e->avatar())
		{
// FIXME: #warning "THE AVATAR SHOULD BE RESIZED TO MATCH THE MAX WIDTH/HEIGHT"
			// maybe now we can match this user ?
			msg->console()->checkDefaultAvatar(e,msg->safeParam(1),e->user(),e->host());
		}
	}

	KviWhoisInfo * i = msg->console()->lookupAsyncWhois(msg->safeParam(1));
	if(i)
	{
		i->szNick = msg->safeParam(1);
		i->szUser = msg->safeParam(2);
		i->szHost = msg->safeParam(3);
		i->szReal = msg->safeTrailing();
		return;
	}

	if(!msg->haltOutput())
	{
		msg->console()->activeWindow()->output(
			KVI_OUT_WHOISUSER,__tr("%c\r!n\r%s\r%c is %c\r!n\r%s\r!%s@\r!h\r%s\r%c"),KVI_TEXT_BOLD,
				msg->safeParam(1),KVI_TEXT_BOLD,KVI_TEXT_UNDERLINE,msg->safeParam(1),
				msg->safeParam(2),msg->safeParam(3),KVI_TEXT_UNDERLINE);

		msg->console()->activeWindow()->output(
			KVI_OUT_WHOISUSER,__tr("%c%s%c's real name: %s"),KVI_TEXT_BOLD,
				msg->safeParam(1),KVI_TEXT_BOLD,msg->safeTrailing());
	}
}

void KviServerParser::parseNumericWhowasUser(KviIrcMessage * msg)
{
	// 314: RPL_WHOWASUSER [I,E,U,D]
	// :prefix 314 <target> <nick> <user> <host> * :<real_name>
	if(!msg->haltOutput())
	{
		msg->console()->activeWindow()->output(
			KVI_OUT_WHOISUSER,__tr("%c\r!n\r%s\r%c was %c\r!n\r%s\r!%s@\r!h\r%s\r%c"),KVI_TEXT_BOLD,
			msg->safeParam(1),KVI_TEXT_BOLD,KVI_TEXT_UNDERLINE,msg->safeParam(1),
			msg->safeParam(2),msg->safeParam(3),KVI_TEXT_UNDERLINE);
		msg->console()->activeWindow()->output(
			KVI_OUT_WHOISUSER,__tr("%c\r!n\r%s\r%c's real name was: %s"),KVI_TEXT_BOLD,
			msg->safeParam(1),KVI_TEXT_BOLD,msg->safeTrailing());
	}
}


void KviServerParser::parseNumericWhoisChannels(KviIrcMessage *msg)
{
	// 319: RPL_WHOISCHANNELS [I,E,U,D]
	// :prefix 319 <target> <nick> :<channel list>

	KviWhoisInfo * i = msg->console()->lookupAsyncWhois(msg->safeParam(1));
	if(i)
	{
		i->szChannels = msg->safeTrailing();
		return;
	}

	if(!msg->haltOutput())
	{
		KviStr channels;
		// channel names are links
		KviStr tmp = msg->safeTrailing();
		while(tmp.hasData())
		{
			KviStr chan;
			tmp.getToken(chan,' ');
			if(chan.hasData())
			{
				if(channels.hasData())channels.append(", ");
				
				if((*(chan.ptr()) == '@') || (*(chan.ptr()) == '+') || (*(chan.ptr()) == '^') || (*(chan.ptr()) == '*') || (*(chan.ptr()) == '%'))
				{
					char flag = *(chan.ptr());
					chan.cutLeft(1);
					channels.append(KviStr::Format,"%c\r!c\r%s\r",flag,chan.ptr());
				} else {
					channels.append(KviStr::Format,"\r!c\r%s\r",chan.ptr());
				}
			}
		} //else channels = msg->safeTrailing();

		msg->console()->activeWindow()->output(
			KVI_OUT_WHOISCHANNELS,__tr("%c\r!n\r%s\r%c's channels: %s"),KVI_TEXT_BOLD,
				msg->safeParam(1),KVI_TEXT_BOLD,channels.ptr());
	}
}

void KviServerParser::parseNumericWhoisIdle(KviIrcMessage *msg)
{
	// 317: RPL_WHOISIDLE [I,E,U,D]
	// :prefix 317 <target> <nick> <number> <number> :seconds idle, signon time

// FIXME: #warning "and NICK LINKS"
	KviWhoisInfo * i = msg->console()->lookupAsyncWhois(msg->safeParam(1));
	if(i)
	{
		i->szIdle = msg->safeParam(2);
		i->szSignon = msg->safeParam(3);
		if(!i->szSignon.isUnsignedNum())i->szSignon = "";
		return;
	}

	if(!msg->haltOutput())
	{
		KviStr idle = msg->safeParam(2);
		KviStr sign = msg->safeParam(3);

		bool bOk;
		unsigned int uTime = idle.toUInt(&bOk);
		if(!bOk)
		{
			UNRECOGNIZED_MESSAGE(msg,__tr("Received a broken RPL_WHOISIDLE: can't evaluate the idle time"));
			return;
		}
		unsigned int uDays = uTime / 86400;
		uTime = uTime % 86400;
		unsigned int uHours = uTime / 3600;
		uTime = uTime % 3600;
		unsigned int uMins = uTime / 60;
		uTime = uTime % 60;
		msg->console()->activeWindow()->output(
			KVI_OUT_WHOISIDLE,__tr("%c\r!n\r%s\r%c's idle time: %u d %u h %u m %u s"),KVI_TEXT_BOLD,
			msg->safeParam(1),KVI_TEXT_BOLD,uDays,uHours,uMins,uTime);

		uTime = sign.toUInt(&bOk);
		if(bOk)
		{
			QDateTime dt;
			dt.setTime_t((time_t)uTime);
			KviStr tmp = dt.toString();
			msg->console()->activeWindow()->output(
				KVI_OUT_WHOISIDLE,__tr("%c\r!n\r%s\r%c's signon time: %s"),KVI_TEXT_BOLD,
				msg->safeParam(1),KVI_TEXT_BOLD,tmp.ptr());
		}
	}
}

void KviServerParser::parseNumericWhoisServer(KviIrcMessage *msg)
{
	// 312: RPL_WHOISSERVER [I,E,U,D] (sent also in response to WHOWAS)
	// :prefix 312 <target> <nick> <server> :<server description / last whowas date>

	KviIrcUserDataBase * db = msg->console()->userDataBase();
	KviIrcUserEntry * e = db->find(msg->safeParam(1));
	if(e)e->setServer(msg->safeParam(2));

	KviWhoisInfo * i = msg->console()->lookupAsyncWhois(msg->safeParam(1));
	if(i)
	{
		i->szServer = msg->safeParam(2);
		return;
	}

// FIXME: #warning "AWHOIS HERE.... and NICK LINKS"
	if(!msg->haltOutput())
	{
		msg->console()->activeWindow()->output(
			KVI_OUT_WHOISSERVER,__tr("%c\r!n\r%s\r%c's server: \r!s\r%s\r : %s"),KVI_TEXT_BOLD,
			msg->safeParam(1),KVI_TEXT_BOLD,msg->safeParam(2),msg->safeTrailing());
	}
}

void KviServerParser::parseNumericWhoisOther(KviIrcMessage *msg)
{
	// *: RPL_WHOIS* [?]
	// :prefix * <target> <nick> :<description>
	// used for RPL_WHOISCHANOP,RPL_WHOISADMIN,
	// RPL_WHOISSADMIN,RPL_WHOISOPERATOR,RPL_WHOISREGNICK,RPL_WHOISSSL

	KviWhoisInfo * i = msg->console()->lookupAsyncWhois(msg->safeParam(1));
	if(i)
	{
		if(i->szServer.hasData())i->szServer.append(',');
		i->szServer.append(msg->safeTrailing());
		return;
	}

// FIXME: #warning "NICK LINKS"
	if(!msg->haltOutput())
	{
		msg->console()->activeWindow()->output(
			KVI_OUT_WHOISOTHER,__tr("%c\r!n\r%s\r%c's info: %s"),KVI_TEXT_BOLD,
			msg->safeParam(1),KVI_TEXT_BOLD,msg->safeTrailing());
	}
}

// FIXME: #warning "WHOWAS MISSING"

void KviServerParser::parseNumericEndOfWhois(KviIrcMessage *msg)
{
	// 318: RPL_ENDOFWHOIS [I,E,U,D]
	// :prefix 318 <target> <nick> :End of /WHOIS list

	KviWhoisInfo * i = msg->console()->lookupAsyncWhois(msg->safeParam(1));
	if(i)
	{
		if(!g_pApp->windowExists(i->pWindow))i->pWindow = msg->console();
		KviParameterList *params = new KviParameterList();
		params->append(new KviStr(i->szNick));
		params->append(new KviStr(i->szUser));
		params->append(new KviStr(i->szHost));
		params->append(new KviStr(i->szReal));
		params->append(new KviStr(i->szServer));
		params->append(new KviStr(i->szIdle));
		params->append(new KviStr(i->szSignon));
		params->append(new KviStr(i->szChannels));
		params->append(new KviStr(msg->safePrefix()));
		params->append(new KviStr(i->szSpecial));
		params->append(new KviStr(i->szMagic));
		g_pUserParser->parseCommandBuffer(i->szCode.ptr(),i->pWindow,params);
		msg->console()->removeAsyncWhois(i);
		return;
	}

// FIXME: #warning "NICK LINKS"
	if(!msg->haltOutput())
	{
		msg->console()->activeWindow()->output(
			KVI_OUT_WHOISOTHER,__tr("%c\r!n\r%s\r%c whois info from \r!s\r%s\r"),KVI_TEXT_BOLD,
			msg->safeParam(1),KVI_TEXT_BOLD,msg->safePrefix());
	}
}


void KviServerParser::parseNumericEndOfWhowas(KviIrcMessage *msg)
{
	// 369: RPL_ENDOFWHOWAS [I,E,U,D]
	// :prefix 369 <target> <nick> :End of /WHOWAS list
	if(!msg->haltOutput())
	{
		msg->console()->activeWindow()->output(
			KVI_OUT_WHOISOTHER,__tr("%c\r!n\r%s\r%c whowas info from \r!s\r%s\r"),KVI_TEXT_BOLD,
			msg->safeParam(1),KVI_TEXT_BOLD,msg->safePrefix());
	}
}

void KviServerParser::parseNumericNoSuchNick(KviIrcMessage *msg)
{
	// 401: ERR_NOSUCHNICK [I,E,U,D]
	// 406: ERR_WASNOSUCHNICK [I,E,U,D]
	// :prefix 401 <target> <nick> :No such nick/channel
	// :prefix 406 <target> <nick> :There was no such nickname
	KviWhoisInfo * i = msg->console()->lookupAsyncWhois(msg->safeParam(1));
	if(i)
	{
		if(!g_pApp->windowExists(i->pWindow))i->pWindow = msg->console();
		KviParameterList *params = new KviParameterList();
		params->append(new KviStr(i->szNick));
		g_pUserParser->parseCommandBuffer(i->szCode.ptr(),i->pWindow,params);
		msg->console()->removeAsyncWhois(i);
		return;
	}

// FIXME: #warning "KVI_OUT_NOSUCHNICKCHANNEL ?"
// FIXME: #warning "QUERIES SHOULD REPORT NO TARGET HERE! (?)"
	if(!msg->haltOutput())
	{
		msg->console()->activeWindow()->output(KVI_OUT_NICKNAMEPROBLEM,__tr("\r!n\r%s\r: %s"),
			msg->safeParam(1),msg->safeTrailing());
	}
}

void KviServerParser::parseNumericCreationTime(KviIrcMessage *msg)
{
	// 329: RPL_CREATIONTIME
	// :prefix 329 <target> <channel> <creation_time>
	KviChannel * chan = msg->console()->findChannel(msg->safeParam(1));
	KviStr tmstr = msg->safeParam(2);
	QDateTime dt;
	dt.setTime_t((time_t)tmstr.toUInt());

	if(!tmstr.isUnsignedNum())
	{
		UNRECOGNIZED_MESSAGE(msg,__tr("Can't evaluate the creation time"));
		return;
	}
	if(chan)
	{
		if(!msg->haltOutput())
		{
			chan->output(KVI_OUT_CREATIONTIME,__tr("Channel was created at %s"),dt.toString().ascii());
		}
	} else {
		msg->console()->activeWindow()->output(KVI_OUT_CREATIONTIME,__tr("Channel \r!c\r%s\r was created at %s"),
			msg->safeParam(1),dt.toString().ascii());
	}
}

void KviServerParser::parseNumericIsOn(KviIrcMessage *msg)
{
	// 303: RPL_ISON
	// :prefix 303 <target> :<ison replies>
	if(msg->console()->connectionInfo()->pNotifyListManager)
	{
		if(msg->console()->connectionInfo()->pNotifyListManager->handleIsOn(msg))return;
	}
	// not handled...output it
// FIXME: #warning "OUTPUT IT!"
}

void KviServerParser::parseNumericUserhost(KviIrcMessage *msg)
{
	// 302: RPL_USERHOST
	// :prefix 302 <target> :<userhost replies>
	if(msg->console()->connectionInfo()->pNotifyListManager)
	{
		if(msg->console()->connectionInfo()->pNotifyListManager->handleUserhost(msg))return;
	}
	// not handled...output it
// FIXME: #warning "OUTPUT IT!"
}

void KviServerParser::parseNumericListStart(KviIrcMessage *msg)
{
	// 321: RPL_LISTSTART [I,E,U,D]
	// :prefix 321 <target> :Channel users name
	if(!(msg->console()->listWindow()))
	{
		// attempt to load the module...
		msg->console()->createListWindow();
	}

	if(msg->console()->listWindow())
	{
		// module loaded
		msg->console()->listWindow()->control(EXTERNAL_SERVER_DATA_PARSER_CONTROL_STARTOFDATA);

	} else {
		if(!msg->haltOutput())
		{
			msg->console()->activeWindow()->output(KVI_OUT_LIST,__tr("Channel list begin: Channel , users , topic"));
		}
	}
}

void KviServerParser::parseNumericList(KviIrcMessage *msg)
{
	// 322: RPL_LIST [I,E,U,D]
	// :prefix 364 <target> <channel> <users> :<topic>
	if(!(msg->console()->listWindow()))
	{
		// attempt to load the module...
		msg->console()->createListWindow();
		if(msg->console()->listWindow())
		{
			msg->console()->listWindow()->control(EXTERNAL_SERVER_DATA_PARSER_CONTROL_STARTOFDATA);
		}
	}

	if(msg->console()->listWindow())
	{
		// module loaded
		msg->console()->listWindow()->processData(msg);
	} else {
		// ops...can't load the module...
		if(!msg->haltOutput())
		{
			msg->console()->activeWindow()->output(KVI_OUT_LIST,__tr("List: %s"),msg->allParams());
		}
	}
}

void KviServerParser::parseNumericListEnd(KviIrcMessage *msg)
{
	// 323: RPL_LISTEND [I,E,U,D]
	// :prefix 323 <target> :End of /LIST
	if(msg->console()->listWindow())
	{
		msg->console()->listWindow()->control(EXTERNAL_SERVER_DATA_PARSER_CONTROL_ENDOFDATA);
	} else {
		if(!msg->haltOutput())
		{
			msg->console()->activeWindow()->output(KVI_OUT_LIST,__tr("End of channel list"));
		}
	}

}

void KviServerParser::parseNumericLinks(KviIrcMessage *msg)
{
	// 364: RPL_LINKS [I,E,U,D]
	// :prefix 364 <target> <host> <parent> :<hops> <description>
	if(!(msg->console()->linksWindow()))
	{
		// attempt to load the module...
		msg->console()->createLinksWindow();
	}

	if(msg->console()->linksWindow())
	{
		// module loaded
		msg->console()->linksWindow()->processData(msg);
	} else {
		// ops...can't load the module...
		if(!msg->haltOutput())
		{
			msg->console()->activeWindow()->output(KVI_OUT_LINKS,__tr("Link: %s"),msg->allParams());
		}
	}
}

void KviServerParser::parseNumericEndOfLinks(KviIrcMessage *msg)
{
	// 365: RPL_ENDOFLINKS [I,E,U,D]
	// :prefix 365 <target> :End of /LINKS
	if(msg->console()->linksWindow())
	{
		msg->console()->linksWindow()->control(EXTERNAL_SERVER_DATA_PARSER_CONTROL_ENDOFDATA);
	} else {
		if(!msg->haltOutput())
		{
			msg->console()->activeWindow()->output(KVI_OUT_LINKS,__tr("End of links"));
		}
	}
}

void KviServerParser::parseNumericBackFromAway(KviIrcMessage * msg)
{
	// 305: RPL_UNAWAY [I,E,U,D]
	// :prefix 305 <target> :You are no longer away

	bool bWasAway = msg->console()->userIsAway();

	msg->console()->changeAwayState(false);

	KviStr tmp(KviStr::Format,"%u",bWasAway ? (unsigned int)(msg->console()->connectionInfo()->tAwayTime) : 0);

	if(TRIGGER_EVENT_2PARAM_RETVALUE(KviEvent_OnMeBack,msg->console(),tmp.ptr(),msg->safeTrailing()))msg->setHaltOutput();


	if(KVI_OPTION_BOOL(KviOption_boolChangeNickAway) && bWasAway)
	{
		KviStr command(KviStr::Format,"nick %s",msg->console()->connectionInfo()->szNickBeforeAway.ptr());
		g_pUserParser->parseCommandBuffer(command.ptr(), msg->console());
	}

	if(!msg->haltOutput())
	{
		if(bWasAway)
		{
			int uTimeDiff = bWasAway ? (time(0) - msg->console()->connectionInfo()->tAwayTime) : 0;
			msg->console()->activeWindow()->output(KVI_OUT_AWAY,__tr("[Leaving away status after %d d %d h %d m %d s]: %s"),
					uTimeDiff / 86400,(uTimeDiff % 86400) / 3600,(uTimeDiff % 3600) / 60,uTimeDiff % 60,
					msg->safeTrailing());
		} else {
			msg->console()->activeWindow()->output(KVI_OUT_AWAY,__tr("[Leaving away status]: %s"),msg->safeTrailing());
		}
	}
}

void KviServerParser::parseNumericAway(KviIrcMessage * msg)
{
	// 306: RPL_NOWAWAY [I,E,U,D]
	// :prefix 305 <target> :You're away man
	msg->console()->changeAwayState(true);
	
	if(KVI_OPTION_BOOL(KviOption_boolChangeNickAway))
	{

		KviStr nick = msg->safeParam(0);
		msg->console()->connectionInfo()->szNickBeforeAway = nick;

		if(KVI_OPTION_BOOL(KviOption_boolAutoGeneratedAwayNick))
		{
			if(nick.len() >= 5)nick.insert(5,"AWAY");
			else nick.append("AWAY");
		}
		else nick = KVI_OPTION_STRING(KviOption_stringCustomAwayNick);
		KviStr command(KviStr::Format,"nick %s",nick.ptr());
		g_pUserParser->parseCommandBuffer(command.ptr(),msg->console());
	}

	if(TRIGGER_EVENT_1PARAM_RETVALUE(KviEvent_OnMeAway,msg->console(),msg->safeTrailing()))msg->setHaltOutput();

	if(!msg->haltOutput())
	{
		msg->console()->activeWindow()->output(KVI_OUT_AWAY,__tr("[Entering away status]: %s"),msg->safeTrailing());
	}
}

void KviServerParser::parseNumericWatch(KviIrcMessage *msg)
{
	// 600: RPL_LOGON
	// :prefix 600 <target> <nick> <user> <host> <logintime> :logged online
	// 601: RPL_LOGON
	// :prefix 601 <target> <nick> <user> <host> <logintime> :logged offline
	// 602: RPL_WATCHOFF
	// :prefix 602 <target> <nick> <user> <host> <logintime> :stopped watching
	// 604: PRL_NOWON
	// :prefix 604 <target> <nick> <user> <host> <logintime> :is online
	// 605: PRL_NOWOFF
	// :prefix 605 <target> <nick> <user> <host> 0 :is offline

	if(msg->console()->connectionInfo()->pNotifyListManager)
	{
		if(msg->console()->connectionInfo()->pNotifyListManager->handleWatchReply(msg))return;
	}
	// not handled...output it

// FIXME: #warning "OUTPUT IT! (In a suitable way) (And handle 602 , 603 , 606 and 607 gracefully)"
	if(!msg->haltOutput())
	{
		msg->console()->output(KVI_OUT_UNHANDLED,
			"[%s][%s] %s",msg->prefix(),msg->command(),msg->allParams());
	}
}

void KviServerParser::parseNumericStats(KviIrcMessage * msg)
{

	if(!msg->haltOutput())
	{
		if(msg->paramCount() > 2)
		{
			KviStr szParms;
			KviStr *p = msg->firstParam();
			for(p = msg->nextParam();p;p = msg->nextParam())
			{
				if(szParms.hasData())szParms.append(' ');
				szParms.append(*p);
			}
			msg->console()->outputNoFmt(KVI_OUT_STATS,szParms.ptr());
		} else {
			msg->console()->outputNoFmt(KVI_OUT_STATS,msg->safeTrailing());
		}
	}
}

void KviServerParser::parseNumericUserMode(KviIrcMessage *msg)
{
	// 321: RPL_UMODEIS [I,E,U,D]
	// :prefix 221 <target> <modeflags>
	parseUserMode(msg,msg->safeParam(1));

	if(!msg->haltOutput())
		msg->console()->activeWindow()->output(KVI_OUT_MODE,__tr("Your user mode is %s"),msg->safeParam(1));
}
