//
//   File : kvi_sp_literal.cpp
//   Creation date : Thu Aug 3 2000 01:29:12 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
#define __KVIRC__

#include "kvi_sparser.h"
#include "kvi_window.h"
#include "kvi_out.h"
#include "kvi_locale.h"
#include "kvi_ircsocket.h"
#include "kvi_options.h"
#include "kvi_ircmask.h"
#include "kvi_channel.h"
#include "kvi_topicw.h"
#include "kvi_frame.h"
#include "kvi_mirccntrl.h"
#include "kvi_query.h"
#include "kvi_userlistview.h"
#include "kvi_antispam.h"
#include "kvi_nickserv.h"
#include "kvi_uparser.h"
#include "kvi_event.h"
#include "kvi_parameterlist.h"
#include "kvi_ircuserdb.h"
#include "kvi_app.h"
#include "kvi_regusersdb.h"
#include "kvi_debug.h"

#include "kvi_settings.h"

#ifdef COMPILE_CRYPT_SUPPORT
	#include "kvi_crypt.h"
	#include "kvi_cryptcontroller.h"
#endif

//#include "kvi_regusersdb.h"
//#include "kvi_iconmanager.h"
#include <qdatetime.h>

// kvi_app.cpp
extern KviRegisteredUserDataBase * g_pRegisteredUserDataBase;

extern KviNickServDataBase * g_pNickServDataBase;

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// PING
//
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void KviServerParser::parseLiteralPing(KviIrcMessage *msg)
{
	// PING
	// <optional_prefix> PING :<argument>

	msg->console()->socket()->sendFmtData("PONG %s",msg->allParams());

	if(TRIGGER_EVENT_2PARAM_RETVALUE(KviEvent_OnPing,msg->console(),msg->safePrefix(),msg->allParams()))
			msg->setHaltOutput();

	if((!msg->haltOutput()) && KVI_OPTION_BOOL(KviOption_boolShowPingPong))
	{
		msg->console()->output(KVI_OUT_SERVERPING,
			__tr("Received ping from \r!s\r%s\r (PING %s) : replied pong"),msg->safePrefix(),msg->allParams());
	}
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// JOIN
//
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void KviServerParser::parseLiteralJoin(KviIrcMessage *msg)
{
	// JOIN
	// :<joiner_mask> JOIN :<channel>

	KviIrcMask joiner(msg->prefix());

	KviStr * channel = msg->trailingString();

	if(!channel)
	{
		// This is broken....
		UNRECOGNIZED_MESSAGE(msg,__tr("Missing channel parameter in JOIN message"));
		return;
	}

	// check for extended join syntax.
	// it is used in splits only (AFAIK)
	// nick!user@host JOIN :#channel\x07[o|v]
	const char * pExt = channel->ptr();
	char  chExtMode = 0;
	while(*pExt && (*pExt != 0x07))pExt++;
	if(*pExt)
	{
		++pExt;
		if(*pExt)
		{
			chExtMode = (*pExt);
			channel->cutRight(2); // assuming that we're at the end (we should be)
		} else {
			channel->cutRight(1); // senseless 0x07 in channel name (not allowed I guess...)
		}
	}

	// Now lookup the channel
	KviChannel * chan = msg->console()->findChannel(channel->ptr());

	bool bIsMe = IS_ME(msg,joiner.nick());

	if(!chan)
	{
		// This must be me...(or desync)
		if(bIsMe)
		{
			msg->console()->setUserInfoFromServer(joiner.username(),joiner.host());
			chan = msg->console()->findDeadChannel(channel->ptr());
			if(chan)chan->resetState(); // Rejoining
			else {
				chan = msg->console()->createChannel(channel->ptr()); // New channel
			}
		} else {
			// Someone is joining an inexsisting channel!!!
			UNRECOGNIZED_MESSAGE(msg,__tr("Someone is joining an inexisting channel...desync ?"));
			return;
		}
// FIXME: #warning "STATUS BAR TEXT"
		msg->console()->socket()->sendFmtData("MODE %s",channel->ptr());

		if(msg->console()->connectionInfo()->bServerSupportsModeIe)
		{
// FIXME: #warning "OPTION TO AVOID REQUESTING THE MODES"
			msg->console()->socket()->sendFmtData("MODE %s e",channel->ptr());
			msg->console()->socket()->sendFmtData("MODE %s I",channel->ptr());
		}
		// MODE %s b MUST BE THE LAST AUTOMATIC CHANNEL QUERY
		// so we get RPL_ENDOFBANLIST as the last reply
		// and we know that the channel is in sync
// FIXME: #warning "OPTION TO AVOID THE WHO REQUEST"
		msg->console()->socket()->sendFmtData("WHO %s",channel->ptr());
		msg->console()->socket()->sendFmtData("MODE %s b",channel->ptr());
// FIXME: #warning "IF VERBOSE SAY THAT WE'RE REQUESTING MODES & BAN LIST"

		int iFlags = 0;
		switch(chExtMode)
		{
			case 'o': iFlags = KVI_USERFLAG_OP;     break;
			case 'v': iFlags = KVI_USERFLAG_VOICE;  break;
			case 'h': iFlags = KVI_USERFLAG_HALFOP; break;
			// we ignore +a and +q actually
		}

		KviUserListEntry * it = chan->join(joiner.nick(),joiner.username(),joiner.host(),iFlags);
		if(iFlags)chan->updateCaption();

// FIXME: #warning "Trigger also OnMeVoice and OnMeOp here ?"
		if(!(it->globalData()->avatar()))
		{
			QPixmap * avatar = KVI_OPTION_PIXMAP(KviOption_pixmapMyAvatar).pixmap();
			if(avatar)
			{
				KviAvatar * av = new KviAvatar(new QPixmap(*avatar),KVI_OPTION_PIXMAP(KviOption_pixmapMyAvatar).path(),true);
				it->globalData()->setAvatar(av);
				msg->console()->avatarChanged(av,joiner.nick(),joiner.user(),joiner.host(),0);
			}
		}

		if(TRIGGER_EVENT_RETVALUE(KviEvent_OnMeJoin,chan))msg->setHaltOutput();

	} else {
		// This must be someone else...(or desync)
		int iFlags = 0;
		switch(chExtMode)
		{
			case 'o': iFlags = KVI_USERFLAG_OP;     break;
			case 'v': iFlags = KVI_USERFLAG_VOICE;  break;
			case 'h': iFlags = KVI_USERFLAG_HALFOP; break;
			// we ignore +a and +q actually
		}

		KviUserListEntry * it = chan->join(joiner.nick(),joiner.username(),joiner.host(),iFlags);

// FIXME: #warning "Trigger also OnVoice and OnOp here ?"
		// Note: checkDefaultAvatar() makes a KviRegisteredUser lookup
		//       if later it is needed, make it return a pointer
		if(!(it->globalData()->avatar()))msg->console()->checkDefaultAvatar(it->globalData(),
			joiner.nick(),joiner.user(),joiner.host());

		if(TRIGGER_EVENT_3PARAM_RETVALUE(KviEvent_OnJoin,chan,joiner.nick(),joiner.username(),joiner.host()))
			msg->setHaltOutput();

// FIXME: #warning "WE COULD OPTIONALLY REQUEST A /WHO FOR THE USERS JOINING THAT WE DON'T KNOW THE HOST OF"
	}

	// Now say it to the world
	if(!msg->haltOutput())
	{
// FIXME: #warning "CHECK IF MESSAGES GO TO CONSOLE OR NOT"

		if(chExtMode != 0)
		{
			chan->output(KVI_OUT_JOIN,
				__tr("\r!n\r%s\r [%s@\r!h\r%s\r] has joined \r!c\r%s\r [implicit +%c umode change]"),
				joiner.nick(),joiner.username(),joiner.host(),channel->ptr(),chExtMode);

		} else {
			chan->output(KVI_OUT_JOIN,
				__tr("\r!n\r%s\r [%s@\r!h\r%s\r] has joined \r!c\r%s\r"),
				joiner.nick(),joiner.username(),joiner.host(),channel->ptr());
		}
	}

	if(!bIsMe && KVI_OPTION_BOOL(KviOption_boolEnableQueryTracing))
	{
		KviStr szChans;
		int iChans = msg->console()->getCommonChannels(joiner.nick(),szChans);
		for(KviQuery *q=msg->console()->queryList()->first();q;q=msg->console()->queryList()->next())
		{
			if(q->haveTarget(joiner.nick()))
			{
				q->output(KVI_OUT_QUERYTRACE,
					__tr("\r!n\r%s\r [%s@\r!h\r%s\r] has just joined \r!c\r%s\r"),joiner.nick(),joiner.username(),
					joiner.host(),msg->safeParam(0));
				q->notifyCommonChannels(joiner.nick(),joiner.username(),joiner.host(),iChans,szChans);
			}
		}
	}
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// PART
//
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void KviServerParser::parseLiteralPart(KviIrcMessage *msg)
{
	// PART
	// :<source_mask> PART <channel> :<part message>

	KviIrcMask source(msg->prefix());
	// Now lookup the channel
	KviChannel * chan = msg->console()->findChannel(msg->safeParam(0));

	if(chan)
	{
		if(IS_ME(msg,source.nick()))
		{
			if(TRIGGER_EVENT_1PARAM_RETVALUE(KviEvent_OnMePart,chan,msg->safeTrailing()))
				msg->setHaltOutput();

			// It's me!
			chan->frame()->closeWindow(chan); // <-- deleted path

			if(!msg->haltOutput())
			{
				if(KVI_OPTION_BOOL(KviOption_boolShowOwnParts))
				{
					msg->console()->output(KVI_OUT_PART,
						__tr("You have left channel \r!c\r%s\r :%s"),msg->safeParam(0),msg->safeTrailing());
				}
			}

		} else {
			// Someone else

			if(TRIGGER_EVENT_4PARAM_RETVALUE(KviEvent_OnPart,chan,source.nick(),
					source.user(),source.host(),msg->safeTrailing()))
				msg->setHaltOutput();

			chan->part(source.nick());

			if(!msg->haltOutput())
			{
				chan->output(KVI_OUT_PART,
					__tr("\r!n\r%s\r [%s@\r!h\r%s\r] has left \r!c\r%s\r :%s"),source.nick(),source.username(),
					source.host(),msg->safeParam(0),msg->safeTrailing());
			}

			if(KVI_OPTION_BOOL(KviOption_boolEnableQueryTracing))
			{
				KviStr szChans;
				int iChans = msg->console()->getCommonChannels(source.nick(),szChans);
				for(KviQuery *q=msg->console()->queryList()->first();q;q=msg->console()->queryList()->next())
				{
					if(q->haveTarget(source.nick()))
					{
						q->output(KVI_OUT_QUERYTRACE,
							__tr("\r!n\r%s\r [%s@\r!h\r%s\r] has just left \r!c\r%s\r :%s"),source.nick(),source.username(),
							source.host(),msg->safeParam(0),msg->safeTrailing());
						q->notifyCommonChannels(source.nick(),source.username(),source.host(),iChans,szChans);
					}
				}
			}

		}
	} else {
		UNRECOGNIZED_MESSAGE(msg,__tr("Received a PART message for an unknown channel...desync ?"));
	}
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// QUIT
//
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void KviServerParser::parseLiteralQuit(KviIrcMessage *msg)
{
	// QUIT
	// :<source_mask> QUIT :<quit message>

	KviIrcMask source(msg->prefix());

	const char * aux = msg->safeTrailing();
	bool bWasSplit = false;
	//determine if signoff string matches "%.% %.%", and only one space (from eggdrop code)
	char *p = strchr(aux, ' ');
	if (p && (p == strrchr(aux,' '))){
		// one space detected. go ahead
		char *z1, *z2;
		*p = 0;
		z1 = strchr(p + 1, '.');
		z2 = strchr(aux, '.');
		if (z1 && z2 && (*(z1 + 1) != 0) && (z1 - 1 != p) && (z2 + 1 != p) && (z2 != aux))
		{
			// server split, or else it looked like it anyway
			*p=' ';
			bWasSplit = true;

			time_t curTime = time(0);
			int diff = ((unsigned int)curTime) - ((unsigned int)msg->console()->connectionInfo()->lastNetsplitTime);
			bool bDuplicate = false;
			if(diff < 6)
			{
				if(kvi_strEqualCI(msg->console()->connectionInfo()->szLastNetsplit.ptr(),aux))
				{
					bDuplicate = true;
				}
			}
			msg->console()->connectionInfo()->szLastNetsplit = aux;
			msg->console()->connectionInfo()->lastNetsplitTime = curTime;
			if(!bDuplicate && !msg->haltOutput())
			{
// FIXME: #warning "OnNetsplit ?"
				msg->console()->output(KVI_OUT_SPLIT,__tr("Netsplit detected :%s"),aux);
			}
		} else *p = ' ';
	}

// FIXME: #warning "Add a netsplit parameter ?"
	if(TRIGGER_EVENT_4PARAM_RETVALUE(KviEvent_OnQuit,msg->console(),source.nick(),
			source.user(),source.host(),msg->safeTrailing()))
		msg->setHaltOutput();

	for(KviChannel *c=msg->console()->channelList()->first();c;c=msg->console()->channelList()->next())
	{
		if(c->part(source.nick()))
		{
			if(!msg->haltOutput())c->output(bWasSplit ? KVI_OUT_QUITSPLIT : KVI_OUT_QUIT,
				__tr("\r!n\r%s\r [%s@\r!h\r%s\r] has quit irc %s%s"),source.nick(),source.username(),
				source.host(),bWasSplit ? ":NETSPLIT " : ":",msg->safeTrailing());
		}
	}

	for(KviQuery *q=msg->console()->queryList()->first();q;q=msg->console()->queryList()->next())
	{
		if(q->haveTarget(source.nick()))
		{
			if(!msg->haltOutput())
			{
				q->output(bWasSplit ? KVI_OUT_QUITSPLIT : KVI_OUT_QUIT,
					__tr("\r!n\r%s\r [%s@\r!h\r%s\r] has quit irc %s%s"),source.nick(),source.username(),
					source.host(),bWasSplit ? ":NETSPLIT " : ":",msg->safeTrailing());

				if(q->targetCount() > 1)
				{
					q->removeTarget(source.nick());
					q->output(KVI_OUT_SYSTEMWARNING,__tr("\r!n\r%s\r, a target of this query, has just quit IRC and " \
							"has been removed from the target list. " \
							"Use \r![!dbl]addtarget %s[!txt]Double click here to re-add %s to the target list" \
							"\r/addtarget %s\r to add him again"),
						source.nick(),source.nick(),source.nick(),source.nick());
				}
			}
		}
	}
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// KICK
//
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void KviServerParser::parseLiteralKick(KviIrcMessage *msg)
{
	// KICK
	// :<source_mask> KICK <channel> <nick> :<kick message>
	KviIrcMask source(msg->prefix());

//	KviStr nickBuffer;
//	KviStr hostBuffer;
//	formatNickLink(source.nick(),nickBuffer);
//	formatHostLink(source.host(),hostBuffer);

	KviStr szChan = msg->safeParam(0);
	KviStr szNick = msg->safeParam(1);

	KviChannel * chan = msg->console()->findChannel(szChan.ptr());

	if(!chan){
		// Ooops , desync with the server.
		UNRECOGNIZED_MESSAGE(msg,__tr("Kick message to an inexisting channel...desync ?"));
		return;
	}

	if(IS_ME(msg,szNick.ptr()))
	{
		// ops...I have been kicked

		if(TRIGGER_EVENT_4PARAM_RETVALUE(KviEvent_OnMeKick,chan,
				source.nick(),source.user(),source.host(),msg->safeTrailing()))
			msg->setHaltOutput();

		if(KVI_OPTION_BOOL(KviOption_boolKeepChannelOpenOnKick))
		{
			chan->userAction(&source,KVI_USERACTION_KICK);

			chan->part(szNick.ptr());
			chan->setDeadChan();

			if(!msg->haltOutput())
			{
// FIXME: #warning "OPTION FOR THIS TO GO TO THE CONSOLE!"
				chan->output(KVI_OUT_MEKICK,
					__tr("You have been kicked from \r!c\r%s\r by \r!n\r%s\r [%s@\r!h\r%s\r] :%s"),
					szChan.ptr(),source.nick(),source.username(),source.host(),msg->safeTrailing());
			}
		} else {
			chan->frame()->closeWindow(chan); // <-- deleted path

			if(!msg->haltOutput())
			{
// FIXME: #warning "This could go also to the active window!"
				msg->console()->output(KVI_OUT_MEKICK,
					__tr("You have been kicked from \r!c\r%s\r by \r!n\r%s\r [%s@\r!h\r%s\r] :%s"),
					szChan.ptr(),source.nick(),source.username(),source.host(),msg->safeTrailing());
			}
		}
		if(KVI_OPTION_BOOL(KviOption_boolRejoinChannelOnKick))
		{
			if(KVI_OPTION_BOOL(KviOption_boolBeVerbose))
			{
// FIXME: #warning "This should go where the 'You have been kicked...' message was printed"
				msg->console()->output(KVI_OUT_SYSTEMMESSAGE,__tr("Attempting to rejoin \r!c\r%s\r"),szChan.ptr());
			}
			msg->console()->socket()->sendFmtData("JOIN %s",szChan.ptr());
		}
	} else {
		// ok...someone else has been kicked

		if(TRIGGER_EVENT_5PARAM_RETVALUE(KviEvent_OnKick,chan,
				source.nick(),source.user(),source.host(),szNick.ptr(),msg->safeTrailing()))
			msg->setHaltOutput();

		KviIrcUserEntry * e = msg->console()->userDataBase()->find(szNick.ptr());

		KviStr szHost;
		KviStr szUser;

		if(e)
		{
			szHost = e->host();
			szUser = e->user();
		} else {
			szHost = "*";
			szUser = "*";
		}


		chan->userAction(&source,KVI_USERACTION_KICK);

		chan->part(szNick.ptr());



		if(!msg->haltOutput())
		{
// FIXME: #warning "OPTION FOR THIS TO GO TO THE CONSOLE!"
			chan->output(KVI_OUT_KICK,
				__tr("\r!n\r%s\r [%s@\r!h\r%s\r] has been kicked from \r!c\r%s\r by \r!n\r%s\r [%s@\r!h\r%s\r] :%s"),
				szNick.ptr(),szUser.ptr(),szHost.ptr(),szChan.ptr(),
				source.nick(),source.username(),source.host(),msg->safeTrailing());
		}

		if(KVI_OPTION_BOOL(KviOption_boolEnableQueryTracing))
		{
			KviStr szChans;
			int iChans = msg->console()->getCommonChannels(szNick.ptr(),szChans);
			for(KviQuery *q=msg->console()->queryList()->first();q;q=msg->console()->queryList()->next())
			{
				if(q->haveTarget(szNick.ptr()))
				{
					q->output(KVI_OUT_QUERYTRACE,
						__tr("\r!n\r%s\r [%s@\r!h\r%s\r] has just been kicked from \r!c\r%s\r by \r!n\r%s\r [%s@\r!h\r%s\r] :%s"),
						szNick.ptr(),szUser.ptr(),szHost.ptr(),szChan.ptr(),
						source.nick(),source.username(),source.host(),msg->safeTrailing());
					q->notifyCommonChannels(szNick.ptr(),szUser.ptr(),szHost.ptr(),iChans,szChans);
				}
			}
		}
	}
}

#ifdef COMPILE_CRYPT_SUPPORT

// FIXME: 	#warning "SIMPLIFY THIS STUFF"

	#define OUTPUT_PRIVMSGORNOTICE(_cons,_target,_type,_type2,_talker,_txt) \
		if(KviCryptSessionInfo * cinf = _target->cryptSessionInfo()) \
		{ \
			if(cinf->bDoDecrypt) \
			{ \
				if(cinf->pEngine->isCryptographicEngine() && (*_txt == KVI_TEXT_CRYPT)) \
				{ \
					KviStr decryptedStuff; \
					if(cinf->pEngine->decrypt(_txt + 1,decryptedStuff)) \
					{ \
						/*_cons->outputPrivmsg(_target,_type,_talker.nick(),_talker.username(),_talker.host(),_txt + 1);*/ \
						_cons->outputPrivmsg(_target,_type2,_talker.nick(),_talker.username(),_talker.host(),decryptedStuff.ptr()); \
					} else { \
						_target->output(KVI_OUT_SYSTEMERROR, \
							__tr("The following message looks like an encrypted one, but the crypting engine failed to decode it: %s"), \
							cinf->pEngine->lastError()); \
						_cons->outputPrivmsg(_target,_type,_talker.nick(),_talker.username(),_talker.host(),_txt + 1); \
					} \
				} else { \
					if(cinf->pEngine->isCryptographicEngine()) \
					{ \
						_cons->outputPrivmsg(_target,_type,_talker.nick(),_talker.username(),_talker.host(),_txt); \
					} else { \
						KviStr decryptedStuff; \
						if(cinf->pEngine->decrypt(_txt,decryptedStuff)) \
						{ \
							/*_cons->outputPrivmsg(_target,_type,_talker.nick(),_talker.username(),_talker.host(),_txt + 1);*/ \
							_cons->outputPrivmsg(_target,_type,_talker.nick(),_talker.username(),_talker.host(),decryptedStuff.ptr()); \
						} else { \
							_target->output(KVI_OUT_SYSTEMERROR, \
								__tr("The following message looks like an encrypted one, but the crypting engine failed to decode it: %s"), \
								cinf->pEngine->lastError()); \
							_cons->outputPrivmsg(_target,_type,_talker.nick(),_talker.username(),_talker.host(),_txt); \
						} \
					} \
				} \
			} else _cons->outputPrivmsg(_target,_type,_talker.nick(),_talker.username(),_talker.host(),_txt); \
		} else _cons->outputPrivmsg(_target,_type,_talker.nick(),_talker.username(),_talker.host(),_txt);

#else

	#define OUTPUT_PRIVMSGORNOTICE(_cons,_target,_type,_type2,_talker,_txt) \
		_cons->outputPrivmsg(_target,_type,_talker.nick(),_talker.username(),_talker.host(),_txt);

#endif


void KviServerParser::parseLiteralPrivmsg(KviIrcMessage *msg)
{
	// PRIVMSG
	// :source PRIVMSG <target> :<message>

	KviIrcMask talker(msg->safePrefix());
	// Ignore it?
	if(KVI_OPTION_BOOL(KviOption_boolEnableIgnoreOnPrivMsg))
	{
		KviRegisteredUser * u = g_pRegisteredUserDataBase->findMatchingUser(talker.nick(),talker.user(),talker.host());
		if(u)
		{
			if(u->getBoolProperty("ignore"))
			{
				if(KVI_OPTION_BOOL(KviOption_boolVerboseIgnore))
					msg->console()->output(KVI_OUT_IGNORE,__tr("Ignoring PRIVMSG from %s: %s"), msg->safePrefix(), msg->safeTrailing());
				return;
			}
		}
	}
// FIXME: 	#warning "Stats for PRIVMSG traffic on a channel"

// FIXME: 	#warning "Highlighting ?"
// FIXME: 	#warning "DEDICATED CTCP WINDOW ?"
	KviStr * pTrailing = msg->trailingString();
	if(pTrailing)
	{
		if(*(pTrailing->ptr()) == 0x01){
			if(pTrailing->len() > 1)
			{
				if(pTrailing->lastCharIs(0x01))pTrailing->cutRight(1);
				pTrailing->cutLeft(1);
				KviCtcpMessage ctcp;
				ctcp.msg = msg;
				ctcp.pData = pTrailing->ptr();
				ctcp.pSource = &talker;
				ctcp.pTarget = msg->safeParam(0);
				ctcp.bIgnored = false;
				ctcp.bIsFlood = false;
				ctcp.bUnknown = false;
				parseCtcpRequest(&ctcp);
				return;
			}
		}
	}

	// Normal PRIVMSG
	if(IS_ME(msg,msg->safeParam(0)))
	{

// FIXME: 	#warning "PROCESS MULTIMEDIA FILE REQUESTS"

	//		if(g_pOptions->m_bListenToMultimediaFileRequests)
	//		{
	//			if(*aux == '!')
	//			{
	//				const char *tmp = aux;
	//				tmp++;
	//				if(kvi_strEqualCI(tmp,m_pFrm->m_global.szCurrentNick.ptr()))
	//				{
	//					// A multimedia file request ?
	//					tmp += m_pFrm->m_global.szCurrentNick.len();
	//					if(((*tmp) == ' ') || ((*tmp) == '\t'))
	//					{
	//						while(((*tmp) == ' ') || ((*tmp) == '\t'))tmp++;
	//						if(*tmp)
	//						{
	//							KviStr file = tmp;
	//							KviStr filePath;
	//							m_pFrm->findMultimediaFileOffert(filePath,file);
	//							if(filePath.hasData())
	//							{
	//								m_pFrm->activeWindow()->output(KVI_OUT_INTERNAL,__tr("%s requests previously offered file %s: sending (%s)"),talker.nick(),file.ptr(),filePath.ptr());
	//								KviStr cmd(KviStr::Format,"DCC SEND %s %s",talker.nick(),filePath.ptr());
	//								m_pFrm->m_pUserParser->parseUserCommand(cmd,m_pConsole);
	//								return;

	//							} else {
	//								m_pFrm->activeWindow()->output(KVI_OUT_INTERNAL,__tr("%s requests file %s: no such file was offered , ignoring"),talker.nick(),file.ptr());
	//								return;
	//							}
	//						}
	//					}
	//				}
	//			}
	//		}



		KviQuery * query = msg->console()->findQuery(talker.nick());
		if(!query)
		{
			// New query requested ?
			if(TRIGGER_EVENT_4PARAM_RETVALUE(KviEvent_OnQueryMessage,msg->console()->activeWindow(),talker.nick(),talker.user(),talker.host(),msg->safeTrailing()))
			{
				msg->setHaltOutput();
				return;
			}

			if(KVI_OPTION_BOOL(KviOption_boolUseAntiSpamOnPrivmsg))
			{
				KviStr * theMsg = msg->trailingString();
				if(theMsg)
				{
					KviStr spamWord;
					if(kvi_mayBeSpam(theMsg,spamWord))
					{
						// FIXME: OnSpam ?

						if(!(msg->haltOutput() || KVI_OPTION_BOOL(KviOption_boolSilentAntiSpam)))
						{
							msg->console()->output(KVI_OUT_SPAM,__tr("Spam privmsg from \r!n\r%s\r [%s@\r!h\r%s\r]: %s (matching spamword \"%s\")"),
								talker.nick(),talker.user(),talker.host(),msg->safeTrailing(),spamWord.ptr());
						}
						return;
					}
				}
			}

			if(KVI_OPTION_BOOL(KviOption_boolCreateQueryOnPrivmsg))
			{
				query = msg->console()->createQuery(talker.nick());
				query->addTarget(talker.nick(),talker.username(),talker.host());
			} else {
				if(!msg->haltOutput())
				{
					msg->console()->activeWindow()->output(KVI_OUT_QUERYPRIVMSG,"[PRIVMSG \r!n\r%s\r]: %s",talker.nick(),msg->safeTrailing());
					return;
				}
			}
		} else {
			if(TRIGGER_EVENT_4PARAM_RETVALUE(KviEvent_OnQueryMessage,query,talker.nick(),talker.user(),talker.host(),msg->safeTrailing())) 
				msg->setHaltOutput();
		}

		query->userAction(&talker,KVI_USERACTION_PRIVMSG);

		
		if(!msg->haltOutput())
		{
			OUTPUT_PRIVMSGORNOTICE(msg->console(),query,KVI_OUT_QUERYPRIVMSG,KVI_OUT_QUERYPRIVMSGCRYPTED,talker,msg->safeTrailing())
		}
		return;
	}
	// Channel PRIVMSG
	KviChannel * chan = msg->console()->findChannel(msg->safeParam(0));

	if(!chan)
	{
		if(!msg->haltOutput())
		{
			KviStr broad(KviStr::Format,__tr("[>> %s] %s"),msg->safeParam(0),msg->safeTrailing());
			OUTPUT_PRIVMSGORNOTICE(msg->console(),msg->console()->activeWindow(),KVI_OUT_BROADCASTPRIVMSG,KVI_OUT_BROADCASTPRIVMSG,talker,broad.ptr())
			return;
		}
	}

	chan->userAction(&talker,KVI_USERACTION_PRIVMSG);

	if(TRIGGER_EVENT_4PARAM_RETVALUE(KviEvent_OnChannelMessage,chan,talker.nick(),talker.user(),talker.host(),msg->safeTrailing()))
		msg->setHaltOutput();

	if(!msg->haltOutput())
	{
		OUTPUT_PRIVMSGORNOTICE(msg->console(),chan,KVI_OUT_CHANPRIVMSG,KVI_OUT_CHANPRIVMSGCRYPTED,talker,msg->safeTrailing())
	}
}


void KviServerParser::parseLiteralNotice(KviIrcMessage *msg)
{
	// NOTICE
	// :source NOTICE <target> :<message>

	KviIrcMask talker(msg->safePrefix());

	if( *(talker.host()) == '*' )
	{
		if( *(talker.user()) == '*')
		{
			KviStr szServ = talker.nick();
			if(szServ.contains('.'))
			{
				// server notice
// FIXME: 		#warning "Dedicated window for server notices ?"
				if(TRIGGER_EVENT_2PARAM_RETVALUE(KviEvent_OnServerNotice, msg->console(), talker.nick(), msg->safeTrailing()))msg->setHaltOutput();
				if(!msg->haltOutput())msg->console()->output(KVI_OUT_SERVERNOTICE,__tr("[\r!s\r%s\r]: %s"),talker.nick(),msg->safeTrailing());
				return;
			}
		}
	}

	// Ignore it?
	if(KVI_OPTION_BOOL(KviOption_boolEnableIgnoreOnNotice))
	{
		KviRegisteredUser * u = g_pRegisteredUserDataBase->findMatchingUser(talker.nick(),talker.user(),talker.host());
		if(u)
		{
			if(u->getBoolProperty("ignore"))
			{
				if(KVI_OPTION_BOOL(KviOption_boolVerboseIgnore))
					msg->console()->output(KVI_OUT_IGNORE,__tr("Ignoring PRIVMSG from %s: %s"), msg->safePrefix(), msg->safeTrailing());
				return;
			}
		}
	}

// FIXME: 	#warning "Highlighting ?"
// FIXME: 	#warning "DEDICATED CTCP WINDOW ?"

	KviStr * pTrailing = msg->trailingString();
	if(pTrailing)
	{
		if(*(pTrailing->ptr()) == 0x01){
			if(pTrailing->len() > 1)
			{
				if(pTrailing->lastCharIs(0x01))pTrailing->cutRight(1);
				pTrailing->cutLeft(1);
				KviCtcpMessage ctcp;
				ctcp.msg = msg;
				ctcp.pData = pTrailing->ptr();
				ctcp.pSource = &talker;
				ctcp.pTarget = msg->safeParam(0);
				ctcp.bIgnored = false;
				ctcp.bIsFlood = false;
				ctcp.bUnknown = false;
				parseCtcpReply(&ctcp);
				return;
			}
		}
	}


	// Normal NOTICE
	if(IS_ME(msg,msg->safeParam(0)))
	{

// FIXME: 	#warning "The NickServ and ChanServ handling should be optional!"

		if(kvi_strEqualCI(talker.nick(),"NickServ"))
		{
			// nickname service... does it ask for identification ?
			msg->console()->output(KVI_OUT_NICKSERV,__tr("\r!n\r%s\r [%s@\r!h\r%s\r]: %s"),talker.nick(),talker.user(),talker.host(),msg->safeTrailing());

			if(KVI_OPTION_BOOL(KviOption_boolEnableAutomaticNickServIdentification))
			{
				KviStr szData = msg->safeTrailing();

				int msgIdx      = szData.findFirstIdx("/msg",false);
				if(msgIdx > 0)
				{
					szData.cutLeft(msgIdx);
					msgIdx = 0;
				} else {
					if(msgIdx == -1)
					{
						// irc.dynamix.com uses /NickServ
						msgIdx = szData.findFirstIdx("/NickServ",false);
						if(msgIdx > 0)
						{
							szData.cutLeft(msgIdx);
							msgIdx = 0;
						}
					}
				}
				int nickServIdx = szData.findFirstIdx("NickServ",false);
				int identifyIdx = szData.findFirstIdx("identify",false);
				int passwordIdx = szData.findFirstIdx("password",false);

				if((identifyIdx != -1) && (nickServIdx != -1) && (msgIdx != -1) && (passwordIdx != -1)
					&& (msgIdx < nickServIdx) && (nickServIdx < identifyIdx) && (identifyIdx < passwordIdx))
				{
					// yep..this is almost surely an identification request
					KviStr szIdentifyCommand = g_pNickServDataBase->findIdentifyCommand(msg->console()->currentNickName(),msg->console()->currentServerName(),&talker);
					if(szIdentifyCommand.hasData())
					{
						// execute it!
						msg->console()->output(KVI_OUT_SYSTEMMESSAGE,__tr("NickServ requests authentication: executing scheduled command"));
						if(!g_pUserParser->parseCommandBuffer(szIdentifyCommand.ptr(),msg->console()))
						{
							msg->console()->output(KVI_OUT_SYSTEMERROR,__tr("The scheduled NickServ identification command looks to be broken: please fix it in the options dialog"));
						}
					}
				}
			}

			return;
		}

		if(kvi_strEqualCI(talker.nick(),"ChanServ"))
		{
			msg->console()->output(KVI_OUT_CHANSERV,__tr("\r!n\r%s\r [%s@\r!h\r%s\r]: %s"),talker.nick(),talker.user(),talker.host(),msg->safeTrailing());
			return;
		}

// FIXME: 	#warning "PROCESS MULTIMEDIA FILE REQUESTS"

	//		if(g_pOptions->m_bListenToMultimediaFileRequests)
	//		{
	//			if(*aux == '!')
	//			{
	//				const char *tmp = aux;
	//				tmp++;
	//				if(kvi_strEqualCI(tmp,m_pFrm->m_global.szCurrentNick.ptr()))
	//				{
	//					// A multimedia file request ?
	//					tmp += m_pFrm->m_global.szCurrentNick.len();
	//					if(((*tmp) == ' ') || ((*tmp) == '\t'))
	//					{
	//						while(((*tmp) == ' ') || ((*tmp) == '\t'))tmp++;
	//						if(*tmp)
	//						{
	//							KviStr file = tmp;
	//							KviStr filePath;
	//							m_pFrm->findMultimediaFileOffert(filePath,file);
	//							if(filePath.hasData())
	//							{
	//								m_pFrm->activeWindow()->output(KVI_OUT_INTERNAL,__tr("%s requests previously offered file %s: sending (%s)"),talker.nick(),file.ptr(),filePath.ptr());
	//								KviStr cmd(KviStr::Format,"DCC SEND %s %s",talker.nick(),filePath.ptr());
	//								m_pFrm->m_pUserParser->parseUserCommand(cmd,m_pConsole);
	//								return;

	//							} else {
	//								m_pFrm->activeWindow()->output(KVI_OUT_INTERNAL,__tr("%s requests file %s: no such file was offered , ignoring"),talker.nick(),file.ptr());
	//								return;
	//							}
	//						}
	//					}
	//				}
	//			}
	//		}

		KviQuery * query = msg->console()->findQuery(talker.nick());
		if(!query)
		{
			// New query requested ?

			if(KVI_OPTION_BOOL(KviOption_boolUseAntiSpamOnNotice))
			{
				KviStr * theMsg = msg->trailingString();
				if(theMsg)
				{
					KviStr spamWord;
					if(kvi_mayBeSpam(theMsg,spamWord))
					{
						if(!(msg->haltOutput() || KVI_OPTION_BOOL(KviOption_boolSilentAntiSpam)))
						{
							msg->console()->output(KVI_OUT_SPAM,__tr("Spam notice from \r!n\r%s\r [%s@\r!h\r%s\r]: %s (matching spamword \"%s\")"),
								talker.nick(),talker.user(),talker.host(),msg->safeTrailing(),spamWord.ptr());
						}
						return;
					}
				}
			}

			if(KVI_OPTION_BOOL(KviOption_boolCreateQueryOnNotice))
			{
				query = msg->console()->createQuery(talker.nick());
				query->addTarget(talker.nick(),talker.username(),talker.host());
			} else {
				if(!msg->haltOutput())
				{
					msg->console()->activeWindow()->output(KVI_OUT_QUERYNOTICE,"[NOTICE \r!n\r%s\r]: %s",talker.nick(),msg->safeTrailing());
					return;
				}
			}
		}

		query->userAction(&talker,KVI_USERACTION_NOTICE);

		if(!msg->haltOutput())
		{
			OUTPUT_PRIVMSGORNOTICE(msg->console(),query,KVI_OUT_QUERYNOTICE,KVI_OUT_QUERYNOTICECRYPTED,talker,msg->safeTrailing())
		}
		return;
	}
	// Channel PRIVMSG
	KviChannel * chan = msg->console()->findChannel(msg->safeParam(0));

	if(!chan)
	{
		if(!msg->haltOutput())
		{
			KviStr broad(KviStr::Format,__tr("[>> %s] %s"),msg->safeParam(0),msg->safeTrailing());
			OUTPUT_PRIVMSGORNOTICE(msg->console(),msg->console()->activeWindow(),KVI_OUT_BROADCASTNOTICE,KVI_OUT_BROADCASTNOTICE,talker,broad.ptr())
			return;
		}
	}else 
	{
		if(TRIGGER_EVENT_2PARAM_RETVALUE(KviEvent_OnChannelNotice,chan,talker.nick(),msg->safeTrailing()))msg->setHaltOutput();
		chan->userAction(&talker,KVI_USERACTION_NOTICE);

		if(!msg->haltOutput())
		{
			OUTPUT_PRIVMSGORNOTICE(msg->console(),chan,KVI_OUT_CHANNELNOTICE,KVI_OUT_CHANNELNOTICECRYPTED,talker,msg->safeTrailing())
		}
	}
}


///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// TOPIC
//
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void KviServerParser::parseLiteralTopic(KviIrcMessage *msg)
{
	// TOPIC
	// :<source_mask> TOPIC <channel> :<topic>
	
	KviIrcMask source(msg->prefix());
	// Now lookup the channel
	KviChannel * chan = msg->console()->findChannel(msg->safeParam(0));

	if(chan)
	{
		if(TRIGGER_EVENT_4PARAM_RETVALUE(KviEvent_OnTopic,chan,
			source.nick(),source.user(),source.host(),msg->safeTrailing()))
				msg->setHaltOutput();

		chan->topicWidget()->setTopic(msg->safeTrailing());
		chan->topicWidget()->setTopicSetBy(source.nick());
		KviStr tmp = QDateTime::currentDateTime().toString();
		chan->topicWidget()->setTopicSetAt(tmp.ptr());

		chan->userAction(&source,KVI_USERACTION_TOPIC);

		if(!msg->haltOutput())
		{
			chan->output(KVI_OUT_TOPIC,
				__tr("\r!n\r%s\r [%s@\r!h\r%s\r] has changed topic to \"%s%c\""),
				source.nick(),source.username(),source.host(),
				msg->safeTrailing(),KVI_TEXT_RESET);
		}
	} else {
		UNRECOGNIZED_MESSAGE(msg,__tr("Received a TOPIC message for an unknown channel...desync ?"));
	}
}

void KviServerParser::parseLiteralNick(KviIrcMessage *msg)
{
	// NICK
	// :source NICK <newnick>
	KviIrcMask source(msg->safePrefix());

	bool bIsMe = IS_ME(msg,source.nick());

	if(bIsMe)
	{
		// We have changed our nick
		msg->console()->nickChange(msg->safeTrailing());

		if(TRIGGER_EVENT_2PARAM_RETVALUE(KviEvent_OnMeNickChange,msg->console(),source.nick(),msg->safeTrailing()))
				msg->setHaltOutput();
	} else {
		if(TRIGGER_EVENT_4PARAM_RETVALUE(KviEvent_OnNickChange,msg->console(),source.nick(),
				source.user(),source.host(),msg->safeTrailing()))
				msg->setHaltOutput();
	}

	for(KviChannel * c = msg->console()->channelList()->first();c;c = msg->console()->channelList()->next())
	{
		if(c->nickChange(source.nick(),msg->safeTrailing()))
		{
			c->output(KVI_OUT_NICK,__tr("\r!n\r%s\r [%s@\r!h\r%s\r] is now known as \r!n\r%s\r"),
				source.nick(),source.username(),source.host(),msg->safeTrailing());
		}
		if(bIsMe)c->updateCaption();
	}

// FIXME: #warning "NEW NICK MIGHT BE REGISTERED AND HAVE AN AVATAR!"

	for(KviQuery * q = msg->console()->queryList()->first();q;q = msg->console()->queryList()->next())
	{
		// some target may have changed the nick
		if(q->nickChange(source.nick(),msg->safeTrailing()))
		{
			q->output(KVI_OUT_NICK,__tr("\r!n\r%s\r [%s@\r!h\r%s\r] is now known as \r!n\r%s\r"),
				source.nick(),source.username(),source.host(),msg->safeTrailing());
		}
		q->updateCaption();
	}

// FIXME: #warning "UPDATE ALL THE OTHER CONNECTION RELATED WINDOW CAPTIONS WHEN bIsMe!!"
}

void KviServerParser::parseLiteralInvite(KviIrcMessage *msg)
{
	// INVITE
	// :source INVITE <target> <channel>

	KviIrcMask source(msg->safePrefix());

	KviStr target(msg->safeParam(0));
	KviStr channel(msg->safeParam(1));

	if(IS_ME(msg,target.ptr()))
	{
		KviWindow * pOut = KVI_OPTION_BOOL(KviOption_boolInviteToConsole) ? (KviWindow *)(msg->console()) : (KviWindow *)(msg->console()->activeWindow());
		if(TRIGGER_EVENT_4PARAM_RETVALUE(KviEvent_OnInvite,pOut,source.nick(),source.username(),source.host(),channel.ptr()))
			msg->setHaltOutput();
		
		if(!msg->haltOutput())
		{
			pOut->output(KVI_OUT_INVITE,__tr("\r!n\r%s\r [%s@\r!h\r%s\r] invites you to channel \r!c\r%s\r (%s)"),
				source.nick(),source.user(),source.host(),
				channel.ptr(),KVI_OPTION_BOOL(KviOption_boolAutoJoinOnInvite) ? __tr("autojoining") : __tr("double click the channel name to join"));
		}

		if(KVI_OPTION_BOOL(KviOption_boolAutoJoinOnInvite))
		{
			msg->console()->socket()->sendFmtData("JOIN %s",channel.ptr());
		}

	} else {
		UNRECOGNIZED_MESSAGE(msg,__tr("Received an INVITE message directed to someone else...mmmh"));
	}
}

void KviServerParser::parseLiteralWallops(KviIrcMessage *msg)
{
	// WALLOPS
	// :source WALLOPS :msg

	KviIrcMask source(msg->safePrefix());

	KviWindow * pOut = KVI_OPTION_BOOL(KviOption_boolWallopsToConsole) ? (KviWindow *)(msg->console()) : (KviWindow *)(msg->console()->activeWindow());
//#warning " FIXME: Event for WALLOPS"
//	if(TRIGGER_EVENT_4PARAM_RETVALUE(KviEvent_OnInvite,pOut,source.nick(),source.username(),source.host(),channel.ptr()))
//			msg->setHaltOutput();

	if(!msg->haltOutput())
	{
		pOut->output(KVI_OUT_WALLOPS,__tr("WALLOPS from \r!n\r%s\r [%s@\r!h\r%s\r]: %s"),
			source.nick(),source.username(),source.host(),msg->safeTrailing());
	}
}

void KviServerParser::parseUserMode(KviIrcMessage *msg,const char * modeflptr)
{
	// changed my user mode
	bool bSet = true;
	while(*modeflptr)
	{
		switch(*modeflptr)
		{
			case '+': bSet = true; break;
			case '-': bSet = false; break;
			default:
				if(msg->console()->setUserMode(*modeflptr,bSet))
				{
					// There was a mode change
					if(g_pEventManager->hasEventHandlers(KviEvent_OnUserMode))
					{
						KviStr * modefl = new KviStr(KviStr::Format,"%c%c",bSet ? '+' : '-',*modeflptr);
						if(g_pUserParser->triggerEvent(KviEvent_OnUserMode,msg->console(),new KviParameterList(modefl)))
							msg->setHaltOutput();
					}
				}
			break;
		}
		++modeflptr;
	}
}



void KviServerParser::parseLiteralMode(KviIrcMessage *msg)
{
	// NICK
	// :source MODE target <params>
	//    :source MODE <me> +|-modeflag
	//    :source MODE <channel> +-modeflags [parameters]

	KviIrcMask source(msg->safePrefix());

//	if(!source.hasHost())
//	{
//		// This is a server or a channel service
//		KviStr snick = source.nick();
//		if(snick.contains('.'))source.setHost(source.nick()); // this is a server
//	}

	KviStr     target(msg->safeParam(0));
	KviStr     modefl(msg->safeParam(1));

	if(IS_ME(msg,target.ptr()))
	{
		parseUserMode(msg,modefl.ptr());
		if(!msg->haltOutput())
			msg->console()->activeWindow()->output(KVI_OUT_MODE,__tr("You have set user mode %s"),modefl.ptr());
	} else {
		// a channel mode
		KviChannel * chan = msg->console()->findChannel(target.ptr());

		if(!chan){
			// Ooops , desync with the server.
			UNRECOGNIZED_MESSAGE(msg,__tr("Mode change for an unknown channel ?"));
			return;
		}

		chan->userAction(&source,KVI_USERACTION_CHANMODE);

		parseChannelMode(source,chan,modefl,msg,2);
	}
}

void KviServerParser::parseChannelMode(KviIrcMask &source,KviChannel * chan,KviStr &modefl,KviIrcMessage *msg,int curParam)
{
	bool bSet    = true;
	const char * aux = modefl.ptr();
	KviStr aParam;

	KviStr nickBuffer;
	KviStr hostBuffer;
	if(source.hasHost())
	{
		formatNickLink(source.nick(),nickBuffer);
		formatHostLink(source.host(),hostBuffer);
	} else {
		nickBuffer = source.nick();
		if(nickBuffer.contains('.'))
		{
			// This looks a lot like a server!
			formatServerLink(source.nick(),nickBuffer);
		} else {
			// Probably a service....whois should work
			formatNickLink(source.nick(),nickBuffer);
		}
		hostBuffer = source.host();
	}
	KviStr auxBuffer;
	KviIrcMask * auxMask;

	int curParamSave = curParam;
	bool bIsMe;


	while(*aux)
	{
		switch(*aux)
		{
			case '+':
				bSet = true;
			break;
			case '-':
				bSet = false;
			break;

			case 'o':
				aParam = msg->safeParam(curParam++);
				chan->op(aParam.ptr(),bSet);
				bIsMe = IS_ME(msg,aParam.ptr());
				if(bIsMe)
				{
					if(TRIGGER_EVENT_3PARAM_RETVALUE(bSet ? KviEvent_OnMeOp : KviEvent_OnMeDeOp,chan,source.nick(),source.username(),source.host()))msg->setHaltOutput();
					chan->updateCaption();
				} else {
					if(TRIGGER_EVENT_4PARAM_RETVALUE(bSet ? KviEvent_OnOp : KviEvent_OnDeOp,chan,source.nick(),source.username(),source.host(),aParam.ptr()))msg->setHaltOutput();
				}
				if(!(msg->haltOutput() || KVI_OPTION_BOOL(KviOption_boolShowCompactModeChanges)))
				{
					chan->output(bSet ? (bIsMe ? KVI_OUT_MEOP : KVI_OUT_OP) : (bIsMe ? KVI_OUT_MEDEOP : KVI_OUT_DEOP),
						__tr("%s [%s@%s] has set mode %co %s"),
						nickBuffer.ptr(),source.username(),hostBuffer.ptr(),
						bSet ? '+' : '-',formatNickLink(aParam.ptr(),auxBuffer));
				}
			break;
			case 'h':
				aParam = msg->safeParam(curParam++);
				chan->halfop(aParam.ptr(),bSet);
				bIsMe = IS_ME(msg,aParam.ptr());
				if(bIsMe)
				{
					if(TRIGGER_EVENT_3PARAM_RETVALUE(bSet ? KviEvent_OnMeHalfOp : KviEvent_OnMeDeHalfOp,chan,source.nick(),source.username(),source.host()))msg->setHaltOutput();
					chan->updateCaption();
				} else {
					if(TRIGGER_EVENT_4PARAM_RETVALUE(bSet ? KviEvent_OnHalfOp : KviEvent_OnDeHalfOp,chan,source.nick(),source.username(),source.host(),aParam.ptr()))msg->setHaltOutput();
				}
				if(!(msg->haltOutput() || KVI_OPTION_BOOL(KviOption_boolShowCompactModeChanges)))
				{
					chan->output(bSet ? (bIsMe ? KVI_OUT_MEHALFOP : KVI_OUT_HALFOP) : (bIsMe ? KVI_OUT_MEDEHALFOP : KVI_OUT_HALFDEOP),
						__tr("%s [%s@%s] has set mode %ch %s"),
						nickBuffer.ptr(),source.username(),hostBuffer.ptr(),
						bSet ? '+' : '-',formatNickLink(aParam.ptr(),auxBuffer));
				}
			break;
			case 'v':
				aParam = msg->safeParam(curParam++);
				chan->voice(aParam.ptr(),bSet);
				bIsMe = IS_ME(msg,aParam.ptr());
				if(bIsMe)
				{
					if(TRIGGER_EVENT_3PARAM_RETVALUE(bSet ? KviEvent_OnMeVoice : KviEvent_OnMeDeVoice,chan,source.nick(),source.username(),source.host()))msg->setHaltOutput();
					chan->updateCaption();
				} else {
					if(TRIGGER_EVENT_4PARAM_RETVALUE(bSet ? KviEvent_OnVoice : KviEvent_OnDeVoice,chan,source.nick(),source.username(),source.host(),aParam.ptr()))msg->setHaltOutput();
				}
				if(!(msg->haltOutput() || KVI_OPTION_BOOL(KviOption_boolShowCompactModeChanges)))
				{
					chan->output(bSet ? (bIsMe ? KVI_OUT_MEVOICE : KVI_OUT_VOICE) : (bIsMe ? KVI_OUT_MEDEVOICE : KVI_OUT_DEVOICE),
						__tr("%s [%s@%s] has set mode %cv %s"),
						nickBuffer.ptr(),source.username(),hostBuffer.ptr(),
						bSet ? '+' : '-',formatNickLink(aParam.ptr(),auxBuffer));
				}
			break;
			case 'b':
				aParam = msg->safeParam(curParam++);
				chan->banMask(aParam.ptr(),bSet,msg->prefix(),QDateTime::currentDateTime().toString().ascii());
				auxMask = new KviIrcMask(aParam.ptr());
				bIsMe = auxMask->matchesFixed(msg->console()->currentNickName(),msg->console()->currentUserName(),msg->console()->currentLocalHostName());
				delete auxMask;

				if(bIsMe)
				{
					if(TRIGGER_EVENT_3PARAM_RETVALUE(bSet ? KviEvent_OnMeBan : KviEvent_OnMeUnban,chan,source.nick(),source.username(),source.host()))msg->setHaltOutput();
				} else {
					if(TRIGGER_EVENT_4PARAM_RETVALUE(bSet ? KviEvent_OnBan : KviEvent_OnUnban,chan,source.nick(),source.username(),source.host(),aParam.ptr()))msg->setHaltOutput();
				}

				if(!(msg->haltOutput() || KVI_OPTION_BOOL(KviOption_boolShowCompactModeChanges)))
				{
					chan->output(bSet ? (bIsMe ? KVI_OUT_MEBAN : KVI_OUT_BAN) : (bIsMe ? KVI_OUT_MEUNBAN : KVI_OUT_UNBAN),
						__tr("%s [%s@%s] has set mode %cb \r!m%cb\r%s\r"),
						nickBuffer.ptr(),source.username(),hostBuffer.ptr(),
						bSet ? '+' : '-',bSet ? '-' : '+',aParam.ptr());
				}
			break;
			case 'I':
				aParam = msg->safeParam(curParam++);
				chan->inviteExceptionMask(aParam.ptr(),bSet,msg->prefix(),QDateTime::currentDateTime().toString().ascii());
				auxMask = new KviIrcMask(aParam.ptr());
				bIsMe = auxMask->matchesFixed(msg->console()->currentNickName(),msg->console()->currentUserName(),msg->console()->currentLocalHostName());
				delete auxMask;

				if(bIsMe)
				{
					if(TRIGGER_EVENT_3PARAM_RETVALUE(bSet ? KviEvent_OnMeInviteException : KviEvent_OnMeInviteExceptionRemove,chan,source.nick(),source.username(),source.host()))msg->setHaltOutput();
				} else {
					if(TRIGGER_EVENT_4PARAM_RETVALUE(bSet ? KviEvent_OnInviteException : KviEvent_OnInviteExceptionRemove,chan,source.nick(),source.username(),source.host(),aParam.ptr()))msg->setHaltOutput();
				}

				if(!(msg->haltOutput() || KVI_OPTION_BOOL(KviOption_boolShowCompactModeChanges)))
				{
					chan->output(bSet ? (bIsMe ? KVI_OUT_MEINVITEEXCEPT : KVI_OUT_INVITEEXCEPT) : (bIsMe ? KVI_OUT_MEINVITEUNEXCEPT : KVI_OUT_INVITEUNEXCEPT),
						__tr("%s [%s@%s] has set mode %cI \r!m%cI\r%s\r"),
						nickBuffer.ptr(),source.username(),hostBuffer.ptr(),
						bSet ? '+' : '-',bSet ? '-' : '+',aParam.ptr());
				}
			break;
			case 'e':
				aParam = msg->safeParam(curParam++);
				chan->banExceptionMask(aParam.ptr(),bSet,msg->prefix(),QDateTime::currentDateTime().toString().ascii());
				auxMask = new KviIrcMask(aParam.ptr());
				bIsMe = auxMask->matchesFixed(msg->console()->currentNickName(),msg->console()->currentUserName(),msg->console()->currentLocalHostName());
				delete auxMask;

				if(bIsMe)
				{
					if(TRIGGER_EVENT_3PARAM_RETVALUE(bSet ? KviEvent_OnMeBanException : KviEvent_OnMeBanExceptionRemove,chan,source.nick(),source.username(),source.host()))msg->setHaltOutput();
				} else {
					if(TRIGGER_EVENT_4PARAM_RETVALUE(bSet ? KviEvent_OnBanException : KviEvent_OnBanExceptionRemove,chan,source.nick(),source.username(),source.host(),aParam.ptr()))msg->setHaltOutput();
				}

				if(!(msg->haltOutput() || KVI_OPTION_BOOL(KviOption_boolShowCompactModeChanges)))
				{
					chan->output(bSet ? (bIsMe ? KVI_OUT_MEBANEXCEPT : KVI_OUT_BANEXCEPT) : (bIsMe ? KVI_OUT_MEBANUNEXCEPT : KVI_OUT_BANUNEXCEPT),
						__tr("%s [%s@%s] has set mode %ce \r!m%ce\r%s\r"),
						nickBuffer.ptr(),source.username(),hostBuffer.ptr(),
						bSet ? '+' : '-',bSet ? '-' : '+',aParam.ptr());
				}
			break;

			case 'k':
				if(bSet)aParam = msg->safeParam(curParam++);
				else aParam = "";
				chan->setChannelKey(aParam.ptr());

                if(bSet) {
                    if(TRIGGER_EVENT_4PARAM_RETVALUE(KviEvent_OnKeySet,chan,source.nick(),source.username(),source.host(),aParam.ptr()))
                        msg->setHaltOutput();
                } else {
                    if(TRIGGER_EVENT_3PARAM_RETVALUE(KviEvent_OnKeyUnset,chan,source.nick(),source.username(),source.host()))
                        msg->setHaltOutput();
                }
				
				if(!(msg->haltOutput() || KVI_OPTION_BOOL(KviOption_boolShowCompactModeChanges)))
				{
					if(bSet)chan->output(KVI_OUT_KEY,
						__tr("%s [%s@%s] has set channel key to \"\r!m-k\r%s\r\""),
						nickBuffer.ptr(),source.username(),hostBuffer.ptr(),aParam.ptr());
					else chan->output(KVI_OUT_KEY,
						__tr("%s [%s@%s] has unset the channel key"),
						nickBuffer.ptr(),source.username(),hostBuffer.ptr());
				}
			break;
			case 'l':
				if(bSet)aParam = msg->safeParam(curParam++);
				else aParam = "";
				chan->setChannelLimit(aParam.ptr());
				
				if(bSet) {
					if(TRIGGER_EVENT_4PARAM_RETVALUE(KviEvent_OnLimitSet,chan,source.nick(),source.username(),source.host(),aParam.ptr()))
						msg->setHaltOutput();
				} else {
					if(TRIGGER_EVENT_3PARAM_RETVALUE(KviEvent_OnLimitUnset,chan,source.nick(),source.username(),source.host()))
						msg->setHaltOutput();
				}
				
				if(!(msg->haltOutput() || KVI_OPTION_BOOL(KviOption_boolShowCompactModeChanges)))
				{
					if(bSet)chan->output(KVI_OUT_LIMIT,
						__tr("%s [%s@%s] has set channel \r!m-l\rlimit to %s\r"),
						nickBuffer.ptr(),source.username(),hostBuffer.ptr(),aParam.ptr());
					else chan->output(KVI_OUT_LIMIT,
						__tr("%s [%s@%s] has unset the channel limit"),
						nickBuffer.ptr(),source.username(),hostBuffer.ptr());
				}
			break;

			default:
				chan->setChannelMode(*aux,bSet);
				if(!(msg->haltOutput() || KVI_OPTION_BOOL(KviOption_boolShowCompactModeChanges)))
				{
					chan->output(KVI_OUT_CHANMODE,
						__tr("%s [%s@%s] has set channel \r!m%c%c\rmode %c%c\r"),
						nickBuffer.ptr(),source.username(),hostBuffer.ptr(),
						bSet ? '-' : '+',*aux,bSet ? '+' : '-',*aux);
				}
			break;
		}
		++aux;
	}

	if(KVI_OPTION_BOOL(KviOption_boolShowCompactModeChanges) && (!msg->haltOutput()) &&
			(!kvi_strEqualCS(modefl.ptr(),"+")))
	{
		KviStr param;
		KviStr params;
		param = msg->safeParam(curParamSave++);
		while(param.hasData())
		{
			if(params.hasData())params.append(' ');
			params.append(param);
			param = msg->safeParam(curParamSave++);
		}

		if(params.hasData())
		{
			chan->output(KVI_OUT_CHANMODE,__tr("%s [%s@%s] has set mode %s %s"),
				nickBuffer.ptr(),source.username(),hostBuffer.ptr(),modefl.ptr(),
				params.ptr());
		} else {
			chan->output(KVI_OUT_CHANMODE,__tr("%s [%s@%s] has set channel mode %s"),
				nickBuffer.ptr(),source.username(),hostBuffer.ptr(),modefl.ptr());
		}
	}
}
