//
//   File : kvi_sp_ctcp.cpp
//   Creation date : Thu Aug 16 2000 13:34:42 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
#define __KVIRC__
// FIXME: #warning "CTCP BEEP == WAKEUP == AWAKE"
// FIXME: #warning "CTCP AVATARREQ or QUERYAVATAR"

#include "kvi_app.h"
#include "kvi_sparser.h"
#include "kvi_window.h"
#include "kvi_out.h"
#include "kvi_locale.h"
#include "kvi_ircsocket.h"
#include "kvi_options.h"
#include "kvi_channel.h"
#include "kvi_defaults.h"
//#include "kvi_ircmask.h"
#include "kvi_channel.h"
#include "kvi_query.h"
#include "kvi_ircuserdb.h"
#include "kvi_iconmanager.h"
#include "kvi_modulemanager.h"
#include "kvi_filetrader.h"
#include "kvi_time.h"
#include "kvi_fileutils.h"
#include "kvi_event.h"
#include "kvi_ctcppagedialog.h"

#include <stdlib.h>

#include <qdatetime.h>

#include <qpixmap.h>
//#include <iostream.h> //XXX

extern KVIRC_API KviFileTrader * g_pFileTrader;
extern KVIRC_API KviCtcpPageDialog * g_pCtcpPageDialog;

/*
	@doc: ctcp_handling
	@title:
		KVIrc & the CTCP protocol
	@short:
		For developers: Client to Client protocol handling in KVIrc
	@body:
		[big]Introduction[/big][br]
		Personally, I think that the CTCP specification is to
		be symbolically printed & burned. It is really too complex
		(you can go mad with the quoting specifications)
		and NO IRC CLIENT supports it completly.
		Here is my personal point of view on the CTCP protocol.[br]
		[big]What is CTCP ?[/big][br]
		CTCP stands for Client-To-Client-Protocol. It is designed
		for exchanging almost arbitrary data between IRC clients;
		the data is embedded into text messages of the underlying
		IRC protocol.[br]
		[big]Basic concepts[/big][br]
		A CTCP message is sent as the <text> part of the PRIVMSG and
		NOTICE IRC commands.[br]
		To differentiate the CTCP message from a normal IRC message
		text we use a delimiter character (Ascii char 1); we will
		use the symbol <0x01> for this delimiter.
		You may receive a CTCP message from server in one of the
		following two ways:[br]
		[b]:<source_mask> PRIVMSG <target> :<0x01><ctcp message><0x01>[/b][br]
		[b]:<source_mask> NOTICE <target>:<0x01><ctcp message><0x01>[/b][br]
		The PRIVMSG is used for CTCP REQUESTS, the NOTICE for CTCP REPLIES.
		The NOTICE form should never generate an automatic reply.[br]
		The two delimiters were used to begin and terminate the
		CTCP message; The origial protocol allowed more than one CTCP
		message inside a single IRC message. [b]Nobody sends more than
		one message at once, no client can recognize it (since it
		complicates the message parsing), it could be even dangerous (see below)[/b].
		It has no real sense unless we want to use the CTCP protocol to embed escape sequences
		into IRC messages: this is not the case.[br]
		Furthermore, sending more ctcp messages in a single IRC message could
		be easily used to flood a client: assume 450 characters available for the irc message
		text part: you could include 50 ctcp messages containing "<0x01>VERSION<0x01>".[br]
		Since the VERSION replies are usually long, there can be 3 or 4 replies per irc message:
		so a client that has no ctcp flood protection (or has it disabled) would 
		be surely disconnected while sending the replies: and all this after
		receiving a single irc message (no flood for the sender).
		From my personal point of view, only [b]one ctcp message per IRC message[/b]
		is allowed and theoretically the trailing <0x01> delimiter could be optional.[br]
		[big]How to extract the CTCP message[/big][br]
		The IRC messages do not allow the following characters to be sent:[br]
		<NUL> (Ascii character 0), <CR> (Carriage return), <LF> (Line feed).[br]
		So finally we have four characters that [b]cannot appear literally into a 
		CTCP message[/b]: <NUL>,<CR>,<LF>,<0x01>.[br]
		To extract a <ctcp_message> from an IRC PRIVMSG or NOTICE command you
		have to perform the following actions:[br]
		Find the <trailing> part of the IRC message (the one just after the ':'
		delimiter, or the last message token).[br]
		Check if the first character of the <trailing> is a <0x01>, if it is
		we have a <ctcp_message> beginning just after <0x01>.
		The trailing (optional) <0x01> can be removed in this phase
		or later, assuming that it is a non valid char in <ctcp messages>.[br]
		In this document I will assume that you have stripped the trailing <0x01>
		and thus from now on we will deal only with the <ctcp message> part.[br]

		[big]Parsing a ctcp message: the quoting dilemma[/big][br]
		Since there are characters that can not appear into a <ctcp message>,
		theoretically we should have to use a quoting mechanism.
		Well, in fact, no actual CTCP message uses the quoting: there
		is no need to include a <NUL> , a <CR> or <LF> inside the actually
		defined messages (The only one could be CTCP SED, but I have never
		seen it in action...is there any client that implements it ?).
		We could also leave the "quoting" to the "single message type semantic":
		a message that needs to include "any character" could have its own
		encoding method (Base64 for example). With the "one ctcp per irc message"
		convention we could even allow <0x01> inside messages: only the leading
		(and eventually trailing) <0x01> would be the delimiter: the other ones
		would be valid characters. Finally, is there any CTCP type that needs
		<0x01> inside a message ? <0x01> is not printable (as well as <CR>,<LF> and <NUL>),
		so only encoded messages (and again we can stick to the single message semantic)
		messages or the ones including special parameters. Some machines might
		allow <0x01> in filenames....well, a file with <0x01> in its name has something
		broken inside, or the creator is a sort of "hacker" (so he also
		knows how to rename a file...) :).[br]
		Anyway, let's be pedantic, and define this quoting method
		Let's use the most intuitive and adopted all around the world:[br]
		the backslash character ('\') as escape.[br]
		An escape sequence is formed by the backslash character and a number
		of following ascii characters. We define the following two types of escape sequences:[br]
		[b]'\XXX'[/b] (where XXX is an [b]octal number[/b] formed by three digits)
		that indicates the ascii character with code that corresponds to the number.[br]
		[b]'\C'[/b] (where C is a [b]ctcp valid ascii non digit character[/b]) that corresponds
		literally to the character C discarding any other semantic that might be associated
		with it (This will become clear later).
		I've choosen the octal rappresentation just to follow a bit the old specification:
		the authors seemed to like it. This point could be discussed in
		some mailing list or sth. The '\C' sequence is useful to include the backslash
		character (escape sequence '\\').[br]

		[big]Let's mess a little more[/big][br]
		A CTCP message is made of [b]space separated parameters[/b].[br]
		The natural way of separating parameters is to use the space character.
		We define a "token" as a sequence of valid ctcp characters not including literal space.
		A <ctcp parameter> is usally a token, but not always:
		filenames can contain spaces inside names (and it happens very often!).
		So one of the parameters of CTCP DCC is not a "space separated" token.
		How do we handle it ? Again a standard is missing. Some clients simply change
		the filename placing underscores instead of spaces; this is a reasonable solution if used with care.
		Other clients attempt to "isolate" the filename token by including it some kind
		of quotes: usually '"' or ''' ASCII characters: this is also a good solution.
		Another one that naturally comes into my mind is to use the previously defined
		quoting to define a "non-breaking space" character: a space after a backslash
		could loose its original semantic. Better yet: use the backslash followed by
		the octal rappresentation of the space character ('\040').
		Anyway , to mantain the compatibility with other popular irc clients (such as mIrc),
		let's include the '"' quotes in our standard: literal (unescaped) '"' quotes
		define a single token string. To include a literal '"' character, "escape" it.
		Additionally, the last parameter of a <ctcp message> may be made of multiple tokens.

		[big]A ctcp parameter extracting example[/big][br]
		An trivial example of a C "ctcp parameter extracting routine" follows.[br]
		An IRC message is made of 510 "useable" characters.
		When a CTCP is sent there is a PRIVMSG or NOTICE token that uses at least 6 characters,
		at least two spaces and a target token (that can not be empty, so it is at least one character)
		and finally one <0x01> escape character. This gives 500 characters as maximum size
		for a complete <ctcp message> and thus for a <ctcp token>.
		In fact the <ctcp message> is always smaller than 500 characters: the <0x01> chars are
		usually two, there is a message source part at the beginning of the IRC message
		that is 10-15 characters long and there is a ':' character before the trailing parameter.
		Anyway, to be really on the "safe side", we use a 512 character buffer for each
		<ctcp token>. Finally, I'll assume that you have already ensured that
		the <ctcp message> that we are extracting from is shorter than 511 characters at all,
		and have provided a buffer big enough to avoid this code to segfault.
		I'm assuming that msg_ptr points somewhere in the <ctcp message> and is null-terminated.[br]
		(There are C++ style comments, you might want to remove it)
		[example]
		const char * decode_escape(const char * msg_ptr,char * buffer)
		{
			// This one decodes an escape sequence
			// and returns the pointer "just after it"
			// and should be called when *msg_ptr points
			// just after a backslash
			char c;
			if((*msg_ptr >= '0') && (*msg_ptr < '8'))
			{
				// a digit follows the backslash
				c = *msg_ptr - '0';
				msg_ptr++;
				if(*msg_ptr >= '0') && (*msg_ptr < '8'))
				{
					c = ((c << 3) + *msg_ptr);
					msg_ptr++;
					if(*msg_ptr >= '0') && (*msg_ptr < '8'))
					{
						c = ((c << 3) + *msg_ptr);
						msg_ptr++;
					} // else broken message, but let's be flexible
				} // else it is broken, but let's be flexible
				// append the character and return
				*buffer = c;
				return msg_ptr;
			} else {
				// simple escape: just append the following
				// character (thus discarding its semantic)
				*buffer = *msg_ptr;
				return ++msg_ptr;
			}
		}

		const char * extract_ctcp_parameter(const char * msg_ptr,char * buffer,int spaceBreaks)
		{
			// this one extracts the "next" ctcp parameter in msg_ptr
			// it skips the leading and trailing spaces.
			// spaceBreaks should be set to 0 if (and only if) the
			// extracted parameter is the last in the CTCP message.
			int inString = 0;
			while(*msg_ptr == ' ')msg_ptr++;
			while(*msg_ptr)
			{
				switch(*msg_ptr)
				{
					case '\\':
						// backslash : escape sequence
						msg_ptr++;
						if(*msg_ptr)msg_ptr = decode_escape(msg_ptr,buffer);
						else return msg_ptr; // senseless backslash
					break;
					case ' ':
						// space : separate tokens ?
						if(inString || (!spaceBreaks))*buffer++ = *msg_ptr++;
						else {
							// not in string and space breaks: end of token
							// skip trailing white space (this could be avoided)
							// and return
							while(*msg_ptr == ' ')msg_ptr++;
							return msg_ptr;
						}
					break;
					case '"':
						// a string begin or end
						inString = !inString;
						msg_ptr++;
					break;
					default:
						// any other char
						*buffer++ = *msg_ptr++;
					break;
				}
			}
			return msg_ptr;
		}
		[/example][br]

		[big]CTCP parameter semantics[/big][br]
		The first <ctcp parameter> of a <ctcp message> is the <ctcp tag>: it defines
		the semantic of the rest of the message.[br]
		Altough it is a convention to specify the <ctcp tag> as uppercase letters,
		and the original specification says that the whole <ctcp messages> are
		case sensitive, I'd prefer to follow the IRC message semantic (just to
		have less "special cases") and treat the whole mssage as [b]case insensitive[/b].[br]
		The remaining tokens depend on the <ctcp tag>. A description of known <ctcp tags>
		and thus <ctcp messages> follows.[br]

		[big]PING[/big][br]
		[b]Syntax: <0x01>PING <data><0x01>[/b][br]
		The PING request is used to check the round trip time from one client to another.
		The receiving client should reply with exactly the same message but sent
		through a NOTICE instead of a PRIVMSG. The <data> usually contains an unsigned
		integer but not necessairly; it is not even mandatory for <data> to be a single token.
		The receiver should ignore the semantic of <data>.[br]
		The reply is intended to be processed by IRC clients.

		[big]VERSION[/big][br]
		[b]Syntax: <0x01>VERSION<0x01>[/b][br]
		The VERSION request asks for informations about another user's IRC client program.
		The reply should be sent thru a NOTICE with the following syntax:[br]
		<0x01>VERSION <client_version_data><0x01>[br]
		The preferred form for <client_version_data> is
		"<client_name>:<client_version>:<client_enviroinement>", but historically
		clients (and users) send a generic reply describing the client name, version
		and eventually the used script name. This CTCP reply is intended to be human
		readable, so any form is accepted.

		[big]USERINFO[/big][br]
		[b]Syntax: <0x01>USERINFO<0x01>[/b][br]
		The USERINFO request asks for informations about another user.
		The reply should be sent thru a NOTICE with the following syntax:[br]
		<0x01>USERINFO <user_info_data><0x01>[br]
		The <user_info_data> should be a human readable "user defined" string;
	
		[big]CLIENTINFO[/big][br]
		[b]Syntax: <0x01>CLIENTINFO<0x01>[/b][br]
		The CLIENTINFO request asks for informations about another user's IRC client program.
		While VERSION requests the client program name and version, CLIENTINFO requests
		informations about CTCP capabilities.[br]
		The reply should be sent thru a NOTICE with the following syntax:[br]
		<0x01>CLIENTINFO <client_info_data><0x01>[br]
		The <client_info_data> should contain a list of supported CTCP request tags.
		The CLIENTINFO reply is intended to be human readable.

		[big]FINGER[/big][br]
		[b]Syntax: <0x01>FINGER<0x01>[/b][br]
		The FINGER request asks for informations about another IRC user.
		The reply should be sent thru a NOTICE with the following syntax:[br]
		<0x01>FINGER <user_info_data><0x01>[br]
		The <user_info_data> should be a human readable string containing
		the system username and possibly the system idle time;

		[big]SOURCE[/big][br]
		[b]Syntax: <0x01>SOURCE<0x01>[/b][br]
		The SOURCE request asks for the client homepage or ftp site informations.
		The reply should be sent thru a NOTICE with the following syntax:[br]
		<0x01>VERSION <homepage_url_data><0x01>[br]
		This CTCP reply is intended to be human readable, so any form is accepted.

		[big]TIME[/big][br]
		[b]Syntax: <0x01>TIME<0x01>[/b][br]
		The TIME request asks for the user local time.
		The reply should be sent thru a NOTICE with the following syntax:[br]
		<0x01>TIME <time and date string><0x01>[br]
		This CTCP reply is intended to be human readable, so any form is accepted.

		[big]ACTION[/big][br]
		[b]Syntax: <0x01>ACTION<0x01>[/b][br]
		The ACTION tag is used to describe an action.[br]
		It should be sent through a NOTICE message and never generate a reply.[br]

		[big]AVATAR (equivalent to ICON or FACE)[/big][br]
		[b]Syntax: <0x01>AVATAR<0x01>[/b][br]
		The AVATAR tag is used to query an user's avatar.[br]

		[big]MULTIMEDIA (equivalent to MM or SOUND)[/big][br]
		[b]Syntax: <0x01>MULTIMEDIA <filename><0x01>[/b][br]
		The MULTIMEDIA tag is used to play a multimedia file on the receiver's side.[br]
		The receiving client should locate the file associated to <filename>,
		and play it. If the file can not be located
		by the receiving client, and the MULTIMEDIA tag was sent through a PRIVMSG format ctcp,
		the receiving client CAN request a [doc:dcc_connection]DCC GET[/doc] <filename> from the source user.
		If the MULTIMEDIA tag was sent through a NOTICE message, the receiving client
		should not generate any reply: the message should be notified to the receiving
		client's user and then be discarded. The <filename> should never contain a leading
		path. If any part of the <filename> looks to be a path component , it should be discarded.
		The client may decide to drop the entire message too. Older clients (including
		older releases of KVIrc) used to request the missing filenames by a particular
		non-standard private message syntax. This convention should be dropped.[br]

		[big]DCC[/big][br]
		[b]Syntax: <0x01>DCC <type> <type dependant parameters><0x01>[/b][br]
		The DCC tag is used to initiate a Direct Client Connection.
		The known dcc types are:[br]
		CHAT[br]
		SEND[br]
		TSEND[br]
		GET[br]
		TGET[br]
		ACCEPT[br]
		RESUME[br]

*/


const char * KviServerParser::decodeCtcpEscape(const char * msg_ptr,KviStr &buffer)
{
	if((*msg_ptr >= '0') && (*msg_ptr < '8'))
	{
		char c = *msg_ptr - '0';
		msg_ptr++;
		if((*msg_ptr >= '0') && (*msg_ptr < '8'))
		{
			c = ((c << 3) + *msg_ptr);
			msg_ptr++;
			if((*msg_ptr >= '0') && (*msg_ptr < '8'))
			{
				c = ((c << 3) + *msg_ptr);
				msg_ptr++;
			}
		}
// FIXME: #warning "bAllowNull, bAllowCrLf"
		buffer.append(c);
		return msg_ptr;
	} else {
		buffer.append(*msg_ptr);
		return ++msg_ptr;
	}
}

const char * KviServerParser::extractCtcpParameter(const char * msg_ptr,KviStr &buffer,bool bSpaceBreaks)
{
	while(*msg_ptr == ' ')msg_ptr++;
	bool bInString = false;
	const char * aux = msg_ptr;
	while(*msg_ptr)
	{
		switch(*msg_ptr)
		{
			case '\\':
				buffer.append(aux,msg_ptr - aux);
				msg_ptr++;
				if(*msg_ptr)msg_ptr = decodeCtcpEscape(msg_ptr,buffer);
				else return msg_ptr;
				
				aux = msg_ptr;
			break;
			case ' ':
				if(bInString || (!bSpaceBreaks))msg_ptr++;
				else
				{
					buffer.append(aux,msg_ptr - aux);
					while(*msg_ptr == ' ')msg_ptr++;
					return msg_ptr;
				}
			break;
			case '"':
				bInString = !bInString;
				msg_ptr++;
			break;
			default:
				msg_ptr++;
			break;
		}
	}
	buffer.append(aux,msg_ptr - aux);
	return msg_ptr;
}



void KviServerParser::parseCtcpRequest(KviCtcpMessage *msg)
{
	msg->pData = extractCtcpParameter(msg->pData,msg->szTag);

	if(TRIGGER_EVENT_6PARAM_RETVALUE(KviEvent_OnCtcpRequest,msg->msg->console(),msg->pSource->nick(),msg->pSource->user(),msg->pSource->host(), \
					msg->pTarget,msg->szTag.ptr(),msg->pData))return;

	for(int i=0;m_ctcpRequestParseProcTable[i].msgName;i++)
	{
		if(kvi_strEqualCI(msg->szTag.ptr(),m_ctcpRequestParseProcTable[i].msgName))
		{
			(this->*(m_ctcpRequestParseProcTable[i].proc))(msg);
			return;
		}
	}

	// unknown
	msg->bUnknown = true;
	echoCtcpRequest(msg);
}

void KviServerParser::parseCtcpReply(KviCtcpMessage *msg)
{
	msg->pData = extractCtcpParameter(msg->pData,msg->szTag);

	if(TRIGGER_EVENT_6PARAM_RETVALUE(KviEvent_OnCtcpReply,msg->msg->console(),msg->pSource->nick(),msg->pSource->user(),msg->pSource->host(), \
					msg->pTarget,msg->szTag.ptr(),msg->pData))return;


	for(int i=0;m_ctcpReplyParseProcTable[i].msgName;i++)
	{
		if(kvi_strEqualCI(msg->szTag.ptr(),m_ctcpReplyParseProcTable[i].msgName))
		{
			(this->*(m_ctcpReplyParseProcTable[i].proc))(msg);
			return;
		}
	}

	// unknown
	msg->bUnknown = true;
	echoCtcpReply(msg);
}


// Ctcp message handlers

bool KviServerParser::checkCtcpFlood(KviCtcpMessage *msg)
{
	if(!KVI_OPTION_BOOL(KviOption_boolUseCtcpFloodProtection))return false;

	unsigned int interval = (unsigned int)(((unsigned int)time(0)) - ((unsigned int)msg->msg->console()->connectionInfo()->lastCtcpTime));

	if(interval < KVI_OPTION_UINT(KviOption_uintCtcpFloodCheckInterval))
	{
		msg->msg->console()->connectionInfo()->uCtcpCount++;
		if(msg->msg->console()->connectionInfo()->uCtcpCount > KVI_OPTION_UINT(KviOption_uintMaxCtcpRequests))
		{
			// This is flood
			msg->bIsFlood = true;
			return true;
		}
	} else {
		msg->msg->console()->connectionInfo()->lastCtcpTime = time(0);
		msg->msg->console()->connectionInfo()->uCtcpCount = 1;
	}
	return false;
}


void KviServerParser::replyCtcp(KviCtcpMessage *msg,const char * data)
{
	msg->msg->console()->socket()->sendFmtData("NOTICE %s :%c%s %s%c",
		msg->pSource->nick(),0x01,msg->szTag.ptr(),data,0x01);
}

void KviServerParser::echoCtcpReply(KviCtcpMessage * msg)
{
	if(!msg->msg->haltOutput())
	{
//		KviStr req = msg->szTag.ptr();
//		if(*(msg->pData))req.append(KviStr::Format," %s",msg->pData);

		KviWindow * pOut = KVI_OPTION_BOOL(KviOption_boolCtcpRepliesToActiveWindow) ? msg->msg->console()->activeWindow() : msg->msg->console();

		bool bIsChannel = false;

		if(!IS_ME(msg->msg,msg->pTarget))
		{
			// Channel ctcp request!
			pOut = msg->msg->console()->findChannel(msg->pTarget);
			if(!pOut)
			{
				pOut = msg->msg->console();
				pOut->output(KVI_OUT_SYSTEMWARNING,
					__tr("The following ctcp reply has unrecognized target \"%s\""),
					msg->pTarget);
			} else bIsChannel = true;
		}

		KviStr buffer1,buffer2;

		pOut->output(
			msg->bUnknown ? KVI_OUT_CTCPREPLYUNKNOWN : KVI_OUT_CTCPREPLY,
			__tr("%s %s reply from %s [%s@%s]: %s"),
			bIsChannel ? __tr("Channel CTCP") : "CTCP",
			msg->szTag.ptr(),formatNickLink(msg->pSource->nick(),buffer1),
			msg->pSource->username(),formatHostLink(msg->pSource->host(),buffer2),msg->pData);
	}
}


void KviServerParser::echoCtcpRequest(KviCtcpMessage *msg)
{
// FIXME: #warning "DEDICATED CTCP WINDOW...MINIMIZED ?"
	if(!msg->msg->haltOutput())
	{
		KviStr req = msg->szTag.ptr();
		if(*(msg->pData))req.append(KviStr::Format," %s",msg->pData);

		KviWindow * pOut = msg->msg->console();

		bool bIsChannel = false;

		if(!IS_ME(msg->msg,msg->pTarget))
		{
			// Channel ctcp request!
			pOut = msg->msg->console()->findChannel(msg->pTarget);
			if(!pOut)
			{
				pOut = msg->msg->console();
				pOut->output(KVI_OUT_SYSTEMWARNING,
					__tr("The following ctcp request has unrecognized target %s"),
					msg->pTarget);
			} else bIsChannel = true;
		}

		KviStr buffer1,buffer2;
		if(msg->bIsFlood)
		{
			if(!TRIGGER_EVENT_6PARAM_RETVALUE(KviEvent_OnCtcpFlood,pOut,msg->pSource->nick(),msg->pSource->username(),msg->pSource->host(),msg->pTarget,msg->szTag.ptr(),msg->pData))
				pOut->output(KVI_OUT_CTCPREQUESTFLOOD,
					__tr("%s %s request from %s [%s@%s] (%s): ignored (flood limit exceeded)"),
					bIsChannel ? __tr("Channel CTCP") : "CTCP",
					msg->szTag.ptr(),formatNickLink(msg->pSource->nick(),buffer1),
					msg->pSource->username(),formatHostLink(msg->pSource->host(),buffer2),req.ptr());
		} else {
			pOut->output(
				msg->bUnknown ? KVI_OUT_CTCPREQUESTUNKNOWN : 
					(msg->bIgnored ? KVI_OUT_CTCPREQUESTIGNORED : KVI_OUT_CTCPREQUESTREPLIED),
				__tr("%s %s request from %s [%s@%s] (%s): %s"),
				bIsChannel ? __tr("Channel CTCP") : "CTCP",
				msg->szTag.ptr(),formatNickLink(msg->pSource->nick(),buffer1),
				msg->pSource->username(),formatHostLink(msg->pSource->host(),buffer2),req.ptr(),
				msg->bUnknown ? __tr("ignored (not recognized)") :
					(msg->bIgnored ? __tr("ignored") : __tr("replied"))
				);
		}
	}
}



void KviServerParser::parseCtcpRequestPing(KviCtcpMessage *msg)
{
	if(!checkCtcpFlood(msg))
	{
		if(!KVI_OPTION_BOOL(KviOption_boolIgnoreCtcpPing))
		{
			replyCtcp(msg,msg->pData);
		} else msg->bIgnored = true;
	}

	echoCtcpRequest(msg);
}

void KviServerParser::parseCtcpReplyPing(KviCtcpMessage * msg)
{
	if(!msg->msg->haltOutput())
	{
		KviWindow * pOut = KVI_OPTION_BOOL(KviOption_boolCtcpRepliesToActiveWindow) ? msg->msg->console()->activeWindow() : msg->msg->console();

		bool bIsChannel = false;

		if(!IS_ME(msg->msg,msg->pTarget))
		{
			// Channel ctcp request!
			pOut = msg->msg->console()->findChannel(msg->pTarget);
			if(!pOut)
			{
				pOut = msg->msg->console();
				pOut->output(KVI_OUT_SYSTEMWARNING,
					__tr("The following ctcp PING reply has unrecognized target \"%s\""),
					msg->pTarget);
			} else bIsChannel = true;
		}

		unsigned int uSecs;
		unsigned int uMSecs = 0;

		KviStr szTime;

		struct timeval tv;
		kvi_gettimeofday(&tv,0);

		msg->pData = extractCtcpParameter(msg->pData,szTime,true);

		if(szTime.contains('.'))
		{
			KviStr szUSecs = szTime;
			szUSecs.cutToFirst('.');
			szTime.cutFromFirst('.');

			bool bOk;
			uMSecs = szUSecs.toUInt(&bOk);
			if(!bOk)
			{
				uMSecs = 0;
				tv.tv_usec = 0;
			}
		} else tv.tv_usec = 0;

		bool bOk;

		uSecs = szTime.toUInt(&bOk);
		if(!bOk)pOut->output(KVI_OUT_SYSTEMWARNING,
				__tr("The following ctcp PING reply has a broken time identifier \"%s\": don't trust the displayed time"),szTime.ptr());



		KviStr buffer1,buffer2;

		unsigned int uDiffSecs = tv.tv_sec - uSecs;
		while(((unsigned int)tv.tv_usec) < uMSecs)
		{
			tv.tv_usec += 1000000;
			if(uDiffSecs > 0)uDiffSecs --;
		}
		unsigned int uDiffMSecs = (tv.tv_usec - uMSecs) / 1000;
		

		pOut->output(
			msg->bUnknown ? KVI_OUT_CTCPREPLYUNKNOWN : KVI_OUT_CTCPREPLY,
			__tr("%s PING reply from %s [%s@%s]: ~ %u secs %u msecs"),
			bIsChannel ? __tr("Channel CTCP") : "CTCP",formatNickLink(msg->pSource->nick(),buffer1),
			msg->pSource->username(),formatHostLink(msg->pSource->host(),buffer2),uDiffSecs,uDiffMSecs);
	}
}


void KviServerParser::parseCtcpRequestVersion(KviCtcpMessage *msg)
{
	if(!checkCtcpFlood(msg))
	{
		if(!KVI_OPTION_BOOL(KviOption_boolIgnoreCtcpVersion))
		{
			KviStr version = KVI_CTCP_VERSION_REPLY;
			if(KVI_OPTION_STRING(KviOption_stringCtcpVersionPostfix).hasData())
			{
				version.append(KviStr::Format," :%s",
					KVI_OPTION_STRING(KviOption_stringCtcpVersionPostfix).ptr());
			}
			replyCtcp(msg,version.ptr());
		} else msg->bIgnored = true;
	}

	echoCtcpRequest(msg);
}


void KviServerParser::parseCtcpRequestUserinfo(KviCtcpMessage *msg)
{
	if(!checkCtcpFlood(msg))
	{
		if(!KVI_OPTION_BOOL(KviOption_boolIgnoreCtcpUserinfo))
		{
			replyCtcp(msg,KVI_OPTION_STRING(KviOption_stringCtcpUserinfoReply).ptr());
		} else msg->bIgnored = true;
	}

	echoCtcpRequest(msg);
}

// FIXME: CTCP SEX , AGE , LOCATION!!! <--- so we will be safe :)
// FIXME: KEEP THIS TABLE UP TO DATE

static const char * ctcpTagTable[][2]=
{
	{ "PING"        , "Returns the parameters without parsing it"                   },
	{ "VERSION"     , "Returns the version of this client"                          },
	{ "CLIENTINFO"  , "With no parameters lists supported ctcp tags," \
						"'CLIENTINFO <tag>' describes the <tag>"                    },
	{ "USERINFO"    , "Returns personal informations about the current client user" },
	{ "FINGER"      , "Returns informations about the current client user"          },
	{ "SOURCE"      , "Returns the client homepage url"                             },
	{ "TIME"        , "Returns the current local time"                              },
	{ "ACTION"      , "Used to describe actions: generates no reply"                },
	{ "AVATAR"      , "Returns the current avatar (may generate a DCC GET) or" \
						" sets your own on this side (if sent thru a NOTICE)"       },
	{ "DCC"         , "Initiates a dcc connection (XDCC,TDCC)"                      },
	{ "PAGE"        , "Logs a message for this client"                              },
	{ 0             , 0                                                             }
};

void KviServerParser::parseCtcpRequestClientinfo(KviCtcpMessage *msg)
{
	if(!checkCtcpFlood(msg))
	{
		if(!KVI_OPTION_BOOL(KviOption_boolIgnoreCtcpClientinfo))
		{
			KviStr szTag;
			msg->pData = extractCtcpParameter(msg->pData,szTag,false);
			szTag.stripWhiteSpace();
			szTag.toUpper();
			if(szTag.isEmpty())
			{
				KviStr reply(KviStr::Format,"%s : Supported tags: ",KVI_CTCP_CLIENTINFO_VERSION);
				for(int i=0;ctcpTagTable[i][0];i++)
				{
					reply.append(ctcpTagTable[i][0]);
					if(ctcpTagTable[i + 1][0])reply.append(',');
				}
				reply.append(" : Use 'CLIENTINFO <tag>' for a description of each tag");
				replyCtcp(msg,reply.ptr());
			} else {
				bool bFound = false;
				for(int i=0;ctcpTagTable[i][0] && !bFound;i++)
				{
					if(kvi_strEqualCS(ctcpTagTable[i][0],szTag.ptr()))
					{
						KviStr reply(KviStr::Format,"%s: %s",ctcpTagTable[i][0],ctcpTagTable[i][1]);
						replyCtcp(msg,reply.ptr());
						bFound = true;
					}
				}
				if(!bFound)
				{
					msg->szTag= "ERRMSG";
					KviStr reply(KviStr::Format,"Unsupported tag %s",szTag.ptr());
					replyCtcp(msg,reply.ptr());
				}
			}
		} else msg->bIgnored = true;
	}

	echoCtcpRequest(msg);
}

void KviServerParser::parseCtcpRequestFinger(KviCtcpMessage *msg)
{
	if(!checkCtcpFlood(msg))
	{
		if(!KVI_OPTION_BOOL(KviOption_boolIgnoreCtcpFinger))
		{
			KviStr username = getenv("USER");
			if(username.isEmpty())username = getenv("LOGNAME");
			if(username.isEmpty())username = msg->msg->console()->currentUserName();
// FIXME: #warning "UTSNAME ?...AND OTHER INFO ?...SYSTEM IDLE TIME ?...KVIRC IDLE TIME ?"

			KviStr reply(KviStr::Format,"%s",username.ptr());
			replyCtcp(msg,reply.ptr());
		} else msg->bIgnored = true;
	}

	echoCtcpRequest(msg);
}

void KviServerParser::parseCtcpRequestSource(KviCtcpMessage *msg)
{
	if(!checkCtcpFlood(msg))
	{
		if(!KVI_OPTION_BOOL(KviOption_boolIgnoreCtcpSource))
		{
			KviStr version = KVI_CTCP_SOURCE_REPLY;
			if(KVI_OPTION_STRING(KviOption_stringCtcpSourcePostfix).hasData())
			{
				version.append(KviStr::Format," :%s",
					KVI_OPTION_STRING(KviOption_stringCtcpSourcePostfix).ptr());
			}
			replyCtcp(msg,version.ptr());
		} else msg->bIgnored = true;
	}

	echoCtcpRequest(msg);
}

void KviServerParser::parseCtcpRequestTime(KviCtcpMessage *msg)
{
	if(!checkCtcpFlood(msg))
	{
		if(!KVI_OPTION_BOOL(KviOption_boolIgnoreCtcpTime))
		{
			KviStr tmp = QDateTime::currentDateTime().toString();
			replyCtcp(msg,tmp.ptr());
		} else msg->bIgnored = true;
	}

	echoCtcpRequest(msg);
}

void KviServerParser::parseCtcpRequestPage(KviCtcpMessage *msg)
{
	if(!checkCtcpFlood(msg))
	{
		if(!KVI_OPTION_BOOL(KviOption_boolIgnoreCtcpPage))
		{
			KVI_OPTION_STRING(KviOption_stringCtcpPageReply).stripWhiteSpace();
			if(KVI_OPTION_STRING(KviOption_stringCtcpPageReply).isEmpty())
			{
				KVI_OPTION_STRING(KviOption_stringCtcpPageReply) = KVI_DEFAULT_CTCP_PAGE_REPLY;
			}
			replyCtcp(msg,KVI_OPTION_STRING(KviOption_stringCtcpPageReply).ptr());
			if(KVI_OPTION_BOOL(KviOption_boolShowDialogOnCtcpPage))
			{
				if(!g_pCtcpPageDialog)g_pCtcpPageDialog = new KviCtcpPageDialog();
				g_pCtcpPageDialog->addPage(msg->pSource->nick(),msg->pSource->user(),msg->pSource->host(),msg->pData);
				g_pCtcpPageDialog->popup();
			}
		} else msg->bIgnored = true;
	}

	echoCtcpRequest(msg);
}

void KviServerParser::parseCtcpRequestAction(KviCtcpMessage *msg)
{
	KviStr szData;
	msg->pData = extractCtcpParameter(msg->pData,szData,false);

	if(TRIGGER_EVENT_3PARAM_RETVALUE(KviEvent_OnAction,msg->msg->console(), msg->pSource->nick(), msg->pSource->username(),msg->pSource->host()))msg->msg->setHaltOutput();

	if(IS_ME(msg->msg,msg->pTarget))				
	{
		KviQuery * query = msg->msg->console()->findQuery(msg->pSource->nick());
		if(!query)
		{
			// New query requested ?
// FIXME: 			#warning "CHECK FOR SPAM!"
			query = msg->msg->console()->createQuery(msg->pSource->nick());
			query->addTarget(msg->pSource->nick(),msg->pSource->username(),msg->pSource->host());
// FIXME: 			#warning "OPTION TO SELECT IF CREATE QUERY ON ACTION!!!"
// FIXME: 			#warning "OPTION TO CREATE MINIMIZED QUERY"
// FIXME: 			#warning "OTHERWISE SAY THE MSG TO THE ACTIVE WINDOW!!!"
		}
		if(!msg->msg->haltOutput())query->output(KVI_OUT_ACTION,"%s %s",msg->pSource->nick(),szData.ptr());
	} else {
		KviChannel * chan = msg->msg->console()->findChannel(msg->pTarget);
		if(chan)
		{
// FIXME: #warning "ESCAPE SEQUENCES IN THIS MSG ?"
			chan->outputMessage(KVI_OUT_ACTION,"%s %s",msg->pSource->nick(),szData.ptr());
			chan->userAction(msg->pSource,KVI_USERACTION_ACTION);
		} else {
			if(!msg->msg->haltOutput())
			{
				KviStr buffer1,buffer2;
				msg->msg->console()->output(KVI_OUT_SYSTEMWARNING,
					__tr("CTCP ACTION from %s [%s@%s] to unknown target %s: %s"),
					formatNickLink(msg->pSource->nick(),buffer1),msg->pSource->username(),
					formatHostLink(msg->pSource->host(),buffer2),msg->pTarget,szData.ptr());
			}
		}
	}
}
// FIXME: #warning "UTSNAME ?...AND OTHER INFO ?...SYSTEM IDLE TIME ?...KVIRC IDLE TIME ?"

void KviServerParser::parseCtcpRequestAvatar(KviCtcpMessage *msg)
{
	// AVATAR
	if(!checkCtcpFlood(msg))
	{
		if(!KVI_OPTION_BOOL(KviOption_boolIgnoreCtcpAvatar))
		{
			KviAvatar * a = msg->msg->console()->currentAvatar();
			if(a)
			{
// FIXME: #warning "TIMEDOUT WOULD ALSO NEED TO EXTEND EXISTING OFFERS LIFETIME"
// FIXME: #warning "OPTION FOR SETTING A FIXED BIND ADDRESS FOR OUTGOING DCC OFFERS"
				KviStr szUserMask;
				msg->pSource->mask(szUserMask);

				KviFileOffer * o;

				KviStr szName = a->name();

				kvi_adjustFilePath(szName);
				szName.cutToLast(KVI_PATH_SEPARATOR_CHAR);

				if(!(o = g_pFileTrader->addOffer(szName.ptr(),a->path(),szUserMask.ptr(),KVI_OPTION_UINT(KviOption_uintAvatarOfferTimeoutInSecs))))
				{
					// Don't delete o...it has been already deleted by g_pFileTrader
					msg->msg->console()->output(KVI_OUT_SYSTEMWARNING,__tr("Can't add a file offer for file %s (huh ? file not readable ?)"),a->path());
				} else {
					if(KVI_OPTION_BOOL(KviOption_boolBeVerbose))
					{
						msg->msg->console()->output(KVI_OUT_SYSTEMMESSAGE,__tr("Added %d secs file offer for file %s (%s) and receiver %s"),
							KVI_OPTION_UINT(KviOption_uintAvatarOfferTimeoutInSecs),szName.ptr(),a->path(),szUserMask.ptr());
					}
				}

				KviStr szReply;
				if(o)szReply.sprintf("%s %u",szName.ptr(),o->fileSize());
				else szReply = szName;

				replyCtcp(msg,szReply.ptr());
			} else replyCtcp(msg,"");
		} else msg->bIgnored = true;
	}

	echoCtcpRequest(msg);

}


void KviServerParser::parseCtcpReplyAvatar(KviCtcpMessage *msg)
{
	KviStr szFile;
	KviStr szSize;
	msg->pData = extractCtcpParameter(msg->pData,szFile,true);
	msg->pData = extractCtcpParameter(msg->pData,szSize,true);

	bool bOk;
	unsigned int uSize = szSize.toUInt(&bOk);
	if(!bOk)uSize = 0;

	szFile.stripWhiteSpace();

	KviStr szData = szFile;

	if(szData.contains(KVI_PATH_SEPARATOR_CHAR))
	{
		// szData is a file path... cut it : we don't want to be hacked :)
		szData.cutToLast(KVI_PATH_SEPARATOR_CHAR,true);
	}

	bool bPrivate = IS_ME(msg->msg,msg->pTarget);

	KviStr textLine;
	KviAvatar * avatar = 0;

	bool bResetAvatar = true;

	KviStr nickLink;
	KviStr hostLink;

	formatNickLink(msg->pSource->nick(),nickLink);
	formatHostLink(msg->pSource->host(),hostLink);

	KviIrcUserEntry * e = msg->msg->console()->userDataBase()->find(msg->pSource->nick());

	if(szData.isEmpty())
	{
		// avatar unset
		textLine.sprintf(__tr("%s [%s@%s] unsets avatar (%s %s)"),
			nickLink.ptr(),msg->pSource->username(),hostLink.ptr(),
			bPrivate ? __tr("private") : __tr("channel notification:"),
			bPrivate ? __tr("notification") : msg->pTarget);

	} else {

// FIXME: #warning "The avatar should be the one with the requested size!!"
		avatar = g_pIconManager->getAvatar(szData.ptr());

		textLine.sprintf(__tr("%s [%s@%s] changes avatar to %s (%u bytes, %s %s)"),
			nickLink.ptr(),msg->pSource->username(),hostLink.ptr(),szData.ptr(),uSize,
			bPrivate ? __tr("private") : __tr("channel notification:"),
			bPrivate ? __tr("notification") : msg->pTarget);

		if((avatar == 0) && e)
		{
			// we have no such file on our HD....
			bResetAvatar = false;
			// request DCC GET ?
			if(KVI_OPTION_BOOL(KviOption_boolRequestMissingAvatars))
			{
// FIXME: #warning "Request avatars only from registered users ?"
// FIXME: #warning "Ask before making the request ?"

				if((uSize <= KVI_OPTION_UINT(KviOption_uintMaximumRequestedAvatarSize)) && (uSize > 0))
				{
					if(!checkCtcpFlood(msg))
					{
						textLine.append(KviStr::Format,
							__tr(": No valid local copy of avatar available; requesting one (DCC GET %s)"),
							szFile.ptr());

						msg->msg->console()->socket()->sendFmtData("PRIVMSG %s :%cDCC GET %s %u%c",
								msg->pSource->nick(),0x01,szFile.ptr(),uSize,0x01);
	
						g_pApp->setAvatarOnFileReceived(msg->msg->console(),
							szFile.ptr(),msg->pSource->nick(),msg->pSource->username(),msg->pSource->host());
					} else {
						textLine.append(__tr(": No valid local copy of avatar available; flood limit exceeded: ignoring"));
					}
				} else {
					textLine.append(KviStr::Format,__tr(": No valid local copy of avatar available; unacceptable/invalid avatar size (%u): ignoring"),uSize);
				}
			} else {
				textLine.append(__tr(": No valid local copy of avatar available; ignoring"));
			}
		}
	}

	if(!e)
	{
		textLine.append(__tr(": No such nickname in the user database; ignoring the change"));
// FIXME: #warning "MAYBE CACHE THE AVATAR ?"
		msg->msg->console()->output(KVI_OUT_AVATAR,textLine.ptr());
		return;
	}

	if(bResetAvatar)e->setAvatar(avatar);
	msg->msg->console()->avatarChanged(avatar,msg->pSource->nick(),msg->pSource->user(),msg->pSource->host(),
												msg->msg->haltOutput() ? 0 : textLine.ptr());

}



typedef void (*dccModuleCtcpDccParseRoutine)(KviDccRequest *par);



void KviServerParser::parseCtcpRequestDcc(KviCtcpMessage *msg)
{
	KviDccRequest p;
	KviStr aux    = msg->pData;
	msg->pData    = extractCtcpParameter(msg->pData,p.szType);
	msg->pData    = extractCtcpParameter(msg->pData,p.szParam1);
	msg->pData    = extractCtcpParameter(msg->pData,p.szParam2);
	msg->pData    = extractCtcpParameter(msg->pData,p.szParam3);
	msg->pData    = extractCtcpParameter(msg->pData,p.szParam4);
	p.ctcpMsg     = msg;
	p.bIpV6       = msg->msg->console()->isIpV6Connection();
	p.pConsole    = msg->msg->console();

	KviStr nickBuffer,hostBuffer;

	if(!checkCtcpFlood(msg))
	{
		if(!KVI_OPTION_BOOL(KviOption_boolIgnoreCtcpDcc))
		{
			if(!msg->msg->haltOutput())
			{
				msg->msg->console()->output(KVI_OUT_DCCREQUEST,
					__tr("Processing DCC %s request from %s [%s@%s] (%s %s)"),
					p.szType.ptr(),formatNickLink(msg->pSource->nick(),nickBuffer),
					msg->pSource->username(),formatHostLink(msg->pSource->host(),hostBuffer),msg->szTag.ptr(),
					aux.ptr());
			}
	
			KviModule * m = g_pModuleManager->getModule("dcc");
			if(!m)
			{
				msg->msg->console()->output(KVI_OUT_DCCERROR,
					__tr("Unable to process the above request: can't load the dcc module (%s)"),g_pModuleManager->lastError().ptr());
			} else {
				dccModuleCtcpDccParseRoutine proc = (dccModuleCtcpDccParseRoutine)m->getSymbol("dccModuleCtcpDccParseRoutine");
				if(!proc)
				{
					msg->msg->console()->outputNoFmt(KVI_OUT_DCCERROR,
						__tr("Unable to process the above request: the dcc module looks to be broken"));
				} else {
					proc(&p);
				}
			}
		} else {
			if(!msg->msg->haltOutput())
			{
				msg->msg->console()->output(KVI_OUT_DCCREQUEST,
					__tr("Ignoring DCC %s request from %s [%s@%s] (%s %s)"),
					p.szType.ptr(),formatNickLink(msg->pSource->nick(),nickBuffer),
					msg->pSource->username(),formatHostLink(msg->pSource->host(),hostBuffer),p.szType.ptr(),
					msg->szTag.ptr(),aux.ptr());
			}
		}
	} else {
		// That's flood
		echoCtcpRequest(msg);
	}
}

void KviServerParser::parseCtcpReplyGeneric(KviCtcpMessage *msg)
{
	echoCtcpReply(msg);
}



//ERRORMSG,ECHO,ERRMSG
//SED,DCC,SOUND/MULTIMEDIA/MM,SCRIPT
