//
//   File : kvi_notifylist.cpp
//   Creation date : Fri Oct 27 2000 23:41:01 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#define __KVIRC__

#define _KVI_DEBUG_CHECK_RANGE_

#include "kvi_debug.h"
#include "kvi_notifylist.h"
#include "kvi_console.h"
#include "kvi_ircsocket.h"
#include "kvi_regusersdb.h"
#include "kvi_userlistview.h"
#include "kvi_channel.h"
#include "kvi_options.h"
#include "kvi_window.h"
#include "kvi_locale.h"
#include "kvi_out.h"
#include "kvi_sparser.h"
#include "kvi_ircmask.h"
#include "kvi_numeric.h"
#include "kvi_event.h"
#include "kvi_parameterlist.h"

// FIXME: #warning "Finish this doc!"

/*
	@doc: notify_list
	@title:
		Notify lists
	@short:
		Tracking users on IRC
	@keyterms:
		notify property, watch property, notify lists
	@body:
		The notify list is a mean of tracking users on IRC.[br]
		Once connected to an IRC Server , you can setup KVirc in order to check
		periodically if your friends are on-line.[br]
		This is basically achieved by setting a property in a [doc:registered_users]registered users database[/doc]
		entry.[br]
		The property is called "notify": you have to set it to the nickname
		that you want to look for.[br]
		So for example, assume to register a frend of yours like Szymon:[br]
		[example]
			[cmd:reguser.add]reguser.add[/cmd] Szymon
			[cmd:reguser.addmask]reguser.addmask[/cmd] Szymon Pragma!*@*.it
		[/example]
		And then want it in the notify list; nothing easier, just set
		hist "notify" property to the nickname that you want him to be "looked for":[br]
		[example]
			[cmd:reguser.setproperty]reguser.setproperty[/cmd] Szymon notify Pragma
		[/example]
		In this way, once in a while, KVIrc will send to the server an ISON message
		with the nickname Pragma. If Szymon is online, you will get notified with a line like:[br]
		"Pragma (Pragma!someuser@somehost.it) is online".[br]
		If Szymon uses often "[Pragma]" as his secondary nickname , you can do the following:[br]
		[example]
			[cmd:reguser.addmask]reguser.addmask[/cmd] Szymon [Pragma]*@*.it
			[cmd:reguser.setproperty]reguser.setproperty[/cmd] Szymon notify "Pragma [Pragma]"
		[/example]
		KVIrc will then look for both nicknames getting online.[br]
		KVIrc supports three notify lists management methods:[br]
		The "stupid ISON method", the "intelligent ISON method" and the "WATCH method".[br]
		The "stupid ISON method" will assume that Szymon is online if any user with nickname
		Pragma (or [Pragma] in the second example) gets online; this means that also Pragma!someuser@somehost.com will be
		assumed to be "Szymon" and will be shown in the notify list.[br]
		This might be a false assumption (since somehod.com does not even match *.it),
		but it is the best result that the "stupid ISON method" can achieve.[br]
		The "intelligent ISON method" will also check the Pragma's username and hostname
		and match it in the registered masks; so in the example above, you will be notified if
		any user that matches Pragma!*@*.it gets online; (but you will NOT be notified if
		(for example) Pragma!someuser@somehost.com gets online).[br]
		So what's the point in including a stupid method ? :) Well...the intelligent
		method "eats" some of your IRC bandwidth: it has to send USERHOST messages
		for every group of 5 users in the notify list. If you have a lot of users
		in the notify list, it might become slow and eventually help in generating
		client to server flood.[br]
		So finally, the intelligent method is the default. If you have "flood" problems,
		or if you think that the notify list is quite slow , try the "stupid" method:
		it is not that bad after all.[br]
		The third notify list management method is the "WATCH method".[br]
		It uses a totally different (and better) approach to the notify lists management,
		and can be used only on the networks that support the WATCH notify method (DalNet afaik).[br]
		KVIrc will attempt to guess if the server you're currently using supports the WATCH command
		and eventually use this last method.[br]
		The WATCH method uses the "notify" property to get the nicknames that have to be
		sent to the server in the /WATCH commands. 
*/

// Basic NotifyListManager: this does completly nothing

KviNotifyListManager::KviNotifyListManager(KviConsole * console)
: QObject(0,"notify_list_manager")
{
	m_pConsole = console;
}

KviNotifyListManager::~KviNotifyListManager()
{
}

void KviNotifyListManager::start()
{
}

void KviNotifyListManager::stop()
{
}

bool KviNotifyListManager::handleUserhost(KviIrcMessage *)
{
	return false;
}

bool KviNotifyListManager::handleIsOn(KviIrcMessage *)
{
	return false;
}

bool KviNotifyListManager::handleWatchReply(KviIrcMessage *)
{
	return false;
}

// kvi_app.cpp
extern KviRegisteredUserDataBase * g_pRegisteredUserDataBase;


//
//  INTELLIGENT NOTIFY LIST MANAGER: NOTIFY PROCESS:
//
//            start()                              stop()
//               |                                   ^
//         buildRegUserDict()                        |
//               |                                   |
//     m_pRegUserDict->isEmpty() ? -- YES ---------->+
//                       |                           |
//                      NO                           |
//                       |                           |
//         newNotifySession()<------- TIMER ---------------- delayedNotifySession() --------------------------------+
//                       |    (can be stopped here)  |              ^                                               |
//                       |                           |              ^                                               |
//                  buildNotifyList()                |              |                                              YES
//                       |                           |              |                                               |
//                m_pNotifyList->isEmpty() ? - YES ->+              |                                               |
//                       |                                          |                                               |
//                      NO                                          |                                               |
//                       |                                          |                                               |
//                  newIsOnSession()<------------- TIMER -------------------- delayedIsOnSession() -- NO - m_pNotifyList->isEmpty() ?
//                               |           (can be stopped here)  |                                               |
//                               |                                  |                                               |
//                            buildIsOnList()                       |                                               |
//                               |                                  |                                               |
//                       m_pIsOnList->isEmpty() ? -- YES ---------->+                                               |
//                               |                                                                                  |
//                              NO                                                                                  |
//                               |                                                                                  |
//                            sendIsOn() - - - - - - - - - - - -> handleIsOn()                                      |
//                                                                      |                                           |
//                                                            (build m_pOnlineList)                                 |
//                                                                      |                                           |
//                                                         m_pOnlineList->isEmpty() ? - YES ----------------------->+
//                                                                      |                                           |
//                                                                     NO                                          YES
//                                                                      |                                           |
//                                                            delayedUserhostSession()<--------------- NO - m_pOnlineList->isEmpty() ?
//                                                                               |                                  ^
//                                                                             TIMER (can be stopped here)          |
//                                                                               |                                  |
//                                                                           newUserhostSession()                   |
//                                                                               |                                  |
//                                                                           buildUserhostList()                    |
//                                                                               |                                  |
//                                                                           m_pUserhostList->isEmpty() ? - YES --->+
//                                                                               |                          ^^^     |  
//                                                                               |             (unexpected!)|||     |
//                                                                               NO                                 |
//                                                                               |                                  |
//                                                                           sendUserhost() - - - - - - - - > handleUserhost()
//


KviIsOnNotifyListManager::KviIsOnNotifyListManager(KviConsole * console)
: KviNotifyListManager(console)
{
	m_pRegUserDict = new QAsciiDict<KviStr>(17,false,true); // case insensitive , copy keys
	m_pRegUserDict->setAutoDelete(true);
	m_pNotifyList  = new KviPtrList<KviStr>;
	m_pNotifyList->setAutoDelete(true);
	m_pIsOnList = new KviPtrList<KviStr>;
	m_pIsOnList->setAutoDelete(true);
	m_pOnlineList = new KviPtrList<KviStr>;
	m_pOnlineList->setAutoDelete(true);
	m_pUserhostList = new KviPtrList<KviStr>;
	m_pUserhostList->setAutoDelete(true);
	m_pDelayedNotifyTimer = new QTimer();
	connect(m_pDelayedNotifyTimer,SIGNAL(timeout()),this,SLOT(newNotifySession()));
	m_pDelayedIsOnTimer = new QTimer();
	connect(m_pDelayedIsOnTimer,SIGNAL(timeout()),this,SLOT(newIsOnSession()));
	m_pDelayedUserhostTimer = new QTimer();
	connect(m_pDelayedUserhostTimer,SIGNAL(timeout()),this,SLOT(newUserhostSession()));
	m_bRunning = false;
}


KviIsOnNotifyListManager::~KviIsOnNotifyListManager()
{
	if(m_bRunning)stop();
	delete m_pDelayedNotifyTimer;
	delete m_pDelayedIsOnTimer;
	delete m_pDelayedUserhostTimer;
	delete m_pRegUserDict;
	delete m_pOnlineList;
	delete m_pNotifyList;
	delete m_pIsOnList;
	delete m_pUserhostList;
}

void KviIsOnNotifyListManager::start()
{
	if(m_bRunning)stop();
	m_bRunning = true;
	m_pConsole->notifyListView()->partAll();

	m_bExpectingIsOn = false;
	m_bExpectingUserhost = false;

	buildRegUserDict();
	if(m_pRegUserDict->isEmpty())
	{
		if(KVI_OPTION_BOOL(KviOption_boolBeVerbose))
			m_pConsole->output(KVI_OUT_SYSTEMMESSAGE,__tr("Notify list: no users to check for: stopping"));
		stop();
		return;
	}
	newNotifySession();
}

void KviIsOnNotifyListManager::buildRegUserDict()
{
	m_pRegUserDict->clear();

	const QAsciiDict<KviRegisteredUser> * d = g_pRegisteredUserDataBase->userDict();
	QAsciiDictIterator<KviRegisteredUser> it(*d);
	while(KviRegisteredUser * u = it.current())
	{
		KviStr notify;
		if(u->getProperty("notify",notify))
		{
			notify.stripWhiteSpace();
			KviStr single;
			const char * aux = notify.ptr();
			while(*aux)
			{
				single = "";
				aux = kvi_extractToken(single,aux,' ');
				if(single.hasData())
				{
					m_pRegUserDict->replace(single.ptr(),new KviStr(u->name()));
				}
			}
		}
		++it;
	}	
}

void KviIsOnNotifyListManager::delayedNotifySession()
{
	unsigned int iTimeout = KVI_OPTION_UINT(KviOption_uintNotifyListCheckTimeInSecs);
	if(iTimeout < 15)
	{
		// life first of all.
		// don't allow the user to suicide
		m_pConsole->output(KVI_OUT_SYSTEMWARNING,
			__tr("The notify list timeout (%d secs) is really too small: resetting to something more reasonable (15 secs)"),
			iTimeout);
		iTimeout = 15;
		KVI_OPTION_UINT(KviOption_uintNotifyListCheckTimeInSecs) = 15;
	}
	m_pDelayedNotifyTimer->start(iTimeout * 1000,true);
}

void KviIsOnNotifyListManager::newNotifySession()
{
	buildNotifyList();
	if(m_pNotifyList->isEmpty())
	{
		if(KVI_OPTION_BOOL(KviOption_boolBeVerbose))
			m_pConsole->output(KVI_OUT_SYSTEMMESSAGE,__tr("Notify list: notify list empty: stopping"));
		stop();
		return;
	}
	newIsOnSession();
}

void KviIsOnNotifyListManager::buildNotifyList()
{
	m_pNotifyList->clear();
	QAsciiDictIterator<KviStr> it(*m_pRegUserDict);
	while(it.current())
	{
		m_pNotifyList->append(new KviStr(it.currentKey()));
		++it;
	}
}

void KviIsOnNotifyListManager::delayedIsOnSession()
{
	unsigned int iTimeout = KVI_OPTION_UINT(KviOption_uintNotifyListIsOnDelayTimeInSecs);
	if(iTimeout < 5)
	{
		// life first of all.
		// don't allow the user to suicide
		m_pConsole->output(KVI_OUT_SYSTEMWARNING,
			__tr("The notify list ISON delay (%d secs) is really too small: resetting to something more reasonable (5 secs)"),
			iTimeout);
		iTimeout = 5;
		KVI_OPTION_UINT(KviOption_uintNotifyListIsOnDelayTimeInSecs) = 5;
	}
	m_pDelayedIsOnTimer->start(iTimeout * 1000,true);
}

void KviIsOnNotifyListManager::newIsOnSession()
{
	buildIsOnList();
	if(m_pIsOnList->isEmpty())delayedNotifySession();
	else sendIsOn();
}

void KviIsOnNotifyListManager::buildIsOnList()
{
	m_pIsOnList->clear();
	m_szIsOnString = "";
	m_pNotifyList->setAutoDelete(false);
	while(KviStr * s = m_pNotifyList->first())
	{
		if(((m_szIsOnString.len() + s->len()) + 1) < 504)
		{
			if(m_szIsOnString.hasData())m_szIsOnString.append(' ');
			m_szIsOnString.append(*s);
			m_pIsOnList->append(s);
			m_pNotifyList->removeFirst();
		} else break;
	}
	m_pNotifyList->setAutoDelete(true);
}

void KviIsOnNotifyListManager::sendIsOn()
{
	if(KVI_OPTION_BOOL(KviOption_boolBeVerbose))
		m_pConsole->output(KVI_OUT_SYSTEMMESSAGE,__tr("Notify list: checking for: %s"),m_szIsOnString.ptr());
	m_pConsole->socket()->sendFmtData("ISON %s",m_szIsOnString.ptr());
	m_szIsOnString = "";
	m_bExpectingIsOn = true;
// FIXME: #warning "And if can't send ?"
}


bool KviIsOnNotifyListManager::handleIsOn(KviIrcMessage *msg)
{
	if(!m_bExpectingIsOn)return false;

	// Check if it is our ISON
	// all the nicks must be on the IsOnList

	KviPtrList<KviStr> tmplist;
	tmplist.setAutoDelete(false);

	KviStr nk;
	const char * aux = msg->trailing();

	while(*aux)
	{
		nk = "";
		aux = kvi_extractToken(nk,aux,' ');
		if(nk.hasData())
		{
			bool bGotIt = false;
			for(KviStr * s = m_pIsOnList->first();s && (!bGotIt);s = m_pIsOnList->next())
			{
				if(kvi_strEqualCI(s->ptr(),nk.ptr()))
				{
					tmplist.append(s);
					bGotIt = true;
				}
			}
			if(!bGotIt)
			{
				// ops...not my userhost!
				if(KVI_OPTION_BOOL(KviOption_boolBeVerbose))
					m_pConsole->output(KVI_OUT_SYSTEMMESSAGE,__tr("Notify list: hey! you've used ISON behind my back ? (I might be confused now...)"));
				return false;
			}
		}
	}

	// Ok...looks to be my ison (still not sure at 100% , but can't do better)

	m_bExpectingIsOn = false;

	m_pOnlineList->clear();

	m_pIsOnList->setAutoDelete(false);

	// Ok...we have an IsOn reply here
	// The nicks in the IsOnList that are also in the reply are online , and go to the OnlineList
	// the remaining in the IsOnList are offline

	KviStr * s;
	
	for(s = tmplist.first();s;s = tmplist.next())
	{
		m_pIsOnList->removeRef(s);
		m_pOnlineList->append(s);
	}

	m_pIsOnList->setAutoDelete(true);
	// Ok...all the users that are online , are on the OnlineList
	// the remaining users are in the m_pIsOnList , and are no longer online

	// first the easy step: remove the users that have just left irc or have never been online
	// we're clearling the m_pIsOnList
	while(KviStr * s = m_pIsOnList->first())
	{
		if(m_pConsole->notifyListView()->findEntry(s->ptr()))
		{
			// has just left IRC... make him part
			m_pConsole->notifyListView()->part(s->ptr());
			KviWindow * out = KVI_OPTION_BOOL(KviOption_boolNotifyListChangesToActiveWindow) ? m_pConsole->activeWindow() : m_pConsole;

			if(!TRIGGER_EVENT_1PARAM_RETVALUE(KviEvent_OnNotifyOffLine,out,s->ptr()))
			{
				out->output(KVI_OUT_NOTIFYOFFLINE,__tr("\r!n\r%s\r has left irc"),s->ptr());
			}
		} // else has never been here

		m_pIsOnList->removeFirst(); // autodelete is true
	}

	// ok... complex step now: the remaining users in the userhost list are online
	// if they have been online before, just remove them from the list
	// otherwise they must be matched for masks
	// and eventually inserted in the notify view later

	KviIrcUserDataBase * db = m_pConsole->userDataBase();

	KviPtrList<KviStr> l;
	l.setAutoDelete(false);

	for(s = m_pOnlineList->first();s;s = m_pOnlineList->next())
	{
		if(KviUserListEntry * ent = m_pConsole->notifyListView()->findEntry(s->ptr()))
		{
			// the user was online from a previous notify session
			// might the mask have been changed ? (heh...this is tricky, maybe too much even)
			if(KVI_OPTION_BOOL(KviOption_boolNotifyListSendUserhostForOnlineUsers))
			{
				// user wants to be sure about online users....
				// check if he is on some channels
				if(ent->globalData()->nRefs() > 1)
				{
					// mmmh...we have more than one ref , so the user is at least in one query or channel
					// look him up on channels , if we find his entry , we can be sure that he is
					// still the right user
					KviPtrList<KviChannel> * chlist = m_pConsole->connectionInfo()->pChannelList;
					for(KviChannel * ch = chlist->first();ch;ch = chlist->next())
					{
						if(KviUserListEntry * le = ch->findEntry(s->ptr()))
						{
							l.append(s); // ok...found on a channel...we don't need an userhost to match him
							KviIrcMask mk;
							mk.setNick(s->ptr());
							mk.setUsername(le->globalData()->user());
							mk.setHost(le->globalData()->host());
							if(!doMatchUser(s->ptr(),mk))return true; // critical problems = have to restart!!!
							break;
						}
					}
				} // else Only one ref...we need an userhost to be sure (don't remove from the list)
			} else {
				// user wants no userhost for online users...we "hope" that everything will go ok.
				l.append(s);
			}
			//l.append(s); // we will remove him from the list
		} else {
			// the user was not online!
			// check if we have a cached mask
			if(db)
			{
				if(KviIrcUserEntry * ue = db->find(s->ptr()))
				{
					// already in the db... do we have a mask ?
					if(ue->hasUser() && ue->hasHost())
					{
						// yup! we have a complete mask to match on
						//KviStr mask(KviStr::Format,"%s!%s@%s",s->ptr(),ue->user(),ue->host());
						KviIrcMask mk;
						mk.setNick(s->ptr());
						mk.setUsername(ue->user());
						mk.setHost(ue->host());
						// lookup the user's name in the m_pRegUserDict
						if(!doMatchUser(s->ptr(),mk))return true; // critical problems = have to restart!!!
						l.append(s); // remove anyway
					}
				}
			}
		}
	}

	for(s = l.first();s;s = l.next())
	{
		m_pOnlineList->removeRef(s); // autodelete is true
	}

	if(m_pOnlineList->isEmpty())
	{
		if(m_pNotifyList->isEmpty())delayedNotifySession();
		else delayedIsOnSession();
	} else delayedUserhostSession();

	return true;
}

// FIXME: #warning "Nickname escapes (links) in the notifylist messages!"

bool KviIsOnNotifyListManager::doMatchUser(const char * notifyString,const KviIrcMask & mask)
{
	KviStr * nam = m_pRegUserDict->find(notifyString);
	if(nam)
	{
		// ok...find the user
		if(KviRegisteredUser * u = g_pRegisteredUserDataBase->findUserByName(nam->ptr()))
		{
			// ok ... match the user
			if(u->matchesFixed(mask))
			{
				// new user online
				if(!(m_pConsole->notifyListView()->findEntry(mask.nick())))
				{
					m_pConsole->notifyListView()->join(mask.nick(),mask.user(),mask.host());
					KviWindow * out = KVI_OPTION_BOOL(KviOption_boolNotifyListChangesToActiveWindow) ? m_pConsole->activeWindow() : m_pConsole;
					if(!TRIGGER_EVENT_1PARAM_RETVALUE(KviEvent_OnNotifyOnLine,out,mask.nick()))
					{
						out->output(KVI_OUT_NOTIFYONLINE,__tr("\r!n\r%s\r (%s@\r!h\r%s\r) is on irc"),mask.nick(),mask.user(),mask.host());
					}
				} // else already online , and matching...all ok
			} else {
				// not matched.... has he been online before ?
				if(m_pConsole->notifyListView()->findEntry(mask.nick()))
				{
					// has been online just a sec ago , but now the mask does not match
					// either reguserdb has changed , or the user went offline and another one got his nick
					// in the meantime... (ugly situation anyway)
					m_pConsole->notifyListView()->part(mask.nick());
					KviWindow * out = KVI_OPTION_BOOL(KviOption_boolNotifyListChangesToActiveWindow) ? m_pConsole->activeWindow() : m_pConsole;
					if(!TRIGGER_EVENT_1PARAM_RETVALUE(KviEvent_OnNotifyOffLine,out,mask.nick()))
					{
						out->output(KVI_OUT_NOTIFYOFFLINE,__tr("\r!n\r%s\r (%s@\r!h\r%s\r) has left irc (registration mask changed or someone else is using his nick now)"),mask.nick(),mask.user(),mask.host());
					}
				} else {
					// has never been online
					if(KVI_OPTION_BOOL(KviOption_boolBeVerbose))
						m_pConsole->output(KVI_OUT_SYSTEMMESSAGE,__tr("Notify list: \r!n\r%s\r seems to be online , but the mask (%s@\r!h\r%s\r) does not match (someone else is using his nickname or your registration mask(s) do not match)"),mask.nick(),mask.user(),mask.host());
				}
			}
		} else {
			// ops... unexpected inconsistency .... reguser db modified ?
			m_pConsole->output(KVI_OUT_SYSTEMWARNING,__tr("Notify list: unexpected inconsistency: registered user db modified ? (restarting)"));
			stop();
			start();
			return false; // critical ... exit from the call stack
		}
	} else {
		// ops...unexpected inconsistency
		m_pConsole->output(KVI_OUT_SYSTEMWARNING,__tr("Notify list: unexpected inconsistency: thought to have \r!n\r%s\r in the reguser dictionary..."),notifyString);
	}
	return true;
}

void KviIsOnNotifyListManager::delayedUserhostSession()
{
	unsigned int iTimeout = KVI_OPTION_UINT(KviOption_uintNotifyListUserhostDelayTimeInSecs);
	if(iTimeout < 5)
	{
		// life first of all.
		// don't allow the user to suicide
		m_pConsole->output(KVI_OUT_SYSTEMWARNING,
			__tr("The notify list USERHOST delay (%d secs) is really too small: resetting to something more reasonable (5 secs)"),
			iTimeout);
		iTimeout = 5;
		KVI_OPTION_UINT(KviOption_uintNotifyListUserhostDelayTimeInSecs) = 5;
	}
	m_pDelayedUserhostTimer->start(iTimeout * 1000,true);
}

void KviIsOnNotifyListManager::newUserhostSession()
{
	buildUserhostList();
	if(m_pUserhostList->isEmpty())
	{
		// this is unexpected!
		m_pConsole->output(KVI_OUT_SYSTEMWARNING,__tr("Notify list: unexpected inconsistency: the userhost list is empty...mhhhhh"));
		if(m_pOnlineList->isEmpty())
		{
			if(m_pNotifyList->isEmpty())delayedNotifySession();
			else delayedIsOnSession();
		} else delayedUserhostSession();
		return;
	}
	sendUserhost();
}

#define MAX_USERHOST_ENTRIES 5

void KviIsOnNotifyListManager::buildUserhostList()
{
	m_szUserhostString = "";
	m_pUserhostList->clear();

	m_pOnlineList->setAutoDelete(false);
	int i = 0;
	KviStr * s;
	while((s = m_pOnlineList->first()) && (i < MAX_USERHOST_ENTRIES))
	{
		if(m_szUserhostString.hasData())m_szUserhostString.append(' ');
		m_szUserhostString.append(*s);
		m_pUserhostList->append(s);
		m_pOnlineList->removeFirst();
		i++;
	}
	m_pOnlineList->setAutoDelete(true);
}

void KviIsOnNotifyListManager::sendUserhost()
{
	if(KVI_OPTION_BOOL(KviOption_boolBeVerbose))
		m_pConsole->output(KVI_OUT_SYSTEMMESSAGE,__tr("Notify list: checking userhost for: %s"),m_szUserhostString.ptr());
	m_pConsole->socket()->sendFmtData("USERHOST %s",m_szUserhostString.ptr());
	m_szUserhostString = "";
	m_bExpectingUserhost = true;
// FIXME: #warning "And if can't send ?"
}

bool KviIsOnNotifyListManager::handleUserhost(KviIrcMessage *msg)
{
	if(!m_bExpectingUserhost)return false;
	// first check for consistency: all the replies must be on the USERHOST list
	KviPtrList<KviIrcMask> tmplist;
	tmplist.setAutoDelete(true);

	KviStr nk;
	const char * aux = msg->trailing();

	while(*aux)
	{
		nk = "";
		aux = kvi_extractToken(nk,aux,' ');
		if(nk.hasData())
		{
			// split it in a mask
			KviStr nick;
			KviStr user;
			KviStr host;

			int idx = nk.findFirstIdx('=');
			if(idx != -1)
			{
				nick = nk.left(idx);
				if(nick.lastCharIs('*'))nick.cutRight(1);
				nk.cutLeft(idx + 1);
				if(nk.firstCharIs('+') || nk.firstCharIs('-'))nk.cutLeft(1);

				idx = nk.findFirstIdx('@');
				if(idx != -1)
				{
					user = nk.left(idx);
					nk.cutLeft(idx + 1);
					host = nk;
				} else {
					user = "*";
					host = nk;
				}
	
				bool bGotIt = false;
				for(KviStr * s = m_pUserhostList->first();s && (!bGotIt);s = m_pUserhostList->next())
				{
					if(kvi_strEqualCI(s->ptr(),nick.ptr()))
					{
						KviIrcMask * mk = new KviIrcMask();
						mk->setNick(nick.ptr());
						mk->setUsername(user.ptr());
						mk->setHost(host.ptr());
	
						tmplist.append(mk);
						bGotIt = true;
						m_pUserhostList->removeRef(s);
					}
				}
	
				if(!bGotIt)
				{
					// ops...not my userhost!
					if(KVI_OPTION_BOOL(KviOption_boolBeVerbose))
						m_pConsole->output(KVI_OUT_SYSTEMWARNING,__tr("Notify list: hey! you've used USERHOST behind my back ? (I might be confused now...)"));
					return false;
				}
			} else {
				if(KVI_OPTION_BOOL(KviOption_boolBeVerbose))
					m_pConsole->output(KVI_OUT_SYSTEMWARNING,__tr("Notify list: mmmh...got a broken USERHOST reply from the server ? (%s)"),nk.ptr());
			}
		}
	}

	// Ok...looks to be my usershot (still not sure at 100% , but can't do better)

	m_bExpectingUserhost = false;

	for(KviIrcMask * mk = tmplist.first();mk;mk = tmplist.next())
	{
		if(!doMatchUser(mk->nick(),*mk))return true; // have to restart!!!
	}

	if(!(m_pUserhostList->isEmpty()))
	{
		// ops...someone is no longer online ?
		while(KviStr * s = m_pUserhostList->first())
		{
			m_pConsole->output(KVI_OUT_SYSTEMMESSAGE,__tr("Notify list: It looks that \r!n\r%s\r went offline while awaiting his USERHOST reply, will recheck in the next loop"),s->ptr());
			m_pUserhostList->removeFirst();
		}
	
	}

	if(m_pOnlineList->isEmpty())
	{
		if(m_pNotifyList->isEmpty())delayedNotifySession();
		else delayedIsOnSession();
	} else delayedUserhostSession();

	return true;
}

void KviIsOnNotifyListManager::stop()
{
	if(!m_bRunning)return;
	m_pDelayedNotifyTimer->stop();
	m_pDelayedIsOnTimer->stop();
	m_pDelayedUserhostTimer->stop();
	m_pConsole->notifyListView()->partAll();
	m_pRegUserDict->clear();
	m_pNotifyList->clear();
	m_pIsOnList->clear();
	m_pOnlineList->clear();
	m_pUserhostList->clear();
	m_szIsOnString = "";
	m_szUserhostString = "";
	m_bRunning = false;
}


///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Stupid notify list manager
//
///////////////////////////////////////////////////////////////////////////////////////////////////

KviStupidNotifyListManager::KviStupidNotifyListManager(KviConsole * console)
: KviNotifyListManager(console)
{
	m_pNickList = new KviPtrList<KviStr>;
	m_pNickList->setAutoDelete(true);
	m_iRestartTimer = 0;
}

KviStupidNotifyListManager::~KviStupidNotifyListManager()
{
	if(m_iRestartTimer)
	{
		killTimer(m_iRestartTimer);
		m_iRestartTimer = 0;
	}
	delete m_pNickList;
}

void KviStupidNotifyListManager::start()
{
	if(m_iRestartTimer)
	{
		killTimer(m_iRestartTimer);
		m_iRestartTimer = 0;
	}
	if(KVI_OPTION_BOOL(KviOption_boolBeVerbose))
		m_pConsole->outputNoFmt(KVI_OUT_SYSTEMMESSAGE,__tr("Starting notify list"));
	buildNickList();
	if(m_pNickList->isEmpty())
	{
		if(KVI_OPTION_BOOL(KviOption_boolBeVerbose))
			m_pConsole->outputNoFmt(KVI_OUT_SYSTEMMESSAGE,__tr("No users in the notify list"));
		return; // Ok...no nicknames in the list
	}
	m_iNextNickToCheck = 0;
	m_pConsole->notifyListView()->partAll();
	sendIsOn();
}

void KviStupidNotifyListManager::sendIsOn()
{
	m_szLastIsOnMsg = "";
	KviStr * nick = m_pNickList->at(m_iNextNickToCheck);
	__range_valid(nick);

	int i = 0;
	while(nick && ((nick->len() + 5 + m_szLastIsOnMsg.len()) < 510))
	{
		m_szLastIsOnMsg.append(KviStr::Format," %s",nick->ptr());
		nick = m_pNickList->next();
		i++;
	}
	if(KVI_OPTION_BOOL(KviOption_boolBeVerbose))
		m_pConsole->output(KVI_OUT_SYSTEMMESSAGE,__tr("Notify list: checking for:%s"),m_szLastIsOnMsg.ptr());
	m_pConsole->socket()->sendFmtData("ISON%s",m_szLastIsOnMsg.ptr());
	m_iNextNickToCheck += i;
}

bool KviStupidNotifyListManager::handleIsOn(KviIrcMessage * msg)
{
	KviStr nk;
	const char * aux = msg->trailing();
	while(*aux)
	{
		nk = "";
		aux = kvi_extractToken(nk,aux,' ');
		if(nk.hasData())
		{
			KviStr nksp(KviStr::Format," %s",nk.ptr());
			m_szLastIsOnMsg.replaceAll(nksp.ptr(),"",false);
			if(!(m_pConsole->notifyListView()->findEntry(nk.ptr())))
			{
				// not yet notified
				m_pConsole->notifyListView()->join(nk.ptr());
				KviWindow * out = KVI_OPTION_BOOL(KviOption_boolNotifyListChangesToActiveWindow) ? m_pConsole->activeWindow() : m_pConsole;
				if(!TRIGGER_EVENT_1PARAM_RETVALUE(KviEvent_OnNotifyOnLine,out,nk.ptr()))
				{
					out->output(KVI_OUT_NOTIFYONLINE,__tr("\r!n\r%s\r is on irc"),nk.ptr());
				}
			}
		}
	}
	// ok...check the users that have left irc now...
	aux = m_szLastIsOnMsg.ptr();
	while(*aux)
	{
		nk = "";
		aux = kvi_extractToken(nk,aux,' ');
		if(nk.hasData())
		{
			if(m_pConsole->notifyListView()->findEntry(nk.ptr()))
			{
				// has just left irc
				m_pConsole->notifyListView()->part(nk.ptr());
				KviWindow * out = KVI_OPTION_BOOL(KviOption_boolNotifyListChangesToActiveWindow) ? m_pConsole->activeWindow() : m_pConsole;
				if(!TRIGGER_EVENT_1PARAM_RETVALUE(KviEvent_OnNotifyOffLine,out,nk.ptr()))
				{
					out->output(KVI_OUT_NOTIFYOFFLINE,__tr("\r!n\r%s\r has left irc"),nk.ptr());
				}
			} // else has never been here...
		}
	}

	if(((unsigned int)m_iNextNickToCheck) >= m_pNickList->count())
	{
		// have to restart
		unsigned int iTimeout = KVI_OPTION_UINT(KviOption_uintNotifyListCheckTimeInSecs);
		if(iTimeout < 5)
		{
			// life first of all.
			// don't allow the user to suicide
			m_pConsole->output(KVI_OUT_SYSTEMWARNING,
				__tr("The notify list timeout (%d secs) is really too small: resetting to something more reasonable (5 secs)"),
				iTimeout);
			iTimeout = 5;
			KVI_OPTION_UINT(KviOption_uintNotifyListCheckTimeInSecs) = 5;
		}
		m_iRestartTimer = startTimer(iTimeout * 1000);
	} else sendIsOn();
	return true;
}

void KviStupidNotifyListManager::timerEvent(QTimerEvent *e)
{
	if(e->timerId() == m_iRestartTimer)
	{
		killTimer(m_iRestartTimer);
		m_iRestartTimer = 0;
		m_iNextNickToCheck = 0;
		sendIsOn();
		return;
	}
	QObject::timerEvent(e);
}

void KviStupidNotifyListManager::stop()
{
	if(m_iRestartTimer)
	{
		killTimer(m_iRestartTimer);
		m_iRestartTimer = 0;
	}
	m_pConsole->notifyListView()->partAll();

	// The ISON Method needs no stopping
}

void KviStupidNotifyListManager::buildNickList()
{
	const QAsciiDict<KviRegisteredUser> * d = g_pRegisteredUserDataBase->userDict();
	QAsciiDictIterator<KviRegisteredUser> it(*d);
	m_pNickList->clear();
	while(it.current())
	{
		KviStr notify;
		if(it.current()->getProperty("notify",notify))
		{
			m_pNickList->append(new KviStr(notify));
		}
		++it;
	}
}

///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Watch notify list manager
//
///////////////////////////////////////////////////////////////////////////////////////////////////

KviWatchNotifyListManager::KviWatchNotifyListManager(KviConsole * console)
: KviNotifyListManager(console)
{
	m_pRegUserDict = new QAsciiDict<KviStr>(17,false,true);
	m_pRegUserDict->setAutoDelete(true);
}

KviWatchNotifyListManager::~KviWatchNotifyListManager()
{
	delete m_pRegUserDict;
}

void KviWatchNotifyListManager::buildRegUserDict()
{
	m_pRegUserDict->clear();

	const QAsciiDict<KviRegisteredUser> * d = g_pRegisteredUserDataBase->userDict();
	QAsciiDictIterator<KviRegisteredUser> it(*d);
	while(KviRegisteredUser * u = it.current())
	{
		KviStr notify;
		if(u->getProperty("notify",notify))
		{
			notify.stripWhiteSpace();
			KviStr single;
			const char * aux = notify.ptr();
			while(*aux)
			{
				single = "";
				aux = kvi_extractToken(single,aux,' ');
				if(single.hasData())
				{
					m_pRegUserDict->replace(single.ptr(),new KviStr(u->name()));
				}
			}
		}
		++it;
	}	
}

void KviWatchNotifyListManager::start()
{
	m_pConsole->notifyListView()->partAll();

	buildRegUserDict();

	KviStr watchStr;

	QAsciiDictIterator<KviStr> it(*m_pRegUserDict);
	while(it.current())
	{
		KviStr nk = it.currentKey();
		if(nk.findFirstIdx('*') == -1)
		{
			if((watchStr.len() + nk.len() + 2) > 501)
			{
				m_pConsole->socket()->sendFmtData("WATCH%s",watchStr.ptr());
				if(KVI_OPTION_BOOL(KviOption_boolBeVerbose))
					m_pConsole->output(KVI_OUT_SYSTEMMESSAGE,__tr("Notify list: adding watch entries for%s"),watchStr.ptr());
				watchStr = "";
			}
			watchStr.append(KviStr::Format," +%s",nk.ptr());
		}
		++it;
	}

	if(watchStr.hasData())
	{
		m_pConsole->socket()->sendFmtData("WATCH%s",watchStr.ptr());
		if(KVI_OPTION_BOOL(KviOption_boolBeVerbose))
			m_pConsole->output(KVI_OUT_SYSTEMMESSAGE,__tr("Notify list: adding watch entries for%s"),watchStr.ptr());
	}
}
void KviWatchNotifyListManager::stop()
{
	m_pConsole->notifyListView()->partAll();
	m_pConsole->socket()->sendFmtData("WATCH clear");
	m_pRegUserDict->clear();
}

bool KviWatchNotifyListManager::doMatchUser(KviIrcMessage * msg,const char * notifyString,const KviIrcMask & mask)
{
	KviStr * nam = m_pRegUserDict->find(notifyString);
	KviWindow * out = KVI_OPTION_BOOL(KviOption_boolNotifyListChangesToActiveWindow) ? m_pConsole->activeWindow() : m_pConsole;

	if(nam)
	{
		// ok...find the user
		if(KviRegisteredUser * u = g_pRegisteredUserDataBase->findUserByName(nam->ptr()))
		{
			// ok ... match the user
			if(u->matchesFixed(mask))
			{
				// new user online
				if(!(m_pConsole->notifyListView()->findEntry(mask.nick())))
				{
					m_pConsole->notifyListView()->join(mask.nick(),mask.user(),mask.host());
					if(!TRIGGER_EVENT_1PARAM_RETVALUE(KviEvent_OnNotifyOnLine,out,mask.nick()))
					{
						out->output(KVI_OUT_NOTIFYONLINE,__tr("\r!n\r%s\r (%s@\r!h\r%s\r) is on irc (watch)"),mask.nick(),mask.user(),mask.host());
					}
				} else {
					// else already online , and matching...all ok
					if(msg->numeric() == RPL_NOWON)
					{
						// This is a reply to a /watch +something (should not happen, unless the user is messing) or to /watch l (user requested)
						if(!TRIGGER_EVENT_1PARAM_RETVALUE(KviEvent_OnNotifyOnLine,out,mask.nick()))
						{
							out->output(KVI_OUT_NOTIFYONLINE,
								__tr("\r!n\r%s\r (%s@\r!h\r%s\r) is on irc (watch: user requested list)"),
								mask.nick(),mask.user(),mask.host());
						}
					} else {
						// This is a RPL_LOGON....we're desynched ?
						if(KVI_OPTION_BOOL(KviOption_boolBeVerbose))
						{
							if(!TRIGGER_EVENT_1PARAM_RETVALUE(KviEvent_OnNotifyOnLine,out,mask.nick()))
							{
								m_pConsole->output(KVI_OUT_SYSTEMMESSAGE,
									__tr("Notify list: \r!n\r%s\r (%s@\r!h\r%s\r) is on irc (watch: watch list desync ?)"),
									mask.nick(),mask.user(),mask.host());
							}
						}
					}
				}
			} else {
				// not matched.... has he been online before ?
				if(m_pConsole->notifyListView()->findEntry(mask.nick()))
				{
					// has been online just a sec ago , but now the mask does not match
					// prolly the reguserdb has been changed
					m_pConsole->notifyListView()->part(mask.nick());
					if(!TRIGGER_EVENT_1PARAM_RETVALUE(KviEvent_OnNotifyOffLine,out,mask.nick()))
					{
						out->output(KVI_OUT_NOTIFYOFFLINE,
							__tr("\r!n\r%s\r (%s@\r!h\r%s\r) has left irc (watch: registration mask changed or desync with the watch service)"),
							mask.nick(),mask.user(),mask.host());
					}
				} else {
					// has never been online
					if(KVI_OPTION_BOOL(KviOption_boolBeVerbose))
						m_pConsole->output(KVI_OUT_SYSTEMMESSAGE,
							__tr("Notify list: \r!n\r%s\r seems to be online , but the mask (%s@\r!h\r%s\r) does not match (watch: someone else is using his nickname or your registration mask(s) do not match)"),
							mask.nick(),mask.user(),mask.host());
				}
			}
		} else {
			// ops... unexpected inconsistency .... reguser db modified ?
			m_pConsole->output(KVI_OUT_SYSTEMWARNING,
				__tr("Notify list: unexpected inconsistency: registered user db modified ? (watch: restarting)"));
			stop();
			start();
			return false; // critical ... exit from the call stack
		}
	} else {
		// not in our dictionary
		// prolly someone used /WATCH behind our back... bad boy!
		if(!(m_pConsole->notifyListView()->findEntry(mask.nick())))
		{
			m_pConsole->notifyListView()->join(mask.nick(),mask.user(),mask.host());
		}
		if(!TRIGGER_EVENT_1PARAM_RETVALUE(KviEvent_OnNotifyOnLine,out,mask.nick()))
		{
			out->output(KVI_OUT_NOTIFYONLINE,__tr("\r!n\r%s\r (%s@\r!h\r%s\r) is on irc (user-added watch)"),mask.nick(),mask.user(),mask.host());
		}
	}
	return true;
}

// FIXME: #warning "DEDICATED WATCH LIST VERBOSITY FLAG ? (To allow the user to use /WATCH l and manual /WATCH)"

bool KviWatchNotifyListManager::handleWatchReply(KviIrcMessage *msg)
{
	// 600: RPL_LOGON
	// :prefix 600 <target> <nick> <user> <host> <logintime> :logged online
	// 601: RPL_LOGON
	// :prefix 601 <target> <nick> <user> <host> <logintime> :logged offline
	// 604: PRL_NOWON
	// :prefix 604 <target> <nick> <user> <host> <logintime> :is online
	// 605: PRL_NOWOFF
	// :prefix 605 <target> <nick> <user> <host> 0 :is offline

// FIXME: #warning "Use the logintime in some way ?"

	const char * nk = msg->safeParam(1);
	const char * us = msg->safeParam(2);
	const char * ho = msg->safeParam(3);

	if((msg->numeric() == RPL_LOGON) || (msg->numeric() == RPL_NOWON))
	{
		KviIrcMask m;
		m.setNick(nk);
		m.setUsername(us);
		m.setHost(ho);

		doMatchUser(msg,nk,m);

		return true;

	} else if(msg->numeric() == RPL_WATCHOFF)
	{
		if(m_pConsole->notifyListView()->findEntry(nk))
		{
			m_pConsole->notifyListView()->part(nk);
			KviWindow * out = KVI_OPTION_BOOL(KviOption_boolNotifyListChangesToActiveWindow) ? m_pConsole->activeWindow() : m_pConsole;
			if(!TRIGGER_EVENT_1PARAM_RETVALUE(KviEvent_OnNotifyOffLine,out,nk))
			{
				out->output(KVI_OUT_NOTIFYOFFLINE,__tr("\r!n\r%s\r (%s@\r!h\r%s\r) has left irc (watch: stopped watching)"),nk,us,ho);
			}
		} else {
			if(KVI_OPTION_BOOL(KviOption_boolBeVerbose))
				m_pConsole->output(KVI_OUT_SYSTEMMESSAGE,__tr("Notify list: stopped watching for \r!n\r%s\r"),nk);
		}
		if(m_pRegUserDict->find(nk))m_pRegUserDict->remove(nk); // kill that

		return true;

	} else if((msg->numeric() == RPL_LOGOFF) || (msg->numeric() == RPL_NOWOFF))
	{
		if(m_pConsole->notifyListView()->findEntry(nk))
		{
			m_pConsole->notifyListView()->part(nk);
			KviWindow * out = KVI_OPTION_BOOL(KviOption_boolNotifyListChangesToActiveWindow) ? m_pConsole->activeWindow() : m_pConsole;
			if(!TRIGGER_EVENT_1PARAM_RETVALUE(KviEvent_OnNotifyOffLine,out,nk))
			{
				out->output(KVI_OUT_NOTIFYOFFLINE,__tr("\r!n\r%s\r (%s@\r!h\r%s\r) has left irc (watch)"),nk,us,ho);
			}
		} else {
			if(KVI_OPTION_BOOL(KviOption_boolBeVerbose))
			{
				if(msg->numeric() == RPL_NOWOFF)
				{
					// This is a reply to a /watch +something
					m_pConsole->output(KVI_OUT_SYSTEMMESSAGE,__tr("Notify list: \r!n\r%s\r is offline (watch)"),nk);
				} else {
					// This is a RPL_LOGOFF for an user that has not matched the reg-mask
					m_pConsole->output(KVI_OUT_SYSTEMMESSAGE,__tr("Notify list: \r!n\r%s\r (%s@\r!h\r%s\r) has left irc (watch: unmatched notify list entry)"),nk,us,ho);
				}
			}
		}
		return true;
	}

	return false;
}


#include "kvi_notifylist.moc"
