//
//   File : kvi_modulemanager.cpp
//   Creation date : Sat Aug 12 2000 20:32:11 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

//#warning "This has to be rewritten for Windoze"

#define __KVIRC__

#define _KVI_MODULEMANAGER_CPP_

#include "kvi_modulemanager.h"
#include "kvi_fileutils.h"
#include "kvi_app.h"
#include "kvi_options.h"
#include "kvi_frame.h"
#include "kvi_console.h"
#include "kvi_locale.h"
#include "kvi_out.h"

#include "kvi_library.h"

#include <qdir.h>

KviModuleManager * g_pModuleManager = 0;


KviModuleManager::KviModuleManager()
{
	m_pModuleDict = new QAsciiDict<KviModule>(17,false);
	m_pModuleDict->setAutoDelete(false);
	m_pCleanupTimer = new QTimer(this);
	connect(m_pCleanupTimer,SIGNAL(timeout()),this,SLOT(cleanupUnusedModules()));
}

KviModuleManager::~KviModuleManager()
{
	unloadAllModules();
	delete m_pModuleDict;
	delete m_pCleanupTimer;
}

void KviModuleManager::loadModulesByCaps(const char * caps,const char * dir)
{
	KviStr szCapsPath(KviStr::Format,"%s%ccaps%c%s%c",dir,KVI_PATH_SEPARATOR_CHAR,
			KVI_PATH_SEPARATOR_CHAR,caps,KVI_PATH_SEPARATOR_CHAR);

	QDir d(szCapsPath.ptr());

	// FIXME: maybe check timestamps ? (old modules)

	QStringList sl = d.entryList(QDir::Files | QDir::Readable | QDir::NoSymLinks);
	for(QStringList::Iterator it = sl.begin();it != sl.end();++it)
	{
		KviStr modname = *it;
		modname.cutToLast(KVI_PATH_SEPARATOR_CHAR);
		getModule((*it).ascii());
	}
}

void KviModuleManager::loadModulesByCaps(const char * caps)
{
	KviStr szDir;
	g_pApp->getLocalKvircDirectory(szDir,KviApp::Plugins);
	loadModulesByCaps(caps,szDir.ptr());
	g_pApp->getGlobalKvircDirectory(szDir,KviApp::Plugins);
	loadModulesByCaps(caps,szDir.ptr());
}

void KviModuleManager::completeModuleNames(const KviStr &path,const KviStr &word,KviPtrList<KviStr> * matches)
{
	QDir d(path.ptr());
#ifdef COMPILE_ON_WINDOWS
	d.setNameFilter("kvi*.dll");
#else
	d.setNameFilter("libkvi*.so");
#endif
	// FIXME: maybe check timestamps ? (old modules)

	QStringList sl = d.entryList(QDir::Files | QDir::Readable | QDir::NoSymLinks);
	for(QStringList::Iterator it = sl.begin();it != sl.end();++it)
	{
		KviStr * modname = new KviStr(*it);
		modname->cutToLast(KVI_PATH_SEPARATOR_CHAR);
		modname->cutToFirst("kvi");
		if(kvi_strEqualCIN(word.ptr(),modname->ptr(),word.len()))
		{
			modname->cutFromLast(".so");
			matches->append(modname);
		} else delete modname;
	}
}

void KviModuleManager::completeModuleNames(const KviStr &word,KviPtrList<KviStr> * matches)
{
	KviStr szDir;
	g_pApp->getLocalKvircDirectory(szDir,KviApp::Plugins);
	completeModuleNames(szDir,word,matches);
	g_pApp->getGlobalKvircDirectory(szDir,KviApp::Plugins);
	completeModuleNames(szDir,word,matches);
}

KviModule * KviModuleManager::findModule(const char * modName)
{
	KviModule * m = m_pModuleDict->find(modName);
	if(m)m->updateAccessTime();
	return m;
}

KviModule * KviModuleManager::getModule(const char * modName)
{
	KviModule * m = m_pModuleDict->find(modName);
	if(!m)
	{
		loadModule(modName);
		m = m_pModuleDict->find(modName);
	}
	if(m)m->updateAccessTime();
	return m;
}

static bool default_module_cmd_load(KviModule *,KviCommand *)
{
	return true;
}

static bool default_module_cmd_unload(KviModule *m,KviCommand *)
{
	g_pModuleManager->unloadModule(m->name());
	return true;
}

bool KviModuleManager::loadModule(const char * modName)
{
	if(findModule(modName))
	{
		//debug("MODULE %s ALREADY IN CORE MEMORY",modName);
		return true;
	}
	KviStr tmp;
#ifdef COMPILE_ON_WINDOWS
	KviStr szName(KviStr::Format,"kvi%s.dll",modName);
#else
	KviStr szName(KviStr::Format,"libkvi%s.so",modName);
#endif
	szName.toLower();

	g_pApp->getLocalKvircDirectory(tmp,KviApp::Plugins,szName.ptr());
	if(!kvi_fileExists(tmp.ptr()))
	{
		g_pApp->getGlobalKvircDirectory(tmp,KviApp::Plugins,szName.ptr());
	}

	kvi_library_t handle = kvi_library_open(tmp.ptr());
	if(!handle)
	{
		m_szLastError = kvi_library_error();
		//debug("ERROR IN LOADING MODULE %s (%s): %s",modName,szName.ptr(),kvi_library_error());
		return false;
	}
	KviModuleInfo * info = (KviModuleInfo *)kvi_library_symbol(handle,"kvirc_module_info");
	if(!info)
	{
		m_szLastError = __tr("No kvirc_module_info symbol exported: not a kvirc module ?");
		//debug("ERROR IN LOADING MODULE %s (%s): No kvirc_module_info symbol exported",modName,szName.ptr());
		kvi_library_close(handle);
		return false;
	}
	KviModule * module = new KviModule(handle,info,modName,szName.ptr());

	if(info->init_routine)
	{
		if(!((info->init_routine)(module)))
		{
			m_szLastError = __tr("Failed to execute the init routine");
			//debug("ERROR IN LOADING MODULE %s (%s): failed to execute the init routine",modName,szName.ptr());
			kvi_library_close(handle);
			delete module;
			return false;
		}
	}
	m_pModuleDict->insert(modName,module);

	registerDefaultCommands(module);

	if(KVI_OPTION_BOOL(KviOption_boolCleanupUnusedModules))
	{
		if(!m_pCleanupTimer->isActive())
		{
			if(KVI_OPTION_UINT(KviOption_uintModuleCleanupTimerInterval) < 30)
				KVI_OPTION_UINT(KviOption_uintModuleCleanupTimerInterval) = 30;
			m_pCleanupTimer->start(KVI_OPTION_UINT(KviOption_uintModuleCleanupTimerInterval) * 1000);
		}
	}
	// be verbose if needed....just make sure that we're not shutting down...
	if(KVI_OPTION_BOOL(KviOption_boolBeVerbose))
	{
		KviFrame * frm = g_pApp->activeFrame();
		if(frm)frm->firstConsole()->output(KVI_OUT_SYSTEMMESSAGE,
			__tr("Loaded module '%s' (%s)"),modName,szName.ptr());
	}
	return true;
}

void KviModuleManager::registerDefaultCommands(KviModule * module)
{
	// Register the default commands
// FIXME: #warning "version and help ?"
	module->registerCommand("load",default_module_cmd_load);
	module->registerCommand("unload",default_module_cmd_unload);
}

bool KviModuleManager::unloadModule(const char * modName)
{
	return unloadModule(findModule(modName));
}

bool KviModuleManager::unloadModule(KviModule * module)
{
	if(!module)return false;
	if(module->moduleInfo()->cleanup_routine)
	{
		(module->moduleInfo()->cleanup_routine)(module);
	}
	KviStr szModName = module->name();
	kvi_library_close(module->handle());
	m_pModuleDict->remove(szModName.ptr());
	delete module;

	if(m_pModuleDict->isEmpty())
	{
		if(m_pCleanupTimer->isActive())m_pCleanupTimer->stop();
	}

	if(KVI_OPTION_BOOL(KviOption_boolBeVerbose) && !g_pApp->closingDown())
	{
		KviFrame * frm = g_pApp->activeFrame();
		if(frm)frm->firstConsole()->output(KVI_OUT_SYSTEMMESSAGE,
			__tr("Unloaded module '%s'"),szModName.ptr());
	}
	return true;
}

void KviModuleManager::cleanupUnusedModules()
{
	QAsciiDictIterator<KviModule> it(*m_pModuleDict);
	while(it.current())
	{
		if(it.current()->secondsSinceLastAccess() > KVI_OPTION_UINT(KviOption_uintModuleCleanupTimeout))
		{
			if(it.current()->moduleInfo()->can_unload)
			{
				if((it.current()->moduleInfo()->can_unload)(it.current()))
				{
					unloadModule(it.current());
					continue;
				}
			} else {
				if(!(it.current()->isLocked()))
				{
					unloadModule(it.current());
					continue;
				}
			}
		}
		++it;
	}
}

void KviModuleManager::unloadAllModules()
{
	QAsciiDictIterator<KviModule> it(*m_pModuleDict);
	while(it.current())unloadModule(it.current());
}

#include "kvi_modulemanager.moc"
