//
//   File : kvi_ircsocket.cpp
//   Creation date : Wed Jul 26 2000 20:36:58 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#define __KVIRC__

#include "kvi_ircsocket.h"
#include "kvi_ircserver.h"
#include "kvi_proxydb.h"
#include "kvi_netutils.h"
#include "kvi_settings.h"
#include "kvi_error.h"
#include "kvi_locale.h"
#include "kvi_malloc.h"
#include "kvi_debug.h"
#include "kvi_string.h"
#include "kvi_channel.h"
#include "kvi_query.h"
#include "kvi_console.h"
#include "kvi_options.h"
#include "kvi_memmove.h"
#include "kvi_sparser.h"
#include "kvi_socket.h"

#ifdef COMPILE_SSL_SUPPORT
	#include "kvi_sslmaster.h"
#endif

#include <qtimer.h>
#include <qsocketnotifier.h>

#ifndef COMPILE_ON_WINDOWS
	#include <unistd.h> //for gettimeofday()
#endif
//#include <fcntl.h>
//#include <errno.h>


// FIXME: #warning "Lag-o-meter"

KviIrcSocketMonitor::KviIrcSocketMonitor(KviConsole * lpConsole)
{
	m_pMyConsole = lpConsole;
	m_pMyConsole->registerSocketMonitor(this);
}

KviIrcSocketMonitor::~KviIrcSocketMonitor()
{
	m_pMyConsole->unregisterSocketMonitor(this);
}

KviIrcSocket::KviIrcSocket(KviConsole * console,const char * name)
: QObject(0,name)
{
	m_pConsole       = console;      // parent console (and irc context)

	m_state          = Ready;        // current socket state

	m_pRsn           = 0;            // read socket notifier
	m_pWsn           = 0;            // write socket notifier
	m_sock           = KVI_INVALID_SOCKET;  // socket

	m_pIrcServer     = 0;            // current server data
	m_pProxy         = 0;            // current proxy data

	m_pTimeoutTimer  = 0;            // timeout for connect()

	m_pReadBuffer    = 0;            // incoming data buffer
	m_uReadBufferLen = 0;            // incoming data buffer length

	m_uReadBytes     = 0;            // total read bytes per session
	m_uSentBytes     = 0;            // total sent bytes per session
	m_uReadPackets   = 0;            // total packets read per session
	m_uSentPackets   = 0;            // total packets sent per session

	m_pSendQueueHead = 0;            // data queue
	m_pSendQueueTail = 0;            //

#ifdef COMPILE_SSL_SUPPORT
	m_pSSL           = 0;
#endif

	m_tAntiFloodLastMessageTime.tv_sec = 0;
	m_tAntiFloodLastMessageTime.tv_usec = 0;

	if(KVI_OPTION_UINT(KviOption_uintSocketQueueFlushTimeout) < 100)
		KVI_OPTION_UINT(KviOption_uintSocketQueueFlushTimeout) = 100; // this is our minimum , we don't want to lag the app

	m_pFlushTimer    = new QTimer(); // queue flush timer
	connect(m_pFlushTimer,SIGNAL(timeout()),this,SLOT(flushSendQueue()));
}

KviIrcSocket::~KviIrcSocket()
{
	reset();
	delete m_pFlushTimer;
}

void KviIrcSocket::reset()
{
#ifdef COMPILE_SSL_SUPPORT
	if(m_pSSL)
	{
		KviSSLMaster::freeSSL(m_pSSL);
		m_pSSL = 0;
	}
#endif
	if(m_pIrcServer)
	{
		delete m_pIrcServer;
		m_pIrcServer = 0;
	}
	if(m_pProxy)
	{
		delete m_pProxy;
		m_pProxy = 0;
	}
	if(m_pRsn)
	{
		delete m_pRsn;
		m_pRsn = 0;
	}
	if(m_pWsn)
	{
		delete m_pWsn;
		m_pWsn = 0;
	}
	if(m_sock > -1)
	{
		kvi_socket_destroy(m_sock);
		m_sock = KVI_INVALID_SOCKET;
	}
	if(m_pTimeoutTimer)
	{
		m_pTimeoutTimer->stop();
		delete m_pTimeoutTimer;
		m_pTimeoutTimer = 0;
	}
	if(m_pReadBuffer)
	{
		kvi_free(m_pReadBuffer);
		m_pReadBuffer    = 0;
		m_uReadBufferLen = 0;
	}
	m_uReadBytes                 = 0;
	m_uSentBytes                 = 0;
	m_uReadPackets               = 0;
	m_uSentPackets               = 0;
	m_tAntiFloodLastMessageTime.tv_sec = 0;
	m_tAntiFloodLastMessageTime.tv_usec = 0;

	if(m_pFlushTimer->isActive())m_pFlushTimer->stop();
	queue_removeAllMessages();

	setState(Ready);
}


#ifdef COMPILE_SSL_SUPPORT

void KviIrcSocket::raiseSSLError()
{
/*	// FIXME: more verbose error ? (ERR_lib_error_string ERR_func_error_string)
	unsigned long uSSLErr = ERR_get_error();
	while(uSSLErr != 0)
	{
		const char * theError = ERR_reason_error_string(uSSLErr);
		m_pConsole->socketEvent(SSLError,(void *)(theError ? theError : __tr("unknown error")));
		uSSLErr = ERR_get_error();
	}
*/
	KviStr buffer;
	while(m_pSSL->getLastErrorString(buffer))
	{
		m_pConsole->socketEvent(SSLError,(void *)(buffer.ptr()));
	}
}

#endif

int KviIrcSocket::connectToIrcHost(KviIrcServer *srv,KviProxy * prx,const char * bindAddress)
{
	// Attempts to estabilish an IRC connection
	// to the server specified by *srv.
	// Uses the proxy *prx if not 0
	if(m_state != Ready)return KviError_anotherConnectionInProgress;
	// Coherent state, thnx.
	reset();

	if(srv->useSSL())
	{
#ifndef COMPILE_SSL_SUPPORT
		return KviError_noSSLSupport;
#endif //!COMPILE_SSL_SUPPORT
	}

	// Copy the server
	m_pIrcServer = new KviIrcServer(*srv);

	bool bTargetIpV6 = false;


	// We're going to check the addresses now

	// check the proxy stuff...
	if(prx)
	{
		// Yeah...here comes the proxy
		m_pProxy = new KviProxy(*prx);
		// check the proxy IP address
		if(m_pProxy->isIpV6())
		{
			// IpV6 proxy :) (STILL QUITE UNTESTED...)
#ifdef COMPILE_IPV6_SUPPORT
			bTargetIpV6 = true;
			if(!kvi_isValidStringIp_V6(m_pProxy->ip()))return KviError_invalidProxyAddress;
			// SOCKSV4 does not support IPV6 addresses
			if(m_pProxy->protocol() == KviProxy::Socks4)return KviError_socksV4LacksIpV6Support;
#else
			return KviError_noIpV6Support;
#endif
		} else {
			// IpV4 proxy
			if(!kvi_isValidStringIp(m_pProxy->ip()))return KviError_invalidProxyAddress;
		}
	}

	// check the irc host ip
#ifdef COMPILE_IPV6_SUPPORT
	if(m_pIrcServer->isIpV6())
	{
		// We have an IpV6 server host (Interesting if proxy is IpV4)
		if(!kvi_isValidStringIp_V6(m_pIrcServer->ip()))return KviError_invalidIpAddress;
		bTargetIpV6 = true;
	} else {
#endif
		// We have an IpV4 server host
		if(!kvi_isValidStringIp(m_pIrcServer->ip()))return KviError_invalidIpAddress;
#ifdef COMPILE_IPV6_SUPPORT
	}
#endif

	KviSockaddr sa(prx ? m_pProxy->ip() : m_pIrcServer->ip(),prx ? m_pProxy->port() : m_pIrcServer->port(),bTargetIpV6);

	if(!sa.socketAddress())return KviError_invalidIpAddress;

	// create the socket
#ifdef COMPILE_IPV6_SUPPORT
	m_sock = kvi_socket_create(bTargetIpV6 ? KVI_SOCKET_PF_INET6 : KVI_SOCKET_PF_INET,KVI_SOCKET_TYPE_STREAM,KVI_SOCKET_PROTO_TCP);
#else
	m_sock = kvi_socket_create(KVI_SOCKET_PF_INET,KVI_SOCKET_TYPE_STREAM,KVI_SOCKET_PROTO_TCP);
#endif

	if(m_sock < 0)return KviError_socketCreationFailed;


	if(bindAddress)
	{
		// we have to bind the socket to a local address

		KviSockaddr localSa(bindAddress,0,bTargetIpV6);

		bool bBindOk = localSa.socketAddress();

		if(bBindOk)
		{
			bBindOk = kvi_socket_bind(m_sock,localSa.socketAddress(),localSa.addressLength());
		}

		if(bBindOk)
		{
			KviStr tmp(KviStr::Format,__tr("Binding to local address %s"),bindAddress);
			m_pConsole->socketEvent(Message,(void *)(tmp.ptr()));
		} else {
			KviStr tmp(KviStr::Format,__tr("Binding to local address %s failed: the kernel will choose the correct interface"),bindAddress);
			m_pConsole->socketEvent(Warning,(void *)(tmp.ptr()));
		}
	}


	// make it non blocking
	if(!kvi_socket_setNonBlocking(m_sock))
	{
		reset();
		return KviError_asyncSocketFailed;
	}

	if(!kvi_socket_connect(m_sock,sa.socketAddress(),sa.addressLength()))
	{

		//debug("Ops...can't connect");
		int err = kvi_socket_error();

		//debug("Error = %d (%d) (%d,%d,%d)",err,WSAEFAULT,size,sizeof(struct sockaddr),sizeof(struct sockaddr_storage));

		if(!kvi_socket_recoverableConnectError(err))
		{
			// Ops...
			int sockError=err;
			if(sockError==0)
			{
				// Zero error ?...let's look closer
				int iSize=sizeof(int);
				if(!kvi_socket_getsockopt(m_sock,SOL_SOCKET,SO_ERROR,
						(void *)&sockError,&iSize))sockError=0;
			}
			// Die
			reset();
			// And declare problems :)
			if(sockError)return kvi_errorFromSystemError(sockError);
			else return KviError_unknownError; //Error 0 ?
		}
	}

	// and setup the WRITE notifier...
	m_pWsn = new QSocketNotifier((int)m_sock,QSocketNotifier::Write);

	QObject::connect(m_pWsn,SIGNAL(activated(int)),this,SLOT(writeNotifierFired(int)));

	m_pWsn->setEnabled(true);
	// set the timer
	if(KVI_OPTION_UINT(KviOption_uintIrcSocketTimeout) < 5)
		KVI_OPTION_UINT(KviOption_uintIrcSocketTimeout) = 5;

	m_pTimeoutTimer = new QTimer();
	QObject::connect(m_pTimeoutTimer,SIGNAL(timeout()),this,SLOT(connectionTimedOut()));
	m_pTimeoutTimer->start(KVI_OPTION_UINT(KviOption_uintIrcSocketTimeout) * 1000,true);
	// and wait for connect
	setState(Connecting);

	return KviError_success;
}

void KviIrcSocket::connectionTimedOut()
{
	m_pConsole->socketEvent(Error,(void *)KviError_connectionTimedOut);
	reset();
}

void KviIrcSocket::writeNotifierFired(int)
{
	if(m_pTimeoutTimer)
	{
		delete m_pTimeoutTimer;
		m_pTimeoutTimer = 0;
	}

	// Check for errors...
	int sockError;
// FIXME: #warning "CHECK_SOCKLEN_T"
	int iSize=sizeof(int);
	if(!kvi_socket_getsockopt(m_sock,SOL_SOCKET,SO_ERROR,(void *)&sockError,&iSize))sockError = -1;
	//sockError = 0;
	if(sockError != 0)
	{
		//debug("Failed here %d",sockError);
		//failed
		if(sockError > 0)sockError = kvi_errorFromSystemError(sockError);
		else sockError = KviError_unknownError; //Error 0 ?
		m_pConsole->socketEvent(Error,(void *)sockError);
		reset();
		return;
	}

	delete m_pWsn;
	m_pWsn = 0;

	//Succesfully connected...
#ifdef COMPILE_SSL_SUPPORT
	if(m_pIrcServer->useSSL())
	{
		m_pSSL = KviSSLMaster::allocSSL(m_pConsole,m_sock,KviSSL::Client);
		if(!m_pSSL)
		{
			raiseSSLError();
			m_pConsole->socketEvent(Error,(void *)KviError_SSLError);
			reset();
			return;
		}
		setState(SSLHandshake);
		doSSLHandshake(0);
	} else {
#endif
		connectionEstabilished();
#ifdef COMPILE_SSL_SUPPORT
	}
#endif
}

#ifdef COMPILE_SSL_SUPPORT
void KviIrcSocket::printSSLPeerCertificate()
{
	KviSSLCertificate * c = m_pSSL->getPeerCertificate();
	if(c)
	{
		m_pConsole->socketEvent(SSLCertificate,(void *)c);
		delete c;
	} else {
		m_pConsole->socketEvent(SSLMessage,(void *)__tr("The server didn't provide a certificate"));
	}
}

void KviIrcSocket::printSSLCipherInfo()
{
	KviSSLCipherInfo * ci = m_pSSL->getCurrentCipherInfo();
	if(ci)
	{
		m_pConsole->socketEvent(SSLCipherInfo,(void *)ci);
		delete ci;
	} else {
		m_pConsole->socketEvent(SSLMessage,(void *)__tr("Unable to determine the current cipher"));
	}
}

#endif

void KviIrcSocket::doSSLHandshake(int)
{
#ifdef COMPILE_SSL_SUPPORT
	__range_valid(m_pSSL);

	if(m_pRsn)
	{
		delete m_pRsn;
		m_pRsn = 0;
	}
	if(m_pWsn)
	{
		delete m_pWsn;
		m_pWsn = 0;
	}

	if(!m_pSSL)
	{
		debug("Ops... I've lost the SSL class ?");
		reset();
		return; // ops ?
	}

	switch(m_pSSL->connect())
	{
		case KviSSL::Success:
			// done!
			printSSLCipherInfo();
			printSSLPeerCertificate();
			connectionEstabilished();
		break;
		case KviSSL::WantRead:
			m_pRsn = new QSocketNotifier((int)m_sock,QSocketNotifier::Read);
			QObject::connect(m_pRsn,SIGNAL(activated(int)),this,SLOT(doSSLHandshake(int)));
			m_pRsn->setEnabled(true);
		break;
		case KviSSL::WantWrite:
			m_pWsn = new QSocketNotifier((int)m_sock,QSocketNotifier::Write);
			QObject::connect(m_pWsn,SIGNAL(activated(int)),this,SLOT(doSSLHandshake(int)));
			m_pWsn->setEnabled(true);
		break;
		case KviSSL::RemoteEndClosedConnection:
			m_pConsole->socketEvent(Error,(void *)KviError_remoteEndClosedConnection);
			reset();
		break;
		case KviSSL::SSLError:
			raiseSSLError();
			m_pConsole->socketEvent(Error,(void *)KviError_SSLError);
			reset();
		break;
		case KviSSL::SyscallError:
		{
			// syscall problem
			int err = kvi_socket_error();
			if(!kvi_socket_recoverableError(err))
			{
				// Declare problems :)
				m_pConsole->socketEvent(Error,(void *)(err ? kvi_errorFromSystemError(err) : KviError_unknownError));
			} else {
				// can recover ? (EAGAIN , EINTR ?)
				m_pWsn = new QSocketNotifier((int)m_sock,QSocketNotifier::Write);
				QObject::connect(m_pWsn,SIGNAL(activated(int)),this,SLOT(doSSLHandshake(int)));
				m_pWsn->setEnabled(true);
				return;
			}
			reset();
		}
		break;
		default:
			m_pConsole->socketEvent(Error,(void *)KviError_SSLError);
			reset();
		break;
	}

/*
	int ret = SSL_connect(m_pSSL);

	switch(SSL_get_error(m_pSSL,ret))
	{
		case SSL_ERROR_NONE:
			// done!
			printSSLCipherInfo();
			printSSLPeerCertificate();
			connectionEstabilished();
		break;
		case SSL_ERROR_WANT_READ:
			m_pRsn = new QSocketNotifier((int)m_sock,QSocketNotifier::Read);
			QObject::connect(m_pRsn,SIGNAL(activated(int)),this,SLOT(doSSLHandshake(int)));
			m_pRsn->setEnabled(true);
		break;
		case SSL_ERROR_WANT_WRITE:
			m_pWsn = new QSocketNotifier((int)m_sock,QSocketNotifier::Write);
			QObject::connect(m_pWsn,SIGNAL(activated(int)),this,SLOT(doSSLHandshake(int)));
			m_pWsn->setEnabled(true);
		break;
		case SSL_ERROR_ZERO_RETURN:
			m_pConsole->socketEvent(Error,(void *)KviError_remoteEndClosedConnection);
			reset();
		break;
		case SSL_ERROR_WANT_X509_LOOKUP:
			m_pConsole->socketEvent(Error,(void *)KviError_SSLError);
			reset();
		break;
		case SSL_ERROR_SYSCALL:
		{
			int errcode = ERR_peek_error();
			if(errcode != 0)
			{
				raiseSSLError();
				m_pConsole->socketEvent(Error,(void *)KviError_SSLError);
			} else {
				if(ret == 0)
				{
					m_pConsole->socketEvent(Error,(void *)KviError_remoteEndClosedConnection);
				} else {
					// syscall problem
					int err = kvi_socket_error();
					if(!kvi_socket_recoverableConnectError(err))
					{
						// Ops...
						int sockError=err;
						if(sockError==0)
						{
							// Zero error ?...let's look closer
							int iSize=sizeof(int);
							if(!kvi_socket_getsockopt(m_sock,SOL_SOCKET,SO_ERROR,
									(void *)&sockError,&iSize))sockError=0;
						}
						// Declare problems :)
						m_pConsole->socketEvent(Error,(void *)(sockError ? kvi_errorFromSystemError(sockError) : KviError_unknownError));
					} else {
						// can recover ? (EAGAIN , EINTR ?)
						m_pWsn = new QSocketNotifier((int)m_sock,QSocketNotifier::Write);
						QObject::connect(m_pWsn,SIGNAL(activated(int)),this,SLOT(doSSLHandshake(int)));
						m_pWsn->setEnabled(true);
						return;
					}
				}
			}
			reset();
		}
		break;
		case SSL_ERROR_SSL:
			raiseSSLError();
			m_pConsole->socketEvent(Error,(void *)KviError_SSLError);
			reset();
		break;
	}
*/
#else  //!COMPILE_SSL_SUPPORT
	debug("Ops.. ssl handshake without ssl support!...aborting!");
	exit(-1);
#endif //!COMPILE_SSL_SUPPORT
}

void KviIrcSocket::connectionEstabilished()
{
	setState(m_pProxy ? ProxyLogin : Connected);
	// create the correct read notifier now...
	// If we're using a proxy , we connect the read notifier
	// to a special slot that will handle the proxy login
	// and then dynamically change the slot to the
	// "normal" (irc) one.

	if(m_sock == KVI_INVALID_SOCKET)return; // ops...disconnected in setState() ????

	if(m_pRsn)
	{
		delete m_pRsn;
		m_pRsn = 0;
	}

	m_pRsn = new QSocketNotifier((int)m_sock,QSocketNotifier::Read);
	if(m_pProxy){
		// proxy connection
		QObject::connect(m_pRsn,SIGNAL(activated(int)),this,SLOT(readProxyData(int)));
		switch(m_pProxy->protocol())
		{
			case KviProxy::Http:
				proxyLoginHttp();
			break;
			case KviProxy::Socks5:
				proxyLoginV5();
			break;
			default:
				proxyLoginV4();
			break;
		}
	} else {
		// normal irc connection
		QObject::connect(m_pRsn,SIGNAL(activated(int)),this,SLOT(readIrcData(int)));
	}
	m_pRsn->setEnabled(true);
}

void KviIrcSocket::proxyLoginHttp()
{
	// Well..this is just plain and easy: connect to the proxy machine
	// and say "CONNECT <irc.server>:<port> HTTP/<version>\n\n"
	// Then expect a server reply header (2 newlines)
	// HTTP 200 = Success
	// HTTP Anything else = Failure

	m_pConsole->socketEvent(ProxyMessage,(void *)(__tr("Using HTTP protocol.")));

	setState(ProxyFinalHttp);
	KviStr tmp(KviStr::Format,"CONNECT %s:%u HTTP/1.0\n\n",m_pIrcServer->hostname(),(unsigned int)(m_pIrcServer->port()));

	sendRawData(tmp.ptr(),tmp.len());
}

void KviIrcSocket::proxyLoginV4()
{
	// SOCKSV4 protocol
	//
	// 1) CONNECT
	//
	// The client connects to the SOCKS server and sends a CONNECT request when
	// it wants to establish a connection to an application server. The client
	// includes in the request packet the IP address and the port number of the
	// destination host, and userid, in the following format.
	//
	//                +----+----+----+----+----+----+----+----+----+----+....+----+
	//                | VN | CD | DSTPORT |      DSTIP        | USERID       |NULL|
	//                +----+----+----+----+----+----+----+----+----+----+....+----+
	// # of bytes:       1    1      2              4           variable       1
	//
	// VN is the SOCKS protocol version number and should be 4. CD is the
	// SOCKS command code and should be 1 for CONNECT request. NULL is a byte
	// of all zero bits.
	//
	// The SOCKS server checks to see whether such a request should be granted
	// based on any combination of source IP address, destination IP address,
	// destination port number, the userid, and information it may obtain by
	// consulting IDENT, cf. RFC 1413.  If the request is granted, the SOCKS
	// server makes a connection to the specified port of the destination host.
	// A reply packet is sent to the client when this connection is established,
	// or when the request is rejected or the operation fails. 
	//
	m_pConsole->socketEvent(ProxyMessage,(void *)(__tr("Using SOCKSV4 protocol.")));

	m_pProxy->normalizeUserAndPass();
	// the protocol does not specify the "userid" format...
	// so build an userid from the pass and/or username...

	KviStr szUserAndPass=m_pProxy->user();
	if(m_pProxy->hasPass()){
		if(szUserAndPass.hasData())szUserAndPass.append(' ');
		szUserAndPass.append(m_pProxy->pass());
	}
	int iLen = szUserAndPass.len()+9;
	// build the request packet
	char *bufToSend = new char[iLen];
	bufToSend[0]=(unsigned char)4; //Version 4
	bufToSend[1]=(unsigned char)1; //Connect
	Q_UINT16 port=(Q_UINT16)htons(m_pIrcServer->port());
	kvi_memmove((void *)(bufToSend+2),(void *)&port,2);

	struct in_addr ircInAddr;

	if(!kvi_stringIpToBinaryIp(m_pIrcServer->ip(),&ircInAddr))
		debug("SOCKET INTERNAL ERROR IN IPV4 (SOCKS4) ADDR CONVERSION");

	Q_UINT32 host=(Q_UINT32)ircInAddr.s_addr;
	kvi_memmove((void *)(bufToSend+4),(void *)&host,4);
	kvi_memmove((void *)(bufToSend+8),(void *)(szUserAndPass.ptr()),szUserAndPass.len());
	// send it into hyperspace...
	setState(ProxyFinalV4);
	sendRawData(bufToSend,iLen);
	delete[] bufToSend;
	// and wait for reply...
}

void KviIrcSocket::proxyLoginV5()
{
	// SOCKSV5 protocol.
	//
	// When a TCP-based client wishes to establish a connection to an object
	// that is reachable only via a firewall (such determination is left up
	// to the implementation), it must open a TCP connection to the
	// appropriate SOCKS port on the SOCKS server system.  The SOCKS service
	// is conventionally located on TCP port 1080.  If the connection
	// request succeeds, the client enters a negotiation for the
	// authentication method to be used, authenticates with the chosen
	// method, then sends a relay request.  The SOCKS server evaluates the
	// request, and either establishes the appropriate connection or denies
	// it.
	//
	// The client connects to the server, and sends a version
	// identifier/method selection message:
	//
	//                    +----+----------+----------+
	//                    |VER | NMETHODS | METHODS  |
	//                    +----+----------+----------+
	//                    | 1  |    1     | 1 to 255 |
	//                    +----+----------+----------+
	//
	// The VER field is set to X'05' for this version of the protocol.  The
	// NMETHODS field contains the number of method identifier octets that
	// appear in the METHODS field.
	// The values currently defined for METHOD are:
	//
	//      o  X'00' NO AUTHENTICATION REQUIRED
	//      o  X'01' GSSAPI
	//      o  X'02' USERNAME/PASSWORD
	//      o  X'03' CHAP
	//      o  X'04' to X'7F' IANA ASSIGNED
	//      o  X'80' to X'FE' RESERVED FOR PRIVATE METHODS
	//      o  X'FF' NO ACCEPTABLE METHODS
	//

	m_pConsole->socketEvent(ProxyMessage,
			(void *)(__tr("Using SOCKSV5 protocol.")));

	m_pProxy->normalizeUserAndPass();
	// the protocol does not specify the "userid" format...
	// so build an userid from the pass and/or username...

	char bufToSend[4];
	bufToSend[0]=(unsigned char)5; //use version 5
	int sendLen = 3;
	if(m_pProxy->hasUser() || m_pProxy->hasPass()){
		// no auth needed.
		bufToSend[1]=(unsigned char)1; //select one method
		bufToSend[2]=(unsigned char)0; //select method 0 : no auth
		m_pConsole->socketEvent(ProxyMessage,
			(void *)(__tr("We can accept auth method 0 (no auth)")));
	} else {
		// we can provide a password and username if needed...
		bufToSend[1]=(unsigned char)2; //select from two methods
		bufToSend[2]=(unsigned char)0; //method 0 or
		bufToSend[3]=(unsigned char)2; //method 2 username/pass auth
		sendLen = 4;
		m_pConsole->socketEvent(ProxyMessage,
			(void *)(__tr("We can accept auth method 0 (no auth) or 2 (user/pass)")));
	}
	// notify the user before sending...since we may get disconnected
	setState(ProxySelectAuthMethodV5);
	sendRawData(bufToSend,sendLen);
	// and wait for response
}

void KviIrcSocket::proxyAuthUserPassV5()
{
	//   Once the SOCKS V5 server has started, and the client has selected the
	//   Username/Password Authentication protocol, the Username/Password
	//   subnegotiation begins.  This begins with the client producing a
	//   Username/Password request:
	//
	//           +----+------+----------+------+----------+
	//           |VER | ULEN |  UNAME   | PLEN |  PASSWD  |
	//           +----+------+----------+------+----------+
	//           | 1  |  1   | 1 to 255 |  1   | 1 to 255 |
	//           +----+------+----------+------+----------+
	//
	//   The VER field contains the current version of the subnegotiation,
	//   which is X'01'. The ULEN field contains the length of the UNAME field
	//   that follows. The UNAME field contains the username as known to the
	//   source operating system. The PLEN field contains the length of the
	//   PASSWD field that follows. The PASSWD field contains the password
	//   association with the given UNAME.
	//
	unsigned int lPass=(unsigned int)m_pProxy->passLen();
	if(lPass>255)lPass=255;
	unsigned int lUser=(unsigned int)m_pProxy->userLen();
	if(lUser>255)lUser=255;
	int iLen=lPass+lUser+3;
	char *bufToSend=new char[iLen];
	bufToSend[0]=(unsigned char)1;                           //version x'01'
	bufToSend[1]=(unsigned char)lUser;                       //length of the username
	kvi_memmove((void *)(bufToSend+2),(void *)m_pProxy->user(),lUser); //username
	bufToSend[2+lUser]=(unsigned char)lPass;             //length of the password
	kvi_memmove((void *)(bufToSend+3+lUser),(void *)m_pProxy->pass(),lPass);
	// spit out the buffer and wait
	setState(ProxyUserPassV5);
	sendRawData(bufToSend,iLen);
	delete[] bufToSend;
	// and wait for response...
}

void KviIrcSocket::proxySendTargetDataV5()
{
	//   Once the method-dependent subnegotiation has completed, the client
	//   sends the request details.  If the negotiated method includes
	//   encapsulation for purposes of integrity checking and/or
	//   confidentiality, these requests MUST be encapsulated in the method-
	//   dependent encapsulation.
	//
	//   The SOCKS request is formed as follows:
	//
	//           +----+-----+------+------+----------+----------+
	//           |VER | CMD | FLAG | ATYP | DST.ADDR | DST.PORT |
	//           +----+-----+------+------+----------+----------+
	//           | 1  |  1  |  1   |  1   | Variable |    2     |
	//           +----+-----+------+------+----------+----------+
	//
	//   Where:
	//
	//      o VER    protocol version: X'05'
	//      o CMD
	//         o CONNECT X'01'
	//         o BIND X'02'
	//         o UDP ASSOCIATE X'03'
	//         o  X'04' to X'7F' IANA ASSIGNED
	//         o  X'80' to X'FF' RESERVED FOR PRIVATE METHODS
	//      o FLAG   command dependent flag (defaults to X'00')
	//      o ATYP   address type of following address
	//        o IP V4 address: X'01'
	//        o DOMAINNAME: X'03'
	//        o IP V6 address: X'04'
	//      o DST.ADDR       desired destination address
	//      o DST.PORT desired destination port in network octet
	//         order
	//
	//      The SOCKS server will typically evaluate the request based on
	//      source and destination addresses, and return one or more reply
	//      messages, as appropriate for the request type.
	//
	//   In an address field (DST.ADDR, BND.ADDR), the ATYP field specifies
	//   the type of address contained within the field:
	//
	//             o  X'01'
	//
	//   The address is a version-4 IP address, with a length of 4 octets.
	//
	//             o  X'03'
	//
	//   The address field contains a fully-qualified domain name.  The first
	//   octet of the address field contains the number of octets of name that
	//   follow, there is no terminating NUL octet.
	//
	//             o  X'04'
	//
	//   The address is a version-6 IP address, with a length of 16 octets.
	int bufLen = m_pIrcServer->isIpV6() ? 22 : 10;
	char * bufToSend = (char *)kvi_malloc(sizeof(char) * bufLen);
	bufToSend[0]=(unsigned char)5;           //Proto 5
	bufToSend[1]=(unsigned char)1;           //CONNECT
	bufToSend[2]=(unsigned char)0;           //RSV
	bufToSend[3]=(unsigned char)m_pIrcServer->isIpV6() ? 4 : 1; // IPV6 : IPV4

	if(m_pIrcServer->isIpV6())
	{
#ifdef COMPILE_IPV6_SUPPORT
		struct in6_addr ircInAddr;

		if(!kvi_stringIpToBinaryIp_V6(m_pIrcServer->ip(),&ircInAddr))debug("SOCKET INTERNAL ERROR IN IPV6 ADDR CONVERSION");
		kvi_memmove((void *)(bufToSend + 4),(void *)(&ircInAddr),4);
		Q_UINT16 port = (Q_UINT16)htons(m_pIrcServer->port());
		kvi_memmove((void *)(bufToSend + 20),(void *)&port,2);
#endif
	} else {
		struct in_addr ircInAddr;

		if(!kvi_stringIpToBinaryIp(m_pIrcServer->ip(),&ircInAddr))debug("SOCKET INTERNAL ERROR IN IPV4 ADDR CONVERSION");
		Q_UINT32 host = (Q_UINT32)ircInAddr.s_addr;
		kvi_memmove((void *)(bufToSend + 4),(void *)&host,4);
		Q_UINT16 port = (Q_UINT16)htons(m_pIrcServer->port());
		kvi_memmove((void *)(bufToSend + 8),(void *)&port,2);
	}

	// send it into hyperspace...
	setState(ProxyFinalV5);
	sendRawData(bufToSend,bufLen);
	kvi_free(bufToSend);
	// and wait for reply...
}

void KviIrcSocket::setState(SockState st)
{
	if(st != m_state)
	{
		m_state = st;
		m_pConsole->socketEvent(StateChange,(void *)st);
	}
}

void KviIrcSocket::readProxyData(int)
{
	char buffer[256];

	int readLength;
#ifdef COMPILE_SSL_SUPPORT
	if(m_pSSL)
	{
		readLength = m_pSSL->read(buffer,256);
		if(readLength <= 0)
		{
			// ssl error....?
			switch(m_pSSL->getProtocolError(readLength))
			{
				case KviSSL::ZeroReturn:
					readLength = 0;
				break;
				case KviSSL::WantRead:
				case KviSSL::WantWrite:
					// hmmm...
					return;
				break;
				case KviSSL::SyscallError:
				{
					int iE = m_pSSL->getLastError(true);
					if(iE != 0)
					{
						raiseSSLError();
						m_pConsole->socketEvent(Error,(void *)KviError_SSLError);
						reset();
						return;
					}
				}
				break;
				case KviSSL::SSLError:
					raiseSSLError();
					m_pConsole->socketEvent(Error,(void *)KviError_SSLError);
					reset();
					return;
				break;
				default:
					m_pConsole->socketEvent(Error,(void *)KviError_SSLError);
					reset();
					return;
				break;

			}
			handleInvalidSocketRead(readLength);
			return;
		}
	} else {
#endif
		readLength = kvi_socket_recv(m_sock,buffer,256);
		if(readLength <= 0)
		{
			handleInvalidSocketRead(readLength);
			return;
		}
#ifdef COMPILE_SSL_SUPPORT
	}
#endif

	// we need at least two bytes...
	if(readLength < 2)
	{
		// a single byte of reply means:
		// - connection through a 1 bps modem
		// - a totally blocked network
		// - remote host is not a SOCKS server
		// Anyway....it is always a meaningless reply
		// better to try again later :)
		m_pConsole->socketEvent(Error,(void *)KviError_unrecognizedProxyReply);
		reset();
		return;
	}
	// handle the reply
	switch(m_state)
	{
		case ProxyFinalV4:
			//V4 final reply
			proxyHandleV4FinalReply((unsigned char)buffer[1]);
			break;
		case ProxySelectAuthMethodV5:
			//V5 method selection reply
			proxyHandleV5MethodReply((unsigned char)buffer[1]);
			break;
		case ProxyUserPassV5:
			//V5 user and pass reply
			proxyHandleV5AuthReply((unsigned char)buffer[1]);
			break;
		case ProxyFinalV5:
			//V5 final reply
			proxyHandleV5FinalReply((unsigned char)buffer[1]);
			break;
		case ProxyFinalHttp:
			//Http final reply
			buffer[readLength] = '\0';
			proxyHandleHttpFinalReply(buffer,readLength);
			break;
		default:
			// what ?
			m_pConsole->socketEvent(Error,(void *)KviError_unrecognizedProxyReply);
			reset();
			break;
	}
}


void KviIrcSocket::proxyHandleV5AuthReply(unsigned char reply)
{
	//   The server verifies the supplied UNAME and PASSWD, and sends the
	//   following response:
	//
	//                        +----+--------+
	//                        |VER | STATUS |
	//                        +----+--------+
	//                        | 1  |   1    |
	//                        +----+--------+
	//
	//   A STATUS field of X'00' indicates success. If the server returns a
	//   `failure' (STATUS value other than X'00') status, it MUST close the
	//   connection.
	//
	if(reply == 0){
		m_pConsole->socketEvent(ProxyMessage,(void *)(__tr("Proxy response: auth OK: access granted")));
		proxySendTargetDataV5();
		return;
	}
	m_pConsole->socketEvent(Error,(void *)KviError_proxyAuthFailed);
	reset();
}

void KviIrcSocket::proxyHandleV5MethodReply(unsigned char reply)
{
	//   The server selects from one of the methods given in METHODS, and
	//   sends a METHOD selection message:
	//
	//                            +----+--------+
	//                            |VER | METHOD |
	//                            +----+--------+
	//                            | 1  |   1    |
	//                            +----+--------+
	//
	//   If the selected METHOD is X'FF', none of the methods listed by the
	//   client are acceptable, and the client MUST close the connection.
	//
	// The values currently defined for METHOD are:
	//
	//      o  X'00' NO AUTHENTICATION REQUIRED
	//      o  X'01' GSSAPI
	//      o  X'02' USERNAME/PASSWORD
	//      o  X'03' CHAP
	//      o  X'04' to X'7F' IANA ASSIGNED
	//      o  X'80' to X'FE' RESERVED FOR PRIVATE METHODS
	//      o  X'FF' NO ACCEPTABLE METHODS
	//
	if(reply == 0){
		m_pConsole->socketEvent(ProxyMessage,(void *)(__tr("Proxy response: Auth method OK: using method 0 (no auth)")));
		proxySendTargetDataV5();
		return;
	}
	if(reply == 2){
		m_pConsole->socketEvent(ProxyMessage,(void *)(__tr("Proxy response: Auth method OK: using method 2 (user/pass)")));
		proxyAuthUserPassV5();
		return;
	}
	//Request rejected
	if(reply == 0xFF)
	{
		m_pConsole->socketEvent(Error,(void *)KviError_proxyNoAcceptableAuthMethod);
		reset();
	} else {
		// unrecognized...
		m_pConsole->socketEvent(Error,(void *)KviError_unrecognizedProxyReply);
		reset();
	}
}

void KviIrcSocket::proxyHandleV5FinalReply(unsigned char reply)
{
	//
	//   The SOCKS request information is sent by the client as soon as it has
	//   established a connection to the SOCKS server, and completed the
	//   authentication negotiations.  The server evaluates the request, and
	//   returns a reply formed as follows:
	//
	//           +----+-----+------+------+----------+----------+
	//           |VER | REP | FLAG | ATYP | BND.ADDR | BND.PORT |
	//           +----+-----+------+------+----------+----------+
	//           | 1  |  1  |  1   |  1   | Variable |    2     |
	//           +----+-----+------+------+----------+----------+
	//
	//   Where:
	//             o  VER    protocol version: X'05'
	//             o  REP    Reply field:
	//                o  X'00' succeeded
	//                o  X'01' general SOCKS server failure
	//                o  X'02' connection not allowed by ruleset
	//                o  X'03' Network unreachable
	//                o  X'04' Host unreachable
	//                o  X'05' Connection refused
	//                o  X'06' TTL expired
	//                o  X'07' Command not supported
	//                o  X'08' Address type not supported
	//                o  X'09' Invalid address
	//                o  X'0A' to X'FF' unassigned
	//             o  FLAG   command dependent flag
	//             o  ATYP   address type of following address
	//                o  IP V4 address: X'01'
	//                o  DOMAINNAME: X'03'
	//                o  IP V6 address: X'04'
	//             o  BND.ADDR       server bound address
	//             o  BND.PORT       server bound port in network octet order
	//
	if(reply==0){
		// Request granted
		m_pConsole->socketEvent(ProxyMessage,(void *)(__tr("Proxy response: target data OK: request granted")));
		// change the slot....
		QObject::disconnect(m_pRsn,SIGNAL(activated(int)),this,SLOT(readProxyData(int)));
		QObject::connect(m_pRsn,SIGNAL(activated(int)),this,SLOT(readIrcData(int)));
		setState(Connected);
	} else {
		//Request rejected
		int err;
		switch(reply){
			case 1: err = KviError_proxyReply01GeneralSOCKSFailure; break;
			case 2: err = KviError_proxyReply02ConnectionNotAllowed; break;
			case 3: err = KviError_proxyReply03NetworkUnreachable; break;
			case 4: err = KviError_proxyReply04HostUnreachable; break;
			case 5: err = KviError_proxyReply05ConnectionRefused; break;
			case 6: err = KviError_proxyReply06TTLExpired; break;
			case 7: err = KviError_proxyReply07CommandNotSupported; break;
			case 8: err = KviError_proxyReply08AddressTypeNotSupported; break;
			case 9: err = KviError_proxyReply09InvalidAddress; break;
			default: err = KviError_unrecognizedProxyReply; break;
		}
		m_pConsole->socketEvent(Error,(void *)err);
		reset();
	}
}

void KviIrcSocket::proxyHandleV4FinalReply(unsigned char reply)
{
	// If the request is granted, the SOCKS
	// server makes a connection to the specified port of the destination host.
	// A reply packet is sent to the client when this connection is established,
	// or when the request is rejected or the operation fails. 
	//
	//
	//                +----+----+----+----+----+----+----+----+
	//                | VN | CD | DSTPORT |      DSTIP        |
	//                +----+----+----+----+----+----+----+----+
	// # of bytes:       1    1      2              4
	//
	// VN is the version of the reply code and should be 0. CD is the result
	// code with one of the following values:
	//
	//        90: request granted
	//        91: request rejected or failed
	//        92: request rejected becasue SOCKS server cannot connect to
	//            identd on the client
	//        93: request rejected because the client program and identd
	//            report different user-ids
	//
	// The remaining fields are ignored.
	//
	// The SOCKS server closes its connection immediately after notifying
	// the client of a failed or rejected request. For a successful request,
	// the SOCKS server gets ready to relay traffic on both directions. This
	// enables the client to do I/O on its connection as if it were directly
	// connected to the application server.
	if(reply==90){
		// Request granted
		m_pConsole->socketEvent(ProxyMessage,(void *)(__tr("Proxy response: target data OK: request granted")));
		// change the slot....
		QObject::disconnect(m_pRsn,SIGNAL(activated(int)),this,SLOT(readProxyData(int)));
		QObject::connect(m_pRsn,SIGNAL(activated(int)),this,SLOT(readIrcData(int)));
		setState(Connected);
	} else {
		//Request rejected
		int err;
		switch(reply){
			case 91: err = KviError_proxyReply91RequestFailed; break;
			case 92: err = KviError_proxyReply92IdentFailed; break;
			case 93: err = KviError_proxyReply93IdentNotMatching; break;
			default: err = KviError_unrecognizedProxyReply; break;
		}
		m_pConsole->socketEvent(Error,(void *)err);
		reset();
	}
	// Just looked out of the window...
	// Hmmmm...strange light outside...
	// Looked at the clock...6:34 !
	// I think I'll go sleep.... :)
}

void KviIrcSocket::proxyHandleHttpFinalReply(const char * buffer,int bufLen)
{
	// Escape character is '^]'.
	// CONNECT warszawa.irc.pl:6667 HTTP/1.0
	//
	// HTTP/1.0 200 Connection established

	KviStr tmp = buffer;
// FIXME: #warning "We could even show the proxy output here...!"
	tmp.cutFromFirst('\n');
	tmp.stripWhiteSpace();

	if(kvi_strEqualCIN(tmp.ptr(),"HTTP",4))
	{
		int idx = tmp.findFirstIdx(" 200 ");
		if(idx != -1)
		{
			if(idx == tmp.findFirstIdx(' '))
			{
				tmp.prepend(__tr("Proxy response: "));
				m_pConsole->socketEvent(ProxyMessage,(void *)(tmp.ptr()));
				// change the slot....
				QObject::disconnect(m_pRsn,SIGNAL(activated(int)),this,SLOT(readProxyData(int)));
				QObject::connect(m_pRsn,SIGNAL(activated(int)),this,SLOT(readIrcData(int)));
				setState(Connected);
				return;
			}
		}
		
	}

	tmp.prepend(__tr("Proxy said something about \""));
	tmp.append(__tr("\" ...that looks like an error to me..."));
	m_pConsole->socketEvent(ProxyError,(void *)(tmp.ptr()));

	m_pConsole->socketEvent(Error,(void *)KviError_proxyHttpFailure);
	reset();
}



void KviIrcSocket::readIrcData(int)
{
	//read data
	char buffer[1025];
	int readLength;
#ifdef COMPILE_SSL_SUPPORT
	if(m_pSSL)
	{
		readLength = m_pSSL->read(buffer,1024);
		if(readLength <= 0)
		{
			// ssl error....?
			switch(m_pSSL->getProtocolError(readLength))
			{
				case KviSSL::ZeroReturn:
					readLength = 0;
				break;
				case KviSSL::WantRead:
				case KviSSL::WantWrite:
					// hmmm...
					return;
				break;
				case KviSSL::SyscallError:
				{
					int iE = m_pSSL->getLastError(true);
					if(iE != 0)
					{
						raiseSSLError();
						m_pConsole->socketEvent(Error,(void *)KviError_SSLError);
						reset();
						return;
					}
				}
				break;
				case KviSSL::SSLError:
					raiseSSLError();
					m_pConsole->socketEvent(Error,(void *)KviError_SSLError);
					reset();
					return;
				break;
				default:
					m_pConsole->socketEvent(Error,(void *)KviError_SSLError);
					reset();
					return;
				break;
	
			}
			handleInvalidSocketRead(readLength);
			return;
		}
	} else {
#endif
		readLength = kvi_socket_recv(m_sock,buffer,1024);
		if(readLength <= 0)
		{
			handleInvalidSocketRead(readLength);
			return;
		}
#ifdef COMPILE_SSL_SUPPORT
	}
#endif

	m_uReadBytes += readLength;
	//terminate our buffer
	(*(buffer+readLength))='\0';
	register char *p=buffer;
	char *beginOfCurData = buffer;
	int   bufLen = 0;
	char *messageBuffer = (char *)kvi_malloc(1);
	//Shut up the socket notifier
	//in case that we enter in a local loop somewhere
	//while processing data...
	m_pRsn->setEnabled(false);
	while(*p){
		if((*p == '\r' )||(*p == '\n')){
			//found a CR or LF...
			//prepare a message buffer
			bufLen = p - beginOfCurData;
			//check for previous unterminated data
			if(m_uReadBufferLen > 0){
				__range_valid(m_pReadBuffer);
				messageBuffer = (char *)kvi_realloc(messageBuffer,bufLen + m_uReadBufferLen + 1);
				kvi_memmove(messageBuffer,m_pReadBuffer,m_uReadBufferLen);
				kvi_memmove((void *)(messageBuffer + m_uReadBufferLen),beginOfCurData,bufLen);
				*(messageBuffer + bufLen + m_uReadBufferLen) = '\0';
				m_uReadBufferLen = 0;
				kvi_free(m_pReadBuffer);
				m_pReadBuffer = 0;
			} else {
				__range_invalid(m_pReadBuffer);
				messageBuffer = (char *)kvi_realloc(messageBuffer,bufLen + 1);
				kvi_memmove(messageBuffer,beginOfCurData,bufLen);
				*(messageBuffer + bufLen) = '\0';
			}
			m_uReadPackets++;

			if(m_pConsole->ircSocketMonitors())
			{
				for(KviIrcSocketMonitor *m =m_pConsole->ircSocketMonitors()->first();m;m =m_pConsole->ircSocketMonitors()->next())
				{
					m->incomingMessage(messageBuffer);
				}
			}

			g_pServerParser->parseMessage(messageBuffer,m_pConsole);
			while(*p && ((*p=='\r')||(*p=='\n')) )p++;
			beginOfCurData = p;
		} else p++;
	}
	//now *p == '\0'
	//beginOfCurData points to '\0' if we have
	//no more stuff to parse , or points to something
	//different than '\r' or '\n'...
	if(*beginOfCurData){
		//Have remaining data...in the local buffer
		bufLen = p - beginOfCurData;
		if(m_uReadBufferLen > 0){
			//and there was more stuff saved... (really slow connection)
			__range_valid(m_pReadBuffer);
			m_pReadBuffer =(char *)kvi_realloc(m_pReadBuffer,m_uReadBufferLen + bufLen);
			kvi_memmove((void *)(m_pReadBuffer+m_uReadBufferLen),beginOfCurData,bufLen);
			m_uReadBufferLen += bufLen;
		} else {
			//
			__range_invalid(m_pReadBuffer);
			m_uReadBufferLen = bufLen;
			m_pReadBuffer =(char *)kvi_malloc(m_uReadBufferLen);
			kvi_memmove(m_pReadBuffer,beginOfCurData,m_uReadBufferLen);
		}
		//The m_pReadBuffer contains at max 1 irc message...
		//that can not be longer than 510 bytes (the message is not CRLF terminated)
		if(m_uReadBufferLen > 510)debug("WARNING : Receiving an invalid irc message from server.");
	}
	kvi_free(messageBuffer);
	//ready to receive
	m_pRsn->setEnabled(true);
}

//=== handleInvalidSocketRead ===============================================//
//
// Checks if the socket error is a transient error
// If it is not a transient error it resets the socket
// and fires the appropriate event.
// Otherwise it does nothing.
//
void KviIrcSocket::handleInvalidSocketRead(int readedLength)
{
	__range_valid(readedLength <= 0);
	if(readedLength==0)
	{
		m_pConsole->socketEvent(Error,(void *)KviError_remoteEndClosedConnection);
		reset();
	} else {
		//check for transmission errors
		int err = kvi_socket_error();
#ifdef COMPILE_ON_WINDOWS
		if((err != EAGAIN) && (err != EINTR) && (err != WSAEWOULDBLOCK))
#else
		if((err != EAGAIN) && (err != EINTR))
#endif
		{
			if(err > 0)m_pConsole->socketEvent(Error,(void *)(kvi_errorFromSystemError(err)));
			else m_pConsole->socketEvent(Error,(void *)KviError_remoteEndClosedConnection);
			reset();
		} //else transient error...wait again...
	}
}

//=== Message send stuff ====================================================//
// Max buffer that can be sent to an IRC server is 512 bytes
// including CRLF. (ircd simply 'cuts' messages to 512 bytes
// and discards the remainig part)
// Note that 510 bytes of data is a reasonably long message :)
//
// 01234567890123456789012345678901234567890123456789
// 01234567890123456789012345678901234567890123456789
// 01234567890123456789012345678901234567890123456789
// 01234567890123456789012345678901234567890123456789
// 01234567890123456789012345678901234567890123456789
// 01234567890123456789012345678901234567890123456789
// 01234567890123456789012345678901234567890123456789
// 01234567890123456789012345678901234567890123456789
// 01234567890123456789012345678901234567890123456789
// 01234567890123456789012345678901234567890123456789
// 0123456789\r\n
//
// We keep a list of data to send , and flush it as soon as we can.
//
bool KviIrcSocket::sendFmtData(const char *fmt,...)
{
	if(m_state == Connected)
	{
		//new buffer	
		KviIrcSocketMsgEntry *ptr = (KviIrcSocketMsgEntry *)kvi_malloc(sizeof(KviIrcSocketMsgEntry));
		ptr->data_ptr = (char *)kvi_malloc(512);
		va_list(list);
		va_start(list,fmt);
		bool bTruncated;
		//sprintf the buffer up to 512 chars (adds a CRLF too)
		ptr->data_len = kvi_irc_vsnprintf(ptr->data_ptr,fmt,list,&bTruncated);
		va_end(list);
		//adjust the buffer size
		if(ptr->data_len < 512)ptr->data_ptr = (char *)kvi_realloc(ptr->data_ptr,ptr->data_len);
		if(bTruncated)m_pConsole->socketEvent(Warning,(void *)(__tr("Socket message truncated to 512 bytes.")));
//#ifdef COMPILE_NEED_CHARSET_TRANSLATION
//		if(g_pOptions->m_bUseCharsetTranslation)
//		{
//			g_pCharsetTranslator->translateToServer(ptr->data_ptr,ptr->data_len);
//		}
//#endif
		queue_insertMessage(ptr);
		flushSendQueue();
		return true;
	}
	return false; //not connected
}

bool KviIrcSocket::sendRawData(const char *buffer,int buflen)
{
	if((m_state != Ready) && (m_state != Connecting)){
		//new buffer	
		KviIrcSocketMsgEntry *ptr = (KviIrcSocketMsgEntry *)kvi_malloc(sizeof(KviIrcSocketMsgEntry));
		ptr->data_ptr = (char *)kvi_malloc(buflen);
		ptr->data_len = buflen;
		kvi_memmove(ptr->data_ptr,buffer,buflen);
//#ifdef COMPILE_NEED_CHARSET_TRANSLATION
//		if(g_pOptions->m_bUseCharsetTranslation)
//		{
//			g_pCharsetTranslator->translateToServer(ptr->data_ptr,ptr->data_len);
//		}
//#endif
		queue_insertMessage(ptr);
		flushSendQueue();
		return true;
	}
	return false; //not connected
}

bool KviIrcSocket::sendData(const char *buffer,int buflen)
{
	if(m_state == Connected){
		//new buffer	
		KviIrcSocketMsgEntry *ptr = (KviIrcSocketMsgEntry *)kvi_malloc(sizeof(KviIrcSocketMsgEntry));
		if(buflen < 0)buflen = strlen(buffer);
		if(buflen > 510){
			buflen = 510;
			m_pConsole->socketEvent(Warning,(void *)(__tr("Socket message truncated to 512 bytes.")));
		}
		ptr->data_ptr = (char *)kvi_malloc(buflen+2);
		ptr->data_len = buflen+2;
		kvi_memmove(ptr->data_ptr,buffer,buflen);
		*(ptr->data_ptr+buflen)='\r';
		*(ptr->data_ptr+buflen+1)='\n';
//#ifdef COMPILE_NEED_CHARSET_TRANSLATION
//		if(g_pOptions->m_bUseCharsetTranslation)
//		{
//			g_pCharsetTranslator->translateToServer(ptr->data_ptr,ptr->data_len);
//		}
//#endif
		queue_insertMessage(ptr);
		flushSendQueue();
		return true;
	}
	return false; //not connected
}

//=== data queue functions ==================================================//
//
// queue_insertMessage : appends a KviIrcSocketMsgEntry to the tail of
//     the message queue. The next_ptr for this message is set to 0.
// queue_removeMessage : removes a message from the head of the queue.
//

void KviIrcSocket::queue_insertMessage(KviIrcSocketMsgEntry *msg_ptr)
{
	__range_valid(msg_ptr);
	__range_valid(msg_ptr->data_ptr);
	__range_valid(msg_ptr->data_len);
	msg_ptr->next_ptr = 0;
	if(m_pSendQueueHead){
		m_pSendQueueTail->next_ptr = msg_ptr;
		m_pSendQueueTail = msg_ptr;
	} else {
		m_pSendQueueHead = msg_ptr;
		m_pSendQueueTail = msg_ptr;
	}
}

bool KviIrcSocket::queue_removeMessage()
{
	__range_valid(m_pSendQueueTail);
	__range_valid(m_pSendQueueHead);
	if(m_pSendQueueHead->data_ptr)kvi_free((void *)m_pSendQueueHead->data_ptr);
	KviIrcSocketMsgEntry *aux_ptr = m_pSendQueueHead;
	m_pSendQueueHead = aux_ptr->next_ptr;
	kvi_free((void *)aux_ptr);
	if(m_pSendQueueHead == 0){
		m_pSendQueueTail = 0;
		return false;
	} else return true;
}

void KviIrcSocket::queue_removeAllMessages()
{
	if(m_pSendQueueHead)while(queue_removeMessage());
}

//=== flushSendQueue ========================================================//
//
// Attempts to send as much as possible to the server
// If fails (happens only on really lagged servers)
// calls itself with a QTimer shot after KVI_OPTION_UINT(KviOption_uintSocketQueueFlushTimeout) ms
// to retry again...
//
void KviIrcSocket::flushSendQueue()
{
	// If we're called from the flush timer , stop it
	if(m_pFlushTimer->isActive())m_pFlushTimer->stop();
//	if(!m_pSendQueueHead)return;
	// Ok...have something to send...
	__range_valid(m_state != Ready);

//	if(m_state != Ready){
	struct timeval curTime;

	while(m_pSendQueueHead)
	{
		if(KVI_OPTION_BOOL(KviOption_boolLimitOutgoingTraffic))
		{
			kvi_gettimeofday(&curTime,0);

			int timeDiff = curTime.tv_usec - m_tAntiFloodLastMessageTime.tv_usec;
			timeDiff += (curTime.tv_sec - m_tAntiFloodLastMessageTime.tv_sec) * 1000000;


			if(((unsigned int)timeDiff) < KVI_OPTION_UINT(KviOption_uintOutgoingTrafficLimitUSeconds))
			{
				// need to wait for a while....
				m_pFlushTimer->start(((KVI_OPTION_UINT(KviOption_uintOutgoingTrafficLimitUSeconds) - timeDiff) / 1000) + 1);
				return;
			} // else can send
		}
		// Write one data buffer...
		int result;
#ifdef COMPILE_SSL_SUPPORT
		if(m_pSSL)
		{
			result = m_pSSL->write(m_pSendQueueHead->data_ptr,m_pSendQueueHead->data_len);
		} else {
#endif
			result = kvi_socket_send(m_sock,m_pSendQueueHead->data_ptr,m_pSendQueueHead->data_len);
#ifdef COMPILE_SSL_SUPPORT
		}
#endif
		if(result == m_pSendQueueHead->data_len)
		{
			// Succesfull send...remove this data buffer
			m_uSentPackets++;
			m_uSentBytes += result;
			if(m_pConsole->ircSocketMonitors())
			{
				for(KviIrcSocketMonitor *m =m_pConsole->ircSocketMonitors()->first();m;m =m_pConsole->ircSocketMonitors()->next())
				{
					m->outgoingMessage(m_pSendQueueHead->data_ptr,result);
				}
			}
			queue_removeMessage();
			if(KVI_OPTION_BOOL(KviOption_boolLimitOutgoingTraffic))
			{
				m_tAntiFloodLastMessageTime.tv_sec = curTime.tv_sec;
				m_tAntiFloodLastMessageTime.tv_usec = curTime.tv_usec;
			}
			// And try next buffer...
			continue;
		} else {
			// Something wrong ?
#ifdef COMPILE_SSL_SUPPORT
			if(result <= 0)
			{
				// ops...might be an SSL error
				switch(m_pSSL->getProtocolError(result))
				{
					case KviSSL::WantWrite:
					case KviSSL::WantRead:
						// Async continue...
						m_pFlushTimer->start(KVI_OPTION_UINT(KviOption_uintSocketQueueFlushTimeout));
						return;
					break;
					case KviSSL::SyscallError:
						if(result == 0)
						{
							raiseSSLError();
							m_pConsole->socketEvent(Error,(void *)KviError_remoteEndClosedConnection);
							reset();
							return;
						} else {
							int iSSLErr = m_pSSL->getLastError(true);
							if(iSSLErr != 0)
							{
								raiseSSLError();
								m_pConsole->socketEvent(Error,(void *)KviError_SSLError);
								reset();
								return;
							} else {
								goto handle_system_error;
							}
						}
					break;
					case KviSSL::SSLError:
						raiseSSLError();
						m_pConsole->socketEvent(Error,(void *)KviError_SSLError);
						reset();
						return;
					break;
					default:
						m_pConsole->socketEvent(Error,(void *)KviError_SSLError);
						reset();
						return;
					break;
				}
			} else {
#else //!COMPILE_SSL_SUPPORT
			if(result >= 0)
			{
				if(result > 0)
				{
#endif //!COMPILE_SSL_SUPPORT

					// Partial send...need to finish it later
					if(m_pConsole->ircSocketMonitors())
					{
						for(KviIrcSocketMonitor *m =m_pConsole->ircSocketMonitors()->first();m;m =m_pConsole->ircSocketMonitors()->next())
						{
							m->outgoingMessage(m_pSendQueueHead->data_ptr,result);
						}
					}
					m_pSendQueueHead->data_len -= result;
					m_pSendQueueHead->data_ptr = (char *)kvi_realloc(m_pSendQueueHead->data_ptr,m_pSendQueueHead->data_len);
					m_uSentBytes += result;
					m_pConsole->socketEvent(Warning,(void *)(__tr("Partial socket write: packet broken into smaller pieces.")));
#ifndef COMPILE_SSL_SUPPORT
				}
#endif //!COMPILE_SSL_SUPPORT
				// Async continue...
				m_pFlushTimer->start(KVI_OPTION_UINT(KviOption_uintSocketQueueFlushTimeout));
				return;
			}

handle_system_error:
			// Oops...error ?
			int err = kvi_socket_error();
#ifdef COMPILE_ON_WINDOWS
			if((err == EAGAIN) || (err == EINTR) || (err == WSAEWOULDBLOCK))
#else
			if((err == EAGAIN)||(err == EINTR))
#endif
			{
				// Transient error...partial send as before...
				m_pConsole->socketEvent(Warning,(void *)(__tr("Partial socket write: packet broken into smaller pieces.")));
				// Async continue...
				m_pFlushTimer->start(KVI_OPTION_UINT(KviOption_uintSocketQueueFlushTimeout));
				return;
			} else {
				// Disconnected... :(
				m_pConsole->socketEvent(Error,(void *)(kvi_errorFromSystemError(err)));
				reset();
				return;
			}
		}
	}
	//flushed completly ...
//	} else debug("WARNING : flushSendQueue called with no irc connection");
}

bool KviIrcSocket::getLocalHostIp(KviStr &szIp,bool bIpV6)
{
	if(m_state != Connected)return false;

	if(bIpV6)
	{
#ifdef COMPILE_IPV6_SUPPORT
		struct sockaddr_in6 name;
		int len = sizeof(name);
		if(!kvi_socket_getsockname(m_sock, (struct sockaddr *)&name,&len))return false;
		//I assume that getsockname returns data in Network byte order...
		//The man page misses to specify that...
		if(!kvi_binaryIpToStringIp_V6(name.sin6_addr,szIp))return false;
		return true;
#else
		return false; // no support
#endif
	}
	struct sockaddr_in name;
	int len = sizeof(name);
	if(!kvi_socket_getsockname(m_sock, (struct sockaddr *)&name,&len))return false;
	//I assume that getsockname returns data in Network byte order...
	//The man page misses to specify that...
	if(!kvi_binaryIpToStringIp(name.sin_addr,szIp))return false;
	return true;
}

#include "kvi_ircsocket.moc"
