//
//   File : kvi_iconmanager.cpp
//   Creation date : Sat Jun 24 2000 14:50:11 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#define __KVIRC__

#define _KVI_ICONMANAGER_CPP_

#include "kvi_iconmanager.h"

#include "kvi_imagelib.h"
#include "kvi_app.h"
#include "kvi_settings.h"
#include "kvi_defaults.h"
#include "kvi_locale.h"

#include <qtimer.h>
#include <qlayout.h>
#include <qlabel.h>

#include <qdragobject.h>
#include <qcursor.h>

// kvi_app.cpp
extern QPixmap * g_pUserChanStatePixmap;

KviIconManager * g_pIconManager = 0;

KviIconWidget::KviIconWidget()
: QWidget(0,"global_icon_widget",WType_TopLevel | WStyle_Customize | WStyle_Title | WStyle_StaysOnTop | WStyle_DialogBorder)
{
	init();
}

KviIconWidget::KviIconWidget(QWidget * par)
: QWidget(par,"global_icon_widget")
{
	init();
}

void KviIconWidget::init()
{
	setCaption(__tr("Icon table"));
	setIcon(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_ICONMANAGER)));
	int rows = KVI_NUM_SMALL_ICONS / 10;
	if((rows * 10) < KVI_NUM_SMALL_ICONS)rows++;
	QGridLayout * g = new QGridLayout(this,rows + 1,11,0,2);
	int i;
	for(i = 0;i < 10;i++)
	{
		KviStr tmp(KviStr::Format,"%d",i);
		QLabel * l = new QLabel(tmp.ptr(),this);
		g->addWidget(l,0,i + 1);
	}
	for(i = 0;i < rows;i++)
	{
		KviStr tmp(KviStr::Format,"%d",i * 10);
		QLabel * l = new QLabel(tmp.ptr(),this);
		g->addWidget(l,i + 1,0);
	}
	for(i = 0;i < KVI_NUM_SMALL_ICONS;i++)
	{
		KviStr tmp(KviStr::Format,"%d",i);
		QLabel * l = new QLabel(this,tmp.ptr());
		l->setPixmap(*(g_pIconManager->getSmallIcon(i)));
		g->addWidget(l,(i / 10) + 1,(i % 10) + 1);
		l->installEventFilter(this);
		l->setAcceptDrops(true);
	}
}

KviIconWidget::~KviIconWidget()
{
}

void KviIconWidget::closeEvent(QCloseEvent * e)
{
	e->ignore();
	emit closed();
}

bool KviIconWidget::eventFilter(QObject * o,QEvent *e)
{
	if(e->type() == QEvent::MouseButtonPress)
	{
		if(o->inherits("QLabel"))
		{
			KviStr szName = ((QLabel *)o)->name();
			bool bOk;
			int iVal = szName.toInt(&bOk);
			if(bOk)emit selected(iVal);

			if(parentWidget() && isVisible() && parentWidget()->inherits("QPopupMenu"))
			{
				parentWidget()->close();
				return true;
			} else {
				if(QPixmap * pix = ((QLabel *)o)->pixmap())
				{
					QDragObject * drobj = new QTextDrag(o->name(),this);
					drobj->setPixmap(*pix,((QLabel *)o)->mapFromGlobal(QCursor::pos()));
					drobj->dragCopy();
					return true;
				}
			}
		}
	} else if(e->type() == QEvent::DragEnter)
	{
		((QDragEnterEvent *)e)->accept(true);
		return true;
	}
	return QWidget::eventFilter(o,e);
}


KviIconManager::KviIconManager()
{
	KviStr buffer;
	g_pApp->findImage(buffer,KVI_SMALLICONS_IMAGELIB_NAME);

//	debug("Loading image %s",buffer.ptr());

	KviImageLibrary l1(buffer.ptr(),16,16);

//	debug("Library loaded: %d",l1.libraryLoaded());

	int i;

	for(i=0;i<KVI_NUM_SMALL_ICONS;i++)
	{
		m_smallIcons[i] = new QPixmap(l1.getImage(i));
	}

	g_pApp->findImage(buffer,KVI_BIGICONS_IMAGELIB_NAME);

	KviImageLibrary l2(buffer.ptr(),32,32);

	for(i=0;i<KVI_NUM_BIG_ICONS;i++)
	{
		m_bigIcons[i] = new QPixmap(l2.getImage(i));
	}

	m_pCachedImages = new QAsciiDict<KviCachedPixmap>(17,true,true);
	m_pCachedImages->setAutoDelete(true);

	m_pCleanupTimer = new QTimer();
	connect(m_pCleanupTimer,SIGNAL(timeout()),this,SLOT(cleanup()));

	m_pIconWidget = 0;

	// Load the userchanstate image
	g_pApp->findImage(buffer,KVI_USERCHANSTATE_IMAGE_NAME);
	g_pUserChanStatePixmap = new QPixmap(buffer.ptr());
}

KviIconManager::~KviIconManager()
{
	delete g_pUserChanStatePixmap;

	if(m_pIconWidget)delete m_pIconWidget;
	int i;

	for(i=0;i<KVI_NUM_SMALL_ICONS;i++)
	{
		delete m_smallIcons[i];
	}
	for(i=0;i<KVI_NUM_BIG_ICONS;i++)
	{
		delete m_bigIcons[i];
	}
	delete m_pCachedImages;
	delete m_pCleanupTimer;
}

void KviIconManager::showIconWidget()
{
	if(!m_pIconWidget)
	{
		m_pIconWidget = new KviIconWidget();
		connect(m_pIconWidget,SIGNAL(closed()),this,SLOT(iconWidgetClosed()));
	}
	m_pIconWidget->show();
	m_pIconWidget->raise();
	m_pIconWidget->setFocus();
}

void KviIconManager::iconWidgetClosed()
{
	delete m_pIconWidget;
	m_pIconWidget = 0;
}

/*
	@doc: image_id
	@title:
		The image identifier
	@keyterms:
		image identifier, using builtin images, how kvirc locates image files
	@type:
		language
	@short:
		Semantics of the <image_id> parameter
	@body:
		Some KVIrc commands and functions accept
		the <image_id> as a parameter.
		The <image_id> indicates an image to be displayed
		by some gui element. Since KVIrc has a set of
		builtin icons, we want the user to be able to use it :).[br]
		The <image_id> can be either a signed integer number
		or a filename.[br]
		If it is a signed integer it is interpreted as
		index of the internal KVIrc image to use:
		positive integers (starting at 0) indicate
		small (16x16) icons (the ones used in the view widget),
		negative integers indicate the bigger (32x32) icons used in
		the toolbars.[br]
		If <image_id> is anything else than a signed integer,
		it is interpreted as a filename.
		The filename can be an absolute file path or a relative one.
		In this last case KVIrc will look for the file in a predefined set
		of directories: First in the local "pics" directory (usually
		in ~/kvirc-version/pics/), then in the local "incoming" directory,
		then in the global "pics" directory , then in the user home (~)
		and in the current directory.[br]
*/

QPixmap * KviIconManager::getImage(const char * id,bool bCanBeNumber,KviStr * pRetPath)
{
	KviStr tmp = id;
	bool bOk;
	int idx = tmp.toInt(&bOk);
	if(bOk && bCanBeNumber)
	{
		// was a number : this is not a filename
		if(idx < 0)
		{
			idx = -idx;
			idx -= 1; // we want it to start from 0
			return getBigIcon(idx % KVI_NUM_BIG_ICONS);
		} else return getSmallIcon(idx % KVI_NUM_SMALL_ICONS);
	}

	KviCachedPixmap * p = m_pCachedImages->find(id);
	if(!p)
	{
		QPixmap * pix = 0;
		KviStr szRetPath;
		if(g_pApp->findUserFile(szRetPath,id))
		{
			pix = new QPixmap(szRetPath.ptr());
			if(pix->isNull())
			{
				delete pix; // it is not an valid image!!! (really bad situation...)
				pix = 0;
			}
		}
		if(pix)
		{
			p = new KviCachedPixmap();
			p->pPixmap = pix;
			m_pCachedImages->insert(id,p);
			if(pRetPath)*pRetPath = szRetPath;
		} else return 0;
	} else {
		if(pRetPath)g_pApp->findUserFile(*pRetPath,id); // MUST BE FOUND!
	}
	p->timestamp = time(0);
	if(!m_pCleanupTimer->isActive())
	{
		m_pCleanupTimer->start(120000); // 2 min timeout.... 
	}
	return p->pPixmap;
}

KviAvatar * KviIconManager::getAvatar(const char * fName)
{
	KviStr retPath;
	bool bWasFullPath = false;
	if(fName && *fName == '/')bWasFullPath = true;
	QPixmap * pix = getImage(fName,false,&retPath);
	if(pix)return new KviAvatar(new QPixmap(*pix),retPath.ptr(),bWasFullPath);
	return 0;
}

void KviIconManager::cleanup()
{
	KviPtrList<KviStr> l;
	l.setAutoDelete(true);
	QAsciiDictIterator<KviCachedPixmap> it(*m_pCachedImages);
	while(it.current())
	{
		time_t curTime = time(0);
		
		if((((unsigned int)it.current()->timestamp) - ((unsigned int)curTime)) > 120)
		{
			// unused since 2 minutes
			// time to cleanup it...
			l.append(new KviStr(it.currentKey()));
		}
		++it;
	}
	for(KviStr * p =l.first();p;p = l.next())
	{
		m_pCachedImages->remove(p->ptr());
	}
	if(m_pCachedImages->isEmpty())m_pCleanupTimer->stop();
}

#include "kvi_iconmanager.moc"
