//
//   File : kvi_app_fs.cpp
//   Last major modification : Thu Dec 24 1998 04:05:26 by Szymon Stefanek
//
//   This file is part of the Kvirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#define __KVIRC__

#include "kvi_debug.h"
#include "kvi_app.h"
#include "kvi_locale.h"
#include "kvi_fileutils.h"
#include "kvi_mediatype.h"


#include <qdir.h>
#include <qfileinfo.h>

extern KviMediaManager * g_pMediaManager;

//============ getGlobalKvircDirectory ================//

void KviApp::getGlobalKvircDirectory(KviStr &szData,KvircSubdir dir,const char *appendFile)
{
	szData=m_szGlobalKvircDir.ptr();
	szData.ensureLastCharIs(KVI_PATH_SEPARATOR_CHAR);

//	debug("Global1 : %s",szData.ptr());

	switch(dir)
	{
		case None          :                                                                           break;
		case Pics          : szData.append("pics");                                                    break;
		case Modules       :
		case Plugins       : szData.append("modules");                                                 break;
		case ConfigPlugins : szData.append(KviStr::Format,"config%smodules",KVI_PATH_SEPARATOR);       break;
		case ConfigScripts : szData.append(KviStr::Format,"config%sscripts",KVI_PATH_SEPARATOR);       break;
		case Help          :
			szData.append("help");
			{
				// Localized help
				KviStr tmp = kvi_getLocaleName();
				tmp.prepend(KVI_PATH_SEPARATOR_CHAR);
				tmp.prepend(szData);
				if(kvi_directoryExists(tmp.ptr()))szData = tmp;
				else {
					tmp = szData; tmp.append(KviStr::Format,"%sen",KVI_PATH_SEPARATOR);
					if(kvi_directoryExists(tmp.ptr()))szData = tmp;
				}
			}
		break;
		case HelpEN        : szData.append(KviStr::Format,"help%sen",KVI_PATH_SEPARATOR);              break;
		case HelpNoIntl    : szData.append("help");                                                    break;
		case Log           : debug("WARNING Global log directory requested!");                         break;
		case Incoming      : debug("WARNING Global incoming directory requested!");                    break;
		case Trash         : debug("WARNING Global trash directory requested!");                       break;
		case Config        : szData.append("config");                                                  break;
		case Audio         : szData.append("audio");                                                   break;
		case Scripts       : szData.append("scripts");                                                 break;
		case MsgColors     : szData.append("msgcolors");                                               break;
		case Charmaps      : szData.append("charmaps");                                                break;
		case Avatars       : szData.append("avatars");                                                 break;
		case DefScript     : szData.append("defscript");                                               break;
		case License       : szData.append("license");                                                 break;
		case Filters       : szData.append("filters");                                                 break;
	}

	if(appendFile)
	{
		szData.ensureLastCharIs(KVI_PATH_SEPARATOR_CHAR);
		szData.append(appendFile);
	}

//	debug("Globa2: %s",szData.ptr());

	kvi_adjustFilePath(szData);

//	debug("Global3: %s",szData.ptr());
}

//=============== getLocalKvircDirectory ================//

void KviApp::getLocalKvircDirectory(KviStr &szData,KvircSubdir dir,const char *appendFile,bool bCreateIfNeeded)
{
	szData=m_szLocalKvircDir.ptr();
	szData.ensureLastCharIs(KVI_PATH_SEPARATOR_CHAR);

	switch(dir)
	{
		case None          :                                                                     break;
		case Pics          : szData.append("pics");                                              break;
		case Config        : szData.append("config");                                            break;
		case ConfigPlugins : szData.append(KviStr::Format,"config%smodules",KVI_PATH_SEPARATOR); break;
		case ConfigScripts : szData.append(KviStr::Format,"config%sscripts",KVI_PATH_SEPARATOR); break;
		case Log           : szData.append("log");                                               break;
		case Incoming      : szData.append("incoming");                                          break;
		case Help          :
		case HelpEN        :
		case HelpNoIntl    : szData.append("help");                                              break;
		case Audio         : szData.append("audio");                                             break;
		case Scripts       : szData.append("scripts");                                           break;
		case Modules       :
		case Plugins       : szData.append("modules");                                           break;
		case Trash         : szData.append("trash");                                             break;
		case MsgColors     : szData.append("msgcolors");                                         break;
		case Charmaps      : szData.append("charmaps");                                          break;
		case Avatars       : szData.append("avatars");                                           break;
		case License       : szData.append("license");                                           break;
		case Filters       : szData.append("filters");                                           break;
//		case DefScript     : szData+="/defscript"      ; break; // should never happen ?
	}
	if(bCreateIfNeeded)kvi_makeDir(szData.ptr());

	if(appendFile)
	{
		szData.ensureLastCharIs(KVI_PATH_SEPARATOR_CHAR);
		szData.append(appendFile);
	}

	kvi_adjustFilePath(szData);
}

//====================== trashFile ====================//
//
// Moves a file to the local trash directory
//

bool KviApp::trashFile(const char *filename)
{
	// Exists ?
	QFileInfo fi(filename);
	if(!fi.exists())return false; // nothing to trash

	// Build the filename
	KviStr lastPart = filename;
	lastPart.cutToLast(KVI_PATH_SEPARATOR_CHAR); // need only the filename
	lastPart.append(".bak_");
	QDateTime tm(QDateTime::currentDateTime());
	lastPart.append(tm.toString());
	lastPart.replaceAll(' ',"_");
	// Ok...have lastPart.bak_Dec_11_31_1999
	// Find the thrash directory (make it if necessary)
	KviStr trashFile;
	getLocalKvircDirectory(trashFile,Trash,lastPart.ptr(),true);
	// Check if a such file already exists
	fi.setFile(trashFile.ptr());
	while(fi.exists()){
		trashFile.append(".rnm");
		fi.setFile(trashFile.ptr());
	}
	// rename the file
	return kvi_renameFile(filename,trashFile.ptr());
}

void KviApp::completeDirectory(const KviStr &word,KviPtrList<KviStr> * matches)
{
	KviStr szDir = word;
	KviStr szFile = word;
	szFile.cutToLast(KVI_PATH_SEPARATOR_CHAR);
	szDir.cutFromLast(KVI_PATH_SEPARATOR_CHAR);
	kvi_adjustFilePath(szDir);
	szDir.ensureLastCharIs(KVI_PATH_SEPARATOR_CHAR);
	QDir d(szDir.ptr());

	// FIXME: maybe check timestamps ? (old modules)
	QStringList sl = d.entryList(QDir::Dirs);
	QStringList::Iterator it;

	for(it = sl.begin();it != sl.end();++it)
	{
		KviStr * t = new KviStr(*it);
		if(kvi_strEqualCS(t->ptr(),".") || kvi_strEqualCS(t->ptr(),".."))
		{
			delete t;
		} else {
#ifdef COMPILE_ON_WINDOWS
			if(kvi_strEqualCIN(szFile.ptr(),t->ptr(),szFile.len()))
#else
			if(kvi_strEqualCSN(szFile.ptr(),t->ptr(),szFile.len()))
#endif
			{
				t->prepend(szDir);
				t->append(KVI_PATH_SEPARATOR_CHAR);
				matches->append(t);
			} else {
				delete t;
			}
		}
	}

	sl = d.entryList(QDir::Files);

	for(it = sl.begin();it != sl.end();++it)
	{
		KviStr * t = new KviStr(*it);
#ifdef COMPILE_ON_WINDOWS
			if(kvi_strEqualCIN(szFile.ptr(),t->ptr(),szFile.len()))
#else
			if(kvi_strEqualCSN(szFile.ptr(),t->ptr(),szFile.len()))
#endif
		{
			t->prepend(szDir);
			matches->append(t);
		} else {
			delete t;
		}
	}
}


//============= getChannelDumpLogFileName ==============//
//
// Returns a valid filename for the channel dump log
//

void KviApp::getChannelDumpLogFileName(KviStr &str)
{
	unsigned int logN = 0;
	KviStr fPrefix="kick_";
	do{
		KviStr fName(KviStr::Format,"%s%u.log",fPrefix.ptr(),logN);
		getLocalKvircDirectory(str,Log,fName.ptr());
		logN++;
		if(logN > 9999){
			// Time to clear logs! :)
			fPrefix+="x";
			logN = 0;
		}
	} while(kvi_fileExists(str.ptr()));
	kvi_adjustFilePath(str);
}

//===================== findUserFile =====================//
//
// Looks for an user file in the standard directories
//

bool KviApp::findUserFile(KviStr &szRetPath,const char *filename)
{
	static KviApp::KvircSubdir localsubdirs[5]={ Avatars , Incoming , Pics , Audio , Log };
	static KviApp::KvircSubdir globalsubdirs[3]={ Avatars , Pics , Audio };

	// Absolute paths can not be "found"... they are obvious
	if(kvi_isAbsolutePath(filename))
	{
		szRetPath=filename;
		kvi_adjustFilePath(szRetPath);
		return kvi_fileExists(filename);
	}

// FIXME: #warning "Use the user defined path ?"

	// lookup the magic by name match only...

	g_pMediaManager->lock();
	KviMediaType * m = g_pMediaManager->findMediaType(filename,false);
	if(m)
	{
		if(m->szSavePath.hasData())
		{
			szRetPath = m->szSavePath;
			szRetPath.ensureLastCharIs(KVI_PATH_SEPARATOR_CHAR);
			szRetPath.append(filename);
			if(kvi_fileExists(szRetPath.ptr()))
			{
				// check again the media type... check the magic too
				KviMediaType * mt = g_pMediaManager->findMediaType(szRetPath.ptr());
				if(mt == m)
				{
					g_pMediaManager->unlock();
					//if(retMediaType)*retMediaType = mt;
					kvi_adjustFilePath(szRetPath);
					return true;
				} // else mime type not matched...we should not be looking there!
			}
		}
	}
	g_pMediaManager->unlock();

// FIXME: #warning "LOOK ALSO IN THE SCRIPT DIRECTORIES"
/*
	if(g_pOptions->m_szLastScriptPath.hasData())
	{
		szP = g_pOptions->m_szLastScriptPath;
		if(!szP.lastCharIs('/'))szP.append('/');
		szP.append(filename);
		if(kvi_fileExists(szP.ptr())){ szRetPath = szP; return true; }
	}
*/

	int i;

	for(i= 0;i<5; i++)
	{
		getLocalKvircDirectory(szRetPath,localsubdirs[i],filename);
		if(kvi_fileExists(szRetPath.ptr()))return true;
	}

	for(i= 0;i<3; i++)
	{
		getLocalKvircDirectory(szRetPath,globalsubdirs[i],filename);
		if(kvi_fileExists(szRetPath.ptr()))return true;
	}

	// Last resort ...
	szRetPath = QDir::homeDirPath();
	szRetPath.ensureLastCharIs(KVI_PATH_SEPARATOR_CHAR);
	szRetPath.append(filename);
	if(kvi_fileExists(szRetPath.ptr()))return true;;

	szRetPath = filename;
	//if(retMediaType)*retMediaType = m;
	return false;

//gotit:
	//if(retMediaType)*retMediaType = g_pMediaManager->findMediaType(szRetPath.ptr());
//	return true;
}

//========================= findImage ===========================//
//
// Find a file inside the default "Image" directories
// This should be thread safe (and in fact IS used by other threads)
// This function should be used only for finding the INTERNAL
// pixmaps... (so we expect them to be in the pics directory or
// Eventually in user's $HOME
//

// FIXME: #warning "Check WHEN findImage is used and when findUserFile is used...we have a mess here"

bool KviApp::findImage(KviStr &szRetPath,const char *filename)
{
	// Find an user file...
	if(kvi_isAbsolutePath(filename)){
		szRetPath=filename;
		return kvi_fileExists(filename);
	}

	KviStr szP;
// FIXME: #warning "UNCOMMENT THIS AFTER OPTIONS HAVE BEEN IMPLEMENTED"
/*
	if(g_pOptions->m_szLastScriptPath.hasData())
	{
		szP = g_pOptions->m_szLastScriptPath;
		if(!szP.lastCharIs('/'))szP.append('/');
		szP.append(filename);
		if(kvi_fileExists(szP.ptr())){ szRetPath = szP; return true; }
		szP = g_pOptions->m_szLastScriptPath;
		if(!szP.lastCharIs('/'))szP.append('/');
		szP.append("pics/");
		szP.append(filename);
		if(kvi_fileExists(szP.ptr())){ szRetPath = szP; return true; }		
	}
*/

	getLocalKvircDirectory(szP,Pics,filename);
	if(kvi_fileExists(szP.ptr())){ szRetPath=szP; return true; }
	getGlobalKvircDirectory(szP,Pics,filename);
	if(kvi_fileExists(szP.ptr())){ szRetPath=szP; return true; }

	// Last resort
	szP = QDir::homeDirPath();
	szP.ensureLastCharIs(KVI_PATH_SEPARATOR_CHAR);
	szP.append(filename);
	if(kvi_fileExists(szP.ptr())){ szRetPath = szP; return true; }
	return false;
}

//=================== getDefaultDccSaveFilePath =================//
//
// Return the best place to save the file
//

void KviApp::getDefaultDccSaveFilePath(KviStr &path,const char *filename)
{
// FIXME: #warning "UNCOMMENT THIS AFTER MIME TYPES AND OPTIONS HAVE BEEN IMPLEMENTED"
/*
	KviMimeType * m = g_pOptions->m_pMimeManager->findMatch(filename,false);
	// the mime manager always returns something
	path = "";
	if(m->savePath.hasData()){
		// check the path validity
		if(*(m->savePath.ptr()) == '/'){
			// at least a valid dir...
			if(kvi_makeDir(m->savePath.ptr())){
				// Ok...the directory was made
				path = m->savePath;
			}
		}
	}
*/
	if(path.isEmpty())getLocalKvircDirectory(path,Incoming,0,true);
	if(path.isEmpty())path = QDir::homeDirPath(); //last resource...
}

//================== getReadOnlyConfigPath ====================//
//
// We want to READ a config file...find it
//

bool KviApp::getReadOnlyConfigPath(KviStr &buffer,const char *config_name,KvircSubdir sbd,bool bNoFail)
{
	// Take a look in the local directory....
	getLocalKvircDirectory(buffer,sbd,config_name);
	if(!kvi_fileExists(buffer.ptr()))
	{
		// No saved config yet... check for defaults
		KviStr tmp;
		getGlobalKvircDirectory(tmp,sbd,config_name);
		if(!kvi_fileExists(tmp.ptr()))
		{
			// No defaults...no such config file at all.
			if(bNoFail)getLocalKvircDirectory(buffer,sbd,config_name);
			return false;
		} else buffer = tmp; // get the defaults this time
	} // else file exists...
	return true;
}
