//
//   File : kvi_process.cpp
//   Creation date : Sun Jan 10 1999 17:20:34 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#define __LIBKVILIB__

#define _KVI_PROCESS_CPP_

#include "kvi_process.h"
#include "kvi_malloc.h"

#ifdef COMPILE_ON_WINDOWS
	#include <winsock2.h> // this will pull in windows.h
	//#include <windows.h>
#else
	#include <sys/types.h>
	#include <unistd.h>
#endif

bool kvi_runProcess(KviStr & szCommand,KviPtrList<KviStr> * pArgs)
{
#ifdef COMPILE_ON_WINDOWS
	KviStr szLocalCommand = szCommand;
	for(KviStr * s = pArgs->first();s;s = pArgs->next())
	{
		szLocalCommand.append(KviStr::Format," %s",s->ptr());
	}
	return WinExec(szLocalCommand.ptr(),SW_SHOW);
#else
	pid_t ret = fork();
	if(ret == -1)return false;
	if(ret == 0)
	{
		char ** pArgv = (char **)kvi_malloc(sizeof(char *) * (2 + pArgs->count()));
		pArgv[0] = szCommand.ptr();
		for(int i=1;i < (1 + pArgs->count());i++)pArgv[i] = pArgs->at(i - 1)->ptr();
		pArgv[1 + pArgs->count()] = 0;
		if(execvp(szCommand.ptr(),pArgv) == -1)
		{
			debug("Failed to execute \"%s\" (execvp() failed)",szCommand.ptr());
			exit(0);
		}

// FIXME: #warning "This SIGSEGVS!"

	} else return true;
#endif
}

//#warning "For Qt 3.0 , this stuff could be killed and substituted by QProcess"


/*

//
//  Simple versions of the KProcess and KProcessController classes
//  original code by (C) Christian Czezatke
//  e9025461@student.tuwien.ac.at
//  Really good work :)
//

//#define _KVI_DEBUG_CLASS_NAME_ "KviProcess"
//#define _KVI_DEBUG_CHECK_RANGE_
#ifndef _GNU_SOURCE
	// need this for SA_NOCLDSTOP...at least on my linux distro
	#define _GNU_SOURCE
#endif

#include "kvi_debug.h"

#include "kvi_process.h"
#include "kvi_malloc.h"
#include "kvi_locale.h"
#include "kvi_settings.h"

#include <qwindowdefs.h> //for qAddPostRoutine
#include <qfileinfo.h>
#include <qtimer.h>


#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>

#ifndef SA_NOCLDSTOP
	// uh ?...
	#include <signal.h>
	#ifndef SA_NOCLDSTOP
		#define SA_NOCLDSTOP 1
	#endif
#endif

//
// KviProcessController class
// Handles signals from dying children.
//


// FIXME: #warning "IN KDE COMPILATION MODE WE MIGHT HAVE PROLEMS WITH THE SINGLA HANDLERS!...WHY NOT USE KProcess THEN ?"

KviProcessController * g_pProcessController = 0;

KviStr KviProcess::m_szShellName("");

KviProcessController::KviProcessController()
: QObject(0,"global_process_controller")
{
	// We keep a list of processes currently alive.
	m_pProcessList = new KviPtrList<KviProcess>;
	m_pProcessList->setAutoDelete(false);      //newer delete the processes from here
 	// Need a pipe to communicate from the signal handler
// FIXME: #warning "USE kvi_error here!"
	if(0>pipe(m_fd))printf(strerror(errno));
	if(fcntl(m_fd[0], F_SETFL, O_NONBLOCK)==-1)printf(strerror(errno));
	// And a notifier for that
	m_pNotifier = new QSocketNotifier(m_fd[0],QSocketNotifier::Read);
	m_pNotifier->setEnabled(true);
	QObject::connect(m_pNotifier, SIGNAL(activated(int)),this, SLOT(slotDoHousekeeping(int)));
	setupSignalHandler();
	// ready
}

void theAbsoluteSigCHLDHandler(int )
{
	// A signal from a child was received (SIGPIPE or SIGCHILD)
	int status;
	pid_t this_pid;
	int saved_errno = errno;
	// since waitpid and write change errno, we have to save it and restore it
	// (Richard Stevens, Advanced programming in the Unix Environment)
	this_pid = waitpid(-1, &status, WNOHANG);
	// J6t: g_pProcessController might be already destroyed
	// debug("SIGCHLD for pid %d",this_pid);
	if((-1 != this_pid) && (g_pProcessController != 0))
	{
		::write(g_pProcessController->m_fd[1], &this_pid, sizeof(this_pid));
		::write(g_pProcessController->m_fd[1], &status, sizeof(status));
	}
	errno = saved_errno;
}

void KviProcessController::setupSignalHandler()
{
	struct sigaction act;
	act.sa_handler=&theAbsoluteSigCHLDHandler;
	sigemptyset(&(act.sa_mask));
	sigaddset(&(act.sa_mask), SIGCHLD);
	act.sa_flags = SA_NOCLDSTOP;
	// CC: take care of SunOS which automatically restarts interrupted system
	// calls (and thus does not have SA_RESTART)
#ifdef SA_RESTART
	act.sa_flags |= SA_RESTART;
#endif
	sigaction( SIGCHLD, &act, 0L);
	// kvi_thread takes care of the following SIGNAL
	//	act.sa_handler=SIG_IGN;
	//	sigemptyset(&(act.sa_mask));
	//	sigaddset(&(act.sa_mask), SIGPIPE);
	//	act.sa_flags = 0;
	//	sigaction( SIGPIPE, &act, 0L);
}

bool KviProcessController::doHousekeeping(int pid,int status,bool bNotifyProcess)
{
	KviProcess * proc = m_pProcessList->first();
	// Cleanup
	while(proc)
	{
		if(proc->pid() == pid)
		{
			// process has exited, so do emit the respective events
			if(bNotifyProcess)proc->processHasExited(status);
			//debug("Got child pid %d : killed",pid);
			return true; // only ONE process has exited
		}
		proc = m_pProcessList->next();
	}
	return false;
}

void KviProcessController::slotDoHousekeeping(int)
{
	// A process has exited , and data was written to the pipe
	// in the handler...here we remove dead processes from the list

	usleep(1000);
	int bytes_read;
	pid_t pid = (pid_t)0;
	int status;
	bytes_read  = ::read(m_fd[0], &pid, sizeof(pid_t));
	bytes_read += ::read(m_fd[0], &status, sizeof(int));
	if(bytes_read != sizeof(int)+sizeof(pid_t))
	{
		if(bytes_read < 0)
		{
			debug(__tr("Error: Could not read info from signal handler!"));
			debug(__tr("Error %d"),errno);
			if((errno == EINTR) || (errno == EINPROGRESS) || (errno == EAGAIN))
			{
				debug(__tr("Retrying"));
				slotDoHousekeeping(0);
				return;
			}
		} else {
			debug(__tr("Error: Could not read info from signal handler!"));
			debug(__tr("Readed %d bytes instead of %d+%d"),bytes_read,sizeof(int),sizeof(pid_t));
		}
	} // else debug("HOUSEKEEPING : CHILD OUT %d",pid);
	doHousekeeping(pid,status,true);
}

KviProcessController::~KviProcessController()
{
	struct sigaction act;

	// Turn off notification for processes that have exited
	act.sa_handler=SIG_IGN;
	sigemptyset(&(act.sa_mask));
	sigaddset(&(act.sa_mask), SIGCHLD);
	act.sa_flags = 0;
	sigaction( SIGCHLD, &act, 0L);

  	close(m_fd[0]);
	close(m_fd[1]);

	delete m_pProcessList;
	delete m_pNotifier;
}

void KviProcessController::addProcess(KviProcess *proc)
{
#ifdef COMPILE_KDE_SUPPORT
	setupSignalHandler(); // ensure that nobody eats our signals
#endif
	m_pProcessList->append(proc);
}

void KviProcessController::removeProcess(KviProcess *proc)
{
	m_pProcessList->removeRef(proc);
}


void kill_process_controller()
{
	// global handler to kill the g_pProcessController
	if(g_pProcessController)
	{
		delete g_pProcessController;
		g_pProcessController = 0;
	}
}

KviProcess::KviProcess()
{
	if(!g_pProcessController)
	{
		g_pProcessController = new KviProcessController();
		qAddPostRoutine(kill_process_controller);
	}

	m_pid = 0;
	m_bIsRunning = false;
	clearSockVariables();
	m_pStdoutNotifier = 0;
	m_pStderrNotifier = 0;
	g_pProcessController->addProcess(this);
}

KviProcess::~KviProcess()
{
	// First kill all sockets
	killSockets();
	// If is running , and killOnClose is reequested , well , KILL it
	if(m_bKillOnClose && m_bIsRunning)sendSignal(SIGKILL);
	// And tell the process controller to forget about us.
	if(g_pProcessController)g_pProcessController->removeProcess(this);
}

void KviProcess::clearSockVariables()
{
	m_stdinSock[0]=-1;
	m_stdinSock[1]=-1;
	m_stdoutSock[0]=-1;
	m_stdoutSock[1]=-1;
	m_stderrSock[0]=-1;
	m_stderrSock[1]=-1;
}

bool KviProcess::setupSockets()
{
	int error = socketpair(AF_UNIX,SOCK_STREAM,0,m_stdinSock);
	if(error != 0)return false;
	error = socketpair(AF_UNIX,SOCK_STREAM,0,m_stderrSock);
	if(error != 0){
		close(m_stdinSock[0]);
		close(m_stdinSock[1]);
		clearSockVariables();
		return false;
	}
	error = socketpair(AF_UNIX,SOCK_STREAM,0,m_stdoutSock);
	if(error != 0){
		close(m_stdinSock[0]);
		close(m_stdinSock[1]);
		close(m_stdoutSock[0]);
		close(m_stdoutSock[1]);
		clearSockVariables();
		return false;
	}
	m_iSocketErrors = 0;
	return true;
}

void KviProcess::killSockets()
{
	if(m_stdinSock[0]!=-1)close(m_stdinSock[0]);
	if(m_stdinSock[1]!=-1)close(m_stdinSock[1]);
	if(m_stdoutSock[0]!=-1)close(m_stdoutSock[0]);
	if(m_stdoutSock[1]!=-1)close(m_stdoutSock[1]);
	if(m_stderrSock[0]!=-1)close(m_stderrSock[0]);
	if(m_stderrSock[1]!=-1)close(m_stderrSock[1]);
	clearSockVariables();
	if(m_pStdoutNotifier != 0)delete m_pStdoutNotifier;
	if(m_pStderrNotifier != 0)delete m_pStderrNotifier;
	m_pStdoutNotifier = 0;
	m_pStderrNotifier = 0;
}

bool KviProcess::run(int (*routine)(void *),void * argument,bool bCommunicate,bool bKillOnClose)
{
	if(m_bIsRunning)return false;
	m_bKillOnClose = bKillOnClose;
	m_bCommunicate = bCommunicate;
	if(bCommunicate){
		if(!setupSockets())return false;
	}

	m_pid = fork();
	switch(m_pid){
		case 0:
			//////////////////////////////////////////
			// CHILD Process : Success
			//
			if(bCommunicate){
				if(!child_setupSockets()){
					killSockets();
					debug(__tr("Could not setup child communication"));
				}
			}//else setpgid(0,0); ?????

			{
				int res = (*routine)(argument);
//#warning "REMOVEME";
//				fprintf(stderr,"EXITING HERE");
				fflush(stdout);
				fflush(stderr);
				exit(res);
			}
			//
			//
			//////////////////////////////////////////
		break;
		case -1: //Parent process : Failed
			killSockets();
			return false;
		break;
		default: //Parent process : Success
			if(bCommunicate){
				if(!parent_setupSockets()){
					killSockets();
					debug(__tr("Could not setup parent communication"));
				}
			}
			m_bIsRunning = true;
			return true;
		break;
	}
	//NEWER Here...
}

// FIXME: #warning "DOUBLE CHECK PROBLEMS WITH THE X Socket!!!"


bool KviProcess::run(const char * commandline,bool bCommunicate,bool bKillOnClose,bool bExecInSubshell)
{
	if(m_bIsRunning)return false;
	if((!commandline) || (!(*commandline)))return false;

	m_bCommunicate = bCommunicate;
	m_bKillOnClose = bKillOnClose;
	//Get the argument list
	KviPtrList<KviStr> argList;
	argList.setAutoDelete(true);

	KviStr arg;
	KviStr shellName;
	if(!findShell(shellName))shellName = "/bin/sh"; // just a default

	if(bExecInSubshell){
		argList.append(new KviStr(shellName.ptr()));
		argList.append(new KviStr("-c"));
		argList.append(new KviStr(commandline));
	} else {
		while(*commandline){
			commandline = kvi_extractToken(arg,commandline);
			if(arg.hasData())argList.append(new KviStr(arg));
		}
	}

	if(argList.isEmpty())return false;
	if(bCommunicate){
		if(!setupSockets())return false;
	}

	char ** pArgs=(char **)kvi_malloc((argList.count()+1) * sizeof(char*));
	uint cur=0;
	for(KviStr * pA=argList.first();pA && (cur<argList.count());pA=argList.next()){
		pArgs[cur]=pA->ptr();
		cur++;
	}
	pArgs[cur]=0; //Null terminator

	m_pid = fork();

	switch(m_pid){
		case 0:
			//////////////////////////////////////////
			// CHILD Process : Success
			//
			if(bCommunicate){
				if(!child_setupSockets()){
					killSockets();
					debug(__tr("Could not setup child communication"));
				}
			}//else setpgid(0,0); ?????
			execvp(pArgs[0],pArgs);
			debug(__tr("execvp failed for file %s : %s"),pArgs[0],strerror(errno));
//			debug("commandline was %s",commandline);
			exit(-1);
			//
			//
			//////////////////////////////////////////
		break;
		case -1: //Parent process : Failed
			killSockets();
			kvi_free(pArgs);
			return false;
		break;
		default: //Parent process : Success
			if(bCommunicate){
				if(!parent_setupSockets()){
					killSockets();
					debug(__tr("Could not setup parent communication"));
				}
			}
			m_bIsRunning = true;
			kvi_free(pArgs);
			return true;
		break;
	}
	//NEWER Here...
}

bool KviProcess::parent_setupSockets()
{
	// Kill the child socket side...
	close(m_stdinSock[PROC_SOCKET_CHILD]);
	close(m_stdoutSock[PROC_SOCKET_CHILD]);
	close(m_stderrSock[PROC_SOCKET_CHILD]);
	m_stdinSock[PROC_SOCKET_CHILD] =-1;
	m_stdoutSock[PROC_SOCKET_CHILD]=-1;
	m_stderrSock[PROC_SOCKET_CHILD]=-1;

	if(fcntl(m_stdinSock[PROC_SOCKET_PARENT],F_SETFL,O_NONBLOCK)==-1)return false;
	if(fcntl(m_stdoutSock[PROC_SOCKET_PARENT],F_SETFL,O_NONBLOCK)==-1)return false;
	if(fcntl(m_stderrSock[PROC_SOCKET_PARENT],F_SETFL,O_NONBLOCK)==-1)return false;

	m_pStdoutNotifier = new QSocketNotifier(m_stdoutSock[PROC_SOCKET_PARENT],QSocketNotifier::Read,this);
	QObject::connect(m_pStdoutNotifier,SIGNAL(activated(int)),this,SLOT(receivedStdout(int)));
	m_pStdoutNotifier->setEnabled(true);

	m_pStderrNotifier = new QSocketNotifier(m_stderrSock[PROC_SOCKET_PARENT],QSocketNotifier::Read,this);
	QObject::connect(m_pStderrNotifier,SIGNAL(activated(int)),this,SLOT(receivedStderr(int)));
	m_pStderrNotifier->setEnabled(true);

	return true;
}

bool KviProcess::child_setupSockets()
{
//	__enter("child_setupSockets");
	// Kill the parent socket side...
	close(m_stdinSock[PROC_SOCKET_PARENT]);
	close(m_stdoutSock[PROC_SOCKET_PARENT]);
	close(m_stderrSock[PROC_SOCKET_PARENT]);
	m_stdinSock[PROC_SOCKET_PARENT] =-1;
	m_stdoutSock[PROC_SOCKET_PARENT]=-1;
	m_stderrSock[PROC_SOCKET_PARENT]=-1;

	if(dup2(m_stdinSock[PROC_SOCKET_CHILD] ,STDIN_FILENO) ==-1)return false;
	if(dup2(m_stdoutSock[PROC_SOCKET_CHILD],STDOUT_FILENO)==-1)return false;
	if(dup2(m_stderrSock[PROC_SOCKET_CHILD],STDERR_FILENO)==-1)return false;

	// linger a while, to send all the output...
	struct linger l;
	l.l_onoff = 1;
	l.l_linger = 100; // linger 1 sec

	if(setsockopt(m_stdoutSock[PROC_SOCKET_CHILD],SOL_SOCKET,SO_LINGER,(char*)&l,sizeof(l))==-1)return false;
	if(setsockopt(m_stderrSock[PROC_SOCKET_CHILD],SOL_SOCKET,SO_LINGER,(char*)&l,sizeof(l))==-1)return false;
	return true;
}

void KviProcess::receivedStdout(int fd)
{
	char buffer[1024];
	int len;
	do {
		//#warning "Check for read() errors here (EAGAIN ?)"
		len =read(fd,buffer,1024);
		if(len > 0)emit processStdout(this,buffer,len); //What means an EOF from a process ?
		else {
			if((len < 0) && ((errno == EAGAIN) || (errno == EINTR)))return;
			m_iSocketErrors++;
			if(m_iSocketErrors > 10)
			{
				// Maybe we have missed the SIGCHLD
				//debug("Problems with process communication (pid=%d)(%d): killing",pid(),errno);
				if(g_pProcessController->doHousekeeping(pid(),0,false))processTerminated(0);
			}
			return;
		}
		//		else m_pStdoutNotifier->setEnabled(false); //I'll follow the example of KProcess that disables the notifier
	} while(len == 1024);
}

void KviProcess::receivedStderr(int fd)
{
	char buffer[1024];
	int len;
	do {
		len =read(fd,buffer,1024);
		if(len > 0)emit processStderr(this,buffer,len); //What means an EOF from a process ?
		else {
			if((len < 0) && ((errno == EAGAIN) || (errno == EINTR)))return;
			m_iSocketErrors++;
			if(m_iSocketErrors > 10)
			{
				// Maybe we have missed the SIGCHLD...
				// debug("Problems with process communication (pid=%d)(%d): killing",pid(),errno);
				if(g_pProcessController->doHousekeeping(pid(),0,false))processTerminated(0);
			}
			return;
		}
//		else m_pStderrNotifier->setEnabled(false); //I'll follow the example of KProcess that disables the notifier
	} while(len == 1024);
}

bool KviProcess::sendSignal(int sig)
{
	if(m_bIsRunning)
	{
		if(::kill(m_pid,sig)==-1)return false;
	}
	return true;
}

void KviProcess::processTerminated(int status)
{
	killSockets();
	m_bIsRunning= false;
	m_bKillOnClose = false;
	emit terminated(this,status);
}

void KviProcess::processHasExited(int status)
{
	// read the last data in the buffer (sometimes the SIGCHLD arrives before the in_socket notifier fires)
	if(m_bCommunicate)
	{
		receivedStderr(m_stderrSock[PROC_SOCKET_PARENT]);
		receivedStdout(m_stdoutSock[PROC_SOCKET_PARENT]);
	}
	processTerminated(status);
}

bool KviProcess::findShell(KviStr &buffer)
{
	if(m_szShellName.hasData())
	{
		buffer = m_szShellName;
		return true;
	}

	KviStr tmp = getenv("SHELL");
	tmp.stripWhiteSpace();
	if(tmp.isEmpty())
	{
		debug(__tr("No SHELL enviroinement variable defined!"));
		return false;
	}
	QFileInfo f(QString(tmp.ptr()));
	if(f.isExecutable())
	{
		m_szShellName = tmp;
		buffer = m_szShellName;
		return true;
	} else debug(__tr("SHELL is defined to %s but seems to be not executable"),tmp.ptr());
	return false;
}

bool KviProcess::writeData(const char *buffer,int len)
{
	// if(len == -1)len = strlen(buffer);
	// debug("Writing %s to process",buffer);
	return (write(m_stdinSock[PROC_SOCKET_PARENT],buffer,len) == len);
}


#include "kvi_process.moc"
*/
