//
//   File : kvi_locale.cpp
//   Creation date : Fri Mar 19 1999 19:08:41 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#define __LIBKVILIB__

//#define _KVI_DEBUG_CHECK_RANGE_
#include "kvi_debug.h"

#include "kvi_bswap.h"

#define _KVI_LOCALE_CPP_
#include "kvi_locale.h"


#include <qglobal.h> //for debug()

#include "kvi_string.h"

static KviStr g_szLang("en"); //default locale name

char *  kvi_getLocaleName()
{
	return g_szLang.ptr();
}


#ifdef COMPILE_LOCALE_STUFF

#include "kvi_fileutils.h"

#include <qdict.h>

/////////////////////////////////////////////////////////////////////////////////////////////////////
//
//   The following code was extracted and adapted from gettext.h and gettextP.h
//   from the GNU gettext package.
//   
//   Internal header for GNU gettext internationalization functions.
//   Copyright (C) 1995, 1997 Free Software Foundation, Inc.
//
//   This program is free software; you can redistribute it and/or modify
//   it under the terms of the GNU General Public License as published by
//   the Free Software Foundation; either version 2, or (at your option)
//   any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU Library General Public
//   License along with the GNU C Library; see the file COPYING.LIB.  If not,
//   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//   Boston, MA 02111-1307, USA. 
//
/////////////////////////////////////////////////////////////////////////////////////////////////////

#include <stdio.h>

#if HAVE_LIMITS_H || _LIBC
	#include <limits.h>
#endif

/* The magic number of the GNU message catalog format.  */
#define KVI_LOCALE_MAGIC 0x950412de
#define KVI_LOCALE_MAGIC_SWAPPED 0xde120495

/* Revision number of the currently used .mo (binary) file format.  */
#define MO_REVISION_NUMBER 0




/* Header for binary .mo file format.  */
struct GnuMoFileHeader
{
	/* The magic number.  */
	Q_UINT32 magic;
	/* The revision number of the file format.  */
	Q_UINT32 revision;
	/* The number of strings pairs.  */
	Q_UINT32 nstrings;
	/* Offset of table with start offsets of original strings.  */
	Q_UINT32 orig_tab_offset;
	/* Offset of table with start offsets of translation strings.  */
	Q_UINT32 trans_tab_offset;
	/* Size of hashing table.  */
	Q_UINT32 hash_tab_size;
	/* Offset of first hashing entry.  */
	Q_UINT32 hash_tab_offset;
};

struct GnuMoStringDescriptor
{
	/* Length of addressed string.  */
	Q_UINT32 length;
	/* Offset of string in file.  */
	Q_UINT32 offset;
};

#define KVI_SWAP_IF_NEEDED(flag,value) (flag ? kvi_swap32(value) : (value))

///////////////////////////////////////////////////////////////////////////////////////////////
//   End of gettext.h & gettextP.h
///////////////////////////////////////////////////////////////////////////////////////////////

//
// This function attempts to determine the current locale
// and then load the corresponding translation file
// from the KVIrc locale directory
// Returns true if the locale was correctly set
// i.e. the locale is C or POSIX (no translation needed)
//     or the locale is correctly defined and the
//     translation map was sucesfully loaded
//

#include <stdlib.h> // for getenv

static QDict<KviStr> * g_pMessages   = 0;
static KviTranslator * g_pTranslator = 0;

static int somePrimeNumbers[90]=
{
	257 , 521 , 769 , 1031, 1087, 1091, 1103, 1117, 1123, 1151, // Incomplete *.mo files
	1163, 1171, 1181, 1193, 1201, 1213, 1217, 1223, 1229, 1231, // Complete *.mo files
	1237, 1249, 1259, 1277, 1283, 1289, 1291, 1297, 1307, 1319,
	1321, 1327, 1361, 1367, 1373, 1381, 1399, 1409, 1423, 1433,
	1447, 1459, 1471, 1481, 1493, 1511, 1523, 1531, 1543, 1553,
	1567, 1571, 1583, 1597, 1609, 1619, 1627, 1637, 1657, 1667, // Too big for KVIrc *.mo files
	1693, 1709, 1721, 1733, 1741, 1753, 1777, 1789, 1811, 1831,
	1907, 2069, 2111, 2221, 2309, 2441, 2531, 2617, 2731, 2837,
	2903, 3121, 3329, 3331, 3767, 4127, 5051, 6089, 7039, 9973
};

int kvi_getFirstBiggerPrime(int number)
{
	for(int i=0;i<90;i++){
		if(somePrimeNumbers[i] >= number)return somePrimeNumbers[i];
	}
	return 9973; //error!
}

bool kvi_initLocale(QApplication * app,const char *localeDir)
{
	// Find the current locale
	g_szLang = getenv("LANG");
	g_szLang.toLower();
	g_szLang.stripWhiteSpace();

	if(g_szLang.isEmpty())
	{
		g_szLang = "en";
		return true; // No locale specified...use default (english is built-in)
	}

	g_szLang.toLower();

	if(g_szLang.findFirstIdx('_') != -1)
	{
		// ops.... the locale is set in the "strange" form (it_IT or things like that)
		// we'll use the first part
		g_szLang.cutFromFirst('_',true);
	}

	if(kvi_strEqualCI(g_szLang.ptr(),"c") || kvi_strEqualCI(g_szLang.ptr(),"en") ||
		kvi_strEqualCI(g_szLang.ptr(),"uk") || kvi_strEqualCI(g_szLang.ptr(),"us") ||
		kvi_strEqualCI(g_szLang.ptr(),"gb") || kvi_strEqualCI(g_szLang.ptr(),"posix")){
		g_szLang = "en";
		return true; // english is (built-in)
	}

	//__debug_1arg("Locale is set to %s",g_szLang.ptr());

	// Compute the name of the messages file to load
	KviStr messagesFile = localeDir;
	messagesFile.append(g_szLang);
	messagesFile.append(".mo");

	// Try to load the header
	QFile f(messagesFile.ptr());
	if(!f.open(IO_ReadOnly)){
		debug("KviLocale: Failed to read messages file %s: probably doesn't exist",messagesFile.ptr());
		return false;
	}

	GnuMoFileHeader hdr;

	if(f.readBlock((char *)&hdr,sizeof(GnuMoFileHeader)) < (int)sizeof(GnuMoFileHeader)){
		debug("KviLocale: Failed to read header of %s",messagesFile.ptr());
		f.close();
		return false;
	}

	bool bMustSwap = false;

	if(hdr.magic != KVI_LOCALE_MAGIC){
		if(hdr.magic == KVI_LOCALE_MAGIC_SWAPPED){
			debug("KviLocale: Swapped magic for file %s: swapping data too",messagesFile.ptr());
			bMustSwap = true;
		} else {
			debug("KviLocale: Bad locale magic for file %s: not a *.mo file ?",messagesFile.ptr());
			f.close();
			return false;
		}
	}

	if(KVI_SWAP_IF_NEEDED(bMustSwap,hdr.revision) != MO_REVISION_NUMBER){
		debug("KviLocale: Invalid *.mo file revision number for file %s",messagesFile.ptr());
		f.close();
		return false;
	}

	int numberOfStrings = KVI_SWAP_IF_NEEDED(bMustSwap,hdr.nstrings);
	if(numberOfStrings <= 0){
		debug("KviLocale: No translated messages found in file %s",messagesFile.ptr());
		f.close();
		return false;
	}
	if(numberOfStrings >= 9972){
		debug("Number of strings too big...sure that it is a KVIrc catalog file ?");
		numberOfStrings = 9972;
	}
	// return back
	f.at(0);

	unsigned int fSize = f.size();
	char * buffer = new char[fSize];

	if(f.readBlock(buffer,fSize) < (int)fSize){
		debug("KviLocale: Error while reading the translation file %s",messagesFile.ptr());
		delete buffer;
		f.close();
		return false;
	}

	// Check for broken *.mo files
	if(fSize < (24 + (sizeof(GnuMoStringDescriptor) * numberOfStrings))){
		debug("KviLocale: Broken translation file %s (too small for all descriptors)",messagesFile.ptr());
		delete buffer;
		f.close();
		return false;
	}

	GnuMoStringDescriptor * origDescriptor  = (GnuMoStringDescriptor *)(buffer + KVI_SWAP_IF_NEEDED(bMustSwap,hdr.orig_tab_offset));
	GnuMoStringDescriptor * transDescriptor = (GnuMoStringDescriptor *)(buffer + KVI_SWAP_IF_NEEDED(bMustSwap,hdr.trans_tab_offset));

	// Check again for broken *.mo files
	int expectedFileSize = KVI_SWAP_IF_NEEDED(bMustSwap,transDescriptor[numberOfStrings - 1].offset) +
							KVI_SWAP_IF_NEEDED(bMustSwap,transDescriptor[numberOfStrings - 1].length);

	if(fSize < (unsigned int)expectedFileSize){
		debug("KviLocale: Broken translation file %s (too small for all the message strings)",messagesFile.ptr());
		delete buffer;
		f.close();
		return false;
	}

	// Ok...we can run now

	int dictSize = kvi_getFirstBiggerPrime(numberOfStrings);
	g_pMessages = new QDict<KviStr>(dictSize);
	g_pMessages->setAutoDelete(true);

	//__debug_2arg("Number of strings : %d , dict size %d",numberOfStrings,dictSize);

	for(int i=0;i < numberOfStrings;i++){
	// FIXME: "Check for NULL inside strings here ?"
	//		debug("original seems to be at %u and %u byttes long",KVI_SWAP_IF_NEEDED(bMustSwap,origDescriptor[i].offset),
	//					KVI_SWAP_IF_NEEDED(bMustSwap,origDescriptor[i].length));
	//		debug("translated seems to be at %u and %u byttes long",KVI_SWAP_IF_NEEDED(bMustSwap,transDescriptor[i].offset),
	//					KVI_SWAP_IF_NEEDED(bMustSwap,transDescriptor[i].length));

		KviStr original((char *)(buffer + KVI_SWAP_IF_NEEDED(bMustSwap,origDescriptor[i].offset)),
								KVI_SWAP_IF_NEEDED(bMustSwap,origDescriptor[i].length));
		// In some (or all?) *.mo files the first string
		// is zero bytes long and the translated one contains
		// informations about the translation
		if(original.len() == 0)continue;
		KviStr translated((char *)(buffer + KVI_SWAP_IF_NEEDED(bMustSwap,transDescriptor[i].offset)),
								KVI_SWAP_IF_NEEDED(bMustSwap,transDescriptor[i].length));
		g_pMessages->insert(original.ptr(),new KviStr(translated));
	}

	delete buffer;
	f.close();

	g_pTranslator = new KviTranslator(app,"kvirc_translator");
	app->installTranslator(g_pTranslator);

	return true;
}

void kvi_destroyLocale(QApplication * app)
{
	if(g_pMessages)delete g_pMessages;
	g_pMessages = 0;

	if(g_pTranslator){
		app->removeTranslator(g_pTranslator);
		delete g_pTranslator;
		g_pTranslator = 0;
	}
}

const char * kvi_translate(const char *text)
{
	if(g_pMessages){
		KviStr * aux = g_pMessages->find(text);
		if(aux)return aux->ptr();
	}
	return text;
}


KviTranslator::KviTranslator(QObject * parent,const char * name)
: QTranslator(parent,name)
{
}

KviTranslator::~KviTranslator()
{
}

QString KviTranslator::find(const char *,const char * message) const
{
	if(g_pMessages)
	{
		KviStr * aux = g_pMessages->find(message);
		if(aux)return __c2q(aux->ptr());
	}
	return QString::null;
}

#include "kvi_locale.moc"

#else //!COMPILE_LOCALE_STUFF

bool kvi_initLocale(QApplication *,const char *)
{
	g_szLang = "en";
	//debug("This program was build without the locale support.");
	return false;
}

void kvi_destroyLocale(QApplication *)
{
}




#endif //!COMPILE_LOCALE_STUFF
