//
//   File : kvi_nickserv.cpp
//   Creation date : Thu Aug 09 2001 17:44:56 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2001 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#define __LIBKVILIB__

#include "kvi_nickserv.h"
#include "kvi_config.h"

/*
	@doc: nickserv_proto
	@title:
		NickServ support
	@keyterms:
		NickServ, automatic authentication with NickServ
	@type:
		generic
	@short:
		Automatic authentication with NickServ
	@body:
		KVIrc supports automatic authentication with the NickServ service.[br]
		This service is commonly implemented on major IRC networks: basically
		it is a program that allows users to register their nickname and protect
		it from being stolen by others.[br] The NickServ protocol is
		not standardized (at the time that I'm writing this doc) and automatic
		authentication is a pure experimental protocol.[br]
		Once you get on IRC with a registered nickname , the NickServ will
		ask you for identification by sending you a NOTICE.[br]
		The message will look in a way similar to the following:[br]
		<b>You're using a registered nickname: if this is your nick,
		please type /msg NickServ IDENTIFY password, otherwise please
		choose another nickname</b>.[br]
		The message is often broken in two or three lines of text.[br]
		KVIrc will try to recognize such messages bu looking for the words
		"NickServ" , "identify"  and "/msg" in the same line.[br]
		As you can see , this is a complete "guess", and might not work on all the netoworks.[br]
		Actually, this has been tested on DalNet and Openprojects: I have no idea about other networks.[br]
		KVIrc will also match the current server name with the one that you provide
		in the authentication rule and will pretend that the NickServ mask match
		the mask that you provide.[br] This is all the security that KVIrc can
		provide you against malicious users that want to stole your NickServ password.[br]
		On DalNet and Openprojects this machinery is proven to work and have no security flaws
		(unless the NickServ protocol changes).[br]
		On most IRC networks the nick NickServ is even juped, to ensure that nobody will abuse of it.[br]
		Please note that many network policies suggest to avoid automatic authentication
		with NickServ.[br]I have implemented it because I know that it works on the networks
		that I'm usually on.[br]You have to check that this protocol works on your network and
		then eventually use it at your own risk.[br]
*/

KviNickServDataBase::KviNickServDataBase()
{
	m_pEntryDict = new QAsciiDict<KviNickServEntryList>(17,false);
	m_pEntryDict->setAutoDelete(true);
}

KviNickServDataBase::~KviNickServDataBase()
{
	delete m_pEntryDict;
}

KviNickServEntryList * KviNickServDataBase::getEntryList(const char * nick)
{
	KviNickServEntryList * l = findEntryList(nick);
	if(!l)
	{
		l = new KviNickServEntryList;
		l->setAutoDelete(true);
		m_pEntryDict->insert(nick,l);
	}
	return l;
}

KviNickServEntry * KviNickServDataBase::findEntry(const char * szNick,const char * szServer,KviIrcMask * nickServMask)
{
	KviNickServEntryList * l = findEntryList(szNick);
	if(!l)return 0; // no way

	for(KviNickServEntry * e = l->first();e;e = l->next())
	{
		if(kvi_matchString(e->szServerMask.ptr(),szServer))
		{
			// might be
			if(nickServMask->matchedBy(e->szNickServMask.ptr()))
			{
				return e;
			}
		}
	}

	return 0;
}

const char * KviNickServDataBase::findIdentifyCommand(const char * szNick,const char * szServer,KviIrcMask * nickServMask)
{
	KviNickServEntry * e = findEntry(szNick,szServer,nickServMask);
	return e ? e->szIdentifyCommand.ptr() : 0;
}

void KviNickServDataBase::insertEntry(const char * szNick,const char * szServerMask,const char * szNickServMask,const char * szIdentifyCommand)
{
	KviNickServEntryList * l = getEntryList(szNick);
	KviNickServEntry * e = new KviNickServEntry;
	e->szServerMask = szServerMask;
	e->iServerNonWildcards = e->szServerMask.len() - e->szServerMask.occurences('*');
	e->szNickServMask = szNickServMask;
	e->szIdentifyCommand = szIdentifyCommand;

	// Now insert

	KviNickServEntry * existing = l->first();
	int idx = 0;

	if(e->iServerNonWildcards != e->szServerMask.len())
	{
		// There are some wildcards...skip all the non wildcard entries
		while(existing)
		{
			if(existing->iServerNonWildcards != existing->szServerMask.len())break;
			existing = l->next();
			idx++;
		}

		// ok...we're pointing at the first wildcarded entry (or to 0)

	}

	while(existing)
	{
		// more non-wildcard chars go first
		// now our entry goes BEFORE the first entry that has less non-wildcards
		if(existing->iServerNonWildcards < e->iServerNonWildcards)
		{
			l->insert(idx,e);
			return;
		}
		existing = l->next(),
		idx++;
	}

	// append at the end

	l->append(e);
}

void KviNickServDataBase::clear()
{
	delete m_pEntryDict;
	m_pEntryDict = new QAsciiDict<KviNickServEntryList>(17,false);
	m_pEntryDict->setAutoDelete(true);
}

void KviNickServDataBase::save(const char * filename)
{
	KviConfig cfg(filename);

	cfg.clear();

	QAsciiDictIterator<KviNickServEntryList> it(*m_pEntryDict);

	int idx = 0;
	int iTotal = 0;

	while(it.current())
	{
		for(KviNickServEntry *e = it.current()->first();e;e = it.current()->next())
		{
			KviStr tmp(KviStr::Format,"szServerMask_%d",idx);
			cfg.writeEntry(tmp.ptr(),e->szServerMask.ptr());
			tmp.sprintf("szNickServMask_%d",idx);
			cfg.writeEntry(tmp.ptr(),e->szNickServMask.ptr());
			tmp.sprintf("szIdentifyCommand_%d",idx);
			cfg.writeEntry(tmp.ptr(),e->szIdentifyCommand.ptr());
			tmp.sprintf("szNick_%d",idx);
			cfg.writeEntry(tmp.ptr(),it.currentKey());
			idx++;
		}
		iTotal += it.current()->count();
		++it;
	}

	cfg.writeEntry("nEntries",iTotal);
}

void KviNickServDataBase::load(const char * filename)
{
	clear();

	KviConfig cfg(filename);

	int iNumEntries = cfg.readIntEntry("nEntries",0);

	for(int idx = 0;idx < iNumEntries;idx++)
	{
		KviStr tmp(KviStr::Format,"szServerMask_%d",idx);
		KviStr szServerMask = cfg.readEntry(tmp.ptr(),"irc.unknown.net");
		tmp.sprintf("szNickServMask_%d",idx);
		KviStr szNickServMask = cfg.readEntry(tmp.ptr(),"NickServ!*@*");
		tmp.sprintf("szIdentifyCommand_%d",idx);
		KviStr szIdentifyCommand = cfg.readEntry(tmp.ptr(),"");
		tmp.sprintf("szNick_%d",idx);
		KviStr szNick = cfg.readEntry(tmp.ptr(),"");
		if(szNick.hasData() && szIdentifyCommand.hasData())
		{
			insertEntry(szNick.ptr(),szServerMask.ptr(),szNickServMask.ptr(),szIdentifyCommand.ptr());
		}
	}
}
