#ifndef _KVI_REGUSERSDB_H_
#define _KVI_REGUSERSDB_H_
//=================================================================================================
//
//   File : kvi_regusersdb.h
//   Creation date : Sat Sep 09 2000 15:30:56 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//=================================================================================================

//
// REGISTERED USERS
//     
//     Here we manage users resigered by mask and their (generic!) properties
//

#include "kvi_settings.h"
#include "kvi_string.h"
#include "kvi_ircmask.h"
#include "kvi_debug.h"

#include "kvi_list.h"
#include <qasciidict.h>


class KviRegisteredUserDataBase;

//=================================================================================================
//
// KviRegisteredUser
//

class KVILIB_API KviRegisteredUser
{
	friend class KviRegisteredUserDataBase;
public:
	KviRegisteredUser(const char * name);
	~KviRegisteredUser();
private:
	KviStr                       m_szName;
	QAsciiDict<KviStr>         * m_pPropertyDict;   // owned properties
	KviPtrList<KviIrcMask>     * m_pMaskList;       // owned masks
protected:
	// mask ownership is transferred! (always!) returns false if the mask was already there
	bool addMask(KviIrcMask * mask);
	bool removeMask(KviIrcMask * mask);
	KviIrcMask * findMask(const KviIrcMask &mask);
public:
	const char * name(){ return m_szName.ptr(); };
	bool matches(const KviIrcMask &mask);
	bool matchesFixed(const KviIrcMask &mask);

	void setProperty(const char * name,const char * value);

	const char * getProperty(const char * name);       // returns 0 if the property is not there
	bool getProperty(const char * name,KviStr &value); // returns false if the property is not there
	bool getBoolProperty(const char * name);           // returns true if the property is there and is true
	// the propertyDict may be 0!
	QAsciiDict<KviStr> * propertyDict(){ return m_pPropertyDict; };
	// this is never zero (but may contain no masks)
	KviPtrList<KviIrcMask> * maskList(){ return m_pMaskList; };
};

//============================================================================================================
//
// KviRegisteredMask
//

class KVILIB_API KviRegisteredMask
{
private:
	KviRegisteredUser * m_pUser;               // pointer , not owned!
	KviIrcMask        * m_pMask;               // pointer , not owned!
	int                 m_iMaskNonWildChars;
public:
	KviRegisteredMask(KviRegisteredUser * u,KviIrcMask * m);
	~KviRegisteredMask(){};
public:
	int                 nonWildChars(){ return m_iMaskNonWildChars; };
	KviRegisteredUser * user(){ return m_pUser; };
	KviIrcMask        * mask(){ return m_pMask; };
};

typedef KviPtrList<KviRegisteredMask> KviRegisteredMaskList;

//=================================================================================================
//
// KviRegisteredUsersDb
//
//    Manages a set of KviRegisteredUser instances stored in the m_pUserDict dictionary
//    The users are identified by masks stored in m_pMaskDict and m_pWildMaskList
//    m_pMaskDict contains lists of non wild-nick KviRegisteredMask that point to users
//    m_pWildMaskList is a list of wild-nick KviRegisteredMask that point to users
//

class KVILIB_API KviRegisteredUserDataBase
{
public:
	KviRegisteredUserDataBase();
	~KviRegisteredUserDataBase();
private:
	QAsciiDict<KviRegisteredUser>     * m_pUserDict; // unique namespace, owns the objects, does not copy keys
	QAsciiDict<KviRegisteredMaskList> * m_pMaskDict; // owns the objects, copies the keys
	KviRegisteredMaskList             * m_pWildMaskList; // owns the objects
public:
	void copyFrom(KviRegisteredUserDataBase * db);
	KviRegisteredUser * addUser(const char * name); // returns 0 if already there
	KviRegisteredUser * getUser(const char * name); // returns existing or adds
	bool removeUser(const char * name);
	KviRegisteredUser * findUserByName(const char * name){ return m_pUserDict->find(name); };
	// mask must be allocated on the heap and the ownership is transferred!
	// returns non zero if there is already an user with this mask (returns the pointer to it!)
	KviRegisteredUser * addMask(KviRegisteredUser * u,KviIrcMask * mask);
	bool removeMaskByPointer(KviIrcMask * mask);
	bool removeMask(const KviIrcMask &mask);
//	KviRegisteredUser * findMatchingUser(const KviIrcMask &mask);
	KviRegisteredUser * findMatchingUser(const char * nick,const char * user,const char * host);
	KviRegisteredUser * findUserWithMask(const KviIrcMask &mask);
	KviRegisteredMask * findExactMask(const KviIrcMask &mask);
	KviRegisteredMask * findMatchingMask(const char * nick,const char * user,const char * host);
	//Only used in few places (actually one) of the code, but lot of times;perfect for inlining...
	//FIXME XXX: Si el usuario con esa mscara no existe petar?
	//bool isIgnoredUser(const char * nick,const char * user,const char * host);
	void load(const char * filename);
	void save(const char * filename);

	QAsciiDict<KviRegisteredUser> * userDict(){ return m_pUserDict; };
};




#endif //_KVI_REGUSERSDB_H_
