//
//   File : kvi_regusersdb.cpp
//   Creation date : Sat Sep 09 2000 15:46:12 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#define __LIBKVILIB__
//#define _KVI_DEBUG_CHECK_RANGE_


#include "kvi_debug.h"
#include "kvi_regusersdb.h"
//#include "kvi_ircmask.h"
#include "kvi_config.h"



/*
	@doc: registered_users
	@title:
		Registered users
	@type:
		generic
	@short:
		Registration of users in KVIrc
	@keyterms:
		registered users, registration mask, registered user properties,
		user properties, notify property, avatar property
	@body:
		[big]Introduction[/big][br]
		The "registered user database" is basically a set of users with associated
		[doc:irc_masks]irc-masks[/doc] and properties.[br]
		It is used to recognize users on IRC and associate properties to them.[br]
		This works more or less like the IRC ban list, K-Line list, or invite list.[br]
		[big]User entry[/big][br]
		A registered user database entry is identified by an [b]unique[/b] name.[br]
		It may be the nickname of the user that you want to match, or the real name (if you know it)
		or any other string (even with spaces). The name is an "internal identifier" for the user entry:
		each name maps to a single entry and each entry has a single name.[br]
		Each entry has a set of registration [doc:irc_masks]irc-masks[/doc]: these masks
		are used to recognize the user on irc.[br]
		[br]
		[big]Registration masks[/big][br]
		The masks have the common IRC mask format: [b]<nick>!<user>@<host>[/b][br]
		The masks may contain '*' and '?' wildcards that match any portion of text.[br]
		[b]*!*@*[/b][br]
		[b]Pragma!*@*[/b][br]
		[b]*!~daemon@*[/b][br]
		[b]Pragma!*daemon@*.it[/b][br]
		[b]Pragma!?daemon@some*.it[/b][br]
		[b]Pragma!~daemon@some.host.it[/b][br]
		Are examples of valid registration masks.[br]
		The masks with wildcards can actually match more than a single user.[br]
		For example the mask *!root@*.host.com will match all the users
		having root as username and coming from the host.com domain.[br]
		For this reason putting wildcards in nicknames could become a problem
		if not used carefully (but may also be used to achieve interesting tricks).[br]
		If you don't use wildcards in nicknames you are sure that
		in a single irc connection , a mask will always refer to a single user.[br]
		You will commonly use the following format:[br]
		<nick>!*<username>@*.<host>.<top>[br]
		or[br]
		<nick>!*<username>@<number>.<number>.<number>.*[br]
		In this way you can be 95% sure that the mask will really match the correct user.[br]
		[br]
		[big]Example of registration and lookups[/big]
		Assume that you want to registere a friend of yours: Derek Riggs.[br]
		Derek often uses "Eddie" as his nickname
		"stranger" as username and has a dial-up connection that makes his IP address appear as
		<variable-number>.somewhere.in.time.org.[br]
		You will add an entry with name "Derek Riggs" and a registration mask like the following:
		Eddie!stranger@*.somewhere.in.time.org.[br]
		If the IRC servers keep adding strange characters ([doc:irc_masks]prefixes[/doc]) at the beginning of his username you may use
		Eddie!*stranger@*.somewhere.in.time.org.[br]
		If Eddie also often connects from the wasted.years.org domain and gets 'eddie' as username there, you might add a second registration mask as follows:
		Eddie!*eddie@*.wasted.years.org.[br]
		An alternative could be use only one mask with *.org as domain and allow any username (Eddie!*@*.org) but this
		could become dangerous since it could match the users that you don't want to.[br]
		On the other hand, if you dislike the users with the nickname Eddie that come from .org
		and you're implementing an auto-kick system, the correct mask to register is "Eddie!*@*.org".[br]
		[br]
		KVirc ties to be smart , and always find the most correct match for an user:
		If you have two masks registered: Pragma!*xor@*.myisp.it and *!*@*.myisp.it,
		kvirc will match Pragma!~xor@233-dyn.myisp.it with the first one even if the second
		one matches too; the firs one is a best match.[br]
		[br]
		[big]Properties[/big][br]
		A registered user has an (eventually empty) set of properties
		defined by name/value pairs. (In versions prior to 3.0.0 flags were used instead,
		but revealed to be insufficient).[br]
		KVirc recognizes some of these proprietes and associates semantic actions to it; other properties
		are left for scripting extension. Property names are case insensitive.[br]
		One of the recognized properties is the "[doc:notify_list]notify[/doc]" property.
		When an user is found to have this property set to a special value
		KVIrc will attempt to track the user presence on IRC.
		Another one is the [doc:avatar]avatar[/doc] property. Its value should be the
		name of the "default" [doc:avatar]avatar image file[/doc] for the specified user.[br]
		The "ignore" property should be set to "1" (or "true") for users that have to be ignored (:D).[br]
		[br]
		[big]The interface to the database[/big][br]
		The [module:reguser]reguser module[/module] is the interface to the "registered users database".[br]
		It provides a set of commands for adding and removing masks and manipulating properties.[br]
*/

//============================================================================================================
//
// KviRegisteredMask
//

KviRegisteredMask::KviRegisteredMask(KviRegisteredUser * u,KviIrcMask * m)
{
	m_pUser = u;
	m_pMask = m;
	m_iMaskNonWildChars = m_pMask->nonWildChars();
}

//============================================================================================================
//
// KviRegisteredUser
//


KviRegisteredUser::KviRegisteredUser(const char * name)
{
	m_szName        = name;
	m_pPropertyDict = 0;
	m_pMaskList     = new KviPtrList<KviIrcMask>;
	m_pMaskList->setAutoDelete(true);
}

KviRegisteredUser::~KviRegisteredUser()
{
	if(m_pPropertyDict)delete m_pPropertyDict;
	delete m_pMaskList;
}

KviIrcMask * KviRegisteredUser::findMask(const KviIrcMask &mask)
{
	for(KviIrcMask * m = m_pMaskList->first();m;m = m_pMaskList->next())
	{
		if(*m == mask)return m;
	}
	return 0;
}

bool KviRegisteredUser::addMask(KviIrcMask * mask)
{
//	debug("USER %s: addMask(%u (%s!%s@%s))",name(),mask,mask->nick(),mask->user(),mask->host());
	if(findMask(*mask))
	{
//		debug(" ops , already there!");
		delete mask;
		return false;
	}
//	debug(" added :)");
	m_pMaskList->append(mask);
	return true;
}

bool KviRegisteredUser::removeMask(KviIrcMask * mask)
{
//	debug("USER %s: removeMask(%u (%s!%s@%s))",name(),mask,mask->nick(),mask->user(),mask->host());
	return m_pMaskList->removeRef(mask);
}

bool KviRegisteredUser::matches(const KviIrcMask &mask)
{
	for(KviIrcMask * m = m_pMaskList->first();m;m = m_pMaskList->next())
	{
		if(m->matches(mask))return true;
	}
	return false;
}

bool KviRegisteredUser::matchesFixed(const KviIrcMask &mask)
{
	for(KviIrcMask * m = m_pMaskList->first();m;m = m_pMaskList->next())
	{
		if(m->matchesFixed(mask))return true;
	}
	return false;
}

void KviRegisteredUser::setProperty(const char * name,const char * value)
{
	if(value && *value)
	{
		if(!m_pPropertyDict)
		{
			m_pPropertyDict = new QAsciiDict<KviStr>(7,false,true);
			m_pPropertyDict->setAutoDelete(true);
		}
		KviStr * val = new KviStr(value);
		val->stripWhiteSpace();
		if(val->hasData())
		{
			m_pPropertyDict->replace(name,val);
		} else {
			delete val;
		}
	} else {
		if(m_pPropertyDict)m_pPropertyDict->remove(name);
	}
}

bool KviRegisteredUser::getProperty(const char * name,KviStr &value)
{
	if(!m_pPropertyDict)return false;
	KviStr * pValue = m_pPropertyDict->find(name);
	if(pValue)value = *pValue;
	else return false;
	return true;
}

const char * KviRegisteredUser::getProperty(const char * name)
{
	if(!m_pPropertyDict)return 0;
	KviStr * pValue = m_pPropertyDict->find(name);
	if(pValue)return pValue->ptr();
	return 0;
}

bool KviRegisteredUser::getBoolProperty(const char * name)
{
	if(!m_pPropertyDict)return false;
	KviStr * pValue = m_pPropertyDict->find(name);
	if(pValue)
	{
		// be flexible , allow more "true" values (pragma)
		if(kvi_strEqualCS(pValue->ptr(),"1"))return true;
		if(kvi_strEqualCI(pValue->ptr(),"true"))return true;
		if(kvi_strEqualCI(pValue->ptr(),"yes"))return true;
	}
	return false;
}

//============================================================================================================
//
// KviRegisteredUserDb
//

KviRegisteredUserDataBase::KviRegisteredUserDataBase()
{
	m_pUserDict = new QAsciiDict<KviRegisteredUser>(31,false,false); // do not copy keys
	m_pUserDict->setAutoDelete(true);

	m_pWildMaskList = new KviRegisteredMaskList;
	m_pWildMaskList->setAutoDelete(true);

	m_pMaskDict = new QAsciiDict<KviRegisteredMaskList>(49,false,true); // copy keys here!
	m_pMaskDict->setAutoDelete(true);
}

KviRegisteredUserDataBase::~KviRegisteredUserDataBase()
{
	delete m_pUserDict;
	delete m_pWildMaskList;
	delete m_pMaskDict;
}

KviRegisteredUser * KviRegisteredUserDataBase::addUser(const char * name)
{
	if(m_pUserDict->find(name))return 0;
	KviRegisteredUser * u = new KviRegisteredUser(name);
	m_pUserDict->replace(u->name(),u); //u->name() because we're NOT copying keys!
	return u;
}

KviRegisteredUser * KviRegisteredUserDataBase::getUser(const char * name)
{
	KviRegisteredUser * u = m_pUserDict->find(name);
	if(!u)
	{
		u = new KviRegisteredUser(name);
		m_pUserDict->replace(u->name(),u); //u->name() because we're NOT copying keys!
	}
	return u;
}

static void append_mask_to_list(KviRegisteredMaskList *l,KviRegisteredUser *u,KviIrcMask *mask)
{
	KviRegisteredMask * newMask = new KviRegisteredMask(u,mask);
	int idx = 0;
	for(KviRegisteredMask * m = l->first();m;m = l->next())
	{
		if(m->nonWildChars() < newMask->nonWildChars())
		{
			l->insert(idx,newMask);
			return;
		}
		idx++;
	}
	l->append(newMask);
}

KviRegisteredUser * KviRegisteredUserDataBase::addMask(KviRegisteredUser * u,KviIrcMask * mask)
{
	__range_valid(u == m_pUserDict->find(u->name()));

//	debug("Adding mask %s!%s@%s (%d) for user %s (%d)",mask->nick(),mask->user(),mask->host(),mask,u->name(),u);

	KviRegisteredMaskList * l;
	if(mask->hasWildNick())
	{
//		debug(" it has a wild nick");
		for(KviRegisteredMask * m = m_pWildMaskList->first();m;m = m_pWildMaskList->next())
		{
			if(*(m->mask()) == *mask)
			{
//				debug(" ...and it is already there :/");
				delete mask;
				return m->user();
			}
		}
//		debug(" ok... not there yet, will add it");
		// not found ...ok... add it
		// masks with more info go first in the list
		l = m_pWildMaskList;
	} else {
		l = m_pMaskDict->find(mask->nick());
//		debug(" the nick is not wild");
		if(l)
		{
//			debug(" I already have a list of masks with this nick");
			// FIXME: #warning "Here we could compare the host and username only: nick matches for sure"
			for(KviRegisteredMask * m = l->first();m;m = l->next())
			{
				if(*(m->mask()) == *mask)
				{
//					debug(" ... and it contains the same mask :/");
					delete mask;
					return m->user();
				}
			}
//			debug(" ok... not there yet, will add it");
			// not found ...ok... add it
		} else {
//			debug(" no list with this nick yet");
			// not found ...ok... add it
			// this is the first mask in the list
			l = new KviRegisteredMaskList;
			l->setAutoDelete(true);
			if(!u->addMask(mask))
			{
				debug(" Ops...got an incoherent regusers action...recovered ?");
				delete l;
			} else {
//				debug(" added to the user mask list too!");
				append_mask_to_list(l,u,mask);
				m_pMaskDict->insert(mask->nick(),l);
			}
			return 0;
		}
	}
//	debug(" finally adding");
	// Ok...add it
	if(!u->addMask(mask))
	{
		debug("ops...got an incoherent regusers action...recovered ?");
		return 0; // ops...already there ?
	}
//		debug(" added to the user mask list too!");
	append_mask_to_list(l,u,mask);
	return 0;
}

void KviRegisteredUserDataBase::copyFrom(KviRegisteredUserDataBase * db)
{
	m_pUserDict->clear();
	m_pWildMaskList->clear();
	m_pMaskDict->clear();

	QAsciiDictIterator<KviRegisteredUser> it(*(db->m_pUserDict));

	while(KviRegisteredUser * theCur = it.current())
	{
		KviRegisteredUser * u = getUser(theCur->name());
		// copy masks
		KviPtrList<KviIrcMask> * l = theCur->maskList();
		for(KviIrcMask * m=l->first();m;m = l->next())
		{
			KviIrcMask * m2 = new KviIrcMask(*m);
			addMask(u,m2);
		}
		// copy properties
		QAsciiDict<KviStr> * pd = theCur->propertyDict();
		if(pd)
		{
			QAsciiDictIterator<KviStr> pdi(*pd);
			while(pdi.current())
			{
				u->setProperty(pdi.currentKey(),pdi.current()->ptr());
				++pdi;
			}
		}
		++it;
	}
}


bool KviRegisteredUserDataBase::removeUser(const char * name)
{
	KviRegisteredUser * u = m_pUserDict->find(name);
//	debug("Removing user %s (%u)",name,u);
	if(!u)return false;
//	debug("The user %s has %d masks in the lists",name,u->maskList()->count());
//	int i =0;
//	for(KviIrcMask * m = u->maskList()->first();m;m = u->maskList()->next())
//	{
//		debug(" %d: %u : %s!%s@%s",i,m,m->nick(),m->user(),m->host());
//		i++;
//	}
	while(KviIrcMask * mask = u->maskList()->first())
	{
//		debug("Removing mask (%u) %s!%s@%s",mask,mask->nick(),mask->user(),mask->host());
		if(!removeMaskByPointer(mask))
			debug("Ops... removeMaskByPointer(%d (%s)) failed ?",mask,name);
	}
	m_pUserDict->remove(name);
	return true;
}

bool KviRegisteredUserDataBase::removeMask(const KviIrcMask &mask)
{
	// find the mask pointer
	KviRegisteredMask * m = findExactMask(mask);
	// and remove it
	if(m)return removeMaskByPointer(m->mask());
	return 0;
}

bool KviRegisteredUserDataBase::removeMaskByPointer(KviIrcMask * mask)
{
//	debug("Removing mask %u by pointer",mask);
	if(mask->hasWildNick())
	{
		// remove from the wild list
		for(KviRegisteredMask * m = m_pWildMaskList->first();m;m = m_pWildMaskList->next())
		{
			if(m->mask() == mask)
			{
				// ok..got it, remove from the list and from the user struct (user struct deletes it!)
				m->user()->removeMask(mask);    // this one deletes m->mask()
				m_pWildMaskList->removeRef(m);  // this one deletes m
				return true;
			}
		}
//#ifdef _KVI_DEBUG_CHECK_RANGE_
//		debug("Ops.. wild mask %d (%s) not found in removeMaskByPointer",mask,mask->nick());
//#endif
		// not found ...opz :)
	} else {
		KviRegisteredMaskList * l = m_pMaskDict->find(mask->nick());
		if(l)
		{
			// FIXME: #warning "Here we could compare the host and username only: nick matches for sure"
			for(KviRegisteredMask * m = l->first();m;m = l->next())
			{
//				debug("Comparing mask %d (%s) with %d (%s)",m->mask(),m->mask()->nick(),mask,mask->nick());
				if(m->mask() == mask)
				{
					KviStr nick = mask->nick();
					m->user()->removeMask(mask); // this one deletes m->mask() (or mask)
					l->removeRef(m);             // this one deletes m
					if(l->count() == 0)m_pMaskDict->remove(nick.ptr());
					return true;
				}
			}
			// not found ...opz
//#ifdef _KVI_DEBUG_CHECK_RANGE_
//		debug("Ops.. mask %d (%s) not found in the list not found in removeMaskByPointer",mask,mask->nick());
//#endif
		}
//#ifdef _KVI_DEBUG_CHECK_RANGE_
//		else debug("Ops.. mask list %d (%s) not found in removeMaskByPointer",mask,mask->nick());
//#endif
	}
	// not found...
	return false;
}



/*
KviRegisteredUser * KviRegisteredUserDataBase::findMatchingUser(const KviIrcMask &mask)
{
	// first lookup the nickname in the maskDict
	KviRegisteredMaskList * l = m_pMaskDict->find(mask.nick());
	if(l)
	{
		for(KviRegisteredMask *m = l->first();m;m = l->next())
		{
			if(m->mask()->matchesFixed(0,mask.user(),mask.host()))return m->user();
		}
	}
	// not found....lookup the wild ones
	for(KviRegisteredMask * m = m_pWildMaskList->first();m;m = m_pWildMaskList->next())
	{
		if(m->mask()->matchesFixed(mask))return m->user();
	}
	return 0; // no match at all
}
*/
KviRegisteredUser * KviRegisteredUserDataBase::findMatchingUser(const char * nick,const char *user,const char * host)
{
	KviRegisteredMask * m = findMatchingMask(nick,user,host);
	if(m)return m->user();
	return 0; // no match at all
}

KviRegisteredMask * KviRegisteredUserDataBase::findMatchingMask(const char * nick,const char *user,const char * host)
{
	// first lookup the nickname in the maskDict
	KviRegisteredMaskList * l = m_pMaskDict->find(nick);
	if(l)
	{
		for(KviRegisteredMask *m = l->first();m;m = l->next())
		{
			if(m->mask()->matchesFixed(0,user,host))return m;
		}
	}
	// not found....lookup the wild ones
	for(KviRegisteredMask * m = m_pWildMaskList->first();m;m = m_pWildMaskList->next())
	{
		if(m->mask()->matchesFixed(nick,user,host))return m;
	}
	return 0; // no match at all
}

KviRegisteredUser * KviRegisteredUserDataBase::findUserWithMask(const KviIrcMask &mask)
{
	KviRegisteredMask * m = findExactMask(mask);
	if(m)return m->user();
	return 0;
}

KviRegisteredMask * KviRegisteredUserDataBase::findExactMask(const KviIrcMask &mask)
{
	// first lookup the nickname in the maskDict
	KviRegisteredMaskList * l = m_pMaskDict->find(mask.nick());
	if(l)
	{
		for(KviRegisteredMask *m = l->first();m;m = l->next())
		{
			if(*(m->mask()) == mask)return m;
		}
	}
	// not found....lookup the wild ones
	for(KviRegisteredMask * m = m_pWildMaskList->first();m;m = m_pWildMaskList->next())
	{
		if(*(m->mask()) == mask)return m;
	}
	return 0; // no match at all
}
/*
bool KviRegisteredUserDataBase::isIgnoredUser(const char * nick,const char * user,const char * host)
{ 
	KviRegisteredUser * u = findMatchingUser(nick,user,host);
	if(u)return u->getBoolProperty("IGNORE");
	else return false;
}
*/
void KviRegisteredUserDataBase::load(const char * filename)
{
	KviConfig cfg(filename);

	KviConfigIterator it(*cfg.dict());
	while(it.current())
	{
		KviStr tmp = it.currentKey();

		KviRegisteredUser * u = addUser(tmp.ptr());
		if(u)
		{
			KviStrDictIterator sdi(*(it.current()));
			while(sdi.current())
			{
				KviStr tmp = sdi.currentKey();
				if(kvi_strEqualCSN("prop_",tmp.ptr(),5))
				{
					tmp.cutLeft(5);
					u->setProperty(tmp.ptr(),sdi.current()->ptr());
				} else if(kvi_strEqualCSN("mask_",tmp.ptr(),5))
				{
					KviIrcMask * mask = new KviIrcMask(sdi.current()->ptr());
					addMask(u,mask);
				}
				++sdi;
			}
		}
		++it;
	}

}


void KviRegisteredUserDataBase::save(const char * filename)
{
	KviConfig cfg(filename);
	cfg.clear();
	cfg.preserveEmptyGroups(true);

	QAsciiDictIterator<KviRegisteredUser> it(*m_pUserDict);

	while(it.current())
	{
		cfg.setGroup(it.current()->name());
		// Write properties
		if(it.current()->propertyDict())
		{
			QAsciiDictIterator<KviStr> pit(*(it.current()->propertyDict()));
			while(pit.current())
			{
				KviStr tmp = "prop_";
				tmp.append(pit.currentKey());
				cfg.writeEntry(tmp.ptr(),pit.current()->ptr());
				++pit;
			}
		}
		// Write masks
		int idx = 0;
		for(KviIrcMask * m = it.current()->maskList()->first();m;m = it.current()->maskList()->next())
		{
			KviStr tmp(KviStr::Format,"mask_%d",idx);
			KviStr mask;
			m->mask(mask);
			cfg.writeEntry(tmp.ptr(),mask.ptr());
			++idx;
		}
		++it;
	}
}
