//
//   File : kvi_filetrader.cpp
//   Creation date : Wed Aug 27 2000 10:33:11 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#define __LIBKVILIB__

#include "kvi_filetrader.h"

#include "kvi_config.h"
#include "kvi_fileutils.h"

#include <qfileinfo.h>

// TODO: Match servers that the file requests come from
// TODO: Max number of downloads ?


/*
	@doc: file_offers
	@title:
		File offers
	@type:
		generic
	@short:
		Automatically sharing your files with other IRC users
	@keyterms:
		file sharing
	@body:
		[big]What is this ?[/big]
		The "file offers" are a simple way to share your files with other IRC users.[br]
		Basically , you setup an offer by selecting a local file, choosing a "visible name" for it.
		Remote users will be able to request you the file and download it automatically by
		issuing a simple DCC GET request.[br]
		[big]Details[/big]
		Each offer refers to an existing file on one of your locally mounted file systems.
		The offer is given a visible name that the remote users will effectively request.
		To share the file /usr/arch/mp3/SonataArctica_SingInSilence_Live.mp3 you will add a file offer
		with /usr/arch/mp3/SonataArctica_SingInSilence_Live.mp3 as real file path , something like
		"SonataArctica_SingInSilence.mp3". A remote user will then request you a DCC GET SonataArctica_SingInSilence.mp3
		and KVIrc will automatically send the file.[br]
		Each file offer has an "user mask" that the requesting remote users must match to
		obtain the file: *!*@* matches any user, Pragma!*@* matches any user with nickname pragma,
		*!*@*.omnikron.net matches any user coming from the omnikron.net domain.[br]
		Each offer can have an expire time: the offer will be automatically removed after
		a defined number of seconds. An expire time of '0' seconds means that the offer should never expire.[br]
		If you have two file offers with the same name and different file, the remote user can
		use an additional "size" parameter in the DCC GET request.[br]
		[big]Security issues[/big]
		This is a nice but unsecure method of sharing files.[br]
		The user mask is a good protection but you have to use it properly!.[br]
		Setting the user mask to Nick!*@* can be easily exploited (just by making an user disconnect
		in one of the well known ways and then by using his nickname).[br]
		On the other side, the remote end must know exactly the visible name of the offer to request
		and noone but you will tell him that name.[br]
		In sum:[br]
		Don't share any really important files: this *might* be like putting it on your webpage :D[br]
		Please don't send complains if someone stoles you /etc/passwd : it is because you have permitted that.[br]
*/

KviFileOffer::KviFileOffer(const char * szAbsPath,const char * szUserMask,time_t expireTime,unsigned int uFileSize)
{
	m_szAbsFilePath = szAbsPath;
	m_szUserMask = szUserMask;
	m_expireTime = expireTime;
	m_uFileSize = uFileSize;
	m_uWildCount = m_szUserMask.occurences('*');
	m_uNonWildCount = m_szUserMask.len() - m_uWildCount;
}

KviFileOffer::~KviFileOffer()
{
}


KviFileTrader::KviFileTrader()
: QObject()
{
	m_pOfferListDict = new QAsciiDict<KviFileOfferList>;
	m_pOfferListDict->setAutoDelete(true);
	m_pCleanupTimer = new QTimer();
	connect(m_pCleanupTimer,SIGNAL(timeout()),this,SLOT(cleanup()));
}

KviFileTrader::~KviFileTrader()
{
	if(m_pCleanupTimer->isActive())m_pCleanupTimer->stop();
	delete m_pCleanupTimer;
	delete m_pOfferListDict;
}

void KviFileTrader::cleanup()
{
	QAsciiDictIterator<KviFileOfferList> it(*m_pOfferListDict);
	time_t curTime = time(0);

	bool bOtherStuffToCleanup = false;
	bool bChanged = false;

	while(KviFileOfferList * l = it.current())
	{
		KviPtrList<KviFileOffer> tmp;
		tmp.setAutoDelete(false);
		for(KviFileOffer * o = l->first();o;o = l->next())
		{
			if(o->expireTime() > 0)
			{
				if(((int)o->expireTime()) <= ((int)curTime))
				{
					tmp.append(o);
					bChanged = true;
				} else {
					bOtherStuffToCleanup = true;
				}
			}
		}
		for(KviFileOffer * fo = tmp.first();fo;fo = tmp.next())l->removeRef(fo);
		if(l->count() == 0)
		{
			m_pOfferListDict->remove(it.currentKey());
		}
		++it;
	}

	if(!bOtherStuffToCleanup)m_pCleanupTimer->stop();
	if(bChanged)emit offerListChanged();
}

void KviFileTrader::clear()
{
	m_pOfferListDict->clear();
	emit offerListChanged();
}

void KviFileTrader::doInsert(KviFileOfferList * l, KviFileOffer * o)
{
	int index = 0;
	for(KviFileOffer * fo =l->first();fo;fo = l->next())
	{
		if(o->wildcardCount() > 0)
		{
			// the new mask has wildcards... if the current one has none, skip it
			if(fo->wildcardCount() > 0)
			{
				// the one in the list has wildcards too...
				// the ones with more non-wild chars go first...
				if(fo->nonWildcardCount() < o->nonWildcardCount())
				{
					// ok...the new one has more non-wildcards , insert
					l->insert(index,o);
					return;
				} else {
					if(o->nonWildcardCount() == fo->nonWildcardCount())
					{
						// the same number of non-wildcards
						// let the number of wildcards decide (it will be eventually equal)
						if(o->wildcardCount() < fo->wildcardCount())
						{
							// the new one has less wildcards... goes first
							l->insert(index,o);
							return;
						} // else the same number of wildcards and non-wildcards...skip
					} // else the existing one has more non-wildcards...skip
				}
			} // else the current has no wildcards...skip
		} else {
			// the new mask has no wildcards....
			if(fo->wildcardCount() > 0)
			{
				// current one has wildcards...insert
				l->insert(index,o);
				return;
			}
			// the current one has no wildcards...
			// the longer masks go first....
			if(fo->maskLength() < o->maskLength())
			{
				// the current one is shorter than the new one...insert
				l->insert(index,o);
				return;
			} // else current one is longer...skip
		}
		index++;
	}
	l->append(o);
}

KviFileOffer * KviFileTrader::addOffer(const char * szName,const char * szAbsPath,const char * szMask,int timeoutInSecs)
{
	QFileInfo inf(szAbsPath);
	if(inf.exists() && inf.isFile() && inf.isReadable() && (inf.size() > 0))
	{
		// First find the list
		KviFileOfferList * l = m_pOfferListDict->find(szName);
		if(!l)
		{
			l = new KviFileOfferList;
			l->setAutoDelete(true);
			m_pOfferListDict->replace(szName,l);
		}
	
		// Now insert
		KviFileOffer * o = new KviFileOffer(szAbsPath,szMask,timeoutInSecs > 0 ? (((int)(time(0))) + timeoutInSecs) : 0,inf.size());

		doInsert(l,o);

		if(((int)o->expireTime()) > 0)
		{
			if(!m_pCleanupTimer->isActive())m_pCleanupTimer->start(60000);
		}

		emit offerListChanged();

		return o;
	} else {
		debug("File %s unreadable: can't add offer",szAbsPath);
		return 0;
	}
}

KviFileOffer * KviFileTrader::lookupOffer(const char * szName,KviIrcMask * mask,unsigned int uFileSize)
{
	KviFileOfferList * l = m_pOfferListDict->find(szName);
	if(!l)return 0;
	for(KviFileOffer * o = l->first();o;o = l->next())
	{
		bool bMatch;
		if(mask)bMatch = mask->matchedBy(o->userMask());
		else bMatch = kvi_strEqualCS(o->userMask(),"*!*@*");
		if(bMatch)
		{
			if(uFileSize > 0)
			{
				if(uFileSize == o->fileSize())return o;
			} else return o;
		}
	}

	return 0;
}
bool KviFileTrader::removeOffer(const char * szName,const char * szMask,unsigned int uFileSize)
{
	KviFileOfferList * l = m_pOfferListDict->find(szName);
	if(!l)return false;
	for(KviFileOffer * o = l->first();o;o = l->next())
	{
		if(kvi_strEqualCI(szMask,o->userMask()))
		{
			bool bMatch = uFileSize > 0 ? uFileSize == o->fileSize() : true;
			if(bMatch)
			{
				l->removeRef(o);
				if(l->count() == 0)m_pOfferListDict->remove(szName);
				emit offerListChanged();
				return true;
			}
		}
	}
	return false;
}

bool KviFileTrader::removeOffer(const char * szName,KviFileOffer * off)
{
	KviFileOfferList * l = m_pOfferListDict->find(szName);
	if(!l)return false;
	for(KviFileOffer * o = l->first();o;o = l->next())
	{
		if(off == o)
		{
			l->removeRef(o);
			if(l->count() == 0)m_pOfferListDict->remove(szName);
			emit offerListChanged();
			return true;
		}
	}
	return false;
}


void KviFileTrader::load(const char * filename)
{
	KviConfig cfg(filename);
	cfg.clear();
	cfg.setGroup("PermanentFileOffers");
	int num = cfg.readIntEntry("NEntries",0);
	for(int idx=0;idx<num;idx++)
	{
		KviStr tmp(KviStr::Format,"%dFName",idx);
		KviStr szName = cfg.readEntry(tmp.ptr(),"");
		tmp.sprintf("%dFilePath",idx);
		KviStr szPath = cfg.readEntry(tmp.ptr(),"");
		tmp.sprintf("%dUserMask",idx);
		KviStr szMask = cfg.readEntry(tmp.ptr(),"");
		if(szMask.hasData() && szPath.hasData() && szName.hasData())addOffer(szName.ptr(),szPath.ptr(),szMask.ptr(),0);
	}
}

void KviFileTrader::save(const char * filename)
{
	KviConfig cfg(filename);
	cfg.clear();
	cfg.setGroup("PermanentFileOffers");

	QAsciiDictIterator<KviFileOfferList> it(*m_pOfferListDict);
	int idx = 0;
	while(KviFileOfferList * l = it.current())
	{
		for(KviFileOffer * o = l->first();o;o = l->next())
		{
			if(((int)(o->expireTime())) == 0)
			{
				KviStr tmp(KviStr::Format,"%dFName",idx);
				cfg.writeEntry(tmp.ptr(),it.currentKey());
				tmp.sprintf("%dFilePath",idx);
				cfg.writeEntry(tmp.ptr(),o->absFilePath());
				tmp.sprintf("%dUserMask",idx);
				cfg.writeEntry(tmp.ptr(),o->userMask());
				++idx;
			}
		}
		++it;
	}
	cfg.writeEntry("NEntries",idx);
}

#include "kvi_filetrader.moc"
