//
//   File : kvi_config.cpp (/usr/build/NEW_kvirc/kvirc/kvilib/kvi_config.cpp)
//   Last major modification : Thu Jan 14 1999 18:03:59 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#define __LIBKVILIB__

#define _KVI_DEBUG_CHECK_RANGE_
#include "kvi_debug.h"

#define _KVI_CONFIG_CPP_
#include "kvi_config.h"
#include "kvi_fileutils.h"
#include "kvi_pixmap.h"
#include "kvi_msgtype.h"
#include "kvi_stringconversion.h"

#ifdef COMPILE_ON_WINDOWS
	#include "kvi_malloc.h"
#endif


KviConfig::KviConfig(const char *filename,bool bReadOnly)
{
	__range_valid(filename);
	m_szFileName           = filename;
	m_pDict                = new QAsciiDict<KviStrDict>(17,false);
	m_pDict->setAutoDelete(true);
	m_bDirty               = false;
	m_szGroup              = KVI_CONFIG_DEFAULT_GROUP;
	m_bPreserveEmptyGroups = false;
	m_bReadOnly            = bReadOnly;
	load();
}

KviConfig::~KviConfig()
{
	if(m_bDirty)save();
	delete m_pDict;
}

void KviConfig::clear()
{
	delete m_pDict;
	m_pDict      = new QAsciiDict<KviStrDict>(17,false);
	m_pDict->setAutoDelete(true);
	m_bDirty     = false;
	m_szGroup    = KVI_CONFIG_DEFAULT_GROUP;
}

void KviConfig::clearGroup(const char *szGroup)
{

	m_pDict->remove(szGroup);
	__range_invalid(m_pDict->find(szGroup));
	if(!m_pDict->find(m_szGroup.ptr()))m_szGroup = KVI_CONFIG_DEFAULT_GROUP; //removed the current one
}

void KviConfig::clearKey(const char *szKey)
{
	KviStrDict * p_group = getCurrentGroup();
	p_group->remove(szKey);
	if(p_group->count() == 0)clearGroup(m_szGroup.ptr());
}

/*
void KviConfig::getContentsString(KviStr &buffer)
{
	buffer = __tr("Contents of config file ");
	buffer.append(m_szFileName.ptr());
	buffer.append('\n');
	int sections = 0;
	int keys     = 0;
	QDictIterator<KviStrDict> it(*m_pDict);
	while(it.current()){
		buffer.append(" Section [");
		buffer.append(it.currentKey());
		buffer.append("]\n");
		int sectionKeys = 0;
		QDictIterator<KviStr> it2(*it.current());
		while(it2.current()){
			buffer.append("  Key [");
			buffer.append(it2.currentKey());
			buffer.append("] : ");
			buffer.append(it2.current()->ptr());
			buffer.append('\n');
			++it2;
			++sectionKeys;
			++keys;
		}
		KviStr tmp(KviStr::Format,__tr("  Total: %d keys"),sectionKeys);
		buffer.append(tmp);
		buffer.append('\n');
		++it;
		++sections;
	}
	KviStr tmp(KviStr::Format,__tr("Total: %d keys in %d sections"),keys,sections);
	buffer.append(tmp);
}
*/

bool KviConfig::load()
{
	QFile f(m_szFileName.ptr());
	if(!f.open(IO_ReadOnly))return false;

	__range_valid(m_pDict->isEmpty());

	KviStrDict * p_group = 0;

	KviStr dataLine;
	bool bContinue;

	do {
		bContinue = kvi_readFirstNonEmptyStrippedLine(&f,dataLine);

		if(dataLine.hasData())
		{
			if(dataLine.at(0)=='[')
			{
				//set the group
				dataLine.cutLeft(1);
				dataLine.cutRight(1);

				p_group = (KviStrDict *)m_pDict->find(dataLine.ptr());

				if(!p_group)
				{
					p_group = new KviStrDict(17,false);
					p_group->setAutoDelete(true);
					m_pDict->insert(dataLine.ptr(),p_group);
				}

			} else {

				//data entry...split in two...

				KviStr name=dataLine.getToken('=');
				name.stripWhiteSpace();
				name.decodeWhiteSpace();
				dataLine.stripWhiteSpace();
				dataLine.decodeWhiteSpace();
				if(name.hasData() && dataLine.hasData())
				{
					//insert (replace items if needed)
					KviStr *p_data=new KviStr(dataLine);
					if(!p_group)
					{
						// ops...we're missing a group
						// use the default one
						p_group = new KviStrDict(17,false);
						p_group->setAutoDelete(true);
						m_pDict->insert(KVI_CONFIG_DEFAULT_GROUP,p_group);
					}
					p_group->replace(name.ptr(),p_data);
				}
			}
		}
	} while (bContinue);

	f.close();
	return true;
}

bool KviConfig::save()
{
//	debug("KviConfig : saving %s",m_szFileName.ptr());
	if(m_bReadOnly)return false;
//	debug("Is not read only!");

	QFile f(m_szFileName.ptr());
	if(!f.open(IO_WriteOnly | IO_Truncate))return false;
//	debug("File opened! %s",m_szFileName.ptr());
	QAsciiDictIterator<KviStrDict> it(*m_pDict);
	while (it.current())
	{
		if((it.current()->count() != 0) || (m_bPreserveEmptyGroups))
		{
			KviStr key(it.currentKey());
			KviStr str(KviStr::Format,"[%s]",key.ptr());
			if(!kvi_writeLine(&f,str.ptr()))return false;
			KviStrDict * dict = (KviStrDict *)it.current();
			QAsciiDictIterator<KviStr> it2(*dict);
			while(it2.current())
			{
				key=it2.currentKey();
				KviStr * p_str = it2.current();
				str.sprintf("%s=%s",key.ptr(),p_str->ptr());
				str.encodeWhiteSpace();
				if(!kvi_writeLine(&f,str.ptr()))return false;
				++it2;
			}
		}
		++it;
	}
	f.close();
	m_bDirty = false;
	return true;
}

void KviConfig::setGroup(const char *szGroup)
{
	m_szGroup = szGroup;
	if(m_bPreserveEmptyGroups)
	{
		if(!hasGroup(szGroup))
		{
			getCurrentGroup(); // we need it to be created.
			m_bDirty = true;
		}
	}
}

bool KviConfig::hasKey(const char *szKey)
{
	KviStrDict * p_group = getCurrentGroup();
	return (p_group->find(szKey) != 0);
}

bool KviConfig::hasGroup(const char * szGroup)
{
	return (m_pDict->find(szGroup) != 0);
}

KviStrDict * KviConfig::getCurrentGroup()
{
	if(m_szGroup.isEmpty())m_szGroup = KVI_CONFIG_DEFAULT_GROUP;
	KviStrDict * p_group = m_pDict->find(m_szGroup.ptr());
	if(!p_group){
		//create the group
		p_group = new KviStrDict(17,false);
		p_group->setAutoDelete(true);
		m_pDict->insert(m_szGroup.ptr(),p_group);
	}
	return p_group;
}

////////////////////////////////// KviStr

void KviConfig::writeEntry(const char *szKey,const char *szValue)
{
	__range_valid(szKey);
	__range_valid(szValue);
	m_bDirty = true;
	KviStrDict * p_group = getCurrentGroup();
	KviStr *p_data=new KviStr(szValue);
//	p_data->replaceAll(" ","\\[#]");
//	p_data->replaceAll("\n","\\[n]");
	p_group->replace(szKey,p_data);
}

// FIXME: #warning "We have probs here ?"

const char * KviConfig::readEntry(const char *szKey,const char *szDefault)
{
	__range_valid(szKey);
	KviStrDict * p_group = getCurrentGroup();
	KviStr * p_str = p_group->find(szKey);
	if(!p_str)
	{
		m_szStrBuffer = szDefault;
	} else {
		m_szStrBuffer = p_str->ptr();
//		m_szStrBuffer.replaceAll("\\[#]"," ");
//		m_szStrBuffer.replaceAll("\\[n]","\n");
	}
	return m_szStrBuffer.ptr();
}

////////////////////////////////// QStringList

QStringList KviConfig::readStringListEntry(const char * szKey,const QStringList &list)
{
	__range_valid(szKey);
	KviStrDict * p_group = getCurrentGroup();
	KviStr * p_str = p_group->find(szKey);
	if(!p_str)return list;
	m_szStrBuffer = p_str->ptr();

//	m_szStrBuffer.replaceAll("\\[n]","\n");
//	m_szStrBuffer.replaceAll("\\[#]"," ");

	return QStringList::split(",\\[ITEM],",m_szStrBuffer.ptr());
}

void KviConfig::writeEntry(const char *szKey,const QStringList &list)
{
	__range_valid(szKey);
	m_bDirty = true;
	KviStrDict * p_group = getCurrentGroup();
	KviStr *p_data=new KviStr(list.join(",\\[ITEM],"));
//	p_data->replaceAll("\n","\\[n]");
//	p_data->replaceAll(" ","\\[#]");
	p_group->replace(szKey,p_data);
}

////////////////////////////////// QValueList<int>

QValueList<int> KviConfig::readIntListEntry(const char * szKey,const QValueList<int> &list)
{
	__range_valid(szKey);
	KviStrDict * p_group = getCurrentGroup();
	KviStr * p_str = p_group->find(szKey);
	if(!p_str)return list;
	m_szStrBuffer = p_str->ptr();
	QValueList<int> ret;
	KviStr tmp;
	int iTmp;
	bool bOk;
	while(m_szStrBuffer.hasData())
	{
		m_szStrBuffer.getToken(tmp,',');
		iTmp = tmp.toInt(&bOk);
		if(bOk)ret.append(iTmp);
	}
	return ret;
}


void KviConfig::writeEntry(const char *szKey,const QValueList<int> &list)
{
	__range_valid(szKey);
	m_bDirty = true;
	KviStrDict * p_group = getCurrentGroup();
	KviStr *p_data=new KviStr();
	for(QValueList<int>::ConstIterator it = list.begin();it != list.end();++it)
	{
		if(p_data->hasData())p_data->append(',');
		p_data->append(KviStr::Format,"%d",*it);
	}
	p_group->replace(szKey,p_data);
}

////////////////////////////////// KviPixmap

// FIXME: #warning "Spaces in image names ?"

void KviConfig::writeEntry(const char *szKey,const KviPixmap &pixmap)
{
	__range_valid(szKey);
	m_bDirty = true;
	KviStrDict * p_group = getCurrentGroup();
	KviStr *p_data=new KviStr(pixmap.path());
	p_group->replace(szKey,p_data);
}

KviPixmap KviConfig::readPixmapEntry(const char * szKey,const KviPixmap &pixDef)
{
	__range_valid(szKey);
	KviStrDict * p_group = getCurrentGroup();
	KviStr * p_str = p_group->find(szKey);
	KviPixmap ret(p_str ? p_str->ptr() : "");
	return ret.pixmap() ? ret : pixDef;
}

////////////////////////////////// KviMsgType

void KviConfig::writeEntry(const char *szKey,const KviMsgType &msg)
{
	__range_valid(szKey);
	m_bDirty = true;
	KviStrDict * p_group = getCurrentGroup();
	KviStr *p_data=new KviStr();
	KviStringConversion::toString(msg,*p_data);
	p_group->replace(szKey,p_data);
}

KviMsgType KviConfig::readMsgTypeEntry(const char * szKey,const KviMsgType &msgDef)
{
	__range_valid(szKey);
	KviStrDict * p_group = getCurrentGroup();
	KviStr * p_str = p_group->find(szKey);
	if(!p_str)return msgDef;
	KviMsgType ret = msgDef;
	KviStringConversion::fromString(p_str->ptr(),ret);
	return ret;
}

////////////////////////////////// QColor

void KviConfig::writeEntry(const char *szKey,const QColor &clr)
{
	__range_valid(szKey);
	m_bDirty = true;
	KviStrDict * p_group = getCurrentGroup();
	KviStr *p_data = new KviStr(KviStr::Format,"%d,%d,%d",clr.red(),clr.green(),clr.blue());
	p_group->replace(szKey,p_data);
}

QColor KviConfig::readColorEntry(const char *szKey,const QColor &clr)
{
	KviStrDict * p_group = getCurrentGroup();
	QColor color(clr);
	KviStr * pointer_that_IS_initialized = p_group->find(szKey);


	if(pointer_that_IS_initialized){

		KviStr str(*pointer_that_IS_initialized);
		str.stripLeftWhiteSpace();

		KviStr red,green,blue;

		str.getToken(red,',');
		str.getToken(green,',');
		str.getToken(blue,',');

		if((red.isUnsignedNum())&&(green.isUnsignedNum())&&(blue.isUnsignedNum())){
			bool bOk;
			int r = red.toInt(&bOk) % 256;
			int g = green.toInt(&bOk) % 256;
			int b = blue.toInt(&bOk) % 256;
			if(r < 0)r = -r;
			if(g < 0)g = -g;
			if(b < 0)b = -b;
			color.setRgb(r,g,b);
		}
	}
	return color;
}

////////////////////////////////// QFont

void KviConfig::getFontProperties(KviStr & buffer,QFont *fnt)
{
	KviStringConversion::toString(*fnt,buffer);
}

void KviConfig::writeEntry(const char *szKey,QFont &fnt)
{
	__range_valid(szKey);
	m_bDirty = true;
	KviStrDict * p_group = getCurrentGroup();
	KviStr tmp;
	getFontProperties(tmp,&fnt);
	KviStr *p_data = new KviStr(tmp);
	p_group->replace(szKey,p_data);
}


void KviConfig::setFontProperties(KviStr & str,QFont *fnt)
{
	KviStringConversion::fromString(str.ptr(),*fnt);
}

QFont KviConfig::readFontEntry(const char *szKey,QFont &fnt)
{
	QFont font(fnt);
	KviStrDict * p_group = getCurrentGroup();
	KviStr * p_str       = p_group->find(szKey);
	if(p_str){
		//FontEntry=Arial,12,9,0,100,italic,underline,strikeout,
		KviStr str(*p_str);
		str.stripLeftWhiteSpace();
		setFontProperties(str,&font);
	}
	return font;
}

////////////////////////////////// bool

void KviConfig::writeEntry(const char *szKey,bool bTrue)
{
	__range_valid(szKey);
	m_bDirty = true;
	KviStrDict * p_group = getCurrentGroup();
	KviStr *p_data = new KviStr(bTrue ? "true" : "false");
	p_group->replace(szKey,p_data);
}

bool KviConfig::readBoolEntry(const char *szKey,bool bTrue)
{
	__range_valid(szKey);
	KviStrDict * p_group = getCurrentGroup();
	KviStr * p_str = p_group->find(szKey);
	if(!p_str)return bTrue;
	return (*p_str == "true");
}

////////////////////////////////// QRect

void KviConfig::writeEntry(const char *szKey,const QRect &rct)
{
	__range_valid(szKey);
	m_bDirty = true;
	KviStrDict * p_group = getCurrentGroup();
	KviStr * p_data = new KviStr();
	KviStringConversion::toString(rct,*p_data);
	p_group->replace(szKey,p_data);
}

QRect KviConfig::readRectEntry(const char *szKey,const QRect &rct)
{
	__range_valid(szKey);
	KviStrDict * p_group = getCurrentGroup();
	KviStr * str = p_group->find(szKey);
	if(!str)return rct;
	QRect ret;
	return KviStringConversion::fromString(str->ptr(),ret) ? ret : rct;
}

////////////////////////////////// unsigned short

void KviConfig::writeEntry(const char *szKey,unsigned short usValue)
{
	__range_valid(szKey);
	m_bDirty = true;
	KviStrDict * p_group = getCurrentGroup();
	KviStr *p_data = new KviStr();
	p_data->setNum(usValue);
	p_group->replace(szKey,p_data);
}

unsigned short int KviConfig::readUShortEntry(const char *szKey,unsigned short int usDefault)
{
	__range_valid(szKey);
	KviStrDict * p_group = getCurrentGroup();
	KviStr * p_str = p_group->find(szKey);
	if(!p_str)return usDefault;
	bool bOk;
	unsigned short int usVal=p_str->toUShort(&bOk);
	return bOk ? usVal : usDefault;
}

/*
////////////////////////////////// unsigned long

Unused code
void KviConfig::writeEntry(const char *szKey,unsigned long lValue)
{
	__range_valid(szKey);
	m_bDirty = true;
	KviStrDict * p_group = getCurrentGroup();
	KviStr *p_data = new KviStr();
	p_data->setNum(lValue);
	p_group->replace(szKey,p_data);
}

unsigned long KviConfig::readULongEntry(const char *szKey,unsigned long lDefault)
{
	__range_valid(szKey);
	KviStrDict * p_group = getCurrentGroup();
	KviStr * p_str = p_group->find(szKey);
	if(!p_str)return lDefault;
	bool bOk;
	unsigned long lVal=p_str->toULong(&bOk);
	return bOk ? lVal : lDefault;
}
*/

////////////////////////////////// int

void KviConfig::writeEntry(const char *szKey,int iValue)
{
	__range_valid(szKey);
	m_bDirty = true;
	KviStrDict * p_group = getCurrentGroup();
	KviStr *p_data = new KviStr();
	p_data->setNum(iValue);
	p_group->replace(szKey,p_data);
}

int KviConfig::readIntEntry(const char *szKey,int iDefault)
{
	__range_valid(szKey);
	KviStrDict * p_group = getCurrentGroup();
	KviStr * p_str = p_group->find(szKey);
	if(!p_str)return iDefault;
	bool bOk;
	int iVal=p_str->toInt(&bOk);
	return bOk ? iVal : iDefault;
}

////////////////////////////////// unsigned int

void KviConfig::writeEntry(const char *szKey,unsigned int iValue)
{
	__range_valid(szKey);
	m_bDirty = true;
	KviStrDict * p_group = getCurrentGroup();
	KviStr *p_data = new KviStr();
	p_data->setNum(iValue);
	p_group->replace(szKey,p_data);
}

unsigned int KviConfig::readUIntEntry(const char *szKey,unsigned int iDefault)
{
	__range_valid(szKey);
	KviStrDict * p_group = getCurrentGroup();
	KviStr * p_str = p_group->find(szKey);
	if(!p_str)return iDefault;
	bool bOk;
	unsigned int iVal=p_str->toUInt(&bOk);
	return bOk ? iVal : iDefault;
}

////////////////////////////////// char

void KviConfig::writeEntry(const char *szKey,char iValue)
{
	__range_valid(szKey);
	m_bDirty = true;
	KviStrDict * p_group = getCurrentGroup();
	KviStr *p_data = new KviStr();
	p_data->setNum(iValue);
	p_group->replace(szKey,p_data);
}

char KviConfig::readCharEntry(const char *szKey,char iDefault)
{
	__range_valid(szKey);
	KviStrDict * p_group = getCurrentGroup();
	KviStr * p_str = p_group->find(szKey);
	if(!p_str)return iDefault;
	bool bOk;
	char iVal=p_str->toChar(&bOk);
	return bOk ? iVal : iDefault;
}

////////////////////////////////// unsigned char

void KviConfig::writeEntry(const char *szKey,unsigned char iValue)
{
	__range_valid(szKey);
	m_bDirty = true;
	KviStrDict * p_group = getCurrentGroup();
	KviStr *p_data = new KviStr();
	p_data->setNum(iValue);
	p_group->replace(szKey,p_data);
}

unsigned char KviConfig::readUCharEntry(const char *szKey,unsigned char iDefault)
{
	__range_valid(szKey);
	KviStrDict * p_group = getCurrentGroup();
	KviStr * p_str = p_group->find(szKey);
	if(!p_str)return iDefault;
	bool bOk;
	unsigned char iVal=p_str->toUChar(&bOk);
	return bOk ? iVal : iDefault;
}


#ifdef COMPILE_ON_WINDOWS

	// On windows we need to override new and delete operators
	// to ensure that always the right new/delete pair is called for an object instance
	// This bug is present in all the classes exported by a module that
	// can be instantiated/destroyed from external modules.
	// (this is a well known bug described in Q122675 of MSDN)

	void * KviConfig::operator new(size_t tSize)
	{
		return kvi_malloc(tSize);
	}

	void KviConfig::operator delete(void * p)
	{
		kvi_free(p);
	}

#endif
