#include "board.h"
#include "board.moc"

#include "piece.h"
#include "settings.h"
#include "defines.h"


#define BOARD_INFO Factory::self()->boardInfo()

BaseBoard::BaseBoard(bool graphic, QWidget *parent)
: QCanvasView(parent, "board"),
  GenericTetris(BOARD_INFO.width, BOARD_INFO.height, BOARD_INFO.withPieces,
                graphic),
  state(GameOver), timer(this), sequences(0), main(0), _next(0)
{
    if (graphic) {
        setVScrollBarMode(AlwaysOff);
        setHScrollBarMode(AlwaysOff);
        setFrameStyle( QFrame::Panel | QFrame::Sunken );

        sequences = new SequenceArray;
        main = new BlockInfo(*sequences);
        setCanvas(main);
        _next = new BlockInfo(*sequences);
        setBlockInfo(main, _next);
        connect(&timer, SIGNAL(timeout()), SLOT(timeout()));
    }
}

void BaseBoard::copy(const GenericTetris &g)
{
    GenericTetris::copy(g);
    state = static_cast<const BaseBoard &>(g).state;
}

void BaseBoard::settingsChanged()
{
    _animations = AppearanceSettingsWidget::readAnimations();

    int size = AppearanceSettingsWidget::readBlockSize();
	Q_ASSERT( graphic() );
    if ( BOARD_INFO.blockSizeMustBeOdd && (size%2)==0 ) size--;

	sequences->setBlockSize(size);
    int w = matrix().width() * size;
    int h = matrix().height() * size;
	main->resize(w, h);
    int fw = 2 * frameWidth();
    setFixedSize(fw + w, fw + h);
    const GPieceInfo *info = Piece::info();
    _next->resize((info->maxWidth() + 2)*size, (info->maxHeight() + 2)*size);

	for (uint i=0; i<matrix().width(); i++)
		for (uint j=0; j<firstClearLine(); j++) {
            Coord c(i, j);
            if ( matrix()[c]==0 ) continue;
			partialMoveBlock(c, 0, 0);
        }
	main->update();

	emit blockSizeChanged();
	updateGeometry();
}

BaseBoard::~BaseBoard()
{
    if ( graphic() ) {
        setBlockInfo(0, 0); // destruct all sprites before deleting canvas
        delete _next;
        delete main;
        delete sequences;
    }
}

void BaseBoard::start()
{
	Q_ASSERT( graphic() );
	state = Normal;
	GenericTetris::start(); // NB: the timer is started by updateLevel !
}

void BaseBoard::stop()
{
	timer.stop();
    state = GameOver;
}

void BaseBoard::pause()
{
	Q_ASSERT( graphic() );
    timer.stop();
    _oldState = state;
    state = Paused;
	showBoard(false);
}

void BaseBoard::gameOver()
{
    stop();
    emit gameOverSignal();
}

void BaseBoard::showCanvas(QCanvas *c, bool show)
{
	QCanvasItemList l = c->allItems();
	QCanvasItemList::Iterator it;
	for (it=l.begin(); it!=l.end(); ++it) {
		if (show) (*it)->show();
		else (*it)->hide();
	}
	c->update();
}

void BaseBoard::showBoard(bool show)
{
	showCanvas(main, show);
}

void BaseBoard::unpause()
{
	Q_ASSERT( graphic() );
	showBoard(true);
    state = _oldState;
	startTimer();
}

void BaseBoard::updateRemoved(uint newRemoved)
{
	GenericTetris::updateRemoved(newRemoved);
    emit removedUpdated(newRemoved);
}

void BaseBoard::updateScore(uint newScore)
{
	GenericTetris::updateScore(newScore);
    emit scoreUpdated(newScore);
}

uint BaseBoard::blockSize() const
{
	return sequences->blockSize();
}

int BaseBoard::firstColumnBlock(uint col) const
{
	for (int j=firstClearLine()-1; j>=0; j--) {
        Coord c(col, j);
		if ( matrix()[c]!=0 ) return j;
    }
	return -1;
}

//-----------------------------------------------------------------------------
void BaseBoard::_beforeRemove(bool first)
{
	if ( graphic() ) {
		state = ( beforeRemove(first) ? BeforeRemove : Normal );
		if ( state==BeforeRemove ) {
			startTimer();
			return;
		}
	}
	remove();
	_afterRemove(true);
}

void BaseBoard::remove()
{
    for (uint j=0; j<firstClearLine(); j++)
        for (uint i=0; i<matrix().width(); i++) {
            Coord c(i, j);
            if ( matrix()[c]==0 || !toBeRemoved(c) ) continue;
            removeBlock(c);
        }
    computeInfos();
    if ( graphic() ) main->update();
}

bool BaseBoard::doFall(bool doAll, bool first, bool lineByLine)
{
    Q_ASSERT( !lineByLine || !doAll );

    if ( !doAll ) {
		if (first) loop = 0;
		else loop++;
    }
	bool final = (doAll || lineByLine || loop==BOARD_INFO.nbFallStages);

    for (uint i=0; i<matrix().width(); i++) {
        // compute heights
        // we must separate this computation since toFall() can depend
        // directly on the disposition of blocks under the current one
        // (for e.g. in kfouleggs)
        // we do not rely on firstClearLine() here since this method is
        // used in kfouleggs to  make gift blocks fall down ...
        uint h = 0;
        QMemArray<uint> heights(matrix().height());
        for (uint j=1; j<matrix().height(); j++) { // first line cannot fall
            Coord src(i, j);
            if ( toFall(src) ) h++;
            heights[j] = h;
        }

        // do move
        for (uint j=1; j<matrix().height(); j++) {
            Coord src(i, j);
            if( heights[j]==0 || matrix()[src]==0 ) continue;
            if (lineByLine) final = false;
            uint k = j - (lineByLine ? 1 : heights[j]);
            Coord dest(i, k);
            if ( final || lineByLine ) moveBlock(src, dest);
            else partialBlockFall(src, dest);
        }
    }

    if (final) computeInfos();
    return final;
}

void BaseBoard::_afterRemove(bool first)
{
	AfterRemoveResult r = afterRemove(!graphic(), first);
	switch (r) {
	  case Done:
          state = Normal;
          afterAfterRemove();
          return;
	  case NeedAfterRemove:
          state = AfterRemove;
          startTimer();
          return;
	  case NeedRemoving:
          _beforeRemove(true);
          return;
	}
}

BaseBoard::AfterRemoveResult BaseBoard::afterRemove(bool doAll, bool first)
{
    return (doFall(doAll, first, false) ? Done : NeedAfterRemove);
}

bool BaseBoard::timeout()
{
    Q_ASSERT( graphic() );
    if ( state==GameOver ) return true;
	switch (state) {
        case BeforeRemove: _beforeRemove(FALSE); break;
        case AfterRemove:  _afterRemove(FALSE);  break;
        default:           return false;
	}
	main->update();
    return true;
}

bool BaseBoard::startTimer()
{
    Q_ASSERT( graphic() );
    if ( state==GameOver ) return true;
	switch (state) {
        case BeforeRemove:
            timer.start(BOARD_INFO.beforeRemoveTime, true);
            break;
        case AfterRemove:
            timer.start(BOARD_INFO.afterRemoveTime, true);
            break;
        default:
            return false;
	}
    return true;
}

bool BaseBoard::beforeRemove(bool first)
{
    if (first) loop = 0;
	else loop++;

	for (uint j=0; j<firstClearLine(); j++)
		for (uint i=0; i<matrix().width(); i++) {
            Coord c(i, j);
			if ( toBeRemoved(c) ) matrix()[c]->toggleLight();
        }

	return ( loop!=BOARD_INFO.nbToggles );
}


//-----------------------------------------------------------------------------
void BaseBoard::partialBlockFall(const Coord &src, const Coord &dest)
{
    Q_ASSERT( loop<BOARD_INFO.nbFallStages );

    float c = (float)(loop+1) / BOARD_INFO.nbFallStages * blockSize();
    int dec = (int)dest.i - src.i;
    float xDec = c * dec;
    dec = (int)src.j - dest.j;
    float yDec = c * dec;
    partialMoveBlock(src, (int)xDec, (int)yDec);
}

uint BaseBoard::findGroup(Matrix<int> &field, const Coord &c) const
{
    uint nb = 0;
    _findGroup(field, c, nb, false);
    return nb;
}

void BaseBoard::setGroup(Matrix<int> &field, const Coord &c, uint nb) const
{
    _findGroup(field, c, nb, true);
}

void BaseBoard::_findGroup(Matrix<int> &field, const Coord &c,
                           uint &nb, bool set) const
{
    if (!set) nb++;
	field[c] = (set ? (int)nb : -1);
    Coord nc;
    uint value = matrix()[c]->value();
    for (uint i=0; i<4; i++)
        if ( matrix().neighbour(c, i, nc) )
            blockInGroup(field, nc, value, nb, set);
}

void BaseBoard::blockInGroup(Matrix<int> &field, const Coord &c, uint value,
                             uint &nb, bool set) const
{
	if ( matrix()[c]==0 ) return;
	if ( matrix()[c]->value()!=value ) return;
	if ( field[c]!=(set ? -1 : 0) ) return;
	_findGroup(field, c, nb, set);
}

QMemArray<uint> BaseBoard::findGroups(Matrix<int> &field, uint minSize,
									  bool exitAtFirstFound) const
{
    field.fill(0);
    QMemArray<uint> groups;
 	for (uint j=0; j<firstClearLine(); j++)
		for (uint i=0; i<matrix().width(); i++) {
            Coord c(i, j);
			if ( matrix()[c]==0 || matrix()[c]->isGarbage() ) continue;
			if ( field[c]!=0 ) continue;
            uint nb = findGroup(field, c);
            setGroup(field, c, nb);
            if ( nb>=minSize ) {
				uint s = groups.size();
				groups.resize(s+1);
				groups[s] = nb;
                if (exitAtFirstFound) return groups;
			}
		}
    return groups;
}

uint BaseBoard::drawCode(const Coord &c) const
{
	uint v = matrix()[c]->value();
	uint code = 0;
    Coord nc;
    for (uint i=0; i<4; i++) {
        if ( !matrix().neighbour(c, i, nc) ) continue;
        if ( matrix()[nc]==0 || matrix()[nc]->value()!=v ) continue;
        code |= Coord::DIRECTIONS[i].code;
    }
	return code;
}

void BaseBoard::computeNeighbours()
{
	for (uint j=0; j<firstClearLine(); j++)
		for (uint i=0; i<matrix().width(); i++) {
            Coord c(i, j);
			if ( matrix()[c]==0 || matrix()[c]->isGarbage() ) continue;
			matrix()[c]->sprite()->setFrame( drawCode(c) );
		}
}
