/***************************************************************************
                          csocket.cpp  -  description
                             -------------------
    begin                : Sat Oct 6 2001
    copyright            : (C) 2001 by Mathias Kster
    email                : mathen@ketelhot.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <errno.h>

#ifndef WIN32
#include <netdb.h>
#include <malloc.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <net/if.h>
#endif

#include <sys/types.h>

#ifdef WIN32
#include <winsock2.h>
#endif

#ifndef MSG_NOSIGNAL
// 0x2000  /* don't raise SIGPIPE */
#define MSG_NOSIGNAL      0
#endif

#include <dclib/casyncdns.h>

#include "csocket.h"

CSocket::CSocket()
{
	SocketType = estTCP;
	iHandle    = INVALID_SOCKET;
	sError     = "";

#ifdef WIN32
	WORD wVersionRequested;
	WSADATA wsaData;
	int err;
 
	wVersionRequested = MAKEWORD( 2, 2 );
 
	err = WSAStartup( wVersionRequested, &wsaData );

	if ( err != 0 )
	{
		/* Tell the user that we could not find a usable */
		/* WinSock DLL.                                  */
		return;
	}
 
	/* Confirm that the WinSock DLL supports 2.2.*/
	/* Note that if the DLL supports versions greater    */
	/* than 2.2 in addition to 2.2, it will still return */
	/* 2.2 in wVersion since that is the version we      */
	/* requested.                                        */
 
	if ( LOBYTE( wsaData.wVersion ) != 2 ||
		HIBYTE( wsaData.wVersion ) != 2 )
	{
		/* Tell the user that we could not find a usable */
		/* WinSock DLL.                                  */
		WSACleanup( );
		return; 
	}
 
	/* The WinSock DLL is acceptable. Proceed. */
#endif
}

CSocket::~CSocket()
{
	if ( iHandle != INVALID_SOCKET)
	{
#ifdef WIN32
		closesocket(iHandle);
#else
		close(iHandle);
#endif
	}
}

/** */
CString CSocket::GetHostByName( CString Host )
{
	CString s = "";
	struct hostent *hp;
	struct sockaddr_in sin;

	if ( Host == "" )
	{
		return s;
	}

	if ((hp = gethostbyname(Host.Data())) == 0)
	{
		return s;
	}

 	memcpy((void *)&sin.sin_addr, hp->h_addr, hp->h_length);

	s = inet_ntoa(sin.sin_addr);

	return s;
}

/** */
CString CSocket::GetInterfaceIP( CString iface )
{
	CString s = "";
#ifndef WIN32
	int sock = -1;
	struct ifreq ifreqa[256];
	int i;
	int count;
	struct ifconf *pifconf;
	struct ifreq * pifreq;
	struct ifreq ifr;

	sock = socket( PF_INET, SOCK_DGRAM, 0 );

	if ( sock == -1 )
	{
		return s;
	}

	pifconf = (struct ifconf*) malloc(sizeof(ifconf));

	pifconf->ifc_len = 255*32;
	pifconf->ifc_ifcu.ifcu_req = ifreqa;

	i = ioctl( sock, SIOCGIFCONF, pifconf );

	if ( i == -1 )
	{
		close(sock);
		free(pifconf);
		return s;
	}

	if ( pifconf->ifc_len == 0 )
	{
		close(sock);
		free(pifconf);
		return s;
	}

	count = pifconf->ifc_len / sizeof( struct ifreq );

	pifreq = pifconf->ifc_ifcu.ifcu_req;

	for(i=0;i<count;i++)
	{
// change for freebsd ...
//		if ( iface == (char*)pifreq[i].ifr_ifrn.ifrn_name )
//		{
//			memcpy( ifr.ifr_name,pifreq[i].ifr_ifrn.ifrn_name,16);
//
		if ( iface == (char*)pifreq[i].ifr_name )
		{
			memcpy (ifr.ifr_name,pifreq[i].ifr_name,16);

			if ( ioctl( sock, SIOCGIFADDR, &ifr ) != -1 )
			{
				s = inet_ntoa(((sockaddr_in*)&ifr.ifr_broadaddr)->sin_addr);
			}

			break;
		}
	}

	close(sock);
	free(pifconf);
#endif
	return s;
}

/** */
int CSocket::GetFreeSendBufferSize()
{
	int tot, free;
#ifndef WIN32
	int unsent;
#endif
	socklen_t ilen;

	free = 0;

	ilen = sizeof(int);

	if ( getsockopt(iHandle, SOL_SOCKET, SO_SNDBUF, (char*)&tot, &ilen) == 0 )
	{
		// quick & dirty fix for free bsd
		free = tot;
#ifndef WIN32
		if ( ioctl(iHandle, TIOCOUTQ, &unsent) == 0 )
		{
			free = tot - unsent;
		}
#endif
	}

	return free;
}

/** */
int CSocket::IsConnect()
{
	int i,err;
	struct timeval tv;
	fd_set readset;

	if ( iHandle == INVALID_SOCKET )
	{
		return -1;
	}

	FD_ZERO(&readset);
	FD_SET(iHandle, &readset);

	tv.tv_sec  = 0;
	tv.tv_usec = 10;

	err = -1;

	i = select(FD_SETSIZE, &readset, &readset, NULL, &tv);

	if ( i > 0 )
	{
		err = 1;
	}
	else if ( i == 0 )
	{
		err = 0;
	}

	FD_CLR(iHandle, &readset);

	return err;
}

/** */
eConnectState CSocket::Connect( CString Host, int Port, int iAsync )
{
	struct hostent *hp;
	struct sockaddr_in sin;
	SOCKET sockfd;

	if ( iHandle != INVALID_SOCKET )
	{
		Disconnect();
	}

	memset((void *)&sin, 0, sizeof(sin));
	sin.sin_family = AF_INET;

	if ( SocketType == estTCP )
	{
		if ( iAsync == 0 )
		{
			// use blocked gethostbyname
			if ((hp = gethostbyname(Host.Data())) == 0)
			{
#ifdef WIN32
				sError = strerror(WSAGetLastError());
#else
				sError = hstrerror(h_errno);
#endif
				return ecsERROR;
			}
			else if ( hp->h_addr == 0 )
			{
				return ecsERROR;
			}
			else
			{
			 	memcpy((void *)&sin.sin_addr, hp->h_addr, hp->h_length);
			}
		}
		else
		{
			// use async dns class
			eAsyncDns ead = pAsyncDns.GetHostByName( Host, &sin.sin_addr );

			if ( ead == eadAGAIN )
			{
				return ecsAGAIN;
			}
			else if ( ead == eadERROR )
			{
				return ecsERROR;
			}
		}
	}
	else
	{
		sin.sin_addr.s_addr = htonl(INADDR_ANY);
	}

	sin.sin_port = htons((unsigned short)Port);

	// socket
	if ( SocketType == estTCP )
	{
		if ((sockfd = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0)
		{
			sError = strerror(errno);
			return ecsERROR;
		}
	}
	else
	{
		if( (sockfd = socket( AF_INET, SOCK_DGRAM, 0 )) < 0 )
		{
			sError = strerror(errno);
			return ecsERROR;
		}
	}

	// set async flag
#ifdef WIN32
	unsigned long flag = iAsync;
	if ( ioctlsocket(sockfd, FIONBIO, &flag ) != 0 )
#else
	int flag = iAsync;
	if ( ioctl(sockfd, FIONBIO, &flag ) != 0 )
#endif
	{
		sError = strerror(errno);
		return ecsERROR;
	}

	// connect
	if ( SocketType == estTCP )
	{
		if ( connect(sockfd, (struct sockaddr *)&sin, sizeof(sin)) != 0)
		{
#ifdef WIN32
			int e = WSAGetLastError();
			if ( (e != WSAEINPROGRESS) && (e != WSAEWOULDBLOCK) )
#else
			if ( errno != EINPROGRESS )
#endif
			{
				sError = strerror(errno);
#ifdef WIN32
				closesocket(sockfd);
#else
				close(sockfd);
#endif
				return ecsERROR;
			}
		}
	}
	else
	{
		if ( bind(sockfd, (struct sockaddr *) &sin, sizeof(sin)) < 0 )
		{
			sError = strerror(errno);
#ifdef WIN32
			closesocket(sockfd);
#else
			close(sockfd);
#endif
			return ecsERROR;
		}
	}

	iHandle = sockfd;

	return ecsSUCCESS;
}

/** */
int CSocket::Disconnect()
{
	if ( iHandle != INVALID_SOCKET )
	{
#ifdef WIN32
		closesocket(iHandle);
#else
		close(iHandle);
#endif
		iHandle = INVALID_SOCKET;
	}

	return 0;
}

/** */
int CSocket::Read( char * buffer, int len, int sec_timeout, int usec_timeout )
{
	int l,i;
	struct timeval tv;
	fd_set readset;
	struct sockaddr cli_addr;
	socklen_t cli_len;

	cli_len = sizeof(cli_addr);

	l = 0;

	if ( iHandle == INVALID_SOCKET )
	{
		return -1;
	}

	FD_ZERO(&readset);
	FD_SET(iHandle, &readset);

	tv.tv_sec  = sec_timeout;
	tv.tv_usec = usec_timeout;

	i = select(FD_SETSIZE, &readset, NULL, NULL, &tv);

	if ( i > 0 )
	{
		// handle udp socket
		if ( SocketType == estUDP )
		{
			if ( (l=recvfrom(iHandle,buffer,len,0,&cli_addr,&cli_len)) < 0 )
			{
#ifdef WIN32
				int e = WSAGetLastError();
				if ( (e != WSAEINPROGRESS) && (e != WSAEWOULDBLOCK) )
				{
#else
				if ( (errno != EINPROGRESS) && (errno != EAGAIN) )
				{
					sError = strerror(errno);
#endif
				}
				else
				{
					i = l = 0;
				}
			}
		}
		// handle tcp socket
		else if ( SocketType == estTCP )
		{
			if ( (l=recv(iHandle,buffer,len,0)) < 0 )
			{
				sError = strerror(errno);
			}
#if 0
			else
			{
				FILE * f=fopen("dcsocket.log","a");
				if(f!=0)
				{
					sError = strerror(errno);
					fprintf(f,"recv: %d\n",l);
					if( l <= len )
					{
						buffer[l]=0;
						fprintf(f,"%s\n",buffer);
					}
					fclose(f);
				}
			}
#endif
		}
	}
	else if ( i < 0 )
	{
		l = -1;
		sError = strerror(errno);
	}

	FD_CLR(iHandle, &readset);

	/** we are disconnected */
	if ( (i==1) && (l==0) )
		l = -1;

	return l;
}

/** */
int CSocket::Write( const char * buffer, int len, int sec_timeout, int usec_timeout )
{
	int i;
	struct timeval tv;
	fd_set writeset;

	if ( iHandle == INVALID_SOCKET )
	{
		return -1;
	}


	FD_ZERO(&writeset);
	FD_SET(iHandle, &writeset);

	tv.tv_sec  = sec_timeout;
	tv.tv_usec = usec_timeout;

	i = select(FD_SETSIZE, NULL, &writeset, NULL, &tv);

	if ( i > 0 )
	{
		// handle tcp socket
		if ( SocketType == estTCP )
		{
			i = send( iHandle, buffer, len, MSG_NOSIGNAL );
		}
		else
		{
			printf("wrong socket type\n");//fflush(stdout);
			i = -1;
		}

		// send error or wrong socket type
		if ( i < 0 )
		{
#ifdef WIN32
			int e = WSAGetLastError();
			if ( (e != WSAEINPROGRESS) && (e != WSAEWOULDBLOCK) )
			{
#else
			if ( errno != EINPROGRESS )
			{
				sError = strerror(errno);
#endif
			}
			else
			{
				i = 0;
			}
		}
#if 0
		else
		{
			FILE * f=fopen("dcsocket.log","a");
			if(f!=0)
			{
				sError = strerror(errno);
				fprintf(f,"send: %d\n%s\n",len,buffer);
				fclose(f);
			}
		}
#endif
	}


	FD_CLR(iHandle, &writeset);

	return i;
}

/** */
int CSocket::Accept()
{
	struct timeval tv;
	fd_set readset;
	struct sockaddr_in serv_addr;
	SOCKET s = -1;
	int i;
	socklen_t sin_size;

	sin_size = sizeof(struct sockaddr_in);

	FD_ZERO(&readset);
	FD_SET(iHandle, &readset);

	tv.tv_sec  = 0;
	tv.tv_usec = 100;

	i = select(FD_SETSIZE, &readset, NULL, NULL, &tv);

	if ( i > 0 )
	{
	    if ((s = accept(iHandle, (struct sockaddr *)&serv_addr, &sin_size)) == INVALID_SOCKET)
	    {
		return -1;
	    }
	}
	
	return s;
}

/** */
int CSocket::Listen( int port )
{
	SOCKET sockfd;
	struct sockaddr_in serv_addr;

	if ((sockfd = socket(AF_INET, SOCK_STREAM, 0)) == INVALID_SOCKET)
	{
#ifdef WIN32
		sError = strerror(WSAGetLastError());
#else
		sError = strerror(errno);
#endif
		return -1;
	}

	serv_addr.sin_family = AF_INET;
	serv_addr.sin_port = htons((unsigned short)port);
	serv_addr.sin_addr.s_addr = INADDR_ANY;

	if (bind(sockfd, (struct sockaddr *)&serv_addr, sizeof(struct sockaddr)) == -1)
	{
#ifdef WIN32
		sError = strerror(WSAGetLastError());
#else
		sError = strerror(errno);
#endif
		return -1;
	}

	if (listen(sockfd, 5) == -1)
	{
#ifdef WIN32
		sError = strerror(WSAGetLastError());
#else
		sError = strerror(errno);
#endif
		return -1;
	}

	iHandle = sockfd;

	return 0;
}

/** */
int CSocket::GetPeerName( CString * host, int * port )
{
	struct sockaddr_in addr;
	socklen_t sin_size;

	sin_size = sizeof(struct sockaddr_in);

	if ( (!host) || (!port) )
	{
		return -1;
	}

	if ( getpeername( iHandle, (struct sockaddr*)&addr, &sin_size ) == -1 )
	{
		sError = strerror(errno);
		return -1;
	}

	*host = inet_ntoa(addr.sin_addr);
	*port = ntohs(addr.sin_port);

	return 0;
}
